/*
 * @(#)ClusterMessageAckInfo.java	1.14 09/19/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.io.*;
import java.util.*;
import java.nio.*;
import com.sun.messaging.jmq.util.UID;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.io.Status;
import com.sun.messaging.jmq.io.SysMessageID;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.FaultInjection;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.multibroker.Cluster;
import com.sun.messaging.jmq.jmsserver.multibroker.ClusterGlobals;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;

/**
 * An instance of this class is intended to be used one direction only
 */

public class ClusterMessageAckInfo 
{
    protected Logger logger = Globals.getLogger();

    private SysMessageID sysid = null;
    private ConsumerUID intid =  null; 
    private int ackType;
    private Long ackackXid = null;
    private Map optionalProps = null;
    private Long transactionID = null;
    private BrokerAddress msgHome;
    private Cluster c = null;

    private GPacket pkt = null;
    private DataInputStream dis = null;
    private static FaultInjection fi = FaultInjection.getInjection(); 

    private ClusterMessageAckInfo(SysMessageID sysid, ConsumerUID intid,
                                  int ackType, Long ackackXid, Map optionalProps, 
                                  Long transactionID, BrokerAddress msgHome, Cluster c) {
        this.sysid = sysid;
        this.intid = intid;
        this.ackType = ackType;
        this.ackackXid = ackackXid;
        this.optionalProps = optionalProps;
        this.transactionID = transactionID;
        this.msgHome =  msgHome;
        this.c = c;
    }

    private ClusterMessageAckInfo(GPacket pkt, Cluster c) {
        this.pkt = pkt;
        this.c = c;
    }

    private ClusterMessageAckInfo(GPacket pkt) {
        this.pkt = pkt;
        this.c = null;
    }

    public static ClusterMessageAckInfo newInstance(SysMessageID mid,
                             ConsumerUID cid, int ackType, Long ackackXid, 
                             Map optionalProps, Long transactionID,
                             BrokerAddress msgHome, Cluster c) {
        return new ClusterMessageAckInfo(mid, cid, ackType, ackackXid, 
                                         optionalProps, transactionID, msgHome, c);
    }

    /**
     * GPacket to Destination 
     *
     * @param pkt The GPacket to be unmarsheled
     */
    public static ClusterMessageAckInfo newInstance(GPacket pkt, Cluster c) {
        return new ClusterMessageAckInfo(pkt, c);
    }

    public GPacket getGPacket() throws IOException { 
        /*
        if (transactionID != null && ackType != ClusterGlobals.MB_MSG_TXN_ACK) {
            if (sysid != null || intid != null) {
                throw new IOException(Globals.getBrokerResources().getKString(
                BrokerResources.E_INTERNAL_BROKER_ERROR, 
                "Unexpected argument for ackType "+ackType + " in transaction"));
            }
            //also check must not be null
        } */

        GPacket gp = GPacket.getInstance();
        gp.setType(ProtocolGlobals.G_MESSAGE_ACK);
        gp.putProp("T", new Integer(ackType));
        c.marshalBrokerAddress(c.getSelfAddress(), gp);
        gp.putProp("messageBrokerSession", new Long(msgHome.getBrokerSessionUID().longValue()));
        if (msgHome.getHAEnabled()) {
        gp.putProp("messageStoreSession", new Long(msgHome.getStoreSessionUID().longValue()));
        }

        if (optionalProps != null) {
            Object pn = null;
            Iterator itr = optionalProps.keySet().iterator();
            while (itr.hasNext()) {
                pn =  itr.next();
                gp.putProp(pn, optionalProps.get(pn));
            }
        }

        if (transactionID != null) {
            gp.putProp("transactionID", transactionID);
        }
        if (ackackXid != null) {
            gp.setBit(gp.A_BIT, true);
            gp.putProp("X", ackackXid);
        }

        if (ackType == ClusterGlobals.MB_MSG_TXN_PREPARE ||
            ackType == ClusterGlobals.MB_MSG_TXN_ROLLEDBACK ||
            (ackType == ClusterGlobals.MB_MSG_CONSUMED && 
             transactionID != null)) {
            return gp;
        }

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);

        sysid.writeID(dos);
        ClusterConsumerInfo.writeConsumerUID(intid, dos);

        dos.flush();
        bos.flush();

        byte[] buf = bos.toByteArray();
        gp.setPayload(ByteBuffer.wrap(buf));

        return gp;
    }

    public int getAckType() {
        assert ( pkt != null );
        return ((Integer) pkt.getProp("T")).intValue();
    }

    public Map getOptionalProps() {
        assert ( pkt != null );
        Set keys = pkt.propsKeySet();
        if (keys == null || keys.size() == 0) return null;
        Map m = new HashMap();
        Object key = null;
        Iterator itr = keys.iterator();
        while (itr.hasNext()) {
            key = itr.next();   
            m.put(key, pkt.getProp(key));
        }
        return m;
    }
    
    public UID getMessageStoreSessionUID() {
        assert ( pkt != null );
        Long ssid = (Long)pkt.getProp("messageStoreSession"); 
        if (ssid == null) return null;
        return new UID(ssid.longValue());
    }

    public UID getMessageBrokerSessionUID() {
        assert ( pkt != null );
        Long bsid = (Long)pkt.getProp("messageBrokerSession"); 
        if (bsid == null) return null;
        return new UID(bsid.longValue());
    }

    public Long getTransactionID() {
        assert ( pkt != null );
        return  (Long)pkt.getProp("transactionID");
    }

    /**
     * must called in the following order: 
     * 
     * initPayloadRead()
     * readPayloadSysMesssageID()
     * readPayloadConsumerUID()
     *
     */
    public void initPayloadRead() {
        assert ( pkt != null );

        byte[] buf = pkt.getPayload().array();
        ByteArrayInputStream bis = new ByteArrayInputStream(buf);
        dis = new DataInputStream(bis);
    }

    public SysMessageID readPayloadSysMessageID() throws IOException {
        assert ( dis != null );

        SysMessageID sysid = new SysMessageID();
        sysid.readID(dis);
        return sysid;
    }

    public ConsumerUID readPayloadConsumerUID() throws Exception { 
        assert ( dis != null );
        ConsumerUID intid =  ClusterConsumerInfo.readConsumerUID(dis);
        if (c != null) {
            BrokerAddress ba = c.unmarshalBrokerAddress(pkt);
            if (ba != null) intid.setBrokerAddress(ba);
        }
        return intid;
    }

    /**
     * Only to be used to aid toString for ack pkt
     */
    public void initToString(SysMessageID mid, ConsumerUID cid) {
        assert ( pkt != null );
        this.sysid = mid;
        this.intid = cid;
    }
    
    public boolean needReply() {
        assert ( pkt != null );
        return pkt.getBit(pkt.A_BIT);
    }

    public GPacket getReplyGPacket(int status, String reason) {
        assert ( pkt != null );

        GPacket gp = GPacket.getInstance();
        gp.setType(ProtocolGlobals.G_MESSAGE_ACK_REPLY);
        gp.putProp("X", (Long)pkt.getProp("X"));
        gp.putProp("T", new Integer(getAckType()));
        if (pkt.getProp("messageBrokerSession") != null) {
            gp.putProp("messageBrokerSession", pkt.getProp("messageBrokerSession"));
        }
        if (pkt.getProp("messageStoreSession") != null) {
            gp.putProp("messageStoreSession", pkt.getProp("messageStoreSession"));
        }
        if (pkt.getProp("transactionID") != null) {
            gp.putProp("transactionID", pkt.getProp("transactionID"));
        }
        gp.putProp("S", new Integer(status));
        if (reason != null) gp.putProp("reason", reason);

        if (pkt.getPayload() != null) {
            gp.setPayload(ByteBuffer.wrap(pkt.getPayload().array()));
        }
        return gp;
    }

    /** 
     * To be used by ack sender
     */
    public String toString() {
        if (pkt == null) {

        StringBuffer buf = new StringBuffer();
        buf.append("\n\tAckType = ").append(ackType);

        buf.append("\n\tMessageBrokerSession = ").append(msgHome.getBrokerSessionUID().longValue());
        if (msgHome.getHAEnabled()) {
            buf.append("\n\tMessageStoreSession = ").append(msgHome.getStoreSessionUID().longValue());
        }

        if (transactionID != null) {
           buf.append("\n\tTransactionID = ").append(transactionID);
        }

        if (ackackXid != null) {
           buf.append("\n\tAckAck = ").append("true");
        }

        buf.append("\n\tMessage Home = ").append(msgHome);

        if (sysid != null) {
            buf.append("\n\tSysMessageID = ").append(sysid);
        }
        if (intid != null) {
            buf.append("\n\tConsumerUID = ").append(intid);
        }
        buf.append("\n");
        return buf.toString();
        }

        assert ( pkt !=  null );

        StringBuffer buf = new StringBuffer();
        buf.append("\n\tAckType = ").append(getAckType());

        if (getMessageBrokerSessionUID() != null) {
        buf.append("\n\tMessageBrokerSession = ").append(getMessageBrokerSessionUID().longValue());
        }
        if (getMessageStoreSessionUID() != null) {
        buf.append("\n\tMessageStoreSession = ").append(getMessageStoreSessionUID().longValue());
        }

        if (getTransactionID() != null) {
        buf.append("\n\tTransactionID = ").append(getTransactionID());
        }

        if (sysid != null) {
            buf.append("\n\tSysMessageID = ").append(sysid);
        }
        if (intid != null) {
            buf.append("\n\tConsumerUID = ").append(intid);
        }
        buf.append("\n");

        return buf.toString();
    }

    public static Long getAckAckXid(GPacket ackack) {
        return (Long)ackack.getProp("X");
    }

    public static Integer getAckAckType(GPacket ackack) {
        return (Integer)ackack.getProp("T");
    }

    public static int getAckAckStatus(GPacket ackack) {
        return ((Integer)ackack.getProp("S")).intValue();
    }

    public static String getAckAckStatusReason(GPacket ackack) {
        return (String)ackack.getProp("reason");
    }

    /**
     * To be used for ackack pkt 
     */
    public static String toString(GPacket ackack) {
        int acktyp = getAckAckType(ackack).intValue();

        StringBuffer buf = new StringBuffer();
        buf.append("\n\tackStatus = ").append(Status.getString(getAckAckStatus(ackack)));

        if (ackack.getProp("reason") != null) {
        buf.append("\n\tReason = ").append(getAckAckStatusReason(ackack));
        }

        buf.append("\n\tAckType = ").append(acktyp);

        if (ackack.getProp("messageBrokerSession") != null) {
            buf.append("\n\tMessageBrokerSession = ").append(ackack.getProp("messageBrokerSession"));
        }
        if (ackack.getProp("messageStoreSession") != null) {
            buf.append("\n\tMessageStoreSession = ").append(ackack.getProp("messageStoreSession"));
        }

        if (ackack.getProp("transactionID") != null) {
            buf.append("\n\tTransactionID = ").append(ackack.getProp("transactionID"));
        }

        if (ackack.getPayload() != null) {
            ClusterMessageAckInfo cai = new ClusterMessageAckInfo(ackack);
            try {
                cai.initPayloadRead();
                buf.append("\n\tSysMessageID = ").append(cai.readPayloadSysMessageID());
                buf.append("\n\tConsumerUID = ").append(cai.readPayloadConsumerUID().longValue());
                buf.append("\n");

            } catch (Exception e) {
                Globals.getLogger().logStack(Logger.WARNING, e.getMessage(), e);
            }
        }
        return buf.toString();
    }

    private static String convertAckTypeToFaultString(int ackType, Long txnID) {
         switch (ackType) {
             case ClusterGlobals.MB_MSG_CONSUMED:
                 return ((txnID == null) ? "":
                         FaultInjection.MSG_REMOTE_ACK_TXNCOMMIT);
             case ClusterGlobals.MB_MSG_TXN_ACK:
                 return FaultInjection.MSG_REMOTE_ACK_TXNACK;
             case ClusterGlobals.MB_MSG_TXN_PREPARE:
                 return FaultInjection.MSG_REMOTE_ACK_TXNPREPARE;
             case ClusterGlobals.MB_MSG_TXN_ROLLEDBACK:
                 return FaultInjection.MSG_REMOTE_ACK_TXNROLLBACK;
        }
        return null;
    }

    public static void CHECKFAULT(HashMap ackCounts, 
                                  int ackType, Long txnID,
                                  String fprefix, String fstage) {
        int ackCount = 0;
        HashMap fips = new HashMap();
        Integer ak = new Integer(ackType);
        synchronized(ackCounts) {
            Integer v = (Integer)ackCounts.get(ak);
            if (v != null) ackCount = v.intValue();
            if (fstage.equals(FaultInjection.STAGE_1)) {
                ackCounts.put(ak, new Integer(++ackCount));
            }
        }
        fips.put(FaultInjection.MSG_ACKCOUNT_PROP, new Integer(ackCount));

        fi.checkFaultAndExit(fprefix+
                             convertAckTypeToFaultString(ackType, txnID)+
                             fstage, null, 2, false);
    }

}
