/*
 * @(#)ClusterDestInfo.java	1.6 04/26/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.io.*;
import java.util.*;
import java.nio.*;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.io.PacketProperties;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;

/**
 * An instance of this class is intended to be used one direction only
 * either Destination -> GPacket or GPacket -> Destination (see assertions)
 */

public class ClusterDestInfo 
{
    private Destination d = null;

    private GPacket pkt = null;
    private String destName = null;
    private int destType = -1;

    private ClusterDestInfo(Destination d) {
        this.d = d;
    }

    private ClusterDestInfo(GPacket pkt) {
        assert (pkt.getType() == ProtocolGlobals.G_REM_DESTINATION || 
                pkt.getType() == ProtocolGlobals.G_UPDATE_DESTINATION);

        this.pkt = pkt;
        destName = (String) pkt.getProp("N");
        destType = ((Integer) pkt.getProp("DT")).intValue();
    }

    /**
     * Destination to GPacket
     *
     * @param d The Destination to be marshaled to GPacket
     */
    public static ClusterDestInfo newInstance(Destination d) {
        return new ClusterDestInfo(d);
    }

    /**
     * GPacket to Destination 
     *
     * @param pkt The GPacket to be unmarsheled
     */
    public static ClusterDestInfo newInstance(GPacket pkt) {
        return new ClusterDestInfo(pkt);
    }

    public GPacket getGPacket(short protocol, boolean changeRecord) { 
        assert (d !=  null);
        assert (protocol == ProtocolGlobals.G_REM_DESTINATION || 
                protocol == ProtocolGlobals.G_UPDATE_DESTINATION);
        GPacket gp = GPacket.getInstance();
        gp.setType(protocol);
        gp.putProp("N", d.getDestinationName());
        gp.putProp("DT", new Integer(d.getType()));

        switch (protocol) {
           case ProtocolGlobals.G_REM_DESTINATION:
           break;

           case ProtocolGlobals.G_UPDATE_DESTINATION:

           if (DestType.isTemporary(d.getType())) {
               gp.putProp("connectionUID", new Long(d.getConnectionUID().longValue()));
           }

           HashMap props = d.getDestinationProperties();
           if (props == null) props = new HashMap();
           ByteArrayOutputStream bos = new ByteArrayOutputStream();
           try {
               PacketProperties.write(props, bos);
               bos.flush();
           }
           catch (IOException e) { /* Ignore */ }

           byte[] buf = bos.toByteArray();
           gp.setPayload(ByteBuffer.wrap(buf));
           break;

        }
        if (changeRecord) gp.putProp("M", new Boolean(true));

        return gp;
    }

    public DestinationUID getDestUID() {
        assert (destName != null);
        return DestinationUID.getUID(destName, DestType.isQueue(destType));
    }

    public int getDestType() {
        assert (pkt != null);
        return destType;
    }

    public String getDestName() {
        assert (pkt != null);
        return destName;
    }

    public Hashtable getDestProps() throws IOException, ClassNotFoundException {
        assert (pkt != null); 
        ByteArrayInputStream bis = new ByteArrayInputStream(pkt.getPayload().array());
        return PacketProperties.parseProperties(bis);
    }

}
