/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.sam;

import htsjdk.samtools.SAMRecord;
import java.util.ArrayList;
import java.util.List;
import org.broadinstitute.gatk.utils.exceptions.ReviewedGATKException;
import org.broadinstitute.gatk.utils.sam.ArtificialSingleSampleReadStream;

public class ArtificialSingleSampleReadStreamAnalyzer {
    protected ArtificialSingleSampleReadStream originalStream;
    protected SAMRecord lastRead;
    protected int totalReads;
    protected boolean allSamplesMatch;
    protected int numContigs;
    protected List<Integer> stacksPerContig;
    protected Integer minReadsPerStack;
    protected Integer maxReadsPerStack;
    protected Integer minDistanceBetweenStacks;
    protected Integer maxDistanceBetweenStacks;
    protected Integer minReadLength;
    protected Integer maxReadLength;
    protected int numUnmappedReads;
    protected int currentContigNumStacks;
    protected int currentStackNumReads;

    public ArtificialSingleSampleReadStreamAnalyzer(ArtificialSingleSampleReadStream originalStream) {
        this.originalStream = originalStream;
        this.reset();
    }

    public void reset() {
        this.lastRead = null;
        this.totalReads = 0;
        this.allSamplesMatch = true;
        this.numContigs = 0;
        this.stacksPerContig = new ArrayList<Integer>();
        this.minReadsPerStack = null;
        this.maxReadsPerStack = null;
        this.minDistanceBetweenStacks = null;
        this.maxDistanceBetweenStacks = null;
        this.minReadLength = null;
        this.maxReadLength = null;
        this.numUnmappedReads = 0;
        this.currentContigNumStacks = 0;
        this.currentStackNumReads = 0;
    }

    public void analyze(Iterable<SAMRecord> stream) {
        for (SAMRecord read : stream) {
            this.update(read);
        }
        this.finalizeStats();
    }

    public void validate() {
        if ((this.originalStream.getNumContigs() == 0 || this.originalStream.getNumStacksPerContig() == 0) && this.originalStream.getNumUnmappedReads() == 0) {
            if (this.totalReads != 0) {
                throw new ReviewedGATKException("got reads from the stream, but the stream was configured to have 0 reads");
            }
            return;
        }
        if (this.totalReads == 0) {
            throw new ReviewedGATKException("got no reads from the stream, but the stream was configured to have > 0 reads");
        }
        if (!this.allSamplesMatch) {
            throw new ReviewedGATKException("some reads had the wrong sample");
        }
        if (this.numContigs != this.originalStream.getNumContigs()) {
            throw new ReviewedGATKException("number of contigs not correct");
        }
        if (this.stacksPerContig.size() != this.originalStream.getNumContigs()) {
            throw new ReviewedGATKException(String.format("bug in analyzer code: calculated sizes for %d contigs even though there were only %d contigs", this.stacksPerContig.size(), this.originalStream.getNumContigs()));
        }
        for (int contigStackCount : this.stacksPerContig) {
            if (contigStackCount == this.originalStream.getNumStacksPerContig()) continue;
            throw new ReviewedGATKException("contig had incorrect number of stacks");
        }
        if (this.originalStream.getNumStacksPerContig() > 0) {
            if (this.minReadsPerStack < this.originalStream.getMinReadsPerStack()) {
                throw new ReviewedGATKException("stack had fewer than the minimum number of reads");
            }
            if (this.maxReadsPerStack > this.originalStream.getMaxReadsPerStack()) {
                throw new ReviewedGATKException("stack had more than the maximum number of reads");
            }
        } else if (this.minReadsPerStack != null || this.maxReadsPerStack != null) {
            throw new ReviewedGATKException("bug in analyzer code: reads per stack was calculated even though 0 stacks per contig was specified");
        }
        if (this.originalStream.getNumStacksPerContig() > 1) {
            if (this.minDistanceBetweenStacks < this.originalStream.getMinDistanceBetweenStacks()) {
                throw new ReviewedGATKException("stacks were separated by less than the minimum distance");
            }
            if (this.maxDistanceBetweenStacks > this.originalStream.getMaxDistanceBetweenStacks()) {
                throw new ReviewedGATKException("stacks were separated by more than the maximum distance");
            }
        } else if (this.minDistanceBetweenStacks != null || this.maxDistanceBetweenStacks != null) {
            throw new ReviewedGATKException("bug in analyzer code: distance between stacks was calculated even though numStacksPerContig was <= 1");
        }
        if (this.minReadLength < this.originalStream.getMinReadLength()) {
            throw new ReviewedGATKException("read was shorter than the minimum allowed length");
        }
        if (this.maxReadLength > this.originalStream.getMaxReadLength()) {
            throw new ReviewedGATKException("read was longer than the maximum allowed length");
        }
        if (this.numUnmappedReads != this.originalStream.getNumUnmappedReads()) {
            throw new ReviewedGATKException(String.format("wrong number of unmapped reads: requested %d but saw %d", this.originalStream.getNumUnmappedReads(), this.numUnmappedReads));
        }
        if ((this.originalStream.getNumContigs() == 0 || this.originalStream.getNumStacksPerContig() == 0) && this.numUnmappedReads != this.totalReads) {
            throw new ReviewedGATKException("stream should have consisted only of unmapped reads, but saw some mapped reads");
        }
    }

    public void update(SAMRecord read) {
        if (read.getReadUnmappedFlag()) {
            ++this.numUnmappedReads;
            if (this.numUnmappedReads == 1 && this.lastRead != null) {
                this.processContigChange();
                --this.numContigs;
            }
        } else if (this.lastRead == null) {
            this.numContigs = 1;
            this.currentContigNumStacks = 1;
            this.currentStackNumReads = 1;
        } else if (!read.getReferenceIndex().equals(this.lastRead.getReferenceIndex())) {
            this.processContigChange();
        } else if (read.getAlignmentStart() != this.lastRead.getAlignmentStart()) {
            this.processStackChangeWithinContig(read);
        } else {
            ++this.currentStackNumReads;
        }
        this.updateReadLength(read.getReadLength());
        this.allSamplesMatch = this.allSamplesMatch && this.readHasCorrectSample(read);
        ++this.totalReads;
        this.lastRead = read;
    }

    private void processContigChange() {
        ++this.numContigs;
        this.stacksPerContig.add(this.currentContigNumStacks);
        this.currentContigNumStacks = 1;
        this.updateReadsPerStack(this.currentStackNumReads);
        this.currentStackNumReads = 1;
    }

    private void processStackChangeWithinContig(SAMRecord read) {
        ++this.currentContigNumStacks;
        this.updateReadsPerStack(this.currentStackNumReads);
        this.currentStackNumReads = 1;
        this.updateDistanceBetweenStacks(read.getAlignmentStart() - this.lastRead.getAlignmentStart());
    }

    private void updateReadsPerStack(int stackReadCount) {
        if (this.minReadsPerStack == null || stackReadCount < this.minReadsPerStack) {
            this.minReadsPerStack = stackReadCount;
        }
        if (this.maxReadsPerStack == null || stackReadCount > this.maxReadsPerStack) {
            this.maxReadsPerStack = stackReadCount;
        }
    }

    private void updateDistanceBetweenStacks(int stackDistance) {
        if (this.minDistanceBetweenStacks == null || stackDistance < this.minDistanceBetweenStacks) {
            this.minDistanceBetweenStacks = stackDistance;
        }
        if (this.maxDistanceBetweenStacks == null || stackDistance > this.maxDistanceBetweenStacks) {
            this.maxDistanceBetweenStacks = stackDistance;
        }
    }

    private void updateReadLength(int readLength) {
        if (this.minReadLength == null || readLength < this.minReadLength) {
            this.minReadLength = readLength;
        }
        if (this.maxReadLength == null || readLength > this.maxReadLength) {
            this.maxReadLength = readLength;
        }
    }

    private boolean readHasCorrectSample(SAMRecord read) {
        return this.originalStream.getReadGroupID().equals(read.getAttribute("RG"));
    }

    public void finalizeStats() {
        if (this.lastRead != null && !this.lastRead.getReadUnmappedFlag()) {
            this.stacksPerContig.add(this.currentContigNumStacks);
            this.updateReadsPerStack(this.currentStackNumReads);
        }
    }
}

