/* GStreamer
 * Copyright (C) <2006> Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 * Copyright (C) <2004> David A. Schleef <ds@schleef.org>
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gstrfbsrc.h"

#include <gst/video/video.h>

#include <string.h>
#include <stdlib.h>

enum
{
  ARG_0,
  ARG_HOST,
  ARG_PORT,
};

#define RGB332_R(x)  ((((x)&0x07) * 0x124)>>3)
#define RGB332_G(x)  ((((x)&0x38) * 0x124)>>6)
#define RGB332_B(x)  ((((x)&0xc0) * 0x149)>>8)

GST_DEBUG_CATEGORY_STATIC (rfbsrc_debug);
#define GST_CAT_DEFAULT rfbsrc_debug

static const GstElementDetails gst_rfb_src_details =
GST_ELEMENT_DETAILS ("Rfb source",
    "Source/Video",
    "Creates a rfb video stream",
    "David A. Schleef <ds@schleef.org>, "
    "Andre Moreira Magalhaes <andre.magalhaes@indt.org.br");

static GstStaticPadTemplate gst_rfb_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-rgb, "
        "bpp = (int) 32, "
        "depth = (int) 24, "
        "endianness = (int) 4321, "
        "red_mask = (int) 0x0000ff00, "
        "green_mask = (int) 0x00ff0000, "
        "blue_mask = (int) 0xff000000, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], " "framerate = (fraction) 30/1")
    );

static void gst_rfb_src_base_init (gpointer g_class);
static void gst_rfb_src_class_init (GstRfbSrcClass * klass);
static void gst_rfb_src_dispose (GObject * object);
static void gst_rfb_src_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_rfb_src_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static gboolean gst_rfb_src_start (GstBaseSrc * bsrc);
static gboolean gst_rfb_src_stop (GstBaseSrc * bsrc);
static gboolean gst_rfb_src_event (GstBaseSrc * bsrc, GstEvent * event);
static GstFlowReturn gst_rfb_src_create (GstPushSrc * psrc,
    GstBuffer ** outbuf);
static void gst_rfb_src_paint_rect (RfbDecoder * decoder, gint x, gint y,
    gint w, gint h, guint8 * data);

GST_BOILERPLATE (GstRfbSrc, gst_rfb_src, GstPushSrc, GST_TYPE_PUSH_SRC);

static void
gst_rfb_src_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rfb_src_template));

  gst_element_class_set_details (element_class, &gst_rfb_src_details);
}

static void
gst_rfb_src_class_init (GstRfbSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->dispose = gst_rfb_src_dispose;
  gobject_class->set_property = gst_rfb_src_set_property;
  gobject_class->get_property = gst_rfb_src_get_property;

  g_object_class_install_property (gobject_class, ARG_HOST,
      g_param_spec_string ("host", "Host to connect to", "Host to connect to",
          "127.0.0.1", G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, ARG_PORT,
      g_param_spec_int ("port", "Port", "Port",
          1, 65535, 5900, G_PARAM_READWRITE));

  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_rfb_src_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_rfb_src_stop);
  gstbasesrc_class->event = GST_DEBUG_FUNCPTR (gst_rfb_src_event);
  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_rfb_src_create);

  GST_DEBUG_CATEGORY_INIT (rfbsrc_debug, "rfbsrc", 0, "Rfb source");
}

static void
gst_rfb_src_init (GstRfbSrc * src, GstRfbSrcClass * klass)
{
  GstBaseSrc *bsrc = GST_BASE_SRC (src);

  gst_pad_use_fixed_caps (GST_BASE_SRC_PAD (bsrc));
  gst_base_src_set_live (bsrc, TRUE);
  gst_base_src_set_format (bsrc, GST_FORMAT_TIME);

  src->host = g_strdup ("127.0.0.1");
  src->port = 5900;
}

static void
gst_rfb_src_dispose (GObject * object)
{
  GstRfbSrc *src = GST_RFB_SRC (object);

  g_free (src->host);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gst_rfb_src_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstRfbSrc *src = GST_RFB_SRC (object);

  switch (prop_id) {
    case ARG_HOST:
      src->host = g_strdup (g_value_get_string (value));
      break;
    case ARG_PORT:
      src->port = g_value_get_int (value);
      break;
    default:
      break;
  }
}

static void
gst_rfb_src_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstRfbSrc *src = GST_RFB_SRC (object);

  switch (prop_id) {
    case ARG_HOST:
      g_value_set_string (value, src->host);
      break;
    case ARG_PORT:
      g_value_set_int (value, src->port);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
gst_rfb_src_start (GstBaseSrc * bsrc)
{
  GstRfbSrc *src = GST_RFB_SRC (bsrc);
  RfbDecoder *decoder;
  GstCaps *caps;

  decoder = rfb_decoder_new ();

  GST_DEBUG_OBJECT (src, "connecting to host %s on port %d",
      src->host, src->port);
  if (!rfb_decoder_connect_tcp (decoder, src->host, src->port)) {
    GST_ELEMENT_ERROR (src, LIBRARY, INIT, (NULL),
        ("Could not connect to host %s on port %d", src->host, src->port));
    rfb_decoder_free (decoder);
    return FALSE;
  }

  src->decoder = decoder;
  src->inter = FALSE;

  while (!decoder->inited) {
    rfb_decoder_iterate (decoder);
  }

  g_object_set (bsrc, "blocksize",
      src->decoder->width * src->decoder->height * 4, NULL);

  src->frame = g_malloc (bsrc->blocksize);
  decoder->paint_rect = gst_rfb_src_paint_rect;
  decoder->decoder_private = src;

  GST_DEBUG_OBJECT (src, "setting caps width to %d and height to %d",
      decoder->width, decoder->height);

  caps =
      gst_caps_copy (gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD (bsrc)));
  gst_caps_set_simple (caps, "width", G_TYPE_INT, decoder->width, "height",
      G_TYPE_INT, decoder->height, NULL);
  gst_pad_set_caps (GST_BASE_SRC_PAD (bsrc), caps);
  gst_caps_unref (caps);

  return TRUE;
}

static gboolean
gst_rfb_src_stop (GstBaseSrc * bsrc)
{
  GstRfbSrc *src = GST_RFB_SRC (bsrc);

  rfb_decoder_free (src->decoder);

  if (src->frame) {
    g_free (src->frame);
    src->frame = NULL;
  }

  return TRUE;
}

static GstFlowReturn
gst_rfb_src_create (GstPushSrc * psrc, GstBuffer ** outbuf)
{
  GstRfbSrc *src = GST_RFB_SRC (psrc);
  RfbDecoder *decoder = src->decoder;
  gulong newsize;
  GstFlowReturn ret;

  rfb_decoder_send_update_request (decoder, src->inter, 0, 0,
      decoder->width, decoder->height);
  // src->inter = TRUE;

  src->go = TRUE;
  while (src->go) {
    rfb_decoder_iterate (decoder);
  }

  newsize = GST_BASE_SRC (psrc)->blocksize;

  /* Create the buffer. */
  ret = gst_pad_alloc_buffer (GST_BASE_SRC_PAD (GST_BASE_SRC (psrc)),
      GST_BUFFER_OFFSET_NONE, newsize,
      GST_PAD_CAPS (GST_BASE_SRC_PAD (GST_BASE_SRC (psrc))), outbuf);

  if (G_UNLIKELY (ret != GST_FLOW_OK)) {
    return GST_FLOW_ERROR;
  }

  memcpy (GST_BUFFER_DATA (*outbuf), src->frame, newsize);
  GST_BUFFER_SIZE (*outbuf) = newsize;

  return GST_FLOW_OK;
}

static gboolean
gst_rfb_src_event (GstBaseSrc * bsrc, GstEvent * event)
{
  GstRfbSrc *src = GST_RFB_SRC (bsrc);
  gdouble x, y;
  gint button;
  GstStructure *structure;
  const gchar *event_type;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_NAVIGATION:
      structure = event->structure;
      event_type = gst_structure_get_string (structure, "event");
      gst_structure_get_double (structure, "pointer_x", &x);
      gst_structure_get_double (structure, "pointer_y", &y);
      button = 0;

      if (strcmp (event_type, "key-press") == 0) {
        const gchar *key = gst_structure_get_string (structure, "key");

        GST_LOG_OBJECT (src, "sending key event for key %d", key[0]);
        rfb_decoder_send_key_event (src->decoder, key[0], 1);
        rfb_decoder_send_key_event (src->decoder, key[0], 0);
      } else if (strcmp (event_type, "mouse-move") == 0) {
        GST_LOG_OBJECT (src, "sending mouse-move event "
            "button_mask=%d, x=%d, y=%d", src->button_mask, (gint) x, (gint) y);
        rfb_decoder_send_pointer_event (src->decoder, src->button_mask,
            (gint) x, (gint) y);
      } else if (strcmp (event_type, "mouse-button-release") == 0) {
        src->button_mask &= ~(1 << button);
        GST_LOG_OBJECT (src, "sending mouse-button-release event "
            "button_mask=%d, x=%d, y=%d", src->button_mask, (gint) x, (gint) y);
        rfb_decoder_send_pointer_event (src->decoder, src->button_mask,
            (gint) x, (gint) y);
      } else if (strcmp (event_type, "mouse-button-press") == 0) {
        src->button_mask |= (1 << button);
        GST_LOG_OBJECT (src, "sending mouse-button-press event "
            "button_mask=%d, x=%d, y=%d", src->button_mask, (gint) x, (gint) y);
        rfb_decoder_send_pointer_event (src->decoder, src->button_mask,
            (gint) x, (gint) y);
      }
      break;
    default:
      break;
  }

  return TRUE;
}

static void
gst_rfb_src_paint_rect (RfbDecoder * decoder, gint x, gint y, gint w, gint h,
    guint8 * data)
{
  gint i, j;
  guint8 *frame;
  guint8 color;
  GstRfbSrc *src;
  gint width;
  gint offset;

  // GST_DEBUG ("painting %d,%d (%dx%d)\n", x, y, w, h);
  src = GST_RFB_SRC (decoder->decoder_private);

  frame = src->frame;
  width = decoder->width;
  for (j = 0; j < h; j++) {
    for (i = 0; i < w; i++) {
      color = data[j * w + i];

      offset = ((j + x) * width + (i + x)) * 4;
      frame[offset + 0] = RGB332_B (color);
      frame[offset + 1] = RGB332_G (color);
      frame[offset + 2] = RGB332_R (color);
      frame[offset + 3] = 0;
    }
  }

  src->go = FALSE;
}

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "rfbsrc", GST_RANK_NONE,
      GST_TYPE_RFB_SRC);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "rfbsrc",
    "Connects to a VNC server and decodes RFB stream",
    plugin_init, VERSION, GST_LICENSE, GST_PACKAGE_NAME, GST_PACKAGE_ORIGIN)
