/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Network Interfaces
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef M4_NETWORK_H
#define M4_NETWORK_H

#ifdef __cplusplus
extern "C" {
#endif

#include <gpac/m4_descriptors.h>

/*handle to service*/
typedef struct _net_service *LPNETSERVICE;

/*handle to channel*/
typedef void *LPNETCHANNEL;

enum
{
	/*down commands: app->plugin*/
	/*media control*/
	CHAN_PLAY,
	CHAN_STOP,
	CHAN_SET_SPEED,
	CHAN_PAUSE,
	CHAN_RESUME,
	CHAN_PREROLL,
	/*channel configuration. the channel is always setup when called - plugin shall not send acknowledgement*/
	CHAN_CONFIG,

	/*down/up commands: the plugin shall update the associated config values and shall not send acknowledgement*/
	CHAN_DURATION,
	CHAN_BUFFER,
	/*up commands: query buffer fullness (min and max values of buffers are also updated in command)*/
	CHAN_BUFFER_QUERY,
	
	/*retrieves DSI from channel - this may be needed in RTSP/SDP when the config is carried in the SDP and omitted in the ESD*/
	CHAN_GET_DSI,
	/*sets input channel to pull mode - plugin shall not send acknowledgement*/
	CHAN_SET_PULL,
	/*set media padding for all AUs fetched (pull mode only). If not supported the channel will have to run in push mode*/
	CHAN_SET_PADDING,
	
	/*query channel capability to pause/resume and seek(play from an arbitrary range)
	a non-interactive channel doesn't have to handle SET_SPEED, PAUSE, RESUME and PREROLL commands but can still
	work in pull mode*/
	CHAN_INTERACTIVE,

	/*map net time (OTB) to media time (up only) - this is needed by some signaling protocols when the 
	real play range is not the requested one */
	CHAN_MAP_TIME,
	/*reconfiguration of channel comming from network (up only) - this is used to override the SL config
	if it differs from the one specified at config*/
	CHAN_RECONFIG,

	/*no service commands define yet*/
	NM_COM_UNKNOWN
};

typedef struct
{
	u32 command_type;

	/*channel the command applies to if channel command*/
	LPNETCHANNEL on_channel;

	/*1 - MEDIA CHANNEL COMMANDS*/

	/*params for CHAN_PLAY, ranges in sec - if range is <0, then it is ignored (eg [2, -1] with speed>0 means 2 +oo) */
	Double start_range, end_range;
	/*params for CHAN_PLAY and CHAN_SPEED*/
	Double speed;
	/*params for preroll, indicates media length of prerolling in sec*/
	Double length;
	
	/*params valid for a CHAN_CONFIG - channel config may happen as soon as the channel is open, even if the plugin 
	hasn't acknowledge creation*/
		/*SL config of the stream as delivered in OD - this allows for emulation of SL - app->channel only
	this member is also used when reconfiguring a channel (CHAN_RECONFIG), channel->app only*/
	SLConfigDescriptor sl_config;		
		/*stream priority packet drops are more tolerable if low priority - app->channel only*/
	u32 priority;
		/*sync ID: all channels with the same sync ID run on the same timeline, thus the plugin should try to match this
	app->channel only*/
	u32 sync_id;
	/*audio frame duration and sample rate if any - this is needed by some RTP payload*/
	u32 frame_duration, sample_rate;

	/*params for CHAN_BUFFER - the recommended buffering limits in ms - this depends on the plugins preferences and 
	on the service type (multicast, vod, ...) - below buffer_min the stream will pause if possible until buffer_max is reached*/
	u32 buffer_min, buffer_max;
	/*params for CHAN_BUFFER_QUERY - amount of media in decoding buffer, in ms*/
	u32 buffer_occupancy;
	/*params for CHAN_DURATION - duration in sec*/
	Double duration;

	/*CHAN_GET_DSI - carries specific info for codec - data shall be allocated by service and is freed by user*/
	char *dsi;
	u32 dsi_len;

	/*CHAN_SET_PULL: no parameters: the plugin returns M4OK if pull is supported or M4NotSupported (or anything else) 
	if only push is supported*/
	/*CHAN_SET_PADDING param: amount of bytes to add when fetching samples (extra alloc)*/
	u32 padding_bytes;
	/*CHAN_INTERACTIVE: no parameters, the plugin returns M4OK if supported or M4NotSupported (or anything else) if 
	no control caps*/
	/*CHAN_MAP_TIME - timestamp conveys the TS where mapping is done (in SL TS resolution). MediaTime for this mapping shall be 
	carried in start_range*/
	u32 map_timestamp;

	/*confirmation state (only used when the command is send back to the user)*/
	M4Err response;

} NetworkCommand;

enum
{
	/*service/channel is setup and waits for connection request (whenever a service is closed
	it shall fall back to NM_Setup status)*/
	NM_Setup = 0,
	/*waiting for server reply*/
	NM_WaitingForAck,
	/*connection OK*/
	NM_Connected,
	/*data exchange on this service/channel*/
	NM_Running,
	/*deconnection OK - a download channel can automatically disconnect when download is done*/
	NM_Disconnected,
	/*service/channel is not (no longer) available/found and should be removed*/
	NM_Unavailable
};


/*MPEG-4 OD types. This type provides a hint to network plugins which may have to generate an OD/IOD on the fly
they occur only if objects/services used in the scene are not referenced through OD IDs but direct URL*/
enum
{
	/*OD expected is of undefined type*/
	NM_OD_UNDEF = 0,
	/*OD expected is of SCENE type (eg, BIFS and OD if needed)*/
	NM_OD_SCENE,
	/*OD expected is of BIFS type (anim streams)*/
	NM_OD_BIFS,
	/*OD expected is of VISUAL type*/
	NM_OD_VIDEO,
	/*OD expected is of AUDIO type*/
	NM_OD_AUDIO,
	/*OD expected is of InputSensor type*/
	NM_OD_INTERACT
};

/*
	network plugins
*/

/*MPEG-4 streaming client interface name*/
#define M4STREAMINGCLIENT			FOUR_CHAR_INT('M', 'S', 'T', 'R')

typedef struct _netinterface
{
	/* interface declaration*/
	M4_DECL_PLUGIN_INTERFACE

	/*retuns 1 if plugin can process this URL, 0 otherwise*/
	Bool (*CanHandleURL)(struct _netinterface *, const char *url);

	/*connects the service to the desired URL - the service handle is used for callbacks. 
	Only one service can be connected to a loaded interface.
	*/
	M4Err (*ConnectService) (struct _netinterface *, LPNETSERVICE serv, const char *url);

	/*disconnects service - the plugin is no longer used after this call - if immediate_shutdown is set the plugin
	shall not attempt to get confirmation from remote side, it will be deleted right away
	
	NOTE: depending on how the client/server exchange is happening, it may happen that the CloseService is called
	in the same context as a reply from your plugin. This can result into deadlocks if you're using threads. 
	You should therefore only try to destroy threads used in the interface shutdown process, which is guarantee
	to be in a different context call.
	*/
	M4Err (*CloseService) (struct _netinterface *, Bool immediate_shutdown);

	/*retrieves MPEG4 iod for accessing this service through an IOD - data is allocated and returned to the caller if OK
	the IOD shall refer to the service attached to the plugin
	@expect_type is a hint in case the service regenerates an IOD. It indicates whether the entry point expected is INLINE, 
	BIFS animation stream, video, audio or input sensor. cf above for types
	Note: once a service is acknowledged as connected, the IOD shall be available synchronously
	*/
	M4Err (*Get_MPEG4_IOD) (struct _netinterface *, u32 expect_type, char **raw_iod, u32 *raw_iod_size);


	/*sends command to the service / channel - cf command structure*/
	M4Err (*ServiceCommand) (struct _netinterface *, NetworkCommand *com);

	/*data channel setup - url is either
	"ES_ID=ID" where ID is the stream ID in this service
	or a control string depending on the service/stream. The URL is first used to load a plugin able to handle it, 
	so the plugin has no redirection to handle
	*/
	M4Err (*ConnectChannel) (struct _netinterface *, LPNETCHANNEL channel, const char *url, Bool upstream);
	/*teardown of data channel*/
	M4Err (*DisconnectChannel) (struct _netinterface *, LPNETCHANNEL channel);

	/*gets service status (if channel is NULL) or channel status*/
	M4Err (*GetStatus) (struct _netinterface *, LPNETCHANNEL channel, u32 *status);

	/*optional - fetch MPEG4 data from channel - data shall not be duplicated and must be released at ReleaseData
	SL info shall be written to provided header - if the data is a real SL packet the flag sl_compressed shall be 
	set to signal the app this is a full SL pdu (@out_sl_hdr is then ignored)
	set to NULL if not supported
	*/
	M4Err (*ChannelGetSLP) (struct _netinterface *, LPNETCHANNEL channel, char **out_data_ptr, u32 *out_data_size, struct tagSLHeader *out_sl_hdr, Bool *sl_compressed, M4Err *out_reception_status, Bool *is_new_data);

	/*optional - release SLP data allocated on channel by the previous call, if any
	set to NULL if not supported*/
	M4Err (*ChannelReleaseSLP) (struct _netinterface *, LPNETCHANNEL channel);

	/*this is needed for plugins to query other plugins, the typical case being 2 ESD URLs pointing to the 
	same media (audio and video streams in an RTSP session). This is always used on loaded plugins but 
	doesn't have to be declared*/
	Bool (*CanHandleURLInService)(struct _netinterface *, const char *url);

/*private*/
	void *priv;
} NetClientPlugin;

/*callback functions - these can be linked with non-GPL plugins*/
/*message from service - error is set if error*/
void NM_OnMessage(LPNETSERVICE service, M4Err error, const char *message);
/*to call on service (if channel is NULL) or channel connect completed*/
void NM_OnConnect(LPNETSERVICE service, LPNETCHANNEL ns, M4Err response);
/*to call on service (if channel is NULL) or channel disconnect completed*/
void NM_OnDisconnect(LPNETSERVICE service, LPNETCHANNEL ns, M4Err response);
/* acknowledgement of service command - service commands handle both services and channels
Most of the time commands are NOT acknowledged, typical acknowledgement are needed for setup and control
with remote servers. 
command can also be triggered from the service (QoS, broadcast announcements)
cf above for command usage
*/
void NM_OnCommand(LPNETSERVICE service, NetworkCommand *com);
/*to call when data packet is recieved. 
@data, data_size: data recieved
@hdr: uncompressed SL header passed with data for stream sync - if not present then data shall be a valid SL packet 
	(header + PDU). Note that using an SLConfig resulting in an empty SLHeader allows sending raw data directly
@reception_status: data reception status. To signal end of stream, set this to M4EOF
*/
void NM_OnSLPRecieved(LPNETSERVICE service, LPNETCHANNEL ns, char *data, u32 data_size, struct tagSLHeader *hdr, M4Err reception_status);
/*returns URL associated with service (so that you don't need to store it)*/
const char *NM_GetServiceURL(LPNETSERVICE service);

/*file downloader interface name*/
#define M4NETDOWNLOADER			FOUR_CHAR_INT('M', 'D', 'N', 'L')

typedef struct _netdownloader
{
	/* interface declaration*/
	M4_DECL_PLUGIN_INTERFACE

	/*returns 1 if URL can be handled*/
	Bool (*CanHandleURL)(const char *url);
	/*connect to given URL and start fetching data. If disable_cache, data is NEVER cached*/
	M4Err (*Connect) (struct _netdownloader *, const char *url, Bool disable_cache);
	/*closes service*/
	M4Err (*Close) (struct _netdownloader *);
	/*returns local cache file name*/
	const char *(*GetCacheFileName) (struct _netdownloader *);

	/*stat info updated by plugin*/
	u32 total_size;
	u32 bytes_done;
	Float bytes_per_sec;
	/*current net status*/
	u32 net_status;
	/*error report - M4EOF indicates done*/
	M4Err error;

	/*callback assigned by loader*/
	void *user_cbck;

	/*callback function for data delivery - assigned by loader. Whenever a plugin needs to communicate 
	a state change, it shall call this with data_size = 0 and update its error/net_status*/
	void (*OnData)(struct _netdownloader *, char *data, u32 data_size);

	/*plugin private*/
	void *priv;
} NetDownloader;


#ifdef __cplusplus
}
#endif

#endif
