/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / RTP input plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "rtp_in.h"
#include <gpac/m4_decoder.h>


u32 payt_get_type(RTPClient *rtp, SDP_RTPMap *map, SDPMedia *media)
{
	u32 i, j;

	if (!stricmp(map->payload_name, "MP4V-ES") ) return RTP_HINT_MPEG4;
	if (!stricmp(map->payload_name, "mpeg4-generic")) return RTP_HINT_MPEG4;

	/*LATM: only without multiplexing (not tested but should be straight AUs)*/
	if (!stricmp(map->payload_name, "MP4A-LATM")) {
		for (i=0; i<ChainGetCount(media->FMTP); i++) {
			SDP_FMTP *fmtp = ChainGetEntry(media->FMTP, i);
			if (fmtp->PayloadType != map->PayloadType) continue;
			//this is our payload. check cpresent is 0
			for (j=0; j<ChainGetCount(fmtp->Attributes); j++) {
				X_Attribute *att = ChainGetEntry(fmtp->Attributes, j);
				if (!stricmp(att->Name, "cpresent") && atoi(att->Value)) return 0;
			}
		}
		return RTP_HINT_MPEG4;
	}
	if (!stricmp(map->payload_name, "MPA")) return RTP_HINT_MPEG12;
	if (!stricmp(map->payload_name, "AMR")) return RTP_HINT_AMR;
	if (!stricmp(map->payload_name, "H263-1998")) return RTP_HINT_H263;
	if (!stricmp(map->payload_name, "H263-2000")) return RTP_HINT_H263;
	return 0;
}


static M4Err payt_set_param(RTPStream *ch, char *param_name, char *param_val)
{
	u32 i, val;
	char valS[3];
	BitStream *bs;

	if (!ch || !param_name) return M4BadParam;

	/*1 - mpeg4-generic / RFC 3016 payload type items*/

	/*PL (not needed when IOD is here)*/
	if (!stricmp(param_name, "Profile-level-id")) ch->sl_map.PL_ID = atoi(param_val);

	/*decoder specific info (not needed when IOD is here)*/
	else if (!stricmp(param_name, "config")) {
		//decode the buffer - the string buffer is MSB hexadecimal
		bs = NewBitStream(NULL, 0, BS_WRITE);
		valS[2] = 0;
		for (i=0; i<strlen(param_val);i+=2) {
			valS[0] = param_val[i];
			valS[1] = param_val[i+1];
			sscanf(valS, "%x", &val);
			BS_WriteInt(bs, val, 8);
		}
		if (ch->sl_map.config) free(ch->sl_map.config);
		ch->sl_map.config = NULL;
		BS_GetContent(bs, (unsigned char **) &ch->sl_map.config, &ch->sl_map.configSize);
		DeleteBitStream(bs);
	}
	/*mpeg4-generic payload type items required*/
	
	/*constant size (size of all AUs) */
	else if (!stricmp(param_name, "ConstantSize")) {
		ch->sl_map.ConstantSize = atoi(param_val);
	}
	/*constant size (size of all AUs) */
	else if (!stricmp(param_name, "ConstantDuration")) {
		ch->sl_map.ConstantDuration = atoi(param_val);
	}
	/*object type indication (not needed when IOD is here)*/
	else if (!stricmp(param_name, "ObjectType")) {
		ch->sl_map.ObjectTypeIndication = atoi(param_val);
	}
	else if (!stricmp(param_name, "StreamType")) ch->sl_map.StreamType = atoi(param_val);
	else if (!stricmp(param_name, "mode")) {
		strcpy(ch->sl_map.mode, param_val);
		/*in case no IOD and no streamType/OTI in the file*/
		if (!stricmp(param_val, "AAC-hbr") || !stricmp(param_val, "AAC-lbr") || !stricmp(param_val, "CELP-vbr") || !stricmp(param_val, "CELP-cbr")) {
			ch->sl_map.StreamType = M4ST_AUDIO;
			ch->sl_map.ObjectTypeIndication = 0x40;
		}

	}

	else if (!stricmp(param_name, "DTSDeltaLength")) ch->sl_map.DTSDeltaLength = atoi(param_val);
	else if (!stricmp(param_name, "CTSDeltaLength")) ch->sl_map.CTSDeltaLength = atoi(param_val);
	else if (!stricmp(param_name, "SizeLength")) ch->sl_map.SizeLength = atoi(param_val);
	else if (!stricmp(param_name, "IndexLength")) ch->sl_map.IndexLength = atoi(param_val);
	else if (!stricmp(param_name, "IndexDeltaLength")) ch->sl_map.IndexDeltaLength = atoi(param_val);
	else if (!stricmp(param_name, "RandomAccessIndication")) ch->sl_map.RandomAccessIndication = atoi(param_val);
	else if (!stricmp(param_name, "StreamStateIndication")) ch->sl_map.StreamStateIndication = atoi(param_val);
	else if (!stricmp(param_name, "AuxiliaryDataSizeLength")) ch->sl_map.AuxiliaryDataSizeLength = atoi(param_val);

	/*AMR config*/
	else if (!stricmp(param_name, "octet-align")) ch->octet_align = 1;
	return M4OK;
}

void payt_setup(RTPStream *ch, SDP_RTPMap *map, SDPMedia *media)
{
	u32 i, j;

	/*reset sl map*/
	memset(&ch->sl_map, 0, sizeof(RTPSLMap));
	
	/*setup channel*/
	RTP_SetupPayload(ch->rtp_ch, map);

	/*then process all FMTPs*/
	for (i=0; i<ChainGetCount(media->FMTP); i++) {
		SDP_FMTP *fmtp = ChainGetEntry(media->FMTP, i);
		//we work with only one PayloadType for now
		if (fmtp->PayloadType != map->PayloadType) continue;
		for (j=0; j<ChainGetCount(fmtp->Attributes); j++) {
			X_Attribute *att = ChainGetEntry(fmtp->Attributes, j);
			payt_set_param(ch, att->Name, att->Value);
		}
	}

	switch (ch->rtptype) {
	case RTP_HINT_MPEG4:
		/*mark if AU header is present*/
		ch->sl_map.auh_first_min_len = ch->sl_map.CTSDeltaLength;
		ch->sl_map.auh_first_min_len += ch->sl_map.DTSDeltaLength;
		ch->sl_map.auh_first_min_len += ch->sl_map.SizeLength;
		ch->sl_map.auh_first_min_len += ch->sl_map.RandomAccessIndication;
		ch->sl_map.auh_first_min_len += ch->sl_map.StreamStateIndication;
		ch->sl_map.auh_min_len = ch->sl_map.auh_first_min_len;
		ch->sl_map.auh_first_min_len += ch->sl_map.IndexLength;
		ch->sl_map.auh_min_len += ch->sl_map.IndexDeltaLength;
		if (!stricmp(map->payload_name, "MP4V-ES")) {
			ch->sl_map.StreamType = M4ST_VISUAL;
			ch->sl_map.ObjectTypeIndication = 0x20;
		}
		else if (!stricmp(map->payload_name, "MP4A-LATM")) {
			ch->sl_map.StreamType = M4ST_AUDIO;
			ch->sl_map.ObjectTypeIndication = 0x40;
		}
		break;
	case RTP_HINT_MPEG12:
		if (!stricmp(map->payload_name, "MPA")) {
			ch->sl_map.StreamType = M4ST_AUDIO;
			ch->sl_map.ObjectTypeIndication = 0x69;
		}
		else if (!stricmp(map->payload_name, "MPV")) {
			ch->sl_map.StreamType = M4ST_VISUAL;
			/*FIXME: *differentiate MPEG1 from MPEG2 video*/
			ch->sl_map.ObjectTypeIndication = 0x6A;
		}
		break;
	case RTP_HINT_AMR:
		{
			BitStream *bs;
			ch->sl_map.StreamType = M4ST_AUDIO;
			ch->sl_map.ObjectTypeIndication = GPAC_QT_CODECS_OTI;
			/*create DSI*/
			bs = NewBitStream(NULL, 0, BS_WRITE);
			BS_WriteInt(bs, FOUR_CHAR_INT('a', 'm', 'r', ' '), 32);
			BS_WriteInt(bs, 0, 4*32);
			BS_GetContent(bs, (unsigned char **) &ch->sl_map.config, &ch->sl_map.configSize);
			DeleteBitStream(bs);
		}
		break;
	case RTP_HINT_H263:
		{
			u32 x, y, w, h;
			BitStream *bs;
			x = y = w = h = 0;
			for (j=0; j<ChainGetCount(media->Attributes); j++) {
				X_Attribute *att = ChainGetEntry(media->Attributes, j);
				if (stricmp(att->Name, "cliprect")) continue;
				/*only get the display area*/
				sscanf(att->Value, "%d,%d,%d,%d", &y, &x, &h, &w);
			}

			ch->sl_map.StreamType = M4ST_VISUAL;
			ch->sl_map.ObjectTypeIndication = GPAC_QT_CODECS_OTI;
			/*create DSI*/
			bs = NewBitStream(NULL, 0, BS_WRITE);
			BS_WriteInt(bs, FOUR_CHAR_INT('h', '2', '6', '3'), 32);
			BS_WriteInt(bs, w, 32);
			BS_WriteInt(bs, h, 32);
			BS_GetContent(bs, (unsigned char **) &ch->sl_map.config, &ch->sl_map.configSize);
			DeleteBitStream(bs);
		}
		break;
	}

}


void RP_ParsePayloadMPEG4(RTPStream *ch, RTPHeader *hdr, char *payload, u32 size)
{
	u32 aux_size, au_size, first_idx, au_hdr_size, pay_start, num_au;
	s32 au_idx;
	BitStream *hdr_bs, *aux_bs;

	hdr_bs = NewBitStream(payload, size, BS_READ);
	aux_bs = NewBitStream(payload, size, BS_READ);

//	printf("parsing packet %d size %d ts %d M %d\n", hdr->SequenceNumber, size, hdr->TimeStamp, hdr->Marker);

	/*global AU header len*/
	au_hdr_size = 0;
	if (ch->sl_map.auh_first_min_len) {
		au_hdr_size = BS_ReadInt(hdr_bs, 16);
		BS_ReadInt(aux_bs, 16);
	}

	/*jump to aux section, skip it and get payload start*/
	BS_ReadInt(aux_bs, au_hdr_size);
	BS_Align(aux_bs);
	if (ch->sl_map.AuxiliaryDataSizeLength) {
		aux_size = BS_ReadInt(aux_bs, ch->sl_map.AuxiliaryDataSizeLength);
		BS_ReadInt(aux_bs, aux_size);
		BS_Align(aux_bs);
	}
	pay_start = (u32) BS_GetPosition(aux_bs);
	DeleteBitStream(aux_bs);

	first_idx = 0;
	au_idx = 0;

	ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp;
	ch->sl_hdr.decodingTimeStamp = hdr->TimeStamp;

	num_au = 0;

	ch->sl_hdr.accessUnitEndFlag = hdr->Marker;
	/*override some defaults for RFC 3016*/
	if (ch->new_au) {
		ch->sl_hdr.accessUnitStartFlag = 1;
	} else {
		ch->sl_hdr.accessUnitStartFlag = 0;
	}

	while (1) {
		/*get default AU size*/
		au_size = ch->sl_map.ConstantSize;
		/*not signaled, assume max one AU per packet*/
		if (!au_size) au_size = size - pay_start;
		
		if ((!num_au && ch->sl_map.auh_first_min_len) || (num_au && ch->sl_map.auh_min_len)) {
			/*AU size*/
			if (ch->sl_map.SizeLength) {
				au_size = BS_ReadInt(hdr_bs, ch->sl_map.SizeLength);
				if (au_size > size - pay_start) au_size = size - pay_start;
				au_hdr_size -= ch->sl_map.SizeLength;
			}
			/*AU index*/
			if (! num_au) {
				au_idx = first_idx = BS_ReadInt(hdr_bs, ch->sl_map.IndexLength);
				au_hdr_size -= ch->sl_map.IndexLength;
			} else {
				au_idx += 1 + (s32) BS_ReadInt(hdr_bs, ch->sl_map.IndexDeltaLength);
				au_hdr_size -= ch->sl_map.IndexDeltaLength;
			}
			/*CTS flag*/
			if (ch->sl_map.CTSDeltaLength) {
				ch->sl_hdr.compositionTimeStampFlag = BS_ReadInt(hdr_bs, 1);
				au_hdr_size -= 1;
			} else {
				/*get CTS from IDX*/
				if (ch->sl_map.ConstantDuration) {
					ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp + (au_idx - first_idx) * ch->sl_map.ConstantDuration;
				} else {
					ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp + (au_idx - first_idx) * ch->unit_duration;
				}
			}

			/*CTS in-band*/
			if (ch->sl_hdr.compositionTimeStampFlag) {
				ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp + (s32) BS_ReadInt(hdr_bs, ch->sl_map.CTSDeltaLength);
				au_hdr_size -= ch->sl_map.CTSDeltaLength;
			}
			/*DTS flag is always present (needed for reconstruction of TSs in case of packet loss)*/
			if (ch->sl_map.DTSDeltaLength) {
				ch->sl_hdr.decodingTimeStampFlag = BS_ReadInt(hdr_bs, 1);
				au_hdr_size -= 1;
			} else {
				/*NO DTS otherwise*/
				ch->sl_hdr.decodingTimeStampFlag = 0;
			}
			if (ch->sl_hdr.decodingTimeStampFlag) {
				s32 ts = hdr->TimeStamp - (s32) BS_ReadInt(hdr_bs, ch->sl_map.DTSDeltaLength);
				ch->sl_hdr.decodingTimeStamp = (ts>0) ? ts : 0;
				au_hdr_size -= ch->sl_map.DTSDeltaLength;
			}
			/*RAP flag*/
			if (ch->sl_map.RandomAccessIndication) {
				ch->sl_hdr.randomAccessPointFlag = BS_ReadInt(hdr_bs, 1);
				au_hdr_size -= 1;
			}
			/*stream state - map directly to seqNum*/
			if (ch->sl_map.StreamStateIndication) {
				ch->sl_hdr.AU_sequenceNumber = BS_ReadInt(hdr_bs, ch->sl_map.StreamStateIndication);
				au_hdr_size -= ch->sl_map.StreamStateIndication;
			}
		}
		/*no header present, update CTS/DTS - note we're sure there's no interleaving*/
		else {
			if (num_au) {
				ch->sl_hdr.compositionTimeStamp += ch->sl_map.ConstantDuration;
				ch->sl_hdr.decodingTimeStamp += ch->sl_map.ConstantDuration;
			}
		}
		/*we cannot map RTP SN to SL SN since an RTP packet may carry several SL ones - only inc by 1 seq nums*/
		ch->sl_hdr.packetSequenceNumber += 1;

		/*force indication of CTS whenever we have a new AU*/
		
		ch->sl_hdr.compositionTimeStampFlag = ch->new_au;

		if (ch->owner->first_packet_drop && (ch->sl_hdr.packetSequenceNumber >= ch->owner->first_packet_drop) ) {
			if ( (ch->sl_hdr.packetSequenceNumber - ch->owner->first_packet_drop) % ch->owner->frequency_drop)
				NM_OnSLPRecieved(ch->owner->service, ch->channel, payload + pay_start, au_size, &ch->sl_hdr, M4OK);
		} else {
			NM_OnSLPRecieved(ch->owner->service, ch->channel, payload + pay_start, au_size, &ch->sl_hdr, M4OK);
		}

		ch->sl_hdr.compositionTimeStampFlag = 0;

		if (au_hdr_size < ch->sl_map.auh_min_len) break;
		pay_start += au_size;
		if (pay_start >= size) break;
		num_au ++;
	}

	ch->new_au = hdr->Marker ? 1 : 0;

	DeleteBitStream(hdr_bs);
}


void RP_ParsePayloadMPEG12Audio(RTPStream *ch, RTPHeader *hdr, char *payload, u32 size)
{
	u16 offset;
	u32 mp3hdr, ts;
	BitStream *bs;

	ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp;
	ch->sl_hdr.decodingTimeStamp = hdr->TimeStamp;

	ch->sl_hdr.accessUnitStartFlag = ch->sl_hdr.accessUnitEndFlag ? 1 : 0;
	if (ch->new_au) ch->sl_hdr.accessUnitStartFlag = 1;

	/*get frag header*/
	bs = NewBitStream(payload, size, BS_READ);
	BS_ReadInt(bs, 16);
	offset = BS_ReadInt(bs, 16);
	DeleteBitStream(bs);
	payload += 4;
	size -= 4;
	mp3hdr = 0;
	while (1) {

		/*frame start if no offset*/
		ch->sl_hdr.accessUnitStartFlag = offset ? 0 : 1;

		/*new frame, store size*/
		ch->sl_hdr.compositionTimeStampFlag = 0;
		if (ch->sl_hdr.accessUnitStartFlag) {
			mp3hdr = FOUR_CHAR_INT((u8) payload[0], (u8) payload[1], (u8) payload[2], (u8) payload[3]);
			ch->sl_hdr.accessUnitLength = MP3_GetFrameSize(mp3hdr);
			ch->sl_hdr.compositionTimeStampFlag = 1;
		}
		if (!ch->sl_hdr.accessUnitLength) break;
		/*fragmented frame*/
		if (ch->sl_hdr.accessUnitLength>size) {
			NM_OnSLPRecieved(ch->owner->service, ch->channel, payload, ch->sl_hdr.accessUnitLength, &ch->sl_hdr, M4OK);
			ch->sl_hdr.accessUnitLength -= size;
			ch->sl_hdr.accessUnitStartFlag = ch->sl_hdr.accessUnitEndFlag = 0;
			return;
		}
		/*complete frame*/
		ch->sl_hdr.accessUnitEndFlag = 1;
		NM_OnSLPRecieved(ch->owner->service, ch->channel, payload, ch->sl_hdr.accessUnitLength, &ch->sl_hdr, M4OK);
		payload += ch->sl_hdr.accessUnitLength;
		size -= ch->sl_hdr.accessUnitLength;
		ch->sl_hdr.accessUnitLength = 0;
		
		/*if fragmented there shall not be other frames in the packet*/
		if (!ch->sl_hdr.accessUnitStartFlag) return;
		if (!size) break;
		offset = 0;
		/*get ts*/
		ts = MP3_GetSamplesPerFrame(mp3hdr);
		ch->sl_hdr.compositionTimeStamp += ts;
		ch->sl_hdr.decodingTimeStamp += ts;
	}
	ch->new_au = 1;
}

void RP_ParsePayloadMPEG12(RTPStream *ch, RTPHeader *hdr, char *payload, u32 size)
{
	switch (ch->sl_map.StreamType) {
	case M4ST_VISUAL:
		break;
	case M4ST_AUDIO:
		RP_ParsePayloadMPEG12Audio(ch, hdr, payload, size);
		break;
	}
}

static const u32 amr_frame_size_bytes[16] = {
	12, 13, 15, 17, 19, 20, 26, 31, 5, 6, 5, 5, 0, 0, 0, 1};

void RP_ParsePayloadAMR(RTPStream *ch, RTPHeader *hdr, char *payload, u32 size)
{
	unsigned char c, type;
	char *data;
	/*we support max 30 frames in one RTP packet...*/
	u32 nbFrame, i, frame_size;
	/*not supported yet*/
	if (!ch->octet_align) return;

	/*process toc and locate start of payload data*/
	nbFrame = 0;
	while (1) {
		c = payload[nbFrame + 1];
		nbFrame++;
		if (!(c & 0x80)) break;
	}
	data = payload + nbFrame + 1;
	ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp;
	/*then each frame*/
	for (i=0; i<nbFrame; i++) {
		c = payload[i + 1];
		type = ((c & 0x78) >> 3);
		frame_size = amr_frame_size_bytes[type];

		ch->sl_hdr.compositionTimeStampFlag = 1;
		ch->sl_hdr.accessUnitStartFlag = 1;
		ch->sl_hdr.accessUnitEndFlag = 0;
		/*send TOC*/
		NM_OnSLPRecieved(ch->owner->service, ch->channel, &payload[i+1], 1, &ch->sl_hdr, M4OK);
		ch->sl_hdr.packetSequenceNumber ++;
		ch->sl_hdr.compositionTimeStampFlag = 0;
		ch->sl_hdr.accessUnitStartFlag = 0;
		ch->sl_hdr.accessUnitEndFlag = 1;
		/*send payload*/
		NM_OnSLPRecieved(ch->owner->service, ch->channel, data, frame_size, &ch->sl_hdr, M4OK);
		data += frame_size;
		ch->sl_hdr.compositionTimeStamp += 160;
	}
}


void RP_ParsePayloadH263(RTPStream *ch, RTPHeader *hdr, char *payload, u32 size)
{
	BitStream *bs;
	Bool P_bit, V_bit;
	u32 plen, plen_bits, offset;
	char blank[2];

	bs = NewBitStream(payload, size, BS_READ);
	/*reserved*/
	BS_ReadInt(bs, 5);
	P_bit = BS_ReadInt(bs, 1);
	V_bit = BS_ReadInt(bs, 1);
	plen = BS_ReadInt(bs, 6);
	plen_bits = BS_ReadInt(bs, 3);

	/*VRC not supported yet*/
	if (V_bit) {
		BS_ReadInt(bs, 8);
	}
	/*extra picture header not supported yet*/
	if (plen) {
		BS_SkipBytes(bs, plen);
	}
	offset = (u32) BS_GetPosition(bs);
	DeleteBitStream(bs);

	blank[0] = blank[1] = 0;
	/*start*/
	if (P_bit) {
		ch->sl_hdr.compositionTimeStamp = hdr->TimeStamp;
		ch->sl_hdr.compositionTimeStampFlag = 1;
		ch->sl_hdr.accessUnitStartFlag = 1;
		ch->sl_hdr.accessUnitEndFlag = 0;
		/*send missing start code*/
		NM_OnSLPRecieved(ch->owner->service, ch->channel, (char *) blank, 2, &ch->sl_hdr, M4OK);
		/*send payload*/
		ch->sl_hdr.compositionTimeStampFlag = 0;
		ch->sl_hdr.accessUnitStartFlag = 0;
		/*if M bit set, end of frame*/
		ch->sl_hdr.accessUnitEndFlag = hdr->Marker;
		NM_OnSLPRecieved(ch->owner->service, ch->channel, payload + offset, size - offset, &ch->sl_hdr, M4OK);
		return;
	}

	/*middle/end of frames - if M bit set, end of frame*/
	ch->sl_hdr.accessUnitEndFlag = hdr->Marker;
	NM_OnSLPRecieved(ch->owner->service, ch->channel, payload + offset, size - offset, &ch->sl_hdr, M4OK);
}

