/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "visual_surface.h"

#include "gl_inc.h"

#define CHECK_GL_EXT(name) ((strstr(ext, name) != NULL) ? 1 : 0)
void R3D_LoadExtensions(Render3D *sr)
{
	const char *ext = glGetString(GL_EXTENSIONS);
	if (!ext) return;
	memset(&sr->compositor->hw_caps, 0, sizeof(HardwareCaps));

	if (CHECK_GL_EXT("GL_ARB_multisample") || CHECK_GL_EXT("GLX_ARB_multisample") || CHECK_GL_EXT("WGL_ARB_multisample")) 
		sr->compositor->hw_caps.multisample = 1;
	if (CHECK_GL_EXT("GL_ARB_texture_non_power_of_two")) 
		sr->compositor->hw_caps.npot_texture = 1;
	if (CHECK_GL_EXT("GL_EXT_abgr")) 
		sr->compositor->hw_caps.abgr_texture = 1;
	if (CHECK_GL_EXT("GL_EXT_bgra")) 
		sr->compositor->hw_caps.bgra_texture = 1;

	if (CHECK_GL_EXT("GL_EXT_texture_rectangle") || CHECK_GL_EXT("GL_NV_texture_rectangle")) 
		sr->compositor->hw_caps.rect_texture = 1;
}


void VS3D_Setup(VisualSurface *surf)
{
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glShadeModel(GL_SMOOTH);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glClearDepth(1.0f);
    glEnable(GL_CULL_FACE);
    glFrontFace(GL_CCW);
    glCullFace(GL_BACK);
	glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

	if (surf->render->compositor->high_speed) {
		glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
		glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
	} else {
		glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
		glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	}
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

	if (surf->render->compositor->antiAlias == M4_AL_All) {
		glEnable(GL_LINE_SMOOTH);
		glEnable(GL_POINT_SMOOTH);
		if (surf->render->poly_aa)
			glEnable(GL_POLYGON_SMOOTH);
		else
			glDisable(GL_POLYGON_SMOOTH);
	} else {
		glDisable(GL_LINE_SMOOTH);
		glDisable(GL_POINT_SMOOTH);
		glDisable(GL_POLYGON_SMOOTH);
	}

	glDisable(GL_COLOR_MATERIAL);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glClear(GL_DEPTH_BUFFER_BIT);
}

void VS3D_SetHeadlight(VisualSurface *surf, Bool bOn)
{
	Float pos[] = { 0.0, 0.0, 1.0, 0.0 };
	Float col[] = { 1.0, 1.0, 1.0, 1.0 };

	glDisable(GL_LIGHTING);
	if (!bOn) {
		glDisable(GL_LIGHT0);
	} else {
		glEnable(GL_LIGHT0);
		glLightfv(GL_LIGHT0, GL_POSITION, pos);
		glLightfv(GL_LIGHT0, GL_DIFFUSE, col);
		glLightfv(GL_LIGHT0, GL_SPECULAR, col);
		glEnable(GL_LIGHTING);
	}
}

void VS3D_SetViewport(VisualSurface *surf, M4Rect vp)
{
	glViewport((s32) vp.x, (s32) vp.y, (s32) vp.width, (s32) vp.height);
}


void VS3D_DrawMeshIntern(VisualSurface *surf, M4Mesh *mesh)
{
	u32 prim_type;
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, sizeof(M4Vertex),  &mesh->vertices[0].pos);

	if (mesh->flags & MESH_HAS_COLOR) {
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(3, GL_FLOAT, sizeof(M4Vertex), &mesh->vertices[0].color);
	} else {
		glDisableClientState(GL_COLOR_ARRAY);
	}

	if (!mesh->mesh_type && !(mesh->flags & MESH_NO_TEXTURE)) {
		glEnableClientState(GL_TEXTURE_COORD_ARRAY );
		glTexCoordPointer(2, GL_FLOAT, sizeof(M4Vertex), &mesh->vertices[0].texcoords);
	}
	
	if (mesh->mesh_type || (mesh->flags & MESH_IS_2D)) {
		glDisableClientState(GL_NORMAL_ARRAY);
		glNormal3f(0, 0, 1.0f);
	} else {
		glEnableClientState(GL_NORMAL_ARRAY );
		glNormalPointer(GL_FLOAT, sizeof(M4Vertex), &mesh->vertices[0].normal);
		/*if texture is transparent DON'T CULL*/
		if (!surf->tx_transparent && (mesh->flags & MESH_IS_SOLID)) {
			glEnable(GL_CULL_FACE);
			glFrontFace((mesh->flags & MESH_IS_CW) ? GL_CW : GL_CCW);
		} else {
			glDisable(GL_CULL_FACE);
		}
	}


	switch (mesh->mesh_type) {
	case MESH_LINESET:
		prim_type = GL_LINES;
		break;
	case MESH_POINTSET:
		prim_type = GL_POINTS;
		break;
	default:
		prim_type = GL_TRIANGLES;
		break;
	}
	glDrawElements(prim_type, mesh->i_count, GL_UNSIGNED_INT, mesh->indices);

	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY );
	glDisableClientState(GL_NORMAL_ARRAY );

	surf->tx_transparent = 0;
}

void VS3D_DrawMeshBoundingVolume(VisualSurface *surf, M4Mesh *mesh)
{
	SFVec3f c, s;
	s = vec_diff(&mesh->bounds.max_edge, &mesh->bounds.min_edge);
	c.x = mesh->bounds.min_edge.x + s.x/2;
	c.y = mesh->bounds.min_edge.y + s.y/2;
	c.z = mesh->bounds.min_edge.z + s.z/2;

	glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
	glPushMatrix();
	
	glTranslatef(c.x, c.y, c.z);

	glDisable(GL_LIGHTING);
	glColor3f(1.0, 1.0, 1.0);
//	glColor3f(0.8, 0.8, 0.8);
	if (surf->render->compositor->draw_bvol==M4_Bounds_Box) {
		glScalef(s.x, s.y, s.z);
		VS3D_DrawMeshIntern(surf, surf->render->unit_bbox);
	} else {
		glScalef(mesh->bounds.radius, mesh->bounds.radius, mesh->bounds.radius);
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		VS3D_DrawMeshIntern(surf, surf->render->unit_bsphere);
	}
	glPopMatrix();
	glPopAttrib();
}

void VS3D_DrawMesh(VisualSurface *surf, M4Mesh *mesh)
{
	if (surf->render->wiremode != M4_WireOnly) VS3D_DrawMeshIntern(surf, mesh);

	if (surf->render->wiremode != M4_WireNone) {
		glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		glDisable(GL_LIGHTING);
		glColor3f(0.0, 0.0, 0.0);
		VS3D_DrawMeshIntern(surf, mesh);
		glPopAttrib();
	}

	if (surf->render->compositor->draw_bvol) VS3D_DrawMeshBoundingVolume(surf, mesh);
}


/*only used for ILS/ILS2D or IFS2D outline*/
void VS3D_StrikeMesh(VisualSurface *surf, M4Mesh *mesh, Float width, u32 dash_style)
{
	u16 style;
	
	if (mesh->mesh_type != MESH_LINESET) return;

	switch (dash_style) {
	case M4StrikeDash:
		style = 0x1F1F;
		break;
	case M4StrikeDot:
		style = 0x3333;
		break;
	case M4StrikeDashDot:
		style = 0x6767;
		break;
	case M4StrikeDashDashDot:
		style = 0x33CF;
		break;
	case M4StrikeDashDotDot:
		style = 0x330F;
		break;
	default:
		style = 0;
		break;
	}

	if (style) {
		u32 factor = (u32) (width*0.5);
		if (!factor) factor = 1;
		glEnable(GL_LINE_STIPPLE);
		glLineStipple(factor, style); 
	}
	glLineWidth(width);

	VS3D_DrawMesh(surf, mesh);
}

void VS3D_SetMaterial2D(VisualSurface *surf, SFColor col, Float alpha)
{
	glDisable(GL_LIGHTING);
	if (alpha != 1.0) glEnable(GL_BLEND);
	glColor4f(col.red, col.green, col.blue, alpha);
}


void VS3D_PushState(VisualSurface *surf)
{
	/*store:
	all on/off state 
	polygon face culling & orientation
	lighting state
	*/
	glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING_BIT | GL_CURRENT_BIT);
}

void VS3D_PopState(VisualSurface *surf)
{
	glPopAttrib();
    glShadeModel(GL_SMOOTH);
    glMatrixMode(GL_TEXTURE);
    glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
}

void VS3D_SetAntiAlias(VisualSurface *surf, Bool bOn)
{
	if (bOn) {
		glEnable(GL_LINE_SMOOTH);
		glEnable(GL_POINT_SMOOTH);
		if (surf->render->poly_aa)
			glEnable(GL_POLYGON_SMOOTH);
		else
			glDisable(GL_POLYGON_SMOOTH);
	} else {
		glDisable(GL_LINE_SMOOTH);
		glDisable(GL_POINT_SMOOTH);
		glDisable(GL_POLYGON_SMOOTH);
	}
}

void VS3D_ClearSurface(VisualSurface *surf, SFColor color, Float alpha)
{
	glClearColor(color.red, color.green, color.blue, alpha);
	glClear(GL_COLOR_BUFFER_BIT);
}

void VS3D_DrawImage(VisualSurface *surf, Float pos_x, Float pos_y, u32 width, u32 height, u32 pixelformat, char *data, Float scale_x, Float scale_y)
{
	u32 gl_format;
	glPixelZoom(scale_x, scale_y);

	gl_format = 0;
	switch (pixelformat) {
	case M4PF_RGB_24:
		gl_format = GL_RGB;
		break;
	case M4PF_RGB_32:
	case M4PF_RGBA:
		gl_format = GL_RGBA;
		break;
	case M4PF_ARGB:
		if (!surf->render->compositor->hw_caps.bgra_texture) return;
		gl_format = GL_BGRA_EXT;
		break;
	default:
		return;
	}

	/*glRasterPos2f doesn't accept point outside the view volume (it invalidates all draw pixel, draw bitmap)
	so we move to the center of the local coord system, draw a NULL bitmap with raster pos displacement*/
	glRasterPos2f(0, 0);
	glBitmap(0, 0, 0, 0, pos_x, -pos_y, NULL);
	glDrawPixels(width, height, gl_format, GL_UNSIGNED_BYTE, data);
	glBitmap(0, 0, 0, 0, -pos_x, pos_y, NULL);

}

void VS3D_GetMatrix(VisualSurface *surf, u32 mat_type, Float *mat)
{
	switch (mat_type) {
	case MAT_MODELVIEW:
		glGetFloatv(GL_MODELVIEW_MATRIX, mat);
		break;
	case MAT_PROJECTION:
		glGetFloatv(GL_PROJECTION_MATRIX, mat);
		break;
	case MAT_TEXTURE:
		glGetFloatv(GL_TEXTURE_MATRIX, mat);
		break;
	}
}

void VS3D_SetMatrixMode(VisualSurface *surf, u32 mat_type)
{
	switch (mat_type) {
	case MAT_MODELVIEW:
		glMatrixMode(GL_MODELVIEW);
		break;
	case MAT_PROJECTION:
		glMatrixMode(GL_PROJECTION);
		break;
	case MAT_TEXTURE:
		glMatrixMode(GL_TEXTURE);
		break;
	}
}

void VS3D_ResetMatrix(VisualSurface *surf)
{
	glLoadIdentity();
}
void VS3D_PushMatrix(VisualSurface *surf)
{
	glPushMatrix();
}
void VS3D_MultMatrix(VisualSurface *surf, Float *mat)
{
	glMultMatrixf(mat);
}
void VS3D_PopMatrix(VisualSurface *surf)
{
	glPopMatrix();
}


void VS3D_SetClip2D(VisualSurface *surf, M4Rect clip)
{
	Double g[4];
	g[2] = 0; 
	g[1] = 0; g[3] = clip.width/2;
	g[0] = -1; glClipPlane(GL_CLIP_PLANE0, g); glEnable(GL_CLIP_PLANE0);
	g[0] = 1; glClipPlane(GL_CLIP_PLANE1, g); glEnable(GL_CLIP_PLANE1);
	g[0] = 0; g[3] = clip.height/2;
	g[1] = -1; glClipPlane(GL_CLIP_PLANE2, g); glEnable(GL_CLIP_PLANE2);
	g[1] = 1; glClipPlane(GL_CLIP_PLANE3, g); glEnable(GL_CLIP_PLANE3);
}



void VS3D_SetMaterial(VisualSurface *surf, u32 material_type, Float *rgba)
{
	switch (material_type) {
	case MATERIAL_AMBIENT:
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, rgba);
		break;
	case MATERIAL_DIFFUSE:
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, rgba);
		break;
	case MATERIAL_SPECULAR:
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, rgba);
		break;
	case MATERIAL_EMISSIVE:
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, rgba);
		break;
	case MATERIAL_NONE:
		glColor4fv(rgba);
		break;
	}
}

void VS3D_SetShininess(VisualSurface *surf, Float shininess)
{
	glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess * 128);
}

void VS3D_SetState(VisualSurface *surf, u32 flag_mask, Bool setOn)
{
	if (setOn) {
		if (flag_mask & F3D_LIGHT) glEnable(GL_LIGHTING);
		if (flag_mask & F3D_BLEND) glEnable(GL_BLEND);
		if (flag_mask & F3D_COLOR) glEnable(GL_COLOR_MATERIAL);
	} else {
		if (flag_mask & F3D_LIGHT) glDisable(GL_LIGHTING);
		if (flag_mask & F3D_BLEND) glDisable(GL_BLEND);
		if (flag_mask & F3D_COLOR) glDisable(GL_COLOR_MATERIAL);
	}
}


M4Err R3D_GetScreenBuffer(VisualRenderer *vr, M4VideoSurface *fb)
{
	u32 i, hy;
	char *tmp;
	Render3D *sr = (Render3D *)vr->user_priv;

	fb->video_buffer = malloc(sizeof(char)*3*sr->out_width * sr->out_height);
	fb->width = sr->out_width;
	fb->pitch = 3*sr->out_width;
	fb->height = sr->out_height;
	fb->os_handle = NULL;
	fb->pixel_format = M4PF_RGB_24;

	/*don't understand why I get BGR when using GL_RGB*/
	glReadPixels(sr->out_x, sr->out_y, sr->out_width, sr->out_height, GL_BGR_EXT, GL_UNSIGNED_BYTE, fb->video_buffer);

	/*flip image (openGL always handle image data bottom to top) */
	tmp = malloc(sizeof(char)*fb->pitch);
	hy = fb->height/2;
	for (i=0; i<hy; i++) {
		memcpy(tmp, fb->video_buffer+ i*fb->pitch, fb->pitch);
		memcpy(fb->video_buffer + i*fb->pitch, fb->video_buffer + (fb->height - 1 - i) * fb->pitch, fb->pitch);
		memcpy(fb->video_buffer + (fb->height - 1 - i) * fb->pitch, tmp, fb->pitch);
	}
	free(tmp);
	return M4OK;
}

M4Err R3D_ReleaseScreenBuffer(VisualRenderer *vr, M4VideoSurface *framebuffer)
{
	free(framebuffer->video_buffer);
	framebuffer->video_buffer = 0;
	return M4OK;
}
