/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "visual_surface.h"
#include "stacks3d.h"

VisualSurface *VS_New(Bool for_3D)
{
	VisualSurface *tmp = malloc(sizeof(VisualSurface));
	memset(tmp, 0, sizeof(VisualSurface));
	tmp->back_stack = NewChain();
	tmp->view_stack = NewChain();
	if (for_3D) {
		tmp->fog_stack = NewChain();
		tmp->navigation_stack = NewChain();
	}
	tmp->is_pixel_metrics = 1;
	return tmp;
}

void VS_Delete(VisualSurface *surf)
{
	BindableStackDelete(surf->back_stack);
	BindableStackDelete(surf->view_stack);
	BindableStackDelete(surf->fog_stack);
	BindableStackDelete(surf->navigation_stack);
	free(surf);
}

DrawableStack *new_drawable(SFNode *owner, SceneRenderer *compositor)
{
	DrawableStack *tmp = malloc(sizeof(DrawableStack));
	if (tmp) stack_setup(tmp, owner, compositor);
	return tmp;
}
void delete_drawable(DrawableStack*d)
{
	drawablestack_predestroy(d);
	free(d);
}

void drawable_Node_PreDestroy(SFNode *n)
{
	DrawableStack *d = Node_GetPrivate(n);
	if (d) delete_drawable(d);
}

void BaseDrawableStack(SceneRenderer *sr, SFNode *node)
{
	Node_SetPrivate(node, new_drawable(node, sr));
	Node_SetPreDestroyFunction(node, drawable_Node_PreDestroy);
}


void delete_strikeinfo(StrikeInfo *info)
{
	if (info->outline) mesh_free(info->outline);
	free(info);
}

stack2D *new_stack2D(SFNode *owner, SceneRenderer *compositor)
{
	stack2D *tmp = malloc(sizeof(stack2D));
	if (tmp) {
		stack_setup(tmp, owner, compositor);
		tmp->path = m4_new_path();
		tmp->strike_list = NewChain();
	}
	return tmp;
}

void stack2D_predestroy(stack2D *d)
{
	Render3D *sr = (Render3D *)d->compositor->visual_renderer->user_priv;
	if (d->mesh) mesh_free(d->mesh);
	assert(d->path);
	m4_path_delete(d->path);

	while (ChainGetCount(d->strike_list)) {
		StrikeInfo *si = ChainGetEntry(d->strike_list, 0);
		ChainDeleteEntry(d->strike_list, 0);
		/*remove from main strike list*/
		ChainDeleteItem(sr->strike_bank, si);
		delete_strikeinfo(si);
	}
	DeleteChain(d->strike_list);
}

void delete_stack2D(stack2D *d)
{
	stack2D_predestroy(d);
	free(d);
}
void stack2D_node_predestroy(SFNode *n)
{
	stack2D *d = Node_GetPrivate(n);
	if (d) delete_stack2D(d);
}

void BaseStack2D(SceneRenderer *sr, SFNode *node)
{
	Node_SetPrivate(node, new_stack2D(node, sr));
	Node_SetPreDestroyFunction(node, stack2D_node_predestroy);
}


void stack2D_setup(stack2D *st, SceneRenderer *sr, SFNode *node)
{
	stack_setup(st, node, sr);
	st->path = m4_new_path();
	st->strike_list = NewChain();
}

void stack2D_reset(stack2D *d)
{
	Render3D *sr = (Render3D *)d->compositor->visual_renderer->user_priv;
	if (d->path) m4_path_reset(d->path);
	while (ChainGetCount(d->strike_list)) {
		StrikeInfo *si = ChainGetEntry(d->strike_list, 0);
		ChainDeleteEntry(d->strike_list, 0);
		ChainDeleteItem(sr->strike_bank, si);
		delete_strikeinfo(si);
	}
}

void VS_GetAspect2D(RenderEffect *eff, Aspect2D *asp)
{
#ifdef M4_DEF_Material2D
	B_Material2D *mat;
#endif
	/*default values*/
	memset(asp, 0, sizeof(Aspect2D));
	asp->fill_color.red = asp->fill_color.green = asp->fill_color.blue = 0.8f;
	/*this is a bug in the spec: by default line width is 1.0, but in meterMetrics this means half of the screen :)*/
	if (eff->surface->is_pixel_metrics) 
		asp->pen_props.width = 1;
	else
		asp->pen_props.width = 1.0f / eff->surface->width;

	asp->line_color = asp->fill_color;
	asp->pen_props.cap = M4LineCapFlat;
	asp->pen_props.join = M4LineJoinMiter;
	asp->pen_props.miterLimit = 4.0;
	asp->line_alpha = asp->alpha = 1.0;


#if defined(M4_DEF_Material2D) && defined(M4_DEF_Appearance) 
	if (!eff->appear) goto exit;
	mat = (B_Material2D *) ((B_Appearance *)eff->appear)->material;
	if (!mat || Node_GetTag((SFNode *)mat) != TAG_Material2D) goto exit;

	asp->line_color = asp->fill_color = mat->emissiveColor;
	asp->line_alpha = asp->alpha = 1.0f - mat->transparency;
	asp->filled = mat->filled;

	if (mat->lineProps == NULL) {
		if (asp->filled || (asp->alpha==0.0f)) asp->pen_props.width =0;
	} else {
		asp->lp = mat->lineProps;
		switch (Node_GetTag(mat->lineProps)) {
#ifdef M4_DEF_LineProperties
		case TAG_LineProperties:
		{
			B_LineProperties *lp = (B_LineProperties *)mat->lineProps;
			asp->pen_props.width = lp->width;
			asp->pen_props.dash = lp->lineStyle;
			asp->line_color = lp->lineColor;
		}
			break;
#endif
#ifdef M4_DEF_XLineProperties
		case TAG_XLineProperties:
		{
			B_XLineProperties *xlp = (B_XLineProperties *)mat->lineProps;
			asp->pen_props.dash = xlp->lineStyle;
			asp->line_color = xlp->lineColor;
			asp->line_alpha = 1.0f - xlp->transparency;
			asp->pen_props.width = xlp->width;
			asp->pen_props.is_scalable = xlp->isScalable;
			asp->pen_props.align = xlp->isCenterAligned ? M4LineCentered : M4LineInside;
			asp->pen_props.cap = xlp->lineCap;
			asp->pen_props.join = xlp->lineJoin;
			asp->pen_props.miterLimit = xlp->miterLimit;
			asp->pen_props.dash_offset = xlp->dashOffset;
			/*dash settings strutc is the same as MFFloat from XLP, typecast without storing*/
			if (xlp->dashes.count) {
				asp->pen_props.dash_set = (M4DashSettings *) &xlp->dashes;
			} else {
				asp->pen_props.dash_set = NULL;
			}
			asp->txh = R3D_GetTextureHandler(xlp->texture);
			asp->tx_trans = xlp->textureTransform;
		}
			break;
#endif
		}
	}

exit:
	/*update line width*/
	if (asp->pen_props.width) {
		Float g[16];
		VS3D_GetMatrix(eff->surface, MAT_MODELVIEW, g);
		if (g[0]<0) g[0] *= -1;
		if (g[5]<0) g[5] *= -1;
		asp->line_scale = (g[5]>g[0]) ? g[5] : g[0];
	}
#endif

}

StrikeInfo *VS_GetStrikeInfo(stack2D *st, Aspect2D *asp, RenderEffect *eff)
{
	StrikeInfo *si;
	u32 now, i;
	Render3D *sr = (Render3D *)st->compositor->visual_renderer->user_priv;
	Bool vect_outline = !sr->raster_outlines;
	if (!asp->pen_props.width || !st->path) return NULL;

	si = NULL;
	for (i=0; i<ChainGetCount(st->strike_list); i++) {
		si = ChainGetEntry(st->strike_list, i);
		/*note this includes default LP (NULL)*/
		if (si->lineProps == asp->lp) break;
		si = NULL;
	}
	/*not found, add*/
	if (!si) {
		si = malloc(sizeof(StrikeInfo));
		memset(si, 0, sizeof(StrikeInfo));
		si->lineProps = asp->lp;
		si->node2D = st->owner;
		ChainAddEntry(st->strike_list, si);
		ChainAddEntry(sr->strike_bank, si);
	}

	if (vect_outline != si->is_vectorial) {
		if (si->outline) mesh_free(si->outline);
		si->outline = NULL;
	}

	/*node changed or outline not build*/
	now = asp->lp ? (1 + R3D_LP_GetLastUpdateTime(asp->lp)) : si->last_update_time;
	if (!si->outline 
		|| (si->is_vectorial && ((now!=si->last_update_time) || (si->line_scale != asp->line_scale) )) ) {
		si->last_update_time = now;
		si->line_scale = asp->line_scale;
		if (si->outline) mesh_free(si->outline);
		si->outline = new_mesh();
		si->is_vectorial = vect_outline;
		if (vect_outline) {
			M4Path *outline_path;
			Float w = asp->pen_props.width;
			if (!asp->pen_props.is_scalable) asp->pen_props.width/=asp->line_scale;
			outline_path = m4_path_get_outline(st->path, asp->pen_props);
			asp->pen_props.width = w;
			TesselatePath(si->outline, outline_path, asp->txh ? 2 : 1);
			m4_path_delete(outline_path);
		} else {
			mesh_get_outline(si->outline, st->path);
		}
	}
	return si;
}


StrikeInfo *VS_GetStrikeInfoIFS(stack2D *st, Aspect2D *asp, RenderEffect *eff)
{
	StrikeInfo *si;
	u32 now, i;
	Render3D *sr = (Render3D *)st->compositor->visual_renderer->user_priv;
	if (!asp->pen_props.width || !st->path) return NULL;

	si = NULL;
	for (i=0; i<ChainGetCount(st->strike_list); i++) {
		si = ChainGetEntry(st->strike_list, i);
		/*note this includes default LP (NULL)*/
		if (si->lineProps == asp->lp) break;
		si = NULL;
	}
	/*not found, add*/
	if (!si) {
		si = malloc(sizeof(StrikeInfo));
		memset(si, 0, sizeof(StrikeInfo));
		si->lineProps = asp->lp;
		si->node2D = st->owner;
		ChainAddEntry(st->strike_list, si);
		ChainAddEntry(sr->strike_bank, si);
	}

	/*for this func the strike is always raster*/
	if (si->is_vectorial) {
		if (si->outline) mesh_free(si->outline);
		si->outline = NULL;
	}

	/*node changed or outline not build*/
	now = asp->lp ? R3D_LP_GetLastUpdateTime(asp->lp) : si->last_update_time;
	if ((now!=si->last_update_time) || (si->line_scale != asp->line_scale) ) {
		si->last_update_time = now;
		si->line_scale = asp->line_scale;
		if (si->outline) mesh_free(si->outline);
		si->outline = NULL;
		si->is_vectorial = 0;
	}
	return si;
}


Float Aspect_GetLineWidth(Aspect2D *asp)
{
	/*for raster outlines are already set to the proper width - note this may not work depending on GL...*/
	Float width = asp->pen_props.width;
	if (asp->pen_props.is_scalable) width *= asp->line_scale;
	return width;
}

void VS_Set2DStrikeAspect(VisualSurface *surf, Aspect2D *asp)
{
	if (asp->txh && tx_enable(asp->txh, asp->tx_trans)) return;
	/*no texture or not ready, use color*/
	VS3D_SetMaterial2D(surf, asp->line_color, asp->line_alpha);
}


void stack2D_draw(stack2D *st, RenderEffect *eff)
{
	Aspect2D asp;
	StrikeInfo *si;

	VS_GetAspect2D(eff, &asp);

	/*fill path*/
	if (VS_setup_texture(eff) || asp.filled) {
		if (asp.filled) VS3D_SetMaterial2D(eff->surface, asp.fill_color, asp.alpha);
		VS3D_DrawMesh(eff->surface, st->mesh);
		/*reset texturing in case of line texture*/
		if (asp.txh) tx_disable(asp.txh);
	}
	/*strike path*/
	si = VS_GetStrikeInfo(st, &asp, eff);
	if (si) {
		VS_Set2DStrikeAspect(eff->surface, &asp);
		if (!si->is_vectorial) {
			VS3D_StrikeMesh(eff->surface, si->outline, Aspect_GetLineWidth(&asp), asp.pen_props.dash);
		} else {
			VS3D_DrawMesh(eff->surface, si->outline);
		}
	}
}

void VS_setup_material(RenderEffect *eff)
{
	SFColor def;
	def.red = def.green = def.blue = 1.0f;

	if (!eff->appear) {
		/*use material2D to disable lighting*/
		VS3D_SetMaterial2D(eff->surface, def, 1.0);
		return;
	}

#ifdef M4_DEF_Appearance
	{
		SFNode *__mat = ((B_Appearance *)eff->appear)->material;
		if (!__mat) {
			VS3D_SetMaterial2D(eff->surface, def, 1.0);
		} 
#ifdef M4_DEF_Material
		else if (Node_GetTag((SFNode *)__mat) == TAG_Material) {
			Float vec[4], diff[4];
			u32 flag = F3D_LIGHT | F3D_COLOR;
			B_Material *mat = (B_Material *)__mat;
			if (mat->transparency>M4_EPSILON_FLOAT) {
				/*using antialiasing with alpha usually gives bad results (non-edge face segments are visible)*/
				VS3D_SetAntiAlias(eff->surface, 0);
				flag |= F3D_BLEND;
			}
			VS3D_SetState(eff->surface, flag, 1);

			vec[0] = mat->diffuseColor.red * mat->ambientIntensity;
			vec[1] = mat->diffuseColor.green * mat->ambientIntensity;
			vec[2] = mat->diffuseColor.blue * mat->ambientIntensity;
			vec[3] = (Float) (1.0 - mat->transparency);
			VS3D_SetMaterial(eff->surface, MATERIAL_AMBIENT, vec);

			diff[0] = mat->diffuseColor.red;
			diff[1] = mat->diffuseColor.green;
			diff[2] = mat->diffuseColor.blue;
			diff[3] = vec[3];
			VS3D_SetMaterial(eff->surface, MATERIAL_DIFFUSE, diff);

			VS3D_SetShininess(eff->surface, mat->shininess);

			vec[0] = mat->specularColor.red;
			vec[1] = mat->specularColor.green;
			vec[2] = mat->specularColor.blue;
			VS3D_SetMaterial(eff->surface, MATERIAL_SPECULAR, vec);
			
			vec[0] = mat->emissiveColor.red;
			vec[1] = mat->emissiveColor.green;
			vec[2] = mat->emissiveColor.blue;
			if (mat->diffuseColor.red || mat->diffuseColor.green || mat->diffuseColor.blue
				|| mat->specularColor.red || mat->specularColor.green || mat->specularColor.blue) {

				VS3D_SetMaterial(eff->surface, MATERIAL_EMISSIVE, vec);
				VS3D_SetMaterial(eff->surface, MATERIAL_NONE, diff);
			} else {
				VS3D_SetState(eff->surface, F3D_LIGHT, 0);
				VS3D_SetMaterial(eff->surface, MATERIAL_EMISSIVE, diff);
				VS3D_SetMaterial(eff->surface, MATERIAL_NONE, vec);
			}
		} 
#endif
#ifdef M4_DEF_Material2D
		else if (Node_GetTag((SFNode *)__mat) == TAG_Material2D) {
			B_Material2D *mat = (B_Material2D *)__mat;
			VS3D_SetState(eff->surface, F3D_LIGHT | F3D_COLOR, 0);
			if (mat->transparency>M4_EPSILON_FLOAT) VS3D_SetState(eff->surface, F3D_BLEND, 1);
			VS3D_SetMaterial2D(eff->surface, mat->emissiveColor, 1.0f - mat->transparency);
		}
#endif
	}
#endif
}



Bool VS_setup_texture(RenderEffect *eff)
{
#ifdef M4_DEF_Appearance
	Bool ret;
	TextureHandler *txh;
	if (!eff->appear) return 0;
	txh = R3D_GetTextureHandler(((B_Appearance *)eff->appear)->texture);
	ret = tx_enable(txh, ((B_Appearance *)eff->appear)->textureTransform);
	if (ret) eff->surface->tx_transparent = txh->transparent;
	return ret;
#else
	return 0;
#endif
}

void VS_setup_appearance(RenderEffect *eff)
{
	if (!VS_setup_texture(eff) || eff->surface->tx_transparent) VS_setup_material(eff);
}


#define DEBUG_VIEW_CULL		0
/*uncomment to disable frustum cull*/
#define DISABLE_VIEW_CULL


Bool node_cull_or_render(RenderEffect *eff, M4BBox *bbox) 
{
	M4BBox b;
	Float irad, rad;
	Bool do_sphere;
	u32 i, p_idx;
	M4Frustum *f;
	SFVec3f cdiff, vertices[8];

	if (eff->cull_flag == CULL_INSIDE) return 1;
	assert(eff->cull_flag != CULL_OUTSIDE);

#ifdef DISABLE_VIEW_CULL
	eff->cull_flag = CULL_INSIDE;
	return 1;
#endif

	/*empty bounds*/
	if (!bbox->is_set) {
		eff->cull_flag = CULL_OUTSIDE;
		return 0;
	}

	/*get bbox in world space*/
	b = *bbox;
	mx_apply_bbox(&eff->model_view, &b);
	f = &eff->surface->frustum;

	/*if camera is inside bbox consider we intersect*/
	if (bbox_point_inside(&b, &f->position)) {
		eff->cull_flag = CULL_INTERSECTS;
#if DEBUG_VIEW_CULL
		fprintf(stdout, "CULL INTER (camera in box test)\n");
#endif
		return 1;
	}
	/*first check: sphere vs frustum sphere intersection, this will discard far objects quite fast*/
	cdiff = vec_diff(&f->center, &b.center);
	rad = b.radius + f->radius;
	if (vec_lensq(&cdiff) > rad*rad) {
		eff->cull_flag = CULL_OUTSIDE;
#if DEBUG_VIEW_CULL
		fprintf(stdout, "CULL OUT (sphere-sphere test)\n");
#endif
		return 0;
	}

	/*second check: sphere vs frustum planes intersection, if any intersection is detected switch 
	to n/p vertex check.*/
	rad = b.radius;
	irad = -b.radius;
	do_sphere = 1;
	/*skip near/far tests in ortho mode*/
	i = f->is_ortho ? 2 : 0;
	for (; i<6; i++) {
		if (do_sphere) {
			Float d = m4_plane_get_distance(&f->planes[i], &b.center);
			if (d<irad) {
				eff->cull_flag = CULL_OUTSIDE;
#if DEBUG_VIEW_CULL
				fprintf(stdout, "CULL OUT (sphere-planes test)\n");
#endif
				return 0;
			}
			/*intersect, move to n-p vertex test*/
			if (d<rad) {
				/*get box vertices*/
				bbox_get_vertices(&b, vertices);
				do_sphere = 0;
			} else {
				continue;
			}
		}
		p_idx = f->p_idx[i];
		/*check p-vertex: if not in plane, we're out (since p-vertex is the closest point to the plane)*/
		if (m4_plane_get_distance(&f->planes[i], &vertices[p_idx])<0) {
			eff->cull_flag = CULL_OUTSIDE;
#if DEBUG_VIEW_CULL
			fprintf(stdout, "CULL OUT (p-vertex test)\n");
#endif
			return 0;
		}
		/*check n-vertex: if not in plane, we're intersecting*/
		if (m4_plane_get_distance(&f->planes[i], &vertices[7-p_idx])<0) {
			eff->cull_flag = CULL_INTERSECTS;
#if DEBUG_VIEW_CULL
			fprintf(stdout, "CULL INTER (n-vertex test)\n");
#endif
			return 1;
		}
	}
	eff->cull_flag = CULL_INSIDE;
#if DEBUG_VIEW_CULL
	fprintf(stdout, "CULL IN (%s test)\n", do_sphere ? "sphere-planes" : "n-p vertex");
#endif
	return 1;
}

static void VP_SetupProjection(VisualSurface *surf, RenderEffect *eff)
{
	M4Rect vp;
	Float pix_w, pix_h;
	SFNode *bindable;

	eff->surface = surf;

	vp.x = vp.y = 0;
	vp.width = pix_w = (Float) surf->width;
	vp.height = pix_h = (Float) surf->height;
	/*1-setup initial AR*/
	/*not composite texture*/
	if (surf->render->surface == surf) {
		/*size info, scale*/
		if (surf->render->compositor->has_size_info) {
			vp.x = (Float) surf->render->out_x; vp.y = (Float) surf->render->out_y;
			vp.width = (Float) surf->render->out_width;
			vp.height = (Float) surf->render->out_height;
			pix_w = (Float) surf->width;
			pix_h = (Float) surf->height;
		}
	}
	VS3D_SetViewport(surf, vp);

	/*store stacks*/
	eff->backgrounds = surf->back_stack;
	eff->viewpoints = surf->view_stack;
	eff->fogs = surf->fog_stack;
	eff->navigations = surf->navigation_stack;
	
	/*take care of all bindables*/
	eff->traversing_mode = TRAVERSE_RENDER_BINDABLE;

	/*load default matrices*/
	mx_init(surf->projection);
	mx_init(surf->modelview);
	/*by default we're ortho (2D, 3D without VP)*/
	surf->frustum.z_near = -512;
	surf->frustum.z_far = 512;
	mx_ortho(&surf->projection, -pix_w/2, pix_w/2, -pix_h/2, pix_h/2, -512, 512);
	surf->frustum.is_ortho = 1;

	/*setup viewpoint (this directly modifies the surface matrices)*/
	bindable = ChainGetEntry(eff->viewpoints, 0);
	if (Bindable_GetIsBound(bindable)) {
		Node_Render(bindable, eff);
	}

	/*setup projection/modelview*/
	VS3D_SetMatrixMode(surf, MAT_PROJECTION);
	VS3D_ResetMatrix(surf);
	VS3D_SetMatrixMode(surf, MAT_MODELVIEW);
	VS3D_ResetMatrix(surf);

	surf->frustum.aspectRatio = ((Float) eff->surface->width) / eff->surface->height;
	/*update frustum*/
	frustum_update(&surf->frustum, &surf->projection, &surf->modelview);

	/*setup surface bounds*/
	eff->bbox.max_edge.x = (Float) pix_w/2;
	eff->bbox.min_edge.x = -eff->bbox.max_edge.x;
	eff->bbox.max_edge.y = (Float) pix_h/2;
	eff->bbox.min_edge.y = -eff->bbox.max_edge.y;
	eff->bbox.max_edge.z = eff->bbox.min_edge.z = 0;
	eff->bbox.is_set = 1;

}

void VS_InitRender(VisualSurface *surf, RenderEffect *eff)
{
	SFNode *bindable;

	VS3D_Setup(surf);
	
	/*take care of all bindables*/
	eff->traversing_mode = TRAVERSE_RENDER_BINDABLE;

	/*setup projection*/
	VP_SetupProjection(surf, eff);

	/*setup navigation info*/
	bindable = eff->navigations ? ChainGetEntry(eff->navigations, 0) : NULL;
	if (Bindable_GetIsBound(bindable)) {
		Node_Render(bindable, eff);
	} else {
		surf->avatar_size.x = 0.25f; surf->avatar_size.y = 1.6f; surf->avatar_size.z = 0.75f;
		surf->visibility = 0.0f;
		surf->speed = 1.0f;
		/*consider that's a 3D surface if navigation stack is attached, and enable headlight*/
		if (surf->navigation_stack) {
			surf->navigation_flags = NAV_ANY | NAV_HEADLIGHT;
		} else {
			surf->navigation_flags = 0;
		}
	}

	/*set headlight if any*/
	VS3D_SetHeadlight(eff->surface, (surf->navigation_flags & NAV_HEADLIGHT) ? 1 : 0);

	VS3D_SetMatrixMode(surf, MAT_PROJECTION);
	VS3D_MultMatrix(surf, surf->projection.m);
	VS3D_SetMatrixMode(surf, MAT_MODELVIEW);
	VS3D_MultMatrix(surf, surf->modelview.m);

	/*setup background*/
	bindable = ChainGetEntry(eff->backgrounds, 0);
	if (Bindable_GetIsBound(bindable)) {
		Node_Render(bindable, eff);
	} else {
		SFColor col;
		Float alpha;
		if (surf->render->surface != surf) {
			col.red = col.green = col.blue = 0;
			alpha = 0.0f;
		} else {
			alpha = 1.0f;
			col = surf->render->compositor->clear_color;
		}
		VS3D_ClearSurface(surf, col, alpha);
	}

	/*setup fogs & lights*/

	/*and go for normal*/
	eff->traversing_mode = TRAVERSE_RENDER;

}
