/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#ifndef M4MESH_H_
#define M4MESH_H_

#include <gpac/intern/m4_render_tools.h>

typedef struct
{
	/*position*/
	SFVec3f pos;	
	/*normal*/
	SFVec3f normal;	
	/*color if used by mesh object*/
	SFColor color;
	/*texture coordinates*/
	SFVec2f texcoords;
} M4Vertex;

/*mesh type used*/
enum
{
	/*default: triangles described by indices (nb triangles = nb indices / 3) */
	MESH_TRIANGLES = 0,
	/*point set: indices is meaningless*/
	MESH_POINTSET,
	/*line set: lines described by indices (nb lines = nb indices / 2) */
	MESH_LINESET,
};

/*mesh flags*/
enum
{
	/*color per vertex is used*/
	MESH_HAS_COLOR = 1, 
	/*mesh is 2D: normal should be ignored and a global normal set to 0 0 1*/
	MESH_IS_2D = 1<<1, 
	/*mesh is 2D: normal should be ignored and a global normal set to 0 0 1*/
	MESH_NO_TEXTURE = 1<<2, 
	/*mesh faces are clockwise*/
	MESH_IS_CW = 1<<3, 
	/*mesh is solid (back face culling + 2 side lighting)*/
	MESH_IS_SOLID = 1<<4, 
};

/*mesh object used by all 2D/3D primitives. 
*/
typedef struct
{
	/*vertex list*/
	u32 v_count, v_alloc;
	M4Vertex *vertices;
	/*triangle indexes*/
	u32 i_count, i_alloc;
	u32 *indices;

	/*one of the above type*/
	u32 mesh_type;

	/*one of the above flags*/
	u32 flags;

	/*bounds info: bounding box and bounding sphere radius*/
	M4BBox bounds;
} M4Mesh;

M4Mesh *new_mesh();
void mesh_free(M4Mesh *mesh);
/*reset mesh*/
void mesh_reset(M4Mesh *mesh);
/*recompute mesh bounds*/
void mesh_update_bounds(M4Mesh *mesh);
/*adds new vertex*/
void mesh_set_vertex_vx(M4Mesh *mesh, M4Vertex *vx);
/*adds new vertex (exported for tesselator only)*/
void mesh_set_vertex(M4Mesh *mesh, Float x, Float y, Float z, Float nx, Float ny, Float nz, Float u, Float v);
/*adds an index (exported for tesselator only)*/
void mesh_set_index(M4Mesh *mesh, u32 idx);
/*adds an index (exported for tesselator only)*/
void mesh_set_triangle(M4Mesh *mesh, u32 id1, u32 id2, u32 id3);

/*inserts a box (lines only) of size 1.0 1.0 1.0*/
void mesh_new_unit_bbox(M4Mesh *mesh);

/*insert base primitives - low res indicates less subdivision steps for circles (cone, cylinder, ellipse, sphere)*/
void mesh_new_rectangle(M4Mesh *mesh, SFVec2f size);
void mesh_new_ellipse(M4Mesh *mesh, Float a_dia, Float b_dia, Bool low_res);
void mesh_new_box(M4Mesh *mesh, SFVec3f size);
void mesh_new_cylinder(M4Mesh *mesh, Float height, Float radius, Bool bottom, Bool side, Bool top, Bool low_res);
void mesh_new_cone(M4Mesh *mesh, Float height, Float radius, Bool bottom, Bool side, Bool low_res);
void mesh_new_sphere(M4Mesh *mesh, Float radius, Bool low_res);
/*inserts ILS/ILS2D and IFS2D outline when not filled*/
void mesh_new_ils(M4Mesh *mesh, SFNode *__coord, MFInt32 *coordIndex, SFNode *__color, MFInt32 *colorIndex, Bool colorPerVertex, Bool do_close);
/*inserts IFS2D*/
void mesh_new_ifs2d(M4Mesh *mesh, SFNode *ifs2d);
/*inserts IFS*/
void mesh_new_ifs(M4Mesh *mesh, SFNode *ifs);
/*inserts PS/PS2D*/
void mesh_new_ps(M4Mesh *mesh, SFNode *__coord, SFNode *__color);
/*builds mesh from path, performing tesselation if desired*/
void mesh_from_path(M4Mesh *mesh, M4Path *path);
/*builds mesh for outline of the given path*/
void mesh_get_outline(M4Mesh *mesh, M4Path *path);

#endif

