/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"

#ifdef M4_DEF_Layer2D

static void DestroyLayer2D(SFNode *node)
{
	Layer2DStack *st = (Layer2DStack *) Node_GetPrivate(node);
	DeleteGroupingNode((GroupingNode *)st);
	DeleteChain(st->backs);
	DeleteChain(st->views);
	free(st);
}

static void RenderLayer2D(SFNode *node, void *rs)
{
	Chain *oldb, *oldv, *oldf, *oldn;
	SFNode *viewport;
	GroupingNode *parent_bck;
	SFNode *back;
	B_Layer2D *l = (B_Layer2D *)node;
	Layer2DStack *st = (Layer2DStack *) Node_GetPrivate(node);
	RenderEffect *eff;


	eff = (RenderEffect *) rs;
	parent_bck = eff->parent;
	eff->parent = (GroupingNode *) st;

	/*layer2D maintains its own stacks*/
	oldb = eff->backgrounds;
	oldv = eff->viewpoints;
	oldf = eff->fogs;
	oldn = eff->navigations;
	eff->backgrounds = st->backs;
	eff->viewpoints = st->views;
	eff->fogs = eff->navigations = NULL;

	if (st->first) {
		/*first time, render back to register with stack/bind*/
		if (l->background) Node_Render((SFNode*) l->background, eff);
		/*first time, render viewport to register with stack/bind*/
		if (l->viewport) Node_Render((SFNode*) l->viewport, eff);
	}

	back = NULL;
	if (ChainGetCount(st->backs)) back = ChainGetEntry(st->backs, 0);

	viewport = NULL;
	if (ChainGetCount(st->views)) viewport = ChainGetEntry(st->views, 0);

	/*note we don't clear dirty flag, this is done in traversing*/
	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		st->clip.width = (Float) eff->surface->width;
		st->clip.height = (Float) eff->surface->height;

		/*setup bounds in local coord system, pixel metrics*/
		if (l->size.x>=0) {
			if (st->compositor->is_pixel_metrics) {
				st->clip.width = l->size.x;
			} else {
				st->clip.width *= l->size.x / 2;
			}
		}
		if (l->size.y>=0) {
			if (st->compositor->is_pixel_metrics) {
				st->clip.height = l->size.y;
			} else {
				st->clip.height *= l->size.y / 2;
			}
		}
		m4_rect_center(&st->clip, st->clip.width, st->clip.height);
		bbox_from_rect(&st->bbox, &st->clip);
	}
	
	if (back && (eff->traversing_mode==TRAVERSE_RENDER) ) {
		eff->traversing_mode = TRAVERSE_RENDER_BINDABLE;
		eff->bbox = st->bbox;
		Node_Render(back, eff);
		eff->traversing_mode = TRAVERSE_RENDER;
	}

	VS3D_PushState(eff->surface);
	/*setup clipping*/
	VS3D_SetClip2D(eff->surface, st->clip);


	/*apply viewport*/
	if (viewport) {
		u32 prev_mode = eff->traversing_mode;
		eff->traversing_mode = TRAVERSE_RENDER_BINDABLE;
		bbox_from_rect(&eff->bbox, &st->clip);

		/*vp changes current modelview transform*/
		VS3D_PushMatrix(eff->surface);
		Node_Render(viewport, eff);
		eff->traversing_mode = prev_mode;
	}
	
	grouping_traverse((GroupingNode *)st, eff);
	VS3D_PopState(eff->surface);

	if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) eff->bbox = st->bbox;

	/*restore modelview transform*/
	if (viewport) VS3D_PopMatrix(eff->surface);

	group_reset_children((GroupingNode*)st);

	/*restore effect*/
	eff->parent = parent_bck;
	eff->backgrounds = oldb;
	eff->viewpoints = oldv;
	eff->fogs = oldf;
	eff->navigations = oldn;

	/*check bindables*/
	if (st->first) {
		st->first = 0;
		/*we missed background or viewport (was not declared as bound during traversal, and is bound now), force redraw*/
		if ( (!back && ChainGetCount(st->backs)) || (!viewport && ChainGetCount(st->views)) ) 
			SR_Invalidate(st->compositor, NULL);
	}


}

void R3D_InitLayer2D(Render3D *sr, SFNode *node)
{
	Layer2DStack *stack = malloc(sizeof(Layer2DStack));
	SetupGroupingNode((GroupingNode*)stack, sr->compositor, node, ((B_Layer2D *)node)->children);
	stack->backs = NewChain();
	stack->views = NewChain();
	stack->first = 1;

	Node_SetPrivate(node, stack);
	Node_SetPreDestroyFunction(node, DestroyLayer2D);
	Node_SetRenderFunction(node, RenderLayer2D);
}

#endif
