/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "common_stacks.h"

#ifdef M4_DEF_Sound2D
/*sound2D wraper - spacialization is not supported yet*/
void RenderSound2D(SFNode *node, void *rs)
{
	BaseRenderEffect *eff = (BaseRenderEffect*) rs;
	B_Sound2D *snd = (B_Sound2D *)node;
	if (snd->source) {
		eff->sound_holder = node;
		Node_Render((SFNode *) snd->source, eff);
		eff->sound_holder = NULL;
		eff->trav_flags |= TF_DONT_CULL;
	}
}

void InitSound2D(LPSCENERENDER sr, SFNode *node)
{
	Node_SetRenderFunction(node, RenderSound2D);
}
#endif



#ifdef M4_DEF_AudioClip

static void DestroyAudioClip(SFNode *node)
{
	AudioClipStack *st = (AudioClipStack *)Node_GetPrivate(node);
	stop_audio(&st->input);
	if (st->time_handle.is_registered) {
		SR_UnregisterTimeNode(st->input.compositor, &st->time_handle);
	}
	free(st);
}


static void AC_Activate(AudioClipStack *st, B_AudioClip *ac)
{
	open_audio(&st->input, &ac->url);
	ac->isActive = 1;
	Node_OnEventOutSTR((SFNode *)ac, "isActive");

	MO_SetSpeed(st->input.stream, st->input.speed);
	/*rerender all graph to get parent audio group*/
	SR_Invalidate(st->input.compositor, NULL);
}

static void AC_Deactivate(AudioClipStack *st, B_AudioClip *ac)
{
	stop_audio(&st->input);
	ac->isActive = 0;
	Node_OnEventOutSTR((SFNode *)ac, "isActive");

	st->time_handle.needs_unregister = 1;
}

static void RenderAudioClip(SFNode *node, void *rs)
{
	BaseRenderEffect *eff = (BaseRenderEffect *)rs;
	B_AudioClip *ac = (B_AudioClip *)node;
	AudioClipStack *st = (AudioClipStack *)Node_GetPrivate(node);
	/*check end of stream*/
	if (st->input.stream && st->input.stream_finished) {
		if (MO_GetLoop(st->input.stream, ac->loop)) {
			restart_audio(&st->input);
		} else if (ac->isActive && MO_ShouldDeactivate(st->input.stream)) {
			/*deactivate*/
			AC_Deactivate(st, ac);
		}
	}
	if (ac->isActive) {
		audio_register_node(&st->input, (BaseRenderEffect*)rs);
	}
	if (st->set_duration && st->input.stream) {
		ac->duration_changed = MO_GetDuration(st->input.stream);
		Node_OnEventOutSTR(node, "duration_changed");
		st->set_duration = 0;
	}

	/*store mute flag*/
	st->input.is_muted = (eff->trav_flags & TF_SWITCHED_OFF);
}

static void AC_UpdateTime(TimeNode *tn)
{
	Double time;
	B_AudioClip *ac = (B_AudioClip *)tn->obj;
	AudioClipStack *st = (AudioClipStack *)Node_GetPrivate(tn->obj);

	if (! ac->isActive) {
		st->start_time = ac->startTime;
		st->input.speed = ac->pitch;
	}
	time = Node_GetSceneTime(tn->obj);
	if ((time<st->start_time) || (st->start_time<0)) return;
	
	if (ac->isActive) {
		if ( (ac->stopTime > st->start_time) && (time>=ac->stopTime)) {
			AC_Deactivate(st, ac);
			return;
		}
	}
	if (!ac->isActive) AC_Activate(st, ac);
}


void InitAudioClip(LPSCENERENDER sr, SFNode *node)
{
	AudioClipStack *st = malloc(sizeof(AudioClipStack));
	memset(st, 0, sizeof(AudioClipStack));
	setup_audio_input(&st->input, sr, node);

	st->time_handle.UpdateTimeNode = AC_UpdateTime;
	st->time_handle.obj = node;
	st->set_duration = 1;

	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderAudioClip);
	Node_SetPreDestroyFunction(node, DestroyAudioClip);

	SR_RegisterTimeNode(sr, &st->time_handle);
}


void AudioClipModified(SFNode *node)
{
	B_AudioClip *ac = (B_AudioClip *)node;
	AudioClipStack *st = (AudioClipStack *) Node_GetPrivate(node);
	if (!st) return;

	/*MPEG4 spec is not clear about that , so this is not forbidden*/
	if (st->input.is_open&& st->input.is_open) {
		if (audio_check_url_changed(&st->input, &ac->url)) {
			stop_audio(&st->input);
			open_audio(&st->input, &ac->url);
			/*force unregister to resetup audio cfg*/
			audio_unregister_node(&st->input);
			SR_Invalidate(st->input.compositor, NULL);
		}
	}

	//update state if we're active
	if (ac->isActive) 
		AC_UpdateTime(&st->time_handle);

	/*make sure we are still registered*/
	if (!st->time_handle.is_registered && !st->time_handle.needs_unregister) 
		SR_RegisterTimeNode(st->input.compositor, &st->time_handle);
	else
		st->time_handle.needs_unregister = 0;
}

#endif


#ifdef M4_DEF_AudioSource

static void DestroyAudioSource(SFNode *node)
{
	AudioSourceStack *st = (AudioSourceStack *)Node_GetPrivate(node);
	stop_audio(&st->input);
	audio_unregister_node(&st->input);
	if (st->time_handle.is_registered) {
		SR_UnregisterTimeNode(st->input.compositor, &st->time_handle);
	}
	free(st);
}


static void AS_Activate(AudioSourceStack *st, B_AudioSource *as)
{
	open_audio(&st->input, &as->url);
	st->is_active = 1;
	MO_SetSpeed(st->input.stream, st->input.speed);
	/*rerender all graph to get parent audio group*/
	SR_Invalidate(st->input.compositor, NULL);
}

static void AS_Deactivate(AudioSourceStack *st, B_AudioSource *as)
{
	stop_audio(&st->input);
	st->is_active = 0;
	st->time_handle.needs_unregister = 1;
}

static void RenderAudioSource(SFNode *node, void *rs)
{
	BaseRenderEffect*eff = (BaseRenderEffect*)rs;
	B_AudioSource *as = (B_AudioSource *)node;
	AudioSourceStack *st = (AudioSourceStack *)Node_GetPrivate(node);
	/*check end of stream*/
	if (st->input.stream && st->input.stream_finished) {
		if (MO_GetLoop(st->input.stream, 0)) {
			restart_audio(&st->input);
		} else if (st->is_active && MO_ShouldDeactivate(st->input.stream)) {
			/*deactivate*/
			AS_Deactivate(st, as);
		}
	}
	if (st->is_active) {
		audio_register_node(&st->input, (BaseRenderEffect*)rs);
	}

	/*store mute flag*/
	st->input.is_muted = (eff->trav_flags & TF_SWITCHED_OFF);
}

static void AS_UpdateTime(TimeNode *tn)
{
	Double time;
	B_AudioSource *as = (B_AudioSource *)tn->obj;
	AudioSourceStack *st = (AudioSourceStack *)Node_GetPrivate(tn->obj);

	if (! st->is_active) {
		st->start_time = as->startTime;
		st->input.speed = as->speed;
	}
	time = Node_GetSceneTime(tn->obj);
	if ((time<st->start_time) || (st->start_time<0)) return;
	
	if (st->input.input_ifce.GetSpeed(st->input.input_ifce.callback) && st->is_active) {
		if ( (as->stopTime > st->start_time) && (time>=as->stopTime)) {
			AS_Deactivate(st, as);
			return;
		}
	}
	if (!st->is_active) AS_Activate(st, as);
}


void InitAudioSource(LPSCENERENDER sr, SFNode *node)
{
	AudioSourceStack *st = malloc(sizeof(AudioSourceStack));
	memset(st, 0, sizeof(AudioSourceStack));
	setup_audio_input(&st->input, sr, node);

	st->time_handle.UpdateTimeNode = AS_UpdateTime;
	st->time_handle.obj = node;

	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderAudioSource);
	Node_SetPreDestroyFunction(node, DestroyAudioSource);

	SR_RegisterTimeNode(sr, &st->time_handle);
}


void AudioSourceModified(SFNode *node)
{
	B_AudioSource *as = (B_AudioSource *)node;
	AudioSourceStack *st = (AudioSourceStack *) Node_GetPrivate(node);
	if (!st) return;

	/*MPEG4 spec is not clear about that , so this is not forbidden*/
	if (st->input.is_open&& st->input.is_open) {
		if (audio_check_url_changed(&st->input, &as->url)) {
			stop_audio(&st->input);
			open_audio(&st->input, &as->url);
			/*force unregister to resetup audio cfg*/
			audio_unregister_node(&st->input);
			SR_Invalidate(st->input.compositor, NULL);
		}
	}

	//update state if we're active
	if (st->is_active) {
		AS_UpdateTime(&st->time_handle);
		if (!st->is_active) return;
	}

	/*make sure we are still registered*/
	if (!st->time_handle.is_registered && !st->time_handle.needs_unregister) 
		SR_RegisterTimeNode(st->input.compositor, &st->time_handle);
	else
		st->time_handle.needs_unregister = 0;
}

#endif

#ifdef M4_DEF_AudioClip

static void DestroyAudioBuffer(SFNode *node)
{
	AudioBufferStack *st = (AudioBufferStack *)Node_GetPrivate(node);
	B_AudioBuffer *ab = (B_AudioBuffer *)node;

	if (ab->isActive) audio_unregister_node(&st->output);
	if (st->time_handle.is_registered) 
		SR_UnregisterTimeNode(st->output.compositor, &st->time_handle);

	DeleteAudioMixer(st->am);
	if (st->buffer) free(st->buffer);
	DeleteChain(st->new_inputs);
	free(st);
}


/*we have no choice but always browsing the children, since a src can be replaced by a new one
without the parent being modified. We just collect the src and check against the current mixer inputs
to reset the mixer or not - the spec is not clear about that btw, shall rebuffering happen if a source is modified or not ...*/
static void RenderAudioBuffer(SFNode *node, void *rs)
{
	u32 count, i, j;
	Bool update_mixer;
	AudioGroup *parent;
	AudioBufferStack *st = (AudioBufferStack *)Node_GetPrivate(node);
	B_AudioBuffer *ab = (B_AudioBuffer *)node;
	BaseRenderEffect*eff = (BaseRenderEffect*) rs;

	parent = eff->audio_parent;
	eff->audio_parent = (AudioGroup *) st;
	count = ChainGetCount(ab->children);
	for (i=0; i<count; i++) {
		SFNode *ptr = ChainGetEntry(ab->children, i);
		Node_Render(ptr, eff);
	}

	AM_Lock(st->am, 1);

	/*if no new inputs don't change mixer config*/
	update_mixer = ChainGetCount(st->new_inputs) ? 1 : 0;
	
	if (ChainGetCount(st->am->inputs) == ChainGetCount(st->new_inputs)) {
		update_mixer = 0;
		/*check mixer*/
		for (i=0; i<ChainGetCount(st->am->inputs); i++) {
			Bool found = 0;
			AudioInterface *ifce = ChainGetEntry(st->am->inputs, i);
			for (j=0; j<ChainGetCount(st->new_inputs); j++) {
				AudioInput *cur = ChainGetEntry(st->new_inputs, j);
				if (ifce == &cur->input_ifce) {
					found = 1;
					break;
				}
			}
			/*not found*/
			if (!found) {
				update_mixer = 1;
				break;
			}
		}
	}

	if (update_mixer) {
		AM_RemoveAllInputs(st->am);
		st->am->force_channel_out = 1;
		st->am->out_channels = ab->numChan;
	}

	while (ChainGetCount(st->new_inputs)) {
		AudioInput *src = ChainGetEntry(st->new_inputs, 0);
		ChainDeleteEntry(st->new_inputs, 0);
		if (update_mixer) AM_AddSource(st->am, &src->input_ifce);
	}

	AM_Lock(st->am, 0);
	eff->audio_parent = parent;

	/*Note the audio buffer is ALWAYS registered untill destroyed since buffer filling shall happen even when inactive*/
	if (!st->output.register_with_parent || !st->output.register_with_renderer) 
		audio_register_node(&st->output, eff);
}



static void AB_Activate(AudioBufferStack *st, B_AudioBuffer *ab)
{
	ab->isActive = 1;
	Node_OnEventOutSTR((SFNode *)ab, "isActive");
	/*rerender all graph to get parent audio group*/
	SR_Invalidate(st->output.compositor, NULL);
	st->done = 0;
	st->read_pos = 0;
}

static void AB_Deactivate(AudioBufferStack *st, B_AudioBuffer *ab)
{
	ab->isActive = 0;
	Node_OnEventOutSTR((SFNode *)ab, "isActive");
	st->time_handle.needs_unregister = 1;
}

static void AB_UpdateTime(TimeNode *tn)
{
	Double time;
	B_AudioBuffer *ab = (B_AudioBuffer *)tn->obj;
	AudioBufferStack *st = (AudioBufferStack *)Node_GetPrivate(tn->obj);

	if (! ab->isActive) {
		st->start_time = ab->startTime;
	}
	time = Node_GetSceneTime(tn->obj);
	if ((time<st->start_time) || (st->start_time<0)) return;
	
	if (ab->isActive) {
		if ( (ab->stopTime > st->start_time) && (time>=ab->stopTime)) {
			AB_Deactivate(st, ab);
			return;
		}
		/*THIS IS NOT NORMATIVE*/
		if ( !ab->loop && st->done) {
			AB_Deactivate(st, ab);
			return;
		}
	}
	if (!ab->isActive) AB_Activate(st, ab);
}




static char *AB_FetchFrame(void *callback, u32 *size, u32 audio_delay_ms)
{
	u32 blockAlign;
	AudioBufferStack *st = (AudioBufferStack *) Node_GetPrivate( ((AudioInput *) callback)->owner);
	B_AudioBuffer *ab = (B_AudioBuffer*)st->output.owner;

	if (!st->is_init) return NULL;
	if (!st->buffer) {
		st->done = 0;
		st->buffer_size = (u32) ceil(ab->length * st->output.input_ifce.bytes_per_sec);
		blockAlign = st->am->out_channels*st->am->out_bits_per_sample / 8;
		/*BLOCK ALIGN*/
		while (st->buffer_size%blockAlign) st->buffer_size++;
		st->buffer = malloc(sizeof(char) * st->buffer_size);
		memset(st->buffer, 0, sizeof(char) * st->buffer_size);
		st->read_pos = st->write_pos = 0;
	}
	if (st->done) return NULL;

	/*even if not active, fill the buffer*/
	if (st->write_pos < st->buffer_size) {
		u32 written;
		while (1) {
			/*just try to completely fill it*/
			written = AM_GetMix(st->am, st->buffer + st->write_pos, st->buffer_size - st->write_pos);
			if (!written) break;
			st->write_pos += written;
			assert(st->write_pos<=st->buffer_size);
		}
	}
	/*not playing*/
	if (! ab->isActive) return NULL;
	*size = st->write_pos - st->read_pos;
	assert(st->output.input_ifce.bytes_per_sec);
	return st->buffer + st->read_pos;
}

static void AB_ReleaseFrame(void *callback, u32 nb_bytes)
{
	AudioBufferStack *st = (AudioBufferStack *) Node_GetPrivate( ((AudioInput *) callback)->owner);
	st->read_pos += nb_bytes;
	assert(st->read_pos<=st->write_pos);
	if (st->read_pos==st->write_pos) {
		if (st->write_pos<st->buffer_size) {
			/*reading faster than buffering - let's still attempt to fill the buffer*/
#if 0
			st->write_pos = st->buffer_size;
			fprintf(stdout, "Warning: AudioBuffer done playing before buffer filling done\n");
#endif
		} else if ( ((B_AudioBuffer*)st->output.owner)->loop) {
			st->read_pos = 0;
		} else {
			st->done = 1;
		}
	}
}

static Bool AB_IsInit(void *callback)
{
	AudioBufferStack *st = (AudioBufferStack *) Node_GetPrivate( ((AudioInput *) callback)->owner);
	return st->is_init;
}

static Float AB_GetSpeed(void *callback)
{
	B_AudioBuffer *ab = (B_AudioBuffer *) ((AudioInput *) callback)->owner;
	return ab->pitch;
}

static Bool AB_GetChannelVolume(void *callback, Float *vol)
{
	Float volume = 1.0;
	AudioInput *ai = (AudioInput *) callback;
	if (ai->snd2D) {
#ifdef M4_DEF_Sound2D
		volume = ((B_Sound2D *)ai->snd2D)->intensity;
#endif
	}
	vol[0] = vol[1] = vol[2] = vol[3] = vol[4] = vol[5] = volume;	
	return (volume==1.0) ? 0 : 1;
}

static Bool AB_IsMuted(void *callback)
{
	/*no mute on AudioBuffer*/
	return 0;
}

static Bool AB_GetConfig(void *callback, u32 *sample_rate, u32 *num_channels, u32 *bitspersample)
{
	AudioBufferStack *st = (AudioBufferStack *) Node_GetPrivate( ((AudioInput *) callback)->owner);

	if (AM_Reconfig(st->am)) {
		if (st->buffer) free(st->buffer);
		st->buffer = NULL;
		st->buffer_size = 0;
	}
	*sample_rate = st->am->out_sample_rate;
	*num_channels = st->am->out_channels;
	*bitspersample = st->am->out_bits_per_sample;
	st->is_init = (st->am->out_sample_rate && st->am->out_channels && st->am->out_bits_per_sample) ? 1 : 0;
	return st->is_init;
}

void AB_AddSource(AudioGroup *_this, AudioInput *src)
{
	AudioBufferStack *st = (AudioBufferStack *)_this;
	if (!src) return;
	/*just collect the input, reconfig is done once all children are rendered*/
	ChainAddEntry(st->new_inputs, src);
}


void setup_audiobufer(AudioInput *ai, LPSCENERENDER sr, SFNode *node)
{
	memset(ai, 0, sizeof(AudioInput));
	ai->owner = node;
	ai->compositor = sr;
	/*NEVER used for audio buffer*/
	ai->stream = NULL;
	/*setup io interface*/
	ai->input_ifce.FetchFrame = AB_FetchFrame;
	ai->input_ifce.ReleaseFrame = AB_ReleaseFrame;
	ai->input_ifce.GetConfig = AB_GetConfig;
	ai->input_ifce.GetChannelVolume = AB_GetChannelVolume;
	ai->input_ifce.GetSpeed = AB_GetSpeed;
	ai->input_ifce.IsInit = AB_IsInit;
	ai->input_ifce.IsMuted = AB_IsMuted;
	ai->input_ifce.callback = ai;
	ai->speed = 1.0;
}

void InitAudioBuffer(LPSCENERENDER sr, SFNode *node)
{
	AudioBufferStack *st = malloc(sizeof(AudioBufferStack));
	memset(st, 0, sizeof(AudioBufferStack));

	/*use our private input*/
	setup_audiobufer(&st->output, sr, node);
	st->add_source = AB_AddSource;

	st->time_handle.UpdateTimeNode = AB_UpdateTime;
	st->time_handle.obj = node;
	st->set_duration = 1;

	st->am = NewAudioMixer();
	st->new_inputs = NewChain();

	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderAudioBuffer);
	Node_SetPreDestroyFunction(node, DestroyAudioBuffer);
	SR_RegisterTimeNode(sr, &st->time_handle);
}


void AudioBufferModified(SFNode *node)
{
	B_AudioBuffer *ab = (B_AudioBuffer *)node;
	AudioBufferStack *st = (AudioBufferStack *) Node_GetPrivate(node);
	if (!st) return;

	//update state if we're active
	if (ab->isActive) 
		AB_UpdateTime(&st->time_handle);

	/*make sure we are still registered*/
	if (!st->time_handle.is_registered && !st->time_handle.needs_unregister) 
		SR_RegisterTimeNode(st->output.compositor, &st->time_handle);
	else
		st->time_handle.needs_unregister = 0;
}

#endif

