/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_render_dev.h>

AudioMixer *NewAudioMixer()
{
	AudioMixer *am;
	am = (AudioMixer *) malloc(sizeof(AudioMixer));
	if (!am) return NULL;
	memset(am, 0, sizeof(AudioMixer));
	am->mx = NewMutex();
	am->inputs = NewChain();
	am->isEmpty = 1;
	return am;
}

void DeleteAudioMixer(AudioMixer *am)
{
	DeleteChain(am->inputs);
	MX_Delete(am->mx);
	free(am);
}

void AM_RemoveAllInputs(AudioMixer *am)
{
	if (am->isEmpty) return;
	AM_Lock(am, 1);
	while (ChainGetCount(am->inputs)) {
		ChainDeleteEntry(am->inputs, 0);
	}
	AM_Lock(am, 0);
}

void AM_Lock(AudioMixer *am, Bool lockIt)
{
	if (lockIt)
		MX_P(am->mx);
	else
		MX_V(am->mx);
}


void AM_AddSource(AudioMixer *am, AudioInterface *src)
{
	u32 i;
	for (i=0; i<ChainGetCount(am->inputs); i++) {
		if ((AudioInterface *)ChainGetEntry(am->inputs, i) == src) return;
	}
	AM_Lock(am, 1);
	ChainAddEntry(am->inputs, src);
	am->must_reconfig = 1;
	am->isEmpty = 0;
	AM_Lock(am, 0);
}

void AM_RemoveSource(AudioMixer *am, AudioInterface *src)
{
	if (am->isEmpty) return;
	AM_Lock(am, 1);
	ChainDeleteItem(am->inputs, src);
	am->isEmpty = ChainGetCount(am->inputs);
	/*we don't ask for reconfig when removing a node*/
	AM_Lock(am, 0);
}


static u32 get_best_samplerate(AudioMixer *am, u32 obj_samplerate, u32 outCH, u32 outBPS)
{
	if (!am->ar) return obj_samplerate;
	if (!am->ar->audio_out || !am->ar->audio_out->QueryOutputSampleRate) return obj_samplerate;
	return am->ar->audio_out->QueryOutputSampleRate(am->ar->audio_out, obj_samplerate, outCH, outBPS);
}

void AM_SetMixerConfig(AudioMixer *am, u32 outSR, u32 outCH, u32 outBPS)
{
	u32 i, count, sr, ch, bps;
	AudioInterface *ai;

	AM_Lock(am, 1);

	am->out_bits_per_sample = outBPS;
	/*force channel is never set for main mixer*/
	if (!am->force_channel_out) am->out_channels = outCH;
	am->out_sample_rate = get_best_samplerate(am, outSR, outCH, outBPS);

	count = ChainGetCount(am->inputs);
	for (i=0; i<count; i++) {
		ai = (AudioInterface *) ChainGetEntry(am->inputs, i);
		if (!ai->GetConfig(ai->callback, &sr, &ch, &bps)) continue;
		ai->fracSampRatio = sr * 0xffff / am->out_sample_rate;
		ai->fracSample = 0;
	}
	AM_Lock(am, 0);
}

/*this is called before each mix*/
Bool AM_Reconfig(AudioMixer *am)
{
	AudioInterface *ai;
	u32 i, count, numInit, max_sample_rate, max_channels, max_bps, cfg_changed;
	if (am->isEmpty || !am->must_reconfig) return 0;

	AM_Lock(am, 1);
	numInit = 0;
	max_sample_rate = am->out_sample_rate;
	max_channels = am->out_channels;
	max_bps = am->out_bits_per_sample;
	cfg_changed = 0;

	count = ChainGetCount(am->inputs);
	for (i=0; i<count; i++) {
		ai = (AudioInterface *) ChainGetEntry(am->inputs, i);
		if (ai->bytes_per_sec && ai->IsInit(ai->callback)) {
			numInit++;
			continue;
		}
		if (!ai->GetConfig(ai->callback, &ai->sr, &ai->chan, &ai->bps)) continue;
		/*update out cfg*/
		if (count==1 && max_sample_rate!=ai->sr) {
			cfg_changed = 1;
			max_sample_rate = ai->sr;
		} else if (max_sample_rate<ai->sr) {
			cfg_changed = 1;
			max_sample_rate = ai->sr;
		}
		if (count==1 && max_bps!=ai->bps) {
			cfg_changed = 1;
			max_bps = ai->bps;
		} else if (max_bps<ai->bps) {
			cfg_changed = 1;
			max_bps = ai->bps;
		}
		if (!am->force_channel_out && count==1 && max_channels!=ai->chan) {
			cfg_changed = 1;
			max_channels = ai->chan;
		} else if (!am->force_channel_out && max_channels <ai->chan) {
			cfg_changed = 1;
			max_channels = ai->chan;
		}

		ai->bytes_per_sec = ai->sr * ai->chan * ai->bps / 8;
		/*cfg has changed, we must reconfig everything*/
		if (cfg_changed) continue;
		/*otherwise setup cfg*/
		ai->fracSampRatio = ai->sr * 0xffff / am->out_sample_rate;
		ai->fracSample = 0;
		memset(&ai->prevChan, 0, sizeof(s16)*MAX_CHANNELS);
		numInit++;
	}

	if (cfg_changed) AM_SetMixerConfig(am, max_sample_rate, max_channels, max_bps);
	/*all our inputs are setup*/
	if (numInit == count) am->must_reconfig = 0;

	AM_Lock(am, 0);
	return cfg_changed;
}

u32 audio_mix(AudioMixer *am, AudioInterface *ai, s16 *dst, u32 dst_size, void *src, u32 src_len, Float *vol, Float speed, u32 *src_read);

static u32 MixInputStream(AudioMixer *am, AudioInterface *ai, void *buffer, u32 buffer_size, u32 audio_delay)
{
	Float speed;
	u32 size, written, read, remain;
	Float pan[6];
	char *data, *ptr;

	speed = ai->GetSpeed(ai->callback);
	if (speed<0) speed *= -1;
	else if (speed==0) buffer_size = 0;

	if (ai->IsMuted(ai->callback)) {
		pan[0] = pan[1] = pan[2] = pan[3] = pan[4] = pan[5] = 0;	
	} else {
		ai->GetChannelVolume(ai->callback, pan);
	}

	ptr = buffer;
	remain = buffer_size;
	while (remain) {
		data = ai->FetchFrame(ai->callback, &size, audio_delay);
		if (!data) break;
		written = audio_mix(am, ai, (s16 *) ptr, remain, data, size, pan, speed, &read);
		ptr += written;
		assert(read<=size);
		assert(written<=remain);
		remain -= written;
		ai->ReleaseFrame(ai->callback, read);
	}
	return buffer_size - remain;
}


u32 AM_GetMix(AudioMixer *am, void *buffer, u32 buffer_size)
{
	AudioInterface *ai;
	Float pan[6];
	u32 i, count, size, in_size, delay;
	char *data, *ptr;

	/*the config has changed we don't write to output since settings change*/
	if (AM_Reconfig(am)) return 0;

	AM_Lock(am, 1);
	count = ChainGetCount(am->inputs);
	if (!count) {
		memset(buffer, 0, buffer_size);
		AM_Lock(am, 0);
		return 0;
	}
	delay = 0;
	if (am->ar && am->ar->resync_clocks) delay = am->ar->audio_delay;

	if (count!=1) goto do_mix;
	if (am->force_channel_out) goto do_mix;
	ai = (AudioInterface *) ChainGetEntry(am->inputs, 0);
	/*this happens if input SR cannot be mapped to output audio hardware*/
	if (ai->sr != am->out_sample_rate) goto do_mix;
	if (ai->chan != am->out_channels) goto do_mix;
	if (ai->GetSpeed(ai->callback)!=1.0) goto do_mix;
	if (ai->GetChannelVolume(ai->callback, pan)) goto do_mix;

	ptr = (char *)buffer;
	in_size = buffer_size;

	while (buffer_size) {
		data = ai->FetchFrame(ai->callback, &size, delay);
		if (!data) break;
		/*don't copy more than possible*/
		if (size > buffer_size) size = buffer_size;
		if (ai->IsMuted(ai->callback)) {
			memset(ptr, 255, size);
		} else {
			memcpy(ptr, data, size);
		}
		buffer_size -= size;
		ptr += size;
		ai->ReleaseFrame(ai->callback, size);
	}

	/*not completely filled*/
	if (buffer_size) {
//		if (!data) printf("NOT ENOUGH INPUT DATA %d remain\n", buffer_size);
		memset(ptr, 0, buffer_size);
	}

	AM_Lock(am, 0);
	return (in_size - buffer_size);

do_mix:

	memset(buffer, 0, buffer_size);
	in_size = 0;
	for (i=0; i<count; i++) {
		ai = ChainGetEntry(am->inputs, i);
		size = MixInputStream(am, ai, buffer, buffer_size, delay);
		if (in_size<size) in_size=size;
	}
	AM_Lock(am, 0);
	return in_size;
}


#define SHORTCLIP(t) t = (t) > M4_SHORT_MAX ? M4_SHORT_MAX : ( ( (t) <M4_SHORT_MIN) ? M4_SHORT_MIN : (t) ) 

u32 audio_mix(AudioMixer *am, AudioInterface *ai, s16 *dst, u32 dst_size, void *src, u32 src_len, Float *pan, Float speed, u32 *src_read)
{
	u32 i, j, in_ch, out_ch;
	s16 *input;
	s32 inChan[MAX_CHANNELS], inChanNext[MAX_CHANNELS], resChan[MAX_CHANNELS];
	u32 written, curPos, inc, src_samp;
	Bool resample = (am->out_sample_rate != ai->sr) ? 1 : 0;

	written = curPos = 0;
	input = (s16 *) src;
	*src_read = src_len;

	inc = (u32) (ai->bps * ai->chan * speed / 8);
	src_samp = (u32) (src_len * 8 / ai->bps / ai->chan);
	in_ch = ai->chan;
	out_ch = am->out_channels;

	for (i=0; i<src_samp; i++) {
		/*1- gather samples*/
		memset(&inChan, 0, sizeof(s32) * MAX_CHANNELS);
		if (curPos) {
			for (j=0; j<in_ch; j++) inChan[j] = *(input + (curPos>>1) - (in_ch-j) );
		} else {
			for (j=0; j<in_ch; j++) inChan[j] = ai->prevChan[j];
		}
		/*2 - resample*/
		if (resample) {
			for (j=0; j<in_ch; j++) {
				inChanNext[j] = *(input + (curPos>>1) + j);
				inChan[j] = (inChan[j] * (0xffff - ai->fracSample) + inChanNext[j] * ai->fracSample);
				inChan[j] >>= 16;
			}
		}

		/*3 - todo: get input mix table (currently only move mono->stereo), and do mix*/
		if (in_ch==1) inChan[1] = inChan[0];

		for (j=0; j<out_ch ; j++) {
			resChan[j] = (s32) (inChan[j] * pan[j]) + *(dst);
			SHORTCLIP(resChan[j]);
			*dst++ = (s16) resChan[j];
			written += 2;
		}

		/*4 - move to next sample*/
		if (resample) {
			ai->fracSample += ai->fracSampRatio;
			if (ai->fracSample >=0xffff) {
				ai->fracSample -= 0xffff;
				curPos += inc;
			}
		} else {
			curPos += inc;
		}
		if ((curPos + inc> src_len) || (dst_size==written) ) break;
	}
	if (resample) {
		for (j=0; j<in_ch; j++) ai->prevChan[j] = inChanNext[j];
	} else {
		for (j=0; j<in_ch; j++) ai->prevChan[j] = inChan[j];
	}
	if (i) *src_read = curPos;
	return written;
}

