/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Authoring Tools sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/m4_author.h>
#include <gpac/intern/m4_scenegraph_dev.h>

#define BT_LINE_SIZE	4000

typedef struct
{
	M4Err last_error;
	u32 line;

	char szMsg[512];

	FILE *input;
	Bool done;
	/*0: no unicode, 1: UTF-16BE, 2: UTF-16LE*/
	u32 unicode_type;

	/*extern node initializer*/
	void (*BTNodeInit)(void *cbk, SFNode *n);
	/*extern error reporter*/
	void (*BTOnError)(void *cbk, char *msg);
	/*callback for above functions*/
	void *bt_cbck;

	/*routes are not created in the graph when parsing, so we need to track insert and delete/replace*/
	Chain *unresolved_routes, *inserted_routes;
	Chain *undef_nodes, *def_nodes;

	char line_buffer[BT_LINE_SIZE], cur_buffer[500];
	s32 line_size, line_pos, line_start_pos;

	/*result context*/
	M4SceneManager *ctx;
	/*current scene graph*/
	LPSCENEGRAPH scene_graph;
	/*set when parsing proto*/
	LPPROTO parsing_proto;

	/*current stream ID, AU time and RAP flag*/
	u32 stream_id;
	u32 au_time;
	Bool au_is_rap;

	/*current BIFS stream & AU*/
	M4StreamContext *bifs_es;
	M4AUContext *bifs_au;
	u32 base_bifs_id;

	/*current OD stream & AU*/
	M4StreamContext *od_es;
	M4AUContext *od_au;
	u32 base_od_id;
} BTParser;

M4Err bt_parse_bifs_command(BTParser *parser, char *name, Chain *cmdList);
LPROUTE bt_parse_route(BTParser *parser, Bool skip_def, SGCommand *com);
void bt_resolve_routes(BTParser *parser);

#define BT_REPORT_ERR(parser) \
	if (parser->BTOnError) parser->BTOnError(parser->bt_cbck, parser->szMsg);	\
	else fprintf(stdout, parser->szMsg);	\

M4Err BTLogErr1(BTParser *parser, char *str)
{
	sprintf(parser->szMsg, "Error line %d: %s\n", parser->line, str);
	BT_REPORT_ERR(parser);
	return parser->last_error = M4BadParam;
}

M4Err BTLogErr2(BTParser *parser, char *str, const char *field)
{
	sprintf(parser->szMsg, "Error line %d: \"%s\": %s\n", parser->line, field, str);
	BT_REPORT_ERR(parser);
	return parser->last_error = M4BadParam;
}


SFNode *BTNewNode(BTParser *parser, u32 tag) 
{
	SFNode *n = SG_NewNode(parser->scene_graph, tag);
	if (n && !parser->parsing_proto) Node_Init(n);
	return n;
}

void bt_check_line(BTParser *parser)
{
	while (
		(parser->line_buffer[parser->line_pos]==' ') 
		|| (parser->line_buffer[parser->line_pos]=='\t') 
		) parser->line_pos++;

	if (parser->line_buffer[parser->line_pos]=='#') parser->line_size = parser->line_pos;
	else if ((parser->line_buffer[parser->line_pos]=='/') && (parser->line_buffer[parser->line_pos+1]=='/') ) parser->line_size = parser->line_pos;

	if (parser->line_size == parser->line_pos) {
		/*string based input - done*/
		if (!parser->input) return;

next_line:
		parser->line_start_pos = ftell(parser->input);
		parser->line_buffer[0] = 0;
		if (parser->unicode_type) {
			u8 c1, c2;
			unsigned short wchar;
			unsigned short l[BT_LINE_SIZE];
			unsigned short *dst = l;
			Bool is_ret = 0;
			u32 last_space_pos, last_space_pos_stream;
			u32 go = BT_LINE_SIZE - 1;
			last_space_pos = last_space_pos_stream = 0;
			while (go && !feof(parser->input) ) {
				c1 = fgetc(parser->input);
				c2 = fgetc(parser->input);
				/*Little-endian order*/
				if (parser->unicode_type==2) {
					if (c2) { wchar = c2; wchar <<=8; wchar |= c1; }
					else wchar = c1;
				} else {
					wchar = c1;
					if (c2) { wchar <<= 8; wchar |= c2;}
				}
				*dst = wchar;
				if (wchar=='\r') is_ret = 1;
				else if (wchar=='\n') {
					dst++;
					break;
				}
				else if (is_ret && wchar!='\n') {
					u32 fpos = ftell(parser->input);
					fseek(parser->input, fpos-2, SEEK_SET);
					is_ret = 1;
					break;
				}
				if (wchar==' ') {
					last_space_pos_stream = ftell(parser->input);
					last_space_pos = dst - l;
				}
				dst++;
				go--;

			}
			*dst = 0;
			/*long line, rewind stream to last space*/
			if (!go) {
				u32 rew_pos = ftell(parser->input) - 2*(dst - &l[last_space_pos]);
				fseek(parser->input, rew_pos, SEEK_SET);
				l[last_space_pos+1] = 0;
			}
			/*check eof*/
			if (l[0]==0xFFFF) {
				parser->done = 1;
				return;
			}
			/*convert to mbc string*/
			dst = l;
			utf8_wcstombs(parser->line_buffer, BT_LINE_SIZE, (const unsigned short **) &dst);

			if (!strlen(parser->line_buffer) && feof(parser->input)) {
				parser->done = 1;
				return;
			}
		} else {
			if ((fgets(parser->line_buffer, BT_LINE_SIZE, parser->input) == NULL) 
				|| (!strlen(parser->line_buffer) && feof(parser->input))) {
				parser->done = 1;
				return;
			}
			/*watchout for long lines*/
			if (1 + strlen(parser->line_buffer) == BT_LINE_SIZE) {
				u32 rew, pos;
				rew = 0;
				while (parser->line_buffer[strlen(parser->line_buffer)-1] != ' ') {
					parser->line_buffer[strlen(parser->line_buffer)-1] = 0;
					rew++;
				}
				pos = ftell(parser->input);
				fseek(parser->input, pos-rew, SEEK_SET);
			}
		}


		while (
			(parser->line_buffer[strlen(parser->line_buffer)-1]=='\n')
			|| (parser->line_buffer[strlen(parser->line_buffer)-1]=='\r')
			|| (parser->line_buffer[strlen(parser->line_buffer)-1]=='\t')
			)
			parser->line_buffer[strlen(parser->line_buffer)-1] = 0;

		
		parser->line_size = strlen(parser->line_buffer);
		parser->line_pos = 0;
		parser->line++;

		while ((parser->line_buffer[parser->line_pos]==' ') || (parser->line_buffer[parser->line_pos]=='\t'))
			parser->line_pos++;
		if ( (parser->line_buffer[parser->line_pos]=='#') 
			|| ( (parser->line_buffer[parser->line_pos]=='/') && (parser->line_buffer[parser->line_pos+1]=='/')) )
			goto next_line;
	}
	if (!parser->line_size) {
		if (!feof(parser->input)) bt_check_line(parser);
		else parser->done = 1;
	}
	else if (!parser->done && (parser->line_size == parser->line_pos)) bt_check_line(parser);
}

Bool bt_check_code(BTParser *parser, char code)
{
	bt_check_line(parser);
	if (parser->line_buffer[parser->line_pos]==code) {
		parser->line_pos++;
		return 1;
	}
	return 0;
}

char *bt_get_next(BTParser *parser, Bool point_break)
{
	u32 has_quote;
	s32 i;
	bt_check_line(parser);
	i=0;
	has_quote = 0;
	while (1) {
		if (parser->line_buffer[parser->line_pos + i] == '\"') {
			if (!has_quote) has_quote = 1;
			else has_quote = 0;
			parser->line_pos += 1;

			if (parser->line_pos+i==parser->line_size) break;
			continue;
		}
		if (!has_quote) {
			if (!parser->line_buffer[parser->line_pos + i]) break;
			else if (parser->line_buffer[parser->line_pos + i] == ' ') break;
			else if (parser->line_buffer[parser->line_pos + i] == '\t') break;
			else if (parser->line_buffer[parser->line_pos + i] == '{') break;
			else if (parser->line_buffer[parser->line_pos + i] == '}') break;
			else if (parser->line_buffer[parser->line_pos + i] == ']') break;
			else if (parser->line_buffer[parser->line_pos + i] == '[') break;
			else if (parser->line_buffer[parser->line_pos + i] == ',') break;
			else if (point_break && parser->line_buffer[parser->line_pos + i] == '.') break;
		}		
		parser->cur_buffer[i] = parser->line_buffer[parser->line_pos + i];
		i++;
		if (parser->line_pos+i==parser->line_size) break;
	}
	parser->cur_buffer[i] = 0;
	parser->line_pos += i;
	return parser->cur_buffer;
}

char *bt_get_string(BTParser *parser)
{
	char *res;
	s32 i, size;

	res = malloc(sizeof(char) * 500);
	size = 500;
	while (parser->line_buffer[parser->line_pos]==' ') parser->line_pos++;
	
	if (parser->line_pos==parser->line_size) {
		if (feof(parser->input)) return NULL;
		bt_check_line(parser);
	}

	i=0;
	while (1) {
		if (parser->line_buffer[parser->line_pos] == '\"') 
			if (parser->line_buffer[parser->line_pos-1] != '\\') break;
		if (i==size) {
			res = realloc(res, sizeof(char) * (size+500));
			size += 500;
		}

		if ((parser->line_buffer[parser->line_pos]=='/') && (parser->line_buffer[parser->line_pos+1]=='/') ) {
			/*this looks like a comment*/
			if (!strstr(&parser->line_buffer[parser->line_pos], "\"")) {
				bt_check_line(parser);
				continue;
			}
		}
		if ((parser->line_buffer[parser->line_pos] != '\\') || (parser->line_buffer[parser->line_pos+1] != '"')) {
			/*handle UTF-8 - WARNING: if parser is in unicode string is already utf8 multibyte chars*/
			if (!parser->unicode_type && parser->line_buffer[parser->line_pos] & 0x80) {
				res[i] = 0xc0 | ( (parser->line_buffer[parser->line_pos] >> 6) & 0x3 );
				i++;
				parser->line_buffer[parser->line_pos] &= 0xbf;
			}
			res[i] = parser->line_buffer[parser->line_pos];
			i++;
		}
		parser->line_pos++;
		if (parser->line_pos==parser->line_size) {
			bt_check_line(parser);
		}

	}
	res[i] = 0;
	parser->line_pos += 1;
	return res;
}

M4Err bt_parse_float(BTParser *parser, const char *name, Float *val)
{
	u32 i;
	char *str = bt_get_next(parser, 0);
	if (!str) 
		return parser->last_error = M4IOErr;
	for (i=0; i<strlen(str); i++) {
		if (!isdigit(str[i]) && (str[i] != '.') && (str[i] != 'E') && (str[i] != 'e') && (str[i] != '-') && (str[i] != '+')) {
			return BTLogErr2(parser, "Number expected", name);
		}
	}
	if (!i) {
		return BTLogErr2(parser, "Number expected", name);
	}
	*val = (Float) atof(str);
	return M4OK;
}
M4Err bt_parse_time(BTParser *parser, const char *name, SFTime *val)
{
	u32 i;
	char *str = bt_get_next(parser, 0);
	if (!str) 
		return parser->last_error = M4IOErr;
	for (i=0; i<strlen(str); i++) {
		if (!isdigit(str[i]) && (str[i] != '.') && (str[i] != 'E') && (str[i] != 'e') && (str[i] != '-') && (str[i] != '+')) {
			return BTLogErr2(parser, "Number expected", name);
		}
	}
	if (!i) {
		return BTLogErr2(parser, "Number expected", name);
	}
	*val = atof(str);
	return M4OK;
}
M4Err bt_parse_int(BTParser *parser, const char *name, SFInt32 *val)
{
	u32 i;
	char *str = bt_get_next(parser, 0);
	if (!str) 
		return parser->last_error = M4IOErr;
	/*URL ODID*/
	if (!strnicmp(str, "od:", 3)) str += 3;

	for (i=0; i<strlen(str); i++) {
		if (!isdigit(str[i]) && (str[i] != 'E') && (str[i] != 'e') && (str[i] != '-')) {
			return BTLogErr2(parser, "Number expected", name);
		}
	}
	if (!i) {
		return BTLogErr2(parser, "Number expected", name);
	}
	*val = atoi(str);
	return M4OK;
}
M4Err bt_parse_bool(BTParser *parser, const char *name, SFBool *val)
{
	char *str = bt_get_next(parser, 0);
	if (!str) 
		return parser->last_error = M4IOErr;
	if (!stricmp(str, "true") || !stricmp(str, "1") ) {
		*val = 1;
	}
	else if (!stricmp(str, "false") || !stricmp(str, "0") ) {
		*val = 0;
	} else {
		return BTLogErr2(parser, "Boolean expected", name);
	}
	return M4OK;
}

M4Err bt_parse_color(BTParser *parser, const char *name, SFColor *col)
{
	u32 i;
	u32 val;
	char *str = bt_get_next(parser, 0);
	if (!str) return parser->last_error = M4IOErr;

	/*HTML code*/
	if (str[0]=='$') {
		sscanf(str, "%x", &val);
		col->red = (Float) ((val>>16) & 0xFF) / 255.0f;
		col->green = (Float) ((val>>8) & 0xFF) / 255.0f;
		col->blue = (Float) (val & 0xFF) / 255.0f;
		return parser->last_error;
	} 

	for (i=0; i<strlen(str); i++) {
		if (!isdigit(str[i]) && (str[i] != '.') && (str[i] != 'E') && (str[i] != 'e') && (str[i] != '-') && (str[i] != '+')) {
			return BTLogErr2(parser, "Number expected", name);
		}
	}
	col->red = (Float) atof(str);
	bt_parse_float(parser, name, & col->green);
	bt_parse_float(parser, name, & col->blue);
	return parser->last_error;
}

void bt_sffield(BTParser *parser, FieldInfo *info, SFNode *n)
{
	switch (info->fieldType) {
	case FT_SFInt32:
		bt_parse_int(parser, info->name, (SFInt32 *)info->far_ptr);
		if (parser->last_error) return;
		break;
	case FT_SFBool:
		bt_parse_bool(parser, info->name, (SFBool *)info->far_ptr);
		if (parser->last_error) return;
		break;
	case FT_SFFloat:
		bt_parse_float(parser, info->name, (SFFloat *)info->far_ptr);
		if (parser->last_error) return;
		break;
	case FT_SFTime:
		bt_parse_time(parser, info->name, (SFTime *)info->far_ptr);
		if (parser->last_error) return;
		break;
	case FT_SFColor:
		bt_parse_color(parser, info->name, (SFColor *)info->far_ptr);
		break;
	case FT_SFVec2f:
		bt_parse_float(parser, info->name, & ((SFVec2f *)info->far_ptr)->x);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFVec2f *)info->far_ptr)->y);
		if (parser->last_error) return;
		break;
	case FT_SFVec3f:
		bt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->x);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->y);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->z);
		if (parser->last_error) return;
		break;
	case FT_SFRotation:
		bt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->xAxis);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->yAxis);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->zAxis);
		if (parser->last_error) return;
		bt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->angle);
		if (parser->last_error) return;
		break;
	case FT_SFString:
		if (bt_check_code(parser, '\"') || bt_check_code(parser, '\'')) {
			char *str = bt_get_string(parser); 
			if (!str) 
				goto err;
			if (((SFString *)info->far_ptr)->buffer) free(((SFString *)info->far_ptr)->buffer);
			((SFString *)info->far_ptr)->buffer = NULL;
			if (strlen(str))
				((SFString *)info->far_ptr)->buffer = str;
			else
				free(str);
		} else {
			goto err;
		}
		break;
	case FT_SFURL:
		if (bt_check_code(parser, '\"') || bt_check_code(parser, '\'')) {
			SFURL *url = (SFURL *)info->far_ptr;
			char *str = bt_get_string(parser);
			if (!str) goto err;
			if (url->url) free(url->url);
			url->url = NULL;
			url->OD_ID = 0;
			if (strchr(str, '#')) {
				url->url = str;
			} else {
				u32 id = 0;
				char *odstr = str;
				if (!strnicmp(str, "od:", 3)) odstr += 3;
				/*be carefull, an url like "11-regression-test.mp4" will return 1 on sscanf :)*/
				if (sscanf(odstr, "%d", &id) == 1) {
					char szURL[20];
					sprintf(szURL, "%d", id);
					if (strcmp(szURL, odstr)) id=0;
				}
				if (id) {
					url->OD_ID = id;
					free(str);
				} else {
					url->url = str;
				}
			}
		} else {
			s32 val;
			bt_parse_int(parser, info->name, & val );
			if (parser->last_error) return;
			((SFURL *)info->far_ptr)->OD_ID = val;
		}
		break;
	case FT_SFCommandBuffer:
	{
		SFCommandBuffer *cb = (SFCommandBuffer *)info->far_ptr;
		if (bt_check_code(parser, '{')) {
			while (!parser->last_error) {
				if (bt_check_code(parser, '}')) break;
				parser->last_error = bt_parse_bifs_command(parser, NULL, cb->commandList);
			}
		}
	}
		break;
	case FT_SFImage:
	{
		u32 i, size, v;
		char *str;
		SFImage *img = (SFImage *)info->far_ptr;
		bt_parse_int(parser, "width", &img->width);
		if (parser->last_error) return;
		bt_parse_int(parser, "height", &img->height);
		if (parser->last_error) return;
		bt_parse_int(parser, "nbComp", &v);
		if (parser->last_error) return;
		img->numComponents = v;
		size = img->width * img->height * img->numComponents;
		if (img->pixels) free(img->pixels);
		img->pixels = malloc(sizeof(char) * size);
		for (i=0; i<size; i++) {
			str = bt_get_next(parser, 0);
			if (strstr(str, "0x")) sscanf(str, "%x", &v);
			else sscanf(str, "%d", &v);
			switch (img->numComponents) {
			case 1:
				img->pixels[i] = (char) v;
				break;
			case 2:
				img->pixels[i] = (char) (v>>8)&0xFF;
				img->pixels[i+1] = (char) (v)&0xFF;
				i++;
				break;
			case 3:
				img->pixels[i] = (char) (v>>16)&0xFF;
				img->pixels[i+1] = (char) (v>>8)&0xFF;
				img->pixels[i+2] = (char) (v)&0xFF;
				i+=2;
				break;
			case 4:
				img->pixels[i] = (char) (v>>24)&0xFF;
				img->pixels[i+1] = (char) (v>>16)&0xFF;
				img->pixels[i+2] = (char) (v>>8)&0xFF;
				img->pixels[i+3] = (char) (v)&0xFF;
				i+=3;
				break;
			}
		}
	}
		break;
	case FT_SFScript:
	{
		SFScript *sc = (SFScript *) info->far_ptr;
		if (!bt_check_code(parser, '\"')) {
			BTLogErr1(parser, "\" expected in Script");
		}
		sc->script_text = bt_get_string(parser);
	}
		break;
	default:
		parser->last_error = M4NotSupported;
		break;

	}
	return;
err:
	BTLogErr2(parser, "Invalid field syntax", info->name);
}

void bt_mffield(BTParser *parser, FieldInfo *info, SFNode *n)
{
	FieldInfo sfInfo;
	Bool force_single = 0;

	if (!bt_check_code(parser, '[')) {
		force_single = 1;
	}

	sfInfo.fieldType = SG_GetSFType(info->fieldType);
	sfInfo.name = info->name;
	MFField_Reset(info->far_ptr, info->fieldType);

	while (!bt_check_code(parser, ']')) {
		MFField_Append(info->far_ptr, info->fieldType, &sfInfo.far_ptr);
		bt_sffield(parser, &sfInfo, n);
		if (parser->last_error) return;

		bt_check_code(parser, ','); 
		if (force_single) break;
	}
}

Bool BTCheckNDT(BTParser *parser, FieldInfo *info, SFNode *node, SFNode *parent)
{
	u32 i, tag;

#ifdef M4_DEF_Script
	if (parent->sgprivate->tag == TAG_Script) return 1;
#endif
	tag = node->sgprivate->tag;
	if (tag == TAG_ProtoNode) {
		tag = Proto_GetRenderingTag(((ProtoInstance *)node)->proto_interface);
		/*this happens with extern protos*/
		if (tag==TAG_UndefinedNode) return 1;
	}

	for (i=0;i<LAST_BIFS_VERSION; i++) {
		if (NDT_GetNodeType(info->NDTtype, tag, i+1)) return 1;
	}
	/*not defined yet, we assume NDT will be correct*/
	if (node->sgprivate->tag == TAG_UndefinedNode) return 1;
	/*not found*/
	sprintf(parser->szMsg, "Error line %d: node %s not valid in field %s\n", parser->line, Node_GetName(node), info->name);
	BT_REPORT_ERR(parser);
	parser->last_error = M4InvalidNode;
	Node_Unregister(node, parent);
	return 0;
}

u32 bt_GetDEFID(BTParser *parser, char *defName)
{
	SFNode *n;
	u32 ID;
	if (sscanf(defName, "N%d", &ID) == 1) {
		ID ++;
		if (parser->bifs_es->next_node_id<ID) parser->bifs_es->next_node_id = ID + 1;
		n = SG_FindNode(parser->scene_graph, ID);
		/*if an existing node use*/
		if (n) {
			u32 nID = 1 + parser->bifs_es->next_node_id;
			parser->bifs_es->next_node_id ++;
			fprintf(stdout, "WARNING: changing node %s ID from %d to %d\n", n->sgprivate->NodeName, n->sgprivate->NodeID-1, nID-1);
			Node_SetDEF(n, nID, n->sgprivate->NodeName);
		}
	} else {
		ID = 1 + parser->bifs_es->next_node_id;
		parser->bifs_es->next_node_id ++;
	}
	return ID;
}

Bool bt_IS_field(BTParser *parser, FieldInfo *info, SFNode *n)
{
	u32 i;
	LPPROTOFIELD pfield;
	FieldInfo pinfo;
	char *str;
	bt_check_line(parser);
	i=0;
	while ((parser->line_buffer[parser->line_pos + i] == ' ') || (parser->line_buffer[parser->line_pos + i] == '\t')) i++;
	if (strnicmp(&parser->line_buffer[parser->line_pos + i] , "IS", 2)) return 0;

	str = bt_get_next(parser, 0);
	str = bt_get_next(parser, 0);

	/*that's an ISed field*/
	pfield = Proto_FindFieldByName(parser->parsing_proto, str);
	if (!pfield) {
		BTLogErr2(parser, "Unknown proto field", str);
		return 1;
	}
	ProtoField_GetField(pfield, &pinfo);
	Proto_SetISField(parser->parsing_proto, pinfo.allIndex, n, info->allIndex);
	return 1;
}

void BT_CheckUnresolvedNodes(BTParser *parser)
{
	u32 i, count;
	count = ChainGetCount(parser->undef_nodes); 
	if (!count) return;
	for (i=0; i<count; i++) {
		SFNode *n = ChainGetEntry(parser->undef_nodes, i);
		assert(n->sgprivate->NodeName);
		fprintf(stdout, "Cannot find node %s\n", n->sgprivate->NodeName);
	}
	parser->last_error = M4BadParam;
}

Bool BT_HasBeenDEF(BTParser *parser, char *node_name)
{
	u32 i, count;
	count = ChainGetCount(parser->def_nodes);
	for (i=0; i<count; i++) {
		SFNode *n = ChainGetEntry(parser->def_nodes, i);
		if (!strcmp(n->sgprivate->NodeName, node_name)) return 1;
	}
	return 0;
}

SFNode *bt_SFNode(BTParser *parser, char *node_name, SFNode *parent)
{
	u32 tag, ID;
	Bool is_script, replace_prev, register_def;
	LPPROTO p;
	SFNode *node, *newnode, *undef_node;
	FieldInfo info;
	char *name;
	char * str;

	if (node_name) {
		str = node_name;
	} else {
		str = bt_get_next(parser, 0);
	}
	name = NULL;
	if (!stricmp(str, "NULL")) return NULL;

	ID = 0;
	register_def = 0;
	replace_prev = 0;
	undef_node = NULL;
	if (!stricmp(str, "DEF")) {
		register_def = 1;
		str = bt_get_next(parser, 0);
		name = strdup(str);
		undef_node = SG_FindNodeByName(parser->scene_graph, str);
		str = bt_get_next(parser, 0);
		if (undef_node) {
			ID = undef_node->sgprivate->NodeID;
			/*if we see twice a DEF N1 then force creation of a new node*/
			if (BT_HasBeenDEF(parser, name)) {
				undef_node = NULL;
			}
		} else {
			ID = bt_GetDEFID(parser, name);
		}
	}
	else if (!stricmp(str, "USE")) {
		str = bt_get_next(parser, 0);
		node = SG_FindNodeByName(parser->scene_graph, str);
		if (node && node->sgprivate->NodeID>parser->bifs_es->max_node_id)
			parser->bifs_es->max_node_id = node->sgprivate->NodeID;

		if (!node) {
			/*create a temp node (undefined)*/
			node = BTNewNode(parser, TAG_UndefinedNode);
			ID = bt_GetDEFID(parser, str);
			Node_SetDEF(node, ID, str);
			ChainAddEntry(parser->undef_nodes, node);
		} 
		Node_Register(node, parent);
		return node;
	}
	p = NULL;
	tag = Node_GetTagByName(str);
	if (!tag) {
		LPSCENEGRAPH sg = parser->scene_graph;
		while (1) {
			p = SG_FindProto(sg, parser->bifs_es->next_proto_id + 1, str);
			if (p) break;
			sg = sg->parent_scene;
			if (!sg) break;
		}
		if (!p) {
			/*locate proto*/
			BTLogErr2(parser, "not a valid/supported node", str);
			parser->last_error = M4InvalidNode;
			return NULL;
		}
		tag = TAG_ProtoNode;
	}
	if (undef_node && (undef_node->sgprivate->tag == tag)) {
		node = undef_node;
	} else {
		if (undef_node) replace_prev = 1;
		if (p) {
			node = Proto_CreateInstance(parser->scene_graph, p);
		} else {
			node = BTNewNode(parser, tag);
		}
	}
	is_script = 0;
#ifdef M4_DEF_Script
	if (tag==TAG_Script) is_script = 1;
#endif

	if (!node) {
		parser->last_error = M4UnknownNode;
		return NULL;
	}
	if (register_def) ChainAddEntry(parser->def_nodes, node);

	/*register before parsing in case a node uses this node...*/
	if (name) {
		if (!undef_node || replace_prev) {
			Node_SetDEF(node, ID, name);
		}
		free(name);
		name = NULL;
	}

	Node_Register(node, parent);

	/*remove temp node*/
	if (replace_prev) {
		Node_ReplaceAllInstances(undef_node, node, 0);
		ChainDeleteItem(parser->undef_nodes, undef_node);
	}


//	printf("create node %s\n", str);

	if (bt_check_code(parser, '{')) {

		while (1) {
			if (bt_check_code(parser, '}')) 
				break;

			str = bt_get_next(parser, 0);
			if (!str) {
				BTLogErr1(parser, "Invalid node syntax");
				parser->last_error = M4InvalidNode;
				goto err;
			}

			parser->last_error = Node_GetFieldByName(node, str, &info);

			if (is_script && parser->last_error) {
#ifdef M4_DEF_Script
				u32 eType, fType;
				LPSCRIPTFIELD sf;
				eType = 0;
				if (!stricmp(str, "eventIn")) eType = SFET_EventIn;
				else if (!stricmp(str, "eventOut")) eType = SFET_EventOut;
				else if (!stricmp(str, "field")) eType = SFET_Field;
				else {
					BTLogErr2(parser, "Unknown script event type", str);
					goto err;
				}
				str = bt_get_next(parser, 0);
				fType = GetFieldTypeByName(str);
				if (fType==FT_Unknown) {
					BTLogErr2(parser, "Unknown script field type", str);
					goto err;
				}
				parser->last_error = 0;
				str = bt_get_next(parser, 0);
				sf = SG_NewScriptField(node, eType, fType, str);
				parser->last_error = Node_GetFieldByName(node, str, &info);

				if (parser->parsing_proto && bt_IS_field(parser, &info, node)) continue;
				if ((eType == SFET_EventIn) || (eType == SFET_EventOut)) continue;
#endif
			}
			
			if (parser->last_error) {
				BTLogErr2(parser, "Unknown field", str);
				goto err;
			}
			
			if (parser->parsing_proto && bt_IS_field(parser, &info, node)) continue;

			switch (info.fieldType) {
			case FT_SFNode:
				newnode = bt_SFNode(parser, NULL, node);
				if (!newnode) goto err;
				if (!BTCheckNDT(parser, &info, newnode, node)) goto err;
				* ((SFNode **)info.far_ptr) = newnode;
				break;
			case FT_MFNode:
				if (!bt_check_code(parser, '[')) break;

				while (!bt_check_code(parser, ']')) {
					newnode = bt_SFNode(parser, NULL, node);
					if (!newnode) goto err;
					if (!BTCheckNDT(parser, &info, newnode, node)) goto err;
					ChainAddEntry(*(Chain **)info.far_ptr, newnode);
					if (parser->last_error!=M4OK) goto err;
				}
				break;
			default:
				if (SG_IsSFField(info.fieldType)) {
					bt_sffield(parser, &info, node);
				} else {
					bt_mffield(parser, &info, node);
				}
				if (parser->last_error) goto err;
				break;
			}
		}
	}
	return node;

err:
	Node_Unregister(node, parent);
	if (name) free(name);
	return NULL;
}
/*
	locate node, if not defined yet parse ahead in current AU
*/
SFNode *bt_peek_node(BTParser *parser, char *defID)
{
	SFNode *n;
	u32 tag, ID;
	char *str, *ret;
	char nName[1000];
	u32 pos, line, line_pos;
	
	n = SG_FindNodeByName(parser->scene_graph, defID);
	if (n) {
		if (n->sgprivate->NodeID > parser->bifs_es->max_node_id) 
			parser->bifs_es->max_node_id = n->sgprivate->NodeID;
		return n;
	}

	pos = parser->line_start_pos;
	line_pos = parser->line_pos;
	line = parser->line;
	strcpy(nName, defID);

	n = NULL;
	while (!parser->done) {
		str = bt_get_next(parser, 0);
		bt_check_code(parser, '[');
		bt_check_code(parser, ']');
		bt_check_code(parser, '{');
		bt_check_code(parser, '}');
		bt_check_code(parser, ',');
		bt_check_code(parser, '.');

		if (!stricmp(str, "AT")) {
			sprintf(parser->szMsg, "Cannot find node %s\n", nName);
			BT_REPORT_ERR(parser);
			parser->last_error = M4BadParam;
			break;
		}
		if (stricmp(str, "DEF")) continue;
		str = bt_get_next(parser, 0);
		ret = strdup(str);
		str = bt_get_next(parser, 0);
		if (!stricmp(str, "ROUTE") || stricmp(ret, nName)) {
			free(ret);
			continue;
		}
		tag = Node_GetTagByName(str);
		if (!tag) {
			LPPROTO p;
			LPSCENEGRAPH sg = parser->scene_graph;
			while (1) {
				p = SG_FindProto(sg, parser->bifs_es->next_proto_id + 1, str);
				if (p) break;
				sg = sg->parent_scene;
				if (!sg) break;
			}
			if (!p) {
				/*locate proto*/
				BTLogErr2(parser, "not a valid/supported node", str);
				parser->last_error = M4InvalidNode;
				return NULL;
			}
			n = Proto_CreateInstance(parser->scene_graph, p);
		} else {
			n = BTNewNode(parser, tag);
		}
		ID = bt_GetDEFID(parser, ret);
		Node_SetDEF(n, ID, ret);
		free(ret);
		/*NO REGISTER on peek (both scene graph or DEF list) because peek is only used to get node type
		and fields, never to insert in the graph*/
		break;
	}
	/*restore context*/
	fseek(parser->input, pos, SEEK_SET);
	parser->line_pos = parser->line_size;
	bt_check_line(parser);
	parser->line = line;
	parser->line_pos = line_pos;
	
	return n;
}

u32 bt_get_route(BTParser *parser, char *name) 
{
	u32 i;
	LPROUTE r = SG_FindRouteByName(parser->scene_graph, name);
	if (r) return r->ID;
	for (i=0; i<ChainGetCount(parser->inserted_routes); i++) {
		SGCommand *com = ChainGetEntry(parser->inserted_routes, i);
		if (com->def_name && !stricmp(com->def_name, name)) return com->RouteID;
	}
	return 0;
}


M4Err bt_parse_proto(BTParser *parser, char *proto_code, Chain *proto_list)
{
	FieldInfo info;
	u32 fType, eType, QPType;
	Bool externProto;
	LPPROTO proto, prevproto;
	LPPROTOFIELD pfield;
	LPSCENEGRAPH sg;
	char *str, *name;
	char szDefName[1024];
	Bool isDEF;

	if (proto_code) 
		str = proto_code;
	else
		str = bt_get_next(parser, 0);
	
	externProto = !stricmp(str, "externProto") ? 1 : 0;
	str = bt_get_next(parser, 0);
	name = strdup(str);
	if (!bt_check_code(parser, '[')) {
		return BTLogErr1(parser, "[ expected in proto declare");
	}
	proto = SG_NewProto(parser->scene_graph, parser->bifs_es->next_proto_id, name, proto_list ? 1 : 0);
	if (proto_list) ChainAddEntry(proto_list, proto);

	parser->bifs_es->next_proto_id++;

	free(name);
	/*get all fields*/
	while (!parser->last_error && !bt_check_code(parser, ']')) {
		str = bt_get_next(parser, 0);
		if (!stricmp(str, "eventIn")) eType = ET_EventIn;
		else if (!stricmp(str, "eventOut")) eType = ET_EventOut;
		else if (!stricmp(str, "field")) eType = ET_Field;
		else if (!stricmp(str, "exposedField")) eType = ET_ExposedField;
		else {
			BTLogErr2(parser, "unknown event type", str);
			goto err;
		}
		str = bt_get_next(parser, 0);
		fType = GetFieldTypeByName(str);
		if (fType==FT_Unknown) {
			BTLogErr2(parser, "unknown field type", str);
			goto err;
		}
		str = bt_get_next(parser, 0);
		pfield = Proto_NewField(proto, fType, eType, str);
		if ((eType==ET_EventIn) || (eType==ET_EventOut)) continue;
		ProtoField_GetField(pfield, &info);
		if (fType==FT_SFNode) {
			str = bt_get_next(parser, 0);
			if (stricmp(str, "NULL")) {
				BTLogErr1(parser, "SFNode protofield must use NULL default value");
				goto err;
			}
		} else if (fType==FT_MFNode) {
			if (bt_check_code(parser, '[')) {
				if (!bt_check_code(parser, ']')) {
					BTLogErr1(parser, "MFNode protofield must use empty default value");
					goto err;
				}
			}
		} else if (SG_IsSFField(fType)) {
			bt_sffield(parser, &info, NULL);
		} else {
			bt_mffield(parser, &info, NULL);
		}
		/*check QP info*/
		if (!bt_check_code(parser, '{')) continue;
		if (bt_check_code(parser, '}')) continue;
		str = bt_get_next(parser, 0);
		if (!stricmp(str, "QP")) {
			u32 nbBits, hasMin;
			Float ftMin, ftMax;
			bt_parse_int(parser, "QPType", &QPType);

			nbBits = 0;
			str = bt_get_next(parser, 0);
			if (!stricmp(str, "nbBits")) {
				bt_parse_int(parser, "nbBits", &nbBits);
				str = bt_get_next(parser, 0);
			}
			hasMin = 0;
			eType = 0;
			if (!stricmp(str, "b")) {
				hasMin = 1;
				if (!bt_check_code(parser, '{')) {
					BTLogErr2(parser, "Invalid proto coding parameter declare", str);
					goto err;
				}
				bt_parse_float(parser, "min", &ftMin);
				bt_parse_float(parser, "max", &ftMax);
				if (!bt_check_code(parser, '}')) {
					BTLogErr1(parser, "Invalid proto coding parameter declare");
					goto err;
				}
				if (SG_GetSFType(fType) == FT_SFInt32) {
					eType = FT_SFInt32;
				} else {
					eType = FT_SFFloat;
				}
			}
			ProtoField_SetQuantizationInfo(pfield, QPType, hasMin, eType, &ftMin, &ftMax, nbBits);
			if (!bt_check_code(parser, '}')) {
				BTLogErr1(parser, "Invalid proto coding parameter declare");
				goto err;
			}
		}
	}

	if (externProto) {
		str = bt_get_next(parser, 0);
		MFField_Alloc(&proto->ExternProto, FT_MFURL, 1);
		if (!strnicmp(str, "od:", 3)) {
			sscanf(str, "od:%d", &proto->ExternProto.vals[0].OD_ID);
		} else {
			if (!sscanf(str, "%d", &proto->ExternProto.vals[0].OD_ID)) {
				proto->ExternProto.vals[0].url = strdup(str);
			} else {
				char szURL[20];
				sprintf(szURL, "%d", proto->ExternProto.vals[0].OD_ID);
				if (strcmp(szURL, str)) {
					proto->ExternProto.vals[0].OD_ID = 0;
					proto->ExternProto.vals[0].url = strdup(str);
				}
			}
		}
		return M4OK;
	} 
	/*parse proto code*/
	if (!bt_check_code(parser, '{')) {
		BTLogErr1(parser, "Invalid Proto body declaration - at least one node must be defined");
		goto err;
	}

	prevproto = parser->parsing_proto;
	sg = parser->scene_graph;
	parser->parsing_proto = proto;
	parser->scene_graph = Proto_GetSceneGraph(proto);

	isDEF = 0;
	while (!bt_check_code(parser, '}')) {
		str = bt_get_next(parser, 0);
		if (!stricmp(str, "PROTO") || !stricmp(str, "EXTERNPROTO")) {
			bt_parse_proto(parser, str, NULL);
		} else if (!stricmp(str, "DEF")) {
			isDEF = 1;
			str = bt_get_next(parser, 0);
			strcpy(szDefName, str);
		} else if (!stricmp(str, "ROUTE")) {
			LPROUTE r = bt_parse_route(parser, 1, NULL);
			if (isDEF) {
				u32 rID = bt_get_route(parser, szDefName);
				if (!rID) {
					rID = 1 + parser->bifs_es->next_route_id;
					parser->bifs_es->next_route_id++;
				}
				Route_SetID(r, rID);
				Route_SetName(r, szDefName);
				isDEF = 0;
			}
		} else {
			SFNode *n = bt_SFNode(parser, str, NULL);
			if (!n) goto err;
			if (isDEF) {
				u32 ID = bt_GetDEFID(parser, szDefName);
				isDEF = 0;
				Node_SetDEF(n, ID, szDefName);
			}
			Proto_AddNodeCode(proto, n);
		}
	}
	bt_resolve_routes(parser);
	BT_CheckUnresolvedNodes(parser);
	parser->scene_graph = sg;
	parser->parsing_proto = prevproto;
	return parser->last_error;

err:
	SG_DeleteProto(proto);
	return parser->last_error;
}


LPROUTE bt_parse_route(BTParser *parser, Bool skip_def, SGCommand *com)
{
	LPROUTE r;
	char *str, nstr[1000], rName[1000];
	u32 rID;
	SFNode *orig, *dest;
	FieldInfo orig_field, dest_field;

	rID = 0;
	strcpy(nstr, bt_get_next(parser, 1));
	if (!skip_def && !stricmp(nstr, "DEF")) {
		str = bt_get_next(parser, 0);
		strcpy(rName, str);
		rID = bt_get_route(parser, rName);
		if (!rID) {
			rID = 1 + parser->bifs_es->next_route_id;
			parser->bifs_es->next_route_id++;
		}
		strcpy(nstr, bt_get_next(parser, 1));
	}
	orig = bt_peek_node(parser, nstr);
	if (!orig) {
		sprintf(parser->szMsg, "Error line %d: cannot find node %s\n", parser->line, nstr);
		BT_REPORT_ERR(parser);
		parser->last_error = M4BadParam;
		return NULL;
	}
	if (!bt_check_code(parser, '.')) {
		BTLogErr1(parser, ". expected in route decl");
		parser->last_error = M4BadParam;
		return NULL;
	}
	str = bt_get_next(parser, 0);
	if (Node_GetFieldByName(orig, str, &orig_field)!= M4OK) {
		sprintf(parser->szMsg, "Error line %d: %s not a field of node %s (%s)\n", parser->line, str, orig->sgprivate->NodeName, Node_GetName(orig));
		BT_REPORT_ERR(parser);
		parser->last_error = M4BadParam;
		return NULL;
	}
	str = bt_get_next(parser, 0);
	if (stricmp(str, "to")) {
		BTLogErr1(parser, "TO expected in route declaration");
		parser->last_error = M4BadParam;
		return NULL;
	}

	strcpy(nstr, bt_get_next(parser, 1));
	dest = bt_peek_node(parser, nstr);
	if (!dest) {
		sprintf(parser->szMsg, "Error line %d: cannot find node %s\n", parser->line, nstr);
		BT_REPORT_ERR(parser);
		parser->last_error = M4BadParam;
		return NULL;
	}
	if (!bt_check_code(parser, '.')) {
		BTLogErr1(parser, ". expected in route decl");
		parser->last_error = M4BadParam;
		return NULL;
	}
	str = bt_get_next(parser, 0);
	if (Node_GetFieldByName(dest, str, &dest_field)!= M4OK) {
		sprintf(parser->szMsg, "Error line %d: %s not a field of node %s (%s)\n", parser->line, str, dest->sgprivate->NodeName, Node_GetName(dest));
		BT_REPORT_ERR(parser);
		parser->last_error = M4BadParam;
		return NULL;
	}
	if (com) {
		com->fromNodeID = orig->sgprivate->NodeID;
		com->fromFieldIndex = orig_field.allIndex;
		com->toNodeID = dest->sgprivate->NodeID;
		com->toFieldIndex = dest_field.allIndex;
		if (rID) {
			com->RouteID = rID;
			com->def_name = strdup(rName);
		}
		return NULL;
	}
	r = SG_NewRoute(parser->scene_graph, orig, orig_field.allIndex, dest, dest_field.allIndex);
	if (r && rID) {
		Route_SetID(r, rID);
		Route_SetName(r, rName);
	}
	return r;
}

void bt_resolve_routes(BTParser *parser)
{
	SGCommand *com;
	/*resolve all commands*/
	while(ChainGetCount(parser->unresolved_routes) ) {
		com = ChainGetEntry(parser->unresolved_routes, 0);
		ChainDeleteEntry(parser->unresolved_routes, 0);
		switch (com->tag) {
		case SG_RouteDelete:
		case SG_RouteReplace:
			com->RouteID = bt_get_route(parser, com->unres_name);
			if (!com->RouteID) {
				BTLogErr2(parser, "Cannot resolve Route DEF", com->unres_name);
				parser->last_error = M4BadParam;
			}
			free(com->unres_name);
			com->unres_name = NULL;
			com->unresolved = 0;
			break;
		}
	}

	while (ChainGetCount(parser->inserted_routes)) ChainDeleteEntry(parser->inserted_routes, 0);
}


static void bd_set_com_node(SGCommand *com, SFNode *node)
{
	com->node = node;
	Node_Register(com->node, NULL);
}

M4Err bt_parse_bifs_command(BTParser *parser, char *name, Chain *cmdList)
{
	s32 pos;
	LPROUTE r;
	SFNode *n, *newnode;
	SGCommand *com;
	CommandFieldInfo *inf;
	FieldInfo info;
	char *str, field[1000];
	if (!name) {
		str = bt_get_next(parser, 0);
	} else {
		str = name;
	}
	com = NULL;
	pos = -2;
	/*REPLACE commands*/
 	if (!stricmp(str, "REPLACE")) {
		str = bt_get_next(parser, 1);
		if (!stricmp(str, "ROUTE")) {
			str = bt_get_next(parser, 0);
			r = SG_FindRouteByName(parser->scene_graph, str);
			if (!r) strcpy(field, str);
			str = bt_get_next(parser, 0);
			if (stricmp(str, "BY")) {
				BTLogErr1(parser, "BY expected");
				return parser->last_error = M4BadParam;
			}
			com = SG_NewCommand(SG_RouteReplace);
			if (r) {
				com->RouteID = r->ID;
			} else {
				com->unres_name = strdup(field);
				com->unresolved = 1;
				ChainAddEntry(parser->unresolved_routes, com);
			}
			bt_parse_route(parser, 1, com);
			ChainAddEntry(cmdList, com);
			return parser->last_error;
		} 
		/*scene replace*/
		if (!stricmp(str, "SCENE")) {
			str = bt_get_next(parser, 0);
			if (stricmp(str, "BY")) {
				BTLogErr1(parser, "BY expected");
				return parser->last_error = M4BadParam;
			}
			bt_resolve_routes(parser);
			while (ChainGetCount(parser->def_nodes)) ChainDeleteEntry(parser->def_nodes, 0);

			com = SG_NewCommand(SG_SceneReplace);
			parser->scene_graph = com->graph = NewSceneGraph(parser->BTNodeInit, parser->bt_cbck, NULL, NULL);
			n = bt_SFNode(parser, NULL, NULL);
			if (parser->last_error) goto err;
			if (n) {
				SG_SetRootNode(parser->scene_graph, n);
			} else {
				SG_Delete(com->graph);
				parser->scene_graph = com->graph = NULL;
			}
			ChainAddEntry(cmdList, com);
			return M4OK;
		}
		if (!stricmp(str, "LAST")) pos = -1;
		else if (!stricmp(str, "BEGIN")) pos = 0;

		bt_check_code(parser, '.');
		strcpy(field, str);
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "unknown node", field);
			return parser->last_error = M4BadParam;
		}
		str = bt_get_next(parser, 0);
		strcpy(field, str);
		if (bt_check_code(parser, '[')) {
			if ( (parser->last_error = bt_parse_int(parser, "index", &pos)) ) return parser->last_error;
			if (!bt_check_code(parser, ']')) {
				BTLogErr1(parser, "] expected");
				return parser->last_error = M4BadParam;
			}
		}
		/*node replace*/
		if (!stricmp(field, "BY")) {
			com = SG_NewCommand(SG_NodeReplace);
			bd_set_com_node(com, n);
			inf = SG_NewFieldCommand(com);
			inf->new_node = bt_SFNode(parser, NULL, n);
			inf->fieldType = FT_SFNode;
			inf->field_ptr = &inf->new_node;
			ChainAddEntry(cmdList, com);
			return parser->last_error;
		}
		str = bt_get_next(parser, 0);
		if (stricmp(str, "BY")) {
			BTLogErr1(parser, "BY expected");
			return parser->last_error = M4BadParam;
		}
		parser->last_error = Node_GetFieldByName(n, field, &info);
		if (parser->last_error) {
			BTLogErr2(parser, "Unknown node field", field);
			return parser->last_error;
		}
		/*field replace*/
		if (pos==-2) {
			com = SG_NewCommand(SG_FieldReplace);
			bd_set_com_node(com, n);
			inf = SG_NewFieldCommand(com);
			inf->fieldIndex = info.allIndex;
			inf->fieldType = info.fieldType;

			switch (info.fieldType) {
			case FT_SFNode:
				inf->new_node = bt_SFNode(parser, NULL, n);
				inf->field_ptr = &inf->new_node;
				if (!BTCheckNDT(parser, &info, inf->new_node, n)) goto err;
				break;
			case FT_MFNode:
				if (!bt_check_code(parser, '[')) break;
				inf->node_list = NewChain();
				inf->field_ptr = &inf->node_list;
				while (!bt_check_code(parser, ']')) {
					newnode = bt_SFNode(parser, NULL, n);
					if (!newnode) goto err;
					if (parser->last_error!=M4OK) goto err;
					if (!BTCheckNDT(parser, &info, newnode, n)) goto err;
					ChainAddEntry(inf->node_list, newnode);
				}
				break;
			default:
				inf->field_ptr = SG_NewFieldPointer(info.fieldType);
				info.far_ptr = inf->field_ptr;
				if (SG_IsSFField(info.fieldType)) {
					bt_sffield(parser, &info, n);
				} else {
					bt_mffield(parser, &info, n);
				}
				if (parser->last_error) goto err;
				break;
			}

			ChainAddEntry(cmdList, com);
			return parser->last_error;
		}
		/*indexed field replace*/
		com = SG_NewCommand(SG_IndexedReplace);
		bd_set_com_node(com, n);
		inf = SG_NewFieldCommand(com);
		inf->pos = pos;
		inf->fieldIndex = info.allIndex;
		if (SG_IsSFField(info.fieldType)) {
			BTLogErr2(parser, "MF type field expected", info.name);
			parser->last_error = M4BadParam;
			goto err;
		}
		inf->fieldType = SG_GetSFType(info.fieldType);
		switch (info.fieldType) {
		case FT_MFNode:
			inf->new_node = bt_SFNode(parser, NULL, n);
			inf->field_ptr = &inf->new_node;
 			break;
		default:
			info.fieldType = inf->fieldType;
			info.far_ptr = inf->field_ptr = SG_NewFieldPointer(inf->fieldType);
			bt_sffield(parser, &info, n);
			break;
		}
		if (parser->last_error) goto err;
		ChainAddEntry(cmdList, com);
		return parser->last_error;
	}
	/*INSERT commands*/
	if (!stricmp(str, "INSERT") || !stricmp(str, "APPEND")) {
		Bool is_append = !stricmp(str, "APPEND") ? 1 : 0;
		str = bt_get_next(parser, 0);
		if (!stricmp(str, "ROUTE")) {
			com = SG_NewCommand(SG_RouteInsert);
			bt_parse_route(parser, 0, com);
			if (parser->last_error) goto err;
			ChainAddEntry(cmdList, com);
			ChainAddEntry(parser->inserted_routes, com);
			return M4OK;
		}
		if (stricmp(str, "AT") && stricmp(str, "TO")) {
			BTLogErr1(parser, is_append ? "TO expected" : "AT expected");
			return parser->last_error = M4BadParam;
		}
		str = bt_get_next(parser, 1);
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "Unknown node", str);
			return parser->last_error = M4BadParam;
		}
		if (!bt_check_code(parser, '.')) {
			BTLogErr2(parser, ". expected", str);
			return parser->last_error = M4BadParam;
		}
		str = bt_get_next(parser, 1);
		strcpy(field, str);
		if (!is_append) {
			if (!bt_check_code(parser, '[')) {
				BTLogErr2(parser, "[ expected", str);
				return parser->last_error = M4BadParam;
			}
			bt_parse_int(parser, "index", &pos);
			if (!bt_check_code(parser, ']')) {
				BTLogErr2(parser, "] expected", str);
				return parser->last_error = M4BadParam;
			}
		} else {
			if (bt_check_code(parser, '[')) {
				BTLogErr1(parser, "[ unexpected in Append command");
				return parser->last_error = M4BadParam;
			}
			pos = -1;
		}
		if (!stricmp(field, "children")) {
			com = SG_NewCommand(SG_NodeInsert);
			bd_set_com_node(com, n);
			inf = SG_NewFieldCommand(com);
			inf->pos = pos;
			inf->new_node = bt_SFNode(parser, NULL, n);
			inf->fieldType = FT_SFNode;
			inf->field_ptr = &inf->new_node;
			if (parser->last_error) goto err;
			return ChainAddEntry(cmdList, com);
		}
		Node_GetFieldByName(n, field, &info);
		if (SG_IsSFField(info.fieldType)) {
			BTLogErr2(parser, "MF type field expected", info.name);
			parser->last_error = M4BadParam;
			goto err;
		}
		com = SG_NewCommand(SG_IndexedInsert);
		bd_set_com_node(com, n);
		inf = SG_NewFieldCommand(com);
		inf->pos = pos;
		inf->fieldIndex = info.allIndex;
		inf->fieldType = SG_GetSFType(info.fieldType);
		switch (info.fieldType) {
		case FT_MFNode:
			inf->new_node = bt_SFNode(parser, NULL, n);
			inf->field_ptr = &inf->new_node;
			break;
		default:
			info.fieldType = inf->fieldType;
			inf->field_ptr = SG_NewFieldPointer(inf->fieldType);
			info.far_ptr = inf->field_ptr;
			bt_sffield(parser, &info, n);
			break;
		}
		if (parser->last_error) goto err;
		ChainAddEntry(cmdList, com);
		return parser->last_error;
	}
	/*DELETE commands*/
	if (!stricmp(str, "DELETE")) {
		str = bt_get_next(parser, 1);
		if (!stricmp(str, "ROUTE")) {
			str = bt_get_next(parser, 0);
			com = SG_NewCommand(SG_RouteDelete);
			com->RouteID = bt_get_route(parser, str);
			if (!com->RouteID) {
				com->unres_name = strdup(str);
				com->unresolved = 1;
				ChainAddEntry(parser->unresolved_routes, com);
			}
			/*for bt<->xmt conversions*/
			com->def_name = strdup(str);
			return ChainAddEntry(cmdList, com);
		}
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "Unknown Node", str);
			return parser->last_error = M4BadParam;
		}
		if (!bt_check_code(parser, '.')) {
			com = SG_NewCommand(SG_NodeDelete);
			bd_set_com_node(com, n);
			return ChainAddEntry(cmdList, com);
		}
		str = bt_get_next(parser, 0);
		Node_GetFieldByName(n, str, &info);
		if (!bt_check_code(parser, '[')) {
			BTLogErr1(parser, "[ expected");
			return parser->last_error = M4BadParam;
		}
		bt_parse_int(parser, "index", &pos);
		if (!bt_check_code(parser, ']')) {
			BTLogErr1(parser, "[ expected");
			return parser->last_error = M4BadParam;
		}
		com = SG_NewCommand(SG_IndexedDelete);
		bd_set_com_node(com, n);
		inf = SG_NewFieldCommand(com);
		inf->fieldIndex = info.allIndex;
		inf->pos = pos;
		return ChainAddEntry(cmdList, com);
	}
	/*Extended BIFS commands*/

	/*GlobalQP commands*/
	if (!stricmp(str, "GLOBALQP")) {
		newnode = bt_SFNode(parser, NULL, NULL);
#ifdef M4_DEF_QuantizationParameter
		if (newnode && (newnode->sgprivate->tag != TAG_QuantizationParameter)) {
			BTLogErr1(parser, "Only QuantizationParameter node allowed in GLOBALQP");
			Node_Unregister(newnode, NULL);
			return parser->last_error = M4BadParam;
		}
		com = SG_NewCommand(SG_GlobalQuantizer);
		com->node = NULL;
		inf = SG_NewFieldCommand(com);
		inf->new_node = newnode;
		inf->field_ptr = &inf->new_node;
		inf->fieldType = FT_SFNode;
		return ChainAddEntry(cmdList, com);
#endif
	}

	/*MultipleReplace commands*/
	if (!stricmp(str, "MULTIPLEREPLACE")) {
		str = bt_get_next(parser, 0);
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "Unknown node", str);
			return parser->last_error = M4BadParam;
		}
		if (!bt_check_code(parser, '{')) {
			BTLogErr1(parser, "{ expected");
			return parser->last_error = M4BadParam;
		}
		com = SG_NewCommand(SG_MultipleReplace);
		bd_set_com_node(com, n);

		while (!bt_check_code(parser, '}')) {
			str = bt_get_next(parser, 0);
			parser->last_error = Node_GetFieldByName(n, str, &info);
			if (parser->last_error) {
				BTLogErr2(parser, "Unknown node field", str);
				goto err;
			}
			inf = SG_NewFieldCommand(com);
			inf->fieldIndex = info.allIndex;
			inf->fieldType = info.fieldType;
			inf->pos = -1;

			switch (info.fieldType) {
			case FT_SFNode:
				inf->new_node = bt_SFNode(parser, NULL, n);
				if (parser->last_error) goto err;
				if (!BTCheckNDT(parser, &info, inf->new_node, n)) goto err;
				inf->field_ptr = &inf->new_node;
				break;
			case FT_MFNode:
				if (!bt_check_code(parser, '[')) {
					BTLogErr1(parser, "[ expected");
					goto err;
				}
				inf->node_list = NewChain();
				info.far_ptr = inf->field_ptr = &inf->node_list;
				while (!bt_check_code(parser, ']')) {
					newnode = bt_SFNode(parser, NULL, n);
					if (parser->last_error!=M4OK) goto err;
					if (!BTCheckNDT(parser, &info, newnode, n)) goto err;
					ChainAddEntry(inf->node_list, newnode);
				}
				break;
			default:
				info.far_ptr = inf->field_ptr = SG_NewFieldPointer(inf->fieldType);
				if (SG_IsSFField(info.fieldType)) {
					bt_sffield(parser, &info, n);
				} else {
					bt_mffield(parser, &info, n);
				}
				if (parser->last_error) goto err;
				break;
			}
		}
		return ChainAddEntry(cmdList, com);
	}

	/*MultipleIndexReplace commands*/
	if (!stricmp(str, "MULTIPLEINDREPLACE")) {
		str = bt_get_next(parser, 1);
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "Unknown node", str);
			return parser->last_error = M4BadParam;
		}
		if (!bt_check_code(parser, '.')) {
			BTLogErr1(parser, ". expected");
			return parser->last_error = M4BadParam;
		}
		str = bt_get_next(parser, 0);
		parser->last_error = Node_GetFieldByName(n, str, &info);
		if (parser->last_error) {
			BTLogErr2(parser, "Unknown field", info.name);
			return parser->last_error = M4BadParam;
		}
		if (SG_IsSFField(info.fieldType)) {
			BTLogErr1(parser, "Only MF field allowed");
			return parser->last_error = M4BadParam;
		}
		if (!bt_check_code(parser, '[')) {
			BTLogErr1(parser, "[ expected");
			return parser->last_error = M4BadParam;
		}

		com = SG_NewCommand(SG_MultipleIndexedReplace);
		bd_set_com_node(com, n);
		info.fieldType = SG_GetSFType(info.fieldType);

		while (!bt_check_code(parser, ']')) {
			u32 pos;
			if (bt_parse_int(parser, "position", &pos)) goto err;
			str = bt_get_next(parser, 0);
			if (stricmp(str, "BY")) {
				BTLogErr1(parser, "BY expected");
				goto err;
			}
			inf = SG_NewFieldCommand(com);
			inf->fieldIndex = info.allIndex;
			inf->fieldType = info.fieldType;
			inf->pos = pos;
			if (inf->fieldType==FT_SFNode) {
				info.far_ptr = inf->field_ptr = &inf->new_node;
				inf->new_node = bt_SFNode(parser, NULL, n);
				if (parser->last_error) goto err;
				if (!BTCheckNDT(parser, &info, inf->new_node, n)) goto err;
				inf->field_ptr = &inf->new_node;
			} else {
				info.far_ptr = inf->field_ptr = SG_NewFieldPointer(inf->fieldType);
				bt_sffield(parser, &info, n);
				if (parser->last_error) goto err;
			}
		}
		return ChainAddEntry(cmdList, com);
	}

	if (!stricmp(str, "XDELETE")) {
		str = bt_get_next(parser, 1);
		n = bt_peek_node(parser, str);
		if (!n) {
			BTLogErr2(parser, "Unknown Node", str);
			return parser->last_error = M4BadParam;
		}
		com = SG_NewCommand(SG_NodeDeleteEx);
		bd_set_com_node(com, n);
		return ChainAddEntry(cmdList, com);
	}

	if (!stricmp(str, "INSERTPROTO")) {
		if (!bt_check_code(parser, '[')) {
			BTLogErr1(parser, "[ expected");
			return parser->last_error = M4BadParam;
		}
		com = SG_NewCommand(SG_ProtoInsert);
		while (!bt_check_code(parser, ']')) {
			parser->last_error = bt_parse_proto(parser, NULL, com->new_proto_list);
			if (parser->last_error) goto err;
		}
		ChainAddEntry(cmdList, com);
		return M4OK;
	}
	if (!stricmp(str, "DELETEPROTO")) {
		if (!bt_check_code(parser, '[')) {
			com = SG_NewCommand(SG_ProtoDeleteAll);
			str = bt_get_next(parser, 0);
			if (stricmp(str, "ALL")) {
				BTLogErr1(parser, "ALL expected");
				goto err;
			}
			return ChainAddEntry(cmdList, com);
		}
		com = SG_NewCommand(SG_ProtoDelete);
		while (!bt_check_code(parser, ']')) {
			LPPROTO proto;
			str = bt_get_next(parser, 0);
			proto = SG_FindProto(parser->scene_graph, 0, str);
			if (!proto) {
				BTLogErr2(parser, "Unknown proto", str);
				goto err;
			}
			com->del_proto_list = realloc(com->del_proto_list, sizeof(u32)*(com->del_proto_list_size+1));
			com->del_proto_list[com->del_proto_list_size] = proto->ID;
			com->del_proto_list_size++;
		}
		ChainAddEntry(cmdList, com);
		return M4OK;
	}

	BTLogErr1(parser, "Unknown command syntax");
	return parser->last_error = M4BadParam;

err:
	if (com) SG_DeleteCommand(com);
	return parser->last_error;
}

Descriptor *bt_parse_descriptor(BTParser *parser, char *name)
{
	char *str, field[500];
	Descriptor *desc, *subdesc;
	u32 type;
	u8 tag;
	if (name) {
		str = name;
	} else {
		str = bt_get_next(parser, 0);
	}
	tag = OD_GetDescriptorTag(str);
	if (!tag) {
		BTLogErr2(parser, "Unknown descriptor", str);
		return NULL;
	}
	desc = OD_NewDescriptor(tag);

	if (!desc) return NULL;
	if (!bt_check_code(parser, '{')) return desc;

	while (1) {
		/*done*/
		if (bt_check_code(parser, '}')) break;
		str = bt_get_next(parser, 0);
		strcpy(field, str);
		type = OD_DescriptorFieldType(desc, str);
		switch (type) {
		case 2:
			if (bt_check_code(parser, '[')) {
				while (!bt_check_code(parser, ']')) {
					subdesc = bt_parse_descriptor(parser, NULL);
					if (!subdesc) {
						OD_DeleteDescriptor(&desc);
						parser->last_error = M4BadParam;
						return NULL;
					}
					OD_AddDescToDesc(desc, subdesc);
				}
			}
			break;
		case 1:
			str = bt_get_next(parser, 0);
			subdesc = bt_parse_descriptor(parser, str);
			if (!subdesc) {
				sprintf(parser->szMsg, "Error line %d: unknown desc %s in field %s\n", parser->line, str, field);
				BT_REPORT_ERR(parser);
				OD_DeleteDescriptor(&desc);
				parser->last_error = M4BadParam;
				return NULL;
			}
			OD_AddDescToDesc(desc, subdesc);
			break;
		default:
			str = bt_get_next(parser, 0);
			parser->last_error = OD_SetDescriptorField(desc, field, str);

			if (parser->last_error) {
				sprintf(parser->szMsg, "Error line %d: unknown desc %s in field %s\n", parser->line, str, field);
				BT_REPORT_ERR(parser);
				OD_DeleteDescriptor(&desc);
				parser->last_error = M4BadParam;
				return NULL;
			}
			break;
		}
	}
	if (desc->tag == BIFSConfig_Tag) {
		u32 w, h, isPM;
		BIFSConfigDescriptor *bcfg = (BIFSConfigDescriptor *)desc;
		isPM = SG_GetSizeInfo(parser->scene_graph, &w, &h);
		if (!isPM || !w || !h) SG_SetSizeInfo(parser->scene_graph, bcfg->pixelWidth, bcfg->pixelHeight, bcfg->pixelMetrics);
		/*for bt->xmt*/
		if (!bcfg->isCommandStream) bcfg->isCommandStream = 1;
		if (!bcfg->version) bcfg->version = 1;
	}
	else if (desc->tag==ESDescriptor_Tag) {
		ESDescriptor *esd  =(ESDescriptor*)desc;
		if (esd->decoderConfig) {
			M4StreamContext *sc = M4SM_NewStream(parser->ctx, esd->ESID, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication);
			/*set default timescale for systems tracks (ignored for other)*/
			if (sc) sc->timeScale = esd->slConfig ? esd->slConfig->timestampResolution : 1000;
			if (!parser->base_bifs_id && (esd->decoderConfig->streamType==M4ST_BIFS)) parser->base_bifs_id = esd->ESID;
			if (!parser->base_od_id && (esd->decoderConfig->streamType==M4ST_OD)) parser->base_od_id = esd->ESID;
		}
	}
	return desc;
}

void bt_parse_od_command(BTParser *parser, char *name)
{
	u32 val;
	char *str;
	Descriptor *desc;
	
	if (!stricmp(name, "UPDATE")) {
		str = bt_get_next(parser, 0);
		/*OD update*/
		if (!stricmp(str, "OD")) {
			ObjectDescriptorUpdate *odU;
			if (!bt_check_code(parser, '[')) {
				BTLogErr1(parser, "[ expected");
				parser->last_error = M4BadParam;
 				return;
			}
			odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
			ChainAddEntry(parser->od_au->commands, odU);
			while (!parser->done) {
				str = bt_get_next(parser, 0);
				if (bt_check_code(parser, ']')) break;
				if (stricmp(str, "ObjectDescriptor") && stricmp(str, "InitialObjectDescriptor")) {
					BTLogErr1(parser, "Object Descriptor expected");
					parser->last_error = M4BadParam;
					break;
				}
				desc = bt_parse_descriptor(parser, str);
				if (!desc) break;
				ChainAddEntry(odU->objectDescriptors, desc);
			}
			return;
		}
		/*ESD update*/
		if (!stricmp(str, "ESD")) {
			ESDescriptorUpdate *esdU;
			str = bt_get_next(parser, 0);
			if (stricmp(str, "in")) {
				BTLogErr1(parser, "in expected");
				parser->last_error = M4BadParam;
 				return;
			}
			esdU = (ESDescriptorUpdate *) OD_NewCommand(ESDUpdate_Tag);
			parser->last_error = bt_parse_int(parser, "OD_ID", &val);
			if (parser->last_error) return;
			esdU->ODID = val;
			ChainAddEntry(parser->od_au->commands, esdU);

			if (!bt_check_code(parser, '[')) {
				str = bt_get_next(parser, 0);
				if (stricmp(str, "sDescr")) {
					BTLogErr1(parser, "esDescr expected");
					parser->last_error = M4BadParam;
 					return;
				}
				if (!bt_check_code(parser, '[')) {
					BTLogErr1(parser, "[ expected");
					parser->last_error = M4BadParam;
 					return;
				}
			}

			while (!parser->done) {
				str = bt_get_next(parser, 0);
				if (bt_check_code(parser, ']')) break;
				if (stricmp(str, "ESDescriptor")) {
					BTLogErr1(parser, "ESDescriptor expected");
					parser->last_error = M4BadParam;
					break;
				}
				desc = bt_parse_descriptor(parser, str);
				if (!desc) break;
				ChainAddEntry(esdU->ESDescriptors, desc);
			}
			return;
		}
		BTLogErr2(parser, "unknown OD command", str);
		return;
	}
	if (!stricmp(name, "REMOVE")) {
		str = bt_get_next(parser, 0);
		/*OD remove*/
		if (!stricmp(str, "OD")) {
			ObjectDescriptorRemove *odR;
			if (!bt_check_code(parser, '[')) {
				BTLogErr1(parser, "[ expected");
				parser->last_error = M4BadParam;
 				return;
			}
			odR = (ObjectDescriptorRemove *) OD_NewCommand(ODRemove_Tag);
			ChainAddEntry(parser->od_au->commands, odR);
			while (!parser->done) {
				u32 id;
				if (bt_check_code(parser, ']')) break;
				bt_parse_int(parser, "ODID", &id);
				if (parser->last_error) return;
				odR->OD_ID = realloc(odR->OD_ID, sizeof(u16) * (odR->NbODs+1));
				odR->OD_ID[odR->NbODs] = id;
				odR->NbODs++;
			}
			return;
		}
		/*ESD remove*/
		if (!stricmp(str, "ESD")) {
			u32 odid;
			ESDescriptorRemove *esdR;
			str = bt_get_next(parser, 0);
			if (stricmp(str, "FROM")) {
				BTLogErr1(parser, "FROM expected");
				parser->last_error = M4BadParam;
 				return;
			}
			bt_parse_int(parser, "ODID", &odid);
			if (parser->last_error) return;

			if (!bt_check_code(parser, '[')) {
				BTLogErr1(parser, "[ expected");
				parser->last_error = M4BadParam;
 				return;
			}
			esdR = (ESDescriptorRemove *) OD_NewCommand(ESDRemove_Tag);
			esdR->ODID = odid;
			ChainAddEntry(parser->od_au->commands, esdR);
			while (!parser->done) {
				u32 id;
				if (bt_check_code(parser, ']')) break;
				bt_parse_int(parser, "ES_ID", &id);
				if (parser->last_error) return;
				esdR->ES_ID = realloc(esdR->ES_ID, sizeof(u16) * (esdR->NbESDs+1));
				esdR->ES_ID[esdR->NbESDs] = id;
				esdR->NbESDs++;
			}
			return;
		}
		BTLogErr2(parser, "unknown OD command", str);
		return;
	}
}


M4Err M4SM_LoadContextFromBT(M4SceneManager *ctx, FILE *input, void (*BTNodeInit)(void *cbk, SFNode *n), void (*BTOnError)(void *cbk, char *msg), void *bt_cbck)
{
	u32 pos;
	char *str;
	BTParser parser;
	SFNode *node;
	SGCommand *com;
	Bool in_com;
	LPROUTE r;
	Bool has_id;
	unsigned char BOM[4];
	char szDEFName[1000];

	memset(&parser, 0, sizeof(BTParser));
	parser.input = input;
	parser.ctx = ctx;

	pos = ftell(input);
	BOM[0] = fgetc(input);
	BOM[1] = fgetc(input);
	BOM[2] = fgetc(input);
	BOM[3] = fgetc(input);
	fseek(input, pos, SEEK_SET);

	/*0: no unicode, 1: UTF-16BE, 2: UTF-16LE*/
	if ((BOM[0]==0xFF) && (BOM[1]==0xFE)) {
		if (!BOM[2] && !BOM[3]) {
			fprintf(stdout, "UTF-32 Text Files not supported\n");
			return M4NotSupported;
		} else {
			parser.unicode_type = 2;
			fseek(input, 2, SEEK_CUR);
		}
	} else if ((BOM[0]==0xFE) && (BOM[1]==0xFF)) {
		if (!BOM[2] && !BOM[3]) {
			fprintf(stdout, "UTF-32 Text Files not supported\n");
			return M4NotSupported;
		} else {
			parser.unicode_type = 1;
			fseek(input, 2, SEEK_CUR);
		}
	} else if ((BOM[0]==0xEF) && (BOM[1]==0xBB) && (BOM[2]==0xBF)) {
		/*we handle UTF8 as asci*/
		parser.unicode_type = 0;
		fseek(input, 3, SEEK_CUR);
	}

	parser.unresolved_routes = NewChain();
	parser.inserted_routes = NewChain();
	parser.undef_nodes = NewChain();
	parser.def_nodes = NewChain();
	parser.BTNodeInit = BTNodeInit;
	parser.BTOnError = BTOnError;
	parser.bt_cbck = bt_cbck;

	/*create at least one empty BIFS stream*/
	parser.bifs_es = M4SM_NewStream(ctx, 0, M4ST_BIFS, 0);
	parser.bifs_au = M4SM_NewAU(parser.bifs_es, 0, 0, 1);

	/*default scene replace - we create it no matter what since it is used to store BIFS config
	when parsing IOD.*/
	com = SG_NewCommand(SG_SceneReplace);
	parser.scene_graph = com->graph = NewSceneGraph(BTNodeInit, bt_cbck, NULL, NULL);
	ChainAddEntry(parser.bifs_au->commands, com);
	parser.bifs_es->scene_context = parser.scene_graph;


	has_id = 0;
	in_com = 0;
	/*parse all top-level items*/
	while (!parser.last_error) {
		str = bt_get_next(&parser, 0);
		if (parser.done) break;
		/*IOD*/
		if (!stricmp(str, "InitialObjectDescriptor") || !stricmp(str, "ObjectDescriptor")) {
			parser.ctx->root_od = (ObjectDescriptor *) bt_parse_descriptor(&parser, str);
		}
		/*explicit command*/
		else if (!stricmp(str, "AT") || !stricmp(str, "RAP")) {
			parser.au_is_rap = 0;
			if (!stricmp(str, "RAP")) {
				parser.au_is_rap = 1;
				str = bt_get_next(&parser, 0);
				if (stricmp(str, "AT")) {
					BTLogErr1(&parser, "\'AT\' expected");
					parser.last_error = M4BadParam;
					break;
				}
			}
			str = bt_get_next(&parser, 0);
			if (str[0] == 'D') {
				parser.au_time += atoi(&str[1]);
			} else {
				u32 i;
				for (i=0; i<strlen(str); i++) {
					if (!isdigit(str[i]) && (str[i] != 'E') && (str[i] != 'e') && (str[i] != '-')) {
						BTLogErr2(&parser, "Number expected", str);
						break;
					}
				}
				if (!i) {
					BTLogErr2(&parser, "Number expected", str);
					break;
				}
				parser.au_time = atoi(str);
			}
			if (parser.last_error) break;
			/*reset all contexts*/
			if (parser.od_au && (parser.od_au->timing != parser.au_time)) parser.od_au = NULL;
			if (parser.bifs_au && (parser.bifs_au->timing != parser.au_time)) {
				BT_CheckUnresolvedNodes(&parser);
				parser.bifs_au = NULL;
			}

			parser.stream_id = 0;
			/*fix for mp4tool bt which doesn't support RAP signaling: assume the first AU
			is always RAP*/
			if (!parser.au_time) parser.au_is_rap = 1;
	
			in_com = 1;

			if (!bt_check_code(&parser, '{')) {
				str = bt_get_next(&parser, 0);
				if (!stricmp(str, "IN")) {
					bt_parse_int(&parser, "IN", &parser.stream_id);
					if (parser.last_error) break;
				}
				if (!bt_check_code(&parser, '{')) {
					BTLogErr1(&parser, "{ expected");
					parser.last_error = M4BadParam;
				}
			}
		}
		else if (!strcmp(str, "PROTO") || !strcmp(str, "EXTERNPROTO")) {
			bt_parse_proto(&parser, str, NULL);
			/*reset nodeID and routeID after each proto (!= namespaces)*/
			if (parser.bifs_es->max_node_id<parser.bifs_es->next_node_id) parser.bifs_es->max_node_id = parser.bifs_es->next_node_id;
			parser.bifs_es->next_node_id = 0;
			if (parser.bifs_es->max_route_id < parser.bifs_es->next_route_id) parser.bifs_es->max_route_id = parser.bifs_es->next_route_id;
			parser.bifs_es->next_route_id = 0;
		}
		/*compatibility for old bt (mp4tool) in ProtoLibs*/
		else if (!strcmp(str, "NULL")) {
		}
		else if (!strcmp(str, "DEF")) {
			str = bt_get_next(&parser, 0);
			strcpy(szDEFName, str);
			has_id = 1;
		}
		/*implicit BIFS command on SFTopNodes only*/
		else if (!strcmp(str, "OrderedGroup") 
			|| !strcmp(str, "Group") 
			|| !strcmp(str, "Layer2D") 
			|| !strcmp(str, "Layer3D")
			 /* SVG Top node */
			|| !strcmp(str,"SVG_Svg"))
		{

			node = bt_SFNode(&parser, str, NULL);
			if (!node) break;
			if (has_id) {
				u32 ID = bt_GetDEFID(&parser, szDEFName);
				has_id = 0;
				Node_SetDEF(node, ID, szDEFName);
			}
			SG_SetRootNode(parser.scene_graph, node);
		} 
		else if (!stricmp(str, "ROUTE")) {
			r = bt_parse_route(&parser, 1, NULL);
			if (r && has_id) {
				u32 ID = bt_get_route(&parser, szDEFName);
				has_id = 0;
				if (ID) {
					Route_SetID(r, ID);
				} else {
					Route_SetID(r, 1 + parser.bifs_es->next_route_id);
					parser.bifs_es->next_route_id++;
				}
				Route_SetName(r, szDEFName);
			}
		}
		/*OD commands*/
		else if (!stricmp(str, "UPDATE") || !stricmp(str, "REMOVE")) {
			if (!parser.stream_id || parser.stream_id==parser.bifs_es->ESID) parser.stream_id = parser.base_od_id;

			if (parser.od_es && (parser.od_es->ESID != parser.stream_id)) {
				M4StreamContext *prev = parser.od_es;
				parser.od_es = M4SM_NewStream(ctx, (u16) parser.stream_id, M4ST_OD, 0);
				/*force new AU if stream changed*/
				if (parser.od_es != prev) parser.bifs_au = NULL;
			}
			if (!parser.od_es) parser.od_es = M4SM_NewStream(ctx, (u16) parser.stream_id, M4ST_OD, 0);
			if (!parser.od_au) parser.od_au = M4SM_NewAU(parser.od_es, parser.au_time, 0, parser.au_is_rap);
			/*also store scene context for special import*/
			if (!parser.od_es->scene_context) parser.od_es->scene_context = parser.scene_graph;
			bt_parse_od_command(&parser, str);
		}
		/*BIFS commands*/
		else if (!stricmp(str, "REPLACE") || !stricmp(str, "INSERT") || !stricmp(str, "APPEND") || !stricmp(str, "DELETE")
			/*BIFS extended commands*/
			|| !stricmp(str, "GLOBALQP") || !stricmp(str, "MULTIPLEREPLACE") || !stricmp(str, "MULTIPLEINDREPLACE") || !stricmp(str, "XDELETE") || !stricmp(str, "DELETEPROTO") || !stricmp(str, "INSERTPROTO") ) {

			if (!parser.stream_id) parser.stream_id = parser.base_bifs_id;
			if (!parser.stream_id || (parser.od_es && (parser.stream_id==parser.od_es->ESID)) ) parser.stream_id = parser.base_bifs_id;

			if (parser.bifs_es->ESID != parser.stream_id) {
				M4StreamContext *prev = parser.bifs_es;
				parser.bifs_es = M4SM_NewStream(ctx, (u16) parser.stream_id, M4ST_BIFS, 0);
				/*force new AU if stream changed*/
				if (parser.bifs_es != prev) {
					BT_CheckUnresolvedNodes(&parser);
					parser.bifs_au = NULL;
				}
			}
			if (!parser.bifs_au) parser.bifs_au = M4SM_NewAU(parser.bifs_es, parser.au_time, 0, parser.au_is_rap);
			if (!parser.bifs_es->scene_context) parser.bifs_es->scene_context = parser.scene_graph;
			bt_parse_bifs_command(&parser, str, parser.bifs_au->commands);
		}
		/*if in command, check command end*/
		else {
			/*check command end*/
			if (in_com && bt_check_code(&parser, '}')) in_com = 0;
			else if (strlen(str)) {
				BTLogErr2(&parser, "Unknown top-level element", str);
				parser.last_error = M4BadParam;
			}
			parser.au_is_rap = 0;
		}
	}
	bt_resolve_routes(&parser);
	DeleteChain(parser.unresolved_routes);
	DeleteChain(parser.inserted_routes);
	BT_CheckUnresolvedNodes(&parser);
	DeleteChain(parser.undef_nodes);
	DeleteChain(parser.def_nodes);

	fprintf(stdout, "Parsing done\n");
	return parser.last_error;
}
