/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Authoring Tools sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_author.h>

//video tools
#define VO_START_CODE					0x00
#define VOL_START_CODE					0x20
#define VOP_START_CODE					0xB6
#define VOS_START_CODE					0xB0
#define GOV_START_CODE					0xB3
#define UDTA_START_CODE					0xB2

typedef struct _tagM4V_Parser
{
	BitStream *bs;
	u32 current_object_type;
	u32 current_object_start;
	u32 tc_dec, prev_tc_dec, tc_disp, prev_tc_disp;
} M4V_Parser;


M4V_Parser *NewMPEG4VideoParser(unsigned char *data, u32 data_size)
{
	M4V_Parser *tmp;
	if (!data || !data_size) return NULL;
	tmp = malloc(sizeof(M4V_Parser));
	memset(tmp, 0, sizeof(M4V_Parser));
	tmp->bs = NewBitStream(data, data_size, BS_READ);
	return tmp;
}

void DeleteMPEG4VideoParser(M4V_Parser *m4v)
{
	DeleteBitStream(m4v->bs);
	free(m4v);
}


s32 M4V_LoadObject(M4V_Parser *m4v)
{
	Bool BS_CheckVideoStartCode(BitStream *bs);
	if (!m4v) return 0;

	while (BS_Available(m4v->bs)) {
		if (BS_CheckVideoStartCode(m4v->bs)) {
			m4v->current_object_start = (u32) BS_GetPosition(m4v->bs);
			BS_SkipBytes(m4v->bs, 3);
			m4v->current_object_type = BS_ReadInt(m4v->bs, 8);
			return (s32) m4v->current_object_type;
		}
		BS_SkipBytes(m4v->bs, 1);
	}
	return -1;
}

const char *MP4T_VideoProfileName(u8 video_pl)
{
	switch (video_pl) {
	case 0x00: return "Reserved (0x00) Profile";
	case 0x01: return "Simple Profile @ Level 1";
	case 0x02: return "Simple Profile @ Level 2";
	case 0x03: return "Simple Profile @ Level 3";
	case 0x08: return "Simple Profile @ Level 0";
	case 0x10: return "Simple Scalable Profile @ Level 0";
	case 0x11: return "Simple Scalable Profile @ Level 1";
	case 0x12: return "Simple Scalable Profile @ Level 2";
	case 0x21: return "Core Profile @ Level 1";
	case 0x22: return "Core Profile @ Level 2";
	case 0x32: return "Main Profile @ Level 2";
	case 0x33: return "Main Profile @ Level 3";
	case 0x34: return "Main Profile @ Level 4";
	case 0x42: return "N-bit Profile @ Level 2";
	case 0x51: return "Scalable Texture Profile @ Level 1";
	case 0x61: return "Simple Face Animation Profile @ Level 1";
	case 0x62: return "Simple Face Animation Profile @ Level 2";
	case 0x63: return "Simple FBA Profile @ Level 1";
	case 0x64: return "Simple FBA Profile @ Level 2";
	case 0x71: return "Basic Animated Texture Profile @ Level 1";
	case 0x72: return "Basic Animated Texture Profile @ Level 2";
	case 0x81: return "Hybrid Profile @ Level 1";
	case 0x82: return "Hybrid Profile @ Level 2";
	case 0x91: return "Advanced Real Time Simple Profile @ Level 1";
	case 0x92: return "Advanced Real Time Simple Profile @ Level 2";
	case 0x93: return "Advanced Real Time Simple Profile @ Level 3";
	case 0x94: return "Advanced Real Time Simple Profile @ Level 4";
	case 0xA1: return "Core Scalable Profile @ Level1";
	case 0xA2: return "Core Scalable Profile @ Level2";
	case 0xA3: return "Core Scalable Profile @ Level3";
	case 0xB1: return "Advanced Coding Efficiency Profile @ Level 1";
	case 0xB2: return "Advanced Coding Efficiency Profile @ Level 2";
	case 0xB3: return "Advanced Coding Efficiency Profile @ Level 3";
	case 0xB4: return "Advanced Coding Efficiency Profile @ Level 4";
	case 0xC1: return "Advanced Core Profile @ Level 1";
	case 0xC2: return "Advanced Core Profile @ Level 2";
	case 0xD1: return "Advanced Scalable Texture @ Level1";
	case 0xD2: return "Advanced Scalable Texture @ Level2";
	case 0xE1: return "Simple Studio Profile @ Level 1";
	case 0xE2: return "Simple Studio Profile @ Level 2";
	case 0xE3: return "Simple Studio Profile @ Level 3";
	case 0xE4: return "Simple Studio Profile @ Level 4";
	case 0xE5: return "Core Studio Profile @ Level 1";
	case 0xE6: return "Core Studio Profile @ Level 2";
	case 0xE7: return "Core Studio Profile @ Level 3";
	case 0xE8: return "Core Studio Profile @ Level 4";
	case 0xF0: return "Advanced Simple Profile @ Level 0";
	case 0xF1: return "Advanced Simple Profile @ Level 1";
	case 0xF2: return "Advanced Simple Profile @ Level 2";
	case 0xF3: return "Advanced Simple Profile @ Level 3";
	case 0xF4: return "Advanced Simple Profile @ Level 4";
	case 0xF5: return "Advanced Simple Profile @ Level 5";
	case 0xF7: return "Advanced Simple Profile @ Level 3b";
	case 0xF8: return "Fine Granularity Scalable Profile @ Level 0";
	case 0xF9: return "Fine Granularity Scalable Profile @ Level 1";
	case 0xFA: return "Fine Granularity Scalable Profile @ Level 2";
	case 0xFB: return "Fine Granularity Scalable Profile @ Level 3";
	case 0xFC: return "Fine Granularity Scalable Profile @ Level 4";
	case 0xFD: return "Fine Granularity Scalable Profile @ Level 5";
	case 0xFF: return "Reserved Profile for escape";
	default: return "ISO Reserved Profile";
	}
}

void M4V_RewritePL(unsigned char *data, u32 dataLen, u8 PL)
{
	u32 pos = 0;
	while (pos+4<dataLen) {
		if (!data[pos] && !data[pos+1] && (data[pos+2]==1) && (data[pos+3]==VOS_START_CODE)) {
			u32 i;
			BitStream *bsr = NewBitStream(data, dataLen, BS_READ);
			BitStream *bsw = NewBitStream(data, dataLen, BS_WRITE);
			for (i=0; i<dataLen; i++) {
				u32 val = BS_ReadInt(bsr, 8);
				if (i==pos+4) {
					BS_WriteInt(bsw, PL, 8);
				} else {
					BS_WriteInt(bsw, val, 8);
				}
			}
			DeleteBitStream(bsr);
			DeleteBitStream(bsw);
			return;
		}
		pos ++;
	}
	/*FIXME - VOS is missing...*/
}

M4Err M4V_ParseConfig(M4V_Parser *m4v, M4VDecoderSpecificInfo *dsi)
{
	s32 o_type;
	u8 go, verid;
	s32 clock_rate;

	if (!m4v || !dsi) return M4BadParam;

	memset(dsi, 0, sizeof(M4VDecoderSpecificInfo));

	go = 1;
	while (go) {
		o_type = M4V_LoadObject(m4v);
		switch (o_type) {
		//vosh
		case VOS_START_CODE:
			dsi->VideoPL = (u8) BS_ReadInt(m4v->bs, 8);
			break;
	
		case VOL_START_CODE:
			verid = 0;
			dsi->RAP_stream = BS_ReadInt(m4v->bs, 1);
			dsi->objectType = BS_ReadInt(m4v->bs, 8);
			if (BS_ReadInt(m4v->bs, 1)) {
				verid = BS_ReadInt(m4v->bs, 4);
				BS_ReadInt(m4v->bs, 3);
			}
			if (BS_ReadInt(m4v->bs, 4) == 0xF) {
//				dsi->ratio_width = 
				BS_ReadInt(m4v->bs, 8);
//				dsi->ratio_height = 
				BS_ReadInt(m4v->bs, 8);
			}
			if (BS_ReadInt(m4v->bs, 1)) {
				BS_ReadInt(m4v->bs, 3);
				if (BS_ReadInt(m4v->bs, 1)) BS_ReadInt(m4v->bs, 79);
			}
			dsi->has_shape = BS_ReadInt(m4v->bs, 2);
			if (dsi->has_shape && (verid!=1) ) BS_ReadInt(m4v->bs, 4);
			BS_ReadInt(m4v->bs, 1);
			//clock rate
			dsi->clock_rate = BS_ReadInt(m4v->bs, 16);
			//marker
			BS_ReadInt(m4v->bs, 1);

			clock_rate = dsi->clock_rate-1;
			if (clock_rate >= 65536) clock_rate = 65535;
			if (clock_rate > 0) {
				for (dsi->NumBitsTimeIncrement = 1; dsi->NumBitsTimeIncrement < 16; dsi->NumBitsTimeIncrement++)	{	
					if (clock_rate == 1) break;
					clock_rate = (clock_rate >> 1);
				}
			} else {
				//fix from vivien for divX
				dsi->NumBitsTimeIncrement = 1;
			}			
			//fixed FPS stream
			dsi->time_increment = 0;
			if (BS_ReadInt(m4v->bs, 1)) {
				dsi->time_increment = BS_ReadInt(m4v->bs, dsi->NumBitsTimeIncrement);
			}
			if (!dsi->has_shape) {
				BS_ReadInt(m4v->bs, 1);
				dsi->width = BS_ReadInt(m4v->bs, 13);
				BS_ReadInt(m4v->bs, 1);
				dsi->height = BS_ReadInt(m4v->bs, 13);
			} else {
				dsi->width = dsi->height = 0;
			}
			//shape will be done later
			BS_Align(m4v->bs);
			break;

		case VOP_START_CODE:
		case GOV_START_CODE:
			go = 0;
			break;
		/*EOS*/
		case -1:
			go = 0;
			m4v->current_object_start = (u32) BS_GetPosition(m4v->bs);
			break;
		//don't interest us
		case UDTA_START_CODE:
		default:
			break;
		}
	}
	return M4OK;
}

M4Err M4V_Reset(M4V_Parser *m4v, u32 start)
{
	BS_Seek(m4v->bs, start);
	m4v->current_object_start = start;
	m4v->current_object_type = 0;
	return M4OK;
}

M4Err M4V_GetFrame(M4V_Parser *m4v, M4VDecoderSpecificInfo dsi, u8 *frame_type, u32 *time_inc, u32 *size, u32 *start, Bool *is_coded)
{
	u8 go, hasVOP, firstObj, secs;
	s32 o_type;
	u32 vop_inc = 0;

	if (!m4v || !size || !start || !frame_type) return M4BadParam;

	*size = 0;
	firstObj = 1;
	hasVOP = 0;
	*is_coded = 0;
	m4v->current_object_type = -1;
	
	M4V_Reset(m4v, m4v->current_object_start);
	go = 1;
	while (go) {
		o_type = M4V_LoadObject(m4v);
		switch (o_type) {
		case VOP_START_CODE:
			//done
			if (hasVOP) {
				go = 0;
				break;
			}
			if (firstObj) {
				*start = m4v->current_object_start;
				firstObj = 0;
			}
			hasVOP = 1;

			/*coding type*/
			*frame_type = BS_ReadInt(m4v->bs, 2);
			/*modulo time base*/
			secs = 0;
			while (BS_ReadInt(m4v->bs, 1) != 0)
				secs ++;
			/*no support for B frames in parsing*/
			secs += (dsi.enh_layer || *frame_type!=2) ? m4v->tc_dec : m4v->tc_disp;
			/*marker*/
			BS_ReadInt(m4v->bs, 1);
			/*vop_time_inc*/
			if (dsi.NumBitsTimeIncrement)
				vop_inc = BS_ReadInt(m4v->bs, dsi.NumBitsTimeIncrement);
			
			m4v->prev_tc_dec = m4v->tc_dec;
			m4v->prev_tc_disp = m4v->tc_disp;
			if (dsi.enh_layer || *frame_type!=2) {
				m4v->tc_disp = m4v->tc_dec;
				m4v->tc_dec = secs;
			}
			*time_inc = secs * dsi.clock_rate + vop_inc;
			/*marker*/
			BS_ReadInt(m4v->bs, 1);
			/*coded*/
			*is_coded = BS_ReadInt(m4v->bs, 1);
			BS_Align(m4v->bs);
			break;
		case GOV_START_CODE:
			if (firstObj) {
				*start = m4v->current_object_start;
				firstObj = 0;
			}
			if (hasVOP) go = 0;
			break;

		case VO_START_CODE:
		case VOL_START_CODE:
		default:
			break;

		case -1:
			*size = (u32) BS_GetPosition(m4v->bs) - *start;
			return M4EOF;
		}
	}
	*size = m4v->current_object_start - *start;
	return M4OK;
}


u32 M4V_GetObjectStartPos(LPM4VPARSER m4v)
{
	return m4v->current_object_start;
}

Bool M4V_IsValidObjectType(LPM4VPARSER m4v)
{
	return ((s32) m4v->current_object_type==-1) ? 0 : 1;
}


M4Err M4V_GetConfig(char *rawdsi, u32 rawdsi_size, M4VDecoderSpecificInfo *dsi)
{
	M4Err e;
	LPM4VPARSER vparse;
	if (!rawdsi || !rawdsi_size) return M4NonCompliantBitStream;
	vparse = NewMPEG4VideoParser(rawdsi, rawdsi_size);
	e = M4V_ParseConfig(vparse, dsi);
	DeleteMPEG4VideoParser(vparse);
	return e;
}

static const char *M4A_ObjectTypesNames[] = {
    "Reserved",
    "AAC Main",
    "AAC LC",
    "AAC SBR",
    "AAC LTP",
    "SBR",
    "AAC Scalable",
    "TwinVQ",
    "CELP", 
    "HVXC",
    "Reserved", 
    "Reserved"
    "TTSI",
    "Main synthetic",
    "Wavetable synthesis",
    "General MIDI",
    "Algorithmic Synthesis and Audio FX",
    "ER AAC LC",
    "Reserved",
    "ER AAC LTP",
    "ER AAC scalable",
    "ER TwinVQ",
    "ER BSAC",
    "ER AAC LD",
    "ER CELP",
    "ER HVXC",
    "ER HILN",
    "ER Parametric",
    "(Reserved)",
    "(Reserved)",
    "(Reserved)",
    "(Reserved)"
};

const char *M4A_GetObjectTypeName(u32 objectType)
{
  if (objectType>=32) return NULL;
  return M4A_ObjectTypesNames[objectType];
}


/*MP3 parsing - credits go to CISCO - MPEG4IP*/
u8 MP3_GetVersion(u32 hdr)
{
	return ((hdr >> 19) & 0x3); 
}

u8 MP3_GetLayer(u32 hdr)
{
	return ((hdr >> 17) & 0x3); 
}

u8 MP3_GetNumChannels(u32 hdr)
{
	if (((hdr >> 6) & 0x3) == 3) return 1;
	return 2;
}


static u16 MP3SamplingRates[4][3] = {
	{ 11025, 12000, 8000 },		/* MPEG-2.5 */
	{ 0, 0, 0 },
	{ 22050, 24000, 16000 },	/* MPEG-2 */
	{ 44100, 48000, 32000 }		/* MPEG-1 */
};

u16 MP3_GetSamplingRate(u32 hdr)
{
	/* extract the necessary fields from the MP3 header */
	u8 version = MP3_GetVersion(hdr);
	u8 sampleRateIndex = (hdr >> 10) & 0x3;
	return MP3SamplingRates[version][sampleRateIndex];
}

u16 MP3_GetSamplesPerFrame(u32 hdr)
{
	u8 version = MP3_GetVersion(hdr);
	u8 layer = MP3_GetLayer(hdr);

	if (layer == 1) {
		if (version == 3) return 1152;
		return 576;
	}
	if (layer == 2) return 1152;
	return 384;
}

u8 MP3_GetObjectTypeIndication(u32 hdr)
{
	switch (MP3_GetVersion(hdr)) {
	case 3:
		return 0x6B;
		break;
	case 2:
	case 0:
		return 0x69;
	default:
		return 0x00;
	}
}
static const char *MP3_Versions[] = 
{
	"MPEG-2.5",
	"Reserved",
	"MPEG-2",
	"MPEG-1"
};

const char *MP3_GetVersionName(u32 hdr)
{
  u32 v = MP3_GetVersion(hdr);
  if (v>3) return NULL;
  return MP3_Versions[v];
}

static u16 MP3BitRates[5][14] = {
	/* MPEG-1, Layer III */
	{ 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320 },
	/* MPEG-1, Layer II */
	{ 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384 },
	/* MPEG-1, Layer I */
	{ 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448 },
	/* MPEG-2 or 2.5, Layer II or III */
	{ 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160 },
	/* MPEG-2 or 2.5, Layer I */
	{ 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256 }
};


u16 MP3_GetFrameSize(u32 hdr)
{
	u32 val;
	u8 bitRateIndex1;
	u8 version = MP3_GetVersion(hdr);
	u8 layer = MP3_GetLayer(hdr);
	u8 bitRateIndex2 = (hdr >> 12) & 0xF;
	u8 sampleRateIndex = (hdr >> 10) & 0x3;
	Bool isPadded = (hdr >> 9) & 0x1;
	u16 frameSize = 0;

	if (version == 3) {
		bitRateIndex1 = layer - 1;
	} else {
		if (layer == 3) {
			bitRateIndex1 = 4;
		} else {
			bitRateIndex1 = 3;
		}
	}

	/* compute frame size */
	val = (MP3SamplingRates[version][sampleRateIndex] << !(version & 1));
	if (!val) return 0;
	frameSize = 144 * 1000 * MP3BitRates[bitRateIndex1][bitRateIndex2-1] / val;

	if (isPadded) {
		if (layer == 3) {
			frameSize += 4;
		} else {
			frameSize++;
		}
	}
	return frameSize;
}

static const u32 m4a_sample_rates[] =
{
    96000, 88200, 64000, 48000, 44100, 32000,
    24000, 22050, 16000, 12000, 11025, 8000
};


u32 M4A_GetCodecType(char *dsi, u32 dsi_size, u32 *nbChannels, u32 *sampleRate)
{
	u32 audioType, srIndex;
	BitStream *bs;
	if (!dsi || !dsi_size || (dsi_size<2) ) return 0;
	bs = NewBitStream(dsi, dsi_size, BS_READ);

	audioType = BS_ReadInt(bs, 5);
	srIndex = BS_ReadInt(bs, 4);
	if (srIndex == 0x0F) BS_ReadInt(bs, 24);
	*nbChannels = BS_ReadInt(bs, 4);
	DeleteBitStream(bs);

	*sampleRate = 0;
    if (srIndex < 12) *sampleRate = m4a_sample_rates[srIndex];
	return audioType;
}



u32 MP3_GetNextHeader(FILE* in)
{
	u8 b, state = 0;
	u32 dropped = 0;
	u8 bytes[4];

	while (1) {
		if (fread(&b, 1, 1, in) == 0) return 0;

		if (state==3) {
			bytes[state] = b;
			return FOUR_CHAR_INT(bytes[0], bytes[1], bytes[2], bytes[3]);
		}
		if (state==2) {
			if (((b & 0xF0) == 0) || ((b & 0xF0) == 0xF0) || ((b & 0x0C) == 0x0C)) {
				if (bytes[1] == 0xFF) state = 1; 
				else state = 0; 
			} else {
				bytes[state] = b;
				state = 3;
			}
		}
		if (state==1) {
			if (((b & 0xE0) == 0xE0) && ((b & 0x18) != 0x08) && ((b & 0x06) != 0)) {
				bytes[state] = b;
				state = 2;
			} else {
				state = 0;
			}
		}

		if (state==0) {
			if (b == 0xFF) {
				bytes[state] = b;
				state = 1;
			} else {
				if ((dropped == 0) && ((b & 0xE0) == 0xE0) && ((b & 0x18) != 0x08) && ((b & 0x06) != 0)) {
					bytes[0] = (u8) 0xFF;
					bytes[1] = b;
					state = 2;
				} else {
					dropped++;
				}
			}
		}
	}
	return 0;
}

