/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_esm_dev.h>

#define M4TERM()	M4Client *term = (M4Client *) user_priv; if (!term) return;

static M4INLINE Channel *get_mpeg4_channel(LPNETSERVICE service, LPNETCHANNEL ns)
{
	Channel *ch = (Channel *)ns;
	if (!service || !ch) return NULL;
	if (ch->chan_id != (u32) ch) return NULL;
	if (ch->service != service) return NULL;
	return ch;
}

static void m4_on_message(void *user_priv, LPNETSERVICE service, M4Err error, const char *message)
{
	M4TERM();

	/*check for UDP timeout*/
	if (error==M4UDPTimeOut) {
		char *sOpt = IF_GetKey(term->user->config, "Network", "AutoReconfigUDP");
		if (sOpt && !stricmp(sOpt, "yes")) {
			sOpt = IF_GetKey(term->user->config, "Network", "UDPNotAvailable");
			/*if option is already set don't bother try reconfig*/
			if (!sOpt || stricmp(sOpt, "yes")) {
				char szMsg[1024];
				sprintf(szMsg, "UDP down (%s) - Retrying with TCP", message);
				M4_OnMessage(term, service->url, szMsg, M4OK);
				/*reconnect top-level*/
				sOpt = strdup(term->root_scene->root_od->net_service->url);
				M4T_CloseURL(term);
				IF_SetKey(term->user->config, "Network", "UDPNotAvailable", "yes");
				M4T_ConnectURL(term, sOpt);
				free(sOpt);
				return;
			}
		}
	}
	M4_OnMessage(term, service->url, message, error);
}

static void m4_on_connect(void *user_priv, LPNETSERVICE service, LPNETCHANNEL netch, M4Err err)
{
	Channel *ch;
	ODManager *root;
	M4TERM();

	root = service->owner;
	if (root && (root->net_service != service)) {
		M4_OnMessage(term, service->url, "Incomaptible plugin type", M4ServiceError);
		return;
	}
	/*this is service connection*/
	if (!netch) {
		if (err) {
			char msg[5000];
			sprintf(msg, "Cannot open %s", service->url);
			M4_OnMessage(term, service->url, msg, err);

			/*destroy service only if attached*/
			if (root) {
				Term_LockNet(term, 1);
				ChainDeleteItem(term->net_services, service);
				if (root) {
					/*take care of remote ODs*/
					while (root->parent_OD) root = root->parent_OD;
					ODM_RemoveOD(root);
				}
				Term_LockNet(term, 0);
				return;
			}
		}

		if (!root) {
			/*channel service connect*/
			u32 i;
			Chain *ODs = NewChain();;
			Term_LockNet(term, 1);
			for (i=0; i<ChainGetCount(term->channels_pending); i++) {
				ChannelSetup *cs = ChainGetEntry(term->channels_pending, i);
				if (cs->ch->service != service) continue;
				ChainDeleteEntry(term->channels_pending, i);
				i--;
				/*even if error do setup (channel needs to be deleted)*/
				if (ODM_SetupChannel(cs->ch, cs->dec, err) == M4OK) {
					if (cs->ch->odm && (ChainFindEntry(ODs, cs->ch->odm)==-1) ) ChainAddEntry(ODs, cs->ch->odm);
				}
				free(cs);
			}
			Term_LockNet(term, 0);
			/*finally setup all ODs concerned (we do this later in case of scalability)*/
			while (ChainGetCount(ODs)) {
				ODManager *odm = ChainGetEntry(ODs, 0);
				ChainDeleteEntry(ODs, 0);
				/*force re-setup*/
				IS_SetupOD(odm->parentscene, odm);
			}
			DeleteChain(ODs);
			return;
		}
		/*setup od*/
		ODM_SetupService(root);

	}

	/*this is channel connection*/
	if (err) {
		M4_OnMessage(term, service->url, "Channel Connection Failed", err);
		return;
	}
	ch = get_mpeg4_channel(service, netch);
	if (!ch) return;

	Term_LockNet(term, 1);
	Channel_OnConnect(ch);
	Term_LockNet(term, 0);

	/*in case the OD user has requested a play send a PLAY on the object (Play request are skiped
	until all channels are connected) */
	if (ch->odm->mo && ch->odm->mo->num_open) {
		ODM_Start(ch->odm);
	}
	/*if this is a channel in the root OD play */
	else if (! ch->odm->parentscene) {
		ODM_Start(ch->odm);
	}
}

static void m4_on_disconnect(void *user_priv, LPNETSERVICE service, LPNETCHANNEL netch, M4Err response)
{
	ODManager *root;
	Channel *ch;
	M4TERM();

	root = service->owner;
	if (root && (root->net_service != service)) {
		M4_OnMessage(term, service->url, "Incomaptible plugin type", M4ServiceError);
		return;
	}
	/*this is service disconnect*/
	if (!netch) {
		Term_LockNet(term, 1);
		/*unregister from valid services*/
		ChainDeleteItem(term->net_services, service);
		/*and queue for destroy*/
		ChainAddEntry(term->net_services_to_remove, service);
		Term_LockNet(term, 0);
		return;
	}
	/*this is channel disconnect*/

	/*no notif in case of failure for disconnection*/
	ch = get_mpeg4_channel(service, netch);
	if (!ch) return;
	/*signal channel state*/
	ch->net_status = NM_Disconnected;
}

static void m4_on_slp_recieved(void *user_priv, LPNETSERVICE service, LPNETCHANNEL netch, char *data, u32 data_size, struct tagSLHeader *hdr, M4Err reception_status)
{
	Channel *ch;
	M4TERM();

	ch = get_mpeg4_channel(service, netch);
	if (!ch) return;
	
	if (reception_status==M4EOF) {
		Channel_EndOfStream(ch);
		return;
	}
	/*otherwise dispatch with error code*/
	Channel_RecieveSLP(service, ch, data, data_size, hdr, reception_status);
}


static void m4_on_command(void *user_priv, LPNETSERVICE service, NetworkCommand *com)
{
	Channel *ch;
	M4TERM();

	if (! com->on_channel) return;

	ch = get_mpeg4_channel(service, com->on_channel);
	if (!ch) return;

	switch (com->command_type) {
	/*SL reconfiguration*/
	case CHAN_RECONFIG:
		Term_LockNet(term, 1);
		Channel_ReconfigSL(ch, &com->sl_config);
		Term_LockNet(term, 0);
		return;
	/*time mapping (TS to media-time)*/
	case CHAN_MAP_TIME:
		ch->SeedTS = com->map_timestamp;
		if (ch->clock && (ch->clock->clockID != ch->esd->ESID))
			ch->ts_offset = (u32) (com->start_range*1000);
		else
			ch->ts_offset = 0;

		Channel_TimeMapped(ch);

		if (ch->odm->codec && !ch->esd->dependsOnESID) {
			ch->odm->codec->SeedTime = (u32) (com->start_range*1000);
		} else if (ch->odm->subscene && ch->odm->subscene->bifs_codec) {
			ch->odm->subscene->bifs_codec->SeedTime = (u32) (com->start_range*1000);
		}
		break;
	/*duration changed*/
	case CHAN_DURATION:
		ODM_SetDuration(ch->odm, ch, (u32) (1000*com->duration));
		break;
	case CHAN_BUFFER_QUERY:
		com->buffer_max = ch->MaxBuffer;
		com->buffer_min = ch->MinBuffer;
		com->buffer_occupancy = ch->BufferTime;
		break;
	default:
		return;
	}
}

void Term_SetupNetwork(M4Client *app)
{
	app->client_sink.on_connect = m4_on_connect;
	app->client_sink.on_disconnect = m4_on_disconnect;
	app->client_sink.on_message = m4_on_message;
	app->client_sink.on_command = m4_on_command;
	app->client_sink.on_slp_recieved = m4_on_slp_recieved;
	app->client_sink.user_priv = app;
}
