/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



/*for OD service types*/
#include <gpac/m4_render.h>
#include <gpac/intern/m4_esm_dev.h>
#include "MediaControl.h"

MediaObject *NewMediaObject();
void MO_UpdateCaps(MediaObject *mo);


/*extern proto fetcher*/
LPSCENEGRAPH IS_GetProtoLib(void *SceneCallback, MFURL *lib_url);
typedef struct
{
	MFURL *url;
	MediaObject *mo;
} ProtoLink;

Double IS_GetSceneTime(void *_is)
{
	Double ret;
	InlineScene *is = (InlineScene *)_is;
	assert(is);
	if (!is->bifs_codec) return 0.0;
	ret = CK_GetTime(is->bifs_codec->ck);
	ret/=1000.0;
	return ret;
}

InlineScene *NewInlineScene(InlineScene *parentScene)
{
	InlineScene *tmp = malloc(sizeof(InlineScene));
	if (! tmp) return NULL;
	memset(tmp, 0, sizeof(InlineScene));

	tmp->ODlist = NewChain();
	tmp->media_objects = NewChain();
	tmp->extern_protos = NewChain();

	/*init inline scene*/
	if (parentScene) {
		tmp->graph = SG_NewSubScene(parentScene->graph);
	} else {
		tmp->graph = NewSceneGraph(M4_NodeInit, tmp, M4_NodeModified, tmp);
	}
	SG_SetPrivate(tmp->graph, tmp);
	SG_SetSceneTime(tmp->graph, IS_GetSceneTime, tmp);
	SG_SetProtoLoader(tmp->graph, IS_GetProtoLib);
	return tmp;
}

void IS_Delete(InlineScene *is)
{
	DeleteChain(is->ODlist);
	
	while (ChainGetCount(is->extern_protos)) {
		ProtoLink *pl = ChainGetEntry(is->extern_protos, 0);
		ChainDeleteEntry(is->extern_protos, 0);
		free(pl);
	}
	DeleteChain(is->extern_protos);

	/*delete the scene graph*/
	SG_Delete(is->graph);

	/*delete all our codecs*/
	if (is->bifs_codec) {
		MM_RemoveCodec(is->root_od->term->mediaman, is->bifs_codec);
		DeleteCodec(is->bifs_codec);
		/*reset pointer to NULL in case nodes try to access scene time*/
		is->bifs_codec = NULL;
	}
	if (is->od_codec) {
		MM_RemoveCodec(is->root_od->term->mediaman, is->od_codec);
		DeleteCodec(is->od_codec);
		is->od_codec = NULL;
	}
	/*don't touch the root_od, will be deleted by the parent scene*/

	/*clean all remaining associations*/
	while (ChainGetCount(is->media_objects)) {
		MediaObject *obj = ChainGetEntry(is->media_objects, 0);
		if (obj->od_man) ((ODManager *)obj->od_man)->mo = NULL;
		ChainDeleteEntry(is->media_objects, 0);
		if (obj->OD_URL) free(obj->OD_URL);
		free(obj);
	}
	DeleteChain(is->media_objects);
	free(is);
}

ODManager *IS_FindODM(InlineScene *is, u16 OD_ID)
{
	u32 i;
	ODManager *odm;
	//browse the OD List only
	for (i=0; i<ChainGetCount(is->ODlist); i++) {
		odm = ChainGetEntry(is->ODlist, i);
		if (odm->OD->objectDescriptorID == OD_ID) return odm;
	}
	return NULL;
}

void IS_Disconnect(InlineScene *is)
{
	ODManager *odm;

	SG_Reset(is->graph);
	is->graph_attached = 0;
	
	while (ChainGetCount(is->ODlist)) {
		odm = ChainGetEntry(is->ODlist, 0);
		ODM_RemoveOD(odm);
	}

	/*remove stream associations*/
	while (ChainGetCount(is->media_objects)) {
		MediaObject *obj = ChainGetEntry(is->media_objects, 0);
		ChainDeleteEntry(is->media_objects, 0);
		if (obj->od_man) ((ODManager*)obj->od_man)->mo = NULL;
		free(obj);
	}
}


void IS_RemoveOD(InlineScene *is, ODManager *odm)
{
	u32 i;
	ODManager *parent;

	for (i=0; i<ChainGetCount(is->ODlist); i++) {
		ODManager *temp = ChainGetEntry(is->ODlist, i);
		if (temp == odm) {
			ChainDeleteEntry(is->ODlist, i);
			break;
		}
	}

	parent = odm;
	while (parent->parent_OD) parent = parent->parent_OD;

	for (i=0; i<ChainGetCount(is->media_objects); i++) {
		MediaObject *obj = ChainGetEntry(is->media_objects, i);
		if (
			/*assigned object*/
			(obj->od_man==odm) || 
			/*remote OD*/
			((obj->OD_ID!=DYNAMIC_OD_ID) && parent->OD && (obj->OD_ID == parent->OD->objectDescriptorID) ) ||
			/*dynamic OD*/
			(obj->OD_URL && parent->OD && parent->OD->URLString && !stricmp(obj->OD_URL, parent->OD->URLString)) 
		) {
			obj->mo_flags = 0;
			if (obj->od_man) ((ODManager*)obj->od_man)->mo = NULL;
			obj->od_man = NULL;
			obj->current_frame = NULL;
			obj->current_size = obj->current_ts = 0;

			/*if graph not attached we can remove the link (this is likely scene shutdown for some error)*/
			if (!is->graph_attached) {
				u32 j;
				for (j=0; j<ChainGetCount(is->extern_protos); j++) {
					ProtoLink *pl = ChainGetEntry(is->extern_protos, j);
					if (pl->mo==obj) {
						pl->mo = NULL;
						break;
					}
				}
				ChainDeleteEntry(is->media_objects, i);
				free(obj);
			}
			/*unregister*/
			if (odm->parent_OD) odm->parent_OD->remote_OD = NULL;
			return;
		}
	}
}

u32 URL_GetODID(MFURL *url)
{
	u32 i, j, tmpid;
	char *str, *s_url;
	u32 id = 0;

	if (!url) return 0;
	
	for (i=0; i<url->count; i++) {
		if (url->vals[i].OD_ID) {
			/*works because OD ID 0 is forbidden in MPEG4*/
			if (!id) {
				id = url->vals[i].OD_ID;
			}
			/*bad url, only one object can be described in MPEG4 urls*/
			else if (id != url->vals[i].OD_ID) return 0;
		} else if (url->vals[i].url && strlen(url->vals[i].url)) {
			/*format: od:ID or od:ID#segment - also check for "ID" in case...*/
			str = url->vals[i].url;
			if (strstr(str, "od:")) str += 3;
			/*remove segment info*/
			s_url = strdup(str);
			j = 0;
			while (j<strlen(s_url)) {
				if (s_url[j]=='#') {
					s_url[j] = 0;
					break;
				}
				j++;
			}
			j = sscanf(s_url, "%d", &tmpid);
			/*be carefull, an url like "11-regression-test.mp4" will return 1 on sscanf :)*/
			if (j==1) {
				char szURL[20];
				sprintf(szURL, "%d", tmpid);
				if (stricmp(szURL, s_url)) j = 0;
			}
			free(s_url);

			if (j!= 1) {
				/*dynamic OD if only one URL specified*/
				if (!i) return DYNAMIC_OD_ID;
				/*otherwise ignore*/
				continue;
			}
			if (!id) {
				id = tmpid;
				continue;
			}
			/*bad url, only one object can be described in MPEG4 urls*/
			else if (id != tmpid) return 0;
		}
	}
	return id;
}


//browse all channels and update buffering info
void IS_UpdateBufferingInfo(InlineScene *is)
{
	u32 i, j, max_buffer, cur_buffer;
	Channel *ch;
	ODManager *odm;
	char message[1024];

	if (!is) return;

	max_buffer = cur_buffer = 0;

	/*get buffering on root OD*/
	for (j=0; j<ChainGetCount(is->root_od->channels); j++) {
		ch = ChainGetEntry(is->root_od->channels, j);
		/*count only re-buffering channels*/
		if (!ch->BufferOn) continue;

		max_buffer += ch->MaxBuffer;
		cur_buffer += (ch->BufferTime>0) ? ch->BufferTime : 1;
	}

	/*get buffering on all ODs*/
	for (i=0; i<ChainGetCount(is->ODlist); i++) {
		odm = ChainGetEntry(is->ODlist, i);

		if (!odm->codec) continue;
		for (j=0; j<ChainGetCount(odm->channels); j++) {
			ch = ChainGetEntry(odm->channels, j);
			/*count only re-buffering channels*/
			if (!ch->BufferOn) continue;

			max_buffer += ch->MaxBuffer;
			cur_buffer += (ch->BufferTime>0) ? ch->BufferTime : 1;
		}
	}

	if (!max_buffer || !cur_buffer || (max_buffer <= cur_buffer)) {
		sprintf(message, "Buffering 100 %c", '%');
	} else {
		Float ft = (Float) (100*cur_buffer);
		ft /= max_buffer;
		sprintf(message, "Buffering %.2f %c", ft, '%');
	}
	//note we signal it as the main service, even if some streams are running under another 
	//service (ESD URL, OD URL)
	M4_OnMessage(is->root_od->term, is->root_od->net_service->url, message, M4OK);
}


void IS_InsertOD(InlineScene *is, char *url)
{
	ODManager *root_od;
	ODManager *odm;
	if (!url || !is) return;

	odm = NewODManager();
	odm->parentscene = is;
	odm->OD = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
	odm->OD->objectDescriptorID = DYNAMIC_OD_ID;
	odm->OD->URLString = strdup(url);
	odm->parentscene = is;
	odm->term = is->root_od->term;
	root_od = is->root_od;
	ChainAddEntry(is->ODlist, odm);
	while (root_od->remote_OD) root_od = root_od->remote_OD;
	ODM_SetupOD(odm, root_od->net_service);
}
#ifdef M4_DEF_Inline

static Bool Inline_SetScene(B_Inline *root)
{
	ODManager *odm;
	MediaObject *mo;
	InlineScene *parent;
	LPSCENEGRAPH graph = Node_GetParentGraph((SFNode *) root);
	parent = SG_GetPrivate(graph);
	if (!parent) return 0;

	mo = IS_GetMediaObject(parent, &root->url, NM_OD_SCENE);
	if (!mo || !mo->od_man) return 0;
	odm = (ODManager *)mo->od_man;
	
	/*we don't handle num_open as with regular ODs since an inline is never "started" by the scene renderer*/
	if (!mo->num_open && !odm->is_open) ODM_Start(odm);
	mo->num_open ++;

	/*handle remote*/
	while (odm->remote_OD) odm = odm->remote_OD;
	if (!odm->subscene) return 0;
	Node_SetPrivate((SFNode *)root, odm->subscene);
	return 1;
}

void Inline_Modified(SFNode *node)
{
	u32 ODID;
	MediaObject *mo;
	B_Inline *pInline = (B_Inline *) node;
	InlineScene *pIS = Node_GetPrivate(node);
	if (!pIS) return;

	mo = (pIS->root_od) ? pIS->root_od->mo : NULL;
	ODID = URL_GetODID(&pInline->url);

	/*disconnect current inline if we're the last one using it (same as regular OD session leave/join)*/
	if (mo) {
		Bool changed = 1;
		if (ODID && (ODID != DYNAMIC_OD_ID)) {
			if (ODID==pIS->root_od->OD->objectDescriptorID) changed = 0;
		} else if (mo->OD_URL && pInline->url.count && pInline->url.vals[0].url) {
			if (!stricmp(mo->OD_URL, pInline->url.vals[0].url)) changed = 0;
		}
		if (mo->num_open) {
			if (!changed) return;
			mo->num_open --;
			if (!mo->num_open) {
				ODM_Stop(pIS->root_od, 1);
				IS_Disconnect(pIS);
			}
		}
	}
	
	if (ODID) Inline_SetScene(pInline);
}


static void IS_CheckMediaRestart(InlineScene *is)
{
	if (!is->needs_restart) ODM_CheckSegmentSwitch(is->root_od);
	if (is->needs_restart) return;

#ifdef M4_DEF_MediaControl
	if (is->root_od->media_ctrl && is->root_od->media_ctrl->control->loop) {
		Clock *ck = ODM_GetMediaClock(is->root_od);
		if (ck->has_seen_eos) {
			u32 now = CK_GetTime(ck);
			u32 dur = is->duration;
			if (is->root_od->media_ctrl->current_seg) {
				/*only process when all segments are played*/
				if (ChainGetCount(is->root_od->media_ctrl->seg) <= is->root_od->media_ctrl->current_seg) {
					is->needs_restart = 1;
					is->root_od->media_ctrl->current_seg = 0;
				}
			}
			else if (dur<now) {
				is->needs_restart = 1;
				is->root_od->media_ctrl->current_seg = 0;
			}
		} else {
			/*trigger render until to watch for restart...*/
			Term_InvalidateScene(is->root_od->term);
		}
	}
#endif
}

void IS_Render(SFNode *n, void *render_stack)
{
	SFNode *root;
	InlineScene *pIS;

	pIS = Node_GetPrivate(n);

	//if no private scene is associated	get the node parent graph, retrieve the IS and find the OD
	if (!pIS) {
		Inline_SetScene((B_Inline *) n);
		pIS = Node_GetPrivate(n);
		if (!pIS) return;
	}

	IS_CheckMediaRestart(pIS);

	/*if we need to restart, shutdown graph and do it*/
	if (pIS->needs_restart) {
		u32 current_seg = 0;
#ifdef M4_DEF_MediaControl
		if (pIS->root_od->media_ctrl) current_seg = pIS->root_od->media_ctrl->current_seg;
#endif
		pIS->needs_restart = 0;
		/*stop main object from playing but don't disconnect channels*/
		ODM_Stop(pIS->root_od, 1);
		/*this will close all ODs inside the scene and reset the graph*/
		IS_Disconnect(pIS);

#ifdef M4_DEF_MediaControl
		if (pIS->root_od->media_ctrl) pIS->root_od->media_ctrl->current_seg = current_seg;
#endif
		/*start*/
		ODM_Start(pIS->root_od);
		return;
	} 
	
	/*if not attached return (attaching the graph cannot be done in render since render is not called while unattached :) */
	if (!pIS->graph_attached) return;

	root = SG_GetRootNode(pIS->graph);
	Node_Render(root, render_stack);
}

#endif

void IS_AttachGraphToRender(InlineScene *is)
{
	is->graph_attached = SG_GetRootNode(is->graph) ? 1 : 0;

	/*main display scene, setup renderer*/
	if (is->root_od->term->root_scene == is) {
		SR_SetSceneGraph(is->root_od->term->renderer, is->graph);
	}
	/*check if inline scene of main display, if so invalidate*/
	else {
		while (is->root_od->parentscene) is = is->root_od->parentscene;
		if (is == is->root_od->term->root_scene) {
			Term_InvalidateScene(is->root_od->term);
		}
	}
}


MediaObject *IS_GetMediaObject(InlineScene *is, MFURL *url, u32 obj_type_hint)
{
	MediaObject *obj;
	u32 i, OD_ID;

	OD_ID = URL_GetODID(url);
	if (!OD_ID) return NULL;

	obj = NULL;
	for (i=0; i<ChainGetCount(is->media_objects); i++) {
		obj = ChainGetEntry(is->media_objects, i);
		/*regular OD scheme*/
		if (OD_ID != DYNAMIC_OD_ID && (obj->OD_ID==OD_ID)) return obj;

		/*dynamic OD scheme*/
		if (OD_ID == DYNAMIC_OD_ID && obj->OD_URL 
			/*locate sub-url in given one (handles viewpoint/segments)*/
			&& strstr(url->vals[0].url, obj->OD_URL) 
			/*if object type unknown (media control, media sensor), return first obj matching URL
			otherwise check types*/
			&& (!obj_type_hint || (obj->type==obj_type_hint)) ) return obj;
	}
	/*create a new object identification*/
	obj = NewMediaObject();
	obj->OD_ID = OD_ID;
	obj->type = obj_type_hint;
	ChainAddEntry(is->media_objects, obj);
	if (OD_ID == DYNAMIC_OD_ID) {
		char *szExt;
		obj->OD_URL = strdup(url->vals[0].url);
		/*remove proto addressing or viewpoint/viewport*/
		switch (obj_type_hint) {
		case NM_OD_SCENE:
			szExt = strrchr(obj->OD_URL, '#');
			if (szExt) szExt[0] = 0;
			break;
		case NM_OD_AUDIO:
			/*little trick to avoid pbs when an audio and a visual node refer to the same service without 
			extensions (eg "file.avi")*/
			szExt = strrchr(obj->OD_URL, '#');
			if (!szExt) {
				szExt = malloc(sizeof(char)* (strlen(obj->OD_URL)+7));
				strcpy(szExt, obj->OD_URL);
				strcat(szExt, "#audio");
				free(obj->OD_URL);
				obj->OD_URL = szExt;
			}
			break;
		}
		IS_InsertOD(is, obj->OD_URL);
		/*safety check!!!*/
		if (ChainFindEntry(is->media_objects, obj)<0) 
			return NULL;
	} else {
		ODManager *odm = IS_FindODM(is, (u16) OD_ID);
		/*this happens with remote ODs*/
		if (odm) {
			while (odm->remote_OD) odm = odm->remote_OD;
			/*OD not attached, try to attach*/
			if (!odm->OD) ODM_SetupService(odm);
		}
	}
	return obj;
}

void IS_SetupOD(InlineScene *is, ODManager *odm)
{
	ODManager *parent;
	MediaObject *obj;
	u32 i;
	/*remote ODs shall NOT be setup*/
	assert(odm->remote_OD==NULL);

	/*get parent*/
	parent = odm;
	while (parent->parent_OD) parent = parent->parent_OD;

	/*an object may already be assigned (when using ESD URLs, setup is performed twice)*/
	if (odm->mo != NULL) goto existing;


	for (i=0; i<ChainGetCount(is->media_objects); i++) {
		obj = ChainGetEntry(is->media_objects, i);
		if (obj->OD_ID==DYNAMIC_OD_ID) {
			assert(obj->OD_URL);
			if (parent->OD->URLString && !stricmp(parent->OD->URLString, obj->OD_URL)) {
				assert(obj->od_man==NULL);
				/*assign FINAL OD, not parent*/
				obj->od_man = odm;
				odm->mo = obj;
				goto existing;
			}
		}
		else if (obj->OD_ID == parent->OD->objectDescriptorID) {
			assert(obj->od_man==NULL);
			obj->od_man = odm;
			odm->mo = obj;
			goto existing;
		}
	}
	/*newly created OD*/
	odm->mo = NewMediaObject();
	ChainAddEntry(is->media_objects, odm->mo);
	odm->mo->od_man = odm;
	odm->mo->OD_ID = parent->OD->objectDescriptorID;

existing:
	/*setup object type*/
	if (!odm->codec) {
		odm->mo->type = NM_OD_SCENE;
	}
	else if (odm->codec->type == M4ST_VISUAL) {
		odm->mo->type = NM_OD_VIDEO;
	} 
	else if (odm->codec->type == M4ST_AUDIO) {
		odm->mo->type = NM_OD_AUDIO;
	}
	else if (odm->codec->type == M4ST_BIFS) {
		odm->mo->type = NM_OD_BIFS;
	}
	
	/*update info*/
	MO_UpdateCaps(odm->mo);
	if (odm->mo->num_open && !odm->is_open) {
		ODM_Start(odm);
		if (odm->mo->speed != 1.0) ODM_SetSpeed(odm, odm->mo->speed);
	}
	/*invalidate scene for all nodes using the OD*/
	Term_InvalidateScene(odm->term);
}

void IS_Restart(InlineScene *is)
{
	is->needs_restart = 1;
}


void IS_SetBIFSDuration(InlineScene *is)
{
	Double dur;
	u32 i, max_dur;
	ODManager *odm;
	Clock *ck;

	/*this is not normative but works in so many cases... set the duration to the max duration
	of all streams sharing the clock*/
	ck = ODM_GetMediaClock(is->root_od);
	max_dur = is->root_od->duration;
	for (i=0; i<ChainGetCount(is->ODlist); i++) {
		odm = ChainGetEntry(is->ODlist, i);
		if (!odm->codec) continue;
		if (ck && ODM_SharesClock(odm, ck)) {
			if (odm->duration>max_dur) max_dur = odm->duration;
		}
	}
	if (is->duration == max_dur) return;

	is->duration = max_dur;
	dur = is->duration;
	dur /= 1000;
	
#ifdef M4_DEF_MediaSensor
	for (i = 0; i < ChainGetCount(is->root_od->ms_stack); i++) {
		MediaSensorStack *media_sens = ChainGetEntry(is->root_od->ms_stack, i);
		if (media_sens->sensor->isActive) {
			media_sens->sensor->mediaDuration = dur;
			Node_OnEventOutSTR((SFNode *) media_sens->sensor, "mediaDuration");
		}
	}
#endif

	if ((is == is->root_od->term->root_scene) && is->root_od->term->user->EventProc) {
		M4Event evt;
		evt.type = M4E_DURATION;
		evt.duration.duration = dur;
		M4USER_SENDEVENT(is->root_od->term->user,&evt);
	}

}

void IS_LoadExternProto(void *_is, MFURL *url)
{
	u32 i;
	InlineScene *is;
	ProtoLink *pl;
	if (!url || !url->count) return;
	is = (InlineScene *)_is;

	for (i=0; i<ChainGetCount(is->extern_protos); i++) {
		pl = ChainGetEntry(is->extern_protos, i);
		if (pl->url == url) return;
		if (pl->url->vals[0].OD_ID == url->vals[0].OD_ID) return;
		if (pl->url->vals[0].url && url->vals[0].url && !stricmp(pl->url->vals[0].url, url->vals[0].url) ) return;
	}
	pl = malloc(sizeof(ProtoLink));
	pl->url = url;
	ChainAddEntry(is->extern_protos, pl);
	pl->mo = IS_GetMediaObject(is, url, NM_OD_SCENE);
	/*this may already be destroyed*/
	if (pl->mo) MO_Play(pl->mo);
}

LPSCENEGRAPH IS_GetProtoLib(void *_is, MFURL *lib_url)
{
	u32 i;
	char *sOpt;
	InlineScene *is = (InlineScene *) _is;
	if (!is || !lib_url->count) return NULL;

	for (i=0; i<ChainGetCount(is->extern_protos); i++) {
		ProtoLink *pl = ChainGetEntry(is->extern_protos, i);
		if (!pl->mo) continue;
		if (URL_GetODID(pl->url) != DYNAMIC_OD_ID) {
			if (URL_GetODID(pl->url) == URL_GetODID(lib_url)) {
				ODManager *odm = pl->mo->od_man;
				if (!odm || !odm->subscene) return NULL;
				return odm->subscene->graph;
			}
		} else if (lib_url->vals[0].url) {
			if (strstr(lib_url->vals[0].url, pl->mo->OD_URL)) {
				ODManager *odm = pl->mo->od_man;
				if (!odm || !odm->subscene) return NULL;
				return odm->subscene->graph;
			}
		}
	}
	if (!lib_url->vals[0].url) return NULL;

	/*look in hardcoded proto libs .. */
	sOpt = IF_GetKey(is->root_od->term->user->config, "Systems", "hardcoded_protos");
	if (sOpt && strstr(sOpt, lib_url->vals[0].url)) return SG_INTERNAL_PROTO;

	return NULL;
}

ODManager *IS_GetProtoSceneByGraph(void *_is, LPSCENEGRAPH sg)
{
	u32 i;
	InlineScene *is = (InlineScene *) _is;
	if (!is) return NULL;
	for (i=0; i<ChainGetCount(is->extern_protos); i++) {
		ProtoLink *pl = ChainGetEntry(is->extern_protos, i);
		ODManager *odm = pl->mo->od_man;
		if (odm && odm->subscene && (odm->subscene->graph==sg)) return odm;
	}
	return NULL;
}


Bool IS_IsProtoLibObject(InlineScene *is, ODManager *odm)
{
	u32 i;
	for (i=0; i<ChainGetCount(is->extern_protos); i++) {
		ProtoLink *pl = ChainGetEntry(is->extern_protos, i);
		if (pl->mo->od_man == odm) return 1;
	}
	return 0;
}


MediaObject *IS_FindObject(InlineScene *is, u16 ODID, char *url)
{
	u32 i;
	if (!url && !ODID) return NULL;
	for (i=0; i<ChainGetCount(is->media_objects); i++) {
		MediaObject *mo = ChainGetEntry(is->media_objects, i);
		if (ODID==DYNAMIC_OD_ID) {
			if (mo->OD_URL && !stricmp(mo->OD_URL, url)) return mo;
		} else if (mo->OD_ID==ODID) return mo;
	}
	return NULL;
}

