import os
import subprocess
from subprocess import PIPE, Popen
import re
import sys

class HardwareDetection(object):
    '''
    A simple class to:
      * Detect the available graphics cards
      * See what drivers support them (If they are
        ATI or NVIDIA cards). If more than one card is 
        available, try to find the highest common 
        driver version which supports them all.
        (READ the comments in the code for further
        details)
      * Return a dictionary such as the following:
      
      
      {
      'fglrx': {
                0: {'compatible': True, 'recommended': True, 'name': 'xorg-driver-fglrx'}
               },
      'nvidia': {
                0: {'compatible': True, 'recommended': False, 'name': 'nvidia-glx-177'},
                1: {'compatible': True, 'recommended': True, 'name': 'nvidia-glx-173'},
                2: {'compatible': True, 'recommended': False, 'name': 'nvidia-glx-96'},
                3: {'compatible': False, 'recommended': False, 'name': 'nvidia-glx-71'}
                }
      }
      
    '''
    
    aliasesPath = '/usr/share/jockey/modaliases/'
    driverDetails = {}
    
    def __init__(self, datadir=aliasesPath):
        '''
        If the modaliases path does not exist
        print none, so that debconf is not triggered.
        '''
        if not os.path.isdir(datadir):
            #print 'none'
            raise IOError, "dir %s not found" % datadir
        self.datadir = datadir
        self.detection()
        self.getData()
        self.getCards()
        self.removeUnsupported()
        
    def detection(self):
        '''
        Detect the models of the graphics cards
        and store them in self.cards
        '''
        self.cards = []
        p1 = Popen(['lspci', '-n'], stdout=PIPE)
        p = p1.communicate()[0].split('\n')
        indentifier1 = re.compile('.*0300: *(.+):(.+) \(.+\)')
        indentifier2 = re.compile('.*0300: *(.+):(.+)')
        for line in p:
            m1 = indentifier1.match(line)
            m2 = indentifier2.match(line)
            if m1:
                id1 = m1.group(1).strip().lower()
                id2 = m1.group(2).strip().lower()
                id = id1 + ':' + id2
                self.cards.append(id)
            elif m2:
                id1 = m2.group(1).strip().lower()
                id2 = m2.group(2).strip().lower()
                id = id1 + ':' + id2
                self.cards.append(id)

    def getData(self):
        '''
        Get the data from the modaliases for each driver
        and store them in self.drivers
        '''
        
        files = os.listdir(self.datadir)
        self.drivers = {}
        
        for file in files:
            a = open(self.datadir + file, 'r')
            lines = a.readlines()
            NvidiaIdentifier = re.compile('.*alias pci:v0000(.+)d0000(.+)sv.*nvidia.*nvidia-glx-(.+).*')
            AtiIdentifier = re.compile('.*alias pci:v0000(.+)d0000(.+)sv\*sd\*bc03sc\*i\*.*fglrx.*')
            for line in lines:
                m1 = NvidiaIdentifier.match(line)
                m2 = AtiIdentifier.match(line)
                if m1:
                    id1 = m1.group(1).strip().lower()#e.g. 10de or 12d2 for nvidia
                    id2 = m1.group(2).strip().lower()#the id which is specific to the model
                    drivername = m1.group(3).strip().lower()#e.g. 173, 177, 96 or 71
                    if id1 in ['10de', '12d2']:#if NVIDIA
                        self.drivers.setdefault('nvidia', {}).setdefault(int(drivername), []).append(id1 + ':' + id2)
                elif m2:
                    id1 = m2.group(1).strip().lower()#e.g. 1002 for ati
                    id2 = m2.group(2).strip().lower()#the id which is specific to the model
                    drivername = 'fglrx'#m1.group(3).strip().lower()#e.g. 173, 177, 96 or 71
                    if id1 in ['1002']:#if ATI
                        self.drivers.setdefault('fglrx', {}).setdefault(drivername, []).append(id1 + ':' + id2)
            a.close()
        
        
        '''
        If the modaliases files don't contain anything useful
        just print none and exit so as not to trigger debconf.
        '''
        if len(self.drivers.keys()) == 0:
            raise ValueError, "modaliases have no useful information"
    
    def getCards(self):
        '''
        See if the detected graphics cards are NVIDIA cards.
        If they are NVIDIA cards, append them to self.nvidiaCards
        '''
        self.driversForCards = {}
        self.nvidiaCards = []
        self.atiCards = []
        '''
        It is possible to override hardware detection (only for testing
        purposes) by setting self.cards to one of the following lists:
        
        self.cards = ['10de:02e2', '10de:002d', '10de:0296', '10de:087f']
        self.cards = ['10de:02e2', '10de:087f']
        self.cards = ['10de:02e2', '10de:087f', '10de:fake']
        self.cards = ['10de:fake']
        self.cards = ['1002:fake'] ATI
        self.cards = ['1002:95c7'] ATI
        '''
        #self.cards = ['1002:fake', '1002:95c7']
        
        for card in self.cards:
            if card[0: card.find(':')] in ['10de', '12d2']:
                self.nvidiaCards.append(card)
            elif card[0: card.find(':')] == '1002':
                self.atiCards.append(card)
        
        self.nvidiaOrderedList = self.drivers['nvidia'].keys()
        self.nvidiaOrderedList.sort(reverse=True)
        
        self.atiOrderedList = self.drivers['fglrx'].keys()
        '''
        See what drivers support each card and fill self.driversForCards
        so as to have something like the following:
        
        self.driversForCards = {
                                 'id_of_card1': [driver1, driver2],
                                 'id_of_card2': [driver2, driver3],
                               }
        '''
        for card in self.nvidiaCards:
            supported = False
            for driver in self.nvidiaOrderedList:
                if card in self.drivers['nvidia'][driver]:
                    supported = True
                    self.driversForCards.setdefault(card, []).append(driver)
            if supported == False:
                self.driversForCards.setdefault(card, []).append(None)
        
        for card in self.atiCards:
            supported = False
            for driver in self.atiOrderedList:
                if card in self.drivers['fglrx'][driver]:
                    supported = True
                    self.driversForCards.setdefault(card, []).append(driver)
            if supported == False:
                self.driversForCards.setdefault(card, []).append(None)

    def removeUnsupported(self):
        '''
        Remove unsupported cards from self.nvidiaCards and from
        self.driversForCards
        '''
        #print self.driversForCards
        unsupportedCards = []
        for card in self.driversForCards:
            if None in self.driversForCards[card]:
                unsupportedCards.append(card)

        for unsupported in unsupportedCards:
            try:
                self.nvidiaCards.remove(unsupported)
            except ValueError:
                self.atiCards.remove(unsupported)
            del self.driversForCards[unsupported]

    def selectDriver(self):
        '''
        If more than one card is available, try to get the highest common driver
        '''
        
        compatibleDrivers = {}
        recommendedDrivers = {}
        availableDrivers = {}
        
        nvidiaCardsNumber = len(self.nvidiaCards)
        atiCardsNumber = len(self.atiCards)
        if nvidiaCardsNumber > 0:#if a NVIDIA card is available
            if nvidiaCardsNumber > 1:#if more than 1 card
                '''
                occurrence stores the number of occurrences (the values of the
                dictionary) of each driver version (the keys of the dictionary)
                
                Example:
                    occurrence = {177: 1, 173: 3}
                    This means that driver 177 supports only 1 card while 173
                    supports 3 cards.
                '''
                occurrence = {}
                #print self.driversForCards
                for card in self.driversForCards:
                    if '1002' not in card:
                        for drv in self.driversForCards[card]:
                            occurrence.setdefault(drv, 0)
                            occurrence[drv] += 1
                
                occurrences = occurrence.keys()
                occurrences.sort(reverse=True)
                '''
                candidates is the list of the likely candidates for the
                installation
                '''
                candidates = []
                for driver in occurrences:
                    if occurrence[driver] == nvidiaCardsNumber:
                        candidates.append(driver)
                
                
                if len(candidates) > 0:
                    '''
                    If more than one driver version works for all the available
                    cards then the newest one is selected.
                    
                    USE-CASE:
                        If a user has the following cards:
                        * GeForce 9300 (supported by driver 177 and 173)
                        * GeForce 7300 (supported by driver 177 and 173)
                        * GeForce 6200 (supported by driver 177 and 173)
                    
                        Driver 177 is selected.
                    '''
                    candidates.sort(reverse=True)
                    
                    
                    if 173 in candidates['nvidia'] and 177 in candidates['nvidia']:
                        '''
                        Prefer the stable driver
                        '''
                        choice = candidates['nvidia'][1]
                    else:
                        choice = candidates['nvidia'][0]
#                    if self.verbose and not self.printonly:
#                        print 'Recommended NVIDIA driver:', choice
                else:
                    '''
                    Otherwise, if there is no single driver version which works 
                    for all the available cards, the newest is selected.
                    
                    USE-CASE:
                        If a user has the following cards:
                        * GeForce 9300 (supported by driver 177 and 173)
                        * GeForce 1 (supported by driver 71)
                        * GeForce 2 (supported by driver 71)
                        
                        The most modern card has the highest priority since
                        no common driver can be found. The other 2 cards 
                        should use the open source driver
                    '''
                    
                    if 173 in occurrences and 177 in occurrences:
                        '''
                        Prefer the stable driver
                        '''
                        choice = occurrences[1]
                    else:
                        choice = occurrences[0]
#                    if self.verbose and not self.printonly:
#                        print 'Recommended NVIDIA driver:', choice
            else:#just one NVIDIA card
                '''
                The choice is easy if only one card is available and/or supported.
                
                The newest driver which supports the card is chosen.
                '''
                it = 0
                for card in self.driversForCards.keys():
                    if '1002' not in card:
                        position = it
                    it += 1
                
                drivers = self.driversForCards[self.driversForCards.keys()[position]]
                
                if 173 in drivers and 177 in drivers:
                    '''
                    Prefer the stable driver
                    '''
                    choice = drivers[1]
                else:
                    choice = drivers[0]
                
                #choice = self.driversForCards[self.driversForCards.keys()[position]][0]
            
            for card in self.driversForCards:
                if '1002' not in card:
                    for drv in self.driversForCards[card]:
                        compatibleDrivers.setdefault('nvidia', []).append('nvidia-glx-' + str(drv))
            
            choice = 'nvidia-glx-' + str(choice)
            
            recommendedDrivers['nvidia'] = choice
        
        
        
        if atiCardsNumber > 0:
            choice = self.driversForCards[self.driversForCards.keys()[0]][0]
            
            compatibleDrivers.setdefault('fglrx', []).append('xorg-driver-' + choice)
            recommendedDrivers['fglrx'] = 'xorg-driver-' + choice
            
        if atiCardsNumber == 0 and nvidiaCardsNumber == 0 :
            '''
            If no card is supported
            '''
            choice = None
            
            compatibleDrivers['nvidia'] = []
            recommendedDrivers['nvidia'] = choice
            compatibleDrivers['fglrx'] = []
            recommendedDrivers['fglrx'] = choice
        
        for driver in self.drivers:
            availableDrivers.setdefault(driver, [])
            temp = []
            for drv in self.drivers[driver]:
                temp.append(drv)
            
            if driver == 'nvidia':
                temp.sort(reverse=True)
                for drv in temp:
                    availableDrivers[driver].append('nvidia-glx-' + str(drv))
            elif driver == 'fglrx':
                for drv in temp:
                    availableDrivers[driver].append('xorg-driver-' + str(drv))
        
        for driver in availableDrivers:
            
            self.driverDetails.setdefault(driver, {})
            it = 0
            for drv in availableDrivers[driver]:
                self.driverDetails[driver].setdefault(it, {})
                self.driverDetails[driver][it]['name'] = drv
                try:
                    self.driverDetails[driver][it]['compatible'] = drv in compatibleDrivers[driver]
                    self.driverDetails[driver][it]['recommended'] = recommendedDrivers[driver] == drv
                except KeyError:
                    self.driverDetails[driver][it]['compatible'] = False
                    self.driverDetails[driver][it]['recommended'] = False
                
                
                it += 1
            
#        print 'DriverDetails =', self.driverDetails
##        print 'Compatible =', compatibleDrivers, '\nRecommended', recommendedDrivers
#        print 'All drivers =', availableDrivers#self.drivers['nvidia'].keys(), self.drivers['fglrx'].keys()
        return self.driverDetails
if __name__ == '__main__':
    a = HardwareDetection()
    print a.selectDriver()
