(* The 14 Standard PDF Fonts (Widths and Kerns). *)
open Utility

(* The raw kern data is a list of integer triples. We need the first two as a
pair, to form a key for the lookup hashtable. *)
let hashtable_of_kerns kerns =
  hashtable_of_dictionary
    (map (fun (c, c', k) -> (c, c'), k) kerns)

(* TimesRoman *)
let times_roman_widths =
  hashtable_of_dictionary
  [32, 250; 33, 333; 34, 408; 35, 500; 36, 500; 37, 833; 38, 778; 39, 333;
   40, 333; 41, 333; 42, 500; 43, 564; 44, 250; 45, 333; 46, 250; 47, 278;
   48, 500; 49, 500; 50, 500; 51, 500; 52, 500; 53, 500; 54, 500; 55, 500;
   56, 500; 57, 500; 58, 278; 59, 278; 60, 564; 61, 564; 62, 564; 63, 444;
   64, 921; 65, 722; 66, 667; 67, 667; 68, 722; 69, 611; 70, 556; 71, 722;
   72, 722; 73, 333; 74, 389; 75, 722; 76, 611; 77, 889; 78, 722; 79, 722;
   80, 556; 81, 722; 82, 667; 83, 556; 84, 611; 85, 722; 86, 722; 87, 944;
   88, 722; 89, 722; 90, 611; 91, 333; 92, 278; 93, 333; 94, 469; 95, 500;
   96, 333; 97, 444; 98, 500; 99, 444; 100, 500; 101, 444; 102, 333; 103, 500;
   104, 500; 105, 278; 106, 278; 107, 500; 108, 278; 109, 778; 110, 500;
   111, 500; 112, 500; 113, 500; 114, 333; 115, 389; 116, 278; 117, 500;
   118, 500; 119, 722; 120, 500; 121, 500; 122, 444; 123, 480; 124, 200;
   125, 480; 126, 541; 161, 333; 162, 500; 163, 500; 164, 167; 165, 500;
   166, 500; 167, 500; 168, 500; 169, 180; 170, 444; 171, 500; 172, 333;
   173, 333; 174, 556; 175, 556; 177, 500; 178, 500; 179, 500; 180, 250;
   182, 453; 183, 350; 184, 333; 185, 444; 186, 444; 187, 500; 188, 1000;
   189, 1000; 191, 444; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 889; 227, 276; 232, 611; 233, 722; 234, 889;
   235, 310; 241, 667; 245, 278; 248, 278; 249, 500; 250, 722; 251, 500]

let times_roman_kerns =
  hashtable_of_kerns
  [65, 121, -92; 65, 119, -92; 65, 118, -74; 65, 117, 0; 65, 39, -111;
   65, 186, 0; 65, 112, 0; 65, 89, -105; 65, 87, -90; 65, 86, -135;
   65, 85, -55; 65, 84, -111; 65, 81, -55; 65, 79, -55; 65, 71, -40;
   65, 67, -40; 66, 46, 0; 66, 44, 0; 66, 85, -10; 66, 65, -35; 68, 46, 0;
   68, 44, 0; 68, 89, -55; 68, 87, -30; 68, 86, -40; 68, 65, -40; 70, 114, 0;
   70, 46, -80; 70, 111, -15; 70, 105, 0; 70, 101, 0; 70, 44, -80; 70, 97, -15;
   70, 65, -74; 71, 46, 0; 71, 44, 0; 74, 117, 0; 74, 46, 0; 74, 111, 0;
   74, 101, 0; 74, 44, 0; 74, 97, 0; 74, 65, -60; 75, 121, -25; 75, 117, -15;
   75, 111, -35; 75, 101, -25; 75, 79, -30; 76, 121, -55; 76, 39, -92;
   76, 186, 0; 76, 89, -100; 76, 87, -74; 76, 86, -100; 76, 84, -92; 78, 46, 0;
   78, 44, 0; 78, 65, -35; 79, 46, 0; 79, 44, 0; 79, 89, -50; 79, 88, -40;
   79, 87, -35; 79, 86, -50; 79, 84, -40; 79, 65, -35; 80, 46, -111; 80, 111, 0;
   80, 101, 0; 80, 44, -111; 80, 97, -15; 80, 65, -92; 81, 46, 0; 81, 44, 0;
   81, 85, -10; 82, 89, -65; 82, 87, -55; 82, 86, -80; 82, 85, -40; 82, 84, -60;
   82, 79, -40; 83, 46, 0; 83, 44, 0; 84, 121, -80; 84, 119, -80; 84, 117, -45;
   84, 59, -55; 84, 114, -35; 84, 46, -74; 84, 111, -80; 84, 105, -35;
   84, 45, -92; 84, 104, 0; 84, 101, -70; 84, 44, -74; 84, 58, -50; 84, 97, -80;
   84, 79, -18; 84, 65, -93; 85, 46, 0; 85, 44, 0; 85, 65, -40; 86, 117, -75;
   86, 59, -74; 86, 46, -129; 86, 111, -129; 86, 105, -60; 86, 45, -100;
   86, 101, -111; 86, 44, -129; 86, 58, -74; 86, 97, -111; 86, 79, -40;
   86, 71, -15; 86, 65, -135; 87, 121, -73; 87, 117, -50; 87, 59, -37;
   87, 46, -92; 87, 111, -80; 87, 105, -40; 87, 45, -65; 87, 104, 0;
   87, 101, -80; 87, 44, -92; 87, 58, -37; 87, 97, -80; 87, 79, -10;
   87, 65, -120; 89, 117, -111; 89, 59, -92; 89, 46, -129; 89, 111, -110;
   89, 105, -55; 89, 45, -111; 89, 101, -100; 89, 44, -129; 89, 58, -92;
   89, 97, -100; 89, 79, -30; 89, 65, -120; 97, 121, 0; 97, 119, -15;
   97, 118, -20; 97, 116, 0; 97, 112, 0; 97, 103, 0; 97, 98, 0; 98, 121, 0;
   98, 118, -15; 98, 117, -20; 98, 46, -40; 98, 108, 0; 98, 44, 0; 98, 98, 0;
   99, 121, -15; 99, 46, 0; 99, 108, 0; 99, 107, 0; 99, 104, 0; 99, 44, 0;
   58, 32, 0; 44, 32, 0; 44, 39, -70; 44, 186, -70; 100, 121, 0; 100, 119, 0;
   100, 118, 0; 100, 46, 0; 100, 100, 0; 100, 44, 0; 101, 121, -15;
   101, 120, -15; 101, 119, -25; 101, 118, -25; 101, 46, 0; 101, 112, 0;
   101, 103, -15; 101, 44, 0; 101, 98, 0; 102, 39, 55; 102, 186, 0; 102, 46, 0;
   102, 111, 0; 102, 108, 0; 102, 105, -20; 102, 102, -25; 102, 101, 0;
   102, 245, -50; 102, 44, 0; 102, 97, -10; 103, 121, 0; 103, 114, 0;
   103, 46, 0; 103, 111, 0; 103, 105, 0; 103, 103, 0; 103, 101, 0; 103, 44, 0;
   103, 97, -5; 104, 121, -5; 105, 118, -25; 107, 121, -15; 107, 111, -10;
   107, 101, -10; 108, 121, 0; 108, 119, -10; 109, 121, 0; 109, 117, 0;
   110, 121, -15; 110, 118, -40; 110, 117, 0; 111, 121, -10; 111, 120, 0;
   111, 119, -25; 111, 118, -15; 111, 103, 0; 112, 121, -10; 46, 39, -70;
   46, 186, -70; 170, 96, 0; 170, 65, -80; 186, 32, 0; 96, 96, -74; 96, 65, -80;
   39, 118, -50; 39, 116, -18; 39, 32, -74; 39, 115, -55; 39, 114, -50;
   39, 39, -74; 39, 186, 0; 39, 108, -10; 39, 100, -50; 114, 121, 0; 114, 118, 0;
   114, 117, 0; 114, 116, 0; 114, 115, 0; 114, 114, 0; 114, 113, 0; 114, 46, -55;
   114, 112, 0; 114, 111, 0; 114, 110, 0; 114, 109, 0; 114, 108, 0; 114, 107, 0;
   114, 105, 0; 114, 45, -20; 114, 103, -18; 114, 101, 0; 114, 100, 0;
   114, 44, -40; 114, 99, 0; 114, 97, 0; 115, 119, 0; 32, 96, 0; 32, 170, 0;
   32, 89, -90; 32, 87, -30; 32, 86, -50; 32, 84, -18; 32, 65, -55; 118, 46, -65;
   118, 111, -20; 118, 101, -15; 118, 44, -65; 118, 97, -25; 119, 46, -65;
   119, 111, -10; 119, 104, 0; 119, 101, 0; 119, 44, -65; 119, 97, -10;
   120, 101, -15; 121, 46, -65; 121, 111, 0; 121, 101, 0; 121, 44, -65;
   121, 97, 0; 122, 111, 0; 122, 101, 0]

(* TimesBold *)
let times_bold_widths =
  hashtable_of_dictionary
  [32, 250; 33, 333; 34, 555; 35, 500; 36, 500; 37, 1000; 38, 833; 39, 333;
   40, 333; 41, 333; 42, 500; 43, 570; 44, 250; 45, 333; 46, 250; 47, 278;
   48, 500; 49, 500; 50, 500; 51, 500; 52, 500; 53, 500; 54, 500; 55, 500;
   56, 500; 57, 500; 58, 333; 59, 333; 60, 570; 61, 570; 62, 570; 63, 500;
   64, 930; 65, 722; 66, 667; 67, 722; 68, 722; 69, 667; 70, 611; 71, 778;
   72, 778; 73, 389; 74, 500; 75, 778; 76, 667; 77, 944; 78, 722; 79, 778;
   80, 611; 81, 778; 82, 722; 83, 556; 84, 667; 85, 722; 86, 722; 87, 1000;
   88, 722; 89, 722; 90, 667; 91, 333; 92, 278; 93, 333; 94, 581; 95, 500;
   96, 333; 97, 500; 98, 556; 99, 444; 100, 556; 101, 444; 102, 333; 103, 500;
   104, 556; 105, 278; 106, 333; 107, 556; 108, 278; 109, 833; 110, 556;
   111, 500; 112, 556; 113, 556; 114, 444; 115, 389; 116, 333; 117, 556;
   118, 500; 119, 722; 120, 500; 121, 500; 122, 444; 123, 394; 124, 220;
   125, 394; 126, 520; 161, 333; 162, 500; 163, 500; 164, 167; 165, 500;
   166, 500; 167, 500; 168, 500; 169, 278; 170, 500; 171, 500; 172, 333;
   173, 333; 174, 556; 175, 556; 177, 500; 178, 500; 179, 500; 180, 250;
   182, 540; 183, 350; 184, 333; 185, 500; 186, 500; 187, 500; 188, 1000;
   189, 1000; 191, 500; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 1000; 227, 300; 232, 667; 233, 778; 234, 1000;
   235, 330; 241, 722; 245, 278; 248, 278; 249, 500; 250, 722; 251, 556]
  
let times_bold_kerns =
  hashtable_of_kerns 
  [65, 121, -74; 65, 119, -90; 65, 118, -100; 65, 117, -50; 65, 39, -74;
   65, 186, 0; 65, 112, -25; 65, 89, -100; 65, 87, -130; 65, 86, -145;
   65, 85, -50; 65, 84, -95; 65, 81, -45; 65, 79, -45; 65, 71, -55;
   65, 67, -55; 66, 46, 0; 66, 44, 0; 66, 85, -10; 66, 65, -30; 68, 46, -20;
   68, 44, 0; 68, 89, -40; 68, 87, -40; 68, 86, -40; 68, 65, -35; 70, 114, 0;
   70, 46, -110; 70, 111, -25; 70, 105, 0; 70, 101, -25; 70, 44, -92;
   70, 97, -25; 70, 65, -90; 71, 46, 0; 71, 44, 0; 74, 117, -15; 74, 46, -20;
   74, 111, -15; 74, 101, -15; 74, 44, 0; 74, 97, -15; 74, 65, -30;
   75, 121, -45; 75, 117, -15; 75, 111, -25; 75, 101, -25; 75, 79, -30;
   76, 121, -55; 76, 39, -110; 76, 186, -20; 76, 89, -92; 76, 87, -92;
   76, 86, -92; 76, 84, -92; 78, 46, 0; 78, 44, 0; 78, 65, -20; 79, 46, 0;
   79, 44, 0; 79, 89, -50; 79, 88, -40; 79, 87, -50; 79, 86, -50; 79, 84, -40;
   79, 65, -40; 80, 46, -110; 80, 111, -20; 80, 101, -20; 80, 44, -92;
   80, 97, -10; 80, 65, -74; 81, 46, -20; 81, 44, 0; 81, 85, -10; 82, 89, -35;
   82, 87, -35; 82, 86, -55; 82, 85, -30; 82, 84, -40; 82, 79, -30; 83, 46, 0;
   83, 44, 0; 84, 121, -74; 84, 119, -74; 84, 117, -92; 84, 59, -74;
   84, 114, -74; 84, 46, -90; 84, 111, -92; 84, 105, -18; 84, 45, -92;
   84, 104, 0; 84, 101, -92; 84, 44, -74; 84, 58, -74; 84, 97, -92;
   84, 79, -18; 84, 65, -90; 85, 46, -50; 85, 44, -50; 85, 65, -60;
   86, 117, -92; 86, 59, -92; 86, 46, -145; 86, 111, -100; 86, 105, -37;
   86, 45, -74; 86, 101, -100; 86, 44, -129; 86, 58, -92; 86, 97, -92;
   86, 79, -45; 86, 71, -30; 86, 65, -135; 87, 121, -60; 87, 117, -50;
   87, 59, -55; 87, 46, -92; 87, 111, -75; 87, 105, -18; 87, 45, -37;
   87, 104, 0; 87, 101, -65; 87, 44, -92; 87, 58, -55; 87, 97, -65;
   87, 79, -10; 87, 65, -120; 89, 117, -92; 89, 59, -92; 89, 46, -92;
   89, 111, -111; 89, 105, -37; 89, 45, -92; 89, 101, -111; 89, 44, -92;
   89, 58, -92; 89, 97, -85; 89, 79, -35; 89, 65, -110; 97, 121, 0; 97, 119, 0;
   97, 118, -25; 97, 116, 0; 97, 112, 0; 97, 103, 0; 97, 98, 0; 98, 121, 0;
   98, 118, -15; 98, 117, -20; 98, 46, -40; 98, 108, 0; 98, 44, 0; 98, 98, -10;
   99, 121, 0; 99, 46, 0; 99, 108, 0; 99, 107, 0; 99, 104, 0; 99, 44, 0;
   58, 32, 0; 44, 32, 0; 44, 39, -55; 44, 186, -45; 100, 121, 0; 100, 119, -15;
   100, 118, 0; 100, 46, 0; 100, 100, 0; 100, 44, 0; 101, 121, 0; 101, 120, 0;
   101, 119, 0; 101, 118, -15; 101, 46, 0; 101, 112, 0; 101, 103, 0; 101, 44, 0;
   101, 98, 0; 102, 39, 55; 102, 186, 50; 102, 46, -15; 102, 111, -25;
   102, 108, 0; 102, 105, -25; 102, 102, 0; 102, 101, 0; 102, 245, -35;
   102, 44, -15; 102, 97, 0; 103, 121, 0; 103, 114, 0; 103, 46, -15; 103, 111, 0;
   103, 105, 0; 103, 103, 0; 103, 101, 0; 103, 44, 0; 103, 97, 0; 104, 121, -15;
   105, 118, -10; 107, 121, -15; 107, 111, -15; 107, 101, -10; 108, 121, 0;
   108, 119, 0; 109, 121, 0; 109, 117, 0; 110, 121, 0; 110, 118, -40;
   110, 117, 0; 111, 121, 0; 111, 120, 0; 111, 119, -10; 111, 118, -10;
   111, 103, 0; 112, 121, 0; 46, 39, -55; 46, 186, -55; 170, 96, 0;
   170, 65, -10; 186, 32, 0; 96, 96, -63; 96, 65, -10; 39, 118, -20;
   39, 116, 0; 39, 32, -74; 39, 115, -37; 39, 114, -20; 39, 39, -63;
   39, 186, 0; 39, 108, 0; 39, 100, -20; 114, 121, 0; 114, 118, -10;
   114, 117, 0; 114, 116, 0; 114, 115, 0; 114, 114, 0; 114, 113, -18;
   114, 46, -100; 114, 112, -10; 114, 111, -18; 114, 110, -15; 114, 109, 0;
   114, 108, 0; 114, 107, 0; 114, 105, 0; 114, 45, -37; 114, 103, -10;
   114, 101, -18; 114, 100, 0; 114, 44, -92; 114, 99, -18; 114, 97, 0;
   115, 119, 0; 32, 96, 0; 32, 170, 0; 32, 89, -55; 32, 87, -30; 32, 86, -45;
   32, 84, -30; 32, 65, -55; 118, 46, -70; 118, 111, -10; 118, 101, -10;
   118, 44, -55; 118, 97, -10; 119, 46, -70; 119, 111, -10; 119, 104, 0;
   119, 101, 0; 119, 44, -55; 119, 97, 0; 120, 101, 0; 121, 46, -70;
   121, 111, -25; 121, 101, -10; 121, 44, -55; 121, 97, 0; 122, 111, 0;
   122, 101, 0]

(* TimesItalic *)
let times_italic_widths =
  hashtable_of_dictionary
  [32, 250; 33, 333; 34, 420; 35, 500; 36, 500; 37, 833; 38, 778; 39, 333;
   40, 333; 41, 333; 42, 500; 43, 675; 44, 250; 45, 333; 46, 250; 47, 278;
   48, 500; 49, 500; 50, 500; 51, 500; 52, 500; 53, 500; 54, 500; 55, 500;
   56, 500; 57, 500; 58, 333; 59, 333; 60, 675; 61, 675; 62, 675; 63, 500;
   64, 920; 65, 611; 66, 611; 67, 667; 68, 722; 69, 611; 70, 611; 71, 722;
   72, 722; 73, 333; 74, 444; 75, 667; 76, 556; 77, 833; 78, 667; 79, 722;
   80, 611; 81, 722; 82, 611; 83, 500; 84, 556; 85, 722; 86, 611; 87, 833;
   88, 611; 89, 556; 90, 556; 91, 389; 92, 278; 93, 389; 94, 422; 95, 500;
   96, 333; 97, 500; 98, 500; 99, 444; 100, 500; 101, 444; 102, 278; 103, 500;
   104, 500; 105, 278; 106, 278; 107, 444; 108, 278; 109, 722; 110, 500;
   111, 500; 112, 500; 113, 500; 114, 389; 115, 389; 116, 278; 117, 500;
   118, 444; 119, 667; 120, 444; 121, 444; 122, 389; 123, 400; 124, 275;
   125, 400; 126, 541; 161, 389; 162, 500; 163, 500; 164, 167; 165, 500;
   166, 500; 167, 500; 168, 500; 169, 214; 170, 556; 171, 500; 172, 333;
   173, 333; 174, 500; 175, 500; 177, 500; 178, 500; 179, 500; 180, 250;
   182, 523; 183, 350; 184, 333; 185, 556; 186, 556; 187, 500; 188, 889;
   189, 1000; 191, 500; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 889; 225, 889; 227, 276; 232, 556; 233, 722; 234, 944;
   235, 310; 241, 667; 245, 278; 248, 278; 249, 500; 250, 667; 251, 500]

let times_italic_kerns =
  hashtable_of_kerns
  [65, 121, -55; 65, 119, -55; 65, 118, -55; 65, 117, -20; 65, 39, -37;
   65, 186, 0; 65, 112, 0; 65, 89, -55; 65, 87, -95; 65, 86, -105; 65, 85, -50;
   65, 84, -37; 65, 81, -40; 65, 79, -40; 65, 71, -35; 65, 67, -30; 66, 46, 0;
   66, 44, 0; 66, 85, -10; 66, 65, -25; 68, 46, 0; 68, 44, 0; 68, 89, -40;
   68, 87, -40; 68, 86, -40; 68, 65, -35; 70, 114, -55; 70, 46, -135;
   70, 111, -105; 70, 105, -45; 70, 101, -75; 70, 44, -135; 70, 97, -75;
   70, 65, -115; 71, 46, 0; 71, 44, 0; 74, 117, -35; 74, 46, -25; 74, 111, -25;
   74, 101, -25; 74, 44, -25; 74, 97, -35; 74, 65, -40; 75, 121, -40;
   75, 117, -40; 75, 111, -40; 75, 101, -35; 75, 79, -50; 76, 121, -30;
   76, 39, -37; 76, 186, 0; 76, 89, -20; 76, 87, -55; 76, 86, -55; 76, 84, -20;
   78, 46, 0; 78, 44, 0; 78, 65, -27; 79, 46, 0; 79, 44, 0; 79, 89, -50;
   79, 88, -40; 79, 87, -50; 79, 86, -50; 79, 84, -40; 79, 65, -55;
   80, 46, -135; 80, 111, -80; 80, 101, -80; 80, 44, -135; 80, 97, -80;
   80, 65, -90; 81, 46, 0; 81, 44, 0; 81, 85, -10; 82, 89, -18; 82, 87, -18;
   82, 86, -18; 82, 85, -40; 82, 84, 0; 82, 79, -40; 83, 46, 0; 83, 44, 0;
   84, 121, -74; 84, 119, -74; 84, 117, -55; 84, 59, -65; 84, 114, -55;
   84, 46, -74; 84, 111, -92; 84, 105, -55; 84, 45, -74; 84, 104, 0;
   84, 101, -92; 84, 44, -74; 84, 58, -55; 84, 97, -92; 84, 79, -18;
   84, 65, -50; 85, 46, -25; 85, 44, -25; 85, 65, -40; 86, 117, -74;
   86, 59, -74; 86, 46, -129; 86, 111, -111; 86, 105, -74; 86, 45, -55;
   86, 101, -111; 86, 44, -129; 86, 58, -65; 86, 97, -111; 86, 79, -30;
   86, 71, 0; 86, 65, -60; 87, 121, -70; 87, 117, -55; 87, 59, -65;
   87, 46, -92; 87, 111, -92; 87, 105, -55; 87, 45, -37; 87, 104, 0;
   87, 101, -92; 87, 44, -92; 87, 58, -65; 87, 97, -92; 87, 79, -25;
   87, 65, -60; 89, 117, -92; 89, 59, -65; 89, 46, -92; 89, 111, -92;
   89, 105, -74; 89, 45, -74; 89, 101, -92; 89, 44, -92; 89, 58, -65;
   89, 97, -92; 89, 79, -15; 89, 65, -50; 97, 121, 0; 97, 119, 0; 97, 118, 0;
   97, 116, 0; 97, 112, 0; 97, 103, -10; 97, 98, 0; 98, 121, 0; 98, 118, 0;
   98, 117, -20; 98, 46, -40; 98, 108, 0; 98, 44, 0; 98, 98, 0; 99, 121, 0;
   99, 46, 0; 99, 108, 0; 99, 107, -20; 99, 104, -15; 99, 44, 0; 58, 32, 0;
   44, 32, 0; 44, 39, -140; 44, 186, -140; 100, 121, 0; 100, 119, 0;
   100, 118, 0; 100, 46, 0; 100, 100, 0; 100, 44, 0; 101, 121, -30;
   101, 120, -20; 101, 119, -15; 101, 118, -15; 101, 46, -15; 101, 112, 0;
   101, 103, -40; 101, 44, -10; 101, 98, 0; 102, 39, 92; 102, 186, 0;
   102, 46, -15; 102, 111, 0; 102, 108, 0; 102, 105, -20; 102, 102, -18;
   102, 101, 0; 102, 245, -60; 102, 44, -10; 102, 97, 0; 103, 121, 0;
   103, 114, 0; 103, 46, -15; 103, 111, 0; 103, 105, 0; 103, 103, -10;
   103, 101, -10; 103, 44, -10; 103, 97, 0; 104, 121, 0; 105, 118, 0;
   107, 121, -10; 107, 111, -10; 107, 101, -10; 108, 121, 0; 108, 119, 0;
   109, 121, 0; 109, 117, 0; 110, 121, 0; 110, 118, -40; 110, 117, 0;
   111, 121, 0; 111, 120, 0; 111, 119, 0; 111, 118, -10; 111, 103, -10;
   112, 121, 0; 46, 39, -140; 46, 186, -140; 170, 96, 0; 170, 65, 0;
   186, 32, 0; 96, 96, -111; 96, 65, 0; 39, 118, -10; 39, 116, -30;
   39, 32, -111; 39, 115, -40; 39, 114, -25; 39, 39, -111; 39, 186, 0;
   39, 108, 0; 39, 100, -25; 114, 121, 0; 114, 118, 0; 114, 117, 0;
   114, 116, 0; 114, 115, -10; 114, 114, 0; 114, 113, -37; 114, 46, -111;
   114, 112, 0; 114, 111, -45; 114, 110, 0; 114, 109, 0; 114, 108, 0;
   114, 107, 0; 114, 105, 0; 114, 45, -20; 114, 103, -37; 114, 101, -37;
   114, 100, -37; 114, 44, -111; 114, 99, -37; 114, 97, -15; 115, 119, 0;
   32, 96, 0; 32, 170, 0; 32, 89, -75; 32, 87, -40; 32, 86, -35; 32, 84, -18;
   32, 65, -18; 118, 46, -74; 118, 111, 0; 118, 101, 0; 118, 44, -74;
   118, 97, 0; 119, 46, -74; 119, 111, 0; 119, 104, 0; 119, 101, 0;
   119, 44, -74; 119, 97, 0; 120, 101, 0; 121, 46, -55; 121, 111, 0;
   121, 101, 0; 121, 44, -55; 121, 97, 0; 122, 111, 0; 122, 101, 0]

(* TimesBoldItalic *)
let times_bold_italic_widths =
  hashtable_of_dictionary
  [32, 250; 33, 389; 34, 555; 35, 500; 36, 500; 37, 833; 38, 778; 39, 333;
   40, 333; 41, 333; 42, 500; 43, 570; 44, 250; 45, 333; 46, 250; 47, 278;
   48, 500; 49, 500; 50, 500; 51, 500; 52, 500; 53, 500; 54, 500; 55, 500;
   56, 500; 57, 500; 58, 333; 59, 333; 60, 570; 61, 570; 62, 570; 63, 500;
   64, 832; 65, 667; 66, 667; 67, 667; 68, 722; 69, 667; 70, 667; 71, 722;
   72, 778; 73, 389; 74, 500; 75, 667; 76, 611; 77, 889; 78, 722; 79, 722;
   80, 611; 81, 722; 82, 667; 83, 556; 84, 611; 85, 722; 86, 667; 87, 889;
   88, 667; 89, 611; 90, 611; 91, 333; 92, 278; 93, 333; 94, 570; 95, 500;
   96, 333; 97, 500; 98, 500; 99, 444; 100, 500; 101, 444; 102, 333; 103, 500;
   104, 556; 105, 278; 106, 278; 107, 500; 108, 278; 109, 778; 110, 556;
   111, 500; 112, 500; 113, 500; 114, 389; 115, 389; 116, 278; 117, 556;
   118, 444; 119, 667; 120, 500; 121, 444; 122, 389; 123, 348; 124, 220;
   125, 348; 126, 570; 161, 389; 162, 500; 163, 500; 164, 167; 165, 500;
   166, 500; 167, 500; 168, 500; 169, 278; 170, 500; 171, 500; 172, 333;
   173, 333; 174, 556; 175, 556; 177, 500; 178, 500; 179, 500; 180, 250;
   182, 500; 183, 350; 184, 333; 185, 500; 186, 500; 187, 500; 188, 1000;
   189, 1000; 191, 500; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 944; 227, 266; 232, 611; 233, 722; 234, 944;
   235, 300; 241, 722; 245, 278; 248, 278; 249, 500; 250, 722; 251, 500]

let times_bold_italic_kerns =
  hashtable_of_kerns
  [65, 121, -74; 65, 119, -74; 65, 118, -74; 65, 117, -30; 65, 39, -74;
   65, 186, 0; 65, 112, 0; 65, 89, -70; 65, 87, -100; 65, 86, -95; 65, 85, -50;
   65, 84, -55; 65, 81, -55; 65, 79, -50; 65, 71, -60; 65, 67, -65; 66, 46, 0;
   66, 44, 0; 66, 85, -10; 66, 65, -25; 68, 46, 0; 68, 44, 0; 68, 89, -50;
   68, 87, -40; 68, 86, -50; 68, 65, -25; 70, 114, -50; 70, 46, -129;
   70, 111, -70; 70, 105, -40; 70, 101, -100; 70, 44, -129; 70, 97, -95;
   70, 65, -100; 71, 46, 0; 71, 44, 0; 74, 117, -40; 74, 46, -10; 74, 111, -40;
   74, 101, -40; 74, 44, -10; 74, 97, -40; 74, 65, -25; 75, 121, -20;
   75, 117, -20; 75, 111, -25; 75, 101, -25; 75, 79, -30; 76, 121, -37;
   76, 39, -55; 76, 186, 0; 76, 89, -37; 76, 87, -37; 76, 86, -37; 76, 84, -18;
   78, 46, 0; 78, 44, 0; 78, 65, -30; 79, 46, 0; 79, 44, 0; 79, 89, -50;
   79, 88, -40; 79, 87, -50; 79, 86, -50; 79, 84, -40; 79, 65, -40;
   80, 46, -129; 80, 111, -55; 80, 101, -50; 80, 44, -129; 80, 97, -40;
   80, 65, -85; 81, 46, 0; 81, 44, 0; 81, 85, -10; 82, 89, -18; 82, 87, -18;
   82, 86, -18; 82, 85, -40; 82, 84, -30; 82, 79, -40; 83, 46, 0; 83, 44, 0;
   84, 121, -37; 84, 119, -37; 84, 117, -37; 84, 59, -74; 84, 114, -37;
   84, 46, -92; 84, 111, -95; 84, 105, -37; 84, 45, -92; 84, 104, 0;
   84, 101, -92; 84, 44, -92; 84, 58, -74; 84, 97, -92; 84, 79, -18;
   84, 65, -55; 85, 46, 0; 85, 44, 0; 85, 65, -45; 86, 117, -55; 86, 59, -74;
   86, 46, -129; 86, 111, -111; 86, 105, -55; 86, 45, -70; 86, 101, -111;
   86, 44, -129; 86, 58, -74; 86, 97, -111; 86, 79, -30; 86, 71, -10;
   86, 65, -85; 87, 121, -55; 87, 117, -55; 87, 59, -55; 87, 46, -74;
   87, 111, -80; 87, 105, -37; 87, 45, -50; 87, 104, 0; 87, 101, -90;
   87, 44, -74; 87, 58, -55; 87, 97, -85; 87, 79, -15; 87, 65, -74;
   89, 117, -92; 89, 59, -92; 89, 46, -74; 89, 111, -111; 89, 105, -55;
   89, 45, -92; 89, 101, -111; 89, 44, -92; 89, 58, -92; 89, 97, -92;
   89, 79, -25; 89, 65, -74; 97, 121, 0; 97, 119, 0; 97, 118, 0; 97, 116, 0;
   97, 112, 0; 97, 103, 0; 97, 98, 0; 98, 121, 0; 98, 118, 0; 98, 117, -20;
   98, 46, -40; 98, 108, 0; 98, 44, 0; 98, 98, -10; 99, 121, 0; 99, 46, 0;
   99, 108, 0; 99, 107, -10; 99, 104, -10; 99, 44, 0; 58, 32, 0; 44, 32, 0;
   44, 39, -95; 44, 186, -95; 100, 121, 0; 100, 119, 0; 100, 118, 0;
   100, 46, 0; 100, 100, 0; 100, 44, 0; 101, 121, 0; 101, 120, 0; 101, 119, 0;
   101, 118, 0; 101, 46, 0; 101, 112, 0; 101, 103, 0; 101, 44, 0; 101, 98, -10;
   102, 39, 55; 102, 186, 0; 102, 46, -10; 102, 111, -10; 102, 108, 0;
   102, 105, 0; 102, 102, -18; 102, 101, -10; 102, 245, -30; 102, 44, -10;
   102, 97, 0; 103, 121, 0; 103, 114, 0; 103, 46, 0; 103, 111, 0; 103, 105, 0;
   103, 103, 0; 103, 101, 0; 103, 44, 0; 103, 97, 0; 104, 121, 0; 105, 118, 0;
   107, 121, 0; 107, 111, -10; 107, 101, -30; 108, 121, 0; 108, 119, 0;
   109, 121, 0; 109, 117, 0; 110, 121, 0; 110, 118, -40; 110, 117, 0;
   111, 121, -10; 111, 120, -10; 111, 119, -25; 111, 118, -15; 111, 103, 0;
   112, 121, 0; 46, 39, -95; 46, 186, -95; 170, 96, 0; 170, 65, 0; 186, 32, 0;
   96, 96, -74; 96, 65, 0; 39, 118, -15; 39, 116, -37; 39, 32, -74;
   39, 115, -74; 39, 114, -15; 39, 39, -74; 39, 186, 0; 39, 108, 0;
   39, 100, -15; 114, 121, 0; 114, 118, 0; 114, 117, 0; 114, 116, 0;
   114, 115, 0; 114, 114, 0; 114, 113, 0; 114, 46, -65; 114, 112, 0;
   114, 111, 0; 114, 110, 0; 114, 109, 0; 114, 108, 0; 114, 107, 0;
   114, 105, 0; 114, 45, 0; 114, 103, 0; 114, 101, 0; 114, 100, 0; 114, 44, -65;
   114, 99, 0; 114, 97, 0; 115, 119, 0; 32, 96, 0; 32, 170, 0; 32, 89, -70;
   32, 87, -70; 32, 86, -70; 32, 84, 0; 32, 65, -37; 118, 46, -37;
   118, 111, -15; 118, 101, -15; 118, 44, -37; 118, 97, 0; 119, 46, -37;
   119, 111, -15; 119, 104, 0; 119, 101, -10; 119, 44, -37; 119, 97, -10;
   120, 101, -10; 121, 46, -37; 121, 111, 0; 121, 101, 0; 121, 44, -37;
   121, 97, 0; 122, 111, 0; 122, 101, 0]

(* Helvetica *)
let helvetica_widths =
  hashtable_of_dictionary
  [32, 278; 33, 278; 34, 355; 35, 556; 36, 556; 37, 889; 38, 667; 39, 222;
   40, 333; 41, 333; 42, 389; 43, 584; 44, 278; 45, 333; 46, 278; 47, 278;
   48, 556; 49, 556; 50, 556; 51, 556; 52, 556; 53, 556; 54, 556; 55, 556;
   56, 556; 57, 556; 58, 278; 59, 278; 60, 584; 61, 584; 62, 584; 63, 556;
   64, 1015; 65, 667; 66, 667; 67, 722; 68, 722; 69, 667; 70, 611; 71, 778;
   72, 722; 73, 278; 74, 500; 75, 667; 76, 556; 77, 833; 78, 722; 79, 778;
   80, 667; 81, 778; 82, 722; 83, 667; 84, 611; 85, 722; 86, 667; 87, 944;
   88, 667; 89, 667; 90, 611; 91, 278; 92, 278; 93, 278; 94, 469; 95, 556;
   96, 222; 97, 556; 98, 556; 99, 500; 100, 556; 101, 556; 102, 278; 103, 556;
   104, 556; 105, 222; 106, 222; 107, 500; 108, 222; 109, 833; 110, 556;
   111, 556; 112, 556; 113, 556; 114, 333; 115, 500; 116, 278; 117, 556;
   118, 500; 119, 722; 120, 500; 121, 500; 122, 500; 123, 334; 124, 260;
   125, 334; 126, 584; 161, 333; 162, 556; 163, 556; 164, 167; 165, 556;
   166, 556; 167, 556; 168, 556; 169, 191; 170, 333; 171, 556; 172, 333;
   173, 333; 174, 500; 175, 500; 177, 556; 178, 556; 179, 556; 180, 278;
   182, 537; 183, 350; 184, 222; 185, 333; 186, 333; 187, 556; 188, 1000;
   189, 1000; 191, 611; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 1000; 227, 370; 232, 556; 233, 778; 234, 1000;
   235, 365; 241, 889; 245, 278; 248, 222; 249, 611; 250, 944; 251, 611]

let helvetica_kerns =
  hashtable_of_kerns
  [65, 121, -40; 65, 119, -40; 65, 118, -40; 65, 117, -30; 65, 89, -100;
   65, 87, -50; 65, 86, -70; 65, 85, -50; 65, 84, -120; 65, 81, -30;
   65, 79, -30; 65, 71, -30; 65, 67, -30; 66, 46, -20; 66, 44, -20;
   66, 85, -10; 67, 46, -30; 67, 44, -30; 68, 46, -70; 68, 44, -70;
   68, 89, -90; 68, 87, -40; 68, 86, -70; 68, 65, -40; 70, 114, -45;
   70, 46, -150; 70, 111, -30; 70, 101, -30; 70, 44, -150; 70, 97, -50;
   70, 65, -80; 74, 117, -20; 74, 46, -30; 74, 44, -30; 74, 97, -20;
   74, 65, -20; 75, 121, -50; 75, 117, -30; 75, 111, -40; 75, 101, -40;
   75, 79, -50; 76, 121, -30; 76, 39, -160; 76, 186, -140; 76, 89, -140;
   76, 87, -70; 76, 86, -110; 76, 84, -110; 79, 46, -40; 79, 44, -40;
   79, 89, -70; 79, 88, -60; 79, 87, -30; 79, 86, -50; 79, 84, -40;
   79, 65, -20; 80, 46, -180; 80, 111, -50; 80, 101, -50; 80, 44, -180;
   80, 97, -40; 80, 65, -120; 81, 85, -10; 82, 89, -50; 82, 87, -30;
   82, 86, -50; 82, 85, -40; 82, 84, -30; 82, 79, -20; 83, 46, -20;
   83, 44, -20; 84, 121, -120; 84, 119, -120; 84, 117, -120; 84, 59, -20;
   84, 114, -120; 84, 46, -120; 84, 111, -120; 84, 45, -140; 84, 101, -120;
   84, 44, -120; 84, 58, -20; 84, 97, -120; 84, 79, -40; 84, 65, -120;
   85, 46, -40; 85, 44, -40; 85, 65, -40; 86, 117, -70; 86, 59, -40;
   86, 46, -125; 86, 111, -80; 86, 45, -80; 86, 101, -80; 86, 44, -125;
   86, 58, -40; 86, 97, -70; 86, 79, -40; 86, 71, -40; 86, 65, -80;
   87, 121, -20; 87, 117, -30; 87, 46, -80; 87, 111, -30; 87, 45, -40;
   87, 101, -30; 87, 44, -80; 87, 97, -40; 87, 79, -20; 87, 65, -50;
   89, 117, -110; 89, 59, -60; 89, 46, -140; 89, 111, -140; 89, 105, -20;
   89, 45, -140; 89, 101, -140; 89, 44, -140; 89, 58, -60; 89, 97, -140;
   89, 79, -85; 89, 65, -110; 97, 121, -30; 97, 119, -20; 97, 118, -20;
   98, 121, -20; 98, 118, -20; 98, 117, -20; 98, 46, -40; 98, 108, -20;
   98, 44, -40; 98, 98, -10; 99, 107, -20; 99, 44, -15; 58, 32, -50;
   44, 39, -100; 44, 186, -100; 101, 121, -20; 101, 120, -30; 101, 119, -20;
   101, 118, -30; 101, 46, -15; 101, 44, -15; 102, 39, 50; 102, 186, 60;
   102, 46, -30; 102, 111, -30; 102, 101, -30; 102, 245, -28; 102, 44, -30;
   102, 97, -30; 103, 114, -10; 104, 121, -30; 107, 111, -20; 107, 101, -20;
   109, 121, -15; 109, 117, -10; 110, 121, -15; 110, 118, -20; 110, 117, -10;
   111, 121, -30; 111, 120, -30; 111, 119, -15; 111, 118, -15; 111, 46, -40;
   111, 44, -40; 249, 122, -55; 249, 121, -70; 249, 120, -85; 249, 119, -70;
   249, 118, -70; 249, 117, -55; 249, 116, -55; 249, 115, -55; 249, 114, -55;
   249, 113, -55; 249, 46, -95; 249, 112, -55; 249, 111, -55; 249, 110, -55;
   249, 109, -55; 249, 108, -55; 249, 107, -55; 249, 106, -55; 249, 105, -55;
   249, 104, -55; 249, 103, -55; 249, 102, -55; 249, 101, -55; 249, 100, -55;
   249, 44, -95; 249, 99, -55; 249, 98, -55; 249, 97, -55; 112, 121, -30;
   112, 46, -35; 112, 44, -35; 46, 32, -60; 46, 39, -100; 46, 186, -100;
   186, 32, -40; 96, 96, -57; 39, 32, -70; 39, 115, -50; 39, 114, -50;
   39, 39, -57; 39, 100, -50; 114, 121, 30; 114, 118, 30; 114, 117, 15;
   114, 116, 40; 114, 59, 30; 114, 46, -50; 114, 112, 30; 114, 110, 25;
   114, 109, 25; 114, 108, 15; 114, 107, 15; 114, 105, 15; 114, 44, -50;
   114, 58, 30; 114, 97, -10; 115, 119, -30; 115, 46, -15; 115, 44, -15;
   59, 32, -50; 32, 96, -60; 32, 170, -30; 32, 89, -90; 32, 87, -40;
   32, 86, -50; 32, 84, -50; 118, 46, -80; 118, 111, -25; 118, 101, -25;
   118, 44, -80; 118, 97, -25; 119, 46, -60; 119, 111, -10; 119, 101, -10;
   119, 44, -60; 119, 97, -15; 120, 101, -30; 121, 46, -100; 121, 111, -20;
   121, 101, -20; 121, 44, -100; 121, 97, -20; 122, 111, -15; 122, 101, -15]
 
(* HelveticaBold *)
let helvetica_bold_widths =
  hashtable_of_dictionary
  [32, 278; 33, 333; 34, 474; 35, 556; 36, 556; 37, 889; 38, 722; 39, 278;
   40, 333; 41, 333; 42, 389; 43, 584; 44, 278; 45, 333; 46, 278; 47, 278;
   48, 556; 49, 556; 50, 556; 51, 556; 52, 556; 53, 556; 54, 556; 55, 556;
   56, 556; 57, 556; 58, 333; 59, 333; 60, 584; 61, 584; 62, 584; 63, 611;
   64, 975; 65, 722; 66, 722; 67, 722; 68, 722; 69, 667; 70, 611; 71, 778;
   72, 722; 73, 278; 74, 556; 75, 722; 76, 611; 77, 833; 78, 722; 79, 778;
   80, 667; 81, 778; 82, 722; 83, 667; 84, 611; 85, 722; 86, 667; 87, 944;
   88, 667; 89, 667; 90, 611; 91, 333; 92, 278; 93, 333; 94, 584; 95, 556;
   96, 278; 97, 556; 98, 611; 99, 556; 100, 611; 101, 556; 102, 333; 103, 611;
   104, 611; 105, 278; 106, 278; 107, 556; 108, 278; 109, 889; 110, 611;
   111, 611; 112, 611; 113, 611; 114, 389; 115, 556; 116, 333; 117, 611;
   118, 556; 119, 778; 120, 556; 121, 556; 122, 500; 123, 389; 124, 280;
   125, 389; 126, 584; 161, 333; 162, 556; 163, 556; 164, 167; 165, 556;
   166, 556; 167, 556; 168, 556; 169, 238; 170, 500; 171, 556; 172, 333;
   173, 333; 174, 611; 175, 611; 177, 556; 178, 556; 179, 556; 180, 278;
   182, 556; 183, 350; 184, 278; 185, 500; 186, 500; 187, 556; 188, 1000;
   189, 1000; 191, 611; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 1000; 227, 370; 232, 611; 233, 778; 234, 1000;
   235, 365; 241, 889; 245, 278; 248, 278; 249, 611; 250, 944; 251, 611]

let helvetica_bold_kerns =
  hashtable_of_kerns
  [65, 121, -30; 65, 119, -30; 65, 118, -40; 65, 117, -30; 65, 89, -110;
   65, 87, -60; 65, 86, -80; 65, 85, -50; 65, 84, -90; 65, 81, -40;
   65, 79, -40; 65, 71, -50; 65, 67, -40; 66, 85, -10; 66, 65, -30;
   68, 46, -30; 68, 44, -30; 68, 89, -70; 68, 87, -40; 68, 86, -40;
   68, 65, -40; 70, 46, -100; 70, 44, -100; 70, 97, -20; 70, 65, -80;
   74, 117, -20; 74, 46, -20; 74, 44, -20; 74, 65, -20; 75, 121, -40;
   75, 117, -30; 75, 111, -35; 75, 101, -15; 75, 79, -30; 76, 121, -30;
   76, 39, -140; 76, 186, -140; 76, 89, -120; 76, 87, -80; 76, 86, -110;
   76, 84, -90; 79, 46, -40; 79, 44, -40; 79, 89, -70; 79, 88, -50;
   79, 87, -50; 79, 86, -50; 79, 84, -40; 79, 65, -50; 80, 46, -120;
   80, 111, -40; 80, 101, -30; 80, 44, -120; 80, 97, -30; 80, 65, -100;
   81, 46, 20; 81, 44, 20; 81, 85, -10; 82, 89, -50; 82, 87, -40; 82, 86, -50;
   82, 85, -20; 82, 84, -20; 82, 79, -20; 84, 121, -60; 84, 119, -60;
   84, 117, -90; 84, 59, -40; 84, 114, -80; 84, 46, -80; 84, 111, -80;
   84, 45, -120; 84, 101, -60; 84, 44, -80; 84, 58, -40; 84, 97, -80;
   84, 79, -40; 84, 65, -90; 85, 46, -30; 85, 44, -30; 85, 65, -50;
   86, 117, -60; 86, 59, -40; 86, 46, -120; 86, 111, -90; 86, 45, -80;
   86, 101, -50; 86, 44, -120; 86, 58, -40; 86, 97, -60; 86, 79, -50;
   86, 71, -50; 86, 65, -80; 87, 121, -20; 87, 117, -45; 87, 59, -10;
   87, 46, -80; 87, 111, -60; 87, 45, -40; 87, 101, -35; 87, 44, -80;
   87, 58, -10; 87, 97, -40; 87, 79, -20; 87, 65, -60; 89, 117, -100; 
   89, 59, -50; 89, 46, -100; 89, 111, -100; 89, 101, -80; 89, 44, -100;
   89, 58, -50; 89, 97, -90; 89, 79, -70; 89, 65, -110; 97, 121, -20;
   97, 119, -15; 97, 118, -15; 97, 103, -10; 98, 121, -20; 98, 118, -20;
   98, 117, -20; 98, 108, -10; 99, 121, -10; 99, 108, -20; 99, 107, -20;
   99, 104, -10; 58, 32, -40; 44, 32, -40; 44, 39, -120; 44, 186, -120;
   100, 121, -15; 100, 119, -15; 100, 118, -15; 100, 100, -10; 101, 121, -15;
   101, 120, -15; 101, 119, -15; 101, 118, -15; 101, 46, 20; 101, 44, 10;
   102, 39, 30; 102, 186, 30; 102, 46, -10; 102, 111, -20; 102, 101, -10;
   102, 44, -10; 103, 103, -10; 103, 101, 10; 104, 121, -20; 107, 111, -15;
   108, 121, -15; 108, 119, -15; 109, 121, -30; 109, 117, -20; 110, 121, -20;
   110, 118, -40; 110, 117, -10; 111, 121, -20; 111, 120, -30; 111, 119, -15;
   111, 118, -20; 112, 121, -15; 46, 32, -40; 46, 39, -120; 46, 186, -120;
   186, 32, -80; 96, 96, -46; 39, 118, -20; 39, 32, -80; 39, 115, -60;
   39, 114, -40; 39, 39, -46; 39, 108, -20; 39, 100, -80; 114, 121, 10;
   114, 118, 10; 114, 116, 20; 114, 115, -15; 114, 113, -20; 114, 46, -60;
   114, 111, -20; 114, 45, -20; 114, 103, -15; 114, 100, -20; 114, 44, -60;
   114, 99, -20; 115, 119, -15; 59, 32, -40; 32, 96, -60; 32, 170, -80;
   32, 89, -120; 32, 87, -80; 32, 86, -80; 32, 84, -100; 118, 46, -80;
   118, 111, -30; 118, 44, -80; 118, 97, -20; 119, 46, -40; 119, 111, -20;
   119, 44, -40; 120, 101, -10; 121, 46, -80; 121, 111, -25; 121, 101, -10;
   121, 44, -80; 121, 97, -30; 122, 101, 10]

(* HelveticaOblique *)
let helvetica_oblique_widths =
  hashtable_of_dictionary
  [32, 278; 33, 278; 34, 355; 35, 556; 36, 556; 37, 889; 38, 667; 39, 222;
   40, 333; 41, 333; 42, 389; 43, 584; 44, 278; 45, 333; 46, 278; 47, 278;
   48, 556; 49, 556; 50, 556; 51, 556; 52, 556; 53, 556; 54, 556; 55, 556;
   56, 556; 57, 556; 58, 278; 59, 278; 60, 584; 61, 584; 62, 584; 63, 556;
   64, 1015; 65, 667; 66, 667; 67, 722; 68, 722; 69, 667; 70, 611; 71, 778;
   72, 722; 73, 278; 74, 500; 75, 667; 76, 556; 77, 833; 78, 722; 79, 778;
   80, 667; 81, 778; 82, 722; 83, 667; 84, 611; 85, 722; 86, 667; 87, 944;
   88, 667; 89, 667; 90, 611; 91, 278; 92, 278; 93, 278; 94, 469; 95, 556;
   96, 222; 97, 556; 98, 556; 99, 500; 100, 556; 101, 556; 102, 278; 103, 556;
   104, 556; 105, 222; 106, 222; 107, 500; 108, 222; 109, 833; 110, 556;
   111, 556; 112, 556; 113, 556; 114, 333; 115, 500; 116, 278; 117, 556;
   118, 500; 119, 722; 120, 500; 121, 500; 122, 500; 123, 334; 124, 260;
   125, 334; 126, 584; 161, 333; 162, 556; 163, 556; 164, 167; 165, 556;
   166, 556; 167, 556; 168, 556; 169, 191; 170, 333; 171, 556; 172, 333;
   173, 333; 174, 500; 175, 500; 177, 556; 178, 556; 179, 556; 180, 278;
   182, 537; 183, 350; 184, 222; 185, 333; 186, 333; 187, 556; 188, 1000;
   189, 1000; 191, 611; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 1000; 227, 370; 232, 556; 233, 778; 234, 1000;
   235, 365; 241, 889; 245, 278; 248, 222; 249, 611; 250, 944; 251, 611]

let helvetica_oblique_kerns =
  hashtable_of_kerns
  [65, 121, -40; 65, 119, -40; 65, 118, -40; 65, 117, -30; 65, 89, -100;
   65, 87, -50; 65, 86, -70; 65, 85, -50; 65, 84, -120; 65, 81, -30;
   65, 79, -30; 65, 71, -30; 65, 67, -30; 66, 46, -20; 66, 44, -20;
   66, 85, -10; 67, 46, -30; 67, 44, -30; 68, 46, -70; 68, 44, -70;
   68, 89, -90; 68, 87, -40; 68, 86, -70; 68, 65, -40; 70, 114, -45;
   70, 46, -150; 70, 111, -30; 70, 101, -30; 70, 44, -150; 70, 97, -50;
   70, 65, -80; 74, 117, -20; 74, 46, -30; 74, 44, -30; 74, 97, -20;
   74, 65, -20; 75, 121, -50; 75, 117, -30; 75, 111, -40; 75, 101, -40;
   75, 79, -50; 76, 121, -30; 76, 39, -160; 76, 186, -140; 76, 89, -140;
   76, 87, -70; 76, 86, -110; 76, 84, -110; 79, 46, -40; 79, 44, -40;
   79, 89, -70; 79, 88, -60; 79, 87, -30; 79, 86, -50; 79, 84, -40;
   79, 65, -20; 80, 46, -180; 80, 111, -50; 80, 101, -50; 80, 44, -180;
   80, 97, -40; 80, 65, -120; 81, 85, -10; 82, 89, -50; 82, 87, -30;
   82, 86, -50; 82, 85, -40; 82, 84, -30; 82, 79, -20; 83, 46, -20;
   83, 44, -20; 84, 121, -120; 84, 119, -120; 84, 117, -120; 84, 59, -20;
   84, 114, -120; 84, 46, -120; 84, 111, -120; 84, 45, -140; 84, 101, -120;
   84, 44, -120; 84, 58, -20; 84, 97, -120; 84, 79, -40; 84, 65, -120;
   85, 46, -40; 85, 44, -40; 85, 65, -40; 86, 117, -70; 86, 59, -40;
   86, 46, -125; 86, 111, -80; 86, 45, -80; 86, 101, -80; 86, 44, -125;
   86, 58, -40; 86, 97, -70; 86, 79, -40; 86, 71, -40; 86, 65, -80;
   87, 121, -20; 87, 117, -30; 87, 46, -80; 87, 111, -30; 87, 45, -40;
   87, 101, -30; 87, 44, -80; 87, 97, -40; 87, 79, -20; 87, 65, -50;
   89, 117, -110; 89, 59, -60; 89, 46, -140; 89, 111, -140; 89, 105, -20;
   89, 45, -140; 89, 101, -140; 89, 44, -140; 89, 58, -60; 89, 97, -140;
   89, 79, -85; 89, 65, -110; 97, 121, -30; 97, 119, -20; 97, 118, -20;
   98, 121, -20; 98, 118, -20; 98, 117, -20; 98, 46, -40; 98, 108, -20;
   98, 44, -40; 98, 98, -10; 99, 107, -20; 99, 44, -15; 58, 32, -50;
   44, 39, -100; 44, 186, -100; 101, 121, -20; 101, 120, -30; 101, 119, -20;
   101, 118, -30; 101, 46, -15; 101, 44, -15; 102, 39, 50; 102, 186, 60;
   102, 46, -30; 102, 111, -30; 102, 101, -30; 102, 245, -28; 102, 44, -30;
   102, 97, -30; 103, 114, -10; 104, 121, -30; 107, 111, -20; 107, 101, -20;
   109, 121, -15; 109, 117, -10; 110, 121, -15; 110, 118, -20; 110, 117, -10;
   111, 121, -30; 111, 120, -30; 111, 119, -15; 111, 118, -15; 111, 46, -40;
   111, 44, -40; 249, 122, -55; 249, 121, -70; 249, 120, -85; 249, 119, -70;
   249, 118, -70; 249, 117, -55; 249, 116, -55; 249, 115, -55; 249, 114, -55;
   249, 113, -55; 249, 46, -95; 249, 112, -55; 249, 111, -55; 249, 110, -55;
   249, 109, -55; 249, 108, -55; 249, 107, -55; 249, 106, -55; 249, 105, -55;
   249, 104, -55; 249, 103, -55; 249, 102, -55; 249, 101, -55; 249, 100, -55;
   249, 44, -95; 249, 99, -55; 249, 98, -55; 249, 97, -55; 112, 121, -30;
   112, 46, -35; 112, 44, -35; 46, 32, -60; 46, 39, -100; 46, 186, -100;
   186, 32, -40; 96, 96, -57; 39, 32, -70; 39, 115, -50; 39, 114, -50;
   39, 39, -57; 39, 100, -50; 114, 121, 30; 114, 118, 30; 114, 117, 15;
   114, 116, 40; 114, 59, 30; 114, 46, -50; 114, 112, 30; 114, 110, 25;
   114, 109, 25; 114, 108, 15; 114, 107, 15; 114, 105, 15; 114, 44, -50;
   114, 58, 30; 114, 97, -10; 115, 119, -30; 115, 46, -15; 115, 44, -15;
   59, 32, -50; 32, 96, -60; 32, 170, -30; 32, 89, -90; 32, 87, -40;
   32, 86, -50; 32, 84, -50; 118, 46, -80; 118, 111, -25; 118, 101, -25;
   118, 44, -80; 118, 97, -25; 119, 46, -60; 119, 111, -10; 119, 101, -10;
   119, 44, -60; 119, 97, -15; 120, 101, -30; 121, 46, -100; 121, 111, -20;
   121, 101, -20; 121, 44, -100; 121, 97, -20; 122, 111, -15; 122, 101, -15]

(* HelveticaBoldOblique *)
let helvetica_bold_oblique_widths =
  hashtable_of_dictionary
  [32, 278; 33, 333; 34, 474; 35, 556; 36, 556; 37, 889; 38, 722; 39, 278;
   40, 333; 41, 333; 42, 389; 43, 584; 44, 278; 45, 333; 46, 278; 47, 278;
   48, 556; 49, 556; 50, 556; 51, 556; 52, 556; 53, 556; 54, 556; 55, 556;
   56, 556; 57, 556; 58, 333; 59, 333; 60, 584; 61, 584; 62, 584; 63, 611;
   64, 975; 65, 722; 66, 722; 67, 722; 68, 722; 69, 667; 70, 611; 71, 778;
   72, 722; 73, 278; 74, 556; 75, 722; 76, 611; 77, 833; 78, 722; 79, 778;
   80, 667; 81, 778; 82, 722; 83, 667; 84, 611; 85, 722; 86, 667; 87, 944;
   88, 667; 89, 667; 90, 611; 91, 333; 92, 278; 93, 333; 94, 584; 95, 556;
   96, 278; 97, 556; 98, 611; 99, 556; 100, 611; 101, 556; 102, 333; 103, 611;
   104, 611; 105, 278; 106, 278; 107, 556; 108, 278; 109, 889; 110, 611;
   111, 611; 112, 611; 113, 611; 114, 389; 115, 556; 116, 333; 117, 611;
   118, 556; 119, 778; 120, 556; 121, 556; 122, 500; 123, 389; 124, 280;
   125, 389; 126, 584; 161, 333; 162, 556; 163, 556; 164, 167; 165, 556;
   166, 556; 167, 556; 168, 556; 169, 238; 170, 500; 171, 556; 172, 333;
   173, 333; 174, 611; 175, 611; 177, 556; 178, 556; 179, 556; 180, 278;
   182, 556; 183, 350; 184, 278; 185, 500; 186, 500; 187, 556; 188, 1000;
   189, 1000; 191, 611; 193, 333; 194, 333; 195, 333; 196, 333; 197, 333;
   198, 333; 199, 333; 200, 333; 202, 333; 203, 333; 205, 333; 206, 333;
   207, 333; 208, 1000; 225, 1000; 227, 370; 232, 611; 233, 778; 234, 1000;
   235, 365; 241, 889; 245, 278; 248, 278; 249, 611; 250, 944; 251, 611] 

let helvetica_bold_oblique_kerns =
  hashtable_of_kerns
  [65, 121, -30; 65, 119, -30; 65, 118, -40; 65, 117, -30; 65, 89, -110;
   65, 87, -60; 65, 86, -80; 65, 85, -50; 65, 84, -90; 65, 81, -40;
   65, 79, -40; 65, 71, -50; 65, 67, -40; 66, 85, -10; 66, 65, -30;
   68, 46, -30; 68, 44, -30; 68, 89, -70; 68, 87, -40; 68, 86, -40;
   68, 65, -40; 70, 46, -100; 70, 44, -100; 70, 97, -20; 70, 65, -80;
   74, 117, -20; 74, 46, -20; 74, 44, -20; 74, 65, -20; 75, 121, -40;
   75, 117, -30; 75, 111, -35; 75, 101, -15; 75, 79, -30; 76, 121, -30;
   76, 39, -140; 76, 186, -140; 76, 89, -120; 76, 87, -80; 76, 86, -110;
   76, 84, -90; 79, 46, -40; 79, 44, -40; 79, 89, -70; 79, 88, -50;
   79, 87, -50; 79, 86, -50; 79, 84, -40; 79, 65, -50; 80, 46, -120;
   80, 111, -40; 80, 101, -30; 80, 44, -120; 80, 97, -30; 80, 65, -100;
   81, 46, 20; 81, 44, 20; 81, 85, -10; 82, 89, -50; 82, 87, -40; 82, 86, -50;
   82, 85, -20; 82, 84, -20; 82, 79, -20; 84, 121, -60; 84, 119, -60;
   84, 117, -90; 84, 59, -40; 84, 114, -80; 84, 46, -80; 84, 111, -80;
   84, 45, -120; 84, 101, -60; 84, 44, -80; 84, 58, -40; 84, 97, -80;
   84, 79, -40; 84, 65, -90; 85, 46, -30; 85, 44, -30; 85, 65, -50;
   86, 117, -60; 86, 59, -40; 86, 46, -120; 86, 111, -90; 86, 45, -80;
   86, 101, -50; 86, 44, -120; 86, 58, -40; 86, 97, -60; 86, 79, -50;
   86, 71, -50; 86, 65, -80; 87, 121, -20; 87, 117, -45; 87, 59, -10;
   87, 46, -80; 87, 111, -60; 87, 45, -40; 87, 101, -35; 87, 44, -80;
   87, 58, -10; 87, 97, -40; 87, 79, -20; 87, 65, -60; 89, 117, -100;
   89, 59, -50; 89, 46, -100; 89, 111, -100; 89, 101, -80; 89, 44, -100;
   89, 58, -50; 89, 97, -90; 89, 79, -70; 89, 65, -110; 97, 121, -20;
   97, 119, -15; 97, 118, -15; 97, 103, -10; 98, 121, -20; 98, 118, -20;
   98, 117, -20; 98, 108, -10; 99, 121, -10; 99, 108, -20; 99, 107, -20;
   99, 104, -10; 58, 32, -40; 44, 32, -40; 44, 39, -120; 44, 186, -120;
   100, 121, -15; 100, 119, -15; 100, 118, -15; 100, 100, -10; 101, 121, -15;
   101, 120, -15; 101, 119, -15; 101, 118, -15; 101, 46, 20; 101, 44, 10;
   102, 39, 30; 102, 186, 30; 102, 46, -10; 102, 111, -20; 102, 101, -10;
   102, 44, -10; 103, 103, -10; 103, 101, 10; 104, 121, -20; 107, 111, -15;
   108, 121, -15; 108, 119, -15; 109, 121, -30; 109, 117, -20; 110, 121, -20;
   110, 118, -40; 110, 117, -10; 111, 121, -20; 111, 120, -30; 111, 119, -15;
   111, 118, -20; 112, 121, -15; 46, 32, -40; 46, 39, -120; 46, 186, -120;
   186, 32, -80; 96, 96, -46; 39, 118, -20; 39, 32, -80; 39, 115, -60;
   39, 114, -40; 39, 39, -46; 39, 108, -20; 39, 100, -80; 114, 121, 10;
   114, 118, 10; 114, 116, 20; 114, 115, -15; 114, 113, -20; 114, 46, -60;
   114, 111, -20; 114, 45, -20; 114, 103, -15; 114, 100, -20; 114, 44, -60;
   114, 99, -20; 115, 119, -15; 59, 32, -40; 32, 96, -60; 32, 170, -80;
   32, 89, -120; 32, 87, -80; 32, 86, -80; 32, 84, -100; 118, 46, -80;
   118, 111, -30; 118, 44, -80; 118, 97, -20; 119, 46, -40; 119, 111, -20;
   119, 44, -40; 120, 101, -10; 121, 46, -80; 121, 111, -25; 121, 101, -10;
   121, 44, -80; 121, 97, -30; 122, 101, 10]

(* Courier *)
let courier_widths =
  hashtable_of_dictionary
  [32, 600; 33, 600; 34, 600; 35, 600; 36, 600; 37, 600; 38, 600; 39, 600;
   40, 600; 41, 600; 42, 600; 43, 600; 44, 600; 45, 600; 46, 600; 47, 600;
   48, 600; 49, 600; 50, 600; 51, 600; 52, 600; 53, 600; 54, 600; 55, 600;
   56, 600; 57, 600; 58, 600; 59, 600; 60, 600; 61, 600; 62, 600; 63, 600;
   64, 600; 65, 600; 66, 600; 67, 600; 68, 600; 69, 600; 70, 600; 71, 600;
   72, 600; 73, 600; 74, 600; 75, 600; 76, 600; 77, 600; 78, 600; 79, 600;
   80, 600; 81, 600; 82, 600; 83, 600; 84, 600; 85, 600; 86, 600; 87, 600;
   88, 600; 89, 600; 90, 600; 91, 600; 92, 600; 93, 600; 94, 600; 95, 600;
   96, 600; 97, 600; 98, 600; 99, 600; 100, 600; 101, 600; 102, 600; 103, 600;
   104, 600; 105, 600; 106, 600; 107, 600; 108, 600; 109, 600; 110, 600;
   111, 600; 112, 600; 113, 600; 114, 600; 115, 600; 116, 600; 117, 600;
   118, 600; 119, 600; 120, 600; 121, 600; 122, 600; 123, 600; 124, 600;
   125, 600; 126, 600; 161, 600; 162, 600; 163, 600; 164, 600; 165, 600;
   166, 600; 167, 600; 168, 600; 169, 600; 170, 600; 171, 600; 172, 600;
   173, 600; 174, 600; 175, 600; 177, 600; 178, 600; 179, 600; 180, 600;
   182, 600; 183, 600; 184, 600; 185, 600; 186, 600; 187, 600; 188, 600;
   189, 600; 191, 600; 193, 600; 194, 600; 195, 600; 196, 600; 197, 600;
   198, 600; 199, 600; 200, 600; 202, 600; 203, 600; 205, 600; 206, 600;
   207, 600; 208, 600; 225, 600; 227, 600; 232, 600; 233, 600; 234, 600;
   235, 600; 241, 600; 245, 600; 248, 600; 249, 600; 250, 600; 251, 600]

let courier_kerns = hashtable_of_kerns []

(* CourierBold *)
let courier_bold_widths =
  hashtable_of_dictionary
  [32, 600; 33, 600; 34, 600; 35, 600; 36, 600; 37, 600; 38, 600; 39, 600;
   40, 600; 41, 600; 42, 600; 43, 600; 44, 600; 45, 600; 46, 600; 47, 600;
   48, 600; 49, 600; 50, 600; 51, 600; 52, 600; 53, 600; 54, 600; 55, 600;
   56, 600; 57, 600; 58, 600; 59, 600; 60, 600; 61, 600; 62, 600; 63, 600;
   64, 600; 65, 600; 66, 600; 67, 600; 68, 600; 69, 600; 70, 600; 71, 600;
   72, 600; 73, 600; 74, 600; 75, 600; 76, 600; 77, 600; 78, 600; 79, 600;
   80, 600; 81, 600; 82, 600; 83, 600; 84, 600; 85, 600; 86, 600; 87, 600;
   88, 600; 89, 600; 90, 600; 91, 600; 92, 600; 93, 600; 94, 600; 95, 600;
   96, 600; 97, 600; 98, 600; 99, 600; 100, 600; 101, 600; 102, 600; 103, 600;
   104, 600; 105, 600; 106, 600; 107, 600; 108, 600; 109, 600; 110, 600;
   111, 600; 112, 600; 113, 600; 114, 600; 115, 600; 116, 600; 117, 600;
   118, 600; 119, 600; 120, 600; 121, 600; 122, 600; 123, 600; 124, 600;
   125, 600; 126, 600; 161, 600; 162, 600; 163, 600; 164, 600; 165, 600;
   166, 600; 167, 600; 168, 600; 169, 600; 170, 600; 171, 600; 172, 600;
   173, 600; 174, 600; 175, 600; 177, 600; 178, 600; 179, 600; 180, 600;
   182, 600; 183, 600; 184, 600; 185, 600; 186, 600; 187, 600; 188, 600;
   189, 600; 191, 600; 193, 600; 194, 600; 195, 600; 196, 600; 197, 600;
   198, 600; 199, 600; 200, 600; 202, 600; 203, 600; 205, 600; 206, 600;
   207, 600; 208, 600; 225, 600; 227, 600; 232, 600; 233, 600; 234, 600;
   235, 600; 241, 600; 245, 600; 248, 600; 249, 600; 250, 600; 251, 600]

let courier_bold_kerns = hashtable_of_kerns []

(* CourierOblique *)
let courier_oblique_widths =
  hashtable_of_dictionary
  [32, 600; 33, 600; 34, 600; 35, 600; 36, 600; 37, 600; 38, 600; 39, 600;
   40, 600; 41, 600; 42, 600; 43, 600; 44, 600; 45, 600; 46, 600; 47, 600;
   48, 600; 49, 600; 50, 600; 51, 600; 52, 600; 53, 600; 54, 600; 55, 600;
   56, 600; 57, 600; 58, 600; 59, 600; 60, 600; 61, 600; 62, 600; 63, 600;
   64, 600; 65, 600; 66, 600; 67, 600; 68, 600; 69, 600; 70, 600; 71, 600;
   72, 600; 73, 600; 74, 600; 75, 600; 76, 600; 77, 600; 78, 600; 79, 600;
   80, 600; 81, 600; 82, 600; 83, 600; 84, 600; 85, 600; 86, 600; 87, 600;
   88, 600; 89, 600; 90, 600; 91, 600; 92, 600; 93, 600; 94, 600; 95, 600;
   96, 600; 97, 600; 98, 600; 99, 600; 100, 600; 101, 600; 102, 600; 103, 600;
   104, 600; 105, 600; 106, 600; 107, 600; 108, 600; 109, 600; 110, 600;
   111, 600; 112, 600; 113, 600; 114, 600; 115, 600; 116, 600; 117, 600;
   118, 600; 119, 600; 120, 600; 121, 600; 122, 600; 123, 600; 124, 600; 
   125, 600; 126, 600; 161, 600; 162, 600; 163, 600; 164, 600; 165, 600;
   166, 600; 167, 600; 168, 600; 169, 600; 170, 600; 171, 600; 172, 600;
   173, 600; 174, 600; 175, 600; 177, 600; 178, 600; 179, 600; 180, 600;
   182, 600; 183, 600; 184, 600; 185, 600; 186, 600; 187, 600; 188, 600;
   189, 600; 191, 600; 193, 600; 194, 600; 195, 600; 196, 600; 197, 600;
   198, 600; 199, 600; 200, 600; 202, 600; 203, 600; 205, 600; 206, 600;
   207, 600; 208, 600; 225, 600; 227, 600; 232, 600; 233, 600; 234, 600;
   235, 600; 241, 600; 245, 600; 248, 600; 249, 600; 250, 600; 251, 600]

let courier_oblique_kerns = hashtable_of_kerns []

(* CourierBoldOblique *)
let courier_bold_oblique_widths =
  hashtable_of_dictionary
  [32, 600; 33, 600; 34, 600; 35, 600; 36, 600; 37, 600; 38, 600; 39, 600;
   40, 600; 41, 600; 42, 600; 43, 600; 44, 600; 45, 600; 46, 600; 47, 600;
   48, 600; 49, 600; 50, 600; 51, 600; 52, 600; 53, 600; 54, 600; 55, 600;
   56, 600; 57, 600; 58, 600; 59, 600; 60, 600; 61, 600; 62, 600; 63, 600;
   64, 600; 65, 600; 66, 600; 67, 600; 68, 600; 69, 600; 70, 600; 71, 600;
   72, 600; 73, 600; 74, 600; 75, 600; 76, 600; 77, 600; 78, 600; 79, 600;
   80, 600; 81, 600; 82, 600; 83, 600; 84, 600; 85, 600; 86, 600; 87, 600;
   88, 600; 89, 600; 90, 600; 91, 600; 92, 600; 93, 600; 94, 600; 95, 600;
   96, 600; 97, 600; 98, 600; 99, 600; 100, 600; 101, 600; 102, 600; 103, 600;
   104, 600; 105, 600; 106, 600; 107, 600; 108, 600; 109, 600; 110, 600;
   111, 600; 112, 600; 113, 600; 114, 600; 115, 600; 116, 600; 117, 600;
   118, 600; 119, 600; 120, 600; 121, 600; 122, 600; 123, 600; 124, 600;
   125, 600; 126, 600; 161, 600; 162, 600; 163, 600; 164, 600; 165, 600;
   166, 600; 167, 600; 168, 600; 169, 600; 170, 600; 171, 600; 172, 600;
   173, 600; 174, 600; 175, 600; 177, 600; 178, 600; 179, 600; 180, 600;
   182, 600; 183, 600; 184, 600; 185, 600; 186, 600; 187, 600; 188, 600;
   189, 600; 191, 600; 193, 600; 194, 600; 195, 600; 196, 600; 197, 600;
   198, 600; 199, 600; 200, 600; 202, 600; 203, 600; 205, 600; 206, 600;
   207, 600; 208, 600; 225, 600; 227, 600; 232, 600; 233, 600; 234, 600;
   235, 600; 241, 600; 245, 600; 248, 600; 249, 600; 250, 600; 251, 600]

let courier_bold_oblique_kerns = hashtable_of_kerns []

(* Symbol *)
let symbol_widths =
  hashtable_of_dictionary
  [32, 250; 33, 333; 34, 713; 35, 500; 36, 549; 37, 833; 38, 778; 39, 439;
   40, 333; 41, 333; 42, 500; 43, 549; 44, 250; 45, 549; 46, 250; 47, 278;
   48, 500; 49, 500; 50, 500; 51, 500; 52, 500; 53, 500; 54, 500; 55, 500;
   56, 500; 57, 500; 58, 278; 59, 278; 60, 549; 61, 549; 62, 549; 63, 444;
   64, 549; 65, 722; 66, 667; 67, 722; 68, 612; 69, 611; 70, 763; 71, 603;
   72, 722; 73, 333; 74, 631; 75, 722; 76, 686; 77, 889; 78, 722; 79, 722;
   80, 768; 81, 741; 82, 556; 83, 592; 84, 611; 85, 690; 86, 439; 87, 768;
   88, 645; 89, 795; 90, 611; 91, 333; 92, 863; 93, 333; 94, 658; 95, 500;
   96, 500; 97, 631; 98, 549; 99, 549; 100, 494; 101, 439; 102, 521; 103, 411;
   104, 603; 105, 329; 106, 603; 107, 549; 108, 549; 109, 576; 110, 521;
   111, 549; 112, 549; 113, 521; 114, 549; 115, 603; 116, 439; 117, 576;
   118, 713; 119, 686; 120, 493; 121, 686; 122, 494; 123, 480; 124, 200;
   125, 480; 126, 549; 161, 620; 162, 247; 163, 549; 164, 167; 165, 713;
   166, 500; 167, 753; 168, 753; 169, 753; 170, 753; 171, 1042; 172, 987;
   173, 603; 174, 987; 175, 603; 176, 400; 177, 549; 178, 411; 179, 549;
   180, 549; 181, 713; 182, 494; 183, 460; 184, 549; 185, 549; 186, 549;
   187, 549; 188, 1000; 189, 603; 190, 1000; 191, 658; 192, 823; 193, 686;
   194, 795; 195, 987; 196, 768; 197, 768; 198, 823; 199, 768; 200, 768;
   201, 713; 202, 713; 203, 713; 204, 713; 205, 713; 206, 713; 207, 713;
   208, 768; 209, 713; 210, 790; 211, 790; 212, 890; 213, 823; 214, 549;
   215, 250; 216, 713; 217, 603; 218, 603; 219, 1042; 220, 987; 221, 603;
   222, 987; 223, 603; 224, 494; 225, 329; 226, 790; 227, 790; 228, 786;
   229, 713; 230, 384; 231, 384; 232, 384; 233, 384; 234, 384; 235, 384;
   236, 494; 237, 494; 238, 494; 239, 494; 241, 329; 242, 274; 243, 686;
   244, 686; 245, 686; 246, 384; 247, 384; 248, 384; 249, 384; 250, 384;
   251, 384; 252, 494; 253, 494; 254, 494]

let symbol_kerns = hashtable_of_kerns []

(* ZapfDingbats *)
let zapfdingbats_widths =
  hashtable_of_dictionary
  [32, 278; 33, 974; 34, 961; 35, 974; 36, 980; 37, 719; 38, 789; 39, 790;
   40, 791; 41, 690; 42, 960; 43, 939; 44, 549; 45, 855; 46, 911; 47, 933;
   48, 911; 49, 945; 50, 974; 51, 755; 52, 846; 53, 762; 54, 761; 55, 571;
   56, 677; 57, 763; 58, 760; 59, 759; 60, 754; 61, 494; 62, 552; 63, 537;
   64, 577; 65, 692; 66, 786; 67, 788; 68, 788; 69, 790; 70, 793; 71, 794;
   72, 816; 73, 823; 74, 789; 75, 841; 76, 823; 77, 833; 78, 816; 79, 831;
   80, 923; 81, 744; 82, 723; 83, 749; 84, 790; 85, 792; 86, 695; 87, 776;
   88, 768; 89, 792; 90, 759; 91, 707; 92, 708; 93, 682; 94, 701; 95, 826;
   96, 815; 97, 789; 98, 789; 99, 707; 100, 687; 101, 696; 102, 689; 103, 786;
   104, 787; 105, 713; 106, 791; 107, 785; 108, 791; 109, 873; 110, 761;
   111, 762; 112, 762; 113, 759; 114, 759; 115, 892; 116, 892; 117, 788;
   118, 784; 119, 438; 120, 138; 121, 277; 122, 415; 123, 392; 124, 392;
   125, 668; 126, 668; 161, 732; 162, 544; 163, 544; 164, 910; 165, 667;
   166, 760; 167, 760; 168, 776; 169, 595; 170, 694; 171, 626; 172, 788;
   173, 788; 174, 788; 175, 788; 176, 788; 177, 788; 178, 788; 179, 788;
   180, 788; 181, 788; 182, 788; 183, 788; 184, 788; 185, 788; 186, 788;
   187, 788; 188, 788; 189, 788; 190, 788; 191, 788; 192, 788; 193, 788;
   194, 788; 195, 788; 196, 788; 197, 788; 198, 788; 199, 788; 200, 788;
   201, 788; 202, 788; 203, 788; 204, 788; 205, 788; 206, 788; 207, 788;
   208, 788; 209, 788; 210, 788; 211, 788; 212, 894; 213, 838; 214, 1016;
   215, 458; 216, 748; 217, 924; 218, 748; 219, 918; 220, 927; 221, 928;
   222, 928; 223, 834; 224, 873; 225, 828; 226, 924; 227, 924; 228, 917;
   229, 930; 230, 931; 231, 463; 232, 883; 233, 836; 234, 836; 235, 867;
   236, 867; 237, 696; 238, 696; 239, 874; 241, 874; 242, 760; 243, 946;
   244, 771; 245, 865; 246, 771; 247, 888; 248, 967; 249, 888; 250, 831;
   251, 873; 252, 927; 253, 970; 254, 918]

let zapfdingbats_kerns = hashtable_of_kerns []

(* Main functions *)
type widths = (int, int) Hashtbl.t

type kerns = ((int * int),  int) Hashtbl.t

type table =
  Pdftext.standard_font * widths * kerns

let tables =
  [Pdftext.TimesRoman, (times_roman_widths, times_roman_kerns);
   Pdftext.TimesBold, (times_bold_widths, times_bold_kerns);
   Pdftext.TimesItalic, (times_italic_widths, times_italic_kerns);
   Pdftext.TimesBoldItalic, (times_bold_italic_widths, times_bold_italic_kerns);
   Pdftext.Helvetica, (helvetica_widths, helvetica_kerns);
   Pdftext.HelveticaBold, (helvetica_bold_widths, helvetica_bold_kerns);
   Pdftext.HelveticaOblique, (helvetica_oblique_widths, helvetica_oblique_kerns);
   Pdftext.HelveticaBoldOblique, (helvetica_bold_oblique_widths, helvetica_bold_oblique_kerns);
   Pdftext.Courier, (courier_widths, courier_kerns);
   Pdftext.CourierBold, (courier_bold_widths, courier_bold_kerns);
   Pdftext.CourierOblique, (courier_oblique_widths, courier_oblique_kerns);
   Pdftext.CourierBoldOblique, (courier_bold_oblique_widths, courier_bold_oblique_kerns);
   Pdftext.Symbol, (symbol_widths, symbol_kerns);
   Pdftext.ZapfDingbats, (zapfdingbats_widths, zapfdingbats_kerns)]

(* Calculate the width of a list of characters, taking account of kerning. *)
let find_kern kerns key =
  try Hashtbl.find kerns key with Not_found -> 0

let find_width widths h =
  try Hashtbl.find widths h with Not_found -> 0

let rec width widths kerns = function
  | [] -> 0
  | [h] -> find_width widths h
  | h::h'::t ->
      find_width widths h +
      find_kern kerns (h, h') +
      width widths kerns (h'::t)

(* \intf The main function. Give a font and the text string. *)
let textwidth f s =
  let widths, kerns = lookup_failnull f tables in
    width widths kerns (map int_of_char (explode s))

