/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 *               2004 Jean-François Wauthy (pollux@xfce.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#include "input_page.h"

static void input_page_class_init (InputPageClass *);
static void input_page_init (InputPage *);
static void input_page_finalize (GObject *);

/* signal handlers */
static void toggled_cb (GtkWidget *, InputPage *);
static void valuechanged_cb (GtkWidget *, InputPage *);

static GtkWidgetClass *parent_class = NULL;

GtkType
input_page_get_type (void)
{
  static GtkType input_page_type = 0;

  if (!input_page_type) {
    static const GTypeInfo input_page_info = {
      sizeof (InputPageClass),
      NULL,
      NULL,
      (GClassInitFunc) input_page_class_init,
      NULL,
      NULL,
      sizeof (InputPage),
      0,
      (GInstanceInitFunc) input_page_init,
      NULL
    };

    input_page_type = g_type_register_static (GTK_TYPE_VBOX, "InputPage", &input_page_info, 0);
  }

  return (input_page_type);
}

static void
input_page_class_init (InputPageClass * klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = input_page_finalize;

  parent_class = gtk_type_class (gtk_vbox_get_type ());
}

static void
input_page_init (InputPage * pg)
{
  GtkWidget *label;
  GtkWidget *hbox;

  gtk_container_set_border_width (GTK_CONTAINER (pg), 6);
  gtk_box_set_spacing (GTK_BOX (pg), 4);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (pg), hbox, FALSE, TRUE, 0);

  label = gtk_label_new (_("Encoding:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  pg->encoding = gtk_combo_new ();
  gtk_widget_show (pg->encoding);
  gtk_box_pack_start (GTK_BOX (hbox), pg->encoding, TRUE, TRUE, 0);

  pg->all = gtk_radio_button_new_with_label (NULL, _("Print all pages"));
  gtk_widget_show (pg->all);
  g_signal_connect (G_OBJECT (pg->all), "toggled", G_CALLBACK (toggled_cb), pg);
  gtk_box_pack_start (GTK_BOX (pg), pg->all, FALSE, TRUE, 0);

  pg->range = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (pg->all), _("Print pages in range from"));
  gtk_widget_show (pg->range);
  gtk_box_pack_start (GTK_BOX (pg), pg->range, FALSE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (pg), hbox, FALSE, TRUE, 0);

  pg->from = gtk_spin_button_new_with_range (1.0, 1000.0, 1.0);
  gtk_widget_set_sensitive (pg->from, FALSE);
  gtk_widget_show (pg->from);
  g_signal_connect (G_OBJECT (pg->from), "value-changed", G_CALLBACK (valuechanged_cb), pg);
  gtk_box_pack_start (GTK_BOX (hbox), pg->from, FALSE, TRUE, 0);

  label = gtk_label_new ("-");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  pg->to = gtk_spin_button_new_with_range (1.0, 1000.0, 1.0);
  gtk_widget_set_sensitive (pg->to, FALSE);
  gtk_widget_show (pg->to);
  g_signal_connect (G_OBJECT (pg->to), "value-changed", G_CALLBACK (valuechanged_cb), pg);
  gtk_box_pack_start (GTK_BOX (hbox), pg->to, FALSE, TRUE, 0);

  pg->cut = gtk_check_button_new_with_label (_("Cut long lines"));
  gtk_widget_show (pg->cut);
  gtk_box_pack_start (GTK_BOX (pg), pg->cut, FALSE, TRUE, 0);

  pg->interpret = gtk_check_button_new_with_label (_("Interpret TAB, BS and FF characters"));
  gtk_widget_show (pg->interpret);
  gtk_box_pack_start (GTK_BOX (pg), pg->interpret, FALSE, TRUE, 0);

  pg->binary = gtk_check_button_new_with_label (_("Force binary printing"));
  gtk_widget_show (pg->binary);
  gtk_box_pack_start (GTK_BOX (pg), pg->binary, FALSE, TRUE, 0);
}

static void
input_page_finalize (GObject * object)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (INPUT_IS_PAGE (object));

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
input_page_new (void)
{
  InputPage *pg;
  GList *list;

  pg = INPUT_PAGE (g_object_new (input_page_get_type (), NULL));

  if ((list = xfprint_option_list (encodings)) != NULL) {
    gtk_combo_set_popdown_strings (GTK_COMBO (pg->encoding), list);
    g_list_free (list);
  }

  return (GTK_WIDGET (pg));
}

void
input_page_set_settings (InputPage * pg, const XfprintSettingsInput * ip)
{
  g_return_if_fail (pg != NULL);
  g_return_if_fail (ip != NULL);

  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (pg->encoding)->entry), xfprint_option_alias (encodings, ip->encoding));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ip->all ? pg->all : pg->range), TRUE);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (pg->from), ip->from);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (pg->to), ip->to);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pg->cut), ip->cut);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pg->interpret), ip->interpret);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pg->binary), ip->binary);
}

void
input_page_get_settings (InputPage * pg, XfprintSettingsInput * ip)
{
  g_return_if_fail (pg != NULL);
  g_return_if_fail (ip != NULL);

  ip->encoding = xfprint_option_name (encodings, gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (pg->encoding)->entry)));

  ip->all = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pg->all));
  ip->from = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (pg->from));
  ip->to = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (pg->to));

  ip->cut = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pg->cut));
  ip->interpret = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pg->interpret));
  ip->binary = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pg->binary));
}

static void
toggled_cb (GtkWidget * btn, InputPage * pg)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pg->all))) {
    gtk_widget_set_sensitive (pg->from, FALSE);
    gtk_widget_set_sensitive (pg->to, FALSE);
  }
  else {
    gtk_widget_set_sensitive (pg->from, TRUE);
    gtk_widget_set_sensitive (pg->to, TRUE);
  }
}

static void
valuechanged_cb (GtkWidget * btn, InputPage * pg)
{
  gint from;
  gint to;

  from = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (pg->from));
  to = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (pg->to));

  if (from <= to)
    return;

  if (btn == pg->from)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (pg->to), from);
  else if (btn == pg->to)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (pg->from), to);
}
