#include <check.h>
#include <utouch/frame-mtdev.h>
#include <grail.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

struct test_data {
	struct evemu_device *evemu;
	utouch_frame_handle fh;
	grail_handle ge;
	FILE *fp;
	struct timeval evtime;
};

static int near(const struct grail_coord *p, float x, float y)
{
	double dx = p->x - x;
	double dy = p->y - y;
	return dx * dx + dy * dy < 1e-8 * (x * x + y * y);
}

static int init_test_data(struct test_data *data, const char *path)
{
	memset(data, 0, sizeof(*data));

	data->fp = fopen(path, "r");
	if (!data->fp)
		return -1;

	data->evemu = evemu_new(NULL);
	if (!data->evemu || evemu_read(data->evemu, data->fp) <= 0)
		return -1;

	data->fh = utouch_frame_new_engine(100, 32, 100);
	if (!data->fh || utouch_frame_init_mtdev(data->fh, data->evemu))
		return -1;

	data->ge = grail_new(data->fh, 10, 0);
	if (!data->ge)
		return -1;

	return 0;
}

static void term_test_data(struct test_data *data)
{
	grail_delete(data->ge);
	utouch_frame_delete_engine(data->fh);
	evemu_delete(data->evemu);
	fclose(data->fp);
}

static int get_event(struct test_data *data, struct input_event *ev)
{
	return evemu_read_event_realtime(data->fp, ev, &data->evtime) > 0;
}

static const struct grail_element *get_element(struct test_data *data,
					       const struct input_event *ev)
{
	const struct utouch_frame *touch;
	const struct grail_frame *frame;

	touch = utouch_frame_pump_mtdev(data->fh, ev);
	if (!touch)
		return 0;
	frame = grail_pump_frame(data->ge, touch);
	if (!frame || !frame->num_ongoing)
		return 0;

	return frame->ongoing[frame->num_ongoing - 1];
}

START_TEST(device_coordinates)
{
	struct test_data data;
	struct input_event ev;
	const struct grail_element *slot;
	struct grail_coord pos = { 0, 0 };

	fail_if(init_test_data(&data, "io/evemu/one-tap.evemu"));

	while (get_event(&data, &ev)) {
		slot = get_element(&data, &ev);
		if (slot)
			pos = slot->center;
	}

	fail_unless(near(&pos, 17087, 14790));

	term_test_data(&data);
}
END_TEST

START_TEST(mapped_coordinates)
{
	static const struct grail_coord min = { 0, 0 }, max = { 1.6, 1 };
	struct test_data data;
	struct input_event ev;
	const struct grail_element *slot;
	struct grail_coord pos = { 0, 0 };

	fail_if(init_test_data(&data, "io/evemu/one-tap.evemu"));
	grail_set_bbox(data.ge, &min, &max);

	while (get_event(&data, &ev)) {
		slot = get_element(&data, &ev);
		if (slot)
			pos = slot->center;
	}

	fail_unless(near(&pos, 0.834303, 0.451338));

	term_test_data(&data);
}
END_TEST

Suite *make_transform_suite()
{
	Suite *s = suite_create("grail-solver");
	TCase *test;

	test = tcase_create("device_coordinates");
	tcase_add_test(test, device_coordinates);
	suite_add_tcase(s, test);

	test = tcase_create("mapped_coordinates");
	tcase_add_test(test, mapped_coordinates);
	suite_add_tcase(s, test);

	return s;
}
