#include <check.h>
#include <grail.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

struct test_data {
	struct grail ge;
	int fd;
	struct grail_coord pos;
};

static int near(const struct grail_coord *p, float x, float y)
{
	double dx = p->x - x;
	double dy = p->y - y;
	return dx * dx + dy * dy < 1e-8 * (x * x + y * y);
}

static int tp_get_clients(struct grail *ge,
			  struct grail_client_info *clients, int max_clients,
			  const struct grail_coord *coords, int num_coords,
			  const grail_mask_t *types, int type_bytes)
{
	memset(&clients[0], 0, sizeof(clients[0]));
	memset(clients[0].mask, ~0, sizeof(clients[0].mask));
	clients[0].id.client = 345;
	clients[0].id.root = 1;
	clients[0].id.event = 2;
	clients[0].id.child = 3;
	ge = 0;
	types = 0;
	type_bytes = 0;
	max_clients = 0;
	coords = 0;
	num_coords = 0;
	return 1;
}

static void tp_event(struct grail *ge, const struct input_event *ev)
{
	ge = 0;
	ev = 0;
}

static void tp_gesture(struct grail *ge, const struct grail_event *ev)
{
	struct test_data *data = ge->priv;
	if (ev->type != GRAIL_TYPE_TAP1)
		return;
	data->pos.x = ev->prop[GRAIL_PROP_TAP_X];
	data->pos.y = ev->prop[GRAIL_PROP_TAP_Y];
}

static int init_test_data(struct test_data *data, const char *path)
{
	int ret;

	memset(data, 0, sizeof(*data));

	data->ge.get_clients = tp_get_clients;
	data->ge.event = tp_event;
	data->ge.gesture = tp_gesture;

	data->fd = open(path, O_RDONLY | O_NONBLOCK);
	ret = grail_open(&data->ge, data->fd);
	if(ret)
		return ret;
	return 0;
}

static void term_test_data(struct test_data *data)
{
	grail_close(&data->ge, data->fd);
	close(data->fd);
}

START_TEST(device_coordinates)
{
	struct test_data data;

	fail_if(init_test_data(&data, "io/evemu/one-tap.evemu"));

	data.ge.priv = &data;
	grail_pull(&data.ge, data.fd);

	fail_unless(near(&data.pos, 17087, 14790));

	term_test_data(&data);
}
END_TEST

START_TEST(mapped_coordinates)
{
	static const struct grail_coord min = { 0, 0 }, max = { 1.6, 1 };
	struct test_data data;

	fail_if(init_test_data(&data, "io/evemu/one-tap.evemu"));
	grail_set_bbox(&data.ge, &min, &max);

	data.ge.priv = &data;
	grail_pull(&data.ge, data.fd);

	fail_unless(near(&data.pos, 0.834303, 0.451338));

	term_test_data(&data);
}
END_TEST

Suite *make_mapping_suite()
{
	Suite *s = suite_create("grail-mapping");
	TCase *test;

	test = tcase_create("device_coordinates");
	tcase_add_test(test, device_coordinates);
	suite_add_tcase(s, test);

	test = tcase_create("mapped_coordinates");
	tcase_add_test(test, mapped_coordinates);
	suite_add_tcase(s, test);

	return s;
}
