#!/usr/bin/python3
# -*- Mode: Python; indent-tabs-mode: nil; tab-width: 4; coding: utf-8 -*-

import apt_check
import io
import os
import subprocess
import unittest
import textwrap


def get_message(*args, **kwds):
    with io.StringIO() as stream:
        apt_check.write_human_readable_summary(stream, *args, **kwds)
        return stream.getvalue()


class TestMotd(unittest.TestCase):
    """ ensure that the tree is pep8 clean """

    def test_esm_disabled_upto_date_esm_avail(self):
        self.assertEqual(
            get_message(upgrades=0, security_updates=0,
                        esm_updates=0, have_esm=False,
                        disabled_esm_updates=23),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is not enabled.

                0 updates can be installed immediately.
                0 of these updates are security updates.

                Enable ESM to receive 23 additional security updates.
                See 'ua enable esm' or https://ubuntu.com/esm
                """).lstrip())

    def test_esm_disabled_security_esm_avail(self):
        self.assertEqual(
            get_message(upgrades=15, security_updates=7,
                        esm_updates=0, have_esm=False,
                        disabled_esm_updates=23),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is not enabled.

                15 updates can be installed immediately.
                7 of these updates are security updates.

                Enable ESM to receive 23 additional security updates.
                See 'ua enable esm' or https://ubuntu.com/esm
                """).lstrip())

    def test_esm_disabled_security_no_esm_avail(self):
        self.assertEqual(
            get_message(upgrades=15, security_updates=7,
                        esm_updates=0, have_esm=False,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is not enabled.

                15 updates can be installed immediately.
                7 of these updates are security updates.
                """).lstrip())

    def test_esm_disabled_nosecurity(self):
        self.assertEqual(
            get_message(upgrades=15, security_updates=0,
                        esm_updates=0, have_esm=False,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is not enabled.

                15 updates can be installed immediately.
                0 of these updates are security updates.
                """).lstrip())

    def test_esm_disabled_noupdates(self):
        self.assertEqual(
            get_message(upgrades=0, security_updates=0,
                        esm_updates=0, have_esm=False,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is not enabled.

                0 updates can be installed immediately.
                0 of these updates are security updates.
                """).lstrip())

    def test_esm_enabled_nosecurity(self):
        self.assertEqual(
            get_message(upgrades=35, security_updates=0,
                        esm_updates=13, have_esm=True,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is enabled.

                35 updates can be installed immediately.
                13 of these updates are provided through ESM.
                0 of these updates are security updates.
                """).lstrip())

    def test_esm_enabled_somesecurity(self):
        self.assertEqual(
            get_message(upgrades=47, security_updates=7,
                        esm_updates=13, have_esm=True,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is enabled.

                47 updates can be installed immediately.
                13 of these updates are provided through ESM.
                7 of these updates are security updates.
                """).lstrip())

    def test_esm_enabled_noupdates(self):
        self.assertEqual(
            get_message(upgrades=0, security_updates=0,
                        esm_updates=0, have_esm=True,
                        disabled_esm_updates=0),
            textwrap.dedent(
                """
                Extended Security Maintenance (ESM) is enabled.

                0 updates can be installed immediately.
                0 of these updates are security updates.
                """).lstrip())


if __name__ == "__main__":
    import logging
    logging.basicConfig(level=logging.DEBUG)
    unittest.main()
