#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/wait.h>


#include "update-notifier.h"
#include "hal.h"

#define CDROM_CHECKER PACKAGE_LIB_DIR"/update-notifier/apt-cdrom-check"

/* reposonses for the dialog */
#define RES_START_PM 1
#define RES_UPGRADE_AUTOMATIC 2

static LibHalContext *hal_ctx;
static DBusConnection *dbus_connection;

void distro_cd_detected(UpgradeNotifier *un, char *mount_point)
{
   GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL,
					      GTK_MESSAGE_QUESTION, 
					      GTK_BUTTONS_NONE,
					      NULL );
   gtk_window_set_title(GTK_WINDOW(dialog),_("Ubuntu CD detected"));
   gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
   gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(dialog),
				 _("<span weight=\"bold\" size=\"larger\">"
				   "Ubuntu CD detected</span> \n\n"
				   "You can start the package manager "
				   "application with it now."));
   gtk_dialog_add_buttons(GTK_DIALOG(dialog),
			  GTK_STOCK_CANCEL,
			  GTK_RESPONSE_REJECT,
			  _("Start package manager"), 
			  RES_START_PM,
			  NULL);


   int res = gtk_dialog_run (GTK_DIALOG (dialog));
   char *cmd;
   switch(res) {
   case RES_START_PM:
      cmd = g_strdup_printf("synaptic --add-cdrom %s",mount_point);
      invoke_with_gksu(cmd, _("Package manager"));
      break;
   }
   gtk_widget_destroy (dialog);
}

#if 0
LibHalFunctions *
up_return_hal_functions ()
{
       static const LibHalFunctions hal_functions = { 
	  hal_mainloop_integration,
	  NULL /*hal_device_added*/,
	  NULL /*hal_device_removed*/,
	  NULL /*hal_device_new_capability*/,
	  NULL /*hal_device_lost_capability*/,
	  hal_property_modified,
	  NULL /*hal_device_condition*/
       };
       return &hal_functions;
}
#endif

/** Internal UP initialization function
 *
 * @functions			The LibHalFunctions to register as callbacks.
 * @return			The LibHalContext of the HAL connection or
 *				NULL on error.
 */
LibHalContext *
up_do_hal_init ()
{
        DBusError error;
	char **devices;
	int nr;

	hal_ctx = libhal_ctx_new ();
	if (!hal_ctx) {
	        g_warning ("failed to initialize HAL!\n");
		return NULL;
	}

	dbus_error_init (&error);
	if (!hal_mainloop_integration (hal_ctx, &error)) {
		g_warning ("hal_initialize failed: %s\n", error.message);
		dbus_error_free (&error);
		return NULL;
	}

	libhal_ctx_set_device_property_modified (hal_ctx,
		  				 hal_property_modified);

	if (!libhal_device_property_watch_all (hal_ctx, &error)) {
	        g_warning ("failed to watch all HAL properties!: %s\n", error.message);
		libhal_ctx_shutdown (hal_ctx, &error);
		return NULL;
	}

	if (!libhal_ctx_init (hal_ctx, &error)) {
		warn ("hal_initialize failed: %s\n", error.message);
		dbus_error_free (&error);
		libhal_ctx_free (hal_ctx);
		return NULL;
	}

	/*
	 * Do something to ping the HAL daemon - the above functions will
	 * succeed even if hald is not running, so long as DBUS is.  But we
	 * want to exit silently if hald is not running, to behave on
	 * pre-2.6 systems.
	 */
	devices = libhal_get_all_devices (hal_ctx, &nr, &error);
	if (!devices) {
		g_warning ("seems that HAL is not running\n");
		libhal_ctx_shutdown (hal_ctx, &error);
		if (dbus_error_is_set (&error))
			dbus_error_free (&error);
		libhal_ctx_free (hal_ctx);
		return NULL;
	}
	libhal_free_string_array (devices);

	return hal_ctx;
}

/** Invoked by GLib in response to a D-BUS disconnect event.
 *
 * @param  data                 Context pointer
 */
static gboolean
up_reconnect_to_hal (gpointer data __attribute__((__unused__)))
{
	static unsigned int retries = 0;

	g_message ("Trying a reconnect ...");
	hal_ctx = up_do_hal_init ();
	if (hal_ctx != NULL) {
		g_message ("Reconnected OK.");
		retries = 0;
		return FALSE;
	} else if (dbus_connection) { 
	   dbus_connection_unref (dbus_connection);
	   dbus_connection = NULL;
	}

	/* Retry later if it failed. */
	if (retries++ < 6000)
		return TRUE;

	/* Too many retries; clean up and bail. */
	warn("gvm_reconnect_to_hal: no reconnection after 6000 retries, "
	     "exiting\n");

	/* Too many retries; clean up and bail. */
	gtk_main_quit ();
	return FALSE;
}

static DBusHandlerResult
up_do_filter_dbus_msg (DBusConnection *connection __attribute__((__unused__)),
                        DBusMessage *message,
                        void *user_data __attribute__((__unused__)))
{
        DBusError error;

	if (dbus_message_is_signal (message,
	                            DBUS_INTERFACE_LOCAL,
	                            "Disconnected")) {
	        g_timeout_add(1000, up_reconnect_to_hal, NULL);
		libhal_ctx_shutdown (hal_ctx, &error);
		libhal_ctx_free (hal_ctx);
		hal_ctx = NULL;
		dbus_connection_unref (dbus_connection);
		dbus_connection = NULL;
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	else
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static dbus_bool_t
hal_mainloop_integration (LibHalContext *ctx, 
			   DBusError *error)
			  
{

	dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, error);

	if (dbus_error_is_set (error))
		return FALSE;

	dbus_connection_set_exit_on_disconnect (dbus_connection, FALSE);

        dbus_connection_setup_with_g_main (dbus_connection, NULL);
	dbus_connection_add_filter (dbus_connection, up_do_filter_dbus_msg, 
				    NULL, NULL);
	
	libhal_ctx_set_dbus_connection (ctx, dbus_connection);
	
	return TRUE;
}

/** Type for callback when a property of a device changes. 
 *
 *  @param  udi                 Unique Device Id
 *  @param  key                 Name of the property that has changed
 *  @param  is_removed          Property removed
 *  @param  is_added            Property added
 */
void 
hal_property_modified(LibHalContext *ctx, const char *udi, const char *key,
		      dbus_bool_t is_removed, dbus_bool_t is_added)
{
   char *mount_point;
   //g_print("hal_device_changed(): %s\n",key);

   /* we are only interessted in mount events from the cdrom*/
   if (g_strcasecmp (key, "volume.is_mounted") != 0)
      return;
   if(!libhal_device_get_property_bool(ctx, udi, key, NULL))
      return;

   char* storage_device = libhal_device_get_property_string (ctx, udi,
						    "block.storage_device",
							     NULL);
   if (!storage_device) {
      g_warning("cannot get storage_device\n");
      return;
   }
   char* media_type = libhal_device_get_property_string (ctx, storage_device, 
						      "storage.drive_type",
						      NULL);
   if(!media_type) {
      g_warning("cannot get storage.drive_type\n");
      return;
   }
   if (g_ascii_strcasecmp(media_type, "cdrom") != 0) {
      g_warning("no cdrom: %s\n",media_type);
      return;
   }
      
   mount_point = libhal_device_get_property_string (ctx, udi, 
						    "volume.mount_point",
						    NULL);
   //g_print("mounted: %s on %s\n", udi, mount_point);

   char *ubuntu_dir = g_strdup_printf("%s/ubuntu",mount_point);
   if(!g_file_test (ubuntu_dir, G_FILE_TEST_IS_SYMLINK)) {
      g_free(ubuntu_dir);
      return;
   }

   //g_print("this looks like a ubuntu-cdrom\n");
   char *cmd = g_strdup_printf(CDROM_CHECKER" %s",mount_point);
   int retval=-1;
   g_spawn_command_line_sync(cmd, NULL, NULL,  &retval, NULL);
   
   //g_print("retval: %i \n", WEXITSTATUS(retval));
   if(WEXITSTATUS(retval) == 1) {
      distro_cd_detected(libhal_ctx_get_user_data(ctx), mount_point);
   }

   g_free(cmd);
   g_free(ubuntu_dir);
}

