/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Dee;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;

namespace Unity.FilesPlace {
  
  const string ACTIVE_SECTION_HINT = "UnityActiveSection";
  const string EXTRA_ACTION_HINT = "UnityExtraAction";
  
  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
  
  /* Helper class used to encapsulate a state in a navigation sequence,
   * used as generic for our Unity.Place.Browser instance */
  private class BrowsingState
  {
    public Section section;
    public PlaceSearch? search;
    public string uri;    
  }

  public class Daemon : GLib.Object, Unity.Activation
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index index;
    private Zeitgeist.Monitor monitor;

    private Bookmarks bookmarks;
    private UrlChecker urls;

    private Unity.PlaceController control;
    private Unity.PlaceEntryInfo files;
    private Unity.Browser<BrowsingState> browser;
    
    /* We use the sections_model in normal operation, but when changing
     * to browsing mode we switch over to use pathbar_model as the sections
     * model. Magic in libunity will ensure us that the Unity Shell is notified
     * (over dbus) that we changed the sections model*/
    private Dee.SharedModel sections_model;
    private Dee.SharedModel pathbar_model;
    private string? browsing_uri = null;
    private Section browsing_root;
    
    /* For each section we have a set of Zeitgeist.Event templates that
     * we use to query Zeitgeist */
    private Gee.List<PtrArray> section_templates;

    /* Store a maping of DateMonth to Dee.ModelIter. We map to the iter and
     * not simply the offset, because in theory anyone on the bus could
     * update the Dee.SharedModel we use for the groups changing the row
     * offsets*/
    private Gee.List<unowned Dee.ModelIter?> months;
    
    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private PlaceSearch? previous_search;
    
    /** Keep track of the previously active section and don't update
     * state if the section is in fact unchanged */
    private uint previous_active_section;
    
    private bool is_dirty;
    private bool all_models_synced;
    
    private Dee.Index entry_results_by_group;

    construct
    {
      sections_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.SectionsModel");
      sections_model.set_schema ("s", "s");

      pathbar_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.PathBarModel");
      pathbar_model.set_schema ("s", "s");

      var groups_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.GroupsModel");
      groups_model.set_schema ("s", "s", "s");

      var global_groups_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.GlobalGroupsModel");
      global_groups_model.set_schema ("s", "s", "s");

      var results_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.ResultsModel");
      results_model.set_schema ("s", "s", "u", "s", "s", "s");

      var global_results_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.GlobalResultsModel");
      global_results_model.set_schema ("s", "s", "u", "s", "s", "s");

      section_templates = new Gee.ArrayList<PtrArray> ();
      prepare_section_templates();

      files = new PlaceEntryInfo ("/com/canonical/unity/filesplace/files");
      files.sections_model = sections_model;
      files.entry_renderer_info.groups_model = groups_model;
      files.entry_renderer_info.results_model = results_model;
      files.global_renderer_info.groups_model = global_groups_model;
      files.global_renderer_info.results_model = global_results_model;      

      files.icon = @"$(Config.PREFIX)/share/unity/themes/files.png";
      
      previous_search = null;
      previous_active_section = Section.LAST_SECTION; /* Must be an illegal section! */
      
      is_dirty = true;
      
      var analyzer = new Dee.Analyzer.for_uint32_column (ResultsColumn.GROUP_ID);
      entry_results_by_group = new Dee.HashIndex (results_model,
                                                  analyzer);

      /* Bring up Zeitgeist interfaces */
      log = new Zeitgeist.Log();
      index = new Zeitgeist.Index();
      
      /* Listen for all file:// related events from Zeitgeist */
      var templates = new PtrArray();
      var event = new Zeitgeist.Event ();
      var subject = new Zeitgeist.Subject ();
      subject.set_uri ("file://*");
      event.add_subject (subject);
      templates.add (event.ref ());
      monitor = new Zeitgeist.Monitor (new Zeitgeist.TimeRange.from_now (),
                                       (owned) templates);
      monitor.events_inserted.connect (on_zeitgeist_changed);
      monitor.events_deleted.connect (on_zeitgeist_changed);
      log.install_monitor (monitor);
      
      bookmarks = new Bookmarks ();
      urls = new UrlChecker ();

      /* Listen for section changes */
      files.notify["active-section"].connect (
        (obj, pspec) => {
          if (!files.active)
            return;
          
          if (!all_models_synced)
            return;
          
          var section = (Section)files.active_section;
          var _results_model = files.entry_renderer_info.results_model;
          var _groups_model = files.entry_renderer_info.groups_model;
          
          if (!(is_dirty || previous_active_section != files.active_section))
            {
              return;
            }
          
          is_dirty = false;
          
          if (search_is_invalid (files.active_search))
            {
              int group_override = section == Section.ALL_FILES ?
                                                           Group.RECENT : -1;
              update_without_search_async.begin(section,
                                                _results_model, _groups_model,
                                                group_override);
            }
          else
            {
              update_search_async.begin (files.active_search, section,
                                         _results_model, _groups_model,
                                         false, entry_results_by_group);
            }
          
          previous_active_section = files.active_section;
        }
      );

      /* Listen for changes to the place entry search */
      files.notify["active-search"].connect (
        (obj, pspec) => {
          var search = files.active_search;
          
          if (!all_models_synced)
            return;
          
          if (!files.active)
            return;
          
          var _results_model = files.entry_renderer_info.results_model;
          var _groups_model = files.entry_renderer_info.groups_model;
          var section = (Section) files.active_section;
          
          if (!(Utils.search_has_really_changed (previous_search, search) || is_dirty))
            return;
          
          is_dirty = false;
          
          if (search_is_invalid (files.active_search))
            {
              int group_override = section == Section.ALL_FILES ?
                                                           Group.RECENT : -1;
              update_without_search_async.begin(section,
                                                _results_model, _groups_model,
                                                group_override);
            }
          else
            {
              update_search_async.begin (search, section,
                                         _results_model, _groups_model,
                                         Utils.check_is_filter_search(search,
                                                                      previous_search),
                                         entry_results_by_group);
             }
          previous_search = search;
        }
      );

      /* Listen for changes to the global search */
      files.notify["active-global-search"].connect (
        (obj, pspec) => {
          if (!all_models_synced)
            return;

          var search = files.active_global_search;
          var _results_model = files.global_renderer_info.results_model;
          var _groups_model = files.global_renderer_info.groups_model;
          
          if (search_is_invalid (search))
            return;
          
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          
          update_global_search_async.begin (search,
                                            _results_model,
                                            _groups_model,
                                            Utils.check_is_filter_search(search, previous_search));
          previous_search = search;
        }
      );
      
      /* Listen for when the place is hidden by the Unity Shell, and reset
       * all state when we are deactivated */
      files.notify["active"].connect (
        (obj, pspec) => {
          debug (@"Activated: $(files.active)");
          if (files.active && is_dirty)
            {
              files.notify_property ("active-section");
            }
          else
            {
              reset_browsing_state ();
            }
        }
      );           

      /* The last thing we do is export the controller. Once that is up,
       * clients will expect the SharedModels to work */
      control = new Unity.PlaceController ("/com/canonical/unity/filesplace");
      control.add_entry (files);
      // control.activation = this; // <-- folder browsing disabled

      try {
        control.export ();
      } catch (IOError error) {
        critical ("Failed to export DBus service for '%s': %s",
                  control.dbus_path, error.message);
      }
      
      /* The browser will automagically be exported/unexported on the bus
       * when we set/unset the 'browser' property on the 'files' EntryInfo.
       * Likewise, setting the browser also sets the UnityPlaceBrowserPath
       * and UnitySectionStyle hints accordingly.
       * Such works the magic of the libunity API :-) */
      browser = new Unity.Browser<BrowsingState> (
                                     "/com/canonical/unity/filesplace/browser");
      browser.back.connect (
        (browser, state, comment) => {
          debug ("Go back to: %s", (state as BrowsingState).uri);
          var f = File.new_for_uri ((state as BrowsingState).uri);
          browse_folder.begin (f);
        }
      );
      
      browser.forward.connect (
        (browser, state, comment) => {
          debug ("Go forward to: %s", (state as BrowsingState).uri);
          var f = File.new_for_uri ((state as BrowsingState).uri);
          browse_folder.begin (f);
        }
      );
      
      /* We should not start manipulating any of our models before they are
       * all synchronized. When they are we set all_models_synced = true */
      sections_model.notify["synchronized"].connect (check_models_synced);
      pathbar_model.notify["synchronized"].connect (check_models_synced);
      groups_model.notify["synchronized"].connect (check_models_synced);
      global_groups_model.notify["synchronized"].connect (check_models_synced);
      results_model.notify["synchronized"].connect (check_models_synced);
      global_results_model.notify["synchronized"].connect (check_models_synced);
      all_models_synced = false;
    }

    /* The check_models_synced() method acts like a latch - once all models
     * have reported themselves to be synchronized we set
     * all_models_synced = true and tell the searches to re-check their state
     * as they should refuse to run when all_models_synced == false */
    private void check_models_synced (Object obj, ParamSpec pspec)
    {
      if ((sections_model as Dee.SharedModel).synchronized &&
          (pathbar_model as Dee.SharedModel).synchronized &&
          (files.entry_renderer_info.groups_model as Dee.SharedModel).synchronized &&
          (files.entry_renderer_info.results_model as Dee.SharedModel).synchronized &&
          (files.global_renderer_info.groups_model as Dee.SharedModel).synchronized &&
          (files.global_renderer_info.results_model as Dee.SharedModel).synchronized) {
        if (all_models_synced == false)
          {
            all_models_synced = true;
            
            populate_sections ();
            populate_groups ();
            
            /* Emitting notify here will make us recheck of the search results
             * need update. In the negative case this is a noop */
            files.notify_property ("active-search");
            files.notify_property ("active-global-search");
          }
      }
    }

    private void populate_sections ()
    {
      var sections = files.sections_model;

      if (sections.get_n_rows() != 0)
        {
          debug ("Sections model already populated. We probably cloned it off Unity. Resetting it.");
          sections.clear ();
        }

      sections.append (_("All Files"), "");
      sections.append ( _("Documents"), "");
      sections.append (_("Folders"), "");
      sections.append (_("Images"), "");
      sections.append (_("Audio"), "");
      sections.append (_("Videos"), "");
      sections.append (_("Presentations"), "");
      sections.append (_("Other"), "");
    }

    private void populate_groups ()
    {
      var groups = files.entry_renderer_info.groups_model;

      if (groups.get_n_rows() != 0)
        {
          debug ("Groups model already populated. We probably cloned it off Unity. Resetting it.");
          groups.clear ();
        }

      /* Always expand the Favorite Folders group */
      files.entry_renderer_info.set_hint ("ExpandedGroups",
                                          @"$((uint)Group.FAVORITE_FOLDERS)");

      groups.append ("UnityFileInfoRenderer",
                     _("Top Results"),
                     ICON_PATH + "group-mostused.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Recent"),
                     ICON_PATH + "group-recent.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Downloads"),
                     ICON_PATH + "group-downloads.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Favorite Folders"),
                     ICON_PATH + "group-favoritefolders.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Files"),
                     ICON_PATH + "group-recent.svg");
      groups.append ("UnityEmptySearchRenderer",
                     "No search results", /* No i18n. Should never be rendered */
                     "");
      groups.append ("UnityEmptySectionRenderer",
                     "Empty section", /* No i18n. Should never be rendered */
                     "");
      groups.append ("UnityFileInfoRenderer",
                     _("Today"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("Yesterday"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("This week"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("Last Week"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("This Month"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("Past Six Months"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("This Year"),
                     ICON_PATH + "group-daterange.svg");
      groups.append ("UnityFileInfoRenderer",
                     _("Last Year"),
                     ICON_PATH + "group-daterange.svg");
      // FIXME: For prehistoric items use actual year, eg "2009"


      months = new Gee.ArrayList<unowned Dee.ModelIter?>();
      months.add(null);
      for (uint i = 1; i <= DateMonth.DECEMBER; i++)
      {
        unowned Dee.ModelIter iter = groups.append ("UnityFileInfoRenderer",
                                                    Utils.get_month_name ((DateMonth)i),
                                                    ICON_PATH + "group-daterange.svg");
        months.add(iter);

      }
      
      /* We only need the global groups up to Group.FILES */
      var global_groups = files.global_renderer_info.groups_model;
      
      if (global_groups.get_n_rows() != 0)
        {
          debug ("Global groups model already populated. We probably cloned it off Unity. Resetting it.");
          global_groups.clear ();
        }
      
      global_groups.append ("UnityFileInfoRenderer",
                            _("Top Results"),
                            ICON_PATH + "group-mostused.svg");
      global_groups.append ("UnityDefaultRenderer",
                            _("Recent"),
                            ICON_PATH + "group-recent.svg");
      global_groups.append ("UnityDefaultRenderer",
                            _("Downloads"),
                            ICON_PATH + "group-downloads.svg");
      global_groups.append ("UnityDefaultRenderer",
                            _("Folders"),
                            ICON_PATH + "group-favoritefolders.svg");
      global_groups.append ("UnityDefaultRenderer",
                            _("Files"),
                            ICON_PATH + "group-recent.svg");
    }

    private void prepare_section_templates ()
    {
      PtrArray templates;
      Event event;

      /* HACK ALERT: All the (event as GLib.Object).ref() are needed because
       *             GPtrArray doesn't grab a ref to the event objects */

      /* Section.ALL_FILES */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.DOCUMENTS
       * FIXME: Filter out presentations: https://bugs.launchpad.net/zeitgeist/+bug/592599 */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_DOCUMENT,
                                               "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.FOLDERS.
       * FIXME: We probably need to be clever here and use something
       *       like subject.origin in stead of NFO_FOLDER */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.IMAGES */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_IMAGE, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);
      
      /* Section.AUDIO */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_AUDIO, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.VIDEOS */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_VIDEO, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.PRESENTATIONS
       * FIXME: Zeitgeist logger needs to user finer granularity
       *        on classification as I am not sure it uses
       *        NFO_PRESENTATION yet */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_PRESENTATION, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.OTHER 
       * Note that subject templates are joined with logical AND */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "");
      event.add_subject (new Subject.full ("file:*",
                                           "!"+NFO_DOCUMENT, "", "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_IMAGE,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_AUDIO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_VIDEO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_PRESENTATION,
                                           "",
                                           "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);
    }

    private bool search_is_invalid (PlaceSearch? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.get_search_string () == null)
        return true;
      
      return search.get_search_string ().strip() == "";
    }

    private string prepare_search_string (PlaceSearch? search)
    {
      var s = search.get_search_string ();

      if (s.has_suffix (" "))
        s = s.strip ();

      if (!s.has_suffix ("*"))
        s = s + "*";

      /* The Xapian query parser (used by the Zeitgeist FTS Extension) seems to
       * handle hyphens in a special way, namely that it forces the joined
       * tokens into a phrase query no matter if it appears as the last word
       * in a query and we have the PARTIAL flag set on the query parser.
       * This makes 'unity-p' not match 'unity-package-search.cc' etc. */
      s = s.delimit ("-", ' ');

      return s;
    }
    
    private async void update_global_search_async (PlaceSearch search,
                                                   Dee.Model results_model,
                                                   Dee.Model groups_model,
                                                   bool is_filter_search)
    {
      if (search_is_invalid (search))
        {
          results_model.clear ();
          return;
        }
      
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      files.freeze_notify ();
      
      var search_string = prepare_search_string (search);
      var templates = section_templates.get((int)Section.ALL_FILES);
      
      try {
        /* Get results ranked by recency */
        var timer = new Timer ();
        var results = yield index.search (search_string,
                                          new Zeitgeist.TimeRange.anytime(),
                                          templates,
                                          Zeitgeist.StorageState.ANY,
                                          20,
                                          ResultType.MOST_RECENT_SUBJECTS,
                                          null);
        
        timer.stop ();
        debug ("Found %u/%u global results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);


        var bookmark_matches = bookmarks.prefix_search (search.get_search_string());
        var checked_url = urls.check_url (search.get_search_string());

        /* Clean up results model */
        if (!is_filter_search)
          {
            /* Don't clear the model before the first results are ready */
            results_model.clear ();
            
            if (checked_url != null)
              {
                results_model.append (checked_url, urls.icon, Group.FILES,
                                      "text/html", search.get_search_string(),
                                      checked_url);
              }
            
            append_bookmarks (bookmark_matches, results_model, Group.FILES);
            
            Unity.FilesPlace.append_events_sorted (results,
                                                   results_model,
                                                   groups_model,
                                                   Section.ALL_FILES,
                                                   Group.FILES);
          }
        else
          {
            /* Remove all rows from the results_model that are not
             * in the new result set or matching bookmarks */
            Set<string> valid_uris = Utils.get_uri_set (results);
            
            foreach (var bookmark in bookmark_matches)
              {
                valid_uris.add (bookmark.uri);
              }
            
            unowned ModelIter iter = results_model.get_first_iter();
            unowned ModelIter end = results_model.get_last_iter ();
            unowned ModelIter current_row;
            string result_uri;
            while (iter != end)
              {
                result_uri = results_model.get_string (iter, ResultsColumn.URI);
                current_row = iter;
                iter = results_model.next (iter);
                if (!(result_uri in valid_uris))
                  results_model.remove (current_row);
              }
            
            /* A possibly checked URL will be filtered out by now,
             * So we *prepend* it, to add it on the first spot
             * if we have one */
            if (checked_url != null)
              {
                results_model.prepend (checked_url, urls.icon, Group.FILES,
                                       "text/html", search.get_search_string(),
                                       checked_url);
              }
          }
      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.get_search_string (), e.message);
      }
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        files.thaw_notify ();
        return false;
      });
      
      search.finished ();
    }
    
    private async void update_search_async  (PlaceSearch search,
                                             Section section,
                                             Dee.Model results_model,
                                             Dee.Model groups_model,
                                             bool is_filter_search,
                                             Dee.Index results_by_group,
                                             int group_override = -1)
    {
      Timer full_timer = new Timer ();
      
      // FIXME: Implement in-folder searching
      if (files.browser != null)
        {
          warning ("In-folder searching not implemented yet");
          return;
        }
      
      if (search_is_invalid (search))
        {
          update_without_search_async.begin (section,
                                             results_model,
                                             groups_model);
          return;
        }
      
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      files.freeze_notify ();
      
      var search_string = prepare_search_string (search);
      var templates = section_templates.get((int)section);  
      
      try {
        /* Get relevancy ranked results for the "Top Results" group */
        var timer = new Timer ();
        var results = yield index.search (search_string,
                                          new Zeitgeist.TimeRange.anytime(),
                                          templates,
                                          Zeitgeist.StorageState.ANY,
                                          50,
                                          Zeitgeist.ResultType.RELEVANCY,
                                          null);
        
        timer.stop ();
        debug ("Found %u/%u Top Results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);

        /* Clean up results model */
        if (!is_filter_search)
          {
            /* Don't clear the results before the first ones are ready */
            results_model.clear ();
          }
        else
          {
            /* Remove everything in the Top Results group. We can not do filter
             * searches here as it's sorted by relevancy alone */
            uint group = Group.TOP_RESULTS;
            var top_results = results_by_group.lookup (@"$group",
                                                       TermMatchFlag.EXACT);
            foreach (var row in top_results)
              {
                results_model.remove (row);
              }
          }
        
        /* First add any matching folder bookmarks.
         * Note: This only works without dupes on incremental search
         *       because the folders are in the Top Results group which we
         *       always clean out on each update */
        var bookmark_matches = bookmarks.prefix_search (search.get_search_string());
        append_bookmarks (bookmark_matches, results_model, Group.TOP_RESULTS);
        
        var seen_uris = Unity.FilesPlace.append_events_sorted (results,
                                                               results_model,
                                                               groups_model,
                                                               section,
                                                               Group.TOP_RESULTS);
        
        /* Get time-grouped results */
        var result_type = section == Section.FOLDERS ?
                               ResultType.MOST_RECENT_ORIGIN : 
                               ResultType.MOST_RECENT_SUBJECTS;
        timer.start ();
        results = yield index.search (search_string,
                                      new Zeitgeist.TimeRange.anytime(),
                                      templates,
                                      Zeitgeist.StorageState.ANY,
                                      100,
                                      result_type,
                                      null);
        
        timer.stop ();
        debug ("Found %u/%u time grouped results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);
        
        if (!is_filter_search)
          {
            Unity.FilesPlace.append_events_sorted (results,
                                                   results_model, groups_model,
                                                   section,
                                                   group_override,
                                                   seen_uris);
          }
        else
          {
            /* We must filter all groups except Top Results and we should
             * not include results that are already in Top Results
             * IMPORTANT: This code assumes Group.TOP_RESULTS is the very first
             *            and that Group.RECENT is the second group! */
            Set<string> timegrouped_uris = Utils.get_uri_set (results);
            timegrouped_uris.remove_all (seen_uris);
            uint n_groups = groups_model.get_n_rows ();
            for (uint group_ = Group.RECENT; group_ < n_groups; group_++)
              Utils.apply_uri_filter (timegrouped_uris,
                                      results_by_group.lookup (@"$group_",
                                                               TermMatchFlag.EXACT));
          }

      } catch (GLib.Error e) {
        warning ("Error performing search '%s': %s",
                 search.get_search_string (), e.message);
      }
      
      check_empty_search (search, results_model);            
      
      full_timer.stop ();
      debug ("FULL SEARCH TIME FOR '%s': %fms",
             search.get_search_string(), full_timer.elapsed()*1000);
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        files.thaw_notify ();
        return false;
      });
      
      search.finished ();
    }

    private async void update_without_search_async (Section section,
                                                    Dee.Model results_model,
                                                    Dee.Model groups_model,
                                                    int group_override = -1)
    {
      bool active_section_hint_new = false;
    
      if (files.browser != null)
        {
          if (section != 0)
            {
              File folder = get_folder_for_pathbar_section (section);
              try {	            
                yield activate (folder.get_uri ());
              } catch (IOError ee) {
	            warning ("Failed to activate URI '%s': %s",
	                     folder.get_uri (), ee.message);
              }
              debug ("Browsed %s from path bar", folder.get_uri ());
              return;
            }
          else
            {
              /* The root section of the pathbar was clicked.
               * Leave folder browsing mode */
              debug ("Root section of pathbar activated. Leaving browsing mode");
              reset_browsing_state ();
              files.active_section = browsing_root;
              files.set_hint (ACTIVE_SECTION_HINT, @"$browsing_root");
              section = browsing_root;
              active_section_hint_new = true;
            }          
        }
      
      /* If we have the UnityActiveSection hint and we didn't just set it,
       * then clear it */
      if (files.get_hint (ACTIVE_SECTION_HINT) != null &&
          !active_section_hint_new)
        {
          // Hack alert: We do the notify() because we need to trigger a DBus
          //             signal that the entry has changed
          files.clear_hint (ACTIVE_SECTION_HINT);
          files.notify_property ("active-section");
          debug ("Clearing active section hint");
        }
      
      var sections_model = files.sections_model;

      if (Section.LAST_SECTION != sections_model.get_n_rows())
        {
          critical ("Section model malformed");
          return;
        }

      if (section > Section.LAST_SECTION)
        {
          critical ("Active section out of bounds: %u", section);
          return;
        }

      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      files.freeze_notify ();

      var templates = section_templates.get((int)section);
      var result_type = section == Section.FOLDERS ?
                               ResultType.MOST_RECENT_ORIGIN : 
                               ResultType.MOST_RECENT_SUBJECTS;

      try {
        var timer = new Timer();
        var events = yield log.find_events (
                                        new Zeitgeist.TimeRange.anytime(),
                                        templates,
                                        Zeitgeist.StorageState.ANY,
                                        100,
                                        result_type,
                                        null);
        timer.stop ();
        debug ("Got %u events for section %u in %fms",
               events.size(), section, timer.elapsed()*1000);

        /* Don't clear the model before we have the results ready */
        results_model.clear();
        
        if (section == Section.ALL_FILES)
          {
            append_bookmarks (bookmarks.list(), results_model);
            yield update_downloads_async (results_model, groups_model);
          }

        Unity.FilesPlace.append_events_sorted (events,
                                               results_model, groups_model,
                                               section,
                                               group_override);
      } catch (GLib.Error e) {
        warning ("Error fetching recetnly used files: %s", e.message);
      }
      
      check_empty_section (section, results_model);
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        files.thaw_notify ();
        return false;
      });
    }

    private void append_bookmarks (GLib.List<Bookmark> bookmarks,
                                   Dee.Model results_model,
                                   Group group = Group.FAVORITE_FOLDERS)
    {
      foreach (var bookmark in bookmarks)
      {
        results_model.append (bookmark.uri, bookmark.icon, group,
                              bookmark.mimetype, bookmark.display_name,
                              bookmark.uri);
      }
    }

    private async void update_downloads_async (Dee.Model results_model,
                                               Dee.Model groups_model)
    {
      // FIXME: The Downloads folder and update on changes
      unowned string download_path =
                 Environment.get_user_special_dir (UserDirectory.DOWNLOAD);
      var download_dir = File.new_for_path (download_path);
      SList<FileInfo> downloads;
      
      try {
        downloads = yield Utils.list_dir (download_dir);
      } catch (GLib.Error e) {
        warning ("Failed to list downloads from directory '%s': %s",
                 download_path, e.message);
        return;
      }
      
      /* Sort files by mtime, we do an ugly nested ternary
       * to avoid potential long/int overflow */
      downloads.sort ((CompareFunc) Utils.cmp_file_info_by_mtime);
      
      foreach (var info in downloads)
      {
        var uri = download_dir.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);
        results_model.append (uri, icon_hint, Group.DOWNLOADS,
                              mimetype, info.get_display_name (), uri);
      }
    }

    private void reset_browsing_state ()
    {
      /* We check for actual changes before resetting the properties, in order
       * to avoid spurious PlaceEntryInfoChanged signals which Unity doesn't
       * like that much */
      if (files.browser != null)
        {
          files.browser = null;
        }
      
      if (files.sections_model != sections_model)
        {
          files.sections_model = sections_model;
        }
      
      files.clear_hints ();
      is_dirty = true;
    }

    /**
     * Override of the default activation handler. The files place daemon
     * can handle activation of folders which puts it into "folder browsing mode"
     */
    public async uint32 activate (string uri)
    {
      /* When Unity asks us to activate "." it's a special request for
       * activating our UnityExtraAction hint. Which for the files' place
       * is  "browse current folder in Nautilus" */
      if ("." == uri)
        {
          browse_current_folder_in_nautilus ();
          return ActivationStatus.ACTIVATED_HIDE_DASH;
        }
      
      var f = File.new_for_uri (uri);
      
      if (f.query_file_type (0, null) != FileType.DIRECTORY)
        {
          debug ("Declined activation of URI '%s': Not a directory", uri);
          return ActivationStatus.NOT_ACTIVATED;
        }
      
      debug ("Browsing folder: %s", uri);

      /* Record the URI in the browser */
      var state = new BrowsingState ();
      state.search = files.active_search;
      state.section = (Section)files.active_section;
      state.uri = uri;
      browser.record_state (state, uri); // FIXME: Nicer comment than just the URI
      browsing_uri = uri;
      
      /* We need to mark us as dirty - otherwise we'll discard the section
       * change signal when changed back to the section of the pathbar root */
      is_dirty = true;
      
      /* If we are entering browsing mode record the active section,
       * and set the UnityExtraAction hint to "Browse current folder
       * in Nautilus" */
  	  if (files.browser == null)    
        {
          browsing_root = (Section) files.active_section;
          
          /* The UnityExtraIcon hint must contain a serialized, GIcon,
           * but a path to an image qualifies as such */
          files.set_hint (EXTRA_ACTION_HINT, ICON_PATH + "open-folder.svg");
        }      

      /* Setting the browser property does all the necessary dbus magic,
       * it is also used to indicate to our selves that we are in browsing
       * mode */          
      files.browser = browser;      
      
      /* Change our files.sections_model over to a special model
       * we use to render the breadcrumbs/path bar. Changing the
       * files.sections_model property will automatically notify Unity
       * over DBus with the PlaceEntryInfoChanged signal */
      files.sections_model = pathbar_model;
       
      browse_folder.begin (f);

      return ActivationStatus.ACTIVATED_SHOW_DASH;
    }
    
    private async void browse_folder (File folder)
    {         
      var results_model = files.entry_renderer_info.results_model;
      SList<FileInfo> file_infos;
      
      // FIXME: Alphabetic sorting of folder contents
      /* Populate the results_model with folder contents */
      try {
        file_infos = yield Utils.list_dir (folder);
      } catch (GLib.Error err) {
        warning ("Failed to browse folder '%s': %s",
                 folder.get_uri (), err.message);
        return;
      }

      results_model.clear ();
      foreach (var info in file_infos) {
	    if (info.get_is_hidden() || info.get_is_backup ())
	      continue;
	    
        var uri = folder.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);
        debug ("Found child: %s", uri);
        results_model.append (uri, icon_hint, 0, // FIXME: which group for folder browsing?
                              mimetype, info.get_display_name (), uri);
      }
      
      yield update_pathbar_model (folder);
    }
    
    private async void update_pathbar_model (File folder)
    {
      /* Update the pathbar model with path relative to the home dir
       * Unity should be showing the pathbar instead of the normal sections */
      // FIXME: Don't .clear() the model, but compute the diff and update in stead
      pathbar_model.clear ();
      string home_path = Environment.get_home_dir ();
      File parent = folder;
      FileInfo finfo;      
      do {
        try {
          if (parent.get_path () == home_path)
            break;

          finfo = parent.query_info (Utils.file_attribs,
	                                   FileQueryInfoFlags.NONE);
        } catch (GLib.Error e) {
          warning ("Failed to compute breadcrumb path: %s", e.message);  
	        break;
        }
        pathbar_model.prepend (finfo.get_display_name (), "");
      } while ((parent = parent.get_parent ()) != null);
      
      string section_name = sections_model.get_string (sections_model.get_iter_at_row (browsing_root), 0);
      pathbar_model.prepend (section_name, "");
    }
    
    /* Calculates the file URI of the path currently being browsed,
     * up to the path element number @path_element */
    private File get_folder_for_pathbar_section (uint path_element)
    {
      uint n_path_sections = pathbar_model.get_n_rows ();
      if (path_element >= n_path_sections - 1)
        {
		      warning ("Path section out of bounds: %u (of %u)",
		               path_element, n_path_sections);
		      return File.new_for_path (Environment.get_home_dir ());
        }
	  	  
      uint to_remove = n_path_sections - path_element - 1;
      File uri = File.new_for_uri (browsing_uri);
      
      while (to_remove > 0)
        {
          uri = uri.get_parent ();
          to_remove--;
        }
	  
      debug ("For section %u found: %s", path_element, uri.get_uri ());
      return uri;
    }
    
    /* Launch default file manager (Nautilus) on the currently browsed folder */
    private void browse_current_folder_in_nautilus ()
    {
      if (browsing_uri == null || browsing_uri == "")
        {
        warning ("Unable to open current folder in file manager. " +
                 "We don't have a current folder!");
          return;
        }
      
      debug ("Opening folder current folder '%s' in file manager",
             browsing_uri);
      
      try {
        AppInfo.launch_default_for_uri (browsing_uri, null);
      } catch (GLib.Error e) {
        warning ("Failed to open current folder '%s' in file manager: %s",
                 browsing_uri, e.message);
      }
    }
    
    public void check_empty_search (PlaceSearch? search,
                                    Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
      
      if (search_is_invalid(search))
        return;
      
      results_model.append ("", "", Group.EMPTY_SEARCH,
                            "", _("Your search did not match any files"), "");      
      
      // FIXME: Use prefered browser
      // FIXME: URL escape search string
      results_model.append (@"http://google.com/#q=$(search.get_search_string())",
                            "", Group.EMPTY_SEARCH,
                            "", _("Search the web"), "");
    }
    
    public void check_empty_section (Section section,
                                     Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
            
      string msg;
      
      switch (section)
      {
        case Section.ALL_FILES:
          msg = _("There are no files in your Home folder");
          break;
        case Section.DOCUMENTS:
          msg = _("There are no documents in your Home folder");
          break;
        case Section.FOLDERS:
          msg = _("There are no folders in your Home folder");
          break;
        case Section.IMAGES:
          msg = _("There are no images in your Home folder");
          break;
        case Section.AUDIO:
          msg = _("There are no audio files in your Home folder");
          break;
        case Section.VIDEOS:
          msg = _("There are no videos in your Home folder");
          break;
        case Section.PRESENTATIONS:
          msg = _("There are no presentations in your Home folder");
          break;
        case Section.OTHER:
          msg = _("There are no other files in your Home folder");
          break;
        default:
          warning ("Unknown section number: %u", section);
          msg = _("There are no files in your Home folder");
          break;
      }
      
      results_model.append ("", "", Group.EMPTY_SECTION, "", msg, "");
    }
    
    private void on_zeitgeist_changed ()
    {
      /* Emitting notify here will make us recheck if the search results
       * need update. In the negative case this is a noop */
      is_dirty = true;
      files.notify_property ("active-section");
    }
    
  } /* End: Daemon class */
  
  /* Appends a set of Zeitgeist.Events to our Dee.Model assuming that
   * these events are already sorted with descending timestamps */
  public Gee.Set<string> append_events_sorted (Zeitgeist.ResultSet events,
                                               Dee.Model results,
                                               Dee.Model groups,
                                               Section section,
                                               int group_override = -1,
                                               Gee.Set<string>? seen_uris = null)
    {
      Gee.Set<string> _seen_uris;
    
      if (seen_uris == null)
        _seen_uris = new Gee.HashSet<string>(str_hash);
      else
        _seen_uris = seen_uris;
    
      foreach (var ev in events)
        {
          if (ev.num_subjects() > 0)
            {
              // FIXME: We only use the first subject...
              Zeitgeist.Subject su = ev.get_subject(0);
              //var timer = new Timer();
              
              string uri = su.get_uri ();
           	  string display_name = su.get_text ();
           	  string mimetype = su.get_mimetype () != null ?
           	                su.get_mimetype () : "application/octet-stream";
           	  File file = File.new_for_uri (uri);
           	  
           	  /* De-dup the results keyed on the subject URIs */
              if (uri in _seen_uris)
                continue;
              _seen_uris.add (uri);
              
              /* Don't check existence on non-native files as http:// and
               * friends are *very* expensive to query */
              if (file.is_native())
                if (!file.query_exists ())
                  continue;
                  
              if (section == Section.FOLDERS)
                {
                  File dir = File.new_for_uri(uri).get_parent();
                  uri = dir.get_uri ();
                  try{
                    FileInfo info = dir.query_info (FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
	                                                FileQueryInfoFlags.NONE);
 	                display_name = info.get_display_name ();
 	              } catch (GLib.Error e) {
 	                /* Bugger, we fall back to basename, which might not be
 	                 * valid UTF8... */
 	                warning ("Unable to get display name for %s", uri);
 	                display_name = dir.get_basename ();
 	              }
	              mimetype = "inode/directory";
                }
                  
              string icon = Utils.get_icon_for_uri (uri, mimetype);
              
              uint group_id;
              string comment = "";
              
              /* We call this as we want the comment string */
              group_id= Utils.get_time_group (ev, groups, out comment);
              
              if (group_override >= 0)
                {
                  group_id = (uint) group_override;
                }
              
              results.append (uri, icon, group_id, mimetype,
                              display_name, comment);
              
              //timer.stop();
              //debug("  + %s: %fms", uri, timer.elapsed()*1000);
            }          
        }
      
      return _seen_uris;
    }
} /* namespace */

