/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;
using Unity;
using Gdk;
using Ubuntuone.Constants;
using Ubuntuone.Webservice;

namespace Unity.MusicLens {
  private const string ERROR_MESSAGE_NOT_LOGGED_IN = _("It seems you don't have an Ubuntu One account, or you are not logged in. To continue, please login and visit the Ubuntu One online store.");
  private const string ERROR_MESSAGE_NO_PAYMENT_METHOD = _("It seems you haven't set yet your preferred Ubuntu One payment method. To add a payment method, please visit the Ubuntu One online store.");
  private const string ERROR_MESSAGE_TECHNICAL_PROBLEM = _("Sorry, we have encountered a technical problem. No money has been taken from your account. To try your purchase again, please visit the Ubuntu One online store.");
  private const string UBUNTUONE_MUSIC_LOGIN = Config.LIBEXECDIR + "/ubuntuone-credentials/ubuntuone-music-login";

  public class MusicStoreScopeProxy : SimpleScope
  {
    private MusicStoreCollection collection;
    private Unity.MusicPreview? music_preview;
    private PreferencesManager preferences = PreferencesManager.get_default ();
    private HashTable<string, Album> album_map;
    private PurchaseService purchase_service;
    private Notify.Notification notification;

    public MusicStoreScopeProxy ()
    {
      base ();

      scope = new Unity.DeprecatedScope ("/com/canonical/unity/scope/musicstore", "musicstore.scope");
      scope.search_in_global = false;
      scope.search_hint = _("Search music");
      scope.visible = true;

      scope.activate_uri.connect (activate);
      scope.preview_uri.connect (preview);

      base.initialize ();

      Notify.init ("Music Store Scope");
      notification = new Notify.Notification("Album name", _("Purchase started"), "");
      collection = new MusicStoreCollection ();
      album_map = new HashTable<string, Album>(str_hash, str_equal);
      purchase_service = new PurchaseService ();

      preferences.notify["remote-content-search"].connect((obj, pspec) => { scope.queue_search_changed(SearchType.DEFAULT); });

      try {
	scope.export ();
      } catch (GLib.IOError e) {
	stdout.printf (e.message);
      }
    }

    protected override int num_results_without_search { get { return 100; } }
    protected override int num_results_global_search { get { return 20; } }
    protected override int num_results_lens_search { get { return 50; } }

    public Unity.ActivationResponse activate (string uri)
    {
      var prv = preview (uri);
      if (prv != null)
      {
        return new Unity.ActivationResponse.with_preview (prv);
      }
      else
      {
        warning ("Failed to generate preview for %s", uri);
        return open_uri (uri);
      }
    }

    public Unity.ActivationResponse open_uri (string uri)
    {
      /* launch the music store streaming client  or the webstore or whatevz */
      try {
        AppInfo.launch_default_for_uri (uri, null);
      } catch (GLib.Error e) {
        warning ("Failed to open uri %s. %s", uri, e.message);
      }
      return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
    }

    public Unity.Preview preview (string uri)
    {
      music_preview = null;
      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);

      if (album != null)
      {
        album_map.insert (uri, album);
        File cover_file = File.new_for_uri (album.artwork_path); //artwork path is a remote uri
        var cover = new FileIcon (cover_file);

        int i = 1;
        music_preview = new Unity.MusicPreview (album.title, album.artist, cover);

        if (tracks != null)
        {
          foreach (Track track in tracks)
          {
            TrackMetadata tm = new TrackMetadata ();
            tm.uri = track.uri;
            tm.track_no = i++; //FIXME: u1ms search doesn't provide track numbers *yet*, this will change soon
            tm.title = track.title;
            tm.length = track.duration;
            music_preview.add_track (tm);
          }
        }

        GLib.Icon? icon = new GLib.FileIcon (File.new_for_path (Config.DATADIR + "/icons/unity-icon-theme/places/svg/service-u1.svg"));
        var download_action = new Unity.PreviewAction ("show_purchase_preview", _("Download"), icon);
        if (album.price > 0 && album.formatted_price != null)
            download_action.extra_text = album.formatted_price;
        else if (album.price == 0)
            download_action.extra_text = _("Free");

        download_action.activated.connect (open_uri);

        music_preview.add_action (download_action);
      }
      return music_preview;
    }

    public override async void perform_search (DeprecatedScopeSearch search,
                                               SearchType search_type,
                                               owned List<FilterParser> filters,
                                               int max_results = -1,
                                               GLib.Cancellable? cancellable = null)
    {
      if (is_search_empty (search))
        return;

      /**
       * only perform the request if the user has not disabled
       * online/commercial suggestions. That will hide the category as well.
       */
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.results_model.clear ();
        return;
      }

      try {
	debug ("model has %u rows before search", search.results_model.get_n_rows ());
	yield collection.search (search, search_type, (owned) filters, max_results, cancellable);
	debug ("model has %u rows after search", search.results_model.get_n_rows ());
      } catch (IOError e) {
	warning ("Failed to search for '%s': %s", search.search_string, e.message);
      }

    }

    delegate Unity.ActivationResponse LinkHandler (string uri);

    private Unity.ActivationResponse build_error_preview (string uri, string error_header, string album_price, string link_text, LinkHandler link_handler, bool has_cancel=true)
    {
      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);
      debug ("album art uri: %s", album.artwork_path);
      File cover_file = File.new_for_uri (album.artwork_path); //artwork path is a remote uri
      var cover = new FileIcon (cover_file);
      var error_preview = new Unity.PaymentPreview.for_error(album.title, album.artist, cover);
      error_preview.header = error_header;
      error_preview.purchase_prize = album_price;
      error_preview.purchase_type = _("Digital CD");
      var error_action = new Unity.PreviewAction ("open_u1_link", link_text, null);
      error_action.activated.connect (() => link_handler (uri));
      error_preview.add_action (error_action);

      if (has_cancel)
      {
        var cancel_action = new Unity.PreviewAction ("cancel", _("Cancel"), null);
        cancel_action.activated.connect (cancel_purchase);
        error_preview.add_action (cancel_action);
      }

      return new Unity.ActivationResponse.with_preview (error_preview);
    }

    public Unity.ActivationResponse open_forgot_password_url (string uri)
    {
      return open_uri (forgotten_password_url ());
    }

    public Unity.ActivationResponse change_payment_method (string uri)
    {
      if (purchase_service.open_url != null)
      {
        debug ("Open url is %s", purchase_service.open_url);
        return open_uri (purchase_service.open_url);
      }

      return open_uri (change_payment_method_url ());
    }

    public Unity.ActivationResponse cancel_purchase (string uri)
    {
      return new Unity.ActivationResponse.with_preview (preview (uri));
    }

    public Unity.ActivationResponse open_sso_login (string uri)
    {
      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);

      debug ("Finding %s", UBUNTUONE_MUSIC_LOGIN);
      var ui_path = Environment.find_program_in_path (UBUNTUONE_MUSIC_LOGIN);
      string cmd_line = "%s --album=\"%s\" --artist=\"%s\" --price=\"%s\" --picture=\"%s\" --url=\"%s\"".printf(
        ui_path, album.title, album.artist, album.formatted_price, album.artwork_path, uri);
      if (ui_path != null) {
        try {
          debug ("Executing '%s'", cmd_line);
          bool was_started = Process.spawn_command_line_async (cmd_line);
          // hide dash
          if (was_started) {
              return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
          }
        } catch (GLib.SpawnError e) {
          debug ("Failed to start ubuntuone-music-login for uri %s due to '%s'", uri, e.message);
        }
      }
      return open_uri (uri);
    }

    bool run_with_timeout (MainLoop ml, uint timeout_ms)
    {
        bool timeout_reached = false;
        var t_id = Timeout.add (timeout_ms, () => {
            timeout_reached = true;
            debug ("Timeout reached");
            ml.quit ();
            return false;
        });

        ml.run ();

        if (!timeout_reached) {
            Source.remove (t_id);
        }

        return !timeout_reached;
    }

    void fetch_account_info_sync () throws PurchaseError
    {
        PurchaseError failure = null;
        MainLoop mainloop = new MainLoop ();
        purchase_service.fetch_account_info.begin((obj, res) => {
            mainloop.quit ();
            try {
                purchase_service.fetch_account_info.end (res);
            } catch (PurchaseError e) {
                failure = e;
            }
        });
        if (!run_with_timeout (mainloop, 10000)) {
            throw new PurchaseError.MISSING_CREDENTIALS_ERROR ("Timeout getting credentials.");
        }
        if (failure != null) {
            throw failure;
        }
    }

    public Unity.ActivationResponse show_purchase_preview (string uri)
    {

      try {
        fetch_account_info_sync ();
        debug ("retrieved account info: %s %s", purchase_service.nickname, purchase_service.email);
      } catch (PurchaseError e) {
        debug ("can't get account info: %s", e.message);
      }

      if (purchase_service.got_credentials () == false)
      {
        debug ("no credentials available, opening sso login. %s", uri);
        return open_sso_login (uri);
      }

      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);

      if (album != null)
      {
        try {
          purchase_service.fetch_payment_info (album.purchase_sku);
          debug ("retrieved payment method: %s", purchase_service.selected_payment_method);
          return new Unity.ActivationResponse.with_preview (purchase_preview (uri, null));
        } catch (PurchaseError e) {
          debug ("can't get default payment method: %s", e.message);
          return build_error_preview (uri, ERROR_MESSAGE_NO_PAYMENT_METHOD, album.formatted_price, _("Go to Ubuntu One"), change_payment_method);
        }
      }
      return open_uri (uri);
    }

    public Unity.ActivationResponse purchase_album (Unity.PreviewAction action, string uri)
    {
      var password = action.hints["password"].get_string();

      if (password == null) {
          var preview = purchase_preview (uri, _("Please enter your password"));
          debug ("empty password.");
          return new Unity.ActivationResponse.with_preview (preview);
      }

      var album = album_map.get (uri);

      File cover_file = File.new_for_uri (album.artwork_path); //artwork path is a remote uri
      try {
        var cover_pixbuf = new Pixbuf.from_stream (cover_file.read ());
        notification.set_icon_from_pixbuf (cover_pixbuf);
      } catch (GLib.Error e) {
        debug ("Cannot set notification icon from uri %s", uri);
      }

      notification.summary = album.title;
      notification.body = _("Authorizing purchase");
      try {
        notification.show ();
      } catch (GLib.Error e) {
        debug ("Error while showing notification: %s", e.message);
      }

      try {
        purchase_service.purchase (album.purchase_sku, password);
        debug ("purchase completed.");
        notification.update (album.title, _("Purchase completed"), "");
        try {
          notification.show ();
        } catch (GLib.Error e) {
          debug ("Error while showing notification: %s", e.message);
        }
        return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
      } catch (PurchaseError e) {
        if (e is PurchaseError.WRONG_PASSWORD_ERROR) {
            debug ("wrong password error: %s", e.message);
            return new Unity.ActivationResponse.with_preview (purchase_preview (uri, _("Wrong password")));
        } else {
            debug ("got purchase error: %s", e.message);
            return build_error_preview (uri, ERROR_MESSAGE_TECHNICAL_PROBLEM, album.formatted_price, _("Continue"), open_uri);
        }
      }
    }

    public Unity.Preview purchase_preview (string uri, string? error_message)
    {
      Unity.PaymentPreview album_purchase_preview = null;
      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);

      if (album != null)
      {
        File cover_file = File.new_for_uri (album.artwork_path); //artwork path is a remote uri

        var cover = new FileIcon (cover_file);
        album_purchase_preview = new Unity.PaymentPreview.for_music(album.title, album.artist, cover);

        album_purchase_preview.header = _("Hi %s, you purchased in the past from Ubuntu One,"
        + " would you like to use the same payment details? Please review your order.").printf (purchase_service.nickname);
        album_purchase_preview.email = purchase_service.email;
        album_purchase_preview.payment_method = purchase_service.selected_payment_method;
        album_purchase_preview.purchase_prize = album.formatted_price;
        album_purchase_preview.purchase_type = _("Digital CD");

        // data

        var data = new HashTable<string, Variant>(str_hash, str_equal);
        if (error_message != null) {
            data["error_message"] = error_message;
        }

        InfoHint info_hint = new InfoHint.with_variant("album_purchase_preview", "", null, data);
        album_purchase_preview.add_info(info_hint);

        // actions

        var purchase_action = new Unity.PreviewAction ("purchase_album", _("Buy Now"), null);
        purchase_action.activated.connect (purchase_album);
        album_purchase_preview.add_action (purchase_action);

        var forgot_password_action = new Unity.PreviewAction ("forgot_password", _("forgotten your Ubuntu One password?"), null);
        forgot_password_action.activated.connect (open_forgot_password_url);
        album_purchase_preview.add_action (forgot_password_action);

        var cancel_action = new Unity.PreviewAction ("cancel_purchase", _("Cancel"), null);
        cancel_action.activated.connect (cancel_purchase);
        album_purchase_preview.add_action (cancel_action);

        var change_payment_method_action = new Unity.PreviewAction ("change_payment_method", _("change payment method"), null);
        change_payment_method_action.activated.connect (change_payment_method);
        album_purchase_preview.add_action (change_payment_method_action);
      }
      return album_purchase_preview;
    }

  }
}
