/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens
{
  public abstract class SimpleScope : Object
  {
    public Unity.Scope scope { get; protected set; }

    protected abstract int num_results_without_search { get; }
    protected abstract int num_results_global_search { get; }
    protected abstract int num_results_lens_search { get; }

    protected abstract void perform_search (LensSearch? search, Dee.Model results_model, List<FilterParser> filters, int max_results = -1);
    
    protected bool is_dirty;
    protected LensSearch? previous_search;

    public SimpleScope ()
    {
      is_dirty = true;
      previous_search = null;
    }

    protected void initialize ()
    {
      /* Listen for filter changes */
      scope.filters_changed.connect(
				    () => { 
				      if (scope.active_search != null)
					{
					  is_dirty = true;
					  scope.notify_property ("active-search");
					}
				    }
				    );
      
      /* Listen for changes to the lens search entry */
      scope.notify["active-search"].connect (
					     (obj, pspec) => { 
					       var search = scope.active_search;
					       if (!(is_dirty || search_has_really_changed (previous_search, search)))
						 return;
					       is_dirty = false;
					       update_search_async.begin (search);
					       previous_search = search;
					     }
					     );
      
      /* Listen for changes to the global search */
      scope.notify["active-global-search"].connect (
						    (obj, pspec) => { 
						      var search = scope.active_global_search;
						      
						      if (search_is_invalid (search))
							return;
						      
						      if (!search_has_really_changed (previous_search, search))
							return;
						      
						      update_global_search_async.begin (search);
						      previous_search = search;
						    }
						    );
    }

    /* based on the filters(?) get the default results to show when the music lens is
     * just opened without filters, like when someone hits Super+M
     */
    private async void update_without_search_async (LensSearch search, Dee.Model results_model)
    {
      perform_search_async.begin (search, results_model, num_results_without_search);
    }
    
    /**
     * results for a global search from just hitting Super. here we want to return
     * a smaller number of results with 0 filters.
     */
    private async void update_global_search_async (LensSearch search)
    {
      var results_model = scope.global_results_model;
      
      if (search_is_invalid (search))
	{
	  results_model.clear ();
	  return;
	}
      
      perform_search_async.begin (search, scope.global_results_model, num_results_global_search);
    }
    
    private async void update_search_async (LensSearch search)
    {
      // just pretend like there's no search
      if (search_is_invalid (search))
	{
	  update_without_search_async.begin (search, scope.results_model);
	  return;
	}
      
      perform_search_async.begin (search, scope.results_model, num_results_lens_search);
    }

    private async void perform_search_async (LensSearch search, Dee.Model results_model, int max_results)
    {       
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();
      
      List<FilterParser> filters = new List<FilterParser> ();
      Filter filter = scope.get_filter ("genre");
      if (filter.filtering)
	filters.append (new GenreFilterParser (filter as CheckOptionFilter));

      filter = scope.get_filter ("decade");
      if (filter.filtering)
	filters.append (new DecadeFilterParser (filter as MultiRangeFilter));
      
      results_model.clear ();
      perform_search (search, results_model, filters, max_results);
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
	  scope.thaw_notify ();
	  return false;
	});
      
      search.finished ();
    }

    protected bool search_is_invalid (LensSearch? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.search_string == null)
        return true;
      
      return search.search_string.strip() == "";
    }
    
    private bool search_has_really_changed (LensSearch? old_search,
					    LensSearch? new_search)
    {
      if (old_search == null && new_search == null)
	return false;
      
      string s1, s2;
      
      if (old_search == null)
        {
	  s1 = new_search.search_string;
	  if (s1 == null || s1.strip() == "")
	    return false;
	  else
	    return true;
        }
      else if (new_search == null)
	{
	  s2 = old_search.search_string;
	  if (s2 == null || s2.strip() == "")
	    return false;
	  else
	    return true;
	}
      else
	{
	  s1 = new_search.search_string;
	  s2 = old_search.search_string;
	  if (s1 == null)
	    {
	      if (s2 == null || s2.strip() == "")
		return false;
	      else
		return true;
	    }
	  else if (s2 == null)
	    {
	      if (s1 == null || s1.strip() == "")
		return false;
	      else
		return true;
	    }
	  else
	    return s1.strip () != s2.strip ();
	}
    }
  }
}
