/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *             Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
using Dee;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;
using GMenu;

namespace Unity.ApplicationsLens {

  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";

  public class Daemon : GLib.Object
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index zg_index;
    
    /* The searcher for online material may be null if it fails to load
     * the Xapian index from the Software Center */
    private Unity.Package.Searcher? pkgsearcher;
    public Unity.Package.Searcher appsearcher;
    
    /* Read the app ratings dumped by the Software Center */
    private Unity.Ratings.Database? ratings;

    private Unity.Lens lens;
    private Unity.Scope scope;
    
    private Unity.ApplicationsLens.Runner runner;

    /* For each section we have a set filtering query we use to restrict
     * Xapian queries to that type */
    private HashTable<string, string> type_queries;

    private Gee.List<string> image_extensions;
    private HashTable<string,Icon> file_icon_cache;
    
    /* We remember the previous search so we can figure out if we should do
     * incremental filtering of the result models */
    private Unity.LensSearch? previous_search;
    private Unity.LensSearch? previous_global_search;
        
    private PtrArray zg_templates;
    
    /* Gnome menu structure - also used to check whether apps are installed */
    private uint app_menu_changed_reindex_timeout = 0;
    private GMenu.Tree app_menu = null;

    private Regex? uri_regex;

    construct
    {
      populate_type_queries ();
      populate_zg_templates ();

      log = new Zeitgeist.Log();
      zg_index = new Zeitgeist.Index();
      
      pkgsearcher = new Unity.Package.Searcher ();
      if (pkgsearcher == null)
        {
          critical ("Failed to load Software Center index. 'Apps Available for Download' will not be listed");
        }
      
      try {
        ratings = new Unity.Ratings.Database ();
      } catch (FileError e) {
        warning (e.message);
        ratings = null;
      }
      
      /* Image file extensions in order of popularity */
      image_extensions = new Gee.ArrayList<string> ();
      image_extensions.add ("png");
      image_extensions.add ("xpm");
      image_extensions.add ("svg");
      image_extensions.add ("tiff");
      image_extensions.add ("ico");
      image_extensions.add ("tif");
      image_extensions.add ("jpg");
      
      previous_search = null;
      previous_global_search = null;
            
      build_app_menu_index ();
      
      file_icon_cache = new HashTable<string,Icon>(str_hash, str_equal);

      scope = new Unity.Scope ("/com/canonical/unity/scope/applications");
      //scope.icon = @"$(Config.PREFIX)/share/unity/themes/applications.png";
      
      /* Listen for changes to the lens scope search */
      scope.notify["active-search"].connect (
        (obj, pspec) => {
          var search = scope.active_search;

          if (!Utils.search_has_really_changed (previous_search, search))
            return;

          update_scope_search.begin (search);
          previous_search = search;
        }
      );

      /* Re-do the search if the filters changed */
      scope.filters_changed.connect (
        () => {
          var search = scope.active_search;
          if (search == null || search.search_string == null)
            return;

          update_scope_search.begin (search);
          previous_search = search;
          
        }
      );

      /* Listen for changes to the global search aka Dash search */
      scope.notify["active-global-search"].connect (
        (obj, pspec) => {
          var search = scope.active_global_search;

          if (!Utils.search_has_really_changed (previous_global_search, search))
            return;
          
          update_global_search.begin (search);
          previous_global_search = search;
        }
      );

      scope.activate_uri.connect (activate);

      /* Listen for changes in the installed applications */
      AppInfoManager.get_instance().changed.connect (on_appinfo_changed);
      
      /* Now start the RunEntry */
      runner = new Unity.ApplicationsLens.Runner (this);
      
      try {
        uri_regex = new Regex ("^[a-z]+:.+$");
      } catch (RegexError e) {
        uri_regex = null;
        critical ("Failed to compile URI regex. URL launching will be disabled");
      }
      
      lens = new Unity.Lens ("/com/canonical/unity/lens/applications", "applications");
      lens.search_hint = _("Search Applications");
      lens.visible = true;
      lens.search_in_global = true;
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);
      lens.export ();
    }

    /* Pre-populates the type queries so it's easier/faster to build our search */
    private void populate_type_queries ()
    {
      type_queries = new HashTable<string, string> (str_hash, str_equal);
      type_queries.insert ("all", "NOT category:XYZ");

      type_queries.insert ("accessories", "(category:Utility AND NOT category:Accessibility)");
      type_queries.insert ("education", "(category:Education AND NOT category:Science)");
      type_queries.insert ("game", "category:Game");
      type_queries.insert ("graphics", "category:Graphics");
      type_queries.insert ("internet", "category:Network");
      type_queries.insert ("fonts", "category:Fonts"); // FIXME: wtf?
      type_queries.insert ("office", "category:Office");
      type_queries.insert ("media", "category:AudioVideo");
      type_queries.insert ("customization", "category:Settings");
      type_queries.insert ("accessibility", "(category:Accessibility AND NOT category:Settings)");
      type_queries.insert ("developer", "category:Development"); // FIXME emacs.desktop should be added
      type_queries.insert ("science-and-engineering", "(category:Science OR category:Engineering)");
      type_queries.insert ("system", "(category:System OR category:Security)");
    }

    private void populate_categories ()
    {
      GLib.List<Unity.Category> categories = new GLib.List<Unity.Category> ();
      File icon_dir = File.new_for_path (ICON_PATH);

      var cat = new Unity.Category (_("Most Frequently Used"),
                                    new FileIcon (icon_dir.get_child ("group-mostused.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Installed"),
                                new FileIcon (icon_dir.get_child ("group-installed.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Apps Available for Download"),
                                new FileIcon (icon_dir.get_child ("group-downloads.svg")));
      categories.append (cat);

      lens.categories = categories;
    }

    private void populate_filters()
    {
      GLib.List<Unity.Filter> filters = new GLib.List<Unity.Filter> ();

      /* Type filter */
      {
        var filter = new RadioOptionFilter ("type", _("Type"));
        filter.sort_type = Unity.OptionsFilter.SortType.DISPLAY_NAME;
        
        filter.add_option ("accessories", _("Accessories"));
        filter.add_option ("education", _("Education"));
        filter.add_option ("game", _("Games"));
        filter.add_option ("graphics", _("Graphics"));
        filter.add_option ("internet", _("Internet"));
        filter.add_option ("fonts", _("Fonts"));
        filter.add_option ("office", _("Office"));
        filter.add_option ("media", _("Media"));
        filter.add_option ("customization", _("Customization"));
        filter.add_option ("accessibility", _("Accessibility"));
        filter.add_option ("developer", _("Developer"));
        filter.add_option ("science-and-engineering", _("Science & Engineering"));
        filter.add_option ("system", _("System"));

        filters.append (filter);
      }

      /* Rating filter */
      {
        var filter = new RatingsFilter("rating", _("Rating"));
        filters.append (filter);
      }

      lens.filters = filters;
    }
  
    /* Load xdg menu info and build a Xapian index over it.
     * Do throttled re-index if the menu changes */
    private bool build_app_menu_index ()
    {            
      if (app_menu == null)
        {
          debug ("Building initial application menu");
        
          /* We need INCLUDE_NODISPLAY to employ proper de-duping between
           * the Installed and Availabale categorys. If a NoDisplay app is installed,
           * eg. Evince, it wont otherwise be in the menu index, only in the
           * S-C index - thus show up in the Available category */
          app_menu = GMenu.Tree.lookup ("unity-lens-applications.menu",
                                        GMenu.TreeFlags.INCLUDE_NODISPLAY);
          
          app_menu.add_monitor ((menu) => {
            /* Reschedule the timeout if we already have one. The menu tree triggers
             * many change events during app installation. This way we wait the full
             * delay *after* the last change event has triggered */
            if (app_menu_changed_reindex_timeout != 0)
              Source.remove (app_menu_changed_reindex_timeout);
            
            app_menu_changed_reindex_timeout =
                                  Timeout.add_seconds (5, build_app_menu_index_and_result_models);
          });
        }
      
      debug ("Indexing application menu");
      appsearcher = new Unity.Package.Searcher.for_menu (app_menu);
      app_menu_changed_reindex_timeout = 0;
      
      return false;
    }

    /* Called when our app_menu structure changes - probably because something
     * has been installed or removed. We rebuild the index and update the
     * result models for global and scope. We need to update both because
     * we can't know exactly what Unity may be showing */
    private bool build_app_menu_index_and_result_models ()
    {
      build_app_menu_index ();
      
      debug ("Updating result models");
      previous_search = null;
      previous_global_search = null;
      scope.notify_property ("active-search");
      scope.notify_property ("active-global-search");
      
      return false;
    }

    private void populate_zg_templates ()
    {
      /* Create a template that activation of applications */
      zg_templates = new PtrArray.sized(1);
      var ev = new Zeitgeist.Event.full (ZG_ACCESS_EVENT, ZG_USER_ACTIVITY, "",
                             new Subject.full ("application://*",
                                               "", //NFO_SOFTWARE,
                                               "",
                                               "", "", "", ""));
      zg_templates.add ((ev as GLib.Object).ref());
    }

    private string prepare_zg_search_string (Unity.LensSearch? search,
                                             string type_id="all")
    {
      string s;
      if (search != null)
        s = search.search_string;
      else
        s = "";

      s = s.strip ();
      
      if (!s.has_suffix ("*") && s != "")
        s = s + "*";
      
      if (s != "")
        s = @"app:($s)";
      else
        return type_queries.lookup(type_id);
      
      if (type_id == "all")
        return s;
      else
        return s + @" AND $(type_queries.lookup(type_id))";
    }

    private async void update_scope_search (Unity.LensSearch? search)
    {
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();

      var model = scope.results_model;
      /* We'll clear the model once we finish waiting for the dbus-call
       * to finish, to prevent flicker. */

      debug ("Searching for: %s", search.search_string);
      
      var filter = scope.get_filter ("type") as RadioOptionFilter;
      Unity.FilterOption? option =  filter.get_active_option ();
      string type_id = option == null ? "all" : option.id;

      string pkg_search_string = prepare_pkg_search_string (search, type_id);

      bool has_search = !Utils.search_is_invalid (search);
      
      Timer timer = new Timer ();

      var zg_search_string = prepare_zg_search_string (search, type_id);

      /* Even though the Installed apps search is super fast, we wait here
       * for the Most Popular search to finish, because otherwise we'll update
       * the Installed category too soon and this will cause flicker
       * in the Dash. (lp:868192) */

      try {
        debug ("SEARCH_STRING: %s", zg_search_string);
        var results = yield zg_index.search (zg_search_string,
                                             new Zeitgeist.TimeRange.anytime(),
                                             zg_templates,
                                             Zeitgeist.StorageState.ANY,
                                             20,
                                             Zeitgeist.ResultType.MOST_POPULAR_SUBJECTS,
                                             null);

        model.clear ();
        append_events_with_category (results, model, Category.MOST_USED);
        
        timer.stop ();
        debug ("Entry search found %u/%u Most Used apps in %fms for query '%s'",
               results.size (), results.estimated_matches (),
               timer.elapsed()*1000, zg_search_string);

      } catch (GLib.Error e) {
        warning ("Error performing search '%s': %s",
                 search.search_string, e.message);
        model.clear ();
      }

      timer.start ();      
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var appresults = appsearcher.search (pkg_search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           has_search ?
                                              Unity.Package.Sort.BY_RELEVANCY :
                                              Unity.Package.Sort.BY_NAME);
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Category.INSTALLED);
      
      timer.stop ();
      debug ("Entry search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, pkg_search_string);

      /* We force a flush of the shared model's revision queue here because
       * generally the search for Most Used apps has a small result set,
       * while the one for Available apps may be huge. The huge result set
       * would slow the arrival of the Most Used results */
      (model as Dee.SharedModel).flush_revision_queue ();
      
      /* If we don't have a search we display 6 random apps */
      if (has_search && pkgsearcher != null)
        {
          timer.start ();
          var pkgresults = pkgsearcher.search (pkg_search_string, 50,
                                               Unity.Package.SearchType.PREFIX,
                                               Unity.Package.Sort.BY_RELEVANCY);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Category.AVAILABLE);
          timer.stop ();
          debug ("Entry search listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, pkg_search_string);
        }
      else if (pkgsearcher != null)
        {
          timer.start ();
          string? filter_query = prepare_pkg_search_string (null, type_id);
          var random_pkgresults = pkgsearcher.get_random_apps (filter_query, 12);
          add_pkg_search_result (random_pkgresults, installed_uris, available_uris,
                                 model, Category.AVAILABLE, 6);
          timer.stop ();
          debug ("Entry search listed %i random Available apps in %fms",
                 random_pkgresults.num_hits, timer.elapsed ()*1000);
        }
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });
      
      if (search != null)
        search.finished ();
    }
    
    private async void update_global_search (Unity.LensSearch? search)
    {
      var model = scope.global_results_model;
    
      model.clear ();
      
      if (Utils.search_is_invalid (search))
      {        
        return;
      }
      
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remember to thaw the notifys again! */
      scope.freeze_notify ();
      
      var search_string = prepare_pkg_search_string (search, "all");
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var timer = new Timer ();
      var appresults = appsearcher.search (search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           Unity.Package.Sort.BY_RELEVANCY);      
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Category.INSTALLED);
      
      timer.stop ();
      debug ("Global search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, search_string);

      // Dowloadable Apps search disabled from global search
      // See https://bugs.launchpad.net/unity-lens-applications/+bug/733669
      /*
      if (pkgsearcher != null)
        {      
          // We force a flush of the shared model's revision queue here because
          // generally the search for Installed apps is orders of magnitude faster
          // than the search for Available apps. So we can update the UI quicker         

          (model as Dee.SharedModel).flush_revision_queue ();
          
          timer.start ();
          var pkgresults = pkgsearcher.search (search_string, 20,
                                               Unity.Package.SearchType.PREFIX,
                                               Unity.Package.Sort.BY_RELEVANCY);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Category.AVAILABLE);
          timer.stop ();
          debug ("Global search listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, search_string);
        }
      */
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });
      
      if (search != null)
        search.finished ();
    }
    
    private string prepare_pkg_search_string (Unity.LensSearch? search, string type_id="all")
    {
      if (Utils.search_is_invalid (search))
        {
          if (type_id == "all")
            return "type:Application";
          else
            return @"type:Application AND $(type_queries.lookup(type_id))";
        }
      else
        {
          var s = search.search_string;

          s = s.strip ();
          
          /* The Xapian query parser seems to handle hyphens in a special way,
           * namely that it forces the joined tokens into a phrase query
           * no matter if it appears as the last word in a query and we have
           * the PARTIAL flag set on the query parser. This makes 'd-f' not
           * match 'd-feet' etc. */
          s = s.delimit ("-", ' ');

          if (type_id == "all")
            return @"type:Application AND $s";
          else
            return @"type:Application AND $(type_queries.lookup(type_id)) AND $s";
        }
    }
    
    public Icon find_pkg_icon (Unity.Package.PackageInfo pkginfo)
    {
      string desktop_id = Path.get_basename (pkginfo.desktop_file);
      bool installed = AppInfoManager.get_instance().lookup (desktop_id) != null;
      
      /* If the app is already installed we should be able to pull the
       * icon from the theme */
      if (installed)
        return new ThemedIcon (pkginfo.icon);
      
      /* App is not installed - we need to find the right icon in the bowels
       * of the software center */
      if (pkginfo.icon.has_prefix ("/"))
        {
          return new FileIcon (File.new_for_path (pkginfo.icon));
        }
      else
        {
          Icon icon = file_icon_cache.lookup (pkginfo.icon);
          
          if (icon != null)
            return icon;
          
          /* If the icon name contains a . it probably already have a
           * type postfix - so test icon name directly */
          string path;
          if ("." in pkginfo.icon)
            {
              path  = @"$(Config.DATADIR)/app-install/icons/$(pkginfo.icon)";
              if (FileUtils.test (path, FileTest.EXISTS))
                {
                  icon = new FileIcon (File.new_for_path (path));
                  file_icon_cache.insert (pkginfo.icon, icon);
                  return icon;
                }
            }
          
          /* Now try appending all the image extensions we know */
          foreach (var ext in image_extensions)
          {
            path = @"$(Config.DATADIR)/app-install/icons/$(pkginfo.icon).$(ext)";
            if (FileUtils.test (path, FileTest.EXISTS))
              {
                /* Got it! Cache the icon path and return the icon */
                icon = new FileIcon (File.new_for_path (path));
                file_icon_cache.insert (pkginfo.icon, icon);
                return icon;
              }
          }
        }
      
      /* Cache the fact that we couldn't find this icon */
      var icon = new ThemedIcon ("applications-other");
      file_icon_cache.insert (pkginfo.icon, icon);
      
      return icon;
    }

    private void on_appinfo_changed (string id, AppInfo? appinfo)
    {
      debug ("Application changed: %s", id);
      //update_scope_results_model.begin ();
    }
    
    private void add_pkg_search_result (Unity.Package.SearchResult results,
                                        Set<string> installed_uris,
                                        Set<string> available_uris,
                                        Dee.Model model,
                                        Category category,
                                        uint max_add=0)
    {
      var appmanager = AppInfoManager.get_instance();
      var ratings_filter = scope.get_filter ("rating") as RatingsFilter;
      uint n_added = 0;
    
      foreach (var pkginfo in results.results)
      {
      	if (pkginfo.desktop_file == null)
          continue;
        
        string desktop_id = Path.get_basename (pkginfo.desktop_file);
        string full_path;

        /* S-C uses "app_name:desktop_id.desktop", get rid of the prefix */
        int colon_pos = desktop_id.index_of (":");
        if (colon_pos > 0)
        {
          desktop_id = desktop_id[colon_pos+1:desktop_id.length];
        }

        AppInfo? app = appmanager.lookup (desktop_id);
        full_path = appmanager.get_path (desktop_id);
        
        /* De-dupe by 'application://foo.desktop' URI. Also note that we need
         * to de-dupe before we chuck out NoDisplay app infos, otherwise they'd
         * show up from alternate sources */
        string uri = @"application://$(desktop_id)";
        if (uri in installed_uris || uri in available_uris)
          continue;
        
        /* Extract basic metadata and register de-dupe keys */
        string display_name;
        string comment;
        switch (category)
        {
          case Category.INSTALLED:
            installed_uris.add (uri);
            display_name = app.get_display_name ();
            comment = app.get_description ();
            break;
          case Category.AVAILABLE:
            available_uris.add (uri);
            display_name = pkginfo.application_name;
            comment = "";
            break;
          default:
            warning (@"Illegal category for package search $(category)");
            continue;
        } 
        
        /* We can only chuck out NoDisplay and OnlyShowIn app infos after
         * we have registered a de-dupe key for them - which is done in the
         * switch block above) */
        if (app != null && !app.should_show ())
          continue;
                
        if (category == Category.AVAILABLE)
          {
            /* If we have an available item, which is not a dupe, but is
             * installed anyway, we weed it out here, because it's probably
             * left out from the Installed section because of some rule in the
             * .menu file */
            if (app != null)
              continue;
            
            /* Filter by app rating in Software Center if enabled */
            if (ratings != null && ratings_filter.rating > 0.00001)
              {
                Unity.Ratings.Result result;
                if (ratings.query (pkginfo.package_name, out result))
                  {
                    if (result.average_rating < ratings_filter.rating * 5 - 0.2)
                      continue;
                  }
                else
                  continue;
              }
            
            /* Apps that are not installed, ie. in the Available category
             * use the 'unity-install://pkgname/Full App Name' URI scheme,
             * but only use that after we've de-duped the results.
             * But only change the URI *after* we've de-duped the results! */
            uri = @"unity-install://$(pkginfo.package_name)/$(pkginfo.application_name)";
            available_uris.add (uri);            
          }
        
        Icon icon = find_pkg_icon (pkginfo);
        
        model.append (uri, icon.to_string (),
                      category,"application/x-desktop",
                      display_name != null ? display_name : "",
                      comment != null ? comment : "", 
		      full_path != null ? "file://" + full_path : "");
        
        /* Stop if we added the number of items requested */
        n_added++;
        if (max_add > 0 && n_added >= max_add)
          return;
      }
    }

    /**
     * Override of the default activation handler. The apps lens daemon
     * can handle activation of installable apps using the Software Center
     */
    public Unity.ActivationResponse activate (string uri)
    {
      string[] args;
      string exec_or_dir = null;
      if (uri.has_prefix ("unity-install://"))
        {
          unowned string pkg = uri.offset (16); // strip off "unity-install://" prefix
          debug ("Installing: %s", pkg);
          args = new string[2];
          args[0] = "software-center";
          args[1] = pkg;
        }
      else if (uri.has_prefix ("unity-runner://"))
        {
          string orig;
          orig = uri.offset (15);
          if (uri_regex != null && uri_regex.match (orig)) {
            try {
              AppInfo.launch_default_for_uri (orig, null);
            } catch (GLib.Error error) {
              warning ("Failed to launch URI %s", orig);
              return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
            }
            return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);

          } else {
            exec_or_dir = Utils.subst_tilde (orig);
            args = exec_or_dir.split (" ", 0);
            for (int i = 0; i < args.length; i++)
              args[i] = Utils.subst_tilde (args[i]);
          }
          this.runner.add_history (orig);
        }
      else
        {
          debug ("Declined activation of URI '%s': Expected URI scheme unity-install:// or unity-runner://", uri);
          return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
        }

      if ((exec_or_dir != null) && FileUtils.test (exec_or_dir, FileTest.IS_DIR))
      {
        try {
            AppInfo.launch_default_for_uri ("file://" + exec_or_dir, null);
        } catch (GLib.Error err) {
            warning ("Failed to open current folder '%s' in file manager: %s",
                     exec_or_dir, err.message);
            return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
        }
      }
      else
      {
          try {
            unowned string home_dir = GLib.Environment.get_home_dir ();
            Process.spawn_async (home_dir, args, null, SpawnFlags.SEARCH_PATH, null, null);
          } catch (SpawnError e) {
            warning ("Failed to spawn software-center or direct URI activation '%s': %s",
                     uri, e.message);
            return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
          }
      }

      return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
      
    }

    /* Appends the subject URIs from a set of Zeitgeist.Events to our Dee.Model
     * assuming that these events are already sorted */
    public void append_events_with_category (Zeitgeist.ResultSet events,
                                          Dee.Model results,
                                          uint category_id)
    {
      foreach (var ev in events)
        {
          string app_uri;
          if (ev.num_subjects () > 0)
            app_uri = ev.get_subject (0).get_uri ();
          else
            {
              warning ("Unexpected event without subject");
              continue;
            }
        
          /* Assert that we indeed have a known application as actor */
          string desktop_id = Utils.get_desktop_id_for_actor (app_uri);
          string full_path;

          var appmanager = AppInfoManager.get_instance ();
          AppInfo? app = appmanager.lookup (desktop_id);

          if (app == null)
            continue;
          
          if (!app.should_show ())
            continue;

          full_path = appmanager.get_path (desktop_id);
          string full_uri = full_path != null ? "file://" + full_path : app_uri;

          results.append (app_uri, app.get_icon().to_string(), category_id,
                          "application/x-desktop", app.get_display_name (),
                          app.get_description (), full_uri);
        }
    }
    
  } /* END: class Daemon */

} /* namespace */
