# ubuntuone.storageprotocol.tests.test_throttling -
#     Throttling tests
#
# Author: Facundo Batista <facundo@canonical.com>
#
# Copyright (C) 2009 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License version 3,
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for directory content serialization/unserialization."""

from __future__ import with_statement

from twisted.internet import defer, reactor
from twisted.trial.unittest import TestCase as TwistedTestCase

from ubuntuone.storageprotocol import client

class FakeClient(object):
    """Fake a Client class that is handy for tests."""

    def __init__(self):
        self.events = []

    def throttleReads(self):
        """Store a throttleReads event."""
        self.events.append("thR")

    def unthrottleReads(self):
        """Store a unthrottleReads event."""
        self.events.append("unthR")

    def throttleWrites(self):
        """Store a throttleWrites event."""
        self.events.append("thW")

    def unthrottleWrites(self):
        """Store a unthrottleWrites event."""
        self.events.append("unthW")


class TestProducingState(TwistedTestCase):
    """Test for filename validation and normalization."""

    timeout = 5

    def setUp(self):
        self.client = FakeClient()
        self.tscf = client.ThrottlingStorageClientFactory(3, 3)
        self.tscf.client = self.client

    def tearDown(self):
        self.tscf.unregisterProtocol(None)

    def test_under_write_limit(self):
        """Don't pas the write limit, no event."""
        d = defer.Deferred()
        self.tscf.registerWritten(2)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, [])
            d.callback(True)

        reactor.callLater(.1, check)
        return d

    def test_under_read_limit(self):
        """Don't pas the read limit, no event."""
        d = defer.Deferred()
        self.tscf.registerRead(2)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, [])
            d.callback(True)

        reactor.callLater(.1, check)
        return d

    def test_above_write_throttles(self):
        """Above the write limit, throttles."""
        d = defer.Deferred()
        self.tscf.registerWritten(4)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW"])
            d.callback(True)

        reactor.callLater(.1, check)
        return d

    def test_above_read_throttles(self):
        """Above the read limit, throttles."""
        d = defer.Deferred()
        self.tscf.registerRead(4)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR"])
            d.callback(True)

        reactor.callLater(.1, check)
        return d

    def test_above_write_throttles_unthrottles(self):
        """Above the write limit, throttles and unthrottles after 1s."""
        d = defer.Deferred()
        self.tscf.registerWritten(4)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW", "unthW"])
            d.callback(True)

        reactor.callLater(1.1, check)
        return d

    def test_above_read_throttles_unthrottles(self):
        """Above the read limit, throttles and unthrottles after 1s."""
        d = defer.Deferred()
        self.tscf.registerRead(4)

        def check():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR", "unthR"])
            d.callback(True)

        reactor.callLater(1.1, check)
        return d

    def test_very_above_write_throttles_unthrottles(self):
        """A lot above the write limit, throttles and unthrottles."""
        d = defer.Deferred()
        self.tscf.registerWritten(8)

        def check1():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW"])
            reactor.callLater(1, check2)

        def check2():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW", "unthW"])
            d.callback(True)

        reactor.callLater(1.1, check1)
        return d

    def test_very_above_read_throttles_unthrottles(self):
        """A lot above the read limit, throttles and unthrottles."""
        d = defer.Deferred()
        self.tscf.registerRead(8)

        def check1():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR"])
            reactor.callLater(1, check2)

        def check2():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR", "unthR"])
            d.callback(True)

        reactor.callLater(1.1, check1)
        return d

    def test_double_write(self):
        """Two writes on a row while throttling."""
        d = defer.Deferred()
        self.tscf.registerWritten(4)

        def check1():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW"])
            self.tscf.registerWritten(1)
            reactor.callLater(2, check2)

        def check2():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thW", "unthW"])
            d.callback(True)

        reactor.callLater(.1, check1)
        return d

    def test_double_read(self):
        """Two read on a row while throttling."""
        d = defer.Deferred()
        self.tscf.registerRead(4)

        def check1():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR"])
            self.tscf.registerWritten(1)
            reactor.callLater(2, check2)

        def check2():
            """Check for the correct events."""
            self.assertEqual(self.client.events, ["thR", "unthR"])
            d.callback(True)

        reactor.callLater(.1, check1)
        return d

