# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

# Unused import QtGui
# pylint: disable=W0611
from PyQt4 import QtGui, QtCore
# pylint: enable=W0611
from twisted.internet import defer

from ubuntuone.controlpanel import backend
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import KILOBYTES
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import preferences_ui


logger = setup_logging('qt.preferences')

CHECKED = QtCore.Qt.Checked
UNCHECKED = QtCore.Qt.Unchecked


def check_state(checked):
    """Return the QCheckBox check state according the value of 'checked'."""
    return CHECKED if checked else UNCHECKED


def tweak_speed(speed):
    """Return the proper speed in kilobytes.

    If speed < 0, return -1.
    If speed == 0, return 1 (almost nothing, the backed won't accept 0.
    If speed > 0, return the kilobytes presentation of that.

    """
    result = -1
    if speed > 0:
        result = speed * KILOBYTES
    elif speed == 0:
        result = 1
    return result


class PreferencesPanel(UbuntuOneBin):
    """The Preferences Tab Panel widget"""

    ui_class = preferences_ui
    logger = logger

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        info = yield self.backend.file_sync_settings_info()
        self.process_info(info)

    @log_call(logger.debug)
    def process_info(self, info):
        """Load settings info into the view."""
        autoconnect = check_state(info[backend.AUTOCONNECT_KEY])
        show_all_notifs = \
            check_state(info[backend.SHOW_ALL_NOTIFICATIONS_KEY])
        share_autosubscribe = \
            check_state(info[backend.SHARE_AUTOSUBSCRIBE_KEY])
        udf_autosubscribe = \
            check_state(info[backend.UDF_AUTOSUBSCRIBE_KEY])
        download_speed = info[backend.DOWNLOAD_KEY] // KILOBYTES
        upload_speed = info[backend.UPLOAD_KEY] // KILOBYTES

        self.ui.autoconnect_checkbox.setCheckState(autoconnect)
        self.ui.show_all_notifications_checkbox.setCheckState(show_all_notifs)
        self.ui.share_autosubscribe_checkbox.setCheckState(share_autosubscribe)
        self.ui.udf_autosubscribe_checkbox.setCheckState(udf_autosubscribe)
        # proper checkState will be set with the spinbox's valueChanged
        # signal handlers
        self.ui.download_speed_spinbox.setValue(download_speed)
        self.ui.upload_speed_spinbox.setValue(upload_speed)
        self.is_processing = False

    def process_speed_settings(self, new_value, checkbox):
        """A new speed limit was set."""
        if new_value < 0:
            checkbox.setCheckState(UNCHECKED)
        else:
            checkbox.setCheckState(CHECKED)

    # Invalid name "on_{down, up}load_speed_spinbox_valueChanged"
    # pylint: disable=C0103

    @QtCore.pyqtSlot(int)
    def on_download_speed_spinbox_valueChanged(self, new_value):
        """A new download speed was set."""
        self.process_speed_settings(new_value,
                                    self.ui.limit_downloads_checkbox)

    @QtCore.pyqtSlot(int)
    def on_upload_speed_spinbox_valueChanged(self, new_value):
        """A new upload speed was set."""
        self.process_speed_settings(new_value,
                                    self.ui.limit_uploads_checkbox)

    def on_limit_downloads_checkbox_stateChanged(self, new_state):
        """Limit download speed checkbox changed."""
        if new_state == UNCHECKED:
            self.ui.download_speed_spinbox.setValue(-1)

    def on_limit_uploads_checkbox_stateChanged(self, new_state):
        """Limit upload speed checkbox changed."""
        if new_state == UNCHECKED:
            self.ui.upload_speed_spinbox.setValue(-1)

    @QtCore.pyqtSlot()
    @defer.inlineCallbacks
    def on_apply_changes_button_clicked(self):
        """The apply changes button was clicked."""
        autoconnect = self.ui.autoconnect_checkbox.checkState()
        notifications = self.ui.show_all_notifications_checkbox.checkState()
        share_autosubscribe = self.ui.share_autosubscribe_checkbox.checkState()
        udf_autosubscribe = self.ui.udf_autosubscribe_checkbox.checkState()
        download_speed = tweak_speed(self.ui.download_speed_spinbox.value())
        upload_speed = tweak_speed(self.ui.upload_speed_spinbox.value())

        settings = {
            backend.AUTOCONNECT_KEY: autoconnect == CHECKED,
            backend.SHOW_ALL_NOTIFICATIONS_KEY: notifications == CHECKED,
            backend.SHARE_AUTOSUBSCRIBE_KEY: share_autosubscribe == CHECKED,
            backend.UDF_AUTOSUBSCRIBE_KEY: udf_autosubscribe == CHECKED,
            backend.DOWNLOAD_KEY: download_speed,
            backend.UPLOAD_KEY: upload_speed,
        }

        self.is_processing = True
        yield self.backend.change_file_sync_settings(settings)
        self.is_processing = False

    @QtCore.pyqtSlot()
    @defer.inlineCallbacks
    def on_restore_defaults_button_clicked(self):
        """The restore defaults button was clicked."""
        self.is_processing = True
        yield self.backend.restore_file_sync_settings()
        settings = yield self.backend.file_sync_settings_info()

        self.process_info(settings)
