# -*- coding: utf-8 -*-
#
# Copyright 2010-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the service when accessing SyncDaemon."""

import os
import uuid

from twisted.internet import defer
from twisted.internet.defer import inlineCallbacks, returnValue
from ubuntuone.platform import tools
from ubuntuone.syncdaemon.interaction_interfaces import bool_str

from ubuntuone.controlpanel import sd_client
from ubuntuone.controlpanel.tests import CustomError, TestCase

# Instance of 'SyncDaemonTool' has no 'foo' member
# pylint: disable=E1101


SAMPLE_LIMITS = {'upload': 999, 'download': 838}


class FakedSyncDaemonTool(object):
    """Fake the SyncDaemonTool."""

    settings = {
        'autoconnect': True,
        'show_all_notifications': True,
        'share_autosubscribe': True,
        'udf_autosubscribe': True,
        'file_sync_enabled': True,
        'throttling_enabled': False,
        'limits': {'download': -1, 'upload': -1},
    }
    status_dict = {
        'name': 'QUEUE_MANAGER',
        'connection': 'With User With Network',
        'description': 'processing the commands pool',
        'is_connected': True,
        'is_error': False,
        'is_online': True,
        'queues': 'IDLE',
    }
    called = {}
    shares = {}
    folders = {}
    home_dir = u'/home/ñandu úser'
    root_dir = os.path.join(home_dir, u'Ubuntu One')
    shares_dir = os.path.join(home_dir, u'.cache/ubuntuone/shares')
    shares_dir_link = os.path.join(root_dir, u'Shared With Me')

    @classmethod
    def create_share(cls, name, accepted=False, subscribed=False):
        """Add a new share (fake)."""
        sid = unicode(uuid.uuid4())
        path = u'/home/tester/.hidden/shares/%s from The Othr User' % name
        share = {
            u'name': name, u'subscribed': bool_str(subscribed),
            u'accepted': bool_str(accepted),
            u'generation': u'36', u'type': u'Share',
            u'node_id': u'ca3a1cec-09d2-485e-9685-1a5180bd6441',
            u'volume_id': sid, u'access_level': u'View',
            u'other_username': u'https://login.ubuntu.com/+id/nHRnYmz',
            u'other_visible_name': u'The Other User',
            u'free_bytes': u'2146703403', u'path': path,
        }
        cls.shares[sid] = share
        return share

    @classmethod
    def create_udf(cls, path, subscribed=False):
        """Create a new faked folder."""
        fid = unicode(uuid.uuid4())
        folder = {}
        if isinstance(path, str):
            path = path.decode('utf-8')
        folder[u'path'] = folder[u'suggested_path'] = path
        folder[u'volume_id'] = fid
        folder[u'subscribed'] = bool_str(subscribed)
        folder[u'node_id'] = u'a18f4cbd-a846-4405-aaa1-b28904817089'
        folder[u'generation'] = u''
        folder[u'type'] = u'UDF'
        cls.folders[fid] = folder
        return folder

    @classmethod
    def _set_share_attr(cls, share_id, attr, value):
        """Set values to shares."""
        if share_id not in cls.shares:
            raise CustomError('share_id %r not in shares %r.' %
                              (share_id, cls.shares))

        value = bool_str(value)
        cls.shares[share_id][attr] = value
        return share_id

    @classmethod
    def _set_folder_attr(cls, folder_id, attr, value):
        """Set values to folders."""
        if folder_id not in cls.folders:
            raise CustomError('folder_id %r not in folders %r.' %
                              (folder_id, cls.folders))

        value = bool_str(value)
        cls.folders[folder_id][attr] = value
        return folder_id

    @inlineCallbacks
    def accept_share(self, share_id):
        """Accept the share with id: share_id."""
        yield self._set_share_attr(share_id, u'accepted', True)

    @inlineCallbacks
    def reject_share(self, share_id):
        """Reject the share with id: share_id."""
        yield self._set_share_attr(share_id, u'accepted', False)

    @inlineCallbacks
    def subscribe_share(self, share_id):
        """Subscribe to a share given its id."""
        yield self._set_share_attr(share_id, u'subscribed', True)

    @inlineCallbacks
    def unsubscribe_share(self, share_id):
        """Unsubscribe from a share given its id."""
        yield self._set_share_attr(share_id, u'subscribed', False)

    @inlineCallbacks
    def get_shares(self):
        """Get the list of shares (accepted or not)."""
        result = yield self.shares.values()
        returnValue(sorted(result))

    def refresh_shares(self):
        """Call refresh_shares method."""
        self.called['refresh_shares'] = None

    def offer_share(self, path, username, name, access_level):
        """Offer a share at the specified path to user with id: username."""
        self.called['offer_share'] = (path, username, name, access_level)

    def list_shared(self):
        """Get the list of the shares "shared"/created/offered."""
        self.called['list_shared'] = None

    def validate_path(self, path):
        """Validate a path for folder creation."""
        return path not in [f['path'] for f in self.folders.itervalues()]

    def create_folder(self, path):
        """Create a user defined folder in the specified path."""
        if path == '':  # simulate an error
            raise CustomError(path)

        folder = self.create_udf(path)
        return folder

    def delete_folder(self, folder_id):
        """Delete a user defined folder given its id."""
        if folder_id not in self.folders:
            raise CustomError(folder_id)

        folder = self.folders.pop(folder_id)
        return folder

    @inlineCallbacks
    def subscribe_folder(self, folder_id):
        """Subscribe to a user defined folder given its id."""
        yield self._set_folder_attr(folder_id, u'subscribed', True)

    @inlineCallbacks
    def unsubscribe_folder(self, folder_id):
        """Unsubscribe from a user defined folder given its id."""
        yield self._set_folder_attr(folder_id, u'subscribed', False)

    @inlineCallbacks
    def get_folders(self):
        """Return the list of folders (a list of dicts)."""
        result = yield self.folders.values()
        returnValue(sorted(result))

    def get_folder_info(self, path):
        """Call the get_info method for a UDF path."""
        for _, content in self.folders.iteritems():
            if content['path'] == path:
                return content
        else:
            return {}

    def get_metadata(self, path):
        """Call FileSystem.get_metadata."""
        self.called['get_metadata'] = path

    def change_public_access(self, path, is_public):
        """Change the public access for a given path."""
        self.called['change_public_access'] = (path, is_public)

    def quit(self):
        """Quit the syncdaemon."""
        self.called['quit'] = None

    def connect(self):
        """Connect syncdaemon."""
        self.called['connect'] = None

    def disconnect(self):
        """Disconnect syncdaemon."""
        self.called['disconnect'] = None

    def get_status(self):
        """Get the current_status dict."""
        return self.status_dict

    def start(self):
        """Start syncdaemon if it's not running."""
        self.called['start'] = None

    def get_throttling_limits(self):
        """Return a dict with the read and write limits."""
        return self.settings['limits']

    def set_throttling_limits(self, read_limit, write_limit):
        """Set the read and write limits."""
        self.settings['limits']["download"] = read_limit
        self.settings['limits']["upload"] = write_limit

    def is_throttling_enabled(self):
        """Check if throttling is enabled."""
        return self.settings['throttling_enabled']

    def enable_throttling(self, enabled):
        """Enable/disable throttling."""
        self.settings['throttling_enabled'] = enabled

    def is_files_sync_enabled(self):
        """Check if files sync is enabled."""
        return self.settings['file_sync_enabled']

    def enable_files_sync(self, enabled):
        """Enable/disable file sync."""
        self.settings['file_sync_enabled'] = enabled

    def is_autoconnect_enabled(self):
        """Check if autoconnect is enabled."""
        return self.settings['autoconnect']

    def enable_autoconnect(self, enabled):
        """Enable/disable autoconnect."""
        self.settings['autoconnect'] = enabled

    def is_show_all_notifications_enabled(self):
        """Check if show_all_notifications is enabled."""
        return self.settings['show_all_notifications']

    def enable_show_all_notifications(self, enabled):
        """Enable/disable show_all_notifications."""
        self.settings['show_all_notifications'] = enabled

    def is_share_autosubscribe_enabled(self):
        """Check if share_autosubscribe is enabled."""
        return self.settings['share_autosubscribe']

    def enable_share_autosubscribe(self, enabled):
        """Enable/disable share_autosubscribe."""
        self.settings['share_autosubscribe'] = enabled

    def is_udf_autosubscribe_enabled(self):
        """Check if udf_autosubscribe is enabled."""
        return self.settings['udf_autosubscribe']

    def enable_udf_autosubscribe(self, enabled):
        """Enable/disable udf_autosubscribe."""
        self.settings['udf_autosubscribe'] = enabled

    def refresh_volumes(self):
        """Request the volumes list to the server."""
        self.called['refresh_volumes'] = None

    def rescan_from_scratch(self, volume_id):
        """Request a rescan from scratch for volume_id."""
        self.called['rescan_from_scratch'] = volume_id

    def get_home_dir(self):
        """Return the root directory."""
        return self.home_dir

    def get_root_dir(self):
        """Return the root directory."""
        return self.root_dir

    def get_shares_dir(self):
        """Return the shares directory."""
        return self.shares_dir

    def get_shares_dir_link(self):
        """Return the shares link directory."""
        return self.shares_dir_link

    def connect_signal(self, signal_name, handler):
        """Set the handler for the status changed signal."""
        self.called[signal_name] = handler


class BaseTestCase(TestCase):
    """Test for the syncdaemon client methods."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(BaseTestCase, self).setUp()
        self.patch(tools, 'SyncDaemonTool', FakedSyncDaemonTool)
        self.sd = sd_client.SyncDaemonClient()
        self.sd.proxy.shares.clear()
        self.sd.proxy.folders.clear()
        self.sd.proxy.called.clear()

    def fake_fail(self, *a):
        """Fake a failure."""
        return defer.fail(CustomError(a))


class ThrottlingTestCase(BaseTestCase):
    """Test for the throttling syncdaemon client methods."""

    @inlineCallbacks
    def test_throttle_limits_returned(self):
        """When SD returns the limits, they are handled."""
        limits = yield self.sd.get_throttling_limits()
        self.assertEqual(limits, self.sd.proxy.settings['limits'])

    @inlineCallbacks
    def test_getting_throttle_limits_throws_an_error(self):
        """Handle error when getting the limits."""
        self.patch(self.sd.proxy, 'get_throttling_limits',
                   self.fake_fail)
        yield self.assertFailure(self.sd.get_throttling_limits(),
                                 CustomError)

    @inlineCallbacks
    def test_throttle_limits_set(self):
        """Setting the limits."""
        yield self.sd.set_throttling_limits(SAMPLE_LIMITS)
        actual = self.sd.get_throttling_limits()
        self.assertEqual(actual, SAMPLE_LIMITS)

    @inlineCallbacks
    def test_setting_throttle_limits_throws_an_error(self):
        """Handle error when setting the limits."""
        self.patch(self.sd.proxy, 'set_throttling_limits',
                   self.fake_fail)
        d = self.sd.set_throttling_limits(SAMPLE_LIMITS)
        yield self.assertFailure(d, CustomError)

    @inlineCallbacks
    def test_throttle_get(self):
        """Getting the throttling state."""
        self.sd.proxy.settings['throttling_enabled'] = False
        throttling = yield self.sd.bandwidth_throttling_enabled()
        self.assertEqual(throttling, False)

        self.sd.proxy.settings['throttling_enabled'] = True
        throttling = yield self.sd.bandwidth_throttling_enabled()
        self.assertEqual(throttling, True)

    @inlineCallbacks
    def test_throttle_get_throws_an_error(self):
        """Handle error when getting the throttling state."""
        self.patch(self.sd.proxy, 'is_throttling_enabled',
                   self.fake_fail)
        yield self.assertFailure(self.sd.bandwidth_throttling_enabled(),
                                 CustomError)

    @inlineCallbacks
    def test_throttle_enable(self):
        """Enabling bw throttling."""
        self.sd.proxy.settings['throttling_enabled'] = False
        yield self.sd.enable_bandwidth_throttling()
        self.assertTrue(self.sd.proxy.settings['throttling_enabled'])

    @inlineCallbacks
    def test_throttle_enable_throws_an_error(self):
        """Handle error when enabling throttling."""
        self.patch(self.sd.proxy, 'enable_throttling',
                   self.fake_fail)
        yield self.assertFailure(self.sd.enable_bandwidth_throttling(),
                                 CustomError)

    @inlineCallbacks
    def test_throttle_disable(self):
        """Disabling bw throttling."""
        self.sd.proxy.settings['throttling_enabled'] = True
        yield self.sd.disable_bandwidth_throttling()
        self.assertFalse(self.sd.proxy.settings['throttling_enabled'])

    @inlineCallbacks
    def test_throttle_disable_throws_an_error(self):
        """Handle error when disabling throttling."""
        self.patch(self.sd.proxy, 'enable_throttling',
                   self.fake_fail)
        yield self.assertFailure(self.sd.disable_bandwidth_throttling(),
                                 CustomError)


class NotificationsTestCase(BaseTestCase):
    """Test for the show_all_notifications syncdaemon client methods."""

    # No value passed for parameter '_' in function call
    # pylint: disable=E1120

    name = 'show_all_notifications'

    @defer.inlineCallbacks
    def setUp(self):
        yield super(NotificationsTestCase, self).setUp()
        self.getter = getattr(self.sd, '%s_enabled' % self.name)
        self.enabler = getattr(self.sd, 'enable_%s' % self.name)
        self.disabler = getattr(self.sd, 'disable_%s' % self.name)

    @inlineCallbacks
    def test_get_value(self):
        """Getting the show_all_notifications state."""
        self.sd.proxy.settings[self.name] = False
        state = yield self.getter()
        self.assertEqual(state, False)
        self.sd.proxy.settings[self.name] = True
        state = yield self.getter()
        self.assertEqual(state, True)

    @inlineCallbacks
    def test_get_value_throws_an_error(self):
        """Handle error when getting the show_all_notifications state."""
        self.patch(self.sd.proxy, 'is_%s_enabled' % self.name,
                   self.fake_fail)
        yield self.assertFailure(self.getter(), CustomError)

    @inlineCallbacks
    def test_enable(self):
        """Enabling show_all_notifications."""
        self.sd.proxy.settings[self.name] = False
        yield self.enabler()
        self.assertTrue(self.sd.proxy.settings[self.name])

    @inlineCallbacks
    def test_enable_throws_an_error(self):
        """Handle error when enabling show_all_notifications."""
        self.patch(self.sd.proxy, 'enable_%s' % self.name,
                   self.fake_fail)
        yield self.assertFailure(self.enabler(), CustomError)

    @inlineCallbacks
    def test_disable(self):
        """Disabling show_all_notifications."""
        self.sd.proxy.settings[self.name] = True
        yield self.disabler()
        value = self.sd.proxy.settings[self.name]
        self.assertFalse(value)

    @inlineCallbacks
    def test_disable_throws_an_error(self):
        """Handle error when disabling show_all_notifications."""
        self.patch(self.sd.proxy, 'enable_%s' % self.name,
                   self.fake_fail)
        yield self.assertFailure(self.disabler(), CustomError)


class AutoconnectTestCase(NotificationsTestCase):
    """Test for the autoconnect syncdaemon client methods."""

    name = 'autoconnect'


class ShareAutosubscribeTestCase(NotificationsTestCase):
    """Test for the share_autosubscribe syncdaemon client methods."""

    name = 'share_autosubscribe'


class UDFAutosubscribeTestCase(NotificationsTestCase):
    """Test for the udf_autosubscribe syncdaemon client methods."""

    name = 'udf_autosubscribe'


class FoldersTestCase(BaseTestCase):
    """Test for the shares syncdaemon client methods."""

    @inlineCallbacks
    def test_get_folders(self):
        """Retrieve folders info list."""
        folder1 = self.sd.proxy.create_udf(path='~/bar/baz')
        folder2 = self.sd.proxy.create_udf(path='~/bar/foo',
                                                 subscribed=False)
        result = yield self.sd.get_folders()

        self.assertEqual(result, sorted([folder1, folder2]))

    @inlineCallbacks
    def test_get_folders_error(self):
        """Handle error when retrieving current syncdaemon status."""
        self.patch(self.sd.proxy, 'get_folders', self.fake_fail)
        yield self.assertFailure(self.sd.get_folders(), CustomError)

    @inlineCallbacks
    def test_validate_path(self):
        """Check if a folder path is valid."""
        path = '~/bar/baz'
        result = yield self.sd.validate_path(path)

        self.assertTrue(result)

        yield self.sd.create_folder(path)
        result = yield self.sd.validate_path(path)

        self.assertFalse(result)

    @inlineCallbacks
    def test_create_folder(self):
        """Create a new folder."""
        path = '~/bar/baz'
        folder_info = yield self.sd.create_folder(path)

        expected = self.sd.proxy.folders[folder_info['volume_id']]
        self.assertEqual(expected, folder_info)

    @inlineCallbacks
    def test_create_folder_error(self):
        """Create a new folder fails."""
        self.patch(self.sd.proxy, 'create_folder', self.fake_fail)
        path = ''
        failure = yield self.assertFailure(self.sd.create_folder(path=path),
                                           CustomError)
        self.assertEqual(failure[0], (path,))

    @inlineCallbacks
    def test_subscribe_folder(self):
        """Subscribe to a folder."""
        folder_info = self.sd.proxy.create_udf(path='~/bar/foo',
                                                     subscribed=False)
        fid = folder_info['volume_id']
        yield self.sd.subscribe_folder(fid)

        result = yield self.sd.get_folders()
        expected, = filter(lambda folder: folder['volume_id'] == fid, result)
        self.assertEqual(expected['subscribed'], 'True')

    @inlineCallbacks
    def test_subscribe_folder_error(self):
        """Subscribe to a folder."""
        self.patch(self.sd.proxy, 'subscribe_folder', self.fake_fail)
        fid = u'does not exist'
        failure = yield self.assertFailure(self.sd.subscribe_folder(fid),
                                           CustomError)
        self.assertEqual(failure[0], (fid,))

    @inlineCallbacks
    def test_unsubscribe_folder(self):
        """Unsubscribe to a folder."""
        folder_info = self.sd.proxy.create_udf(path='~/bar/foo',
                                                     subscribed=True)
        fid = folder_info['volume_id']

        yield self.sd.unsubscribe_folder(fid)

        self.assertEqual(self.sd.proxy.folders[fid]['subscribed'], '')

    @inlineCallbacks
    def test_unsubscribe_folder_error(self):
        """Unsubscribe to a folder."""
        self.patch(self.sd.proxy, 'unsubscribe_folder', self.fake_fail)
        fid = u'does not exist'
        failure = yield self.assertFailure(self.sd.unsubscribe_folder(fid),
                                           CustomError)
        self.assertEqual(failure[0], (fid,))


class SharesTestCase(BaseTestCase):
    """Test for the shares syncdaemon client methods."""

    @inlineCallbacks
    def test_get_shares(self):
        """Retrieve shares info list."""
        share1 = self.sd.proxy.create_share(name='first, test me!')
        share2 = self.sd.proxy.create_share(name='and test me more!',
                                                accepted=True)
        share3 = self.sd.proxy.create_share(name='last but not least',
                                                accepted=True, subscribed=True)

        result = yield self.sd.get_shares()

        self.assertEqual(result, sorted([share1, share2, share3]))

    @inlineCallbacks
    def test_get_shares_error(self):
        """Handle error when retrieving current syncdaemon status."""
        self.patch(self.sd.proxy, 'get_shares', self.fake_fail)
        yield self.assertFailure(self.sd.get_shares(), CustomError)

    @inlineCallbacks
    def test_subscribe_share(self):
        """Subscribe to a share."""
        share = self.sd.proxy.create_share(name='to be subscribed',
                                               accepted=True, subscribed=False)
        sid = share['volume_id']

        yield self.sd.subscribe_share(sid)

        result = yield self.sd.get_shares()
        expected, = filter(lambda share: share['volume_id'] == sid, result)
        self.assertEqual(expected['subscribed'], 'True')

    @inlineCallbacks
    def test_subscribe_share_error(self):
        """Subscribe to a share."""
        sid = u'does not exist'
        yield self.assertFailure(self.sd.subscribe_share(sid), CustomError)

    @inlineCallbacks
    def test_unsubscribe_share(self):
        """Unsubscribe to a share."""
        share = self.sd.proxy.create_share(name='to be unsubscribed',
                                                accepted=True, subscribed=True)
        sid = share['volume_id']

        yield self.sd.unsubscribe_share(sid)

        result = yield self.sd.get_shares()
        expected, = filter(lambda share: share['volume_id'] == sid, result)
        self.assertEqual(expected['subscribed'], '')

    @inlineCallbacks
    def test_unsubscribe_share_error(self):
        """Unsubscribe to a share."""
        sid = u'does not exist'
        yield self.assertFailure(self.sd.unsubscribe_share(sid), CustomError)


class FileSyncTestCase(BaseTestCase):
    """Test for the file sync syncdaemon client methods."""

    @inlineCallbacks
    def test_file_sync_enabled(self):
        """Retrieve whether file sync is enabled."""
        expected = object()
        self.sd.proxy.settings['file_sync_enabled'] = expected

        enabled = yield self.sd.file_sync_enabled()

        self.assertEqual(expected, enabled)

    @inlineCallbacks
    def test_enable_file_sync(self):
        """Enable the file sync service."""
        yield self.sd.enable_file_sync()

        self.assertTrue(self.sd.proxy.settings['file_sync_enabled'])

    @inlineCallbacks
    def test_disable_file_sync(self):
        """Disable the file sync service."""
        yield self.sd.disable_file_sync()

        self.assertFalse(self.sd.proxy.settings['file_sync_enabled'])

    @inlineCallbacks
    def test_connect_file_sync(self):
        """Set if file sync is enabled or not."""
        yield self.sd.connect_file_sync()

        self.assertEqual(self.sd.proxy.called['connect'], None)

    @inlineCallbacks
    def test_disconnect_file_sync(self):
        """Set if file sync is enabled or not."""
        yield self.sd.disconnect_file_sync()

        self.assertEqual(self.sd.proxy.called['disconnect'], None)

    @inlineCallbacks
    def test_start_file_sync(self):
        """Set if file sync is enabled or not."""
        yield self.sd.start_file_sync()

        self.assertEqual(self.sd.proxy.called['start'], None)

    @inlineCallbacks
    def test_stop_file_sync(self):
        """Set if file sync is enabled or not."""
        yield self.sd.stop_file_sync()

        self.assertEqual(self.sd.proxy.called['quit'], None)

    @inlineCallbacks
    def test_set_status_changed_handler(self):
        """Connect a handler to the status changed signal."""
        sample_handler = object()
        yield self.sd.set_status_changed_handler(sample_handler)
        self.assertEqual(sample_handler, self.sd.proxy.called['StatusChanged'])


class BasicTestCase(BaseTestCase):
    """Test for the basic syncdaemon client methods."""

    @inlineCallbacks
    def test_get_root_dir(self):
        """Retrieve current syncdaemon root dir."""
        root = yield self.sd.get_root_dir()

        self.assertEqual(self.sd.proxy.root_dir, root)

    @inlineCallbacks
    def test_get_shares_dir(self):
        """Retrieve current syncdaemon shares dir."""
        result = yield self.sd.get_shares_dir()

        self.assertEqual(self.sd.proxy.shares_dir, result)

    @inlineCallbacks
    def test_get_shares_dir_link(self):
        """Retrieve current syncdaemon shares dir."""
        result = yield self.sd.get_shares_dir_link()

        self.assertEqual(self.sd.proxy.shares_dir_link, result)

    @inlineCallbacks
    def test_get_current_status(self):
        """Retrieve current syncdaemon status."""
        status = yield self.sd.get_current_status()

        self.assertEqual(self.sd.proxy.status_dict, status)

    @inlineCallbacks
    def test_get_current_status_error(self):
        """Handle error when retrieving current syncdaemon status."""
        self.patch(self.sd.proxy, 'get_status', self.fake_fail)
        yield self.assertFailure(self.sd.get_current_status(), CustomError)


class PublicFilesTestCase(BaseTestCase):
    """Test for the public files syncdaemon client methods."""

    @inlineCallbacks
    def test_set_public_files_list_handler(self):
        """Connect a handler to the public files signal."""
        sample_handler = object()
        yield self.sd.set_public_files_list_handler(sample_handler)
        self.assertEqual(sample_handler,
            self.sd.proxy.called['PublicFilesList'])

    @inlineCallbacks
    def test_set_public_access_changed_handler(self):
        """Connect a handler to the public access changed signal."""
        sample_handler = object()
        yield self.sd.set_public_access_changed_handler(sample_handler)
        self.assertEqual(sample_handler,
            self.sd.proxy.called['PublicAccessChanged'])

    @inlineCallbacks
    def test_set_public_access_change_error_handler(self):
        """Connect a handler to the public access change error signal."""
        sample_handler = object()
        yield self.sd.set_public_access_change_error_handler(sample_handler)
        self.assertEqual(sample_handler,
            self.sd.proxy.called['PublicAccessChangeError'])
