/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "syncdaemon-credentials.h"

G_DEFINE_TYPE(SyncdaemonCredentials, syncdaemon_credentials, G_TYPE_OBJECT)

struct _SyncdaemonCredentialsPrivate {
	gchar *consumer_key;
	gchar *consumer_secret;
	gchar *token;
	gchar *token_secret;
};

static void
syncdaemon_credentials_finalize (GObject *object)
{
	SyncdaemonCredentials *credentials = SYNCDAEMON_CREDENTIALS (object);

	if (credentials->priv != NULL) {
		if (credentials->priv->consumer_key != NULL)
			g_free (credentials->priv->consumer_key);
		if (credentials->priv->consumer_secret != NULL)
			g_free (credentials->priv->consumer_secret);
		if (credentials->priv->token != NULL)
			g_free (credentials->priv->token);
		if (credentials->priv->token_secret != NULL)
			g_free (credentials->priv->token_secret);

		g_free (credentials->priv);
	}

	G_OBJECT_CLASS (syncdaemon_credentials_parent_class)->finalize (object);
}

static void
syncdaemon_credentials_class_init (SyncdaemonCredentialsClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_credentials_finalize;
}

static void
syncdaemon_credentials_init (SyncdaemonCredentials *credentials)
{
	credentials->priv = g_new0 (SyncdaemonCredentialsPrivate, 1);
}

/**
 * syncdaemon_credentials_new:
 */
SyncdaemonCredentials *
syncdaemon_credentials_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_CREDENTIALS, NULL);
}

/**
 * syncdaemon_credentials_new_from_hash_table:
 */
SyncdaemonCredentials *
syncdaemon_credentials_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonCredentials *credentials;

	credentials = g_object_new (SYNCDAEMON_TYPE_CREDENTIALS, NULL);
	if (hash != NULL) {
		syncdaemon_credentials_set_consumer_key (
			credentials,
			g_hash_table_lookup (hash, "consumer_key"));
		syncdaemon_credentials_set_consumer_secret (
			credentials,
			g_hash_table_lookup (hash, "consumer_secret"));
		syncdaemon_credentials_set_token (
			credentials,
			g_hash_table_lookup (hash, "token"));
		syncdaemon_credentials_set_token_secret (
			credentials,
			g_hash_table_lookup (hash, "token_secret"));
	}

	return credentials;
}

/**
 * syncdaemon_credentials_get_oauth_consumer_token:
 */
const gchar *
syncdaemon_credentials_get_consumer_key (SyncdaemonCredentials *credentials)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials), NULL);

	return (const gchar *) credentials->priv->consumer_key;
}

/**
 * syncdaemon_credentials_set_consumer_key:
 */
void
syncdaemon_credentials_set_consumer_key (SyncdaemonCredentials *credentials, const gchar *consumer_key)
{
	g_return_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials));

	if (credentials->priv->consumer_key != NULL)
		g_free (credentials->priv->consumer_key);

	credentials->priv->consumer_key = g_strdup (consumer_key);
}

/**
 * syncdaemon_credentials_get_consumer_secret:
 */
const gchar *
syncdaemon_credentials_get_consumer_secret (SyncdaemonCredentials *credentials)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials), NULL);

	return (const gchar *) credentials->priv->consumer_secret;
}

/**
 * syncdaemon_credentials_set_consumer_secret:
 */
void
syncdaemon_credentials_set_consumer_secret (SyncdaemonCredentials *credentials, const gchar *consumer_secret)
{
	g_return_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials));

	if (credentials->priv->consumer_secret != NULL)
		g_free (credentials->priv->consumer_secret);

	credentials->priv->consumer_secret = g_strdup (consumer_secret);
}

/**
 * syncdaemon_credentials_get_token:
 */
const gchar *
syncdaemon_credentials_get_token (SyncdaemonCredentials *credentials)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials), NULL);

	return (const gchar *) credentials->priv->token;
}

/**
 * syncdaemon_credentials_set_token:
 */
void
syncdaemon_credentials_set_token (SyncdaemonCredentials *credentials, const gchar *token)
{
	g_return_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials));

	if (credentials->priv->token != NULL)
		g_free (credentials->priv->token);

	credentials->priv->token = g_strdup (token);
}

/**
 * syncdaemon_credentials_get_token_secret:
 */
const gchar *
syncdaemon_credentials_get_token_secret (SyncdaemonCredentials *credentials)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials), NULL);

	return (const gchar *) credentials->priv->token_secret;
}

/**
 * syncdaemon_credentials_set_token_secret:
 */
void
syncdaemon_credentials_set_token_secret (SyncdaemonCredentials *credentials, const gchar *token_secret)
{
	g_return_if_fail (SYNCDAEMON_IS_CREDENTIALS (credentials));

	if (credentials->priv->token_secret != NULL)
		g_free (credentials->priv->token_secret);

	credentials->priv->token_secret = g_strdup (token_secret);
}
