# ubuntuone.platform.windows- windows platform imports
#
# Author: Lucio Torre <lucio.torre@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Windows tools to interact with the os."""

import logging
import os
import sys

from functools import wraps
from contextlib import contextmanager
from pywintypes import error as PyWinError
from win32api import MoveFileEx, GetUserName
from win32file import FindFilesW
from win32com.client import Dispatch

from win32file import (
    MOVEFILE_COPY_ALLOWED,
    MOVEFILE_REPLACE_EXISTING,
    MOVEFILE_WRITE_THROUGH
)
from win32security import (
    ACL,
    ACL_REVISION,
    DACL_SECURITY_INFORMATION,
    GetFileSecurity,
    LookupAccountName,
    LookupAccountSid,
    SetFileSecurity,
)
from ntsecuritycon import (
    FILE_GENERIC_READ,
    FILE_GENERIC_WRITE,
)

# ugly trick to stop pylint for complaining about
# WindowsError on Linux
if sys.platform != 'win32':
    WindowsError = None

platform = 'win32'

LONG_PATH_PREFIX = '\\\\?\\'
EVERYONE_GROUP = 'Everyone'
ADMINISTRATORS_GROUP = 'Administrators'


def _int_to_bin(n):
    """Convert an int to a bin string of 32 bits."""
    return "".join([str((n >> y) & 1) for y in range(32 - 1, -1, -1)])


def _set_as_long_path(path):
    """Allows to access files longer than 260 chars."""
    if path and not path.startswith(LONG_PATH_PREFIX):
        # return the path with the required prefix added
        return LONG_PATH_PREFIX + path
    return path


def _get_group_sid(group_name):
    """Return the SID for a group with the given name."""
    return LookupAccountName('', group_name)[0]


def _set_file_attributes(path, groups):
    """Set file attributes using the wind32api."""
    security_descriptor = GetFileSecurity(path, DACL_SECURITY_INFORMATION)
    dacl = ACL()
    for group_name in groups:
        # set the attributes of the group only if not null
        if groups[group_name]:
            group_sid = _get_group_sid(group_name)
            dacl.AddAccessAllowedAce(ACL_REVISION, groups[group_name],
                group_sid)
    # the dacl has all the info of the dff groups passed in the parameters
    security_descriptor.SetSecurityDescriptorDacl(1, dacl, 0)
    SetFileSecurity(path, DACL_SECURITY_INFORMATION, security_descriptor)


def _has_read_mask(number):
    """Return if the read flag is present."""
    # get the bin representation of the mask
    binary = _int_to_bin(number)
    # there is actual no documentation of this in MSDN but if bt 28 is set,
    # the mask has full access, more info can be found here:
    # http://www.iu.hio.no/cfengine/docs/cfengine-NT/node47.html
    if binary[28] == '1':
        return True
    # there is no documentation in MSDN about this, but if bit 0 and 3 are true
    # we have the read flag, more info can be found here:
    # http://www.iu.hio.no/cfengine/docs/cfengine-NT/node47.html
    return binary[0] == '1' and binary[3] == '1'


def abspath(paths_indexes=[], paths_names=[]):
    """Ensure that the paths are absolute."""
    def decorator(function):
        """Decorate the funtion to make sure the paths are absolute."""
        @wraps(function)
        def abspath_wrapper(*args, **kwargs):
            """Set the paths to be absolute."""
            fixed_args = list(args)
            for current_path in paths_indexes:
                fixed_args[current_path] = \
                    os.path.abspath(args[current_path])
            fixed_args = tuple(fixed_args)
            for current_key in paths_names:
                try:
                    kwargs[current_key] = \
                        os.path.abspath(kwargs[current_key])
                except KeyError:
                    logging.warn('Patameter %s could not be made a long path',
                        current_key)
            return function(*fixed_args, **kwargs)
        return abspath_wrapper
    return decorator


def longpath(paths_indexes=[], paths_names=[]):
    """Ensure that the parameters are long paths."""
    def decorator(function):
        """Decorate function so that the given params are long paths.."""
        @wraps(function)
        def long_path_wrapper(*args, **kwargs):
            """Set the paths to be long paths."""
            fixed_args = list(args)
            for current_path in paths_indexes:
                fixed_args[current_path] = \
                    _set_as_long_path(args[current_path])
            fixed_args = tuple(fixed_args)
            for current_key in paths_names:
                try:
                    kwargs[current_key] = \
                        _set_as_long_path(kwargs[current_key])
                except KeyError:
                    logging.warn('Patameter %s could not be made a long path',
                        current_key)
            return function(*fixed_args, **kwargs)
        return long_path_wrapper
    return decorator


@longpath(paths_indexes=[0])
def set_no_rights(path):
    # set the groups to be empty which will remove all the rights of the file
    groups = {}
    _set_file_attributes(path, groups)


@longpath(paths_indexes=[0])
def set_file_readonly(path):
    """Change path permissions to readonly in a file."""
    # we use the win32 api because chmod just sets the readonly flag and
    # we want to have imore control over the permissions
    groups = {}
    groups[EVERYONE_GROUP] = FILE_GENERIC_READ
    groups[ADMINISTRATORS_GROUP] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    groups[GetUserName()] = FILE_GENERIC_READ
    # the above equals more or less to 0444
    _set_file_attributes(path, groups)


@longpath(paths_indexes=[0])
def set_file_readwrite(path):
    """Change path permissions to readwrite in a file."""
    groups = {}
    groups[EVERYONE_GROUP] = FILE_GENERIC_READ
    groups[ADMINISTRATORS_GROUP] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    groups[GetUserName()] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    # the above equals more or less to 0774
    _set_file_attributes(path, groups)


@longpath(paths_indexes=[0])
def set_dir_readonly(path):
    """Change path permissions to readonly in a dir."""
    groups = {}
    groups[EVERYONE_GROUP] = FILE_GENERIC_READ
    groups[ADMINISTRATORS_GROUP] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    groups[GetUserName()] = FILE_GENERIC_READ
    # the above equals more or less to 0444
    _set_file_attributes(path, groups)


@longpath(paths_indexes=[0])
def set_dir_readwrite(path):
    """Change path permissions to readwrite in a dir."""
    groups = {}
    groups[EVERYONE_GROUP] = FILE_GENERIC_READ
    groups[ADMINISTRATORS_GROUP] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    groups[GetUserName()] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
    # the above equals more or less to 0774
    _set_file_attributes(path, groups)


@contextmanager
@longpath(paths_indexes=[0])
def allow_writes(path):
    """A very simple context manager to allow writting in RO dirs."""
    # get the old dacl of the file so that we can reset it back when done
    security_descriptor = GetFileSecurity(path, DACL_SECURITY_INFORMATION)
    old_dacl = security_descriptor.GetSecurityDescriptorDacl()
    set_dir_readwrite(path)
    yield
    # set the old dacl back
    security_descriptor.SetSecurityDescriptorDacl(1, old_dacl, 0)
    SetFileSecurity(path, DACL_SECURITY_INFORMATION, security_descriptor)


@longpath(paths_indexes=[0])
def remove_file(path):
    """Remove a file."""
    os.remove(path)


@longpath(paths_indexes=[0])
def remove_dir(path):
    """Remove a dir."""
    os.rmdir(path)


@longpath(paths_indexes=[0])
def path_exists(path):
    """Return if the path exists."""
    return os.path.exists(path)


@longpath(paths_indexes=[0])
def make_dir(path, recursive=False):
    """Make a dir, optionally creating all the middle ones."""
    if recursive:
        try:
            os.makedirs(path)
        except WindowsError, e:
            raise OSError(str(e))
    else:
        try:
            os.mkdir(path)
        except WindowsError, e:
            raise OSError(str(e))


@longpath(paths_indexes=[0])
def open_file(path, mode='r'):
    """Open a file."""
    return open(path, mode)


@abspath(paths_indexes=[0, 1])
@longpath(paths_indexes=[0, 1])
def rename(path_from, path_to):
    """Rename a file or directory."""
    # to ensure the same behavious as on linux, use the MoveEx function from
    # win32 which will allow to replace the destionation path if exists and
    # the user has the rights see:
    # http://msdn.microsoft.com/en-us/library/aa365240(v=vs.85).aspx
    try:
        MoveFileEx(
            path_from, path_to,
            MOVEFILE_COPY_ALLOWED |
            MOVEFILE_REPLACE_EXISTING |
            MOVEFILE_WRITE_THROUGH)
    except PyWinError, e:
        raise OSError(str(e))


@abspath(paths_indexes=[0, 1])
def make_link(target, destination):
    """Create a link from the target in the given destination."""
    # append the correct file type
    if not destination.endswith('.lnk'):
        destination += '.lnk'
    # ensure that the dir containing the link exists
    dirname = os.path.dirname(destination)
    if dirname != '' and not os.path.exists(dirname):
        make_dir(dirname, recursive=True)
    shell = Dispatch('WScript.Shell')
    shortcut = shell.CreateShortCut(destination)
    shortcut.Targetpath = target
    shortcut.save()


@longpath(paths_indexes=[0])
def listdir(directory):
    """List a directory."""
    # os.listdir combines / and \ in its search which breacks things on
    # windows, we use the win32 api instead.
    result = []
    for file_info in FindFilesW(os.path.join(directory, '*.*')):
        name = file_info[8]
        # ignore . and .. which windows returns and linux does not
        if name != '.' and name != '..':
            result.append(name)
    return result


@longpath(paths_indexes=[0])
def access(path):
    """Return if the path is at least readable."""
    # for a file to be readable it has to be readable either by the user or
    # by the everyone group
    security_descriptor = GetFileSecurity(path, DACL_SECURITY_INFORMATION)
    dacl = security_descriptor.GetSecurityDescriptorDacl()
    sids = []
    for index in range(0, dacl.GetAceCount()):
        # add the sid of the ace if it can read to test that we remove
        # the r bitmask and test if the bitmask is the same, if not, it means
        # we could read and removed it.
        ace = dacl.GetAce(index)
        if _has_read_mask(ace[1]):
            sids.append(ace[2])
    accounts = [LookupAccountSid('', x)[0] for x in sids]
    return GetUserName() in accounts or EVERYONE_GROUP in accounts


@longpath(paths_indexes=[0])
def stat_path(path, look_for_link=True):
    """Return stat info about a path."""
    # if the path end with .lnk, that means we are dealing with a link
    # and we should return the stat of th target path
    if path.endswith('.lnk'):
        shell = Dispatch('WScript.Shell')
        shortcut = shell.CreateShortCut(path)
        path = shortcut.Targetpath
    if look_for_link and os.path.exists(path + '.lnk'):
        return stat_path(path + '.lnk')
    return os.lstat(path)
