/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "libsyncdaemon.h"

#define TEST_DESCRIPTION "processing queues"
#define TEST_NAME "QUEUE_MANAGER"
#define TEST_STATUS "IDLE"
#define TEST_PATH "/home/user/transfer.file"
#define TEST_ID   "abcdefeghijk"
#define TEST_HASH "1234567890abcdef"

static GMainLoop *main_loop = NULL;
static SyncdaemonDaemon *the_daemon = NULL;

static void
show_hash_table (GHashTable *hash)
{
	GHashTableIter iter;
	gchar *key, *value;

	g_hash_table_iter_init (&iter, hash);
	while (g_hash_table_iter_next (&iter, (gpointer *) &key, (gpointer *) &value))
		g_print ("\tKey %s -> %s\n", key, value);
}

static void
test_daemon (void)
{
	const gchar *str;

	syncdaemon_daemon_connect (the_daemon);

	str = syncdaemon_daemon_get_root_dir (the_daemon);
	g_assert (str != NULL);

	syncdaemon_daemon_has_network (the_daemon);
}

static void
test_config (void)
{
	gboolean orig_enabled, enabled;

	/* Test set/get_throttling_enabled */
	orig_enabled = syncdaemon_daemon_get_bandwidth_throttling (the_daemon);
	syncdaemon_daemon_set_bandwidth_throttling (the_daemon, !orig_enabled);
	enabled = syncdaemon_daemon_get_bandwidth_throttling (the_daemon);
	g_assert (orig_enabled != enabled);

	syncdaemon_daemon_set_bandwidth_throttling (the_daemon, orig_enabled);
	enabled = syncdaemon_daemon_get_bandwidth_throttling (the_daemon);
	g_assert (orig_enabled == enabled);

	/* FIXME: we don't test get/set_files_sync since disabling it means the
	   syncdaemon quits and doesn't start anymore */

	/* Test set/get_udf_autosubscribe */
	orig_enabled = syncdaemon_daemon_get_udf_autosubscribe (the_daemon);
	syncdaemon_daemon_set_udf_autosubscribe (the_daemon, !orig_enabled);
	enabled = syncdaemon_daemon_get_udf_autosubscribe (the_daemon);
	g_assert (orig_enabled != enabled);

	syncdaemon_daemon_set_udf_autosubscribe (the_daemon, orig_enabled);
	enabled = syncdaemon_daemon_get_udf_autosubscribe (the_daemon);
	g_assert (orig_enabled == enabled);
}

static void
test_credentials (void)
{
	SyncdaemonCredentials *credentials;
	GHashTable *hash;

	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "consumer_key", TEST_ID);
	g_hash_table_insert (hash, "consumer_secret", TEST_ID);
	g_hash_table_insert (hash, "token", TEST_ID);
	g_hash_table_insert (hash, "token_secret", TEST_ID);

	credentials = syncdaemon_credentials_new_from_hash_table (hash);
	g_assert (SYNCDAEMON_IS_CREDENTIALS (credentials));

	g_assert (g_strcmp0 (syncdaemon_credentials_get_consumer_key (credentials), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_credentials_get_consumer_secret (credentials), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_credentials_get_token (credentials), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_credentials_get_token_secret (credentials), TEST_ID) == 0);

	g_object_unref (G_OBJECT (credentials));
	g_hash_table_destroy (hash);
}

static void
test_file_info (void)
{
	GHashTable *hash;
	SyncdaemonFileInfo *finfo;

	/* Test creating an empty SyncdaemonFileInfo */
	finfo = syncdaemon_file_info_new ();
	g_assert (SYNCDAEMON_IS_FILE_INFO (finfo));

	syncdaemon_file_info_set_path (finfo, TEST_PATH);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_path (finfo), TEST_PATH) == 0);
	syncdaemon_file_info_set_volume_id (finfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_volume_id (finfo), TEST_ID) == 0);
	syncdaemon_file_info_set_node_id (finfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_node_id (finfo), TEST_ID) == 0);
	syncdaemon_file_info_set_public_url (finfo, TEST_PATH);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_public_url (finfo), TEST_PATH) == 0);
	syncdaemon_file_info_set_is_public (finfo, TRUE);
	g_assert (syncdaemon_file_info_get_is_public (finfo));

	g_object_unref (G_OBJECT (finfo));

	/* Test creating a SyncdaemonFileInfo from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "volume_id", TEST_ID);
	g_hash_table_insert (hash, "node_id", TEST_ID);
	g_hash_table_insert (hash, "public_url", TEST_PATH);
	g_hash_table_insert (hash, "path", TEST_PATH);
	g_hash_table_insert (hash, "is_public", "True");

	finfo = syncdaemon_file_info_new_from_hash_table (hash);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_path (finfo), TEST_PATH) == 0);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_volume_id (finfo), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_node_id (finfo), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_file_info_get_public_url (finfo), TEST_PATH) == 0);
	g_assert (syncdaemon_file_info_get_is_public (finfo));

	g_object_unref (G_OBJECT (finfo));
	g_hash_table_destroy (hash);
}

static void
test_folder_info (void)
{
	GHashTable *hash;
	SyncdaemonFolderInfo *finfo;

	/* Test creating an empty SyncdaemonFolderInfo */
	finfo = syncdaemon_folder_info_new ();
	g_assert (SYNCDAEMON_IS_FOLDER_INFO (finfo));

	syncdaemon_folder_info_set_path (finfo, TEST_PATH);
	g_assert (g_strcmp0 (syncdaemon_folder_info_get_path (finfo), TEST_PATH) == 0);
	syncdaemon_folder_info_set_volume_id (finfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_folder_info_get_volume_id (finfo), TEST_ID) == 0);
	syncdaemon_folder_info_set_subscribed (finfo, TRUE);
	g_assert (syncdaemon_folder_info_get_subscribed (finfo));

	g_object_unref (G_OBJECT (finfo));

	/* Test creating a SyncdaemonFolderInfo from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "path", TEST_PATH);
	g_hash_table_insert (hash, "subscribed", "True");
	g_hash_table_insert (hash, "volume_id", TEST_ID);

	finfo = syncdaemon_folder_info_new_from_hash_table (hash);
	g_assert (g_strcmp0 (syncdaemon_folder_info_get_path (finfo), TEST_PATH) == 0);
	g_assert (g_strcmp0 (syncdaemon_folder_info_get_volume_id (finfo), TEST_ID) == 0);
	g_assert (syncdaemon_folder_info_get_subscribed (finfo));

	g_object_unref (G_OBJECT (finfo));
	g_hash_table_destroy (hash);
}

static void
test_folders (void)
{
	GSList *folders_list, *l;

	/* Get list of UDFs */
	folders_list = syncdaemon_daemon_get_folders (the_daemon);
	g_assert (g_slist_length (folders_list) >= 1);
	
	for (l = folders_list; l != NULL; l = l->next) {
		SyncdaemonFolderInfo *folder_info = (SyncdaemonFolderInfo *) l->data;

		g_assert (SYNCDAEMON_IS_FOLDER_INFO (folder_info));
	}

	g_slist_free (folders_list);
}

static void
test_metadata (void)
{
	GHashTable *hash;
	SyncdaemonMetadata *metadata;

	/* Test creating an empty SyncdaemonMetadata */
	metadata = syncdaemon_metadata_new ();
	g_assert (SYNCDAEMON_IS_METADATA (metadata));

	syncdaemon_metadata_set_path (metadata, TEST_PATH);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_path (metadata), TEST_PATH) == 0);

	syncdaemon_metadata_set_local_hash (metadata, TEST_HASH);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_local_hash (metadata), TEST_HASH) == 0);

	syncdaemon_metadata_set_server_hash (metadata, TEST_HASH);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_server_hash (metadata), TEST_HASH) == 0);

	syncdaemon_metadata_set_share_id (metadata, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_share_id (metadata), TEST_ID) == 0);

	syncdaemon_metadata_set_node_id (metadata, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_node_id (metadata), TEST_ID) == 0);

	g_object_unref (G_OBJECT (metadata));

	/* Test creating a SyncdaemonMetadata from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "path", TEST_PATH);
	g_hash_table_insert (hash, "local_hash", TEST_HASH);
	g_hash_table_insert (hash, "server_hash", TEST_HASH);
	g_hash_table_insert (hash, "share_id", TEST_ID);
	g_hash_table_insert (hash, "node_id", TEST_ID);

	metadata = syncdaemon_metadata_new_from_hash_table (hash);
	g_assert (SYNCDAEMON_IS_METADATA (metadata));
	g_assert (g_strcmp0 (syncdaemon_metadata_get_path (metadata), TEST_PATH) == 0);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_local_hash (metadata), TEST_HASH) == 0);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_server_hash (metadata), TEST_HASH) == 0);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_share_id (metadata), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_metadata_get_node_id (metadata), TEST_ID) == 0);

	g_object_unref (G_OBJECT (metadata));
	g_hash_table_destroy (hash);
}

static void
test_public_files (void)
{
	SyncdaemonPublicfilesInterface *public;

	public = (SyncdaemonPublicfilesInterface *) syncdaemon_daemon_get_publicfiles_interface (the_daemon);
	g_assert (SYNCDAEMON_IS_PUBLICFILES_INTERFACE (public));
}

static void
test_share_info (void)
{
	SyncdaemonShareInfo *sinfo;
	GHashTable *hash;

	/* Test creating an empty SyncdaemonShareInfo */
	sinfo = syncdaemon_share_info_new ();
	g_assert (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	syncdaemon_share_info_set_name (sinfo, TEST_NAME);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_name (sinfo), TEST_NAME) == 0);

	syncdaemon_share_info_set_share_type (sinfo, SYNCDAEMON_SHARE_INFO_TYPE_SHARED);
	g_assert (syncdaemon_share_info_get_share_type (sinfo) == SYNCDAEMON_SHARE_INFO_TYPE_SHARED);

	syncdaemon_share_info_set_path (sinfo, TEST_PATH);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_path (sinfo), TEST_PATH) == 0);

	syncdaemon_share_info_set_username (sinfo, TEST_NAME);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_username (sinfo), TEST_NAME) == 0);

	syncdaemon_share_info_set_user_visible_name (sinfo, TEST_NAME);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_user_visible_name (sinfo), TEST_NAME) == 0);

	syncdaemon_share_info_set_node_id (sinfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_node_id (sinfo), TEST_ID) == 0);

	syncdaemon_share_info_set_volume_id (sinfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_volume_id (sinfo), TEST_ID) == 0);

	syncdaemon_share_info_set_accepted (sinfo, TRUE);
	g_assert (syncdaemon_share_info_get_accepted (sinfo) == TRUE);

	syncdaemon_share_info_set_allow_modifications (sinfo, FALSE);
	g_assert (syncdaemon_share_info_get_allow_modifications (sinfo) == FALSE);

	g_object_unref (G_OBJECT (sinfo));

	/* Test creating a SyncdaemonShareInfo from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "name", TEST_NAME);
	g_hash_table_insert (hash, "type", "Shared");
	g_hash_table_insert (hash, "path", TEST_PATH);
	g_hash_table_insert (hash, "other_username", TEST_NAME);
	g_hash_table_insert (hash, "other_visible_name", TEST_NAME);
	g_hash_table_insert (hash, "node_id", TEST_ID);
	g_hash_table_insert (hash, "volume_id", TEST_ID);
	g_hash_table_insert (hash, "accepted", "True");
	g_hash_table_insert (hash, "access_level", "View");

	sinfo = syncdaemon_share_info_new_from_hash_table (hash);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_name (sinfo), TEST_NAME) == 0);
	g_assert (syncdaemon_share_info_get_share_type (sinfo) == SYNCDAEMON_SHARE_INFO_TYPE_SHARED);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_path (sinfo), TEST_PATH) == 0);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_username (sinfo), TEST_NAME) == 0);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_user_visible_name (sinfo), TEST_NAME) == 0);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_node_id (sinfo), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_share_info_get_volume_id (sinfo), TEST_ID) == 0);
	g_assert (syncdaemon_share_info_get_accepted (sinfo) == TRUE);
	g_assert (syncdaemon_share_info_get_allow_modifications (sinfo) == FALSE);

	g_object_unref (G_OBJECT (sinfo));
}

static void
test_shares (void)
{
	GSList *list;

	/* Get shared folders */
	list = syncdaemon_daemon_get_shared_folders (the_daemon);
	while (list != NULL) {
		SyncdaemonShareInfo *share_info = list->data;

		g_assert (SYNCDAEMON_IS_SHARE_INFO (share_info));
		g_assert (syncdaemon_share_info_get_share_type (share_info) == SYNCDAEMON_SHARE_INFO_TYPE_SHARED);

		list = g_slist_remove (list, share_info);
	}

	/* Get shares */
	list = syncdaemon_daemon_get_shares (the_daemon);
	while (list != NULL) {
		SyncdaemonShareInfo *share_info = list->data;

		g_assert (SYNCDAEMON_IS_SHARE_INFO (share_info));
		g_assert (syncdaemon_share_info_get_share_type (share_info) == SYNCDAEMON_SHARE_INFO_TYPE_SHARE);

		list = g_slist_remove (list, share_info);
	}
}

static void
test_status (void)
{
	GSList *list;

	/* Test getting current downloads */
	list = syncdaemon_daemon_get_current_downloads (the_daemon);
	while (list != NULL) {
		SyncdaemonTransferInfo *tinfo = (SyncdaemonTransferInfo *) list->data;

		g_assert (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));
		list = g_slist_remove (list, tinfo);
	}

	/* Test getting current uploads */
	list = syncdaemon_daemon_get_current_uploads (the_daemon);
	while (list != NULL) {
		SyncdaemonTransferInfo *tinfo = (SyncdaemonTransferInfo *) list->data;

		g_assert (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));
		list = g_slist_remove (list, tinfo);
	}
}

static void
test_status_info (void)
{
	SyncdaemonStatusInfo *sinfo;
	GHashTable *hash;

	/* Test creating a SyncdaemonStatusInfo from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "connection", TEST_DESCRIPTION);
	g_hash_table_insert (hash, "description", TEST_DESCRIPTION);
	g_hash_table_insert (hash, "is_connected", "True");
	g_hash_table_insert (hash, "is_online", "False");
	g_hash_table_insert (hash, "name", TEST_NAME);
	g_hash_table_insert (hash, "queues", TEST_STATUS);

	sinfo = syncdaemon_status_info_new_from_hash_table (hash);
	g_assert (g_strcmp0 (syncdaemon_status_info_get_connection (sinfo), TEST_DESCRIPTION) == 0);
	g_assert (g_strcmp0 (syncdaemon_status_info_get_description (sinfo), TEST_DESCRIPTION) == 0);
	g_assert (syncdaemon_status_info_get_connected (sinfo));
	g_assert (syncdaemon_status_info_get_online (sinfo) == FALSE);
	g_assert (g_strcmp0 (syncdaemon_status_info_get_name (sinfo), TEST_NAME) == 0);
	g_assert (g_strcmp0 (syncdaemon_status_info_get_queues (sinfo), TEST_STATUS) == 0);

	g_object_unref (G_OBJECT (sinfo));
	g_hash_table_destroy (hash);
}

static void
test_transfer_info (void)
{
	GHashTable *hash;
	SyncdaemonTransferInfo *tinfo;

	/* Test creating an empty SyncdaemonTransferInfo */
	tinfo = syncdaemon_transfer_info_new (TEST_PATH);
	g_assert (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_path (tinfo), TEST_PATH) == 0);

	syncdaemon_transfer_info_set_share_id (tinfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_share_id (tinfo), TEST_ID) == 0);

	syncdaemon_transfer_info_set_node_id (tinfo, TEST_ID);
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_node_id (tinfo), TEST_ID) == 0);

	syncdaemon_transfer_info_set_bytes_transferred (tinfo, 100);
	g_assert (syncdaemon_transfer_info_get_bytes_transferred (tinfo) == 100);

	syncdaemon_transfer_info_set_total_size (tinfo, 1000);
	g_assert (syncdaemon_transfer_info_get_total_size (tinfo) == 1000);

	g_object_unref (G_OBJECT (tinfo));

	/* Test creating a SyncdaemonTransferInfo from a hash table */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_hash_table_insert (hash, "path", TEST_PATH);
	g_hash_table_insert (hash, "share_id", TEST_ID);
	g_hash_table_insert (hash, "node_id", TEST_ID);
	g_hash_table_insert (hash, "n_bytes_read", "100");
	g_hash_table_insert (hash, "deflated_size", "1000");

	tinfo = syncdaemon_transfer_info_new_from_hash_table (hash);
	g_assert (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_path (tinfo), TEST_PATH) == 0);
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_share_id (tinfo), TEST_ID) == 0);
	g_assert (g_strcmp0 (syncdaemon_transfer_info_get_node_id (tinfo), TEST_ID) == 0);
	g_assert (syncdaemon_transfer_info_get_bytes_transferred (tinfo) == 100);
	g_assert (syncdaemon_transfer_info_get_total_size (tinfo) == 1000);

	g_object_unref (G_OBJECT (tinfo));
	g_hash_table_destroy (hash);
}

static void
daemon_connected_cb (SyncdaemonDaemon *daemon, gpointer user_data)
{
	g_debug ("Daemon is now connected, and root_dir = %s", syncdaemon_daemon_get_root_dir (the_daemon));
	syncdaemon_daemon_disconnect (the_daemon);
}

static void
daemon_disconnected_cb (SyncdaemonDaemon *daemon, gpointer user_data)
{
	g_debug ("Daemon is now disconnected");
	syncdaemon_daemon_connect (the_daemon);
}

static void
daemon_error_cb (SyncdaemonDaemon *daemon, const gchar *signal, GHashTable *extra_args, gpointer user_data)
{
	g_assert (signal != NULL);
	g_debug ("Got an error: %s:", signal);
	show_hash_table (extra_args);
}

static void
daemon_event_cb (SyncdaemonDaemon *daemon, GHashTable *event_dict, gpointer user_data)
{
	g_debug ("Got an event:");
	show_hash_table (event_dict);
}

static void
daemon_ready_cb (SyncdaemonDaemon daemon, gpointer user_data)
{
	g_test_run ();

	g_main_loop_quit (main_loop);
}

int
main (int argc, gchar *argv[])
{
	g_type_init ();
	g_test_init (&argc, &argv, NULL);

	/* Initialize main object */
	the_daemon = syncdaemon_daemon_new ();
	g_signal_connect (G_OBJECT (the_daemon), "ready", G_CALLBACK (daemon_ready_cb), NULL);
	g_signal_connect (G_OBJECT (the_daemon), "connected", G_CALLBACK (daemon_connected_cb), NULL);
	g_signal_connect (G_OBJECT (the_daemon), "disconnected", G_CALLBACK (daemon_disconnected_cb), NULL);
	g_signal_connect (G_OBJECT (the_daemon), "event", G_CALLBACK (daemon_event_cb), NULL);
	g_signal_connect (G_OBJECT (the_daemon), "error", G_CALLBACK (daemon_error_cb), NULL);

	/* Add test functions */
	g_test_add_func ("/testlibsyncdaemon/TestDaemon", test_daemon);
	g_test_add_func ("/testlibsyncdaemon/TestConfig", test_config);
	g_test_add_func ("/testlibsyncdaemon/TestCredentials", test_credentials);
	g_test_add_func ("/testlibsyncdaemon/TestFileInfo", test_file_info);
	g_test_add_func ("/testlibsyncdaemon/TestFolderInfo", test_folder_info);
	g_test_add_func ("/testlibsyncdaemon/TestFolders", test_folders);
	g_test_add_func ("/testlibsyncdaemon/TestMetadata", test_metadata);
	g_test_add_func ("/testlibsyncdaemon/TestPublicFiles", test_public_files);
	g_test_add_func ("/testlibsyncdaemon/TestShareInfo", test_share_info);
	g_test_add_func ("/testlibsyncdaemon/TestShares", test_shares);
	g_test_add_func ("/testlibsyncdaemon/TestStatus", test_status);
	g_test_add_func ("/testlibsyncdaemon/TestStatusInfo", test_status_info);
	g_test_add_func ("/testlibsyncdaemon/TestTransferInfo", test_transfer_info);

	/* Run main loop and tests suite */
	main_loop = g_main_loop_new (NULL, TRUE);
	g_main_loop_run (main_loop);

	return 0;
}
