# -*- coding: utf-8 -*-
#
# Author: John R. Lenton <john.lenton@canonical.com>
# Author: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the action queue module."""

from __future__ import with_statement

import base64
from functools import wraps
import logging
import operator
import os
import shutil
import unittest
import urllib2
import uuid

import dbus
from mocker import Mocker
from StringIO import StringIO
from oauth import oauth
from twisted.internet import defer, threads, reactor
from twisted.internet import error as twisted_error
from twisted.python.failure import DefaultException, Failure
from twisted.web import server
from twisted.trial.unittest import TestCase as TwistedTestCase

from contrib.testing.testcase import (
    BaseTwistedTestCase, MementoHandler, DummyClass, FakeMain
)

from ubuntuone.storageprotocol import (
    client, errors, protocol_pb2, request
)
from ubuntuone.syncdaemon import states
from ubuntuone.syncdaemon.action_queue import (
    ActionQueue, ActionQueueCommand, ChangePublicAccess, CreateUDF,
    DeleteVolume, Download, ListVolumes, ActionQueueProtocol,
    NoisyRequestQueue, RequestQueue, UploadProgressWrapper, Upload,
    CreateShare, DeleteShare, GetPublicFiles, GetDelta, GetDeltaFromScratch,
    TRANSFER_PROGRESS_THRESHOLD, Unlink, Move, MakeFile, MakeDir,
    DeltaList, ZipQueue, DeferredMap, UniqueRequestQueue
)
from ubuntuone.syncdaemon.event_queue import EventQueue, EVENTS
from ubuntuone.syncdaemon.marker import MDMarker


PATH = u'~/Documents/pdfs/moño/'
NAME = u'UDF-me'
VOLUME = uuid.UUID('12345678-1234-1234-1234-123456789abc')
NODE = uuid.UUID('FEDCBA98-7654-3211-2345-6789ABCDEF12')
USER = u'Dude'
SHARE = uuid.uuid4()

NO_OP = lambda *args, **kwargs: None


def fire_and_check(f, deferred, check):
    """Callback a deferred."""
    @wraps(f)
    def inner(*args, **kwargs):
        """Execute f and fire the deferred."""
        result = f(*args, **kwargs)
        error = check()
        if not error:
            deferred.callback(True)
        else:
            deferred.errback(error)
        return result
    return inner


class FakeCommand(object):
    """Yet another fake action queue command."""

    is_runnable = True
    cleaned = False
    executed = False
    when_executing = 'ok'

    def __init__(self, share_id=None, node_id=None):
        self.share_id = share_id
        self.node_id = node_id

    def run(self):
        """Run that just succeeds."""
        self.executed = True
        if isinstance(self.when_executing, Exception):
            return defer.fail(self.when_executing)
        else:
            return defer.succeed(self.when_executing)

    def cleanup_and_retry(self):
        """Mark as cleaned."""
        self.cleaned = True


class FakedEventQueue(EventQueue):
    """Faked event queue."""

    def __init__(self, fs=None):
        """Initialize a faked event queue."""
        super(FakedEventQueue, self).__init__(fs=fs)
        self.events = []

    def push(self, event_name, *args, **kwargs):
        """Faked event pushing."""
        self.events.append((event_name, args, kwargs))
        super(FakedEventQueue, self).push(event_name, *args, **kwargs)


class FakedVolume(object):
    """Faked volume."""
    volume_id = None
    generation = None
    free_bytes = None


class TestingProtocol(ActionQueue.protocol):
    """Protocol for testing."""

    def connectionMade(self):
        """connectionMade."""
        ActionQueue.protocol.connectionMade(self)

        # assure we're connected
        events = [x[0] for x in self.factory.event_queue.events]
        assert 'SYS_CONNECTION_MADE' in events

        self.factory.event_queue.events = [] # reset events
        if hasattr(self, 'testing_deferred'):
            self.testing_deferred.callback(True)


class TestActionQueue(ActionQueue):
    """AQ class that uses the testing protocol."""
    protocol = TestingProtocol


class BasicTestCase(BaseTwistedTestCase):
    """Basic test case for ActionQueue."""

    timeout = 5

    def setUp(self):
        """Init."""
        BaseTwistedTestCase.setUp(self)

        self.root = self.mktemp('root')
        self.home = self.mktemp('home')
        self.data = self.mktemp('data')
        self.shares = self.mktemp('shares')
        self.partials = self.mktemp('partials')

        # set up Fake Main to use the testing AQ, the port will
        # be decided later
        self._orig_fakemain_aq = (FakeMain._fake_AQ_class,
                                  FakeMain._fake_AQ_params)
        FakeMain._fake_AQ_class = TestActionQueue
        FakeMain._fake_AQ_params = ('127.0.0.1', 0, False)

        self.main = FakeMain(root_dir=self.root, shares_dir=self.shares,
                             data_dir=self.data, partials_dir=self.partials)

        self.action_queue = self.main.action_q
        self.action_queue.connection_timeout = 3
        self.action_queue.event_queue.events = []

        def keep_a_copy(f):
            """Keep a copy of the pushed events."""
            @wraps(f)
            def recording(event_name, *args, **kwargs):
                """Keep a copy of the pushed events."""
                value = (event_name, args, kwargs)
                if event_name != 'SYS_STATE_CHANGED' and \
                   not event_name.startswith('VM_'):
                    self.action_queue.event_queue.events.append(value)
                return f(event_name, *args, **kwargs)
            return recording

        self.main.event_q.push = keep_a_copy(self.main.event_q.push)

        self.handler = MementoHandler()
        self.handler.setLevel(logging.INFO)
        self._logger = logging.getLogger('ubuntuone.SyncDaemon')
        self._logger.addHandler(self.handler)

        dbus.service.BusName.__del__ = lambda _: None

    def tearDown(self):
        """Cleanup."""
        self._logger.removeHandler(self.handler)
        self.main.shutdown()

        # restore Fake Main settings
        cls, params = self._orig_fakemain_aq
        FakeMain._fake_AQ_class = cls
        FakeMain._fake_AQ_params = params

        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        shutil.rmtree(self.partials)

        for record in self.handler.records:
            exc_info = getattr(record, 'exc_info', None)
            if exc_info is not None:
                raise exc_info[0], exc_info[1], exc_info[2]

        BaseTwistedTestCase.tearDown(self)

    def user_connect(self):
        """User requested to connect to server."""
        token = {'token': 'bla', 'token_secret': 'ble',
                 'consumer_key': 'foo', 'consumer_secret': 'bar'}
        self.action_queue.event_queue.push('SYS_USER_CONNECT', token)


class BasicTests(BasicTestCase):
    """Basic tests to check ActionQueue."""

    def test_content_queue_has_only_one_op_per_node(self):
        """Check that the content queue uniquifies operations per node."""
        # totally fake, we don't care: the messages are only validated on run
        self.action_queue.download('foo', 'bar', 0, 0)
        self.action_queue.upload('foo', 'bar', 0, 0, 0, 0, 0)
        self.assertEqual(len(self.action_queue.content_queue.waiting), 1)

    @defer.inlineCallbacks
    def test_content_queue_has_only_one_op_per_node_even_with_markers(self):
        """ Check that the content queue uniquifies operations per node.

        Even when some of the operations were added using markers.
        """
        m = MDMarker('bar')
        yield self.action_queue.download('share', m, 0, 0)
        self.action_queue.uuid_map.set('bar', 'bah')
        self.action_queue.upload('share', 'bah', 0, 0, 0, 0, 0)
        self.assertEqual(len(self.action_queue.content_queue.waiting), 1)

    @defer.inlineCallbacks
    def test_get_root_and_demark(self):
        """Get the received Root and demark its mdid."""
        # get the marker
        d = self.action_queue.uuid_map.get('mdid')

        # we received a root!
        self.main.event_q.push('SYS_ROOT_RECEIVED', 'node_id', 'mdid')

        # it should be demarked with the root node_id
        root_node_id = yield d
        self.assertEqual(root_node_id, 'node_id')


class TestRequestQueue(TwistedTestCase):
    """Tests for the RequestQueue."""

    def setUp(self):
        """Set up."""

        class FakeAQ(object):
            """Fake AQ."""
            event_queue = self.eq = FakedEventQueue()

        class TestQueue(RequestQueue):
            """Inheritance to test."""

            _condition_checked = False

            def check_conditions(self):
                """Not really in RQ."""
                self._condition_checked = True

        self.rq = TestQueue(name='META_QUEUE', action_queue=FakeAQ())

        # add a Memento handler to the logger
        self.log_handler = MementoHandler()
        self.log_handler.setLevel(logging.DEBUG)
        logging.getLogger('ubuntuone.SyncDaemon').addHandler(self.log_handler)

    def tearDown(self):
        """Tear down."""
        self.eq.shutdown()

    def test_len_nothing(self):
        """Len with nothing queued."""
        self.assertEqual(len(self.rq), 0)

    def test_len_heap(self):
        """Len with nothing waiting and one processing."""
        self.rq._head = "foo"
        self.assertEqual(len(self.rq), 0)

    def test_len_waiting(self):
        """Len with something in the queue."""
        self.rq.waiting.append(1)
        self.assertEqual(len(self.rq), 1)

    def test_len_mixed(self):
        """Len with something processing and something queued."""
        self.rq._head = "foo"
        self.rq.waiting.append(1)
        self.rq.waiting.append(1)
        self.assertEqual(len(self.rq), 2)

    def test_get_full_queue(self):
        """Check that it returns head and waiting."""
        self.rq._head = 1
        self.rq.waiting.append(2)
        self.rq.waiting.append(3)
        self.assertEqual(self.rq.full_queue(), [1, 2, 3])

    def test_done_empties_head(self):
        """After done, the head should be empty."""
        self.rq._head = 1
        self.rq.done()
        self.assertTrue(self.rq._head is None)

    def test_done_waiting_send_signal_waiting(self):
        """It should send the WAITING signal if something waiting."""
        self.rq.waiting.append(1)
        self.rq.done()
        self.assertEqual(self.eq.events, [('SYS_META_QUEUE_WAITING', (), {})])

    def test_done_waiting_send_signal_no(self):
        """It should send the DONE signal if nothing waiting."""
        self.rq.done()
        self.assertEqual(self.eq.events, [('SYS_META_QUEUE_DONE', (), {})])

    def test_queue_adds_to_waiting(self):
        """Command queues is appended to waiting."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.append(cmd1)

        # queue and test
        self.rq.queue(cmd2)
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])

    def test_queue_waiting_if_first_and_no_head(self):
        """It should send the WAITING signal ."""
        # set up
        cmd = FakeCommand()

        # queue and test
        self.rq.queue(cmd)
        self.assertEqual(self.rq._head, None)
        self.assertEqual(list(self.rq.waiting), [cmd])
        self.assertEqual(self.eq.events, [('SYS_META_QUEUE_WAITING', (), {})])

    def test_queue_nowaiting_if_not_first(self):
        """It should not send the WAITING signal if no first cmd."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.append(cmd1)

        # queue and test
        self.rq.queue(cmd2)
        self.assertEqual(self.rq._head, None)
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])
        self.assertEqual(self.eq.events, [])

    def test_queue_nowaiting_if_head(self):
        """It should not send the WAITING signal if processing."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq._head = cmd1

        # queue and test
        self.rq.queue(cmd2)
        self.assertEqual(self.rq._head, cmd1)
        self.assertEqual(list(self.rq.waiting), [cmd2])
        self.assertEqual(self.eq.events, [])

    def test_queue_check_condition_for_runnable_cmd(self):
        """When a runnable command is queued, it should check conditions."""
        # set up
        cmd = FakeCommand()

        # queue and test
        self.rq.queue(cmd)
        self.assertTrue(self.rq._condition_checked)

    def test_queue_not_check_condition_for_not_runnable_cmd(self):
        """When a not runnable command is queued, it should not check."""
        # set up
        cmd = FakeCommand()
        cmd.is_runnable = False

        # queue and test
        self.rq.queue(cmd)
        self.assertFalse(self.rq._condition_checked)

    def test_queuetop_adds_to_waiting(self):
        """Command queues is inserted in the beggining of waiting."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.append(cmd1)

        # queue and test
        self.rq.queue(cmd2, on_top=True)
        self.assertEqual(list(self.rq.waiting), [cmd2, cmd1])

    def test_queuetop_waiting_if_first_and_no_head(self):
        """It should send the WAITING signal ."""
        # set up
        cmd = FakeCommand()

        # queue and test
        self.rq.queue(cmd, on_top=True)
        self.assertEqual(self.rq._head, None)
        self.assertEqual(list(self.rq.waiting), [cmd])
        self.assertEqual(self.eq.events, [('SYS_META_QUEUE_WAITING', (), {})])

    def test_queuetop_nowaiting_if_not_first(self):
        """It should not send the WAITING signal if no first cmd."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.append(cmd1)

        # queue and test
        self.rq.queue(cmd2, on_top=True)
        self.assertEqual(self.rq._head, None)
        self.assertEqual(list(self.rq.waiting), [cmd2, cmd1])
        self.assertEqual(self.eq.events, [])

    def test_queuetop_nowaiting_if_head(self):
        """It should not send the WAITING signal if processing."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq._head = cmd1

        # queue and test
        self.rq.queue(cmd2, on_top=True)
        self.assertEqual(self.rq._head, cmd1)
        self.assertEqual(list(self.rq.waiting), [cmd2])
        self.assertEqual(self.eq.events, [])

    def test_queuetop_check_condition_for_runnable_cmd(self):
        """When a runnable command is queued, it should check conditions."""
        # set up
        cmd = FakeCommand()

        # queue and test
        self.rq.queue(cmd, on_top=True)
        self.assertTrue(self.rq._condition_checked)

    def test_queuetop_not_check_condition_for_not_runnable_cmd(self):
        """When a not runnable command is queued, it should not check."""
        # set up
        cmd = FakeCommand()
        cmd.is_runnable = False

        # queue and test
        self.rq.queue(cmd, on_top=True)
        self.assertFalse(self.rq._condition_checked)

    def test_cleanup_and_retry_head(self):
        """Cleanup and retry with something in head."""
        # set up
        cmd = FakeCommand()
        assert cmd.cleaned == False
        self.rq._head = cmd

        # cleanup and test
        self.rq.cleanup_and_retry()
        self.assertTrue(cmd.cleaned)
        self.assertTrue(self.rq._head is None)

    @defer.inlineCallbacks
    def test_run_nothing_without_waiting(self):
        """Run should do nothing but logging if no waiting."""
        assert len(self.rq.waiting) == 0
        yield self.rq.run()
        self.assertTrue(self.log_handler.check_warning("queue empty"))

    @defer.inlineCallbacks
    def test_run_something_if_notworking(self):
        """Run should put something waiting to work if wasn't processing."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.extend([cmd1, cmd2])

        # run and test
        yield self.rq.run()
        self.assertEqual(list(self.rq.waiting), [cmd2])

    @defer.inlineCallbacks
    def test_run_nothing_if_working(self):
        """Run should do nothing if already processing something in head."""
        # set up
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        self.rq.waiting.extend([cmd1, cmd2])
        self.rq._head = FakeCommand()

        # run and test
        yield self.rq.run()
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])

    @defer.inlineCallbacks
    def test_run_get_next_command(self):
        """Run gets next commands and keeps it in head."""
        # set up
        self.rq.waiting.append(FakeCommand())
        cmd = FakeCommand()
        self.rq._get_next_runnable_command = lambda: cmd
        self.rq.done = lambda: None

        # run and test
        yield self.rq.run()
        self.assertEqual(self.rq._head, cmd)

    @defer.inlineCallbacks
    def test_run_execute_next_command(self):
        """Run executes next command."""
        # set up
        self.rq.waiting.append(FakeCommand())
        cmd = FakeCommand()
        self.rq._get_next_runnable_command = lambda: cmd

        # run and test
        yield self.rq.run()
        self.assertTrue(cmd.executed)

    @defer.inlineCallbacks
    def test_run_done_after_ok(self):
        """Run programs a call to done() after command finished ok."""
        # set up
        self.rq.waiting.append(FakeCommand())
        called = []
        self.rq.done = lambda: called.append(True)
        cmd = FakeCommand()
        cmd.when_executing = "ok"
        self.rq._get_next_runnable_command = lambda: cmd

        # run and test
        yield self.rq.run()
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_run_done_after_bad(self):
        """Run programs a call to done() after command finished with error."""
        # set up
        self.rq.waiting.append(FakeCommand())
        called = []
        self.rq.done = lambda: called.append(True)
        cmd = FakeCommand()
        cmd.when_executing = Exception("something bad")
        self.rq._get_next_runnable_command = lambda: cmd

        # run and test
        try:
            yield self.rq.run()
        except Exception, e:
            # *we* are making it fail
            self.assertEqual(str(e), "something bad")
        self.assertTrue(called)

    def test_remove_empty(self):
        """Don't remove if waiting is empty."""
        assert not self.rq.waiting, "test badly set up"
        cmd = FakeCommand()
        self.rq.remove(cmd)
        self.assertFalse(self.rq.waiting)

    def test_remove_other(self):
        """Don't remove if waiting has other command."""
        cmd1 = FakeCommand()
        self.rq.waiting.append(cmd1)
        cmd2 = FakeCommand()
        self.rq.remove(cmd2)
        self.assertEqual(list(self.rq.waiting), [cmd1])

    def test_remove_command(self):
        """Remove for the command."""
        cmd = FakeCommand()
        self.rq.waiting.append(cmd)
        self.rq.remove(cmd)
        self.assertFalse(self.rq.waiting)

    def test_remove_mixed(self):
        """Remove ok in a mixed situation."""
        cmd1 = FakeCommand()
        cmd2 = FakeCommand()
        cmd3 = FakeCommand()
        self.rq.waiting.append(cmd1)
        self.rq.waiting.append(cmd2)
        self.rq.waiting.append(cmd3)
        self.rq.remove(cmd2)
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd3])


class TestNoisyRQ(unittest.TestCase):
    """Tests for NoisyRequestQueue."""

    def test_noisy_rq_blurts_about_head(self):
        """Test NRQ calls its callback when head is set."""
        setter = lambda h, w: setattr(self, 'result', (h, tuple(w)))
        rq = NoisyRequestQueue('name', None, setter)
        rq._head = 'blah'
        self.assertEqual(self.result, ('blah', ()))

    def test_noisy_rq_blurts_about_waiting(self):
        """Test NRQ calls its callback when the waiting queue is altered."""

        class BlackHole(object):
            """The universal tool."""
            __call__ = __getattr__ = lambda *_, **__: BlackHole()

        def cb(head, waiting):
            """NRQ testing callback"""
            evts.append((head, tuple(waiting)))

        evts = []
        cmd = FakeCommand()
        cmd2 = FakeCommand()
        rq = NoisyRequestQueue('name', BlackHole(), cb)
        rq.queue(cmd2)
        rq.queue(cmd, on_top=True)
        rq.run()
        self.assertEqual(evts, [(None, ()),           # __init__
                                (None, (cmd2,)),      # queue
                                (None, (cmd, cmd2)),  # queue_top
                                (cmd, (cmd2,)),       # run
                                (None, (cmd2,)),      # done
                                ])


class TestDeferredMap(TwistedTestCase):
    """Test the deferred map."""

    def setUp(self):
        """Set up."""
        self.dm = DeferredMap()

    def test_one_get_returns_stored_deferred(self):
        """Get will return the stored deferred."""
        d = self.dm.get('foo')
        self.assertEqual(self.dm.waiting, {'foo': [d]})

    def test_two_gets_returns_second_deferred_other_key(self):
        """A second get for other key will return other deferred."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('bar')
        self.assertEqual(self.dm.waiting, {'foo': [d1], 'bar': [d2]})

    def test_two_gets_returns_second_deferred_same_key(self):
        """A second get for the same key will return other deferred."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('foo')
        self.assertEqual(self.dm.waiting, {'foo': [d1, d2]})

    def test_mixed_gets(self):
        """Several gets with different keys."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('bar')
        d3 = self.dm.get('foo')
        d4 = self.dm.get('baz')
        self.assertEqual(self.dm.waiting,
                         {'foo': [d1, d3], 'bar': [d2], 'baz': [d4]})

    def test_set_to_nothing(self):
        """It's ok to set a key that is not being waited."""
        self.dm.set('not there', 'value')

    @defer.inlineCallbacks
    def test_set_fires_deferred_single(self):
        """The set fires the unique waiting deferred with the value."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('bar')
        d3 = self.dm.get('foo')
        self.assertEqual(self.dm.waiting, {'foo': [d1, d3], 'bar': [d2]})

        self.dm.set('bar', 'value')
        res = yield d2
        self.assertEqual(res, 'value')
        self.assertEqual(self.dm.waiting, {'foo': [d1, d3]})

    @defer.inlineCallbacks
    def test_set_fires_deferred_multiple(self):
        """The set fires the multiple waiting deferreds with the value."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('bar')
        d3 = self.dm.get('foo')
        self.assertEqual(self.dm.waiting, {'foo': [d1, d3], 'bar': [d2]})

        self.dm.set('foo', 'value')
        res1 = yield d1
        res2 = yield d3
        self.assertEqual(res1, 'value')
        self.assertEqual(res2, 'value')
        self.assertEqual(self.dm.waiting, {'bar': [d2]})

    def test_err_to_nothing(self):
        """It's ok to err a key that is not being waited."""
        self.dm.err('not there', 'failure')

    @defer.inlineCallbacks
    def test_err_fires_deferred_single(self):
        """The set fires the unique waiting deferred with the failure."""
        d1 = self.dm.get('foo')
        d2 = self.dm.get('bar')
        d3 = self.dm.get('foo')
        self.assertEqual(self.dm.waiting, {'foo': [d1, d3], 'bar': [d2]})

        exc = Exception('problem!')
        self.dm.err('bar', Failure(exc))
        try:
            yield d2
        except Exception, e:
            self.assertEqual(e, exc)
        else:
            self.fail("It didn't fired the deferred with a failure!")
        self.assertEqual(self.dm.waiting, {'foo': [d1, d3]})



class TestUniqueRequestQueue(TwistedTestCase):
    """Tests for the UniqueRequestQueue."""

    def setUp(self):
        """Set up."""

        class FakeAQ(object):
            """Fake AQ."""
            event_queue = self.eq = FakedEventQueue()

        self.urq = UniqueRequestQueue(name='META_QUEUE', action_queue=FakeAQ())

        # add a Memento handler to the logger
        self.handler = MementoHandler()
        self.handler.setLevel(logging.DEBUG)
        self._logger = logging.getLogger('ubuntuone.SyncDaemon')
        self._logger.addHandler(self.handler)

    def tearDown(self):
        """Tear down."""
        self._logger.removeHandler(self.handler)
        self.eq.shutdown()

    def test_queue_one_ok(self):
        """First command is queued."""
        cmd = FakeCommand('a', 'b')
        self.urq.queue(cmd)
        self.assertEqual(list(self.urq.waiting), [cmd])

    def test_queue_second_ok_if_different(self):
        """Second command is queued if different than first."""
        cmd1 = FakeCommand('a', 'b')
        cmd2 = FakeCommand('a', 'c')
        self.urq.queue(cmd1)
        self.urq.queue(cmd2)
        self.assertEqual(list(self.urq.waiting), [cmd1, cmd2])

    def test_queue_second_equal_first_removes(self):
        """Second command removes the first one if equal."""
        cmd1 = FakeCommand('a', 'b')
        cmd2 = FakeCommand('a', 'b')
        self.urq.queue(cmd1)
        self.urq.queue(cmd2)
        self.assertEqual(list(self.urq.waiting), [cmd2])
        self.assertTrue(self.handler.check_debug("Request removed"))

    def test_queue_complex(self):
        """A more complex scenario."""
        cmd1 = FakeCommand('a', 'b')
        cmd2 = FakeCommand('a', 'c')
        cmd3 = FakeCommand('a', 'd')
        cmd4 = FakeCommand('a', 'e')
        cmd5 = FakeCommand('a', 'd') # same than cmd3
        for cmd in cmd1, cmd2, cmd3, cmd4, cmd5:
            self.urq.queue(cmd)
        self.assertEqual(list(self.urq.waiting), [cmd1, cmd2, cmd4, cmd5])


class TestZipQueue(TwistedTestCase):
    """Test the zipping queue."""

    def setUp(self):
        """Set up."""
        self.zq = ZipQueue()

    @defer.inlineCallbacks
    def test_zip_calls_compress_in_thread(self):
        """Test that self._compress is called in another thread."""
        called = []
        def fake_compress(deferred, upload):
            """Fake the _compress method."""
            self.assertEqual(upload, 'foo')
            called.append(True)
            deferred.callback(True)

        self.zq._compress = fake_compress
        yield self.zq.zip('foo')
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_zip_acquire_lock(self):
        """Test that it acquires the lock."""
        called = []
        self.zq._compress = lambda deferred, upload: deferred.callback(True)

        def fake_acquire():
            """Fake the acquire method."""
            self.zq.tokens = 1
            called.append(True)
            return defer.succeed(True)

        self.zq.acquire = fake_acquire
        yield self.zq.zip('foo')
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_zip_release_lock_ok(self):
        """Test that it releases the lock when all ok."""
        called = []
        self.zq._compress = lambda deferred, upload: deferred.callback(True)
        self.zq.release = lambda: called.append(True)

        yield self.zq.zip('foo')
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_zip_release_lock_error(self):
        """Test that it releases the lock even on error."""
        called = []
        exc = Exception('bad')
        self.zq._compress = lambda deferred, upload: deferred.errback(exc)
        self.zq.release = lambda: called.append(True)

        try:
            yield self.zq.zip('foo')
        except Exception, e:
            # need to silent the exception we're generating in the test
            self.assertEqual(e, exc)
        self.assertTrue(called)


class FactoryBaseTestCase(BasicTestCase):
    """Helper for by-pass Twisted."""

    def _start_sample_webserver(self):
        """Start a web server serving content at its root"""
        # start listening on `decide yourself` port, and fix AQ with it
        website = server.Site(None)
        webport = reactor.listenTCP(0, website)
        self.action_queue.port = webport.getHost().port

        transport_class = webport.transport
        def save_an_instance(skt, protocol, addr, sself, s, sreactor):
            self.server_transport = transport_class(skt, protocol, addr, sself,
                                                    s, sreactor)
            return self.server_transport
        webport.transport = save_an_instance

        self.addCleanup(webport.stopListening)
        return webport

    def _connect_factory(self):
        """Connect the instance factory."""
        self.server = self._start_sample_webserver()
        orig = self.action_queue.buildProtocol

        d = defer.Deferred()
        def faked_buildProtocol(*args, **kwargs):
            """Override buildProtocol to hook a deferred."""
            protocol = orig(*args, **kwargs)
            protocol.testing_deferred = d
            return protocol

        self.action_queue.buildProtocol = faked_buildProtocol
        self.action_queue.connect()
        return d

    def _disconnect_factory(self):
        """Disconnect the instance factory."""
        if self.action_queue.client is not None:
            orig = self.action_queue.client.connectionLost

            d = defer.Deferred()
            def faked_connectionLost(reason):
                """Receive connection lost and fire tearDown."""
                orig(reason)
                d.callback(True)

            self.action_queue.client.connectionLost = faked_connectionLost
        else:
            d = defer.succeed(True)

        if self.action_queue.connect_in_progress:
            self.action_queue.disconnect()

        return d


class ConnectionTestCase(FactoryBaseTestCase):
    """Test TCP/SSL connection mechanism for ActionQueue."""

    def assert_connection_state_reset(self):
        """Test connection state is properly reset."""
        self.assertTrue(self.action_queue.client is None)
        self.assertTrue(self.action_queue.connector is None)
        self.assertEqual(False, self.action_queue.connect_in_progress)

    def test_init(self):
        """Test connection init state."""
        self.assert_connection_state_reset()

    @defer.inlineCallbacks
    def test_connect_if_already_connected(self):
        """Test that double connections are avoided."""
        yield self._connect_factory()

        assert self.action_queue.connector is not None
        assert self.action_queue.connect_in_progress == True
        # double connect, it returns None instead of a Deferred
        result = self.action_queue.connect()
        self.assertTrue(result is None, 'not connecting again')

        yield self._disconnect_factory()

    @defer.inlineCallbacks
    def test_disconnect_if_connected(self):
        """self.action_queue.connector.disconnect was called."""
        yield self._connect_factory()

        self.action_queue.event_queue.events = [] # cleanup events
        assert self.action_queue.connector.state == 'connected'
        self.action_queue.disconnect()

        self.assert_connection_state_reset()
        self.assertEqual([], self.action_queue.event_queue.events)

        yield self._disconnect_factory()

    @defer.inlineCallbacks
    def test_clientConnectionFailed(self):
        """Test clientConnectionFailed.

        The connection will not be completed since the server will be down. So,
        self.action_queue.connector will never leave the 'connecting' state.
        When interrupting the connection attempt, twisted automatically calls
        self.action_queue.clientConnectionFailed.

        """
        self.action_queue.event_queue.events = []
        orig = self.action_queue.clientConnectionFailed

        d = defer.Deferred()
        def faked_clientConnectionFailed(connector, reason):
            """Receive connection failed and check."""
            orig(connector, reason)
            self.assert_connection_state_reset()
            self.assertEqual([('SYS_CONNECTION_FAILED', (), {})],
                             self.action_queue.event_queue.events)
            self.action_queue.clientConnectionFailed = orig
            d.callback(True)

        self.action_queue.clientConnectionFailed = faked_clientConnectionFailed
        # factory will never finish the connection, server was never started
        self.action_queue.connect()
        # stopConnecting() will be called since the connection is in progress
        assert self.action_queue.connector.state == 'connecting'
        self.action_queue.connector.disconnect()

        yield d

    @defer.inlineCallbacks
    def test_clientConnectionLost(self):
        """Test clientConnectionLost

        The connection will be completed successfully.
        So, self.action_queue.connector will be in the 'connected' state.
        When disconnecting the connector, twisted automatically calls
        self.action_queue.clientConnectionLost.

        """
        yield self._connect_factory()

        self.action_queue.event_queue.events = []
        orig = self.action_queue.clientConnectionLost

        d = defer.Deferred()
        def faked_clientConnectionLost(connector, reason):
            """Receive connection lost and check."""
            orig(connector, reason)
            self.assert_connection_state_reset()
            self.assertEqual([('SYS_CONNECTION_LOST', (), {})],
                             self.action_queue.event_queue.events)
            self.action_queue.clientConnectionLost = orig
            d.callback(True)

        self.action_queue.clientConnectionLost = faked_clientConnectionLost
        # loseConnection() will be called since the connection was completed
        assert self.action_queue.connector.state == 'connected'
        self.action_queue.connector.disconnect()
        yield d

        yield self._disconnect_factory()

    @defer.inlineCallbacks
    def test_server_disconnect(self):
        """Test factory's connection when the server goes down."""

        yield self._connect_factory()

        self.action_queue.event_queue.events = []
        orig = self.action_queue.clientConnectionLost

        d = defer.Deferred()
        def faked_connectionLost(*args, **kwargs):
            """Receive connection lost and check."""
            orig(*args, **kwargs)
            self.assert_connection_state_reset()
            self.assertEqual([('SYS_CONNECTION_LOST', (), {})],
                             self.action_queue.event_queue.events)
            self.action_queue.clientConnectionLost = orig
            d.callback(True)

        self.action_queue.clientConnectionLost = faked_connectionLost
        # simulate a server failure!
        yield self.server_transport.loseConnection()
        yield d
        yield self._disconnect_factory()

    def test_buildProtocol(self):
        """Test buildProtocol."""
        protocol = self.action_queue.buildProtocol(addr=None)
        self.assertTrue(protocol is self.action_queue.client)
        self.assertTrue(self.action_queue is self.action_queue.client.factory)

        # callbacks are connected
        # pylint: disable-msg=W0212
        aq = self.action_queue
        self.assertEqual(aq.client._share_change_callback,
                         aq._share_change_callback)
        self.assertEqual(aq.client._share_answer_callback,
                         aq._share_answer_callback)
        self.assertEqual(aq.client._free_space_callback,
                         aq._free_space_callback)
        self.assertEqual(aq.client._account_info_callback,
                         aq._account_info_callback)
        self.assertEqual(aq.client._volume_created_callback,
                         aq._volume_created_callback)
        self.assertEqual(aq.client._volume_deleted_callback,
                         aq._volume_deleted_callback)
        self.assertEqual(aq.client._volume_new_generation_callback,
                         aq._volume_new_generation_callback)

    @defer.inlineCallbacks
    def test_connector_gets_assigned_on_connect(self):
        """Test factory's connector gets assigned on connect."""
        yield self._connect_factory()

        self.assertTrue(self.action_queue.connector is not None)

        yield self._disconnect_factory()

    @defer.inlineCallbacks
    def test_cleanup_doesnt_disconnect(self):
        """cleanup() doesn't disconnect the factory."""
        yield self._connect_factory()

        self.action_queue.cleanup()
        self.assertTrue(self.action_queue.connector is not None)
        self.assertEqual(self.action_queue.connector.state, 'connected')

        yield self._disconnect_factory()


class NetworkmanagerTestCase(FactoryBaseTestCase):
    """Base test case generating a connected factory."""

    timeout = 15

    def fake_answer(self, answer):
        """Push an event faking a server answer."""
        return (lambda *_: self.action_queue.event_queue.push(answer))

    def setUp(self):
        """Init."""
        super(NetworkmanagerTestCase, self).setUp()
        self.action_queue.event_queue.push('SYS_NET_CONNECTED')
        self.main.start()

    @defer.inlineCallbacks
    def test_wrong_disconnect(self):
        """Test factory's connection when SYS_NET_DISCONNECTED."""

        d1 = self.main.wait_for('SYS_CONNECTION_MADE')
        d2 = self.main.wait_for('SYS_CONNECTION_LOST')

        self.server = self._start_sample_webserver()
        self.user_connect()
        yield d1

        self.action_queue.event_queue.push('SYS_NET_DISCONNECTED')
        yield d2

    @defer.inlineCallbacks
    def test_disconnect_twice(self):
        """Test connection when SYS_NET_DISCONNECTED is received twice."""

        d1 = self.main.wait_for('SYS_CONNECTION_MADE')
        d2 = self.main.wait_for('SYS_CONNECTION_LOST')

        self.server = self._start_sample_webserver()
        self.user_connect()
        yield d1

        self.action_queue.event_queue.push('SYS_NET_DISCONNECTED')
        yield d2

        self.action_queue.event_queue.events = []
        self.action_queue.event_queue.push('SYS_NET_DISCONNECTED')
        self.assertEqual([('SYS_NET_DISCONNECTED', (), {})],
                          self.action_queue.event_queue.events,
                       'No new events after a missplaced SYS_NET_DISCONNECTED')


    @defer.inlineCallbacks
    def test_net_connected_if_already_connected(self):
        """Test connection when SYS_NET_CONNECTED is received twice."""

        d1 = self.main.wait_for('SYS_CONNECTION_MADE')

        self.server = self._start_sample_webserver()
        self.user_connect()
        yield d1

        self.action_queue.event_queue.events = []
        self.action_queue.event_queue.push('SYS_NET_CONNECTED')
        self.assertEqual([('SYS_NET_CONNECTED', (), {})],
                          self.action_queue.event_queue.events,
                          'No new events after a missplaced SYS_NET_CONNECTED')

    @defer.inlineCallbacks
    def test_messy_mix(self):
        """Test connection when a messy mix of events is received."""
        orig_waiting = states.MAX_WAITING
        states.MAX_WAITING = 1

        self.action_queue.event_queue.events = []
        self.server = self._start_sample_webserver()

        conn_made = self.main.wait_for('SYS_CONNECTION_MADE')
        self.user_connect()
        yield conn_made

        events = ['SYS_NET_CONNECTED', 'SYS_NET_DISCONNECTED',
                  'SYS_NET_CONNECTED', 'SYS_NET_CONNECTED',
                  'SYS_NET_DISCONNECTED', 'SYS_NET_DISCONNECTED',
                  'SYS_NET_CONNECTED']

        for i in events:
            self.action_queue.event_queue.push(i)

        yield self.main.wait_for_nirvana()

        expected = ['SYS_NET_CONNECTED', # from the DBus fake NetworkManager
                    'SYS_NET_CONNECTED', 'SYS_NET_DISCONNECTED',
                    'SYS_CONNECTION_LOST', 'SYS_CONNECTION_RETRY',
                    'SYS_NET_CONNECTED', 'SYS_NET_CONNECTED',
                    'SYS_CONNECTION_MADE', 'SYS_NET_DISCONNECTED',
                    'SYS_NET_DISCONNECTED']

        avoid = ('SYS_STATE_CHANGED', 'SYS_LOCAL_RESCAN_DONE',
                 'SYS_PROTOCOL_VERSION_OK', 'SYS_SET_CAPABILITIES_OK',
                 'SYS_AUTH_OK', 'SYS_SERVER_RESCAN_DONE')
        actual = [event for (event, args, kwargs) in
                  self.action_queue.event_queue.events
                  if event not in avoid]
        self.assertEqual(sorted(expected), sorted(actual))

        states.MAX_WAITING = orig_waiting


class ConnectedBaseTestCase(FactoryBaseTestCase):
    """Base test case generating a connected factory."""

    @defer.inlineCallbacks
    def setUp(self):
        """Init."""
        FactoryBaseTestCase.setUp(self)
        yield self._connect_factory()
        assert self.action_queue.connector.state == 'connected'

    @defer.inlineCallbacks
    def tearDown(self):
        """Clean up."""
        yield self._disconnect_factory()
        FactoryBaseTestCase.tearDown(self)


class VolumeManagementTestCase(ConnectedBaseTestCase):
    """Test Volume managemenr for ActionQueue."""

    def setUp(self):
        """Init."""
        res = super(VolumeManagementTestCase, self).setUp()

        self.patch(self.action_queue.main.vm,
                   'handle_SV_VOLUME_DELETED', NO_OP)

        return res

    def test_volume_created_push_event(self):
        """Volume created callback push proper event."""
        volume = FakedVolume()
        self.action_queue._volume_created_callback(volume)
        self.assertEqual([('SV_VOLUME_CREATED', (), {'volume': volume})],
                          self.action_queue.event_queue.events)

    def test_volume_deleted_push_event(self):
        """Volume deleted callback push proper event."""
        volume_id = VOLUME
        self.action_queue._volume_deleted_callback(volume_id)
        self.assertEqual([('SV_VOLUME_DELETED', (), {'volume_id': volume_id})],
                          self.action_queue.event_queue.events)

    def test_volume_new_generation_push_event_root(self):
        """Volume New Generation callback push proper event with root."""
        volume = request.ROOT
        self.action_queue._volume_new_generation_callback(volume, 77)
        event = ('SV_VOLUME_NEW_GENERATION', (),
                 {'volume_id': volume, 'generation': 77})
        self.assertTrue(event in self.action_queue.event_queue.events)

    def test_volume_new_generation_push_event_uuid(self):
        """Volume New Generation callback push proper event with uuid."""
        volume = uuid.uuid4()
        self.action_queue._volume_new_generation_callback(volume, 77)
        event = ('SV_VOLUME_NEW_GENERATION', (),
                 {'volume_id': volume, 'generation': 77})
        self.assertTrue(event in self.action_queue.event_queue.events)

    def test_valid_events(self):
        """Volume events are valid in EventQueue."""
        new_events = ('SV_VOLUME_CREATED', 'SV_VOLUME_DELETED',
                      'AQ_CREATE_UDF_OK', 'AQ_CREATE_UDF_ERROR',
                      'AQ_LIST_VOLUMES', 'AQ_LIST_VOLUMES_ERROR',
                      'AQ_DELETE_VOLUME_OK', 'AQ_DELETE_VOLUME_ERROR',
                      'SV_VOLUME_NEW_GENERATION')
        for event in new_events:
            self.assertTrue(event in EVENTS)

        self.assertEqual(('volume',), EVENTS['SV_VOLUME_CREATED'])
        self.assertEqual(('volume_id',), EVENTS['SV_VOLUME_DELETED'])
        self.assertEqual(('volume_id', 'node_id', 'marker'),
                          EVENTS['AQ_CREATE_UDF_OK'])
        self.assertEqual(('error', 'marker'), EVENTS['AQ_CREATE_UDF_ERROR'])
        self.assertEqual(('volumes',), EVENTS['AQ_LIST_VOLUMES'])
        self.assertEqual(('error',), EVENTS['AQ_LIST_VOLUMES_ERROR'])
        self.assertEqual(('volume_id',), EVENTS['AQ_DELETE_VOLUME_OK'])
        self.assertEqual(('volume_id', 'error',),
                         EVENTS['AQ_DELETE_VOLUME_ERROR'])
        self.assertEqual(('volume_id', 'generation',),
                         EVENTS['SV_VOLUME_NEW_GENERATION'])

    def test_create_udf(self):
        """Test volume creation."""
        path = PATH
        name = NAME
        res = self.action_queue.create_udf(path, name, marker=None)
        self.assertTrue(res is None) # this is what start returns

    def test_list_volumes(self):
        """Test volume listing."""
        res = self.action_queue.list_volumes()
        self.assertTrue(res is None) # this is what start returns

    def test_delete_volume(self):
        """Test volume deletion."""
        volume_id = VOLUME
        res = self.action_queue.delete_volume(volume_id)
        self.assertTrue(res is None) # this is what start returns


class ActionQueueCommandTestCase(ConnectedBaseTestCase):
    """Test for the generic functionality of ActionQueueCommand."""

    def setUp(self):
        """Set up."""
        res = super(ActionQueueCommandTestCase, self).setUp()
        self.handler.setLevel(logging.DEBUG)

        class MyCommand(ActionQueueCommand):
            logged_attrs = ('a', 'b', 'c', 'd')
            a = 3
            b = 'foo'
            c = u'año'

            def _run(self):
                return defer.succeed(True)

        request_queue = RequestQueue(name='META_QUEUE',
                                     action_queue=self.action_queue)
        self.cmd = MyCommand(request_queue)
        self.cmd.log = self.cmd.make_logger()

        return res

    def test_runnable(self):
        """All commands are runnable by default."""
        self.assertTrue(self.cmd.is_runnable)

    def test_dump_to_dict(self):
        """Test to dict dumping."""
        d = self.cmd.to_dict()
        self.assertEqual(d, dict(a=3, b='foo', c=u'año', d=None))

    @defer.inlineCallbacks
    def test_demark_not_marker(self):
        """Test demark with not a marker."""
        self.cmd.possible_markers = 'foo',
        self.cmd.foo = 'not a marker'
        yield self.cmd.demark()
        self.assertEqual(self.cmd.foo, 'not a marker')

    @defer.inlineCallbacks
    def test_demark_with_marker_future(self):
        """Test demark with a marker not ready.

        Here, on purpose, set up everything and trigger later.
        """
        marker = MDMarker('foo')
        self.cmd.possible_markers = 'foo',
        self.cmd.foo = marker
        d = self.cmd.demark()

        self.action_queue.uuid_map.set(marker, 'node_id')
        yield d
        self.assertEqual(self.cmd.foo, 'node_id')
        self.assertTrue(self.handler.check_debug(
                        "waiting for the real value of marker:foo"))

    @defer.inlineCallbacks
    def test_demark_with_marker_ready(self):
        """Test demark with a marker that had data."""
        marker = MDMarker('foo')
        self.cmd.possible_markers = 'foo',
        self.cmd.foo = marker
        d = self.cmd.demark()
        self.action_queue.uuid_map.set(marker, 'node_id')
        yield d
        self.assertEqual(self.cmd.foo, 'node_id')
        self.assertTrue(self.handler.check_debug(
                        "waiting for the real value of marker:foo"))

    @defer.inlineCallbacks
    def test_demark_mixed_markers(self):
        """Test demark with both a marker and not."""
        # call demark with both
        marker = MDMarker('foo')
        self.cmd.possible_markers = 'foo', 'bar'
        self.cmd.foo = 'notamarker'
        self.cmd.bar = marker
        d = self.cmd.demark()
        self.action_queue.uuid_map.set(marker, 'node_id')
        yield d

        # check
        self.assertEqual(self.cmd.foo, 'notamarker')
        self.assertEqual(self.cmd.bar, 'node_id')
        self.assertTrue(self.handler.check_debug(
                        "waiting for the real value of marker:foo"))
        self.assertFalse(self.handler.check_debug(
                         "waiting for the real value of 'notamarker'"))

    @defer.inlineCallbacks
    def test_demark_marker_future_got_ok(self):
        """Test demark getting a marker triggered ok later."""
        # don't have the info now
        marker = MDMarker('foo')
        self.cmd.possible_markers = 'foo',
        self.cmd.foo = marker
        d = self.cmd.demark()
        self.assertFalse(self.handler.check_debug("for marker:foo"))

        # set and check
        self.action_queue.uuid_map.set(marker, 'node_id')
        yield d
        self.assertEqual(self.cmd.foo, 'node_id')
        self.assertTrue(self.handler.check_debug(
                        "for marker:foo got value 'node_id'"))

    @defer.inlineCallbacks
    def test_demark_marker_future_got_failure(self):
        """Test demark getting a marker triggered with failure later."""
        # don't have the info now
        marker = MDMarker('foo')
        self.cmd.possible_markers = 'foo',
        self.cmd.foo = marker
        d = self.cmd.demark()
        self.assertFalse(self.handler.check_error("failed marker:foo"))

        # set the marker and check
        self.action_queue.uuid_map.err(marker, Failure(Exception('bad')))
        yield d
        self.assertTrue(self.handler.check_error("failed marker:foo"))
        try:
            yield self.cmd.markers_resolved_deferred
        except Exception, e:
            self.assertEqual(str(e), 'bad')
        else:
            self.fail("An exception should have been raised!")

    @defer.inlineCallbacks
    def test_demark_two_markers_ok(self):
        """Test demark with two markers that finish ok."""
        # call demark with both
        marker1 = MDMarker('foo')
        marker2 = MDMarker('bar')
        self.cmd.possible_markers = 'foo', 'bar'
        self.cmd.foo = marker1
        self.cmd.bar = marker2
        d = self.cmd.demark()
        self.action_queue.uuid_map.set(marker1, 'data1')
        self.action_queue.uuid_map.set(marker2, 'data2')
        yield d

        # check
        self.assertEqual(self.cmd.foo, 'data1')
        self.assertEqual(self.cmd.bar, 'data2')
        yield self.cmd.markers_resolved_deferred

    @defer.inlineCallbacks
    def test_demark_two_markers_one_fail(self):
        """Test demark with two markers that one ends in failure."""
        # call demark with both
        marker1 = MDMarker('foo')
        marker2 = MDMarker('bar')
        self.cmd.possible_markers = 'foo', 'bar'
        self.cmd.foo = marker1
        self.cmd.bar = marker2
        d = self.cmd.demark()
        self.action_queue.uuid_map.set(marker1, 'data ok')
        self.action_queue.uuid_map.err(marker2, Failure(Exception('data bad')))
        yield d

        # check
        try:
            yield self.cmd.markers_resolved_deferred
        except Exception, e:
            self.assertEqual(str(e), 'data bad')
        else:
            self.fail("An exception should have been raised!")

    @defer.inlineCallbacks
    def test_run_sets_running(self):
        """Set the flag."""
        assert not self.cmd.running
        d = defer.Deferred()
        def check():
            """Checks."""
            self.assertTrue(self.cmd.running)
            d.callback(True)

        self.cmd._start = check
        self.cmd.run()
        # wait the deferred to be sure it was checked
        yield d

    @defer.inlineCallbacks
    def test_run_call_start_if_not_started(self):
        """Call ._start if it's not started."""
        called = []
        self.cmd._start = lambda: called.append(True) or defer.succeed(True)
        self.cmd.start_done = False
        self.cmd.markers_resolved_deferred.callback(True)
        yield self.cmd.run()
        self.assertTrue(called)
        self.assertTrue(self.cmd.start_done)

    def test_run_dont_call_start_if_started(self):
        """Do not call ._start if it's started."""
        called = []
        self.cmd._start = lambda: called.append(True) or defer.Deferred()
        self.cmd.start_done = True
        self.cmd.run()
        self.assertFalse(called)

    def test_run_waits_markers_dereferencing(self):
        """Don't call _ready_to_run until have the markers."""
        called = []
        self.cmd._ready_to_run = lambda: called.append(True)
        self.cmd.run()
        self.assertFalse(called)
        self.cmd.markers_resolved_deferred.callback(True)
        self.assertTrue(called)

    def test_run_not_enderrback_if_ok(self):
        """If _start is ok, do not call end_errback."""
        called = []
        self.cmd.end_errback = lambda _: called.append(True)
        # default _start will always succeed
        self.cmd.run()
        self.assertFalse(called)

    def test_run_enderrback_if_problem(self):
        """If _start fails, call end_errback."""
        called = []
        self.cmd.end_errback = lambda _: called.append(True)
        self.cmd._start = lambda: defer.fail(Exception('foo'))
        self.cmd.run()
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_run_done_running_if_ok(self):
        """If _start is ok, done running."""
        # default _start will always succeed
        self.cmd.markers_resolved_deferred.callback(True)
        yield self.cmd.run()
        self.assertFalse(self.cmd.running)

    def test_run_done_running_if_problem_in_start(self):
        """If _start fails, done running."""
        called = []
        self.cmd.end_errback = lambda _: called.append(True)
        self.cmd._start = lambda: defer.fail(Exception('foo'))
        self.cmd.run()
        self.assertFalse(self.cmd.running)
        self.assertTrue(called)

    def test_run_done_running_if_problem_in_markers(self):
        """If deferring markers fails, done running."""
        called = []
        self.cmd.end_errback = lambda _: called.append(True)
        self.cmd.markers_resolved_deferred.errback(Failure(Exception('foo')))
        self.cmd.run()
        self.assertFalse(self.cmd.running)
        self.assertTrue(called)

    def test_queue_calls_prequeued(self):
        """Do the pre-queue when queued."""
        called = []
        self.cmd.pre_queue_setup = lambda: called.append(True)
        self.cmd.queue()
        self.assertTrue(called)

    def test_prequeued_calls_demark(self):
        """Pre-queue implies dereferencing markers."""
        called = []
        self.cmd.demark = lambda: called.append(True)
        self.cmd.pre_queue_setup()
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_start_default(self):
        """Default _start just returns a triggered deferred and sets done."""
        yield self.cmd._start()

    def test_possible_markers_default(self):
        """Default value for possible markers."""
        self.assertEqual(self.cmd.possible_markers, ())

    @defer.inlineCallbacks
    def test_marker_storing(self):
        """Pre-queueing deferres the marker storing."""
        self.cmd.pre_queue_setup()
        yield self.cmd.markers_resolved_deferred


class CreateUDFTestCase(ConnectedBaseTestCase):
    """Test for CreateUDF ActionQueueCommand."""

    def setUp(self):
        """Init."""
        res = super(CreateUDFTestCase, self).setUp()

        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.marker = VOLUME
        self.command = CreateUDF(request_queue, PATH, NAME, marker=self.marker)

        vm = self.action_queue.main.vm
        self.patch(vm, 'handle_AQ_CREATE_UDF_OK', NO_OP)
        self.patch(vm, 'handle_AQ_CREATE_UDF_ERROR', NO_OP)

        return res

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        self.assertTrue(isinstance(self.command, ActionQueueCommand))

    def test_init(self):
        """Test creation."""
        self.assertEqual(PATH, self.command.path)
        self.assertEqual(NAME, self.command.name)
        self.assertEqual(self.marker, self.command.marker)

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        res = self.command._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's create_udf is called."""
        original = self.command.action_queue.client.create_udf
        self.called = False

        def check(path, name):
            """Take control over client's feature."""
            self.called = True
            self.assertEqual(PATH, path)
            self.assertEqual(NAME, name)

        self.command.action_queue.client.create_udf = check

        self.command._run()

        self.assertTrue(self.called, 'command was called')

        self.command.action_queue.client.create_udf = original

    def test_handle_success_push_event(self):
        """Test AQ_CREATE_UDF_OK is pushed on success."""
        request = client.CreateUDF(self.action_queue.client, PATH, NAME)
        request.volume_id = VOLUME
        request.node_id = NODE
        res = self.command.handle_success(success=request)
        events = [('AQ_CREATE_UDF_OK', (), {'volume_id': VOLUME,
                                            'node_id': NODE,
                                            'marker': self.marker})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertEqual(request, res)

    def test_handle_failure_push_event(self):
        """Test AQ_CREATE_UDF_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        res = self.command.handle_failure(failure=failure)
        events = [('AQ_CREATE_UDF_ERROR', (),
                    {'error': msg, 'marker': self.marker})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)


class ActionQueueCommandErrorsTestCase(ConnectedBaseTestCase):
    """Test the error handling in ActionQueueCommand."""

    def setUp(self):
        res = super(ActionQueueCommandErrorsTestCase, self).setUp()

        self.deferred = defer.Deferred()

        class MyLogger(object):
            """Fake logger that just stores error and warning calls."""
            def __init__(self):
                self.logged = None

            def error(self, *a):
                """Mark that this method was called."""
                self.logged = "error"

            def warn(self, *a):
                """Mark that this method was called."""
                self.logged = "warn"

            def debug(self, *a):
                """Nothing."""

        class MyCommand(ActionQueueCommand):
            """Inherit ACQ to provide a retry signaller and a custom log."""
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself, request_queue):
                super(MyCommand, innerself).__init__(request_queue)
                innerself.log = MyLogger()

            def retry(innerself):
                """Signal the retry."""
                self.deferred.callback(True)

        self.rq = RequestQueue(name='foo', action_queue=self.action_queue)
        self.command = MyCommand(self.rq)
        return res

    def test_suppressed_yes_knownerrors(self):
        """Check that the log is in warning for the known errors."""
        def send_failure_and_check(errnum, exception_class):
            """Send the failure."""
            # prepare what to send
            protocol_msg = protocol_pb2.Message()
            protocol_msg.type = protocol_pb2.Message.ERROR
            protocol_msg.error.type = errnum
            err = exception_class("request", protocol_msg)

            # set up and test
            self.command.log.logged = None
            self.command.end_errback(failure=Failure(err))
            self.assertEqual(self.command.log.logged, "warn",
                             "Bad log in exception %s" % (exception_class,))

        known_errors = [x for x in errors._error_mapping.items()
                        if x[1] != errors.InternalError]
        for errnum, exception_class in known_errors:
            send_failure_and_check(errnum, exception_class)

    def test_suppressed_no_internalerror(self):
        """Check that the log is in error for InternalError."""
        # prepare what to send
        protocol_msg = protocol_pb2.Message()
        protocol_msg.type = protocol_pb2.Message.ERROR
        protocol_msg.error.type = protocol_pb2.Error.INTERNAL_ERROR
        err = errors.InternalError("request", protocol_msg)

        # set up and test
        self.command.end_errback(failure=Failure(err))
        self.assertEqual(self.command.log.logged, "error")

    def test_suppressed_yes_cancelled(self):
        """Check that the log is in warning for Cancelled."""
        err = errors.RequestCancelledError("CANCELLED")
        self.command.end_errback(failure=Failure(err))
        self.assertEqual(self.command.log.logged, "warn")

    def test_suppressed_yes_and_retry_when_connectiondone(self):
        """Check that the log is in warning and retries for ConnectionDone."""
        self.command.running = True
        err = twisted_error.ConnectionDone()
        self.command.end_errback(failure=Failure(err))
        self.assertEqual(self.command.log.logged, "warn")
        return self.deferred

    def test_retry_connectionlost(self):
        """Check that it retries when ConnectionLost."""
        self.command.running = True
        err = twisted_error.ConnectionLost()
        self.command.end_errback(failure=Failure(err))
        return self.deferred

    def test_retry_tryagain(self):
        """Check that it retries when TryAgain."""
        # prepare what to send
        self.command.running = True
        protocol_msg = protocol_pb2.Message()
        protocol_msg.type = protocol_pb2.Message.ERROR
        protocol_msg.error.type = protocol_pb2.Error.TRY_AGAIN
        err = errors.TryAgainError("request", protocol_msg)

        # set up and test
        self.command.end_errback(failure=Failure(err))
        return self.deferred


class ListVolumesTestCase(ConnectedBaseTestCase):
    """Test for ListVolumes ActionQueueCommand."""

    def setUp(self):
        """Init."""
        res = super(ListVolumesTestCase, self).setUp()

        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.command = ListVolumes(request_queue)

        return res

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        self.assertTrue(isinstance(self.command, ActionQueueCommand))

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        res = self.command._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's list_volumes is called."""
        original = self.command.action_queue.client.list_volumes
        self.called = False

        def check():
            """Take control over client's feature."""
            self.called = True

        self.command.action_queue.client.list_volumes = check

        self.command._run()

        self.assertTrue(self.called, 'command was called')

        self.command.action_queue.client.list_volumes = original

    def test_handle_success_push_event(self):
        """Test AQ_LIST_VOLUMES is pushed on success."""
        request = client.ListVolumes(self.action_queue.client)
        request.volumes = [FakedVolume(), FakedVolume()]
        res = self.command.handle_success(success=request)
        event = ('AQ_LIST_VOLUMES', (), {'volumes': request.volumes})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertEqual(request, res)

    def test_handle_failure_push_event(self):
        """Test AQ_LIST_VOLUMES_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        res = self.command.handle_failure(failure=failure)
        event = ('AQ_LIST_VOLUMES_ERROR', (), {'error': msg})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)


class DeleteVolumeTestCase(ConnectedBaseTestCase):
    """Test for DeleteVolume ActionQueueCommand."""

    def setUp(self):
        """Init."""
        res = super(DeleteVolumeTestCase, self).setUp()

        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.command = DeleteVolume(request_queue, VOLUME)

        vm = self.action_queue.main.vm
        self.patch(vm, 'handle_AQ_DELETE_VOLUME_OK', NO_OP)
        self.patch(vm, 'handle_AQ_DELETE_VOLUME_ERROR', NO_OP)

        return res

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        self.assertTrue(isinstance(self.command, ActionQueueCommand))

    def test_init(self):
        """Test creation."""
        self.assertEqual(VOLUME, self.command.volume_id)

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        res = self.command._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's delete_volume is called."""
        original = self.command.action_queue.client.delete_volume
        self.called = False

        def check(volume_id):
            """Take control over client's feature."""
            self.called = True
            self.assertEqual(VOLUME, volume_id)

        self.command.action_queue.client.delete_volume = check

        self.command._run()

        self.assertTrue(self.called, 'command was called')

        self.command.action_queue.client.delete_volume = original

    def test_handle_success_push_event(self):
        """Test AQ_DELETE_VOLUME_OK is pushed on success."""
        request = client.DeleteVolume(self.action_queue.client,
                                      volume_id=VOLUME)
        res = self.command.handle_success(success=request)
        events = [('AQ_DELETE_VOLUME_OK', (), {'volume_id': VOLUME})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertEqual(request, res)

    def test_handle_failure_push_event(self):
        """Test AQ_DELETE_VOLUME_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        res = self.command.handle_failure(failure=failure)
        events = [('AQ_DELETE_VOLUME_ERROR', (),
                  {'volume_id': VOLUME, 'error': msg})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)


class FilterEventsTestCase(BasicTestCase):
    """Tests for event filtering when a volume is not of our interest."""

    def setUp(self):
        """Init."""
        BasicTestCase.setUp(self)
        self.vm = self.main.vm
        self.old_home = os.environ.get('HOME', None)
        os.environ['HOME'] = self.home

    def tearDown(self):
        """Clean up."""
        if self.old_home is None:
            os.environ.pop('HOME')
        else:
            os.environ['HOME'] = self.old_home

        BasicTestCase.tearDown(self)


class ChangePublicAccessTests(ConnectedBaseTestCase):
    """Tests for the ChangePublicAccess ActionQueueCommand."""

    def setUp(self):
        super(ChangePublicAccessTests, self).setUp()
        self.user_connect()
        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.command = ChangePublicAccess(request_queue, VOLUME, NODE, True)

    def test_change_public_access(self):
        """Test the change_public_access method.."""
        res = self.action_queue.change_public_access(VOLUME, NODE, True)
        self.assertTrue(res is None) # this is what start returns

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        self.assertTrue(isinstance(self.command, ActionQueueCommand))

    def test_init(self):
        """Test creation."""
        self.assertEqual(VOLUME, self.command.share_id)
        self.assertEqual(NODE, self.command.node_id)
        self.assertEqual(True, self.command.is_public)

    def test_run_defers_work_to_thread(self):
        """Test that work is deferred to a thread."""
        original = threads.deferToThread
        self.called = False

        def check(function):
            self.called = True
            self.assertEqual(
                self.command._change_public_access_http, function)
            return defer.Deferred()

        threads.deferToThread = check
        try:
            res = self.command._run()
        finally:
            threads.deferToThread = original

        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')
        self.assertTrue(self.called, "deferToThread was called")

    def test_change_public_access_http(self):
        """Test the blocking portion of the command."""
        self.called = False
        def check(request):
            self.called = True
            url = 'https://one.ubuntu.com/files/api/set_public/%s:%s' % (
                base64.urlsafe_b64encode(VOLUME.bytes).strip("="),
                base64.urlsafe_b64encode(NODE.bytes).strip("="))
            self.assertEqual(url, request.get_full_url())
            self.assertEqual("is_public=True", request.get_data())
            return StringIO(
                '{"is_public": true, "public_url": "http://example.com"}')

        from ubuntuone.syncdaemon import action_queue
        action_queue.urlopen = check
        try:
            res = self.command._change_public_access_http()
        finally:
            action_queue.urlopen = urllib2.urlopen

        self.assertEqual(
            {'is_public': True, 'public_url': 'http://example.com'}, res)

    def test_handle_success_push_event(self):
        """Test AQ_CHANGE_PUBLIC_ACCESS_OK is pushed on success."""
        response = {'is_public': True, 'public_url': 'http://example.com'}
        res = self.command.handle_success(success=response)
        event = ('AQ_CHANGE_PUBLIC_ACCESS_OK', (),
                 {'share_id': VOLUME, 'node_id': NODE, 'is_public': True,
                  'public_url': 'http://example.com'})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertEqual(response, res)

    def test_handle_failure_push_event(self):
        """Test AQ_CHANGE_PUBLIC_ACCESS_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(urllib2.HTTPError(
                "http://example.com", 500, "Error", [], StringIO(msg)))
        res = self.command.handle_failure(failure=failure)
        event = ('AQ_CHANGE_PUBLIC_ACCESS_ERROR', (),
                 {'share_id': VOLUME, 'node_id': NODE, 'error': msg})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        res = [getattr(self.command, x) for x in self.command.possible_markers]
        self.assertEqual(res, [NODE])


class GetPublicFilesTestCase(ConnectedBaseTestCase):
    """Tests for GetPublicFiles ActionQueueCommand."""

    def setUp(self):
        super(GetPublicFilesTestCase, self).setUp()
        self.user_connect()
        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.command = GetPublicFiles(request_queue)

    def test_init(self):
        """Test __init__ method."""
        default_url = 'https://one.ubuntu.com/files/api/public_files'
        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        command = GetPublicFiles(request_queue)
        self.assertEqual(command._url, default_url)
        custom_url = 'http://example.com:1234/files/api/public_files'
        command_2 = GetPublicFiles(request_queue,
                                   base_url='http://example.com:1234')
        self.assertEqual(command_2._url, custom_url)

    def test_change_public_access(self):
        """Test the get_public_files method.."""
        res = self.action_queue.get_public_files()
        self.assertTrue(res is None) # this is what start returns

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        self.assertTrue(isinstance(self.command, ActionQueueCommand))

    def test_run_defers_work_to_thread(self):
        """Test that work is deferred to a thread."""
        original = threads.deferToThread
        self.called = False

        def check(function):
            self.called = True
            self.assertEqual(
                self.command._get_public_files_http, function)
            return defer.Deferred()

        threads.deferToThread = check
        try:
            res = self.command._run()
        finally:
            threads.deferToThread = original

        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')
        self.assertTrue(self.called, "deferToThread was called")

    def test_get_public_files_http(self):
        """Test the blocking portion of the command."""
        self.called = False
        node_id = uuid.uuid4()
        nodekey = '%s' % (base64.urlsafe_b64encode(node_id.bytes).strip("="))
        node_id_2 = uuid.uuid4()
        nodekey_2 = '%s' % (base64.urlsafe_b64encode(
                                                node_id_2.bytes).strip("="))
        volume_id = uuid.uuid4()
        def check(request):
            self.called = True
            url = 'https://one.ubuntu.com/files/api/public_files'
            self.assertEqual(url, request.get_full_url())
            return StringIO(
                '[{"nodekey": "%s", "volume_id": null,"public_url": '
                '"http://example.com"}, '
                '{"nodekey": "%s", "volume_id": "%s", "public_url": '
                '"http://example.com"}]' % (nodekey, nodekey_2, volume_id))

        from ubuntuone.syncdaemon import action_queue
        action_queue.urlopen = check
        try:
            res = self.command._get_public_files_http()
        finally:
            action_queue.urlopen = urllib2.urlopen
        self.assertEqual([{'node_id': str(node_id), 'volume_id': '',
                          'public_url': 'http://example.com'},
                          {'node_id': str(node_id_2),
                           'volume_id': str(volume_id),
                           'public_url': 'http://example.com'}], res)

    def test_handle_success_push_event(self):
        """Test AQ_PUBLIC_FILES_LIST_OK is pushed on success."""
        response = [{'node_id': uuid.uuid4(), 'volume_id':None,
                    'public_url': 'http://example.com'}]
        res = self.command.handle_success(success=response)
        event = ('AQ_PUBLIC_FILES_LIST_OK', (), {'public_files': response,})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertEqual(response, res)

    def test_handle_failure_push_event(self):
        """Test AQ_PUBLIC_FILES_LIST_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(urllib2.HTTPError(
                "http://example.com", 500, "Error", [], StringIO(msg)))
        res = self.command.handle_failure(failure=failure)
        event = ('AQ_PUBLIC_FILES_LIST_ERROR', (), {'error': msg})
        self.assertIn(event, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)


class DownloadUnconnectedTestCase(FactoryBaseTestCase):
    """Test for Download ActionQueueCommand, no connection"""

    def setUp(self):
        """Init."""
        super(DownloadUnconnectedTestCase, self).setUp()

        self.rq = request_queue = RequestQueue(name='FOO',
                                               action_queue=self.action_queue)
        self.command = Download(request_queue, share_id='a_share_id',
                               node_id='a_node_id', server_hash='server_hash',
                               fileobj_factory=lambda: None)
        self.command.pre_queue_setup() # create the logger

    def test_progress_information_setup(self):
        """Test the setting up of the progress information in ._run()."""
        calls = []
        d = defer.Deferred()
        class FakedRequest():
            """Fake Request."""
            pass
        class FakedClient(object):
            """Fake Client."""
            def get_content_request(innerself, *args, **kwds):
                """Fake a get content request with its deferred."""
                calls.append(kwds.get('offset'))
                req = FakedRequest()
                req.deferred = d
                return req

        self.command.action_queue.connect_in_progress = False
        self.command.action_queue.client = FakedClient()
        self.command._run()

        self.assertEqual(self.command.n_bytes_read_last, 0)
        self.assertEqual(calls, [0])

        calls = []
        dloading = self.command.action_queue.downloading['a_share_id',
                                                         'a_node_id']
        dloading['n_bytes_read'] = 20
        self.command._run()

        self.assertEqual(self.command.n_bytes_read_last, 20)
        self.assertEqual(calls, [20])


class DownloadTestCase(ConnectedBaseTestCase):
    """Test for Download ActionQueueCommand."""

    def setUp(self):
        """Init."""
        res = super(DownloadTestCase, self).setUp()

        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)

        class MyDownload(Download):
            """Just to allow monkeypatching."""

        self.command = MyDownload(request_queue, share_id='a_share_id',
                                  node_id='a_node_id', server_hash='server_hash',
                                  fileobj_factory=lambda: None)
        self.command.pre_queue_setup() # create the logger

        return res

    def test_AQ_DOWNLOAD_FILE_PROGRESS_is_valid_event(self):
        """AQ_DOWNLOAD_FILE_PROGRESS is a valid event."""
        event = 'AQ_DOWNLOAD_FILE_PROGRESS'
        self.assertTrue(event in EVENTS)
        self.assertEqual(('share_id', 'node_id', 'n_bytes_read',
                          'deflated_size'), EVENTS[event])

    def test_progress_hook(self):
        # would first get the node attribute including this
        self.command.action_queue.downloading['a_share_id', 'a_node_id'] = {}
        self.command.nacb(deflated_size = 2*TRANSFER_PROGRESS_THRESHOLD)
        self.command.progress_start(0)

        self.command.progress_hook(5)
        self.assertEqual([], self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_read_last, 5)

        self.command.progress_hook(TRANSFER_PROGRESS_THRESHOLD)
        self.assertEqual([], self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_read_last,
                         TRANSFER_PROGRESS_THRESHOLD)

        self.command.progress_start(5)
        self.command.progress_hook(TRANSFER_PROGRESS_THRESHOLD+5)
        kwargs = {'share_id': 'a_share_id', 'node_id': 'a_node_id',
                  'deflated_size': 2*TRANSFER_PROGRESS_THRESHOLD,
                  'n_bytes_read': 5+TRANSFER_PROGRESS_THRESHOLD}
        events = [('AQ_DOWNLOAD_FILE_PROGRESS', (), kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_read_last,
                         5+TRANSFER_PROGRESS_THRESHOLD)

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        res = [getattr(self.command, x) for x in self.command.possible_markers]
        self.assertEqual(res, ['a_node_id'])

    def test_cancel_set_cancelled(self):
        """Set the command to cancelled."""
        assert not self.command.cancelled, "test badly set up"
        self.command.cancel()
        self.assertTrue(self.command.cancelled)

    def test_cancel_download_req_is_none(self):
        """It's ok to have download_req in None when cancelling."""
        assert self.command.download_req is None, "test badly set up"
        self.command.cancel()

    def test_cancel_download_req_is_something(self):
        """download_req is also cancelled."""
        # set up the mocker
        mocker = Mocker()
        obj = mocker.mock()
        obj.cancel()

        # test
        with mocker:
            self.command.download_req = obj
            self.command.cancel()

    def test_cancel_remove(self):
        """Remove the command from the queue."""
        # set up the mocker
        mocker = Mocker()
        obj = mocker.mock()
        obj.remove(self.command)

        # test
        with mocker:
            self.command._queue = obj
            self.command.cancel()

    def test_cancel_clean_up(self):
        """Clean up."""
        called = []
        self.command.cleanup = lambda: called.append(True)
        self.command.cancel()
        self.assertTrue(called)


class UploadUnconnectedTestCase(FactoryBaseTestCase):
    """Test for Upload ActionQueueCommand, no connection"""

    def setUp(self):
        """Init."""
        super(UploadUnconnectedTestCase, self).setUp()

        self.rq = request_queue = RequestQueue(name='FOO',
                                               action_queue=self.action_queue)
        self.command = Upload(request_queue, share_id='a_share_id',
                              node_id='a_node_id', previous_hash='prev_hash',
                              hash='yadda', crc32=0, size=0,
                              fileobj_factory=lambda: None,
                              tempfile_factory=lambda: None)
        self.command.pre_queue_setup() # create the logger

    def test_upload_progress_wrapper_setup(self):
        """Test the setting up of the progress wrapper in ._run()."""
        calls = []
        d = defer.Deferred()
        class FakedRequest():
            """Fake Request."""
            pass
        class FakedClient(object):
            """Fake Client."""
            def put_content_request(innerself, *args):
                """Fake a put content request with its deferred."""
                calls.append(args)
                req = FakedRequest()
                req.deferred = d
                return req

        self.command.action_queue.connect_in_progress = False
        self.command.action_queue.client = FakedClient()
        self.command._run()

        self.assertEqual(len(calls), 1)
        upload_wrapper = calls[0][-1]
        uploading = self.action_queue.uploading['a_share_id', 'a_node_id']
        self.assertTrue(upload_wrapper.data_dict is uploading)
        self.assertEqual(upload_wrapper.progress_hook,
                         self.command.progress_hook)
        self.assertEqual(self.command.n_bytes_written_last, 0)

        self.command.n_bytes_written_last = 20
        self.command._run()
        self.assertEqual(self.command.n_bytes_written_last, 0)


class UploadProgressWrapperTestCase(BaseTwistedTestCase):
    """Test for the UploadProgressWrapper helper class."""

    def test_read(self):
        """Test the read method."""
        progress = []

        def progress_hook(n_bytes_written):
            progress.append(n_bytes_written)

        info = {'n_bytes_written': 0}
        f = StringIO("x"*10+"y"*5)
        upw = UploadProgressWrapper(f, info, progress_hook)

        res = upw.read(10)
        self.assertEqual(res, "x"*10)
        self.assertEqual(progress, [0])

        res = upw.read(5)
        self.assertEqual(res, "y"*5)
        self.assertEqual(progress, [0, 10])


class UploadTestCase(ConnectedBaseTestCase):
    """Test for Upload ActionQueueCommand."""

    def setUp(self):
        """Init."""
        super(UploadTestCase, self).setUp()

        self.rq = request_queue = RequestQueue(name='FOO',
                                               action_queue=self.action_queue)

        class MyUpload(Upload):
            """Just to allow monkeypatching."""

        self.command = MyUpload(request_queue, share_id=str(uuid.uuid4()),
                                node_id='a_node_id', previous_hash='prev_hash',
                                hash='yadda', crc32=0, size=0,
                                fileobj_factory=lambda: None,
                                tempfile_factory=lambda: None)
        self.command.pre_queue_setup() # create the logger

    def test_upload_in_progress(self):
        """Test Upload retries on UploadInProgress."""
        d = defer.Deferred()
        self.command.retry = lambda: d.callback(True)
        self.command.running = True

        # send the failure
        protocol_msg = protocol_pb2.Message()
        protocol_msg.type = protocol_pb2.Message.ERROR
        protocol_msg.error.type = protocol_pb2.Error.UPLOAD_IN_PROGRESS
        err = errors.UploadInProgressError("request", protocol_msg)
        self.command.end_errback(failure=Failure(err))
        return d

    def test_handle_success_push_event(self):
        """Test AQ_UPLOAD_FINISHED is pushed on success."""
        # create a request and fill it with succesful information
        request = client.PutContent(self.action_queue.client, VOLUME, 'node',
                                    'prvhash', 'newhash', 'crc32', 'size',
                                    'deflated', 'fd')
        request.new_generation = 13

        # trigger success in the command
        self.command.handle_success(request)

        # check for successful event
        kwargs = dict(share_id=self.command.share_id, node_id='a_node_id',
                      hash='yadda', new_generation=13)
        events = [('AQ_UPLOAD_FINISHED', (), kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)

    def test_handle_failure_push_event(self):
        """Test AQ_UPLOAD_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        res = self.command.handle_failure(failure=failure)
        kwargs = dict(share_id=self.command.share_id, node_id='a_node_id',
                      hash='yadda', error=msg)
        events = [('AQ_UPLOAD_ERROR', (), kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)

    def test_handle_failure_removes_temp_file(self):
        """Test temp file is removed on failure."""
        class TempFile(object): pass
        self.command.tempfile = TempFile()
        self.command.tempfile.name = os.path.join(self.tmpdir, 'remove-me.zip')
        open(self.command.tempfile.name, 'w').close()
        assert os.path.exists(self.command.tempfile.name)

        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        self.command.handle_failure(failure=failure)

        self.assertFalse(os.path.exists(self.command.tempfile.name))

    def test_handle_failure_push_quota_exceeded_if_quota_exceeded_error(self):
        """Test SYS_QUOTA_EXCEEDED is pushed on QuotaExceededError."""
        protocol_msg = protocol_pb2.Message()
        protocol_msg.type = protocol_pb2.Message.ERROR
        protocol_msg.error.type = protocol_pb2.Error.QUOTA_EXCEEDED
        protocol_msg.free_space_info.share_id = self.command.share_id
        protocol_msg.free_space_info.free_bytes = 1331564676
        error = errors.QuotaExceededError("request", protocol_msg)
        failure = Failure(error)

        res = self.command.handle_failure(failure=failure)
        volume_id = self.command.share_id
        kwargs = dict(share_id=volume_id, node_id='a_node_id',
                      hash='yadda', error=str(error))
        event1 = ('SYS_QUOTA_EXCEEDED', (), {'volume_id': volume_id,
                                             'free_bytes': 1331564676})
        event2 = ('AQ_UPLOAD_ERROR', (), kwargs)
        self.assertTrue(event1 in self.command.action_queue.event_queue.events)
        self.assertTrue(event2 in self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)

    @defer.inlineCallbacks
    def test_compress_failed_pushes_upload_error(self):
        msg = 'Zip can not be accomplished.'
        error = DefaultException(msg)
        self.action_queue.zip_queue.zip = lambda upload: defer.fail(error)
        yield self.command.run()
        kwargs = dict(share_id=self.command.share_id, node_id='a_node_id',
                      hash='yadda', error=msg)
        events = [('AQ_UPLOAD_ERROR', (), kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)

    def test_AQ_UPLOAD_FILE_PROGRESS_is_valid_event(self):
        """AQ_UPLOAD_FILE_PROGRESS is a valid event."""
        event = 'AQ_UPLOAD_FILE_PROGRESS'
        self.assertTrue(event in EVENTS)
        self.assertEqual(('share_id', 'node_id', 'n_bytes_written',
                          'deflated_size'), EVENTS[event])

    def test_progress_hook(self):
        """Test the progress hook."""
        self.command.deflated_size = 2*TRANSFER_PROGRESS_THRESHOLD
        self.command.n_bytes_written_last = 0

        self.command.progress_hook(5)
        self.assertEqual([], self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_written_last, 5)

        self.command.progress_hook(TRANSFER_PROGRESS_THRESHOLD)
        self.assertEqual([], self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_written_last,
                         TRANSFER_PROGRESS_THRESHOLD)

        self.command.n_bytes_written_last = 5
        self.command.progress_hook(TRANSFER_PROGRESS_THRESHOLD+14)

        kwargs = {'share_id': self.command.share_id, 'node_id': 'a_node_id',
                  'deflated_size': 2*TRANSFER_PROGRESS_THRESHOLD,
                  'n_bytes_written': 14+TRANSFER_PROGRESS_THRESHOLD }
        events = [('AQ_UPLOAD_FILE_PROGRESS', (), kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertEqual(self.command.n_bytes_written_last,
                         14+TRANSFER_PROGRESS_THRESHOLD)

    def test_runnable_space_ok(self):
        """The upload is runnable if space ok."""
        self.action_queue.have_sufficient_space_for_upload = lambda *a: True
        self.assertTrue(self.command.is_runnable)

    def test_runnable_space_bad(self):
        """The upload is not runnable without free space."""
        self.action_queue.have_sufficient_space_for_upload = lambda *a: False
        self.assertFalse(self.command.is_runnable)

    def test_runnable_space_bad_cancelled(self):
        """The upload is runnable if cancelled even with no free space."""
        self.action_queue.have_sufficient_space_for_upload = lambda *a: False
        self.command.cancelled = True
        self.assertTrue(self.command.is_runnable)

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        res = [getattr(self.command, x) for x in self.command.possible_markers]
        self.assertEqual(res, ['a_node_id'])

    def test_cancel_set_cancelled(self):
        """Set the command to cancelled."""
        assert not self.command.cancelled, "test badly set up"
        self.command.cancel()
        self.assertTrue(self.command.cancelled)

    def test_cancel_upload_req_is_none(self):
        """It's ok to have upload_req in None when cancelling."""
        assert self.command.upload_req is None, "test badly set up"
        self.command.cancel()

    def test_cancel_upload_req_is_something(self):
        """upload_req is also cancelled."""
        # set up the mocker
        mocker = Mocker()
        obj = mocker.mock()
        obj.cancel()
        obj.producer

        # test
        with mocker:
            self.command.upload_req = obj
            self.command.cancel()

    def test_cancel_remove(self):
        """Remove the command from the queue."""
        # set up the mocker
        mocker = Mocker()
        obj = mocker.mock()
        obj.remove(self.command)

        # test
        with mocker:
            self.command._queue = obj
            self.command.cancel()

    def test_cancel_clean_up(self):
        """Clean up."""
        called = []
        self.command.cleanup = lambda: called.append(True)
        self.command.cancel()
        self.assertTrue(called)


class CreateShareTestCase(ConnectedBaseTestCase):
    """Test for CreateShare ActionQueueCommand."""

    @defer.inlineCallbacks
    def setUp(self):
        """Init."""
        yield super(CreateShareTestCase, self).setUp()
        self.request_queue = RequestQueue(name='foo',
                                          action_queue=self.action_queue)
        self.orig_create_share_http = CreateShare._create_share_http

    @defer.inlineCallbacks
    def tearDown(self):
        yield super(CreateShareTestCase, self).tearDown()
        CreateShare._create_share_http = self.orig_create_share_http

    @defer.inlineCallbacks
    def test_access_level_modify_http(self):
        """Test proper handling of the access level in the http case."""
        # replace _create_share_http with a fake, just to check the args
        d = defer.Deferred()
        def check_create_http(self, node_id, user, name, read_only, deferred):
            """Fire the deferred with the args."""
            d.callback((node_id, user, name, read_only))
            deferred.callback(None)
        CreateShare._create_share_http = check_create_http
        command = CreateShare(self.request_queue, 'node_id',
                                   'share_to@example.com', 'share_name',
                                   'Modify', 'marker')
        self.assertTrue(command.use_http, 'CreateShare should be in http mode')

        command._run()
        node_id, user, name, read_only = yield d
        self.assertEqual('node_id', node_id)
        self.assertEqual('share_to@example.com', user)
        self.assertEqual('share_name', name)
        self.assertFalse(read_only)

    @defer.inlineCallbacks
    def test_access_level_view_http(self):
        """Test proper handling of the access level in the http case."""
        # replace _create_share_http with a fake, just to check the args
        d = defer.Deferred()
        def check_create_http(self, node_id, user, name, read_only, deferred):
            """Fire the deferred with the args."""
            d.callback((node_id, user, name, read_only))
            deferred.callback(None)
        CreateShare._create_share_http = check_create_http
        command = CreateShare(self.request_queue, 'node_id',
                                   'share_to@example.com', 'share_name',
                                   'View', 'marker')
        self.assertTrue(command.use_http, 'CreateShare should be in http mode')
        command._run()
        node_id, user, name, read_only = yield d
        self.assertEqual('node_id', node_id)
        self.assertEqual('share_to@example.com', user)
        self.assertEqual('share_name', name)
        self.assertTrue(read_only)

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        cmd = CreateShare(self.request_queue, 'node_id', 'shareto@example.com',
                          'share_name', 'View', 'marker')
        res = [getattr(cmd, x) for x in cmd.possible_markers]
        self.assertEqual(res, ['node_id'])


class RequestQueueManagerTestCase(FactoryBaseTestCase):
    """Test how RequestQueue manages the queues."""

    def setUp(self):
        FactoryBaseTestCase.setUp(self)

        self.queue = self.action_queue.meta_queue
        self.cmd = FakeCommand()

    def _events(self):
        """Helper method to see only the events."""
        return [x[0] for x in self.action_queue.event_queue.events]

    def test_empty_gets_one(self):
        """Queue and get the event."""
        self.queue.queue(self.cmd)
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING'])

    def test_with_one_gets_second(self):
        """Queue a second one, no event received."""
        self.queue.queue(self.cmd)
        self.queue.queue(self.cmd)
        # only get the event of the first one
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING'])

    @defer.inlineCallbacks
    def test_empty_run(self):
        """No event received when running nothing."""
        yield self.queue.run()
        self.assertEqual(self._events(), [])

    @defer.inlineCallbacks
    def test_with_one_run(self):
        """Run will execute the command."""
        self.queue.queue(self.cmd)
        yield self.queue.run()
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING',
                                          'SYS_META_QUEUE_DONE'])

    @defer.inlineCallbacks
    def test_with_two_run(self):
        """Run will execute both commands."""
        # first queuing, get the event
        self.queue.queue(self.cmd)
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING'])

        # second queuing, don't get the event
        self.queue.queue(self.cmd)
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING'])

        # first run, will get the waiting for the second event
        yield self.queue.run()
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING',
                                          'SYS_META_QUEUE_WAITING'])

        # second run, now we're done
        yield self.queue.run()
        self.assertEqual(self._events(), ['SYS_META_QUEUE_WAITING',
                                          'SYS_META_QUEUE_WAITING',
                                          'SYS_META_QUEUE_DONE'])

    def test_len_empty(self):
        """Counter return that it's empty."""
        self.assertEqual(len(self.queue), 0)

    def test_len_with_one(self):
        """Counter return that it has one."""
        self.queue.queue(self.cmd)
        self.assertEqual(len(self.queue), 1)

    def test_len_with_two(self):
        """Counter return that it has two."""
        self.queue.queue(self.cmd)
        self.queue.queue(self.cmd)
        self.assertEqual(len(self.queue), 2)

    @defer.inlineCallbacks
    def test_len_run_decreases(self):
        """Counter behaviour when adding/running."""
        self.queue.queue(self.cmd)
        self.assertEqual(len(self.queue), 1)
        self.queue.queue(self.cmd)
        self.assertEqual(len(self.queue), 2)
        yield self.queue.run()
        self.assertEqual(len(self.queue), 1)
        yield self.queue.run()
        self.assertEqual(len(self.queue), 0)


class DeleteShareTestCase(ConnectedBaseTestCase):
    """Test for DeleteShare ActionQueueCommand."""

    @defer.inlineCallbacks
    def setUp(self):
        """Init."""
        yield super(DeleteShareTestCase, self).setUp()
        request_queue = RequestQueue(name='fu', action_queue=self.action_queue)
        self.command = DeleteShare(request_queue, SHARE)
        vm = self.action_queue.main.vm
        self.patch(vm, 'handle_AQ_DELETE_SHARE_OK', NO_OP)
        self.patch(vm, 'handle_AQ_DELETE_SHARE_ERROR', NO_OP)

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        res = self.command._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's delete_volume is called."""
        self.called = False
        def check(share_id):
            """Take control over client's feature."""
            self.called = True
            self.assertEqual(SHARE, share_id)
        self.patch(self.command.action_queue.client, 'delete_share', check)
        self.command._run()
        self.assertTrue(self.called, "command wasn't called")

    def test_handle_success_push_event(self):
        """Test AQ_DELETE_SHARE_OK is pushed on success."""
        request = client.DeleteShare(self.action_queue.client, SHARE)
        res = self.command.handle_success(success=request)
        events = [('AQ_DELETE_SHARE_OK', (), {'share_id': SHARE})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertEqual(request, res)

    def test_handle_failure_push_event(self):
        """Test AQ_DELETE_SHARE_ERROR is pushed on failure."""
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))
        res = self.command.handle_failure(failure=failure)
        events = [('AQ_DELETE_SHARE_ERROR', (),
                  {'share_id': SHARE, 'error': msg})]
        self.assertEqual(events, self.command.action_queue.event_queue.events)
        self.assertTrue(res is None)


class SimpleAQTestCase(BasicTestCase):
    """Simple tests for AQ API."""

    def test_aq_query_volumes(self):
        """Check the API of AQ.query_volumes."""
        self.main.start()
        d = defer.Deferred()
        def list_volumes():
            """Fake list_volumes."""
            result = DummyClass()
            result.volumes = ['foo', 'bar']
            return defer.succeed(result)

        self.action_queue.client = DummyClass()
        self.action_queue.client.list_volumes = list_volumes
        d = self.action_queue.query_volumes()
        def check(result):
            self.assertIn('foo', result)
            self.assertIn('bar', result)
            return result
        d.addCallback(check)
        return d

    def test_have_sufficient_space_for_upload_if_free_space_is_none(self):
        """Check have_sufficient_space_for_upload.

        If free_space is None, SYS_QUOTA_EXCEEDED is not pushed.

        """
        self.patch(self.action_queue.main.vm, 'get_free_space',
                   lambda share_id: None)  # free space is None
        volume_id = 'test share'
        res = self.action_queue.have_sufficient_space_for_upload(volume_id,
                                                                 upload_size=1)
        self.assertTrue(res, "Must have enough space to upload.")
        events = map(operator.itemgetter(0),
                     self.action_queue.event_queue.events)
        self.assertNotIn('SYS_QUOTA_EXCEEDED', events)

    def test_have_sufficient_space_for_upload_if_no_free_space(self):
        """Check have_sufficient_space_for_upload pushes SYS_QUOTA_EXCEEDED."""
        self.patch(self.action_queue.main.vm, 'get_free_space',
                   lambda share_id: 0) # no free space, always
        volume_id = 'test share'
        res = self.action_queue.have_sufficient_space_for_upload(volume_id,
                                                                 upload_size=1)
        self.assertEqual(res, False, "Must not have enough space to upload.")
        msg = 'SYS_QUOTA_EXCEEDED must have been pushed to event queue.'
        expected = ('SYS_QUOTA_EXCEEDED', (), {'volume_id': volume_id,
                                               'free_bytes': 0})
        self.assertTrue(expected in self.action_queue.event_queue.events, msg)

    def test_have_sufficient_space_for_upload_if_free_space(self):
        """Check have_sufficient_space_for_upload doesn't push any event."""
        self.patch(self.action_queue.main.vm, 'get_free_space',
                   lambda share_id: 1) # free space, always
        res = self.action_queue.have_sufficient_space_for_upload(share_id=None,
                                                                 upload_size=0)
        self.assertEqual(res, True, "Must have enough space to upload.")
        msg = 'No event must have been pushed to event queue.'
        self.assertEqual(self.action_queue.event_queue.events, [], msg)

    def test_SYS_QUOTA_EXCEEDED_is_valid_event(self):
        """SYS_QUOTA_EXCEEDED is a valid event."""
        event = 'SYS_QUOTA_EXCEEDED'
        self.assertTrue(event in EVENTS)
        self.assertEqual(('volume_id', 'free_bytes'), EVENTS[event])

    def test_SYS_USER_CONNECT_is_valid_event(self):
        """SYS_USER_CONNECT is a valid event."""
        event = 'SYS_USER_CONNECT'
        self.assertIn(event, EVENTS)
        self.assertEqual(('access_token',), EVENTS[event])

    def test_handle_SYS_USER_CONNECT(self):
        """handle_SYS_USER_CONNECT stores credentials."""
        self.assertEqual(self.action_queue.token, None)
        self.assertEqual(self.action_queue.consumer, None)

        self.user_connect()

        expected = oauth.OAuthToken('bla', 'ble')
        self.assertEqual(self.action_queue.token.key, expected.key)
        self.assertEqual(self.action_queue.token.secret, expected.secret)

        expected = oauth.OAuthConsumer('foo', 'bar')
        self.assertEqual(self.action_queue.consumer.key, expected.key)
        self.assertEqual(self.action_queue.consumer.secret, expected.secret)


class SpecificException(Exception):
    """The specific exception."""


class SillyClass(object):
    """Silly class that accepts the set of any attribute.

    We can't use object() directly, since its raises AttributeError.

    """


class ErrorHandlingTestCase(BasicTestCase):
    """Error handling tests for ActionQueue."""

    def setUp(self):
        """Init."""
        BasicTestCase.setUp(self)

        self.called = False
        self.action_queue.client = SillyClass()
        self.patch(self.main, 'restart', lambda: None)

        self.main.start()

    def fail_please(self, an_exception):
        """Raise the given exception."""
        def inner(*args, **kwargs):
            """A request to the server that fails."""
            self.called = True
            return defer.fail(an_exception)
        return inner

    def succeed_please(self, result):
        """Return the given result."""
        def inner(*args, **kwargs):
            """A request to the server that succeeds."""
            self.called = True
            return defer.succeed(result)
        return inner

    def mock_caps(self, accepted):
        """Reply to query caps with False."""
        def gset_caps(caps):
            """get/set caps helper."""
            req = SillyClass()
            req.caps = caps
            req.accepted = accepted
            return defer.succeed(req)
        return gset_caps

    def test_valid_event(self):
        """SYS_SERVER_ERROR is valid in EventQueue."""
        event = 'SYS_SERVER_ERROR'
        self.assertTrue(event in EVENTS)
        self.assertEqual(('error',), EVENTS[event])

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_no_error(self):
        """_send_request_and_handle_errors is correct when no error."""

        event = 'SYS_SPECIFIC_OK'
        EVENTS[event] = () # add event to the global valid events list
        self.addCleanup(lambda: EVENTS.pop(event))

        result = object()
        request = self.succeed_please(result)
        kwargs = dict(request=request, request_error=SpecificException,
                      event_error='YADDA_YADDA', event_ok=event,
                      args=(1, 2), kwargs={})
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        actual_result = yield d

        self.assertTrue(self.called, 'the request was called')
        self.assertEqual(actual_result, result)
        self.assertEqual((event, (), {}),
                         self.action_queue.event_queue.events[-1])

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertIn(request.__name__, record.message)
        self.assertIn('OK', record.message)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_with_no_event_ok(self):
        """_send_request_and_handle_errors does not push event if is None."""
        original_events = self.action_queue.event_queue.events[:]

        result = object()
        request = self.succeed_please(result)
        kwargs = dict(request=request, request_error=SpecificException,
                      event_error='YADDA_YADDA', event_ok=None)
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        actual_result = yield d

        self.assertTrue(self.called, 'the request was called')
        self.assertEqual(actual_result, result)
        self.assertEqual(original_events,
                         self.action_queue.event_queue.events)

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertIn(request.__name__, record.message)
        self.assertIn('OK', record.message)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_valid_error(self):
        """_send_request_and_handle_errors is correct when expected error."""

        event = 'SYS_SPECIFIC_ERROR'
        EVENTS[event] = ('error',) # add event to the global valid events list
        self.addCleanup(lambda: EVENTS.pop(event))

        exc = SpecificException('The request failed! please be happy.')
        request = self.fail_please(exc)
        kwargs = dict(request=request, request_error=SpecificException,
                      event_error=event, event_ok='YADDA_YADDA')
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        yield d

        self.assertTrue(self.called, 'the request was called')
        self.assertEqual((event, (), {'error': str(exc)}),
                         self.action_queue.event_queue.events[-1])

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertEqual(record.levelno, logging.INFO)
        self.assertIn(request.__name__, record.message)
        self.assertIn(event, record.message)
        self.assertIn(str(exc), record.message)

    @defer.inlineCallbacks
    def assert_send_request_and_handle_errors_on_server_error(self, serr):
        """_send_request_and_handle_errors is correct when server error."""
        # XXX: we need to replace this list with and exception list
        # once bug #557718 is resolved
        msg = protocol_pb2.Message()
        msg.type = protocol_pb2.Message.ERROR
        msg.error.type = serr
        msg.error.comment = 'Error message for %s.' % serr
        exc = errors.error_to_exception(serr)(request=None, message=msg)

        request = self.fail_please(exc)
        kwargs = dict(request=request, request_error=SpecificException,
                      event_error='BAR', event_ok='FOO')
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        yield d

        event = 'SYS_SERVER_ERROR'
        self.assertEqual((event, (), {'error': str(exc)}),
                         self.action_queue.event_queue.events[-1])

        # assert over logging
        self.assertTrue(len(self.handler.records) > 0)
        record = self.handler.records[-1]
        self.assertEqual(record.levelno, logging.INFO)
        self.assertIn(request.__name__, record.message)
        self.assertIn(event, record.message)
        self.assertIn(str(exc), record.message)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_try_again(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.TRY_AGAIN
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_internal_error(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.INTERNAL_ERROR
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_protocol_error(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.PROTOCOL_ERROR
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_unsupported_version(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.UNSUPPORTED_VERSION
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_authetication_failed(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.AUTHENTICATION_FAILED
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_no_permission(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.NO_PERMISSION
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_already_exists(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.ALREADY_EXISTS
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_does_not_exist(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.DOES_NOT_EXIST
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_not_a_dir(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.NOT_A_DIRECTORY
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_not_empty(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.NOT_EMPTY
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_not_available(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.NOT_AVAILABLE
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_upload_in_porgress(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.UPLOAD_IN_PROGRESS
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_upload_corrupt(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.UPLOAD_CORRUPT
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_upload_canceled(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.UPLOAD_CANCELED
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_conflict(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.CONFLICT
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_quota_exceeded(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.QUOTA_EXCEEDED
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_invalid_filename(self):
        """_send_request_and_handle_errors is correct when server error."""
        serr = protocol_pb2.Error.INVALID_FILENAME
        yield self.assert_send_request_and_handle_errors_on_server_error(serr)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_unknown_error(self):
        """_send_request_and_handle_errors is correct when unknown error."""
        # XXX: we need to replace this list with and exception list
        # once bug #557718 is resolved
        serr = protocol_pb2.Error.AUTHENTICATION_REQUIRED
        msg = protocol_pb2.Message()
        msg.type = protocol_pb2.Message.ERROR
        msg.error.type = serr
        msg.error.comment = 'Error message for %s.' % serr
        exc = errors.error_to_exception(serr)(request=None, message=msg)

        request = self.fail_please(exc)
        kwargs = dict(request=request, request_error=SpecificException,
                  event_error='BAR', event_ok='FOO')
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        yield d

        event = 'SYS_UNKNOWN_ERROR'
        self.assertIn((event, (), {}), self.action_queue.event_queue.events)

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertEqual(record.levelno, logging.INFO)
        self.assertIn(request.__name__, record.message)
        self.assertIn(event, record.message)
        self.assertIn(str(exc), record.message)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_no_protocol_error(self):
        """_send_request_and_handle_errors is ok when no-protocol error."""

        event = 'SYS_UNKNOWN_ERROR'
        error_msg = 'Error message for any Exception.'
        exc = Exception(error_msg)
        request = self.fail_please(exc)
        kwargs = dict(request=request, request_error=SpecificException,
                      event_error='BAR', event_ok='FOO')
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        yield d

        self.assertIn((event, (), {}),
                      self.action_queue.event_queue.events)

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertEqual(record.levelno, logging.INFO)
        self.assertIn(request.__name__, record.message)
        self.assertIn(event, record.message)
        self.assertIn(str(exc), record.message)

    @defer.inlineCallbacks
    def test_send_request_and_handle_errors_on_client_mismatch(self):
        """_send_request_and_handle_errors is correct when client mismatch."""

        def change_client(*args, **kwargs):
            """Change AQ's client while doing the request."""
            self.action_queue.client = object()

        self.action_queue.event_queue.events = [] # event cleanup
        kwargs = dict(request=change_client, request_error=SpecificException,
                      event_error='BAR', event_ok='FOO')
        d = self.action_queue._send_request_and_handle_errors(**kwargs)
        yield d

        self.assertEqual([], self.action_queue.event_queue.events)

        # assert over logging
        self.assertEqual(1, len(self.handler.records))
        record = self.handler.records[0]
        self.assertIn(change_client.__name__, record.message)
        self.assertIn('Client mismatch', record.message)

    @defer.inlineCallbacks
    def test_check_version_when_unsupported_version_exception(self):
        """Test error handling after UnsupportedVersionError."""
        # raise a UnsupportedVersionError
        msg = protocol_pb2.Message()
        msg.type = protocol_pb2.Message.ERROR
        msg.error.type = protocol_pb2.Error.UNSUPPORTED_VERSION
        msg.error.comment = 'This is a funny comment.'
        exc = errors.UnsupportedVersionError(request=None, message=msg)

        self.action_queue.client.protocol_version = self.fail_please(exc)
        yield self.action_queue.check_version()
        event = ('SYS_PROTOCOL_VERSION_ERROR', (), {'error': str(exc)})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])

    @defer.inlineCallbacks
    def test_set_capabilities_when_query_caps_not_accepted(self):
        """Test error handling when the query caps are not accepeted."""

        # query_caps returns False
        self.action_queue.client.query_caps = self.mock_caps(accepted=False)

        yield self.action_queue.set_capabilities(caps=None)
        msg = "The server doesn't have the requested capabilities"
        event = ('SYS_SET_CAPABILITIES_ERROR', (), {'error': msg})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])
        self.assertNotIn(('SYS_SET_CAPABILITIES_OK', (), {}),
                          self.action_queue.event_queue.events)

    @defer.inlineCallbacks
    def test_set_capabilities_when_set_caps_not_accepted(self):
        """Test error handling when the query caps are not accepted."""

        # query_caps returns True and set_caps returns False
        self.action_queue.client.query_caps = self.mock_caps(accepted=True)
        self.action_queue.client.set_caps = self.mock_caps(accepted=False)

        caps = 'very difficult cap'
        yield self.action_queue.set_capabilities(caps=caps)
        msg = "The server denied setting '%s' capabilities" % caps
        event = ('SYS_SET_CAPABILITIES_ERROR', (), {'error': msg})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])
        self.assertNotIn(('SYS_SET_CAPABILITIES_OK', (), {}),
                          self.action_queue.event_queue.events)

    @defer.inlineCallbacks
    def test_set_capabilities_when_client_is_none(self):
        """Test error handling when the client is None."""

        self.action_queue.client = None

        yield self.action_queue.set_capabilities(caps=None)
        msg = "'NoneType' object has no attribute 'query_caps'"
        event = ('SYS_SET_CAPABILITIES_ERROR', (), {'error': msg})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])
        self.assertNotIn(('SYS_SET_CAPABILITIES_OK', (), {}),
                          self.action_queue.event_queue.events)

    @defer.inlineCallbacks
    def test_set_capabilities_when_set_caps_is_accepted(self):
        """Test error handling when the query caps are not accepeted."""

        # query_caps returns True and set_caps returns True
        self.action_queue.client.query_caps = self.mock_caps(accepted=True)
        self.action_queue.client.set_caps = self.mock_caps(accepted=True)

        yield self.action_queue.set_capabilities(caps=None)
        event = ('SYS_SET_CAPABILITIES_OK', (), {})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])

    @defer.inlineCallbacks
    def test_authenticate_when_authenticated(self):
        """Test error handling after authenticate with no error."""
        request = client.Authenticate(self.action_queue.client,
                                      {'dummy_token': 'credentials'})
        request.session_id = str(uuid.uuid4())
        self.action_queue.client.oauth_authenticate = \
            self.succeed_please(result=request)
        yield self.action_queue.authenticate()
        event = ('SYS_AUTH_OK', (), {})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])

    @defer.inlineCallbacks
    def test_authenticate_when_authentication_failed_exception(self):
        """Test error handling after AuthenticationFailedError."""
        # raise a AuthenticationFailedError
        msg = protocol_pb2.Message()
        msg.type = protocol_pb2.Message.ERROR
        msg.error.type = protocol_pb2.Error.AUTHENTICATION_FAILED
        msg.error.comment = 'This is a funny comment.'
        exc = errors.AuthenticationFailedError(request=None, message=msg)

        self.action_queue.client.oauth_authenticate = self.fail_please(exc)
        yield self.action_queue.authenticate()
        event = ('SYS_AUTH_ERROR', (), {'error': str(exc)})
        self.assertEqual(event, self.action_queue.event_queue.events[-1])


class GetDeltaTestCase(ConnectedBaseTestCase):
    """Test for GetDelta ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(GetDeltaTestCase, self).setUp()
        self.handler.setLevel(logging.DEBUG)
        self.rq = RequestQueue(name='META_QUEUE',
                               action_queue=self.action_queue)
        return d

    def test_action_queue_get_delta(self):
        """Test AQ get delta."""
        res = self.action_queue.get_delta(VOLUME, 0)
        self.assertTrue(res is None) # this is what start returns

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        cmd = GetDelta(self.rq, VOLUME, 0)
        self.assertTrue(isinstance(cmd, ActionQueueCommand))

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        cmd = GetDelta(self.rq, VOLUME, 0)
        res = cmd._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's get delta is called."""
        called = []
        self.patch(self.action_queue.client, 'get_delta',
                   lambda *a: called.append(a))

        cmd = GetDelta(self.rq, VOLUME, 35)
        cmd._run()
        self.assertEqual(called[0], (VOLUME, 35))

    def test_handle_success_push_event(self):
        """Test AQ_DELTA_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.GetDelta(self.action_queue.client,
                                  share_id=VOLUME, from_generation=21)
        request.response = ['foo', 'bar']
        request.end_generation = 76
        request.full = True
        request.free_bytes = 1231234

        # create a command and trigger it success
        cmd = GetDelta(self.rq, VOLUME, 21)
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        delta_info = dict(volume_id=VOLUME, delta_content=['foo', 'bar'],
                          end_generation=76,
                          full=True, free_bytes=1231234)
        self.assertEqual(received, ('AQ_DELTA_OK', (), delta_info))
        self.assertTrue(isinstance(received[2]["delta_content"], DeltaList))

    def test_handle_generic_failure_push_event(self):
        """Test AQ_DELTA_ERROR is pushed on failure."""
        # create a failure
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))

        # create a command and trigger it success
        cmd = GetDelta(self.rq, VOLUME, 77)
        cmd.handle_failure(failure=failure)

        # check for event
        received = self.action_queue.event_queue.events[0]
        self.assertEqual(received, ('AQ_DELTA_ERROR', (),
                                    {'volume_id': VOLUME, 'error': msg}))

    def test_handle_notpossible_failure_push_event(self):
        """Test AQ_DELTA_NOT_POSSIBLE is pushed on that failure."""
        # create a failure
        msg = protocol_pb2.Message()
        msg.type = protocol_pb2.Message.ERROR
        msg.error.type = protocol_pb2.Error.CANNOT_PRODUCE_DELTA
        msg.error.comment = 'Something went wrong'
        failure = Failure(errors.CannotProduceDelta(self.rq, msg))

        # create a command and trigger it success
        cmd = GetDelta(self.rq, VOLUME, 2)
        cmd.handle_failure(failure=failure)

        # check for event
        received = self.action_queue.event_queue.events[0]
        self.assertEqual(received, ('AQ_DELTA_NOT_POSSIBLE', (),
                                    {'volume_id': VOLUME}))

    def test_queued_mixed_types(self):
        """Command gets queued if other command is waiting."""
        cmd1 = FakeCommand()
        self.rq.waiting.append(cmd1)
        cmd2 = GetDelta(self.rq, 'vol2', 0)
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])
        self.assertTrue(self.handler.check_debug("queueing in the"))

    def test_queued_two_different(self):
        """Two different queued commands is ok."""
        cmd1 = GetDelta(self.rq, 'vol1', 0)
        cmd1.queue()
        cmd2 = GetDelta(self.rq, 'vol2', 0)
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])

    def test_queued_two_equals_one_in_head(self):
        """Second cmd equal to first is queued if later already processing."""
        # create first command and run the queue for it to get into _head
        cmd1 = GetDelta(self.rq, 'vol1', 0)
        cmd1.queue()
        self.rq.run()

        # create second command and start it, then check
        cmd2 = GetDelta(self.rq, 'vol1', 0)
        cmd2.queue()
        self.assertEqual(self.rq._head, cmd1)
        self.assertEqual(list(self.rq.waiting), [cmd2])

    def test_queued_two_equal_second_bigger(self):
        """When two equals, only survive the one with smaller gen, first."""
        cmd1 = GetDelta(self.rq, 'vol', 3)
        cmd1.queue()
        cmd2 = GetDelta(self.rq, 'vol', 5)
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1])
        self.assertTrue(self.handler.check_debug("not queueing self"))

    def test_queued_two_equal_second_smaller(self):
        """When two equals, only survive the one with smaller gen, second."""
        cmd1 = GetDelta(self.rq, 'vol', 5)
        cmd1.queue()
        cmd2 = GetDelta(self.rq, 'vol', 3)
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd2])
        self.assertTrue(self.handler.check_debug("removing previous command"))

    def test_queued_three_equal(self):
        """When several equals, only survive the one with smaller gen."""
        cmd1 = GetDelta(self.rq, 'vol', 5)
        cmd1.queue()
        cmd2 = GetDelta(self.rq, 'vol', 3)
        cmd2.queue()
        cmd3 = GetDelta(self.rq, 'vol', 7)
        cmd3.queue()
        self.assertEqual(list(self.rq.waiting), [cmd2])


class GetDeltaFromScratchTestCase(ConnectedBaseTestCase):
    """Test for GetDelta ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(GetDeltaFromScratchTestCase, self).setUp()
        self.handler.setLevel(logging.DEBUG)
        self.rq = RequestQueue(name='META_QUEUE',
                               action_queue=self.action_queue)
        return d

    def test_action_queue_get_delta(self):
        """Test AQ get delta."""
        res = self.action_queue.rescan_from_scratch(VOLUME)
        self.assertTrue(res is None) # this is what start returns

    def test_is_action_queue_command(self):
        """Test proper inheritance."""
        cmd = GetDeltaFromScratch(self.rq, VOLUME)
        self.assertTrue(isinstance(cmd, ActionQueueCommand))

    def test_run_returns_a_deferred(self):
        """Test a deferred is returned."""
        cmd = GetDelta(self.rq, VOLUME, 0)
        res = cmd._run()
        self.assertTrue(isinstance(res, defer.Deferred), 'deferred returned')

    def test_run_calls_protocol(self):
        """Test protocol's get delta is called."""
        called = []
        self.patch(self.action_queue.client, 'get_delta',
                   lambda *a, **b: called.append((a, b)))

        cmd = GetDeltaFromScratch(self.rq, VOLUME)
        cmd._run()
        self.assertEqual(called[0], ((VOLUME,), {'from_scratch': True}))

    def test_handle_success_push_event(self):
        """Test AQ_DELTA_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.GetDelta(self.action_queue.client,
                                  share_id=VOLUME, from_scratch=True)
        request.response = ['foo', 'bar']
        request.end_generation = 76
        request.full = True
        request.free_bytes = 1231234

        # create a command and trigger it success
        cmd = GetDeltaFromScratch(self.rq, VOLUME)
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        delta_info = dict(volume_id=VOLUME, delta_content=['foo', 'bar'],
                          end_generation=76,
                          free_bytes=1231234)
        self.assertEqual(received, ('AQ_RESCAN_FROM_SCRATCH_OK', (),
                                    delta_info))
        self.assertTrue(isinstance(received[2]["delta_content"], DeltaList))

    def test_handle_generic_failure_push_event(self):
        """Test AQ_DELTA_ERROR is pushed on failure."""
        # create a failure
        msg = 'Something went wrong'
        failure = Failure(DefaultException(msg))

        # create a command and trigger it success
        cmd = GetDeltaFromScratch(self.rq, VOLUME)
        cmd.handle_failure(failure=failure)

        # check for event
        received = self.action_queue.event_queue.events[0]
        self.assertEqual(received, ('AQ_RESCAN_FROM_SCRATCH_ERROR', (),
                                    {'volume_id': VOLUME, 'error': msg}))

    def test_queued_mixed_types(self):
        """Command gets queued if other command is waiting."""
        cmd1 = FakeCommand()
        self.rq.waiting.append(cmd1)
        cmd2 = GetDeltaFromScratch(self.rq, 'vol2')
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])
        self.assertTrue(self.handler.check_debug("queueing in the"))

    def test_queued_two_different(self):
        """Two different queued commands is ok."""
        cmd1 = GetDeltaFromScratch(self.rq, 'vol1')
        cmd1.queue()
        cmd2 = GetDeltaFromScratch(self.rq, 'vol2')
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1, cmd2])

    def test_queued_two_equals_one_in_head(self):
        """Second cmd equal to first is queued if later already processing."""
        # create first command and run the queue for it to get into _head
        cmd1 = GetDeltaFromScratch(self.rq, 'vol1')
        cmd1.queue()
        self.rq.run()

        # create second command and start it, then check
        cmd2 = GetDeltaFromScratch(self.rq, 'vol1')
        cmd2.queue()
        self.assertEqual(self.rq._head, cmd1)
        self.assertEqual(list(self.rq.waiting), [cmd2])

    def test_queued_two_equal(self):
        """When two equals, only survive the first one."""
        cmd1 = GetDeltaFromScratch(self.rq, 'vol')
        cmd1.queue()
        cmd2 = GetDeltaFromScratch(self.rq, 'vol')
        cmd2.queue()
        self.assertEqual(list(self.rq.waiting), [cmd1])
        self.assertTrue(self.handler.check_debug("not queueing self"))


class UnlinkTestCase(ConnectedBaseTestCase):
    """Test for Unlink ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(UnlinkTestCase, self).setUp()
        self.rq = RequestQueue(name='META_QUEUE', action_queue=self.action_queue)
        return d

    def test_handle_success_push_event(self):
        """Test AQ_UNLINK_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.Unlink(self.action_queue.client, VOLUME, 'node_id')
        request.new_generation = 13

        # create a command and trigger it success
        cmd = Unlink(self.rq, VOLUME, 'parent_id', 'node_id')
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(share_id=VOLUME, parent_id='parent_id',
                    node_id='node_id', new_generation=13)
        self.assertEqual(received, ('AQ_UNLINK_OK', (), info))

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        cmd = Unlink(self.rq, VOLUME, 'parent_id', 'node_id')
        res = [getattr(cmd, x) for x in cmd.possible_markers]
        self.assertEqual(res, ['node_id', 'parent_id'])


class MoveTestCase(ConnectedBaseTestCase):
    """Test for Move ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(MoveTestCase, self).setUp()
        self.rq = RequestQueue(name='META_QUEUE', action_queue=self.action_queue)
        return d

    def test_handle_success_push_event(self):
        """Test AQ_MOVE_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.Move(self.action_queue.client, VOLUME, 'node',
                                'new_parent', 'new_name')
        request.new_generation = 13

        # create a command and trigger it success
        cmd = Move(self.rq, VOLUME, 'node', 'o_parent', 'n_parent', 'n_name')
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(share_id=VOLUME, node_id='node', new_generation=13)
        self.assertEqual(received, ('AQ_MOVE_OK', (), info))

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        cmd = Move(self.rq, VOLUME, 'node', 'o_parent', 'n_parent', 'n_name')
        res = [getattr(cmd, x) for x in cmd.possible_markers]
        self.assertEqual(res, ['node', 'o_parent', 'n_parent'])


class MakeFileTestCase(ConnectedBaseTestCase):
    """Test for MakeFile ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(MakeFileTestCase, self).setUp()
        self.rq = RequestQueue(name='META_QUEUE', action_queue=self.action_queue)
        return d

    def test_handle_success_push_event(self):
        """Test AQ_FILE_NEW_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.MakeFile(self.action_queue.client, VOLUME,
                                  'parent', 'name')
        request.new_id = 'new_id'
        request.new_generation = 13

        # create a command and trigger it success
        cmd = MakeFile(self.rq, VOLUME, 'parent', 'name', 'marker')
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(marker='marker', new_id='new_id', new_generation=13,
                    volume_id=VOLUME)
        self.assertEqual(received, ('AQ_FILE_NEW_OK', (), info))

    def test_handle_failure_push_event(self):
        """Test AQ_FILE_NEW_ERROR is pushed on error."""
        # create a request and fill it with succesful information
        request = client.MakeFile(self.action_queue.client, VOLUME,
                                  'parent', 'name')
        request.new_id = 'new_id'
        request.new_generation = 13

        # create a command and trigger it fail
        cmd = MakeFile(self.rq, VOLUME, 'parent', 'name', 'marker')
        failure = Failure(Exception('foo'))
        cmd.handle_failure(failure)

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(marker='marker', failure=failure)
        self.assertEqual(received, ('AQ_FILE_NEW_ERROR', (), info))

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        cmd = MakeFile(self.rq, VOLUME, 'parent', 'name', 'marker')
        res = [getattr(cmd, x) for x in cmd.possible_markers]
        self.assertEqual(res, [ 'parent'])


class MakeDirTestCase(ConnectedBaseTestCase):
    """Test for MakeDir ActionQueueCommand."""

    def setUp(self):
        """Init."""
        d = super(MakeDirTestCase, self).setUp()
        self.rq = RequestQueue(name='META_QUEUE', action_queue=self.action_queue)
        return d

    def test_handle_success_push_event(self):
        """Test AQ_DIR_NEW_OK is pushed on success."""
        # create a request and fill it with succesful information
        request = client.MakeDir(self.action_queue.client, VOLUME,
                                 'parent', 'name')
        request.new_id = 'new_id'
        request.new_generation = 13

        # create a command and trigger it success
        cmd = MakeDir(self.rq, VOLUME, 'parent', 'name', 'marker')
        res = cmd.handle_success(request)
        assert res is request

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(marker='marker', new_id='new_id', new_generation=13,
                    volume_id=VOLUME)
        self.assertEqual(received, ('AQ_DIR_NEW_OK', (), info))

    def test_handle_failure_push_event(self):
        """Test AQ_DIR_NEW_ERROR is pushed on error."""
        # create a request and fill it with succesful information
        request = client.MakeDir(self.action_queue.client, VOLUME,
                                 'parent', 'name')
        request.new_id = 'new_id'
        request.new_generation = 13

        # create a command and trigger it fail
        cmd = MakeDir(self.rq, VOLUME, 'parent', 'name', 'marker')
        failure = Failure(Exception('foo'))
        cmd.handle_failure(failure)

        # check for successful event
        received = self.action_queue.event_queue.events[0]
        info = dict(marker='marker', failure=failure)
        self.assertEqual(received, ('AQ_DIR_NEW_ERROR', (), info))

    def test_possible_markers(self):
        """Test that it returns the correct values."""
        cmd = MakeDir(self.rq, VOLUME, 'parent', 'name', 'marker')
        res = [getattr(cmd, x) for x in cmd.possible_markers]
        self.assertEqual(res, ['parent'])


class TestDeltaList(unittest.TestCase):
    """Tests for DeltaList."""

    def test_is_list(self):
        """A DeltaList is a list."""
        l = [1, 2, 3]
        a = DeltaList(l)
        self.assertTrue(isinstance(a, list))

    def test_is_equal_list(self):
        """A DeltaList is equal to the list it represents."""
        l = [1, 2, 3]
        a = DeltaList(l)
        self.assertEqual(a, l)

    def test_repr(self):
        """A DeltaList has a short representation."""
        a = DeltaList(["a"*1000])
        self.assertTrue(len(repr(a)) < 100)
        self.assertTrue(len(str(a)) < 100)


class AuthenticateTestCase(ConnectedBaseTestCase):
    """Tests for authenticate."""

    def setUp(self):
        """Init."""
        d = super(AuthenticateTestCase, self).setUp()
        self.rq = RequestQueue(name='META_QUEUE', action_queue=self.action_queue)
        return d

    @defer.inlineCallbacks
    def test_session_id_is_logged(self):
        """Test that session_id is logged after auth ok."""
        request = client.Authenticate(self.action_queue.client,
                                      {'dummy_token': 'credentials'})
        request.session_id = str(uuid.uuid4())
        self.action_queue.client.oauth_authenticate = \
                lambda *args: defer.succeed(request)

        yield self.action_queue.authenticate()

        self.assertTrue(self.handler.check_note('Session ID: %r' %
                                                str(request.session_id)))


class ActionQueueProtocolTests(TwistedTestCase):
    """Test the ACQ class."""

    def setUp(self):
        """Set up."""
        # create an AQP and put a factory to it
        self.aqp = ActionQueueProtocol()
        obj = Mocker().mock()
        obj.event_queue.push('SYS_CONNECTION_MADE')
        self.aqp.factory = obj

        # set up the logger
        self.handler = MementoHandler()
        self.handler.setLevel(logging.DEBUG)
        self.aqp.log.addHandler(self.handler)

    def tearDown(self):
        """Tear down."""
        self.aqp.log.removeHandler(self.handler)
        task = self.aqp._looping_ping
        if task is not None and task.running:
            task.stop()

    def test_connection_made(self):
        """Connection is made."""
        mocker = Mocker()
        obj = mocker.mock()
        obj.event_queue.push('SYS_CONNECTION_MADE')
        self.aqp.factory = obj

        # test
        with mocker:
            self.aqp.connectionMade()
        self.assertTrue(self.handler.check_info('Connection made.'))
        self.assertFalse(self.aqp._looping_ping is None)

    def test_connection_lost(self):
        """Connection is lost."""
        self.aqp.connectionLost('foo')
        self.assertTrue(self.handler.check_info(
                        'Connection lost, reason: foo.'))
        self.assertTrue(self.aqp._looping_ping is None)

    def test_ping_connection_made_twice(self):
        """If connection made is called twice, don't create two tasks."""
        self.aqp.connectionMade()
        task1 = self.aqp._looping_ping
        self.aqp.connectionMade()
        task2 = self.aqp._looping_ping
        self.assertTrue(task1 is task2)

    def test_ping_connection_lost_twice(self):
        """If connection lost is called twice, don't stop None."""
        self.aqp.connectionMade()
        self.assertFalse(self.aqp._looping_ping is None)
        self.aqp.connectionLost('reason')
        self.assertTrue(self.aqp._looping_ping is None)
        self.aqp.connectionLost('reason')
        self.assertTrue(self.aqp._looping_ping is None)

    @defer.inlineCallbacks
    def test_ping_task_calls_ping(self):
        """The task will call the _do_ping method."""
        self.aqp._ping_delay = .1
        deferred = defer.Deferred()

        def fake_ping():
            """Stop the loop and trigger the deferred test."""
            self.aqp._looping_ping.stop()
            deferred.callback(True)

        self.aqp._do_ping = fake_ping
        self.aqp.connectionMade()
        yield deferred

    def test_ping_do_ping(self):
        """Ping and log."""
        # mock the request
        mocker = Mocker()
        req = mocker.mock()
        req.rtt
        mocker.result(1.123123)
        self.aqp.ping = lambda: defer.succeed(req)

        # ping will be called, and req accessed, otherwise mocker will complain
        with mocker:
            self.aqp._do_ping()

        # check also the log
        self.assertTrue(self.handler.check_debug('Ping! rtt: 1.123 segs'))

