# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for main.Main class """
import dbus
from dbus.mainloop.glib import DBusGMainLoop
import os
import shutil

from twisted.internet import defer, reactor

from ubuntuone.syncdaemon.main import Main
from ubuntuone.syncdaemon.dbus_interface import DBusInterface
from contrib.testing.testcase import (
    BaseTwistedTestCase,
)

DBusInterface.test =True


class MainTests(BaseTwistedTestCase):
    """ Basic tests to check main.Main """

    def setUp(self):
        """ Sets up a test. """
        BaseTwistedTestCase.setUp(self)
        self.root = self.mktemp('root')
        self.shares = self.mktemp('shares')
        self.data = self.mktemp('data')
        self.partials_dir = self.mktemp('partials_dir')
        # monkeypatch busName.__del__ to avoid errors on gc
        # we take care of releasing the name in shutdown
        dbus.service.BusName.__del__ = lambda _: None

    def build_main(self, **kwargs):
        """
        Build and return a Main object, using reasonable defaults for
        the tests, plus whatever extra kwargs are passed in.
        """
        params = dict(root_dir=self.root,
                      shares_dir=self.shares,
                      data_dir=self.data,
                      partials_dir=self.partials_dir,
                      host='localhost', port=0,
                      dns_srv=False, ssl=False,
                      mark_interval=60,
                      handshake_timeout=2,
                      glib_loop=DBusGMainLoop(set_as_default=True))
        params.update(kwargs)
        m = Main(**params)
        m.local_rescan = lambda *_: m.event_q.push('SYS_LOCAL_RESCAN_DONE')
        return m

    def tearDown(self):
        """Clean up the tests."""
        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        BaseTwistedTestCase.tearDown(self)

    def test_main_initialization(self):
        """test that creating a Main instance works as expected."""
        main = self.build_main()
        main.shutdown()

    def test_main_start(self):
        """Test that Main.start works."""
        main = self.build_main()
        main.start()
        main.shutdown()

    def test_main_restarts_on_critical_error(self):
        """Test that Main restarts when syncdaemon gets into UNKNOWN_ERROR."""
        self.restarted = False
        main = self.build_main()
        main.restart = lambda: setattr(self, 'restarted', True)
        main.start()
        main.event_q.push('SYS_UNKNOWN_ERROR')
        self.assertTrue(self.restarted)
        main.shutdown()

    def test_handshake_timeout(self):
        """Check connecting times out."""
        d0 = defer.Deferred()

        class Handler:
            """Trivial event handler."""
            def handle_SYS_HANDSHAKE_TIMEOUT(self):
                """Pass the test when we get this event."""
                reactor.callLater(0, d0.callback, None)

        main = self.build_main(handshake_timeout=.5)

        def fake_connect(*a):
            """Only connect when States told so."""
            main.event_q.push('SYS_CONNECTION_MADE')
            return defer.Deferred()
        main.action_q.connect = fake_connect

        # fake the following to not be executed
        main.get_root = lambda *_: defer.Deferred()
        main.action_q.check_version = lambda *_: defer.Deferred()

        main.event_q.subscribe(Handler())
        main.start()
        main.event_q.push('SYS_NET_CONNECTED')
        main.event_q.push('SYS_USER_CONNECT', '')
        d0.addCallback(lambda _: main.shutdown())
        return d0

    def test_create_dirs_already_exists_dirs(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        self.assertFalse(os.path.exists(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        self.assertTrue(os.path.exists(main.shares_dir_link))
        # check that the shares link is actually a link
        self.assertTrue(os.path.islink(main.shares_dir_link))
        self.assertEquals(link, main.shares_dir_link)
        main.shutdown()

    def test_create_dirs_already_exists_symlink_too(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        os.symlink(self.shares, link)
        self.assertTrue(os.path.exists(link))
        self.assertTrue(os.path.islink(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertTrue(os.path.islink(main.shares_dir_link))
        main.shutdown()

    def test_create_dirs_already_exists_but_not_symlink(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        os.makedirs(link)
        self.assertTrue(os.path.exists(link))
        self.assertFalse(os.path.islink(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertEquals(main.shares_dir_link, link)
        self.assertFalse(os.path.islink(main.shares_dir_link))
        main.shutdown()

    def test_create_dirs_none_exists(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        # remove the existing dirs
        os.rmdir(self.root)
        os.rmdir(self.shares)
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertTrue(os.path.exists(link))
        self.assertTrue(os.path.islink(main.shares_dir_link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main.shutdown()
