# -*- coding: utf-8 -*-
#
# Author: Rodney Dawes <rodney.dawes@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for the ubuntuone-login script """

import dbus.service
import new
import os

from contrib.testing.testcase import DBusTwistedTestCase, FakeLogin
from twisted.internet import defer
from twisted.python.failure import Failure
from ubuntuone.syncdaemon import dbus_interface

class InvalidSignalError(Exception):
    """Exception for when we get the wrong signal called."""
    pass

class LoginTests(DBusTwistedTestCase):
    """Basic tests for the ubuntuone-login script """

    _path = os.path.join(os.getcwd(), "bin", "ubuntuone-login")
    u1login = new.module('u1login')
    execfile(_path, u1login.__dict__)

    def setUp(self):
        DBusTwistedTestCase.setUp(self)
        self.oauth = FakeLogin(self.bus)
        self._old_path = dbus_interface.DBUS_PATH_AUTH
        dbus_interface.DBUS_PATH_AUTH = '/oauthdesktop'

    def tearDown(self):
        # collect all signal receivers registered during the test
        signal_receivers = set()
        with self.bus._signals_lock:
            for group in self.bus._signal_recipients_by_object_path.values():
                for matches in group.values():
                    for match in matches.values():
                        signal_receivers.update(match)
        d = self.cleanup_signal_receivers(signal_receivers)
        def shutdown(r):
            self.oauth.shutdown()
            dbus_interface.DBUS_PATH_AUTH = self._old_path
        d.addBoth(shutdown)
        d.addBoth(lambda _: DBusTwistedTestCase.tearDown(self))
        return d

    def test_new_credentials(self):
        """ Test logging in """
        def new_creds(realm=None, consumer_key=None, sender=None):
            """ Override the callback """
            d.callback(True)

        def auth_denied():
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def got_oauth_error(message=None):
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def set_up_desktopcouch_pairing(consumer_key):
            """ Override the method """
            return

        def main():
            """ Override LoginMain.main """
            return

        login = self.u1login.LoginMain()
        login.main = main
        login.new_credentials = new_creds
        login.auth_denied = auth_denied
        login.got_oauth_error = got_oauth_error
        login.set_up_desktopcouch_pairing = set_up_desktopcouch_pairing

        login._connect_dbus_signals()

        client = self.bus.get_object(self.u1login.DBUS_IFACE_AUTH_NAME,
                                     '/oauthdesktop',
                                     follow_name_owner_changes=True)
        d = defer.Deferred()

        def login_handler():
            """ login handler """
            return

        iface = dbus.Interface(client, self.u1login.DBUS_IFACE_AUTH_NAME)
        iface.login('http://localhost', self.u1login.OAUTH_CONSUMER,
                    reply_handler=login_handler,
                    error_handler=self.error_handler)
        return d

    def test_auth_denied(self):
        """ Test that denying authorization works correctly. """

        def new_creds(realm=None, consumer_key=None, sender=None):
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def auth_denied():
            """ Override the callback """
            d.callback(True)

        def got_oauth_error(message=None):
            """ override the callback """
            d.errback(Failure(InvalidSignalError()))

        login = self.u1login.LoginMain()
        login.main = self.main
        login.new_credentials = new_creds
        login.auth_denied = auth_denied
        login.got_oauth_error = got_oauth_error

        login._connect_dbus_signals()

        self.oauth.processor.next_login_with(self.oauth.processor.got_denial)

        client = self.bus.get_object(self.u1login.DBUS_IFACE_AUTH_NAME,
                                     '/oauthdesktop',
                                     follow_name_owner_changes=True)
        d = defer.Deferred()

        def login_handler():
            """ login handler """
            return

        iface = dbus.Interface(client, self.u1login.DBUS_IFACE_AUTH_NAME)
        iface.login('http://localhost', self.u1login.OAUTH_CONSUMER,
                    reply_handler=login_handler,
                    error_handler=self.error_handler)
        return d

    def test_oauth_error(self):
        """ Test that getting an error works correctly. """

        def new_creds(realm=None, consumer_key=None, sender=None):
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def auth_denied():
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def got_oauth_error(message=None):
            """ override the callback """
            d.callback(True)

        login = self.u1login.LoginMain()
        login.main = self.main
        login.new_credentials = new_creds
        login.auth_denied = auth_denied
        login.got_oauth_error = got_oauth_error

        login._connect_dbus_signals()

        self.oauth.processor.next_login_with(self.oauth.processor.got_error,
                                             ('error!',))

        client = self.bus.get_object(self.u1login.DBUS_IFACE_AUTH_NAME,
                                     '/oauthdesktop',
                                     follow_name_owner_changes=True)
        d = defer.Deferred()

        def login_handler():
            """ login handler """
            return

        iface = dbus.Interface(client, self.u1login.DBUS_IFACE_AUTH_NAME)
        iface.login('http://localhost', self.u1login.OAUTH_CONSUMER,
                    reply_handler=login_handler,
                    error_handler=self.error_handler)
        return d
