#
# Author: Facundo Batista <facundo@canonical.com>
# Author: Natalia Bidart <natalia.bidart@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Local Re-scanner."""

from __future__ import with_statement

import unittest
import os
import shutil

from twisted.internet import defer, reactor

from contrib.testing import testcase
from ubuntuone.syncdaemon.local_rescan import LocalRescan
from ubuntuone.syncdaemon import (
    event_queue, filesystem_manager, volume_manager
)
from ubuntuone.storageprotocol import (
    hash as storage_hash, volumes
)


class FakeEQ(object):
    """Fake EQ"""
    def __init__(self):
        self.pushed = []

    def push(self, event, *args):
        """Store stuff as pushed."""
        self.pushed.append((event,) + args)

    def _fake(self, *a):
        """fake"""
    inotify_add_watch = inotify_has_watch = freeze_rollback = is_frozen = _fake
    inotify_rm_watch = freeze_begin = add_to_mute_filter = _fake

    def freeze_commit(self, events):
        """just store events"""
        self.pushed.extend(events)


class FakeAQ(object):
    """Fake AQ"""
    def __init__(self):
        self.unlinked = []

    def unlink(self, share_id, parent_id, node_id):
        """Store stuff as unlinked."""
        self.unlinked.append((share_id, parent_id, node_id))


class BaseTestCase(testcase.BaseTwistedTestCase):
    """ Base test case """

    def setUp(self):
        """ Setup the test """
        testcase.BaseTwistedTestCase.setUp(self)
        self.home_dir = self.mktemp('ubuntuonehacker')
        self.shares_dir = self.mktemp('shares')
        usrdir = self.mktemp("usrdir")
        self.fsmdir = self.mktemp("fsmdir")
        self.vm = testcase.FakeVolumeManager(usrdir)
        self.partials_dir = self.mktemp("partials")
        self.fsm = filesystem_manager.FileSystemManager(self.fsmdir,
                                                        self.partials_dir,
                                                        self.vm)
        self.fsm.create(usrdir, "")
        self.eq = FakeEQ()
        self.fsm.register_eq(self.eq)
        self.aq = FakeAQ()

    def tearDown(self):
        self.rmtree(self.tmpdir)
        testcase.BaseTwistedTestCase.tearDown(self)

    @staticmethod
    def create_share(share_id, share_name, fsm, shares_dir,
                     access_level='Modify'):
        """ creates a share """
        share_path = os.path.join(shares_dir, share_name)
        os.makedirs(share_path)
        share = volume_manager.Share(path=share_path, volume_id=share_id,
                                     access_level=access_level)
        fsm.vm.add_share(share)
        return share

    def create_udf(self, udf_id, node_id, suggested_path, path, subscribed):
        """Create an UDF and add it to the volume manager."""
        volume = volumes.UDFVolume(udf_id, node_id, suggested_path)
        udf = volume_manager.UDF.from_udf_volume(volume, path)
        udf.subscribed = subscribed
        self.fsm.vm.add_udf(udf)
        return udf

    def create_node(self, path, is_dir, real=True, which_share=None):
        """Creates a node, really (maybe) and in the metadata."""
        if which_share is None:
            which_share = self.share
        filepath = os.path.join(which_share.path, path)
        if real:
            if is_dir:
                os.mkdir(filepath)
            else:
                open(filepath, "w").close()

        self.fsm.create(filepath, which_share.volume_id, is_dir=is_dir)
        self.fsm.set_node_id(filepath, "uuid"+path)

        # for files we put hashes to signal them not non-content
        if not is_dir:
            self.fsm.set_by_path(filepath, local_hash="h", server_hash="h")
        return filepath


class CollectionTests(BaseTestCase):
    """Test to check how LocalRescan gathers the dirs to scan."""

    def test_init(self):
        """Test the init params."""
        self.assertRaises(TypeError, LocalRescan, 1)
        self.assertRaises(TypeError, LocalRescan, 1, 2)
        self.assertRaises(TypeError, LocalRescan, 1, 2, 3)
        self.assertRaises(TypeError, LocalRescan, 1, 2, 3, 4, 5)

    def test_empty_ro(self):
        """Test with one empty View share."""
        # create the share

        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # pylint: disable-msg=W0212
        toscan = []
        def f():
            """helper"""
            toscan.extend(x[1] for x in lr._queue)
            return defer.Deferred()

        lr._queue_scan = f
        lr.start()
        self.assertEqual(toscan, [self.vm.root.path])

    def test_empty_rw(self):
        """Test with one empty Modify share."""
        # create the share
        share = self.create_share('share_id', 'ro_share',  self.fsm,
                                       self.shares_dir, access_level='Modify')
        self.fsm.create(share.path, "share_id", is_dir=True)
        self.fsm.set_node_id(share.path, "uuid")

        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # pylint: disable-msg=W0212
        toscan = []
        def f():
            """helper"""
            toscan.extend(x[1] for x in lr._queue)
            return defer.Deferred()

        lr._queue_scan = f
        lr.start()
        self.assertEqual(sorted(toscan), [share.path, self.vm.root.path])

    def test_not_empty_rw(self):
        """Test with a Modify share with info."""
        # create the share
        share = self.create_share('share_id', 'ro_share',  self.fsm,
                                       self.shares_dir, access_level='Modify')
        self.fsm.create(share.path, "share_id", is_dir=True)
        self.fsm.set_node_id(share.path, "uuid1")

        # create a node in the share
        filepath = os.path.join(share.path, "a")
        self.fsm.create(filepath, "share_id")
        self.fsm.set_node_id(filepath, "uuid2")
        open(filepath, "w").close()

        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # pylint: disable-msg=W0212
        toscan = []
        def f():
            """helper"""
            toscan.extend(x[1] for x in lr._queue)
            return defer.Deferred()

        lr._queue_scan = f
        lr.start()
        self.assertEqual(sorted(toscan), [share.path, self.vm.root.path])


class VolumeTestCase(BaseTestCase):
    """Test how LocalRescan manages volumes."""

    def setUp(self):
        """Init."""
        BaseTestCase.setUp(self)

        self.env_var = 'HOME'
        self.old_value = os.environ.get(self.env_var, None)
        os.environ[self.env_var] = self.home_dir

        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        self.volumes = []
        self.expected = [self.vm.root.path] # root volume has to be scanned
        paths = ['~/Documents', '~/PDFs', '~/yadda/yadda/doo']
        for i, suggested_path in enumerate(paths):
            # create UDF
            path = os.path.expanduser(suggested_path).encode("utf8")
            os.makedirs(path)
            udf_id, node_id = 'udf_id%i' % i, 'node_id%i' % i
            udf = self.create_udf(udf_id, node_id, suggested_path, path, True)
            self.volumes.append(udf)

            # make FSM aware of it
            self.fsm.create(udf.path, udf_id, is_dir=True)
            self.fsm.set_node_id(udf.path, node_id)

            # one more to assert over
            self.expected.append(path)

        self.expected.sort()

    def tearDown(self):
        """Cleanup."""
        self.lr = None
        self.expected = []

        if self.old_value is None:
            os.environ.pop(self.env_var)
        else:
            os.environ[self.env_var] = self.old_value

        BaseTestCase.tearDown(self)

    def test_start_with_udf(self):
        """LR start() on an udf."""
        # pylint: disable-msg=W0212
        toscan = []
        def f():
            """helper"""
            for _, path, _, _, udfmode in self.lr._queue:
                toscan.append(path)
                self.assertFalse(udfmode)
            return defer.Deferred()

        self.lr._queue_scan = f
        self.lr.start()
        self.assertEqual(self.expected, sorted(toscan))

    def test_scan_with_udf_normal(self):
        """LR scan_dir() on an udf, normal."""
        # pylint: disable-msg=W0212
        udf_path = self.expected[0]
        def f():
            """helper"""
            self.assertTrue(len(self.lr._queue), 1)
            _, path, _, _, udfmode = self.lr._queue[0]
            self.assertEqual(path, udf_path)
            self.assertFalse(udfmode)
            return defer.Deferred()

        self.lr._queue_scan = f
        self.lr.scan_dir("mdid", udf_path)

    def test_scan_with_udf_udfmode(self):
        """LR scan_dir() on an udf, udf mode."""
        # pylint: disable-msg=W0212
        udf_path = self.expected[0]
        def f():
            """helper"""
            self.assertTrue(len(self.lr._queue), 1)
            _, path, _, _, udfmode = self.lr._queue[0]
            self.assertEqual(path, udf_path)
            self.assertTrue(udfmode)
            return defer.Deferred()

        self.lr._queue_scan = f
        self.lr.scan_dir("mdid", udf_path, udfmode=True)

    def __test_scan_dir_with_udf(self):
        udf = self.volumes[0]
        mdobj = self.fsm.get_by_path(udf.path)
        d = self.lr.scan_dir(mdobj.mdid, udf.path)
        # scan_dir would fail if volumes are not included
        self.assertTrue(isinstance(d, defer.Deferred))
        return d

    def test_start_without_udf_itself(self):
        """LR start() having removed UDFs."""
        vol_to_unsub = self.volumes[0]
        vol_to_keep = self.volumes[1:]
        shutil.rmtree(vol_to_unsub.path)
        d = self.lr.start()

        def check(_):
            """Removed UDF should be desubscribed."""
            # these should remain ok
            for vol in vol_to_keep:
                self.assertTrue(vol.subscribed)
            # this should be unsubscribed
            self.assertFalse(vol_to_unsub.subscribed)

        d.addCallback(check)
        return d

    def test_start_without_udf_ancestors(self):
        """LR start() having removed UDFs parents."""
        vol_to_unsub = self.volumes[-1]  # grab the one that has lot of parents
        vol_to_keep = self.volumes[:-1]
        shutil.rmtree(os.path.dirname(vol_to_unsub.path))
        d = self.lr.start()

        def check(_):
            """Removed UDF should be desubscribed."""
            # these should remain ok
            for vol in vol_to_keep:
                self.assertTrue(vol.subscribed)
            # this should be unsubscribed
            self.assertFalse(vol_to_unsub.subscribed)

        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_start_with_udf_unsubscribed(self):
        """LR start() having removed UDFs."""
        unsub_vol = self.volumes.pop(0)
        path_idx = self.expected.index(unsub_vol.path)
        unsub_path = self.expected.pop(path_idx)
        assert unsub_vol.path == unsub_path
        self.fsm.vm.unsubscribe_udf(unsub_vol.volume_id)
        toscan = []
        def f():
            """helper"""
            for _, path, _, _, udfmode in self.lr._queue:
                toscan.append(path)
            return defer.succeed(None)

        self.lr._queue_scan = f
        yield self.lr.start()
        self.assertEqual(self.expected, sorted(toscan))


class TwistedBase(BaseTestCase):
    """Base class for twisted tests."""

    timeout = 2

    def setUp(self):
        """set up the test."""
        BaseTestCase.setUp(self)
        self.deferred = defer.Deferred()
        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # create a share
        self.share = self.create_share('share_id', 'ro_share',  self.fsm,
                                       self.shares_dir, access_level='Modify')
        self.fsm.create(self.share.path, "share_id", is_dir=True)
        self.fsm.set_node_id(self.share.path, "uuidshare")

    def startTest(self, check_function):
        """Start the test using lr.start()."""
        self.deferred.addCallback(lambda x: self.lr.start())
        self.deferred.addCallback(check_function)
        self.deferred.callback(None)

    @defer.inlineCallbacks
    def scanTest(self, check_function):
        """Start the test using lr.scan_dir()."""
        yield lambda x: self.lr.scan_dir("mdid", self.udf_path, udfmode=True)
        yield check_function
        self.deferred.callback(None)


class ComparationTests(TwistedBase):
    """Test LocalRescan checking differences between disk and metadata."""

    timeout = 20

    def setUp(self):
        TwistedBase.setUp(self)

        # create an udf
        udf_path = os.path.join(self.home_dir, "myudf")
        os.mkdir(udf_path)
        self.udf = self.create_udf('udf_id', 'udf_root_node_id', "~/myudf",
                                   udf_path, True)
        self.fsm.create(self.udf.path, 'udf_id', is_dir=True)
        self.fsm.set_node_id(self.udf.path, 'udf_root_node_id')

    def test_empty(self):
        """Test with an empty share."""
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])

        self.startTest(check)
        return self.deferred

    def test_equals(self):
        """Test with a share with the same files as metadata."""
        # create a node in the share
        self.create_node("a", is_dir=False)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])

        self.startTest(check)
        return self.deferred

    def test_disc_more_file_empty_normal(self):
        """Test having an empty file more in disc than in MD, normal volume."""
        # create a node in the share
        self.create_node("a", is_dir=False)

        # and another file in disk
        path = os.path.join(self.share.path, "b")
        open(path, "w").close()

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)])

        self.startTest(check)
        return self.deferred

    def test_disc_more_file_empty_udf(self):
        """Test having an empty file more in disc than in MD, udf mode."""
        # create a node in the share
        self.create_node("a", is_dir=False, which_share=self.udf)

        # and another file in disk
        path = os.path.join(self.share.path, "b")
        open(path, "w").close()

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)])

        self.scanTest(check)
        return self.deferred

    def test_disc_more_file_content(self):
        """Test having a file (with content) more in disc than in metadata."""
        # create a node in the share
        self.create_node("a", is_dir=False)

        # and another file in disk
        path = os.path.join(self.share.path, "b")
        with open(path, "w") as fh:
            fh.write("foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)])

        self.startTest(check)
        return self.deferred

    def test_disc_symlink(self):
        """Test having a symlink in disc."""
        # create a node in the share
        source = self.create_node("a", is_dir=False)

        # and a symlink to ignore!
        symlpath = os.path.join(self.share.path, "b")
        os.symlink(source, symlpath)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])

        self.startTest(check)
        return self.deferred

    def test_disc_more_dir_normal(self):
        """Test having a dir more in disc than in metadata, normal volume."""
        # create a node in the share
        self.create_node("a", is_dir=False)

        # and another dir in disk
        otherpath = os.path.join(self.share.path, "b")
        os.mkdir(otherpath)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_DIR_CREATE', otherpath)])

        self.startTest(check)
        return self.deferred

    def test_disc_more_dir_udf(self):
        """Test having a dir more in disc than in metadata, udf_mode."""
        # create a node in the share
        self.create_node("a", is_dir=False, which_share=self.udf)

        # and another dir in disk
        otherpath = os.path.join(self.share.path, "b")
        os.mkdir(otherpath)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_DIR_CREATE', otherpath)])

        self.scanTest(check)
        return self.deferred

    def test_disc_less_file_normal(self):
        """Test having less in disc than in metadata, normal volume."""
        # create a node in the share, but no in disk
        filepath = self.create_node("a", is_dir=False, real=False)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_DELETE', filepath)])

        self.startTest(check)
        return self.deferred

    def test_disc_less_file_udf(self):
        """Test having less in disc than in metadata, udf mode."""
        # create a node in the share, but no in disk
        filepath = self.create_node("a", is_dir=False, real=False,
                                    which_share=self.udf)
        assert self.fsm.has_metadata(path=filepath)
        parentpath = os.path.dirname(filepath)
        self.fsm.set_by_path(parentpath, local_hash="foo", server_hash="foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=filepath))
            parent = self.fsm.get_by_path(parentpath)
            self.assertEqual(parent.local_hash, "")
            self.assertEqual(parent.server_hash, "")

        self.scanTest(check)
        return self.deferred

    def test_no_file_no_hash(self):
        """Test useless metadata."""
        path = os.path.join(self.share.path, "b")
        self.fsm.create(path, self.share.volume_id)
        self.assertTrue(self.fsm.has_metadata(path=path))

        def check(_):
            """Check."""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_disc_less_dir_normal(self):
        """Test having less in disc than in metadata, normal volume."""
        # create a node in the share, but no in disk
        filepath = self.create_node("a", is_dir=True, real=False)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_DIR_DELETE', filepath)])

        self.startTest(check)
        return self.deferred

    def test_disc_less_dir_udf(self):
        """Test having less in disc than in metadata, udf mode."""
        # create a node in the share, but no in disk
        filepath = self.create_node("a", is_dir=True, real=False,
                                    which_share=self.udf)
        assert self.fsm.has_metadata(path=filepath)
        parentpath = os.path.dirname(filepath)
        self.fsm.set_by_path(parentpath, local_hash="foo", server_hash="foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=filepath))
            parent = self.fsm.get_by_path(parentpath)
            self.assertEqual(parent.local_hash, "")
            self.assertEqual(parent.server_hash, "")

        self.scanTest(check)
        return self.deferred

    def test_differenttype_dir(self):
        """Test that it should be a dir but now it's a file."""
        # create one type in the share, other in disk
        thispath = self.create_node("a", is_dir=True, real=False)
        open(thispath, "w").close()

        def check(_):
            """checks all is ok."""
            # don't sort, as the DELETE must come before the CREATE
            events = self.eq.pushed
            self.assertEqual(events[0], ('FS_DIR_DELETE', thispath))
            self.assertEqual(events[1], ('FS_FILE_CREATE', thispath))
            self.assertEqual(events[2], ('FS_FILE_CLOSE_WRITE', thispath))

        self.startTest(check)
        return self.deferred

    def test_differenttype_file(self):
        """Test that it should be a file but now it's a dir."""
        # create one type in the share, other in disk
        thispath = self.create_node("a", is_dir=False, real=False)
        os.mkdir(thispath)

        def check(_):
            """checks all is ok."""
            # don't sort, as the DELETE must come before the CREATE
            events = self.eq.pushed
            self.assertEqual(events[0], ('FS_FILE_DELETE', thispath))
            self.assertEqual(events[1], ('FS_DIR_CREATE', thispath))

        self.startTest(check)
        return self.deferred

    def test_complex_scenario(self):
        """Several dirs, several files, some differences."""
        self.create_node("a", is_dir=True)
        self.create_node("a/b", is_dir=True)
        sh1 = self.create_node("a/b/e", is_dir=True, real=False)
        self.create_node("a/c", is_dir=True)
        self.create_node("a/c/d", is_dir=False)
        sh2 = self.create_node("a/c/e", is_dir=False, real=False)
        self.create_node("d", is_dir=True)
        sh3 = self.create_node("e", is_dir=False, real=False)
        sh4 = self.create_node("f", is_dir=True, real=False)
        sh5 = os.path.join(self.share.path, "j")
        open(sh5, "w").close()
        sh6 = os.path.join(self.share.path, "a", "c", "q")
        open(sh6, "w").close()
        sh7 = os.path.join(self.share.path, "k")
        os.mkdir(sh7)
        sh8 = os.path.join(self.share.path, "a", "p")
        os.mkdir(sh8)

        # scan!
        def check(_):
            """check"""
            self.assertEqual(sorted(self.eq.pushed), [
                ('FS_DIR_CREATE', sh8),
                ('FS_DIR_CREATE', sh7),
                ('FS_DIR_DELETE', sh1),
                ('FS_DIR_DELETE', sh4),
                ('FS_FILE_CLOSE_WRITE', sh6),
                ('FS_FILE_CLOSE_WRITE', sh5),
                ('FS_FILE_CREATE', sh6),
                ('FS_FILE_CREATE', sh5),
                ('FS_FILE_DELETE', sh2),
                ('FS_FILE_DELETE', sh3),
            ])

        self.startTest(check)
        return self.deferred

    def test_deep_and_wide(self):
        """Lot of files in a dir, and lots of dirs."""
        # almost all known, to force the system to go deep
        dirs = "abcdefghijklmnopq" * 20
        for i in range(1, len(dirs)+1):
            dirpath = os.path.join(*dirs[:i])
            self.create_node(dirpath, is_dir=True)
        basedir = os.path.join(*dirs)
        self.create_node(os.path.join(basedir, "file1"), is_dir=False)
        path = os.path.join(basedir, "file2")
        sh1 = self.create_node(path, is_dir=False, real=False)

        # some files in some dirs
        files = "rstuvwxyz"
        for f in files:
            path = os.path.join(*dirs[:3]+f)
            self.create_node(path, is_dir=False)
            path = os.path.join(*dirs[:6]+f)
            self.create_node(path, is_dir=False)
        sh2 = os.path.join(self.share.path, *dirs[:6]+"q")
        open(sh2, "w").close()

        def check(_):
            """check"""
            self.assertEqual(sorted(self.eq.pushed), [
                ('FS_FILE_CLOSE_WRITE', sh2),
                ('FS_FILE_CREATE', sh2),
                ('FS_FILE_DELETE', sh1),
            ])

        # scan!
        self.startTest(check)
        return self.deferred

    def test_subtree_removal_normal(self):
        """A whole subtree was removed, normal volume."""
        self.create_node("a", is_dir=True)
        sh1 = self.create_node("a/b", is_dir=True)
        sh2 = self.create_node("a/b/c", is_dir=True)
        sh3 = self.create_node("a/b/c/d", is_dir=False)

        # remove the whole subtree
        shutil.rmtree(sh1)

        # scan!
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [
                ('FS_FILE_DELETE', sh3),
                ('FS_DIR_DELETE', sh2),
                ('FS_DIR_DELETE', sh1),
            ])

        self.startTest(check)
        return self.deferred

    def test_subtree_removal_udf(self):
        """A whole subtree was removed, udf mode."""
        self.create_node("a", is_dir=True, which_share=self.udf)
        sh1 = self.create_node("a/b", is_dir=True, which_share=self.udf)
        sh2 = self.create_node("a/b/c", is_dir=True, which_share=self.udf)
        sh3 = self.create_node("a/b/c/d", is_dir=False, which_share=self.udf)

        # remove the whole subtree
        parentpath = os.path.dirname(sh1)
        self.fsm.set_by_path(parentpath, local_hash="foo", server_hash="foo")
        shutil.rmtree(sh1)

        # scan!
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=sh1))
            self.assertFalse(self.fsm.has_metadata(path=sh2))
            self.assertFalse(self.fsm.has_metadata(path=sh3))
            parent = self.fsm.get_by_path(parentpath)
            self.assertEqual(parent.local_hash, "")
            self.assertEqual(parent.server_hash, "")

        self.scanTest(check)
        return self.deferred

    def test_one_dir_only(self):
        """Specific subtree only."""
        self.create_node("a", is_dir=True)
        self.create_node("a/b", is_dir=True)

        # one in both, one only in share, one only in disk
        self.create_node("a/b/c", is_dir=True)
        sh1 = self.create_node("a/b/d", is_dir=True, real=False)
        sh2 = os.path.join(self.share.path, "a", "b", "e")
        open(sh2, "w").close()

        # more differences, but not in dir to check
        self.create_node("a/c", is_dir=False)
        os.mkdir(os.path.join(self.share.path, "a", "k"))

        # scan!
        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)
        toscan = os.path.join(self.share.path, "a", "b")

        def check(_):
            """checks all is ok."""
            self.assertEqual(len(self.eq.pushed), 3)
            events = sorted(self.eq.pushed)
            self.assertEqual(events[0], ('FS_DIR_DELETE', sh1))
            self.assertEqual(events[1], ('FS_FILE_CLOSE_WRITE', sh2))
            self.assertEqual(events[2], ('FS_FILE_CREATE', sh2))

        self.deferred.addCallback(lambda _: lr.scan_dir("mdid", toscan))
        self.deferred.addCallback(check)
        self.deferred.callback(None)
        return self.deferred

    def test_one_nonexistant_dir(self):
        """Specific subtree for a dir that's not in a share or not at all."""
        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # real dir, but not in share
        self.assertRaises(ValueError, lr.scan_dir, "mdid", "/tmp")

        # no dir at all
        self.assertRaises(ValueError, lr.scan_dir, "mdid", "no-dir-at-all")

        # inside a share, but no real dir
        # this does not generate a direct error, but sends an event
        nodir = os.path.join(self.share.path, "no-dir-at-all")
        lr.scan_dir("mdid", nodir)

        # inside a share, and real, but no really a dir
        nodir = self.create_node("a", is_dir=False)
        self.assertRaises(ValueError, lr.scan_dir, "mdid", nodir)

        # need to wait the generated event before finishing the test
        reactor.callLater(.2, self.deferred.callback, None)
        return self.deferred

    def test_one_dir_ro_share(self):
        """The dir is in a share that's RO, no error but no action."""
        # create the share
        share = self.create_share('share_id', 'ro_share2',  self.fsm,
                                       self.shares_dir, access_level='View')
        self.fsm.create(share.path, "share_id", is_dir=True)
        self.fsm.set_node_id(share.path, "uuidshare")

        lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)
        lr.scan_dir("mdid", share.path)

    def test_content_changed(self):
        """Test that it detects the content change."""
        # create one type in the share, other in disk
        path = self.create_node("a", is_dir=False)
        with open(path, "w") as fh:
            fh.write("foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CLOSE_WRITE', path)])

        self.startTest(check)
        return self.deferred

    def test_inode_changed(self):
        """Test that it detects a change using the filedate."""
        # two files with same dates
        pathx = os.path.join(self.share.path, "x")
        pathy = os.path.join(self.share.path, "y")
        open(pathx, "w").close()
        open(pathy, "w").close()
        self.create_node("x", is_dir=False)

        # move the second into the first one
        os.rename(pathy, pathx)

        # a & m times will be the same, but not the inode or change time
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CLOSE_WRITE', pathx)])

        self.startTest(check)
        return self.deferred

    def test_scandir_no_dir_normal(self):
        """Attempt to scan a dir that is not there."""
        nodir = os.path.join(self.share.path, "no-dir-at-all")
        self.lr.scan_dir("mdid", nodir)

        # scan!
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [
                ('LR_SCAN_ERROR', "mdid", False),
            ])

        self.deferred.addCallback(check)
        # trigger the control later, as it the scan error es slightly delayed
        reactor.callLater(.2, self.deferred.callback, None)
        return self.deferred

    def test_scandir_no_dir_udfmode(self):
        """Attempt to scan a dir that is not there."""
        nodir = os.path.join(self.share.path, "no-dir-at-all")
        self.lr.scan_dir("mdid", nodir, udfmode=True)

        # scan!
        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [
                ('LR_SCAN_ERROR', "mdid", True),
            ])

        self.deferred.addCallback(check)
        # trigger the control later, as it the scan error es slightly delayed
        reactor.callLater(.2, self.deferred.callback, None)
        return self.deferred

    def test_no_read_perms_file(self):
        """Test with a file that we can't read"""
        # and another file in disk
        path = os.path.join(self.share.path, "b")
        open(path, "w").close()
        os.chmod(path, 0000)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])

        self.startTest(check)
        return self.deferred

    def test_no_read_perms_dir(self):
        """Test with a dir that we can't read"""
        # and another file in disk
        path = os.path.join(self.share.path, "b")
        os.makedirs(path)
        os.chmod(path, 0000)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [])

        self.startTest(check)
        return self.deferred


class InotifyTests(TwistedBase):
    """Test LocalRescan pushing events to the EventQueue."""
    timeout = 2

    def setUp(self):
        """Setup the test."""
        TwistedBase.setUp(self)
        self.eq = event_queue.EventQueue(self.fsm)
        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)
        self.real_os_stat = os.stat

    def tearDown(self):
        os.stat = self.real_os_stat
        self.eq.shutdown()
        TwistedBase.tearDown(self)

    def test_man_in_the_middle(self):
        """Intercept normal work and change the disk."""
        for c in "abcdefghijk":
            self.create_node(c, is_dir=False)

        # remove a couple, create some new
        sh1 = os.path.join(self.share.path, "d")
        os.remove(sh1)
        sh2 = os.path.join(self.share.path, "f")
        os.remove(sh2)
        sh3 = os.path.join(self.share.path, "jj")
        open(sh3, "w").close()
        sh4 = os.path.join(self.share.path, "kk")
        os.mkdir(sh4)

        # this sh5 will be written in the middle of the scan
        sh5 = os.path.join(self.share.path, "zz")

        should_receive_events = [
            ('FS_DIR_CREATE', sh4),
            ('FS_FILE_CLOSE_WRITE', sh3),
            ('FS_FILE_CLOSE_WRITE', sh5),
            ('FS_FILE_CREATE', sh3),
            ('FS_FILE_CREATE', sh5),
            ('FS_FILE_DELETE', sh1),
            ('FS_FILE_DELETE', sh2),
        ]

        # helper class, pylint: disable-msg=C0111
        class HitMe(object):
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself):
                innerself.hist = []
            def handle_default(innerself, event_name, path):
                innerself.hist.append((event_name, path))
        hm = HitMe()
        self.eq.subscribe(hm)

        # we need to intercept compare, as the stat interception needs
        # to be done after compare() starts.
        # use os.stat to get in the middle of the process, to put some
        # dirt in the testing scenario

        # pylint: disable-msg=W0212
        real_compare = self.lr._compare

        def middle_compare(*a1, **k1):
            """Changes os.stat."""
            self.lr._compare = real_compare

            def middle_stat(*a2, **k2):
                """Dirt!"""
                os.stat = self.real_os_stat
                open(sh5, "w").close()
                reactor.iterate(.1)
                return self.real_os_stat(*a2, **k2)

            os.stat = middle_stat
            return real_compare(*a1, **k1)

        self.lr._compare = middle_compare

        def check(_):
            """check"""
            self.assertEqual(sorted(hm.hist), should_receive_events)

        self.deferred.addCallback(lambda x: self.lr.start())
        self.deferred.addCallback(check)
        self.deferred.callback(None)
        return self.deferred

class QueuingTests(BaseTestCase):
    """Test that simultaneus calls are queued."""
    timeout = 2

    def setUp(self):
        """set up the test."""
        BaseTestCase.setUp(self)
        self.deferred = defer.Deferred()
        self.eq = event_queue.EventQueue(self.fsm)
        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        # create two shares
        self.share1 = self.create_share('share_id1', 'ro_share_1',  self.fsm,
                                       self.shares_dir, access_level='Modify')
        self.fsm.create(self.share1.path, "share_id1", is_dir=True)
        self.fsm.set_node_id(self.share1.path, "uuidshare1")
        self.share2 = self.create_share('share_id2', 'ro_share_2',  self.fsm,
                                       self.shares_dir, access_level='Modify')
        self.fsm.create(self.share2.path, "share_id2", is_dir=True)
        self.fsm.set_node_id(self.share2.path, "uuidshare2")

        self.real_os_stat = os.stat

    def tearDown(self):
        os.stat = self.real_os_stat
        self.eq.shutdown()
        BaseTestCase.tearDown(self)

    def test_intercept_generate_second(self):
        """Intercept the first work and generate a second scan."""
        # fill and alter first share
        for c in "abcdefgh":
            self.create_node(c, is_dir=False, which_share=self.share1)
        sh1 = os.path.join(self.share1.path, "d")
        os.remove(sh1)
        sh2 = os.path.join(self.share1.path, "jj")
        open(sh2, "w").close()

        # fill and alter second share
        for c in "abcdefgh":
            self.create_node(c, is_dir=False, which_share=self.share2)
        sh3 = os.path.join(self.share2.path, "e")
        os.remove(sh3)
        sh4 = os.path.join(self.share2.path, "kk")
        open(sh4, "w").close()

        should_receive_events = [
            ('FS_FILE_CLOSE_WRITE', sh2),
            ('FS_FILE_CLOSE_WRITE', sh4),
            ('FS_FILE_CREATE', sh2),
            ('FS_FILE_CREATE', sh4),
            ('FS_FILE_DELETE', sh1),
            ('FS_FILE_DELETE', sh3),
        ]

        # helper class, pylint: disable-msg=C0111
        class HitMe(object):
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself):
                innerself.hist = []
            def handle_default(innerself, event_name, path):
                innerself.hist.append((event_name, path))
        hm = HitMe()
        self.eq.subscribe(hm)

        # we need to intercept compare, as the stat interception needs
        # to be done after compare() starts.
        # use os.stat to get in the middle of the process, to put some
        # dirt in the testing scenario

        # pylint: disable-msg=W0212
        real_compare = self.lr._compare

        def middle_compare(*a1, **k1):
            """Changes os.stat."""
            self.lr._compare = real_compare

            def middle_stat(*a2, **k2):
                """Dirt!"""
                os.stat = self.real_os_stat
                self.lr.scan_dir("mdid", self.share2.path)
                return self.real_os_stat(*a2, **k2)

            os.stat = middle_stat
            return real_compare(*a1, **k1)

        self.lr._compare = middle_compare

        def check(_):
            """check"""
            self.assertEqual(sorted(hm.hist), should_receive_events)

        self.deferred.addCallback(lambda _:
                                  self.lr.scan_dir("mdid", self.share1.path))
        self.deferred.addCallback(check)
        self.deferred.callback(None)
        return self.deferred


class PushTests(TwistedBase):
    """Test LocalRescan pushing events to the EventQueue."""
    timeout = 2

    def setUp(self):
        """Setup the test."""
        TwistedBase.setUp(self)
        self.eq = event_queue.EventQueue(self.fsm)
        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

    def tearDown(self):
        """Cleanup the test."""
        self.eq.shutdown()
        TwistedBase.tearDown(self)

    def test_one_dir_create(self):
        """Check that an example dir create is really pushed."""
        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_DIR_CREATE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event
        filepath = os.path.join(self.share.path, "a")
        os.mkdir(filepath)

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(filepath, result[0])

        self.startTest(check)
        return self.deferred

    def test_one_file_create(self):
        """Check that an example file create is really pushed."""
        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_FILE_CREATE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event
        filepath = os.path.join(self.share.path, "a")
        open(filepath, "w").close()

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(filepath, result[0])

        self.startTest(check)
        return self.deferred

    def test_one_dir_delete(self):
        """Check that an example dir delete is really pushed."""

        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_DIR_DELETE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event
        filepath = os.path.join(self.share.path, "a")
        self.fsm.create(filepath, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(filepath, "uuid1")

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(filepath, result[0])

        self.startTest(check)
        return self.deferred

    def test_one_file_delete(self):
        """Check that an example file delete is really pushed."""
        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_FILE_DELETE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event
        filepath = os.path.join(self.share.path, "a")
        self.fsm.create(filepath, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(filepath, "uuid1")
        self.fsm.set_by_path(filepath, local_hash="hash", server_hash="hash")

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(filepath, result[0])

        self.startTest(check)
        return self.deferred

    def test_file_changed(self):
        """Check that an example close write is pushed."""
        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_FILE_CLOSE_WRITE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event
        filepath = os.path.join(self.share.path, "a")
        self.fsm.create(filepath, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(filepath, "uuid1")
        with open(filepath, "w") as fh:
            fh.write("foo")

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(filepath, result[0])

        self.startTest(check)
        return self.deferred

    def test_file_changed_in_nestedstruct(self):
        """Check that an example close write is pushed."""
        # helper class, pylint: disable-msg=C0111
        result = []
        class Listener(object):
            def handle_FS_FILE_CLOSE_WRITE(self, path):
                result.append(path)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # create nested struct
        path = os.path.join(self.share.path, "a")
        self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid1")
        os.mkdir(path)
        path = os.path.join(self.share.path, "a", "b")
        self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid2")
        os.mkdir(path)
        path = os.path.join(self.share.path, "a", "b", "c")
        self.fsm.create(path, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path, "uuid3")
        open(path, "w").close()

        # push some event
        with open(path, "w") as fh:
            fh.write("foo")

        def check(_):
            """ asserts that GoodListener was called. """
            self.assertEquals(1, len(result))
            self.assertEquals(path, result[0])

        self.startTest(check)
        return self.deferred

    def test_conflict_file(self):
        """Found a .conflict file."""
        # helper class, pylint: disable-msg=C0111
        class Listener(object):
            def __init__(self):
                self.hit = False
            def handle_default(self, *args):
                self.hit = True

        listener = Listener()
        self.eq.subscribe(listener)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(listener)
        self.addCleanup(cleanup)

        # push some event
        path = os.path.join(self.share.path, "foobar.u1conflict")
        open(path, "w").close()

        def check(_):
            """checks no messages and file still there"""
            self.assertFalse(listener.hit)
            self.assertTrue(os.path.exists(path))

        self.startTest(check)
        return self.deferred

    def test_conflict_dir(self):
        """Found a .conflict dir."""
        # helper class, pylint: disable-msg=C0111
        class Listener(object):
            def __init__(self):
                self.hit = False
            def handle_default(self, *args):
                self.hit = True

        listener = Listener()
        self.eq.subscribe(listener)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(listener)
        self.addCleanup(cleanup)

        # push some event
        path = os.path.join(self.share.path, "foobar.u1conflict")
        os.mkdir(path)

        def check(_):
            """checks no messages and file still there"""
            self.assertFalse(listener.hit)
            self.assertTrue(os.path.exists(path))

        self.startTest(check)
        return self.deferred


class BadStateTests(TwistedBase):
    """Test what happens with those files left in a bad state last time."""

    def _hash(self, path):
        """Hashes a file."""
        hasher = storage_hash.content_hash_factory()
        with open(path) as fh:
            while True:
                cont = fh.read(65536)
                if not cont:
                    break
                hasher.update(cont)
        return hasher.content_hash()

    def test_no_uuid_empty(self):
        """Found an empty file that does not have uuid yet."""
        path = os.path.join(self.share.path, "a")
        self.fsm.create(path, self.share.volume_id, is_dir=False)
        open(path, "w").close()

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)])
            self.assertEqual(self.fsm.has_metadata(path=path), False)

        self.startTest(check)
        return self.deferred

    def test_no_uuid_all_family(self):
        """Fast creation paths: all without uuid."""
        path1 = os.path.join(self.share.path, "a")
        self.fsm.create(path1, self.share.volume_id, is_dir=True)
        os.mkdir(path1)
        path2 = os.path.join(self.share.path, "a", "b")
        self.fsm.create(path2, self.share.volume_id, is_dir=False)
        open(path2, "w").close()

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path2),
                                              ('FS_FILE_CLOSE_WRITE', path2)])
            self.assertEqual(self.fsm.has_metadata(path=path1), True)
            self.assertEqual(self.fsm.has_metadata(path=path2), False)

        self.deferred.addCallback(lambda _: self.lr.scan_dir("mdid", path1))
        self.deferred.addCallback(check)
        self.deferred.callback(None)
        return self.deferred

    def test_no_uuid_content(self):
        """Found a non empty file that does not have uuid yet."""
        path = os.path.join(self.share.path, "a")
        self.fsm.create(path, self.share.volume_id, is_dir=False)
        with open(path, "w") as fh:
            fh.write("foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)])
            self.assertEqual(self.fsm.has_metadata(path=path), False)

        self.startTest(check)
        return self.deferred

    def test_LOCAL(self):
        """We were uploading the file, but it was interrupted."""
        # create the file in metadata and real
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path, "uuid")
        with open(path, "w") as fh:
            fh.write("foo")

        # start a put file, and assume we got interrupted (no server hash)
        pathhash = self._hash(path)
        self.fsm.set_by_mdid(mdid, local_hash=pathhash)
        self.fsm.refresh_stat(path)

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CLOSE_WRITE', path)])

        self.startTest(check)
        return self.deferred

    def test_SERVER_file_empty(self):
        """We were downloading the file, but it was interrupted."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        open(path, "w").close()
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path))

        def check(_):
            """arrange the metadata so later server_rescan will do ok"""
            mdobj = self.fsm.get_by_mdid(mdid)
            self.assertFalse(mdobj.info.is_partial)
            self.assertEqual(mdobj.server_hash, mdobj.local_hash)
            self.assertFalse(os.path.exists(partial_path))

        self.startTest(check)
        return self.deferred

    def test_SERVER_no_file(self):
        """We just queued the Download, and it was interrupted."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False, node_id="uuid")
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))

        # this mimic Sync.get_file
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)

        # now, for some reason, we lose the partial file
        os.remove(partial_path)

        def check(_):
            """arrange the metadata so later server_rescan will do ok"""
            self.assertEqual(self.eq.pushed, [])
            mdobj = self.fsm.get_by_mdid(mdid)
            self.assertFalse(mdobj.info.is_partial)
            self.assertEqual(mdobj.server_hash, mdobj.local_hash)

        self.startTest(check)
        return self.deferred

    def test_SERVER_file_content(self):
        """We were downloading the file, but it was interrupted, and changed"""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path))

        # also change the original file
        with open(path, "w") as fh:
            fh.write("I see dead people")

        def check(_):
            """arrange the metadata so later server_rescan will do ok"""
            mdobj = self.fsm.get_by_mdid(mdid)
            self.assertFalse(mdobj.info.is_partial)
            self.assertEqual(mdobj.server_hash, mdobj.local_hash)
            self.assertFalse(os.path.exists(partial_path))
            self.assertEqual(self.eq.pushed, [('FS_FILE_CLOSE_WRITE', path)])

        self.startTest(check)
        return self.deferred

    def test_SERVER_dir(self):
        """We were downloading the dir, but it was interrupted."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=True)
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path))

        # also put a file inside the directory, to check that LR enters in it
        filepath = os.path.join(path, "file")
        open(filepath, "w").close()

        def check(_):
            """arrange the metadata so later server_rescan will do ok"""
            mdobj = self.fsm.get_by_mdid(mdid)
            self.assertFalse(mdobj.info.is_partial)
            self.assertEqual(mdobj.server_hash, mdobj.local_hash)
            self.assertFalse(os.path.exists(partial_path))
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', filepath),
                                              ('FS_FILE_CLOSE_WRITE', filepath)
                                             ])

        self.startTest(check)
        return self.deferred

    def test_partial_nomd(self):
        """Found a .partial with no metadata at all."""
        path = os.path.join(self.fsm.partials_dir, 'anduuid' + ".u1partial.foo")
        open(path, "w").close()

        def check(_):
            """checks everything's ok"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(os.path.exists(path), path)

        self.startTest(check)
        return self.deferred

    def test_directory_bad_changed(self):
        """Found a dir with 'changed' not SERVER nor NONE."""
        path1 = os.path.join(self.share.path, "onedir")
        self.fsm.create(path1, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path1, "uuid1")
        os.mkdir(path1)
        self.fsm.set_by_path(path1, server_hash="foo")

        # create a child to it
        path2 = os.path.join(path1, "file_inside")
        self.fsm.create(path2, self.share.volume_id)
        self.fsm.set_node_id(path2, "uuid2")
        open(path2, "w").close()

        def check(_):
            """move to conflict, remove MD"""
            self.assertFalse(os.path.exists(path1))
            self.assertTrue(os.path.exists(path1 + ".u1conflict"))
            self.assertFalse(self.fsm.has_metadata(path=path1))

            # check MD is gone also for children
            self.assertFalse(self.fsm.has_metadata(path=path2),
                             "no metadata for %r" % path2)

        self.startTest(check)
        return self.deferred

    def test_file_partial_dir(self):
        """Found a .partial of a file, but MD says it's a dir."""
        # create the dir in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        fh.close()
        partial_path = os.path.join(self.fsm.partials_dir,
                                    mdid + ".u1partial.a")
        self.assertTrue(os.path.exists(partial_path))


        # now change the dir for a file, for LR to find it
        os.rmdir(path)
        open(path, "w").close()

        def check(_):
            """remove everything"""
            self.assertFalse(os.path.exists(path))
            self.assertFalse(os.path.exists(partial_path), partial_path)
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_file_noSERVER(self):
        """We were downloading the file, it was interrupted, but no SERVER."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        open(path, "w").close()
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path))

        # trick it to not return "SERVER"
        real_mdobj = self.fsm.fs[mdid]
        real_mdobj["info"]["is_partial"] = False
        self.fsm.fs[mdid] = real_mdobj

        def check(_):
            """remove everything"""
            self.assertFalse(os.path.exists(path), path)
            self.assertFalse(os.path.exists(partial_path), partial_path)
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_no_file_broken_metadata(self):
        """Found broken metadata but no file."""
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path, "uuid")

        # break the metadata
        real_mdobj = self.fsm.fs[mdid]
        real_mdobj["info"]["is_partial"] = True
        self.fsm.fs[mdid] = real_mdobj

        def check(_):
            """checks everything's ok"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_no_dir_broken_metadata_deep(self):
        """Found broken metadata but no dir."""
        path1 = os.path.join(self.share.path, "a")
        mdid1 = self.fsm.create(path1, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path1, "uuid1")

        path2 = os.path.join(self.share.path, "a", "b")
        mdid2 = self.fsm.create(path2, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path2, "uuid2")

        # break both metadatas
        real_mdobj = self.fsm.fs[mdid1]
        real_mdobj["info"]["is_partial"] = True
        self.fsm.fs[mdid1] = real_mdobj

        real_mdobj = self.fsm.fs[mdid2]
        real_mdobj["info"]["is_partial"] = True
        self.fsm.fs[mdid2] = real_mdobj

        def check(_):
            """checks everything's ok"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path1))
            self.assertFalse(self.fsm.has_metadata(path=path2))

        self.startTest(check)
        return self.deferred

    def test_no_dir_broken_metadata(self):
        """Found broken metadata but no dir."""
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid")

        # break the metadata
        real_mdobj = self.fsm.fs[mdid]
        real_mdobj["info"]["is_partial"] = True
        self.fsm.fs[mdid] = real_mdobj

        def check(_):
            """checks everything's ok"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_no_dir_LOCAL_metadata(self):
        """Found metadata with 'changed' in LOCAL but no dir."""
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid")

        # break the metadata
        real_mdobj = self.fsm.fs[mdid]
        real_mdobj["info"]["is_partial"] = False
        real_mdobj["server_hash"] = "different-than-local"
        self.fsm.fs[mdid] = real_mdobj

        def check(_):
            """checks everything's ok"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_just_created_no_real_hash(self):
        """Found a file that has content, but MD doesn't know it."""
        # this is a case when a file is written with content, the make_file
        # to the server was ok (we have the file uuid), but before
        # HQ finishes everything is stopped; when started again, it needs
        # to generate the corresponding events, to start the hash&upload
        # process again
        path = os.path.join(self.share.path, "a")
        self.fsm.create(path, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path, "uuid")
        self.fsm.set_by_path(path, local_hash="", server_hash="")
        with open(path, "w") as fh:
            fh.write("foo")

        def check(_):
            """check"""
            self.assertEqual(self.eq.pushed, [('FS_FILE_CREATE', path),
                                              ('FS_FILE_CLOSE_WRITE', path)
                                             ])

        self.startTest(check)
        return self.deferred

    def test_notcontent_file(self):
        """The file is created but never started to download."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        # open(path, "w").close()
        self.fsm.create(path, self.share.volume_id, is_dir=False, node_id="1")

        def check(_):
            """No event, and no MD"""
            self.assertEqual(self.eq.pushed, [])
            self.assertFalse(self.fsm.has_metadata(path=path))

        self.startTest(check)
        return self.deferred

    def test_remove_only_partial_of_childs(self):
        """We were downloading the file, but it was interrupted.
        and we have a partial of another fiel in a subdir."""
        # create the file in metadata
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=False)
        partial_path = os.path.join(self.fsm.partials_dir,
                                mdid + ".u1partial." + os.path.basename(path))
        self.fsm.set_node_id(path, "uuid")

        # start the download, never complete it
        self.fsm.set_by_mdid(mdid, server_hash="blah-hash-blah")
        self.fsm.create_partial("uuid", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path))

        # create another file with a partial, but one level deeper
        dir = os.path.join(self.share.path, "dir")
        os.mkdir(dir)
        path_b = os.path.join(self.share.path, "dir", "b")
        open(path_b, "w").close()
        self.fsm.create(dir, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(dir, "uuid2")

        mdid_b = self.fsm.create(path_b, self.share.volume_id, is_dir=False)
        partial_path_b = os.path.join(self.fsm.partials_dir,
                                mdid_b + ".u1partial." + os.path.basename(path_b))
        self.fsm.set_node_id(path_b, "uuid1")


        # start the download, never complete it
        self.fsm.set_by_mdid(mdid_b, server_hash="blah-hash-blah-b")
        self.fsm.create_partial("uuid1", self.share.volume_id)
        fh = self.fsm.get_partial_for_writing("uuid1", self.share.volume_id)
        fh.write("foobar")
        self.assertTrue(os.path.exists(partial_path_b))

        def check(_):
            """arrange the metadata so later server_rescan will do ok"""
            mdobj = self.fsm.get_by_mdid(mdid)
            self.assertFalse(mdobj.info.is_partial)
            self.assertEqual(mdobj.server_hash, mdobj.local_hash)
            self.assertFalse(os.path.exists(partial_path))
            self.assertFalse(os.path.exists(partial_path_b))
            self.assertTrue(os.path.exists(path_b))

        self.startTest(check)
        return self.deferred


class TrashTests(TwistedBase):
    """Test handling trash."""

    def test_nothing(self):
        """No trash."""
        def check(_):
            """check"""
            self.assertEqual(self.aq.unlinked, [])

        self.startTest(check)
        return self.deferred

    def test_one(self):
        """Something in the trash."""
        path = os.path.join(self.share.path, "a")
        mdid = self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid")
        self.fsm.delete_to_trash(mdid, "parent_id")

        def check(_):
            """check"""
            self.assertEqual(self.aq.unlinked,
                             [(self.share.volume_id, "parent_id", "uuid")])

        self.startTest(check)
        return self.deferred

    def test_two(self):
        """Two nodes (file and dir) in the trash."""
        path = os.path.join(self.share.path, "a")
        mdid1 = self.fsm.create(path, self.share.volume_id, is_dir=True)
        self.fsm.set_node_id(path, "uuid1")
        self.fsm.delete_to_trash(mdid1, "parent_id")

        path = os.path.join(self.share.path, "b")
        mdid2 = self.fsm.create(path, self.share.volume_id, is_dir=False)
        self.fsm.set_node_id(path, "uuid2")
        self.fsm.delete_to_trash(mdid2, "parent_id")

        def check(_):
            """check"""
            self.assertEqual(sorted(self.aq.unlinked),
                             [(self.share.volume_id, "parent_id", "uuid1"),
                              (self.share.volume_id, "parent_id", "uuid2")])

        self.startTest(check)
        return self.deferred


class ParentWatchForUDFTestCase(BaseTestCase):
    """Tests over watches for UDF's parent dir."""

    timeout = 2

    def setUp(self):
        """Init."""
        BaseTestCase.setUp(self)
        self._deferred = defer.Deferred()
        self.eq = event_queue.EventQueue(self.fsm)
        self.original_add = self.eq.inotify_add_watch
        self.original_has = self.eq.inotify_has_watch
        self.watches = []
        self.eq.inotify_add_watch = lambda path: self.watches.append(path)
        self.eq.inotify_has_watch = lambda path: path in self.watches

        self.lr = LocalRescan(self.vm, self.fsm, self.eq, self.aq)

        self.env_var = 'HOME'
        self.old_value = os.environ.get(self.env_var, None)
        os.environ[self.env_var] = self.home_dir

        # create UDF
        suggested_path = u'~/Documents/Reading/Books/PDFs'
        path = os.path.expanduser(suggested_path).encode("utf8")
        os.makedirs(path)
        udf_id, node_id = 'udf_id', 'node_id'
        self.udf = self.create_udf(udf_id, node_id, suggested_path, path, True)
        self.ancestors = self.udf.ancestors # need a fake HOME

        # make FSM aware of it
        self.fsm.create(self.udf.path, udf_id, is_dir=True)
        self.fsm.set_node_id(self.udf.path, node_id)

    def tearDown(self):
        """Cleanup."""
        self.eq.inotify_add_watch = self.original_add
        self.eq.inotify_has_watch = self.original_has

        self.eq.shutdown()
        self.eq = None

        if self.old_value is None:
            os.environ.pop(self.env_var)
        else:
            os.environ[self.env_var] = self.old_value

        BaseTestCase.tearDown(self)

    def test_ancestors_have_watch(self):
        """UDF's ancestors have a watch."""

        def ancestors_have_watch(_):
            """Check."""
            expected = set(self.ancestors)
            actual = set(self.watches)
            if expected.issubset(actual):
                # make the test finish ok
                self._deferred.callback(True)
            else:
                difference = expected.symmetric_difference(actual)
                msg = 'Expected (%s)\n\n' \
                      'Is not subset of real watches (%s).\n\n' \
                      'Set symmetric difference is: %s.' % \
                      (expected, actual, difference)
                self._deferred.errback(msg)

        d = self.lr.start()
        d.addCallback(ancestors_have_watch)
        return self._deferred

    def test_watch_is_not_added_if_present(self):
        """Watches are not added if present."""

        for path in self.ancestors:
            self.eq.inotify_add_watch(path)

        def no_watch_added(_):
            """Check."""
            for path in self.udf.ancestors:
                self.assertEquals(1, self.watches.count(path))

        d = self.lr.start()
        d.addCallback(no_watch_added)
        d.addCallback(lambda _:
            self._deferred.callback(None))
        return self._deferred


def test_suite():
    # pylint: disable-msg=C0111
    return unittest.TestLoader().loadTestsFromName(__name__)

if __name__ == "__main__":
    unittest.main()
