# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for the DBUS interface """

import inspect
import os
import unittest
import uuid

from ubuntuone.storageprotocol.sharersp import (
    NotifyShareHolder,
)
from ubuntuone.syncdaemon import dbus_interface
from ubuntuone.syncdaemon.dbus_interface import (
    DBUS_IFACE_STATUS_NAME,
    DBUS_IFACE_EVENTS_NAME,
    DBUS_IFACE_FS_NAME,
    DBUS_IFACE_SYNC_NAME,
    DBUS_IFACE_SHARES_NAME,
    DBUS_IFACE_CONFIG_NAME,
    DBUS_IFACE_FOLDERS_NAME,
    DBUS_IFACE_PUBLIC_FILES_NAME,
    EventListener,
)
from ubuntuone.syncdaemon.volume_manager import Share, Shared, UDF
from ubuntuone.syncdaemon.tools import DBusClient
from ubuntuone.syncdaemon import event_queue, states, main, config
from contrib.testing.testcase import (
    DBusTwistedTestCase,
    FakeLogin,
)
from twisted.internet import defer, reactor
from ubuntuone.syncdaemon.marker import MDMarker


class FakeCommand(object):
    """A fake command."""

    def __init__(self, share_id, node_id):
        self.share_id = share_id
        self.node_id = node_id

    def is_runnable(self):
        """Always runnable."""
        return True

    def run(self):
        """Just succeed."""
        return defer.succeed(None)

    def __str__(self):
        name = self.__class__.__name__
        tmpl = name + "(" + "share_id=%s, node_id=%s" + ")"
        return tmpl % (self.share_id,
                       self.node_id)


class DBusInterfaceTests(DBusTwistedTestCase):
    """ Basic tests to the objects exposed with D-Bus"""

    def test_status(self):
        """ Test the Status Object, registering it to the session bus, and
        calling current_uploads method.
        """
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def waiting_handler(result):
            """ waiting handler """
            self.assertEquals(result, [])
            d.callback(True)

        def status_handler(result):
            """ reply handler """
            state = states.StateManager.QUEUE_MANAGER
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            client.call_method('waiting_content',
                               reply_handler=waiting_handler,
                               error_handler=self.error_handler)

        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            client.call_method('current_status',
                               reply_handler=status_handler,
                               error_handler=self.error_handler)

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            client.call_method('current_downloads',
                               reply_handler=downloads_handler,
                               error_handler=self.error_handler)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_syncDaemon(self):
        """ Test the SyncDaemon object, registering it to the session bus, and
        calling connect and disconnect methods.
        """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def disconnect_handler(result):
            """ reply handler """
            self.assertEquals(None, result)
            d.callback(True)

        def connect_handler(result):
            """ reply handler """
            self.assertEquals(None, result)
            client.call_method("disconnect", reply_handler=disconnect_handler,
                               error_handler=self.error_handler)

        client.call_method("connect", reply_handler=connect_handler,
                           error_handler=self.error_handler)

        return d

    def test_filesystem(self):
        """ Test the FileSystem Object, registering it to the session bus, and
        excercising the API.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """ reply handler """
            d.callback(metadata)

        def callback(result):
            """ callback to check the result. """
            self.assertEquals(path, str(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_push_event(self):
        """ Test the exposed method: push_event """
        push_deferred = defer.Deferred()
        queue_deferred = defer.Deferred()
        push_deferred.addCallback(lambda _: queue_deferred)
        class Listener(object):
            """ A basic listener to handle the pushed event. """

            def handle_FS_FILE_CREATE(self, path):
                """ FS_FILE_CREATE handling """
                self.assertEquals('bar', path)

        listener = Listener()
        self.event_q.subscribe(listener)
        self.event_q.unsubscribe(self.dbus_iface.event_listener)

        def empty_queue_callback():
            """
            Called when the event queue empties
            """
            queue_deferred.callback(True)

        def default(path):
            """ default callback to assert the pushed event. """
            self.event_q.unsubscribe(listener)
            self.event_q.subscribe(self.dbus_iface.event_listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_callback)

        queue_deferred.addCallback(default)

        client = DBusClient(self.bus, '/events',
                   DBUS_IFACE_EVENTS_NAME)
        client.call_method(
            'push_event', 'FS_FILE_CREATE', ('bar',),
            reply_handler=lambda *_: push_deferred.callback(True),
            error_handler=self.error_handler)
        self.event_q.add_empty_event_queue_callback(empty_queue_callback)
        return push_deferred

    def test_waiting_content(self):
        """
        Test Status.waiting_content and Status.schedule_next with fake
        data in the AQ
        """
        # prepare the VM so it lies for us
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        a_path = os.path.join(share_path, "path_a")
        b_path = os.path.join(share_path, "path_b")
        c_path = os.path.join(share_path, "path_c")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.create(b_path, "share_id")
        self.fs_manager.create(c_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id_a")
        self.fs_manager.set_node_id(b_path, "node_id_b")
        self.fs_manager.set_node_id(c_path, "node_id_c")
        # inject the fake data
        self.action_q.content_queue._head = FakeCommand("share_id",
                                                        "node_id_a")
        self.action_q.content_queue.waiting.extend([
                FakeCommand("share_id", "node_id_b"),
                FakeCommand("share_id", "node_id_c")])
        # OK, testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        self.second = False
        def waiting_handler(result):
            """ waiting_content reply handler """
            self.assertEquals(3, len(result))
            # the second time we're called, the result should be reversed
            if self.second:
                node_a, node_c, node_b = result
            else:
                node_a, node_b, node_c = result

            self.assertEquals(a_path, str(node_a['path']))
            self.assertEquals(b_path, str(node_b['path']))
            self.assertEquals('share_id', str(node_a['share']))
            self.assertEquals('share_id', str(node_b['share']))
            self.assertEquals('node_id_a', str(node_a['node']))
            self.assertEquals('node_id_b', str(node_b['node']))

            if self.second:
                # OK, we're done
                d.callback(True)
            else:
                # the last shall be the first, et cetera.
                client.call_method('schedule_next', 'share_id', 'node_id_c',
                                   reply_handler=reschedule_handler,
                                   error_handler=self.error_handler)
        def reschedule_handler(result):
            """ schedule_next reply handler """
            self.assertEqual(result, None)
            self.second = True
            client.call_method('waiting_content',
                               reply_handler=waiting_handler,
                               error_handler=self.error_handler)
        client.call_method('waiting_content',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        return d

    def test_waiting_metadata(self):
        """Test Status.waiting_metadata with fake data in the AQ."""
        # inject the fake data
        self.action_q.meta_queue._head = FakeCommand("n_a_foo", "n_a_bar")
        self.action_q.meta_queue.waiting.extend([
                FakeCommand("n_b_foo", "n_b_bar"),
                FakeCommand("n_c_foo", "n_c_bar")])

        # testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def waiting_handler(result):
            """Reply handler."""
            self.assertEquals(3, len(result))
            node_a, node_b, node_c = result
            self.assertEquals(str(FakeCommand("n_a_foo", "n_a_bar")), node_a)
            self.assertEquals(str(FakeCommand("n_b_foo", "n_b_bar")), node_b)
            self.assertEquals(str(FakeCommand("n_c_foo", "n_c_bar")), node_c)
            d.callback(True)
        client.call_method('waiting_metadata',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        return d

    def test_contq_changed(self):
        """Test the Status.ContentQueueChanged signal."""
        # prepare the VM so it lies for us
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        a_path = os.path.join(share_path, "path_a")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id")

        # testing time
        d = defer.Deferred()
        def content_queue_change_helper(result):
            """Signal handler, test checker."""
            head_path = result['head']['path']
            assert head_path == '' or head_path.endswith('/path_a')
            assert result['head']['command'] == 'FakeCommand'
            assert result.get('FakeCommand') in (None, {'size': '0',
                                                        'count': '1'})
            d.callback(None)

        match = self.bus.add_signal_receiver(content_queue_change_helper,
                                             'ContentQueueChanged')
        self.signal_receivers.add(match)
        # inject the fake data
        self.action_q.content_queue.queue(FakeCommand("share_id", "node_id"))
        return d

    def test_current_downloads(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict(
            deflated_size=10, size=100, n_bytes_read=1)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertEquals('10', str(result[0]['deflated_size']))
            self.assertEquals('1', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict(
            deflated_size=100, n_bytes_written=10)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertEquals('100', str(result[0]['deflated_size']))
            self.assertEquals('10', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads_with_marker(self):
        """ Test Status.current_uploads with fake data in the AQ """
        self.action_q.uploading[('', MDMarker('a_marker_uuid'))] = dict(
            deflated_size=100, n_bytes_written=10)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_two_current_downloads(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict(
            deflated_size=10, size=100, n_bytes_read=8)
        share1_path = os.path.join(self.shares_dir, 'share1')
        self.main.vm.add_share(Share(path=share1_path, volume_id='share_id_1'))
        down_path_1 = os.path.join(share1_path, "down_path_1")
        self.fs_manager.create(down_path_1, "share_id_1")
        self.fs_manager.set_node_id(down_path_1, "node_id_1")
        self.action_q.downloading[('share_id_1', 'node_id_1')] = dict(
            deflated_size=10, size=100, n_bytes_read=5)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(2, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertEquals('10', str(result[0]['deflated_size']))
            self.assertEquals('8', str(result[0]['n_bytes_read']))
            self.assertEquals(down_path_1, str(result[1]['path']))
            self.assertEquals('10', str(result[1]['deflated_size']))
            self.assertEquals('5', str(result[1]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_two_current_uploads(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict(
            deflated_size=100, n_bytes_written=10)
        share1_path = os.path.join(self.shares_dir, 'share1')
        self.main.vm.add_share(Share(path=share1_path, volume_id='share_id_1'))
        up_path_1 = os.path.join(share1_path, "up_path_1")
        self.fs_manager.create(up_path_1, "share_id_1")
        self.fs_manager.set_node_id(up_path_1, "node_id_1")
        self.action_q.uploading[('share_id_1', 'node_id_1')] = dict(
            deflated_size=80, n_bytes_written=20)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(2, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertEquals('100', str(result[0]['deflated_size']))
            self.assertEquals('10', str(result[0]['n_bytes_written']))
            self.assertEquals(up_path_1, str(result[1]['path']))
            self.assertEquals('80', str(result[1]['deflated_size']))
            self.assertEquals('20', str(result[1]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_downloads_deflated_size_NA(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict()
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEquals('0', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads_deflated_size_NA(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict()
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEquals('0', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_nm_signals(self):
        """ Test that NM signals are received and handled properly. """
        result = None
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        # class-closure, cannot use self, pylint: disable-msg=E0213
        class Listener(object):
            def handle_SYS_NET_CONNECTED(innerself):
                self.nm.emit_disconnected()
            def handle_SYS_NET_DISCONNECTED(innerself):
                self.assertTrue(result)

        listener = Listener()
        self.event_q.subscribe(listener)

        def empty_queue_callback():
            d.callback(True)

        def callback(result):
            self.event_q.unsubscribe(listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_callback)
        d.addCallback(callback)

        self.nm.emit_connected()
        self.event_q.add_empty_event_queue_callback(empty_queue_callback)
        return d

    def test_get_shares(self):
        """ Test Shares.get_shares method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id',
                                     access_level='Read', accepted=False))
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def check(shares):
            """ handle get_shares reply """
            self.assertEquals(1, len(shares))
            for share in shares:
                if share['volume_id'] == '':
                    self.assertEquals('', str(share['volume_id']))
                    self.assertEquals(self.root_dir, str(share['path']))
                    self.assertEquals('Modify', str(share['access_level']))
                    self.assertEquals('False', str(share['accepted']))
                else:
                    self.assertEquals('share_id', str(share['volume_id']))
                    self.assertEquals(share_path, str(share['path']))
                    self.assertEquals('Read', str(share['access_level']))
                    self.assertEquals('False', str(share['accepted']))

        def shares_handler(shares):
            d.callback(shares)

        client.call_method('get_shares', reply_handler=shares_handler,
                           error_handler=self.error_handler)
        return d

    def test_accept_share(self):
        """ Test the accept_share method in dbus_interface.Share """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id',
                                     access_level='Read', accepted=False,
                                     node_id="node_id"))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('accept_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """the async checker"""
            self.assertEquals('Yes', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(True, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_reject_share(self):
        """ Test the reject_share method in dbus_interface.Share """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id',
                                     access_level='Read', accepted=False))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('reject_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """the async checker"""
            self.assertEquals('No', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(False, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_get_root(self):
        """ Check SycnDaemon.get_root exposed method """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """ handle get_rootdir reply """
            current_root = self.main.get_rootdir()
            self.assertEquals(current_root, str(result))
            d.callback(True)
        client.call_method('get_rootdir',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_create_share(self):
        """ Test share offering """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.main.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_a_dir', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, 'test_user',
                           'share_a_dir', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_create_shares(self):
        """Test share offering to multiple users at once"""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.main.root_dir, a_dir))
            self.assertEquals(u'test_user', username[:-1])
            self.assertEquals(u'share_a_dir', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_shares',
                           a_dir, ['test_user1', 'test_user2', 'test_user3'],
                           'share_a_dir', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_query_by_path(self):
        """ test that query_by_path method work as expected. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def query(items):
            self.assertEquals(1, len(items))
            self.assertEquals(('', 'node_id', ''), items[0])
        self.action_q.query = query
        client.call_method('query_by_path', a_dir,
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        return d

    def test_get_shared(self):
        """ Test Shares.get_shared API. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK', 'share_id', args[-1])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(results):
            """ handle get_shared reply """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals(a_dir, str(shared['path']))
            self.assertEquals('node_id', str(shared['node_id']))
            self.assertEquals('share_id', str(shared['volume_id']))
            self.assertEquals('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared_missing_path(self):
        """ Test Shares.get_shared, but without having the path. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK', 'share_id', args[-1])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        # remove the md of the subtree from fsm
        self.fs_manager.delete_file(a_dir)
        # set the path to None
        share = self.main.vm.shared['share_id']
        share.path = None
        self.main.vm.shared['share_id'] = share
        d = defer.Deferred()
        def reply_handler(results):
            """ handle get_shared reply """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals('', str(shared['path']))
            self.assertEquals('node_id', str(shared['node_id']))
            self.assertEquals('share_id', str(shared['volume_id']))
            self.assertEquals('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_refresh_shares(self):
        """ Just check that refresh_shares method API works. """
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """ handle get_rootdir reply """
            d.callback(True)
        client.call_method('refresh_shares',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_quit(self):
        """ test the quit exposed method. """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, we need to call quit but don't quit
        # pylint: disable-msg=C0111,W0601,W0602
        def fake_quit():
            pass
        self.main.quit = fake_quit
        def handler(result):
            d.callback(True)
        d.addCallback(self.assertTrue)
        client.call_method('quit',
                           reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_change_public_access(self):
        """Test the change_public_access exposed method."""
        client = DBusClient(
            self.bus, '/publicfiles', DBUS_IFACE_PUBLIC_FILES_NAME)
        d = defer.Deferred()
        def handler(result):
            d.callback(True)
        d.addCallback(self.assertTrue)
        share_id = str(uuid.uuid4())
        node_id = str(uuid.uuid4())
        client.call_method('change_public_access', share_id, node_id, True,
                           reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_get_public_files(self):
        """Test the get_public_files exposed method."""
        public_files = []
        expected = []
        udf_id = str(uuid.uuid4())
        udf_path = self.main.vm._build_udf_path('~/foo/bar')
        udf = UDF(str(udf_id), str('udf_node_id'), '~/foo/bar', udf_path, True)
        self.main.vm.udfs[udf_id] = udf
        for i in xrange(5):
            if i % 2:
                volume_id = udf_id
                path = os.path.join(udf.path, "foo_%d" % i)
            else:
                volume_id = ''
                path = os.path.join(self.root_dir, "foo_%d" % i)
            node_id = str(uuid.uuid4())
            public_url = 'http://example.com/%d' % i
            self.fs_manager.create(path, volume_id)
            self.fs_manager.set_node_id(path, node_id)
            public_files.append(dict(volume_id=volume_id, node_id=node_id,
                                     public_url=public_url))
            expected.append(dict(volume_id=volume_id, node_id=node_id,
                                     public_url=public_url, path=path))
        def fake_get_public_files():
            self.main.event_q.push('AQ_PUBLIC_FILES_LIST_OK', public_files)
        self.main.action_q.get_public_files = fake_get_public_files

        d = defer.Deferred()
        def public_files_list_handler(files):
            """Handler for PublicFilesList signal."""
            d.callback(files)
        match = self.bus.add_signal_receiver(public_files_list_handler,
                                             signal_name='PublicFilesList')
        self.signal_receivers.add(match)
        def check(files):
            self.assertEquals(len(public_files), len(files))
            self.assertEquals(expected, files)
        d.addCallback(check)
        client = DBusClient(
            self.bus, '/publicfiles', DBUS_IFACE_PUBLIC_FILES_NAME)
        client.call_method('get_public_files',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)

        return d


class DBusInterfaceUnicodeTests(DBusTwistedTestCase):
    """ Unicode variant of basic tests to the objects exposed with D-Bus"""

    def test_filesystem_unicode(self):
        """ Test the FileSystem Object, registering it to the session bus, and
        excercising the API.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """ reply handler """
            d.callback(metadata)

        def callback(result):
            """ callback to check the result. """
            self.assertEquals(path, unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_query_by_path_unicode(self):
        """ test that query_by_path method work as expected. """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def query(items):
            self.assertEquals(1, len(items))
            self.assertEquals(('', 'node_id', ''), items[0])
        self.action_q.query = query
        client.call_method('query_by_path', a_dir,
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        return d

    def test_create_share_unicode(self):
        """ Test share offering """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_ñoño', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, u'test_user',
                           u'share_ñoño', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared_unicode(self):
        """ test that list_shared method behaves with unicode data """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        share = Shared(path=a_dir, volume_id='shared_id', name=u'ñoño_shared',
                      access_level='View', other_username=u'test_username',
                      node_id='node_id')
        self.main.vm.add_shared(share)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()

        def check(results):
            """ check the returned values. """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals(a_dir, shared['path'].encode('utf-8'))
            self.assertEquals('node_id', str(shared['node_id']))
            self.assertEquals('shared_id', str(shared['volume_id']))
            self.assertEquals('View', str(shared['access_level']))

        d.addCallback(check)
        client.call_method('get_shared',
                           reply_handler=d.callback, error_handler=d.errback)
        return d


class DBusSignalTest(DBusTwistedTestCase):
    """ Test the DBus signals. """

    def test_event_signal(self):
        """ Test the Event signal. """
        client = DBusClient(self.bus, '/events', DBUS_IFACE_EVENTS_NAME)
        d = defer.Deferred()
        def handler(event_dict):
            """ signal handler """
            self.assertEquals({'path':'foo', 'name':'FS_FILE_CREATE'},
                              event_dict)
            d.callback(True)

        match = self.bus.add_signal_receiver(handler, signal_name='Event')
        self.signal_receivers.add(match)
        client.send_signal('Event', {'name':'FS_FILE_CREATE', 'path':'foo'})

        return d

    def test_download_started(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path):
            """ handler for DownloadStarted signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_STARTED', '', 'node_id', '')
        return d

    def test_download_finished(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_FINISHED', '', 'node_id', '')
        return d

    def test_download_error(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_ERROR',
                               '', 'node_id', '', error='AN_ERROR')
        return d

    def test_download_cancelled(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('CANCELLED', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_CANCELLED', '', 'node_id', 'hash')
        return d

    def test_upload_started(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path):
            """ handler for UploadStarted signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_STARTED', '', 'node_id', '')
        return d

    def test_upload_finished(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """ handler for UploadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_FINISHED', '', 'node_id', '')
        return d

    def test_upload_error(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """ handler for UploadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_ERROR',
                               '', 'node_id', 'AN_ERROR', 'hash')
        return d

    def test_status_changed(self):
        """Test the DBus signals in Status."""
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        def start(result):
            """Start the test."""
            match = self.bus.add_signal_receiver(status_handler,
                                     signal_name='StatusChanged')
            self.signal_receivers.add(match)
            bool_str = self.dbus_iface.status.bool_str
            state = states.StateManager.READY
            state_dict = {'name':state.name,
                          'description':state.description,
                          'is_error':bool_str(state.is_error),
                          'is_connected':bool_str(state.is_connected),
                          'is_online':bool_str(state.is_online)}
            client.send_signal('StatusChanged', state_dict)
        ready = self.main.wait_for_nirvana()
        ready.addCallback(start)
        d = defer.Deferred()
        def status_handler(result):
            """ handler for StatusChanged signal. """
            self.log.debug('status_handler, received: %r' % result)
            state = states.StateManager.READY
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            d.callback(True)

        return d

    def test_all_events_signal(self):
        """ Tests that we are listening all events """
        # configure DBusInterface to send a signal for each event
        self.dbus_iface.send_events = True
        # create the required items in the fs manager
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        a_path = os.path.join(share_path, "a_path")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id")
        handled_events = {}
        # filter only events we handle
        for (name, kind, _, _) in \
                                  inspect.classify_class_attrs(EventListener):
            if kind == 'method' and name.startswith('handle_') \
               and not name == 'handle_default':
                key = name.split('handle_')[1]
                handled_events[key] = event_queue.EVENTS[key]

        event_names = sorted(handled_events.keys())
        events_copy = handled_events.copy()
        d = defer.Deferred()
        def event_handler(event_dict):
            """ event handler that pop the received event from a list """
            event_args = events_copy.pop(str(event_dict.pop('event_name')))
            self.assertEquals(len(event_args), len(event_dict))
            if len(events_copy) == 0:
                cleanup(None)
                d.callback(True)

        def callback(result):
            """ default callback """
            return result

        def errback(error):
            """ error callback """
            missed_events = sorted(events_copy.keys())
            if missed_events:
                self.fail('Missed: ' + str(missed_events))
            else:
                return error

        d.addCallbacks(callback, errback)
        def cleanup(result):
            """ remove the signal receiver """
            self.event_q.subscribe(self.main.vm)
        d.addCallback(cleanup)
        # unsubscribe the vm subscribed in FakeMain as we are going to push
        # fake events
        self.event_q.unsubscribe(self.main.vm)
        match = self.bus.add_signal_receiver(event_handler, signal_name='Event')
        self.signal_receivers.add(match)
        for event_name in event_names:
            args = event_queue.EVENTS[event_name]
            self.event_q.push(event_name, *args)
        return d

    def test_invalid_filename(self):
        """Test the FS_INVALID_NAME signal."""
        d = defer.Deferred()

        def handler(dirname, filename):
            """Handler for InvalidName signal."""
            self.assertTrue(isinstance(dirname, unicode))
            self.assertTrue(isinstance(filename, str))
            d.callback(True)

        match = self.bus.add_signal_receiver(handler,
                                             signal_name='InvalidName',
                                             byte_arrays=True)
        self.signal_receivers.add(match)
        self.main.event_q.push('FS_INVALID_NAME', u'test/dir', 'testpath')
        return d

    def test_share_changed(self):
        """ Test the ShareChanged signal. """
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Write')

        self.main.vm.add_share(Share(path=share_path,
                                     volume_id=str(share_holder.share_id),
                                     node_id=str(share_holder.subtree),
                                     access_level='Read', accepted=False))
        d = defer.Deferred()
        def check(share):
            """ handler for ShareChanged signal. """
            self.assertEquals(str(share_holder.share_id), str(share['volume_id']))
            self.assertEquals(str(share_holder.subtree), str(share['node_id']))
            self.assertEquals(share_path, str(share['path']))
            self.assertEquals('Write', str(share['access_level']))
            self.assertEquals('', str(share['accepted']))
        d.addCallback(check)
        def share_handler(result):
            d.callback(result)
        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareChanged')
        self.signal_receivers.add(match)
        self.main.event_q.push('SV_SHARE_CHANGED', share_holder)
        return d

    def test_share_deleted(self):
        """ Test the ShareDeleted signal. """
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Read')

        self.main.vm.add_share(Share.from_notify_holder(share_holder, share_path))
        d = defer.Deferred()
        def share_handler(share_dict):
            """ handler for ShareDeletedsignal. """
            d.callback(share_dict)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareDeleted')
        self.signal_receivers.add(match)

        def check(share_dict):
            """Check the result."""
            expected_dict = dict(volume_id=str(share_holder.share_id),
                                 node_id=str(share_holder.subtree),
                                 name=u'fake_share',
                                 other_username=u'test_username',
                                 other_visible_name=u'visible_name',
                                 free_bytes='',
                                 path=share_path,
                                 accepted='',
                                 access_level='Read')
            expected_dict['type'] = 'Share'
            for k, v in share_dict.items():
                self.assertEquals(expected_dict[str(k)], str(v))
        d.addCallback(check)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareDeleted')
        self.signal_receivers.add(match)

        self.main.event_q.push('SV_SHARE_DELETED', share_holder.share_id)
        return d

    def test_share_created(self):
        """ Test the ShareCreated signal. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        share_id = uuid.uuid4()
        d = defer.Deferred()
        def share_handler(result):
            """ handler for ShareCreated signal. """
            self.assertEquals(str(share_id), str(result['volume_id']))
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreated')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_OK', share_id, marker)
        return d

    def test_share_created_error(self):
        """ Test the ShareCreated signal. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        error_msg = 'a error message'
        d = defer.Deferred()
        def share_handler(info, error):
            """ handler for ShareCreated signal. """
            self.assertEquals(str(marker), str(info['marker']))
            self.assertEquals(error, error_msg)
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreateError')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_ERROR', marker, error_msg)
        return d

    def test_signal_error(self):
        """ test sending SignalError. """
        d = defer.Deferred()
        def error_handler(type, args):
            """ Error signal handler. """
            self.assertEquals('node_id', args['node_id'].decode('utf-8'))
            self.assertEquals('', args['share_id'].decode('utf-8'))
            d.callback(True)

        error_match = self.bus.add_signal_receiver(error_handler,
                                     signal_name='SignalError',
                                     dbus_interface=DBUS_IFACE_SYNC_NAME)
        self.signal_receivers.add(error_match)
        # force a invalid AQ_UPLOAD_FINISHED
        self.main.event_q.push('AQ_UPLOAD_FINISHED', '', 'node_id', 'hash')
        return d

    def test_new_share(self):
        """Test the NewShare signal."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Read')
        share = Share.from_notify_holder(share_holder, share_path)
        # mark the share as accepted
        share.accepted = True
        d = defer.Deferred()
        def share_handler(share_dict):
            """Handler for NewShare signal."""
            d.callback(share_dict)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='NewShare')
        self.signal_receivers.add(match)

        def check(share_dict):
            """Check the result."""
            expected_dict = dict(volume_id=share.volume_id,
                                 node_id=share.node_id,
                                 name=u'fake_share',
                                 other_username=u'test_username',
                                 other_visible_name=u'visible_name',
                                 free_bytes='',
                                 path=share_path,
                                 accepted='True',
                                 access_level='Read')
            expected_dict['type'] = 'Share'
            for k, v in share_dict.items():
                self.assertEquals(expected_dict[str(k)], str(v))
        d.addCallback(check)
        self.main.vm.add_share(share)
        return d

    def test_public_access_changed(self):
        """Test the PublicAccessChanged signal."""
        d = defer.Deferred()
        def access_changed_handler(file_info):
            """Handler for PublicAccessChanged signal."""
            d.callback(file_info)

        match = self.bus.add_signal_receiver(access_changed_handler,
                                             signal_name='PublicAccessChanged')
        self.signal_receivers.add(match)

        share_id = "share"
        node_id = "node_id"
        is_public = True
        public_url = 'http://example.com'

        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(share_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check(file_info):
            """Check the result."""
            expected_dict = dict(share_id=str(share_id),
                                 node_id=str(node_id),
                                 is_public=str(is_public),
                                 public_url=public_url,
                                 path=path)
            self.assertEquals(expected_dict, file_info)
        d.addCallback(check)
        self.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_OK',
                          share_id=share_id, node_id=node_id,
                          is_public=is_public, public_url=public_url)
        return d

    def test_public_access_change_error(self):
        """Test the PublicAccessChangeError signal."""
        d = defer.Deferred()
        def access_change_error_handler(file_info, error):
            """Handler for PublicAccessChangeError signal."""
            d.callback((file_info, error))

        match = self.bus.add_signal_receiver(
            access_change_error_handler, signal_name='PublicAccessChangeError')
        self.signal_receivers.add(match)

        share_id = "share"
        node_id = "node_id"
        expected_error = 'error message'

        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(share_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check((file_info, error)):
            """Check the result."""
            expected_dict = dict(share_id=str(share_id),
                                 node_id=str(node_id),
                                 path=path)
            self.assertEquals(expected_dict, file_info)
            self.assertEquals(expected_error, error)
        d.addCallback(check)
        self.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_ERROR',
                          share_id=share_id, node_id=node_id,
                          error=expected_error)
        return d

    def test_root_mismatch(self):
        """Test RootMismatch signal."""
        d = defer.Deferred()
        def root_mismatch_handler(root_id, new_root_id):
            """Handler for RootMismatch signal."""
            d.callback((root_id, new_root_id))

        match = self.bus.add_signal_receiver(root_mismatch_handler,
                                             signal_name='RootMismatch')
        self.signal_receivers.add(match)

        def check((root_id, new_root_id)):
            """Check the result."""
            self.assertEquals('root_id', root_id)
            self.assertEquals('another_root_id', new_root_id)
        d.addCallback(check)
        self.event_q.push('SYS_ROOT_RECEIVED', 'root_id')
        self.event_q.push('SYS_ROOT_RECEIVED', 'another_root_id')
        return d

    def test_public_files_list(self):
        """Test the PublicAccessChanged signal."""
        d = defer.Deferred()
        def public_files_list_handler(files):
            """Handler for PublicFilesList signal."""
            d.callback(files)

        match = self.bus.add_signal_receiver(public_files_list_handler,
                                             signal_name='PublicFilesList')
        self.signal_receivers.add(match)

        volume_id = "share"
        node_id = "node_id"
        public_url = 'http://example.com'

        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id=volume_id))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(volume_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check(files):
            """Check the result."""
            expected_list = [dict(volume_id=str(volume_id),
                                 node_id=str(node_id),
                                 public_url=public_url,
                                 path=path)]
            self.assertEquals(expected_list, files)
        d.addCallback(check)
        self.event_q.push('AQ_PUBLIC_FILES_LIST_OK',
                          [dict(volume_id=volume_id, node_id=node_id,
                          public_url=public_url)])
        return d

    def test_public_files_list_error(self):
        """Test the PublicFilesListError signal."""
        d = defer.Deferred()
        def access_change_error_handler(error):
            """Handler for PublicFilesListError signal."""
            d.callback(error)

        match = self.bus.add_signal_receiver(
            access_change_error_handler, signal_name='PublicFilesListError')
        self.signal_receivers.add(match)

        expected_error = 'error message'

        def check(error):
            """Check the result."""
            self.assertEquals(expected_error, error)
        d.addCallback(check)
        self.event_q.push('AQ_PUBLIC_FILES_LIST_ERROR',
                          error=expected_error)
        return d


class TestDBusRestart(DBusTwistedTestCase):
    """
    Test main's restart method (and its interaction with dbus).
    """
    def test_restart(self):
        """
        start things up, then fire a restart, check it tries to
        restart.
        """
        d = defer.Deferred()
        def _handler(*a):
            """async helper"""
            d.callback(True)
        # shutdown will fail when trying to restart because of our
        # half-backed dbus. That's OK, we don't actually want it
        # restarted :)
        self.dbus_iface._restart_error_handler = d.callback
        try:
            self.main.restart()
        except SystemExit:
            pass
        return d


class ConfigTests(DBusTwistedTestCase):
    """ Basic tests to the Config object exposed via D-Bus"""

    def get_client(self):
        """returns a Config DBusClient"""
        return DBusClient(self.bus, '/config', DBUS_IFACE_CONFIG_NAME)

    def test_get_throttling_limits(self):
        """test get_throttling_limits exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        def reply_handler(result):
            """ handle the reply """
            self.assertEquals(aq.readLimit, result['download'])
            self.assertEquals(aq.writeLimit, result['upload'])
            self.assertEquals(100, result['download'])
            self.assertEquals(200, result['upload'])
            d.callback(True)
        def reply_handler_None(result):
            """ handle the reply """
            self.assertEquals(-1, result['download'])
            self.assertEquals(-1, result['upload'])
            aq.readLimit = 100
            aq.writeLimit = 200
            client.call_method('get_throttling_limits',
                               reply_handler=reply_handler,
                               error_handler=self.error_handler)
        client.call_method('get_throttling_limits',
                           reply_handler=reply_handler_None,
                           error_handler=self.error_handler)
        return d


    def test_set_throttling_limits(self):
        """test set_throttling_limits exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """ handle the reply """
            aq = self.main.action_q
            self.assertEquals(aq.readLimit, 100)
            self.assertEquals(aq.writeLimit, 500)
            d.callback(True)
        client.call_method('set_throttling_limits', 100, 500,
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_enable_bandwidth_throttling(self):
        """test enable_bandwidth_throttling exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = False
        def reply_handler(_):
            """ handle the reply """
            self.assertTrue(aq.throttling_enabled)
            d.callback(True)
        client.call_method('enable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_bandwidth_throttling(self):
        """test disable_bandwidth_throttling exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = True
        def reply_handler(_):
            """ handle the reply """
            self.assertFalse(aq.throttling_enabled)
            d.callback(True)
        client.call_method('disable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_bandwidth_throttling_enabled(self):
        """test bandwidth_throttling_enabled exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler_enabled(result):
            """ handle the reply """
            self.assertEquals(1, result)
            d.callback(True)

        def reply_handler_disabled(result):
            """ handle the reply """
            self.assertEquals(0, result)
            self.main.action_q.throttling_enabled = True
            client.call_method('bandwidth_throttling_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)
        client.call_method('bandwidth_throttling_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_autosubscribe_enabled(self):
        """Test for Config.udf_autosubscribe_enabled."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler_disabled(result):
            """ handle the reply """
            self.assertTrue(result)
            d.callback(True)

        def reply_handler_enabled(result):
            """ handle the reply """
            self.assertFalse(result)
            config.get_user_config().set_udf_autosubscribe(False)
            client.call_method('udf_autosubscribe_enabled',
                               reply_handler=reply_handler_disabled,
                               error_handler=self.error_handler)
        client.call_method('udf_autosubscribe_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_enable_autosubscribe(self):
        """Test for Config.enable_udf_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """ handle the reply """
            self.assertTrue(config.get_user_config().get_udf_autosubscribe())
            d.callback(True)
        client.call_method('enable_udf_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_autosubscribe(self):
        """Test for Config.disable_udf_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """ handle the reply """
            self.assertFalse(config.get_user_config().get_udf_autosubscribe())
            d.callback(True)
        client.call_method('disable_udf_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d


class TestDBusOAuth(DBusTwistedTestCase):
    """Tests the interaction between dbus_interface and oauthdesktop"""

    def setUp(self):
        DBusTwistedTestCase.setUp(self)
        self.oauth = FakeLogin(self.bus)
        self._old_path = dbus_interface.DBUS_PATH_AUTH
        dbus_interface.DBUS_PATH_AUTH = '/oauthdesktop'

    def tearDown(self):
        # collect all signal receivers registered during the test
        signal_receivers = set()
        with self.bus._signals_lock:
            for group in self.bus._signal_recipients_by_object_path.values():
                for matches in group.values():
                    for match in matches.values():
                        signal_receivers.update(match)
        d = self.cleanup_signal_receivers(signal_receivers)
        def shutdown(r):
            self.oauth.shutdown()
            dbus_interface.DBUS_PATH_AUTH = self._old_path
        d.addCallback(shutdown)
        d.addCallback(lambda _: DBusTwistedTestCase.tearDown(self))
        return d

    def test_new_credentials_signal(self):
        """NewCredentials signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('my_token',))
        d = self.dbus_iface._request_token()
        def check(info):
            realm, key = info
            expected = (u'http://test.ubuntuone.com', u'ubuntuone')
            self.assertEquals((realm, key), expected)
        d.addCallbacks(check, self.fail)
        return d

    def test_no_credentials_signal(self):
        """NoCredentials signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_no_token)
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'NoCredentials')
        d.addCallbacks(self.fail, check)
        return d

    def test_auth_denied_signal(self):
        """AuthorizationDenied signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_denial)
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'AuthorizationDenied')
        d.addCallbacks(self.fail, check)
        return d

    def test_oauth_error_signal(self):
        """OAuthError signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_error,
                                            ('error!',))
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'OAuthError: error!')
        d.addCallbacks(self.fail, check)
        return d

    def test_connect_without_token(self):
        """Test connecting without having a token"""
        # replace the get_access_token method in Main
        def get_access_token():
            raise main.NoAccessToken('no token')
        self.main.get_access_token = get_access_token
        self.dbus_iface.disconnect()
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('the token',))
        d = self.dbus_iface.connect()
        def check(result):
            self.assertEquals(result, None)
        d.addCallbacks(check, self.fail)
        return d

    def test_connect_with_token(self):
        """Test connecting with a token"""
        self.dbus_iface.disconnect()
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('the token',))
        d = self.dbus_iface.connect()
        # check that the deferred was fired
        self.assertEquals(d.called, True)
        self.assertEquals(d.result, None)
        d.addCallbacks(lambda r: self.assertEquals(r, None), self.fail)
        return d

    def test_error_on_login(self):
        """Test connecting without having a token and getting an error in the
        call to oauthdesktop login().
        """
        # replace the get_access_token method in Main
        def get_access_token():
            raise main.NoAccessToken('no token')
        self.main.get_access_token = get_access_token
        self.dbus_iface.disconnect()
        def broken_login(*args):
            raise ValueError('oops, login is broken!')
        self.oauth.processor.next_login_with(broken_login)
        d = self.dbus_iface.connect()
        def check(result):
            self.assertIn('ValueError: oops, login is broken!',
                          result.getErrorMessage())
        d.addCallbacks(self.fail, check)
        return d


class FolderTests(DBusTwistedTestCase):
    """Tests for the Folder object exposed via dbus."""

    def setUp(self):
        """ setup the test """
        self.home_dir = self.mktemp('ubuntuonehacker')

        d = defer.maybeDeferred(DBusTwistedTestCase.setUp, self)
        def create_client(r):
            self.folders_client = DBusClient(self.bus, '/folders',
                                             DBUS_IFACE_FOLDERS_NAME)
            self._old_home = os.environ['HOME']
            os.environ['HOME'] = self.home_dir
        d.addCallback(create_client)
        return d

    def tearDown(self):
        os.environ['HOME'] = self._old_home
        self.rmtree(self.home_dir)
        return DBusTwistedTestCase.tearDown(self)

    def _create_udf(self, id, node_id, suggested_path, subscribed=True):
        """Create an UDF and returns it and the volume"""
        path = self.main.vm._build_udf_path(suggested_path)
        udf = UDF(str(id), str(node_id), suggested_path, path, subscribed)
        return udf

    def test_get_udf_dict(self):
        """Test for Folders._get_udf_dict."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        udf_dict = self.dbus_iface.folders._get_udf_dict(udf)
        # check the path it's unicode
        self.assertEquals(udf_dict['path'], udf.path.decode('utf-8'))
        self.assertEquals(udf_dict['volume_id'], udf.id)
        self.assertEquals(udf_dict['suggested_path'], udf.suggested_path)
        self.assertEquals(udf_dict['node_id'], udf.node_id)
        self.assertFalse(udf_dict['subscribed'])

    @defer.inlineCallbacks
    def test_get_folders(self):
        """Test for Folders.get_folders."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        d = defer.Deferred()
        self.folders_client.call_method('get_folders',
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        info = yield d
        self.assertEquals(len(info), 0)
        # add a udf
        yield self.main.vm.add_udf(udf)
        d2 = defer.Deferred()
        self.folders_client.call_method('get_folders',
                                        reply_handler=d2.callback,
                                        error_handler=self.error_handler)
        info = yield d2
        udf_dict = self.dbus_iface.folders._get_udf_dict(udf)
        self.assertEquals(1, len(info))
        for key, value in udf_dict.items():
            self.assertEquals(info[0][key], value)

    def test_get_info(self):
        """Test for Folders.get_info."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        d = defer.Deferred()
        self.folders_client.call_method('get_info', udf.path,
                                        reply_handler=self.error_handler,
                                        error_handler=d.callback)
        def check_error(f):
            """Check we get an error."""
            # check the error type
            self.assertEquals('org.freedesktop.DBus.Python.KeyError',
                              f._dbus_error_name)
            # add a udf
            add_deferred = self.main.vm.add_udf(udf)
            def get_info(_):
                """Call get_info once we created the udf."""
                d = defer.Deferred()
                self.folders_client.call_method('get_info', udf.path,
                                            reply_handler=d.callback,
                                            error_handler=self.error_handler)
                return d
            add_deferred.addCallback(get_info)
            return add_deferred
        def check(info):
            """Check we get the udf info."""
            udf_dict = self.dbus_iface.folders._get_udf_dict(udf)
            self.assertEquals(info, udf_dict)
        d.addCallback(check_error)
        d.addCallback(check)
        return d

    def test_create(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño/')
        id = uuid.uuid4()
        node_id = uuid.uuid4()
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            # check that the marker it's the full path to the udf
            udf_path = os.path.join(os.path.expanduser(path), name)
            if str(marker) != udf_path:
                d.errback(ValueError("marker != path - "
                                     "marker: %r path: %r" % (marker, udf_path)))
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=id,
                                                       node_id=node_id,
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        def created_handler(info):
            """FolderCreated handler."""
            d.callback(info)
        match = self.bus.add_signal_receiver(created_handler,
                                             signal_name='FolderCreated')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(info):
            """Check the FolderCreated info."""
            self.assertTrue(os.path.exists(info['path']), info['path'])
            self.assertEquals(info['path'], os.path.normpath(path.encode('utf-8')))
            mdobj = self.main.fs.get_by_path(path.encode('utf-8'))
            udf = self.main.vm.get_volume(mdobj.share_id)
            self.assertNotEquals(None, udf)
            self.assertEquals(udf.path, os.path.normpath(path.encode('utf-8')))
            udf_dict = self.dbus_iface.folders._get_udf_dict(udf)
            self.assertEquals(info, udf_dict)
            self.main.vm.udf_deleted(udf.volume_id)

        d.addCallback(check)
        d.addErrback(self.error_handler)
        return d

    def test_create_server_error(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf, that fails."""
            self.main.event_q.push("AQ_CREATE_UDF_ERROR", marker, "Oops, error!")
        self.main.action_q.create_udf = create_udf
        def create_handler(info, error):
            """FolderCreateError handler."""
            d.callback((info, error))
        match = self.bus.add_signal_receiver(create_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(result):
            """Check the result."""
            info, error = result
            self.assertEquals(info['path'], path)
            self.assertEquals(error, 'Oops, error!')
        d.addCallback(check)
        return d

    def test_create_client_error(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf, that fails."""
            raise ValueError("I'm broken.")
        self.main.action_q.create_udf = create_udf
        def create_handler(info, error):
            """FolderCreateError handler."""
            d.callback((info, error))
        match = self.bus.add_signal_receiver(create_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(result):
            """Check the result."""
            info, error = result
            self.assertEquals(info['path'], path)
            self.assertEquals(error, "UNKNOWN_ERROR: I'm broken.")
        d.addCallback(check)
        return d

    def test_create_error_signal(self):
        """Test for FolderCreateError."""
        path = os.path.join(self.home_dir, u'ñoño')
        d = defer.Deferred()
        def create_error_handler(info, error):
            """FolderCreateError handler"""
            self.assertEquals(info['path'], path.decode('utf-8'))
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(create_error_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        # TODO: once create_udf is implemented remove this callLater
        reactor.callLater(0, self.main.event_q.push,
                          'VM_UDF_CREATE_ERROR', path, "I'm broken")
        return d

    def test_delete(self):
        """Test for Folders.delete."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        self.main.vm.add_udf(udf)
        d = defer.Deferred()
        def delete_volume(path):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=udf.id)
        self.main.action_q.delete_volume = delete_volume
        def deleted_handler(info):
            """FolderDeleted handler."""
            self.assertRaises(KeyError, self.main.fs.get_by_path,
                              info['path'].decode('utf-8'))
            self.assertRaises(KeyError, self.main.vm.get_volume, info['volume_id'])
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_handler,
                                             signal_name='FolderDeleted')
        self.signal_receivers.add(match)
        def check_deleted(info):
            """the callback"""
            self.assertNotIn(udf.volume_id, self.main.vm.udfs)
            self.assertRaises(KeyError, self.main.fs.get_by_path, udf.path)
            self.assertRaises(KeyError, self.main.vm.get_volume, udf.volume_id)
        self.folders_client.call_method('delete', udf.volume_id,
                                        reply_handler=check_deleted,
                                        error_handler=self.error_handler)
        return d

    def test_delete_error_signal(self):
        """Test for FolderDeleteError."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        self.main.vm.add_udf(udf)
        d = defer.Deferred()
        # patch delete_volume to fail
        def delete_volume(path):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_ERROR",
                                   volume_id=udf.volume_id, error="I'm broken")
        self.main.action_q.delete_volume = delete_volume
        def deleted_error_handler(info, error):
            """FolderDeleteError handler"""
            self.assertEquals(info['volume_id'], udf.volume_id)
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='FolderDeleteError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('delete', udf.volume_id,
                                        reply_handler=lambda *args: None,
                                        error_handler=self.error_handler)
        d.addCallback(lambda _: self.main.vm.udf_deleted(udf.volume_id))
        return d

    @defer.inlineCallbacks
    def test_subscribe(self):
        """Test for Folders.subscribe and that it fires a dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        def subscribe_handler(info):
            """FolderSubscribed handler."""
            d.callback(info)
        match = self.bus.add_signal_receiver(subscribe_handler,
                                             signal_name='FolderSubscribed')
        self.signal_receivers.add(match)
        self.folders_client.call_method('subscribe', udf.volume_id,
                                        reply_handler=lambda x: None,
                                        error_handler=self.error_handler)
        def check(info):
            """Check that the folder is subscribed."""
            self.assertTrue(info['subscribed'],
                            "UDF %s isn't subscribed" % udf.volume_id)
        d.addCallback(check)
        d.addCallback(lambda _: self.main.vm.udf_deleted(udf.volume_id))
        yield d

    @defer.inlineCallbacks
    def test_unsubscribe(self):
        """Test for Folders.unsubscribe."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        self.folders_client.call_method('unsubscribe', udf.volume_id,
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        def check(r):
            """Check that the folder it's not subscribed."""
            self.assertFalse(self.main.vm.udfs[udf.volume_id].subscribed,
                            "UDF %s is subscribed" % udf.volume_id)
        d.addCallback(check)
        yield d

    @defer.inlineCallbacks
    def test_unsubscribe_signal(self):
        """Test for Folders.unsubscribe fired dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        signal_deferred = defer.Deferred()
        d = defer.Deferred()
        def unsubscribe_handler(info):
            """FolderUnSubscribed handler."""
            self.assertFalse(info['subscribed'],
                             "UDF %s is subscribed" % udf.volume_id)
            signal_deferred.callback(info)
        match = self.bus.add_signal_receiver(unsubscribe_handler,
                                             signal_name='FolderUnSubscribed')
        self.signal_receivers.add(match)
        self.folders_client.call_method('unsubscribe', udf.volume_id,
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        def check(r):
            """Check that the folder it's not subscribed."""
            a_udf = self.main.vm.udfs[udf.volume_id]
            self.assertFalse(a_udf.subscribed,
                            "UDF %s is subscribed" % a_udf.volume_id)
            return signal_deferred
        d.addCallback(check)
        d.addCallback(lambda _: self.main.vm.udf_deleted(udf.volume_id))
        yield d

    def test_refresh_volumes(self):
        """Just check that refresh_volumes method works."""
        client = DBusClient(self.bus, '/folders', DBUS_IFACE_FOLDERS_NAME)
        d = defer.Deferred()
        def list_volumes():
            """Stub list_volumes."""
            d.callback(True)
        self.main.action_q.list_volumes = list_volumes
        client.call_method('refresh_volumes',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        return d


class ShareTests(DBusTwistedTestCase):
    """Share specific tests"""

    def test_delete_share(self):
        """Test for Shares.delete_share."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = Share(path=share_path, volume_id='share_id',
                      node_id='node_id', accepted=True)
        self.main.vm.add_share(share)
        d = defer.Deferred()
        def delete_volume(volume_id):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=volume_id)
        self.main.action_q.delete_volume = delete_volume
        def deleted_handler(info):
            """FolderDeleted handler."""
            self.assertRaises(KeyError, self.main.fs.get_by_path,
                              info['path'].decode('utf-8'))
            self.assertRaises(KeyError, self.main.vm.get_volume, info['volume_id'])
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_handler,
                                             signal_name='ShareDeleted')
        self.signal_receivers.add(match)
        def check_deleted(info):
            """the callback"""
            self.assertNotIn(share.volume_id, self.main.vm.shares)
            self.assertRaises(KeyError, self.main.fs.get_by_path, share.path)
            self.assertRaises(KeyError, self.main.vm.get_volume, share.volume_id)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        client.call_method('delete_share', share.volume_id,
                           reply_handler=check_deleted,
                           error_handler=self.error_handler)
        return d

    def test_delete_share_error_signal(self):
        """Test for Shares.delete_share with an error."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = Share(path=share_path, volume_id='share_id',
                      node_id='node_id', accepted=True)
        self.main.vm.add_share(share)
        d = defer.Deferred()
        # patch delete_volume to fail
        def delete_volume(volume_id):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_ERROR",
                                   volume_id=volume_id, error="I'm broken")
        self.main.action_q.delete_volume = delete_volume
        def deleted_error_handler(info, error):
            """FolderDeleteError handler"""
            self.assertEquals(info['volume_id'], share.volume_id)
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='ShareDeleteError')
        self.signal_receivers.add(match)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        client.call_method('delete_share', share.volume_id,
                           reply_handler=lambda *args: None,
                           error_handler=self.error_handler)
        return d


def test_suite():
    # pylint: disable-msg=C0111
    return unittest.TestLoader().loadTestsFromName(__name__)
