# ubuntuone.syncdaemon.states - SyncDaemon states
#
# Author: John Lenton <john.lenton@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
The states of the SyncDaemon
"""

from pprint import pformat
from inspect import getsource
from difflib import unified_diff

def compare(this, that):
    this = vars(this)
    if this['enter'] is not None:
        this['enter'] = getsource(this['enter']).strip()
    that = vars(that)
    if that['enter'] is not None:
        that['enter'] = getsource(that['enter']).strip()
    if this != that:
        print "\n".join(unified_diff(pformat(this).split('\n'),
                                     pformat(that).split('\n'),
                                     'new '+this['name'], 'old'))
        assert False


class BadTransition(Exception):
    """
    An event arrived that the state didn't know how to handle.
    """

class UndefinedState(Exception):
    """
    The target state doesn't exist.
    """

class SyncDaemonState(object):
    """
    A SyncDaemon state.
    """
    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__,
                            self.name)

    def __init__(self, name, description,
                 sys_net_connected_trn, sys_connect_trn,
                 sys_net_disconnected_trn, sys_disconnect_trn,
                 sys_connection_lost_trn,
                 is_error=False, is_connected=False, is_online=False,
                 enter=None,
                 **other_transitions):
        self.name = name
        self.description = description
        self.is_error = is_error
        self.is_connected = is_connected
        self.is_online = is_online
        self.enter = enter
        self.has_network = any((self.is_connected,
                                "WITH_NETWORK" in name,
                                "CONNECTING" in name,
                                "CONNECTED_CLEANUP" in name))
        self.wants_to_connect = any((self.is_connected,
                                     "WAITING" in name,
                                     "CONNECTING" in name))
        self.is_handshake = 'SYS_HANDSHAKE_TIMEOUT' in other_transitions
        self.transitions = dict(SYS_NET_CONNECTED=sys_net_connected_trn,
                                SYS_CONNECT=sys_connect_trn,
                                SYS_NET_DISCONNECTED=sys_net_disconnected_trn,
                                SYS_DISCONNECT=sys_disconnect_trn,
                                SYS_CONNECTION_LOST=sys_connection_lost_trn,
                                SYS_UNKNOWN_ERROR='UNKNOWN_ERROR',
                                **other_transitions)

    def next(self, event):
        """
        Determine the next state, given the current state and the event.
        That makes us a Moore FSM, right?
        """
        if event in self.transitions:
            new = self.transitions[event]
            if new in globals():
                return globals()[new]
            else:
                raise UndefinedState("State %s is undefined" % new)
        else:
            raise BadTransition("State %s can't handle the %s event"
                                % (self.name, event))


    def clone_with_queue(self, qname, **extra_transitions):
        new_trns = self.transitions.copy()
        del new_trns['SYS_UNKNOWN_ERROR']
        suffix = '_WITH_' + qname
        mqw_trn = new_trns.pop('SYS_META_QUEUE_WAITING')
        mqd_trn = new_trns.pop('SYS_META_QUEUE_DONE')
        cqw_trn = new_trns.pop('SYS_CONTENT_QUEUE_WAITING')
        cqd_trn = new_trns.pop('SYS_CONTENT_QUEUE_DONE')
        if self.name.startswith('START_'):
            bq_trn = self.name[6:] + '_WITH_BOTHQ'
            expected_name = 'START_' + dict(METAQ=mqw_trn,
                                            CONTQ=cqw_trn, BOTHQ=bq_trn)[qname]
        else:
            bq_trn = self.name + '_WITH_BOTHQ'
            expected_name = dict(METAQ=mqw_trn,
                                 CONTQ=cqw_trn, BOTHQ=bq_trn)[qname]
        mqw = dict(METAQ=mqw_trn, CONTQ=bq_trn, BOTHQ=bq_trn)[qname]
        mqd = dict(METAQ=mqd_trn, CONTQ=cqw_trn, BOTHQ=cqw_trn)[qname]
        cqw = dict(METAQ=bq_trn, CONTQ=cqw_trn, BOTHQ=bq_trn)[qname]
        cqd = dict(METAQ=mqw_trn, CONTQ=cqd_trn, BOTHQ=mqw_trn)[qname]
        new = SyncDaemonState(self.name + suffix,
                              self.description + ', ' + qname + ' waiting',
                              new_trns.pop('SYS_NET_CONNECTED') + suffix,
                              new_trns.pop('SYS_CONNECT') + suffix,
                              new_trns.pop('SYS_NET_DISCONNECTED') + suffix,
                              new_trns.pop('SYS_DISCONNECT') + suffix,
                              new_trns.pop('SYS_CONNECTION_LOST') + suffix,
                              self.is_error, self.is_connected, self.is_online,
                              self.enter,
                              SYS_META_QUEUE_WAITING=mqw,
                              SYS_META_QUEUE_DONE=mqd,
                              SYS_CONTENT_QUEUE_WAITING=cqw,
                              SYS_CONTENT_QUEUE_DONE=cqd)
        new.is_handshake = self.is_handshake
        new.with_q = qname
        assert new.name == expected_name, "%r != %r" % (new.name, expected_name)
        for evt, trn in new_trns.items():
            if trn not in ['BAD_VERSION', 'CAPABILITIES_MISMATCH', 'AUTH_FAILED']:
                trn = trn + suffix
            new.transitions[evt] = trn
        for evt, trn in extra_transitions.items():
            new.transitions[evt] = trn
        return new

    def clone_with_metaq(self, **extra_transitions):
        return self.clone_with_queue('METAQ', **extra_transitions)

    def clone_with_contq(self, **extra_transitions):
        return self.clone_with_queue('CONTQ', **extra_transitions)

    def clone_with_bothq(self, **extra_transitions):
        return self.clone_with_queue('BOTHQ', **extra_transitions)

    def clone_with_no_start(self, new_description, **extra_transitions):
        if not self.name.startswith('START_'):
            raise ValueError, "Not a START_ transition"
        name = self.name[6:]
        new_trns = self.transitions.copy()
        del new_trns['SYS_UNKNOWN_ERROR']
        new_trns.update(extra_transitions)
        new = SyncDaemonState(name,
                              new_description,
                              new_trns.pop('SYS_NET_CONNECTED'),
                              new_trns.pop('SYS_CONNECT'),
                              new_trns.pop('SYS_NET_DISCONNECTED'),
                              new_trns.pop('SYS_DISCONNECT'),
                              new_trns.pop('SYS_CONNECTION_LOST'),
                              self.is_error, self.is_connected, self.is_online,
                              None,
                              **new_trns)
        if getattr(self, 'with_q', object) is not object:
            new.with_q = self.with_q
        return new

class NonActiveConnectedSDState(SyncDaemonState):
    """
    States that handle network disconnections with no cleanup
    required.
    """
    def __init__(self, name, description, is_online=False, with_q=None,
                 **other_transitions):
        ready_waiting = 'READY_WAITING'
        ready_w_network = 'READY_WITH_NETWORK'
        start_connecting = 'START_CONNECTING'
        self.with_q = with_q
        if with_q is not None:
            ready_waiting += '_WITH_' + with_q
            ready_w_network += '_WITH_' + with_q
            start_connecting += '_WITH_' + with_q
        if name.startswith('START_'):
            same = name[6:]
        else:
            same = name
        super(NonActiveConnectedSDState, self).__init__(name, description,
                                                        same, same,
                                                        ready_waiting,
                                                        ready_w_network,
                                                        start_connecting,
                                                        is_connected=True,
                                                        is_online=is_online,
                                                        **other_transitions)

class WorkingSDState(SyncDaemonState):
    """
    States that handle network disconnections by doing cleanup.
    """
    def __init__(self, name, description, with_q=None, **other_transitions):
        self.with_q = with_q
        if with_q is not None:
            suffix = '_WITH_' + with_q
        else:
            suffix = ''
        if name.startswith('START_'):
            next_name = name[6:]
        else:
            next_name = name
        super(WorkingSDState, self).__init__(
            name, description, next_name, next_name,
            'START_CLEANUP_WAITING' + suffix,
            'START_CONNECTED_CLEANUP' + suffix,
            'START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST' + suffix,
            is_connected=True, is_online=True,
            **other_transitions)

class AQErrorState(SyncDaemonState):
    """
    States that are errors.
    """
    def __init__(self, name, description, enter=None):
        super(AQErrorState, self).__init__(name, description,
                                           name, name, name, name, name,
                                           is_error=True, enter=enter)

    def next(self, event):
        """
        The next state of an error is always itself.
        """
        return self


INIT = SyncDaemonState(
    'INIT', 'just initialized',
    'INIT_WITH_NETWORK', 'INIT_WAITING',
    'INIT', 'INIT', 'INIT',
    SYS_WAIT_FOR_LOCAL_RESCAN='READING',
    )

INIT_WITH_NETWORK = SyncDaemonState(
    'INIT_WITH_NETWORK',
    'just initialized, but already has network',
    'INIT_WITH_NETWORK', 'INIT_WAITING_WITH_NETWORK',
    'INIT', 'INIT_WITH_NETWORK', 'INIT_WITH_NETWORK',
    SYS_WAIT_FOR_LOCAL_RESCAN='READING_WITH_NETWORK',
    )

INIT_WAITING = SyncDaemonState(
    'INIT_WAITING',
    'just initialized, but the user already wants to connect',
    'INIT_WAITING_WITH_NETWORK', 'INIT_WAITING',
    'INIT_WAITING', 'INIT', 'INIT_WAITING',
    SYS_WAIT_FOR_LOCAL_RESCAN='READING_WAITING',
    )

INIT_WAITING_WITH_NETWORK = SyncDaemonState(
    'INIT_WAITING_WITH_NETWORK',
    'just initialized, user wants to connect, and network is available',
    'INIT_WAITING_WITH_NETWORK', 'INIT_WAITING_WITH_NETWORK',
    'INIT_WAITING', 'INIT_WITH_NETWORK', 'INIT_WAITING_WITH_NETWORK',
    SYS_WAIT_FOR_LOCAL_RESCAN='READING_WAITING_WITH_NETWORK',
    )

READING = SyncDaemonState(
    'READING',
    'doing local rescan',
    'READING_WITH_NETWORK', 'READING_WAITING',
    'READING', 'READING', 'READING',
    SYS_LOCAL_RESCAN_DONE='READY_FOR_NETWORK',
    SYS_META_QUEUE_WAITING='READING_WITH_METAQ',
    SYS_META_QUEUE_DONE='READING',
    SYS_CONTENT_QUEUE_WAITING='READING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READING',
    )
READING_WITH_METAQ = READING.clone_with_metaq()
READING_WITH_CONTQ = READING.clone_with_contq()
READING_WITH_BOTHQ = READING.clone_with_bothq()

READING_WITH_NETWORK = SyncDaemonState(
    'READING_WITH_NETWORK',
    'doing local rescan; network is available',
    'READING_WITH_NETWORK', 'READING_WAITING_WITH_NETWORK',
    'READING', 'READING_WITH_NETWORK', 'READING_WITH_NETWORK',
    SYS_LOCAL_RESCAN_DONE='READY_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='READING_WITH_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='READING_WITH_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='READING_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READING_WITH_NETWORK',
    )
READING_WITH_NETWORK_WITH_METAQ = READING_WITH_NETWORK.clone_with_metaq()
READING_WITH_NETWORK_WITH_CONTQ = READING_WITH_NETWORK.clone_with_contq()
READING_WITH_NETWORK_WITH_BOTHQ = READING_WITH_NETWORK.clone_with_bothq()

READING_WAITING = SyncDaemonState(
    'READING_WAITING',
    'doing local rescan; user wants to connect',
    'READING_WAITING_WITH_NETWORK', 'READING_WAITING',
    'READING_WAITING', 'READING', 'READING_WAITING',
    SYS_LOCAL_RESCAN_DONE='READY_WAITING',
    SYS_META_QUEUE_WAITING='READING_WAITING_WITH_METAQ',
    SYS_META_QUEUE_DONE='READING_WAITING',
    SYS_CONTENT_QUEUE_WAITING='READING_WAITING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READING_WAITING',
    )
READING_WAITING_WITH_METAQ = READING_WAITING.clone_with_metaq()
READING_WAITING_WITH_CONTQ = READING_WAITING.clone_with_contq()
READING_WAITING_WITH_BOTHQ = READING_WAITING.clone_with_bothq()

READING_WAITING_WITH_NETWORK = SyncDaemonState(
    'READING_WAITING_WITH_NETWORK',
    'doing local rescan; user wants to connect, network is available',
    'READING_WAITING_WITH_NETWORK', 'READING_WAITING_WITH_NETWORK',
    'READING_WAITING', 'READING_WITH_NETWORK',
    'READING_WAITING_WITH_NETWORK',
    SYS_LOCAL_RESCAN_DONE='START_CONNECTING',
    SYS_META_QUEUE_WAITING='READING_WAITING_WITH_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='READING_WAITING_WITH_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='READING_WAITING_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READING_WAITING_WITH_NETWORK',
    )
READING_WAITING_WITH_NETWORK_WITH_METAQ = \
    READING_WAITING_WITH_NETWORK.clone_with_metaq()
READING_WAITING_WITH_NETWORK_WITH_CONTQ = \
    READING_WAITING_WITH_NETWORK.clone_with_contq()
READING_WAITING_WITH_NETWORK_WITH_BOTHQ = \
    READING_WAITING_WITH_NETWORK.clone_with_bothq()

READY_FOR_NETWORK = SyncDaemonState(
    'READY_FOR_NETWORK',
    'ready to connect as soon as the user says so and the network comes up',
    'READY_WITH_NETWORK', 'READY_WAITING',
    'READY_FOR_NETWORK', 'READY_FOR_NETWORK', 'READY_FOR_NETWORK',
    SYS_META_QUEUE_WAITING='READY_FOR_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='READY_FOR_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='READY_FOR_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READY_FOR_NETWORK',
    )
READY_FOR_NETWORK_WITH_METAQ = READY_FOR_NETWORK.clone_with_metaq()
READY_FOR_NETWORK_WITH_CONTQ = READY_FOR_NETWORK.clone_with_contq()
READY_FOR_NETWORK_WITH_BOTHQ = READY_FOR_NETWORK.clone_with_bothq()

READY_WITH_NETWORK = SyncDaemonState(
    'READY_WITH_NETWORK',
    'ready to connect, network up, user yet to push "go"',
    'READY_WITH_NETWORK', 'START_CONNECTING',
    'READY_FOR_NETWORK', 'READY_WITH_NETWORK', 'READY_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='READY_WITH_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='READY_WITH_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='READY_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READY_WITH_NETWORK',
    )
READY_WITH_NETWORK_WITH_METAQ = READY_WITH_NETWORK.clone_with_metaq()
READY_WITH_NETWORK_WITH_CONTQ = READY_WITH_NETWORK.clone_with_contq()
READY_WITH_NETWORK_WITH_BOTHQ = READY_WITH_NETWORK.clone_with_bothq()

READY_WAITING = SyncDaemonState(
    'READY_WAITING',
    'ready to connect; user said "go", network is down',
    'START_CONNECTING', 'READY_WAITING',
    'READY_WAITING', 'READY_FOR_NETWORK', 'READY_WAITING',
    SYS_META_QUEUE_WAITING='READY_WAITING_WITH_METAQ',
    SYS_META_QUEUE_DONE='READY_WAITING',
    SYS_CONTENT_QUEUE_WAITING='READY_WAITING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='READY_WAITING',
)
READY_WAITING_WITH_METAQ = READY_WAITING.clone_with_metaq()
READY_WAITING_WITH_CONTQ = READY_WAITING.clone_with_contq()
READY_WAITING_WITH_BOTHQ = READY_WAITING.clone_with_bothq()

START_STANDOFF_WAITING_WITH_NETWORK = SyncDaemonState(
    'START_STANDOFF_WAITING_WITH_NETWORK',
    'disconnect and wait for SYS_CONNECTION_LOST, then go on to connect',
    'STANDOFF_WAITING_WITH_NETWORK', 'STANDOFF_WAITING_WITH_NETWORK',
    'STANDOFF_WAITING', 'STANDOFF_WITH_NETWORK',
    'START_CONNECTING',
    SYS_META_QUEUE_WAITING='STANDOFF_WAITING_WITH_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='STANDOFF_WAITING_WITH_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='STANDOFF_WAITING_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='STANDOFF_WAITING_WITH_NETWORK',
    enter=lambda m: m.action_q.disconnect(),
)
START_STANDOFF_WAITING_WITH_NETWORK_WITH_METAQ = \
    START_STANDOFF_WAITING_WITH_NETWORK.clone_with_metaq()
START_STANDOFF_WAITING_WITH_NETWORK_WITH_CONTQ = \
    START_STANDOFF_WAITING_WITH_NETWORK.clone_with_contq()
START_STANDOFF_WAITING_WITH_NETWORK_WITH_BOTHQ = \
    START_STANDOFF_WAITING_WITH_NETWORK.clone_with_bothq()
STANDOFF_WAITING_WITH_NETWORK = \
    START_STANDOFF_WAITING_WITH_NETWORK.clone_with_no_start(
    'wait for SYS_CONNECTION_LOST, then go on to connect')
STANDOFF_WAITING_WITH_NETWORK_WITH_METAQ = \
    STANDOFF_WAITING_WITH_NETWORK.clone_with_metaq()
STANDOFF_WAITING_WITH_NETWORK_WITH_CONTQ = \
    STANDOFF_WAITING_WITH_NETWORK.clone_with_contq()
STANDOFF_WAITING_WITH_NETWORK_WITH_BOTHQ = \
    STANDOFF_WAITING_WITH_NETWORK.clone_with_bothq()

START_STANDOFF_WITH_NETWORK = SyncDaemonState(
    'START_STANDOFF_WITH_NETWORK',
    'disconnect and wait for SYS_CONNECTION_LOST;'
        ' network present, but asked not to connect',
    'STANDOFF_WITH_NETWORK', 'STANDOFF_WAITING_WITH_NETWORK',
    'STANDOFF', 'STANDOFF_WITH_NETWORK',
    'READY_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='STANDOFF_WITH_NETWORK_WITH_METAQ',
    SYS_META_QUEUE_DONE='STANDOFF_WITH_NETWORK',
    SYS_CONTENT_QUEUE_WAITING='STANDOFF_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='STANDOFF_WITH_NETWORK',
    enter=lambda m: m.action_q.disconnect(),
)
START_STANDOFF_WITH_NETWORK_WITH_METAQ = \
    START_STANDOFF_WITH_NETWORK.clone_with_metaq()
START_STANDOFF_WITH_NETWORK_WITH_CONTQ = \
    START_STANDOFF_WITH_NETWORK.clone_with_contq()
START_STANDOFF_WITH_NETWORK_WITH_BOTHQ = \
    START_STANDOFF_WITH_NETWORK.clone_with_bothq()
STANDOFF_WITH_NETWORK = START_STANDOFF_WITH_NETWORK.clone_with_no_start(
    'wait for SYS_CONNECTION_LOST; network present, but asked not to connect')
STANDOFF_WITH_NETWORK_WITH_METAQ = STANDOFF_WITH_NETWORK.clone_with_metaq()
STANDOFF_WITH_NETWORK_WITH_CONTQ = STANDOFF_WITH_NETWORK.clone_with_contq()
STANDOFF_WITH_NETWORK_WITH_BOTHQ = STANDOFF_WITH_NETWORK.clone_with_bothq()

START_STANDOFF_WAITING = SyncDaemonState(
    'START_STANDOFF_WAITING',
    'disconnect and wait for SYS_CONNECTION_LOST;'
        ' wanting to connect but no network present',
    'STANDOFF_WAITING_WITH_NETWORK', 'STANDOFF_WAITING',
    'STANDOFF_WAITING', 'STANDOFF',
    'READY_WAITING',
    SYS_META_QUEUE_WAITING='STANDOFF_WAITING_WITH_METAQ',
    SYS_META_QUEUE_DONE='STANDOFF_WAITING',
    SYS_CONTENT_QUEUE_WAITING='STANDOFF_WAITING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='STANDOFF_WAITING',
    enter=lambda m: m.action_q.disconnect(),
)
START_STANDOFF_WAITING_WITH_METAQ = START_STANDOFF_WAITING.clone_with_metaq()
START_STANDOFF_WAITING_WITH_CONTQ = START_STANDOFF_WAITING.clone_with_contq()
START_STANDOFF_WAITING_WITH_BOTHQ = START_STANDOFF_WAITING.clone_with_bothq()
STANDOFF_WAITING = START_STANDOFF_WAITING.clone_with_no_start(
    'wait for SYS_CONNECTION_LOST; wanting to connect but no network present')
STANDOFF_WAITING_WITH_METAQ = STANDOFF_WAITING.clone_with_metaq()
STANDOFF_WAITING_WITH_CONTQ = STANDOFF_WAITING.clone_with_contq()
STANDOFF_WAITING_WITH_BOTHQ = STANDOFF_WAITING.clone_with_bothq()

START_STANDOFF = SyncDaemonState(
    'START_STANDOFF',
    'disconnect and wait for SYS_CONNECTION_LOST;'
        ' no network present, not wanting to connect',
    'STANDOFF_WITH_NETWORK', 'STANDOFF_WAITING',
    'STANDOFF', 'STANDOFF',
    'READY_FOR_NETWORK',
    SYS_META_QUEUE_WAITING='STANDOFF_WITH_METAQ',
    SYS_META_QUEUE_DONE='STANDOFF',
    SYS_CONTENT_QUEUE_WAITING='STANDOFF_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='STANDOFF',
    enter=lambda m: m.action_q.disconnect(),
)
START_STANDOFF_WITH_METAQ = START_STANDOFF.clone_with_metaq()
START_STANDOFF_WITH_CONTQ = START_STANDOFF.clone_with_contq()
START_STANDOFF_WITH_BOTHQ = START_STANDOFF.clone_with_bothq()
STANDOFF = START_STANDOFF.clone_with_no_start(
    'wait for SYS_CONNECTION_LOST; no network present, not wanting to connect')
STANDOFF_WITH_METAQ = STANDOFF.clone_with_metaq()
STANDOFF_WITH_CONTQ = STANDOFF.clone_with_contq()
STANDOFF_WITH_BOTHQ = STANDOFF.clone_with_bothq()

START_CONNECTING = SyncDaemonState(
    'START_CONNECTING',
    'started waiting for the socket to come up',
    'CONNECTING', 'CONNECTING',
    'READY_WAITING', 'READY_WITH_NETWORK', 'START_CONNECTING',
    SYS_CONNECTION_MADE='START_CONNECTED',
    SYS_META_QUEUE_WAITING='CONNECTING_WITH_METAQ',
    SYS_META_QUEUE_DONE='CONNECTING',
    SYS_CONTENT_QUEUE_WAITING='CONNECTING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='CONNECTING',
    SYS_HANDSHAKE_TIMEOUT='START_STANDOFF_WAITING_WITH_NETWORK',
    enter=lambda m: m.action_q.connect(),
    )
START_CONNECTING_WITH_METAQ = START_CONNECTING.clone_with_metaq()
START_CONNECTING_WITH_CONTQ = START_CONNECTING.clone_with_contq()
START_CONNECTING_WITH_BOTHQ = START_CONNECTING.clone_with_bothq()

CONNECTING = START_CONNECTING.clone_with_no_start(
    'waiting for the socket to come up')
CONNECTING_WITH_METAQ = CONNECTING.clone_with_metaq()
CONNECTING_WITH_CONTQ = CONNECTING.clone_with_contq()
CONNECTING_WITH_BOTHQ = CONNECTING.clone_with_bothq()

START_CONNECTED = NonActiveConnectedSDState(
    'START_CONNECTED',
    'socket came up! start checking protocol version',
    SYS_PROTOCOL_VERSION_ERROR='BAD_VERSION',
    SYS_PROTOCOL_VERSION_OK='START_SET_CAPABILITIES',
    SYS_META_QUEUE_WAITING='CONNECTED_WITH_METAQ',
    SYS_META_QUEUE_DONE='CONNECTED',
    SYS_CONTENT_QUEUE_WAITING='CONNECTED_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='CONNECTED',
    SYS_HANDSHAKE_TIMEOUT='START_STANDOFF_WAITING_WITH_NETWORK',
    enter=lambda m: m.check_version(),
    )
START_CONNECTED_WITH_METAQ = START_CONNECTED.clone_with_metaq()
START_CONNECTED_WITH_CONTQ = START_CONNECTED.clone_with_contq()
START_CONNECTED_WITH_BOTHQ = START_CONNECTED.clone_with_bothq()
CONNECTED = START_CONNECTED.clone_with_no_start('socket came up!'
                                                ' checking protocol version')
CONNECTED_WITH_METAQ = CONNECTED.clone_with_metaq()
CONNECTED_WITH_CONTQ = CONNECTED.clone_with_contq()
CONNECTED_WITH_BOTHQ = CONNECTED.clone_with_bothq()

BAD_VERSION = AQErrorState(
    'BAD_VERSION',
    'Protocol version mismatch. You probably need to upgrade',
    )

START_SET_CAPABILITIES = NonActiveConnectedSDState(
    'START_SET_CAPABILITIES',
    'protocol version is OK! start checking capabilities',
    SYS_SET_CAPABILITIES_ERROR='CAPABILITIES_MISMATCH',
    SYS_SET_CAPABILITIES_OK='START_AUTHENTICATING',
    SYS_META_QUEUE_WAITING='SET_CAPABILITIES_WITH_METAQ',
    SYS_META_QUEUE_DONE='SET_CAPABILITIES',
    SYS_CONTENT_QUEUE_WAITING='SET_CAPABILITIES_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='SET_CAPABILITIES',
    SYS_HANDSHAKE_TIMEOUT='START_STANDOFF_WAITING_WITH_NETWORK',
    enter=lambda m: m.set_capabilities(),
    )
START_SET_CAPABILITIES_WITH_METAQ = START_SET_CAPABILITIES.clone_with_metaq()
START_SET_CAPABILITIES_WITH_CONTQ = START_SET_CAPABILITIES.clone_with_contq()
START_SET_CAPABILITIES_WITH_BOTHQ = START_SET_CAPABILITIES.clone_with_bothq()
SET_CAPABILITIES = START_SET_CAPABILITIES.clone_with_no_start(
    'protocol version is OK! checking capabilities')
SET_CAPABILITIES_WITH_METAQ = SET_CAPABILITIES.clone_with_metaq()
SET_CAPABILITIES_WITH_CONTQ = SET_CAPABILITIES.clone_with_contq()
SET_CAPABILITIES_WITH_BOTHQ = SET_CAPABILITIES.clone_with_bothq()

CAPABILITIES_MISMATCH = AQErrorState(
    'CAPABILITIES_MISMATCH',
    'Capabilities mismatch. You probably need to upgrade',
    )

START_AUTHENTICATING = NonActiveConnectedSDState(
    'START_AUTHENTICATING',
    'Start doing the OAuth dance',
    SYS_OAUTH_OK='START_SCANNING',
    SYS_OAUTH_ERROR='AUTH_FAILED',
    SYS_META_QUEUE_WAITING='AUTHENTICATING_WITH_METAQ',
    SYS_META_QUEUE_DONE='AUTHENTICATING',
    SYS_CONTENT_QUEUE_WAITING='AUTHENTICATING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='AUTHENTICATING',
    SYS_HANDSHAKE_TIMEOUT='START_STANDOFF_WAITING_WITH_NETWORK',
    enter=lambda m: m.authenticate(),
    )
START_AUTHENTICATING_WITH_METAQ = START_AUTHENTICATING.clone_with_metaq()
START_AUTHENTICATING_WITH_CONTQ = START_AUTHENTICATING.clone_with_contq()
START_AUTHENTICATING_WITH_BOTHQ = START_AUTHENTICATING.clone_with_bothq()
AUTHENTICATING = START_AUTHENTICATING.clone_with_no_start(
    'Doing the OAuth dance')
AUTHENTICATING_WITH_METAQ = AUTHENTICATING.clone_with_metaq()
AUTHENTICATING_WITH_CONTQ = AUTHENTICATING.clone_with_contq()
AUTHENTICATING_WITH_BOTHQ = AUTHENTICATING.clone_with_bothq()

AUTH_FAILED = AQErrorState(
    'AUTH_FAILED',
    'OAuth failed')

EXCESSIVE_TIMEOUTS = AQErrorState(
    'EXCESSIVE_TIMEOUTS',
    'Too many timeouts. Giving up.')

UNKNOWN_ERROR = AQErrorState(
    'UNKNOWN_ERROR',
    "Some kind of strange error happened and I can't continue",
    enter=lambda m: m.restart())

START_SCANNING = NonActiveConnectedSDState(
    'START_SCANNING',
    'start doing server rescan',
    SYS_SERVER_RESCAN_STARTING='SCANNING',
    SYS_META_QUEUE_WAITING='SCANNING_WITH_METAQ',
    SYS_META_QUEUE_DONE='SCANNING',
    SYS_CONTENT_QUEUE_WAITING='SCANNING_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='SCANNING',
    enter=lambda m: m.server_rescan(),
    )
START_SCANNING_WITH_METAQ = START_SCANNING.clone_with_metaq()
START_SCANNING_WITH_CONTQ = START_SCANNING.clone_with_contq()
START_SCANNING_WITH_BOTHQ = START_SCANNING.clone_with_bothq()
SCANNING = START_SCANNING.clone_with_no_start('doing server rescan',
                                              SYS_SERVER_RESCAN_DONE='IDLE')
SCANNING_WITH_METAQ = SCANNING.clone_with_metaq(
    SYS_SERVER_RESCAN_DONE='START_WORKING_ON_METADATA')
SCANNING_WITH_CONTQ = SCANNING.clone_with_contq(
    SYS_SERVER_RESCAN_DONE='START_WORKING_ON_CONTENT')
SCANNING_WITH_BOTHQ = SCANNING.clone_with_bothq(
    SYS_SERVER_RESCAN_DONE='START_WORKING_ON_METADATA_WITH_CONTQ')

IDLE = NonActiveConnectedSDState(
    'IDLE',
    'nothing on the wire right now',
    is_online=True,
    SYS_META_QUEUE_WAITING='START_WORKING_ON_METADATA',
    SYS_META_QUEUE_DONE='IDLE',
    SYS_CONTENT_QUEUE_WAITING='START_WORKING_ON_CONTENT',
    SYS_CONTENT_QUEUE_DONE='IDLE',
    )

START_WORKING_ON_METADATA = WorkingSDState(
    'START_WORKING_ON_METADATA',
    'start working on metadata',
    SYS_META_QUEUE_WAITING='WORKING_ON_METADATA',
    SYS_META_QUEUE_DONE='IDLE',
    SYS_CONTENT_QUEUE_WAITING='WORKING_ON_METADATA_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='WORKING_ON_METADATA',
    enter=lambda m: m.action_q.meta_queue.run(),
    )
WORKING_ON_METADATA = START_WORKING_ON_METADATA.clone_with_no_start(
    'working on metadata')

START_WORKING_ON_CONTENT = WorkingSDState(
    'START_WORKING_ON_CONTENT',
    'start working on content',
    SYS_META_QUEUE_WAITING='START_WORKING_ON_BOTH',
    SYS_META_QUEUE_DONE='WORKING_ON_CONTENT',
    SYS_CONTENT_QUEUE_WAITING='WORKING_ON_CONTENT',
    SYS_CONTENT_QUEUE_DONE='IDLE',
    enter=lambda m: m.action_q.content_queue.run(),
    )
WORKING_ON_CONTENT = START_WORKING_ON_CONTENT.clone_with_no_start(
    'working on content')

START_WORKING_ON_METADATA_WITH_CONTQ = WorkingSDState(
    'START_WORKING_ON_METADATA_WITH_CONTQ',
    'start working on metadata; content work is waiting also',
    with_q='CONTQ',
    SYS_META_QUEUE_WAITING='WORKING_ON_METADATA_WITH_CONTQ',
    SYS_META_QUEUE_DONE='START_WORKING_ON_CONTENT',
    SYS_CONTENT_QUEUE_WAITING='WORKING_ON_METADATA_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='WORKING_ON_METADATA',
    enter=lambda m: m.action_q.meta_queue.run(),
    )
WORKING_ON_METADATA_WITH_CONTQ = \
    START_WORKING_ON_METADATA_WITH_CONTQ.clone_with_no_start(
    'working on metadata; content work is waiting')

START_WORKING_ON_BOTH = WorkingSDState(
    'START_WORKING_ON_BOTH',
    'start working on both content and metadata',
    SYS_META_QUEUE_WAITING='WORKING_ON_BOTH',
    SYS_META_QUEUE_DONE='WORKING_ON_CONTENT',
    SYS_CONTENT_QUEUE_WAITING='WORKING_ON_BOTH',
    SYS_CONTENT_QUEUE_DONE='WORKING_ON_METADATA',
    enter=lambda m: m.action_q.meta_queue.run(),
    )
WORKING_ON_BOTH = START_WORKING_ON_BOTH.clone_with_no_start(
    'working on both content and metadata')

CLEANUP = SyncDaemonState(
    'CLEANUP',
    'doing cleanup; net down, user asked for shutdown',
    'CLEANUP_WITH_NETWORK', 'CLEANUP_WAITING',
    'CLEANUP', 'CLEANUP', 'CLEANUP_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP',
    SYS_CLEANUP_FINISHED='START_STANDOFF',
    SYS_META_QUEUE_WAITING='CLEANUP_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP', # cancelling meta_q.head can do this
    SYS_CONTENT_QUEUE_DONE='CLEANUP', # ditto
    )
CLEANUP_WITH_METAQ = CLEANUP.clone_with_metaq()
CLEANUP_WITH_CONTQ = CLEANUP.clone_with_contq()
CLEANUP_WITH_BOTHQ = CLEANUP.clone_with_bothq()

CLEANUP_WITH_CONNECTION_LOST = SyncDaemonState(
    'CLEANUP_WITH_CONNECTION_LOST',
    'doing cleanup; net down, user asked for shutdown, connection already lost',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WITH_CONNECTION_LOST',
    SYS_CLEANUP_FINISHED='START_STANDOFF',
    SYS_META_QUEUE_WAITING='CLEANUP_WITH_CONNECTION_LOST_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WITH_CONNECTION_LOST_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WITH_CONNECTION_LOST',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WITH_CONNECTION_LOST',
    )
CLEANUP_WITH_CONNECTION_LOST_WITH_METAQ = \
    CLEANUP_WITH_CONNECTION_LOST.clone_with_metaq()
CLEANUP_WITH_CONNECTION_LOST_WITH_CONTQ = \
    CLEANUP_WITH_CONNECTION_LOST.clone_with_contq()
CLEANUP_WITH_CONNECTION_LOST_WITH_BOTHQ = \
    CLEANUP_WITH_CONNECTION_LOST.clone_with_bothq()

START_CLEANUP_WAITING = SyncDaemonState(
    'START_CLEANUP_WAITING',
    'start doing cleanup; network dropped',
    'CLEANUP_WAITING_WITH_NETWORK', 'CLEANUP_WAITING',
    'CLEANUP_WAITING', 'CLEANUP',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WAITING',
    SYS_META_QUEUE_WAITING='CLEANUP_WAITING_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WAITING_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WAITING',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WAITING',
    enter=lambda m: m.action_q.cleanup(),
    )
START_CLEANUP_WAITING_WITH_CONTQ = START_CLEANUP_WAITING.clone_with_contq()
# START_CLEANUP_WAITING_WITH_BOTHQ = START_CLEANUP_WAITING.clone_with_bothq()
CLEANUP_WAITING = START_CLEANUP_WAITING.clone_with_no_start(
    'doing cleanup; network dropped',
    SYS_CLEANUP_FINISHED='START_STANDOFF_WAITING')
CLEANUP_WAITING_WITH_METAQ = CLEANUP_WAITING.clone_with_metaq()
CLEANUP_WAITING_WITH_CONTQ = CLEANUP_WAITING.clone_with_contq()
CLEANUP_WAITING_WITH_BOTHQ = CLEANUP_WAITING.clone_with_bothq()


CLEANUP_WITH_NETWORK = SyncDaemonState(
    'CLEANUP_WITH_NETWORK',
    'doing cleanup; net hiccup, user asked for shutdown',
    'CLEANUP_WITH_NETWORK', 'CLEANUP_WAITING_WITH_NETWORK',
    'CLEANUP', 'CLEANUP_WITH_NETWORK',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WITH_NETWORK',
    SYS_CLEANUP_FINISHED='START_STANDOFF_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WITH_NETWORK',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WITH_NETWORK',
    )
CLEANUP_WITH_NETWORK_WITH_METAQ = CLEANUP_WITH_NETWORK.clone_with_metaq()
CLEANUP_WITH_NETWORK_WITH_CONTQ = CLEANUP_WITH_NETWORK.clone_with_contq()
CLEANUP_WITH_NETWORK_WITH_BOTHQ = CLEANUP_WITH_NETWORK.clone_with_bothq()

CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST = SyncDaemonState(
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    'doing cleanup; net hiccup, user asked to shutdown, '
        'already got SYS_CONNECTION_LOST',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_FINISHED='READY_WITH_NETWORK',
    SYS_META_QUEUE_WAITING=\
        'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING=\
        'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
)
CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_METAQ = \
    CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_metaq()
CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ = \
    CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_contq()
CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_BOTHQ = \
    CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_bothq()

CLEANUP_WAITING_WITH_NETWORK = SyncDaemonState(
    'CLEANUP_WAITING_WITH_NETWORK',
    'doing cleanup; network hiccup',
    'CLEANUP_WAITING_WITH_NETWORK', 'CLEANUP_WAITING_WITH_NETWORK',
    'CLEANUP_WAITING', 'CLEANUP_WITH_NETWORK',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WAITING_WITH_NETWORK',
    SYS_CLEANUP_FINISHED='START_STANDOFF_WAITING_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='CLEANUP_WAITING_WITH_NETWORK_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WAITING_WITH_NETWORK_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WAITING_WITH_NETWORK',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WAITING_WITH_NETWORK',
    )
CLEANUP_WAITING_WITH_NETWORK_WITH_METAQ = \
    CLEANUP_WAITING_WITH_NETWORK.clone_with_metaq()
CLEANUP_WAITING_WITH_NETWORK_WITH_CONTQ = \
    CLEANUP_WAITING_WITH_NETWORK.clone_with_contq()
CLEANUP_WAITING_WITH_NETWORK_WITH_BOTHQ = \
    CLEANUP_WAITING_WITH_NETWORK.clone_with_bothq()

START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST = SyncDaemonState(
    'START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    'start doing cleanup; network hiccup, already got SYS_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_META_QUEUE_WAITING=\
        'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING=\
        'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    enter=lambda m: m.action_q.cleanup(),
)
START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ = \
    START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_contq()
CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST = \
    START_CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_no_start(
    'doing cleanup; network hiccup, already got SYS_CONNECTION_LOST',
    SYS_CLEANUP_FINISHED='START_CONNECTING')
CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_METAQ = \
    CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_metaq()
CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ = \
    CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_contq()
CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_BOTHQ = \
    CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST.clone_with_bothq()

CLEANUP_WAITING_WITH_CONNECTION_LOST = SyncDaemonState(
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    'doing cleanup; network down, connection already lost',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    'CLEANUP_WITH_CONNECTION_LOST',
    'CLEANUP_WAITING_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WAITING_WITH_CONNECTION_LOST',
    SYS_CLEANUP_FINISHED='READY_WAITING',
    SYS_META_QUEUE_WAITING='CLEANUP_WAITING_WITH_CONNECTION_LOST_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WAITING_WITH_CONNECTION_LOST_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WAITING_WITH_CONNECTION_LOST',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WAITING_WITH_CONNECTION_LOST',
    )
CLEANUP_WAITING_WITH_CONNECTION_LOST_WITH_METAQ = \
    CLEANUP_WAITING_WITH_CONNECTION_LOST.clone_with_metaq()
CLEANUP_WAITING_WITH_CONNECTION_LOST_WITH_CONTQ = \
    CLEANUP_WAITING_WITH_CONNECTION_LOST.clone_with_contq()
CLEANUP_WAITING_WITH_CONNECTION_LOST_WITH_BOTHQ = \
    CLEANUP_WAITING_WITH_CONNECTION_LOST.clone_with_bothq()

# XXX CONNECTED_CLEANUP should use the network to do a "better" cleanup
# XXX the way it stands the cloning methods are useless
START_CONNECTED_CLEANUP = SyncDaemonState(
    'START_CONNECTED_CLEANUP',
    'start doing cleanup using the network'
        ' (not yet implemented; actually starts a plain cleanup)',
    'CLEANUP_WITH_NETWORK', 'CLEANUP_WAITING_WITH_NETWORK',
    'CLEANUP', 'CLEANUP_WITH_NETWORK',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST',
    SYS_CLEANUP_STARTED='CLEANUP_WITH_NETWORK',
    SYS_CLEANUP_FINISHED='START_STANDOFF_WITH_NETWORK',
    SYS_META_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_METAQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_CONTQ',
    SYS_META_QUEUE_DONE='CLEANUP_WITH_NETWORK',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WITH_NETWORK',
    enter=lambda m: m.action_q.cleanup(),
    )

START_CONNECTED_CLEANUP_WITH_CONTQ = SyncDaemonState(
    'START_CONNECTED_CLEANUP_WITH_CONTQ',
    'start doing cleanup using the network'
        ' (not yet implemented; actually starts a plain cleanup)',
    'CLEANUP_WITH_NETWORK_WITH_CONTQ',
    'CLEANUP_WAITING_WITH_NETWORK_WITH_CONTQ',
    'CLEANUP_WITH_CONTQ', 'CLEANUP_WITH_NETWORK_WITH_CONTQ',
    'CLEANUP_WITH_NETWORK_WITH_CONNECTION_LOST_WITH_CONTQ',
    SYS_CLEANUP_STARTED='CLEANUP_WITH_NETWORK_WITH_CONTQ',
    SYS_CLEANUP_FINISHED='START_STANDOFF_WITH_NETWORK_WITH_CONTQ',
    SYS_META_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_BOTHQ',
    SYS_META_QUEUE_DONE='CLEANUP_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_WAITING='CLEANUP_WITH_NETWORK_WITH_CONTQ',
    SYS_CONTENT_QUEUE_DONE='CLEANUP_WITH_NETWORK',
    enter=lambda m: m.action_q.cleanup(),
    )
