# ubuntuone.syncdaemon.dbus_interface - DBus Interface
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" DBUS interface module """
from itertools import groupby, chain

import dbus.service

from ubuntuone.syncdaemon.event_queue import EVENTS
from ubuntuone.syncdaemon.interfaces import IMarker
from ubuntuone.syncdaemon.config import get_user_config
import logging

# Disable the "Invalid Name" check here, as we have lots of DBus style names
# pylint: disable-msg=C0103

DBUS_IFACE_NAME = 'com.ubuntuone.SyncDaemon'
DBUS_IFACE_SYNC_NAME = 'com.ubuntuone.SyncDaemon.SyncDaemon'
DBUS_IFACE_STATUS_NAME = DBUS_IFACE_NAME+'.Status'
DBUS_IFACE_EVENTS_NAME = DBUS_IFACE_NAME+'.Events'
DBUS_IFACE_FS_NAME = DBUS_IFACE_NAME+'.FileSystem'
DBUS_IFACE_SHARES_NAME = DBUS_IFACE_NAME+'.Shares'
DBUS_IFACE_CONFIG_NAME = DBUS_IFACE_NAME+'.Config'

# NetworkManager State constants
NM_STATE_UNKNOWN = 0
NM_STATE_ASLEEP = 1
NM_STATE_CONNECTING = 2
NM_STATE_CONNECTED = 3
NM_STATE_DISCONNECTED = 4
# NM state -> events mapping
NM_STATE_EVENTS = {NM_STATE_CONNECTED: 'SYS_NET_CONNECTED',
                   NM_STATE_DISCONNECTED: 'SYS_NET_DISCONNECTED'}


logger = logging.getLogger("ubuntuone.SyncDaemon.DBus")

def get_classname(thing):
    """
    Get the clasname of the thing.
    If we could forget 2.5, we could do attrgetter('__class__.__name__')
    Alas, we can't forget it yet.
    """
    return thing.__class__.__name__


class DBusExposedObject(dbus.service.Object):
    """ Base class that provides some helper methods to
    DBus exposed objects.
    """

    def __init__(self, bus_name, path):
        """ creates the instance. """
        dbus.service.Object.__init__(self, bus_name=bus_name,
                                     object_path=self.path)

    def bool_str(self, value):
        """ return a string value that can be converted back to bool. """
        return 'True' if value else ''

    @dbus.service.signal(DBUS_IFACE_SYNC_NAME, signature='sa{ss}')
    def SignalError(self, signal, extra_args):
        """ An error ocurred while trying to emit a signal. """
        pass

    def emit_signal_error(self, signal, extra_args):
        """ emit's a Error signal. """
        self.SignalError(signal, extra_args)


class Status(DBusExposedObject):
    """ Represent the status of the syncdaemon """

    def __init__(self, bus_name, dbus_iface):
        """ Creates the instance.

        @param bus: the BusName of this DBusExposedObject.
        """
        self.dbus_iface = dbus_iface
        self.action_queue = dbus_iface.action_queue
        self.fs_manager = dbus_iface.fs_manager
        self.path = '/status'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.method(DBUS_IFACE_STATUS_NAME,
                         in_signature='', out_signature='a{ss}')
    def current_status(self):
        """ return the current status of the system, one of: local_rescan,
        offline, trying_to_connect, server_rescan or online.
        """
        logger.debug('called current_status')
        state = self.dbus_iface.main.state.state
        self.emit_status_changed(state)
        state_dict = {'name':state.name,
                      'description':state.description,
                      'is_error':self.bool_str(state.is_error),
                      'is_connected':self.bool_str(state.is_connected),
                      'is_online':self.bool_str(state.is_online)}
        return state_dict

    @dbus.service.method(DBUS_IFACE_STATUS_NAME, out_signature='aa{ss}')
    def current_downloads(self):
        """ return list of files with a download in progress. """
        logger.debug('called current_downloads')
        current_downloads = []
        for download in self.action_queue.downloading:
            try:
                relpath = self.fs_manager.get_by_node_id(*download).path
            except KeyError:
                # the path has gone away! ignore this download
                continue
            path = self.fs_manager.get_abspath(download[0], relpath)
            info = self.action_queue.downloading[download]
            entry = {'path':path,
                     'share_id':download[0],
                     'node_id':download[1],
                     'n_bytes_read':str(info.get('n_bytes_read', 0))}
            try:
                entry['deflated_size'] = str(info['deflated_size'])
                # the idea is to do nothing, pylint: disable-msg=W0704
            except KeyError:
                # ignore the deflated_size key
                pass
            current_downloads.append(entry)
        return current_downloads

    @dbus.service.method(DBUS_IFACE_STATUS_NAME, out_signature='aa{ss}')
    def waiting_content(self):
        """
        returns a list of files that are waiting to be up- or downloaded
        """
        logger.debug('called waiting_content')
        waiting_content = []
        for cmd in self.action_queue.content_queue.waiting:
            try:
                if IMarker.providedBy(cmd.node_id):
                    # it's a marker! so it's the mdid :)
                    relpath = self.fs_manager.get_by_mdid(cmd.node_id).path
                else:
                    relpath = self.fs_manager.get_by_node_id(cmd.share_id,
                                                             cmd.node_id).path
            except KeyError:
                pass
            else:
                path=self.fs_manager.get_abspath(cmd.share_id, relpath)
                waiting_content.append(dict(path=path,
                                            share=cmd.share_id,
                                            node=cmd.node_id,
                                            operation=cmd.__class__.__name__))
        return waiting_content

    @dbus.service.method(DBUS_IFACE_STATUS_NAME, in_signature='ss')
    def schedule_next(self, share_id, node_id):
        """
        Make the command on the given share and node be next in the
        queue of waiting commands.
        """
        logger.debug('called schedule_next')
        self.action_queue.content_queue.schedule_next(share_id, node_id)

    @dbus.service.method(DBUS_IFACE_STATUS_NAME, out_signature='aa{ss}')
    def current_uploads(self):
        """ return a list of files with a upload in progress """
        logger.debug('called current_uploads')
        current_uploads = []
        for upload in self.action_queue.uploading:
            share_id, node_id = upload
            if IMarker.providedBy(node_id):
                continue
            try:
                relpath = self.fs_manager.get_by_node_id(share_id, node_id).path
            except KeyError:
                # the path has gone away! ignore this upload
                continue
            path = self.fs_manager.get_abspath(share_id, relpath)
            info = self.action_queue.uploading[upload]
            entry = {'path':path,
                     'share_id':upload[0],
                     'node_id':upload[1],
                     'n_bytes_written':str(info.get('n_bytes_written', 0))}
            try:
                entry['deflated_size'] = str(info['deflated_size'])
                # the idea is to do nothing, pylint: disable-msg=W0704
            except KeyError:
                # ignore the deflated_size key
                pass
            current_uploads.append(entry)
        return current_uploads

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME)
    def DownloadStarted(self, path):
        """ Fire a D-BUS signal, notifying a download has started.  """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME,
                         signature='sa{ss}')
    def DownloadFinished(self, path, info):
        """ Fire a D-BUS signal, notifying a download has finished.  """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME)
    def UploadStarted(self, path):
        """ Fire a D-BUS signal, notifying an upload has started.  """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME,
                         signature='sa{ss}')
    def UploadFinished(self, path, info):
        """ Fire a D-BUS signal, notifying an upload has finished.  """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME)
    def StatusChanged(self, status):
        """ Fire a D-BUS signal, notifying that the status of the
        system changed.
        """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME, signature='a{ss}')
    def AccountChanged(self, account_info):
        """ Fire a D-BUS signal, notifying that account information
        has changed.
        """
        pass

    @dbus.service.signal(DBUS_IFACE_STATUS_NAME, signature='a{sa{ss}}')
    def ContentQueueChanged(self, message):
        """
        Fire a D-BUS signal, notifying that the content queue has
        changed.
        """

    def emit_content_queue_changed(self, head, waiting):
        """
        Emits the signal.
        """
        if head is None or getattr(head, 'node_id', None) is None:
            head_path = ''
            head_cmd = ''
        else:
            waiting = chain((head,), waiting)
            head_cmd = get_classname(head)
            try:
                if IMarker.providedBy(head.node_id):
                    # it's a marker! so it's the mdid :)
                    relpath = self.fs_manager.get_by_mdid(head.node_id).path
                else:
                    relpath = self.fs_manager.get_by_node_id(head.share_id,
                                                             head.node_id).path
            except KeyError:
                head_path = ''
            else:
                head_path = self.fs_manager.get_abspath(head.share_id, relpath)
        message = dict(
            (k, {'size': '0', 'count': str(len(tuple(v)))})
            for (k, v) in groupby(sorted(map(get_classname, waiting))))
        message['head'] = {'path': head_path,
                           'command': head_cmd,
                           'size': '0'}
        self.ContentQueueChanged(message)

    def emit_status_changed(self, state):
        """ Emits the signal """
        state_dict = {'name':state.name,
                      'description':self.bool_str(state.description),
                      'is_error':self.bool_str(state.is_error),
                      'is_connected':self.bool_str(state.is_connected),
                      'is_online':self.bool_str(state.is_online)}
        self.StatusChanged(state_dict)

    def emit_download_started(self, download):
        """ Emits the signal """
        self.DownloadStarted(download)

    def emit_download_finished(self, download, **info):
        """ Emits the signal """
        for k, v in info.copy().items():
            info[str(k)] = str(v)
        self.DownloadFinished(download, info)

    def emit_upload_started(self, upload):
        """ Emits the signal """
        self.UploadStarted(upload)

    def emit_upload_finished(self, upload, **info):
        """ Emits the signal """
        for k, v in info.copy().items():
            info[str(k)] = str(v)
        self.UploadFinished(upload, info)

    def emit_account_changed(self, account_info):
        """ Emits the signal """
        info_dict = {'purchased_bytes': unicode(account_info.purchased_bytes)}
        self.AccountChanged(info_dict)


class Events(DBusExposedObject):
    """ The events of the system translated to D-BUS signals """

    def __init__(self, bus_name, event_queue):
        """ Creates the instance.

        @param bus: the BusName of this DBusExposedObject.
        """
        self.event_queue = event_queue
        self.path = '/events'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.signal(DBUS_IFACE_EVENTS_NAME,
                         signature='a{ss}')
    def Event(self, event_dict):
        """ Fire a D-BUS signal, notifying an event. """
        pass

    def emit_event(self, event):
        """ Emits the signal """
        event_dict = {}
        for key, value in event.iteritems():
            event_dict[str(key)] = str(value)
        self.Event(event_dict)

    @dbus.service.method(DBUS_IFACE_EVENTS_NAME, in_signature='sas')
    def push_event(self, event_name, args):
        """ Push a event to the event queue """
        logger.debug('push_event: %r with %r', event_name, args)
        str_args = []
        for arg in args:
            str_args.append(str(arg))
        self.event_queue.push(str(event_name), *str_args)


class EventListener(object):
    """ An Event Queue Listener """

    def __init__(self, dbus_iface):
        """ created the instance """
        self.dbus_iface = dbus_iface

    def _fire_event(self, event_dict):
        """ Calls emit_event in a reactor thread. """
        self.dbus_iface.events.emit_event(event_dict)

    def handle_default(self, event_name, *args, **kwargs):
        """ handle all events """
        if not self.dbus_iface.send_events:
            return
        event_dict = {'event_name' : event_name, }
        event_args = list(EVENTS[event_name])
        event_dict.update(kwargs)
        for key in set(event_args).intersection(kwargs.keys()):
            event_args.pop(event_args.index(key))
        for i in xrange(0, len(event_args)):
            event_dict[event_args[i]] = args[i]
        event_dict.update(kwargs)
        self._fire_event(event_dict)

    def handle_AQ_DOWNLOAD_STARTED(self, share_id, node_id, server_hash):
        """ handle AQ_DOWNLOAD_STARTED """
        self.handle_default('AQ_DOWNLOAD_STARTED', share_id, node_id,
                            server_hash)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_download_started(path)
        except KeyError, e:
            args = dict(message='The md is gone before sending '
                        'DownloadStarted signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id))
            self.dbus_iface.status.emit_signal_error('DownloadStarted', args)

    def handle_AQ_DOWNLOAD_FINISHED(self, share_id, node_id, server_hash):
        """ handle AQ_DOWNLOAD_FINISHED """
        self.handle_default('AQ_DOWNLOAD_FINISHED', share_id,
                            node_id, server_hash)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_download_finished(path)
        except KeyError, e:
            # file is gone before we got this
            args = dict(message='The md is gone before sending '
                        'DownloadFinished signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id))
            self.dbus_iface.status.emit_signal_error('DownloadFinished', args)

    def handle_AQ_DOWNLOAD_CANCELLED(self, share_id, node_id, server_hash):
        """ handle AQ_DOWNLOAD_CANCELLED """
        self.handle_AQ_DOWNLOAD_ERROR(share_id, node_id, server_hash,
                                      'CANCELLED', 'AQ_DOWNLOAD_CANCELLED')

    def handle_AQ_DOWNLOAD_ERROR(self, share_id, node_id, server_hash, error,
                                 event='AQ_DOWNLOAD_ERROR'):
        """ handle AQ_DOWNLOAD_ERROR """
        self.handle_default(event, share_id, node_id,
                            server_hash, error)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_download_finished(path, error=error)
        except KeyError, e:
            # file is gone before we got this
            args = dict(message='The md is gone before sending '
                        'DownloadFinished signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id),
                        download_error=str(error))
            self.dbus_iface.status.emit_signal_error('DownloadFinished', args)

    def handle_AQ_UPLOAD_STARTED(self, share_id, node_id, hash):
        """ handle AQ_UPLOAD_STARTED """
        self.handle_default('AQ_UPLOAD_STARTED', share_id, node_id, hash)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_upload_started(path)
        except KeyError, e:
            args = dict(message='The md is gone before sending '
                        'UploadStarted signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id))
            self.dbus_iface.status.emit_signal_error('UploadStarted', args)

    def handle_AQ_UPLOAD_FINISHED(self, share_id, node_id, hash):
        """ handle AQ_UPLOAD_FINISHED """
        self.handle_default('AQ_UPLOAD_FINISHED', share_id, node_id, hash)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_upload_finished(path)
        except KeyError, e:
            # file is gone before we got this
            args = dict(message='The metadata is gone before sending '
                        'UploadFinished signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id))
            self.dbus_iface.status.emit_signal_error('UploadFinished', args)

    def handle_SV_ACCOUNT_CHANGED(self, account_info):
        """ handle SV_ACCOUNT_CHANGED """
        self.handle_default('SV_ACCOUNT_CHANGED', account_info)
        self.dbus_iface.status.emit_account_changed(account_info)

    def handle_AQ_UPLOAD_ERROR(self, share_id, node_id, error, hash):
        """ handle AQ_UPLOAD_ERROR """
        self.handle_default('AQ_UPLOAD_ERROR', share_id, node_id, error, hash)
        try:
            mdobj = self.dbus_iface.fs_manager.get_by_node_id(share_id, node_id)
            if mdobj.is_dir:
                return
            path = self.dbus_iface.fs_manager.get_abspath(share_id, mdobj.path)
            self.dbus_iface.status.emit_upload_finished(path, error=error)
        except KeyError, e:
            # file is gone before we got this
            args = dict(message='The metadata is gone before sending '
                        'UploadFinished signal',
                        error=str(e),
                        share_id=str(share_id),
                        node_id=str(node_id),
                        upload_error=str(error))
            self.dbus_iface.status.emit_signal_error('UploadFinished', args)

    def handle_SYS_STATE_CHANGED(self, state):
        """ handle SYS_STATE_CHANGED """
        self.handle_default('SYS_STATE_CHANGED', state)
        self.dbus_iface.status.emit_status_changed(state)

    def handle_SV_SHARE_CHANGED(self, message, share):
        """ handle SV_SHARE_CHANGED event, emit's ShareChanged signal. """
        self.handle_default('SV_SHARE_CHANGED', message, share)
        self.dbus_iface.shares.emit_share_changed(message, share)

    def handle_SV_FREE_SPACE(self, share_id, free_bytes):
        """ handle SV_FREE_SPACE event, emit ShareChanged signal. """
        self.handle_default('SV_FREE_SPACE', share_id, free_bytes)
        self.dbus_iface.shares.emit_free_space(share_id, free_bytes)

    def handle_AQ_CREATE_SHARE_OK(self, share_id, marker):
        """ handle AQ_CREATE_SHARE_OK event, emit's ShareCreated signal. """
        self.handle_default('AQ_CREATE_SHARE_OK', share_id, marker)
        share = self.dbus_iface.volume_manager.shared.get(share_id)
        share_dict = {}
        if share:
            # pylint: disable-msg=W0212
            share_dict.update(self.dbus_iface.shares._get_share_dict(share))
        else:
            share_dict.update(dict(share_id=share_id))
        self.dbus_iface.shares.emit_share_created(share_dict)

    def handle_AQ_CREATE_SHARE_ERROR(self, marker, error):
        """ handle AQ_CREATE_SHARE_ERROR event, emit's ShareCreateError signal.
        """
        self.handle_default('AQ_CREATE_SHARE_ERROR', marker, error)
        self.dbus_iface.shares.emit_share_create_error(dict(marker=marker),
                                                       error)

    def handle_AQ_ANSWER_SHARE_OK(self, share_id, answer):
        """ handle AQ_ANSWER_SHARE_OK event, emit's ShareAnswerOk signal. """
        self.handle_default('AQ_ANSWER_SHARE_OK', share_id, answer)
        self.dbus_iface.shares.emit_share_answer_response(share_id, answer)

    def handle_AQ_ANSWER_SHARE_ERROR(self, share_id, answer, error):
        """ handle AQ_ANSWER_SHARE_ERROR event, emit's ShareAnswerError signal.
        """
        self.handle_default('AQ_ANSWER_SHARE_ERROR', share_id, answer, error)
        self.dbus_iface.shares.emit_share_answer_response(share_id, answer,
                                                          error)


class SyncDaemon(DBusExposedObject):
    """ The Daemon dbus interface. """

    def __init__(self, bus_name, dbus_iface):
        """ Creates the instance.

        @param bus: the BusName of this DBusExposedObject.
        """
        self.dbus_iface = dbus_iface
        self.path = '/'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='', out_signature='')
    def connect(self):
        """ Connect to the server. """
        logger.debug('connect requetsed')
        self.dbus_iface.connect()

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='', out_signature='')
    def disconnect(self):
        """ Disconnect from the server. """
        logger.debug('disconnect requetsed')
        self.dbus_iface.disconnect()

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='', out_signature='s')
    def get_rootdir(self):
        """ Returns the root dir/mount point. """
        logger.debug('called get_rootdir')
        return self.dbus_iface.main.get_rootdir()

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='d', out_signature='b',
                         async_callbacks=('reply_handler', 'error_handler'))
    def wait_for_nirvana(self, last_event_interval,
                         reply_handler=None, error_handler=None):
        """ call the reply handler when there are no more
        events or transfers.
        """
        logger.debug('called wait_for_nirvana')
        d = self.dbus_iface.main.wait_for_nirvana(last_event_interval)
        d.addCallbacks(reply_handler, error_handler)
        return d

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='s', out_signature='')
    def query_by_path(self, path):
        """  Request a query of the node identified by path. """
        logger.debug('query_by_path: %r', path)
        mdobj = self.dbus_iface.fs_manager.get_by_path(path.encode('utf-8'))
        items = [(mdobj.share_id, mdobj.node_id, "")]
        self.dbus_iface.action_queue.query(items)

    @dbus.service.method(DBUS_IFACE_SYNC_NAME,
                         in_signature='', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def quit(self, reply_handler=None, error_handler=None):
        """ shutdown the syncdaemon. """
        logger.debug('Quit requested')
        if reply_handler:
            reply_handler()
        self.dbus_iface.quit()


class FileSystem(DBusExposedObject):
    """ A dbus interface to the FileSystem Manager. """

    def __init__(self, bus_name, fs_manager):
        """ Creates the instance. """
        self.fs_manager = fs_manager
        self.path = '/filesystem'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.method(DBUS_IFACE_FS_NAME,
                         in_signature='s', out_signature='a{ss}')
    def get_metadata(self, path):
        """ returns the dict with the attributes of the metadata
        for the specified path.
        """
        logger.debug('get_metadata: %r', path)
        mdobj = self.fs_manager.get_by_path(path.encode('utf-8'))
        md_dict = {}
        for k, v in mdobj.__dict__.items():
            if k == 'info':
                continue
            md_dict[str(k)] = str(v)
        if mdobj.__dict__.get('info', None):
            for k, v in mdobj.info.__dict__.items():
                md_dict['info_'+str(k)] = str(v)
        return md_dict


class Shares(DBusExposedObject):
    """ A dbus interface to interact wiht shares """

    def __init__(self, bus_name, fs_manager, volume_manager):
        """ Creates the instance. """
        self.fs_manager = fs_manager
        self.vm = volume_manager
        self.path = '/shares'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='', out_signature='aa{ss}')
    def get_shares(self):
        """ returns a list of dicts, each dict represents a share """
        logger.debug('called get_shares')
        shares = []
        for share_id, share in self.vm.shares.items():
            if share_id == '':
                continue
            share_dict = self._get_share_dict(share)
            shares.append(share_dict)
        return shares

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='s', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def accept_share(self, share_id, reply_handler=None, error_handler=None):
        """ Accepts a share, a ShareAnswerOk|Error signal will be fired in the
        future as a success/failure indicator.
        """
        logger.debug('accept_share: %r', share_id)
        if str(share_id) in self.vm.shares:
            self.vm.accept_share(str(share_id), True)
            reply_handler()
        else:
            error_handler(ValueError("The share with id: %s don't exists" % \
                                     str(share_id)))

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='s', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def reject_share(self, share_id, reply_handler=None, error_handler=None):
        """ Rejects a share. """
        logger.debug('reject_share: %r', share_id)
        if str(share_id) in self.vm.shares:
            self.vm.accept_share(str(share_id), False)
            reply_handler()
        else:
            error_handler(ValueError("The share with id: %s don't exists" % \
                                     str(share_id)))

    @dbus.service.signal(DBUS_IFACE_SHARES_NAME,
                         signature='a{ss}')
    def ShareChanged(self, share_dict):
        """ A share changed, share_dict contains all the share attributes. """
        pass

    @dbus.service.signal(DBUS_IFACE_SHARES_NAME,
                         signature='a{ss}')
    def ShareDeleted(self, share_dict):
        """ A share was deleted, share_dict contains all available
        share attributes. """
        pass

    def emit_share_changed(self, message, share):
        """ emits ShareChanged or ShareDeleted signal for the share
        notification.
        """
        if message == 'deleted':
            self.ShareDeleted(self._get_share_dict(share))
        elif message == 'changed':
            share = self.vm.shares[share.share_id]
            self.ShareChanged(self._get_share_dict(share))

    def emit_free_space(self, share_id, free_bytes):
        """ emits ShareChanged when free space changes """
        if share_id in self.vm.shares:
            share = self.vm.shares[share_id]
            share_dict = self._get_share_dict(share)
            share_dict['free_bytes'] = unicode(free_bytes)
            self.ShareChanged(share_dict)

    def _get_share_dict(self, share):
        """ get a dict with all the attributes of: share. """
        share_dict = share.__dict__.copy()
        for k, v in share_dict.items():
            if v is None:
                share_dict[unicode(k)] = ''
            elif k == 'path':
                share_dict[unicode(k)] = v.decode('utf-8')
            else:
                share_dict[unicode(k)] = unicode(v)
        return share_dict

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='ssss', out_signature='')
    def create_share(self, path, username, name, access_level):
        """ Share a subtree to the user identified by username.

        @param path: that path to share (the root of the subtree)
        @param username: the username to offer the share to
        @param name: the name of the share
        @param access_level: 'View' or 'Modify'
        """
        logger.debug('create share: %r, %r, %r, %r',
                     path, username, name, access_level)
        path = path.encode("utf8")
        username = unicode(username)
        name = unicode(name)
        access_level = str(access_level)
        try:
            self.fs_manager.get_by_path(path)
        except KeyError:
            raise ValueError("path '%r' does not exist" % path)
        self.vm.create_share(path, username, name, access_level)

    @dbus.service.signal(DBUS_IFACE_SHARES_NAME,
                         signature='a{ss}')
    def ShareCreated(self, share_info):
        """ The requested share was succesfully created. """
        pass

    @dbus.service.signal(DBUS_IFACE_SHARES_NAME,
                         signature='a{ss}s')
    def ShareCreateError(self, share_info, error):
        """ An error ocurred while creating the share. """
        pass

    def emit_share_created(self, share_info):
        """ emits ShareCreated signal """
        self.ShareCreated(share_info)

    def emit_share_create_error(self, share_info, error):
        """ emits ShareDeleted signal """
        path = self.fs_manager.get_by_mdid(str(share_info['marker'])).path
        share_info.update(dict(path=path))
        self.ShareCreateError(share_info, error)

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='', out_signature='')
    def refresh_shares(self):
        """ Refresh the share list, requesting it to the server. """
        self.vm.refresh_shares()

    @dbus.service.method(DBUS_IFACE_SHARES_NAME,
                         in_signature='', out_signature='aa{ss}')
    def get_shared(self):
        """ returns a list of dicts, each dict represents a shared share.
        A share might not have the path set, as we might be still fetching the
        nodes from the server. In this cases the path is ''
        """
        logger.debug('called get_shared')
        shares = []
        for share_id, share in self.vm.shared.items():
            if share_id == '':
                continue
            share_dict = self._get_share_dict(share)
            shares.append(share_dict)
        return shares

    @dbus.service.signal(DBUS_IFACE_SHARES_NAME,
                         signature='a{ss}')
    def ShareAnswerResponse(self, answer_info):
        """The answer to share was succesfull"""
        pass

    def emit_share_answer_response(self, share_id, answer, error=None):
        """ emits ShareCreated signal """
        answer_info = dict(share_id=share_id, answer=answer)
        if error:
            answer_info['error'] = error
        self.ShareAnswerResponse(answer_info)


class Config(DBusExposedObject):
    """ The Syncdaemon config/settings dbus interface. """

    def __init__(self, bus_name, dbus_iface):
        """ Creates the instance.

        @param bus: the BusName of this DBusExposedObject.
        """
        self.dbus_iface = dbus_iface
        self.path = '/config'
        DBusExposedObject.__init__(self, bus_name=bus_name,
                                   path=self.path)

    @dbus.service.method(DBUS_IFACE_CONFIG_NAME,
                         in_signature='', out_signature='a{si}',
                         async_callbacks=('reply_handler', 'error_handler'))
    def get_throttling_limits(self, reply_handler=None, error_handler=None):
        """Get the read/write limit from AQ and return a dict.
        Returns a dict(download=int, upload=int), if int is -1 the value isn't
        configured.
        The values are bytes/second
        """
        try:
            aq = self.dbus_iface.action_queue
            download = -1
            upload = -1
            if aq.readLimit is not None:
                download = aq.readLimit
            if aq.writeLimit is not None:
                upload = aq.writeLimit
            info = dict(download=download,
                        upload=upload)
            if reply_handler:
                reply_handler(info)
            else:
                return info
            # pylint: disable-msg=W0703
        except Exception, e:
            if error_handler:
                error_handler(e)
            else:
                raise

    @dbus.service.method(DBUS_IFACE_CONFIG_NAME,
                         in_signature='ii', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def set_throttling_limits(self, download, upload,
                         reply_handler=None, error_handler=None):
        """Set the read and write limits. The expected values are bytes/sec."""
        try:
            # modify and save the config file
            config = get_user_config()
            config.set_throttling_read_limit(download)
            config.set_throttling_write_limit(upload)
            config.save()
            # modify AQ settings
            aq = self.dbus_iface.action_queue
            if download == -1:
                download = None
            if upload == -1:
                upload = None
            aq.readLimit = download
            aq.writeLimit = upload
            if reply_handler:
                reply_handler()
            # pylint: disable-msg=W0703
        except Exception, e:
            if error_handler:
                error_handler(e)
            else:
                raise

    @dbus.service.method(DBUS_IFACE_CONFIG_NAME,
                         in_signature='', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def enable_bandwidth_throttling(self, reply_handler=None,
                                    error_handler=None):
        """Enable bandwidth throttling."""
        try:
            # modify and save the config file
            config = get_user_config()
            config.set_throttling(True)
            config.save()
            # modify AQ settings
            self.dbus_iface.action_queue.enable_throttling(True)
            if reply_handler:
                reply_handler()
            # pylint: disable-msg=W0703
        except Exception, e:
            if error_handler:
                error_handler(e)
            else:
                raise

    @dbus.service.method(DBUS_IFACE_CONFIG_NAME,
                         in_signature='', out_signature='',
                         async_callbacks=('reply_handler', 'error_handler'))
    def disable_bandwidth_throttling(self, reply_handler=None,
                                     error_handler=None):
        """Disable bandwidth throttling."""
        try:
            # modify and save the config file
            config = get_user_config()
            config.set_throttling(True)
            config.save()
            # modify AQ settings
            self.dbus_iface.action_queue.enable_throttling(False)
            if reply_handler:
                reply_handler()
            # pylint: disable-msg=W0703
        except Exception, e:
            if error_handler:
                error_handler(e)
            else:
                raise

    @dbus.service.method(DBUS_IFACE_CONFIG_NAME,
                         in_signature='', out_signature='b',
                         async_callbacks=('reply_handler', 'error_handler'))
    def bandwidth_throttling_enabled(self, reply_handler=None,
                                     error_handler=None):
        """Returns True (actually 1) if bandwidth throttling is enabled and
        False (0) otherwise.
        """
        enabled = self.dbus_iface.action_queue.throttling
        if reply_handler:
            reply_handler(enabled)
        else:
            return enabled


class DBusInterface(object):
    """ Holder of all DBus exposed objects """
    test = False

    def __init__(self, bus, main, system_bus=None, send_events=False):
        """ Create the instance and add the exposed object to the
        specified bus.
        """
        self.bus = bus
        self.main = main
        self.event_queue = main.event_q
        self.action_queue = main.action_q
        self.fs_manager = main.fs
        self.volume_manager = main.vm
        self.send_events = send_events
        self.busName = dbus.service.BusName(DBUS_IFACE_NAME, bus=self.bus)
        self.status = Status(self.busName, self)
        self.events = Events(self.busName, self.event_queue)
        self.event_listener = EventListener(self)
        self.sync = SyncDaemon(self.busName, self)
        self.fs = FileSystem(self.busName, self.fs_manager)
        self.shares = Shares(self.busName, self.fs_manager,
                             self.volume_manager)
        self.config = Config(self.busName, self)
        if system_bus is None and not DBusInterface.test:
            logger.debug('using the real system bus')
            self.system_bus = self.bus.get_system()
        elif system_bus is None and DBusInterface.test:
            # this is just for the case when test_sync instatiate Main for
            # running it's tests as pqm don't have a system bus running
            logger.debug('using the session bus as system bus')
            self.system_bus = self.bus
        else:
            self.system_bus = system_bus
        if self.event_queue:
            self.event_queue.subscribe(self.event_listener)
            # on initialization, fake a SYS_NET_CONNECTED if appropriate
            if DBusInterface.test:
                # testing under sync; just do it
                logger.debug('using the fake NetworkManager')
                self.connection_state_changed(NM_STATE_CONNECTED)
            else:
                def error_handler(error):
                    """
                    Handle errors from NM
                    """
                    logger.error(
                        "Error while getting the NetworkManager state %s",
                        error)
                    # If we get an error back from NetworkManager, we should
                    # just try to connect anyway; it probably means that
                    # NetworkManager is down or broken or something.
                    self.connection_state_changed(NM_STATE_CONNECTED)
                try:
                    nm = self.system_bus.get_object(
                        'org.freedesktop.NetworkManager',
                        '/org/freedesktop/NetworkManager',
                        follow_name_owner_changes=True)
                except dbus.DBusException, e:
                    if e.get_dbus_name() == \
                        'org.freedesktop.DBus.Error.ServiceUnknown':
                        # NetworkManager isn't running.
                        logger.warn("Unable to connect to NetworkManager. "
                                      "Assuming we have network.")
                        self.connection_state_changed(NM_STATE_CONNECTED)
                    else:
                        raise
                else:
                    nm.Get('org.freedesktop.NetworkManager', 'State',
                           reply_handler=self.connection_state_changed,
                           error_handler=error_handler,
                           dbus_interface='org.freedesktop.DBus.Properties')
        # register a handler to NM StateChanged signal
        self.system_bus.add_signal_receiver(self.connection_state_changed,
                               signal_name='StateChanged',
                               dbus_interface='org.freedesktop.NetworkManager',
                               path='/org/freedesktop/NetworkManager')
        logger.info('DBusInterface initialized.')

    def shutdown(self, with_restart=False):
        """ remove the registered object from the bus and unsubscribe from the
        event queue.
        """
        logger.info('Shuttingdown DBusInterface!')
        self.status.remove_from_connection()
        self.events.remove_from_connection()
        self.sync.remove_from_connection()
        self.fs.remove_from_connection()
        self.shares.remove_from_connection()
        self.config.remove_from_connection()
        self.event_queue.unsubscribe(self.event_listener)
        # remove the NM's StateChanged signal receiver
        self.system_bus.remove_signal_receiver(self.connection_state_changed,
                               signal_name='StateChanged',
                               dbus_interface='org.freedesktop.NetworkManager',
                               path='/org/freedesktop/NetworkManager')
        self.bus.release_name(self.busName.get_name())
        if with_restart:
            # this is what activate_name_owner boils down to, except that
            # activate_name_owner blocks, which is a luxury we can't allow
            # ourselves.
            self.bus.call_async(dbus.bus.BUS_DAEMON_NAME,
                                dbus.bus.BUS_DAEMON_PATH,
                                dbus.bus.BUS_DAEMON_IFACE,
                                'StartServiceByName', 'su',
                                (DBUS_IFACE_NAME, 0),
                                self._restart_reply_handler,
                                self._restart_error_handler)

    def _restart_reply_handler(self, *args):
        """
        This is called by the restart async call.

        It's here to be stepped on from tests; in production we are
        going away and don't really care if the async call works or
        not: there is nothing we can do about it.
        """
    _restart_error_handler = _restart_reply_handler

    def connection_state_changed(self, state):
        """ Push a connection state changed event to the Event Queue. """
        event = NM_STATE_EVENTS.get(state, None)
        if event is not None:
            self.event_queue.push(event)

    def connect(self):
        """ Push the SYS_CONNECT event with the tokens in the keyring. """
        from ubuntuone.syncdaemon.main import NoAccessToken
        try:
            access_token = self.main.get_access_token()
            self.event_queue.push('SYS_CONNECT', access_token)
        except NoAccessToken, e:
            logger.exception("Can't get the auth token")

    def disconnect(self):
        """ Push the SYS_DISCONNECT event. """
        self.event_queue.push('SYS_DISCONNECT')

    def quit(self):
        """ calls Main.quit. """
        logger.debug('Calling Main.quit')
        self.main.quit()
