# -*- coding: utf-8 -*-
#
# Author: Natalia Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the SSO account code."""

import os

# Unable to import 'lazr.restfulclient.*'
# pylint: disable=F0401
from lazr.restfulclient.errors import HTTPError
# pylint: enable=F0401
from twisted.trial.unittest import TestCase

from ubuntu_sso.account import (Account, AuthenticationError, EmailTokenError,
    InvalidEmailError, InvalidPasswordError, NewPasswordError, SERVICE_URL,
    RegistrationError, ResetPasswordTokenError,
    SSO_STATUS_OK, SSO_STATUS_ERROR)
from ubuntu_sso.tests import (APP_NAME, CAPTCHA_ID, CAPTCHA_PATH,
    CAPTCHA_SOLUTION, EMAIL, EMAIL_TOKEN, NAME, PASSWORD, RESET_PASSWORD_TOKEN,
    TOKEN, TOKEN_NAME)


CANT_RESET_PASSWORD_CONTENT = "CanNotResetPassowrdError: " \
                              "Can't reset password for this account"
RESET_TOKEN_INVALID_CONTENT = "AuthToken matching query does not exist."
EMAIL_ALREADY_REGISTERED = 'a@example.com'
STATUS_UNKNOWN = {'status': 'yadda-yadda'}
STATUS_ERROR = {'status': SSO_STATUS_ERROR,
                'errors': {'something': ['Bla', 'Ble']}}
STATUS_OK = {'status': SSO_STATUS_OK}
STATUS_EMAIL_UNKNOWN = {'status': 'yadda-yadda'}
STATUS_EMAIL_ERROR = {'errors': {'email_token': ['Error1', 'Error2']}}
STATUS_EMAIL_OK = {'email': EMAIL}


class FakedCaptchas(object):
    """Fake the captcha generator."""

    def new(self):
        """Return a fix captcha)."""
        return {'image_url': 'file://%s' % CAPTCHA_PATH,
                'captcha_id': CAPTCHA_ID}


class FakedRegistrations(object):
    """Fake the registrations service."""

    def register(self, email, password, displayname,
                 captcha_id, captcha_solution):
        """Fake registration. Return a fix result."""
        if email == EMAIL_ALREADY_REGISTERED:
            return {'status': SSO_STATUS_ERROR,
                    'errors': {'email': 'Email already registered'}}
        elif captcha_id is None and captcha_solution is None:
            return STATUS_UNKNOWN
        elif captcha_id != CAPTCHA_ID or captcha_solution != CAPTCHA_SOLUTION:
            return STATUS_ERROR
        else:
            return STATUS_OK

    def request_password_reset_token(self, email):
        """Fake password reset token. Return a fix result."""
        if email is None:
            return STATUS_UNKNOWN
        elif email != EMAIL:
            raise HTTPError(response=None, content=CANT_RESET_PASSWORD_CONTENT)
        else:
            return STATUS_OK

    def set_new_password(self, email, token, new_password):
        """Fake the setting of new password. Return a fix result."""
        if email is None and token is None and new_password is None:
            return STATUS_UNKNOWN
        elif email != EMAIL or token != RESET_PASSWORD_TOKEN:
            raise HTTPError(response=None, content=RESET_TOKEN_INVALID_CONTENT)
        else:
            return STATUS_OK


class FakedAuthentications(object):
    """Fake the authentications service."""

    def authenticate(self, token_name):
        """Fake authenticate. Return a fix result."""
        if not token_name.startswith(TOKEN_NAME):
            raise HTTPError(response=None, content=None)
        else:
            return TOKEN


class FakedAccounts(object):
    """Fake the accounts service."""

    def __init__(self):
        self.preferred_email = EMAIL

    def validate_email(self, email_token):
        """Fake the email validation. Return a fix result."""
        if email_token is None:
            return STATUS_EMAIL_UNKNOWN
        elif email_token == EMAIL_ALREADY_REGISTERED:
            return {'status': SSO_STATUS_ERROR,
                    'errors': {'email': 'Email already registered'}}
        elif email_token != EMAIL_TOKEN:
            return STATUS_EMAIL_ERROR
        else:
            return STATUS_EMAIL_OK

    # pylint: disable=E0202, C0103

    def me(self):
        """Fake the 'me' information."""
        return {u'username': u'Wh46bKY',
                u'preferred_email': self.preferred_email,
                u'displayname': u'',
                u'unverified_emails': [u'aaaaaa@example.com'],
                u'verified_emails': [],
                u'openid_identifier': u'Wh46bKY'}


class FakedSSOServer(object):
    """Fake an SSO server."""

    def __init__(self, authorizer, service_root):
        self.captchas = FakedCaptchas()
        self.registrations = FakedRegistrations()
        self.authentications = FakedAuthentications()
        self.accounts = FakedAccounts()


class AccountTestCase(TestCase):
    """Test suite for the SSO login processor."""

    def setUp(self):
        """Init."""
        self.processor = Account(sso_service_class=FakedSSOServer)
        self.register_kwargs = dict(email=EMAIL, password=PASSWORD,
                                    displayname=NAME,
                                    captcha_id=CAPTCHA_ID,
                                    captcha_solution=CAPTCHA_SOLUTION)
        self.login_kwargs = dict(email=EMAIL, password=PASSWORD,
                                 token_name=TOKEN_NAME)

    def tearDown(self):
        """Clean up."""
        self.processor = None

    def test_generate_captcha(self):
        """Captcha can be generated."""
        filename = self.mktemp()
        self.addCleanup(lambda: os.remove(filename))
        captcha_id = self.processor.generate_captcha(filename)
        self.assertEqual(CAPTCHA_ID, captcha_id, 'captcha id must be correct.')
        self.assertTrue(os.path.isfile(filename), '%s must exist.' % filename)

        with open(CAPTCHA_PATH) as f:
            expected = f.read()
        with open(filename) as f:
            actual = f.read()
        self.assertEqual(expected, actual, 'captcha image must be correct.')

    def test_register_user_checks_valid_email(self):
        """Email is validated."""
        self.register_kwargs['email'] = 'notavalidemail'
        self.assertRaises(InvalidEmailError,
                          self.processor.register_user, **self.register_kwargs)

    def test_register_user_checks_valid_password(self):
        """Password is validated."""
        self.register_kwargs['password'] = ''
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        # 7 chars, one less than expected
        self.register_kwargs['password'] = 'tesT3it'
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        self.register_kwargs['password'] = 'test3it!'  # no upper case
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        self.register_kwargs['password'] = 'testIt!!'  # no number
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

    # register

    def test_register_user_if_status_ok(self):
        """A user is succesfuy registered into the SSO server."""
        result = self.processor.register_user(**self.register_kwargs)
        self.assertEqual(EMAIL, result, 'registration was successful.')

    def test_register_user_if_status_error(self):
        """Proper error is raised if register fails."""
        self.register_kwargs['captcha_id'] = CAPTCHA_ID * 2  # incorrect
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, STATUS_ERROR['errors'])
            self.assertEqual(val, "\n".join(STATUS_ERROR['errors'][k]))

    def test_register_user_if_status_error_with_string_message(self):
        """Proper error is raised if register fails."""
        self.register_kwargs['email'] = EMAIL_ALREADY_REGISTERED
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, {'email': 'Email already registered'})
            self.assertEqual(val, 'Email already registered')

    def test_register_user_if_status_unknown(self):
        """Proper error is raised if register returns an unknown status."""
        self.register_kwargs['captcha_id'] = None
        self.register_kwargs['captcha_solution'] = None
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        self.assertIn('Received unknown status: %s' % STATUS_UNKNOWN, failure)

    # login

    def test_login_if_http_error(self):
        """Proper error is raised if authentication fails."""
        self.login_kwargs['token_name'] = APP_NAME * 2  # invalid token name
        self.assertRaises(AuthenticationError,
                          self.processor.login, **self.login_kwargs)

    def test_login_if_no_error(self):
        """A user can be succesfully logged in into the SSO service."""
        result = self.processor.login(**self.login_kwargs)
        self.assertEqual(TOKEN, result, 'authentication was successful.')

    # is_validated

    def test_is_validated(self):
        """If preferred email is not None, user is validated."""
        result = self.processor.is_validated(token=TOKEN)
        self.assertTrue(result, 'user must be validated.')

    def test_is_not_validated(self):
        """If preferred email is None, user is not validated."""
        service = FakedSSOServer(None, None)
        service.accounts.preferred_email = None
        result = self.processor.is_validated(sso_service=service,
                                             token=TOKEN)
        self.assertFalse(result, 'user must not be validated.')

    def test_is_not_validated_empty_result(self):
        """If preferred email is None, user is not validated."""
        service = FakedSSOServer(None, None)
        service.accounts.me = lambda: {}
        result = self.processor.is_validated(sso_service=service,
                                             token=TOKEN)
        self.assertFalse(result, 'user must not be validated.')

    # validate_email

    def test_validate_email_if_status_ok(self):
        """A email is succesfuy validated in the SSO server."""
        self.login_kwargs['email_token'] = EMAIL_TOKEN  # valid email token
        result = self.processor.validate_email(**self.login_kwargs)
        self.assertEqual(TOKEN, result, 'email validation was successful.')

    def test_validate_email_if_status_error(self):
        """Proper error is raised if email validation fails."""
        self.login_kwargs['email_token'] = EMAIL_TOKEN * 2  # invalid token
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, STATUS_EMAIL_ERROR['errors'])
            self.assertEqual(val, "\n".join(STATUS_EMAIL_ERROR['errors'][k]))

    def test_validate_email_if_status_error_with_string_message(self):
        """Proper error is raised if register fails."""
        self.login_kwargs['email_token'] = EMAIL_ALREADY_REGISTERED
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, {'email': 'Email already registered'})
            self.assertEqual(val, 'Email already registered')

    def test_validate_email_if_status_unknown(self):
        """Proper error is raised if email validation returns unknown."""
        self.login_kwargs['email_token'] = None
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, failure)

    # reset_password

    def test_request_password_reset_token_if_status_ok(self):
        """A reset password token is succesfuly sent."""
        result = self.processor.request_password_reset_token(email=EMAIL)
        self.assertEqual(EMAIL, result,
                         'password reset token must be successful.')

    def test_request_password_reset_token_if_http_error(self):
        """Proper error is raised if password token request fails."""
        exc = self.assertRaises(ResetPasswordTokenError,
                                self.processor.request_password_reset_token,
                                email=EMAIL * 2)
        self.assertIn(CANT_RESET_PASSWORD_CONTENT, exc)

    def test_request_password_reset_token_if_status_unknown(self):
        """Proper error is raised if password token request returns unknown."""
        exc = self.assertRaises(ResetPasswordTokenError,
                                self.processor.request_password_reset_token,
                                email=None)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, exc)

    def test_set_new_password_if_status_ok(self):
        """A new password is succesfuy set."""
        result = self.processor.set_new_password(email=EMAIL,
                                                 token=RESET_PASSWORD_TOKEN,
                                                 new_password=PASSWORD)
        self.assertEqual(EMAIL, result,
                         'new password must be set successfully.')

    def test_set_new_password_if_http_error(self):
        """Proper error is raised if setting a new password fails."""
        exc = self.assertRaises(NewPasswordError,
                                self.processor.set_new_password,
                                email=EMAIL * 2,
                                token=RESET_PASSWORD_TOKEN * 2,
                                new_password=PASSWORD)
        self.assertIn(RESET_TOKEN_INVALID_CONTENT, exc)

    def test_set_new_password_if_status_unknown(self):
        """Proper error is raised if setting a new password returns unknown."""
        exc = self.assertRaises(NewPasswordError,
                                self.processor.set_new_password,
                                email=None, token=None, new_password=None)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, exc)


class EnvironOverridesTestCase(TestCase):
    """Some URLs can be set from the environment for testing/QA purposes."""

    def test_override_service_url(self):
        """The service url can be set from the env var USSOC_SERVICE_URL."""
        fake_url = 'this is not really a URL'
        old_url = os.environ.get('USSOC_SERVICE_URL')
        os.environ['USSOC_SERVICE_URL'] = fake_url
        try:
            proc = Account(sso_service_class=FakedSSOServer)
            self.assertEqual(proc.service_url, fake_url)
        finally:
            if old_url:
                os.environ['USSOC_SERVICE_URL'] = old_url
            else:
                del os.environ['USSOC_SERVICE_URL']

    def test_no_override_service_url(self):
        """If the environ is unset, the default service url is used."""
        proc = Account(sso_service_class=FakedSSOServer)
        self.assertEqual(proc.service_url, SERVICE_URL)
