# -*- coding: utf-8 -*-
#
# test_networkstate - tests for ubuntu_sso.networkstate
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the network state detection code."""

from ubuntu_sso.networkstate import (NetworkManagerState,
                                     ONLINE, OFFLINE, UNKNOWN)
from ubuntu_sso.networkstate.linux import  (DBUS_UNKNOWN_SERVICE,
                                            NM_STATE_DISCONNECTED,
                                            NM_STATE_CONNECTING,
                                            NM_STATE_CONNECTED)

from mocker import ARGS, KWARGS, ANY, MockerTestCase


class TestException(Exception):
    """An exception to test error conditions."""
    def get_dbus_name(self):
        """A fake dbus name for this exception."""
        return "Test Exception Message"


class TestNmNotAvailableException(Exception):
    """An exception to test unavailability conditions."""
    def get_dbus_name(self):
        """The real name of the dbus error when NM is not running."""
        return DBUS_UNKNOWN_SERVICE


class NetworkManagerStateTestCase(MockerTestCase):
    """Test NetworkManager state retrieval code."""

    def setUp(self):
        """Setup the mocker dbus object tree."""
        self.dbusmock = self.mocker.mock()
        self.dbusmock.SystemBus()
        sysbusmock = self.mocker.mock()
        self.mocker.result(sysbusmock)

        sysbusmock.get_object(ARGS, KWARGS)
        proxymock = self.mocker.mock()
        self.mocker.result(proxymock)

        self.dbusmock.Interface(proxymock, ANY)
        ifmock = self.mocker.mock()
        self.mocker.result(ifmock)

        ifmock.connect_to_signal(ARGS, KWARGS)
        signalmock = self.mocker.mock()
        self.mocker.result(signalmock)

        proxymock.Get(ARGS, KWARGS)
        signalmock.remove()

        self.mocker.replay()

    def test_nm_online(self):
        """Check the connected case."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, ONLINE)

        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
        nms.got_state(NM_STATE_CONNECTED)

    def test_nm_offline(self):
        """Check the disconnected case."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, OFFLINE)

        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
        nms.got_state(NM_STATE_DISCONNECTED)

    def test_nm_connecting_then_online(self):
        """Check the waiting for connection case."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, ONLINE)

        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
        nms.got_state(NM_STATE_CONNECTING)
        nms.state_changed(NM_STATE_CONNECTED)

    def test_nm_connecting_then_offline(self):
        """Check the waiting but fail case."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, OFFLINE)

        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
        nms.got_state(NM_STATE_CONNECTING)
        nms.state_changed(NM_STATE_DISCONNECTED)


class NetworkManagerStateErrorsTestCase(MockerTestCase):
    """Test NetworkManager state retrieval code."""

    # Statement seems to have no effect
    # pylint: disable=W0104

    def setUp(self):
        """Setup the mocker dbus object tree."""
        self.dbusmock = self.mocker.mock()
        self.dbusmock.SystemBus()
        self.sysbusmock = self.mocker.mock()
        self.mocker.result(self.sysbusmock)
        self.sysbusmock.get_object(ARGS, KWARGS)

    def mock_except_while_getting_proxy(self, exc):
        """Simulate an exception while getting the DBus proxy object."""
        self.mocker.throw(exc)
        self.dbusmock.exceptions.DBusException
        self.mocker.result(exc)
        self.mocker.replay()

    def mock_dbus_error_while_getting_state(self, exc):
        """Simulate an exception while getting the State."""
        proxymock = self.mocker.mock()
        self.mocker.result(proxymock)

        self.dbusmock.Interface(proxymock, ANY)
        ifmock = self.mocker.mock()
        self.mocker.result(ifmock)

        ifmock.connect_to_signal(ARGS, KWARGS)
        signalmock = self.mocker.mock()
        self.mocker.result(signalmock)

        proxymock.Get(ARGS, KWARGS)
        self.dbusmock.exceptions.DBusException
        self.mocker.result(exc)
        signalmock.remove()
        self.mocker.replay()

    def test_nm_not_running(self):
        """Check the case when NM is not running."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, UNKNOWN)

        self.mock_dbus_error_while_getting_state(TestNmNotAvailableException)
        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
        nms.got_error(TestNmNotAvailableException())

    def test_dbus_problem(self):
        """Check the case when DBus throws some other exception."""

        def got_state_cb(state):
            """State was given."""
            self.assertEquals(state, UNKNOWN)

        self.mock_except_while_getting_proxy(TestException)
        nms = NetworkManagerState(got_state_cb, self.dbusmock)
        nms.find_online_state()
