# -*- coding: utf-8 -*-
#
# test_main - tests for ubuntu_sso.main
#
# Author: Natalia Bidart <natalia.bidart@canonical.com>
# Author: Alejandro J. Cura <alecu@canonical.com>
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests share by diff platforms."""

import os

from unittest import TestCase
from mocker import MockerTestCase, MATCH
from ubuntu_sso.main import (
    CredentialsManagement,
    SSOCredentialsRoot,
    SSOCredentials,
    SSOLogin,
    U1_PING_URL)


class EnvironOverridesTestCase(TestCase):
    """Some URLs can be set from the environment for testing/QA purposes."""

    def test_override_ping_url(self):
        """The ping url can be set from the environ via USSOC_PING_URL."""
        fake_url = 'this is not really a URL'
        old_url = os.environ.get('USSOC_PING_URL')
        os.environ['USSOC_PING_URL'] = fake_url
        try:
            creds = SSOCredentialsRoot()
            self.assertEqual(creds.ping_url, fake_url)
        finally:
            if old_url:
                os.environ['USSOC_PING_URL'] = old_url
            else:
                del os.environ['USSOC_PING_URL']

    def test_no_override_ping_url(self):
        """If the environ is unset, the default ping url is used."""
        creds = SSOCredentialsRoot()
        self.assertEqual(creds.ping_url, U1_PING_URL)


class SSOLoginMockedTestCase(MockerTestCase):
    """Test that the call are relied correctly."""

    def setUp(self):
        """Setup tests."""
        super(SSOLoginMockedTestCase, self).setUp()
        self.root = self.mocker.mock()
        mockbusname = self.mocker.mock()
        mockbus = self.mocker.mock()
        mockbusname.get_bus()
        self.mocker.result(mockbus)
        self.login = SSOLogin(mockbus)
        self.login.root = self.root
        self.mocker.reset()

    def test_generate_captcha(self):
        """Test that the call is relayed."""
        app_name = 'app'
        filename = 'file'
        self.root.generate_captcha(app_name, filename, MATCH(callable),
                                   MATCH(callable), MATCH(callable))
        self.mocker.replay()
        self.login.generate_captcha(app_name, filename)

    def test_register_user(self):
        """Test that the call is relayed."""
        app_name = 'app'
        email = 'email'
        password = 'pwd'
        name = 'display name'
        captcha_id = 'id'
        captcha_solution = 'hello'
        self.root.register_user(app_name, email, password, name, captcha_id,
                                captcha_solution, MATCH(callable),
                                MATCH(callable), MATCH(callable))
        self.mocker.replay()
        self.login.register_user(app_name, email, password, name, captcha_id,
                                 captcha_solution)

    def test_login(self):
        """Test that the call is relayed."""
        app_name = 'app'
        email = 'email'
        password = 'password'
        self.root.login(app_name, email, password, MATCH(callable),
                        MATCH(callable), MATCH(callable),
                        MATCH(callable))
        self.mocker.mock()
        self.mocker.replay()
        self.login.login(app_name, email, password)

    def test_validate_email(self):
        """Test that the call is relayed."""
        app_name = 'app'
        email = 'email'
        password = 'passwrd'
        email_token = 'token'
        self.root.validate_email(app_name, email, password, email_token,
                                 MATCH(callable), MATCH(callable),
                                 MATCH(callable))
        self.mocker.replay()
        self.login.validate_email(app_name, email, password, email_token)

    def test_request_password_reset_tolen(self):
        """Test that the call is relayed."""
        app_name = 'app'
        email = 'email'
        self.root.request_password_reset_token(app_name, email,
                                               MATCH(callable),
                                               MATCH(callable),
                                               MATCH(callable))
        self.mocker.replay()
        self.login.request_password_reset_token(app_name, email)

    def test_set_new_password(self):
        """Test that the call is relayed."""
        app_name = 'app'
        email = 'email'
        token = 'token'
        new_password = 'new'
        self.root.set_new_password(app_name, email, token, new_password,
                                   MATCH(callable), MATCH(callable),
                                   MATCH(callable))
        self.mocker.replay()
        self.login.set_new_password(app_name, email, token, new_password)


class SSOCredentialsMockedTestCase(MockerTestCase):
    """Test that the call are relied correctly."""

    def setUp(self):
        """Setup tests."""
        super(SSOCredentialsMockedTestCase, self).setUp()
        self.root = self.mocker.mock()
        mockbusname = self.mocker.mock()
        mockbus = self.mocker.mock()
        mockbusname.get_bus()
        self.mocker.result(mockbus)
        self.cred = SSOCredentials(mockbus)
        self.cred.root = self.root
        self.mocker.reset()

    def test_find_credentials(self):
        """Test that the call is relayed."""
        app_name = 'app'
        result_cb = error_cb = lambda: None
        self.root.find_credentials(app_name, result_cb, error_cb)
        self.mocker.mock()
        self.mocker.replay()
        self.cred.find_credentials(app_name, result_cb, error_cb)

    def test_login_or_register_to_get_credentials(self):
        """Test that the call is relayed."""
        app_name = 'app'
        terms = 'terms'
        help_text = 'help'
        window_id = 'id'
        self.root.login_or_register_to_get_credentials(app_name, terms,
                                                       help_text, window_id,
                                                       MATCH(callable),
                                                       MATCH(callable),
                                                       MATCH(callable))
        self.mocker.replay()
        self.cred.login_or_register_to_get_credentials(app_name, terms,
                                                       help_text, window_id)

    def test_clear_token(self):
        """Test that the call is relayed."""
        app_name = 'app'
        result_cb = error_cb = lambda: None
        self.root.clear_token(app_name, result_cb, error_cb)
        self.mocker.replay()
        self.cred.clear_token(app_name, result_cb, error_cb)


class CredentialsManagementMockedTestCase(MockerTestCase):
    """Test that the call are relied correctly."""

    def setUp(self):
        """Setup tests."""
        super(CredentialsManagementMockedTestCase, self).setUp()
        self.root = self.mocker.mock()
        self.cred = CredentialsManagement(None, None)
        self.cred.root = self.root

    def test_find_credentials(self):
        """Test that the call is relayed."""
        app_name = 'app'
        args = 'args'
        self.root.find_credentials(app_name, args, MATCH(callable),
                                   MATCH(callable))
        self.mocker.replay()
        self.cred.find_credentials(app_name, args)

    def test_clear_credentials(self):
        """Test that the call is relayed."""
        app_name = 'app'
        args = 'args'
        self.root.clear_credentials(app_name, args, MATCH(callable),
                                    MATCH(callable))
        self.mocker.replay()
        self.cred.clear_credentials(app_name, args)

    def test_store_credentials(self):
        """Test that the call is relayed."""
        app_name = 'app'
        args = 'args'
        self.root.store_credentials(app_name, args, MATCH(callable),
                                    MATCH(callable))
        self.mocker.replay()
        self.cred.store_credentials(app_name, args)

    def test_register(self):
        """Test that the call is relayed."""
        app_name = 'app'
        args = 'args'
        self.root.register(app_name, args)
        self.mocker.replay()
        self.cred.register(app_name, args)

    def test_login(self):
        """Test that the call is relayed."""
        app_name = 'app'
        args = 'args'
        self.root.login(app_name, args)
        self.mocker.replay()
        self.cred.login(app_name, args)
