#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2002-2004 Zuza Software Foundation
# 
# This file is part of translate.
#
# translate is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# translate is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with translate; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

"""module for parsing .xliff files for translation"""

from xml.dom import minidom

def writexml(self, writer, indent="", addindent="", newl=""):
    """a replacement to writexml that formats it more like typical .ts files"""
    # indent = current indentation
    # addindent = indentation to add to higher levels
    # newl = newline string
    writer.write(indent+"<" + self.tagName)

    attrs = self._get_attributes()
    a_names = attrs.keys()
    a_names.sort()

    for a_name in a_names:
        writer.write(" %s=\"" % a_name)
        minidom._write_data(writer, attrs[a_name].value)
        writer.write("\"")
    if self.childNodes:
        if len(self.childNodes) == 1 and self.childNodes[0].nodeType == self.TEXT_NODE:
          writer.write(">")
          for node in self.childNodes:
              node.writexml(writer,"","","")
          writer.write("</%s>%s" % (self.tagName,newl))
        else:
          writer.write(">%s"%(newl))
          for node in self.childNodes:
              node.writexml(writer,indent+addindent,addindent,newl)
          writer.write("%s</%s>%s" % (indent,self.tagName,newl))
    else:
        writer.write("/>%s"%(newl))

# commented out modifications to minidom classes
'''
Element_writexml = minidom.Element.writexml
for elementclassname in dir(minidom):
  elementclass = getattr(minidom, elementclassname)
  if not isinstance(elementclass, type(minidom.Element)):
    continue
  if not issubclass(elementclass, minidom.Element):
    continue
  if elementclass.writexml != Element_writexml:
    continue
  elementclass.writexml = writexml
'''

def getText(nodelist):
  """joins together the text from all the text nodes in the nodelist and their children"""
  rc = []
  for node in nodelist:
    if node.nodeType == node.TEXT_NODE:
      rc.append(node.data)
    elif node.nodeType == node.ELEMENT_NODE:
      rc += getText(node.childNodes)
  return ''.join(rc)

# TODO: handle comments
# TODO: handle translation types

class XliffParser(object):
  """a parser for xliff files"""
  def __init__(self, inputfile=None):
    """make a new XliffParser, reading from the given inputfile if required"""
    self.filename = getattr(inputfile, "filename", None)
    if inputfile is None:
      self.document = minidom.parseString('''<?xml version="1.0" ?><xliff version='1.1' xmlns='urn:oasis:names:tc:xliff:document:1.1' xmlns:po='urn:oasis:names:tc:xliff:document:1.1:po-guide'></xliff>''')
    else:
      self.document = minidom.parse(inputfile)
      assert self.document.documentElement.tagName == "xliff"

  def addtransunit(self, filename, transunitnode, createifmissing=False):
    """adds the given trans-unit (will create the nodes required if asked). Returns success"""
    filenode = self.getfilenode(filename)
    if filenode is None:
      if not createifmissing:
        return False
      filenode = self.createfilenode(filename)
      self.document.documentElement.appendChild(filenode)
    for transunit in self.gettransunitnodes(filenode):
      pass
    if not createifmissing:
      return False
    filenode.appendChild(transunitnode)
    # transunitnode.setIdAttribute("message1")
    return True

  def createfilenode(self, filename):
    """creates a filenode with the given filename"""
    filenode = self.document.createElement("file")
    filenode.setAttribute("original", filename)
    return filenode

  def getnodetext(self, node):
    """returns the node's text by iterating through the child nodes"""
    return getText(node.childNodes)

  def getxml(self,pretty=True):
    """return the ts file as xml"""
    if pretty:
        xml = self.document.toprettyxml(indent="    ", encoding="utf-8")
    else:
        xml = self.document.toxml(encoding="utf-8")
    xml = "\n".join([line for line in xml.split("\n") if line.strip()])
    return xml

  def getfilename(self, filenode):
    """returns the name of the given file"""
    return filenode.getAttribute("original")

  def getfilenode(self, filename):
    """finds the filenode with the given name"""
    filenodes = self.document.getElementsByTagName("file")
    for filenode in filenodes:
      if self.getfilename(filenode) == filename:
        return filenode
    return None

  def gettransunitnodes(self, filenode=None):
    """returns all the transunitnodes, limiting to the given file (name or node) if given"""
    if filenode is None:
      return self.document.getElementsByTagName("trans-unit")
    else:
      if isinstance(filenode, (str, unicode)):
        # look up the file node by name
        filenode = self.getfilenode(filenode)
        if filenode is None:
          return []
      return filenode.getElementsByTagName("trans-unit")

  def gettransunitsource(self, transunit):
    """returns the transunit source for a given node"""
    sourcenode = transunit.getElementsByTagName("source")[0]
    return self.getnodetext(sourcenode)

  def gettransunittarget(self, transunit):
    """returns the transunit target for a given node"""
    try:
        translationnode = transunit.getElementsByTagName("target")[0]
    except IndexError,e:
        return None
    return self.getnodetext(translationnode)

  def iteritems(self):
    """iterates through (file, transunits)"""
    for filenode in self.document.getElementsByTagName("file"):
      yield self.getfilename(filenode), self.gettransunitnodes(filenode)

  def __del__(self):
    """clean up the document if required"""
    if hasattr(self, "document"):
      self.document.unlink()

