#!/usr/bin/env python3
# SPDX-License-Identifier: LGPL-2.1+
#
# sys-script.py
#
# © 2017 Canonical Ltd.
# Author: Dan Streetman <dan.streetman@canonical.com>

import os, sys
import shutil

def d(path, mode):
    os.mkdir(path, mode)

def l(path, src):
    os.symlink(src, path)

def f(path, mode, contents):
    with open(path, "wb") as f:
        f.write(contents)
    os.chmod(path, mode)

if len(sys.argv) < 2:
    exit("Usage: {} <target dir>".format(sys.argv[0]))

if not os.path.isdir(sys.argv[1]):
    exit("Target dir {} not found".format(sys.argv[1]))

os.chdir(sys.argv[1])

if os.path.exists('sys'):
    shutil.rmtree('sys')
d('sys', 0o755)
d('sys/kernel', 0o775)
f('sys/kernel/kexec_crash_loaded', 0o664, b'0\n')
f('sys/kernel/kexec_loaded', 0o664, b'0\n')
f('sys/kernel/uevent_helper', 0o664, b'\n')
f('sys/kernel/vmcoreinfo', 0o664, b'f15380 1000\n')
f('sys/kernel/notes', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00Z\r!\x804\xc2\x8atA<"f5(\xe7m\xe8-i-')
f('sys/kernel/uevent_seqnum', 0o664, b'1407\n')
d('sys/kernel/slab', 0o775)
d('sys/kernel/slab/kmalloc-96', 0o775)
f('sys/kernel/slab/kmalloc-96/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-96/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/slab_size', 0o664, b'168\n')
f('sys/kernel/slab/kmalloc-96/total_objects', 0o664, b'768\n')
f('sys/kernel/slab/kmalloc-96/slabs', 0o664, b'32\n')
f('sys/kernel/slab/kmalloc-96/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-96/alloc_calls', 0o664, b'''      5 sched_create_group+0x1f/0x220 age=8351870/8359354/8366881 pid=1333-2177 cpus=0-1
     10 __register_sysctl_paths+0x56/0x340 age=8370387/8378535/8379888 pid=1-901 cpus=0-1
     15 __create_workqueue_key+0x31/0x290 age=8370409/8377818/8380009 pid=1-900 cpus=0-1
     59 load_module+0x19c0/0x1b30 age=8303111/8368469/8376350 pid=105-2754 cpus=0-1
      5 __vmalloc_area_node+0xfb/0x140 age=8355400/8371519/8376226 pid=147-1813 cpus=0
     22 __register_chrdev_region+0x2d/0x1a0 age=8309177/8375269/8380004 pid=1-2711 cpus=0-1
      1 do_sys_poll+0xdb/0x450 age=9671 pid=2262 cpus=1
     98 d_alloc+0x19d/0x1f0 age=22395/7516545/8376320 pid=317-13453 cpus=0-1
      8 sys_eventfd2+0x51/0xb0 age=8339333/8339377/8339421 pid=2400 cpus=0
      1 mb_cache_create+0x62/0x1f0 age=8379846 pid=1 cpus=1
      9 sysfs_setattr+0x19a/0x1f0 age=61999/6511912/8377277 pid=63-13308 cpus=0-1
      1 sysfs_open_file+0x1c5/0x230 age=0 pid=13466 cpus=0
     17 copy_semundo+0x71/0xc0 age=3847042/8077586/8366263 pid=1370-2590 cpus=0-1
      2 __crypto_alloc_tfm+0x43/0x1b0 age=8373678/8373678/8373678 pid=126 cpus=1
      2 pcim_iomap_table+0x45/0x60 age=8379289/8379547/8379805 pid=1 cpus=0-1
     26 pci_create_attr+0x3d/0x140 age=8378994/8378994/8378995 pid=1 cpus=0
      8 pci_enable_msi+0xb1/0x2b0 age=8356336/8373963/8379844 pid=1-1541 cpus=0-1
      1 fb_add_videomode+0x89/0xf0 age=8379841 pid=1 cpus=1
      1 soft_cursor+0x92/0x220 age=8379841 pid=1 cpus=1
     20 acpi_os_create_semaphore+0x36/0xaa age=8379969/8380118/8380142 pid=0-1 cpus=0
      6 acpi_ds_build_internal_package_obj+0xaf/0x1df age=8379981/8379985/8379993 pid=1 cpus=0
      2 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379986/8379986/8379987 pid=1 cpus=0
      1 acpi_ev_create_gpe_block+0x5f/0x3bf age=8379999 pid=1 cpus=0
     62 pnp_new_resource+0x25/0x60 age=8379891/8379894/8379895 pid=1 cpus=0
      1 tty_register_driver+0x1b3/0x2a0 age=8379806 pid=1 cpus=1
      5 kbd_connect+0x4e/0xe0 age=8374352/8375743/8379035 pid=17-341 cpus=0-1
     17 __class_create+0x3d/0xa0 age=8374616/8377737/8380008 pid=1-215 cpus=0-1
      2 scsi_probe_and_add_lun+0x65e/0xd80 age=8379084/8379252/8379421 pid=1 cpus=0-1
      3 inet_rtm_newaddr+0xdb/0x220 age=8325560/8353789/8367912 pid=1197-1541 cpus=0-1
      2 cache_add_dev+0x7c/0x546 age=8379855/8379855/8379856 pid=1 cpus=0
     22 kernel_param_sysfs_setup+0x2f/0xc9 age=8380005/8380005/8380005 pid=1 cpus=0
     15 acpi_system_init+0x12e/0x260 age=8379898/8379898/8379899 pid=1 cpus=0
      1 pci_mmcfg_insert_resources+0x5d/0x123 age=8378994 pid=1 cpus=0
      1 0xffffffffa0065104 age=8356008 pid=1766 cpus=0
      1 bitmap_create+0x373/0xad0 [md_mod] age=8370145 pid=916 cpus=0
    256 r1bio_pool_alloc+0x2f/0x60 [raid1] age=8370154/8370154/8370154 pid=916 cpus=0
      8 ip6addrlbl_add+0x75/0x330 [ipv6] age=8360215/8360215/8360215 pid=1549 cpus=0
      1 ip6_route_net_init+0x22/0xd0 [ipv6] age=8360216 pid=1549 cpus=0
      6 snd_mixer_oss_build_input+0x431/0x5c0 [snd_mixer_oss] age=8355594/8355594/8355595 pid=1813 cpus=0
''')
f('sys/kernel/slab/kmalloc-96/objs_per_slab', 0o664, b'24\n')
f('sys/kernel/slab/kmalloc-96/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-96/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/object_size', 0o664, b'96\n')
f('sys/kernel/slab/kmalloc-96/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/objects_partial', 0o664, b'95\n')
f('sys/kernel/slab/kmalloc-96/objects', 0o664, b'743\n')
f('sys/kernel/slab/kmalloc-96/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-96/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-96/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-96/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-96/partial', 0o664, b'5\n')
f('sys/kernel/slab/kmalloc-96/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-96/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-96/free_calls', 0o664, b'''    507 <not-available> age=4303047439 pid=0 cpus=0
      1 free_notes_attrs+0x4b/0x50 age=8375086 pid=383 cpus=0
      1 audit_send_list+0x88/0xa0 age=8361087 pid=1538 cpus=1
      6 __vunmap+0xe9/0x120 age=8309178/8363011/8375091 pid=126-2711 cpus=0-1
      1 do_sys_poll+0x32b/0x450 age=9672 pid=2262 cpus=1
     30 d_callback+0x22/0x40 age=22587/7808930/8375651 pid=0-13433 cpus=0-1
     20 sysfs_release+0x63/0xa0 age=22407/6699376/8376561 pid=165-13453 cpus=0-1
      1 release_sysfs_dirent+0x22/0xd0 age=8367199 pid=1309 cpus=1
    116 free_rb_tree_fname+0x5c/0xb0 age=53501/7931736/8376455 pid=163-13373 cpus=0-1
      7 kobject_uevent_env+0x11a/0x470 age=67088/6542620/8379296 pid=1-419 cpus=0-1
      1 msi_free_irqs+0xad/0x110 age=8356355 pid=1541 cpus=0
      1 acpi_os_execute_deferred+0x34/0x39 age=8379958 pid=13 cpus=1
      1 acpi_walk_resources+0xa4/0xbd age=8379895 pid=1 cpus=0
      1 acpi_pci_irq_add_prt+0x30b/0x324 age=8379922 pid=1 cpus=0
      6 scsi_execute_req+0x9f/0xf0 age=8335607/8359788/8379422 pid=1-1456 cpus=0-1
      9 netlink_destroy_callback+0x21/0x30 age=8325561/8347615/8367913 pid=1197-2527 cpus=0-1
     12 huft_free+0x1e/0x2f age=8379857/8379857/8379857 pid=1 cpus=0
      2 r1bio_pool_free+0x9/0x10 [raid1] age=8364428/8364431/8364434 pid=0 cpus=0
''')
f('sys/kernel/slab/kmalloc-96/aliases', 0o664, b'0\n')
d('sys/kernel/slab/blkdev_requests', 0o775)
f('sys/kernel/slab/blkdev_requests/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/ctor', 0o664, b'')
f('sys/kernel/slab/blkdev_requests/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/slab_size', 0o664, b'376\n')
f('sys/kernel/slab/blkdev_requests/total_objects', 0o664, b'84\n')
f('sys/kernel/slab/blkdev_requests/slabs', 0o664, b'4\n')
f('sys/kernel/slab/blkdev_requests/poison', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_requests/alloc_calls', 0o664, b'     12 mempool_alloc_slab+0x11/0x20 age=61529/5606309/8378864 pid=1-13295 cpus=0-1\n')
f('sys/kernel/slab/blkdev_requests/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/blkdev_requests/shrink', 0o664, b'')
f('sys/kernel/slab/blkdev_requests/trace', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/object_size', 0o664, b'304\n')
f('sys/kernel/slab/blkdev_requests/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/objects_partial', 0o664, b'8\n')
f('sys/kernel/slab/blkdev_requests/objects', 0o664, b'50\n')
f('sys/kernel/slab/blkdev_requests/order', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_requests/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_requests/store_user', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_requests/align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_requests/partial', 0o664, b'2\n')
f('sys/kernel/slab/blkdev_requests/validate', 0o664, b'')
f('sys/kernel/slab/blkdev_requests/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_requests/free_calls', 0o664, b'''      7 <not-available> age=4303046881 pid=0 cpus=0
      5 mempool_free_slab+0x12/0x20 age=63050/1742015/8378750 pid=0-1456 cpus=0
''')
f('sys/kernel/slab/blkdev_requests/aliases', 0o664, b'0\n')
d('sys/kernel/slab/tw_sock_TCP', 0o775)
f('sys/kernel/slab/tw_sock_TCP/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/ctor', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCP/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/slab_size', 0o664, b'256\n')
f('sys/kernel/slab/tw_sock_TCP/total_objects', 0o664, b'16\n')
f('sys/kernel/slab/tw_sock_TCP/slabs', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/poison', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/tw_sock_TCP/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/tw_sock_TCP/shrink', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCP/trace', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/object_size', 0o664, b'136\n')
f('sys/kernel/slab/tw_sock_TCP/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/objects', 0o664, b'16\n')
f('sys/kernel/slab/tw_sock_TCP/order', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/store_user', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/align', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/partial', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCP/validate', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCP/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCP/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/tw_sock_TCP/aliases', 0o664, b'0\n')
d('sys/kernel/slab/Acpi-Parse', 0o775)
f('sys/kernel/slab/Acpi-Parse/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/ctor', 0o664, b'')
f('sys/kernel/slab/Acpi-Parse/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/slab_size', 0o664, b'120\n')
f('sys/kernel/slab/Acpi-Parse/total_objects', 0o664, b'68\n')
f('sys/kernel/slab/Acpi-Parse/slabs', 0o664, b'2\n')
f('sys/kernel/slab/Acpi-Parse/poison', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Parse/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-Parse/objs_per_slab', 0o664, b'34\n')
f('sys/kernel/slab/Acpi-Parse/shrink', 0o664, b'')
f('sys/kernel/slab/Acpi-Parse/trace', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/object_size', 0o664, b'48\n')
f('sys/kernel/slab/Acpi-Parse/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/objects', 0o664, b'68\n')
f('sys/kernel/slab/Acpi-Parse/order', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Parse/store_user', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Parse/align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Parse/validate', 0o664, b'')
f('sys/kernel/slab/Acpi-Parse/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Parse/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-Parse/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sgpool-8', 0o775)
f('sys/kernel/slab/sgpool-8/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/ctor', 0o664, b'')
f('sys/kernel/slab/sgpool-8/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/sgpool-8/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/sgpool-8/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-8/poison', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379297/8379297/8379297 pid=1 cpus=0\n')
f('sys/kernel/slab/sgpool-8/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/sgpool-8/shrink', 0o664, b'')
f('sys/kernel/slab/sgpool-8/trace', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/object_size', 0o664, b'256\n')
f('sys/kernel/slab/sgpool-8/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/objects', 0o664, b'42\n')
f('sys/kernel/slab/sgpool-8/order', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/align', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-8/validate', 0o664, b'')
f('sys/kernel/slab/sgpool-8/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-8/free_calls', 0o664, b'      2 <not-available> age=4303046847 pid=0 cpus=0\n')
f('sys/kernel/slab/sgpool-8/aliases', 0o664, b'0\n')
d('sys/kernel/slab/dnotify_cache', 0o775)
f('sys/kernel/slab/dnotify_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/ctor', 0o664, b'')
f('sys/kernel/slab/dnotify_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/slab_size', 0o664, b'112\n')
f('sys/kernel/slab/dnotify_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/dnotify_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dnotify_cache/objs_per_slab', 0o664, b'36\n')
f('sys/kernel/slab/dnotify_cache/shrink', 0o664, b'')
f('sys/kernel/slab/dnotify_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/object_size', 0o664, b'40\n')
f('sys/kernel/slab/dnotify_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/dnotify_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/dnotify_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/dnotify_cache/validate', 0o664, b'')
f('sys/kernel/slab/dnotify_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/dnotify_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dnotify_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-16', 0o775)
f('sys/kernel/slab/kmalloc-16/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-16/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/slab_size', 0o664, b'88\n')
f('sys/kernel/slab/kmalloc-16/total_objects', 0o664, b'2254\n')
f('sys/kernel/slab/kmalloc-16/slabs', 0o664, b'49\n')
f('sys/kernel/slab/kmalloc-16/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-16/alloc_calls', 0o664, b'''      2 arch_acpi_processor_init_pdc+0x71/0x180 age=8375737/8375768/8375799 pid=330 cpus=0
      2 arch_acpi_processor_init_pdc+0xab/0x180 age=8375737/8375768/8375799 pid=330 cpus=0
      5 sched_create_group+0x40/0x220 age=8351840/8359324/8366851 pid=1333-2177 cpus=0-1
      5 sched_create_group+0x62/0x220 age=8351840/8359324/8366851 pid=1333-2177 cpus=0-1
      2 setup_modinfo_version+0x19/0x30 age=8374630/8375434/8376238 pid=122-126 cpus=0-1
    365 load_module+0x1ac4/0x1b30 age=8303081/8369498/8376320 pid=105-2754 cpus=0-1
     13 mempool_create_node+0x4b/0xf0 age=8379859/8379929/8379974 pid=1 cpus=0
      1 krealloc+0x1e/0x60 age=8375802 pid=330 cpus=0
      4 __vmalloc_area_node+0xfb/0x140 age=8374701/8375120/8375833 pid=126-392 cpus=0
      7 alloc_vfsmnt+0x97/0x180 age=61473/7184623/8379853 pid=1-13348 cpus=0-1
      2 proc_symlink+0x4d/0xb0 age=8380112/8380112/8380112 pid=0 cpus=0
   1275 sysfs_new_dirent+0x10c/0x120 age=22645/7761348/8379979 pid=1-13295 cpus=0-1
      2 ext3_fill_super+0x691/0x1a20 age=8369671/8374021/8378371 pid=1-962 cpus=0
    406 kvasprintf+0x55/0x90 age=22645/7760583/8380112 pid=0-13295 cpus=0-1
      1 bit_cursor+0x223/0x6a0 age=8356170 pid=7 cpus=0
     12 acpi_ds_build_internal_package_obj+0xaf/0x1df age=8379944/8379946/8379954 pid=1 cpus=0
     14 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379945/8379955/8379961 pid=1 cpus=0
      2 acpi_ut_copy_simple_object+0x8f/0x11c age=8375657/8375665/8375674 pid=396 cpus=0
      5 acpi_irq_stats_init+0x1bd/0x268 age=8379969/8379969/8379969 pid=1 cpus=0
     14 pnp_add_id+0x1e/0xe0 age=8379861/8379863/8379866 pid=1 cpus=0
      9 reserve_range+0x39/0x130 age=8379850/8379850/8379850 pid=1 cpus=0
      1 trackpoint_detect+0x94/0x190 age=8378732 pid=17 cpus=0
      2 proto_register+0xf2/0x260 age=8360199/8370021/8379844 pid=1-1549 cpus=0
      2 neigh_sysctl_register+0x1cd/0x330 age=8360183/8366975/8373768 pid=126-1549 cpus=0-1
      1 nl_pid_hash_rehash+0x180/0x1a0 age=8361057 pid=1533 cpus=0
      1 genl_register_family+0x1b6/0x1e0 age=8379853 pid=1 cpus=0
      1 __devinet_sysctl_register+0xb8/0x120 age=8373768 pid=126 cpus=1
      1 pci_acpi_scan_root+0x3f/0x230 age=8379897 pid=1 cpus=0
      1 acpi_parse_mcfg+0x61/0x140 age=8379976 pid=1 cpus=0
      1 pci_mmcfg_arch_init+0x26/0x129 age=8379976 pid=1 cpus=0
      1 icmp_sk_init+0x32/0x13f age=8379826 pid=1 cpus=0
      5 hub_probe+0x26b/0x820 [usbcore] age=8374065/8374558/8374911 pid=143-147 cpus=0-1
     12 usb_cache_string+0x65/0xa0 [usbcore] age=22646/4900582/8374931 pid=143-419 cpus=0-1
      1 usb_create_ep_files+0x283/0x350 [usbcore] age=8374798 pid=143 cpus=0
      1 0xffffffffa006595f age=8360191 pid=1549 cpus=0
     13 snd_info_create_entry+0x30/0xa0 [snd] age=8355564/8372743/8374175 pid=107-1813 cpus=0-1
      1 snd_oss_info_register+0x40/0xc0 [snd] age=8374813 pid=107 cpus=1
      1 async_chainiv_givencrypt+0xf9/0x110 [crypto_blkcipher] age=8374570 pid=215 cpus=0
      1 eseqiv_alloc+0x6d/0x80 [crypto_blkcipher] age=8374570 pid=215 cpus=0
      1 run+0xa9/0x500 [raid1] age=8370124 pid=916 cpus=0
      1 __addrconf_sysctl_register+0xcd/0x140 [ipv6] age=8360183 pid=1549 cpus=0
''')
f('sys/kernel/slab/kmalloc-16/objs_per_slab', 0o664, b'46\n')
f('sys/kernel/slab/kmalloc-16/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-16/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/object_size', 0o664, b'16\n')
f('sys/kernel/slab/kmalloc-16/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/objects', 0o664, b'2254\n')
f('sys/kernel/slab/kmalloc-16/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-16/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-16/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-16/partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-16/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-16/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-16/free_calls', 0o664, b'''   1912 <not-available> age=4303047409 pid=0 cpus=0
      2 free_sched_group+0x62/0x80 age=8375109/8375395/8375681 pid=0-72 cpus=0-1
      2 free_sched_group+0x6b/0x80 age=8375109/8375395/8375681 pid=0-72 cpus=0-1
      3 free_sect_attrs+0x2e/0x50 age=8375055/8375055/8375055 pid=383 cpus=0
      3 __vunmap+0xe9/0x120 age=8373650/8374619/8375152 pid=181-754 cpus=0
      7 vfs_rename+0x301/0x450 age=125333/5028818/8367033 pid=829-13249 cpus=0
     10 bio_free_map_data+0x14/0x30 age=23581/2550049/8379364 pid=0-1738 cpus=0-1
     10 bio_free_map_data+0x1d/0x30 age=53581/4218001/8379379 pid=0-558 cpus=0-1
     60 remove_kevent+0x44/0x60 age=61364/3011504/8354072 pid=1547-2544 cpus=0-1
     49 release_sysfs_dirent+0x8c/0xd0 age=7824392/8263442/8379778 pid=1-2141 cpus=0-1
      9 kobject_release+0xe1/0x140 age=7833231/8247578/8373891 pid=419-1541 cpus=0-1
     70 kobject_uevent_env+0x11a/0x470 age=409229/8261185/8379979 pid=1-12942 cpus=0-1
      2 bit_cursor+0x23b/0x6a0 age=8356171/8367984/8379797 pid=1-7 cpus=0-1
      1 acpi_ds_create_operand+0x12c/0x209 age=8379867 pid=1 cpus=0
      2 acpi_ns_get_node+0x92/0xa1 age=8374467/8374504/8374542 pid=215 cpus=0-1
      8 acpi_ut_delete_internal_obj+0x15f/0x16f age=8366439/8374588/8375781 pid=330-1358 cpus=0
     16 acpi_ut_delete_internal_object_list+0x28/0x2f age=8374469/8378935/8379920 pid=1-396 cpus=0
      2 reserve_range+0x11d/0x130 age=8379851/8379851/8379851 pid=1 cpus=0
      3 release_firmware+0x57/0x70 age=8356214/8363521/8367182 pid=1285-1541 cpus=0-1
      8 module_add_driver+0x66/0xd0 age=8373538/8375876/8379266 pid=1-181 cpus=0-1
      1 power_supply_uevent+0x1a0/0x210 age=8375647 pid=7 cpus=0
      2 get_modalias+0xd4/0x120 age=8355835/8367906/8379977 pid=1-1772 cpus=0
      1 nl_pid_hash_rehash+0x18e/0x1a0 age=8356437 pid=1708 cpus=0
      2 fib_hash_free+0x35/0x40 age=8367899/8367899/8367899 pid=1185 cpus=0
      2 wireless_send_event+0x172/0x330 age=8329605/8342854/8356103 pid=1545 cpus=0-1
     10 usb_get_configuration+0x42d/0x1480 [usbcore] age=22733/6708190/8374932 pid=143-419 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-16/aliases', 0o664, b'0\n')
d('sys/kernel/slab/filp', 0o775)
f('sys/kernel/slab/filp/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/filp/ctor', 0o664, b'')
f('sys/kernel/slab/filp/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/filp/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/filp/total_objects', 0o664, b'4158\n')
f('sys/kernel/slab/filp/slabs', 0o664, b'198\n')
f('sys/kernel/slab/filp/poison', 0o664, b'1\n')
f('sys/kernel/slab/filp/alloc_calls', 0o664, b'   4081 get_empty_filp+0x44/0x1a0 age=0/7917861/8378089 pid=1-20296 cpus=0-1\n')
f('sys/kernel/slab/filp/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/filp/shrink', 0o664, b'')
f('sys/kernel/slab/filp/trace', 0o664, b'0\n')
f('sys/kernel/slab/filp/object_size', 0o664, b'288\n')
f('sys/kernel/slab/filp/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/filp/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/filp/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/filp/objects_partial', 0o664, b'685\n')
f('sys/kernel/slab/filp/objects', 0o664, b'4064\n')
f('sys/kernel/slab/filp/order', 0o664, b'1\n')
f('sys/kernel/slab/filp/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/filp/store_user', 0o664, b'1\n')
f('sys/kernel/slab/filp/align', 0o664, b'0\n')
f('sys/kernel/slab/filp/partial', 0o664, b'33\n')
f('sys/kernel/slab/filp/validate', 0o664, b'')
f('sys/kernel/slab/filp/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/filp/free_calls', 0o664, b'''    866 <not-available> age=4303047162 pid=0 cpus=0
   3217 file_free_rcu+0x16/0x20 age=10/7799670/8377025 pid=0-20296 cpus=0-1
''')
f('sys/kernel/slab/filp/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-2048', 0o775)
f('sys/kernel/slab/kmalloc-2048/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-2048/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/slab_size', 0o664, b'2120\n')
f('sys/kernel/slab/kmalloc-2048/total_objects', 0o664, b'600\n')
f('sys/kernel/slab/kmalloc-2048/slabs', 0o664, b'40\n')
f('sys/kernel/slab/kmalloc-2048/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-2048/alloc_calls', 0o664, b'''     53 load_module+0x18d5/0x1b30 age=8303006/8368517/8376245 pid=105-2754 cpus=0-1
      1 mempool_create_node+0x4b/0xf0 age=8370049 pid=916 cpus=0
     20 sget+0xc4/0x460 age=61397/7958673/8380037 pid=0-13348 cpus=0-1
     15 alloc_fdtable+0x81/0x160 age=4219/6197046/8357231 pid=1532-12336 cpus=0-1
      2 journal_init_common+0x1a/0x130 age=8369592/8373942/8378292 pid=1-962 cpus=0
     15 alloc_disk_node+0x1b/0x120 age=61949/7263912/8379312 pid=1-13295 cpus=0-1
      1 pci_create_bus+0x3a/0x240 age=8379822 pid=1 cpus=0
     21 alloc_pci_dev+0x1a/0x40 age=8379818/8379820/8379822 pid=1 cpus=0
      1 framebuffer_alloc+0x3a/0x80 age=8379737 pid=1 cpus=1
     74 acpi_add_single_object+0x4e/0xd3c age=8379823/8379836/8379852 pid=1 cpus=0
      1 acpi_irq_stats_init+0xcb/0x268 age=8379894 pid=1 cpus=0
     11 pnp_alloc_dev+0x35/0x120 age=8379786/8379789/8379791 pid=1 cpus=0
     11 init_dev+0x12e/0x6f0 age=7217110/8250882/8378132 pid=33-2593 cpus=0-1
      3 init_dev+0x2aa/0x6f0 age=7217110/7969221/8364310 pid=1258-2593 cpus=0-1
      1 tty_register_driver+0x1b3/0x2a0 age=8379706 pid=1 cpus=1
      2 tty_write+0x160/0x280 age=913316/4463710/8014104 pid=1206-3110 cpus=0
      2 kobj_map_init+0x22/0xa0 age=8379899/8379968/8380037 pid=0-1 cpus=0
      3 mousedev_create+0x36/0x2d0 age=53460/5603607/8378947 pid=1-419 cpus=0
      1 atkbd_connect+0x33/0x290 age=8378940 pid=17 cpus=1
      2 thermal_zone_device_register+0x6f/0x310 age=8374790/8374807/8374825 pid=329 cpus=0-1
     15 sk_prot_alloc+0x83/0xb0 age=8328308/8367505/8379903 pid=1-2586 cpus=0-1
      1 reqsk_queue_alloc+0x112/0x120 age=8356786 pid=1690 cpus=0
    255 __netdev_alloc_skb+0x1f/0x40 age=8356300/8356301/8356303 pid=1541 cpus=0
     10 neigh_sysctl_register+0x8d/0x330 age=8360107/8368337/8379760 pid=1-1549 cpus=0-1
      6 __devinet_sysctl_register+0x74/0x120 age=8373573/8377097/8379760 pid=1-126 cpus=0-1
      7 pci_add_new_bus+0x25/0x160 age=8379818/8379818/8379820 pid=1 cpus=0
      1 i8042_create_aux_port+0x36/0x10f age=8378959 pid=1 cpus=0
      1 i8042_probe+0x190/0x68d age=8378959 pid=1 cpus=0
      1 netdev_init+0x32/0xc2 age=8379782 pid=1 cpus=0
      1 netdev_init+0x66/0xc2 age=8379782 pid=1 cpus=0
      1 fib_net_init+0x21/0x14c age=8379760 pid=1 cpus=0
     10 usb_alloc_dev+0x36/0x2c0 [usbcore] age=22875/5876181/8374868 pid=143-419 cpus=0-1
      2 acpi_thermal_add+0x36/0x4bb [thermal] age=8374792/8374809/8374827 pid=329 cpus=0-1
      5 evdev_open+0xaf/0x1e0 [evdev] age=8364377/8364927/8365226 pid=1443 cpus=0-1
      8 evdev_connect+0x54/0x1f0 [evdev] age=53452/7334685/8375002 pid=187-419 cpus=0-1
      1 rtc_device_register+0xa4/0x260 [rtc_core] age=8374707 pid=208 cpus=0
      1 iwl3945_bg_request_scan+0x5d3/0x660 [iwl3945] age=8356027 pid=733 cpus=0
      1 iwl3945_pci_probe+0xd26/0x1020 [iwl3945] age=8373700 pid=126 cpus=1
      1 patch_ad1981+0x1b/0x1d0 [snd_hda_intel] age=8374101 pid=107 cpus=1
      1 rfkill_allocate+0x24/0xc0 [rfkill] age=8374401 pid=215 cpus=0
      1 fuse_fill_super+0x26d/0x6d0 [fuse] age=8335736 pid=2476 cpus=0
''')
f('sys/kernel/slab/kmalloc-2048/objs_per_slab', 0o664, b'15\n')
f('sys/kernel/slab/kmalloc-2048/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-2048/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/object_size', 0o664, b'2048\n')
f('sys/kernel/slab/kmalloc-2048/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-2048/objects_partial', 0o664, b'59\n')
f('sys/kernel/slab/kmalloc-2048/objects', 0o664, b'584\n')
f('sys/kernel/slab/kmalloc-2048/order', 0o664, b'3\n')
f('sys/kernel/slab/kmalloc-2048/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-2048/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-2048/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-2048/partial', 0o664, b'5\n')
f('sys/kernel/slab/kmalloc-2048/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-2048/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-2048/free_calls', 0o664, b'''    424 <not-available> age=4303047334 pid=0 cpus=0
      1 __vunmap+0xe9/0x120 age=8374513 pid=107 cpus=1
     79 free_fdtable_rcu+0x71/0xb0 age=7254/7749386/8377555 pid=0-13365 cpus=0-1
      4 show_stat+0x50e/0x530 age=23516/2125165/8360113 pid=1258-1868 cpus=0
      4 acpi_add_single_object+0xc9a/0xd3c age=8379824/8379831/8379839 pid=1 cpus=0
     20 release_one_tty+0x145/0x190 age=8352386/8360775/8370275 pid=55-2136 cpus=0-1
      1 atkbd_connect+0x1ae/0x290 age=8378923 pid=17 cpus=0
      8 sk_free+0xcd/0x100 age=656267/7395001/8360982 pid=1533-12333 cpus=0-1
     18 skb_release_data+0x85/0xd0 age=61516/7352252/8379750 pid=1-13343 cpus=0-1
      5 __scm_destroy+0x41/0x50 age=7217111/8111550/8352414 pid=2135-2593 cpus=0-1
      1 huft_build+0x163/0x63e age=8379752 pid=1 cpus=0
      1 inflate_fixed+0x17c/0x193 age=8379752 pid=1 cpus=0
      1 usb_release_dev+0x59/0x70 [usbcore] age=53835 pid=419 cpus=0
      1 acpi_processor_get_throttling_info+0x2e4/0x58c [processor] age=8375644 pid=330 cpus=0
      1 acpi_thermal_add+0x457/0x4bb [thermal] age=8374828 pid=329 cpus=1
      1 snd_ctl_ioctl+0x2cf/0x960 [snd] age=8373099 pid=786 cpus=0
''')
f('sys/kernel/slab/kmalloc-2048/aliases', 0o664, b'0\n')
d('sys/kernel/slab/cfq_queue', 0o775)
f('sys/kernel/slab/cfq_queue/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/ctor', 0o664, b'')
f('sys/kernel/slab/cfq_queue/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/slab_size', 0o664, b'208\n')
f('sys/kernel/slab/cfq_queue/total_objects', 0o664, b'95\n')
f('sys/kernel/slab/cfq_queue/slabs', 0o664, b'5\n')
f('sys/kernel/slab/cfq_queue/poison', 0o664, b'1\n')
f('sys/kernel/slab/cfq_queue/alloc_calls', 0o664, b'     67 cfq_get_queue+0xc6/0x220 age=168/7825503/8379604 pid=1-13466 cpus=0-1\n')
f('sys/kernel/slab/cfq_queue/objs_per_slab', 0o664, b'19\n')
f('sys/kernel/slab/cfq_queue/shrink', 0o664, b'')
f('sys/kernel/slab/cfq_queue/trace', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/object_size', 0o664, b'136\n')
f('sys/kernel/slab/cfq_queue/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/objects_partial', 0o664, b'35\n')
f('sys/kernel/slab/cfq_queue/objects', 0o664, b'92\n')
f('sys/kernel/slab/cfq_queue/order', 0o664, b'0\n')
f('sys/kernel/slab/cfq_queue/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/cfq_queue/store_user', 0o664, b'1\n')
f('sys/kernel/slab/cfq_queue/align', 0o664, b'8\n')
f('sys/kernel/slab/cfq_queue/partial', 0o664, b'2\n')
f('sys/kernel/slab/cfq_queue/validate', 0o664, b'')
f('sys/kernel/slab/cfq_queue/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/cfq_queue/free_calls', 0o664, b'''     11 <not-available> age=4303047621 pid=0 cpus=0
     56 cfq_put_queue+0x6f/0xe0 age=45755/7905388/8375272 pid=163-26529 cpus=0-1
''')
f('sys/kernel/slab/cfq_queue/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ext3_xattr', 0o775)
f('sys/kernel/slab/ext3_xattr/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/ctor', 0o664, b'')
f('sys/kernel/slab/ext3_xattr/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/slab_size', 0o664, b'160\n')
f('sys/kernel/slab/ext3_xattr/total_objects', 0o664, b'25\n')
f('sys/kernel/slab/ext3_xattr/slabs', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/poison', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/alloc_calls', 0o664, b'      4 mb_cache_entry_alloc+0x15/0x50 age=5766034/7671829/8319899 pid=2676-9046 cpus=0\n')
f('sys/kernel/slab/ext3_xattr/objs_per_slab', 0o664, b'25\n')
f('sys/kernel/slab/ext3_xattr/shrink', 0o664, b'')
f('sys/kernel/slab/ext3_xattr/trace', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/object_size', 0o664, b'88\n')
f('sys/kernel/slab/ext3_xattr/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/objects', 0o664, b'25\n')
f('sys/kernel/slab/ext3_xattr/order', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/align', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/partial', 0o664, b'0\n')
f('sys/kernel/slab/ext3_xattr/validate', 0o664, b'')
f('sys/kernel/slab/ext3_xattr/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ext3_xattr/free_calls', 0o664, b'      4 <not-available> age=4303047503 pid=0 cpus=0\n')
f('sys/kernel/slab/ext3_xattr/aliases', 0o664, b'0\n')
d('sys/kernel/slab/bio', 0o775)
f('sys/kernel/slab/bio/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/bio/ctor', 0o664, b'')
f('sys/kernel/slab/bio/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/bio/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/bio/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/bio/slabs', 0o664, b'2\n')
f('sys/kernel/slab/bio/poison', 0o664, b'1\n')
f('sys/kernel/slab/bio/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379518/8379518/8379518 pid=1 cpus=0\n')
f('sys/kernel/slab/bio/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/bio/shrink', 0o664, b'')
f('sys/kernel/slab/bio/trace', 0o664, b'0\n')
f('sys/kernel/slab/bio/object_size', 0o664, b'104\n')
f('sys/kernel/slab/bio/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/bio/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/bio/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/bio/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/bio/objects', 0o664, b'42\n')
f('sys/kernel/slab/bio/order', 0o664, b'0\n')
f('sys/kernel/slab/bio/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/bio/store_user', 0o664, b'1\n')
f('sys/kernel/slab/bio/align', 0o664, b'8\n')
f('sys/kernel/slab/bio/partial', 0o664, b'0\n')
f('sys/kernel/slab/bio/validate', 0o664, b'')
f('sys/kernel/slab/bio/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/bio/free_calls', 0o664, b'      2 <not-available> age=4303046952 pid=0 cpus=0\n')
f('sys/kernel/slab/bio/aliases', 0o664, b'0\n')
d('sys/kernel/slab/TCP', 0o775)
f('sys/kernel/slab/TCP/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/TCP/ctor', 0o664, b'')
f('sys/kernel/slab/TCP/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/TCP/slab_size', 0o664, b'2304\n')
f('sys/kernel/slab/TCP/total_objects', 0o664, b'28\n')
f('sys/kernel/slab/TCP/slabs', 0o664, b'2\n')
f('sys/kernel/slab/TCP/poison', 0o664, b'1\n')
f('sys/kernel/slab/TCP/alloc_calls', 0o664, b'      3 sk_prot_alloc+0x1e/0xb0 age=926836/5879023/8356238 pid=1206-1920 cpus=0\n')
f('sys/kernel/slab/TCP/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/TCP/shrink', 0o664, b'')
f('sys/kernel/slab/TCP/trace', 0o664, b'0\n')
f('sys/kernel/slab/TCP/object_size', 0o664, b'2200\n')
f('sys/kernel/slab/TCP/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/TCP/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/TCP/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/TCP/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/TCP/objects', 0o664, b'28\n')
f('sys/kernel/slab/TCP/order', 0o664, b'3\n')
f('sys/kernel/slab/TCP/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/TCP/store_user', 0o664, b'1\n')
f('sys/kernel/slab/TCP/align', 0o664, b'0\n')
f('sys/kernel/slab/TCP/partial', 0o664, b'0\n')
f('sys/kernel/slab/TCP/validate', 0o664, b'')
f('sys/kernel/slab/TCP/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/TCP/free_calls', 0o664, b'      3 sk_free+0x80/0x100 age=932105/5881414/8356684 pid=0-1762 cpus=0\n')
f('sys/kernel/slab/TCP/aliases', 0o664, b'0\n')
d('sys/kernel/slab/nsproxy', 0o775)
f('sys/kernel/slab/nsproxy/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/ctor', 0o664, b'')
f('sys/kernel/slab/nsproxy/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/nsproxy/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/slabs', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/poison', 0o664, b'1\n')
f('sys/kernel/slab/nsproxy/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/nsproxy/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/nsproxy/shrink', 0o664, b'')
f('sys/kernel/slab/nsproxy/trace', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/object_size', 0o664, b'56\n')
f('sys/kernel/slab/nsproxy/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/objects', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/order', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/nsproxy/store_user', 0o664, b'1\n')
f('sys/kernel/slab/nsproxy/align', 0o664, b'8\n')
f('sys/kernel/slab/nsproxy/partial', 0o664, b'0\n')
f('sys/kernel/slab/nsproxy/validate', 0o664, b'')
f('sys/kernel/slab/nsproxy/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/nsproxy/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/nsproxy/aliases', 0o664, b'0\n')
d('sys/kernel/slab/inotify_watch_cache', 0o775)
f('sys/kernel/slab/inotify_watch_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/ctor', 0o664, b'')
f('sys/kernel/slab/inotify_watch_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/inotify_watch_cache/total_objects', 0o664, b'168\n')
f('sys/kernel/slab/inotify_watch_cache/slabs', 0o664, b'6\n')
f('sys/kernel/slab/inotify_watch_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/inotify_watch_cache/alloc_calls', 0o664, b'    126 sys_inotify_add_watch+0x15e/0x1d0 age=60497/8227702/8376765 pid=72-13355 cpus=0-1\n')
f('sys/kernel/slab/inotify_watch_cache/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/inotify_watch_cache/shrink', 0o664, b'')
f('sys/kernel/slab/inotify_watch_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/object_size', 0o664, b'72\n')
f('sys/kernel/slab/inotify_watch_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/objects_partial', 0o664, b'24\n')
f('sys/kernel/slab/inotify_watch_cache/objects', 0o664, b'164\n')
f('sys/kernel/slab/inotify_watch_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/inotify_watch_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/inotify_watch_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/inotify_watch_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/inotify_watch_cache/validate', 0o664, b'')
f('sys/kernel/slab/inotify_watch_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/inotify_watch_cache/free_calls', 0o664, b'''    122 <not-available> age=4303047477 pid=0 cpus=0
      4 free_inotify_user_watch+0x60/0x70 age=104648/4219698/8334832 pid=2423-13211 cpus=0
''')
f('sys/kernel/slab/inotify_watch_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fs_cache', 0o775)
f('sys/kernel/slab/fs_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/ctor', 0o664, b'')
f('sys/kernel/slab/fs_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/fs_cache/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/fs_cache/total_objects', 0o664, b'126\n')
f('sys/kernel/slab/fs_cache/slabs', 0o664, b'6\n')
f('sys/kernel/slab/fs_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/fs_cache/alloc_calls', 0o664, b'     79 __copy_fs_struct+0x28/0xc0 age=4126/7761540/8379883 pid=1-12336 cpus=0-1\n')
f('sys/kernel/slab/fs_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/fs_cache/shrink', 0o664, b'')
f('sys/kernel/slab/fs_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/object_size', 0o664, b'104\n')
f('sys/kernel/slab/fs_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/objects_partial', 0o664, b'75\n')
f('sys/kernel/slab/fs_cache/objects', 0o664, b'117\n')
f('sys/kernel/slab/fs_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fs_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fs_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/fs_cache/partial', 0o664, b'4\n')
f('sys/kernel/slab/fs_cache/validate', 0o664, b'')
f('sys/kernel/slab/fs_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fs_cache/free_calls', 0o664, b'''      7 <not-available> age=4303047241 pid=0 cpus=0
     72 put_fs_struct+0x37/0x40 age=7168/7706405/8375333 pid=145-20292 cpus=0-1
''')
f('sys/kernel/slab/fs_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-512', 0o775)
f('sys/kernel/slab/kmalloc-512/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-512/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/slab_size', 0o664, b'584\n')
f('sys/kernel/slab/kmalloc-512/total_objects', 0o664, b'504\n')
f('sys/kernel/slab/kmalloc-512/slabs', 0o664, b'36\n')
f('sys/kernel/slab/kmalloc-512/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-512/alloc_calls', 0o664, b'''     14 __register_sysctl_paths+0x56/0x340 age=8360125/8372514/8379788 pid=1-1549 cpus=0
     14 param_sysfs_setup+0x87/0x140 age=8355309/8375663/8379919 pid=1-1813 cpus=0-1
     59 module_add_modinfo_attrs+0x25/0xf0 age=8303025/8368384/8376264 pid=105-2754 cpus=0-1
      2 __vmalloc_area_node+0xfb/0x140 age=8374650/8375296/8375942 pid=126-144 cpus=0
      1 dma_kmalloc_cache+0xca/0x150 age=8375776 pid=163 cpus=0
    107 kmem_cache_create+0x4e/0x2e0 age=8360130/8377702/8380056 pid=0-1549 cpus=0-1
      2 __percpu_alloc_mask+0xbc/0x140 age=8379768/8379768/8379768 pid=1 cpus=1
     17 sys_inotify_init1+0xaa/0x220 age=8333709/8350157/8376641 pid=72-2544 cpus=0-1
      4 sys_epoll_create1+0x41/0x100 age=2416994/6871289/8356805 pid=1690-20296 cpus=0-1
      2 __crypto_alloc_tfm+0x43/0x1b0 age=8373592/8373592/8373592 pid=126 cpus=1
      1 crypto_alloc_instance+0x2b/0xe0 age=8373592 pid=745 cpus=0
      3 elevator_alloc+0x67/0xc0 age=62000/5606780/8379335 pid=1-13295 cpus=0-1
      3 cfq_init_queue+0x1b/0x120 age=62000/5606780/8379335 pid=1-13295 cpus=0-1
      1 fb_alloc_cmap+0x66/0x150 age=8379756 pid=1 cpus=1
      1 fb_alloc_cmap+0x81/0x150 age=8379756 pid=1 cpus=1
      1 fb_alloc_cmap+0x9c/0x150 age=8379756 pid=1 cpus=1
      1 fbcon_startup+0xd6/0x2f0 age=8379755 pid=1 cpus=1
      1 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379896 pid=1 cpus=0
      1 make_acpi_ec+0x1a/0xcb age=8379913 pid=1 cpus=0
      1 acpi_irq_stats_init+0x83/0x268 age=8379913 pid=1 cpus=0
      1 acpi_irq_stats_init+0xa7/0x268 age=8379913 pid=1 cpus=0
      5 alloc_tty_driver+0x1c/0x40 age=8309091/8365595/8379725 pid=1-2711 cpus=1
      2 vt_ioctl+0x1874/0x1c30 age=8356613/8356613/8356613 pid=1703 cpus=0
      8 set_inverse_transl+0xa0/0xb0 age=8355024/8367389/8379755 pid=1-1898 cpus=1
      2 con_clear_unimap+0x2a/0xd0 age=8355024/8367389/8379755 pid=1-1898 cpus=1
      7 vc_allocate+0x99/0x1b0 age=8356563/8358748/8368241 pid=1135-1729 cpus=0-1
      1 hpet_alloc+0x6b/0x3c0 age=8379798 pid=1 cpus=0
      4 uart_open+0x36c/0x4b0 age=8368845/8368914/8369120 pid=1059-1073 cpus=0-1
     43 bus_add_driver+0x4f/0x280 age=8303025/8373842/8379841 pid=1-2754 cpus=0-1
      9 bus_register+0x35/0x290 age=8373434/8378695/8379923 pid=1-770 cpus=0-1
     30 __class_register+0x28/0x1e0 age=8374523/8378099/8379922 pid=1-352 cpus=0-1
      1 mousedev_open+0xc8/0x1d0 age=8354269 pid=1738 cpus=0
      1 cpufreq_add_dev+0x8f/0x5c0 age=8366381 pid=1358 cpus=0
      4 sock_alloc_send_skb+0x1cd/0x200 age=8330763/8342704/8350746 pid=1738 cpus=0-1
      1 alloc_netdev_mq+0x6c/0x180 age=8373712 pid=126 cpus=1
      4 inetdev_init+0x28/0x190 age=8373592/8375784/8379779 pid=1-126 cpus=0-1
      2 fib_hash_table+0x1c/0x80 age=8379779/8379779/8379779 pid=1 cpus=0
      1 ioapic_init_sysfs+0x74/0xcf age=8379769 pid=1 cpus=1
      3 snd_malloc_sgbuf_pages+0xbe/0x1f0 [snd_page_alloc] age=8374119/8374119/8374119 pid=107 cpus=1
      1 usb_get_configuration+0x18f/0x1480 [usbcore] age=22677 pid=419 cpus=0
      1 usb_get_configuration+0x5f2/0x1480 [usbcore] age=8372692 pid=419 cpus=0
      2 acpi_processor_get_throttling_info+0x21f/0x58c [processor] age=8375662/8375686/8375710 pid=330 cpus=0
     48 snd_info_create_entry+0x1b/0xa0 [snd] age=8355304/8371157/8374867 pid=107-1813 cpus=0-1
      4 snd_timer_new+0x40/0x190 [snd_timer] age=8374097/8374262/8374757 pid=107 cpus=1
      2 ieee80211_key_alloc+0x37/0xe0 [mac80211] age=3544694/3544694/3544694 pid=1545 cpus=0
      3 snd_pcm_new_stream+0x161/0x520 [snd_pcm] age=8374119/8374119/8374119 pid=107 cpus=1
      1 snd_hda_bus_new+0x31/0xf4 [snd_hda_intel] age=8374122 pid=107 cpus=1
      1 crypto_blkcipher_type+0x54/0xffffffffffffff8e [crypto_blkcipher] age=8374450 pid=215 cpus=0
      1 md_import_device+0x32/0x290 [md_mod] age=8370110 pid=916 cpus=0
      1 bitmap_create+0x5e/0xad0 [md_mod] age=8370066 pid=916 cpus=0
      1 run+0x42/0x500 [raid1] age=8370068 pid=916 cpus=0
      1 ip6_route_net_init+0x4d/0xd0 [ipv6] age=8360130 pid=1549 cpus=0
''')
f('sys/kernel/slab/kmalloc-512/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/kmalloc-512/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-512/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/object_size', 0o664, b'512\n')
f('sys/kernel/slab/kmalloc-512/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-512/objects_partial', 0o664, b'79\n')
f('sys/kernel/slab/kmalloc-512/objects', 0o664, b'443\n')
f('sys/kernel/slab/kmalloc-512/order', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-512/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-512/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-512/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-512/partial', 0o664, b'10\n')
f('sys/kernel/slab/kmalloc-512/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-512/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-512/free_calls', 0o664, b'''    163 <not-available> age=4303047353 pid=0 cpus=0
      4 __vunmap+0xe9/0x120 age=8374515/8375319/8376179 pid=122-215 cpus=0
      1 kmem_cache_shrink+0x5d/0x1b0 age=8379872 pid=1 cpus=0
     33 load_elf_binary+0xa7e/0x1e20 age=8303117/8365402/8377217 pid=62-2754 cpus=0-1
     19 load_elf_binary+0xc37/0x1e20 age=8355044/8370854/8376371 pid=197-1912 cpus=0-1
      1 crypto_larval_destroy+0x2b/0x40 age=8373593 pid=126 cpus=1
    210 skb_release_data+0x85/0xd0 age=22792/8177097/8379923 pid=1-20296 cpus=0-1
      1 acpi_processor_get_power_info+0x2eb/0x569 [processor] age=8375661 pid=330 cpus=1
      1 skcipher_geniv_alloc+0xea/0x3f0 [crypto_blkcipher] age=8374478 pid=215 cpus=0
      1 snd_mixer_oss_build_input+0x402/0x5c0 [snd_mixer_oss] age=8355509 pid=1813 cpus=0
''')
f('sys/kernel/slab/kmalloc-512/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-256', 0o775)
f('sys/kernel/slab/biovec-256/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-256/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/slab_size', 0o664, b'4224\n')
f('sys/kernel/slab/biovec-256/total_objects', 0o664, b'7\n')
f('sys/kernel/slab/biovec-256/slabs', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379465/8379465/8379465 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-256/objs_per_slab', 0o664, b'7\n')
f('sys/kernel/slab/biovec-256/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-256/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/object_size', 0o664, b'4096\n')
f('sys/kernel/slab/biovec-256/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/objects_partial', 0o664, b'2\n')
f('sys/kernel/slab/biovec-256/objects', 0o664, b'2\n')
f('sys/kernel/slab/biovec-256/order', 0o664, b'3\n')
f('sys/kernel/slab/biovec-256/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-256/partial', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/validate', 0o664, b'')
f('sys/kernel/slab/biovec-256/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-256/free_calls', 0o664, b'      2 <not-available> age=4303046899 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-256/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-128', 0o775)
f('sys/kernel/slab/kmalloc-128/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-128/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/slab_size', 0o664, b'200\n')
f('sys/kernel/slab/kmalloc-128/total_objects', 0o664, b'360\n')
f('sys/kernel/slab/kmalloc-128/slabs', 0o664, b'18\n')
f('sys/kernel/slab/kmalloc-128/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-128/alloc_calls', 0o664, b'''     17 param_sysfs_setup+0x87/0x140 age=8366402/8376536/8379939 pid=1-1358 cpus=0-1
      1 mempool_create_node+0x4b/0xf0 age=8379025 pid=1 cpus=0
      4 __vmalloc_area_node+0xfb/0x140 age=8303049/8349608/8370350 pid=900-2754 cpus=0
      2 shmem_fill_super+0x38/0x200 age=8377584/8378686/8379788 pid=1-46 cpus=0-1
     56 __percpu_alloc_mask+0xbc/0x140 age=61988/7483637/8379799 pid=1-13295 cpus=0-1
     11 cdev_alloc+0x1a/0x50 age=61988/7621848/8379938 pid=1-13295 cpus=0-1
    110 d_alloc+0x19d/0x1f0 age=22451/5346628/8376065 pid=401-13439 cpus=0-1
      2 bm_register_write+0x63/0x630 age=8354053/8354053/8354053 pid=2054 cpus=1
      2 alloc_disk_node+0x6c/0x120 age=62016/4220683/8379351 pid=1-13295 cpus=1
     20 ida_pre_get+0x86/0x90 age=8298034/8372144/8380076 pid=0-2761 cpus=0-1
      5 acpi_ds_build_internal_package_obj+0xaf/0x1df age=8379905/8379914/8379918 pid=1 cpus=0
      2 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379921/8379921/8379921 pid=1 cpus=0
      1 acpi_ev_create_gpe_block+0xb2/0x3bf age=8379933 pid=1 cpus=0
      4 acpi_add_single_object+0x5d8/0xd3c age=8379862/8379869/8379873 pid=1 cpus=0
     41 con_insert_unipair+0xd6/0x110 age=8355044/8369520/8379775 pid=1-1898 cpus=1
      2 thermal_zone_bind_cooling_device+0xde/0x2a0 age=8374828/8374828/8374828 pid=329 cpus=0
      4 led_trigger_register_simple+0x2b/0x80 age=8375615/8375659/8375793 pid=392-396 cpus=0
      1 sock_kmalloc+0x5c/0x70 age=8328344 pid=2586 cpus=0
      3 alloc_netdev_mq+0x6c/0x180 age=8373612/8376531/8379814 pid=1-126 cpus=0-1
      2 neigh_table_init_no_netlink+0xd2/0x250 age=8360154/8369976/8379799 pid=1-1549 cpus=0
      8 neigh_parms_alloc+0x5d/0x110 age=8360146/8367976/8379799 pid=1-1549 cpus=0-1
      4 fz_hash_alloc+0x4a/0x60 age=8325493/8357250/8367862 pid=1185-1541 cpus=0-1
      1 get_local_pda+0x39/0x9b age=8380015 pid=1 cpus=0
      1 mnt_init+0xf9/0x202 age=8380076 pid=0 cpus=0
      1 net_ns_init+0x49/0x16d age=8379942 pid=1 cpus=0
      3 sg_add+0x8c/0x400 [sg] age=61988/5604850/8376284 pid=169-13295 cpus=0
      3 usb_get_configuration+0x5f2/0x1480 [usbcore] age=22696/22696/22696 pid=419 cpus=0
     15 usb_get_configuration+0x1388/0x1480 [usbcore] age=22696/3922325/8372712 pid=419 cpus=0
      1 0xffffffffa0065117 age=8374777 pid=107 cpus=1
      1 ieee80211_led_init+0x1b/0x230 [mac80211] age=8373612 pid=126 cpus=1
      1 ieee80211_led_init+0x83/0x230 [mac80211] age=8373612 pid=126 cpus=1
      1 ieee80211_led_init+0xeb/0x230 [mac80211] age=8373611 pid=126 cpus=1
      1 ieee80211_led_init+0x153/0x230 [mac80211] age=8373611 pid=126 cpus=1
      1 snd_hda_codec_new+0x1ec/0x559 [snd_hda_intel] age=8374142 pid=107 cpus=1
      1 md_register_thread+0x35/0xd0 [md_mod] age=8370088 pid=916 cpus=0
      1 fib6_net_init+0x7e/0x130 [ipv6] age=8360149 pid=1549 cpus=0
''')
f('sys/kernel/slab/kmalloc-128/objs_per_slab', 0o664, b'20\n')
f('sys/kernel/slab/kmalloc-128/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-128/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/object_size', 0o664, b'128\n')
f('sys/kernel/slab/kmalloc-128/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/objects_partial', 0o664, b'83\n')
f('sys/kernel/slab/kmalloc-128/objects', 0o664, b'343\n')
f('sys/kernel/slab/kmalloc-128/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-128/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-128/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-128/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-128/partial', 0o664, b'5\n')
f('sys/kernel/slab/kmalloc-128/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-128/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-128/free_calls', 0o664, b'''    197 <not-available> age=4303047372 pid=0 cpus=0
      3 call_usermodehelper_freeinfo+0x27/0x30 age=8360145/8367948/8373612 pid=745-1532 cpus=0
      2 __vunmap+0xe9/0x120 age=8375480/8375882/8376284 pid=105-169 cpus=0
      5 percpu_free+0x2d/0x60 age=7838031/7838092/7838171 pid=419-3347 cpus=0
      1 cdev_dynamic_release+0x19/0x20 age=7838171 pid=419 cpus=0
     49 d_callback+0x22/0x40 age=53327/5302082/8375461 pid=0-13314 cpus=0-1
     51 load_elf_binary+0xaf/0x1e20 age=25315/7326461/8377585 pid=46-21952 cpus=0-1
      1 free_rb_tree_fname+0x5c/0xb0 age=8366816 pid=1274 cpus=0
      1 disk_release+0x2e/0x50 age=7838031 pid=3347 cpus=0
      2 free_bitmap+0x29/0x80 age=8368053/8369024/8369995 pid=55 cpus=0-1
      1 acpi_evaluate_reference+0x100/0x117 age=8374829 pid=329 cpus=0
      6 acpi_get_object_info+0x1b6/0x1d4 age=8379736/8379841/8379873 pid=1 cpus=0-1
      4 do_acpi_find_child+0x48/0x51 age=8379209/8379658/8379850 pid=1 cpus=0-1
      1 unpack_to_rootfs+0x105/0xa39 age=8379790 pid=1 cpus=0
      1 sg_remove+0x1ca/0x240 [sg] age=7838171 pid=419 cpus=0
      6 usb_release_interface_cache+0x2c/0x60 [usbcore] age=7833192/7833192/7833192 pid=419 cpus=1
      2 usb_release_interface_cache+0x47/0x60 [usbcore] age=7833192/7833192/7833192 pid=419 cpus=1
      1 acpi_processor_get_throttling_info+0x15e/0x58c [processor] age=8375683 pid=330 cpus=0
''')
f('sys/kernel/slab/kmalloc-128/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ip_dst_cache', 0o775)
f('sys/kernel/slab/ip_dst_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/ctor', 0o664, b'')
f('sys/kernel/slab/ip_dst_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/ip_dst_cache/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/ip_dst_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/ip_dst_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/alloc_calls', 0o664, b'      1 dst_alloc+0x2b/0x90 age=920684 pid=1206 cpus=0\n')
f('sys/kernel/slab/ip_dst_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/ip_dst_cache/shrink', 0o664, b'')
f('sys/kernel/slab/ip_dst_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/object_size', 0o664, b'312\n')
f('sys/kernel/slab/ip_dst_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/objects', 0o664, b'42\n')
f('sys/kernel/slab/ip_dst_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/ip_dst_cache/validate', 0o664, b'')
f('sys/kernel/slab/ip_dst_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ip_dst_cache/free_calls', 0o664, b'      1 <not-available> age=4303046733 pid=0 cpus=0\n')
f('sys/kernel/slab/ip_dst_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/journal_head', 0o775)
f('sys/kernel/slab/journal_head/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/journal_head/ctor', 0o664, b'')
f('sys/kernel/slab/journal_head/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/slab_size', 0o664, b'168\n')
f('sys/kernel/slab/journal_head/total_objects', 0o664, b'7680\n')
f('sys/kernel/slab/journal_head/slabs', 0o664, b'320\n')
f('sys/kernel/slab/journal_head/poison', 0o664, b'1\n')
f('sys/kernel/slab/journal_head/alloc_calls', 0o664, b'   7662 journal_add_journal_head+0x9f/0x1b0 age=0/2379/651354 pid=1336-13466 cpus=0-1\n')
f('sys/kernel/slab/journal_head/objs_per_slab', 0o664, b'24\n')
f('sys/kernel/slab/journal_head/shrink', 0o664, b'')
f('sys/kernel/slab/journal_head/trace', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/object_size', 0o664, b'96\n')
f('sys/kernel/slab/journal_head/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/objects', 0o664, b'7680\n')
f('sys/kernel/slab/journal_head/order', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/journal_head/store_user', 0o664, b'1\n')
f('sys/kernel/slab/journal_head/align', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/partial', 0o664, b'0\n')
f('sys/kernel/slab/journal_head/validate', 0o664, b'')
f('sys/kernel/slab/journal_head/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/journal_head/free_calls', 0o664, b'''   7589 <not-available> age=4303047557 pid=0 cpus=0
     74 __journal_remove_journal_head+0xcb/0x160 age=1585/52253/654521 pid=32-13248 cpus=0-1
''')
f('sys/kernel/slab/journal_head/aliases', 0o664, b'0\n')
d('sys/kernel/slab/scsi_data_buffer', 0o775)
f('sys/kernel/slab/scsi_data_buffer/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/ctor', 0o664, b'')
f('sys/kernel/slab/scsi_data_buffer/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/slab_size', 0o664, b'96\n')
f('sys/kernel/slab/scsi_data_buffer/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/poison', 0o664, b'1\n')
f('sys/kernel/slab/scsi_data_buffer/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/scsi_data_buffer/objs_per_slab', 0o664, b'42\n')
f('sys/kernel/slab/scsi_data_buffer/shrink', 0o664, b'')
f('sys/kernel/slab/scsi_data_buffer/trace', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/object_size', 0o664, b'24\n')
f('sys/kernel/slab/scsi_data_buffer/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/objects', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/order', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/scsi_data_buffer/store_user', 0o664, b'1\n')
f('sys/kernel/slab/scsi_data_buffer/align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_data_buffer/validate', 0o664, b'')
f('sys/kernel/slab/scsi_data_buffer/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/scsi_data_buffer/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/scsi_data_buffer/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fuse_request', 0o775)
f('sys/kernel/slab/fuse_request/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/ctor', 0o664, b'')
f('sys/kernel/slab/fuse_request/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/slab_size', 0o664, b'728\n')
f('sys/kernel/slab/fuse_request/total_objects', 0o664, b'44\n')
f('sys/kernel/slab/fuse_request/slabs', 0o664, b'2\n')
f('sys/kernel/slab/fuse_request/poison', 0o664, b'1\n')
f('sys/kernel/slab/fuse_request/alloc_calls', 0o664, b'      1 fuse_request_alloc+0x1a/0x40 [fuse] age=8336120 pid=2476 cpus=0\n')
f('sys/kernel/slab/fuse_request/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/fuse_request/shrink', 0o664, b'')
f('sys/kernel/slab/fuse_request/trace', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/object_size', 0o664, b'656\n')
f('sys/kernel/slab/fuse_request/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/objects', 0o664, b'44\n')
f('sys/kernel/slab/fuse_request/order', 0o664, b'2\n')
f('sys/kernel/slab/fuse_request/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fuse_request/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fuse_request/align', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/partial', 0o664, b'0\n')
f('sys/kernel/slab/fuse_request/validate', 0o664, b'')
f('sys/kernel/slab/fuse_request/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fuse_request/free_calls', 0o664, b'      1 <not-available> age=4303047718 pid=0 cpus=0\n')
f('sys/kernel/slab/fuse_request/aliases', 0o664, b'0\n')
d('sys/kernel/slab/pid', 0o775)
f('sys/kernel/slab/pid/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/pid/ctor', 0o664, b'')
f('sys/kernel/slab/pid/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/pid/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/pid/total_objects', 0o664, b'252\n')
f('sys/kernel/slab/pid/slabs', 0o664, b'12\n')
f('sys/kernel/slab/pid/poison', 0o664, b'1\n')
f('sys/kernel/slab/pid/alloc_calls', 0o664, b'    207 alloc_pid+0x26/0x430 age=4190/8015525/8379960 pid=0-13212 cpus=0-1\n')
f('sys/kernel/slab/pid/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/pid/shrink', 0o664, b'')
f('sys/kernel/slab/pid/trace', 0o664, b'0\n')
f('sys/kernel/slab/pid/object_size', 0o664, b'80\n')
f('sys/kernel/slab/pid/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/pid/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/pid/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/pid/objects_partial', 0o664, b'131\n')
f('sys/kernel/slab/pid/objects', 0o664, b'236\n')
f('sys/kernel/slab/pid/order', 0o664, b'0\n')
f('sys/kernel/slab/pid/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/pid/store_user', 0o664, b'1\n')
f('sys/kernel/slab/pid/align', 0o664, b'8\n')
f('sys/kernel/slab/pid/partial', 0o664, b'7\n')
f('sys/kernel/slab/pid/validate', 0o664, b'')
f('sys/kernel/slab/pid/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/pid/free_calls', 0o664, b'''     69 <not-available> age=4303047306 pid=0 cpus=0
    138 put_pid+0x36/0x40 age=7228/7841898/8377070 pid=0-13258 cpus=0-1
''')
f('sys/kernel/slab/pid/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ext2_inode_cache', 0o775)
f('sys/kernel/slab/ext2_inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/ext2_inode_cache/ctor', 0o664, b'init_once+0x0/0x60\n')
f('sys/kernel/slab/ext2_inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/slab_size', 0o664, b'1456\n')
f('sys/kernel/slab/ext2_inode_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/ext2_inode_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/ext2_inode_cache/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/ext2_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/ext2_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/object_size', 0o664, b'1384\n')
f('sys/kernel/slab/ext2_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/ext2_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ext2_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ext2_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/ext2_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/ext2_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ext2_inode_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/ext2_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-1024', 0o775)
f('sys/kernel/slab/kmalloc-1024/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-1024/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/slab_size', 0o664, b'1096\n')
f('sys/kernel/slab/kmalloc-1024/total_objects', 0o664, b'580\n')
f('sys/kernel/slab/kmalloc-1024/slabs', 0o664, b'20\n')
f('sys/kernel/slab/kmalloc-1024/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-1024/alloc_calls', 0o664, b'''     23 __register_sysctl_paths+0x56/0x340 age=8360116/8368981/8379769 pid=1-1549 cpus=0-1
      5 param_sysfs_setup+0x87/0x140 age=8355550/8371926/8379909 pid=1-1805 cpus=0-1
      5 load_module+0x18d5/0x1b30 age=8328407/8365286/8375764 pid=208-2578 cpus=0
      2 __vmalloc_area_node+0xfb/0x140 age=8360170/8367347/8374524 pid=107-1549 cpus=0
      4 __percpu_alloc_mask+0xbc/0x140 age=8379760/8379760/8379760 pid=1 cpus=0
    113 alloc_pipe_info+0x25/0x60 age=125624/8270722/8377065 pid=1-13212 cpus=0-1
      2 alloc_fdtable+0x81/0x160 age=2417000/5386894/8356789 pid=1690 cpus=0-1
      1 mb_cache_create+0xb6/0x1f0 age=8379750 pid=1 cpus=1
      1 mb_cache_create+0x11e/0x1f0 age=8379750 pid=1 cpus=1
      8 add_partition+0x45/0x250 age=61959/7339549/8379206 pid=1-13295 cpus=0
      2 __crypto_alloc_tfm+0x43/0x1b0 age=3544684/3544684/3544684 pid=1545 cpus=0
      1 pci_create_bus+0x82/0x240 age=8379831 pid=1 cpus=0
     14 pcie_port_device_register+0x254/0x4a0 age=8379746/8379746/8379748 pid=1 cpus=1
      1 acpi_ev_create_gpe_block+0x10b/0x3bf age=8379903 pid=1 cpus=0
      1 acpi_tb_resize_root_table_list+0x66/0xdb age=8375731 pid=330 cpus=0
      2 tty_buffer_request_room+0x136/0x190 age=8214639/8272696/8330753 pid=0-2262 cpus=0
      1 tty_register_driver+0x1b3/0x2a0 age=8309081 pid=2711 cpus=1
     10 tty_write+0x160/0x280 age=7214010/8238951/8377659 pid=34-2590 cpus=0-1
      2 set_inverse_trans_unicode+0xce/0xf0 age=8355014/8367379/8379745 pid=1-1898 cpus=1
      1 uart_register_driver+0x31/0x1c0 age=8379710 pid=1 cpus=1
    166 device_create_vargs+0x8c/0x100 age=22658/8173620/8379911 pid=1-13295 cpus=0-1
      8 platform_device_alloc+0x37/0x90 age=8367172/8376785/8379911 pid=1-1285 cpus=0-1
      3 scsi_alloc_target+0x63/0x2e0 age=61990/5606770/8379325 pid=1-13295 cpus=0-1
      2 sd_probe+0x86/0x430 age=61986/4220653/8379321 pid=1-13295 cpus=1
      1 ahci_port_start+0x34/0xe0 age=8379708 pid=1 cpus=1
      1 psmouse_connect+0x5b/0x2c0 age=8378898 pid=17 cpus=1
      2 thermal_cooling_device_register+0x7f/0x290 age=8375643/8375667/8375692 pid=330 cpus=0-1
      1 hid_add_field+0xcc/0x3a0 age=53473 pid=419 cpus=0
      4 sock_alloc_send_skb+0x1cd/0x200 age=8344775/8349266/8350926 pid=1738 cpus=0-1
      3 __rtnl_register+0x79/0x90 age=8360119/8373266/8379912 pid=1-1549 cpus=0
      6 qdisc_alloc+0x34/0xd0 age=8356043/8356079/8356257 pid=1541 cpus=0
      1 dmi_id_init+0x2a4/0x30c age=8379910 pid=1 cpus=0
      5 hub_probe+0xdd/0x820 [usbcore] age=8373999/8374492/8374845 pid=143-147 cpus=0-1
      5 usb_create_hcd+0x39/0x150 [usbcore] age=8374071/8374526/8374892 pid=143-147 cpus=0
     26 usb_set_configuration+0x441/0x5f0 [usbcore] age=22663/3237429/8374857 pid=143-419 cpus=0-1
     10 usb_get_configuration+0x92/0x1480 [usbcore] age=22667/5876069/8374865 pid=143-419 cpus=0-1
     34 usb_create_ep_files+0x51/0x350 [usbcore] age=22579/5431762/8374732 pid=143-419 cpus=0-1
      1 acpi_battery_add+0x29/0x1d3 [battery] age=8375630 pid=396 cpus=0
      1 snd_card_new+0x38/0x370 [snd] age=8374136 pid=107 cpus=0
      1 sta_info_alloc+0x3c/0x1d0 [mac80211] age=8329523 pid=742 cpus=0
      2 snd_pcm_new+0x47/0x130 [snd_pcm] age=8374109/8374109/8374109 pid=107 cpus=1
      1 rs_alloc_sta+0x25/0x120 [iwl3945] age=8329523 pid=742 cpus=0
      1 get_alloc_hash+0x9b/0x140 [snd_hda_intel] age=8373108 pid=786 cpus=0
      1 azx_probe+0xb3/0xc20 [snd_hda_intel] age=8374129 pid=107 cpus=0
      1 azx_probe+0x4a9/0xc20 [snd_hda_intel] age=8374122 pid=107 cpus=0
      1 snd_hda_codec_new+0x3e/0x559 [snd_hda_intel] age=8374112 pid=107 cpus=1
      1 md_probe+0xa1/0x360 [md_mod] age=8370183 pid=916 cpus=0
      8 loop_alloc+0x28/0x140 [loop] age=8369639/8369639/8369639 pid=951 cpus=0
      4 ipv6_add_dev+0x69/0x330 [ipv6] age=8360116/8360117/8360119 pid=1549 cpus=0
      1 ipv6_route_sysctl_init+0x22/0xb0 [ipv6] age=8360115 pid=1549 cpus=0
      2 snd_seq_create_port+0x35/0x180 [snd_seq] age=8355549/8355549/8355549 pid=1805 cpus=0
''')
f('sys/kernel/slab/kmalloc-1024/objs_per_slab', 0o664, b'29\n')
f('sys/kernel/slab/kmalloc-1024/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-1024/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/object_size', 0o664, b'1024\n')
f('sys/kernel/slab/kmalloc-1024/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-1024/objects_partial', 0o664, b'196\n')
f('sys/kernel/slab/kmalloc-1024/objects', 0o664, b'544\n')
f('sys/kernel/slab/kmalloc-1024/order', 0o664, b'3\n')
f('sys/kernel/slab/kmalloc-1024/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-1024/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-1024/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-1024/partial', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-1024/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-1024/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-1024/free_calls', 0o664, b'''    216 <not-available> age=4303047343 pid=0 cpus=0
      1 __vunmap+0xe9/0x120 age=8360137 pid=1549 cpus=0
     64 __free_pipe_info+0x5c/0x70 age=55301/8227248/8377668 pid=34-13363 cpus=0-1
     19 acpi_ds_delete_walk_state+0xc9/0xcd age=8355814/8376228/8380000 pid=0-1772 cpus=0-1
      1 acpi_tb_resize_root_table_list+0xbf/0xdb age=8375732 pid=330 cpus=0
      2 release_one_tty+0xc7/0x190 age=8352476/8361380/8370284 pid=55-1253 cpus=0-1
      1 vt_ioctl+0x273/0x1c30 age=8356598 pid=1703 cpus=0
     14 device_create_release+0x9/0x10 age=8352395/8354628/8356598 pid=1703-2139 cpus=0-1
      1 fw_dev_release+0x20/0x30 age=8356148 pid=1541 cpus=0
    184 skb_release_data+0x85/0xd0 age=22639/7036434/8379911 pid=0-13303 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-1024/aliases', 0o664, b'0\n')
d('sys/kernel/slab/files_cache', 0o775)
f('sys/kernel/slab/files_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/ctor', 0o664, b'')
f('sys/kernel/slab/files_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/files_cache/slab_size', 0o664, b'896\n')
f('sys/kernel/slab/files_cache/total_objects', 0o664, b'108\n')
f('sys/kernel/slab/files_cache/slabs', 0o664, b'6\n')
f('sys/kernel/slab/files_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/files_cache/alloc_calls', 0o664, b'     80 dup_fd+0x2e/0x420 age=4136/7769280/8379905 pid=0-12336 cpus=0-1\n')
f('sys/kernel/slab/files_cache/objs_per_slab', 0o664, b'18\n')
f('sys/kernel/slab/files_cache/shrink', 0o664, b'')
f('sys/kernel/slab/files_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/object_size', 0o664, b'768\n')
f('sys/kernel/slab/files_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/objects_partial', 0o664, b'29\n')
f('sys/kernel/slab/files_cache/objects', 0o664, b'101\n')
f('sys/kernel/slab/files_cache/order', 0o664, b'2\n')
f('sys/kernel/slab/files_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/files_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/files_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/files_cache/partial', 0o664, b'2\n')
f('sys/kernel/slab/files_cache/validate', 0o664, b'')
f('sys/kernel/slab/files_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/files_cache/free_calls', 0o664, b'''      9 <not-available> age=4303047250 pid=0 cpus=0
     31 put_files_struct+0xb4/0xe0 age=7177/7367129/8366999 pid=1202-20292 cpus=0-1
     40 free_fdtable_rcu+0xa0/0xb0 age=61634/7953135/8376143 pid=0-12335 cpus=0-1
''')
f('sys/kernel/slab/files_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/UDP-Lite', 0o775)
f('sys/kernel/slab/UDP-Lite/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/ctor', 0o664, b'')
f('sys/kernel/slab/UDP-Lite/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/UDP-Lite/slab_size', 0o664, b'1344\n')
f('sys/kernel/slab/UDP-Lite/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/poison', 0o664, b'1\n')
f('sys/kernel/slab/UDP-Lite/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDP-Lite/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/UDP-Lite/shrink', 0o664, b'')
f('sys/kernel/slab/UDP-Lite/trace', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/object_size', 0o664, b'1232\n')
f('sys/kernel/slab/UDP-Lite/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/objects', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/order', 0o664, b'2\n')
f('sys/kernel/slab/UDP-Lite/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/UDP-Lite/store_user', 0o664, b'1\n')
f('sys/kernel/slab/UDP-Lite/align', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/partial', 0o664, b'0\n')
f('sys/kernel/slab/UDP-Lite/validate', 0o664, b'')
f('sys/kernel/slab/UDP-Lite/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/UDP-Lite/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDP-Lite/aliases', 0o664, b'0\n')
d('sys/kernel/slab/mnt_cache', 0o775)
f('sys/kernel/slab/mnt_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/ctor', 0o664, b'')
f('sys/kernel/slab/mnt_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/mnt_cache/total_objects', 0o664, b'36\n')
f('sys/kernel/slab/mnt_cache/slabs', 0o664, b'3\n')
f('sys/kernel/slab/mnt_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/alloc_calls', 0o664, b'     25 alloc_vfsmnt+0x1f/0x180 age=61162/8041266/8379801 pid=0-13348 cpus=0-1\n')
f('sys/kernel/slab/mnt_cache/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/mnt_cache/shrink', 0o664, b'')
f('sys/kernel/slab/mnt_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/object_size', 0o664, b'224\n')
f('sys/kernel/slab/mnt_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/objects_partial', 0o664, b'4\n')
f('sys/kernel/slab/mnt_cache/objects', 0o664, b'28\n')
f('sys/kernel/slab/mnt_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/mnt_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/validate', 0o664, b'')
f('sys/kernel/slab/mnt_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/mnt_cache/free_calls', 0o664, b'''     24 <not-available> age=4303047098 pid=0 cpus=0
      1 free_vfsmnt+0x2c/0x40 age=7837771 pid=3347 cpus=0
''')
f('sys/kernel/slab/mnt_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/skbuff_head_cache', 0o775)
f('sys/kernel/slab/skbuff_head_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/ctor', 0o664, b'')
f('sys/kernel/slab/skbuff_head_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_head_cache/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/skbuff_head_cache/total_objects', 0o664, b'588\n')
f('sys/kernel/slab/skbuff_head_cache/slabs', 0o664, b'49\n')
f('sys/kernel/slab/skbuff_head_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_head_cache/alloc_calls', 0o664, b'    550 __alloc_skb+0x44/0x150 age=1665/3999351/8355948 pid=733-1738 cpus=0-1\n')
f('sys/kernel/slab/skbuff_head_cache/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/skbuff_head_cache/shrink', 0o664, b'')
f('sys/kernel/slab/skbuff_head_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/object_size', 0o664, b'208\n')
f('sys/kernel/slab/skbuff_head_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/objects_partial', 0o664, b'18\n')
f('sys/kernel/slab/skbuff_head_cache/objects', 0o664, b'570\n')
f('sys/kernel/slab/skbuff_head_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_head_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_head_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_head_cache/partial', 0o664, b'3\n')
f('sys/kernel/slab/skbuff_head_cache/validate', 0o664, b'')
f('sys/kernel/slab/skbuff_head_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_head_cache/free_calls', 0o664, b'''    333 <not-available> age=4303046979 pid=0 cpus=0
    217 __kfree_skb+0x3a/0xa0 age=1666/662694/8356220 pid=0-13466 cpus=0-1
''')
f('sys/kernel/slab/skbuff_head_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fuse_inode', 0o775)
f('sys/kernel/slab/fuse_inode/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/ctor', 0o664, b'fuse_inode_init_once+0x0/0x10 [fuse]\n')
f('sys/kernel/slab/fuse_inode/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/slab_size', 0o664, b'1344\n')
f('sys/kernel/slab/fuse_inode/total_objects', 0o664, b'12\n')
f('sys/kernel/slab/fuse_inode/slabs', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/poison', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/alloc_calls', 0o664, b'      1 fuse_alloc_inode+0x1a/0xe0 [fuse] age=8336111 pid=2476 cpus=0\n')
f('sys/kernel/slab/fuse_inode/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/fuse_inode/shrink', 0o664, b'')
f('sys/kernel/slab/fuse_inode/trace', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/object_size', 0o664, b'1248\n')
f('sys/kernel/slab/fuse_inode/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/objects_partial', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/objects', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/order', 0o664, b'2\n')
f('sys/kernel/slab/fuse_inode/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/align', 0o664, b'0\n')
f('sys/kernel/slab/fuse_inode/partial', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/validate', 0o664, b'')
f('sys/kernel/slab/fuse_inode/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fuse_inode/free_calls', 0o664, b'      1 <not-available> age=4303047710 pid=0 cpus=0\n')
f('sys/kernel/slab/fuse_inode/aliases', 0o664, b'0\n')
d('sys/kernel/slab/skbuff_fclone_cache', 0o775)
f('sys/kernel/slab/skbuff_fclone_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/ctor', 0o664, b'')
f('sys/kernel/slab/skbuff_fclone_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/slab_size', 0o664, b'512\n')
f('sys/kernel/slab/skbuff_fclone_cache/total_objects', 0o664, b'32\n')
f('sys/kernel/slab/skbuff_fclone_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/skbuff_fclone_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/skbuff_fclone_cache/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/skbuff_fclone_cache/shrink', 0o664, b'')
f('sys/kernel/slab/skbuff_fclone_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/object_size', 0o664, b'420\n')
f('sys/kernel/slab/skbuff_fclone_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/objects', 0o664, b'32\n')
f('sys/kernel/slab/skbuff_fclone_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/skbuff_fclone_cache/validate', 0o664, b'')
f('sys/kernel/slab/skbuff_fclone_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/skbuff_fclone_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/skbuff_fclone_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/buffer_head', 0o775)
f('sys/kernel/slab/buffer_head/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/buffer_head/ctor', 0o664, b'init_buffer_head+0x0/0x20\n')
f('sys/kernel/slab/buffer_head/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/slab_size', 0o664, b'176\n')
f('sys/kernel/slab/buffer_head/total_objects', 0o664, b'31487\n')
f('sys/kernel/slab/buffer_head/slabs', 0o664, b'1369\n')
f('sys/kernel/slab/buffer_head/poison', 0o664, b'1\n')
f('sys/kernel/slab/buffer_head/alloc_calls', 0o664, b'  31488 alloc_buffer_head+0x19/0x50 age=1/3079345/8378152 pid=1-32767 cpus=0-1\n')
f('sys/kernel/slab/buffer_head/objs_per_slab', 0o664, b'23\n')
f('sys/kernel/slab/buffer_head/shrink', 0o664, b'')
f('sys/kernel/slab/buffer_head/trace', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/object_size', 0o664, b'104\n')
f('sys/kernel/slab/buffer_head/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/objects', 0o664, b'31487\n')
f('sys/kernel/slab/buffer_head/order', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/buffer_head/store_user', 0o664, b'1\n')
f('sys/kernel/slab/buffer_head/align', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/partial', 0o664, b'0\n')
f('sys/kernel/slab/buffer_head/validate', 0o664, b'')
f('sys/kernel/slab/buffer_head/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/buffer_head/free_calls', 0o664, b'''  14436 <not-available> age=4303047199 pid=0 cpus=0
  17053 free_buffer_head+0x20/0x40 age=1227/1928220/8379062 pid=1-32759 cpus=0-1
''')
f('sys/kernel/slab/buffer_head/aliases', 0o664, b'0\n')
d('sys/kernel/slab/blkdev_queue', 0o775)
f('sys/kernel/slab/blkdev_queue/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/ctor', 0o664, b'')
f('sys/kernel/slab/blkdev_queue/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/slab_size', 0o664, b'2496\n')
f('sys/kernel/slab/blkdev_queue/total_objects', 0o664, b'26\n')
f('sys/kernel/slab/blkdev_queue/slabs', 0o664, b'2\n')
f('sys/kernel/slab/blkdev_queue/poison', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_queue/alloc_calls', 0o664, b'     12 blk_alloc_queue_node+0x1c/0xc0 age=61520/7678497/8378855 pid=1-13295 cpus=0-1\n')
f('sys/kernel/slab/blkdev_queue/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/blkdev_queue/shrink', 0o664, b'')
f('sys/kernel/slab/blkdev_queue/trace', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/object_size', 0o664, b'2424\n')
f('sys/kernel/slab/blkdev_queue/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/objects_partial', 0o664, b'10\n')
f('sys/kernel/slab/blkdev_queue/objects', 0o664, b'23\n')
f('sys/kernel/slab/blkdev_queue/order', 0o664, b'3\n')
f('sys/kernel/slab/blkdev_queue/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_queue/store_user', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_queue/align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_queue/partial', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_queue/validate', 0o664, b'')
f('sys/kernel/slab/blkdev_queue/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_queue/free_calls', 0o664, b'     12 <not-available> age=4303046873 pid=0 cpus=0\n')
f('sys/kernel/slab/blkdev_queue/aliases', 0o664, b'0\n')
d('sys/kernel/slab/RAW', 0o775)
f('sys/kernel/slab/RAW/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/RAW/ctor', 0o664, b'')
f('sys/kernel/slab/RAW/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/RAW/slab_size', 0o664, b'1280\n')
f('sys/kernel/slab/RAW/total_objects', 0o664, b'12\n')
f('sys/kernel/slab/RAW/slabs', 0o664, b'1\n')
f('sys/kernel/slab/RAW/poison', 0o664, b'1\n')
f('sys/kernel/slab/RAW/alloc_calls', 0o664, b'      3 sk_prot_alloc+0x1e/0xb0 age=8379168/8379171/8379177 pid=1 cpus=0\n')
f('sys/kernel/slab/RAW/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/RAW/shrink', 0o664, b'')
f('sys/kernel/slab/RAW/trace', 0o664, b'0\n')
f('sys/kernel/slab/RAW/object_size', 0o664, b'1208\n')
f('sys/kernel/slab/RAW/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/RAW/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/RAW/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/RAW/objects_partial', 0o664, b'3\n')
f('sys/kernel/slab/RAW/objects', 0o664, b'3\n')
f('sys/kernel/slab/RAW/order', 0o664, b'2\n')
f('sys/kernel/slab/RAW/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/RAW/store_user', 0o664, b'1\n')
f('sys/kernel/slab/RAW/align', 0o664, b'0\n')
f('sys/kernel/slab/RAW/partial', 0o664, b'1\n')
f('sys/kernel/slab/RAW/validate', 0o664, b'')
f('sys/kernel/slab/RAW/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/RAW/free_calls', 0o664, b'      3 <not-available> age=4303046751 pid=0 cpus=0\n')
f('sys/kernel/slab/RAW/aliases', 0o664, b'0\n')
d('sys/kernel/slab/blkdev_ioc', 0o775)
f('sys/kernel/slab/blkdev_ioc/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/ctor', 0o664, b'')
f('sys/kernel/slab/blkdev_ioc/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/slab_size', 0o664, b'200\n')
f('sys/kernel/slab/blkdev_ioc/total_objects', 0o664, b'100\n')
f('sys/kernel/slab/blkdev_ioc/slabs', 0o664, b'5\n')
f('sys/kernel/slab/blkdev_ioc/poison', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_ioc/alloc_calls', 0o664, b'     64 alloc_io_context+0x19/0xa0 age=3410/8058449/8378873 pid=1-13466 cpus=0-1\n')
f('sys/kernel/slab/blkdev_ioc/objs_per_slab', 0o664, b'20\n')
f('sys/kernel/slab/blkdev_ioc/shrink', 0o664, b'')
f('sys/kernel/slab/blkdev_ioc/trace', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/object_size', 0o664, b'128\n')
f('sys/kernel/slab/blkdev_ioc/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/objects_partial', 0o664, b'38\n')
f('sys/kernel/slab/blkdev_ioc/objects', 0o664, b'98\n')
f('sys/kernel/slab/blkdev_ioc/order', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_ioc/store_user', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_ioc/align', 0o664, b'0\n')
f('sys/kernel/slab/blkdev_ioc/partial', 0o664, b'2\n')
f('sys/kernel/slab/blkdev_ioc/validate', 0o664, b'')
f('sys/kernel/slab/blkdev_ioc/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/blkdev_ioc/free_calls', 0o664, b'''      9 <not-available> age=4303046891 pid=0 cpus=0
     55 put_io_context+0xb2/0xd0 age=45025/8043308/8374605 pid=145-13349 cpus=0-1
''')
f('sys/kernel/slab/blkdev_ioc/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sigqueue', 0o775)
f('sys/kernel/slab/sigqueue/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/ctor', 0o664, b'')
f('sys/kernel/slab/sigqueue/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/slab_size', 0o664, b'232\n')
f('sys/kernel/slab/sigqueue/total_objects', 0o664, b'34\n')
f('sys/kernel/slab/sigqueue/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sigqueue/poison', 0o664, b'1\n')
f('sys/kernel/slab/sigqueue/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/sigqueue/objs_per_slab', 0o664, b'17\n')
f('sys/kernel/slab/sigqueue/shrink', 0o664, b'')
f('sys/kernel/slab/sigqueue/trace', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/object_size', 0o664, b'160\n')
f('sys/kernel/slab/sigqueue/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/objects', 0o664, b'34\n')
f('sys/kernel/slab/sigqueue/order', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sigqueue/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sigqueue/align', 0o664, b'8\n')
f('sys/kernel/slab/sigqueue/partial', 0o664, b'0\n')
f('sys/kernel/slab/sigqueue/validate', 0o664, b'')
f('sys/kernel/slab/sigqueue/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sigqueue/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/sigqueue/aliases', 0o664, b'0\n')
d('sys/kernel/slab/TCPv6', 0o775)
f('sys/kernel/slab/TCPv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/ctor', 0o664, b'')
f('sys/kernel/slab/TCPv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/TCPv6/slab_size', 0o664, b'2432\n')
f('sys/kernel/slab/TCPv6/total_objects', 0o664, b'26\n')
f('sys/kernel/slab/TCPv6/slabs', 0o664, b'2\n')
f('sys/kernel/slab/TCPv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/TCPv6/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/TCPv6/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/TCPv6/shrink', 0o664, b'')
f('sys/kernel/slab/TCPv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/object_size', 0o664, b'2336\n')
f('sys/kernel/slab/TCPv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/objects', 0o664, b'26\n')
f('sys/kernel/slab/TCPv6/order', 0o664, b'3\n')
f('sys/kernel/slab/TCPv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/TCPv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/TCPv6/align', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/partial', 0o664, b'0\n')
f('sys/kernel/slab/TCPv6/validate', 0o664, b'')
f('sys/kernel/slab/TCPv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/TCPv6/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/TCPv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kioctx', 0o775)
f('sys/kernel/slab/kioctx/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/ctor', 0o664, b'')
f('sys/kernel/slab/kioctx/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/slab_size', 0o664, b'640\n')
f('sys/kernel/slab/kioctx/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/slabs', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/poison', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kioctx/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/kioctx/shrink', 0o664, b'')
f('sys/kernel/slab/kioctx/trace', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/object_size', 0o664, b'512\n')
f('sys/kernel/slab/kioctx/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/objects', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/order', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/align', 0o664, b'8\n')
f('sys/kernel/slab/kioctx/partial', 0o664, b'0\n')
f('sys/kernel/slab/kioctx/validate', 0o664, b'')
f('sys/kernel/slab/kioctx/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kioctx/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kioctx/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-32', 0o775)
f('sys/kernel/slab/kmalloc-32/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-32/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/slab_size', 0o664, b'104\n')
f('sys/kernel/slab/kmalloc-32/total_objects', 0o664, b'624\n')
f('sys/kernel/slab/kmalloc-32/slabs', 0o664, b'16\n')
f('sys/kernel/slab/kmalloc-32/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-32/alloc_calls', 0o664, b'''      1 mtrr_file_add+0xb0/0xe0 age=8355511 pid=1738 cpus=0
      2 arch_acpi_processor_init_pdc+0x8e/0x180 age=8375727/8375758/8375789 pid=330 cpus=0
      1 pm_qos_add_requirement+0x34/0xe0 age=8376080 pid=122 cpus=1
     59 setup_modinfo_srcversion+0x19/0x30 age=8303075/8368435/8376316 pid=105-2754 cpus=0-1
     42 load_module+0x1455/0x1b30 age=8303075/8366312/8375832 pid=107-2754 cpus=0-1
    151 load_module+0x1ac4/0x1b30 age=8303071/8368871/8376310 pid=105-2754 cpus=0-1
      3 mempool_create_node+0x4b/0xf0 age=62046/5606826/8379381 pid=1-13295 cpus=0-1
      1 strndup_user+0x6d/0xc0 age=8374925 pid=107 cpus=0
     16 __vmalloc_area_node+0xfb/0x140 age=8328464/8366300/8376086 pid=107-2578 cpus=0-1
      1 do_kern_mount+0xca/0x110 age=8335800 pid=2476 cpus=0
      1 do_sys_poll+0xdb/0x450 age=1303 pid=1333 cpus=1
      1 alloc_vfsmnt+0x97/0x180 age=8335801 pid=2476 cpus=0
      1 single_open+0x3c/0xb0 age=8355816 pid=1738 cpus=0
     66 sysfs_new_dirent+0x10c/0x120 age=8360190/8377926/8379815 pid=1-1549 cpus=0-1
     30 do_shmat+0x169/0x410 age=8274161/8330033/8348616 pid=1738-2621 cpus=0-1
     21 register_blkdev+0x60/0x150 age=8369695/8377753/8379765 pid=1-951 cpus=0-1
     39 kvasprintf+0x55/0x90 age=8360190/8378048/8379814 pid=1-1549 cpus=0-1
      7 pci_save_state+0x1a4/0x250 age=8373637/8378406/8379803 pid=1-126 cpus=0-1
      2 pcim_enable_device+0x86/0xb0 age=8379255/8379510/8379765 pid=1 cpus=0-1
     15 acpi_ds_build_internal_package_obj+0xaf/0x1df age=8379932/8379938/8379947 pid=1 cpus=0
     21 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=3847501/7731852/8379949 pid=1-330 cpus=0
      1 acpi_install_gpe_handler+0xa7/0x13a age=8379959 pid=1 cpus=0
      1 acpi_ev_create_gpe_block+0x282/0x3bf age=8379959 pid=1 cpus=0
      1 acpi_pci_bind_root+0x8d/0x13e age=8379883 pid=1 cpus=0
     18 acpi_pci_bind+0x85/0x28d age=8379879/8379879/8379882 pid=1 cpus=0
     15 rand_initialize_disk+0x1d/0x30 age=62014/7263977/8379377 pid=1-13295 cpus=0-1
      1 ata_host_start+0xcb/0x1e0 age=8379764 pid=1 cpus=1
      1 hidinput_connect+0x2372/0x24e0 age=53529 pid=419 cpus=0
      1 sock_kmalloc+0x5c/0x70 age=8325518 pid=1849 cpus=1
      2 proto_register+0x7c/0x260 age=8360191/8370012/8379834 pid=1-1549 cpus=0
      2 neigh_hash_alloc+0x42/0x50 age=8325417/8327412/8329408 pid=0-1849 cpus=1
      1 nl_pid_hash_rehash+0x180/0x1a0 age=947530 pid=1189 cpus=0
     28 unix_bind+0xd8/0x3a0 age=8356458/8357177/8366824 pid=1336-1738 cpus=0-1
      3 ipc_init_proc_interface+0x2e/0x7c age=8379805/8379805/8379805 pid=1 cpus=1
     13 usb_cache_string+0x65/0xa0 [usbcore] age=22649/4521883/8374921 pid=143-419 cpus=0-1
      6 usb_get_configuration+0x18f/0x1480 [usbcore] age=67053/6989971/8374921 pid=143-419 cpus=0-1
      2 _snd_ctl_register_ioctl+0x2b/0x80 [snd] age=8374675/8374675/8374675 pid=107 cpus=0
      2 snd_oss_info_register+0x40/0xc0 [snd] age=8355350/8355452/8355555 pid=1813 cpus=0
      6 adjust_io+0xeb/0x120 [rsrc_nonstatic] age=8373110/8373182/8373542 pid=352-800 cpus=0
      5 adjust_memory+0xe5/0x1a0 [rsrc_nonstatic] age=8373110/8373278/8373535 pid=352-800 cpus=0
      1 ieee80211_sta_set_extra_ie+0x7d/0xe0 [mac80211] age=8329597 pid=1545 cpus=0
      2 ieee80211_rx_bss_info+0x4df/0xa20 [mac80211] age=8330170/8342783/8355397 pid=0 cpus=0
      2 ieee80211_rx_bss_info+0x568/0xa20 [mac80211] age=22736/4188782/8354829 pid=742-1951 cpus=0
      2 ieee80211_rx_bss_info+0x5f7/0xa20 [mac80211] age=22736/4189294/8355852 pid=742-1772 cpus=0
      1 rate_control_alloc+0x32/0x110 [mac80211] age=8373757 pid=126 cpus=1
      2 ieee80211_rate_control_register+0x72/0xf0 [mac80211] age=8374580/8374635/8374690 pid=126 cpus=0-1
      2 azx_probe+0x9a4/0xc20 [snd_hda_intel] age=8374165/8374165/8374165 pid=107 cpus=1
      1 async_chainiv_givencrypt+0x71/0x110 [crypto_blkcipher] age=8374560 pid=215 cpus=0
      1 run+0x66/0x500 [raid1] age=8370114 pid=916 cpus=0
      1 acpi_cpufreq_cpu_init+0x4d/0x45c [acpi_cpufreq] age=8366427 pid=1358 cpus=0
''')
f('sys/kernel/slab/kmalloc-32/objs_per_slab', 0o664, b'39\n')
f('sys/kernel/slab/kmalloc-32/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-32/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/object_size', 0o664, b'32\n')
f('sys/kernel/slab/kmalloc-32/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/objects_partial', 0o664, b'149\n')
f('sys/kernel/slab/kmalloc-32/objects', 0o664, b'617\n')
f('sys/kernel/slab/kmalloc-32/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-32/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-32/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-32/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-32/partial', 0o664, b'4\n')
f('sys/kernel/slab/kmalloc-32/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-32/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-32/free_calls', 0o664, b'''    409 <not-available> age=4303047399 pid=0 cpus=0
      4 __vunmap+0xe9/0x120 age=8366429/8372533/8374823 pid=215-1358 cpus=0-1
      1 vfs_rename+0x301/0x450 age=8274772 pid=2851 cpus=0
      8 do_sys_poll+0x32b/0x450 age=1304/6728883/8333312 pid=1333 cpus=0-1
      1 seq_release_private+0x2d/0x60 age=8352378 pid=2141 cpus=0
      5 single_release+0x2e/0x40 age=8343903/8358822/8366889 pid=1258-1806 cpus=0
      1 setxattr+0xa6/0x130 age=8355591 pid=1816 cpus=0
      6 bio_free_map_data+0x25/0x30 age=53571/4350975/8375998 pid=0-318 cpus=0
      6 remove_kevent+0x44/0x60 age=7219423/7961750/8346613 pid=2271-2423 cpus=0-1
     57 load_elf_binary+0xa72/0x1e20 age=22671/8213994/8376585 pid=102-13419 cpus=0-1
      1 generic_acl_set+0x15e/0x1a0 age=8355591 pid=1816 cpus=0
     67 kobject_uevent_env+0x11a/0x470 age=62016/8246870/8379965 pid=1-13295 cpus=0-1
      2 pci_get_subsys+0x74/0xa0 age=8375728/8375759/8375790 pid=330 cpus=0
      8 acpi_evaluate_integer+0xbf/0xd1 age=3847577/7810351/8379888 pid=1-2097 cpus=0
      3 acpi_ds_create_operand+0x12c/0x209 age=3847578/6867437/8379889 pid=1-2097 cpus=0
      3 acpi_ns_get_node+0x92/0xa1 age=8374447/8376288/8379960 pid=1-215 cpus=0-1
      5 acpi_ut_delete_internal_obj+0x15f/0x16f age=8366428/8373878/8375771 pid=330-1358 cpus=0
      3 acpi_pci_bind+0x277/0x28d age=8379881/8379881/8379883 pid=1 cpus=0
      1 get_modalias+0xd4/0x120 age=8355825 pid=1772 cpus=0
      2 ioctl_standard_iw_point+0x179/0x350 age=63360/84363/105367 pid=1541 cpus=0
      1 pci_bus_assign_resources+0xfe/0x4d0 age=8379836 pid=1 cpus=0
      1 sg_clean+0x3e/0x80 [usbcore] age=59566 pid=13294 cpus=0
      9 usb_get_device_descriptor+0x9b/0xa0 [usbcore] age=22724/6523118/8374922 pid=143-419 cpus=0-1
      1 ieee80211_rx_bss_info+0x552/0xa20 [mac80211] age=22737 pid=742 cpus=0
      2 ieee80211_rx_bss_info+0x5e1/0xa20 [mac80211] age=22737/4189295/8355853 pid=742-1772 cpus=0
''')
f('sys/kernel/slab/kmalloc-32/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-64', 0o775)
f('sys/kernel/slab/kmalloc-64/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-64/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/slab_size', 0o664, b'136\n')
f('sys/kernel/slab/kmalloc-64/total_objects', 0o664, b'15300\n')
f('sys/kernel/slab/kmalloc-64/slabs', 0o664, b'510\n')
f('sys/kernel/slab/kmalloc-64/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-64/alloc_calls', 0o664, b'''     30 alternatives_smp_module_add+0x81/0x160 age=8303055/8368397/8380085 pid=0-2754 cpus=0-1
     39 __request_region+0x39/0x120 age=8373811/8378273/8379949 pid=1-352 cpus=0-1
      1 audit_log_start+0x353/0x3f0 age=8379797 pid=1 cpus=1
     16 request_irq+0x9a/0xf0 age=8356279/8374005/8379942 pid=1-1541 cpus=0-1
      3 devm_request_irq+0x4b/0xc0 age=8379232/8379403/8379747 pid=1 cpus=0-1
      1 strndup_user+0x6d/0xc0 age=8374545 pid=215 cpus=0
     95 __get_vm_area_node+0x9b/0x220 age=8303058/8371551/8380085 pid=0-2754 cpus=0-1
     17 __vmalloc_area_node+0xfb/0x140 age=8309125/8366469/8376299 pid=105-2711 cpus=0-1
      1 add_swap_extent+0x57/0xc0 age=8372039 pid=867 cpus=0
     33 dma_pool_alloc+0x6d/0x200 age=8309120/8340771/8374904 pid=143-2711 cpus=0-1
     40 alloc_kmem_cache_cpu+0x81/0x90 age=8360159/8368022/8379787 pid=1-1549 cpus=0-1
    164 __percpu_alloc_mask+0x52/0x140 age=61997/8067718/8380085 pid=0-13295 cpus=0-1
    184 __percpu_alloc_mask+0xbc/0x140 age=62029/8100825/8380085 pid=0-13295 cpus=0-1
    604 d_alloc+0x19d/0x1f0 age=22685/7859083/8376462 pid=72-24701 cpus=0-1
     17 alloc_fdtable+0x57/0x160 age=4267/6101781/8357279 pid=1532-12336 cpus=0-1
     17 alloc_fdtable+0xb4/0x160 age=4267/6101781/8357279 pid=1532-12336 cpus=0-1
      1 bioset_create+0x21/0xc0 age=8379947 pid=1 cpus=0
      1 bd_claim_by_disk+0x6f/0x270 age=8370131 pid=916 cpus=0
     14 posix_acl_alloc+0x1b/0x30 age=53326/7759393/8352180 pid=2175-13390 cpus=0-1
      5 proc_reg_open+0x3f/0x170 age=8355799/8361268/8366755 pid=1327-1738 cpus=0
  12754 ext3_init_block_alloc_info+0x22/0x80 age=1/498494/8368528 pid=1-23373 cpus=0-1
      4 ext3_readdir+0x617/0x650 age=8352064/8352079/8352114 pid=1370 cpus=0
      8 ext3_htree_store_dirent+0x37/0x130 age=8352064/8352072/8352099 pid=1370 cpus=0
    277 kobject_create+0x1a/0x40 age=22618/7771015/8380085 pid=0-13295 cpus=0-1
      1 pci_create_sysfs_dev_files+0x23e/0x3b0 age=8378937 pid=1 cpus=0
     11 acpi_os_validate_address+0x3d/0xa3 age=8379914/8379927/8379932 pid=1 cpus=0
    124 acpi_ds_build_internal_package_obj+0xaf/0x1df age=3847484/8233379/8379936 pid=1-330 cpus=0
      6 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379920/8379925/8379934 pid=1 cpus=0
      1 acpi_ev_system_memory_region_setup+0x68/0x8f age=8379914 pid=1 cpus=0
      3 acpi_add_single_object+0x596/0xd3c age=8379878/8379880/8379882 pid=1 cpus=0
      3 acpi_add_single_object+0x5d8/0xd3c age=8379886/8379888/8379890 pid=1 cpus=0
     27 acpi_add_single_object+0x9f7/0xd3c age=8379871/8379888/8379900 pid=1 cpus=0
     55 acpi_ec_add_query_handler+0x31/0x91 age=8379870/8379870/8379870 pid=1 cpus=0
      1 find_dock+0x2c9/0x426 age=8379950 pid=1 cpus=0
      6 find_dock_devices+0x5e/0x92 age=8379950/8379950/8379950 pid=1 cpus=0
      8 acpi_pci_link_add+0x32/0x1db age=8379843/8379844/8379846 pid=1 cpus=0
     39 acpi_pci_irq_add_prt+0x181/0x324 age=8379862/8379863/8379865 pid=1 cpus=0
     15 init_dev+0x3c8/0x6f0 age=7217158/8282400/8378180 pid=33-2593 cpus=0-1
     15 init_dev+0x437/0x6f0 age=7217158/8282400/8378180 pid=33-2593 cpus=0-1
      3 init_dev+0x5b2/0x6f0 age=7217158/7969269/8364358 pid=1258-2593 cpus=0-1
      3 init_dev+0x637/0x6f0 age=7217158/7969269/8364358 pid=1258-2593 cpus=0-1
      2 kobj_map_init+0x36/0xa0 age=8379947/8380016/8380085 pid=0-1 cpus=0
     38 kobj_map+0x79/0x1a0 age=61997/7936136/8379947 pid=1-13295 cpus=0-1
      3 dmam_alloc_coherent+0x3f/0xa0 age=8379232/8379403/8379747 pid=1 cpus=0-1
      1 scsi_probe_and_add_lun+0x65e/0xd80 age=62028 pid=13295 cpus=0
    170 scsi_dev_info_list_add+0x3d/0x120 age=8379831/8379831/8379831 pid=1 cpus=0
      1 ahci_init_one+0x137/0xcb0 age=8379748 pid=1 cpus=1
      1 cpufreq_stat_notifier_policy+0xa1/0x2e0 age=8366410 pid=1358 cpus=0
      1 cpufreq_stat_notifier_policy+0x17e/0x2e0 age=8366410 pid=1358 cpus=0
      1 hid_parse_report+0xa9/0x2c0 age=53512 pid=419 cpus=0
     10 __dev_addr_add+0x69/0xe0 age=8325501/8345496/8360156 pid=8-1849 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-64/objs_per_slab', 0o664, b'30\n')
f('sys/kernel/slab/kmalloc-64/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-64/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/object_size', 0o664, b'64\n')
f('sys/kernel/slab/kmalloc-64/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/objects', 0o664, b'15300\n')
f('sys/kernel/slab/kmalloc-64/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-64/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-64/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-64/partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-64/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-64/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-64/free_calls', 0o664, b'''  13029 <not-available> age=4303047386 pid=0 cpus=0
      1 iounmap+0xc2/0x100 age=8379917 pid=1 cpus=0
      1 memtype_get_idx+0x75/0xc0 age=782 pid=13466 cpus=0
      2 __release_region+0xc6/0xd0 age=8379754/8379754/8379754 pid=1 cpus=1
      3 __request_region+0x86/0x120 age=8379828/8379828/8379828 pid=1 cpus=0
      1 generic_smp_call_function_single_interrupt+0xce/0xe0 age=8355499 pid=0 cpus=1
     53 __stop_machine+0x216/0x220 age=8303059/8367652/8376299 pid=105-2754 cpus=0-1
      2 free_irq+0xef/0x190 age=8356302/8367658/8379015 pid=1-1541 cpus=0
     28 __vunmap+0xcd/0x120 age=8309125/8366969/8376298 pid=105-2711 cpus=0-1
      8 __vunmap+0xe9/0x120 age=8328457/8364062/8376068 pid=107-2578 cpus=0-1
      1 percpu_free+0x2d/0x60 age=61987 pid=13295 cpus=0
     10 do_sys_poll+0x32b/0x450 age=61481/7500944/8334758 pid=1333-2262 cpus=0-1
    187 d_callback+0x22/0x40 age=22704/7997936/8376459 pid=0-2156 cpus=0-1
    198 free_fdtable_rcu+0x7a/0xb0 age=1292/3883175/8376371 pid=0-13456 cpus=0-1
    224 free_fdtable_rcu+0x82/0xb0 age=1292/3491449/8377150 pid=0-13456 cpus=0-1
     42 proc_reg_release+0x96/0x120 age=3564/7013222/8376067 pid=126-2586 cpus=0-1
     50 proc_reg_open+0x138/0x170 age=656297/2956669/8367924 pid=1184-12336 cpus=0-1
   1046 free_rb_tree_fname+0x5c/0xb0 age=5905/4315279/8376402 pid=102-24701 cpus=0-1
     79 ext3_htree_free_dir_info+0x19/0x20 age=5905/6235661/8376402 pid=102-24701 cpus=0-1
     26 ext3_clear_inode+0xa5/0xb0 age=12954/2421572/8354304 pid=1253-8445 cpus=0-1
      1 dynamic_kobj_release+0x2c/0x40 age=8356191 pid=1541 cpus=0
     66 kobject_uevent_env+0x11a/0x470 age=22624/7608199/8379955 pid=1-2711 cpus=0-1
     16 acpi_ds_call_control_method+0xfe/0x180 age=3020/4710292/8379848 pid=1-13466 cpus=0-1
      1 acpi_ev_execute_reg_method+0x129/0x13a age=8379946 pid=1 cpus=0
      1 acpi_ev_asynch_execute_gpe_method+0xcc/0x119 age=8379905 pid=12 cpus=1
     33 acpi_evaluate_object+0x1ea/0x1fc age=3015/5838124/8379955 pid=1-13466 cpus=0-1
      4 acpi_get_object_info+0x1c4/0x1d4 age=8379876/8379888/8379895 pid=1 cpus=0
      1 acpi_walk_resources+0xa4/0xbd age=8379755 pid=1 cpus=1
     41 acpi_ut_evaluate_object+0x18d/0x19b age=8379223/8379861/8379919 pid=1 cpus=0-1
      4 acpi_ut_delete_internal_obj+0x15f/0x16f age=8366416/8373412/8375744 pid=330-1358 cpus=0
      7 acpi_bus_get_ejd+0x73/0x80 age=8379955/8379955/8379955 pid=1 cpus=0
     11 pnp_show_options+0x459/0x600 age=1578/1602/1627 pid=13466 cpus=0
     11 pnp_show_current_resources+0x16f/0x180 age=1578/1603/1628 pid=13466 cpus=0
     19 release_one_tty+0x53/0x190 age=804540/7566113/8370327 pid=55-2106 cpus=0-1
     24 release_one_tty+0x60/0x190 age=804540/7729452/8370327 pid=55-2141 cpus=0-1
      6 release_one_tty+0x167/0x190 age=8352429/8353224/8354794 pid=1967-2141 cpus=0
      4 devres_remove_group+0x9e/0xe0 age=8379026/8379313/8379753 pid=1 cpus=0-1
      1 ata_acpi_on_devcfg+0x385/0x6a0 age=8379406 pid=25 cpus=0
      3 serio_free_event+0x1d/0x30 age=8378463/8378812/8378999 pid=17 cpus=0-1
      6 input_register_device+0x17c/0x230 age=53514/6988336/8378983 pid=17-803 cpus=0-1
      1 ip_cork_release+0x1f/0x50 age=8333854 pid=2536 cpus=1
      1 unix_sock_destructor+0x75/0xd0 age=7217178 pid=2344 cpus=0
     13 huft_free+0x1e/0x2f age=8379804/8379804/8379804 pid=1 cpus=0
      6 hub_port_init+0x522/0x740 [usbcore] age=22825/4210550/8374275 pid=419 cpus=0-1
      1 usb_release_interface_cache+0x2c/0x60 [usbcore] age=8373868 pid=419 cpus=0
      1 usb_release_interface_cache+0x47/0x60 [usbcore] age=8373868 pid=419 cpus=0
      1 __param_str_act+0x16c/0xfffffffffffffb13 [thermal] age=8375113 pid=383 cpus=0
      3 snd_card_file_remove+0x73/0x130 [snd] age=8333050/8337203/8339297 pid=2400-2425 cpus=0
      2 hid_probe+0x222/0xf30 [usbhid] age=53517/4212911/8372306 pid=419-803 cpus=0
''')
f('sys/kernel/slab/kmalloc-64/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sgpool-128', 0o775)
f('sys/kernel/slab/sgpool-128/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/ctor', 0o664, b'')
f('sys/kernel/slab/sgpool-128/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-128/slab_size', 0o664, b'4224\n')
f('sys/kernel/slab/sgpool-128/total_objects', 0o664, b'14\n')
f('sys/kernel/slab/sgpool-128/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-128/poison', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-128/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379262/8379262/8379262 pid=1 cpus=0\n')
f('sys/kernel/slab/sgpool-128/objs_per_slab', 0o664, b'7\n')
f('sys/kernel/slab/sgpool-128/shrink', 0o664, b'')
f('sys/kernel/slab/sgpool-128/trace', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/object_size', 0o664, b'4096\n')
f('sys/kernel/slab/sgpool-128/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/objects', 0o664, b'14\n')
f('sys/kernel/slab/sgpool-128/order', 0o664, b'3\n')
f('sys/kernel/slab/sgpool-128/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-128/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-128/align', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-128/validate', 0o664, b'')
f('sys/kernel/slab/sgpool-128/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-128/free_calls', 0o664, b'      2 <not-available> age=4303046812 pid=0 cpus=0\n')
f('sys/kernel/slab/sgpool-128/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kcopyd_job', 0o775)
f('sys/kernel/slab/kcopyd_job/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/ctor', 0o664, b'')
f('sys/kernel/slab/kcopyd_job/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/slab_size', 0o664, b'544\n')
f('sys/kernel/slab/kcopyd_job/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/slabs', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/poison', 0o664, b'1\n')
f('sys/kernel/slab/kcopyd_job/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kcopyd_job/objs_per_slab', 0o664, b'15\n')
f('sys/kernel/slab/kcopyd_job/shrink', 0o664, b'')
f('sys/kernel/slab/kcopyd_job/trace', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/object_size', 0o664, b'472\n')
f('sys/kernel/slab/kcopyd_job/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/objects', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/order', 0o664, b'1\n')
f('sys/kernel/slab/kcopyd_job/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kcopyd_job/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kcopyd_job/align', 0o664, b'8\n')
f('sys/kernel/slab/kcopyd_job/partial', 0o664, b'0\n')
f('sys/kernel/slab/kcopyd_job/validate', 0o664, b'')
f('sys/kernel/slab/kcopyd_job/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kcopyd_job/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kcopyd_job/aliases', 0o664, b'0\n')
d('sys/kernel/slab/dm_target_io', 0o775)
f('sys/kernel/slab/dm_target_io/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/ctor', 0o664, b'')
f('sys/kernel/slab/dm_target_io/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/slab_size', 0o664, b'96\n')
f('sys/kernel/slab/dm_target_io/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/slabs', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/poison', 0o664, b'1\n')
f('sys/kernel/slab/dm_target_io/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dm_target_io/objs_per_slab', 0o664, b'42\n')
f('sys/kernel/slab/dm_target_io/shrink', 0o664, b'')
f('sys/kernel/slab/dm_target_io/trace', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/object_size', 0o664, b'24\n')
f('sys/kernel/slab/dm_target_io/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/objects', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/order', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/dm_target_io/store_user', 0o664, b'1\n')
f('sys/kernel/slab/dm_target_io/align', 0o664, b'8\n')
f('sys/kernel/slab/dm_target_io/partial', 0o664, b'0\n')
f('sys/kernel/slab/dm_target_io/validate', 0o664, b'')
f('sys/kernel/slab/dm_target_io/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/dm_target_io/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dm_target_io/aliases', 0o664, b'0\n')
d('sys/kernel/slab/signal_cache', 0o775)
f('sys/kernel/slab/signal_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/ctor', 0o664, b'')
f('sys/kernel/slab/signal_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/signal_cache/slab_size', 0o664, b'1024\n')
f('sys/kernel/slab/signal_cache/total_objects', 0o664, b'160\n')
f('sys/kernel/slab/signal_cache/slabs', 0o664, b'10\n')
f('sys/kernel/slab/signal_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/signal_cache/alloc_calls', 0o664, b'    124 copy_process+0x93c/0x1220 age=4145/7851187/8379914 pid=0-12336 cpus=0-1\n')
f('sys/kernel/slab/signal_cache/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/signal_cache/shrink', 0o664, b'')
f('sys/kernel/slab/signal_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/object_size', 0o664, b'896\n')
f('sys/kernel/slab/signal_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/objects_partial', 0o664, b'70\n')
f('sys/kernel/slab/signal_cache/objects', 0o664, b'150\n')
f('sys/kernel/slab/signal_cache/order', 0o664, b'2\n')
f('sys/kernel/slab/signal_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/signal_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/signal_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/signal_cache/partial', 0o664, b'5\n')
f('sys/kernel/slab/signal_cache/validate', 0o664, b'')
f('sys/kernel/slab/signal_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/signal_cache/free_calls', 0o664, b'''     42 <not-available> age=4303047259 pid=0 cpus=0
     82 __cleanup_signal+0x20/0x30 age=7186/7586663/8375843 pid=1-13288 cpus=0-1
''')
f('sys/kernel/slab/signal_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/radix_tree_node', 0o775)
f('sys/kernel/slab/radix_tree_node/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/ctor', 0o664, b'radix_tree_node_ctor+0x0/0x10\n')
f('sys/kernel/slab/radix_tree_node/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/slab_size', 0o664, b'624\n')
f('sys/kernel/slab/radix_tree_node/total_objects', 0o664, b'6747\n')
f('sys/kernel/slab/radix_tree_node/slabs', 0o664, b'519\n')
f('sys/kernel/slab/radix_tree_node/poison', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/alloc_calls', 0o664, b'   6737 radix_tree_preload+0x3b/0xb0 age=16/6612251/8379044 pid=1-32767 cpus=0-1\n')
f('sys/kernel/slab/radix_tree_node/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/radix_tree_node/shrink', 0o664, b'')
f('sys/kernel/slab/radix_tree_node/trace', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/object_size', 0o664, b'552\n')
f('sys/kernel/slab/radix_tree_node/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/objects', 0o664, b'6747\n')
f('sys/kernel/slab/radix_tree_node/order', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/store_user', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/align', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/partial', 0o664, b'0\n')
f('sys/kernel/slab/radix_tree_node/validate', 0o664, b'')
f('sys/kernel/slab/radix_tree_node/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/radix_tree_node/free_calls', 0o664, b'''   4146 <not-available> age=4303047064 pid=0 cpus=0
   2591 radix_tree_node_rcu_free+0x41/0x50 age=39290/6239072/8378908 pid=0-32584 cpus=0-1
''')
f('sys/kernel/slab/radix_tree_node/aliases', 0o664, b'0\n')
d('sys/kernel/slab/uid_cache', 0o775)
f('sys/kernel/slab/uid_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/ctor', 0o664, b'')
f('sys/kernel/slab/uid_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/uid_cache/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/uid_cache/total_objects', 0o664, b'24\n')
f('sys/kernel/slab/uid_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/uid_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/uid_cache/alloc_calls', 0o664, b'      5 alloc_uid+0xbd/0x1e0 age=8351086/8358570/8366097 pid=1333-2177 cpus=0-1\n')
f('sys/kernel/slab/uid_cache/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/uid_cache/shrink', 0o664, b'')
f('sys/kernel/slab/uid_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/object_size', 0o664, b'216\n')
f('sys/kernel/slab/uid_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/objects', 0o664, b'24\n')
f('sys/kernel/slab/uid_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/uid_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/uid_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/uid_cache/validate', 0o664, b'')
f('sys/kernel/slab/uid_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/uid_cache/free_calls', 0o664, b'''      2 <not-available> age=4303046654 pid=0 cpus=0
      3 remove_user_sysfs_dir+0xd0/0x100 age=8357175/8367109/8374935 pid=7 cpus=0
''')
f('sys/kernel/slab/uid_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/tw_sock_TCPv6', 0o775)
f('sys/kernel/slab/tw_sock_TCPv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/ctor', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCPv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCPv6/slab_size', 0o664, b'256\n')
f('sys/kernel/slab/tw_sock_TCPv6/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/slabs', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCPv6/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/tw_sock_TCPv6/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/tw_sock_TCPv6/shrink', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCPv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/object_size', 0o664, b'168\n')
f('sys/kernel/slab/tw_sock_TCPv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/objects', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/order', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCPv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCPv6/align', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/partial', 0o664, b'0\n')
f('sys/kernel/slab/tw_sock_TCPv6/validate', 0o664, b'')
f('sys/kernel/slab/tw_sock_TCPv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/tw_sock_TCPv6/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/tw_sock_TCPv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sighand_cache', 0o775)
f('sys/kernel/slab/sighand_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sighand_cache/ctor', 0o664, b'sighand_ctor+0x0/0x40\n')
f('sys/kernel/slab/sighand_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/slab_size', 0o664, b'2304\n')
f('sys/kernel/slab/sighand_cache/total_objects', 0o664, b'154\n')
f('sys/kernel/slab/sighand_cache/slabs', 0o664, b'11\n')
f('sys/kernel/slab/sighand_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/alloc_calls', 0o664, b'''    123 copy_process+0x82a/0x1220 age=4154/7846897/8379923 pid=0-12336 cpus=0-1
      1 flush_old_exec+0x4e1/0x8a0 age=8378174 pid=1 cpus=0
''')
f('sys/kernel/slab/sighand_cache/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/sighand_cache/shrink', 0o664, b'')
f('sys/kernel/slab/sighand_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/sighand_cache/object_size', 0o664, b'2184\n')
f('sys/kernel/slab/sighand_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sighand_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sighand_cache/destroy_by_rcu', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/objects_partial', 0o664, b'61\n')
f('sys/kernel/slab/sighand_cache/objects', 0o664, b'145\n')
f('sys/kernel/slab/sighand_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/sighand_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/sighand_cache/partial', 0o664, b'5\n')
f('sys/kernel/slab/sighand_cache/validate', 0o664, b'')
f('sys/kernel/slab/sighand_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sighand_cache/free_calls', 0o664, b'''     46 <not-available> age=4303047268 pid=0 cpus=0
     78 __cleanup_sighand+0x27/0x30 age=7195/7547046/8375852 pid=1-13288 cpus=0-1
''')
f('sys/kernel/slab/sighand_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/names_cache', 0o775)
f('sys/kernel/slab/names_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/ctor', 0o664, b'')
f('sys/kernel/slab/names_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/names_cache/slab_size', 0o664, b'4224\n')
f('sys/kernel/slab/names_cache/total_objects', 0o664, b'14\n')
f('sys/kernel/slab/names_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/names_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/names_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/names_cache/objs_per_slab', 0o664, b'7\n')
f('sys/kernel/slab/names_cache/shrink', 0o664, b'')
f('sys/kernel/slab/names_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/object_size', 0o664, b'4096\n')
f('sys/kernel/slab/names_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/objects', 0o664, b'14\n')
f('sys/kernel/slab/names_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/names_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/names_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/names_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/names_cache/validate', 0o664, b'')
f('sys/kernel/slab/names_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/names_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/names_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/dm_io', 0o775)
f('sys/kernel/slab/dm_io/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/ctor', 0o664, b'')
f('sys/kernel/slab/dm_io/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/slab_size', 0o664, b'104\n')
f('sys/kernel/slab/dm_io/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/slabs', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/poison', 0o664, b'1\n')
f('sys/kernel/slab/dm_io/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dm_io/objs_per_slab', 0o664, b'39\n')
f('sys/kernel/slab/dm_io/shrink', 0o664, b'')
f('sys/kernel/slab/dm_io/trace', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/object_size', 0o664, b'32\n')
f('sys/kernel/slab/dm_io/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/objects', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/order', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/dm_io/store_user', 0o664, b'1\n')
f('sys/kernel/slab/dm_io/align', 0o664, b'8\n')
f('sys/kernel/slab/dm_io/partial', 0o664, b'0\n')
f('sys/kernel/slab/dm_io/validate', 0o664, b'')
f('sys/kernel/slab/dm_io/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/dm_io/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/dm_io/aliases', 0o664, b'0\n')
d('sys/kernel/slab/key_jar', 0o775)
f('sys/kernel/slab/key_jar/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/ctor', 0o664, b'')
f('sys/kernel/slab/key_jar/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/key_jar/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/key_jar/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/slabs', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/poison', 0o664, b'1\n')
f('sys/kernel/slab/key_jar/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/key_jar/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/key_jar/shrink', 0o664, b'')
f('sys/kernel/slab/key_jar/trace', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/object_size', 0o664, b'232\n')
f('sys/kernel/slab/key_jar/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/objects', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/order', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/key_jar/store_user', 0o664, b'1\n')
f('sys/kernel/slab/key_jar/align', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/partial', 0o664, b'0\n')
f('sys/kernel/slab/key_jar/validate', 0o664, b'')
f('sys/kernel/slab/key_jar/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/key_jar/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/key_jar/aliases', 0o664, b'0\n')
d('sys/kernel/slab/tcp_bind_bucket', 0o775)
f('sys/kernel/slab/tcp_bind_bucket/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/ctor', 0o664, b'')
f('sys/kernel/slab/tcp_bind_bucket/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/tcp_bind_bucket/total_objects', 0o664, b'64\n')
f('sys/kernel/slab/tcp_bind_bucket/slabs', 0o664, b'2\n')
f('sys/kernel/slab/tcp_bind_bucket/poison', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/alloc_calls', 0o664, b'      3 inet_bind_bucket_create+0x27/0x80 age=926731/5878918/8356133 pid=1206-1920 cpus=0\n')
f('sys/kernel/slab/tcp_bind_bucket/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/tcp_bind_bucket/shrink', 0o664, b'')
f('sys/kernel/slab/tcp_bind_bucket/trace', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/object_size', 0o664, b'40\n')
f('sys/kernel/slab/tcp_bind_bucket/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/objects_partial', 0o664, b'2\n')
f('sys/kernel/slab/tcp_bind_bucket/objects', 0o664, b'34\n')
f('sys/kernel/slab/tcp_bind_bucket/order', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/store_user', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/align', 0o664, b'0\n')
f('sys/kernel/slab/tcp_bind_bucket/partial', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/validate', 0o664, b'')
f('sys/kernel/slab/tcp_bind_bucket/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/tcp_bind_bucket/free_calls', 0o664, b'''      1 <not-available> age=4303046680 pid=0 cpus=0
      2 inet_bind_bucket_destroy+0x29/0x30 age=1107644/4731733/8355822 pid=0-1707 cpus=0
''')
f('sys/kernel/slab/tcp_bind_bucket/aliases', 0o664, b'0\n')
d('sys/kernel/slab/mm_struct', 0o775)
f('sys/kernel/slab/mm_struct/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/ctor', 0o664, b'')
f('sys/kernel/slab/mm_struct/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/mm_struct/slab_size', 0o664, b'1216\n')
f('sys/kernel/slab/mm_struct/total_objects', 0o664, b'91\n')
f('sys/kernel/slab/mm_struct/slabs', 0o664, b'7\n')
f('sys/kernel/slab/mm_struct/poison', 0o664, b'1\n')
f('sys/kernel/slab/mm_struct/alloc_calls', 0o664, b'''     52 mm_alloc+0x15/0x50 age=4100/7453183/8378125 pid=1-20296 cpus=0-1
     27 dup_mm+0x61/0x390 age=8335613/8355052/8376513 pid=71-2467 cpus=0-1
''')
f('sys/kernel/slab/mm_struct/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/mm_struct/shrink', 0o664, b'')
f('sys/kernel/slab/mm_struct/trace', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/object_size', 0o664, b'1144\n')
f('sys/kernel/slab/mm_struct/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/objects_partial', 0o664, b'23\n')
f('sys/kernel/slab/mm_struct/objects', 0o664, b'88\n')
f('sys/kernel/slab/mm_struct/order', 0o664, b'2\n')
f('sys/kernel/slab/mm_struct/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/mm_struct/store_user', 0o664, b'1\n')
f('sys/kernel/slab/mm_struct/align', 0o664, b'0\n')
f('sys/kernel/slab/mm_struct/partial', 0o664, b'2\n')
f('sys/kernel/slab/mm_struct/validate', 0o664, b'')
f('sys/kernel/slab/mm_struct/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/mm_struct/free_calls', 0o664, b'''      6 <not-available> age=4303047216 pid=0 cpus=0
     73 __mmdrop+0x47/0x60 age=7143/7714522/8376982 pid=1-20295 cpus=0-1
''')
f('sys/kernel/slab/mm_struct/aliases', 0o664, b'0\n')
d('sys/kernel/slab/dentry', 0o775)
f('sys/kernel/slab/dentry/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/dentry/ctor', 0o664, b'')
f('sys/kernel/slab/dentry/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/dentry/slab_size', 0o664, b'328\n')
f('sys/kernel/slab/dentry/total_objects', 0o664, b'49656\n')
f('sys/kernel/slab/dentry/slabs', 0o664, b'4138\n')
f('sys/kernel/slab/dentry/poison', 0o664, b'1\n')
f('sys/kernel/slab/dentry/alloc_calls', 0o664, b'  49656 d_alloc+0x22/0x1f0 age=0/4250434/8379825 pid=0-32757 cpus=0-1\n')
f('sys/kernel/slab/dentry/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/dentry/shrink', 0o664, b'')
f('sys/kernel/slab/dentry/trace', 0o664, b'0\n')
f('sys/kernel/slab/dentry/object_size', 0o664, b'256\n')
f('sys/kernel/slab/dentry/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/dentry/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/dentry/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/dentry/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/dentry/objects', 0o664, b'49656\n')
f('sys/kernel/slab/dentry/order', 0o664, b'0\n')
f('sys/kernel/slab/dentry/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/dentry/store_user', 0o664, b'1\n')
f('sys/kernel/slab/dentry/align', 0o664, b'8\n')
f('sys/kernel/slab/dentry/partial', 0o664, b'0\n')
f('sys/kernel/slab/dentry/validate', 0o664, b'')
f('sys/kernel/slab/dentry/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/dentry/free_calls', 0o664, b'''  30093 <not-available> age=4303047137 pid=0 cpus=0
   2265 d_free+0x6c/0x80 age=519/5532797/8376904 pid=1-32728 cpus=0-1
  17299 d_callback+0x31/0x40 age=1040/2433896/8379551 pid=0-30210 cpus=0-1
''')
f('sys/kernel/slab/dentry/aliases', 0o664, b'0\n')
d('sys/kernel/slab/UDPv6', 0o775)
f('sys/kernel/slab/UDPv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/ctor', 0o664, b'')
f('sys/kernel/slab/UDPv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/slab_size', 0o664, b'1472\n')
f('sys/kernel/slab/UDPv6/total_objects', 0o664, b'22\n')
f('sys/kernel/slab/UDPv6/slabs', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDPv6/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/UDPv6/shrink', 0o664, b'')
f('sys/kernel/slab/UDPv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/object_size', 0o664, b'1368\n')
f('sys/kernel/slab/UDPv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/objects', 0o664, b'22\n')
f('sys/kernel/slab/UDPv6/order', 0o664, b'3\n')
f('sys/kernel/slab/UDPv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/align', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/partial', 0o664, b'0\n')
f('sys/kernel/slab/UDPv6/validate', 0o664, b'')
f('sys/kernel/slab/UDPv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/UDPv6/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDPv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/mqueue_inode_cache', 0o775)
f('sys/kernel/slab/mqueue_inode_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/mqueue_inode_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/slab_size', 0o664, b'1536\n')
f('sys/kernel/slab/mqueue_inode_cache/total_objects', 0o664, b'21\n')
f('sys/kernel/slab/mqueue_inode_cache/slabs', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/alloc_calls', 0o664, b'      1 mqueue_alloc_inode+0x15/0x30 age=8380010 pid=1 cpus=1\n')
f('sys/kernel/slab/mqueue_inode_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/mqueue_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/mqueue_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/object_size', 0o664, b'1456\n')
f('sys/kernel/slab/mqueue_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/objects_partial', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/objects', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/mqueue_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/mqueue_inode_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/mqueue_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/mqueue_inode_cache/free_calls', 0o664, b'      1 <not-available> age=4303047603 pid=0 cpus=0\n')
f('sys/kernel/slab/mqueue_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-128', 0o775)
f('sys/kernel/slab/biovec-128/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-128/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/slab_size', 0o664, b'2176\n')
f('sys/kernel/slab/biovec-128/total_objects', 0o664, b'15\n')
f('sys/kernel/slab/biovec-128/slabs', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379474/8379474/8379474 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-128/objs_per_slab', 0o664, b'15\n')
f('sys/kernel/slab/biovec-128/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-128/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/object_size', 0o664, b'2048\n')
f('sys/kernel/slab/biovec-128/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/objects', 0o664, b'15\n')
f('sys/kernel/slab/biovec-128/order', 0o664, b'3\n')
f('sys/kernel/slab/biovec-128/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-128/validate', 0o664, b'')
f('sys/kernel/slab/biovec-128/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-128/free_calls', 0o664, b'      2 <not-available> age=4303046908 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-128/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-192', 0o775)
f('sys/kernel/slab/kmalloc-192/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-192/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/slab_size', 0o664, b'264\n')
f('sys/kernel/slab/kmalloc-192/total_objects', 0o664, b'255\n')
f('sys/kernel/slab/kmalloc-192/slabs', 0o664, b'17\n')
f('sys/kernel/slab/kmalloc-192/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-192/alloc_calls', 0o664, b'''     10 sched_create_group+0x91/0x220 age=8351861/8359345/8366872 pid=1333-2177 cpus=0-1
     10 sched_create_group+0xae/0x220 age=8351861/8359345/8366872 pid=1333-2177 cpus=0-1
     10 groups_alloc+0x41/0xe0 age=17070/6159814/8366872 pid=1333-20296 cpus=0-1
      9 param_sysfs_setup+0x87/0x140 age=8303102/8368759/8379996 pid=1-2754 cpus=0-1
      2 futex_lock_pi+0x776/0xb00 age=125614/1986240/3846867 pid=2411-13213 cpus=0-1
     18 mempool_create_node+0x2e/0xf0 age=62077/7917175/8379995 pid=1-13295 cpus=0-1
      4 __vmalloc_area_node+0xfb/0x140 age=8370390/8373664/8374956 pid=107-901 cpus=0
     30 __percpu_alloc_mask+0xbc/0x140 age=8370400/8377809/8380000 pid=1-900 cpus=0-1
      4 d_alloc+0x19d/0x1f0 age=61160/6282991/8356935 pid=1706-13349 cpus=0-1
      1 bm_register_write+0x63/0x630 age=8354110 pid=2054 cpus=1
      1 sysfs_open_file+0xd6/0x230 age=0 pid=13466 cpus=0
     15 ipc_rcu_alloc+0x4a/0x70 age=8274192/8330064/8348647 pid=2177-2621 cpus=0-1
     28 kset_create_and_add+0x35/0xa0 age=8355999/8378311/8380000 pid=1-1766 cpus=0-1
      1 acpi_ds_build_internal_package_obj+0xaf/0x1df age=8379979 pid=1 cpus=0
      3 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379976/8379976/8379977 pid=1 cpus=0
      1 acpi_power_add+0x4a/0x1d6 age=8379890 pid=1 cpus=0
      2 ata_host_alloc+0x4f/0xf0 age=8379280/8379538/8379796 pid=1 cpus=0-1
      8 cpuidle_add_state_sysfs+0xc3/0x140 age=125753/125759/125766 pid=13 cpus=0-1
      1 hid_parse_report+0x43/0x2c0 age=53560 pid=419 cpus=0
      2 hid_add_field+0x18a/0x3a0 age=53560/53560/53560 pid=419 cpus=0
      6 fib_create_info+0x61/0x970 age=8324536/8346547/8367903 pid=1197-1541 cpus=0-1
      1 unix_sysctl_register+0x2a/0x80 age=8379847 pid=1 cpus=0
      1 rtnetlink_init+0x38/0x113 age=8379999 pid=1 cpus=0
      1 inet_diag_init+0x15/0x6b age=8378998 pid=1 cpus=0
     41 usb_alloc_urb+0x19/0x50 [usbcore] age=22746/1043846/8374920 pid=143-419 cpus=0-1
      1 usb_get_configuration+0x5f2/0x1480 [usbcore] age=22753 pid=419 cpus=0
      2 usb_get_configuration+0x1388/0x1480 [usbcore] age=8372394/8372581/8372769 pid=419 cpus=0
      1 sr_probe+0x76/0x540 [sr_mod] age=8375854 pid=163 cpus=0
     18 snd_ctl_new+0x28/0x90 [snd] age=8374174/8374174/8374174 pid=107 cpus=1
      1 iwl3945_pci_probe+0xd3f/0x1020 [iwl3945] age=8373796 pid=126 cpus=1
      1 blkcipher_walk_next+0x29c/0x390 [crypto_blkcipher] age=8374562 pid=215 cpus=1
      3 dm_register_target+0x1f/0xf0 [dm_mod] age=8370400/8370400/8370401 pid=900 cpus=0
      1 ipv6_icmp_sysctl_init+0x22/0x40 [ipv6] age=8360202 pid=1549 cpus=0
      1 snd_seq_pool_new+0x1b/0x80 [snd_seq] age=8355636 pid=1805 cpus=0
''')
f('sys/kernel/slab/kmalloc-192/objs_per_slab', 0o664, b'15\n')
f('sys/kernel/slab/kmalloc-192/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-192/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/object_size', 0o664, b'192\n')
f('sys/kernel/slab/kmalloc-192/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/objects_partial', 0o664, b'14\n')
f('sys/kernel/slab/kmalloc-192/objects', 0o664, b'254\n')
f('sys/kernel/slab/kmalloc-192/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-192/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-192/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-192/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-192/partial', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-192/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-192/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-192/free_calls', 0o664, b'''    149 <not-available> age=4303047430 pid=0 cpus=0
      5 free_sched_group+0x3b/0x80 age=363179/6767246/8375130 pid=0-1614 cpus=0-1
      1 free_sched_group+0x50/0x80 age=8369988 pid=0 cpus=0
      1 do_exit+0x713/0x960 age=8333169 pid=2558 cpus=1
      7 groups_free+0x56/0x60 age=17071/4081875/8369994 pid=0-1690 cpus=0-1
      1 __vunmap+0xe9/0x120 age=8356237 pid=1748 cpus=1
      1 do_readv_writev+0xe4/0x1e0 age=8331191 pid=2262 cpus=0
      4 getxattr+0x8c/0x140 age=0/6266155/8357946 pid=1610-13466 cpus=0-1
     20 sysfs_release+0x7c/0xa0 age=53608/7118781/8376453 pid=226-2519 cpus=0-1
      1 ipc_immediate_free+0x9/0x10 age=8330259 pid=2568 cpus=0
      1 acpi_pci_irq_add_prt+0x30b/0x324 age=8379911 pid=1 cpus=0
      6 cpuidle_remove_state_sysfs+0x4f/0x70 age=125768/1366349/3847513 pid=13 cpus=0
      4 free_fib_info+0x34/0x60 age=8367904/8367909/8367920 pid=1185-1197 cpus=0-1
     36 urb_destroy+0x23/0x30 [usbcore] age=22749/2596918/8374921 pid=143-13294 cpus=0-1
      1 sg_clean+0x3e/0x80 [usbcore] age=61399 pid=13294 cpus=0
      1 0xffffffffa006521f age=8355637 pid=1805 cpus=0
''')
f('sys/kernel/slab/kmalloc-192/aliases', 0o664, b'0\n')
d('sys/kernel/slab/task_struct', 0o775)
f('sys/kernel/slab/task_struct/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/ctor', 0o664, b'')
f('sys/kernel/slab/task_struct/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/slab_size', 0o664, b'4400\n')
f('sys/kernel/slab/task_struct/total_objects', 0o664, b'224\n')
f('sys/kernel/slab/task_struct/slabs', 0o664, b'32\n')
f('sys/kernel/slab/task_struct/poison', 0o664, b'1\n')
f('sys/kernel/slab/task_struct/alloc_calls', 0o664, b'    203 copy_process+0xa4/0x1220 age=4172/8008848/8379941 pid=0-13212 cpus=0-1\n')
f('sys/kernel/slab/task_struct/objs_per_slab', 0o664, b'7\n')
f('sys/kernel/slab/task_struct/shrink', 0o664, b'')
f('sys/kernel/slab/task_struct/trace', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/object_size', 0o664, b'4320\n')
f('sys/kernel/slab/task_struct/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/task_struct/objects_partial', 0o664, b'57\n')
f('sys/kernel/slab/task_struct/objects', 0o664, b'211\n')
f('sys/kernel/slab/task_struct/order', 0o664, b'3\n')
f('sys/kernel/slab/task_struct/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/task_struct/store_user', 0o664, b'1\n')
f('sys/kernel/slab/task_struct/align', 0o664, b'16\n')
f('sys/kernel/slab/task_struct/partial', 0o664, b'10\n')
f('sys/kernel/slab/task_struct/validate', 0o664, b'')
f('sys/kernel/slab/task_struct/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/task_struct/free_calls', 0o664, b'''     93 <not-available> age=4303047286 pid=0 cpus=0
    110 free_task+0x30/0x40 age=7208/7706854/8377050 pid=0-13258 cpus=0-1
''')
f('sys/kernel/slab/task_struct/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sgpool-16', 0o775)
f('sys/kernel/slab/sgpool-16/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/ctor', 0o664, b'')
f('sys/kernel/slab/sgpool-16/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/slab_size', 0o664, b'640\n')
f('sys/kernel/slab/sgpool-16/total_objects', 0o664, b'24\n')
f('sys/kernel/slab/sgpool-16/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-16/poison', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379289/8379289/8379289 pid=1 cpus=0\n')
f('sys/kernel/slab/sgpool-16/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/sgpool-16/shrink', 0o664, b'')
f('sys/kernel/slab/sgpool-16/trace', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/object_size', 0o664, b'512\n')
f('sys/kernel/slab/sgpool-16/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/objects', 0o664, b'24\n')
f('sys/kernel/slab/sgpool-16/order', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/align', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-16/validate', 0o664, b'')
f('sys/kernel/slab/sgpool-16/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-16/free_calls', 0o664, b'      2 <not-available> age=4303046838 pid=0 cpus=0\n')
f('sys/kernel/slab/sgpool-16/aliases', 0o664, b'0\n')
d('sys/kernel/slab/request_sock_TCPv6', 0o775)
f('sys/kernel/slab/request_sock_TCPv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/ctor', 0o664, b'')
f('sys/kernel/slab/request_sock_TCPv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCPv6/slab_size', 0o664, b'256\n')
f('sys/kernel/slab/request_sock_TCPv6/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/slabs', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCPv6/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/request_sock_TCPv6/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/request_sock_TCPv6/shrink', 0o664, b'')
f('sys/kernel/slab/request_sock_TCPv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/object_size', 0o664, b'136\n')
f('sys/kernel/slab/request_sock_TCPv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/objects', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/order', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCPv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCPv6/align', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/partial', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCPv6/validate', 0o664, b'')
f('sys/kernel/slab/request_sock_TCPv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCPv6/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/request_sock_TCPv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/Acpi-Namespace', 0o775)
f('sys/kernel/slab/Acpi-Namespace/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/ctor', 0o664, b'')
f('sys/kernel/slab/Acpi-Namespace/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/slab_size', 0o664, b'104\n')
f('sys/kernel/slab/Acpi-Namespace/total_objects', 0o664, b'1755\n')
f('sys/kernel/slab/Acpi-Namespace/slabs', 0o664, b'45\n')
f('sys/kernel/slab/Acpi-Namespace/poison', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Namespace/alloc_calls', 0o664, b'   1709 acpi_ns_create_node+0x34/0x45 age=8375345/8379655/8379737 pid=0-330 cpus=0\n')
f('sys/kernel/slab/Acpi-Namespace/objs_per_slab', 0o664, b'39\n')
f('sys/kernel/slab/Acpi-Namespace/shrink', 0o664, b'')
f('sys/kernel/slab/Acpi-Namespace/trace', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/object_size', 0o664, b'32\n')
f('sys/kernel/slab/Acpi-Namespace/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/objects_partial', 0o664, b'32\n')
f('sys/kernel/slab/Acpi-Namespace/objects', 0o664, b'1709\n')
f('sys/kernel/slab/Acpi-Namespace/order', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Namespace/store_user', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Namespace/align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Namespace/partial', 0o664, b'2\n')
f('sys/kernel/slab/Acpi-Namespace/validate', 0o664, b'')
f('sys/kernel/slab/Acpi-Namespace/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Namespace/free_calls', 0o664, b'   1709 <not-available> age=4303047034 pid=0 cpus=0\n')
f('sys/kernel/slab/Acpi-Namespace/aliases', 0o664, b'0\n')
d('sys/kernel/slab/posix_timers_cache', 0o775)
f('sys/kernel/slab/posix_timers_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/ctor', 0o664, b'')
f('sys/kernel/slab/posix_timers_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/posix_timers_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/posix_timers_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/posix_timers_cache/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/posix_timers_cache/shrink', 0o664, b'')
f('sys/kernel/slab/posix_timers_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/object_size', 0o664, b'248\n')
f('sys/kernel/slab/posix_timers_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/posix_timers_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/posix_timers_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/posix_timers_cache/validate', 0o664, b'')
f('sys/kernel/slab/posix_timers_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/posix_timers_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/posix_timers_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/scsi_cmd_cache', 0o775)
f('sys/kernel/slab/scsi_cmd_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/ctor', 0o664, b'')
f('sys/kernel/slab/scsi_cmd_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/slab_size', 0o664, b'448\n')
f('sys/kernel/slab/scsi_cmd_cache/total_objects', 0o664, b'36\n')
f('sys/kernel/slab/scsi_cmd_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/scsi_cmd_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/alloc_calls', 0o664, b'      7 scsi_pool_alloc_command+0x25/0x80 age=67290/7192324/8380005 pid=1-419 cpus=0-1\n')
f('sys/kernel/slab/scsi_cmd_cache/objs_per_slab', 0o664, b'18\n')
f('sys/kernel/slab/scsi_cmd_cache/shrink', 0o664, b'')
f('sys/kernel/slab/scsi_cmd_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/object_size', 0o664, b'320\n')
f('sys/kernel/slab/scsi_cmd_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/objects', 0o664, b'36\n')
f('sys/kernel/slab/scsi_cmd_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_cmd_cache/validate', 0o664, b'')
f('sys/kernel/slab/scsi_cmd_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/scsi_cmd_cache/free_calls', 0o664, b'''      5 <not-available> age=4303047639 pid=0 cpus=0
      2 scsi_pool_free_command+0x4c/0x60 age=67789/4223648/8379508 pid=0 cpus=0
''')
f('sys/kernel/slab/scsi_cmd_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/eventpoll_pwq', 0o775)
f('sys/kernel/slab/eventpoll_pwq/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/ctor', 0o664, b'')
f('sys/kernel/slab/eventpoll_pwq/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/eventpoll_pwq/total_objects', 0o664, b'112\n')
f('sys/kernel/slab/eventpoll_pwq/slabs', 0o664, b'4\n')
f('sys/kernel/slab/eventpoll_pwq/poison', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_pwq/alloc_calls', 0o664, b'     65 ep_ptable_queue_proc+0x35/0xa0 age=2416434/8173328/8356246 pid=1690-20296 cpus=0-1\n')
f('sys/kernel/slab/eventpoll_pwq/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/eventpoll_pwq/shrink', 0o664, b'')
f('sys/kernel/slab/eventpoll_pwq/trace', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/object_size', 0o664, b'72\n')
f('sys/kernel/slab/eventpoll_pwq/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/objects_partial', 0o664, b'7\n')
f('sys/kernel/slab/eventpoll_pwq/objects', 0o664, b'91\n')
f('sys/kernel/slab/eventpoll_pwq/order', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_pwq/store_user', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_pwq/align', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_pwq/partial', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_pwq/validate', 0o664, b'')
f('sys/kernel/slab/eventpoll_pwq/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_pwq/free_calls', 0o664, b'''     61 <not-available> age=4303046794 pid=0 cpus=0
      4 ep_unregister_pollwait+0x61/0x80 age=2416452/6864185/8356241 pid=1690-2073 cpus=0-1
''')
f('sys/kernel/slab/eventpoll_pwq/aliases', 0o664, b'0\n')
d('sys/kernel/slab/journal_handle', 0o775)
f('sys/kernel/slab/journal_handle/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/journal_handle/ctor', 0o664, b'')
f('sys/kernel/slab/journal_handle/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/journal_handle/total_objects', 0o664, b'64\n')
f('sys/kernel/slab/journal_handle/slabs', 0o664, b'2\n')
f('sys/kernel/slab/journal_handle/poison', 0o664, b'1\n')
f('sys/kernel/slab/journal_handle/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/journal_handle/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/journal_handle/shrink', 0o664, b'')
f('sys/kernel/slab/journal_handle/trace', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/object_size', 0o664, b'56\n')
f('sys/kernel/slab/journal_handle/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/objects', 0o664, b'64\n')
f('sys/kernel/slab/journal_handle/order', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/journal_handle/store_user', 0o664, b'1\n')
f('sys/kernel/slab/journal_handle/align', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/partial', 0o664, b'0\n')
f('sys/kernel/slab/journal_handle/validate', 0o664, b'')
f('sys/kernel/slab/journal_handle/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/journal_handle/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/journal_handle/aliases', 0o664, b'0\n')
d('sys/kernel/slab/bsg_cmd', 0o775)
f('sys/kernel/slab/bsg_cmd/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/ctor', 0o664, b'')
f('sys/kernel/slab/bsg_cmd/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/bsg_cmd/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/slabs', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/poison', 0o664, b'1\n')
f('sys/kernel/slab/bsg_cmd/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/bsg_cmd/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/bsg_cmd/shrink', 0o664, b'')
f('sys/kernel/slab/bsg_cmd/trace', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/object_size', 0o664, b'312\n')
f('sys/kernel/slab/bsg_cmd/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/objects', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/order', 0o664, b'1\n')
f('sys/kernel/slab/bsg_cmd/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/bsg_cmd/store_user', 0o664, b'1\n')
f('sys/kernel/slab/bsg_cmd/align', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/partial', 0o664, b'0\n')
f('sys/kernel/slab/bsg_cmd/validate', 0o664, b'')
f('sys/kernel/slab/bsg_cmd/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/bsg_cmd/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/bsg_cmd/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kiocb', 0o775)
f('sys/kernel/slab/kiocb/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/ctor', 0o664, b'')
f('sys/kernel/slab/kiocb/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/kiocb/slab_size', 0o664, b'320\n')
f('sys/kernel/slab/kiocb/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/slabs', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/poison', 0o664, b'1\n')
f('sys/kernel/slab/kiocb/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kiocb/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/kiocb/shrink', 0o664, b'')
f('sys/kernel/slab/kiocb/trace', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/object_size', 0o664, b'240\n')
f('sys/kernel/slab/kiocb/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/objects', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/order', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kiocb/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kiocb/align', 0o664, b'8\n')
f('sys/kernel/slab/kiocb/partial', 0o664, b'0\n')
f('sys/kernel/slab/kiocb/validate', 0o664, b'')
f('sys/kernel/slab/kiocb/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kiocb/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kiocb/aliases', 0o664, b'0\n')
d('sys/kernel/slab/file_lock_cache', 0o775)
f('sys/kernel/slab/file_lock_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/file_lock_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/slab_size', 0o664, b'296\n')
f('sys/kernel/slab/file_lock_cache/total_objects', 0o664, b'26\n')
f('sys/kernel/slab/file_lock_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/file_lock_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/file_lock_cache/alloc_calls', 0o664, b'''      2 flock_lock_file_wait+0x2d6/0x2f0 age=8355055/8355767/8356479 pid=1690-1807 cpus=0
      1 __posix_lock_file+0x3f/0x560 age=8349601 pid=2262 cpus=0
''')
f('sys/kernel/slab/file_lock_cache/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/file_lock_cache/shrink', 0o664, b'')
f('sys/kernel/slab/file_lock_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/object_size', 0o664, b'224\n')
f('sys/kernel/slab/file_lock_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/objects', 0o664, b'26\n')
f('sys/kernel/slab/file_lock_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/file_lock_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/file_lock_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/file_lock_cache/validate', 0o664, b'')
f('sys/kernel/slab/file_lock_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/file_lock_cache/free_calls', 0o664, b'      3 locks_free_lock+0x3e/0x60 age=8351291/8355422/8359832 pid=1549-2189 cpus=0\n')
f('sys/kernel/slab/file_lock_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/inet_peer_cache', 0o775)
f('sys/kernel/slab/inet_peer_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/ctor', 0o664, b'')
f('sys/kernel/slab/inet_peer_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/inet_peer_cache/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/inet_peer_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/inet_peer_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/inet_peer_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/inet_peer_cache/shrink', 0o664, b'')
f('sys/kernel/slab/inet_peer_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/object_size', 0o664, b'64\n')
f('sys/kernel/slab/inet_peer_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/inet_peer_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/inet_peer_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/inet_peer_cache/validate', 0o664, b'')
f('sys/kernel/slab/inet_peer_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/inet_peer_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/inet_peer_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/arp_cache', 0o775)
f('sys/kernel/slab/arp_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/ctor', 0o664, b'')
f('sys/kernel/slab/arp_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/slab_size', 0o664, b'448\n')
f('sys/kernel/slab/arp_cache/total_objects', 0o664, b'36\n')
f('sys/kernel/slab/arp_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/arp_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/alloc_calls', 0o664, b'      1 neigh_create+0x8b/0x530 age=1133499 pid=952 cpus=0\n')
f('sys/kernel/slab/arp_cache/objs_per_slab', 0o664, b'18\n')
f('sys/kernel/slab/arp_cache/shrink', 0o664, b'')
f('sys/kernel/slab/arp_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/object_size', 0o664, b'348\n')
f('sys/kernel/slab/arp_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/objects', 0o664, b'36\n')
f('sys/kernel/slab/arp_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/arp_cache/validate', 0o664, b'')
f('sys/kernel/slab/arp_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/arp_cache/free_calls', 0o664, b'      1 neigh_destroy+0x10c/0x160 age=1135741 pid=0 cpus=0\n')
f('sys/kernel/slab/arp_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-4', 0o775)
f('sys/kernel/slab/biovec-4/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-4/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-4/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/biovec-4/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/biovec-4/slabs', 0o664, b'2\n')
f('sys/kernel/slab/biovec-4/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-4/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379500/8379500/8379500 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-4/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/biovec-4/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-4/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/object_size', 0o664, b'64\n')
f('sys/kernel/slab/biovec-4/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/objects', 0o664, b'42\n')
f('sys/kernel/slab/biovec-4/order', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-4/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-4/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-4/validate', 0o664, b'')
f('sys/kernel/slab/biovec-4/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-4/free_calls', 0o664, b'      2 <not-available> age=4303046934 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-4/aliases', 0o664, b'0\n')
d('sys/kernel/slab/scsi_sense_cache', 0o775)
f('sys/kernel/slab/scsi_sense_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/ctor', 0o664, b'')
f('sys/kernel/slab/scsi_sense_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/scsi_sense_cache/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/scsi_sense_cache/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/scsi_sense_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/scsi_sense_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/scsi_sense_cache/alloc_calls', 0o664, b'      7 scsi_pool_alloc_command+0x4a/0x80 age=67299/7192333/8380014 pid=1-419 cpus=0-1\n')
f('sys/kernel/slab/scsi_sense_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/scsi_sense_cache/shrink', 0o664, b'')
f('sys/kernel/slab/scsi_sense_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/object_size', 0o664, b'96\n')
f('sys/kernel/slab/scsi_sense_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/objects', 0o664, b'42\n')
f('sys/kernel/slab/scsi_sense_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/scsi_sense_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/scsi_sense_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_sense_cache/validate', 0o664, b'')
f('sys/kernel/slab/scsi_sense_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/scsi_sense_cache/free_calls', 0o664, b'''      5 <not-available> age=4303047648 pid=0 cpus=0
      2 scsi_pool_free_command+0x40/0x60 age=67798/4223657/8379517 pid=0 cpus=0
''')
f('sys/kernel/slab/scsi_sense_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/cfq_io_context', 0o775)
f('sys/kernel/slab/cfq_io_context/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/ctor', 0o664, b'')
f('sys/kernel/slab/cfq_io_context/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/slab_size', 0o664, b'240\n')
f('sys/kernel/slab/cfq_io_context/total_objects', 0o664, b'85\n')
f('sys/kernel/slab/cfq_io_context/slabs', 0o664, b'5\n')
f('sys/kernel/slab/cfq_io_context/poison', 0o664, b'1\n')
f('sys/kernel/slab/cfq_io_context/alloc_calls', 0o664, b'     69 cfq_set_request+0x13d/0x390 age=177/7837888/8379613 pid=1-13466 cpus=0-1\n')
f('sys/kernel/slab/cfq_io_context/objs_per_slab', 0o664, b'17\n')
f('sys/kernel/slab/cfq_io_context/shrink', 0o664, b'')
f('sys/kernel/slab/cfq_io_context/trace', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/object_size', 0o664, b'168\n')
f('sys/kernel/slab/cfq_io_context/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/objects_partial', 0o664, b'16\n')
f('sys/kernel/slab/cfq_io_context/objects', 0o664, b'84\n')
f('sys/kernel/slab/cfq_io_context/order', 0o664, b'0\n')
f('sys/kernel/slab/cfq_io_context/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/cfq_io_context/store_user', 0o664, b'1\n')
f('sys/kernel/slab/cfq_io_context/align', 0o664, b'8\n')
f('sys/kernel/slab/cfq_io_context/partial', 0o664, b'1\n')
f('sys/kernel/slab/cfq_io_context/validate', 0o664, b'')
f('sys/kernel/slab/cfq_io_context/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/cfq_io_context/free_calls', 0o664, b'''     12 <not-available> age=4303047630 pid=0 cpus=0
     57 cfq_cic_free_rcu+0x21/0xc0 age=61842/7747595/8376055 pid=0-2535 cpus=0-1
''')
f('sys/kernel/slab/cfq_io_context/aliases', 0o664, b'0\n')
d('sys/kernel/slab/idr_layer_cache', 0o775)
f('sys/kernel/slab/idr_layer_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/ctor', 0o664, b'idr_cache_ctor+0x0/0x10\n')
f('sys/kernel/slab/idr_layer_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/slab_size', 0o664, b'616\n')
f('sys/kernel/slab/idr_layer_cache/total_objects', 0o664, b'442\n')
f('sys/kernel/slab/idr_layer_cache/slabs', 0o664, b'34\n')
f('sys/kernel/slab/idr_layer_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/idr_layer_cache/alloc_calls', 0o664, b'    406 idr_pre_get+0x5d/0x80 age=60336/8278969/8380018 pid=0-13355 cpus=0-1\n')
f('sys/kernel/slab/idr_layer_cache/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/idr_layer_cache/shrink', 0o664, b'')
f('sys/kernel/slab/idr_layer_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/object_size', 0o664, b'544\n')
f('sys/kernel/slab/idr_layer_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/objects_partial', 0o664, b'82\n')
f('sys/kernel/slab/idr_layer_cache/objects', 0o664, b'420\n')
f('sys/kernel/slab/idr_layer_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/idr_layer_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/idr_layer_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/idr_layer_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/idr_layer_cache/partial', 0o664, b'8\n')
f('sys/kernel/slab/idr_layer_cache/validate', 0o664, b'')
f('sys/kernel/slab/idr_layer_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/idr_layer_cache/free_calls', 0o664, b'''    304 <not-available> age=4303047315 pid=0 cpus=0
      3 idr_destroy+0x2f/0x40 age=104486/3178486/4715486 pid=13211-13803 cpus=0-1
      4 idr_layer_rcu_free+0x17/0x20 age=8334501/8362397/8377168 pid=0-1155 cpus=0-1
     95 ida_get_new_above+0x141/0x210 age=812584/8292482/8380019 pid=0-2593 cpus=0-1
''')
f('sys/kernel/slab/idr_layer_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/vm_area_struct', 0o775)
f('sys/kernel/slab/vm_area_struct/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/ctor', 0o664, b'')
f('sys/kernel/slab/vm_area_struct/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/slab_size', 0o664, b'240\n')
f('sys/kernel/slab/vm_area_struct/total_objects', 0o664, b'12087\n')
f('sys/kernel/slab/vm_area_struct/slabs', 0o664, b'711\n')
f('sys/kernel/slab/vm_area_struct/poison', 0o664, b'1\n')
f('sys/kernel/slab/vm_area_struct/alloc_calls', 0o664, b'''   2751 dup_mm+0x194/0x390 age=8335622/8348807/8376522 pid=71-2467 cpus=0-1
     52 install_special_mapping+0x3d/0xe0 age=4109/7453125/8378118 pid=1-20296 cpus=0-1
   4006 split_vma+0x61/0x140 age=4107/7930710/8366050 pid=1206-20296 cpus=0-1
      3 copy_vma+0x125/0x1b0 age=8353805/8353815/8353824 pid=1738 cpus=0
     54 do_brk+0x2ec/0x360 age=4107/7486452/8378131 pid=1-20296 cpus=0-1
   5097 mmap_region+0x195/0x5c0 age=4105/7877577/8378131 pid=1-20296 cpus=0-1
     52 bprm_mm_init+0xa0/0x1b0 age=4109/7453192/8378134 pid=1-20296 cpus=0-1
''')
f('sys/kernel/slab/vm_area_struct/objs_per_slab', 0o664, b'17\n')
f('sys/kernel/slab/vm_area_struct/shrink', 0o664, b'')
f('sys/kernel/slab/vm_area_struct/trace', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/object_size', 0o664, b'168\n')
f('sys/kernel/slab/vm_area_struct/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/objects_partial', 0o664, b'47\n')
f('sys/kernel/slab/vm_area_struct/objects', 0o664, b'12049\n')
f('sys/kernel/slab/vm_area_struct/order', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/vm_area_struct/store_user', 0o664, b'1\n')
f('sys/kernel/slab/vm_area_struct/align', 0o664, b'0\n')
f('sys/kernel/slab/vm_area_struct/partial', 0o664, b'5\n')
f('sys/kernel/slab/vm_area_struct/validate', 0o664, b'')
f('sys/kernel/slab/vm_area_struct/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/vm_area_struct/free_calls', 0o664, b'''   6424 <not-available> age=4303047229 pid=0 cpus=0
   5590 remove_vma+0x53/0x70 age=4110/7683818/8376767 pid=71-20296 cpus=0-1
      1 vma_adjust+0x32c/0x570 age=8364245 pid=1371 cpus=0
''')
f('sys/kernel/slab/vm_area_struct/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-1', 0o775)
f('sys/kernel/slab/biovec-1/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-1/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/slab_size', 0o664, b'96\n')
f('sys/kernel/slab/biovec-1/total_objects', 0o664, b'126\n')
f('sys/kernel/slab/biovec-1/slabs', 0o664, b'3\n')
f('sys/kernel/slab/biovec-1/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379509/8379509/8379509 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-1/objs_per_slab', 0o664, b'42\n')
f('sys/kernel/slab/biovec-1/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-1/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/object_size', 0o664, b'16\n')
f('sys/kernel/slab/biovec-1/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/objects_partial', 0o664, b'2\n')
f('sys/kernel/slab/biovec-1/objects', 0o664, b'86\n')
f('sys/kernel/slab/biovec-1/order', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-1/partial', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/validate', 0o664, b'')
f('sys/kernel/slab/biovec-1/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-1/free_calls', 0o664, b'      2 <not-available> age=4303046943 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-1/aliases', 0o664, b'0\n')
d('sys/kernel/slab/secpath_cache', 0o775)
f('sys/kernel/slab/secpath_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/ctor', 0o664, b'')
f('sys/kernel/slab/secpath_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/secpath_cache/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/secpath_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/secpath_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/secpath_cache/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/secpath_cache/shrink', 0o664, b'')
f('sys/kernel/slab/secpath_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/object_size', 0o664, b'56\n')
f('sys/kernel/slab/secpath_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/secpath_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/secpath_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/secpath_cache/validate', 0o664, b'')
f('sys/kernel/slab/secpath_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/secpath_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/secpath_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/UNIX', 0o775)
f('sys/kernel/slab/UNIX/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/ctor', 0o664, b'')
f('sys/kernel/slab/UNIX/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/UNIX/slab_size', 0o664, b'1472\n')
f('sys/kernel/slab/UNIX/total_objects', 0o664, b'506\n')
f('sys/kernel/slab/UNIX/slabs', 0o664, b'23\n')
f('sys/kernel/slab/UNIX/poison', 0o664, b'1\n')
f('sys/kernel/slab/UNIX/alloc_calls', 0o664, b'    450 sk_prot_alloc+0x1e/0xb0 age=61033/8220474/8376385 pid=71-20296 cpus=0-1\n')
f('sys/kernel/slab/UNIX/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/UNIX/shrink', 0o664, b'')
f('sys/kernel/slab/UNIX/trace', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/object_size', 0o664, b'1376\n')
f('sys/kernel/slab/UNIX/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/objects_partial', 0o664, b'178\n')
f('sys/kernel/slab/UNIX/objects', 0o664, b'486\n')
f('sys/kernel/slab/UNIX/order', 0o664, b'3\n')
f('sys/kernel/slab/UNIX/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/UNIX/store_user', 0o664, b'1\n')
f('sys/kernel/slab/UNIX/align', 0o664, b'0\n')
f('sys/kernel/slab/UNIX/partial', 0o664, b'9\n')
f('sys/kernel/slab/UNIX/validate', 0o664, b'')
f('sys/kernel/slab/UNIX/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/UNIX/free_calls', 0o664, b'''    235 <not-available> age=4303046663 pid=0 cpus=0
    215 sk_free+0x80/0x100 age=61050/8083366/8375930 pid=76-20292 cpus=0-1
''')
f('sys/kernel/slab/UNIX/aliases', 0o664, b'0\n')
d('sys/kernel/slab/RAWv6', 0o775)
f('sys/kernel/slab/RAWv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/ctor', 0o664, b'')
f('sys/kernel/slab/RAWv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/slab_size', 0o664, b'1472\n')
f('sys/kernel/slab/RAWv6/total_objects', 0o664, b'22\n')
f('sys/kernel/slab/RAWv6/slabs', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/alloc_calls', 0o664, b'      5 sk_prot_alloc+0x1e/0xb0 age=8360545/8360552/8360555 pid=1549 cpus=0\n')
f('sys/kernel/slab/RAWv6/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/RAWv6/shrink', 0o664, b'')
f('sys/kernel/slab/RAWv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/object_size', 0o664, b'1376\n')
f('sys/kernel/slab/RAWv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/objects_partial', 0o664, b'5\n')
f('sys/kernel/slab/RAWv6/objects', 0o664, b'5\n')
f('sys/kernel/slab/RAWv6/order', 0o664, b'3\n')
f('sys/kernel/slab/RAWv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/align', 0o664, b'0\n')
f('sys/kernel/slab/RAWv6/partial', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/validate', 0o664, b'')
f('sys/kernel/slab/RAWv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/RAWv6/free_calls', 0o664, b'      5 <not-available> age=4303047773 pid=0 cpus=0\n')
f('sys/kernel/slab/RAWv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sgpool-64', 0o775)
f('sys/kernel/slab/sgpool-64/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/ctor', 0o664, b'')
f('sys/kernel/slab/sgpool-64/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-64/slab_size', 0o664, b'2176\n')
f('sys/kernel/slab/sgpool-64/total_objects', 0o664, b'30\n')
f('sys/kernel/slab/sgpool-64/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-64/poison', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-64/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379271/8379271/8379271 pid=1 cpus=0\n')
f('sys/kernel/slab/sgpool-64/objs_per_slab', 0o664, b'15\n')
f('sys/kernel/slab/sgpool-64/shrink', 0o664, b'')
f('sys/kernel/slab/sgpool-64/trace', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/object_size', 0o664, b'2048\n')
f('sys/kernel/slab/sgpool-64/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/objects', 0o664, b'30\n')
f('sys/kernel/slab/sgpool-64/order', 0o664, b'3\n')
f('sys/kernel/slab/sgpool-64/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-64/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-64/align', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-64/validate', 0o664, b'')
f('sys/kernel/slab/sgpool-64/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-64/free_calls', 0o664, b'      2 <not-available> age=4303046820 pid=0 cpus=0\n')
f('sys/kernel/slab/sgpool-64/aliases', 0o664, b'0\n')
d('sys/kernel/slab/bdev_cache', 0o775)
f('sys/kernel/slab/bdev_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/ctor', 0o664, b'init_once+0x0/0x100\n')
f('sys/kernel/slab/bdev_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/slab_size', 0o664, b'1536\n')
f('sys/kernel/slab/bdev_cache/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/bdev_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/bdev_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/alloc_calls', 0o664, b'     13 bdev_alloc_inode+0x15/0x30 age=61383/7093434/8379778 pid=0-13333 cpus=0\n')
f('sys/kernel/slab/bdev_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/bdev_cache/shrink', 0o664, b'')
f('sys/kernel/slab/bdev_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/bdev_cache/object_size', 0o664, b'1432\n')
f('sys/kernel/slab/bdev_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/bdev_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/bdev_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/bdev_cache/objects_partial', 0o664, b'11\n')
f('sys/kernel/slab/bdev_cache/objects', 0o664, b'32\n')
f('sys/kernel/slab/bdev_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/bdev_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/bdev_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/validate', 0o664, b'')
f('sys/kernel/slab/bdev_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/bdev_cache/free_calls', 0o664, b'''      9 <not-available> age=4303047074 pid=0 cpus=0
      4 bdev_destroy_inode+0x1f/0x30 age=61405/4218130/8374840 pid=488-13325 cpus=0-1
''')
f('sys/kernel/slab/bdev_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fasync_cache', 0o775)
f('sys/kernel/slab/fasync_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/ctor', 0o664, b'')
f('sys/kernel/slab/fasync_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/slab_size', 0o664, b'96\n')
f('sys/kernel/slab/fasync_cache/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/fasync_cache/slabs', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/alloc_calls', 0o664, b'      1 fasync_helper+0x115/0x130 age=8354059 pid=1738 cpus=0\n')
f('sys/kernel/slab/fasync_cache/objs_per_slab', 0o664, b'42\n')
f('sys/kernel/slab/fasync_cache/shrink', 0o664, b'')
f('sys/kernel/slab/fasync_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/object_size', 0o664, b'24\n')
f('sys/kernel/slab/fasync_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/objects_partial', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/objects', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/fasync_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/validate', 0o664, b'')
f('sys/kernel/slab/fasync_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fasync_cache/free_calls', 0o664, b'      1 <not-available> age=4303047449 pid=0 cpus=0\n')
f('sys/kernel/slab/fasync_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/UDPLITEv6', 0o775)
f('sys/kernel/slab/UDPLITEv6/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/ctor', 0o664, b'')
f('sys/kernel/slab/UDPLITEv6/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/UDPLITEv6/slab_size', 0o664, b'1472\n')
f('sys/kernel/slab/UDPLITEv6/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/poison', 0o664, b'1\n')
f('sys/kernel/slab/UDPLITEv6/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDPLITEv6/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/UDPLITEv6/shrink', 0o664, b'')
f('sys/kernel/slab/UDPLITEv6/trace', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/object_size', 0o664, b'1368\n')
f('sys/kernel/slab/UDPLITEv6/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/objects', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/order', 0o664, b'3\n')
f('sys/kernel/slab/UDPLITEv6/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/UDPLITEv6/store_user', 0o664, b'1\n')
f('sys/kernel/slab/UDPLITEv6/align', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/partial', 0o664, b'0\n')
f('sys/kernel/slab/UDPLITEv6/validate', 0o664, b'')
f('sys/kernel/slab/UDPLITEv6/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/UDPLITEv6/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/UDPLITEv6/aliases', 0o664, b'0\n')
d('sys/kernel/slab/Acpi-Operand', 0o775)
f('sys/kernel/slab/Acpi-Operand/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/ctor', 0o664, b'')
f('sys/kernel/slab/Acpi-Operand/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/Acpi-Operand/total_objects', 0o664, b'2688\n')
f('sys/kernel/slab/Acpi-Operand/slabs', 0o664, b'96\n')
f('sys/kernel/slab/Acpi-Operand/poison', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Operand/alloc_calls', 0o664, b'   2653 acpi_ut_allocate_object_desc_dbg+0x39/0x75 age=2625/8339457/8379700 pid=0-13466 cpus=0-1\n')
f('sys/kernel/slab/Acpi-Operand/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/Acpi-Operand/shrink', 0o664, b'')
f('sys/kernel/slab/Acpi-Operand/trace', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/object_size', 0o664, b'72\n')
f('sys/kernel/slab/Acpi-Operand/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/objects_partial', 0o664, b'171\n')
f('sys/kernel/slab/Acpi-Operand/objects', 0o664, b'2663\n')
f('sys/kernel/slab/Acpi-Operand/order', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Operand/store_user', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Operand/align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-Operand/partial', 0o664, b'7\n')
f('sys/kernel/slab/Acpi-Operand/validate', 0o664, b'')
f('sys/kernel/slab/Acpi-Operand/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-Operand/free_calls', 0o664, b'''   2267 <not-available> age=4303046997 pid=0 cpus=0
    386 acpi_os_release_object+0x9/0xd age=2626/8137233/8379689 pid=0-13466 cpus=0-1
''')
f('sys/kernel/slab/Acpi-Operand/aliases', 0o664, b'0\n')
d('sys/kernel/slab/xfrm_dst_cache', 0o775)
f('sys/kernel/slab/xfrm_dst_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/ctor', 0o664, b'')
f('sys/kernel/slab/xfrm_dst_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/slab_size', 0o664, b'448\n')
f('sys/kernel/slab/xfrm_dst_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/xfrm_dst_cache/objs_per_slab', 0o664, b'18\n')
f('sys/kernel/slab/xfrm_dst_cache/shrink', 0o664, b'')
f('sys/kernel/slab/xfrm_dst_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/object_size', 0o664, b'344\n')
f('sys/kernel/slab/xfrm_dst_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/xfrm_dst_cache/validate', 0o664, b'')
f('sys/kernel/slab/xfrm_dst_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/xfrm_dst_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/xfrm_dst_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/eventpoll_epi', 0o775)
f('sys/kernel/slab/eventpoll_epi/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/ctor', 0o664, b'')
f('sys/kernel/slab/eventpoll_epi/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_epi/slab_size', 0o664, b'256\n')
f('sys/kernel/slab/eventpoll_epi/total_objects', 0o664, b'96\n')
f('sys/kernel/slab/eventpoll_epi/slabs', 0o664, b'6\n')
f('sys/kernel/slab/eventpoll_epi/poison', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_epi/alloc_calls', 0o664, b'     65 sys_epoll_ctl+0x1a4/0x4d0 age=2416443/8173337/8356255 pid=1690-20296 cpus=0-1\n')
f('sys/kernel/slab/eventpoll_epi/objs_per_slab', 0o664, b'16\n')
f('sys/kernel/slab/eventpoll_epi/shrink', 0o664, b'')
f('sys/kernel/slab/eventpoll_epi/trace', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/object_size', 0o664, b'128\n')
f('sys/kernel/slab/eventpoll_epi/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/objects_partial', 0o664, b'15\n')
f('sys/kernel/slab/eventpoll_epi/objects', 0o664, b'79\n')
f('sys/kernel/slab/eventpoll_epi/order', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_epi/store_user', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_epi/align', 0o664, b'0\n')
f('sys/kernel/slab/eventpoll_epi/partial', 0o664, b'2\n')
f('sys/kernel/slab/eventpoll_epi/validate', 0o664, b'')
f('sys/kernel/slab/eventpoll_epi/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/eventpoll_epi/free_calls', 0o664, b'''     61 <not-available> age=4303046803 pid=0 cpus=0
      4 ep_remove+0xae/0xc0 age=2416461/6864194/8356250 pid=1690-2073 cpus=0-1
''')
f('sys/kernel/slab/eventpoll_epi/aliases', 0o664, b'0\n')
d('sys/kernel/slab/revoke_table', 0o775)
f('sys/kernel/slab/revoke_table/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/ctor', 0o664, b'')
f('sys/kernel/slab/revoke_table/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/slab_size', 0o664, b'88\n')
f('sys/kernel/slab/revoke_table/total_objects', 0o664, b'46\n')
f('sys/kernel/slab/revoke_table/slabs', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/poison', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/alloc_calls', 0o664, b'      4 journal_init_revoke_table+0x1b/0xb0 age=8369805/8374155/8378505 pid=1-962 cpus=0\n')
f('sys/kernel/slab/revoke_table/objs_per_slab', 0o664, b'46\n')
f('sys/kernel/slab/revoke_table/shrink', 0o664, b'')
f('sys/kernel/slab/revoke_table/trace', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/object_size', 0o664, b'16\n')
f('sys/kernel/slab/revoke_table/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/objects_partial', 0o664, b'4\n')
f('sys/kernel/slab/revoke_table/objects', 0o664, b'4\n')
f('sys/kernel/slab/revoke_table/order', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/store_user', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/align', 0o664, b'0\n')
f('sys/kernel/slab/revoke_table/partial', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/validate', 0o664, b'')
f('sys/kernel/slab/revoke_table/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/revoke_table/free_calls', 0o664, b'      4 <not-available> age=4303047546 pid=0 cpus=0\n')
f('sys/kernel/slab/revoke_table/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sgpool-32', 0o775)
f('sys/kernel/slab/sgpool-32/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/ctor', 0o664, b'')
f('sys/kernel/slab/sgpool-32/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-32/slab_size', 0o664, b'1152\n')
f('sys/kernel/slab/sgpool-32/total_objects', 0o664, b'28\n')
f('sys/kernel/slab/sgpool-32/slabs', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-32/poison', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-32/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379280/8379280/8379280 pid=1 cpus=0\n')
f('sys/kernel/slab/sgpool-32/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/sgpool-32/shrink', 0o664, b'')
f('sys/kernel/slab/sgpool-32/trace', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/object_size', 0o664, b'1024\n')
f('sys/kernel/slab/sgpool-32/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/objects', 0o664, b'28\n')
f('sys/kernel/slab/sgpool-32/order', 0o664, b'2\n')
f('sys/kernel/slab/sgpool-32/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-32/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-32/align', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/partial', 0o664, b'0\n')
f('sys/kernel/slab/sgpool-32/validate', 0o664, b'')
f('sys/kernel/slab/sgpool-32/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sgpool-32/free_calls', 0o664, b'      2 <not-available> age=4303046829 pid=0 cpus=0\n')
f('sys/kernel/slab/sgpool-32/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fib6_nodes', 0o775)
f('sys/kernel/slab/fib6_nodes/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/ctor', 0o664, b'')
f('sys/kernel/slab/fib6_nodes/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/fib6_nodes/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/fib6_nodes/total_objects', 0o664, b'64\n')
f('sys/kernel/slab/fib6_nodes/slabs', 0o664, b'2\n')
f('sys/kernel/slab/fib6_nodes/poison', 0o664, b'1\n')
f('sys/kernel/slab/fib6_nodes/alloc_calls', 0o664, b'''      3 fib6_add+0x112/0x6a0 [ipv6] age=8328809/8339785/8360575 pid=8-2080 cpus=0-1
      1 fib6_add+0x2a2/0x6a0 [ipv6] age=8329973 pid=8 cpus=1
      1 fib6_add+0x2b6/0x6a0 [ipv6] age=8329973 pid=8 cpus=1
''')
f('sys/kernel/slab/fib6_nodes/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/fib6_nodes/shrink', 0o664, b'')
f('sys/kernel/slab/fib6_nodes/trace', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/object_size', 0o664, b'48\n')
f('sys/kernel/slab/fib6_nodes/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/objects_partial', 0o664, b'5\n')
f('sys/kernel/slab/fib6_nodes/objects', 0o664, b'5\n')
f('sys/kernel/slab/fib6_nodes/order', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fib6_nodes/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fib6_nodes/align', 0o664, b'0\n')
f('sys/kernel/slab/fib6_nodes/partial', 0o664, b'2\n')
f('sys/kernel/slab/fib6_nodes/validate', 0o664, b'')
f('sys/kernel/slab/fib6_nodes/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fib6_nodes/free_calls', 0o664, b'      5 <not-available> age=4303047799 pid=0 cpus=0\n')
f('sys/kernel/slab/fib6_nodes/aliases', 0o664, b'0\n')
d('sys/kernel/slab/flow_cache', 0o775)
f('sys/kernel/slab/flow_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/ctor', 0o664, b'')
f('sys/kernel/slab/flow_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/slab_size', 0o664, b'168\n')
f('sys/kernel/slab/flow_cache/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/slabs', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/flow_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/flow_cache/objs_per_slab', 0o664, b'24\n')
f('sys/kernel/slab/flow_cache/shrink', 0o664, b'')
f('sys/kernel/slab/flow_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/object_size', 0o664, b'96\n')
f('sys/kernel/slab/flow_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/objects', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/flow_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/flow_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/flow_cache/validate', 0o664, b'')
f('sys/kernel/slab/flow_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/flow_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/flow_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fat_cache', 0o775)
f('sys/kernel/slab/fat_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/fat_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/slab_size', 0o664, b'104\n')
f('sys/kernel/slab/fat_cache/total_objects', 0o664, b'39\n')
f('sys/kernel/slab/fat_cache/slabs', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/alloc_calls', 0o664, b'     14 fat_cache_add+0x11f/0x1c0 age=61297/61310/61330 pid=13349 cpus=0\n')
f('sys/kernel/slab/fat_cache/objs_per_slab', 0o664, b'39\n')
f('sys/kernel/slab/fat_cache/shrink', 0o664, b'')
f('sys/kernel/slab/fat_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/object_size', 0o664, b'32\n')
f('sys/kernel/slab/fat_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/objects', 0o664, b'39\n')
f('sys/kernel/slab/fat_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/fat_cache/validate', 0o664, b'')
f('sys/kernel/slab/fat_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fat_cache/free_calls', 0o664, b'     14 <not-available> age=4303047585 pid=0 cpus=0\n')
f('sys/kernel/slab/fat_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sock_inode_cache', 0o775)
f('sys/kernel/slab/sock_inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/sock_inode_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/slab_size', 0o664, b'1280\n')
f('sys/kernel/slab/sock_inode_cache/total_objects', 0o664, b'504\n')
f('sys/kernel/slab/sock_inode_cache/slabs', 0o664, b'42\n')
f('sys/kernel/slab/sock_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/alloc_calls', 0o664, b'    481 sock_alloc_inode+0x1a/0x70 age=61330/8214562/8379530 pid=1-20296 cpus=0-1\n')
f('sys/kernel/slab/sock_inode_cache/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/sock_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/sock_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/sock_inode_cache/object_size', 0o664, b'1192\n')
f('sys/kernel/slab/sock_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sock_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sock_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sock_inode_cache/objects_partial', 0o664, b'83\n')
f('sys/kernel/slab/sock_inode_cache/objects', 0o664, b'491\n')
f('sys/kernel/slab/sock_inode_cache/order', 0o664, b'2\n')
f('sys/kernel/slab/sock_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/sock_inode_cache/partial', 0o664, b'8\n')
f('sys/kernel/slab/sock_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/sock_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sock_inode_cache/free_calls', 0o664, b'''    276 <not-available> age=4303046961 pid=0 cpus=0
    205 sock_destroy_inode+0x14/0x20 age=61348/8035806/8376228 pid=76-20292 cpus=0-1
''')
f('sys/kernel/slab/sock_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ip_fib_hash', 0o775)
f('sys/kernel/slab/ip_fib_hash/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/ctor', 0o664, b'')
f('sys/kernel/slab/ip_fib_hash/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/ip_fib_hash/total_objects', 0o664, b'56\n')
f('sys/kernel/slab/ip_fib_hash/slabs', 0o664, b'2\n')
f('sys/kernel/slab/ip_fib_hash/poison', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_hash/alloc_calls', 0o664, b'     11 fn_hash_insert+0x5a6/0x800 age=8323831/8347847/8367198 pid=1197-1541 cpus=0-1\n')
f('sys/kernel/slab/ip_fib_hash/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/ip_fib_hash/shrink', 0o664, b'')
f('sys/kernel/slab/ip_fib_hash/trace', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/object_size', 0o664, b'72\n')
f('sys/kernel/slab/ip_fib_hash/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/objects_partial', 0o664, b'11\n')
f('sys/kernel/slab/ip_fib_hash/objects', 0o664, b'11\n')
f('sys/kernel/slab/ip_fib_hash/order', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_hash/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_hash/align', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_hash/partial', 0o664, b'2\n')
f('sys/kernel/slab/ip_fib_hash/validate', 0o664, b'')
f('sys/kernel/slab/ip_fib_hash/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_hash/free_calls', 0o664, b'''      7 <not-available> age=4303046724 pid=0 cpus=0
      4 fn_hash_delete+0x22c/0x290 age=8367201/8367201/8367201 pid=1195 cpus=1
''')
f('sys/kernel/slab/ip_fib_hash/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc_dma-512', 0o775)
f('sys/kernel/slab/kmalloc_dma-512/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc_dma-512/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/slab_size', 0o664, b'584\n')
f('sys/kernel/slab/kmalloc_dma-512/total_objects', 0o664, b'14\n')
f('sys/kernel/slab/kmalloc_dma-512/slabs', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kmalloc_dma-512/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/kmalloc_dma-512/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc_dma-512/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/object_size', 0o664, b'512\n')
f('sys/kernel/slab/kmalloc_dma-512/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/cache_dma', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/objects', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc_dma-512/order', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc_dma-512/partial', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc_dma-512/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc_dma-512/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/kmalloc_dma-512/aliases', 0o664, b'0\n')
d('sys/kernel/slab/inode_cache', 0o775)
f('sys/kernel/slab/inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/inode_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/slab_size', 0o664, b'1144\n')
f('sys/kernel/slab/inode_cache/total_objects', 0o664, b'11676\n')
f('sys/kernel/slab/inode_cache/slabs', 0o664, b'834\n')
f('sys/kernel/slab/inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/inode_cache/alloc_calls', 0o664, b'  11655 alloc_inode+0x251/0x280 age=665/7686826/8379810 pid=0-13466 cpus=0-1\n')
f('sys/kernel/slab/inode_cache/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/object_size', 0o664, b'1072\n')
f('sys/kernel/slab/inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/objects', 0o664, b'11676\n')
f('sys/kernel/slab/inode_cache/order', 0o664, b'2\n')
f('sys/kernel/slab/inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/inode_cache/free_calls', 0o664, b'''  11113 <not-available> age=4303047110 pid=0 cpus=0
    542 destroy_inode+0x4f/0x60 age=1016/6891707/8377435 pid=7-15896 cpus=0-1
''')
f('sys/kernel/slab/inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/inotify_event_cache', 0o775)
f('sys/kernel/slab/inotify_event_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/ctor', 0o664, b'')
f('sys/kernel/slab/inotify_event_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/slab_size', 0o664, b'112\n')
f('sys/kernel/slab/inotify_event_cache/total_objects', 0o664, b'72\n')
f('sys/kernel/slab/inotify_event_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/inotify_event_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/inotify_event_cache/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/inotify_event_cache/objs_per_slab', 0o664, b'36\n')
f('sys/kernel/slab/inotify_event_cache/shrink', 0o664, b'')
f('sys/kernel/slab/inotify_event_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/object_size', 0o664, b'40\n')
f('sys/kernel/slab/inotify_event_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/objects', 0o664, b'72\n')
f('sys/kernel/slab/inotify_event_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/inotify_event_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/inotify_event_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/inotify_event_cache/validate', 0o664, b'')
f('sys/kernel/slab/inotify_event_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/inotify_event_cache/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/inotify_event_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/UDP', 0o775)
f('sys/kernel/slab/UDP/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/UDP/ctor', 0o664, b'')
f('sys/kernel/slab/UDP/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/UDP/slab_size', 0o664, b'1344\n')
f('sys/kernel/slab/UDP/total_objects', 0o664, b'24\n')
f('sys/kernel/slab/UDP/slabs', 0o664, b'2\n')
f('sys/kernel/slab/UDP/poison', 0o664, b'1\n')
f('sys/kernel/slab/UDP/alloc_calls', 0o664, b'      5 sk_prot_alloc+0x1e/0xb0 age=8327731/8349275/8355457 pid=1545-2586 cpus=0-1\n')
f('sys/kernel/slab/UDP/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/UDP/shrink', 0o664, b'')
f('sys/kernel/slab/UDP/trace', 0o664, b'0\n')
f('sys/kernel/slab/UDP/object_size', 0o664, b'1232\n')
f('sys/kernel/slab/UDP/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/UDP/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/UDP/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/UDP/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/UDP/objects', 0o664, b'24\n')
f('sys/kernel/slab/UDP/order', 0o664, b'2\n')
f('sys/kernel/slab/UDP/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/UDP/store_user', 0o664, b'1\n')
f('sys/kernel/slab/UDP/align', 0o664, b'0\n')
f('sys/kernel/slab/UDP/partial', 0o664, b'0\n')
f('sys/kernel/slab/UDP/validate', 0o664, b'')
f('sys/kernel/slab/UDP/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/UDP/free_calls', 0o664, b'''      1 <not-available> age=4303046759 pid=0 cpus=0
      4 sk_free+0x80/0x100 age=8327731/8350484/8364461 pid=1371-2586 cpus=0-1
''')
f('sys/kernel/slab/UDP/aliases', 0o664, b'0\n')
d('sys/kernel/slab/Acpi-ParseExt', 0o775)
f('sys/kernel/slab/Acpi-ParseExt/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/ctor', 0o664, b'')
f('sys/kernel/slab/Acpi-ParseExt/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/Acpi-ParseExt/total_objects', 0o664, b'56\n')
f('sys/kernel/slab/Acpi-ParseExt/slabs', 0o664, b'2\n')
f('sys/kernel/slab/Acpi-ParseExt/poison', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-ParseExt/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-ParseExt/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/Acpi-ParseExt/shrink', 0o664, b'')
f('sys/kernel/slab/Acpi-ParseExt/trace', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/object_size', 0o664, b'72\n')
f('sys/kernel/slab/Acpi-ParseExt/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/objects', 0o664, b'56\n')
f('sys/kernel/slab/Acpi-ParseExt/order', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-ParseExt/store_user', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-ParseExt/align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-ParseExt/validate', 0o664, b'')
f('sys/kernel/slab/Acpi-ParseExt/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-ParseExt/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-ParseExt/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-8', 0o775)
f('sys/kernel/slab/kmalloc-8/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-8/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/slab_size', 0o664, b'80\n')
f('sys/kernel/slab/kmalloc-8/total_objects', 0o664, b'3366\n')
f('sys/kernel/slab/kmalloc-8/slabs', 0o664, b'66\n')
f('sys/kernel/slab/kmalloc-8/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-8/alloc_calls', 0o664, b'''      1 cache_k8_northbridges+0x5f/0x130 age=8379864 pid=1 cpus=0
      1 pm_qos_add_requirement+0x51/0xe0 age=8376100 pid=122 cpus=1
      5 setup_modinfo_version+0x19/0x30 age=8355991/8370769/8376336 pid=169-1766 cpus=0-1
    379 load_module+0x1ac4/0x1b30 age=8303091/8368856/8376330 pid=105-2754 cpus=0-1
     57 strndup_user+0x6d/0xc0 age=8303095/8368234/8376336 pid=105-2754 cpus=0-1
      4 krealloc+0x1e/0x60 age=8355270/8362240/8377141 pid=71-1849 cpus=0-1
      5 __vmalloc_area_node+0xfb/0x140 age=8373659/8374450/8375164 pid=181-754 cpus=0-1
     17 alloc_vfsmnt+0x97/0x180 age=8354102/8377143/8380122 pid=0-2064 cpus=0-1
      1 proc_symlink+0x4d/0xb0 age=8374163 pid=107 cpus=1
   1954 sysfs_new_dirent+0x10c/0x120 age=22655/7531182/8380122 pid=0-13295 cpus=0-1
    690 kvasprintf+0x55/0x90 age=22668/7820060/8380122 pid=0-13295 cpus=0-1
      1 proc_bus_pci_open+0x1d/0x50 age=8355838 pid=1738 cpus=0
      5 pcie_port_device_register+0x25/0x4a0 age=8379822/8379823/8379824 pid=1 cpus=1
     19 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379961/8379970/8379972 pid=1 cpus=0
      9 acpi_ev_pci_config_region_setup+0x162/0x270 age=8379256/8379854/8379947 pid=1-12 cpus=0-1
      2 acpi_ut_copy_simple_object+0x8f/0x11c age=8375678/8375686/8375694 pid=396 cpus=0
     35 acpi_irq_stats_init+0x1bd/0x268 age=8379979/8379979/8379979 pid=1 cpus=0
      8 neigh_sysctl_register+0x1cd/0x330 age=8360192/8368781/8379845 pid=1-1549 cpus=0-1
      7 netlink_kernel_create+0xab/0x1a0 age=8378987/8379768/8379988 pid=1 cpus=0-1
      5 __devinet_sysctl_register+0xb8/0x120 age=8373658/8377862/8379845 pid=1-126 cpus=0-1
      1 fz_hash_alloc+0x4a/0x60 age=8324525 pid=1541 cpus=0
      1 init_vdso_vars+0x4c/0x24a age=8379835 pid=1 cpus=1
     30 netlink_proto_init+0xf1/0x16b age=8379988/8379988/8379988 pid=1 cpus=0
      5 hub_probe+0x20b/0x820 [usbcore] age=8374075/8374568/8374921 pid=143-147 cpus=0-1
      3 usb_cache_string+0x65/0xa0 [usbcore] age=22742/37518/67072 pid=419 cpus=0
     10 usb_get_configuration+0xbd/0x1480 [usbcore] age=22743/5876145/8374941 pid=143-419 cpus=0-1
     35 snd_info_create_entry+0x30/0xa0 [snd] age=8355370/8370655/8374933 pid=107-1813 cpus=0-1
      1 async_chainiv_givencrypt_first+0x56/0x80 [crypto_blkcipher] age=8374580 pid=215 cpus=0
      1 md_seq_open+0x2d/0x90 [md_mod] age=8366135 pid=1371 cpus=0
      1 bitmap_create+0x3aa/0xad0 [md_mod] age=8370125 pid=916 cpus=0
      5 __addrconf_sysctl_register+0xcd/0x140 [ipv6] age=8360192/8360193/8360195 pid=1549 cpus=0
      1 fib6_net_init+0x5d/0x130 [ipv6] age=8360195 pid=1549 cpus=0
''')
f('sys/kernel/slab/kmalloc-8/objs_per_slab', 0o664, b'51\n')
f('sys/kernel/slab/kmalloc-8/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-8/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/object_size', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-8/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/objects', 0o664, b'3366\n')
f('sys/kernel/slab/kmalloc-8/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-8/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-8/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-8/partial', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-8/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-8/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-8/free_calls', 0o664, b'''   2657 <not-available> age=4303047420 pid=0 cpus=0
      3 free_sect_attrs+0x2e/0x50 age=8375066/8375066/8375066 pid=383 cpus=0
      6 load_module+0x73f/0x1b30 age=8374844/8375309/8376003 pid=143-332 cpus=0-1
      1 free_module+0xb9/0xf0 age=8375066 pid=383 cpus=0
      1 krealloc+0x3a/0x60 age=8375814 pid=330 cpus=0
     41 __vunmap+0xe9/0x120 age=8355636/8372979/8376319 pid=105-1805 cpus=0-1
      1 vfs_rename+0x301/0x450 age=8367176 pid=72 cpus=1
    254 release_sysfs_dirent+0x8c/0xd0 age=125751/7762571/8379789 pid=1-3347 cpus=0-1
     61 kobject_release+0xe1/0x140 age=125751/6939731/8378264 pid=1-3347 cpus=0-1
      1 match_number+0x95/0xb0 age=8335823 pid=2476 cpus=0
      2 proc_bus_pci_release+0x18/0x30 age=8355841/8356005/8356169 pid=1738 cpus=0
     49 acpi_ds_create_operand+0x12c/0x209 age=125757/8029619/8379980 pid=1-1442 cpus=0-1
    128 acpi_ns_get_node+0x92/0xa1 age=8366449/8379363/8379989 pid=1-1358 cpus=0-1
      2 acpi_ut_delete_internal_obj+0x15f/0x16f age=8375669/8375730/8375792 pid=330-396 cpus=0
      2 module_add_driver+0x66/0xd0 age=8375989/8375992/8375996 pid=144 cpus=0
      2 sd_media_changed+0xca/0x210 age=7839601/7929600/8019600 pid=2796 cpus=0
      3 get_modalias+0xd4/0x120 age=8355846/8370834/8379988 pid=1-1772 cpus=0-1
      1 fib_hash_free+0x35/0x40 age=8367910 pid=1185 cpus=0
      1 usb_release_dev+0x39/0x70 [usbcore] age=7833240 pid=419 cpus=1
      2 usb_release_dev+0x45/0x70 [usbcore] age=7833240/7835719/7838199 pid=419 cpus=0-1
      4 sg_clean+0x3e/0x80 [usbcore] age=62037/2121033/8297977 pid=2760-13294 cpus=0
     59 usb_control_msg+0xef/0x110 [usbcore] age=22658/5520135/8374943 pid=143-419 cpus=0-1
      7 usb_set_configuration+0x338/0x5f0 [usbcore] age=53566/5999072/8374935 pid=143-419 cpus=0-1
      4 usb_destroy_configuration+0x78/0x140 [usbcore] age=7824401/7967435/8373902 pid=419 cpus=0-1
      7 sr_media_change+0xeb/0x2a0 [sr_mod] age=53597/5984195/8374963 pid=558-1456 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-8/aliases', 0o664, b'0\n')
d('sys/kernel/slab/revoke_record', 0o775)
f('sys/kernel/slab/revoke_record/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/ctor', 0o664, b'')
f('sys/kernel/slab/revoke_record/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/revoke_record/total_objects', 0o664, b'64\n')
f('sys/kernel/slab/revoke_record/slabs', 0o664, b'2\n')
f('sys/kernel/slab/revoke_record/poison', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/revoke_record/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/revoke_record/shrink', 0o664, b'')
f('sys/kernel/slab/revoke_record/trace', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/object_size', 0o664, b'32\n')
f('sys/kernel/slab/revoke_record/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/objects', 0o664, b'64\n')
f('sys/kernel/slab/revoke_record/order', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/store_user', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/align', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/partial', 0o664, b'0\n')
f('sys/kernel/slab/revoke_record/validate', 0o664, b'')
f('sys/kernel/slab/revoke_record/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/revoke_record/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/revoke_record/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ext3_inode_cache', 0o775)
f('sys/kernel/slab/ext3_inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/ext3_inode_cache/ctor', 0o664, b'init_once+0x0/0x70\n')
f('sys/kernel/slab/ext3_inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/slab_size', 0o664, b'1544\n')
f('sys/kernel/slab/ext3_inode_cache/total_objects', 0o664, b'27342\n')
f('sys/kernel/slab/ext3_inode_cache/slabs', 0o664, b'1302\n')
f('sys/kernel/slab/ext3_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/ext3_inode_cache/alloc_calls', 0o664, b'  27338 ext3_alloc_inode+0x15/0x60 age=0/2288559/8378475 pid=1-27944 cpus=0-1\n')
f('sys/kernel/slab/ext3_inode_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/ext3_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/ext3_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/object_size', 0o664, b'1472\n')
f('sys/kernel/slab/ext3_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/objects', 0o664, b'27342\n')
f('sys/kernel/slab/ext3_inode_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/ext3_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ext3_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ext3_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/ext3_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/ext3_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ext3_inode_cache/free_calls', 0o664, b'''  25981 <not-available> age=4303047521 pid=0 cpus=0
   1358 ext3_destroy_inode+0x2f/0x90 age=13089/2437961/8371953 pid=542-32759 cpus=0-1
''')
f('sys/kernel/slab/ext3_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ndisc_cache', 0o775)
f('sys/kernel/slab/ndisc_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/ctor', 0o664, b'')
f('sys/kernel/slab/ndisc_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/slab_size', 0o664, b'448\n')
f('sys/kernel/slab/ndisc_cache/total_objects', 0o664, b'18\n')
f('sys/kernel/slab/ndisc_cache/slabs', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/alloc_calls', 0o664, b'      1 neigh_create+0x8b/0x530 age=8360557 pid=1549 cpus=0\n')
f('sys/kernel/slab/ndisc_cache/objs_per_slab', 0o664, b'18\n')
f('sys/kernel/slab/ndisc_cache/shrink', 0o664, b'')
f('sys/kernel/slab/ndisc_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/object_size', 0o664, b'360\n')
f('sys/kernel/slab/ndisc_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/objects_partial', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/objects', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/ndisc_cache/partial', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/validate', 0o664, b'')
f('sys/kernel/slab/ndisc_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ndisc_cache/free_calls', 0o664, b'      1 <not-available> age=4303047782 pid=0 cpus=0\n')
f('sys/kernel/slab/ndisc_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ip_fib_alias', 0o775)
f('sys/kernel/slab/ip_fib_alias/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/ctor', 0o664, b'')
f('sys/kernel/slab/ip_fib_alias/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/slab_size', 0o664, b'104\n')
f('sys/kernel/slab/ip_fib_alias/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/slabs', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/poison', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_alias/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/ip_fib_alias/objs_per_slab', 0o664, b'39\n')
f('sys/kernel/slab/ip_fib_alias/shrink', 0o664, b'')
f('sys/kernel/slab/ip_fib_alias/trace', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/object_size', 0o664, b'32\n')
f('sys/kernel/slab/ip_fib_alias/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/objects', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/order', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_alias/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_alias/align', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/partial', 0o664, b'0\n')
f('sys/kernel/slab/ip_fib_alias/validate', 0o664, b'')
f('sys/kernel/slab/ip_fib_alias/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ip_fib_alias/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/ip_fib_alias/aliases', 0o664, b'0\n')
d('sys/kernel/slab/proc_inode_cache', 0o775)
f('sys/kernel/slab/proc_inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/proc_inode_cache/ctor', 0o664, b'init_once+0x0/0x10\n')
f('sys/kernel/slab/proc_inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/slab_size', 0o664, b'1192\n')
f('sys/kernel/slab/proc_inode_cache/total_objects', 0o664, b'1677\n')
f('sys/kernel/slab/proc_inode_cache/slabs', 0o664, b'129\n')
f('sys/kernel/slab/proc_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/proc_inode_cache/alloc_calls', 0o664, b'   1627 proc_alloc_inode+0x1b/0x90 age=61424/8163415/8379746 pid=0-19241 cpus=0-1\n')
f('sys/kernel/slab/proc_inode_cache/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/proc_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/proc_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/object_size', 0o664, b'1120\n')
f('sys/kernel/slab/proc_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/objects_partial', 0o664, b'78\n')
f('sys/kernel/slab/proc_inode_cache/objects', 0o664, b'1651\n')
f('sys/kernel/slab/proc_inode_cache/order', 0o664, b'2\n')
f('sys/kernel/slab/proc_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/proc_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/proc_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/proc_inode_cache/partial', 0o664, b'8\n')
f('sys/kernel/slab/proc_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/proc_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/proc_inode_cache/free_calls', 0o664, b'''   1124 <not-available> age=4303047043 pid=0 cpus=0
    503 proc_destroy_inode+0x14/0x20 age=61430/7970819/8372808 pid=1-12336 cpus=0-1
''')
f('sys/kernel/slab/proc_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-16', 0o775)
f('sys/kernel/slab/biovec-16/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-16/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/biovec-16/total_objects', 0o664, b'63\n')
f('sys/kernel/slab/biovec-16/slabs', 0o664, b'3\n')
f('sys/kernel/slab/biovec-16/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379491/8379491/8379491 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-16/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/biovec-16/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-16/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/object_size', 0o664, b'256\n')
f('sys/kernel/slab/biovec-16/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/objects_partial', 0o664, b'2\n')
f('sys/kernel/slab/biovec-16/objects', 0o664, b'44\n')
f('sys/kernel/slab/biovec-16/order', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-16/partial', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/validate', 0o664, b'')
f('sys/kernel/slab/biovec-16/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-16/free_calls', 0o664, b'      2 <not-available> age=4303046925 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-16/aliases', 0o664, b'0\n')
d('sys/kernel/slab/Acpi-State', 0o775)
f('sys/kernel/slab/Acpi-State/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/ctor', 0o664, b'')
f('sys/kernel/slab/Acpi-State/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/slab_size', 0o664, b'152\n')
f('sys/kernel/slab/Acpi-State/total_objects', 0o664, b'52\n')
f('sys/kernel/slab/Acpi-State/slabs', 0o664, b'2\n')
f('sys/kernel/slab/Acpi-State/poison', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-State/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-State/objs_per_slab', 0o664, b'26\n')
f('sys/kernel/slab/Acpi-State/shrink', 0o664, b'')
f('sys/kernel/slab/Acpi-State/trace', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/object_size', 0o664, b'80\n')
f('sys/kernel/slab/Acpi-State/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/objects', 0o664, b'52\n')
f('sys/kernel/slab/Acpi-State/order', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-State/store_user', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-State/align', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/partial', 0o664, b'0\n')
f('sys/kernel/slab/Acpi-State/validate', 0o664, b'')
f('sys/kernel/slab/Acpi-State/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/Acpi-State/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/Acpi-State/aliases', 0o664, b'0\n')
d('sys/kernel/slab/ip6_dst_cache', 0o775)
f('sys/kernel/slab/ip6_dst_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/ctor', 0o664, b'')
f('sys/kernel/slab/ip6_dst_cache/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/slab_size', 0o664, b'384\n')
f('sys/kernel/slab/ip6_dst_cache/total_objects', 0o664, b'42\n')
f('sys/kernel/slab/ip6_dst_cache/slabs', 0o664, b'2\n')
f('sys/kernel/slab/ip6_dst_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/alloc_calls', 0o664, b'      4 dst_alloc+0x2b/0x90 age=8329964/8337614/8360566 pid=8-1549 cpus=0-1\n')
f('sys/kernel/slab/ip6_dst_cache/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/ip6_dst_cache/shrink', 0o664, b'')
f('sys/kernel/slab/ip6_dst_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/object_size', 0o664, b'304\n')
f('sys/kernel/slab/ip6_dst_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/objects_partial', 0o664, b'4\n')
f('sys/kernel/slab/ip6_dst_cache/objects', 0o664, b'4\n')
f('sys/kernel/slab/ip6_dst_cache/order', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/ip6_dst_cache/partial', 0o664, b'2\n')
f('sys/kernel/slab/ip6_dst_cache/validate', 0o664, b'')
f('sys/kernel/slab/ip6_dst_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/ip6_dst_cache/free_calls', 0o664, b'      4 <not-available> age=4303047790 pid=0 cpus=0\n')
f('sys/kernel/slab/ip6_dst_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/task_xstate', 0o775)
f('sys/kernel/slab/task_xstate/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/ctor', 0o664, b'')
f('sys/kernel/slab/task_xstate/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/slab_size', 0o664, b'592\n')
f('sys/kernel/slab/task_xstate/total_objects', 0o664, b'169\n')
f('sys/kernel/slab/task_xstate/slabs', 0o664, b'13\n')
f('sys/kernel/slab/task_xstate/poison', 0o664, b'1\n')
f('sys/kernel/slab/task_xstate/alloc_calls', 0o664, b'''     99 arch_dup_task_struct+0x51/0xa0 age=125559/8275901/8376574 pid=71-13212 cpus=0-1
     51 init_fpu+0xef/0x110 age=4157/7436734/8378165 pid=1-20296 cpus=0-1
''')
f('sys/kernel/slab/task_xstate/objs_per_slab', 0o664, b'13\n')
f('sys/kernel/slab/task_xstate/shrink', 0o664, b'')
f('sys/kernel/slab/task_xstate/trace', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/object_size', 0o664, b'512\n')
f('sys/kernel/slab/task_xstate/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/task_xstate/objects_partial', 0o664, b'57\n')
f('sys/kernel/slab/task_xstate/objects', 0o664, b'161\n')
f('sys/kernel/slab/task_xstate/order', 0o664, b'1\n')
f('sys/kernel/slab/task_xstate/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/task_xstate/store_user', 0o664, b'1\n')
f('sys/kernel/slab/task_xstate/align', 0o664, b'16\n')
f('sys/kernel/slab/task_xstate/partial', 0o664, b'5\n')
f('sys/kernel/slab/task_xstate/validate', 0o664, b'')
f('sys/kernel/slab/task_xstate/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/task_xstate/free_calls', 0o664, b'''     53 <not-available> age=4303047277 pid=0 cpus=0
     97 free_thread_xstate+0x24/0x40 age=7199/7795238/8375118 pid=0-20296 cpus=0-1
''')
f('sys/kernel/slab/task_xstate/aliases', 0o664, b'0\n')
d('sys/kernel/slab/scsi_io_context', 0o775)
f('sys/kernel/slab/scsi_io_context/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/ctor', 0o664, b'')
f('sys/kernel/slab/scsi_io_context/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/slab_size', 0o664, b'184\n')
f('sys/kernel/slab/scsi_io_context/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/poison', 0o664, b'1\n')
f('sys/kernel/slab/scsi_io_context/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/scsi_io_context/objs_per_slab', 0o664, b'22\n')
f('sys/kernel/slab/scsi_io_context/shrink', 0o664, b'')
f('sys/kernel/slab/scsi_io_context/trace', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/object_size', 0o664, b'112\n')
f('sys/kernel/slab/scsi_io_context/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/objects', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/order', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/scsi_io_context/store_user', 0o664, b'1\n')
f('sys/kernel/slab/scsi_io_context/align', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/partial', 0o664, b'0\n')
f('sys/kernel/slab/scsi_io_context/validate', 0o664, b'')
f('sys/kernel/slab/scsi_io_context/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/scsi_io_context/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/scsi_io_context/aliases', 0o664, b'0\n')
d('sys/kernel/slab/anon_vma', 0o775)
f('sys/kernel/slab/anon_vma/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/ctor', 0o664, b'anon_vma_ctor+0x0/0x40\n')
f('sys/kernel/slab/anon_vma/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/slab_size', 0o664, b'144\n')
f('sys/kernel/slab/anon_vma/total_objects', 0o664, b'3304\n')
f('sys/kernel/slab/anon_vma/slabs', 0o664, b'118\n')
f('sys/kernel/slab/anon_vma/poison', 0o664, b'1\n')
f('sys/kernel/slab/anon_vma/alloc_calls', 0o664, b'   3234 anon_vma_prepare+0xe0/0x100 age=4177/7981073/8378201 pid=1-20296 cpus=0-1\n')
f('sys/kernel/slab/anon_vma/objs_per_slab', 0o664, b'28\n')
f('sys/kernel/slab/anon_vma/shrink', 0o664, b'')
f('sys/kernel/slab/anon_vma/trace', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/object_size', 0o664, b'72\n')
f('sys/kernel/slab/anon_vma/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/destroy_by_rcu', 0o664, b'1\n')
f('sys/kernel/slab/anon_vma/objects_partial', 0o664, b'185\n')
f('sys/kernel/slab/anon_vma/objects', 0o664, b'3265\n')
f('sys/kernel/slab/anon_vma/order', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/anon_vma/store_user', 0o664, b'1\n')
f('sys/kernel/slab/anon_vma/align', 0o664, b'0\n')
f('sys/kernel/slab/anon_vma/partial', 0o664, b'8\n')
f('sys/kernel/slab/anon_vma/validate', 0o664, b'')
f('sys/kernel/slab/anon_vma/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/anon_vma/free_calls', 0o664, b'''   1575 <not-available> age=4303047296 pid=0 cpus=0
   1659 anon_vma_unlink+0x5f/0x70 age=7225/7638870/8377152 pid=56-20292 cpus=0-1
''')
f('sys/kernel/slab/anon_vma/aliases', 0o664, b'0\n')
d('sys/kernel/slab/uhci_urb_priv', 0o775)
f('sys/kernel/slab/uhci_urb_priv/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/ctor', 0o664, b'')
f('sys/kernel/slab/uhci_urb_priv/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/slab_size', 0o664, b'128\n')
f('sys/kernel/slab/uhci_urb_priv/total_objects', 0o664, b'32\n')
f('sys/kernel/slab/uhci_urb_priv/slabs', 0o664, b'1\n')
f('sys/kernel/slab/uhci_urb_priv/poison', 0o664, b'1\n')
f('sys/kernel/slab/uhci_urb_priv/alloc_calls', 0o664, b'      1 uhci_urb_enqueue+0xb6/0xa30 [uhci_hcd] age=53794 pid=419 cpus=0\n')
f('sys/kernel/slab/uhci_urb_priv/objs_per_slab', 0o664, b'32\n')
f('sys/kernel/slab/uhci_urb_priv/shrink', 0o664, b'')
f('sys/kernel/slab/uhci_urb_priv/trace', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/object_size', 0o664, b'56\n')
f('sys/kernel/slab/uhci_urb_priv/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/objects', 0o664, b'32\n')
f('sys/kernel/slab/uhci_urb_priv/order', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/uhci_urb_priv/store_user', 0o664, b'1\n')
f('sys/kernel/slab/uhci_urb_priv/align', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/partial', 0o664, b'0\n')
f('sys/kernel/slab/uhci_urb_priv/validate', 0o664, b'')
f('sys/kernel/slab/uhci_urb_priv/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/uhci_urb_priv/free_calls', 0o664, b'      1 uhci_free_urb_priv+0x79/0xd0 [uhci_hcd] age=53806 pid=0 cpus=0\n')
f('sys/kernel/slab/uhci_urb_priv/aliases', 0o664, b'0\n')
d('sys/kernel/slab/shmem_inode_cache', 0o775)
f('sys/kernel/slab/shmem_inode_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/ctor', 0o664, b'init_once+0x0/0x40\n')
f('sys/kernel/slab/shmem_inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/slab_size', 0o664, b'1400\n')
f('sys/kernel/slab/shmem_inode_cache/total_objects', 0o664, b'1173\n')
f('sys/kernel/slab/shmem_inode_cache/slabs', 0o664, b'51\n')
f('sys/kernel/slab/shmem_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/shmem_inode_cache/alloc_calls', 0o664, b'   1144 shmem_alloc_inode+0x15/0x30 age=21584/7506265/8379043 pid=1-13453 cpus=0-1\n')
f('sys/kernel/slab/shmem_inode_cache/objs_per_slab', 0o664, b'23\n')
f('sys/kernel/slab/shmem_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/shmem_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/object_size', 0o664, b'1328\n')
f('sys/kernel/slab/shmem_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/objects_partial', 0o664, b'42\n')
f('sys/kernel/slab/shmem_inode_cache/objects', 0o664, b'1169\n')
f('sys/kernel/slab/shmem_inode_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/shmem_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/shmem_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/shmem_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/shmem_inode_cache/partial', 0o664, b'2\n')
f('sys/kernel/slab/shmem_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/shmem_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/shmem_inode_cache/free_calls', 0o664, b'''    741 <not-available> age=4303046628 pid=0 cpus=0
    403 shmem_destroy_inode+0x24/0x30 age=21598/7406739/8375758 pid=72-13332 cpus=0-1
''')
f('sys/kernel/slab/shmem_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/fat_inode_cache', 0o775)
f('sys/kernel/slab/fat_inode_cache/reclaim_account', 0o664, b'1\n')
f('sys/kernel/slab/fat_inode_cache/ctor', 0o664, b'init_once+0x0/0x60\n')
f('sys/kernel/slab/fat_inode_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/slab_size', 0o664, b'1272\n')
f('sys/kernel/slab/fat_inode_cache/total_objects', 0o664, b'100\n')
f('sys/kernel/slab/fat_inode_cache/slabs', 0o664, b'4\n')
f('sys/kernel/slab/fat_inode_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/fat_inode_cache/alloc_calls', 0o664, b'     76 fat_alloc_inode+0x15/0x30 age=61228/61276/61657 pid=13348-13354 cpus=0-1\n')
f('sys/kernel/slab/fat_inode_cache/objs_per_slab', 0o664, b'25\n')
f('sys/kernel/slab/fat_inode_cache/shrink', 0o664, b'')
f('sys/kernel/slab/fat_inode_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/object_size', 0o664, b'1200\n')
f('sys/kernel/slab/fat_inode_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/objects', 0o664, b'100\n')
f('sys/kernel/slab/fat_inode_cache/order', 0o664, b'3\n')
f('sys/kernel/slab/fat_inode_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/fat_inode_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/fat_inode_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/fat_inode_cache/validate', 0o664, b'')
f('sys/kernel/slab/fat_inode_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/fat_inode_cache/free_calls', 0o664, b'     76 <not-available> age=4303047594 pid=0 cpus=0\n')
f('sys/kernel/slab/fat_inode_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-256', 0o775)
f('sys/kernel/slab/kmalloc-256/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-256/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/slab_size', 0o664, b'328\n')
f('sys/kernel/slab/kmalloc-256/total_objects', 0o664, b'516\n')
f('sys/kernel/slab/kmalloc-256/slabs', 0o664, b'43\n')
f('sys/kernel/slab/kmalloc-256/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-256/alloc_calls', 0o664, b'''      2 mempool_kmalloc+0x11/0x20 age=8379928/8379928/8379928 pid=1 cpus=0
      2 __vmalloc_area_node+0xfb/0x140 age=8374585/8375388/8376192 pid=122-126 cpus=0-1
     32 dma_pool_create+0x90/0x1e0 age=8374074/8374505/8374905 pid=143-147 cpus=0-1
     28 __percpu_alloc_mask+0xbc/0x140 age=8360136/8365750/8379780 pid=1-1549 cpus=0
      2 seq_open+0x84/0xa0 age=8355780/8360929/8366079 pid=1371-1738 cpus=0
     19 inotify_init+0x1b/0x90 age=8333719/8353284/8379778 pid=1-2544 cpus=0-1
      1 mounts_open_common+0x11e/0x210 age=8365844 pid=1371 cpus=0
    347 __proc_create+0xa5/0x150 age=8355314/8375854/8380066 pid=0-1813 cpus=0-1
      6 start_this_handle+0x3f5/0x410 age=4241/38502/142468 pid=21-13466 cpus=0-1
      3 elevator_alloc+0x1b/0xc0 age=62010/5606790/8379345 pid=1-13295 cpus=0-1
      2 acpi_ds_build_internal_buffer_obj+0xe1/0x121 age=8379913/8379914/8379915 pid=1 cpus=0
      1 find_dock+0x414/0x426 age=8379931 pid=1 cpus=0
      6 con_insert_unipair+0x96/0x110 age=8355034/8367399/8379765 pid=1-1898 cpus=1
      2 neigh_resolve_output+0x226/0x2e0 age=1134112/4736669/8339226 pid=0-2401 cpus=0-1
      1 genl_register_family+0x1b6/0x1e0 age=8375014 pid=126 cpus=0
      5 ip_mc_inc_group+0x6b/0x280 age=8325482/8352347/8367852 pid=1185-1849 cpus=0-1
      2 cache_add_dev+0x17b/0x546 age=8379779/8379779/8379780 pid=1 cpus=0-1
     10 audit_register_class+0x1e/0xa1 age=8379779/8379779/8379779 pid=1 cpus=1
      1 proc_net_ns_init+0x1b/0x81 age=8380066 pid=0 cpus=0
      3 snd_malloc_sgbuf_pages+0xfa/0x1f0 [snd_page_alloc] age=8374129/8374129/8374129 pid=107 cpus=1
      1 usb_get_configuration+0x18f/0x1480 [usbcore] age=8372709 pid=419 cpus=0
      1 acpi_ac_add+0x3a/0x1b1 [ac] age=8375784 pid=392 cpus=0
      1 acpi_processor_register_performance+0x2a0/0x3a6 [processor] age=8366391 pid=1358 cpus=0
      3 snd_ctl_open+0x8e/0x180 [snd] age=8333026/8337179/8339273 pid=2400-2425 cpus=0
      8 ieee80211_rx_bss_add+0x4e/0x140 [mac80211] age=8354793/8355457/8355979 pid=0-1951 cpus=0
      2 ipv6_add_addr+0x199/0x3c0 [ipv6] age=8329536/8344837/8360138 pid=8-1549 cpus=0-1
      5 ipv6_dev_mc_inc+0x141/0x3f0 [ipv6] age=8329536/8354016/8360138 pid=8-1549 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-256/objs_per_slab', 0o664, b'12\n')
f('sys/kernel/slab/kmalloc-256/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-256/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/object_size', 0o664, b'256\n')
f('sys/kernel/slab/kmalloc-256/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/objects_partial', 0o664, b'60\n')
f('sys/kernel/slab/kmalloc-256/objects', 0o664, b'504\n')
f('sys/kernel/slab/kmalloc-256/order', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-256/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-256/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-256/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-256/partial', 0o664, b'6\n')
f('sys/kernel/slab/kmalloc-256/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-256/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-256/free_calls', 0o664, b'''    389 <not-available> age=4303047363 pid=0 cpus=0
      1 __vunmap+0xe9/0x120 age=8374878 pid=107 cpus=1
     38 do_execve+0x2b1/0x2d0 age=7303/7294190/8377088 pid=71-13465 cpus=0-1
     51 seq_release+0x20/0x30 age=23541/7877166/8376202 pid=105-2535 cpus=0-1
      3 free_proc_entry+0x73/0x90 age=8356279/8369643/8378992 pid=1-1541 cpus=0
      3 __journal_drop_transaction+0xe6/0x170 age=8360317/8363466/8366128 pid=32 cpus=0
      1 acpi_pci_irq_add_prt+0xc2/0x324 age=8379845 pid=1 cpus=0
      2 acpi_pci_bind+0x26b/0x28d age=8379844/8379845/8379846 pid=1 cpus=0
      1 wireless_send_event+0x172/0x330 age=8329538 pid=742 cpus=1
      3 usb_string+0x17c/0x1e0 [usbcore] age=8374367/8374614/8374886 pid=143 cpus=0
      3 snd_ctl_release+0xf9/0x130 [snd] age=8333027/8337180/8339274 pid=2400-2425 cpus=0
      1 ieee80211_set_associated+0x421/0x520 [mac80211] age=8329538 pid=742 cpus=1
''')
f('sys/kernel/slab/kmalloc-256/aliases', 0o664, b'0\n')
d('sys/kernel/slab/biovec-64', 0o775)
f('sys/kernel/slab/biovec-64/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/ctor', 0o664, b'')
f('sys/kernel/slab/biovec-64/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/biovec-64/slab_size', 0o664, b'1152\n')
f('sys/kernel/slab/biovec-64/total_objects', 0o664, b'28\n')
f('sys/kernel/slab/biovec-64/slabs', 0o664, b'2\n')
f('sys/kernel/slab/biovec-64/poison', 0o664, b'1\n')
f('sys/kernel/slab/biovec-64/alloc_calls', 0o664, b'      2 mempool_alloc_slab+0x11/0x20 age=8379482/8379482/8379482 pid=1 cpus=0\n')
f('sys/kernel/slab/biovec-64/objs_per_slab', 0o664, b'14\n')
f('sys/kernel/slab/biovec-64/shrink', 0o664, b'')
f('sys/kernel/slab/biovec-64/trace', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/object_size', 0o664, b'1024\n')
f('sys/kernel/slab/biovec-64/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/objects', 0o664, b'28\n')
f('sys/kernel/slab/biovec-64/order', 0o664, b'2\n')
f('sys/kernel/slab/biovec-64/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/biovec-64/store_user', 0o664, b'1\n')
f('sys/kernel/slab/biovec-64/align', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/partial', 0o664, b'0\n')
f('sys/kernel/slab/biovec-64/validate', 0o664, b'')
f('sys/kernel/slab/biovec-64/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/biovec-64/free_calls', 0o664, b'      2 <not-available> age=4303046917 pid=0 cpus=0\n')
f('sys/kernel/slab/biovec-64/aliases', 0o664, b'0\n')
d('sys/kernel/slab/sysfs_dir_cache', 0o775)
f('sys/kernel/slab/sysfs_dir_cache/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/ctor', 0o664, b'')
f('sys/kernel/slab/sysfs_dir_cache/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/slab_size', 0o664, b'152\n')
f('sys/kernel/slab/sysfs_dir_cache/total_objects', 0o664, b'11518\n')
f('sys/kernel/slab/sysfs_dir_cache/slabs', 0o664, b'443\n')
f('sys/kernel/slab/sysfs_dir_cache/poison', 0o664, b'1\n')
f('sys/kernel/slab/sysfs_dir_cache/alloc_calls', 0o664, b'  11488 sysfs_new_dirent+0x38/0x120 age=22320/7764235/8379787 pid=0-13295 cpus=0-1\n')
f('sys/kernel/slab/sysfs_dir_cache/objs_per_slab', 0o664, b'26\n')
f('sys/kernel/slab/sysfs_dir_cache/shrink', 0o664, b'')
f('sys/kernel/slab/sysfs_dir_cache/trace', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/object_size', 0o664, b'80\n')
f('sys/kernel/slab/sysfs_dir_cache/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/objects', 0o664, b'11518\n')
f('sys/kernel/slab/sysfs_dir_cache/order', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/sysfs_dir_cache/store_user', 0o664, b'1\n')
f('sys/kernel/slab/sysfs_dir_cache/align', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/partial', 0o664, b'0\n')
f('sys/kernel/slab/sysfs_dir_cache/validate', 0o664, b'')
f('sys/kernel/slab/sysfs_dir_cache/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/sysfs_dir_cache/free_calls', 0o664, b'''  11006 <not-available> age=4303047087 pid=0 cpus=0
    482 release_sysfs_dirent+0x5c/0xd0 age=61294/6319346/8379456 pid=1-2141 cpus=0-1
''')
f('sys/kernel/slab/sysfs_dir_cache/aliases', 0o664, b'0\n')
d('sys/kernel/slab/request_sock_TCP', 0o775)
f('sys/kernel/slab/request_sock_TCP/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/ctor', 0o664, b'')
f('sys/kernel/slab/request_sock_TCP/hwcache_align', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCP/slab_size', 0o664, b'192\n')
f('sys/kernel/slab/request_sock_TCP/total_objects', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/slabs', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/poison', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCP/alloc_calls', 0o664, b'No data\n')
f('sys/kernel/slab/request_sock_TCP/objs_per_slab', 0o664, b'21\n')
f('sys/kernel/slab/request_sock_TCP/shrink', 0o664, b'')
f('sys/kernel/slab/request_sock_TCP/trace', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/object_size', 0o664, b'88\n')
f('sys/kernel/slab/request_sock_TCP/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/objects_partial', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/objects', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/order', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCP/store_user', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCP/align', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/partial', 0o664, b'0\n')
f('sys/kernel/slab/request_sock_TCP/validate', 0o664, b'')
f('sys/kernel/slab/request_sock_TCP/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/request_sock_TCP/free_calls', 0o664, b'No data\n')
f('sys/kernel/slab/request_sock_TCP/aliases', 0o664, b'0\n')
d('sys/kernel/slab/kmalloc-4096', 0o775)
f('sys/kernel/slab/kmalloc-4096/reclaim_account', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/ctor', 0o664, b'')
f('sys/kernel/slab/kmalloc-4096/hwcache_align', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/slab_size', 0o664, b'4168\n')
f('sys/kernel/slab/kmalloc-4096/total_objects', 0o664, b'385\n')
f('sys/kernel/slab/kmalloc-4096/slabs', 0o664, b'55\n')
f('sys/kernel/slab/kmalloc-4096/poison', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-4096/alloc_calls', 0o664, b'''      1 load_module+0x18d5/0x1b30 age=8375716 pid=330 cpus=0
      1 __vmalloc_area_node+0xfb/0x140 age=8371983 pid=867 cpus=0
     24 __percpu_alloc_mask+0xbc/0x140 age=8360098/8363374/8379742 pid=1-1549 cpus=0
      1 seq_read+0x2f0/0x360 age=8365263 pid=1371 cpus=0
      2 ext3_fill_super+0x68/0x1a20 age=8369599/8373987/8378375 pid=1-962 cpus=0
      4 journal_init_revoke_table+0x4d/0xb0 age=8369583/8373933/8378283 pid=1-962 cpus=0
      2 journal_init_inode+0x8d/0x130 age=8369583/8373933/8378283 pid=1-962 cpus=0
      1 fat_fill_super+0x40/0xdd0 age=61388 pid=13348 cpus=1
      7 scsi_host_alloc+0x35/0x360 age=66976/7192011/8379690 pid=1-419 cpus=0-1
      3 scsi_alloc_sdev+0x6a/0x270 age=61972/5606752/8379307 pid=1-13295 cpus=0-1
      8 input_allocate_device+0x1a/0xb0 age=53455/7335718/8378931 pid=17-419 cpus=0-1
      1 reqsk_queue_alloc+0x112/0x120 age=8354535 pid=1920 cpus=0
      3 alloc_netdev_mq+0x4f/0x180 age=8373564/8375671/8379766 pid=1-126 cpus=0-1
      1 pidmap_init+0x15/0x4e age=8380028 pid=0 cpus=0
      1 netlink_proto_init+0x47/0x16b age=8379894 pid=1 cpus=0
      2 acpi_processor_add+0x25/0x6d [processor] age=8375653/8375684/8375715 pid=330 cpus=0
      1 yenta_probe+0x38/0x6cd [yenta_socket] age=8373766 pid=352 cpus=0
    281 iwl3945_rx_allocate+0xd5/0x160 [iwl3945] age=2011/7577/13684 pid=733 cpus=0
      1 skcipher_geniv_alloc+0x304/0x3f0 [crypto_blkcipher] age=8374372 pid=215 cpus=1
      2 bitmap_get_counter+0x153/0x220 [md_mod] age=8370027/8370029/8370031 pid=916 cpus=0
      1 bitmap_create+0x2bc/0xad0 [md_mod] age=8370031 pid=916 cpus=0
      6 __addrconf_sysctl_register+0x7b/0x140 [ipv6] age=8360098/8360099/8360101 pid=1549 cpus=0
      1 acm_probe+0x34c/0xa50 [cdc_acm] age=22641 pid=419 cpus=0
''')
f('sys/kernel/slab/kmalloc-4096/objs_per_slab', 0o664, b'7\n')
f('sys/kernel/slab/kmalloc-4096/shrink', 0o664, b'')
f('sys/kernel/slab/kmalloc-4096/trace', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/object_size', 0o664, b'4096\n')
f('sys/kernel/slab/kmalloc-4096/cpu_slabs', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/cache_dma', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/destroy_by_rcu', 0o664, b'0\n')
f('sys/kernel/slab/kmalloc-4096/objects_partial', 0o664, b'17\n')
f('sys/kernel/slab/kmalloc-4096/objects', 0o664, b'367\n')
f('sys/kernel/slab/kmalloc-4096/order', 0o664, b'3\n')
f('sys/kernel/slab/kmalloc-4096/sanity_checks', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-4096/store_user', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-4096/align', 0o664, b'8\n')
f('sys/kernel/slab/kmalloc-4096/partial', 0o664, b'5\n')
f('sys/kernel/slab/kmalloc-4096/validate', 0o664, b'')
f('sys/kernel/slab/kmalloc-4096/red_zone', 0o664, b'1\n')
f('sys/kernel/slab/kmalloc-4096/free_calls', 0o664, b'''    187 <not-available> age=4303047324 pid=0 cpus=0
      5 seq_release+0x18/0x30 age=61430/6709712/8374782 pid=107-1371 cpus=0-1
      1 cryptomgr_probe+0x81/0xf0 age=8373564 pid=745 cpus=0
     30 kobject_uevent_env+0x122/0x470 age=22643/7542644/8379894 pid=1-1549 cpus=0-1
      1 show_uevent+0xee/0x110 age=2013 pid=13466 cpus=0
      1 input_dev_release+0x23/0x40 age=8378913 pid=17 cpus=0
    130 skb_release_data+0x85/0xd0 age=2011/73374/8354535 pid=0-13466 cpus=0-1
''')
f('sys/kernel/slab/kmalloc-4096/aliases', 0o664, b'0\n')
d('sys/kernel/debug', 0o775)
d('sys/kernel/debug/x86', 0o775)
f('sys/kernel/debug/x86/pat_memtype_list', 0o664, b'')
d('sys/kernel/debug/bdi', 0o775)
d('sys/kernel/debug/bdi/7:1', 0o775)
f('sys/kernel/debug/bdi/7:1/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:0', 0o775)
f('sys/kernel/debug/bdi/7:0/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189292 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:2', 0o775)
f('sys/kernel/debug/bdi/7:2/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:4', 0o775)
f('sys/kernel/debug/bdi/7:4/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:3', 0o775)
f('sys/kernel/debug/bdi/7:3/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/11:0', 0o775)
f('sys/kernel/debug/bdi/11:0/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189292 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:5', 0o775)
f('sys/kernel/debug/bdi/7:5/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/8:0', 0o775)
f('sys/kernel/debug/bdi/8:0/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:      13696 kB
BdiDirtyThresh:      83368 kB
DirtyThresh:        189292 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/default', 0o775)
f('sys/kernel/debug/bdi/default/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189292 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/0:16', 0o775)
f('sys/kernel/debug/bdi/0:16/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189272 kB
BackgroundThresh:    94636 kB
''')
d('sys/kernel/debug/bdi/8:16', 0o775)
f('sys/kernel/debug/bdi/8:16/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189272 kB
BackgroundThresh:    94636 kB
''')
d('sys/kernel/debug/bdi/9:0', 0o775)
f('sys/kernel/debug/bdi/9:0/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189292 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:7', 0o775)
f('sys/kernel/debug/bdi/7:7/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/debug/bdi/7:6', 0o775)
f('sys/kernel/debug/bdi/7:6/stats', 0o664, b'''BdiWriteback:            0 kB
BdiReclaimable:          0 kB
BdiDirtyThresh:          0 kB
DirtyThresh:        189288 kB
BackgroundThresh:    94644 kB
''')
d('sys/kernel/uids', 0o775)
d('sys/kernel/uids/100', 0o775)
f('sys/kernel/uids/100/cpu_share', 0o664, b'1024\n')
d('sys/kernel/uids/103', 0o775)
f('sys/kernel/uids/103/cpu_share', 0o664, b'1024\n')
d('sys/kernel/uids/51', 0o775)
f('sys/kernel/uids/51/cpu_share', 0o664, b'1024\n')
d('sys/kernel/uids/105', 0o775)
f('sys/kernel/uids/105/cpu_share', 0o664, b'1024\n')
d('sys/kernel/uids/2702', 0o775)
f('sys/kernel/uids/2702/cpu_share', 0o664, b'1024\n')
d('sys/kernel/uids/0', 0o775)
f('sys/kernel/uids/0/cpu_share', 0o664, b'2048\n')
d('sys/power', 0o775)
f('sys/power/state', 0o664, b'mem\n')
d('sys/firmware', 0o775)
d('sys/firmware/memmap', 0o775)
d('sys/firmware/memmap/8', 0o775)
f('sys/firmware/memmap/8/start', 0o664, b'0xfec00000\n')
f('sys/firmware/memmap/8/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/8/end', 0o664, b'0xfec0ffff\n')
d('sys/firmware/memmap/11', 0o775)
f('sys/firmware/memmap/11/start', 0o664, b'0xfed1c000\n')
f('sys/firmware/memmap/11/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/11/end', 0o664, b'0xfed8ffff\n')
d('sys/firmware/memmap/6', 0o775)
f('sys/firmware/memmap/6/start', 0o664, b'0x7ff00000\n')
f('sys/firmware/memmap/6/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/6/end', 0o664, b'0x7fffffff\n')
d('sys/firmware/memmap/5', 0o775)
f('sys/firmware/memmap/5/start', 0o664, b'0x7fedf000\n')
f('sys/firmware/memmap/5/type', 0o664, b'ACPI Non-volatile Storage\n')
f('sys/firmware/memmap/5/end', 0o664, b'0x7fefffff\n')
d('sys/firmware/memmap/9', 0o775)
f('sys/firmware/memmap/9/start', 0o664, b'0xfed00000\n')
f('sys/firmware/memmap/9/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/9/end', 0o664, b'0xfed003ff\n')
d('sys/firmware/memmap/2', 0o775)
f('sys/firmware/memmap/2/start', 0o664, b'0xdc000\n')
f('sys/firmware/memmap/2/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/2/end', 0o664, b'0xfffff\n')
d('sys/firmware/memmap/12', 0o775)
f('sys/firmware/memmap/12/start', 0o664, b'0xfee00000\n')
f('sys/firmware/memmap/12/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/12/end', 0o664, b'0xfee00fff\n')
d('sys/firmware/memmap/4', 0o775)
f('sys/firmware/memmap/4/start', 0o664, b'0x7fed0000\n')
f('sys/firmware/memmap/4/type', 0o664, b'ACPI Tables\n')
f('sys/firmware/memmap/4/end', 0o664, b'0x7fedefff\n')
d('sys/firmware/memmap/3', 0o775)
f('sys/firmware/memmap/3/start', 0o664, b'0x100000\n')
f('sys/firmware/memmap/3/type', 0o664, b'System RAM\n')
f('sys/firmware/memmap/3/end', 0o664, b'0x7fecffff\n')
d('sys/firmware/memmap/10', 0o775)
f('sys/firmware/memmap/10/start', 0o664, b'0xfed14000\n')
f('sys/firmware/memmap/10/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/10/end', 0o664, b'0xfed19fff\n')
d('sys/firmware/memmap/0', 0o775)
f('sys/firmware/memmap/0/start', 0o664, b'0x0\n')
f('sys/firmware/memmap/0/type', 0o664, b'System RAM\n')
f('sys/firmware/memmap/0/end', 0o664, b'0x9efff\n')
d('sys/firmware/memmap/7', 0o775)
f('sys/firmware/memmap/7/start', 0o664, b'0xf0000000\n')
f('sys/firmware/memmap/7/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/7/end', 0o664, b'0xf3ffffff\n')
d('sys/firmware/memmap/1', 0o775)
f('sys/firmware/memmap/1/start', 0o664, b'0x9f000\n')
f('sys/firmware/memmap/1/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/1/end', 0o664, b'0x9ffff\n')
d('sys/firmware/memmap/13', 0o775)
f('sys/firmware/memmap/13/start', 0o664, b'0xff800000\n')
f('sys/firmware/memmap/13/type', 0o664, b'reserved\n')
f('sys/firmware/memmap/13/end', 0o664, b'0xffffffff\n')
d('sys/firmware/edd', 0o775)
d('sys/firmware/edd/int13_dev80', 0o775)
l('sys/firmware/edd/int13_dev80/pci_dev', '../../../devices/pci0000:00/0000:00:1f.2')
f('sys/firmware/edd/int13_dev80/version', 0o664, b'0x30\n')
f('sys/firmware/edd/int13_dev80/raw_data', 0o664, b'J\x00\x01\x00\xff?\x00\x00\x10\x00\x00\x00?\x00\x00\x000"\xa5\x0b\x00\x00\x00\x00\x00\x02\xc6\x00@\x00\xdd\xbe,\x00\x00\x00PCI ATA     \x00\x1f\x02\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa5')
f('sys/firmware/edd/int13_dev80/sectors', 0o664, b'195371568\n')
f('sys/firmware/edd/int13_dev80/default_sectors_per_track', 0o664, b'63\n')
f('sys/firmware/edd/int13_dev80/legacy_max_cylinder', 0o664, b'1022\n')
f('sys/firmware/edd/int13_dev80/interface', 0o664, b'ATA     \tdevice: 0\n')
f('sys/firmware/edd/int13_dev80/host_bus', 0o664, b'PCI \t00:1f.2  channel: 1\n')
f('sys/firmware/edd/int13_dev80/legacy_max_head', 0o664, b'239\n')
f('sys/firmware/edd/int13_dev80/mbr_signature', 0o664, b'0x00087b5f\n')
f('sys/firmware/edd/int13_dev80/info_flags', 0o664, b'DMA boundary error transparent\n')
f('sys/firmware/edd/int13_dev80/default_heads', 0o664, b'16\n')
f('sys/firmware/edd/int13_dev80/default_cylinders', 0o664, b'16383\n')
f('sys/firmware/edd/int13_dev80/legacy_sectors_per_track', 0o664, b'63\n')
f('sys/firmware/edd/int13_dev80/extensions', 0o664, b'''Fixed disk access
Enhanced Disk Drive support
''')
d('sys/firmware/acpi', 0o775)
d('sys/firmware/acpi/tables', 0o775)
f('sys/firmware/acpi/tables/DSDT', 0o664, b'''DSDT4\xd2\x00\x00\x01\x9aLENOVOTP-7I    \x11\x00\x00MSFT\x0e\x00\x00\x01\x10 \\_PR_[\x83\x0bCPU0\x00\x10\x10\x00\x00\x06[\x83\x0bCPU1\x01\x10\x10\x00\x00\x06\x10\x83I\x0b\\_SB_\x14C2_INI\x00\xa0(\x93\\SCMP\\_OS_\rMicrosoft Windows\x00\x00p
\x01\\W98F\xa1C\x11\xa0F\t[\x12\\_OSI`\xa0,\\_OSI\rWindows 2001\x00p
\x01\\WNTFp
\x01\\WXPFp
\x00\\WSPV\xa0 \\_OSI\rWindows 2001 SP1\x00p
\x01\\WSPV\xa0 \\_OSI\rWindows 2001 SP2\x00p
\x02\\WSPV\xa0\x1c\\_OSI\rWindows 2006\x00p
\x01\\WVIS\xa1I\x07\xa0+\x93\\SCMP\\_OS_\rMicrosoft Windows NT\x00\x00p
\x01\\WNTF\xa1J\x04\xa0G\x04\x93\\SCMP\\_OS_\rMicrosoft WindowsME: Millennium Edition\x00\x00p
\x01\\WMEFp
\x01\\W98F\xa0\x12\x92\x95\\_REV
\x02p
\x01\\H8DRp
\x01\\OSIF\\/\x05_SB_PCI0LPC_MOU_MHIDp\\SRAH\\/\x03_SB_PCI0RID_\xa0\x1eVIGDp\\SRHE\\/\x04_SB_PCI0VID_RID_\xa1\x1ap\\SRHE\\/\x04_SB_PCI0AGP_RID_p\\SRE0\\/\x04_SB_PCI0EXP0RID_p\\SRE1\\/\x04_SB_PCI0EXP1RID_p\\SRE2\\/\x04_SB_PCI0EXP2RID_p\\SRE3\\/\x04_SB_PCI0EXP3RID_p\\SRU0\\/\x04_SB_PCI0USB0RID_p\\SRU1\\/\x04_SB_PCI0USB1RID_p\\SRU2\\/\x04_SB_PCI0USB2RID_p\\SRU3\\/\x04_SB_PCI0USB3RID_p\\SRU7\\/\x04_SB_PCI0USB7RID_p\\SRPB\\/\x04_SB_PCI0PCI1RID_p\\SRLP\\/\x04_SB_PCI0LPC_RID_p\\SRSA\\/\x04_SB_PCI0IDE0RID_p\\SRSA\\/\x04_SB_PCI0SATARID_p\\SRSM\\/\x04_SB_PCI0SMBURID_[\x82K\x11LNKA\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x01\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRA\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRA
\x80\\/\x04_SB_PCI0LPC_PIRA\x08BUFA\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFA
\x01IRA1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRA
\x8f`\xa0\x0eVPIR`y
\x01`IRA1\xa1\x08p
\x00IRA1\xa4BUFA\x14F\x04_SRS\x01\x8bh
\x01IRA2\x82IRA2`{\\/\x04_SB_PCI0LPC_PIRA
pa}av`apa\\/\x04_SB_PCI0LPC_PIRA[\x82K\x11LNKB\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x02\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRB\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRB
\x80\\/\x04_SB_PCI0LPC_PIRB\x08BUFB\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFB
\x01IRB1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRB
\x8f`\xa0\x0eVPIR`y
\x01`IRB1\xa1\x08p
\x00IRB1\xa4BUFB\x14F\x04_SRS\x01\x8bh
\x01IRB2\x82IRB2`{\\/\x04_SB_PCI0LPC_PIRB
pa}av`apa\\/\x04_SB_PCI0LPC_PIRB[\x82K\x11LNKC\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x03\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRC\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRC
\x80\\/\x04_SB_PCI0LPC_PIRC\x08BUFC\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFC
\x01IRC1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRC
\x8f`\xa0\x0eVPIR`y
\x01`IRC1\xa1\x08p
\x00IRC1\xa4BUFC\x14F\x04_SRS\x01\x8bh
\x01IRC2\x82IRC2`{\\/\x04_SB_PCI0LPC_PIRC
pa}av`apa\\/\x04_SB_PCI0LPC_PIRC[\x82K\x11LNKD\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x04\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRD\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRD
\x80\\/\x04_SB_PCI0LPC_PIRD\x08BUFD\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFD
\x01IRD1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRD
\x8f`\xa0\x0eVPIR`y
\x01`IRD1\xa1\x08p
\x00IRD1\xa4BUFD\x14F\x04_SRS\x01\x8bh
\x01IRD2\x82IRD2`{\\/\x04_SB_PCI0LPC_PIRD
pa}av`apa\\/\x04_SB_PCI0LPC_PIRD[\x82K\x11LNKE\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x05\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRE\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRE
\x80\\/\x04_SB_PCI0LPC_PIRE\x08BUFE\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFE
\x01IRE1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRE
\x8f`\xa0\x0eVPIR`y
\x01`IRE1\xa1\x08p
\x00IRE1\xa4BUFE\x14F\x04_SRS\x01\x8bh
\x01IRE2\x82IRE2`{\\/\x04_SB_PCI0LPC_PIRE
pa}av`apa\\/\x04_SB_PCI0LPC_PIRE[\x82K\x11LNKF\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x06\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRF\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRF
\x80\\/\x04_SB_PCI0LPC_PIRF\x08BUFF\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFF
\x01IRF1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRF
\x8f`\xa0\x0eVPIR`y
\x01`IRF1\xa1\x08p
\x00IRF1\xa4BUFF\x14F\x04_SRS\x01\x8bh
\x01IRF2\x82IRF2`{\\/\x04_SB_PCI0LPC_PIRF
pa}av`apa\\/\x04_SB_PCI0LPC_PIRF[\x82K\x11LNKG\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x07\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRG\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRG
\x80\\/\x04_SB_PCI0LPC_PIRG\x08BUFG\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFG
\x01IRG1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRG
\x8f`\xa0\x0eVPIR`y
\x01`IRG1\xa1\x08p
\x00IRG1\xa4BUFG\x14F\x04_SRS\x01\x8bh
\x01IRG2\x82IRG2`{\\/\x04_SB_PCI0LPC_PIRG
pa}av`apa\\/\x04_SB_PCI0LPC_PIRG[\x82K\x11LNKH\x08_HID\x0cA\xd0\x0c\x0f\x08_UID
\x08\x14(_STA\x00\xa0\x1c\x92VPIR\\/\x04_SB_PCI0LPC_PIRH\xa4
\t\xa1\x04\xa4
\x0b\x08_PRS\x11\t
\x06#\xf8\x0e\x18y\x00\x14/_DIS\x00}\\/\x04_SB_PCI0LPC_PIRH
\x80\\/\x04_SB_PCI0LPC_PIRH\x08BUFH\x11\t
\x06#\x00\x00\x18y\x00\x8bBUFH
\x01IRH1\x14:_CRS\x00{\\/\x04_SB_PCI0LPC_PIRH
\x8f`\xa0\x0eVPIR`y
\x01`IRH1\xa1\x08p
\x00IRH1\xa4BUFH\x14F\x04_SRS\x01\x8bh
\x01IRH2\x82IRH2`{\\/\x04_SB_PCI0LPC_PIRH
pa}av`apa\\/\x04_SB_PCI0LPC_PIRH\x149VPIR\x01p
\x01`\xa0
{h
\x80\x00p
\x00`\xa1!{h
\x0fa\xa0\t\x95a
\x03p
\x00`\xa1\x10\xa0\x0e\x91\x93a
\x08\x93a
\rp
\x00`\xa4`[\x82HhMEM_\x08_HID\x0cA\xd0\x0c\x01\x08ME98\x11B\x04
>\x86\t\x00\x01\x00\x00\x00\x00\x00\x00
\x00\x86\t\x00\x00\x00\x00\x0e\x00\x00\x00\x02\x00\x86\t\x00\x01\x00\x00\x10\x00\x00\x00\xee\x01\x86\t\x00\x00\x00\x00\xc0\xfe\x00\x00\x14\x00\x86\t\x00\x00\x00\x10\xd4\xfe\x00\xf0+\x01y\x00\x8aME98
\x1cMEB0\x8aME98
 MEL0\x08MGAP\x11\x11
\x0e\x86\t\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00y\x00\x8aMGAP
\x04MGPB\x8aMGAP
\x08MGPL\x08MEMS\x11B\r
\xce\x86\t\x00\x01\x00\x00\x00\x00\x00\x00
\x00\x86\t\x00\x00\x00\x00\x0c\x00\x00\x00\x00\x00\x86\t\x00\x00\x00@\x0c\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x80\x0c\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\xc0\x0c\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x00\r\x00\x00\x00\x00\x00\x86\t\x00\x00\x00@\r\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x80\r\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\xc0\r\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x00\x0e\x00\x00\x00\x00\x00\x86\t\x00\x00\x00@\x0e\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x80\x0e\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\xc0\x0e\x00\x00\x00\x00\x00\x86\t\x00\x00\x00\x00\x0f\x00\x00\x00\x01\x00\x86\t\x00\x01\x00\x00\x10\x00\x00\x00\xee\x01\x86\t\x00\x00\x00\x00\xc0\xfe\x00\x00\x14\x00\x86\t\x00\x00\x00\x10\xd4\xfe\x00\xf0+\x01y\x00\x8aMEMS
\x14MC0L\x8aMEMS
 MC4L\x8aMEMS
,MC8L\x8aMEMS
8MCCL\x8aMEMS
DMD0L\x8aMEMS
PMD4L\x8aMEMS
\\MD8L\x8aMEMS
hMDCL\x8aMEMS
tME0L\x8aMEMS
\x80ME4L\x8aMEMS
\x8cME8L\x8aMEMS
\x98MECL\x8dMEMS
xMC0W\x8dMEMS
\xd8MC4W\x8dMEMS\x0b8\x01MC8W\x8dMEMS\x0b\x98\x01MCCW\x8dMEMS\x0b\xf8\x01MD0W\x8dMEMS\x0bX\x02MD4W\x8dMEMS\x0b\xb8\x02MD8W\x8dMEMS\x0b\x18\x03MDCW\x8dMEMS\x0bx\x03ME0W\x8dMEMS\x0b\xd8\x03ME4W\x8dMEMS\x0b8\x04ME8W\x8dMEMS\x0b\x98\x04MECW\x8aMEMS
\xacMEB1\x8aMEMS
\xb0MEL1\x8aMEMS
\xbcMEL2\x8aMEMS
\xc8MEL3\x14F=_CRS\x00\xa0M\x17\\W98Ft\\MEMXMEB0MEL0p\\GAPAMGPBp\\GAPLMGPL\xa0I\x0c\x90MGPBMGPLt\x87ME98
\x02`\x08MBF0\x11\x02`r`\x87MGAP`\x08MBF1\x11\x02`pME98MBF0sMBF0MGAPMBF1\xa0O\x07\x92\x93\\/\x05_SB_PCI0LPC_TPM__STA
\x0ft\x87MBF1
\x02`\x08MBF2\x11\x02`r`\x87\\/\x05_SB_PCI0LPC_TPM__CRS`\x08MBF3\x11\x02`pMBF1MBF2sMBF2\\/\x05_SB_PCI0LPC_TPM__CRSMBF3\xa4MBF3\xa1\x06\xa4MBF1\xa1I\x08\xa0O\x07\x92\x93\\/\x05_SB_PCI0LPC_TPM__STA
\x0ft\x87ME98
\x02`\x08MBF4\x11\x02`r`\x87\\/\x05_SB_PCI0LPC_TPM__CRS`\x08MBF5\x11\x02`pME98MBF4sMBF4\\/\x05_SB_PCI0LPC_TPM__CRSMBF5\xa4MBF5\xa1\x06\xa4ME98{\\/\x03_SB_PCI0PAM1
\x03`\xa0\x18`p\x0b\x00@MC0L\xa0\r{`
\x02\x00p
\x01MC0W{\\/\x03_SB_PCI0PAM1
0`\xa0\x18`p\x0b\x00@MC4L\xa0\r{`
 \x00p
\x01MC4W{\\/\x03_SB_PCI0PAM2
\x03`\xa0\x18`p\x0b\x00@MC8L\xa0\r{`
\x02\x00p
\x01MC8W{\\/\x03_SB_PCI0PAM2
0`\xa0\x18`p\x0b\x00@MCCL\xa0\r{`
 \x00p
\x01MCCW{\\/\x03_SB_PCI0PAM3
\x03`\xa0\x18`p\x0b\x00@MD0L\xa0\r{`
\x02\x00p
\x01MD0W{\\/\x03_SB_PCI0PAM3
0`\xa0\x18`p\x0b\x00@MD4L\xa0\r{`
 \x00p
\x01MD4W{\\/\x03_SB_PCI0PAM4
\x03`\xa0\x18`p\x0b\x00@MD8L\xa0\r{`
\x02\x00p
\x01MD8W{\\/\x03_SB_PCI0PAM4
0`\xa0\x18`p\x0b\x00@MDCL\xa0\r{`
 \x00p
\x01MDCW{\\/\x03_SB_PCI0PAM5
\x03`\xa0\x18`p\x0b\x00@ME0L\xa0\r{`
\x02\x00p
\x01ME0W{\\/\x03_SB_PCI0PAM5
0`\xa0\x18`p\x0b\x00@ME4L\xa0\r{`
 \x00p
\x01ME4W{\\/\x03_SB_PCI0PAM6
\x03`\xa0\x18`p\x0b\x00@ME8L\xa0\r{`
\x02\x00p
\x01ME8W{\\/\x03_SB_PCI0PAM6
0`\xa0\x18`p\x0b\x00@MECL\xa0\r{`
 \x00p
\x01MECWt\\MEMXMEB1MEL1\xa0-\x92\x93\\/\x05_SB_PCI0LPC_TPM__STA
\x0fp\x0c\x00\x00@\x01MEL2p
\x00MEL3\xa4MEMS[\x82I\rLID_\x08_HID\x0cA\xd0\x0c\r\x14<_LID\x00\xa0\x1e\\H8DR\xa4\\/\x05_SB_PCI0LPC_EC__HPLD\xa1\x16\xa0\x0f{\\RBEC
F
\x04\x00\xa4
\x01\xa1\x04\xa4
\x00\x14&_PRW\x00\xa0\x15\x90\\W98F\x92\\WMEF\xa4\x12\x06\x02
\x18
\x04\xa1\t\xa4\x12\x06\x02
\x18
\x03\x14D\x06_PSW\x01\xa0?\\H8DR\xa0\x1chp
\x01\\/\x05_SB_PCI0LPC_EC__HWLO\xa1\x1bp
\x00\\/\x05_SB_PCI0LPC_EC__HWLO\xa1\x1c\xa0\rh\\MBEC
2
\xff
\x04\xa1\x0c\\MBEC
2
\xfb
\x00[\x82L\tSLPB\x08_HID\x0cA\xd0\x0c\x0e\x14&_PRW\x00\xa0\x15\x90\\W98F\x92\\WMEF\xa4\x12\x06\x02
\x18
\x04\xa1\t\xa4\x12\x06\x02
\x18
\x03\x14D\x06_PSW\x01\xa0?\\H8DR\xa0\x1chp
\x01\\/\x05_SB_PCI0LPC_EC__HWFN\xa1\x1bp
\x00\\/\x05_SB_PCI0LPC_EC__HWFN\xa1\x1c\xa0\rh\\MBEC
2
\xff
\x10\xa1\x0c\\MBEC
2
\xef
\x00[\x82\x8fH\x07PCI0[\x82\x8a\x96\x04LPC_\x08_ADR\x0c\x00\x00\x1f\x00\x08_S3D
\x03\x08RID_
\x00[\x82M\x1aSIO_\x08_HID\x0cA\xd0\x0c\x02\x08_UID
\x00\x08SCRS\x11B\x10
\xfeG\x01\x10\x00\x10\x00\x01\x10G\x01\x90\x00\x90\x00\x01\x10G\x01$\x00$\x00\x01\x02G\x01(\x00(\x00\x01\x02G\x01,\x00,\x00\x01\x02G\x010\x000\x00\x01\x02G\x014\x004\x00\x01\x02G\x018\x008\x00\x01\x02G\x01<\x00<\x00\x01\x02G\x01\xa4\x00\xa4\x00\x01\x02G\x01\xa8\x00\xa8\x00\x01\x02G\x01\xac\x00\xac\x00\x01\x02G\x01\xb0\x00\xb0\x00\x01\x06G\x01\xb8\x00\xb8\x00\x01\x02G\x01\xbc\x00\xbc\x00\x01\x02G\x01P\x00P\x00\x01\x04G\x01r\x00r\x00\x01\x06G\x01N\x16N\x16\x01\x02G\x01.\x00.\x00\x01\x02G\x01\x00\x10\x00\x10\x01\x80G\x01\x80\x11\x80\x11\x01@G\x01\x00\x08\x00\x08\x01\x10G\x01\xe0\x15\xe0\x15\x01\x10G\x01\x00\x16\x00\x16\x01`\x86\t\x00\x01\x00\x00\x00\xf0\x00\x00\x00\x04\x86\t\x00\x01\x00\xc0\xd1\xfe\x00@\x00\x00\x86\t\x00\x01\x00@\xd1\xfe\x00@\x00\x00\x86\t\x00\x01\x00\x80\xd1\xfe\x00\x10\x00\x00\x86\t\x00\x01\x00\x90\xd1\xfe\x00\x10\x00\x00y\x00\x14M\x08_CRS\x00\xa0 \x93\\/\x05_SB_PCI0LPC_TPM__STA
\x0f\xa4SCRS\xa1D\x06t\x87SCRS
\x02`\x08BUF0\x11\x02`r`\x87\\/\x05_SB_PCI0LPC_TPM__CRS`\x08BUF1\x11\x02`pSCRSBUF0sBUF0\\/\x05_SB_PCI0LPC_TPM__CRSBUF1\xa4BUF1[\x80LPCS\x02
\x00\x0b\x00\x01[\x81G
LPCS\x00\x00@0PIRA\x08PIRB\x08PIRC\x08PIRD\x08SERQ\x08\x00\x18PIRE\x08PIRF\x08PIRG\x08PIRH\x08\x00@
XU1A\x03\x00\x01XU2A\x03\x00\x01XPA_\x02\x00\x02XFA_\x01\x00\x03XU1E\x01XU2E\x01XPE_\x01XFE_\x01\x00\x0cXG1E\x01\x00\x06XG1A\t\x00\x10XG2E\x01\x00\x03XG2A\x0c\x00@\x0b\x00\x02CLKR\x01GYEN\x01\x00\x03C4C3\x01\x00\x02EXPE\x01\x00\x05[\x80LPIO\x01\x0b\x80\x11
@[\x81O
LPIO\x03GU00\x08GU01\x08GU02\x08GU03\x08GI00\x08GI01\x08GI02\x08GI03\x08\x00 GL00\x08GL01\x08GL02\x08GL03\x08\x00@\x04GB00\x08GB01\x08GB02\x08GB03\x08\x00@\x08GV00\x08GV01\x08GV02\x08GV03\x08GU04\x08GU05\x08GU06\x08GU07\x08GI04\x08GI05\x08GI06\x08GI07\x08GL04\x08GL05\x08GL06\x08GL07\x08[\x80PMIO\x01\x0b\x00\x10
\x80[\x81\x10PMIO\x00\x00@!\x00\x01SWGE\x01[\x823PIC_\x08_HID\x0bA\xd0\x08_CRS\x11 
\x1dG\x01 \x00 \x00\x01\x02G\x01\xa0\x00\xa0\x00\x01\x02G\x01\xd0\x04\xd0\x04\x01\x02"\x04\x00y\x00[\x82%TIMR\x08_HID\x0cA\xd0\x01\x00\x08_CRS\x11\x10
\rG\x01@\x00@\x00\x01\x04"\x01\x00y\x00[\x82B\x05HPET\x08_HID\x0cA\xd0\x01\x03\x14*_STA\x00\xa0\x08W98F\xa4
\x00\xa1\x17\xa0\x10\x90\\WNTF\x92\\WXPF\xa4
\x00\xa1\x04\xa4
\x0f\xa4
\x00\x08_CRS\x11\x11
\x0e\x86\t\x00\x00\x00\x00\xd0\xfe\x00\x04\x00\x00y\x00[\x825DMAC\x08_HID\x0cA\xd0\x02\x00\x08_CRS\x11 
\x1dG\x01\x00\x00\x00\x00\x01\x10G\x01\x80\x00\x80\x00\x01\x10G\x01\xc0\x00\xc0\x00\x01 *\x10\x05y\x00[\x82"SPKR\x08_HID\x0cA\xd0\x08\x00\x08_CRS\x11\r

G\x01a\x00a\x00\x01\x01y\x00[\x82%FPU_\x08_HID\x0cA\xd0\x0c\x04\x08_CRS\x11\x10
\rG\x01\xf0\x00\xf0\x00\x01\x01"\x00 y\x00[\x82%RTC_\x08_HID\x0cA\xd0\x0b\x00\x08_CRS\x11\x10
\rG\x01p\x00p\x00\x01\x02"\x00\x01y\x00[\x82-KBD_\x08_HID\x0cA\xd0\x03\x03\x08_CRS\x11\x18
\x15G\x01`\x00`\x00\x01\x01G\x01d\x00d\x00\x01\x01"\x02\x00y\x00[\x82J\x05MOU_\x08_HID\x0c$M7\x80\x08_CID\x0cA\xd0\x0f\x13\x08_CRS\x11\x08
\x05"\x00\x10y\x00\x141MHID\x00\xa0\x1e\\/\x04_SB_PCI0LPC_PADDp\x0c$M7\x80_HID\xa1\x0bp\x0c$M\x00W_HID[\x80IMGA\x01\x0b\xe0\x15
\x10[\x81!IMGA\x01\x00\x08\x00\x08\x00\x08WAKR\x10\x008GAIX\x08\x00\x08GADT\x08\x00\x08[\x86I\x0bGAIXGADT\x01\x00H VAUX\x02\x00\x01WOLE\x01\x00\x04\x00@\x07\x00\x01CBPW\x01CBSL\x01VDPW\x01PDNE\x01BLPL\x01\x00\x01LEDS\x01TP4R\x01PADR\x01BPAD\x01\x00\x01\x00\x01PADD\x01\x00
DLAN\x01\x00G\x06BUSC\x01BUSD\x01\x00\x01DSCI\x01\x00\x04EPWG\x01\x00\x02DSCS\x01\x00\x04CSON\x01URST\x01\x00N\x06GDR0\x01GDR1\x01GDR2\x01\x00\x05GDT0\x01GDT1\x01GDT2\x01\x00E(GAID\x08[\x80NCFG\x01
.
\x02[\x81\x10NCFG\x01INDX\x08DATA\x08[\x86B
INDXDATA\x01\x008LDN_\x08\x00@\x0cSIOD\x08\x00\x08SCF2\x08\x00\x08SCF4\x08\x00\x02PPSE\x01\x00\x04PNF_\x01FDCD\x01PPD_\x01SP2D\x01SP1D\x01\x00\x01GPSR\x02\x00\x01SRID\x08\x00\x08SCF9\x08\x00\x08\x00(LDA_\x01\x00\x07\x00H\x17IOHI\x08IOLW\x08\x00@\x07IRQN\x04IRQW\x01\x00\x03IRQT\x01IRQL\x01\x00\x06\x00\x10DMA0\x03\x00\x05DMA1\x03\x00\x05[\x86#INDXDATA\x01\x00@xPTRS\x01PPMC\x01\x00\x02PERA\x01PMDS\x03[\x86#INDXDATA\x01\x00@xSTRS\x01SPMC\x01SBSY\x01\x00\x04SBSE\x01[\x86\x1cINDXDATA\x01\x00@xGPPS\x08GPPC\x08GPER\x08[\x80CFGS\x01\x0bN\x16
\x02[\x81\x10CFGS\x01NDXS\x08ATAS\x08[\x86J
NDXSATAS\x01\x008LDNS\x08\x00@\x0c\x00\x08\x00\x08\x00\x08\x00\x08\x00\x08\x00\x02PSES\x01\x00\x04PNFS\x01DCDS\x01PPDS\x01SP2S\x01SP1S\x01\x00\x01PSRS\x02\x00\x01RIDS\x08\x00\x08CCSS\x02CCES\x01MCSS\x01MESS\x01\x00\x03\x00\x08\x00(LDAS\x01\x00\x07\x00H\x17OHIS\x08OLWS\x08\x00@\x07RQNS\x04RQWS\x01\x00\x03RQTS\x01RQLS\x01\x00\x06\x00\x10MA0S\x03\x00\x05MA1S\x03\x00\x05[\x86#NDXSATAS\x01\x00@xTRSS\x01PMCS\x01BSYS\x01\x00\x04SESS\x01[\x80NSDL\x01\x0bL\x16
\x01[\x81\x0bNSDL\x01DLPC\x08[\x80NSIO\x01\x0b\x80\x16
\x0c[\x81\'NSIO\x01DRST\x01DLPD\x01\x00\x06\x00\x04DKI0\x01\x00\x03\x008DKI1\x01DKI2\x01[\x80DSIO\x01\x0b \x16
\x0c[\x81\x0fDSIO\x01\x00\x01DUSB\x01\x00\x06[\x82K\x18FDC_\x08_HID\x0cA\xd0\x07\x00\x14 _STA\x00\xa0\x14\\LFDC\xa0\x08XFE_\xa4
\x0f\xa1\x04\xa4
\r\xa1\x04\xa4
\x00\x14)_DIS\x00p
\x00XFE_p
\x00LDN_p
\x00IRQNp
\x00LDA_p
\x01FDCD\x08_CRS\x11\x1b
\x18G\x01\xf0\x03\xf0\x03\x01\x06G\x01\xf7\x03\xf7\x03\x01\x01"@\x00*\x04\x00y\x00\x08_PRS\x11\x1b
\x18G\x01\xf0\x03\xf0\x03\x01\x06G\x01\xf7\x03\xf7\x03\x01\x01"@\x00*\x04\x00y\x00\x14M\x04_SRS\x01p
\x00LDN_p
\x00LDA_p
\x03IOHIp
\xf0IOLWp
\x06IRQNp
\x02DMA0p
\x00FDCDp
\x01LDA_p
\x00XFA_p
\x01XFE_\x14\x1b_PSC\x00p
\x00LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x00LDN_p
\x01LDA_\x14\x14_PS3\x00p
\x00LDN_p
\x00LDA_\x14#SLFD\x01\xa0\x0eh\\MISA\x0b\xf3\x03
\xf3
\x04\xa1\r\\MISA\x0b\xf3\x03
\xf3
\x00[\x824FDD0\x08_ADR
\x00\x08_FDI\x12"\x10
\x00
\x04
O
\x12
\x01
\xdf
\x02
%
\x02
\x12
\x1b
\xff
l
\xf6
\x0f
\x05\x10\x86/\x01\\/\x03_SB_PCI0LPC_[\x82A-LURT\x08_UID
\x00\x14-_STA\x00\xa0\x16\x93\\/\x03_SB_LCIO_STA
\x00\xa4
\x00\xa1\x0f\xa0\x08XU1E\xa4
\x0f\xa1\x04\xa4
\r\x08_HID\x0cA\xd0\x05\x01\x08_PRW\x12\x06\x02
\x18
\x03\x14D\x06_PSW\x01\xa0?\\H8DR\xa0\x1chp
\x01\\/\x05_SB_PCI0LPC_EC__HWRI\xa1\x1bp
\x00\\/\x05_SB_PCI0LPC_EC__HWRI\xa1\x1c\xa0\rh\\MBEC
2
\xff
@\xa1\x0c\\MBEC
2
\xbf
\x00\x14)_DIS\x00p
\x00XU1Ep
\x03LDN_p
\x00IRQNp
\x00LDA_p
\x01SP1D\x08U1BF\x11\x10
\rG\x01\x00\x00\x00\x00\x01\x08"\x00\x00y\x00\x8bU1BF
\x02U1MN\x8bU1BF
\x04U1MX\x8bU1BF
\tU1IQ\x14J\x04_CRS\x00p
\x03LDN_}yIOHI
\x08\x00IOLW`p`U1MNp`U1MXpIRQN`\xa0\r`y
\x01IRQNU1IQ\xa1\x08p
\x00U1IQ\xa4U1BF\x08_PRS\x11O\x06
k1\x00G\x01\xf8\x03\xf8\x03\x01\x08"\x10\x001\x01G\x01\xf8\x02\xf8\x02\x01\x08"\x08\x001\x01G\x01\xe8\x03\xe8\x03\x01\x08"\x10\x001\x01G\x01\xe8\x02\xe8\x02\x01\x08"\x08\x001\x02G\x01\xf8\x03\xf8\x03\x01\x08"\xa8\x001\x02G\x01\xf8\x02\xf8\x02\x01\x08"\xb0\x001\x02G\x01\xe8\x03\xe8\x03\x01\x08"\xa8\x001\x02G\x01\xe8\x02\xe8\x02\x01\x08"\xb0\x008y\x00\x14O\x0c_SRS\x01\x8ch
\x02RUIL\x8ch
\x03RUIH\x8bh
\x02RUIO\x8bh
\tRUIQp
\x03LDN_p
\x00LDA_pRUILIOLWpRUIHIOHI\xa0\x12RUIQ\x82RUIQ`pv`IRQN\xa1\x08p
\x00IRQNp
\x00SP1Dp
\x01LDA_\xa0\x10\x93RUIO\x0b\xf8\x03p
\x00XU1A\xa1E\x04\xa0\x10\x93RUIO\x0b\xf8\x02p
\x01XU1A\xa11\xa0\x10\x93RUIO\x0b\xe8\x03p
\x07XU1A\xa1\x1e\xa0\x10\x93RUIO\x0b\xe8\x02p
\x05XU1A\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\xb5\x01p
\x01XU1E\x14\x1b_PSC\x00p
\x03LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x06_PS0\x00\x14\x06_PS3\x00[\x82G-DURT\x08_UID
\x01\x143_STA\x00\xa0\x1c\x92{
\x08\\/\x04_SB_PCI0LPC_DLPC\x00\xa4
\x00\xa1\x0f\xa0\x08XU1E\xa4
\x0f\xa1\x04\xa4
\r\x08_HID\x0cA\xd0\x05\x01\x08_PRW\x12\x06\x02
\x18
\x03\x14D\x06_PSW\x01\xa0?\\H8DR\xa0\x1chp
\x01\\/\x05_SB_PCI0LPC_EC__HWRI\xa1\x1bp
\x00\\/\x05_SB_PCI0LPC_EC__HWRI\xa1\x1c\xa0\rh\\MBEC
2
\xff
@\xa1\x0c\\MBEC
2
\xbf
\x00\x14)_DIS\x00p
\x00XU1Ep
\x03LDN_p
\x00IRQNp
\x00LDA_p
\x01SP1D\x08U1BF\x11\x10
\rG\x01\x00\x00\x00\x00\x01\x08"\x00\x00y\x00\x8bU1BF
\x02U1MN\x8bU1BF
\x04U1MX\x8bU1BF
\tU1IQ\x14J\x04_CRS\x00p
\x03LDN_}yIOHI
\x08\x00IOLW`p`U1MNp`U1MXpIRQN`\xa0\r`y
\x01IRQNU1IQ\xa1\x08p
\x00U1IQ\xa4U1BF\x08_PRS\x11O\x06
k1\x00G\x01\xf8\x03\xf8\x03\x01\x08"\x10\x001\x01G\x01\xf8\x02\xf8\x02\x01\x08"\x08\x001\x01G\x01\xe8\x03\xe8\x03\x01\x08"\x10\x001\x01G\x01\xe8\x02\xe8\x02\x01\x08"\x08\x001\x02G\x01\xf8\x03\xf8\x03\x01\x08"\xa8\x001\x02G\x01\xf8\x02\xf8\x02\x01\x08"\xb0\x001\x02G\x01\xe8\x03\xe8\x03\x01\x08"\xa8\x001\x02G\x01\xe8\x02\xe8\x02\x01\x08"\xb0\x008y\x00\x14O\x0c_SRS\x01\x8ch
\x02RUIL\x8ch
\x03RUIH\x8bh
\x02RUIO\x8bh
\tRUIQp
\x03LDN_p
\x00LDA_pRUILIOLWpRUIHIOHI\xa0\x12RUIQ\x82RUIQ`pv`IRQN\xa1\x08p
\x00IRQNp
\x00SP1Dp
\x01LDA_\xa0\x10\x93RUIO\x0b\xf8\x03p
\x00XU1A\xa1E\x04\xa0\x10\x93RUIO\x0b\xf8\x02p
\x01XU1A\xa11\xa0\x10\x93RUIO\x0b\xe8\x03p
\x07XU1A\xa1\x1e\xa0\x10\x93RUIO\x0b\xe8\x02p
\x05XU1A\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\xb5\x01p
\x01XU1E\x14\x1b_PSC\x00p
\x03LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x06_PS0\x00\x14\x06_PS3\x00[\x82L1LLPT\x08_UID
\x00\x14=_STA\x00\xa0\x16\x93\\/\x03_SB_LCIO_STA
\x00\xa4
\x00\xa1\x1f\xa0\x18\x92\x93\\PMOD
\x03\xa0\x08XPE_\xa4
\x0f\xa1\x04\xa4
\r\xa1\x04\xa4
\x00\x08_HID\x0cA\xd0\x04\x00\x14)_DIS\x00p
\x00XPE_p
\x01LDN_p
\x00IRQNp
\x00LDA_p
\x01PPD_\x08PPBF\x11\x10
\rG\x01\x00\x00\x00\x00\x01\x00"\x00\x00y\x00\x8bPPBF
\x02LPN0\x8bPPBF
\x04LPX0\x8cPPBF
\x07LPL0\x8bPPBF
\tLPIQ\x14@\x07_CRS\x00\xa0\x0e\x93\\PMOD
\x03\xa4PPBFp
\x01LDN_}yIOHI
\x08\x00IOLW`p`LPN0p`LPX0\xa0\r\x93`\x0b\xbc\x03p
\x03LPL0\xa1\x08p
\x08LPL0pIRQN`\xa0\r`y
\x01IRQNLPIQ\xa1\x08p
\x00LPIQ\xa4PPBF\x14\x19_PRS\x00\xa0\x0b\\PMOD\xa4PEPP\xa1\x06\xa4PLPT\x08PLPT\x11O\x04
K0G\x01\xbc\x03\xbc\x03\x01\x03"\x80\x000G\x01x\x03x\x03\x01\x08"\x80\x000G\x01x\x02x\x02\x01\x08" \x000G\x01\xbc\x03\xbc\x03\x01\x03" \x000G\x01x\x03x\x03\x01\x08" \x000G\x01x\x02x\x02\x01\x08"\x80\x008y\x00\x08PEPP\x116
30G\x01x\x03x\x03\x01\x08"\x80\x000G\x01x\x02x\x02\x01\x08" \x000G\x01x\x03x\x03\x01\x08" \x000G\x01x\x02x\x02\x01\x08"\x80\x008y\x00\x14H\x0f_SRS\x01\x8ch
\x02RLIL\x8ch
\x03RLIH\x8bh
\x02RLIO\x8bh
\tRLIQp
\x01LDN_p
\x00LDA_pRLILIOLWpRLIHIOHI\xa0\x12RLIQ\x82RLIQ`pv`IRQN\xa1\x08p
\x00IRQN\xa0 \x93\\PMOD
\x00\xa0\r\\PDIRp
\x01PMDS\xa1\x08p
\x00PMDS\xa1\x1b\xa0\x10\x93\\PMOD
\x01p
\x02PMDS\xa1\x08p
\x03PMDSp
\x00PPD_p
\x01LDA_\xa0\x10\x93RLIO\x0bx\x03p
\x00XPA_\xa11\xa0\x10\x93RLIO\x0bx\x02p
\x01XPA_\xa1\x1e\xa0\x10\x93RLIO\x0b\xbc\x03p
\x02XPA_\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\xce\x01p
\x01XPE_\x14\x1b_PSC\x00p
\x01LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x01LDN_p
\x01LDA_\x14\x14_PS3\x00p
\x01LDN_p
\x00LDA_[\x82C2DLPT\x08_UID
\x01\x14D\x04_STA\x00\xa0\x1c\x92{
\x08\\/\x04_SB_PCI0LPC_DLPC\x00\xa4
\x00\xa1\x1f\xa0\x18\x92\x93\\PMOD
\x03\xa0\x08XPE_\xa4
\x0f\xa1\x04\xa4
\r\xa1\x04\xa4
\x00\x08_HID\x0cA\xd0\x04\x00\x14)_DIS\x00p
\x00XPE_p
\x01LDN_p
\x00IRQNp
\x00LDA_p
\x01PPD_\x08PPBF\x11\x10
\rG\x01\x00\x00\x00\x00\x01\x00"\x00\x00y\x00\x8bPPBF
\x02LPN0\x8bPPBF
\x04LPX0\x8cPPBF
\x07LPL0\x8bPPBF
\tLPIQ\x14@\x07_CRS\x00\xa0\x0e\x93\\PMOD
\x03\xa4PPBFp
\x01LDN_}yIOHI
\x08\x00IOLW`p`LPN0p`LPX0\xa0\r\x93`\x0b\xbc\x03p
\x03LPL0\xa1\x08p
\x08LPL0pIRQN`\xa0\r`y
\x01IRQNLPIQ\xa1\x08p
\x00LPIQ\xa4PPBF\x14\x19_PRS\x00\xa0\x0b\\PMOD\xa4PEPP\xa1\x06\xa4PLPT\x08PLPT\x11O\x04
K0G\x01\xbc\x03\xbc\x03\x01\x03"\x80\x000G\x01x\x03x\x03\x01\x08"\x80\x000G\x01x\x02x\x02\x01\x08" \x000G\x01\xbc\x03\xbc\x03\x01\x03" \x000G\x01x\x03x\x03\x01\x08" \x000G\x01x\x02x\x02\x01\x08"\x80\x008y\x00\x08PEPP\x116
30G\x01x\x03x\x03\x01\x08"\x80\x000G\x01x\x02x\x02\x01\x08" \x000G\x01x\x03x\x03\x01\x08" \x000G\x01x\x02x\x02\x01\x08"\x80\x008y\x00\x14H\x0f_SRS\x01\x8ch
\x02RLIL\x8ch
\x03RLIH\x8bh
\x02RLIO\x8bh
\tRLIQp
\x01LDN_p
\x00LDA_pRLILIOLWpRLIHIOHI\xa0\x12RLIQ\x82RLIQ`pv`IRQN\xa1\x08p
\x00IRQN\xa0 \x93\\PMOD
\x00\xa0\r\\PDIRp
\x01PMDS\xa1\x08p
\x00PMDS\xa1\x1b\xa0\x10\x93\\PMOD
\x01p
\x02PMDS\xa1\x08p
\x03PMDSp
\x00PPD_p
\x01LDA_\xa0\x10\x93RLIO\x0bx\x03p
\x00XPA_\xa11\xa0\x10\x93RLIO\x0bx\x02p
\x01XPA_\xa1\x1e\xa0\x10\x93RLIO\x0b\xbc\x03p
\x02XPA_\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\xce\x01p
\x01XPE_\x14\x1b_PSC\x00p
\x01LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x01LDN_p
\x01LDA_\x14\x14_PS3\x00p
\x01LDN_p
\x00LDA_[\x82E7LECP\x08_UID
\x00\x14<_STA\x00\xa0\x16\x93\\/\x03_SB_LCIO_STA
\x00\xa4
\x00\xa1\x1e\xa0\x17\x93\\PMOD
\x03\xa0\x08XPE_\xa4
\x0f\xa1\x04\xa4
\r\xa1\x04\xa4
\x00\x08_HID\x0cA\xd0\x04\x01\x14)_DIS\x00p
\x00XPE_p
\x01LDN_p
\x00IRQNp
\x00LDA_p
\x01PPD_\x08EPBF\x11\x1b
\x18G\x01\x00\x00\x00\x00\x01\x00G\x01\x00\x00\x00\x00\x01\x00"\x00\x00*\x00\x00y\x00\x8bEPBF
\x02ECN0\x8bEPBF
\x04ECX0\x8cEPBF
\x07ECL0\x8bEPBF

ECN1\x8bEPBF
\x0cECX1\x8cEPBF
\x0fECL1\x8bEPBF
\x11ECIQ\x8bEPBF
\x14ECDQ\x14N
_CRS\x00\xa0\x0f\x92\x93\\PMOD
\x03\xa4EPBFp
\x01LDN_}yIOHI
\x08\x00IOLW`p`ECN0p`ECX0r`\x0b\x00\x04ECN1r`\x0b\x00\x04ECX1\xa0\x14\x93`\x0b\xbc\x03p
\x03ECL0p
\x03ECL1\xa1\x0fp
\x08ECL0p
\x08ECL1pIRQN`\xa0\r`y
\x01IRQNECIQ\xa1\x08p
\x00ECIQpDMA0`\xa0\r\x95`
\x04y
\x01`ECDQ\xa1\x08p
\x00ECDQ\xa4EPBF\x08_PRS\x11A\t
\x8d0G\x01x\x03x\x03\x01\x08G\x01x\x07x\x07\x01\x08"\x80\x00*\x0b\x000G\x01x\x02x\x02\x01\x08G\x01x\x06x\x06\x01\x08" \x00*\x0b\x000G\x01\xbc\x03\xbc\x03\x01\x03G\x01\xbc\x07\xbc\x07\x01\x03"\x80\x00*\x0b\x000G\x01x\x03x\x03\x01\x08G\x01x\x07x\x07\x01\x08" \x00*\x0b\x000G\x01x\x02x\x02\x01\x08G\x01x\x06x\x06\x01\x08"\x80\x00*\x0b\x000G\x01\xbc\x03\xbc\x03\x01\x03G\x01\xbc\x07\xbc\x07\x01\x03" \x00*\x0b\x008y\x00\x14A\x0f_SRS\x01\x8ch
\x02RLIL\x8ch
\x03RLIH\x8bh
\x02RLIO\x8bh
\x11RLIQ\x8ch
\x14RLDQp
\x01LDN_p
\x00LDA_p
\x07PMDSp
\x01PERApRLILIOLWpRLIHIOHI\xa0\x12RLIQ\x82RLIQ`pv`IRQN\xa1\x08p
\x00IRQN\xa0\x16{RLDQ
\x0f\x00\x82RLDQ`pv`DMA0\xa1\x08p
\x04DMA0p
\x00PPD_p
\x01LDA_\xa0\x10\x93RLIO\x0bx\x03p
\x00XPA_\xa11\xa0\x10\x93RLIO\x0bx\x02p
\x01XPA_\xa1\x1e\xa0\x10\x93RLIO\x0b\xbc\x03p
\x02XPA_\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\x9f\x01p
\x01XPE_\x14\x1b_PSC\x00p
\x01LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x01LDN_p
\x01LDA_\x14\x14_PS3\x00p
\x01LDN_p
\x00LDA_[\x82L7DECP\x08_UID
\x01\x14C\x04_STA\x00\xa0\x1c\x92{
\x08\\/\x04_SB_PCI0LPC_DLPC\x00\xa4
\x00\xa1\x1e\xa0\x17\x93\\PMOD
\x03\xa0\x08XPE_\xa4
\x0f\xa1\x04\xa4
\r\xa1\x04\xa4
\x00\x08_HID\x0cA\xd0\x04\x01\x14)_DIS\x00p
\x00XPE_p
\x01LDN_p
\x00IRQNp
\x00LDA_p
\x01PPD_\x08EPBF\x11\x1b
\x18G\x01\x00\x00\x00\x00\x01\x00G\x01\x00\x00\x00\x00\x01\x00"\x00\x00*\x00\x00y\x00\x8bEPBF
\x02ECN0\x8bEPBF
\x04ECX0\x8cEPBF
\x07ECL0\x8bEPBF

ECN1\x8bEPBF
\x0cECX1\x8cEPBF
\x0fECL1\x8bEPBF
\x11ECIQ\x8bEPBF
\x14ECDQ\x14N
_CRS\x00\xa0\x0f\x92\x93\\PMOD
\x03\xa4EPBFp
\x01LDN_}yIOHI
\x08\x00IOLW`p`ECN0p`ECX0r`\x0b\x00\x04ECN1r`\x0b\x00\x04ECX1\xa0\x14\x93`\x0b\xbc\x03p
\x03ECL0p
\x03ECL1\xa1\x0fp
\x08ECL0p
\x08ECL1pIRQN`\xa0\r`y
\x01IRQNECIQ\xa1\x08p
\x00ECIQpDMA0`\xa0\r\x95`
\x04y
\x01`ECDQ\xa1\x08p
\x00ECDQ\xa4EPBF\x08_PRS\x11A\t
\x8d0G\x01x\x03x\x03\x01\x08G\x01x\x07x\x07\x01\x08"\x80\x00*\x0b\x000G\x01x\x02x\x02\x01\x08G\x01x\x06x\x06\x01\x08" \x00*\x0b\x000G\x01\xbc\x03\xbc\x03\x01\x03G\x01\xbc\x07\xbc\x07\x01\x03"\x80\x00*\x0b\x000G\x01x\x03x\x03\x01\x08G\x01x\x07x\x07\x01\x08" \x00*\x0b\x000G\x01x\x02x\x02\x01\x08G\x01x\x06x\x06\x01\x08"\x80\x00*\x0b\x000G\x01\xbc\x03\xbc\x03\x01\x03G\x01\xbc\x07\xbc\x07\x01\x03" \x00*\x0b\x008y\x00\x14A\x0f_SRS\x01\x8ch
\x02RLIL\x8ch
\x03RLIH\x8bh
\x02RLIO\x8bh
\x11RLIQ\x8ch
\x14RLDQp
\x01LDN_p
\x00LDA_p
\x07PMDSp
\x01PERApRLILIOLWpRLIHIOHI\xa0\x12RLIQ\x82RLIQ`pv`IRQN\xa1\x08p
\x00IRQN\xa0\x16{RLDQ
\x0f\x00\x82RLDQ`pv`DMA0\xa1\x08p
\x04DMA0p
\x00PPD_p
\x01LDA_\xa0\x10\x93RLIO\x0bx\x03p
\x00XPA_\xa11\xa0\x10\x93RLIO\x0bx\x02p
\x01XPA_\xa1\x1e\xa0\x10\x93RLIO\x0b\xbc\x03p
\x02XPA_\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\x9f\x01p
\x01XPE_\x14\x1b_PSC\x00p
\x01LDN_\xa0\x08LDA_\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x01LDN_p
\x01LDA_\x14\x14_PS3\x00p
\x01LDN_p
\x00LDA_\x10#\\/\x04_SB_PCI0LPC_LURT\x08_EJD\r_SB.LCIO\x00\x10#\\/\x04_SB_PCI0LPC_DURT\x08_EJD\r_SB.GDCK\x00\x10#\\/\x04_SB_PCI0LPC_LLPT\x08_EJD\r_SB.LCIO\x00\x10#\\/\x04_SB_PCI0LPC_DLPT\x08_EJD\r_SB.GDCK\x00\x10#\\/\x04_SB_PCI0LPC_LECP\x08_EJD\r_SB.LCIO\x00\x10#\\/\x04_SB_PCI0LPC_DECP\x08_EJD\r_SB.GDCK\x00[\x82H.FIR_\x08_HID\x0c$M\x00q\x08_CID\x0cA\xd0\x05\x11\x14\x1d_STA\x00\xa0\x08HDIR\xa4
\x00\xa0\x08XU2E\xa4
\x0f\xa1\x04\xa4
\r\x14)_DIS\x00p
\x00XU2Ep
\x02LDNSp
\x00RQNSp
\x00LDASp
\x01SP2S\x08U2BF\x11\x13
\x10G\x01\x00\x00\x00\x00\x01\x08"\x00\x00*\x00\x00y\x00\x8bU2BF
\x02IRMN\x8bU2BF
\x04IRMX\x8bU2BF
\tIRIQ\x8cU2BF
\x0cIRDQ\x14G\x06_CRS\x00p
\x02LDNS}yOHIS
\x08\x00OLWS`p`IRMNp`IRMXpRQNS`\xa0\r`y
\x01RQNSIRIQ\xa1\x08p
\x00IRIQpMA0S`\xa0\r\x95`
\x04y
\x01`IRDQ\xa1\x08p
\x00IRDQ\xa4U2BF\x08_PRS\x11G\x08
\x831\x00G\x01\xf8\x02\xf8\x02\x01\x08"\x08\x00*\x0b\x001\x01G\x01\xf8\x03\xf8\x03\x01\x08"\x10\x00*\x0b\x001\x01G\x01\xe8\x02\xe8\x02\x01\x08"\x08\x00*\x0b\x001\x01G\x01\xe8\x03\xe8\x03\x01\x08"\x10\x00*\x0b\x001\x02G\x01\xf8\x02\xf8\x02\x01\x08"\xb0\x00*\x0b\x001\x02G\x01\xf8\x03\xf8\x03\x01\x08"\xa8\x00*\x0b\x001\x02G\x01\xe8\x02\xe8\x02\x01\x08"\xb0\x00*\x0b\x001\x02G\x01\xe8\x03\xe8\x03\x01\x08"\xa8\x00*\x0b\x008y\x00\x14E\x10_SRS\x01\x8ch
\x02RIIL\x8ch
\x03RIIH\x8bh
\x02RIIO\x8bh
\tRIIQ\x8ch
\x0cRIDQp
\x02LDNSp
\x00LDASpRIILOLWSpRIIHOHIS\xa0\x12RIIQ\x82RIIQ`pv`RQNS\xa1\x08p
\x00RQNS\xa0\x16{RIDQ
\x0f\x00\x82RIDQ`pv`MA0S\xa1\x08p
\x04MA0Sp
\x04MA1Sp
\x01SESSp
\x00SP2Sp
\x01LDAS\xa0\x10\x93RIIO\x0b\xf8\x03p
\x00XU2A\xa1E\x04\xa0\x10\x93RIIO\x0b\xf8\x02p
\x01XU2A\xa11\xa0\x10\x93RIIO\x0b\xe8\x03p
\x07XU2A\xa1\x1e\xa0\x10\x93RIIO\x0b\xe8\x02p
\x05XU2A\xa1\x0b[2\x02\x00\x00\x02\x90\x0b\xa4\x01p
\x01XU2E\x14\x1b_PSC\x00p
\x02LDNS\xa0\x08LDAS\xa4
\x00\xa1\x04\xa4
\x03\x14\x14_PS0\x00p
\x02LDNSp
\x01LDAS\x14\x14_PS3\x00p
\x02LDNSp
\x00LDAS\x10C\x0e\\/\x03_SB_PCI0LPC_\x14A\rLCON\x01\xa0K
h\xa0G
EPWGp
\x01DLPDp
\x07DLPC\xa2\x0e\x92{
\x08DLPC\x00["
\x01p
\x01DRST\xa0L\x07\x92\x93SIOD
\xff}SCF9
\xa0SCF9\xa2\x0e\x92{
\x10SCF9\x00["
\x01}SCF2
\xebSCF2p
\x07LDN_{\x0b \x16
\xff`p`IOLWz\x0b \x16
\x08`{`
\xffIOHIp
\x01LDA_p
\x01GPPSp
\x03GPPCp
\x02GPPSp
\x03GPPCp
\x00DUSB\xa1\x1dp
\x01DUSBp
\x00DRSTp
\x00DLPDp
\x00DLPC[\x82GKTPM_\x08_HID\x0c\x06\x8d\x12\x00\x08_CID\x0cA\xd0\x0c1\x14\x1d_STA\x00\xa0\x0e{\\TPMP
\x01\x00p
\x0f`\xa1\x05p
\x00`\xa4`\x08_CRS\x11\x11
\x0e\x86\t\x00\x01\x00\x00\xd4\xfe\x00\x10\x00\x00y\x00\x14GF_DSM\x0c\x08TMPB\x11\x05
\x02\x00\x00\x8cTMPB
\x00LPCT\x8cTMPB
\x01SSUM\x08LRSP
\x00\x08PPRC
\x00\x08RQS1\x12\x06\x02
\x0c
\r\x08PTOP\x12\x06\x02
\x00
\x00\x08RTOP\x12\x08\x03
\x00
\x00
\x00\x8ah
\x00IID0\x8ah
\x04IID1\x8ah
\x08IID2\x8ah
\x0cIID3\x08UID0\x11\x13
\x10\xa6\xfa\xdd=\x1b6\xb4N\xa4$\x8d\x10\x08\x9d\x16S\x8aUID0
\x00EID0\x8aUID0
\x04EID1\x8aUID0
\x08EID2\x8aUID0
\x0cEID3\xa0I,\x90\x90\x93IID0EID0\x93IID1EID1\x90\x93IID2EID2\x93IID3EID3\xa0\x0b\x93j
\x00\xa4\x11\x04
\x01?\xa0\x0b\x93j
\x01\xa4\r1.0\x00\xa0L\x1c\x93j
\x02p
\x00PPRCTPHY
\x00p\\PH02LPCT\xa0@\x1aLPCTp
\x00SSUMrSSUM\\PH01SSUMrSSUM\\PH02SSUMrSSUM\\PH03SSUMrSSUM\\PH04SSUMrSSUM\\PH05SSUMrSSUM\\PH06SSUMrSSUM\\PH07SSUMrSSUM\\PH08SSUMrSSUM\\PH09SSUMrSSUM\\PH0ASSUMrSSUM\\PH0BSSUM\xa0\x0cSSUMp
\x02PPRC\xa1K\x0ep\x83\x88k
\x00\x00\\PH04p\x89RQS1\x01\\PH04\x00
\x00
\x00`\xa0\x0c\x92\x93`\xffp
\x01PPRC\xa1M\x0bp
\x00\\PH03p
\x00SSUMrSSUM\\PH01SSUMrSSUM\\PH02SSUMrSSUM\\PH03SSUMrSSUM\\PH04SSUMrSSUM\\PH05SSUMrSSUM\\PH06SSUMrSSUM\\PH07SSUMrSSUM\\PH08SSUMrSSUM\\PH09SSUMrSSUM\\PH0ASSUMrSSUM\\PH0BSSUMt
\x00SSUM\\PH03TPHY
\x01\xa1\x08p
\x02PPRC\xa4PPRC\xa0\x1e\x93j
\x03TPHY
\x00p\\PH04\x88PTOP
\x01\x00\xa4PTOP\xa0\x08\x93j
\x04\xa4
\x01\xa0I\x08\x93j
\x05TPHY
\x00p\\PH07`p\\PH06LRSPp}y`
\x08`LRSP\x00LRSP\xa0\x1c\x90\x93\\PH06
\xf0\x93\\PH07
\xffp\x0c\xf0\xff\xff\xffLRSP\xa1\x1e\xa0\x1c\x90\x93\\PH06
\xf1\x93\\PH07
\xffp\x0c\xf1\xff\xff\xffLRSPpLRSP\x88RTOP
\x02\x00p\\PH05\x88RTOP
\x01\x00\xa4RTOP\xa0\x08\x93j
\x06\xa4
\x02\x08UID1\x11\x13
\x10\xedT`7\x13\xccuF\x90\x1cGV\xd7\xf2\xd4]\x8aUID1
\x00EID4\x8aUID1
\x04EID5\x8aUID1
\x08EID6\x8aUID1
\x0cEID7\xa0@\t\x90\x90\x93IID0EID4\x93IID1EID5\x90\x93IID2EID6\x93IID3EID7pj[1pk[1\xa0\x0b\x93j
\x00\xa4\x11\x04
\x01\x01\xa0O\x04\x93j
\x01\xa0\x12\x93\x83\x88k
\x00\x00
\x00p
\x00\\TCG0\xa0\x12\x93\x83\x88k
\x00\x00
\x01p
\x01\\TCG0\xa0\x0f\x93\\TCG0
\x00TMOR
\x00\xa0\x0f\x93\\TCG0
\x01TMOR
\x01\xa4
\x00\xa4
\x01\xa4\x11\x04
\x01\x00[\x80TSES\x00\x0c\x00\x00\xe8\xff\x0b\x00\x02[\x81F\x08TSES\x01\x00@\x06TSCM\x08\x00@(TCT0\x08TCT1\x08TCT2\x08\x00@\x05TMT0\x08\x00H\x04TMT1\x08\x00H\x04TMT2\x08\x00HzTLD0\x08TLD1\x08TLD2\x08TLDL\x18TLDB\x18TLDS\x10TSB0\x08TSB1\x08TSB2\x08TSBL\x08TSBBH\x04TSBF\x10TSBC\x10\x00@\x1aTSIM\x08[\x81\x0fTSES\x01\x00H.TCTAH\x11[\x81\x0fTSES\x01\x00@\xbaTCTLH\x05[\x81\x0fTSES\x01\x00H\xbfTCTBH\x08\x148TCSZ\x02\xa01\x92\x93TSCM
\x12\xa0\x0b\x93
\x01hpiTCZ1\xa1\x1b\xa0\x0b\x93
\x02hpiTCZ2\xa1\r\xa0\x0b\x93
\x03hpiTCZ3\x14I\x0cTCST\x02\xa0A\x0c\x92\x93TSCM
\x12p
\x00`\xa0\x1e\x93
\x01h\xa0\x18\x94iTMT0p
\x04TSIMpiTMT0p
\x01`\xa1B\x04\xa0\x1e\x93
\x02h\xa0\x18\x94iTMT1p
\x04TSIMpiTMT1p
\x01`\xa1 \xa0\x1e\x93
\x03h\xa0\x18\x94iTMT2p
\x04TSIMpiTMT2p
\x01`\xa0@\x05`p
\x00TCT2\x08TCTC\x11\x03
#pTCTATCTCp
"cp
\x00a\xa2\x12cp\x83\x88TCTCc\x00brabavcp\x80a\x00bub{b
\xffapaTCT2p
\xfeTSIM\x14I\x08TCBS\x04\xa0A\x08\x92\x93TSCM
\x12\xa0F\x07\x93{h
\x07\x00
\x00p
\x12TSIMphTSBLpiTSBBpjTSBFpkTSBCp
\x00TSB2\x08TCTD\x11\x03
#pTCTBTCTDp
\x10`p
\x00a\xa2\x12`p\x83\x88TCTD`\x00brabav`p\x80a\x00bub{b
\xffapaTSB2p
\xfeTSIM\x14@\x07TSCL\x02\xa0H\x06\x92\x93TSCM
\x12p
\x15TSIMrTLDSiTLDSp
\x00TLD2\x08TLDD\x11\x03
\x0bpTCTLTLDDp

`p
\x00a\xa2\x12`p\x83\x88TLDD`\x00brabav`p\x80a\x00bub{b
\xffapaTLD2p
\xfeTSIM[\x82\x8e\x00\x02EC__\x08_HID\x0cA\xd0\x0c\t\x08_UID
\x00\x08_GPE
\x1c\x14\x13_REG\x02\xa0\x0c\x93h
\x03pi\\H8DR[\x80ECOR\x03
\x00\x0b\x00\x01[\x81J+ECOR\x01HDBM\x01\x00\x01\x00\x01HFNE\x01\x00\x01\x00\x01HLDM\x01\x00\x01\x00\x01BTCM\x01\x00\x01\x00\x01\x00\x01HBPR\x01BTPC\x01\x00\x01HDUE\x01\x00\x07\x00\x01HETE\x01\x00\x0eHSPA\x01\x00\x07HSUN\x08HSRP\x08\x00 HLCL\x08\x00\x08HFNS\x02\x00\x06\x00\x04HAAA\x03\x00\x01HAM0\x08HAM1\x08HAM2\x08HAM3\x08HAM4\x08HAM5\x08HAM6\x08HAM7\x08HAM8\x08HAM9\x08HAMA\x08HAMB\x08HAMC\x08HAMD\x08HAME\x08HAMF\x08HT00\x01HT01\x01HT02\x01HT03\x01HT10\x01HT11\x01HT12\x01HT13\x01\x00\x10HANT\x08\x00\x10\x00\x01\x00\x01HANA\x02\x00\x01\x00\x01\x00\x1aHATR\x08HT0H\x08HT0L\x08HT1H\x08HT1L\x08HFSP\x08\x00\x06HMUT\x01\x00\x01HBRV\x08HWPM\x01HWLB\x01HWLO\x01HWDK\x01HWFN\x01HWBT\x01HWRI\x01HWBU\x01HWLU\x01\x00\x07\x00\x07HPLO\x01\x00\x08\x00\x10HB0S\x07HB0A\x01HB1S\x07HB1A\x01HCMU\x01\x00\x02OVRQ\x01DCBD\x01DCWL\x01DCWW\x01HB1I\x01\x00\x01KBLT\x01BTPW\x01BTDT\x01HUBS\x01BDPW\x01BDDT\x01HUBB\x01\x00@\x05\x00\x01BTWK\x01HPLD\x01\x00\x01HPAC\x01BTST\x01\x00\x02HPBU\x01\x00\x01HBID\x04HBCS\x01HPNF\x01\x00\x01GSTS\x01\x00\x02HLBU\x01BDST\x01HCBL\x01\x00)HWAK\x10HMPR\x08HMST\x05\x00\x02HMDN\x01HMAD\x08HMCM\x08\x00@\x10HMBC\x08\x00\x18TMP0\x08TMP1\x08TMP2\x08TMP3\x08TMP4\x08TMP5\x08TMP6\x08TMP7\x08\x00\x08HIID\x08\x00\x08HFNI\x08\x00 HDEC\x08HDEO\x08\x00\x18HDAA\x03HDAB\x03HDAC\x02\x00@\x11HDEN HDEP HDEM\x08HDES\x08\x00@\x07ATMX\x08HWAT\x08\x00H\x11\x00\x04HDDD\x01\x14A\x04_INI\x00\xa0\r\\H8DRp
\x00HSPA\xa1\x0c\\MBEC
\x05
\xfe
\x00BINI\\/\x06_SB_PCI0LPC_EC__HKEYWGIN\x08_CRS\x11\x15
\x12G\x01b\x00b\x00\x01\x01G\x01f\x00f\x00\x01\x01y\x00\x14!LED_\x02}hi`\xa0\x0c\\H8DRp`HLCL\xa1\t\\WBEC
\x0c`\x08BAON
\x00\x08WBON
\x00\x14N\x18BEEP\x01\xa0\x0c\x93h
\x05p
\x00WBONpWBONb\xa0O\x04BAON\xa0$\x93h
\x00p
\x00BAON\xa0\rWBONp
\x03`p
\x08a\xa1\tp
\x00`p
\x00a\xa1#p
\xff`p
\xffa\xa0\x0c\x93h
\x11p
\x00WBON\xa0\x0c\x93h
\x10p
\x01WBON\xa1G\x04ph`p
\xffa\xa0\x13\x93h
\x0fph`p
\x08ap
\x01BAON\xa0\x14\x93h
\x11p
\x00`p
\x00ap
\x00WBON\xa0\x14\x93h
\x10p
\x03`p
\x08ap
\x01WBON\xa02\x93h
\x03p
\x00WBON\xa0%bp
\x07`\xa0\x1e\x91\x93\\SPS_
\x03\x93\\SPS_
\x04p
\x00bp
\xff`p
\xffa\xa0\x14\x93h
\x07\xa0\x0ebp
\x00bp
\xff`p
\xffa\xa0C\x04\x90\\H8DR\x92\\W98F\xa0\x1a\x90b\x92WBONp
\x00HSRPp
\x00HSUN["
d\xa0\x0c\x92\x93a
\xffpaHSRP\xa0\x0c\x92\x93`
\xffp`HSUN\xa1>\xa0\x1e\x90b\x92WBON\\WBEC
\x07
\x00\\WBEC
\x06
\x00["
d\xa0\x0e\x92\x93a
\xff\\WBEC
\x07a\xa0\x0e\x92\x93`
\xff\\WBEC
\x06`\xa0\x05\x93h
\x03\xa0
\x93h
\x07["\x0b\xf4\x01\x14C\tEVNT\x01\xa07\\H8DR\xa0\x18h}HAM7
\x01HAM7}HAM5
\x04HAM5\xa1\x17{HAM7
\xfeHAM7{HAM5
\xfbHAM5\xa1C\x05\xa0(h\\MBEC
\x17
\xff
\x01\\MBEC
\x15
\xff
\x04\xa0\x0f\\W98F\\WBEC
\x18
\xff\xa1\'\\MBEC
\x17
\xfe
\x00\\MBEC
\x15
\xfb
\x00\xa0\x0f\\W98F\\WBEC
\x18
\x00\x14K\x07PNST\x01\xa0=\x90hBSTA
\x02\xa0\x1b\x90\\H8DR\x92\\W98Fp
\x01HBPRp
\x01HUBB\xa1\x17\\MBEC
\x01
\xff
 \\MBEC
;
\xff
\x80\xa15\xa0\x1b\x90\\H8DR\x92\\W98Fp
\x00HBPRp
\x00HUBB\xa1\x17\\MBEC
\x01
\xdf
\x00\\MBEC
;
\x7f
\x00[\x84I\x07PUBS\x03\x00\x00\x14+_STA\x00\xa0\x0c\\H8DRpHUBS`\xa1\x0c{\\RBEC
;
\x10`\xa0\x05`\xa4
\x01\xa1\x04\xa4
\x00\x14!_ON_\x00\xa0\r\\H8DRp
\x01HUBS\xa1\x0c\\MBEC
;
\xff
\x10\x14!_OFF\x00\xa0\r\\H8DRp
\x00HUBS\xa1\x0c\\MBEC
;
\xef
\x00\x14E\x0fLPMD\x00p
\x00`p
\x00ap
\x00b\xa0B\x06\\H8DR\xa0J\x05HPLO\xa0J\x04HPAC\xa0C\x04\x95HWAT
Z\xa0\x16\x91\x92HB0A\x95{HB0S
\x0f\x00
\x03p
\x01a\xa0\x16\x91\x92HB1A\x95{HB1S
\x0f\x00
\x03p
\x01b\xa0\x0b\x90abp\\LPST`\xa1\x08p\\LPST`\xa1L\x07\xa0I\x07{\\RBEC
4
\x80\x00\xa0B\x06{\\RBEC
F
\x10\x00\xa0D\x05\x95\\RBEC
\xc9
Zp\\RBEC
8c\xa0\x14\x91\x92{c
\x80\x00\x95{c
\x0f\x00
\x03p
\x01ap\\RBEC
9c\xa0\x14\x91\x92{c
\x80\x00\x95{c
\x0f\x00
\x03p
\x01b\xa0\x0b\x90abp\\LPST`\xa1\x08p\\LPST`\xa4`[\x01MCPU\x07\x14D\x04_Q10\x00\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
\x01\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x01\x10\x14G\x04_Q11\x00\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
\x02\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x02\x10\xa1\x02\xa3\x14$_Q12\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x03\x10\x14A\x05_Q13\x00\xa0:\\/\x06_SB_PCI0LPC_EC__HKEYDHKC\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x04\x10\xa1\x0e\x86\\._SB_SLPB
\x80\x14M\x10_Q64\x00\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
\x10\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x05\x10\xa1H\x0c\xa0>\x92{\\/\x06_SB_PCI0LPC_EC__HKEYWGFL
 \x00\\/\x06_SB_PCI0LPC_EC__HKEYBPWC
\x01\xa1F\x08\\/\x06_SB_PCI0LPC_EC__HKEYBPWC
\x00\xa0F\x06\x92\\WVIS\xa0>\x92{\\/\x06_SB_PCI0LPC_EC__HKEYWGFL
\x02\x00\\/\x06_SB_PCI0LPC_EC__HKEYWPWC
\x01\xa1\x1e\\/\x06_SB_PCI0LPC_EC__HKEYWPWC
\x00\x14D\x04_Q65\x00\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
 \\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x06\x10\x14C
_Q16\x00\xa0\'\\VPDFp
\x00\\VPDF\xa0\x18VIGD\\/\x04_SB_PCI0VID_VSPD\xa1C\x07\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
@\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x07\x10\xa13\xa0\x18VIGD\\/\x04_SB_PCI0VID_VSWT\xa1\x18\\/\x05_SB_PCI0AGP_VID_VSWT\x14B\x05_Q17\x00\xa0<\\/\x06_SB_PCI0LPC_EC__HKEYMHKK
\x80\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x08\x10\xa1\r\xa0\x0b\x92\\WNTFVEXP\x14F\x04_Q18\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00\x01\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\t\x10\xa3\x14E\x04_Q66\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00\x02\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b
\x10\x14E\x04_Q1A\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00\x04\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x0b\x10\x14$_Q1B\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x0c\x10\x14E\x04_Q62\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00\x10\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\r\x10\x14E\x04_Q60\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00 \\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x0e\x10\x14E\x04_Q61\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00@\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x0f\x10\x14N\x04_Q1F\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x02\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x12\x10\\UCMS
\x0e\x14G\x04_Q67\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x04\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x13\x10\x14O\x0e_Q26\x00\\UCMS
\x12DSPD
\x00["\x0b\xf4\x01\x86AC__
\x80\x86\\._TZ_THM0
\x80\x86\\._TZ_THM1
\x80\xa0\x0e\\WXPF[#MCPU\xff\xff\xa07\\SPEN\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\xa1\x08\\STEP
\x00\xa0
\\WXPF["
d\xa0\'\\OSC4\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81\xa0\x0c\\WXPF[\'MCPU\xa0#\x90\x92\\WXPF\\WNTFp
\x00\\/\x04_SB_PCI0LPC_C4C3ATMCDSPD
\x01\x14K\x12_Q27\x00\\UCMS
\x12["\x0b\xf4\x01\x86AC__
\x80\x86\\._TZ_THM0
\x80\x86\\._TZ_THM1
\x80\xa0\x0e\\WXPF[#MCPU\xff\xff\xa07\\SPEN\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\xa1\x08\\STEP
\x01\xa0
\\WXPF["
d\xa0\'\\OSC4\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81\xa0\x0c\\WXPF[\'MCPU\xa01\x90\\CWAC\x92\\CWAS\xa0#\x90\x92\\WXPF\\WNTFp
\x01\\/\x04_SB_PCI0LPC_C4C3ATMC\xa09\x91\x93\\/\x03_SB_GDCKGDID\x0c$M\x00L\x93\\/\x03_SB_GDCKGDID\x0c$M\x00D\x86\\._SB_GDCK
\x01\x14H\x06_Q2A\x00\xa0\x1aVIGD\\/\x04_SB_PCI0VID_VLOC
\x01\xa1\x1a\\/\x05_SB_PCI0AGP_VID_VLOC
\x01\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x02P\x86\\._SB_LID_
\x80\x148_Q2B\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x01P\\UCMS
\r\x86\\._SB_LID_
\x80\x14\x06_Q3D\x00\x14>_Q48\x00\xa07\\SPEN\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\xa1\x08\\STEP
\x04\x14>_Q49\x00\xa07\\SPEN\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\xa1\x08\\STEP
\x05\x14\x10_Q7F\x00[2\x01\x00\x00\x01\x80\x0b+\x03\x14$_Q4E\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x11`\x14$_Q4F\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x12`\x14\x1d_Q75\x00\\/\x05_SB_PCI0LPC_EC__TATR\x14$_Q46\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x12`\x14 _Q22\x00\xa0\x0cHB0A\x86BAT0
\x80\xa0\x0cHB1A\x86BAT1
\x80\x14\r_Q4A\x00\x86BAT0
\x81\x14\r_Q4B\x00\x86BAT0
\x80\x14
_Q4C\x00_Q38\x14%_Q4D\x00\xa0\x1e{^.BAT1B1ST^.BAT1XB1S\x00\x86BAT1
\x80\x14\r_Q24\x00\x86BAT0
\x80\x14%_Q25\x00\xa0\x1e{^.BAT1B1ST^.BAT1XB1S\x00\x86BAT1
\x80[\x811ECOR\x01\x00@PSBRC\x10SBFC\x10SBAE\x10SBRS\x10SBAC\x10SBVO\x10SBAF\x10SBBS\x10[\x81\x1aECOR\x01\x00@P\x00\x0fSBCM\x01SBMD\x10SBCC\x10[\x81\'ECOR\x01\x00@PSBDC\x10SBDV\x10SBOM\x10SBSI\x10SBDT\x10SBSN\x10[\x81\x0eECOR\x01\x00@PSBCH [\x81\x0fECOR\x01\x00@PSBMN@\x08[\x81\x0fECOR\x01\x00@PSBDN@\x08[\x01BATM\x07\x14C\x16GBIF\x03[#BATM\xff\xff\xa0C\x12j}h
\x01HIIDpSBCMg\x7fg
\x01\x88i
\x00\x00phHIID\xa0
gwSBFC

a\xa1\x07pSBFCapa\x88i
\x02\x00}h
\x02HIID\xa0
gwSBDC

`\xa1\x07pSBDC`p`\x88i
\x01\x00xa
\x14b\x88i
\x05\x00\xa0
gp
\xc8\x88i
\x06\x00\xa1!\xa0\x15SBDVx\x0c@\r\x03\x00SBDVb\x88i
\x06\x00\xa1\tp
\x00\x88i
\x06\x00pSBDV\x88i
\x04\x00pSBSN`\x08SERN\x11\t
\x06     \x00p
\x04b\xa2\x15`x`

a`ra
0\x88SERNb\x00vbpSERN\x88i

\x00}h
\x06HIIDpSBDN\x88i
\t\x00}h
\x04HIID\x08BTYP\x11\x08
\x05\x00\x00\x00\x00\x00pSBCHBTYPpBTYP\x88i
\x0b\x00}h
\x05HIIDpSBMN\x88i
\x0c\x00\xa1\'p\x0c\xff\xff\xff\xff\x88i
\x01\x00p
\x00\x88i
\x05\x00p
\x00\x88i
\x06\x00p\x0c\xff\xff\xff\xff\x88i
\x02\x00[\'BATM\xa4i\x14J\rGBST\x04[#BATM\xff\xff\xa0
{i
 \x00p
\x02`\xa1\x12\xa0
{i
@\x00p
\x01`\xa1\x05p
\x00`\xa0\x06{i
\x0f\x00\xa1\x06}`
\x04`\xa0\x19\x93{i
\x0f\x00
\x0fp
\x04`p
\x00ap
\x00bp
\x00c\xa1@\x06phHIIDpSBVOc\xa0
jwSBRC

b\xa1\x07pSBRCbpSBACa\xa0\x1c\x92\x95a\x0b\x00\x80\xa0\x0e{`
\x01\x00t\x0c\x00\x00\x01\x00aa\xa1\x05p
\x00a\xa1\r\xa0\x0b\x92{`
\x02\x00p
\x00a\xa0\rjwcaaxa\x0b\xe8\x03gap`\x88k
\x00\x00pa\x88k
\x01\x00pb\x88k
\x02\x00pc\x88k
\x03\x00[\'BATM\xa4k[\x82G\x15BAT0\x08_HID\x0cA\xd0\x0c
\x08_UID
\x00\x08_PCL\x12\x07\x01\\_SB_\x08B0ST
\x00\x08BT0I\x12#\r
\x00\x0c\xff\xff\xff\xff\x0c\xff\xff\xff\xff
\x01\x0b0*
\x00
\x00
\x01
\x01\r\x00\r\x00\r\x00\r\x00\x08BT0P\x12\x02\x04\x14D\x04_STA\x00\xa0\x0f\\H8DRpHB0AB0ST\xa1\x1e\xa0\x13{\\RBEC
8
\x80\x00p
\x01B0ST\xa1\x08p
\x00B0ST\xa0\x08B0ST\xa4
\x1f\xa1\x04\xa4
\x0f\x14G\x04_BIF\x00p
\x00gp

f\xa2+\x90\x92gf\xa0\x1fHB0A\xa0\x13\x93{HB0S
\x0f\x00
\x0f["\x0b\xe8\x03vf\xa1\x05p
\x01g\xa1\x05p
\x00f\xa4GBIF
\x00BT0Ig\x14#_BST\x00\x7f\x83\x88BT0I
\x00\x00
\x01`\xa4GBST
\x00HB0S`BT0P\x14I\x04_BTP\x01{HAM4
\xefHAM4\xa06hpha\xa0\x11\x92\x83\x88BT0I
\x00\x00xa

`a{a
\xffHT0L{za
\x08\x00
\xffHT0H}HAM4
\x10HAM4[\x82A\x18BAT1\x08_HID\x0cA\xd0\x0c
\x08_UID
\x01\x08_PCL\x12\x07\x01\\_SB_\x08B1ST
\x00\x08XB1S
\x01\x08BT1I\x12#\r
\x00\x0c\xff\xff\xff\xff\x0c\xff\xff\xff\xff
\x01\x0b0*
\x00
\x00
\x01
\x01\r\x00\r\x00\r\x00\r\x00\x08BT1P\x12\x02\x04\x14G\x06_STA\x00\xa0\x0f\\H8DRpHB1AB1ST\xa1\x1e\xa0\x13{\\RBEC
9
\x80\x00p
\x01B1ST\xa1\x08p
\x00B1ST\xa0\x1fB1ST\xa0\x08XB1S\xa4
\x1f\xa1\x10\xa0\t\\WNTF\xa4
\x00\xa1\x04\xa4
\x1f\xa1\x10\xa0\t\\WNTF\xa4
\x00\xa1\x04\xa4
\x0f\x14G\x04_BIF\x00p
\x00gp

f\xa2+\x90\x92gf\xa0\x1fHB1A\xa0\x13\x93{HB1S
\x0f\x00
\x0f["\x0b\xe8\x03vf\xa1\x05p
\x01g\xa1\x05p
\x00f\xa4GBIF
\x10BT1Ig\x14#_BST\x00\x7f\x83\x88BT1I
\x00\x00
\x01`\xa4GBST
\x10HB1S`BT1P\x14I\x04_BTP\x01{HAM4
\xdfHAM4\xa06hpha\xa0\x11\x92\x83\x88BT1I
\x00\x00xa

`a{a
\xffHT1L{za
\x08\x00
\xffHT1H}HAM4
 HAM4[\x82>AC__\x08_HID\rACPI0003\x00\x08_UID
\x00\x08_PCL\x12\x07\x01\\_SB_\x14\x0b_PSR\x00\xa4HPAC\x14\t_STA\x00\xa4
\x0f[\x82IQHKEY\x08_HID\x0c$M\x00h\x14\t_STA\x00\xa4
\x0f\x14
MHKV\x00\xa4\x0b\x00\x01\x08DHKC
\x00\x08DHKB
\x01[\x01XDHK\x07\x08DHKH
\x00\x08DHKW
\x00\x08DHKS
\x00\x08DHKD
\x00\x08DHKN\x0b\x0c\x08\x08DHKT
\x00\x08DHWW
\x00\x14\x0cMHKA\x00\xa4\x0c\xff\xff\xff\x00\x14\x0bMHKN\x00\xa4DHKN\x14\x18MHKK\x01\xa0\rDHKC\xa4{DHKNh\x00\xa1\x03\xa4\x00\x14@\x05MHKM\x02[#XDHK\xff\xff\xa0\x06\x94h
 \xa3\xa13y\x01vh`\xa0){`\x0c\xff\xff\xff\x00\x00\xa0\x0ci}`DHKNDHKN\xa1\x12{DHKN\x7f`\x0c\xff\xff\xff\xff\x00DHKN\xa1\x02\xa3[\'XDHK\x14\x13MHKS\x00\x86\\._SB_SLPB
\x80\x14\x0cMHKC\x01phDHKC\x14I\x08MHKP\x00[#XDHK\xff\xff\xa0\x11DHWWpDHWWap\x00DHWW\xa1O\x05\xa0\x11DHKWpDHKWap\x00DHKW\xa1J\x04\xa0\x11DHKDpDHKDap\x00DHKD\xa15\xa0\x11DHKSpDHKSap\x00DHKS\xa1!\xa0\x11DHKTpDHKTap\x00DHKT\xa1\rpDHKHap\x00DHKH[\'XDHK\xa4a\x14>MHKE\x01phDHKB[#XDHK\xff\xffp\x00DHKHp\x00DHKWp\x00DHKSp\x00DHKDp\x00DHKTp\x00DHWW[\'XDHK\x14E\x0bMHKQ\x01\xa0M
DHKB\xa0@\tDHKC[#XDHK\xff\xff\xa0\x06\x95h\x0b\x00\x10\xa1M\x06\xa0\x0c\x95h\x0b\x00 phDHKH\xa1M\x05\xa0\x0c\x95h\x0b\x000phDHKW\xa1M\x04\xa0\x0c\x95h\x0b\x00@phDHKS\xa1=\xa0\x0c\x95h\x0b\x00PphDHKD\xa1.\xa0\x0c\x95h\x0b\x00`phDHKH\xa1\x1f\xa0\x0c\x95h\x0b\x00pphDHKT\xa1\x10\xa0\x0c\x95h\x0b\x00\x80phDHWW\xa1\x01[\'XDHK\x86HKEY
\x80\xa1\x15\xa0\x13\x93h\x0b\x04\x10\x86\\._SB_SLPB
\x80\x14I\x05MHKB\x01\xa0&\x93h
\x00\\/\x05_SB_PCI0LPC_EC__BEEP
\x11p
\x00\\LIDB\xa1*\xa0&\x93h
\x01\\/\x05_SB_PCI0LPC_EC__BEEP
\x10p
\x01\\LIDB\xa1\x01\x14<MHKD\x00\xa0\x1aVIGD\\/\x04_SB_PCI0VID_VLOC
\x00\xa1\x1a\\/\x05_SB_PCI0AGP_VID_VLOC
\x00\x14>MHQC\x01\xa01\\WNTF\xa0\x0b\x93h
\x00\xa4\\CWAC\xa1\x1e\xa0\x0b\x93h
\x01\xa4\\CWAP\xa1\x10\xa0\x0b\x93h
\x02\xa4\\CWAT\xa1\x02\xa3\xa1\x02\xa3\xa4
\x00\x14A\x08MHGC\x00\xa0C\x07\\WNTF[#XDHK\xff\xff\xa0:\\OSC4\xa0 \\/\x06_SB_PCI0LPC_EC__AC___PSRp
\x03`\xa1\x12\xa0
\\C4NAp
\x03`\xa1\x05p
\x04`\xa1 \xa0\x18\\/\x04_SB_PCI0LPC_C4C3p
\x04`\xa1\x05p
\x03`[\'XDHK\xa4`\xa1\x02\xa3\xa4
\x00\x14A\x13MHSC\x01\xa0F\x12\x90\\CWAC\\WNTF[#XDHK\xff\xff\xa0J\x08\\OSC4\xa0>\x93h
\x03\xa08\x92\\CWAS\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81p
\x01\\CWASp
\x01\\C4NA\xa1C\x04\xa0=\x93h
\x04\xa07\\CWAS\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81p
\x00\\CWASp
\x00\\C4NA\xa1\x02\xa3\xa1O\x07\xa0+\x93h
\x03\xa0%\x92\\CWASp
\x00\\/\x04_SB_PCI0LPC_C4C3p
\x01\\CWAS\xa1@\x05\xa0J\x04\x93h
\x04\xa0C\x04\\CWAS\xa03\x92\\/\x06_SB_PCI0LPC_EC__AC___PSRp
\x01\\/\x04_SB_PCI0LPC_C4C3p
\x00\\CWAS\xa1\x02\xa3[\'XDHK\xa1\x02\xa3\x10G\x14\\/\x05_SB_PCI0LPC_EC__HKEY\x14M\x04TDSC\x01p
\x01`\xa2\x1b`p\\/\x05_SB_PCI0LPC_EC__HANT`ph\\/\x05_SB_PCI0LPC_EC__HANT\xa0\x0c\x93h
!\\ATCC
\x00\x14\x0cTDSS\x01\\ATCCh\x14N\x08TDSG\x01{h
\xff`p`\\/\x05_SB_PCI0LPC_EC__HDAAzh
\x08`{`
\xffapa\\/\x05_SB_PCI0LPC_EC__HDABzh
\x10`{`
\xffapa\\/\x05_SB_PCI0LPC_EC__HDACzh
\x18`{`
\xffapa\\/\x05_SB_PCI0LPC_EC__HANA\x14!TDGC\x00p\\/\x05_SB_PCI0LPC_EC__HDDD`\xa4`\x14!TDGS\x00p\\/\x05_SB_PCI0LPC_EC__HAAA`\xa4`[\x81\x18ECOR\x11\x00@*HSD0\x08HSD1\x08HSD2\x08[\x81\x0eECOR\x11\x00@+HSDL\x10[\x81\x19ECOR\x11\x00@+HSDBH\x04HSDF\x10HSDC\x10[\x01I2CM\x07\x14?CHKS\x00p\x0b\xe8\x03`\xa2\x13HMPR["
\x01v`\xa0\x07\x92`\xa4\x0b\x80\x80\xa0\x19HMDN\xa0\x0fHMST\xa4}\x0b\x00\x80HMST\x00\xa1\x03\xa4\x00\xa1\x05\xa4\x0b\x81\x80\x14 I2RT\x00p
\x02HMADp
!HMCMp
\x0bHMPR\xa4CHKS\x14&I2NT\x01p
\x02HMADp
"HMCMphHSD0p
\x06HMPR\xa4CHKS\x14K\x0eTATR\x00\xa0C\x0e\\H8DR[#I2CM\xff\xffI2RTpHSD0`\xa0%\x93`
 \\/\x04_SB_PCI0LPC_TCSZHSD1HSD2I2NT`\xa1M\t\xa0%\x93`
!\\/\x04_SB_PCI0LPC_TCSTHSD1HSD2I2NT`\xa1D\x07\xa0I\x04\x93`
"\x08TSDC\x11\x0c
\t        \x00pHSDBTSDC\\/\x04_SB_PCI0LPC_TCBSHSD1TSDCHSDFHSDCI2NT`\xa1\'\xa0%\x93`
#\\/\x04_SB_PCI0LPC_TSCLHSD1HSDLI2NT`[\'I2CM\x14\rTATE\x00p
\x01HETE\x08_ADR
\x00\x08_S3D
\x02\x08RID_
\x00\x08LRRT\x12G\x13\x0e\x12\x15\x04\x0c\xff\xff\x01\x00
\x00\\._SB_LNKA
\x00\x12\x15\x04\x0c\xff\xff\x02\x00
\x00\\._SB_LNKA
\x00\x12\x15\x04\x0c\xff\xff\x1b\x00
\x01\\._SB_LNKB
\x00\x12\x15\x04\x0c\xff\xff\x1c\x00
\x00\\._SB_LNKE
\x00\x12\x15\x04\x0c\xff\xff\x1c\x00
\x01\\._SB_LNKF
\x00\x12\x15\x04\x0c\xff\xff\x1c\x00
\x02\\._SB_LNKG
\x00\x12\x15\x04\x0c\xff\xff\x1c\x00
\x03\\._SB_LNKH
\x00\x12\x15\x04\x0c\xff\xff\x1d\x00
\x00\\._SB_LNKA
\x00\x12\x15\x04\x0c\xff\xff\x1d\x00
\x01\\._SB_LNKB
\x00\x12\x15\x04\x0c\xff\xff\x1d\x00
\x02\\._SB_LNKC
\x00\x12\x15\x04\x0c\xff\xff\x1d\x00
\x03\\._SB_LNKD
\x00\x12\x15\x04\x0c\xff\xff\x1f\x00
\x00\\._SB_LNKH
\x00\x12\x15\x04\x0c\xff\xff\x1f\x00
\x02\\._SB_LNKA
\x00\x12\x15\x04\x0c\xff\xff\x1f\x00
\x01\\._SB_LNKA
\x00\x08ARRT\x12G\x0c\x0e\x12\r\x04\x0c\xff\xff\x01\x00
\x00
\x00
\x10\x12\r\x04\x0c\xff\xff\x02\x00
\x00
\x00
\x10\x12\r\x04\x0c\xff\xff\x1b\x00
\x01
\x00
\x11\x12\r\x04\x0c\xff\xff\x1c\x00
\x00
\x00
\x14\x12\r\x04\x0c\xff\xff\x1c\x00
\x01
\x00
\x15\x12\r\x04\x0c\xff\xff\x1c\x00
\x02
\x00
\x16\x12\r\x04\x0c\xff\xff\x1c\x00
\x03
\x00
\x17\x12\r\x04\x0c\xff\xff\x1d\x00
\x00
\x00
\x10\x12\r\x04\x0c\xff\xff\x1d\x00
\x01
\x00
\x11\x12\r\x04\x0c\xff\xff\x1d\x00
\x02
\x00
\x12\x12\r\x04\x0c\xff\xff\x1d\x00
\x03
\x00
\x13\x12\r\x04\x0c\xff\xff\x1f\x00
\x00
\x00
\x17\x12\r\x04\x0c\xff\xff\x1f\x00
\x02
\x00
\x10\x12\r\x04\x0c\xff\xff\x1f\x00
\x01
\x00
\x10\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4ARRT\xa1\x06\xa4LRRT\x08_HID\x0cA\xd0
\x08\x08_CID\x0cA\xd0
\x03\x08_BBN
\x00[\x80MHCS\x02
\x00\x0b\x00\x01[\x815MHCS\x03\x00@HPAM0\x08PAM1\x08PAM2\x08PAM3\x08PAM4\x08PAM5\x08PAM6\x08\x00(\x00\x03TOUD\x05\x08_CRS\x11E\x1c\x0b\xc0\x01\x88\r\x00\x02\x0c\x00\x00\x00\x00\x00\xff\x00\x00\x00\x00\x01G\x01\xf8\x0c\xf8\x0c\x01\x08\x88\r\x00\x01\x0c\x03\x00\x00\x00\x00\xf7\x0c\x00\x00\xf8\x0c\x88\r\x00\x01\x0c\x03\x00\x00\x00\r\xff\xff\x00\x00\x00\xf3\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00
\x00\xff\xff\x0b\x00\x00\x00\x00\x00\x00\x00\x02\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00\x0c\x00\xff?\x0c\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00@\x0c\x00\xff\x7f\x0c\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x80\x0c\x00\xff\xbf\x0c\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\xc0\x0c\x00\xff\xff\x0c\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00\r\x00\xff?\r\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00@\r\x00\xff\x7f\r\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x80\r\x00\xff\xbf\r\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\xc0\r\x00\xff\xff\r\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00\x0e\x00\xff?\x0e\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00@\x0e\x00\xff\x7f\x0e\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x80\x0e\x00\xff\xbf\x0e\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\xc0\x0e\x00\xff\xff\x0e\x00\x00\x00\x00\x00\x00@\x00\x00\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00\x10\x00\xff\xff\xbf\xfe\x00\x00\x00\x00\x00\x00\xb0\xfe\x87\x17\x00\x00\x0c\x03\x00\x00\x00\x00\x00\x00\xd4\xfe\xff\x0f\xd4\xfe\x00\x00\x00\x00\x00\x10\x00\x00y\x00\x8a_CRS
hC0LN\x8a_CRS
\x82C4LN\x8a_CRS
\x9cC8LN\x8a_CRS
\xb6CCLN\x8a_CRS
\xd0D0LN\x8a_CRS
\xeaD4LN\x8a_CRS\x0b\x04\x01D8LN\x8a_CRS\x0b\x1e\x01DCLN\x8a_CRS\x0b8\x01E0LN\x8a_CRS\x0bR\x01E4LN\x8a_CRS\x0bl\x01E8LN\x8a_CRS\x0b\x86\x01ECLN\x8a_CRS\x0b\x94\x01XXMN\x8a_CRS\x0b\x98\x01XXMX\x8a_CRS\x0b\xa0\x01XXLN\x8a_CRS\x0b\xae\x01F4MN\x8a_CRS\x0b\xb2\x01F4MX\x8a_CRS\x0b\xba\x01F4LN\x14A\x12_INI\x08\xa0\x11\x92\\OSIF\\._SB__INIyTOUD
\x1b`p`\\MEMXp`XXMNrtXXMXXXMN\x00
\x01XXLN\xa0\x15\x92\x93{\\TPMP
\x01\x00
\x01p
\x00F4LN\xa0\x10{PAM1
\x03\x00p
\x00C0LN\xa0\x10{PAM1
0\x00p
\x00C4LN\xa0\x10{PAM2
\x03\x00p
\x00C8LN\xa0\x10{PAM2
0\x00p
\x00CCLN\xa0\x10{PAM3
\x03\x00p
\x00D0LN\xa0\x10{PAM3
0\x00p
\x00D4LN\xa0\x10{PAM4
\x03\x00p
\x00D8LN\xa0\x10{PAM4
0\x00p
\x00DCLN\xa0\x10{PAM5
\x03\x00p
\x00E0LN\xa0\x10{PAM5
0\x00p
\x00E4LN\xa0\x10{PAM6
\x03\x00p
\x00E8LN\xa0\x10{PAM6
0\x00p
\x00ECLN\x08SUPP
\x00\x08CTRL
\x00\x14E,_OSC\x04\x8ak
\x00CDW1\x8ak
\x04CDW2\x8ak
\x08CDW3\x8ah
\x00IID0\x8ah
\x04IID1\x8ah
\x08IID2\x8ah
\x0cIID3\x08UID0\x11\x13
\x10[M\xdb3\xf7\x1f\x1c@\x96WtA\xc0=\xd7f\x8aUID0
\x00EID0\x8aUID0
\x04EID1\x8aUID0
\x08EID2\x8aUID0
\x0cEID3\xa0A#\x90\x90\x93IID0EID0\x93IID1EID1\x90\x93IID2EID2\x93IID3EID3pCDW2SUPPpCDW3CTRL{CTRL
\x1dCTRL\xa0H\x1b\x80{CDW1
\x01\x00\x00\xa0F\x0f{CTRL
\x01\x00\xa03\x92\\VIGDp
\x00\\/\x04_SB_PCI0AGP_HPGPp
\x00\\/\x04_SB_PCI0AGP_GMGPp
\x00\\/\x04_SB_PCI0EXP0HPCEp
\x01\\/\x04_SB_PCI0EXP0HPCSp
\x01\\/\x04_SB_PCI0EXP0ABP_p
\x01\\/\x04_SB_PCI0EXP0PDS_p
\x00\\/\x04_SB_PCI0EXP2HPCEp
\x01\\/\x04_SB_PCI0EXP2HPCSp
\x01\\/\x04_SB_PCI0EXP2ABP_p
\x01\\/\x04_SB_PCI0EXP2PDS_p
\x01\\NHPS\xa0D\x0b{CTRL
\x04\x00\xa03\x92\\VIGDp
\x00\\/\x04_SB_PCI0AGP_PMGPp
\x00\\/\x04_SB_PCI0AGP_GMGPp
\x00\\/\x04_SB_PCI0EXP0PMCEp
\x01\\/\x04_SB_PCI0EXP0PMCSp
\x00\\/\x04_SB_PCI0EXP2PMCEp
\x01\\/\x04_SB_PCI0EXP2PMCSp
\x00\\/\x04_SB_PCI0LPC_EXPEp
\x01\\NPME\xa0\x11\x92\x93i
\x01}CDW1

CDW1\xa0\x16\x92\x93CDW3CTRL}CDW1
\x10CDW1pCTRLCDW3\xa1\x0c}CDW1
\x06CDW1\xa4k[\x01MDGS\x07\x08VDEE
\x01\x08VDDA\x11\x03
\x02\x8dVDDA
\x00VUPC\x8dVDDA
\x01VQDL\x8dVDDA
\x02VQDC\x8dVDDA
\x03VQDT\x8dVDDA
\x04VQDD\x8dVDDA
\x05VSDL\x8dVDDA
\x06VSDC\x8dVDDA
\x07VSDT\x8dVDDA
\x08VSDD\x8dVDDA

MSWT\x8dVDDA
\x0bVWST[\x82OJVID_\x08_ADR\x0c\x00\x00\x02\x00\x08RID_
\x00[\x80VPCG\x02
\x00\x0b\x00\x01[\x81\x0eVPCG\x03\x00@jVPWR \x08_S3D
\x03\x145_INI\x00\\VUPS
\x02p\\VCDLVQDLp\\VCDCVQDCp\\VCDTVQDTp\\VCDDVQDD\x14\x07_PS0\x00\xa3\x14\x07_PS1\x00\xa3\x14\x07_PS2\x00\xa3\x14\x07_PS3\x00\xa3\x140VSWT\x00\xa0\x0f\\WVISp\\VEVT
\x07`\xa1
p\\VEVT
\x05`{
\x0f`a\xa0\taASWTa
\x01\x14E\x08VLOC\x01\xa0M\x07\x93h\\/\x03_SB_LID__LID\\VSLDh\xa0C\x06\x93VPWR
\x00\xa0?hp\\VEVT
\x01`\xa0\x13\\WXPF\x86\\._SB_PCI0
\x00\xa1\x1a\xa0\x18\\WNTF\x86\\/\x03_SB_PCI0VID_
\x00["\x0b\xee\x02\xa1
p\\VEVT
\x02`{
\x0f`a\xa0\taASWTa
\x00\x14G\x07_DOS\x01\xa0:\x93h
\x02p
\x14`\xa20`v`[#MDGS\xff\xff\xa0\x19\x93
\x00MSWTp
\x01MSWTp
\x00`phVDEE[\'MDGS["
\xc8\xa14[#MDGS\xff\xff\xa0\x0f\x93VDEE
\x02p
\x00MSWT\xa0\x0c\x94h
\x02p
\x01VDEE\xa1\x07phVDEE[\'MDGS\x14\x13_DOD\x00\xa4\x12\x0b\x03\x0b\x00\x01\x0b\x00\x03\x0b\x00\x04\x14L\x0bASWT\x02\xa0\x14\x93
\x01VDEE{
\x01ia\\VSDSha\xa1O\tp
\x14`\xa2A\x08`v`[#MDGS\xff\xff\xa0I\x06\x93
\x00MSWTp
\x00`\xa0\r{
\x01i\x00p
\x01VUPC\xa1\x08p
\x00VUPC\xa0\r{
\x01h\x00p
\x01VQDL\xa1\x08p
\x00VQDL\xa0\r{
\x02h\x00p
\x01VQDC\xa1\x08p
\x00VQDC\xa0\r{
\x08h\x00p
\x01VQDD\xa1\x08p
\x00VQDD[\'MDGS["
\xc8\xa0\r{
\x02i\x00\x86VID_
\x81\xa1\x08\x86VID_
\x80\x14I\x06VDSW\x01\xa0A\x06\x93VPWR
\x00\xa0D\x04hp\\VEVT
\x03`\xa0\x15\\WVIS{
\x0f`a\xa0\taASWTa
\x00\xa1!\xa0\x07\\WXPF\xa3\xa1\x17\xa0\x15\\WNTF{
\x0f`a\xa0\taASWTa
\x00\xa1\x12p\\VEVT
\x04`ASWT
\x01
\x00\x14\x1eVSPD\x00p\\VEVT
\x06`{
\x0f`a\xa0\taASWTa
\x01[\x82H\x06LCD0\x08_ADR\x0b\x00\x04\x14\x1c_DCS\x00\\VUPS
\x00\xa0\t\\VCDL\xa4
\x1f\xa1\x04\xa4
\x1d\x14\x0b_DGS\x00\xa4VQDL\x140_DSS\x01{h
\x01VSDL\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01[\x82@\x08CRT0\x08_ADR\x0b\x00\x01\x144_DCS\x00\\VUPS
\x01\xa0\x15\\VCSS\xa0\t\\VCDC\xa4
\x1f\xa1\x04\xa4
\x1d\xa1\x10\xa0\t\\VCDC\xa4
\x0f\xa1\x04\xa4
\r\x14\x0b_DGS\x00\xa4VQDC\x140_DSS\x01{h
\x01VSDC\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01[\x82H\x06DVI0\x08_ADR\x0b\x00\x03\x14\x1c_DCS\x00\\VUPS
\x00\xa0\t\\VCDD\xa4
\x1f\xa1\x04\xa4
\x1d\x14\x0b_DGS\x00\xa4VQDD\x140_DSS\x01{h
\x01VSDD\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01\x14?DSWT\x01\xa0\tVSDLp
\x01`\xa1\x05p
\x00`\xa0
VSDC}
\x02``\xa0
VSDD}
\x08``\xa0\x0f`\xa0\x0cVUPC\\VSDS`h\xa1\x02\xa3[\x82BUAGP_\x08_ADR\x0c\x00\x00\x01\x00\x08_S3D
\x03\x08RID_
\x00\x08LART\x12*\x02\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKA
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKB
\x00\x08AART\x12\x1a\x02\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x10\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x11\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4AART\xa1\x06\xa4LART[\x01MDGS\x07\x08VDEE
\x01\x08VDDA\x11\x03
\x02\x8dVDDA
\x00VUPC\x8dVDDA
\x01VQDL\x8dVDDA
\x02VQDC\x8dVDDA
\x03VQDT\x8dVDDA
\x04VQDD\x8dVDDA
\x05VSDL\x8dVDDA
\x06VSDC\x8dVDDA
\x07VSDT\x8dVDDA
\x08VSDD\x8dVDDA

MSWT\x8dVDDA
\x0bVWST[\x82BAVID_\x08_ADR
\x00[\x80VPCG\x02
\x00\x0b\x00\x01[\x81\x0eVPCG\x03\x00@*VPWR \x08_S3D
\x03\x145_INI\x00\\VUPS
\x02p\\VCDLVQDLp\\VCDCVQDCp\\VCDTVQDTp\\VCDDVQDD\x14\x07_PS0\x00\xa3\x14\x07_PS1\x00\xa3\x14\x07_PS2\x00\xa3\x14\x07_PS3\x00\xa3\x140VSWT\x00\xa0\x0f\\WVISp\\VEVT
\x07`\xa1
p\\VEVT
\x05`{
\x0f`a\xa0\taASWTa
\x01\x14@\x05VLOC\x01\xa0H\x04\x93h\\/\x03_SB_LID__LID\\VSLDh\xa0.\x93VPWR
\x00\xa0\x0bhp\\VEVT
\x01`\xa1
p\\VEVT
\x02`{
\x0f`a\xa0\taASWTa
\x00\x14G\x07_DOS\x01\xa0:\x93h
\x02p
\x14`\xa20`v`[#MDGS\xff\xff\xa0\x19\x93
\x00MSWTp
\x01MSWTp
\x00`phVDEE[\'MDGS["
\xc8\xa14[#MDGS\xff\xff\xa0\x0f\x93VDEE
\x02p
\x00MSWT\xa0\x0c\x94h
\x02p
\x01VDEE\xa1\x07phVDEE[\'MDGS\x14\x13_DOD\x00\xa4\x12\x0b\x03\x0b\x00\x01\x0b\x10\x02\x0b\x10\x01\x14L\x0bASWT\x02\xa0\x14\x93
\x01VDEE{
\x01ia\\VSDSha\xa1O\tp
\x14`\xa2A\x08`v`[#MDGS\xff\xff\xa0I\x06\x93
\x00MSWTp
\x00`\xa0\r{
\x01i\x00p
\x01VUPC\xa1\x08p
\x00VUPC\xa0\r{
\x01h\x00p
\x01VQDL\xa1\x08p
\x00VQDL\xa0\r{
\x02h\x00p
\x01VQDC\xa1\x08p
\x00VQDC\xa0\r{
\x08h\x00p
\x01VQDD\xa1\x08p
\x00VQDD[\'MDGS["
\xc8\xa0\r{
\x02i\x00\x86VID_
\x81\xa1\x08\x86VID_
\x80[\x82C\tLCD0\x08_ADR\x0b\x10\x01\x14\x1c_DCS\x00\\VUPS
\x00\xa0\t\\VCDL\xa4
\x1f\xa1\x04\xa4
\x1d\x14*_DDC\x01\\VDDC\xa0\x0b\x93h
\x01\xa4\\DDC1\xa1\x12\xa0\x0b\x93h
\x02\xa4\\DDC2\xa1\x04\xa4
\x00\x14\x0b_DGS\x00\xa4VQDL\x140_DSS\x01{h
\x01VSDL\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01[\x82@\x08CRT0\x08_ADR\x0b\x00\x01\x144_DCS\x00\\VUPS
\x01\xa0\x15\\VCSS\xa0\t\\VCDC\xa4
\x1f\xa1\x04\xa4
\x1d\xa1\x10\xa0\t\\VCDC\xa4
\x0f\xa1\x04\xa4
\r\x14\x0b_DGS\x00\xa4VQDC\x140_DSS\x01{h
\x01VSDC\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01[\x82H\x06DVI0\x08_ADR\x0b\x10\x02\x14\x1c_DCS\x00\\VUPS
\x00\xa0\t\\VCDD\xa4
\x1f\xa1\x04\xa4
\x1d\x14\x0b_DGS\x00\xa4VQDD\x140_DSS\x01{h
\x01VSDD\xa0!{h\x0c\x00\x00\x00\x80\x00\xa0\x0f{h\x0c\x00\x00\x00@\x00DSWT
\x02\xa1\x07DSWT
\x01\x14?DSWT\x01\xa0\tVSDLp
\x01`\xa1\x05p
\x00`\xa0
VSDC}
\x02``\xa0
VSDD}
\x08``\xa0\x0f`\xa0\x0cVUPC\\VSDS`h\xa1\x02\xa3[\x80PEGC\x02
\x00\x0b\x00\x01[\x81\x18PEGC\x03\x00@vGMGP\x01HPGP\x01PMGP\x01[\x82I\x12EXP0\x08_ADR\x0c\x00\x00\x1c\x00\x08RID_
\x00[\x80P0CS\x02
\x00\x0b\x00\x01[\x81M\x04P0CS\x03\x00@-ABP_\x01\x00\x02PDC_\x01\x00\x02PDS_\x01\x00\x01\x00(RID0\x10PSP0\x01PPP0\x01\x00F<\x00\x06HPCE\x01PMCE\x01\x00\x18\x00\x06HPCS\x01PMCS\x01\x08_PRW\x12\x06\x02
\t
\x04\x08LPRT\x12C\x05\x04\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKA
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKB
\x00\x12\x13\x04\x0b\xff\xff
\x02\\._SB_LNKC
\x00\x12\x13\x04\x0b\xff\xff
\x03\\._SB_LNKD
\x00\x08APRT\x122\x04\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x10\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x11\x12\x0b\x04\x0b\xff\xff
\x02
\x00
\x12\x12\x0b\x04\x0b\xff\xff
\x03
\x00
\x13\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4APRT\xa1\x06\xa4LPRT[\x82D\x0fEXP1\x08_ADR\x0c\x01\x00\x1c\x00\x08RID_
\x00[\x80P1CS\x02
\x00\x0b\x00\x01[\x81\x18P1CS\x03\x00@0RID1\x10PSP1\x01PPP1\x01\x08_PRW\x12\x06\x02
\t
\x04\x08LPRT\x12C\x05\x04\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKB
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKC
\x00\x12\x13\x04\x0b\xff\xff
\x02\\._SB_LNKD
\x00\x12\x13\x04\x0b\xff\xff
\x03\\._SB_LNKA
\x00\x08APRT\x122\x04\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x11\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x12\x12\x0b\x04\x0b\xff\xff
\x02
\x00
\x13\x12\x0b\x04\x0b\xff\xff
\x03
\x00
\x10\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4APRT\xa1\x06\xa4LPRT[\x82H\x14EXP2\x08_ADR\x0c\x02\x00\x1c\x00\x08RID_
\x00\x08XCPF
\x00[\x80P2CS\x02
\x00\x0b\x00\x01[\x81M\x04P2CS\x03\x00@-ABP_\x01\x00\x02PDC_\x01\x00\x02PDS_\x01\x00\x01\x00(RID2\x10PSP2\x01PPP2\x01\x00F<\x00\x06HPCE\x01PMCE\x01\x00\x18\x00\x06HPCS\x01PMCS\x01\x08_PRW\x12\x06\x02
\t
\x04\x08LPRT\x12C\x05\x04\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKC
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKD
\x00\x12\x13\x04\x0b\xff\xff
\x02\\._SB_LNKA
\x00\x12\x13\x04\x0b\xff\xff
\x03\\._SB_LNKB
\x00\x08APRT\x122\x04\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x12\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x13\x12\x0b\x04\x0b\xff\xff
\x02
\x00
\x10\x12\x0b\x04\x0b\xff\xff
\x03
\x00
\x11\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4APRT\xa1\x06\xa4LPRT[\x82\x16EXUP\x08_ADR
\x00\x14\t_RMV\x00\xa4
\x01[\x82B\x10EXP3\x08_ADR\x0c\x03\x00\x1c\x00\x08RID_
\x00[\x80P3CS\x02
\x00\x0b\x00\x01[\x81\x18P3CS\x03\x00@0RID3\x10PSP3\x01PPP3\x01\x08_PRW\x12\x06\x02
\t
\x04\x08LPRT\x12C\x05\x04\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKD
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKA
\x00\x12\x13\x04\x0b\xff\xff
\x02\\._SB_LNKB
\x00\x12\x13\x04\x0b\xff\xff
\x03\\._SB_LNKC
\x00\x08APRT\x122\x04\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x13\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x10\x12\x0b\x04\x0b\xff\xff
\x02
\x00
\x11\x12\x0b\x04\x0b\xff\xff
\x03
\x00
\x12\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4APRT\xa1\x06\xa4LPRT[\x82\x0cEXPD\x08_ADR
\x00[\x82O\x16PCI1\x08_ADR\x0c\x00\x00\x1e\x00\x08_S3D
\x02\x08RID_
\x00\x08LPRT\x12G\t\x07\x12\x13\x04\x0b\xff\xff
\x00\\._SB_LNKA
\x00\x12\x13\x04\x0b\xff\xff
\x01\\._SB_LNKB
\x00\x12\x13\x04\x0b\xff\xff
\x02\\._SB_LNKC
\x00\x12\x15\x04\x0c\xff\xff\x01\x00
\x00\\._SB_LNKA
\x00\x12\x15\x04\x0c\xff\xff\x02\x00
\x00\\._SB_LNKF
\x00\x12\x15\x04\x0c\xff\xff\x02\x00
\x01\\._SB_LNKG
\x00\x12\x15\x04\x0c\xff\xff\x08\x00
\x00\\._SB_LNKE
\x00\x08APRT\x12O\x05\x07\x12\x0b\x04\x0b\xff\xff
\x00
\x00
\x10\x12\x0b\x04\x0b\xff\xff
\x01
\x00
\x11\x12\x0b\x04\x0b\xff\xff
\x02
\x00
\x12\x12\r\x04\x0c\xff\xff\x01\x00
\x00
\x00
\x10\x12\r\x04\x0c\xff\xff\x02\x00
\x00
\x00
\x15\x12\r\x04\x0c\xff\xff\x02\x00
\x01
\x00
\x16\x12\r\x04\x0c\xff\xff\x08\x00
\x00
\x00
\x14\x14\x19_PRT\x00\xa0\x0b\\GPIC\xa4APRT\xa1\x06\xa4LPRT\x08_PRW\x12\x06\x02
\x0b
\x04[\x82\'CDBS\x08_ADR
\x00\x14\x13_S3D\x00\xa0\t\\WMEF\xa4
\x02\xa4
\x03\x08_SUN
\x01[\x82L\xb8IDE0\x08_ADR\x0c\x01\x00\x1f\x00\x08_S3D
\x03\x08RID_
\x00[\x80IDCS\x02
\x00\x0b\x00\x01[\x81@\x12IDCS\x03\x00@ PFT0\x01PIE0\x01PPE0\x01PDT0\x01PFT1\x01PIE1\x01PPE1\x01PDT1\x01PRC0\x02\x00\x02PIS0\x02PSIE\x01PIDE\x01SFT0\x01SIE0\x01SPE0\x01SDT0\x01SFT1\x01SIE1\x01SPE1\x01SDT1\x01SRC0\x02\x00\x02SIS0\x02SSIE\x01SIDE\x01PRC1\x02PIS1\x02SRC1\x02SIS1\x02\x00\x18PSD0\x01PSD1\x01SSD0\x01SSD1\x01\x00\x0cPCT0\x02\x00\x02PCT1\x02\x00\x02SCT0\x02\x00\x02SCT1\x02\x00\x02\x00@\x04PCB0\x01PCB1\x01SCB0\x01SCB1\x01PCR0\x01PCR1\x01SCR0\x01SCR1\x01\x00\x02WRPP\x01\x00\x01FPB0\x01FPB1\x01FSB0\x01FSB1\x01PSIG\x02SSIG\x02[\x81\x1fIDCS\x03\x00@ PTI0\x04PTI1\x04\x00\x08STI0\x04STI1\x04\x14\'GPCT\x04\xa0\t\x92}hi\x00\xa4
\x00\xa0\t\x90\x92hi\xa4\x0b\x84\x03\xa4wt
\trjk\x00\x00
\x1e\x00\x14*GDCT\x04\xa0\x06\x92h\xa4
\x00\xa0\x05i\xa4
\x14\xa0\x0cj\xa4wt
\x04k\x00
\x0f\x00\xa4wt
\x04k\x00
\x1e\x00\x142MTIM\x02p
\x00`\xa0\x07h}`
\x01`\xa0\x0b\x92\x95h
\x02}`
\x02`\xa0\x08\x92i}`
\x04`\xa0\x08\x92h}`
\x08`\xa4`\x14\x1aMISP\x01\xa0\x06\x92h\xa4
\x00\xa0\t\x92\x94h
\x02\xa4
\x01\xa4
\x02\x14\x1cMRCT\x01\xa0\t\x92\x94h
\x02\xa4
\x00\xa0\x08\x93h
\x03\xa4
\x01\xa4
\x03[\x82O\x95PRIM\x08_ADR
\x00\x08BGTM\x11\x03
\x14\x8aBGTM
\x00GTP0\x8aBGTM
\x04GTD0\x8aBGTM
\x08GTP1\x8aBGTM
\x0cGTD1\x8aBGTM
\x10GTMF\x08BPI0
\x00\x08BDM0
\x00\x08BPI1
\x00\x08BDM1
\x00\x08DRE0
\x00\x08DRE1
\x00\x08DIP0
\x00\x08DIP1
\x00\x14M\x11_GTM\x00pGPCTPFT0PDT0PIS0PRC0GTP0pGDCTPSD0FPB0PCB0PCT0GTD0\xa0\x0f\x92GTD0pGTP0GTD0\xa0H\x04PSIEpGPCTPFT1PDT1PIS1PRC1GTP1pGDCTPSD1FPB1PCB1PCT1GTD1\xa0\x0f\x92GTD1pGTP1GTD1\xa1\x0fp
\x00GTP1p
\x00GTD1p
\x00GTMF\xa0\x10PSD0}GTMF
\x01GTMF\xa0\x1f\x92GTP0}GTMF
\x01GTMFp
xGTP0p
\x14GTD0\xa0\x10PIE0}GTMF
\x02GTMF\xa0\x10PSD1}GTMF
\x04GTMF\xa0\x10PIE1}GTMF
\x08GTMF}GTMF
\x10GTMF\xa4BGTM\x14@G_STM\x03\x8ah
\x00STP0\x8ah
\x04STD0\x8ah
\x08STP1\x8ah
\x0cSTD1\x8ah
\x10STMFp
\x00DRE0p
\x00DIP0\xa0G$\x93\x87i\x0b\x00\x02\x8bi
\x00M000\x8bi
bM049\x8bi
fM051\x8bi
jM053\x8bi
|M062\x8bi
~M063\x8bi
\x80M064\x8bi
\x82M065\x8bi
\x88M068\x8bi
\x9cM078\x8bi
\xacM086\x8bi
\xb0M088\x8bi
\xeeM119\x8bi\x0b(\x01M148\xa0\x1f\x93{M148\x0b\x00\xc0\x00\x0b\x00@\xa0\x10{M148
\x04\x00p
\x01DRE0\xa0\x1b{M086\x0b\x00\x80\x00\xa0\x10{M119
\x01\x00p
\x01DRE0\xa0E\t\\W98Fp\\UUDMM053M088STD0\xa0\x10STD0}STMF
\x01STMF\xa1\x0c{STMF
\xfeSTMFp\\UMDMM053M063M062M065STP0\xa0 \x92STP0p\\UPIOM053M064M051M068STP0\xa0\x15{M049\x0b\x00\x08\x00}STMF
\x02STMF\xa1\x0c{STMF
\xfdSTMFp\\DPIOSTP0{STMF
\x02\x00`p\\DUDMSTD0{STMF
\x01\x00a\xa0&\x92\x95\\/\x05_SB_PCI0LPC_EC__BGID
\x00
\x0cp
\x00`p
\xffapMTIM`{M000\x0b\x00\x80\x00PTI0pMISP`PIS0pMRCT`PRC0\xa0\x0c\x93a
\xffp
\x00PSD0\xa1C\x06p
\x01PSD0\xa0\x0c\x92\x94a
\x02paPCT0\xa1\x18\xa0\r{a
\x01\x00p
\x01PCT0\xa1\x08p
\x02PCT0\xa0\r\x92\x95a
\x03p
\x01PCB0\xa1\x08p
\x00PCB0\xa0\x0c\x93a
\x05p
\x01FPB0\xa1\x08p
\x00FPB0p
\x01PCR0p\\FDMA`aBDM0p\\FPIO`BPI0\xa0J\x1e\x93\x87j\x0b\x00\x02\x8bj
\x00S000\x8bj
bS049\x8bj
fS051\x8bj
jS053\x8bj
|S062\x8bj
~S063\x8bj
\x80S064\x8bj
\x82S065\x8bj
\x88S068\x8bj
\xb0S088\xa0E\t\\W98Fp\\UUDMS053S088STD1\xa0\x10STD1}STMF
\x04STMF\xa1\x0c{STMF
\xfbSTMFp\\UMDMS053S063S062S065STP1\xa0 \x92STP1p\\UPIOS053S064S051S068STP1\xa0\x15{S049\x0b\x00\x08\x00}STMF
\x08STMF\xa1\x0c{STMF
\xf7STMFp\\DPIOSTP1{STMF
\x08\x00`p\\DUDMSTD1{STMF
\x04\x00a\xa0=STP1pMTIM`{S000\x0b\x00\x80\x00PTI1\xa0${STMF
\x10\x00pMISP`PIS1pMRCT`PRC1p
\x01PSIE\xa1\x0fp
\x00PTI1p
\x00PSIE\xa0\x0c\x93a
\xffp
\x00PSD1\xa1C\x06p
\x01PSD1\xa0\x0c\x92\x94a
\x02paPCT1\xa1\x18\xa0\r{a
\x01\x00p
\x01PCT1\xa1\x08p
\x02PCT1\xa0\r\x92\x95a
\x03p
\x01PCB1\xa1\x08p
\x00PCB1\xa0\x0c\x93a
\x05p
\x01FPB1\xa1\x08p
\x00FPB1p
\x01PCR1p\\FDMA`aBDM1p\\FPIO`BPI1[\x82I4MSTR\x08_ADR
\x00\x08HDTF\x11\x1f
\x1c\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef\x8cHDTF
\x0fHDMA\x8cHDTF
\x16HPIO\x8cHDTF
\rHFLC\x08ERTF\x11&
#\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef_\x00\x00\x00\x00\xa0\xef\x8cERTF
\x0fEDMA\x8cERTF
\x16EPIO\x8cERTF
\rEFLC\x08HPTF\x11&
#\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef\x10\x03\x00\x00\x00\xa0\xef\x8cHPTF
\x0fPDMA\x8cHPTF
\x16PPIO\x8cHPTF
\rPFLC\x08HXTF\x11-
*\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef_\x00\x00\x00\x00\xa0\xef\x10\x03\x00\x00\x00\xa0\xef\x8cHXTF
\x0fXDMA\x8cHXTF
\x16XPIO\x8cHXTF
\rXFLC\x08IDTF\x11\x11
\x0e\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef\x8cIDTF
\x01IDMA\x8cIDTF
\x08IPIO\x08DDTF\x11\x1f
\x1c\x03\x00\x00\x00\x00\xa0\xef\x03\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xe3\x00\x00\x00\x00\x00\xa0\xe3\x8cDDTF
\x01DDMA\x8cDDTF
\x08DPIO\x8cDDTF
\x0fDTAT\x8cDDTF
\x16DTFT\x14@\x14_GTF\x00p\\/\x05_SB_PCI0LPC_EC__BGID
\x00`\xa0"\\OSSSp
\xe1HFLCp
\xe1EFLCp
\xe1PFLCp
\xe1XFLC\xa0M\x08\x93`
\x06\xa0H\x04^^DRE0\xa0"^^DIP0p^^BDM0XDMAp^^BPI0XPIO\xa4HXTF\xa1\x1cp^^BDM0EDMAp^^BPI0EPIO\xa4ERTF\xa0"^^DIP0p^^BDM0PDMAp^^BPI0PPIO\xa4HPTFp^^BDM0HDMAp^^BPI0HPIO\xa4HDTFp\x00a\xa0\x08\x93`
\x03p\x01a\xa0\x08\x93`

p\x01a\xa0\x08\x93`
\x0bp\x01a\xa01ap\\CDFLDTFTp\\CDAHDTATp^^BDM0DDMAp^^BPI0DPIO\xa4DDTF\xa1\x1cp^^BDM0IDMAp^^BPI0IPIO\xa4IDTF\x14\x1e_EJ0\x01\\/\x05_SB_PCI0LPC_EC__BEJ0h\x14)_STA\x00\xa0\x1d\\/\x05_SB_PCI0LPC_EC__BSTA
\x01\xa4
\x0f\xa1\x04\xa4
\x00[\x82K"SATA\x08_ADR\x0c\x02\x00\x1f\x00\x08_S3D
\x03\x08RID_
\x00[\x80IDCS\x02
\x00\x0b\x00\x01[\x81@\x12IDCS\x03\x00@ PFT0\x01PIE0\x01PPE0\x01PDT0\x01PFT1\x01PIE1\x01PPE1\x01PDT1\x01PRC0\x02\x00\x02PIS0\x02PSIE\x01PIDE\x01SFT0\x01SIE0\x01SPE0\x01SDT0\x01SFT1\x01SIE1\x01SPE1\x01SDT1\x01SRC0\x02\x00\x02SIS0\x02SSIE\x01SIDE\x01PRC1\x02PIS1\x02SRC1\x02SIS1\x02\x00\x18PSD0\x01PSD1\x01SSD0\x01SSD1\x01\x00\x0cPCT0\x02\x00\x02PCT1\x02\x00\x02SCT0\x02\x00\x02SCT1\x02\x00\x02\x00@\x04PCB0\x01PCB1\x01SCB0\x01SCB1\x01PCR0\x01PCR1\x01SCR0\x01SCR1\x01\x00\x02WRPP\x01\x00\x01FPB0\x01FPB1\x01FSB0\x01FSB1\x01PSIG\x02SSIG\x02[\x81\x1fIDCS\x03\x00@ PTI0\x04PTI1\x04\x00\x08STI0\x04STI1\x04\x14\'GPCT\x04\xa0\t\x92}hi\x00\xa4
\x00\xa0\t\x90\x92hi\xa4\x0b\x84\x03\xa4wt
\trjk\x00\x00
\x1e\x00\x14*GDCT\x04\xa0\x06\x92h\xa4
\x00\xa0\x05i\xa4
\x14\xa0\x0cj\xa4wt
\x04k\x00
\x0f\x00\xa4wt
\x04k\x00
\x1e\x00\x142MTIM\x02p
\x00`\xa0\x07h}`
\x01`\xa0\x0b\x92\x95h
\x02}`
\x02`\xa0\x08\x92i}`
\x04`\xa0\x08\x92h}`
\x08`\xa4`\x14\x1aMISP\x01\xa0\x06\x92h\xa4
\x00\xa0\t\x92\x94h
\x02\xa4
\x01\xa4
\x02\x14\x1cMRCT\x01\xa0\t\x92\x94h
\x02\xa4
\x00\xa0\x08\x93h
\x03\xa4
\x01\xa4
\x03[\x82\x1dSMBU\x08_ADR\x0c\x03\x00\x1f\x00\x08_S3D
\x03\x08RID_
\x00[\x82J\x0eUSB0\x08_ADR\x0c\x00\x00\x1d\x00\x08_S3D
\x02\x08RID_
\x00[\x80U0CS\x02
\xc4
\x04[\x81\rU0CS\x03U0EN\x02\x00\x1e\x08_PR0\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR1\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR2\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PRW\x12\x1d\x03
\x03
\x03\\/\x05_SB_PCI0LPC_EC__PUBS\x141_PSW\x01\xa0\thp
\x03U0EN\xa1\x08p
\x00U0EN\\/\x05_SB_PCI0LPC_EC__PNSTh[\x82J\x07USB1\x08_ADR\x0c\x01\x00\x1d\x00\x08_S3D
\x02\x08RID_
\x00[\x80U1CS\x02
\xc4
\x04[\x81\rU1CS\x03U1EN\x02\x00\x1e\x08_PRW\x12\x06\x02
\x04
\x03\x14\x19_PSW\x01\xa0\thp
\x03U1EN\xa1\x08p
\x00U1EN[\x82\x1aURTH\x08_ADR
\x00[\x82\x0cUPEX\x08_ADR
\x02[\x82N\x0eUSB2\x08_ADR\x0c\x02\x00\x1d\x00\x08_S3D
\x02\x08RID_
\x00[\x80U2CS\x02
\xc4
\x04[\x81\rU2CS\x03U2EN\x02\x00\x1e\x08_PR0\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR1\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR2\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PRW\x12\x1d\x03
\x0c
\x03\\/\x05_SB_PCI0LPC_EC__PUBS\x14\x19_PSW\x01\xa0\thp
\x03U2EN\xa1\x08p
\x00U2EN[\x82\x1aURTH\x08_ADR
\x00[\x82\x0cUPDK\x08_ADR
\x02[\x82\x16USB3\x08_ADR\x0c\x03\x00\x1d\x00\x08RID_
\x00[\x82N\x0fUSB7\x08_ADR\x0c\x07\x00\x1d\x00\x08_S3D
\x03\x08RID_
\x00[\x80U7CS\x02
`
\x04[\x81\x14U7CS\x03\x00\x10PWKI\x01PWUC\x06\x00\t\x08_PR0\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR1\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x08_PR2\x12\x19\x01\\/\x05_SB_PCI0LPC_EC__PUBS\x14\x14_INI\x00p
\x01PWKIp
\x0fPWUC\x08_PRW\x12\x1d\x03
\r
\x03\\/\x05_SB_PCI0LPC_EC__PUBS[\x82(URTH\x08_ADR
\x00[\x82\x0cUPDK\x08_ADR
\x06[\x82\x0cUPEX\x08_ADR
\x04[\x821HDEF\x08_ADR\x0c\x00\x00\x1b\x00\x08_S3D
\x03\x08RID_
\x00\x08_PRW\x12\x06\x02
\x05
\x04\x14\x07_PSW\x01\xa3\x10C\xbe\\/\x04_SB_PCI0LPC_EC__\x08BDEV
\xff\x08BSTS
\x00\x08BHKE
\x00\x08BXCN
\x00\x142_Q2C\x00\xa0+\x93BSTS
\x00pBGID
\x00BDEV\xa0\rBXCNNXREBDEV\xa1\tNBREBDEV\x14)_Q2D\x00pBGID
\x00BDEV\xa0\rBXCNNXRCBDEV\xa1\tNBINBDEV\x14D\x07_Q38\x00pBGID
\x00`\xa01\x93`
\x0fBDIS\xa0\x17BXCNpBDEV`p
\x0fBDEVNXEJ`\xa1\x0fNBEJBDEVp`BDEV\xa12\xa0\x16HPBU\xa0\x10BXCNp`BDEVNXIN`\xa1\x19p`BDEV\xa0
BXCNNXRC`\xa1\x06NBIN`\x14B\rNBRE\x01\xa0&\x93h
\r\xa0 \\LFDC\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x03\xa0F\x04\x95h
\x0c\xa0#\x93\\BIDE
\x03\x86\\/\x05_SB_PCI0SATASCNDMSTR
\x03\xa1\x1b\x86\\/\x05_SB_PCI0IDE0PRIMMSTR
\x03\xa0I\x04\x93h
\x10\xa0+\x91HPACHB0A\xa0 \\WNTF\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x03\xa1\x16LED_
\x04
\xc0BEEP
\x0fp
\x02BSTS\xa0\x12\x93h
\x11\x86\\._SB_LCIO
\x03\x14K\x0eNBEJ\x01\xa0N\x0c\x93BSTS
\x00\xa0&\x93h
\r\xa0 \\LFDC\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x01\xa0F\x04\x95h
\x0c\xa0#\x93\\BIDE
\x03\x86\\/\x05_SB_PCI0SATASCNDMSTR
\x01\xa1\x1b\x86\\/\x05_SB_PCI0IDE0PRIMMSTR
\x01\xa0C\x04\x93h
\x10\xa0 \\WNTF\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x01\xa1\x1b\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x81\xa0\x12\x93h
\x11\x86\\._SB_LCIO
\x01LED_
\x04
\x00BEEP
\x00p
\x00BSTS\x14B\x15NBIN\x01\xa08\x93h
\r\xa02\\LFDCBEN_
\x00BSFDLED_
\x04
\x80\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x01\xa0B\x06\x95h
\x0c\xa0\x0b\x93h
\x06BEN_
\x02\xa1\x07BEN_
\x01LED_
\x04
\x80\xa0#\x93\\BIDE
\x03\x86\\/\x05_SB_PCI0SATASCNDMSTR
\x01\xa1\x1b\x86\\/\x05_SB_PCI0IDE0PRIMMSTR
\x01\xa0I\x06\x93h
\x10LED_
\x04
\x80\xa0>\\WNTFp
\x01\\/\x06_SB_PCI0LPC_EC__BAT1XB1S\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x01\xa1\x1b\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x81\xa07\x93h
\x11\xa01\x92\\/\x04_SB_PCI0LPC_CSONLED_
\x04
\x80\xa0\x13\\WNTF\x86\\._SB_LCIO
\x01BEEP
\x00p
\x00BSTS\x14J\x07BEJ0\x01\xa0A\x06hBDISLED_
\x04
\x00\\BHDP
\x01
\x00p
\x01BSTS\xa0*BHKEp
\x00BHKE\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x030\xa0\x16\x93BDEV
\x11p
\x0fBDEVp
\x00BSTS\xa1\x10LED_
\x04
\x80p
\x00BSTS\x14\x1dBEJ3\x01\xa0\rhBDISp
\x01BSTS\xa1\x08p
\x00BSTS\x14I\x07BPTS\x01p
\x01HDBM\xa0\x17\x92\x93BSTS
\x00p
\x0fBDEVp
\x00BSTSp
\x00BHKEp
\x01`\xa0\x1d\x92\x93BDEV
\x0f\xa0\x13\x90\x92\\LFDC\x93BDEV
\rp
\x00`\xa1\x05p
\x00`\xa0
\x92\x95h
\x04p
\x00`\xa0\x08`BUWK
\x01\xa1\x0fLED_
\x04
\x00BUWK
\x00\x14D\x14BWAK\x01BUWK
\x00pBGID
\x00`\xa0\x10\x90\x92\\LFDC\x93`
\rBDIS\\/\x05_SB_PCI0LPC_FDC__INI\xa0K\x05\\LFDC\xa0C\x05\x92\x93`
\r\xa0K\x04\x93\\/\x05_SB_PCI0LPC_FDC_FD0S\\/\x05_SB_PCI0LPC_EC__HPNF\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x01\xa0J
\x93BSTS
\x00\xa0.\x92\x93`BDEV\xa0\x10BXCNp`BDEVNXRC`\xa1\x14NBEJBDEVp`BDEVNBIN`\xa1A\x07\xa0N\x06\x91\\LFDC\x92\x93BDEV
\r\xa0M\x05\x92\x93`
\x0fLED_
\x04
\x80\xa0M\x04HPBU}yh
\x08\x00\x0b\x05 BHKE\\/\x06_SB_PCI0LPC_EC__HKEYMHKQBHKE\xa0\x06\x92\x94h
\x02\xa1\x13\xa0
BXCNNXRE`\xa1\x06NBRE`\x14N\x0fBDIS\x00\xa0F\x0f\x92\\/\x04_SB_PCI0LPC_CSON\xa0F\x06\x93\\BIDE
\x03p
\x00\\/\x04_SB_PCI0SATASIE0p
\x00\\/\x04_SB_PCI0SATASTI0p
\x00\\/\x04_SB_PCI0LPC_URST[!
\x0fp
\x01\\/\x04_SB_PCI0SATASSIG\xa1N\x05p
\x00\\/\x04_SB_PCI0IDE0PIE0p
\x00\\/\x04_SB_PCI0IDE0PTI0p
\x00\\/\x04_SB_PCI0LPC_URST[!
\x0fp
\x01\\/\x04_SB_PCI0IDE0PSIGp
\x01\\/\x04_SB_PCI0LPC_CSONBSFD\x14A\x0eBEN_\x01\xa0I\r\\/\x04_SB_PCI0LPC_CSONp
\x00\\/\x04_SB_PCI0LPC_URSTp
\x00\\/\x04_SB_PCI0LPC_CSON["
\x0f\xa0C\th\xa05\x93\\BIDE
\x03p
\x00\\/\x04_SB_PCI0SATASSIGp
\x01\\/\x04_SB_PCI0SATASIDE\xa1-p
\x00\\/\x04_SB_PCI0IDE0PSIGp
\x01\\/\x04_SB_PCI0IDE0PIDE[!
-p
\x01\\/\x04_SB_PCI0LPC_URST\xa0
\x93h
\x02["\x0b\xd0\x07\xa1\x06["\x0b\x90\x01\x14N\x05BSTA\x01\xa0\x17\\/\x04_SB_PCI0LPC_CSON\xa4
\x00BINI\xa0\r\x93h
\x00\xa4\x93BDEV
\r\xa0\r\x93h
\x01\xa4\x95BDEV
\x0c\xa0\r\x93h
\x02\xa4\x93BDEV
\x0e\xa0\r\x93h
\x03\xa4\x93BDEV
\x11\xa4
\x00\x14D\x06BUWK\x01\xa0?\\H8DR\xa0\x1chp
\x01\\/\x05_SB_PCI0LPC_EC__HWBU\xa1\x1bp
\x00\\/\x05_SB_PCI0LPC_EC__HWBU\xa1\x1c\xa0\rh\\MBEC
2
\xff
\x80\xa1\x0c\\MBEC
2
\x7f
\x00\x14\x1aBINI\x00\xa0\x13\x93BDEV
\xffpBGID
\x00BDEV\x14F\x0bBGID\x01\xa0\x06hp
\xff`\xa1E
\xa0\x0c\\H8DRpHBID`\xa1\x10zRBEC
G
\x02`{`
\x0f`{`
\x03`\xa0\t\x93`
\x00p
\x03`\xa1\x11\xa0\t\x93`
\x02p
\x06`\xa1\x05p
\x0f`\xa0)\x93`
\x0f\xa0\x10\\H8DR\xa0\tHB1Ap
\x10`\xa1\x12\xa0\x10{\\RBEC
9
\x80\x00p
\x10`\xa09\x93`
\x0f\xa03\x92\\/\x04_SB_PCI0LPC_CSON\xa0\x1d\x92{\\/\x04_SB_PCI0LPC_GL00
@\x00p
\x11`\xa4`\x14(BSFD\x00\xa0\x13BSTA
\x00\\MISA\x0b\xf3\x03
\xf3
\x00\xa1\r\\MISA\x0b\xf3\x03
\xf3
\x04\x14A\x0bNXRE\x01\xa0\x14\x93h
\x0fLED_
\x04
\x00p
\x00BSTS\xa0!\x93h
\r\xa0\x1b\\LFDCLED_
\x04
\xc0\x86\\._SB_SWAP
\x83\xa0\x1a\x95h
\x0cLED_
\x04
\xc0\x86\\._SB_SWAP
\x83\xa0\x1a\x93h
\x0eLED_
\x04
\xc0\x86\\._SB_SWAP
\x83\xa0<\x93h
\x10\xa0\x1f\x91HPACHB0ALED_
\x04
\xc0\x86\\._SB_SWAP
\x83\xa1\x16LED_
\x04
\xc0BEEP
\x0fp
\x02BSTS\x14E\x0cNXRC\x01\xa0+\x93h
\r\xa0%\\LFDCLED_
\x04
\x80BEN_
\x00BSFD\x86\\._SB_SWAP
\x80\xa0.\x95h
\x0cLED_
\x04
\x80\xa0\x0b\x93h
\x06BEN_
\x02\xa1\x07BEN_
\x01\x86\\._SB_SWAP
\x80\xa0 \x93h
\x0eLED_
\x04
\x80BEN_
\x00\x86\\._SB_SWAP
\x80\xa04\x93h
\x10\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x81LED_
\x04
\x80\x86\\._SB_SWAP
\x80BEEP
\x00p
\x00BSTS\x14I\x04NXEJ\x01\xa0\x1f\x93h
\x10\x86\\/\x05_SB_PCI0LPC_EC__BAT1
\x81\x86\\._SB_SWAP
\x82LED_
\x04
\x00BEEP
\x00p
\x00BSTS\x14\x13NXIN\x01\x86\\._SB_SWAP
\x81\x10C\x10\\/\x04_SB_PCI0LPC_FDC_\x08XFDS
\x00\x08DCFD
\x00\x14C\x05_INI\x00p
\x00XFDS\xa07\\H8DR}\\/\x05_SB_PCI0LPC_EC__HAMA
\x0c\\/\x05_SB_PCI0LPC_EC__HAMA\xa1\x0c\\MBEC
\x1a
\xff
\x0c\x08FDEB\x11\x17
\x14\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x8cFDEB
\x00FD0S\x14C\x06_FDE\x00\xa0&\x91\\/\x05_SB_PCI0LPC_EC__BSTA
\x00DCFDp
\x01FD0S\xa1/\xa0$\x91\\/\x05_SB_PCI0LPC_EC__HPNFXFDSp
\x00FD0S\xa1\x08p
\x01FD0S\xa4FDEB\x10E\x06\\/\x05_SB_PCI0LPC_FDC_FDD0\x14K\x04_EJ0\x01\xa02\\/\x05_SB_PCI0LPC_EC__BSTA
\x00\\/\x05_SB_PCI0LPC_EC__BEJ0h\xa1\x10\xa0\x05DCFD\xa1\x08p
\x01XFDS\x10A\x12\\/\x04_SB_PCI0LPC_EC__\x14C\t_Q52\x00\xa02\\/\x05_SB_PCI0LPC_FDC_XFDSp
\x00\\/\x05_SB_PCI0LPC_FDC_XFDS\xa1H\x05\xa02\x91\\/\x05_SB_PCI0LPC_EC__BSTA
\x00\\/\x05_SB_PCI0LPC_FDC_DCFD\xa1"\xa0 \\LFDC\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x01\x14G\x07_Q53\x00p
\x00\\/\x05_SB_PCI0LPC_FDC_XFDS\xa02\x91\\/\x05_SB_PCI0LPC_EC__BSTA
\x00\\/\x05_SB_PCI0LPC_FDC_DCFD\xa1"\xa0 \\LFDC\x86\\/\x05_SB_PCI0LPC_FDC_FDD0
\x01\x10F\x04\\/\x05_SB_PCI0LPC_EC__BAT1\x14,_EJ0\x01p
\x00B1STp
\x00XB1S\\/\x05_SB_PCI0LPC_EC__BEJ0h\x10@3\\_SB_[\x82G2SWAP\x08_HID\x0c$M\x00i\x14\x15_STA\x00\xa0\t\\WMEF\xa4
\x0f\xa1\x04\xa4
\x00\x14"XCNN\x01ph\\/\x05_SB_PCI0LPC_EC__BXCN\xa4
\t\x14\tXSWP\x00\xa4
\x01\x14\x1eXEJ0\x01\\/\x05_SB_PCI0LPC_EC__BEJ0h\x14\x1eXEJ3\x01\\/\x05_SB_PCI0LPC_EC__BEJ3h\x14@\x1fXDID\x00\x08XPCK\x12\x17\x06
\x00
\x00\x0c\xff\xff\xff\xff\x0c\xff\xff\xff\xff\x0c\xff\xff\xff\xff
\x00p\\/\x05_SB_PCI0LPC_EC__BDEV`p`\x88XPCK
\x00\x00\xa0D\r\x95`
\x0c\xa0J\x06\x93\\BIDE
\x03p\\/\x04_SB_PCI0SATA_ADR\x88XPCK
\x02\x00p\\/\x05_SB_PCI0SATASCND_ADR\x88XPCK
\x03\x00p\\/\x06_SB_PCI0SATASCNDMSTR_ADR\x88XPCK
\x04\x00\xa1B\x06p\\/\x04_SB_PCI0IDE0_ADR\x88XPCK
\x02\x00p\\/\x05_SB_PCI0IDE0PRIM_ADR\x88XPCK
\x03\x00p\\/\x06_SB_PCI0IDE0PRIMMSTR_ADR\x88XPCK
\x04\x00\xa0O\x05\x93`
\r\xa0K\x04\\LFDCp\\/\x05_SB_PCI0LPC_FDC__HID\x88XPCK
\x02\x00p\\/\x06_SB_PCI0LPC_FDC_FDD0_ADR\x88XPCK
\x04\x00\xa1\x0cp
\x0f\x88XPCK
\x00\x00\xa0N\x04\x93`
\x10p\\/\x06_SB_PCI0LPC_EC__BAT1_HID\x88XPCK
\x02\x00p\\/\x06_SB_PCI0LPC_EC__BAT1_UID\x88XPCK
\x04\x00p\x7f\\/\x04_SB_PCI0LPC_CSON
\x01\x00\x88XPCK
\x05\x00\xa4XPCK\x14H\x05XSTM\x01\x08XDMY\x11\x03
\x14\xa0\'\x93\\BIDE
\x03\\/\x05_SB_PCI0SATASCND_STMXDMYh
\x00\xa1\x1f\\/\x05_SB_PCI0IDE0PRIM_STMXDMYh
\x00\x14K\x04XGTF\x00\xa0%\x93\\BIDE
\x03\xa4\\/\x06_SB_PCI0SATASCNDMSTR_GTF\xa1\x1d\xa4\\/\x06_SB_PCI0IDE0PRIMMSTR_GTF\x10L\xe9\\_SB_[\x82C\xe9GDCK\x08_HID\x0c$M\x00y\x08_CID\x0cA\xd0\x0c\x15\x08DOID\x0c\xff\xff\xff\xff\x08DIDB\x0c\xff\xff\xff\xff\x08FLAG
\x00\x08WUCT
\x00\x08DHKE
\x00[\x02DEVT\x14?DSTA\x00UDCKUDKT\xa0\x1a\x91\x93GDID\x0c$M\x00L\x93GDID\x0c$M\x00Dp
\x0f`\xa1\x13\xa0\x0b\x92\\W98Fp
\x00`\xa1\x05p
\x0c`\xa4`\x14G\x07DPTS\x01\xa0O\x06\x90\x92\x95h
\x01\x92\x94h
\x04p
\x00DHKE\xa0\x18DFLG
\x02
\x02p
\x00DOIDDFLG
\x01
\x02\xa0/\x91\x93GDID\x0c$M\x00L\x93GDID\x0c$M\x00D\\/\x05_SB_PCI0LPC_EC__DDWK
\x01pGDIDDIDBDFLG
\x00\x0b\x00\x01\x08DDTM
\x00\x14J\x18DWAK\x01p\x0c\xff\xff\xff\xffDOID\xa0H\x17\x90\x92\x95h
\x01\x92\x94h
\x04p
\x00DDTM\xa0\x1d\x91\x93DIDB\x0c$M\x00L\x93DIDB\x0c$M\x00Dp
\x01DDTM\xa0B\rDDTMp
\x00DDTM\xa0\x12\x93GDID\x0c$M\x00Lp
\x01DDTM\xa0\x12\x93GDID\x0c$M\x00Dp
\x01DDTM\xa0;DDTM\xa0"\x93h
\x04\xa0\x1c{\\/\x04_SB_PCI0LPC_WAKR
\x08\x00DGPEyh
\x08DHKE\xa0
\x92DFLG
\x02
\x08\xa1B\x06p
\x01\\/\x04_SB_PCI0LPC_BUSD\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x01\\/\x05_SB_PCI0LPC_EC__DATT
\x01
\x00\x86\\._SB_GDCK
\x00\\DHDP
\x00\xa1;p
\x00DDTM\xa0\x12\x93GDID\x0c$M\x00Lp
\x01DDTM\xa0\x12\x93GDID\x0c$M\x00Dp
\x01DDTM\xa0\tDDTMWDCK\xa1\x02\xa3DFLG
\x01\x0b\x00\x01DFLG
\x01
\x02DFLG
\x01
\x10DFLG
\x01
 p
\x01\\/\x04_SB_PCI0LPC_DSCI\x14\x0eDGPE\x00DFLG
\x00
\x08\x14N\x12DDCK\x01\xa0J\x08h\\/\x05_SB_PCI0LPC_EC__LED_
\x08
\x00\\/\x05_SB_PCI0LPC_EC__LED_
\t
\x80\\/\x04_SB_PCI0LPC_LCON
\x01BCON
\x01\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x00\\/\x05_SB_PCI0LPC_EC__DATT
\x01
\x01\xa1H\t\\/\x05_SB_PCI0LPC_EC__LED_
\x08
\x80\\/\x05_SB_PCI0LPC_EC__LED_
\t
\xc0DFLG
\x00
\x02\\DHDP
\x00BCON
\x00\\/\x04_SB_PCI0LPC_LCON
\x00\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x01\\/\x05_SB_PCI0LPC_EC__DATT
\x01
\x00\xa4
\x01\x14C\x05DEJ0\x01\xa0\rhp
\x00DOIDUDKIDFLG
\x01
\x02\\/\x05_SB_PCI0LPC_EC__LED_
\x08
\x00\\/\x05_SB_PCI0LPC_EC__LED_
\t
\x00\x14\x1fDEJ3\x01\xa0\x18hDFLG
\x00
\x10\xa0\r\x93\\SPS_
\x03PDE3\x14\x11DEJ4\x01\xa0
hDFLG
\x00
 \x14\x07PDE3\x00\xa3\x08HIDE
\x00\x14@\x08WDCK\x00\xa0/\x91DFLG
\x02
\x10DFLG
\x02
 DDCK
\x01\xa0\x16\\W98FDFLG
\x00\x0b\x00\x02p
\x05HIDE\xa1H\x04\xa06\x93\\/\x05_SB_PCI0LPC_EC__BGID
\x00
\x11\\/\x05_SB_PCI0LPC_EC__NBRE
\x11\xa1\x0e\x86\\._SB_GDCK
\x00\x14O\x07UDCK\x00\xa0G\x07DFLG
\x02
\x08\xa0L\x06\x92DFLG
\x02\x0b\x00\x01}DHKE\x0b\x04 DHKE\xa0(\x93\\UOPT
\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQDHKE\xa0\x13\\W98F\x86\\._SB_GDCK
\x01\xa1\x0e\x86\\._SB_GDCK
\x03DFLG
\x01
\x08\x14\x1dUDKI\x00\xa0\r\\WNTFp
\x01WUCT\xa1\x08p
\x05WUCT\x14C\x05UDKT\x00\xa0K\x04{DHKE\x0b\x04 \x00\xa0?\x93GDID
\x00\xa06\x92vWUCTp
\x00DHKE\xa0\'\x93\\UOPT
\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x03@\x145GDID\x00\xa0)\x93DOID\x0c\xff\xff\xff\xffp\x0c\xff\xff\xff\xff\\/\x03_SB_GDCKG_IDpRDIDDOID\xa4DOID\x14O\tRDID\x00p
\x00`\xa0A\t\\/\x04_SB_PCI0LPC_EPWG\xa0@\x06\x93\\/\x03_SB_GDCKGGID
\x00\xa0B\x04\\H8DR\xa01\\/\x05_SB_PCI0LPC_EC__EEPRp\\/\x05_SB_PCI0LPC_EC__HDEP`\xa1\x08p\x0c$M\x00L`\xa1\x08p\\DCKI`\xa0\x1a\x93\\/\x03_SB_GDCKGGID
\x01p\x0c$M\x00D`\xa4`\x14G\x08RDSR\x00p
\x00`\xa0I\x07\x92\x93GDID
\x00\xa0F\x05\x93\\/\x03_SB_GDCKGGID
\x00\xa08\\H8DR\xa01\\/\x05_SB_PCI0LPC_EC__EEPRp\\/\x05_SB_PCI0LPC_EC__HDEN`\xa1\x08p\\DCKS`\xa0\x17\x93\\/\x03_SB_GDCKGGID
\x01p
\x00`\xa4`\x14E\x17BCON\x01p\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x02`\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x01}\\/\x04_SB_PCI0LPC_SERQ
@\\/\x04_SB_PCI0LPC_SERQ{\\/\x04_SB_PCI0LPC_SERQ
\x7f\\/\x04_SB_PCI0LPC_SERQp\\/\x04_SB_PCI0LPC_CLKRa\xa0.hp
\x00\\/\x04_SB_PCI0LPC_CLKRp
\x01\\/\x04_SB_PCI0LPC_BUSC\xa1\x17p
\x01\\/\x04_SB_PCI0LPC_BUSDpa\\/\x04_SB_PCI0LPC_CLKR}\\/\x04_SB_PCI0LPC_SERQ
\xc0\\/\x04_SB_PCI0LPC_SERQ{\\/\x04_SB_PCI0LPC_SERQ
\xbf\\/\x04_SB_PCI0LPC_SERQ\xa0 \x93`
\x00\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x00\x149DFLG\x02\xa0\x0f\x93h
\x00}FLAGiFLAG\xa0\x11\x93h
\x01{FLAG\x80i\x00FLAG\xa0\x0b{FLAGi\x00\xa4
\x01\xa1\x04\xa4
\x00\x10L\x1b\\/\x04_SB_PCI0LPC_EC__\x14A\x11_Q37\x00\xa0I\x10\\/\x04_SB_PCI0LPC_EPWGp\x0b\xd0\x07`["
dp\x0c\xff\xff\xff\xff\\/\x03_SB_GDCKG_ID\xa20\x90\x93\\/\x03_SB_GDCKGGID
\x07`["
\x01p\x0c\xff\xff\xff\xff\\/\x03_SB_GDCKG_IDv`p\x0c\xff\xff\xff\xff\\/\x03_SB_GDCKDOID\xa0O\x08\x91\x93\\/\x03_SB_GDCKGDID\x0c$M\x00L\x93\\/\x03_SB_GDCKGDID\x0c$M\x00D\xa06\x93\\/\x05_SB_PCI0LPC_EC__BGID
\x00
\x11\\/\x05_SB_PCI0LPC_EC__NBRE
\x11\xa1*\xa0(HPACp
\x00\\/\x04_SB_PCI0LPC_DRST\x86\\._SB_GDCK
\x00\x14D\tEEPR\x00p
\x00`\xa0F\x08\\H8DRp
\x00HDEOp

HDEC\xa2\x0e\x92{HDEC
\xc0\x00["
\x01\xa0A\x06\x92{HDEC
@\x00p
\x00apHDENbp
\x04c\xa2\x11cra{b
\xff\x00azb
\x08bvcpHDEPbp
\x04c\xa2\x11cra{b
\xff\x00azb
\x08bvcraHDEMaraHDESa\xa0\x0b\x92{a
\xff\x00p
\x01`\xa4`\x14D\x04_STA\x00pGGID`\xa0\x0b\x92\\W98Fp
\x00a\xa1\x05p
\x0ca\xa0\x1b\x91\x93`
\x00\x93`
\x01p\\/\x03_SB_GDCKDSTAa\xa0\x06\x93`
\x02\xa3\xa4a\x14H\x0c_INI\x00\\/\x05_SB_PCI0LPC_EC__DATT
\x02
\x01\xa0>\x93GGID
\x07\\/\x05_SB_PCI0LPC_EC__DATT
\x01
\x00\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x01\xa17\\/\x05_SB_PCI0LPC_EC__DATT
\x01
\x01\\/\x05_SB_PCI0LPC_EC__DATT
\x00
\x00\\/\x05_SB_PCI0LPC_EC__DDWK
\x00p
\x01\\/\x04_SB_PCI0LPC_DSCI\x14K\x04_DCK\x01p
\x00`\xa0"\x91\x93GGID
\x00\x93GGID
\x01p\\/\x03_SB_GDCKDDCKh`\xa0\x1a\\VIGD\\/\x04_SB_PCI0VID_VDSWh\xa4`\x08UDOP
\x00\x148_EJ0\x01\xa0\x18\x93GGID
\x00\\/\x03_SB_GDCKDEJ0h\xa0\x18\x93GGID
\x01\\/\x03_SB_GDCKDEJ0h\x148XEJ3\x01\xa0\x18\x93GGID
\x00\\/\x03_SB_GDCKDEJ3h\xa0\x18\x93GGID
\x01\\/\x03_SB_GDCKDEJ3h\x148_EJ4\x01\xa0\x18\x93GGID
\x00\\/\x03_SB_GDCKDEJ4h\xa0\x18\x93GGID
\x01\\/\x03_SB_GDCKDEJ4h\x146PEJ3\x00\xa0\x17\x93GGID
\x00\\/\x03_SB_GDCKPDE3\xa0\x17\x93GGID
\x01\\/\x03_SB_GDCKPDE3\x14A\x04_BDN\x00p
\x00`\xa0\x19\x93GGID
\x00p\\/\x03_SB_GDCKRDID`\xa0\x19\x93GGID
\x01p\\/\x03_SB_GDCKRDID`\xa4`\x14A\x04_UID\x00p
\x00`\xa0\x19\x93GGID
\x00p\\/\x03_SB_GDCKRDSR`\xa0\x19\x93GGID
\x01p\\/\x03_SB_GDCKRDSR`\xa4`\x14.GPTS\x01\\/\x03_SB_GDCKDPTSh\\/\x05_SB_PCI0LPC_EC__RPTSh\x14H\x04GWAK\x01\\/\x03_SB_GDCKDWAKh\\/\x05_SB_PCI0LPC_EC__RWAKh\\/\x05_SB_PCI0LPC_EC__DDWK
\x00\x146GGPE\x00\xa0\x17\x93GGID
\x00\\/\x03_SB_GDCKDGPE\xa0\x17\x93GGID
\x01\\/\x03_SB_GDCKDGPE\x08G_ID\x0c\xff\xff\xff\xff\x14N\x06GGID\x00pG_ID`\xa0N\x05\x93`\x0c\xff\xff\xff\xffp\\/\x04_SB_PCI0LPC_DKI0`p\\/\x04_SB_PCI0LPC_DKI1ap\\/\x04_SB_PCI0LPC_DKI2b}`ya
\x01\x00`}`yb
\x02\x00`p`G_ID\xa4`\x10F*\\/\x04_SB_PCI0LPC_EC__\x14M\x07_Q50\x00p\\/\x03_SB_GDCKGGID`\xa0D\x06\x91\x93`
\x00\x93`
\x01\\/\x05_SB_PCI0LPC_EC__LED_
\x08
\x80\\/\x05_SB_PCI0LPC_EC__LED_
\t
\xc0\xa0\x13\\W98F\x86\\._SB_GDCK
\x01\xa1\x0e\x86\\._SB_GDCK
\x03\x14F\x19DATT\x02p
\x00`\xa0B\x08\x93h
\x00\xa0(\x93i
\x01\xa0\x11\\H8DR}HAM6
\x80HAM6\xa1\x0c\\MBEC
\x16
\xff
\x80p
\x01`\xa0$\x93i
\x00\xa0\x11\\H8DR{HAM6
\x7fHAM6\xa1\x0c\\MBEC
\x16
\x7f
\x00\xa0-\x93i
\x02\xa0\x14\\H8DR\xa0\r{HAM6
\x80\x00p
\x01`\xa1\x12\xa0\x10{\\RBEC
\x16
\x80\x00p
\x01`\xa0B\x08\x93h
\x01\xa0(\x93i
\x01\xa0\x11\\H8DR}HAMA
\x01HAMA\xa1\x0c\\MBEC
\x1a
\xff
\x01p
\x01`\xa0$\x93i
\x00\xa0\x11\\H8DR{HAMA
\xfeHAMA\xa1\x0c\\MBEC
\x1a
\xfe
\x00\xa0-\x93i
\x02\xa0\x14\\H8DR\xa0\r{HAMA
\x01\x00p
\x01`\xa1\x12\xa0\x10{\\RBEC
\x1a
\x01\x00p
\x01`\xa0B\x08\x93h
\x02\xa0(\x93i
\x01\xa0\x11\\H8DR}HAMB
\x01HAMB\xa1\x0c\\MBEC
\x1b
\xff
\x01p
\x01`\xa0$\x93i
\x00\xa0\x11\\H8DR{HAMB
\xfeHAMB\xa1\x0c\\MBEC
\x1a
\xfe
\x00\xa0-\x93i
\x02\xa0\x14\\H8DR\xa0\r{HAMB
\x01\x00p
\x01`\xa1\x12\xa0\x10{\\RBEC
\x1b
\x01\x00p
\x01`\xa4`\x14K\x07DDWK\x01p
\x00`\xa0#\x93h
\x01\xa0\x0c\\H8DRp\x01HWDK\xa1\x0c\\MBEC
2
\xff
\x08p
\x01`\xa0\x1f\x93h
\x00\xa0\x0c\\H8DRp\x00HWDK\xa1\x0c\\MBEC
2
\xf7
\x00\xa0)\x93h
\x02\xa0\x10\\H8DR\xa0\tHWDKp
\x01`\xa1\x12\xa0\x10{\\RBEC
2
\x08\x00p
\x01`\xa4`\x10#\\/\x04_SB_PCI0LPC_EC__\x14\x07RPTS\x01\xa3\x14\x06RWAK\x01\x10B\x10\\/\x04_SB_PCI0LPC_EC__\x14N\x04_Q1C\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00 \x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x16\x10\\UCMS
\x00\x14N\x04_Q1D\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x10\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x15\x10\\UCMS
\x01\x14N\x04_Q1E\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00@\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x17\x10\\UCMS
\x02\x10L\x13\\/\x04_SB_PCI0LPC_EC__\x08BRTF
\x01\x14N\x08_Q14\x00\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0b\x00\x80\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x10\x10\xa0?\\NBCF\xa0\x1c\\VIGD\x86\\/\x04_SB_PCI0VID_LCD0
\x86\xa1\x1b\x86\\/\x05_SB_PCI0AGP_VID_LCD0
\x86\xa1\x08\\UCMS
\x04\x14@\t_Q15\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x01\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x11\x10\xa0?\\NBCF\xa0\x1c\\VIGD\x86\\/\x04_SB_PCI0VID_LCD0
\x87\xa1\x1b\x86\\/\x05_SB_PCI0AGP_VID_LCD0
\x87\xa1\x08\\UCMS
\x05\x10D\x06\\/\x04_SB_PCI0LPC_EC__\x14N\x04_Q19\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x80\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x18\x10\\UCMS
\x03\x10D\x06\\/\x04_SB_PCI0LPC_EC__\x14N\x04_Q63\x00\xa0?\\/\x06_SB_PCI0LPC_EC__HKEYMHKK\x0c\x00\x00\x08\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x14\x10\\UCMS
\x0b\x10C\t\\/\x04_SB_PCI0LPC_EC__\x14
_Q70\x00FNST\x14
_Q72\x00FNST\x14
_Q73\x00FNST\x14L\x05FNST\x00\xa0\x12\\H8DRpHFNS`pHFNEa\xa1\x17{\\RBEC
\x0e
\x03`{\\RBEC
\x00
\x08a\xa0)a\xa0\x0c\x93`
\x00\\UCMS
\x11\xa0\x0c\x93`
\x01\\UCMS
\x0f\xa0\x0c\x93`
\x02\\UCMS
\x10\x10I<\\/\x05_SB_PCI0LPC_EC__HKEY\x08WGFL
\x00\x14\x1eWLSW\x00\xa4\\/\x05_SB_PCI0LPC_EC__GSTS\x14B\x04GWAN\x00p
\x00`\xa0\x0e{WGFL
\x01\x00}`
\x01`\xa0\x0b{WGFL
\x08\x00\xa4`\xa0
WPWS}`
\x02`\xa0\x0e{WGFL
\x04\x00}`
\x04`\xa4`\x14K\x04SWAN\x01\xa0\x0c{h
\x02\x00WPWC
\x01\xa1\x07WPWC
\x00\xa0\x18{h
\x04\x00}WGFL
\x04WGFL\\WGSV
\x02\xa1\x15{WGFL\x80
\x04\x00WGFL\\WGSV
\x03\x14B\x04GBDC\x00p
\x00`\xa0\x0e{WGFL
\x10\x00}`
\x01`\xa0\x0b{WGFL
\x80\x00\xa4`\xa0
BPWS}`
\x02`\xa0\x0e{WGFL
@\x00}`
\x04`\xa4`\x14K\x04SBDC\x01\xa0\x0c{h
\x02\x00BPWC
\x01\xa1\x07BPWC
\x00\xa0\x18{h
\x04\x00}WGFL
@WGFL\\BLTH
\x02\xa1\x15{WGFL\x80
@\x00WGFL\\BLTH
\x03\x14;WPWS\x00\xa0\x1f\\H8DRp\\/\x05_SB_PCI0LPC_EC__DCWW`\xa1\x12pz{\\RBEC
:
@\x00
\x06\x00`\xa4`\x14\x19WTGL\x00\xa0\x12{WGFL
\x01\x00WPWC\x92WPWS\x14B\tWPWC\x01\xa0N\x04\x90h\x90{WGFL
\x01\x00\x92{WGFL
\x08\x00\xa0\x1f\\H8DRp\x01\\/\x05_SB_PCI0LPC_EC__DCWW\xa1\x0c\\MBEC
:
\xff
@}WGFL
\x02WGFL\xa1;\xa0\x1f\\H8DRp\x00\\/\x05_SB_PCI0LPC_EC__DCWW\xa1\x0c\\MBEC
:
\xbf
\x00{WGFL\x80
\x02\x00WGFL\x14;BPWS\x00\xa0\x1f\\H8DRp\\/\x05_SB_PCI0LPC_EC__DCBD`\xa1\x12pz{\\RBEC
:
\x10\x00
\x04\x00`\xa4`\x14\x19BTGL\x00\xa0\x12{WGFL
\x10\x00BPWC\x92BPWS\x14B\tBPWC\x01\xa0N\x04\x90h\x90{WGFL
\x10\x00\x92{WGFL
\x80\x00\xa0\x1f\\H8DRp\x01\\/\x05_SB_PCI0LPC_EC__DCBD\xa1\x0c\\MBEC
:
\xff
\x10}WGFL
 WGFL\xa1;\xa0\x1f\\H8DRp\x00\\/\x05_SB_PCI0LPC_EC__DCBD\xa1\x0c\\MBEC
:
\xef
\x00{WGFL\x80
 \x00WGFL\x14;WGIN\x00p
\x00WGFLp\\WGSV
\x01WGFL\xa0\x10WPWS}WGFL
\x02WGFL\xa0\x10BPWS}WGFL
 WGFL\x146WGPS\x01\xa0\r\x92\x95h
\x04\\BLTH
\x05\xa0\x10\x92{WGFL
\x04\x00WPWC
\x00\xa0\x10\x92{WGFL
@\x00BPWC
\x00\x14&WGWK\x01\xa0\x0f{WGFL
 \x00BPWC
\x01\xa0\x0f{WGFL
\x02\x00WPWC
\x01\x109\\/\x04_SB_PCI0LPC_EC__\x14$_Q41\x00\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b\x00p\x10N\x0b\\_SB_[\x82E\x0bLCIO\x08_HID\x0c0\xae\x00\x01\x08_CID\x0cA\xd0
\x06\x14)_STA\x00\xa0\x1d\\/\x05_SB_PCI0LPC_EC__BSTA
\x03\xa4
\x0f\xa1\x04\xa4
\x00\x14@\x07_EJ0\x01\\/\x05_SB_PCI0LPC_EC__BEJ0h\xa0@\x05\x91\x93\\/\x03_SB_GDCKGDID\x0c$M\x00L\x93\\/\x03_SB_GDCKGDID\x0c$M\x00Dp
\x00\\/\x04_SB_PCI0LPC_DRST\x86\\._SB_GDCK
\x00\x10#\\/\x04_SB_PCI0EXP3EXPD\x08_EJD\r_SB.GDCK\x00\x10\'\\/\x05_SB_PCI0USB2URTHUPDK\x08_EJD\r_SB.GDCK\x00\x10\'\\/\x05_SB_PCI0USB7URTHUPDK\x08_EJD\r_SB.GDCK\x00\x101\\/\x05_SB_PCI0USB1URTHUPEX\x08_EJD\r_SB.PCI0.EXP2.EXUP\x00\x101\\/\x05_SB_PCI0USB7URTHUPEX\x08_EJD\r_SB.PCI0.EXP2.EXUP\x00\x102\\/\x04_SB_PCI0EXP2EXUP\x08_EJD\r_SB.PCI0.USB7.URTH.UPEX\x00\x08\\_S0_\x12
\x04
\x00
\x00
\x00
\x00\x08\\_S3_\x12
\x04
\x05
\x05
\x00
\x00\x08\\_S4_\x12
\x04
\x06
\x06
\x00
\x00\x08\\_S5_\x12
\x04
\x07
\x07
\x00
\x00\x14O\x1d\\_PTS\x01p
\x01`\xa0\x0c\x93h\\SPS_p
\x00`\xa0\x0f\x91\x93h
\x00\x92\x95h
\x06p
\x00`\xa0E\x1b`ph\\SPS_\\/\x06_SB_PCI0LPC_EC__HKEYMHKE
\x00\xa0\x1f\\/\x05_SB_PCI0LPC_EC__KBLT\\UCMS
\r\xa0G\x05\x93h
\x01p\\/\x05_SB_PCI0LPC_EC__HFNI\\FNIDp
\x00\\/\x05_SB_PCI0LPC_EC__HFNIp
\x00\\/\x05_SB_PCI0LPC_EC__HFSP\xa02\x93h
\x03\\VVPD
\x03\\TRAPp\\/\x06_SB_PCI0LPC_EC__AC___PSR\\ACST\xa0)\x93h
\x04\\/\x03_SB_SLPB_PSW
\x00\xa0\r\\SPEN\\STEP
\x07\\TRAP\xa0
\x93h
\x05\\TRAP\\/\x05_SB_PCI0LPC_EC__BPTSh\xa0 \x92\x95h
\x04p
\x00\\/\x05_SB_PCI0LPC_EC__HWLB\xa1\x1bp
\x01\\/\x05_SB_PCI0LPC_EC__HWLB\xa0<\x92\x93h
\x05p
\x01\\/\x05_SB_PCI0LPC_EC__HCMU\\/\x03_SB_GDCKGPTSh\xa0\x0b\\W98F\\CBRI\\/\x06_SB_PCI0LPC_EC__HKEYWGPSh\x08WAKI\x12\x06\x02
\x00
\x00\x14AJ\\_WAK\x01\xa0\x10\x91\x93h
\x00\x92\x95h
\x05\xa4WAKIp
\x00\\SPS_p
\x00\\/\x05_SB_PCI0LPC_EC__HCMUp
\x80\\/\x05_SB_PCI0LPC_EC__HFSP\\/\x05_SB_PCI0LPC_EC__EVNT
\x01\\/\x06_SB_PCI0LPC_EC__HKEYMHKE
\x01\\/\x05_SB_PCI0LPC_EC__FNST\\UCMS
\rp
\x00\\LIDB\xa0"\x93h
\x01p\\/\x05_SB_PCI0LPC_EC__HFNI\\FNID\xa0A\x13\x93h
\x03THRM
\x00\xa0E\x04\\WXPFp
\x00\\/\x04_SB_PCI0LPC_C4C3\xa0\'\\OSC4\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81\xa1D\x07\xa0A\x07\\WNTF\xa02\\/\x06_SB_PCI0LPC_EC__AC___PSRp
\x00\\/\x04_SB_PCI0LPC_C4C3\xa16\xa0\x1c\\CWASp
\x00\\/\x04_SB_PCI0LPC_C4C3\xa1\x17p
\x01\\/\x04_SB_PCI0LPC_C4C3\xa0:\x92\x93\\ACST\\/\x06_SB_PCI0LPC_EC__AC___PSR\\/\x05_SB_PCI0LPC_EC__ATMC\xa0.\x90{\\CFGD\x0c\x00\x00\x00\x01\x00\x90\\WXPF\x92\x94\\WSPV
\x01\xa0\x10{\\CFGD
\xf0\x00PPMS
\x00\xa0@\x13\x93h
\x04\xa0\x0bDTSETHRM
\x02\xa0\x13\\W98F\x86\\._SB_SLPB
\x02\xa0\x1f\\WMEF\\/\x05_SB_PCI0LPC_EC__BEEP
\x05\xa0!\x92\\W98Fp
\x00\\/\x05_SB_PCI0LPC_EC__HSPA\xa0.\\WXPF\xa0\'\\OSC4\x86\\._PR_CPU0
\x81\xa0\x13\\MPEN\x86\\._PR_CPU1
\x81\xa1D\x04\xa0A\x04\\WNTF\xa09\x91\x92\\/\x06_SB_PCI0LPC_EC__AC___PSR\\CWASp
\x00\\/\x04_SB_PCI0LPC_C4C3\xa0\r\\SPEN\\STEP
\x08\\/\x05_SB_PCI0LPC_EC__ATMC\xa0.\x90{\\CFGD\x0c\x00\x00\x00\x01\x00\x90\\WXPF\x92\x94\\WSPV
\x01\xa0\x10{\\CFGD
\xf0\x00PPMS
\x00\xa0O\x07\x7f\\/\x04_SB_PCI0EXP2PDS_\\/\x04_SB_PCI0EXP2XCPF\x00\xa0*\\/\x04_SB_PCI0EXP2PDS_p
\x01\\/\x04_SB_PCI0EXP2XCPF\xa1\x17p
\x00\\/\x04_SB_PCI0EXP2XCPF\x86\\/\x03_SB_PCI0EXP2
\x00\\/\x03_SB_GDCKGWAKh\\/\x05_SB_PCI0LPC_EC__BWAKh\\/\x06_SB_PCI0LPC_EC__HKEYWGWKh\x86\\._TZ_THM0
\x80\x86\\._TZ_THM1
\x80\\VSLD\\/\x03_SB_LID__LID\xa01\x90\\W98F\x92\\WMEF\x86\\/\x03_SB_PCI0USB0
\x00\x86\\/\x03_SB_PCI0USB1
\x00\xa09\x95h
\x04\xa03{\\RRBF
\x02\x00yh
\x08`p}\x0b\x13 `\x00`\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ`p\x00\\RRBF\xa4WAKI\x10@\x1e\\_SI_\x14H\x1d_SST\x01\xa0;\x93h
\x00\\/\x05_SB_PCI0LPC_EC__LED_
\x00
\x00\\/\x05_SB_PCI0LPC_EC__LED_
\x07
\x00\xa0B\x06\x93h
\x01\xa0%\x91\\SPS_\\WNTF\\/\x05_SB_PCI0LPC_EC__BEEP
\x05\\/\x05_SB_PCI0LPC_EC__LED_
\x00
\x80\\/\x05_SB_PCI0LPC_EC__LED_
\x07
\x00\xa0;\x93h
\x02\\/\x05_SB_PCI0LPC_EC__LED_
\x00
\x80\\/\x05_SB_PCI0LPC_EC__LED_
\x07
\xc0\xa0K\x0b\x93h
\x03\xa0"\x94\\SPS_
\x03\\/\x05_SB_PCI0LPC_EC__BEEP
\x07\xa1O\x04\xa01\x93\\SPS_
\x03\\/\x05_SB_PCI0LPC_EC__BEEP
\x03\\/\x03_SB_GDCKPEJ3\xa1\x1a\\/\x05_SB_PCI0LPC_EC__BEEP
\x04\xa0\t\x93\\SPS_
\x03\xa1\x1c\\/\x05_SB_PCI0LPC_EC__LED_
\x00
\x80\\/\x05_SB_PCI0LPC_EC__LED_
\x07
\xc0\xa09\x93h
\x04\\/\x05_SB_PCI0LPC_EC__BEEP
\x03\\/\x05_SB_PCI0LPC_EC__LED_
\x07
\xc0\x10C<\\_GPE[\x01MGPE\x07\x14N
_L18\x00p\\/\x05_SB_PCI0LPC_EC__HWAK`p`\\RRBF["

\xa0\x06{`
\x02\x00\xa0){`
\x04\x00\xa0\x13\\W98F\x86\\._SB_SLPB
\x02\xa1\x0e\x86\\._SB_LID_
\x02\xa0"{`
\x08\x00\\/\x03_SB_GDCKGGPE\x86\\._SB_SLPB
\x02\xa0\x13{`
\x10\x00\x86\\._SB_SLPB
\x02\xa0\x06{`
@\x00\xa0\x13{`
\x80\x00\x86\\._SB_SLPB
\x02\x14K\x0f_L09\x00\xa0<\\/\x04_SB_PCI0EXP0PSP0p
\x01\\/\x04_SB_PCI0EXP0PSP0\x86\\/\x03_SB_PCI0EXP0
\x02\xa0<\\/\x04_SB_PCI0EXP1PSP1p
\x01\\/\x04_SB_PCI0EXP1PSP1\x86\\/\x03_SB_PCI0EXP1
\x02\xa0<\\/\x04_SB_PCI0EXP2PSP2p
\x01\\/\x04_SB_PCI0EXP2PSP2\x86\\/\x03_SB_PCI0EXP2
\x02\xa0<\\/\x04_SB_PCI0EXP3PSP3p
\x01\\/\x04_SB_PCI0EXP3PSP3\x86\\/\x03_SB_PCI0EXP3
\x02\x14D\x17_L01\x00\xa0@\x0f\\/\x04_SB_PCI0EXP2HPCSp
\x01\\/\x04_SB_PCI0EXP2HPCS\xa0*\\/\x04_SB_PCI0EXP2ABP_p
\x01\\/\x04_SB_PCI0EXP2ABP_\xa0I\t\\/\x04_SB_PCI0EXP2PDC_p
\x01\\/\x04_SB_PCI0EXP2PDC_p
\x00\\/\x04_SB_PCI0EXP2XCPF\x86\\/\x03_SB_PCI0EXP2
\x00\xa0E\x04\\/\x04_SB_PCI0EXP2PDS_p
\x01\\/\x04_SB_PCI0EXP2XCPF["
d\x86\\/\x04_SB_PCI0EXP2EXUP
\x01[#MGPE\xff\xff\xa0M\x06\\/\x04_SB_PCI0EXP0HPCSp
\x01\\/\x04_SB_PCI0EXP0HPCS\xa0A\x04\\/\x04_SB_PCI0EXP0PDC_["
\xc8p
\x01\\/\x04_SB_PCI0EXP0PDC_\x86\\/\x03_SB_PCI0EXP0
\x00[\'MGPE\x14D\t_L02\x00p
\x00\\/\x04_SB_PCI0LPC_SWGE\xa0A\x04\\/\x06_SB_PCI0LPC_EC__HKEYDHKC\xa0#DT02\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b"`\x86\\._TZ_THM1
\x80\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\x10@(\\/\x05_SB_PCI0LPC_EC__HKEY\x14H\x0cMHQT\x01\xa0M\x0b\x90\\WNTF\\TATC\xa0\x0e\x93h
\x00p\\TATC`\xa4`\xa1@
\xa0K\x04\x93h
\x01p\\TDFA`r`y\\TDTA
\x04\x00`r`y\\TDFD
\x08\x00`r`y\\TDTD
\x0c\x00`r`y\\TNFT
\x10\x00`r`y\\TNTT
\x14\x00`\xa4`\xa1A\x05\xa02\x93h
\x02p\\TCFA`r`y\\TCTA
\x04\x00`r`y\\TCFD
\x08\x00`r`y\\TCTD
\x0c\x00`\xa4`\xa1\x1b\xa0\x05\x93h
\x03\xa1\x13\xa0\x0e\x93h
\x04p\\TATW`\xa4`\xa1\x02\xa3\xa4
\x00\x14O\x07MHAT\x01\xa0D\x07\x90\\WNTF\\TATCp{h
\xff\x00`\xa0
\x92ATMV`\xa4
\x00p{zh
\x08\x00
\xff\x00`\xa0
\x92ATMV`\xa4
\x00p{h
\x0f\x00\\TCFAp{zh
\x04\x00
\x0f\x00\\TCTAp{zh
\x08\x00
\x0f\x00\\TCFDp{zh
\x0c\x00
\x0f\x00\\TCTDATMC\xa4
\x01\xa4
\x00\x14@\rMHGT\x01\xa0E\x0c\x90\\WNTF\\TATCpy\\TSFT
\x10\x00`r`y\\TSTT
\x14\x00`p{h
\xff\x00a\xa0\x0b\x92ATMVa\xa4\x0b\xff\xffp{h
\x0f\x00a\xa0\r\x93a
\x00r`\\TIF0`\xa1&\xa0\r\x93a
\x01r`\\TIF1`\xa1\x16\xa0\r\x93a
\x02r`\\TIF2`\xa1\x06r`
\xff`p{zh
\x04\x00
\x0f\x00a\xa0\x11\x93a
\x00r`y\\TIT0
\x08\x00`\xa12\xa0\x11\x93a
\x01r`y\\TIT1
\x08\x00`\xa1\x1e\xa0\x11\x93a
\x02r`y\\TIT2
\x08\x00`\xa1
r`y
\xff
\x08\x00`\xa4`\xa4
\x00\x14L\x04ATMV\x01p{h
\x0f\x00ap\\TNFT`\xa0\x08\x92\x95a`\xa4
\x00p{zh
\x04\x00
\x0f\x00bp\\TNTT`\xa0\x08\x92\x95b`\xa4
\x00\xa0\x0f\\TATL\xa0\x08\x7fab\x00\xa4
\x00\xa4
\x01\x10F\x10\\/\x04_SB_PCI0LPC_EC__\x14@\x0fATMC\x00\xa0H\x0e\x90\\WNTF\\TATC\xa0E\x06HPACp\\TCFA`p\\TCTAap}ya
\x04\x00`\x00ATMX\xa0\x1f\x93\\TCTA
\x00p\\TCR0\\TCRTp\\TPS0\\TPSV\xa1#\xa0\x1f\x93\\TCTA
\x01p\\TCR1\\TCRTp\\TPS1\\TPSV\xa1\x01\xa1A\x06p\\TCFD`p\\TCTDap}ya
\x04\x00`\x00ATMX\xa0\x1f\x93\\TCTD
\x00p\\TCR0\\TCRTp\\TPS0\\TPSV\xa1#\xa0\x1f\x93\\TCTD
\x01p\\TCR1\\TCRTp\\TPS1\\TPSV\xa1\x01\x86\\._TZ_THM0
\x81THRM
\x01\x10C\x1f\\_TZ_[\x85L\x0cTHM0\x14\r_CRT\x00\xa4C2K_
\x7f\x14G\x0b_TMP\x00\xa0B\x05\\H8DRp\\/\x05_SB_PCI0LPC_EC__TMP0`p\\/\x05_SB_PCI0LPC_EC__HT12ap\\/\x05_SB_PCI0LPC_EC__HT13b\xa1$p\\RBEC
x`p{\\RBEC
 
@\x00ap{\\RBEC
 
\x80\x00b\xa0\tb\xa4C2K_
\x80\xa0\'\x92\\/\x06_SB_PCI0LPC_EC__HKEYDHKC\xa0\ta\xa4C2K_
\x80\xa4C2K_`[\x85@\x0fTHM1\x143_PSL\x00\xa0\x1e\\MPEN\xa4\x12\x16\x02\\._PR_CPU0\\._PR_CPU1\xa4\x12\x0c\x01\\._PR_CPU0\x14\x0c_CRT\x00\xa4\\TCRT\x14\x0c_PSV\x00\xa4\\TPSV\x14\x0c_TC1\x00\xa4\\TTC1\x14\x0c_TC2\x00\xa4\\TTC2\x14\x0c_TSP\x00\xa4\\TTSP\x14D\x07_TMP\x00\xa0J\x05\\DTSETHRM
\x02pDTS1`\xa0\x11\x92\x95DTS0DTS1pDTS0`\xa0.\x92\\/\x06_SB_PCI0LPC_EC__HKEYDHKC\xa0\x10DT02r\\TCRT
\x01`\xa4`\xa4C2K_`\xa1\x11\xa4\\/\x03_TZ_THM0_TMP\x14+C2K_\x01rwh

\x00\x0b\xac
`\xa0\x0c\x92\x94`\x0b\xac
p\x0b\xb8\x0b`\xa0\x0b\x94`\x0b\xac\x0fp\x0b\xb8\x0b`\xa4`\x10O\x13\\/\x04_SB_PCI0LPC_EC__\x14I\x12_Q40\x00\x86\\._TZ_THM0
\x80\xa08\\H8DRp\\/\x05_SB_PCI0LPC_EC__HT11`p\\/\x05_SB_PCI0LPC_EC__HT12a\xa1\x1bp{\\RBEC
 
 \x00`p{\\RBEC
 
@\x00a\xa0=\\/\x06_SB_PCI0LPC_EC__HKEYDHKC\xa0 a\\/\x06_SB_PCI0LPC_EC__HKEYMHKQ\x0b"`\xa0\x06VIGD\xa3\xa1\x06\\VTHR\xa0C\x07\\SPEN\xa0\'\\OSPX\x86\\._PR_CPU0
\x80\xa0\x13\\MPEN\x86\\._PR_CPU1
\x80\xa1C\x04\xa07\x91\\/\x05_SB_PCI0LPC_EC__HT00\\/\x05_SB_PCI0LPC_EC__HT10\\STEP
\t\xa1\x08\\STEP

\x08GPIC
\x00\x14\r_PIC\x01ph\\GPIC[\x80MNVS\x00\x0c\x00@\xef\x7f\x0b\x00\x10[\x81B+MNVS\x03\x00\x80\x80\x07GAPA GAPL DCKI DCKS VCDL\x01VCDC\x01VCDT\x01VCDD\x01VIGD\x01VCSS\x01VCDB\x01VCIN\x01VPDF\x01\x00\x07VLID\x04VVPO\x04\x00\x08CDFL\x08CDAH\x08PMOD\x02PDIR\x01PDMA\x01\x00\x04LFDC\x01\x00\x07C2NA\x01C3NA\x01C4NA\x01\x00\x05SPEN\x01\x00\x01\x00\x01\x00\x01MPEN\x01\x00\x03OSPX\x01OSC4\x01OSSS\x01NHPS\x01NPME\x01\x00\x03UOPT\x08BTID LWST\x08LPST\x08TCRT\x10TPSV\x10TTC1\x10TTC2\x10TTSP\x10SRAH\x08SRHE\x08SRE0\x08SRE1\x08SRE2\x08SRE3\x08SRE4\x08SRE5\x08SRU0\x08SRU1\x08SRU2\x08SRU3\x08SRU7\x08SRPB\x08SRLP\x08SRSA\x08SRSM\x08CWAC\x01CWAS\x01\x00\x06CWAP\x10CWAT\x10DBGC\x01\x00\x07FS1L\x10FS1M\x10FS1H\x10FS2L\x10FS2M\x10FS2H\x10FS3L\x10FS3M\x10FS3H\x10TATC\x01\x00\x06TATL\x01TATW\x08TNFT\x04TNTT\x04TDFA\x04TDTA\x04TDFD\x04TDTD\x04TCFA\x04TCTA\x04TCFD\x04TCTD\x04TSFT\x04TSTT\x04TIT0\x08TCR0\x10TPS0\x10TIT1\x08TCR1\x10TPS1\x10TIT2\x08TCR2\x10TPS2\x10TIF0\x08TIF1\x08TIF2\x08\x00 TCZ1\x08TCZ2\x08TCZ3\x08BTHI\x01\x00\x07HDIR\x01HDEH\x01\x00\x06TPMP\x01TPMS\x01\x00\x06BIDE\x08\x00\x01DTSE\x01\x00\x06DTS0\x08DTS1\x08DT00\x01DT01\x01DT02\x01DT03\x01\x00\x04PH01\x08PH02\x08PH03\x08PH04\x08PH05\x08PH06\x08PH07\x08PH08\x08PH09\x08PH0A\x08PH0B\x08LIDB\x01\x00\x07TCG0\x01\x00\x07[\x81\x13MNVS\x01\x00\x80\x00\x07DDC1@@\x00@@[\x81\x10MNVS\x01\x00\x80\x00\x07DDC2@\x80[\x80SMI0\x01
\xb2
\x01[\x81\x0bSMI0\x01APMC\x08[\x81(MNVS\x00\x00\x80\xe0\x07CMD_\x08ERR_ PAR0 PAR1 PAR2 PAR3 [\x01MSMI\x07\x14F\x05SMI_\x05[#MSMI\xff\xffphCMD_piPAR0pjPAR1pkPAR2plPAR3p
\xf5APMC\xa2\x13\x93ERR_
\x01["
dp
\xf5APMCpPAR0`[\'MSMI\xa4`\x14\x14RPCI\x01\xa4SMI_
\x00
\x00h
\x00
\x00\x14\x12WPCI\x02SMI_
\x00
\x01hi
\x00\x14\x11MPCI\x03SMI_
\x00
\x02hij\x14\x14RBEC\x01\xa4SMI_
\x00
\x03h
\x00
\x00\x14\x12WBEC\x02SMI_
\x00
\x04hi
\x00\x14\x11MBEC\x03SMI_
\x00
\x05hij\x14\x14RISA\x01\xa4SMI_
\x00
\x06h
\x00
\x00\x14\x12WISA\x02SMI_
\x00
\x07hi
\x00\x14\x11MISA\x03SMI_
\x00
\x08hij\x14\x14VEXP\x00SMI_
\x01
\x00
\x00
\x00
\x00\x14\x13VUPS\x01SMI_
\x01
\x01h
\x00
\x00\x14\x12VSDS\x02SMI_
\x01
\x02hi
\x00\x14\x14VDDC\x00SMI_
\x01
\x03
\x00
\x00
\x00\x14\x13VVPD\x01SMI_
\x01
\x04h
\x00
\x00\x14\x13VNRS\x01SMI_
\x01
\x05h
\x00
\x00\x14\x15GLPW\x00\xa4SMI_
\x01
\x06
\x00
\x00
\x00\x14\x13VSLD\x01SMI_
\x01
\x07h
\x00
\x00\x14\x14VEVT\x01\xa4SMI_
\x01
\x08h
\x00
\x00\x14\x15VTHR\x00\xa4SMI_
\x01
\t
\x00
\x00
\x00\x14\x13VBRC\x01SMI_
\x01

h
\x00
\x00\x14\x15VBRG\x00\xa4SMI_
\x01
\x0e
\x00
\x00
\x00\x14\x14UCMS\x01\xa4SMI_
\x02h
\x00
\x00
\x00\x14\x13BHDP\x02\xa4SMI_
\x03
\x00hi
\x00\x14\x14DHDP\x01\xa4SMI_
\x03
\x01h
\x00
\x00\x14\x13STEP\x01SMI_
\x04h
\x00
\x00
\x00\x14\x14TRAP\x00SMI_
\x05
\x00
\x00
\x00
\x00\x14\x14CBRI\x00SMI_
\x05
\x01
\x00
\x00
\x00\x14"DSPD\x01\xa0\x18\x92\x94h
\x01rh
\x02hSMI_
\x05h
\x00
\x00
\x00\xa1\x02\xa3\x14\x14BLTH\x01\xa4SMI_
\x06h
\x00
\x00
\x00\x14\x14FISP\x00SMI_
\x07
\x00
\x00
\x00
\x00\x14\x13ATCC\x01SMI_
\x08h
\x00
\x00
\x00\x14\x14WGSV\x01\xa4SMI_
\th
\x00
\x00
\x00\x14\x14THRM\x01\xa4SMI_

h
\x00
\x00
\x00\x14\x14PPMS\x01\xa4SMI_
\x0bh
\x00
\x00
\x00\x14\x13TPHY\x01SMI_
\x0ch
\x00
\x00
\x00\x14\x13TMOR\x01SMI_
\rh
\x00
\x00
\x00\x143DPIO\x02\xa0\x06\x92h\xa4
\x00\xa0\x08\x94h
\xf0\xa4
\x00\xa0\x10\x94h
\xb4\xa0\x05i\xa4
\x02\xa1\x04\xa4
\x01\xa0\x08\x94h
x\xa4
\x03\xa4
\x04\x14=DUDM\x02\xa0\x06\x92i\xa4
\xff\xa0\x08\x94h
Z\xa4
\x00\xa0\x08\x94h
<\xa4
\x01\xa0\x08\x94h
-\xa4
\x02\xa0\x08\x94h
\x1e\xa4
\x03\xa0\x08\x94h
\x14\xa4
\x04\xa4
\x05\x14(DMDM\x02\xa0\x05i\xa4
\x00\xa0\x06\x92h\xa4
\x00\xa0\x08\x94h
\x96\xa4
\x01\xa0\x08\x94h
x\xa4
\x02\xa4
\x03\x14A\x05UUDM\x02\xa0
\x92{h
\x04\x00\xa4
\x00\xa0\t{i
 \x00\xa4
\x14\xa0\t{i
\x10\x00\xa4
\x1e\xa0\t{i
\x08\x00\xa4
-\xa0\t{i
\x04\x00\xa4
<\xa0\t{i
\x02\x00\xa4
Z\xa0\t{i
\x01\x00\xa4
x\xa4
\x00\x14H\x04UMDM\x04\xa0
\x92{h
\x02\x00\xa4
\x00\xa0\x08{i
\x04\x00\xa4k\xa0\x14{i
\x02\x00\xa0\t\x92\x94k
x\xa4
\xb4\xa1\x03\xa4k\xa0\x14{j
\x04\x00\xa0\t\x92\x94k
\xb4\xa4
\xf0\xa1\x03\xa4k\xa4
\x00\x14K\x04UPIO\x04\xa0\x16\x92{h
\x02\x00\xa0\x08\x93j
\x02\xa4
\xf0\xa1\x05\xa4\x0b\x84\x03\xa0\x08{i
\x02\x00\xa4k\xa0\x14{i
\x01\x00\xa0\t\x92\x94k
x\xa4
\xb4\xa1\x03\xa4k\xa0\x08\x93j
\x02\xa4
\xf0\xa1\x05\xa4\x0b\x84\x03\x14-FDMA\x02\xa0\x0c\x92\x93i
\xff\xa4}i
@\x00\xa0\x10\x92\x95h
\x03\xa4}th
\x02\x00
 \x00\xa0\x05h\xa4
\x12\xa4
\x00\x14\x1fFPIO\x01\xa0\x0c\x92\x95h
\x03\xa4}h
\x08\x00\xa0\x08\x93h
\x01\xa4
\x01\xa4
\x00\x14J\x05SCMP\x02p\x87h`\xa0\x08\x92\x93`\x87i\xa4\x01u`\x08STR1\x11\x02`\x08STR2\x11\x02`phSTR1piSTR2p\x00a\xa2"\x95a`p\x83\x88STR1a\x00bp\x83\x88STR2a\x00c\xa0\x07\x92\x93bc\xa4\x01ua\xa4\x00\x08SPS_
\x00\x08OSIF
\x00\x08W98F
\x00\x08WNTF
\x00\x08WMEF
\x00\x08WXPF
\x00\x08WVIS
\x00\x08WSPV
\x00\x08LNUX
\x00\x08H8DR
\x00\x08MEMX
\x00\x08ACST
\x00\x08FNID
\x00\x08RRBF
\x00\x08NBCF
\x00''')
f('sys/firmware/acpi/tables/SSDT1', 0o664, b'''SSDT\xaa\x01\x00\x00\x01\x89LENOVOTP-7I    \x11\x00\x00MSFT\x0e\x00\x00\x01\x10\x0e\\\x00\x14
KOU2\x00[!
d\x10I\x0b\\/\x04_SB_PCI0VID_LCD0\x14&_BCL\x00p
\x01\\NBCF\xa4\x12\x16

d
\x1e
\x1e
(
2
<
F
P
Z
d\x08BCLP\x12\x12\x08
\x1e
(
2
<
F
P
Z
d\x08BCLL\x12\x12\x08
\x00
\x01
\x02
\x03
\x04
\x05
\x06
\x07\x14*_BCM\x01p\x89BCLP\x01h\x00
\x00
\x00`\xa0\x15\x92\x93`\xffp\x83\x88BCLL`\x00a\\VBRCa\x14!_BQC\x00p\\VBRG`\xa0\x0f\x92\x94`
\x07\xa4\x83\x88BCLP`\x00\xa1\x03\xa4\x00\x10L\x0b\\/\x05_SB_PCI0AGP_VID_LCD0\x14%_BCL\x00p
\x01NBCF\xa4\x12\x16

d
\x1e
\x1e
(
2
<
F
P
Z
d\x08BCLP\x12\x12\x08
\x1e
(
2
<
F
P
Z
d\x08BCLL\x12\x12\x08
\x00
\x01
\x02
\x03
\x04
\x05
\x06
\x07\x14*_BCM\x01p\x89BCLP\x01h\x00
\x00
\x00`\xa0\x15\x92\x93`\xffp\x83\x88BCLL`\x00a\\VBRCa\x14!_BQC\x00p\\VBRG`\xa0\x0f\x92\x94`
\x07\xa4\x83\x88BCLP`\x00\xa1\x03\xa4\x00''')
f('sys/firmware/acpi/tables/SLIC', 0o664, b'SLICv\x01\x00\x00\x01\x1dLENOVOTP-7I    \x11\x00\x00 LTP\x00\x00\x00\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x06\x02\x00\x00\x00$\x00\x00RSA1\x00\x04\x00\x00\x01\x00\x01\x00i\x16J\x9f\xb1K:\xfb\x80 \xaa\xaf\xc4\xf9>\xc1\x80I\xeeje&r\x1e\xcd\xbf_/\x96\xd6\xc0\n\x92\xf5\x06\xb5\x00\xb2;)\x02\xe2L\x8d\xc2\xf2\xbcAw\x9cp\xf0\xf3\x1b\t\xd2cZ\xdc\xa8\x83\xf8^\xc9\x15\x95\xf9\xfa\xfd\xdc\x05\xb7Mg\x7f-\xb3\x843 \xe1\xd1y*\xa7jw\xd1\xb6 *vB\xc5\xd5\xe9\xb6C@UD\xc3\xc97\x99_A\x97p\xf3\xd1\xf6\x07\xec{\x1a)\xa1\xc1\xf1\x91\xfdH\x86n>\xce\xcb\x01\x00\x00\x00\xb6\x00\x00\x00\x00\x00\x02\x00LENOVOTP-7I   WINDOWS \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x8bz\xb4\xe1\x0e{\x7fc\xd3M%N\xd7\tBd\x9c\x89\xf6F\xb8L\xd5M\x86sy\xf0\x15\xf2\x98\xad\x806\t\xbd\x8c%wn\xb8C\xf4\xb1\xf0W\x08x\x8d\xc6\x18T\xe1`G\xf07\xfdH\xd2\xfdM\x07\xdd\x91O\x9e*\xd6\xdd4\xeb\x9ac\x9a\xb8O%\xd4o\xf0\x95\xbb\xce\xbd:X\x04\xc5l\xb8\xa8\xd89\xf5\x02Oh\x84\x9c)K\xd0\x9b\x16\xb5\xf5A\xb0\x9d\xb8A\x07\x9dJ\x11\xaf\x19\xbd\xb7\xc7\x05\xb6\xa7\x93H\xf6\xcb')
f('sys/firmware/acpi/tables/SSDT3', 0o664, b'''SSDT\xa6\x00\x00\x00\x01hLENOVOTP-7I    \x11\x00\x00INTL\x13\x05\x05 \x10A\x08\\._PR_CPU1\x08_TPC
\x00\x14\x16_PTC\x00\xa4\\/\x03_PR_CPU0_PTC\x14\x16_TSS\x00\xa4\\/\x03_PR_CPU0_TSS\x14?_TSD\x00\xa0\'\x90{CFGD\x0c\x00\x00\x00\x01\x00\x92{PDC1
\x04\x00\xa4\x12\x0f\x01\x12\x0c\x05
\x05
\x00
\x00
\xfd
\x02\xa4\x12\x0f\x01\x12\x0c\x05
\x05
\x00
\x01
\xfc
\x01''')
f('sys/firmware/acpi/tables/SSDT5', 0o664, b'''SSDT\xd8\x01\x00\x00\x01\tLENOVOTP-7I    \x11\x00\x00INTL\x13\x05\x05 \x10C\x1b\\/\x03_SB_PCI0SATA[\x82@\x1aPRT0\x08DRE0
\x00\x08DIP0
\x00\x08HDTF\x11\x11
\x0e\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x8cHDTF
\rHFLC\x08ERTF\x11\x18
\x15\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5_\x00\x00\x00\x00\xa0\xef\x8cERTF
\rEFLC\x08HPTF\x11\x18
\x15\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5\x10\x03\x00\x00\x00\xa0\xef\x8cHPTF
\rPFLC\x08HXTF\x11\x1f
\x1c\x02\x00\x00\x00\x00\xa0\xef\x00\x00\x00\x00\x00\xa0\xf5_\x00\x00\x00\x00\xa0\xef\x10\x03\x00\x00\x00\xa0\xef\x8cHXTF
\rXFLC\x08_ADR\x0b\xff\xff\x14L\x08_SDD\x01p
\x00DRE0p
\x00DIP0\xa0F\x07\x93\x87h\x0b\x00\x02\x8bh
\x9cM078\x8bh
\xacM086\x8bh
\xeeM119\x8bh\x0b(\x01M148\xa0\x1f\x93{M148\x0b\x00\xc0\x00\x0b\x00@\xa0\x10{M148
\x04\x00p
\x01DRE0\xa0\x1b{M086\x0b\x00\x80\x00\xa0\x10{M119
\x01\x00p
\x01DRE0\xa0\x10{M078
\x08\x00p
\x01DIP0\x14B\x05_GTF\x00\xa0"\\OSSSp
\xe1XFLCp
\xe1EFLCp
\xe1HFLCp
\xe1PFLC\xa0\x17DRE0\xa0
DIP0\xa4HXTF\xa1\x06\xa4ERTF\xa0
DIP0\xa4HPTF\xa4HDTF''')
f('sys/firmware/acpi/tables/HPET', 0o664, b'HPET8\x00\x00\x00\x01\xaaLENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x01\xa2\x86\x80\x00\x00\x00\x00\x00\x00\xd0\xfe\x00\x00\x00\x00\x00\x80\x00\x00')
f('sys/firmware/acpi/tables/FACP', 0o664, b'FACP\xf4\x00\x00\x00\x03\xf0LENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x00@\xef\x7f^\x19\xed\x7f\x00\x02\t\x00\xb2\x00\x00\x00\xf0\xf1\x00\xf3\x00\x10\x00\x00\x00\x00\x00\x00\x04\x10\x00\x00\x00\x00\x00\x00 \x10\x00\x00\x08\x10\x00\x00(\x10\x00\x00\x00\x00\x00\x00\x04\x02\x01\x04\x08\x00\x00\xf4\x01\x00U\x00\x00\x00\x00\x00\x01\x03\r\x002\x12\x00\x00\xad\xc2\x00\x00\x01\x08\x00\x00\xf9\x0c\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x00@\xef\x7f\x00\x00\x00\x00^\x19\xed\x7f\x00\x00\x00\x00\x01 \x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x10\x00\x00\x04\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x08\x00\x00 \x10\x00\x00\x00\x00\x00\x00\x01 \x00\x00\x08\x10\x00\x00\x00\x00\x00\x00\x01 \x00\x00(\x10\x00\x00\x00\x00\x00\x00\x01 \x00\x00,\x10\x00\x00\x00\x00\x00\x00')
f('sys/firmware/acpi/tables/APIC', 0o664, b'''APICh\x00\x00\x00\x01lLENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x00\x00\xe0\xfe\x01\x00\x00\x00\x00\x08\x00\x00\x01\x00\x00\x00\x00\x08\x01\x01\x01\x00\x00\x00\x01\x0c\x01\x00\x00\x00\xc0\xfe\x00\x00\x00\x00\x02
\x00\x00\x02\x00\x00\x00\x00\x00\x02
\x00\t\t\x00\x00\x00\r\x00\x04\x06\x00\x05\x00\x01\x04\x06\x01\x05\x00\x01''')
f('sys/firmware/acpi/tables/SSDT2', 0o664, b'''SSDT_\x02\x00\x00\x01\xc2LENOVOTP-7I    \x11\x00\x00INTL\x13\x05\x05 \x10J#\\._PR_CPU0\x08_TPC
\x00\x14M\x06_PTC\x00\xa07{PDC0
\x04\x00\xa4\x12,\x02\x11\x14
\x11\x82\x0c\x00\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00y\x00\x11\x14
\x11\x82\x0c\x00\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00y\x00\xa4\x12,\x02\x11\x14
\x11\x82\x0c\x00\x01\x04\x01\x00\x10\x10\x00\x00\x00\x00\x00\x00y\x00\x11\x14
\x11\x82\x0c\x00\x01\x04\x01\x00\x10\x10\x00\x00\x00\x00\x00\x00y\x00\x08TSSI\x12A\x07\x08\x12\r\x05
d\x0b\xe8\x03
\x00
\x00
\x00\x12\r\x05
X\x0bk\x03
\x00
\x0f
\x00\x12\r\x05
K\x0b\xee\x02
\x00
\x0e
\x00\x12\r\x05
?\x0bq\x02
\x00
\r
\x00\x12\r\x05
2\x0b\xf4\x01
\x00
\x0c
\x00\x12\r\x05
&\x0bw\x01
\x00
\x0b
\x00\x12\x0c\x05
\x19
\xfa
\x00


\x00\x12\x0c\x05
\r
}
\x00
\t
\x00\x08TSSM\x12A\x07\x08\x12\r\x05
d\x0b\xe8\x03
\x00
\x00
\x00\x12\r\x05
X\x0bk\x03
\x00
\x1e
\x00\x12\r\x05
K\x0b\xee\x02
\x00
\x1c
\x00\x12\r\x05
?\x0bq\x02
\x00
\x1a
\x00\x12\r\x05
2\x0b\xf4\x01
\x00
\x18
\x00\x12\r\x05
&\x0bw\x01
\x00
\x16
\x00\x12\x0c\x05
\x19
\xfa
\x00
\x14
\x00\x12\x0c\x05
\r
}
\x00
\x12
\x00\x08TSSF
\x00\x14C\x08_TSS\x00\xa0G\x06\x90\x92TSSF[\x12_PSS\x00p_PSS`p\x87`avap\x83\x88\x83\x88`a\x00
\x01\x00bp
\x00c\xa25\x95c\x87TSSIpxwbt
\x08c\x00\x00
\x08\x00\x00dpd\x88\x83\x88TSSIc\x00
\x01\x00pd\x88\x83\x88TSSMc\x00
\x01\x00ucp\xffTSSF\xa0\x0e{PDC0
\x04\x00\xa4TSSM\xa4TSSI\x14?_TSD\x00\xa0\'\x90{CFGD\x0c\x00\x00\x00\x01\x00\x92{PDC0
\x04\x00\xa4\x12\x0f\x01\x12\x0c\x05
\x05
\x00
\x00
\xfd
\x02\xa4\x12\x0f\x01\x12\x0c\x05
\x05
\x00
\x00
\xfc
\x01''')
f('sys/firmware/acpi/tables/BOOT', 0o664, b'BOOT(\x00\x00\x00\x01\xa8LENOVOTP-7I    \x11\x00\x00 LTP\x01\x00\x00\x005\x00\x00\x00')
f('sys/firmware/acpi/tables/TCPA', 0o664, b'TCPA2\x00\x00\x00\x02YLENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\xce\x1a\xee\x7f\x00\x00\x00\x00')
f('sys/firmware/acpi/tables/SSDT4', 0o664, b'''SSDT\xf7\x04\x00\x00\x01\xa6LENOVOTP-7I    \x11\x00\x00INTL\x13\x05\x05 \x10E\x08\\\x00\x08SSDT\x12C\x05\x0c\rCPU0IST \x00\x0c6\x1d\xef\x7f\x0c\xc4\x02\x00\x00\rCPU1IST \x00\x0cn\x1c\xef\x7f\x0c\xc8\x00\x00\x00\rCPU0CST \x00\x0c\x7f \xef\x7f\x0cZ\x06\x00\x00\rCPU1CST \x00\x0c\xfa\x1f\xef\x7f\x0c\x85\x00\x00\x00\x08CFGD\x0c\xf1i;\x11\x08\\PDC0\x0c\x00\x00\x00\x80\x08\\PDC1\x0c\x00\x00\x00\x80\x08\\SDTL
\x00\x10@!\\._PR_CPU0\x08HI0_
\x00\x08HC0_
\x00\x14H\x06_PDC\x01\x8ah
\x00REVS\x8ah
\x04SIZEp\x87h`pt`
\x08\x00a[\x13h
@wa
\x08\x00TEMP\x08STS0\x11\x07
\x04\x00\x00\x00\x00sSTS0TEMPb_OSC\x11\x13
\x10\x16\xa6w@\x0c)\xbeG\x9e\xbd\xd8pXq9SREVSSIZEb\x14L\x18_OSC\x04\x8ak
\x00STS0\x8ak
\x04CAP0\x8ah
\x00IID0\x8ah
\x04IID1\x8ah
\x08IID2\x8ah
\x0cIID3\x08UID0\x11\x13
\x10\x16\xa6w@\x0c)\xbeG\x9e\xbd\xd8pXq9S\x8aUID0
\x00EID0\x8aUID0
\x04EID1\x8aUID0
\x08EID2\x8aUID0
\x0cEID3\xa02\x92\x90\x90\x93IID0EID0\x93IID1EID1\x90\x93IID2EID2\x93IID3EID3p
\x06STS0\xa4k\xa0\x0f\x92\x93i
\x01p

STS0\xa4k}{PDC0\x0c\xff\xff\xff\x7f\x00CAP0PDC0\xa0L\x05{CFGD
\x01\x00\xa0A\x05\x90\x90{CFGD\x0c\x00\x00\x00\x01\x00\x93{PDC0
\t\x00
\t\x92{SDTL
\x01\x00}SDTL
\x01SDTL[\x80IST0\x00\x83\x88SSDT
\x01\x00\x83\x88SSDT
\x02\x00[ IST0HI0_\xa0I\x05{CFGD
\xf0\x00\xa0N\x04\x90\x90{CFGD\x0c\x00\x00\x00\x01\x00{PDC0
\x18\x00\x92{SDTL
\x02\x00}SDTL
\x02SDTL[\x80CST0\x00\x83\x88SSDT
\x07\x00\x83\x88SSDT
\x08\x00[ CST0HC0_\xa4k\x10K#\\._PR_CPU1\x08HI1_
\x00\x08HC1_
\x00\x14H\x06_PDC\x01\x8ah
\x00REVS\x8ah
\x04SIZEp\x87h`pt`
\x08\x00a[\x13h
@wa
\x08\x00TEMP\x08STS1\x11\x07
\x04\x00\x00\x00\x00sSTS1TEMPb_OSC\x11\x13
\x10\x16\xa6w@\x0c)\xbeG\x9e\xbd\xd8pXq9SREVSSIZEb\x14L\x18_OSC\x04\x8ak
\x00STS1\x8ak
\x04CAP1\x8ah
\x00IID0\x8ah
\x04IID1\x8ah
\x08IID2\x8ah
\x0cIID3\x08UID1\x11\x13
\x10\x16\xa6w@\x0c)\xbeG\x9e\xbd\xd8pXq9S\x8aUID1
\x00EID0\x8aUID1
\x04EID1\x8aUID1
\x08EID2\x8aUID1
\x0cEID3\xa02\x92\x90\x90\x93IID0EID0\x93IID1EID1\x90\x93IID2EID2\x93IID3EID3p
\x06STS1\xa4k\xa0\x0f\x92\x93i
\x01p

STS1\xa4k}{PDC1\x0c\xff\xff\xff\x7f\x00CAP1PDC1\xa0L\x05{CFGD
\x01\x00\xa0A\x05\x90\x90{CFGD\x0c\x00\x00\x00\x01\x00\x93{PDC1
\t\x00
\t\x92{SDTL
\x10\x00}SDTL
\x10SDTL[\x80IST1\x00\x83\x88SSDT
\x04\x00\x83\x88SSDT
\x05\x00[ IST1HI1_\xa0I\x05{CFGD
\xf0\x00\xa0N\x04\x90\x90{CFGD\x0c\x00\x00\x00\x01\x00{PDC1
\x18\x00\x92{SDTL
 \x00}SDTL
 SDTL[\x80CST1\x00\x83\x88SSDT

\x00\x83\x88SSDT
\x0b\x00[ CST1HC1_\xa4k\x14*_INI\x00\xa0\x0c\\DTSETHRM
\x00\xa0\x16\x90\\WXPF\x92\x94\\WSPV
\x01PPMS
\x00''')
f('sys/firmware/acpi/tables/ECDT', 0o664, b'ECDTR\x00\x00\x00\x01ULENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x01\x08\x00\x00f\x00\x00\x00\x00\x00\x00\x00\x01\x08\x00\x00b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1c\\_SB.PCI0.LPC.EC\x00')
f('sys/firmware/acpi/tables/FACS', 0o664, b'FACS@\x00\x00\x00/\x15\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00')
f('sys/firmware/acpi/tables/MCFG', 0o664, b'MCFG<\x00\x00\x00\x01\x82LENOVOTP-7I    \x11\x00\x00LNVO\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0\x00\x00\x00\x00\x00\x00\x00?\x00\x00\x00\x00')
d('sys/firmware/acpi/interrupts', 0o775)
f('sys/firmware/acpi/interrupts/gpe1A', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe12', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe11', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe10', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/ff_pmtimer', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe0C', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe04', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe06', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe14', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe0B', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/sci', 0o664, b'   14240\n')
f('sys/firmware/acpi/interrupts/gpe08', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe15', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe17', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe0A', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/ff_rt_clk', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe09', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe19', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe03', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe01', 0o664, b'       1\tenable\n')
f('sys/firmware/acpi/interrupts/ff_slp_btn', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/error', 0o664, b'       0\n')
f('sys/firmware/acpi/interrupts/gpe1C', 0o664, b'   14039\tenable\n')
f('sys/firmware/acpi/interrupts/ff_pwr_btn', 0o664, b'       0\tenable\n')
f('sys/firmware/acpi/interrupts/gpe0D', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe05', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe02', 0o664, b'     200\tenable\n')
f('sys/firmware/acpi/interrupts/gpe1E', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe18', 0o664, b'       0\tenable\n')
f('sys/firmware/acpi/interrupts/gpe1D', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe1F', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe0E', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe00', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe1B', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe0F', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe07', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe16', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe13', 0o664, b'       0  invalid\n')
f('sys/firmware/acpi/interrupts/gpe_all', 0o664, b'   14240\n')
f('sys/firmware/acpi/interrupts/ff_gbl_lock', 0o664, b'       0\tenable\n')
d('sys/bus', 0o755)
d('sys/bus/pci_express', 0o755)
f('sys/bus/pci_express/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/pci_express/devices', 0o755)
l('sys/bus/pci_express/devices/0000:00:1c.2:pcie02', '../../../devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02')
l('sys/bus/pci_express/devices/0000:00:1c.2:pcie00', '../../../devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00')
l('sys/bus/pci_express/devices/0000:00:1c.0:pcie03', '../../../devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03')
l('sys/bus/pci_express/devices/0000:00:01.0:pcie03', '../../../devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03')
l('sys/bus/pci_express/devices/0000:00:1c.2:pcie03', '../../../devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03')
l('sys/bus/pci_express/devices/0000:00:1c.1:pcie02', '../../../devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02')
l('sys/bus/pci_express/devices/0000:00:1c.0:pcie00', '../../../devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00')
l('sys/bus/pci_express/devices/0000:00:1c.3:pcie02', '../../../devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02')
l('sys/bus/pci_express/devices/0000:00:1c.0:pcie02', '../../../devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02')
l('sys/bus/pci_express/devices/0000:00:1c.1:pcie03', '../../../devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03')
l('sys/bus/pci_express/devices/0000:00:1c.3:pcie03', '../../../devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03')
l('sys/bus/pci_express/devices/0000:00:1c.1:pcie00', '../../../devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00')
l('sys/bus/pci_express/devices/0000:00:1c.3:pcie00', '../../../devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00')
l('sys/bus/pci_express/devices/0000:00:01.0:pcie00', '../../../devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00')
d('sys/bus/acpi', 0o775)
f('sys/bus/acpi/drivers_autoprobe', 0o664, b'1\n')
d('sys/bus/acpi/drivers', 0o775)
d('sys/bus/acpi/drivers/thinkpad_hotkey', 0o775)
l('sys/bus/acpi/drivers/thinkpad_hotkey/IBM0068:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/IBM0068:00')
d('sys/bus/acpi/drivers/button', 0o775)
l('sys/bus/acpi/drivers/button/PNP0C0E:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0E:00')
l('sys/bus/acpi/drivers/button/PNP0C0D:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0D:00')
l('sys/bus/acpi/drivers/button/LNXPWRBN:00', '../../../../devices/LNXSYSTM:00/LNXPWRBN:00')
d('sys/bus/acpi/drivers/processor', 0o775)
l('sys/bus/acpi/drivers/processor/ACPI0007:01', '../../../../devices/LNXSYSTM:00/ACPI0007:01')
l('sys/bus/acpi/drivers/processor/ACPI0007:00', '../../../../devices/LNXSYSTM:00/ACPI0007:00')
d('sys/bus/acpi/drivers/power', 0o775)
l('sys/bus/acpi/drivers/power/LNXPOWER:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/LNXPOWER:00')
d('sys/bus/acpi/drivers/pci_link', 0o775)
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:05', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:05')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:07', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:07')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:04', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:04')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:02', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:02')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:00')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:01', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:01')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:06', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:06')
l('sys/bus/acpi/drivers/pci_link/PNP0C0F:03', '../../../../devices/LNXSYSTM:00/device:00/PNP0C0F:03')
d('sys/bus/acpi/drivers/battery', 0o775)
l('sys/bus/acpi/drivers/battery/PNP0C0A:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/PNP0C0A:00')
d('sys/bus/acpi/drivers/ec', 0o775)
l('sys/bus/acpi/drivers/ec/PNP0C09:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00')
d('sys/bus/acpi/drivers/pci_root', 0o775)
l('sys/bus/acpi/drivers/pci_root/PNP0A08:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00')
d('sys/bus/acpi/drivers/thermal', 0o775)
l('sys/bus/acpi/drivers/thermal/LNXTHERM:01', '../../../../devices/LNXSYSTM:00/LNXTHERM:00/LNXTHERM:01')
l('sys/bus/acpi/drivers/thermal/LNXTHERM:02', '../../../../devices/LNXSYSTM:00/LNXTHERM:00/LNXTHERM:02')
d('sys/bus/acpi/drivers/ac', 0o775)
l('sys/bus/acpi/drivers/ac/ACPI0003:00', '../../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/ACPI0003:00')
d('sys/bus/acpi/devices', 0o775)
l('sys/bus/acpi/devices/device:11', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:11')
l('sys/bus/acpi/devices/device:21', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:21')
l('sys/bus/acpi/devices/PNP0000:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0000:00')
l('sys/bus/acpi/devices/device:0b', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0b')
l('sys/bus/acpi/devices/device:1f', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1d/device:1e/device:1f')
l('sys/bus/acpi/devices/LNXTHERM:01', '../../../devices/LNXSYSTM:00/LNXTHERM:00/LNXTHERM:01')
l('sys/bus/acpi/devices/PNP0103:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0103:00')
l('sys/bus/acpi/devices/LNXPOWER:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/LNXPOWER:00')
l('sys/bus/acpi/devices/device:1c', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1a/device:1b/device:1c')
l('sys/bus/acpi/devices/device:06', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:06')
l('sys/bus/acpi/devices/ACPI0003:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/ACPI0003:00')
l('sys/bus/acpi/devices/PNP0C0A:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/PNP0C0A:00')
l('sys/bus/acpi/devices/device:0e', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0d/device:0e')
l('sys/bus/acpi/devices/ATM1200:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/ATM1200:00')
l('sys/bus/acpi/devices/device:08', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:06/device:07/device:08')
l('sys/bus/acpi/devices/PNP0C0F:05', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:05')
l('sys/bus/acpi/devices/device:18', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:18')
l('sys/bus/acpi/devices/device:0c', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0c')
l('sys/bus/acpi/devices/PNP0303:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0303:00')
l('sys/bus/acpi/devices/device:17', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:16/device:17')
l('sys/bus/acpi/devices/device:02', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:02')
l('sys/bus/acpi/devices/ACPI0007:01', '../../../devices/LNXSYSTM:00/ACPI0007:01')
l('sys/bus/acpi/devices/device:13', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:13')
l('sys/bus/acpi/devices/PNP0C0E:00', '../../../devices/LNXSYSTM:00/device:00/PNP0C0E:00')
l('sys/bus/acpi/devices/device:04', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:02/device:04')
l('sys/bus/acpi/devices/PNP0A08:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00')
l('sys/bus/acpi/devices/PNP0C04:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C04:00')
l('sys/bus/acpi/devices/LNXTHERM:00', '../../../devices/LNXSYSTM:00/LNXTHERM:00')
l('sys/bus/acpi/devices/device:15', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:13/device:14/device:15')
l('sys/bus/acpi/devices/IBM0079:00', '../../../devices/LNXSYSTM:00/device:00/IBM0079:00')
l('sys/bus/acpi/devices/device:1d', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1d')
l('sys/bus/acpi/devices/PNP0C0D:00', '../../../devices/LNXSYSTM:00/device:00/PNP0C0D:00')
l('sys/bus/acpi/devices/device:23', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:21/device:22/device:23')
l('sys/bus/acpi/devices/device:19', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:19')
l('sys/bus/acpi/devices/device:12', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:11/device:12')
l('sys/bus/acpi/devices/device:1a', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1a')
l('sys/bus/acpi/devices/device:00', '../../../devices/LNXSYSTM:00/device:00')
l('sys/bus/acpi/devices/PNP0800:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0800:00')
l('sys/bus/acpi/devices/PNP0C02:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C02:00')
l('sys/bus/acpi/devices/device:10', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0f/device:10')
l('sys/bus/acpi/devices/PNP0100:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0100:00')
l('sys/bus/acpi/devices/device:0d', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0d')
l('sys/bus/acpi/devices/IBM0068:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/IBM0068:00')
l('sys/bus/acpi/devices/device:25', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:25')
l('sys/bus/acpi/devices/PNP0C01:00', '../../../devices/LNXSYSTM:00/device:00/PNP0C01:00')
l('sys/bus/acpi/devices/device:01', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01')
l('sys/bus/acpi/devices/PNP0C0F:07', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:07')
l('sys/bus/acpi/devices/PNP0C0F:04', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:04')
l('sys/bus/acpi/devices/device:1b', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1a/device:1b')
l('sys/bus/acpi/devices/device:24', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:21/device:22/device:24')
l('sys/bus/acpi/devices/device:1e', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:1d/device:1e')
l('sys/bus/acpi/devices/ACPI0007:00', '../../../devices/LNXSYSTM:00/ACPI0007:00')
l('sys/bus/acpi/devices/LNXTHERM:02', '../../../devices/LNXSYSTM:00/LNXTHERM:00/LNXTHERM:02')
l('sys/bus/acpi/devices/PNP0C09:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00')
l('sys/bus/acpi/devices/device:20', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:20')
l('sys/bus/acpi/devices/PNP0C0F:02', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:02')
l('sys/bus/acpi/devices/LNXSYSTM:00', '../../../devices/LNXSYSTM:00')
l('sys/bus/acpi/devices/device:05', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:02/device:05')
l('sys/bus/acpi/devices/PNP0C0F:00', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:00')
l('sys/bus/acpi/devices/device:0f', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:0f')
l('sys/bus/acpi/devices/device:16', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:16')
l('sys/bus/acpi/devices/device:09', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:06/device:07/device:09')
l('sys/bus/acpi/devices/device:07', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:06/device:07')
l('sys/bus/acpi/devices/device:03', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:02/device:03')
l('sys/bus/acpi/devices/device:22', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:21/device:22')
l('sys/bus/acpi/devices/PNP0C0F:01', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:01')
l('sys/bus/acpi/devices/PNP0C0F:06', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:06')
l('sys/bus/acpi/devices/device:0a', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:06/device:07/device:0a')
l('sys/bus/acpi/devices/PNP0200:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0200:00')
l('sys/bus/acpi/devices/device:14', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:13/device:14')
l('sys/bus/acpi/devices/IBM0057:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/IBM0057:00')
l('sys/bus/acpi/devices/PNP0B00:00', '../../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0B00:00')
l('sys/bus/acpi/devices/PNP0C0F:03', '../../../devices/LNXSYSTM:00/device:00/PNP0C0F:03')
l('sys/bus/acpi/devices/LNXPWRBN:00', '../../../devices/LNXSYSTM:00/LNXPWRBN:00')
d('sys/bus/serio', 0o755)
f('sys/bus/serio/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/serio/drivers', 0o755)
d('sys/bus/serio/drivers/psmouse', 0o755)
l('sys/bus/serio/drivers/psmouse/module', '../../../../module/psmouse')
l('sys/bus/serio/drivers/psmouse/serio1', '../../../../devices/platform/i8042/serio1')
f('sys/bus/serio/drivers/psmouse/bind_mode', 0o644, b'auto\n')
f('sys/bus/serio/drivers/psmouse/description', 0o644, b'PS/2 mouse driver\n')
d('sys/bus/serio/drivers/atkbd', 0o755)
l('sys/bus/serio/drivers/atkbd/serio0', '../../../../devices/platform/i8042/serio0')
l('sys/bus/serio/drivers/atkbd/module', '../../../../module/atkbd')
f('sys/bus/serio/drivers/atkbd/bind_mode', 0o644, b'auto\n')
f('sys/bus/serio/drivers/atkbd/description', 0o644, b'AT and PS/2 keyboard driver\n')
d('sys/bus/serio/devices', 0o755)
l('sys/bus/serio/devices/serio0', '../../../devices/platform/i8042/serio0')
l('sys/bus/serio/devices/serio1', '../../../devices/platform/i8042/serio1')
d('sys/bus/scsi', 0o755)
f('sys/bus/scsi/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/scsi/drivers', 0o755)
d('sys/bus/scsi/drivers/sd', 0o755)
l('sys/bus/scsi/drivers/sd/7:0:0:0', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0')
l('sys/bus/scsi/drivers/sd/0:0:0:0', '../../../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0')
d('sys/bus/scsi/drivers/sr', 0o755)
l('sys/bus/scsi/drivers/sr/4:0:0:0', '../../../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0')
d('sys/bus/scsi/devices', 0o755)
l('sys/bus/scsi/devices/target4:0:0', '../../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0')
l('sys/bus/scsi/devices/host0', '../../../devices/pci0000:00/0000:00:1f.2/host0')
l('sys/bus/scsi/devices/host5', '../../../devices/pci0000:00/0000:00:1f.1/host5')
l('sys/bus/scsi/devices/host1', '../../../devices/pci0000:00/0000:00:1f.2/host1')
l('sys/bus/scsi/devices/target0:0:0', '../../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0')
l('sys/bus/scsi/devices/host4', '../../../devices/pci0000:00/0000:00:1f.1/host4')
l('sys/bus/scsi/devices/7:0:0:0', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0')
l('sys/bus/scsi/devices/target7:0:0', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0')
l('sys/bus/scsi/devices/0:0:0:0', '../../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0')
l('sys/bus/scsi/devices/host2', '../../../devices/pci0000:00/0000:00:1f.2/host2')
l('sys/bus/scsi/devices/host7', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7')
l('sys/bus/scsi/devices/4:0:0:0', '../../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0')
l('sys/bus/scsi/devices/host3', '../../../devices/pci0000:00/0000:00:1f.2/host3')
d('sys/bus/usb', 0o755)
f('sys/bus/usb/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/usb/drivers', 0o755)
d('sys/bus/usb/drivers/usb-storage', 0o755)
l('sys/bus/usb/drivers/usb-storage/module', '../../../../module/usb_storage')
l('sys/bus/usb/drivers/usb-storage/5-1:1.0', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0')
d('sys/bus/usb/drivers/cdc_acm', 0o755)
l('sys/bus/usb/drivers/cdc_acm/5-2:1.1', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1')
l('sys/bus/usb/drivers/cdc_acm/5-2:1.0', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0')
l('sys/bus/usb/drivers/cdc_acm/module', '../../../../module/cdc_acm')
d('sys/bus/usb/drivers/usbhid', 0o755)
l('sys/bus/usb/drivers/usbhid/3-1:1.0', '../../../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0')
l('sys/bus/usb/drivers/usbhid/module', '../../../../module/usbhid')
d('sys/bus/usb/drivers/hub', 0o755)
l('sys/bus/usb/drivers/hub/3-0:1.0', '../../../../devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0')
l('sys/bus/usb/drivers/hub/1-0:1.0', '../../../../devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0')
l('sys/bus/usb/drivers/hub/4-0:1.0', '../../../../devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0')
l('sys/bus/usb/drivers/hub/module', '../../../../module/usbcore')
l('sys/bus/usb/drivers/hub/2-0:1.0', '../../../../devices/pci0000:00/0000:00:1d.1/usb2/2-0:1.0')
l('sys/bus/usb/drivers/hub/5-0:1.0', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0')
d('sys/bus/usb/drivers/usb', 0o755)
l('sys/bus/usb/drivers/usb/usb5', '../../../../devices/pci0000:00/0000:00:1d.7/usb5')
l('sys/bus/usb/drivers/usb/usb1', '../../../../devices/pci0000:00/0000:00:1d.0/usb1')
l('sys/bus/usb/drivers/usb/4-2', '../../../../devices/pci0000:00/0000:00:1d.3/usb4/4-2')
l('sys/bus/usb/drivers/usb/5-1', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1')
l('sys/bus/usb/drivers/usb/5-2', '../../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2')
l('sys/bus/usb/drivers/usb/4-1', '../../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1')
l('sys/bus/usb/drivers/usb/module', '../../../../module/usbcore')
l('sys/bus/usb/drivers/usb/usb4', '../../../../devices/pci0000:00/0000:00:1d.3/usb4')
l('sys/bus/usb/drivers/usb/usb3', '../../../../devices/pci0000:00/0000:00:1d.2/usb3')
l('sys/bus/usb/drivers/usb/usb2', '../../../../devices/pci0000:00/0000:00:1d.1/usb2')
l('sys/bus/usb/drivers/usb/3-1', '../../../../devices/pci0000:00/0000:00:1d.2/usb3/3-1')
d('sys/bus/usb/drivers/usbfs', 0o755)
l('sys/bus/usb/drivers/usbfs/module', '../../../../module/usbcore')
d('sys/bus/usb/devices', 0o755)
l('sys/bus/usb/devices/4-1:1.3', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3')
l('sys/bus/usb/devices/5-2:1.4', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4')
l('sys/bus/usb/devices/usb5', '../../../devices/pci0000:00/0000:00:1d.7/usb5')
l('sys/bus/usb/devices/4-1:1.0', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0')
l('sys/bus/usb/devices/5-2:1.2', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2')
l('sys/bus/usb/devices/3-0:1.0', '../../../devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0')
l('sys/bus/usb/devices/usb1', '../../../devices/pci0000:00/0000:00:1d.0/usb1')
l('sys/bus/usb/devices/4-2', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-2')
l('sys/bus/usb/devices/5-2:1.1', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1')
l('sys/bus/usb/devices/1-0:1.0', '../../../devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0')
l('sys/bus/usb/devices/5-1', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1')
l('sys/bus/usb/devices/5-2:1.10', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10')
l('sys/bus/usb/devices/4-0:1.0', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0')
l('sys/bus/usb/devices/5-2:1.0', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0')
l('sys/bus/usb/devices/5-2', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2')
l('sys/bus/usb/devices/5-2:1.9', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9')
l('sys/bus/usb/devices/3-1:1.0', '../../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0')
l('sys/bus/usb/devices/5-2:1.13', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13')
l('sys/bus/usb/devices/5-2:1.5', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5')
l('sys/bus/usb/devices/4-1', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1')
l('sys/bus/usb/devices/4-1:1.2', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2')
l('sys/bus/usb/devices/usb4', '../../../devices/pci0000:00/0000:00:1d.3/usb4')
l('sys/bus/usb/devices/2-0:1.0', '../../../devices/pci0000:00/0000:00:1d.1/usb2/2-0:1.0')
l('sys/bus/usb/devices/5-2:1.11', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11')
l('sys/bus/usb/devices/usb3', '../../../devices/pci0000:00/0000:00:1d.2/usb3')
l('sys/bus/usb/devices/5-2:1.12', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12')
l('sys/bus/usb/devices/5-2:1.3', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3')
l('sys/bus/usb/devices/5-2:1.6', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6')
l('sys/bus/usb/devices/4-1:1.1', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1')
l('sys/bus/usb/devices/5-1:1.0', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0')
l('sys/bus/usb/devices/5-0:1.0', '../../../devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0')
l('sys/bus/usb/devices/4-2:1.0', '../../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0')
l('sys/bus/usb/devices/usb2', '../../../devices/pci0000:00/0000:00:1d.1/usb2')
l('sys/bus/usb/devices/3-1', '../../../devices/pci0000:00/0000:00:1d.2/usb3/3-1')
d('sys/bus/pci', 0o755)
f('sys/bus/pci/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/pci/drivers', 0o755)
d('sys/bus/pci/drivers/pcieport-driver', 0o755)
l('sys/bus/pci/drivers/pcieport-driver/0000:00:1c.2', '../../../../devices/pci0000:00/0000:00:1c.2')
l('sys/bus/pci/drivers/pcieport-driver/0000:00:1c.1', '../../../../devices/pci0000:00/0000:00:1c.1')
l('sys/bus/pci/drivers/pcieport-driver/0000:00:1c.3', '../../../../devices/pci0000:00/0000:00:1c.3')
l('sys/bus/pci/drivers/pcieport-driver/0000:00:1c.0', '../../../../devices/pci0000:00/0000:00:1c.0')
l('sys/bus/pci/drivers/pcieport-driver/0000:00:01.0', '../../../../devices/pci0000:00/0000:00:01.0')
d('sys/bus/pci/drivers/uhci_hcd', 0o755)
l('sys/bus/pci/drivers/uhci_hcd/0000:00:1d.3', '../../../../devices/pci0000:00/0000:00:1d.3')
l('sys/bus/pci/drivers/uhci_hcd/module', '../../../../module/uhci_hcd')
l('sys/bus/pci/drivers/uhci_hcd/0000:00:1d.0', '../../../../devices/pci0000:00/0000:00:1d.0')
l('sys/bus/pci/drivers/uhci_hcd/0000:00:1d.2', '../../../../devices/pci0000:00/0000:00:1d.2')
l('sys/bus/pci/drivers/uhci_hcd/0000:00:1d.1', '../../../../devices/pci0000:00/0000:00:1d.1')
d('sys/bus/pci/drivers/HDA Intel', 0o755)
l('sys/bus/pci/drivers/HDA Intel/0000:00:1b.0', '../../../../devices/pci0000:00/0000:00:1b.0')
l('sys/bus/pci/drivers/HDA Intel/module', '../../../../module/snd_hda_intel')
d('sys/bus/pci/drivers/ata_piix', 0o755)
l('sys/bus/pci/drivers/ata_piix/0000:00:1f.1', '../../../../devices/pci0000:00/0000:00:1f.1')
d('sys/bus/pci/drivers/yenta_cardbus', 0o755)
l('sys/bus/pci/drivers/yenta_cardbus/module', '../../../../module/yenta_socket')
l('sys/bus/pci/drivers/yenta_cardbus/0000:15:00.0', '../../../../devices/pci0000:00/0000:00:1e.0/0000:15:00.0')
d('sys/bus/pci/drivers/e1000e', 0o755)
l('sys/bus/pci/drivers/e1000e/0000:02:00.0', '../../../../devices/pci0000:00/0000:00:1c.0/0000:02:00.0')
l('sys/bus/pci/drivers/e1000e/module', '../../../../module/e1000e')
d('sys/bus/pci/drivers/iwl3945', 0o755)
l('sys/bus/pci/drivers/iwl3945/module', '../../../../module/iwl3945')
l('sys/bus/pci/drivers/iwl3945/0000:03:00.0', '../../../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0')
d('sys/bus/pci/drivers/ehci_hcd', 0o755)
l('sys/bus/pci/drivers/ehci_hcd/module', '../../../../module/ehci_hcd')
d('sys/bus/pci/drivers/agpgart-intel', 0o755)
l('sys/bus/pci/drivers/agpgart-intel/module', '../../../../module/intel_agp')
d('sys/bus/pci/drivers/ahci', 0o755)
l('sys/bus/pci/drivers/ahci/module', '../../../../module/ahci')
l('sys/bus/pci/drivers/ahci/0000:00:1f.2', '../../../../devices/pci0000:00/0000:00:1f.2')
d('sys/bus/pci/devices', 0o755)
l('sys/bus/pci/devices/0000:00:1d.3', '../../../devices/pci0000:00/0000:00:1d.3')
l('sys/bus/pci/devices/0000:00:1c.2', '../../../devices/pci0000:00/0000:00:1c.2')
l('sys/bus/pci/devices/0000:00:1b.0', '../../../devices/pci0000:00/0000:00:1b.0')
l('sys/bus/pci/devices/0000:02:00.0', '../../../devices/pci0000:00/0000:00:1c.0/0000:02:00.0')
l('sys/bus/pci/devices/0000:00:1c.1', '../../../devices/pci0000:00/0000:00:1c.1')
l('sys/bus/pci/devices/0000:00:00.0', '../../../devices/pci0000:00/0000:00:00.0')
l('sys/bus/pci/devices/0000:01:00.0', '../../../devices/pci0000:00/0000:00:01.0/0000:01:00.0')
l('sys/bus/pci/devices/0000:00:1c.3', '../../../devices/pci0000:00/0000:00:1c.3')
l('sys/bus/pci/devices/0000:00:1f.0', '../../../devices/pci0000:00/0000:00:1f.0')
l('sys/bus/pci/devices/0000:00:1c.0', '../../../devices/pci0000:00/0000:00:1c.0')
l('sys/bus/pci/devices/0000:15:00.0', '../../../devices/pci0000:00/0000:00:1e.0/0000:15:00.0')
l('sys/bus/pci/devices/0000:00:1f.3', '../../../devices/pci0000:00/0000:00:1f.3')
l('sys/bus/pci/devices/0000:00:1d.0', '../../../devices/pci0000:00/0000:00:1d.0')
l('sys/bus/pci/devices/0000:00:1f.2', '../../../devices/pci0000:00/0000:00:1f.2')
l('sys/bus/pci/devices/0000:00:1d.2', '../../../devices/pci0000:00/0000:00:1d.2')
l('sys/bus/pci/devices/0000:03:00.0', '../../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0')
l('sys/bus/pci/devices/0000:00:01.0', '../../../devices/pci0000:00/0000:00:01.0')
l('sys/bus/pci/devices/0000:00:1f.1', '../../../devices/pci0000:00/0000:00:1f.1')
l('sys/bus/pci/devices/0000:00:1e.0', '../../../devices/pci0000:00/0000:00:1e.0')
l('sys/bus/pci/devices/0000:00:1d.1', '../../../devices/pci0000:00/0000:00:1d.1')
d('sys/bus/pnp', 0o755)
f('sys/bus/pnp/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/pnp/drivers', 0o755)
d('sys/bus/pnp/drivers/system', 0o755)
l('sys/bus/pnp/drivers/system/00:02', '../../../../devices/pnp0/00:02')
l('sys/bus/pnp/drivers/system/00:00', '../../../../devices/pnp0/00:00')
d('sys/bus/pnp/drivers/i8042 kbd', 0o755)
l('sys/bus/pnp/drivers/i8042 kbd/00:08', '../../../../devices/pnp0/00:08')
d('sys/bus/pnp/drivers/rtc_cmos', 0o755)
l('sys/bus/pnp/drivers/rtc_cmos/00:07', '../../../../devices/pnp0/00:07')
d('sys/bus/pnp/drivers/i8042 aux', 0o755)
l('sys/bus/pnp/drivers/i8042 aux/00:09', '../../../../devices/pnp0/00:09')
d('sys/bus/pnp/devices', 0o755)
l('sys/bus/pnp/devices/00:04', '../../../devices/pnp0/00:04')
l('sys/bus/pnp/devices/00:0a', '../../../devices/pnp0/00:0a')
l('sys/bus/pnp/devices/00:03', '../../../devices/pnp0/00:03')
l('sys/bus/pnp/devices/00:02', '../../../devices/pnp0/00:02')
l('sys/bus/pnp/devices/00:00', '../../../devices/pnp0/00:00')
l('sys/bus/pnp/devices/00:09', '../../../devices/pnp0/00:09')
l('sys/bus/pnp/devices/00:07', '../../../devices/pnp0/00:07')
l('sys/bus/pnp/devices/00:06', '../../../devices/pnp0/00:06')
l('sys/bus/pnp/devices/00:08', '../../../devices/pnp0/00:08')
l('sys/bus/pnp/devices/00:05', '../../../devices/pnp0/00:05')
l('sys/bus/pnp/devices/00:01', '../../../devices/pnp0/00:01')
d('sys/bus/pcmcia', 0o755)
f('sys/bus/pcmcia/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/platform', 0o755)
f('sys/bus/platform/drivers_autoprobe', 0o644, b'1\n')
d('sys/bus/platform/drivers', 0o755)
d('sys/bus/platform/drivers/pcspkr', 0o755)
l('sys/bus/platform/drivers/pcspkr/pcspkr', '../../../../devices/platform/pcspkr')
l('sys/bus/platform/drivers/pcspkr/module', '../../../../module/pcspkr')
d('sys/bus/platform/drivers/serial8250', 0o755)
l('sys/bus/platform/drivers/serial8250/serial8250', '../../../../devices/platform/serial8250')
d('sys/bus/platform/drivers/thinkpad_acpi', 0o755)
l('sys/bus/platform/drivers/thinkpad_acpi/module', '../../../../module/thinkpad_acpi')
l('sys/bus/platform/drivers/thinkpad_acpi/thinkpad_acpi', '../../../../devices/platform/thinkpad_acpi')
f('sys/bus/platform/drivers/thinkpad_acpi/version', 0o644, b'ThinkPad ACPI Extras v0.21\n')
f('sys/bus/platform/drivers/thinkpad_acpi/debug_level', 0o644, b'0x0000\n')
f('sys/bus/platform/drivers/thinkpad_acpi/interface_version', 0o644, b'0x00020200\n')
d('sys/bus/platform/drivers/vesafb', 0o755)
l('sys/bus/platform/drivers/vesafb/vesafb.0', '../../../../devices/platform/vesafb.0')
d('sys/bus/platform/drivers/thinkpad_hwmon', 0o755)
l('sys/bus/platform/drivers/thinkpad_hwmon/module', '../../../../module/thinkpad_acpi')
l('sys/bus/platform/drivers/thinkpad_hwmon/thinkpad_hwmon', '../../../../devices/platform/thinkpad_hwmon')
f('sys/bus/platform/drivers/thinkpad_hwmon/version', 0o644, b'ThinkPad ACPI Extras v0.21\n')
f('sys/bus/platform/drivers/thinkpad_hwmon/fan_watchdog', 0o644, b'0\n')
f('sys/bus/platform/drivers/thinkpad_hwmon/debug_level', 0o644, b'0x0000\n')
f('sys/bus/platform/drivers/thinkpad_hwmon/interface_version', 0o644, b'0x00020200\n')
d('sys/bus/platform/drivers/i8042', 0o755)
l('sys/bus/platform/drivers/i8042/i8042', '../../../../devices/platform/i8042')
d('sys/bus/platform/devices', 0o755)
l('sys/bus/platform/devices/pcspkr', '../../../devices/platform/pcspkr')
l('sys/bus/platform/devices/vesafb.0', '../../../devices/platform/vesafb.0')
l('sys/bus/platform/devices/dock.0', '../../../devices/platform/dock.0')
l('sys/bus/platform/devices/serial8250', '../../../devices/platform/serial8250')
l('sys/bus/platform/devices/thinkpad_acpi', '../../../devices/platform/thinkpad_acpi')
l('sys/bus/platform/devices/thinkpad_hwmon', '../../../devices/platform/thinkpad_hwmon')
l('sys/bus/platform/devices/microcode', '../../../devices/platform/microcode')
l('sys/bus/platform/devices/i8042', '../../../devices/platform/i8042')
d('sys/fs', 0o775)
d('sys/fs/fuse', 0o775)
d('sys/fs/fuse/connections', 0o775)
d('sys/fs/fuse/connections/16', 0o775)
f('sys/fs/fuse/connections/16/waiting', 0o664, b'0\n')
f('sys/fs/fuse/connections/16/abort', 0o664, b'')
d('sys/dev', 0o755)
d('sys/dev/char', 0o755)
l('sys/dev/char/4:39', '../../devices/virtual/tty/tty39')
l('sys/dev/char/116:33', '../../devices/virtual/sound/timer')
l('sys/dev/char/10:229', '../../devices/virtual/misc/fuse')
l('sys/dev/char/4:23', '../../devices/virtual/tty/tty23')
l('sys/dev/char/4:19', '../../devices/virtual/tty/tty19')
l('sys/dev/char/252:1', '../../devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00')
l('sys/dev/char/4:48', '../../devices/virtual/tty/tty48')
l('sys/dev/char/7:129', '../../devices/virtual/vc/vcsa1')
l('sys/dev/char/4:34', '../../devices/virtual/tty/tty34')
l('sys/dev/char/189:0', '../../devices/pci0000:00/0000:00:1d.0/usb1')
l('sys/dev/char/4:61', '../../devices/virtual/tty/tty61')
l('sys/dev/char/4:5', '../../devices/virtual/tty/tty5')
l('sys/dev/char/10:1', '../../devices/virtual/misc/psaux')
l('sys/dev/char/4:47', '../../devices/virtual/tty/tty47')
l('sys/dev/char/254:2', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0')
l('sys/dev/char/252:7', '../../devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00')
l('sys/dev/char/7:1', '../../devices/virtual/vc/vcs1')
l('sys/dev/char/4:37', '../../devices/virtual/tty/tty37')
l('sys/dev/char/4:3', '../../devices/virtual/tty/tty3')
l('sys/dev/char/252:3', '../../devices/pci0000:00/0000:00:1d.1/usb2/usb_endpoint/usbdev2.1_ep00')
l('sys/dev/char/1:3', '../../devices/virtual/mem/null')
l('sys/dev/char/14:12', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/adsp')
l('sys/dev/char/7:0', '../../devices/virtual/vc/vcs')
l('sys/dev/char/252:14', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02')
l('sys/dev/char/4:52', '../../devices/virtual/tty/tty52')
l('sys/dev/char/252:18', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04')
l('sys/dev/char/1:5', '../../devices/virtual/mem/zero')
l('sys/dev/char/13:66', '../../devices/platform/pcspkr/input/input2/event2')
l('sys/dev/char/10:223', '../../devices/virtual/misc/uinput')
l('sys/dev/char/7:2', '../../devices/virtual/vc/vcs2')
l('sys/dev/char/4:40', '../../devices/virtual/tty/tty40')
l('sys/dev/char/4:31', '../../devices/virtual/tty/tty31')
l('sys/dev/char/252:21', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02')
l('sys/dev/char/4:11', '../../devices/virtual/tty/tty11')
l('sys/dev/char/4:17', '../../devices/virtual/tty/tty17')
l('sys/dev/char/252:31', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84')
l('sys/dev/char/189:256', '../../devices/pci0000:00/0000:00:1d.2/usb3')
l('sys/dev/char/252:5', '../../devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00')
l('sys/dev/char/7:4', '../../devices/virtual/vc/vcs4')
l('sys/dev/char/4:38', '../../devices/virtual/tty/tty38')
l('sys/dev/char/4:66', '../../devices/platform/serial8250/tty/ttyS2')
l('sys/dev/char/4:0', '../../devices/virtual/tty/tty0')
l('sys/dev/char/21:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0')
l('sys/dev/char/7:134', '../../devices/virtual/vc/vcsa6')
l('sys/dev/char/116:24', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c')
l('sys/dev/char/4:29', '../../devices/virtual/tty/tty29')
l('sys/dev/char/21:2', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2')
l('sys/dev/char/4:27', '../../devices/virtual/tty/tty27')
l('sys/dev/char/4:21', '../../devices/virtual/tty/tty21')
l('sys/dev/char/13:64', '../../devices/platform/i8042/serio0/input/input0/event0')
l('sys/dev/char/252:30', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83')
l('sys/dev/char/4:51', '../../devices/virtual/tty/tty51')
l('sys/dev/char/254:1', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0')
l('sys/dev/char/4:18', '../../devices/virtual/tty/tty18')
l('sys/dev/char/7:3', '../../devices/virtual/vc/vcs3')
l('sys/dev/char/252:32', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02')
l('sys/dev/char/252:27', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81')
l('sys/dev/char/252:12', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81')
l('sys/dev/char/1:1', '../../devices/virtual/mem/mem')
l('sys/dev/char/189:386', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2')
l('sys/dev/char/10:62', '../../devices/virtual/misc/network_latency')
l('sys/dev/char/252:4', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81')
l('sys/dev/char/5:1', '../../devices/virtual/tty/console')
l('sys/dev/char/1:7', '../../devices/virtual/mem/full')
l('sys/dev/char/7:5', '../../devices/virtual/vc/vcs5')
l('sys/dev/char/4:4', '../../devices/virtual/tty/tty4')
l('sys/dev/char/4:15', '../../devices/virtual/tty/tty15')
l('sys/dev/char/4:9', '../../devices/virtual/tty/tty9')
l('sys/dev/char/4:20', '../../devices/virtual/tty/tty20')
l('sys/dev/char/10:228', '../../devices/virtual/misc/hpet')
l('sys/dev/char/4:59', '../../devices/virtual/tty/tty59')
l('sys/dev/char/252:2', '../../devices/pci0000:00/0000:00:1d.1/usb2/2-0:1.0/usb_endpoint/usbdev2.1_ep81')
l('sys/dev/char/4:7', '../../devices/virtual/tty/tty7')
l('sys/dev/char/4:35', '../../devices/virtual/tty/tty35')
l('sys/dev/char/254:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0')
l('sys/dev/char/189:128', '../../devices/pci0000:00/0000:00:1d.1/usb2')
l('sys/dev/char/10:184', '../../devices/virtual/misc/microcode')
l('sys/dev/char/7:131', '../../devices/virtual/vc/vcsa3')
l('sys/dev/char/116:1', '../../devices/virtual/sound/seq')
l('sys/dev/char/252:17', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84')
l('sys/dev/char/10:61', '../../devices/virtual/misc/network_throughput')
l('sys/dev/char/7:130', '../../devices/virtual/vc/vcsa2')
l('sys/dev/char/252:0', '../../devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81')
l('sys/dev/char/4:33', '../../devices/virtual/tty/tty33')
l('sys/dev/char/4:10', '../../devices/virtual/tty/tty10')
l('sys/dev/char/13:71', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7')
l('sys/dev/char/252:10', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81')
l('sys/dev/char/253:0', '../../devices/pnp0/00:07/rtc/rtc0')
l('sys/dev/char/252:19', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00')
l('sys/dev/char/7:10', '../../devices/virtual/vc/vcs10')
l('sys/dev/char/4:42', '../../devices/virtual/tty/tty42')
l('sys/dev/char/10:144', '../../devices/virtual/misc/nvram')
l('sys/dev/char/4:63', '../../devices/virtual/tty/tty63')
l('sys/dev/char/252:25', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81')
l('sys/dev/char/189:520', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2')
l('sys/dev/char/4:55', '../../devices/virtual/tty/tty55')
l('sys/dev/char/13:33', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1')
l('sys/dev/char/13:70', '../../devices/virtual/input/input6/event6')
l('sys/dev/char/252:29', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01')
l('sys/dev/char/4:45', '../../devices/virtual/tty/tty45')
l('sys/dev/char/4:56', '../../devices/virtual/tty/tty56')
l('sys/dev/char/10:60', '../../devices/virtual/misc/device-mapper')
l('sys/dev/char/5:2', '../../devices/virtual/tty/ptmx')
l('sys/dev/char/252:8', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81')
l('sys/dev/char/14:3', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/dsp')
l('sys/dev/char/13:68', '../../devices/LNXSYSTM:00/device:00/PNP0C0D:00/input/input4/event4')
l('sys/dev/char/13:67', '../../devices/LNXSYSTM:00/LNXPWRBN:00/input/input3/event3')
l('sys/dev/char/1:4', '../../devices/virtual/mem/port')
l('sys/dev/char/116:0', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0')
l('sys/dev/char/4:46', '../../devices/virtual/tty/tty46')
l('sys/dev/char/4:41', '../../devices/virtual/tty/tty41')
l('sys/dev/char/189:512', '../../devices/pci0000:00/0000:00:1d.7/usb5')
l('sys/dev/char/4:62', '../../devices/virtual/tty/tty62')
l('sys/dev/char/252:15', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83')
l('sys/dev/char/7:133', '../../devices/virtual/vc/vcsa5')
l('sys/dev/char/4:2', '../../devices/virtual/tty/tty2')
l('sys/dev/char/1:8', '../../devices/virtual/mem/random')
l('sys/dev/char/7:138', '../../devices/virtual/vc/vcsa10')
l('sys/dev/char/4:64', '../../devices/platform/serial8250/tty/ttyS0')
l('sys/dev/char/4:1', '../../devices/virtual/tty/tty1')
l('sys/dev/char/4:28', '../../devices/virtual/tty/tty28')
l('sys/dev/char/13:32', '../../devices/platform/i8042/serio1/input/input1/mouse0')
l('sys/dev/char/4:44', '../../devices/virtual/tty/tty44')
l('sys/dev/char/4:36', '../../devices/virtual/tty/tty36')
l('sys/dev/char/4:43', '../../devices/virtual/tty/tty43')
l('sys/dev/char/7:128', '../../devices/virtual/vc/vcsa')
l('sys/dev/char/4:60', '../../devices/virtual/tty/tty60')
l('sys/dev/char/7:135', '../../devices/virtual/vc/vcsa7')
l('sys/dev/char/4:6', '../../devices/virtual/tty/tty6')
l('sys/dev/char/4:49', '../../devices/virtual/tty/tty49')
l('sys/dev/char/189:518', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1')
l('sys/dev/char/29:0', '../../devices/platform/vesafb.0/graphics/fb0')
l('sys/dev/char/252:23', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00')
l('sys/dev/char/10:227', '../../devices/virtual/misc/mcelog')
l('sys/dev/char/4:14', '../../devices/virtual/tty/tty14')
l('sys/dev/char/4:54', '../../devices/virtual/tty/tty54')
l('sys/dev/char/7:132', '../../devices/virtual/vc/vcsa4')
l('sys/dev/char/5:0', '../../devices/virtual/tty/tty')
l('sys/dev/char/252:26', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00')
l('sys/dev/char/1:2', '../../devices/virtual/mem/kmem')
l('sys/dev/char/189:259', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1')
l('sys/dev/char/252:11', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02')
l('sys/dev/char/4:26', '../../devices/virtual/tty/tty26')
l('sys/dev/char/252:33', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00')
l('sys/dev/char/4:22', '../../devices/virtual/tty/tty22')
l('sys/dev/char/10:63', '../../devices/virtual/misc/cpu_dma_latency')
l('sys/dev/char/4:13', '../../devices/virtual/tty/tty13')
l('sys/dev/char/116:17', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p')
l('sys/dev/char/116:16', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p')
l('sys/dev/char/252:16', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03')
l('sys/dev/char/4:8', '../../devices/virtual/tty/tty8')
l('sys/dev/char/252:28', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82')
l('sys/dev/char/21:1', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1')
l('sys/dev/char/4:24', '../../devices/virtual/tty/tty24')
l('sys/dev/char/7:7', '../../devices/virtual/vc/vcs7')
l('sys/dev/char/252:20', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81')
l('sys/dev/char/166:0', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0')
l('sys/dev/char/4:58', '../../devices/virtual/tty/tty58')
l('sys/dev/char/4:57', '../../devices/virtual/tty/tty57')
l('sys/dev/char/189:384', '../../devices/pci0000:00/0000:00:1d.3/usb4')
l('sys/dev/char/4:50', '../../devices/virtual/tty/tty50')
l('sys/dev/char/7:6', '../../devices/virtual/vc/vcs6')
l('sys/dev/char/4:25', '../../devices/virtual/tty/tty25')
l('sys/dev/char/4:16', '../../devices/virtual/tty/tty16')
l('sys/dev/char/4:65', '../../devices/platform/serial8250/tty/ttyS1')
l('sys/dev/char/13:65', '../../devices/platform/i8042/serio1/input/input1/event1')
l('sys/dev/char/13:69', '../../devices/LNXSYSTM:00/device:00/PNP0C0E:00/input/input5/event5')
l('sys/dev/char/252:22', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83')
l('sys/dev/char/1:11', '../../devices/virtual/mem/kmsg')
l('sys/dev/char/189:385', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1')
l('sys/dev/char/4:12', '../../devices/virtual/tty/tty12')
l('sys/dev/char/252:13', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82')
l('sys/dev/char/1:9', '../../devices/virtual/mem/urandom')
l('sys/dev/char/4:67', '../../devices/platform/serial8250/tty/ttyS3')
l('sys/dev/char/13:63', '../../devices/virtual/input/mice')
l('sys/dev/char/252:24', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00')
l('sys/dev/char/252:9', '../../devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00')
l('sys/dev/char/4:53', '../../devices/virtual/tty/tty53')
l('sys/dev/char/4:30', '../../devices/virtual/tty/tty30')
l('sys/dev/char/14:4', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/audio')
l('sys/dev/char/14:0', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/mixer')
l('sys/dev/char/252:6', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81')
l('sys/dev/char/4:32', '../../devices/virtual/tty/tty32')
d('sys/dev/block', 0o755)
l('sys/dev/block/7:1', '../../devices/virtual/block/loop1')
l('sys/dev/block/7:0', '../../devices/virtual/block/loop0')
l('sys/dev/block/8:17', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1')
l('sys/dev/block/7:2', '../../devices/virtual/block/loop2')
l('sys/dev/block/7:4', '../../devices/virtual/block/loop4')
l('sys/dev/block/8:8', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8')
l('sys/dev/block/7:3', '../../devices/virtual/block/loop3')
l('sys/dev/block/11:0', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0')
l('sys/dev/block/7:5', '../../devices/virtual/block/loop5')
l('sys/dev/block/8:5', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5')
l('sys/dev/block/8:10', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10')
l('sys/dev/block/8:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda')
l('sys/dev/block/8:9', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9')
l('sys/dev/block/8:7', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7')
l('sys/dev/block/8:16', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb')
l('sys/dev/block/9:0', '../../devices/virtual/block/md0')
l('sys/dev/block/7:7', '../../devices/virtual/block/loop7')
l('sys/dev/block/8:6', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6')
l('sys/dev/block/7:6', '../../devices/virtual/block/loop6')
l('sys/dev/block/8:1', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1')
d('sys/module', 0o775)
d('sys/module/arc4', 0o775)
f('sys/module/arc4/initstate', 0o664, b'live\n')
f('sys/module/arc4/srcversion', 0o664, b'0E7B177AF22D87B5B21A577\n')
f('sys/module/arc4/refcnt', 0o664, b'2\n')
d('sys/module/arc4/sections', 0o775)
f('sys/module/arc4/sections/.exit.text', 0o664, b'0xffffffffa006b0dc\n')
f('sys/module/arc4/sections/.note.gnu.build-id', 0o664, b'0xffffffffa006b0f0\n')
f('sys/module/arc4/sections/.strtab', 0o664, b'0xffffffffa006b400\n')
f('sys/module/arc4/sections/.bss', 0o664, b'0xffffffffa006ba00\n')
f('sys/module/arc4/sections/.text', 0o664, b'0xffffffffa006b000\n')
f('sys/module/arc4/sections/.init.text', 0o664, b'0xffffffffa009a000\n')
f('sys/module/arc4/sections/.data', 0o664, b'0xffffffffa006b520\n')
f('sys/module/arc4/sections/.symtab', 0o664, b'0xffffffffa006b118\n')
f('sys/module/arc4/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa006b640\n')
d('sys/module/arc4/notes', 0o775)
f('sys/module/arc4/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00[P \tR\xc2\xa7\xdd\xf0H\xaa\x06\x1a,\xb3\xa1\xa4\x9a\x13G')
d('sys/module/vt', 0o775)
d('sys/module/vt/parameters', 0o775)
f('sys/module/vt/parameters/default_grn', 0o664, b'0,0,170,85,0,0,170,170,85,85,255,255,85,85,255,255\n')
f('sys/module/vt/parameters/default_red', 0o664, b'0,170,0,170,0,170,0,170,85,255,85,255,85,255,85,255\n')
f('sys/module/vt/parameters/italic', 0o664, b'2\n')
f('sys/module/vt/parameters/underline', 0o664, b'3\n')
f('sys/module/vt/parameters/default_blu', 0o664, b'0,0,0,0,170,170,170,170,85,85,85,85,255,255,255,255\n')
f('sys/module/vt/parameters/default_utf8', 0o664, b'1\n')
d('sys/module/cdrom', 0o775)
f('sys/module/cdrom/initstate', 0o664, b'live\n')
f('sys/module/cdrom/srcversion', 0o664, b'D868CCB0789DA27F51280A8\n')
f('sys/module/cdrom/refcnt', 0o664, b'1\n')
d('sys/module/cdrom/holders', 0o775)
l('sys/module/cdrom/holders/sr_mod', '../../sr_mod')
d('sys/module/cdrom/sections', 0o775)
f('sys/module/cdrom/sections/__ksymtab_strings', 0o664, b'0xffffffffa002fe20\n')
f('sys/module/cdrom/sections/.exit.text', 0o664, b'0xffffffffa002e0f8\n')
f('sys/module/cdrom/sections/.note.gnu.build-id', 0o664, b'0xffffffffa002e120\n')
f('sys/module/cdrom/sections/.strtab', 0o664, b'0xffffffffa0030f60\n')
f('sys/module/cdrom/sections/__ksymtab', 0o664, b'0xffffffffa002fc10\n')
f('sys/module/cdrom/sections/.rodata', 0o664, b'0xffffffffa002e160\n')
f('sys/module/cdrom/sections/__param', 0o664, b'0xffffffffa002fd30\n')
f('sys/module/cdrom/sections/.bss', 0o664, b'0xffffffffa0032380\n')
f('sys/module/cdrom/sections/.text', 0o664, b'0xffffffffa0029000\n')
f('sys/module/cdrom/sections/.init.text', 0o664, b'0xffffffffa0034000\n')
f('sys/module/cdrom/sections/__kcrctab', 0o664, b'0xffffffffa002fcd0\n')
f('sys/module/cdrom/sections/.data', 0o664, b'0xffffffffa0031b80\n')
f('sys/module/cdrom/sections/.symtab', 0o664, b'0xffffffffa002fef8\n')
f('sys/module/cdrom/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0031fc0\n')
f('sys/module/cdrom/sections/.rodata.str1.1', 0o664, b'0xffffffffa002f896\n')
d('sys/module/cdrom/notes', 0o775)
f('sys/module/cdrom/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00C\x03\x04\xe9\x8a\x17\x19\x8b\xc7Y\xea\x1a\xa35\x85K\x9e\x1e\xf8=')
d('sys/module/rfkill', 0o775)
f('sys/module/rfkill/version', 0o664, b'1.0\n')
f('sys/module/rfkill/initstate', 0o664, b'live\n')
f('sys/module/rfkill/srcversion', 0o664, b'6B4E68548C57FD365EB4E71\n')
f('sys/module/rfkill/refcnt', 0o664, b'2\n')
d('sys/module/rfkill/holders', 0o775)
l('sys/module/rfkill/holders/thinkpad_acpi', '../../thinkpad_acpi')
d('sys/module/rfkill/parameters', 0o775)
f('sys/module/rfkill/parameters/default_state', 0o664, b'1\n')
d('sys/module/rfkill/sections', 0o775)
f('sys/module/rfkill/sections/__ksymtab_gpl', 0o664, b'0xffffffffa01d9f70\n')
f('sys/module/rfkill/sections/.smp_locks', 0o664, b'0xffffffffa01d9e80\n')
f('sys/module/rfkill/sections/__bug_table', 0o664, b'0xffffffffa01d9e36\n')
f('sys/module/rfkill/sections/__ksymtab_strings', 0o664, b'0xffffffffa01d9fe0\n')
f('sys/module/rfkill/sections/.exit.text', 0o664, b'0xffffffffa01d9c88\n')
f('sys/module/rfkill/sections/.note.gnu.build-id', 0o664, b'0xffffffffa01d9cb0\n')
f('sys/module/rfkill/sections/.strtab', 0o664, b'0xffffffffa01dae08\n')
f('sys/module/rfkill/sections/__ksymtab', 0o664, b'0xffffffffa01d9ee0\n')
f('sys/module/rfkill/sections/.rodata', 0o664, b'0xffffffffa01d9cd8\n')
f('sys/module/rfkill/sections/__param', 0o664, b'0xffffffffa01d9fb8\n')
f('sys/module/rfkill/sections/.bss', 0o664, b'0xffffffffa01dbe00\n')
f('sys/module/rfkill/sections/.text', 0o664, b'0xffffffffa01d9000\n')
f('sys/module/rfkill/sections/.init.text', 0o664, b'0xffffffffa009a000\n')
f('sys/module/rfkill/sections/__kcrctab', 0o664, b'0xffffffffa01d9f40\n')
f('sys/module/rfkill/sections/.data', 0o664, b'0xffffffffa01db780\n')
f('sys/module/rfkill/sections/.symtab', 0o664, b'0xffffffffa01da088\n')
f('sys/module/rfkill/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01dba40\n')
f('sys/module/rfkill/sections/.rodata.str1.1', 0o664, b'0xffffffffa01d9d36\n')
f('sys/module/rfkill/sections/__kcrctab_gpl', 0o664, b'0xffffffffa01d9fa0\n')
d('sys/module/rfkill/notes', 0o775)
f('sys/module/rfkill/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xd1\x1fs\x98\x9b\x97\xc4\xec\x9dd\xac:\x94m\x1e\xc0Y\xcfd"')
d('sys/module/uinput', 0o775)
f('sys/module/uinput/version', 0o664, b'0.3\n')
f('sys/module/uinput/initstate', 0o664, b'live\n')
f('sys/module/uinput/srcversion', 0o664, b'2C3E0DB9E2496AD70174B85\n')
f('sys/module/uinput/refcnt', 0o664, b'0\n')
d('sys/module/uinput/sections', 0o775)
f('sys/module/uinput/sections/.smp_locks', 0o664, b'0xffffffffa0206448\n')
f('sys/module/uinput/sections/.exit.text', 0o664, b'0xffffffffa0206278\n')
f('sys/module/uinput/sections/.note.gnu.build-id', 0o664, b'0xffffffffa020628c\n')
f('sys/module/uinput/sections/.strtab', 0o664, b'0xffffffffa0206c88\n')
f('sys/module/uinput/sections/.rodata', 0o664, b'0xffffffffa02062c0\n')
f('sys/module/uinput/sections/.bss', 0o664, b'0xffffffffa02074c0\n')
f('sys/module/uinput/sections/.text', 0o664, b'0xffffffffa0205000\n')
f('sys/module/uinput/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/uinput/sections/.data', 0o664, b'0xffffffffa02070c0\n')
f('sys/module/uinput/sections/.symtab', 0o664, b'0xffffffffa0206490\n')
f('sys/module/uinput/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0207100\n')
f('sys/module/uinput/sections/.rodata.str1.1', 0o664, b'0xffffffffa0206398\n')
d('sys/module/uinput/notes', 0o775)
f('sys/module/uinput/notes/.note.gnu.build-id', 0o664, b"\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00'\n{\xdf\xe9m\x80\xea\xa8k\x8f=\x90\x8c\xe1\xf7{\x06\x15S")
d('sys/module/fuse', 0o775)
f('sys/module/fuse/initstate', 0o664, b'live\n')
f('sys/module/fuse/srcversion', 0o664, b'A6C92EE44A51B1B0DC4377D\n')
f('sys/module/fuse/refcnt', 0o664, b'3\n')
d('sys/module/fuse/sections', 0o775)
f('sys/module/fuse/sections/.smp_locks', 0o664, b'0xffffffffa0249d78\n')
f('sys/module/fuse/sections/__bug_table', 0o664, b'0xffffffffa024a032\n')
f('sys/module/fuse/sections/.exit.text', 0o664, b'0xffffffffa02493d0\n')
f('sys/module/fuse/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0249414\n')
f('sys/module/fuse/sections/.strtab', 0o664, b'0xffffffffa024c4e8\n')
f('sys/module/fuse/sections/.rodata', 0o664, b'0xffffffffa0249440\n')
f('sys/module/fuse/sections/.bss', 0o664, b'0xffffffffa024e300\n')
f('sys/module/fuse/sections/.text', 0o664, b'0xffffffffa0240000\n')
f('sys/module/fuse/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/fuse/sections/.data', 0o664, b'0xffffffffa024dba0\n')
f('sys/module/fuse/sections/.symtab', 0o664, b'0xffffffffa024a208\n')
f('sys/module/fuse/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa024df40\n')
f('sys/module/fuse/sections/.rodata.str1.1', 0o664, b'0xffffffffa0249e50\n')
d('sys/module/fuse/notes', 0o775)
f('sys/module/fuse/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x7fh\xfe:\x1f\xc6\xddW\x96\x80\x16\x9fK\x7f\xba,\x8a\xc6\xf6E')
d('sys/module/hid', 0o775)
d('sys/module/hid/parameters', 0o775)
f('sys/module/hid/parameters/pb_fnmode', 0o664, b'1\n')
d('sys/module/uhci_hcd', 0o775)
f('sys/module/uhci_hcd/initstate', 0o664, b'live\n')
f('sys/module/uhci_hcd/srcversion', 0o664, b'E3F4B6BEC99D6670259FCC9\n')
f('sys/module/uhci_hcd/refcnt', 0o664, b'0\n')
d('sys/module/uhci_hcd/drivers', 0o775)
l('sys/module/uhci_hcd/drivers/pci:uhci_hcd', '../../../bus/pci/drivers/uhci_hcd')
d('sys/module/uhci_hcd/parameters', 0o775)
f('sys/module/uhci_hcd/parameters/ignore_oc', 0o664, b'N\n')
d('sys/module/uhci_hcd/sections', 0o775)
f('sys/module/uhci_hcd/sections/.smp_locks', 0o664, b'0xffffffffa00c6850\n')
f('sys/module/uhci_hcd/sections/.exit.text', 0o664, b'0xffffffffa00c621c\n')
f('sys/module/uhci_hcd/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00c6254\n')
f('sys/module/uhci_hcd/sections/.strtab', 0o664, b'0xffffffffa00c7690\n')
f('sys/module/uhci_hcd/sections/.rodata', 0o664, b'0xffffffffa00c6280\n')
f('sys/module/uhci_hcd/sections/__param', 0o664, b'0xffffffffa00c6888\n')
f('sys/module/uhci_hcd/sections/.bss', 0o664, b'0xffffffffa00c8400\n')
f('sys/module/uhci_hcd/sections/.text', 0o664, b'0xffffffffa00c2000\n')
f('sys/module/uhci_hcd/sections/.init.text', 0o664, b'0xffffffffa0027000\n')
f('sys/module/uhci_hcd/sections/.data', 0o664, b'0xffffffffa00c7ee0\n')
f('sys/module/uhci_hcd/sections/.symtab', 0o664, b'0xffffffffa00c68b0\n')
f('sys/module/uhci_hcd/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00c8040\n')
f('sys/module/uhci_hcd/sections/.rodata.str1.1', 0o664, b'0xffffffffa00c67d2\n')
d('sys/module/uhci_hcd/notes', 0o775)
f('sys/module/uhci_hcd/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00t\xd0\xa5\xd2A\xa2\x874\x12\xb5\xf6\x9c\xff\xb9\xee`\x8d\x9f[\x04')
d('sys/module/crypto_blkcipher', 0o775)
f('sys/module/crypto_blkcipher/initstate', 0o664, b'live\n')
f('sys/module/crypto_blkcipher/srcversion', 0o664, b'723F00BD2391209F4D65272\n')
f('sys/module/crypto_blkcipher/refcnt', 0o664, b'1\n')
d('sys/module/crypto_blkcipher/holders', 0o775)
l('sys/module/crypto_blkcipher/holders/ecb', '../../ecb')
d('sys/module/crypto_blkcipher/sections', 0o775)
f('sys/module/crypto_blkcipher/sections/__ksymtab_gpl', 0o664, b'0xffffffffa01f1610\n')
f('sys/module/crypto_blkcipher/sections/.smp_locks', 0o664, b'0xffffffffa01f18c0\n')
f('sys/module/crypto_blkcipher/sections/__bug_table', 0o664, b'0xffffffffa01f185f\n')
f('sys/module/crypto_blkcipher/sections/__ksymtab_strings', 0o664, b'0xffffffffa01f1748\n')
f('sys/module/crypto_blkcipher/sections/.exit.text', 0o664, b'0xffffffffa01f138c\n')
f('sys/module/crypto_blkcipher/sections/.note.gnu.build-id', 0o664, b'0xffffffffa01f13b0\n')
f('sys/module/crypto_blkcipher/sections/.strtab', 0o664, b'0xffffffffa01f2b10\n')
f('sys/module/crypto_blkcipher/sections/.rodata', 0o664, b'0xffffffffa01f13e0\n')
f('sys/module/crypto_blkcipher/sections/.bss', 0o664, b'0xffffffffa01f3ec0\n')
f('sys/module/crypto_blkcipher/sections/.text', 0o664, b'0xffffffffa01ef000\n')
f('sys/module/crypto_blkcipher/sections/.init.text', 0o664, b'0xffffffffa0027000\n')
f('sys/module/crypto_blkcipher/sections/.data', 0o664, b'0xffffffffa01f39e0\n')
f('sys/module/crypto_blkcipher/sections/.symtab', 0o664, b'0xffffffffa01f18f8\n')
f('sys/module/crypto_blkcipher/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01f3b00\n')
f('sys/module/crypto_blkcipher/sections/.rodata.str1.1', 0o664, b'0xffffffffa01f1440\n')
f('sys/module/crypto_blkcipher/sections/__kcrctab_gpl', 0o664, b'0xffffffffa01f16e0\n')
d('sys/module/crypto_blkcipher/notes', 0o775)
f('sys/module/crypto_blkcipher/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00-_P~\xd2\x1eM\xee\xb0!\xc0\xd8\xb7\x00\x1b\xe4#-\xc8\xe0')
d('sys/module/soundcore', 0o775)
f('sys/module/soundcore/initstate', 0o664, b'live\n')
f('sys/module/soundcore/srcversion', 0o664, b'E4F49ED9C4CFD1A5A923330\n')
f('sys/module/soundcore/refcnt', 0o664, b'1\n')
d('sys/module/soundcore/holders', 0o775)
l('sys/module/soundcore/holders/snd', '../../snd')
d('sys/module/soundcore/sections', 0o775)
f('sys/module/soundcore/sections/__ksymtab_strings', 0o664, b'0xffffffffa0067bf0\n')
f('sys/module/soundcore/sections/.exit.text', 0o664, b'0xffffffffa0067850\n')
f('sys/module/soundcore/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0067874\n')
f('sys/module/soundcore/sections/.strtab', 0o664, b'0xffffffffa00686d0\n')
f('sys/module/soundcore/sections/__ksymtab', 0o664, b'0xffffffffa0067b00\n')
f('sys/module/soundcore/sections/.rodata', 0o664, b'0xffffffffa00678a0\n')
f('sys/module/soundcore/sections/.bss', 0o664, b'0xffffffffa00692c0\n')
f('sys/module/soundcore/sections/.text', 0o664, b'0xffffffffa0067000\n')
f('sys/module/soundcore/sections/.init.text', 0o664, b'0xffffffffa006b000\n')
f('sys/module/soundcore/sections/__kcrctab', 0o664, b'0xffffffffa0067ba0\n')
f('sys/module/soundcore/sections/.data', 0o664, b'0xffffffffa0068ec0\n')
f('sys/module/soundcore/sections/.symtab', 0o664, b'0xffffffffa0067cc8\n')
f('sys/module/soundcore/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0068f00\n')
f('sys/module/soundcore/sections/.rodata.str1.1', 0o664, b'0xffffffffa00679f8\n')
d('sys/module/soundcore/notes', 0o775)
f('sys/module/soundcore/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00^\x97fIs2\xf4Ck\xe0$]\xd8)]\xe2\x9c\xea\x1d\x0c')
d('sys/module/pcspkr', 0o775)
f('sys/module/pcspkr/initstate', 0o664, b'live\n')
f('sys/module/pcspkr/srcversion', 0o664, b'5757E9C81E627525BA9D165\n')
f('sys/module/pcspkr/refcnt', 0o664, b'0\n')
d('sys/module/pcspkr/drivers', 0o775)
l('sys/module/pcspkr/drivers/platform:pcspkr', '../../../bus/platform/drivers/pcspkr')
d('sys/module/pcspkr/sections', 0o775)
f('sys/module/pcspkr/sections/.exit.text', 0o664, b'0xffffffffa009e15c\n')
f('sys/module/pcspkr/sections/.note.gnu.build-id', 0o664, b'0xffffffffa009e250\n')
f('sys/module/pcspkr/sections/.strtab', 0o664, b'0xffffffffa009e6e8\n')
f('sys/module/pcspkr/sections/.devexit.text', 0o664, b'0xffffffffa009e16e\n')
f('sys/module/pcspkr/sections/.bss', 0o664, b'0xffffffffa009ed80\n')
f('sys/module/pcspkr/sections/.text', 0o664, b'0xffffffffa009e000\n')
f('sys/module/pcspkr/sections/.init.text', 0o664, b'0xffffffffa00a0000\n')
f('sys/module/pcspkr/sections/.data', 0o664, b'0xffffffffa009e900\n')
f('sys/module/pcspkr/sections/.devinit.text', 0o664, b'0xffffffffa009e1ad\n')
f('sys/module/pcspkr/sections/.symtab', 0o664, b'0xffffffffa009e298\n')
f('sys/module/pcspkr/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa009e9c0\n')
f('sys/module/pcspkr/sections/.rodata.str1.1', 0o664, b'0xffffffffa009e274\n')
d('sys/module/pcspkr/notes', 0o775)
f('sys/module/pcspkr/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xe3\xfd\xb1\xf7b\xd2\x8f\xc1\xd2C\xfc\xfe\xdf8\x80\x82\x85A\xc9!')
d('sys/module/sr_mod', 0o775)
f('sys/module/sr_mod/initstate', 0o664, b'live\n')
f('sys/module/sr_mod/srcversion', 0o664, b'3484F077F1A71803F453437\n')
f('sys/module/sr_mod/refcnt', 0o664, b'0\n')
d('sys/module/sr_mod/parameters', 0o775)
f('sys/module/sr_mod/parameters/xa_test', 0o664, b'0\n')
d('sys/module/sr_mod/sections', 0o775)
f('sys/module/sr_mod/sections/.smp_locks', 0o664, b'0xffffffffa0061488\n')
f('sys/module/sr_mod/sections/.exit.text', 0o664, b'0xffffffffa006126c\n')
f('sys/module/sr_mod/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0061290\n')
f('sys/module/sr_mod/sections/.strtab', 0o664, b'0xffffffffa00623f0\n')
f('sys/module/sr_mod/sections/.rodata', 0o664, b'0xffffffffa00612c0\n')
f('sys/module/sr_mod/sections/__param', 0o664, b'0xffffffffa0061750\n')
f('sys/module/sr_mod/sections/.bss', 0o664, b'0xffffffffa00630c0\n')
f('sys/module/sr_mod/sections/.text', 0o664, b'0xffffffffa005f000\n')
f('sys/module/sr_mod/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/sr_mod/sections/.data', 0o664, b'0xffffffffa0062aa0\n')
f('sys/module/sr_mod/sections/.symtab', 0o664, b'0xffffffffa0061778\n')
f('sys/module/sr_mod/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0062d00\n')
f('sys/module/sr_mod/sections/.rodata.str1.1', 0o664, b'0xffffffffa0061308\n')
d('sys/module/sr_mod/notes', 0o775)
f('sys/module/sr_mod/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xae\xa7\xdcHo\xe7\xe1\xa0#\x94\x9a\xfeV8\xa6O$@\xc1(')
d('sys/module/cfg80211', 0o775)
f('sys/module/cfg80211/initstate', 0o664, b'live\n')
f('sys/module/cfg80211/srcversion', 0o664, b'C43F275505B181F0D1625BE\n')
f('sys/module/cfg80211/refcnt', 0o664, b'2\n')
d('sys/module/cfg80211/holders', 0o775)
l('sys/module/cfg80211/holders/mac80211', '../../mac80211')
l('sys/module/cfg80211/holders/iwl3945', '../../iwl3945')
d('sys/module/cfg80211/parameters', 0o775)
f('sys/module/cfg80211/parameters/ieee80211_regdom', 0o664, b'US\n')
d('sys/module/cfg80211/sections', 0o775)
f('sys/module/cfg80211/sections/.smp_locks', 0o664, b'0xffffffffa00bd2a0\n')
f('sys/module/cfg80211/sections/__bug_table', 0o664, b'0xffffffffa00bd166\n')
f('sys/module/cfg80211/sections/__ksymtab_strings', 0o664, b'0xffffffffa00bd448\n')
f('sys/module/cfg80211/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00bce58\n')
f('sys/module/cfg80211/sections/.strtab', 0o664, b'0xffffffffa00be8c0\n')
f('sys/module/cfg80211/sections/__ksymtab', 0o664, b'0xffffffffa00bd370\n')
f('sys/module/cfg80211/sections/.rodata', 0o664, b'0xffffffffa00bce80\n')
f('sys/module/cfg80211/sections/.data.read_mostly', 0o664, b'0xffffffffa00bfe60\n')
f('sys/module/cfg80211/sections/__param', 0o664, b'0xffffffffa00bd518\n')
f('sys/module/cfg80211/sections/.bss', 0o664, b'0xffffffffa00c02c0\n')
f('sys/module/cfg80211/sections/.text', 0o664, b'0xffffffffa00b9000\n')
f('sys/module/cfg80211/sections/__kcrctab', 0o664, b'0xffffffffa00bd400\n')
f('sys/module/cfg80211/sections/.data', 0o664, b'0xffffffffa00bf760\n')
f('sys/module/cfg80211/sections/.symtab', 0o664, b'0xffffffffa00bd540\n')
f('sys/module/cfg80211/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00bff00\n')
f('sys/module/cfg80211/sections/.rodata.str1.1', 0o664, b'0xffffffffa00bd078\n')
d('sys/module/cfg80211/notes', 0o775)
f('sys/module/cfg80211/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00}-\xdb\xa6\xb1\xf6I\x90\xaf\xbdX\xed\x8d\xac\x86o(\xc8zC')
d('sys/module/snd_seq', 0o775)
f('sys/module/snd_seq/initstate', 0o664, b'live\n')
f('sys/module/snd_seq/srcversion', 0o664, b'94E19F6DBC8F63B241414FE\n')
f('sys/module/snd_seq/refcnt', 0o664, b'0\n')
d('sys/module/snd_seq/parameters', 0o775)
f('sys/module/snd_seq/parameters/seq_default_timer_device', 0o664, b'0\n')
f('sys/module/snd_seq/parameters/seq_default_timer_class', 0o664, b'1\n')
f('sys/module/snd_seq/parameters/seq_client_load', 0o664, b'-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1\n')
f('sys/module/snd_seq/parameters/seq_default_timer_sclass', 0o664, b'0\n')
f('sys/module/snd_seq/parameters/seq_default_timer_resolution', 0o664, b'0\n')
f('sys/module/snd_seq/parameters/seq_default_timer_subdevice', 0o664, b'0\n')
f('sys/module/snd_seq/parameters/seq_default_timer_card', 0o664, b'-1\n')
d('sys/module/snd_seq/sections', 0o775)
f('sys/module/snd_seq/sections/.smp_locks', 0o664, b'0xffffffffa02b1d30\n')
f('sys/module/snd_seq/sections/__ksymtab_strings', 0o664, b'0xffffffffa02b1748\n')
f('sys/module/snd_seq/sections/.exit.text', 0o664, b'0xffffffffa02b0ff0\n')
f('sys/module/snd_seq/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02b10bc\n')
f('sys/module/snd_seq/sections/.strtab', 0o664, b'0xffffffffa02b4590\n')
f('sys/module/snd_seq/sections/__ksymtab', 0o664, b'0xffffffffa02b1610\n')
f('sys/module/snd_seq/sections/.rodata', 0o664, b'0xffffffffa02b10e0\n')
f('sys/module/snd_seq/sections/__param', 0o664, b'0xffffffffa02b1320\n')
f('sys/module/snd_seq/sections/.bss', 0o664, b'0xffffffffa02b6f80\n')
f('sys/module/snd_seq/sections/.text', 0o664, b'0xffffffffa02a9000\n')
f('sys/module/snd_seq/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd_seq/sections/__kcrctab', 0o664, b'0xffffffffa02b16e0\n')
f('sys/module/snd_seq/sections/.data', 0o664, b'0xffffffffa02b6860\n')
f('sys/module/snd_seq/sections/.symtab', 0o664, b'0xffffffffa02b1fc8\n')
f('sys/module/snd_seq/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02b6bc0\n')
f('sys/module/snd_seq/sections/.rodata.str1.1', 0o664, b'0xffffffffa02b18b6\n')
d('sys/module/snd_seq/notes', 0o775)
f('sys/module/snd_seq/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00O@T\xdaF\x1cl\x11\xfev\xdd\xaa \xea\x87\x92P\xfe\xb1\xf8')
d('sys/module/usb_storage', 0o775)
f('sys/module/usb_storage/initstate', 0o664, b'live\n')
f('sys/module/usb_storage/srcversion', 0o664, b'D07F7C47643AB7AFB6BEF6F\n')
f('sys/module/usb_storage/refcnt', 0o664, b'1\n')
d('sys/module/usb_storage/drivers', 0o775)
l('sys/module/usb_storage/drivers/usb:usb-storage', '../../../bus/usb/drivers/usb-storage')
d('sys/module/usb_storage/parameters', 0o775)
f('sys/module/usb_storage/parameters/swi_tru_install', 0o664, b'1\n')
f('sys/module/usb_storage/parameters/delay_use', 0o664, b'5\n')
d('sys/module/usb_storage/sections', 0o775)
f('sys/module/usb_storage/sections/.smp_locks', 0o664, b'0xffffffffa02e1820\n')
f('sys/module/usb_storage/sections/.exit.text', 0o664, b'0xffffffffa02e056c\n')
f('sys/module/usb_storage/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02e0580\n')
f('sys/module/usb_storage/sections/.strtab', 0o664, b'0xffffffffa02e42d0\n')
f('sys/module/usb_storage/sections/.rodata', 0o664, b'0xffffffffa02e05b0\n')
f('sys/module/usb_storage/sections/__param', 0o664, b'0xffffffffa02e18b0\n')
f('sys/module/usb_storage/sections/.bss', 0o664, b'0xffffffffa02e9400\n')
f('sys/module/usb_storage/sections/.text', 0o664, b'0xffffffffa02dd000\n')
f('sys/module/usb_storage/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/usb_storage/sections/.data', 0o664, b'0xffffffffa02e5dc0\n')
f('sys/module/usb_storage/sections/.symtab', 0o664, b'0xffffffffa02e1900\n')
f('sys/module/usb_storage/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02e9040\n')
f('sys/module/usb_storage/sections/.rodata.str1.1', 0o664, b'0xffffffffa02e0ab1\n')
d('sys/module/usb_storage/notes', 0o775)
f('sys/module/usb_storage/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00B~(&&u\xc4Ax\xd8|\x80n\xc4G(i\xf3kG')
d('sys/module/acpi_cpufreq', 0o775)
f('sys/module/acpi_cpufreq/initstate', 0o664, b'live\n')
f('sys/module/acpi_cpufreq/srcversion', 0o664, b'09166B6001DCE2189668F16\n')
f('sys/module/acpi_cpufreq/refcnt', 0o664, b'1\n')
d('sys/module/acpi_cpufreq/parameters', 0o775)
f('sys/module/acpi_cpufreq/parameters/acpi_pstate_strict', 0o664, b'0\n')
d('sys/module/acpi_cpufreq/sections', 0o775)
f('sys/module/acpi_cpufreq/sections/.smp_locks', 0o664, b'0xffffffffa0254d68\n')
f('sys/module/acpi_cpufreq/sections/.exit.text', 0o664, b'0xffffffffa0254c4c\n')
f('sys/module/acpi_cpufreq/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0254c6c\n')
f('sys/module/acpi_cpufreq/sections/.strtab', 0o664, b'0xffffffffa0255620\n')
f('sys/module/acpi_cpufreq/sections/.rodata', 0o664, b'0xffffffffa0254ca0\n')
f('sys/module/acpi_cpufreq/sections/__param', 0o664, b'0xffffffffa0254d70\n')
f('sys/module/acpi_cpufreq/sections/.bss', 0o664, b'0xffffffffa0256000\n')
f('sys/module/acpi_cpufreq/sections/.text', 0o664, b'0xffffffffa0254000\n')
f('sys/module/acpi_cpufreq/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/acpi_cpufreq/sections/.data', 0o664, b'0xffffffffa0255bc0\n')
f('sys/module/acpi_cpufreq/sections/.symtab', 0o664, b'0xffffffffa0254dc8\n')
f('sys/module/acpi_cpufreq/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0255c40\n')
f('sys/module/acpi_cpufreq/sections/.rodata.str1.1', 0o664, b'0xffffffffa0254d98\n')
d('sys/module/acpi_cpufreq/notes', 0o775)
f('sys/module/acpi_cpufreq/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xa4\xfd\x13\xe7*\xbd\x99\x95\x99C\xf6\x95\x04\xf1r\xf9\xf7}aL')
d('sys/module/button', 0o775)
f('sys/module/button/initstate', 0o664, b'live\n')
f('sys/module/button/srcversion', 0o664, b'2713981C971D8EC4C059338\n')
f('sys/module/button/refcnt', 0o664, b'0\n')
d('sys/module/button/sections', 0o775)
f('sys/module/button/sections/.smp_locks', 0o664, b'0xffffffffa00a7c98\n')
f('sys/module/button/sections/.exit.text', 0o664, b'0xffffffffa00a781c\n')
f('sys/module/button/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00a7898\n')
f('sys/module/button/sections/.strtab', 0o664, b'0xffffffffa00a84d8\n')
f('sys/module/button/sections/.rodata', 0o664, b'0xffffffffa00a78c0\n')
f('sys/module/button/sections/.bss', 0o664, b'0xffffffffa00a8f40\n')
f('sys/module/button/sections/.text', 0o664, b'0xffffffffa00a7000\n')
f('sys/module/button/sections/.init.text', 0o664, b'0xffffffffa00aa000\n')
f('sys/module/button/sections/.data', 0o664, b'0xffffffffa00a89e0\n')
f('sys/module/button/sections/.symtab', 0o664, b'0xffffffffa00a7cb0\n')
f('sys/module/button/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00a8b80\n')
f('sys/module/button/sections/.rodata.str1.1', 0o664, b'0xffffffffa00a7b60\n')
d('sys/module/button/notes', 0o775)
f('sys/module/button/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00;\xc1\xa4qIi\xa7\xa0g\x7fb\xcb\xb4\xba\x80\xb1\x17Z#\xb5')
d('sys/module/processor', 0o775)
f('sys/module/processor/initstate', 0o664, b'live\n')
f('sys/module/processor/srcversion', 0o664, b'8A1C302AFF80867DBAC4684\n')
f('sys/module/processor/refcnt', 0o664, b'4\n')
d('sys/module/processor/holders', 0o775)
l('sys/module/processor/holders/acpi_cpufreq', '../../acpi_cpufreq')
l('sys/module/processor/holders/thermal', '../../thermal')
d('sys/module/processor/parameters', 0o775)
f('sys/module/processor/parameters/ignore_ppc', 0o664, b'0\n')
f('sys/module/processor/parameters/latency_factor', 0o664, b'2\n')
d('sys/module/processor/sections', 0o775)
f('sys/module/processor/sections/.smp_locks', 0o664, b'0xffffffffa0077788\n')
f('sys/module/processor/sections/__ex_table', 0o664, b'0xffffffffa0077768\n')
f('sys/module/processor/sections/__bug_table', 0o664, b'0xffffffffa0077621\n')
f('sys/module/processor/sections/__ksymtab_strings', 0o664, b'0xffffffffa00776b8\n')
f('sys/module/processor/sections/.exit.text', 0o664, b'0xffffffffa00763a9\n')
f('sys/module/processor/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00765c4\n')
f('sys/module/processor/sections/.strtab', 0o664, b'0xffffffffa0079438\n')
f('sys/module/processor/sections/.fixup', 0o664, b'0xffffffffa00765ad\n')
f('sys/module/processor/sections/__ksymtab', 0o664, b'0xffffffffa0077640\n')
f('sys/module/processor/sections/.rodata', 0o664, b'0xffffffffa00765f0\n')
f('sys/module/processor/sections/.data.read_mostly', 0o664, b'0xffffffffa007b7d8\n')
f('sys/module/processor/sections/__param', 0o664, b'0xffffffffa00777f0\n')
f('sys/module/processor/sections/.bss', 0o664, b'0xffffffffa007bbc0\n')
f('sys/module/processor/sections/.text', 0o664, b'0xffffffffa0072000\n')
f('sys/module/processor/sections/.init.text', 0o664, b'0xffffffffa007d000\n')
f('sys/module/processor/sections/__kcrctab', 0o664, b'0xffffffffa0077690\n')
f('sys/module/processor/sections/.data', 0o664, b'0xffffffffa007ab90\n')
f('sys/module/processor/sections/.cpuinit.data', 0o664, b'0xffffffffa007b040\n')
f('sys/module/processor/sections/.symtab', 0o664, b'0xffffffffa0077890\n')
f('sys/module/processor/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa007b800\n')
f('sys/module/processor/sections/.rodata.str1.1', 0o664, b'0xffffffffa0076863\n')
f('sys/module/processor/sections/.cpuinit.text', 0o664, b'0xffffffffa0075abc\n')
f('sys/module/processor/sections/.ref.text', 0o664, b'0xffffffffa007641a\n')
d('sys/module/processor/notes', 0o775)
f('sys/module/processor/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xa1\x01$C\tS,+Ue\\\x8cg\xb0Z\xe9\xa1\xaf[\xe5')
d('sys/module/oprofile', 0o775)
d('sys/module/oprofile/parameters', 0o775)
f('sys/module/oprofile/parameters/timer', 0o664, b'0\n')
d('sys/module/spurious', 0o775)
d('sys/module/spurious/parameters', 0o775)
f('sys/module/spurious/parameters/noirqdebug', 0o664, b'N\n')
f('sys/module/spurious/parameters/irqfixup', 0o664, b'0\n')
d('sys/module/raid1', 0o775)
f('sys/module/raid1/initstate', 0o664, b'live\n')
f('sys/module/raid1/srcversion', 0o664, b'EA08FDD37C2EEFA95DE4413\n')
f('sys/module/raid1/refcnt', 0o664, b'1\n')
d('sys/module/raid1/sections', 0o775)
f('sys/module/raid1/sections/.smp_locks', 0o664, b'0xffffffffa0236438\n')
f('sys/module/raid1/sections/__bug_table', 0o664, b'0xffffffffa02369c0\n')
f('sys/module/raid1/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0236414\n')
f('sys/module/raid1/sections/.strtab', 0o664, b'0xffffffffa02376f8\n')
f('sys/module/raid1/sections/.bss', 0o664, b'0xffffffffa0238280\n')
f('sys/module/raid1/sections/.text', 0o664, b'0xffffffffa0232000\n')
f('sys/module/raid1/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/raid1/sections/.data', 0o664, b'0xffffffffa0237e20\n')
f('sys/module/raid1/sections/.symtab', 0o664, b'0xffffffffa0236a08\n')
f('sys/module/raid1/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0237ec0\n')
f('sys/module/raid1/sections/.rodata.str1.1', 0o664, b'0xffffffffa0236618\n')
d('sys/module/raid1/notes', 0o775)
f('sys/module/raid1/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xcd\xdf0\xe8v}\x0cuP}\x93\xe1\x19\xcd\xed~\xf7\x14\xdd@')
d('sys/module/edd', 0o775)
f('sys/module/edd/version', 0o664, b'0.16\n')
f('sys/module/edd/initstate', 0o664, b'live\n')
f('sys/module/edd/srcversion', 0o664, b'ED68C1ACC4B2D1B19C7BEA7\n')
f('sys/module/edd/refcnt', 0o664, b'0\n')
d('sys/module/edd/sections', 0o775)
f('sys/module/edd/sections/.exit.text', 0o664, b'0xffffffffa02a1de8\n')
f('sys/module/edd/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02a1e6c\n')
f('sys/module/edd/sections/.strtab', 0o664, b'0xffffffffa02a2b88\n')
f('sys/module/edd/sections/.rodata', 0o664, b'0xffffffffa02a1ea0\n')
f('sys/module/edd/sections/.bss', 0o664, b'0xffffffffa02a3880\n')
f('sys/module/edd/sections/.text', 0o664, b'0xffffffffa02a1000\n')
f('sys/module/edd/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/edd/sections/.data', 0o664, b'0xffffffffa02a3120\n')
f('sys/module/edd/sections/.symtab', 0o664, b'0xffffffffa02a22b8\n')
f('sys/module/edd/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02a34c0\n')
f('sys/module/edd/sections/.rodata.str1.1', 0o664, b'0xffffffffa02a1f18\n')
d('sys/module/edd/notes', 0o775)
f('sys/module/edd/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xbe\x92\xcfW\xc8l\x08\xaf\xb6\x1a\xb7!E\xb8\xd1\x9eU\xd2\xddM')
d('sys/module/hwmon', 0o775)
f('sys/module/hwmon/initstate', 0o664, b'live\n')
f('sys/module/hwmon/srcversion', 0o664, b'9344A0FAA4A298DC9AEBDC8\n')
f('sys/module/hwmon/refcnt', 0o664, b'1\n')
d('sys/module/hwmon/holders', 0o775)
l('sys/module/hwmon/holders/thinkpad_acpi', '../../thinkpad_acpi')
d('sys/module/hwmon/sections', 0o775)
f('sys/module/hwmon/sections/__ksymtab_gpl', 0o664, b'0xffffffffa007d1d0\n')
f('sys/module/hwmon/sections/__ksymtab_strings', 0o664, b'0xffffffffa007d200\n')
f('sys/module/hwmon/sections/.exit.text', 0o664, b'0xffffffffa007d138\n')
f('sys/module/hwmon/sections/.note.gnu.build-id', 0o664, b'0xffffffffa007d14c\n')
f('sys/module/hwmon/sections/.strtab', 0o664, b'0xffffffffa007d770\n')
f('sys/module/hwmon/sections/.bss', 0o664, b'0xffffffffa007de80\n')
f('sys/module/hwmon/sections/.text', 0o664, b'0xffffffffa007d000\n')
f('sys/module/hwmon/sections/.init.text', 0o664, b'0xffffffffa009a000\n')
f('sys/module/hwmon/sections/.data', 0o664, b'0xffffffffa007da20\n')
f('sys/module/hwmon/sections/.symtab', 0o664, b'0xffffffffa007d230\n')
f('sys/module/hwmon/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa007dac0\n')
f('sys/module/hwmon/sections/.rodata.str1.1', 0o664, b'0xffffffffa007d170\n')
f('sys/module/hwmon/sections/__kcrctab_gpl', 0o664, b'0xffffffffa007d1f0\n')
d('sys/module/hwmon/notes', 0o775)
f('sys/module/hwmon/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x0fR\x0b\x94!\x19\xde\x14\x93\xac\x9e@\xa2&\xcam,\xba\x12\xd1')
d('sys/module/backlight', 0o775)
f('sys/module/backlight/initstate', 0o664, b'live\n')
f('sys/module/backlight/srcversion', 0o664, b'AFD1565C65F3DA6D942CB52\n')
f('sys/module/backlight/refcnt', 0o664, b'1\n')
d('sys/module/backlight/holders', 0o775)
l('sys/module/backlight/holders/thinkpad_acpi', '../../thinkpad_acpi')
d('sys/module/backlight/sections', 0o775)
f('sys/module/backlight/sections/__ksymtab_strings', 0o664, b'0xffffffffa0131700\n')
f('sys/module/backlight/sections/.exit.text', 0o664, b'0xffffffffa01315ec\n')
f('sys/module/backlight/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0131600\n')
f('sys/module/backlight/sections/.strtab', 0o664, b'0xffffffffa0131de0\n')
f('sys/module/backlight/sections/__ksymtab', 0o664, b'0xffffffffa01316d0\n')
f('sys/module/backlight/sections/.bss', 0o664, b'0xffffffffa01326c0\n')
f('sys/module/backlight/sections/.text', 0o664, b'0xffffffffa0131000\n')
f('sys/module/backlight/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/backlight/sections/__kcrctab', 0o664, b'0xffffffffa01316f0\n')
f('sys/module/backlight/sections/.data', 0o664, b'0xffffffffa0132220\n')
f('sys/module/backlight/sections/.symtab', 0o664, b'0xffffffffa0131738\n')
f('sys/module/backlight/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0132300\n')
f('sys/module/backlight/sections/.rodata.str1.1', 0o664, b'0xffffffffa0131624\n')
d('sys/module/backlight/notes', 0o775)
f('sys/module/backlight/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00d\xb6_\xb64[\x83\x90Fm(\x91J\xb1\xce\xf5\xaa\xd2\xf4}')
d('sys/module/snd', 0o775)
f('sys/module/snd/initstate', 0o664, b'live\n')
f('sys/module/snd/srcversion', 0o664, b'D48337942305C9A41A7CC12\n')
f('sys/module/snd/refcnt', 0o664, b'13\n')
d('sys/module/snd/holders', 0o775)
l('sys/module/snd/holders/snd_seq', '../../snd_seq')
l('sys/module/snd/holders/snd_timer', '../../snd_timer')
l('sys/module/snd/holders/snd_pcm', '../../snd_pcm')
l('sys/module/snd/holders/snd_hda_intel', '../../snd_hda_intel')
l('sys/module/snd/holders/snd_mixer_oss', '../../snd_mixer_oss')
l('sys/module/snd/holders/snd_pcm_oss', '../../snd_pcm_oss')
l('sys/module/snd/holders/snd_seq_device', '../../snd_seq_device')
d('sys/module/snd/parameters', 0o775)
f('sys/module/snd/parameters/slots', 0o664, b'snd-hda-intel,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>\n')
f('sys/module/snd/parameters/major', 0o664, b'116\n')
f('sys/module/snd/parameters/cards_limit', 0o664, b'1\n')
d('sys/module/snd/sections', 0o775)
f('sys/module/snd/sections/__ksymtab_strings', 0o664, b'0xffffffffa00d32e8\n')
f('sys/module/snd/sections/.exit.text', 0o664, b'0xffffffffa00d1bc4\n')
f('sys/module/snd/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00d1c8c\n')
f('sys/module/snd/sections/.strtab', 0o664, b'0xffffffffa00d6cc8\n')
f('sys/module/snd/sections/__ksymtab', 0o664, b'0xffffffffa00d2d00\n')
f('sys/module/snd/sections/.rodata', 0o664, b'0xffffffffa00d1cc0\n')
f('sys/module/snd/sections/__param', 0o664, b'0xffffffffa00d3270\n')
f('sys/module/snd/sections/.bss', 0o664, b'0xffffffffa00da480\n')
f('sys/module/snd/sections/.text', 0o664, b'0xffffffffa00ca000\n')
f('sys/module/snd/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd/sections/__kcrctab', 0o664, b'0xffffffffa00d30a0\n')
f('sys/module/snd/sections/.data', 0o664, b'0xffffffffa00d9be0\n')
f('sys/module/snd/sections/.symtab', 0o664, b'0xffffffffa00d3770\n')
f('sys/module/snd/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00da0c0\n')
f('sys/module/snd/sections/.rodata.str1.1', 0o664, b'0xffffffffa00d2558\n')
d('sys/module/snd/notes', 0o775)
f('sys/module/snd/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xee\x11+\xc0\xd4\x8f\x10\x02v\xf8\t\xb4\xa8\xf6[z\xd3\xb3\xb7\x87')
d('sys/module/mac80211', 0o775)
f('sys/module/mac80211/initstate', 0o664, b'live\n')
f('sys/module/mac80211/srcversion', 0o664, b'330E8FB2F2933F478C444A4\n')
f('sys/module/mac80211/refcnt', 0o664, b'1\n')
d('sys/module/mac80211/holders', 0o775)
l('sys/module/mac80211/holders/iwl3945', '../../iwl3945')
d('sys/module/mac80211/parameters', 0o775)
f('sys/module/mac80211/parameters/ieee80211_default_rc_algo', 0o664, b'pid\n')
d('sys/module/mac80211/sections', 0o775)
f('sys/module/mac80211/sections/__ksymtab_gpl', 0o664, b'0xffffffffa0127a50\n')
f('sys/module/mac80211/sections/.smp_locks', 0o664, b'0xffffffffa01270a0\n')
f('sys/module/mac80211/sections/__bug_table', 0o664, b'0xffffffffa0124183\n')
f('sys/module/mac80211/sections/__ksymtab_strings', 0o664, b'0xffffffffa0127630\n')
f('sys/module/mac80211/sections/.exit.text', 0o664, b'0xffffffffa0122de4\n')
f('sys/module/mac80211/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0122df4\n')
f('sys/module/mac80211/sections/.strtab', 0o664, b'0xffffffffa012b890\n')
f('sys/module/mac80211/sections/__ksymtab', 0o664, b'0xffffffffa01272a0\n')
f('sys/module/mac80211/sections/.rodata', 0o664, b'0xffffffffa0122e20\n')
f('sys/module/mac80211/sections/__param', 0o664, b'0xffffffffa0127a28\n')
f('sys/module/mac80211/sections/.bss', 0o664, b'0xffffffffa012fa40\n')
f('sys/module/mac80211/sections/.text', 0o664, b'0xffffffffa0109000\n')
f('sys/module/mac80211/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/mac80211/sections/__kcrctab', 0o664, b'0xffffffffa0127500\n')
f('sys/module/mac80211/sections/.data', 0o664, b'0xffffffffa012f340\n')
f('sys/module/mac80211/sections/.symtab', 0o664, b'0xffffffffa0127a80\n')
f('sys/module/mac80211/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa012f680\n')
f('sys/module/mac80211/sections/.rodata.str1.1', 0o664, b'0xffffffffa0123c40\n')
f('sys/module/mac80211/sections/__kcrctab_gpl', 0o664, b'0xffffffffa0127a70\n')
d('sys/module/mac80211/notes', 0o775)
f('sys/module/mac80211/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00"i\xe2\xa0\x92\xbc7\xc4#\xbc\xda\xf11\x02\x13\xd5\x155\x1cS')
d('sys/module/printk', 0o775)
d('sys/module/printk/parameters', 0o775)
f('sys/module/printk/parameters/time', 0o664, b'N\n')
d('sys/module/rtc_lib', 0o775)
f('sys/module/rtc_lib/initstate', 0o664, b'live\n')
f('sys/module/rtc_lib/srcversion', 0o664, b'7884E7FF8728D88C6BD1427\n')
f('sys/module/rtc_lib/refcnt', 0o664, b'1\n')
d('sys/module/rtc_lib/holders', 0o775)
l('sys/module/rtc_lib/holders/rtc_core', '../../rtc_core')
d('sys/module/rtc_lib/sections', 0o775)
f('sys/module/rtc_lib/sections/__ksymtab_strings', 0o664, b'0xffffffffa009c3f8\n')
f('sys/module/rtc_lib/sections/.note.gnu.build-id', 0o664, b'0xffffffffa009c31c\n')
f('sys/module/rtc_lib/sections/.strtab', 0o664, b'0xffffffffa009c8f0\n')
f('sys/module/rtc_lib/sections/__ksymtab', 0o664, b'0xffffffffa009c380\n')
f('sys/module/rtc_lib/sections/.rodata', 0o664, b'0xffffffffa009c340\n')
f('sys/module/rtc_lib/sections/.bss', 0o664, b'0xffffffffa009cf80\n')
f('sys/module/rtc_lib/sections/.text', 0o664, b'0xffffffffa009c000\n')
f('sys/module/rtc_lib/sections/__kcrctab', 0o664, b'0xffffffffa009c3d0\n')
f('sys/module/rtc_lib/sections/.data', 0o664, b'0xffffffffa009cba8\n')
f('sys/module/rtc_lib/sections/.symtab', 0o664, b'0xffffffffa009c440\n')
f('sys/module/rtc_lib/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa009cbc0\n')
d('sys/module/rtc_lib/notes', 0o775)
f('sys/module/rtc_lib/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xb4\xe2\xcfLw\x17j\xb2\xc7\xf1 \xfa\xc7G\x99+b^+S')
d('sys/module/mousedev', 0o775)
d('sys/module/mousedev/parameters', 0o775)
f('sys/module/mousedev/parameters/tap_time', 0o664, b'200\n')
f('sys/module/mousedev/parameters/yres', 0o664, b'768\n')
f('sys/module/mousedev/parameters/xres', 0o664, b'1024\n')
d('sys/module/rtc_cmos', 0o775)
f('sys/module/rtc_cmos/initstate', 0o664, b'live\n')
f('sys/module/rtc_cmos/srcversion', 0o664, b'7BE4B8C2A52556D0C62CB0B\n')
f('sys/module/rtc_cmos/refcnt', 0o664, b'0\n')
d('sys/module/rtc_cmos/sections', 0o775)
f('sys/module/rtc_cmos/sections/.exit.text', 0o664, b'0xffffffffa00fe370\n')
f('sys/module/rtc_cmos/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00fe4e4\n')
f('sys/module/rtc_cmos/sections/.strtab', 0o664, b'0xffffffffa00ff1a8\n')
f('sys/module/rtc_cmos/sections/.rodata', 0o664, b'0xffffffffa00fe520\n')
f('sys/module/rtc_cmos/sections/.bss', 0o664, b'0xffffffffa00ffd40\n')
f('sys/module/rtc_cmos/sections/.text', 0o664, b'0xffffffffa00fd000\n')
f('sys/module/rtc_cmos/sections/.init.text', 0o664, b'0xffffffffa007d000\n')
f('sys/module/rtc_cmos/sections/.data', 0o664, b'0xffffffffa00ff800\n')
f('sys/module/rtc_cmos/sections/.devinit.text', 0o664, b'0xffffffffa00fe44f\n')
f('sys/module/rtc_cmos/sections/.symtab', 0o664, b'0xffffffffa00fe6e0\n')
f('sys/module/rtc_cmos/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00ff980\n')
f('sys/module/rtc_cmos/sections/.rodata.str1.1', 0o664, b'0xffffffffa00fe6a2\n')
d('sys/module/rtc_cmos/notes', 0o775)
f('sys/module/rtc_cmos/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xf4\x82\xed4\xe7\x10\xb9\x7f\xa9 \x0c\xedIf0\x86u\xd41\x87')
d('sys/module/battery', 0o775)
f('sys/module/battery/initstate', 0o664, b'live\n')
f('sys/module/battery/srcversion', 0o664, b'5A2CE8D302B31A112CFB57A\n')
f('sys/module/battery/refcnt', 0o664, b'0\n')
d('sys/module/battery/parameters', 0o775)
f('sys/module/battery/parameters/cache_time', 0o664, b'1000\n')
d('sys/module/battery/sections', 0o775)
f('sys/module/battery/sections/.exit.text', 0o664, b'0xffffffffa008008c\n')
f('sys/module/battery/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00800ac\n')
f('sys/module/battery/sections/.strtab', 0o664, b'0xffffffffa0081318\n')
f('sys/module/battery/sections/.rodata', 0o664, b'0xffffffffa00800d0\n')
f('sys/module/battery/sections/__param', 0o664, b'0xffffffffa0080798\n')
f('sys/module/battery/sections/.bss', 0o664, b'0xffffffffa0082440\n')
f('sys/module/battery/sections/.text', 0o664, b'0xffffffffa007f000\n')
f('sys/module/battery/sections/.init.text', 0o664, b'0xffffffffa0084000\n')
f('sys/module/battery/sections/.data', 0o664, b'0xffffffffa0081a90\n')
f('sys/module/battery/sections/.symtab', 0o664, b'0xffffffffa00807c0\n')
f('sys/module/battery/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0082080\n')
f('sys/module/battery/sections/.rodata.str1.1', 0o664, b'0xffffffffa008025b\n')
d('sys/module/battery/notes', 0o775)
f('sys/module/battery/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xe3\xa8\xcc.j\xb3\xda\xc8\x88\x03\xca\xf6\xf0*\x9c-\xffE!\xd3')
d('sys/module/yenta_socket', 0o775)
f('sys/module/yenta_socket/initstate', 0o664, b'live\n')
f('sys/module/yenta_socket/srcversion', 0o664, b'B0538CB6E6ED071E80A4575\n')
f('sys/module/yenta_socket/refcnt', 0o664, b'1\n')
d('sys/module/yenta_socket/drivers', 0o775)
l('sys/module/yenta_socket/drivers/pci:yenta_cardbus', '../../../bus/pci/drivers/yenta_cardbus')
d('sys/module/yenta_socket/parameters', 0o775)
f('sys/module/yenta_socket/parameters/isa_probe', 0o664, b'Y\n')
f('sys/module/yenta_socket/parameters/disable_clkrun', 0o664, b'N\n')
f('sys/module/yenta_socket/parameters/pwr_irqs_off', 0o664, b'N\n')
d('sys/module/yenta_socket/sections', 0o775)
f('sys/module/yenta_socket/sections/__bug_table', 0o664, b'0xffffffffa0104aa6\n')
f('sys/module/yenta_socket/sections/.exit.text', 0o664, b'0xffffffffa0103c44\n')
f('sys/module/yenta_socket/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0104330\n')
f('sys/module/yenta_socket/sections/.strtab', 0o664, b'0xffffffffa0105e80\n')
f('sys/module/yenta_socket/sections/.rodata', 0o664, b'0xffffffffa0104354\n')
f('sys/module/yenta_socket/sections/__param', 0o664, b'0xffffffffa0104ad8\n')
f('sys/module/yenta_socket/sections/.bss', 0o664, b'0xffffffffa0107800\n')
f('sys/module/yenta_socket/sections/.text', 0o664, b'0xffffffffa0101000\n')
f('sys/module/yenta_socket/sections/.init.text', 0o664, b'0xffffffffa0084000\n')
f('sys/module/yenta_socket/sections/.data', 0o664, b'0xffffffffa0106ac0\n')
f('sys/module/yenta_socket/sections/.devinit.text', 0o664, b'0xffffffffa0103c60\n')
f('sys/module/yenta_socket/sections/.symtab', 0o664, b'0xffffffffa0104b78\n')
f('sys/module/yenta_socket/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0107440\n')
f('sys/module/yenta_socket/sections/.rodata.str1.1', 0o664, b'0xffffffffa0104388\n')
d('sys/module/yenta_socket/notes', 0o775)
f('sys/module/yenta_socket/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00H\xa8\xe9\xf1\x06e\xb4\xe4\xd7\xa1\x7f\xf8G)\x1a\xa3\xec\xe7m<')
d('sys/module/acpi', 0o775)
d('sys/module/acpi/parameters', 0o775)
f('sys/module/acpi/parameters/acpica_version', 0o664, b'20080609\n')
d('sys/module/cdc_acm', 0o775)
f('sys/module/cdc_acm/initstate', 0o664, b'live\n')
f('sys/module/cdc_acm/srcversion', 0o664, b'5DED3866AA046376E4E36DD\n')
f('sys/module/cdc_acm/refcnt', 0o664, b'0\n')
d('sys/module/cdc_acm/drivers', 0o775)
l('sys/module/cdc_acm/drivers/usb:cdc_acm', '../../../bus/usb/drivers/cdc_acm')
d('sys/module/cdc_acm/sections', 0o775)
f('sys/module/cdc_acm/sections/.smp_locks', 0o664, b'0xffffffffa02d9660\n')
f('sys/module/cdc_acm/sections/.exit.text', 0o664, b'0xffffffffa02d949c\n')
f('sys/module/cdc_acm/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02d94c8\n')
f('sys/module/cdc_acm/sections/.strtab', 0o664, b'0xffffffffa02da838\n')
f('sys/module/cdc_acm/sections/.rodata', 0o664, b'0xffffffffa02d9500\n')
f('sys/module/cdc_acm/sections/.bss', 0o664, b'0xffffffffa02db8c0\n')
f('sys/module/cdc_acm/sections/.text', 0o664, b'0xffffffffa02d7000\n')
f('sys/module/cdc_acm/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/cdc_acm/sections/.data', 0o664, b'0xffffffffa02db0e0\n')
f('sys/module/cdc_acm/sections/.symtab', 0o664, b'0xffffffffa02d9968\n')
f('sys/module/cdc_acm/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02db500\n')
f('sys/module/cdc_acm/sections/.rodata.str1.1', 0o664, b'0xffffffffa02d9698\n')
d('sys/module/cdc_acm/notes', 0o775)
f('sys/module/cdc_acm/notes/.note.gnu.build-id', 0o664, b"\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xe1=\x89\xe9\xe8L\t\xfe|o\x9c+\xfe\x81\xe8\xc6P'\xa4$")
d('sys/module/snd_timer', 0o775)
f('sys/module/snd_timer/initstate', 0o664, b'live\n')
f('sys/module/snd_timer/srcversion', 0o664, b'1F1836030C5B604CF478CFC\n')
f('sys/module/snd_timer/refcnt', 0o664, b'2\n')
d('sys/module/snd_timer/holders', 0o775)
l('sys/module/snd_timer/holders/snd_seq', '../../snd_seq')
l('sys/module/snd_timer/holders/snd_pcm', '../../snd_pcm')
d('sys/module/snd_timer/parameters', 0o775)
f('sys/module/snd_timer/parameters/timer_tstamp_monotonic', 0o664, b'1\n')
f('sys/module/snd_timer/parameters/timer_limit', 0o664, b'1\n')
d('sys/module/snd_timer/sections', 0o775)
f('sys/module/snd_timer/sections/.smp_locks', 0o664, b'0xffffffffa00f8908\n')
f('sys/module/snd_timer/sections/__bug_table', 0o664, b'0xffffffffa00f888f\n')
f('sys/module/snd_timer/sections/__ksymtab_strings', 0o664, b'0xffffffffa00f8a98\n')
f('sys/module/snd_timer/sections/.exit.text', 0o664, b'0xffffffffa00f8520\n')
f('sys/module/snd_timer/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00f8590\n')
f('sys/module/snd_timer/sections/.strtab', 0o664, b'0xffffffffa00f9e90\n')
f('sys/module/snd_timer/sections/__ksymtab', 0o664, b'0xffffffffa00f8910\n')
f('sys/module/snd_timer/sections/.rodata', 0o664, b'0xffffffffa00f85c0\n')
f('sys/module/snd_timer/sections/__param', 0o664, b'0xffffffffa00f8a48\n')
f('sys/module/snd_timer/sections/.bss', 0o664, b'0xffffffffa00fb200\n')
f('sys/module/snd_timer/sections/.text', 0o664, b'0xffffffffa00f5000\n')
f('sys/module/snd_timer/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd_timer/sections/__kcrctab', 0o664, b'0xffffffffa00f89e0\n')
f('sys/module/snd_timer/sections/.data', 0o664, b'0xffffffffa00fad00\n')
f('sys/module/snd_timer/sections/.symtab', 0o664, b'0xffffffffa00f8b88\n')
f('sys/module/snd_timer/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00fae40\n')
f('sys/module/snd_timer/sections/.rodata.str1.1', 0o664, b'0xffffffffa00f8723\n')
d('sys/module/snd_timer/notes', 0o775)
f('sys/module/snd_timer/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xb2uP\xf49\x98_\xdcY\xc1\xbfY\xa4yn\xb0\xbe\x9c\xf2F')
d('sys/module/e1000e', 0o775)
f('sys/module/e1000e/version', 0o664, b'0.3.3.3-k2\n')
f('sys/module/e1000e/initstate', 0o664, b'live\n')
f('sys/module/e1000e/srcversion', 0o664, b'5B93BF2B618524ED19305C8\n')
f('sys/module/e1000e/refcnt', 0o664, b'0\n')
d('sys/module/e1000e/drivers', 0o775)
l('sys/module/e1000e/drivers/pci:e1000e', '../../../bus/pci/drivers/e1000e')
d('sys/module/e1000e/parameters', 0o775)
f('sys/module/e1000e/parameters/copybreak', 0o664, b'256\n')
d('sys/module/e1000e/sections', 0o775)
f('sys/module/e1000e/sections/.devinit.data', 0o664, b'0xffffffffa0024fe0\n')
f('sys/module/e1000e/sections/.smp_locks', 0o664, b'0xffffffffa001e730\n')
f('sys/module/e1000e/sections/__bug_table', 0o664, b'0xffffffffa001e590\n')
f('sys/module/e1000e/sections/.exit.text', 0o664, b'0xffffffffa001c64f\n')
f('sys/module/e1000e/sections/.note.gnu.build-id', 0o664, b'0xffffffffa001c750\n')
f('sys/module/e1000e/sections/.strtab', 0o664, b'0xffffffffa0021ac0\n')
f('sys/module/e1000e/sections/.devexit.text', 0o664, b'0xffffffffa001c672\n')
f('sys/module/e1000e/sections/.rodata', 0o664, b'0xffffffffa001c780\n')
f('sys/module/e1000e/sections/__param', 0o664, b'0xffffffffa001e5f0\n')
f('sys/module/e1000e/sections/.bss', 0o664, b'0xffffffffa0025800\n')
f('sys/module/e1000e/sections/.text', 0o664, b'0xffffffffa0009000\n')
f('sys/module/e1000e/sections/.init.text', 0o664, b'0xffffffffa0027000\n')
f('sys/module/e1000e/sections/.data', 0o664, b'0xffffffffa00243c0\n')
f('sys/module/e1000e/sections/.devinit.text', 0o664, b'0xffffffffa001b380\n')
f('sys/module/e1000e/sections/.symtab', 0o664, b'0xffffffffa001e8b0\n')
f('sys/module/e1000e/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0025440\n')
f('sys/module/e1000e/sections/.rodata.str1.1', 0o664, b'0xffffffffa001d6f0\n')
d('sys/module/e1000e/notes', 0o775)
f('sys/module/e1000e/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xcbJ\xb1\x8eG^\xf7\xba\xbf\xa6\x1b\x03\xfasqDl\xf7"\xf1')
d('sys/module/aes_x86_64', 0o775)
f('sys/module/aes_x86_64/initstate', 0o664, b'live\n')
f('sys/module/aes_x86_64/srcversion', 0o664, b'4E315E23114A08593D5D731\n')
f('sys/module/aes_x86_64/refcnt', 0o664, b'2\n')
d('sys/module/aes_x86_64/sections', 0o775)
f('sys/module/aes_x86_64/sections/.exit.text', 0o664, b'0xffffffffa02d471c\n')
f('sys/module/aes_x86_64/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02d4730\n')
f('sys/module/aes_x86_64/sections/.strtab', 0o664, b'0xffffffffa02d4b60\n')
f('sys/module/aes_x86_64/sections/.bss', 0o664, b'0xffffffffa02d5200\n')
f('sys/module/aes_x86_64/sections/.text', 0o664, b'0xffffffffa02d3000\n')
f('sys/module/aes_x86_64/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/aes_x86_64/sections/.data', 0o664, b'0xffffffffa02d4d20\n')
f('sys/module/aes_x86_64/sections/.symtab', 0o664, b'0xffffffffa02d4758\n')
f('sys/module/aes_x86_64/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02d4e40\n')
d('sys/module/aes_x86_64/notes', 0o775)
f('sys/module/aes_x86_64/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x1e3\x8c\xa9\xdd\xa1\xeb\\\xcf\xd3\x0f\xd1\xfc\xaa\xd2y\x08+f\xca')
d('sys/module/dm_mod', 0o775)
f('sys/module/dm_mod/initstate', 0o664, b'live\n')
f('sys/module/dm_mod/srcversion', 0o664, b'458AEB289C7AA031FFD0011\n')
f('sys/module/dm_mod/refcnt', 0o664, b'0\n')
d('sys/module/dm_mod/sections', 0o775)
f('sys/module/dm_mod/sections/__ksymtab_gpl', 0o664, b'0xffffffffa02122b0\n')
f('sys/module/dm_mod/sections/.smp_locks', 0o664, b'0xffffffffa0210eb8\n')
f('sys/module/dm_mod/sections/__bug_table', 0o664, b'0xffffffffa0211f54\n')
f('sys/module/dm_mod/sections/__ksymtab_strings', 0o664, b'0xffffffffa0212310\n')
f('sys/module/dm_mod/sections/.exit.text', 0o664, b'0xffffffffa0210c20\n')
f('sys/module/dm_mod/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0210c54\n')
f('sys/module/dm_mod/sections/.strtab', 0o664, b'0xffffffffa02151f0\n')
f('sys/module/dm_mod/sections/__ksymtab', 0o664, b'0xffffffffa02120d0\n')
f('sys/module/dm_mod/sections/.rodata', 0o664, b'0xffffffffa0210c80\n')
f('sys/module/dm_mod/sections/__param', 0o664, b'0xffffffffa02120a8\n')
f('sys/module/dm_mod/sections/.bss', 0o664, b'0xffffffffa02177c0\n')
f('sys/module/dm_mod/sections/.text', 0o664, b'0xffffffffa0209000\n')
f('sys/module/dm_mod/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/dm_mod/sections/__kcrctab', 0o664, b'0xffffffffa0212210\n')
f('sys/module/dm_mod/sections/.data', 0o664, b'0xffffffffa0216f40\n')
f('sys/module/dm_mod/sections/.init.data', 0o664, b'0xffffffffa00652c0\n')
f('sys/module/dm_mod/sections/.symtab', 0o664, b'0xffffffffa02124a8\n')
f('sys/module/dm_mod/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0217400\n')
f('sys/module/dm_mod/sections/.rodata.str1.1', 0o664, b'0xffffffffa0211c4d\n')
f('sys/module/dm_mod/sections/__kcrctab_gpl', 0o664, b'0xffffffffa02122f0\n')
d('sys/module/dm_mod/notes', 0o775)
f('sys/module/dm_mod/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00^(\xdd\x16OD6\xfb\x89\xe6\xca3Vl\xb0\xd7MBp\x84')
d('sys/module/usbhid', 0o775)
f('sys/module/usbhid/initstate', 0o664, b'live\n')
f('sys/module/usbhid/srcversion', 0o664, b'3D7B82FE16E5E534DE4C12B\n')
f('sys/module/usbhid/refcnt', 0o664, b'0\n')
d('sys/module/usbhid/drivers', 0o775)
l('sys/module/usbhid/drivers/usb:usbhid', '../../../bus/usb/drivers/usbhid')
d('sys/module/usbhid/parameters', 0o775)
f('sys/module/usbhid/parameters/mousepoll', 0o664, b'0\n')
f('sys/module/usbhid/parameters/rdesc_quirks', 0o664, b'<NULL>,<NULL>,<NULL>,<NULL>\n')
f('sys/module/usbhid/parameters/quirks', 0o664, b'<NULL>,<NULL>,<NULL>,<NULL>\n')
d('sys/module/usbhid/sections', 0o775)
f('sys/module/usbhid/sections/__ksymtab_gpl', 0o664, b'0xffffffffa0201e60\n')
f('sys/module/usbhid/sections/.smp_locks', 0o664, b'0xffffffffa0201700\n')
f('sys/module/usbhid/sections/__ksymtab_strings', 0o664, b'0xffffffffa0201e78\n')
f('sys/module/usbhid/sections/.exit.text', 0o664, b'0xffffffffa0200c6c\n')
f('sys/module/usbhid/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0200c84\n')
f('sys/module/usbhid/sections/.strtab', 0o664, b'0xffffffffa0202e68\n')
f('sys/module/usbhid/sections/.rodata', 0o664, b'0xffffffffa0200cc0\n')
f('sys/module/usbhid/sections/__param', 0o664, b'0xffffffffa0201de0\n')
f('sys/module/usbhid/sections/.bss', 0o664, b'0xffffffffa0203d80\n')
f('sys/module/usbhid/sections/.text', 0o664, b'0xffffffffa01fe000\n')
f('sys/module/usbhid/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/usbhid/sections/.data', 0o664, b'0xffffffffa02037c0\n')
f('sys/module/usbhid/sections/.symtab', 0o664, b'0xffffffffa0201e90\n')
f('sys/module/usbhid/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02039c0\n')
f('sys/module/usbhid/sections/.rodata.str1.1', 0o664, b'0xffffffffa02017a8\n')
f('sys/module/usbhid/sections/__kcrctab_gpl', 0o664, b'0xffffffffa0201e70\n')
d('sys/module/usbhid/notes', 0o775)
f('sys/module/usbhid/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00F\xd8\xa9\xb5\x91:~\xd6KY>\x94\xe1\x1d\xab\xf6\xe5\xeeu*')
d('sys/module/iwl3945', 0o775)
f('sys/module/iwl3945/version', 0o664, b'1.2.26ks\n')
f('sys/module/iwl3945/initstate', 0o664, b'live\n')
f('sys/module/iwl3945/srcversion', 0o664, b'5C079549ABD48E07B20F3C7\n')
f('sys/module/iwl3945/refcnt', 0o664, b'0\n')
d('sys/module/iwl3945/drivers', 0o775)
l('sys/module/iwl3945/drivers/pci:iwl3945', '../../../bus/pci/drivers/iwl3945')
d('sys/module/iwl3945/parameters', 0o775)
f('sys/module/iwl3945/parameters/disable_hw_scan', 0o664, b'0\n')
f('sys/module/iwl3945/parameters/queues_num', 0o664, b'8\n')
f('sys/module/iwl3945/parameters/debug', 0o664, b'0\n')
f('sys/module/iwl3945/parameters/disable', 0o664, b'0\n')
f('sys/module/iwl3945/parameters/hwcrypto', 0o664, b'0\n')
f('sys/module/iwl3945/parameters/antenna', 0o664, b'0\n')
f('sys/module/iwl3945/parameters/qos_enable', 0o664, b'1\n')
d('sys/module/iwl3945/sections', 0o775)
f('sys/module/iwl3945/sections/.smp_locks', 0o664, b'0xffffffffa015e0e0\n')
f('sys/module/iwl3945/sections/__bug_table', 0o664, b'0xffffffffa015e300\n')
f('sys/module/iwl3945/sections/.exit.text', 0o664, b'0xffffffffa015b2e4\n')
f('sys/module/iwl3945/sections/.note.gnu.build-id', 0o664, b'0xffffffffa015b2fc\n')
f('sys/module/iwl3945/sections/.strtab', 0o664, b'0xffffffffa0160f20\n')
f('sys/module/iwl3945/sections/.devexit.text', 0o664, b'0xffffffffa015b044\n')
f('sys/module/iwl3945/sections/.rodata', 0o664, b'0xffffffffa015b320\n')
f('sys/module/iwl3945/sections/__param', 0o664, b'0xffffffffa015e4b0\n')
f('sys/module/iwl3945/sections/.bss', 0o664, b'0xffffffffa0163d80\n')
f('sys/module/iwl3945/sections/.text', 0o664, b'0xffffffffa014b000\n')
f('sys/module/iwl3945/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/iwl3945/sections/.data', 0o664, b'0xffffffffa0162f00\n')
f('sys/module/iwl3945/sections/.symtab', 0o664, b'0xffffffffa015e5c8\n')
f('sys/module/iwl3945/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01639c0\n')
f('sys/module/iwl3945/sections/.rodata.str1.1', 0o664, b'0xffffffffa015bc0c\n')
d('sys/module/iwl3945/notes', 0o775)
f('sys/module/iwl3945/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xd1^"`\xf9\xf2&\x0b\xe0T\x9b\xe1,\xab\x88\x9bos\x0b\xdb')
d('sys/module/rsrc_nonstatic', 0o775)
f('sys/module/rsrc_nonstatic/initstate', 0o664, b'live\n')
f('sys/module/rsrc_nonstatic/srcversion', 0o664, b'8554CBDFE6293476856E830\n')
f('sys/module/rsrc_nonstatic/refcnt', 0o664, b'1\n')
d('sys/module/rsrc_nonstatic/holders', 0o775)
l('sys/module/rsrc_nonstatic/holders/yenta_socket', '../../yenta_socket')
d('sys/module/rsrc_nonstatic/parameters', 0o775)
f('sys/module/rsrc_nonstatic/parameters/probe_mem', 0o664, b'1\n')
d('sys/module/rsrc_nonstatic/sections', 0o775)
f('sys/module/rsrc_nonstatic/sections/__ksymtab_strings', 0o664, b'0xffffffffa00ef8a0\n')
f('sys/module/rsrc_nonstatic/sections/.exit.text', 0o664, b'0xffffffffa00ef624\n')
f('sys/module/rsrc_nonstatic/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00ef69c\n')
f('sys/module/rsrc_nonstatic/sections/.strtab', 0o664, b'0xffffffffa00f0260\n')
f('sys/module/rsrc_nonstatic/sections/__ksymtab', 0o664, b'0xffffffffa00ef860\n')
f('sys/module/rsrc_nonstatic/sections/.devexit.text', 0o664, b'0xffffffffa00ef636\n')
f('sys/module/rsrc_nonstatic/sections/.rodata', 0o664, b'0xffffffffa00ef6c0\n')
f('sys/module/rsrc_nonstatic/sections/__param', 0o664, b'0xffffffffa00ef878\n')
f('sys/module/rsrc_nonstatic/sections/.bss', 0o664, b'0xffffffffa00f0dc0\n')
f('sys/module/rsrc_nonstatic/sections/.text', 0o664, b'0xffffffffa00ee000\n')
f('sys/module/rsrc_nonstatic/sections/.init.text', 0o664, b'0xffffffffa0084000\n')
f('sys/module/rsrc_nonstatic/sections/__kcrctab', 0o664, b'0xffffffffa00ef870\n')
f('sys/module/rsrc_nonstatic/sections/.ref.data', 0o664, b'0xffffffffa00f09c0\n')
f('sys/module/rsrc_nonstatic/sections/.data', 0o664, b'0xffffffffa00f0820\n')
f('sys/module/rsrc_nonstatic/sections/.devinit.text', 0o664, b'0xffffffffa00ef665\n')
f('sys/module/rsrc_nonstatic/sections/.symtab', 0o664, b'0xffffffffa00ef8b8\n')
f('sys/module/rsrc_nonstatic/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00f0a00\n')
f('sys/module/rsrc_nonstatic/sections/.rodata.str1.1', 0o664, b'0xffffffffa00ef78d\n')
d('sys/module/rsrc_nonstatic/notes', 0o775)
f('sys/module/rsrc_nonstatic/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xd5n\x94\x07\x89\xe1\x08\xe3\x07C\xb9\x83\x0c\x19*6\xe8\xe3\x10R')
d('sys/module/thinkpad_acpi', 0o775)
f('sys/module/thinkpad_acpi/version', 0o664, b'0.21\n')
f('sys/module/thinkpad_acpi/initstate', 0o664, b'live\n')
f('sys/module/thinkpad_acpi/srcversion', 0o664, b'FA276F7EFAA90BF39100654\n')
f('sys/module/thinkpad_acpi/refcnt', 0o664, b'0\n')
d('sys/module/thinkpad_acpi/drivers', 0o775)
l('sys/module/thinkpad_acpi/drivers/platform:thinkpad_hwmon', '../../../bus/platform/drivers/thinkpad_hwmon')
l('sys/module/thinkpad_acpi/drivers/platform:thinkpad_acpi', '../../../bus/platform/drivers/thinkpad_acpi')
d('sys/module/thinkpad_acpi/sections', 0o775)
f('sys/module/thinkpad_acpi/sections/.smp_locks', 0o664, b'0xffffffffa01e6088\n')
f('sys/module/thinkpad_acpi/sections/__bug_table', 0o664, b'0xffffffffa01e60e0\n')
f('sys/module/thinkpad_acpi/sections/.note.gnu.build-id', 0o664, b'0xffffffffa01e3a80\n')
f('sys/module/thinkpad_acpi/sections/.strtab', 0o664, b'0xffffffffa01e98b8\n')
f('sys/module/thinkpad_acpi/sections/.rodata', 0o664, b'0xffffffffa01e3ac0\n')
f('sys/module/thinkpad_acpi/sections/__param', 0o664, b'0xffffffffa01e61a0\n')
f('sys/module/thinkpad_acpi/sections/.bss', 0o664, b'0xffffffffa01ed600\n')
f('sys/module/thinkpad_acpi/sections/.text', 0o664, b'0xffffffffa01dd000\n')
f('sys/module/thinkpad_acpi/sections/.init.text', 0o664, b'0xffffffffa01ef000\n')
f('sys/module/thinkpad_acpi/sections/.data', 0o664, b'0xffffffffa01ebd60\n')
f('sys/module/thinkpad_acpi/sections/.init.data', 0o664, b'0xffffffffa01f1700\n')
f('sys/module/thinkpad_acpi/sections/.symtab', 0o664, b'0xffffffffa01e6498\n')
f('sys/module/thinkpad_acpi/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01ed240\n')
f('sys/module/thinkpad_acpi/sections/.rodata.str1.1', 0o664, b'0xffffffffa01e3df0\n')
d('sys/module/thinkpad_acpi/notes', 0o775)
f('sys/module/thinkpad_acpi/notes/.note.gnu.build-id', 0o664, b"\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00]\xca\x1f\xba\xb1J'\xa8\x14\x83cj=h\x11\xe60;\x08\xd6")
d('sys/module/psmouse', 0o775)
d('sys/module/psmouse/drivers', 0o775)
l('sys/module/psmouse/drivers/serio:psmouse', '../../../bus/serio/drivers/psmouse')
d('sys/module/psmouse/parameters', 0o775)
f('sys/module/psmouse/parameters/proto', 0o664, b'''auto

''')
f('sys/module/psmouse/parameters/resync_time', 0o664, b'0\n')
f('sys/module/psmouse/parameters/resolution', 0o664, b'200\n')
f('sys/module/psmouse/parameters/resetafter', 0o664, b'5\n')
f('sys/module/psmouse/parameters/rate', 0o664, b'100\n')
f('sys/module/psmouse/parameters/smartscroll', 0o664, b'Y\n')
d('sys/module/md_mod', 0o775)
f('sys/module/md_mod/initstate', 0o664, b'live\n')
f('sys/module/md_mod/srcversion', 0o664, b'1DB954BE8FA97D4755DC15B\n')
f('sys/module/md_mod/refcnt', 0o664, b'1\n')
d('sys/module/md_mod/holders', 0o775)
l('sys/module/md_mod/holders/raid1', '../../raid1')
d('sys/module/md_mod/parameters', 0o775)
f('sys/module/md_mod/parameters/start_ro', 0o664, b'0\n')
f('sys/module/md_mod/parameters/start_dirty_degraded', 0o664, b'0\n')
d('sys/module/md_mod/sections', 0o775)
f('sys/module/md_mod/sections/__ksymtab_gpl', 0o664, b'0xffffffffa022a790\n')
f('sys/module/md_mod/sections/.smp_locks', 0o664, b'0xffffffffa022a038\n')
f('sys/module/md_mod/sections/__bug_table', 0o664, b'0xffffffffa022a4b8\n')
f('sys/module/md_mod/sections/__ksymtab_strings', 0o664, b'0xffffffffa022a7f0\n')
f('sys/module/md_mod/sections/.exit.text', 0o664, b'0xffffffffa02283a0\n')
f('sys/module/md_mod/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02284c8\n')
f('sys/module/md_mod/sections/.strtab', 0o664, b'0xffffffffa022d7a0\n')
f('sys/module/md_mod/sections/__ksymtab', 0o664, b'0xffffffffa022a590\n')
f('sys/module/md_mod/sections/.rodata', 0o664, b'0xffffffffa0228500\n')
f('sys/module/md_mod/sections/__param', 0o664, b'0xffffffffa022a740\n')
f('sys/module/md_mod/sections/.bss', 0o664, b'0xffffffffa0230880\n')
f('sys/module/md_mod/sections/.text', 0o664, b'0xffffffffa021a000\n')
f('sys/module/md_mod/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/md_mod/sections/__kcrctab', 0o664, b'0xffffffffa022a6b0\n')
f('sys/module/md_mod/sections/.data', 0o664, b'0xffffffffa022f660\n')
f('sys/module/md_mod/sections/.symtab', 0o664, b'0xffffffffa022a968\n')
f('sys/module/md_mod/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02304c0\n')
f('sys/module/md_mod/sections/.rodata.str1.1', 0o664, b'0xffffffffa022869e\n')
f('sys/module/md_mod/sections/__kcrctab_gpl', 0o664, b'0xffffffffa022a7d0\n')
d('sys/module/md_mod/notes', 0o775)
f('sys/module/md_mod/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xbd\xb6!\x84\x88Z\x8bnL\x0f\xf0\xbe\xf0]\n"\xd8\xf3\xb0*')
d('sys/module/snd_pcm', 0o775)
f('sys/module/snd_pcm/initstate', 0o664, b'live\n')
f('sys/module/snd_pcm/srcversion', 0o664, b'96530AD1426FF7F73757C4F\n')
f('sys/module/snd_pcm/refcnt', 0o664, b'2\n')
d('sys/module/snd_pcm/holders', 0o775)
l('sys/module/snd_pcm/holders/snd_hda_intel', '../../snd_hda_intel')
l('sys/module/snd_pcm/holders/snd_pcm_oss', '../../snd_pcm_oss')
d('sys/module/snd_pcm/parameters', 0o775)
f('sys/module/snd_pcm/parameters/preallocate_dma', 0o664, b'1\n')
f('sys/module/snd_pcm/parameters/maximum_substreams', 0o664, b'4\n')
d('sys/module/snd_pcm/sections', 0o775)
f('sys/module/snd_pcm/sections/.smp_locks', 0o664, b'0xffffffffa0141af8\n')
f('sys/module/snd_pcm/sections/__ex_table', 0o664, b'0xffffffffa0141b30\n')
f('sys/module/snd_pcm/sections/__ksymtab_strings', 0o664, b'0xffffffffa01415f0\n')
f('sys/module/snd_pcm/sections/.exit.text', 0o664, b'0xffffffffa01401d4\n')
f('sys/module/snd_pcm/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0140260\n')
f('sys/module/snd_pcm/sections/.strtab', 0o664, b'0xffffffffa0145428\n')
f('sys/module/snd_pcm/sections/.fixup', 0o664, b'0xffffffffa01401fe\n')
f('sys/module/snd_pcm/sections/__ksymtab', 0o664, b'0xffffffffa01410b0\n')
f('sys/module/snd_pcm/sections/.rodata', 0o664, b'0xffffffffa01402a0\n')
f('sys/module/snd_pcm/sections/__param', 0o664, b'0xffffffffa0141bb0\n')
f('sys/module/snd_pcm/sections/.bss', 0o664, b'0xffffffffa0149640\n')
f('sys/module/snd_pcm/sections/.text', 0o664, b'0xffffffffa0134000\n')
f('sys/module/snd_pcm/sections/.init.text', 0o664, b'0xffffffffa007d000\n')
f('sys/module/snd_pcm/sections/__kcrctab', 0o664, b'0xffffffffa0141430\n')
f('sys/module/snd_pcm/sections/.data', 0o664, b'0xffffffffa0148b60\n')
f('sys/module/snd_pcm/sections/.symtab', 0o664, b'0xffffffffa0141c00\n')
f('sys/module/snd_pcm/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0149280\n')
f('sys/module/snd_pcm/sections/.rodata.str1.1', 0o664, b'0xffffffffa0140930\n')
d('sys/module/snd_pcm/notes', 0o775)
f('sys/module/snd_pcm/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xc8N\x86\x9c\x8dq\x14\x04\xe7\xed\xee\x9b\xe0\x8f\xe4\x8fvF{\xfe')
d('sys/module/snd_hda_intel', 0o775)
f('sys/module/snd_hda_intel/initstate', 0o664, b'live\n')
f('sys/module/snd_hda_intel/srcversion', 0o664, b'B76633A09F11336101E5480\n')
f('sys/module/snd_hda_intel/refcnt', 0o664, b'3\n')
d('sys/module/snd_hda_intel/drivers', 0o775)
l('sys/module/snd_hda_intel/drivers/pci:HDA Intel', '../../../bus/pci/drivers/HDA Intel')
d('sys/module/snd_hda_intel/parameters', 0o775)
f('sys/module/snd_hda_intel/parameters/bdl_pos_adj', 0o664, b'1,-1,-1,-1,-1,-1,-1,-1\n')
f('sys/module/snd_hda_intel/parameters/enable', 0o664, b'Y,Y,Y,Y,Y,Y,Y,Y\n')
f('sys/module/snd_hda_intel/parameters/id', 0o664, b'<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>\n')
f('sys/module/snd_hda_intel/parameters/probe_mask', 0o664, b'1,-1,-1,-1,-1,-1,-1,-1\n')
f('sys/module/snd_hda_intel/parameters/enable_msi', 0o664, b'0\n')
f('sys/module/snd_hda_intel/parameters/index', 0o664, b'-1,-1,-1,-1,-1,-1,-1,-1\n')
f('sys/module/snd_hda_intel/parameters/single_cmd', 0o664, b'N\n')
f('sys/module/snd_hda_intel/parameters/model', 0o664, b'<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>,<NULL>\n')
f('sys/module/snd_hda_intel/parameters/position_fix', 0o664, b'0,0,0,0,0,0,0,0\n')
d('sys/module/snd_hda_intel/sections', 0o775)
f('sys/module/snd_hda_intel/sections/.devinit.data', 0o664, b'0xffffffffa01d7340\n')
f('sys/module/snd_hda_intel/sections/.exit.text', 0o664, b'0xffffffffa0182a58\n')
f('sys/module/snd_hda_intel/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0183f4c\n')
f('sys/module/snd_hda_intel/sections/.strtab', 0o664, b'0xffffffffa0193910\n')
f('sys/module/snd_hda_intel/sections/.devexit.text', 0o664, b'0xffffffffa0182a6a\n')
f('sys/module/snd_hda_intel/sections/.rodata', 0o664, b'0xffffffffa0183f80\n')
f('sys/module/snd_hda_intel/sections/__param', 0o664, b'0xffffffffa0189de8\n')
f('sys/module/snd_hda_intel/sections/.bss', 0o664, b'0xffffffffa01d7800\n')
f('sys/module/snd_hda_intel/sections/.text', 0o664, b'0xffffffffa0165000\n')
f('sys/module/snd_hda_intel/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/snd_hda_intel/sections/.data', 0o664, b'0xffffffffa019bd40\n')
f('sys/module/snd_hda_intel/sections/.devinit.text', 0o664, b'0xffffffffa0182aa0\n')
f('sys/module/snd_hda_intel/sections/.symtab', 0o664, b'0xffffffffa0189f50\n')
f('sys/module/snd_hda_intel/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01d7440\n')
f('sys/module/snd_hda_intel/sections/.rodata.str1.1', 0o664, b'0xffffffffa0186d8b\n')
d('sys/module/snd_hda_intel/notes', 0o775)
f('sys/module/snd_hda_intel/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x002\xfe\x91\xd2K\x86\x1d\x92P\xdcX/c\x07=\xf3\xceY\xbc-')
d('sys/module/ehci_hcd', 0o775)
f('sys/module/ehci_hcd/initstate', 0o664, b'live\n')
f('sys/module/ehci_hcd/srcversion', 0o664, b'6EBBF13EC8AD467D8AB11A4\n')
f('sys/module/ehci_hcd/refcnt', 0o664, b'0\n')
d('sys/module/ehci_hcd/drivers', 0o775)
l('sys/module/ehci_hcd/drivers/pci:ehci_hcd', '../../../bus/pci/drivers/ehci_hcd')
d('sys/module/ehci_hcd/parameters', 0o775)
f('sys/module/ehci_hcd/parameters/park', 0o664, b'0\n')
f('sys/module/ehci_hcd/parameters/ignore_oc', 0o664, b'N\n')
f('sys/module/ehci_hcd/parameters/log2_irq_thresh', 0o664, b'0\n')
d('sys/module/ehci_hcd/sections', 0o775)
f('sys/module/ehci_hcd/sections/.smp_locks', 0o664, b'0xffffffffa00e4048\n')
f('sys/module/ehci_hcd/sections/__bug_table', 0o664, b'0xffffffffa00e3f22\n')
f('sys/module/ehci_hcd/sections/.exit.text', 0o664, b'0xffffffffa00e3b98\n')
f('sys/module/ehci_hcd/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00e3bac\n')
f('sys/module/ehci_hcd/sections/.strtab', 0o664, b'0xffffffffa00e54a8\n')
f('sys/module/ehci_hcd/sections/.rodata', 0o664, b'0xffffffffa00e3be0\n')
f('sys/module/ehci_hcd/sections/__param', 0o664, b'0xffffffffa00e42f0\n')
f('sys/module/ehci_hcd/sections/.bss', 0o664, b'0xffffffffa00e6480\n')
f('sys/module/ehci_hcd/sections/.text', 0o664, b'0xffffffffa00dd000\n')
f('sys/module/ehci_hcd/sections/.init.text', 0o664, b'0xffffffffa006b000\n')
f('sys/module/ehci_hcd/sections/.data', 0o664, b'0xffffffffa00e5f40\n')
f('sys/module/ehci_hcd/sections/.symtab', 0o664, b'0xffffffffa00e4368\n')
f('sys/module/ehci_hcd/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00e60c0\n')
f('sys/module/ehci_hcd/sections/.rodata.str1.1', 0o664, b'0xffffffffa00e3dd0\n')
d('sys/module/ehci_hcd/notes', 0o775)
f('sys/module/ehci_hcd/notes/.note.gnu.build-id', 0o664, b"\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00 \xc7\x01\xa9Mi;\xcaNB'\x08T\xc4\xf9\x80\xd7\x10\xb3h")
d('sys/module/sg', 0o775)
f('sys/module/sg/version', 0o664, b'3.5.34\n')
f('sys/module/sg/initstate', 0o664, b'live\n')
f('sys/module/sg/srcversion', 0o664, b'DC7DE342543B6D7AB46D718\n')
f('sys/module/sg/refcnt', 0o664, b'0\n')
d('sys/module/sg/parameters', 0o775)
f('sys/module/sg/parameters/allow_dio', 0o664, b'0\n')
f('sys/module/sg/parameters/scatter_elem_sz', 0o664, b'32768\n')
f('sys/module/sg/parameters/def_reserved_size', 0o664, b'32768\n')
d('sys/module/sg/sections', 0o775)
f('sys/module/sg/sections/.smp_locks', 0o664, b'0xffffffffa00054d8\n')
f('sys/module/sg/sections/__ex_table', 0o664, b'0xffffffffa0005530\n')
f('sys/module/sg/sections/__bug_table', 0o664, b'0xffffffffa00054e8\n')
f('sys/module/sg/sections/.exit.text', 0o664, b'0xffffffffa00049f0\n')
f('sys/module/sg/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0004afc\n')
f('sys/module/sg/sections/.strtab', 0o664, b'0xffffffffa0006710\n')
f('sys/module/sg/sections/.fixup', 0o664, b'0xffffffffa0004a33\n')
f('sys/module/sg/sections/.rodata', 0o664, b'0xffffffffa0004b20\n')
f('sys/module/sg/sections/__param', 0o664, b'0xffffffffa0005630\n')
f('sys/module/sg/sections/.bss', 0o664, b'0xffffffffa0007640\n')
f('sys/module/sg/sections/.text', 0o664, b'0xffffffffa0000000\n')
f('sys/module/sg/sections/.init.text', 0o664, b'0xffffffffa0009000\n')
f('sys/module/sg/sections/.data', 0o664, b'0xffffffffa0007040\n')
f('sys/module/sg/sections/.symtab', 0o664, b'0xffffffffa00056a8\n')
f('sys/module/sg/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0007280\n')
f('sys/module/sg/sections/.rodata.str1.1', 0o664, b'0xffffffffa0004bc0\n')
d('sys/module/sg/notes', 0o775)
f('sys/module/sg/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00q\x06\xb5<\xb3\x13\xd4\xe6\xc5W\xbb;{\x14\x93\xd5\x07\xc7\x13\xfa')
d('sys/module/snd_mixer_oss', 0o775)
f('sys/module/snd_mixer_oss/initstate', 0o664, b'live\n')
f('sys/module/snd_mixer_oss/srcversion', 0o664, b'9D2BFE821DD0ACFA2CFF364\n')
f('sys/module/snd_mixer_oss/refcnt', 0o664, b'1\n')
d('sys/module/snd_mixer_oss/holders', 0o775)
l('sys/module/snd_mixer_oss/holders/snd_pcm_oss', '../../snd_pcm_oss')
d('sys/module/snd_mixer_oss/sections', 0o775)
f('sys/module/snd_mixer_oss/sections/__ksymtab_strings', 0o664, b'0xffffffffa02bba78\n')
f('sys/module/snd_mixer_oss/sections/.exit.text', 0o664, b'0xffffffffa02bb508\n')
f('sys/module/snd_mixer_oss/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02bb54c\n')
f('sys/module/snd_mixer_oss/sections/.strtab', 0o664, b'0xffffffffa02bc470\n')
f('sys/module/snd_mixer_oss/sections/__ksymtab', 0o664, b'0xffffffffa02bba60\n')
f('sys/module/snd_mixer_oss/sections/.rodata', 0o664, b'0xffffffffa02bb580\n')
f('sys/module/snd_mixer_oss/sections/.bss', 0o664, b'0xffffffffa02bd240\n')
f('sys/module/snd_mixer_oss/sections/.text', 0o664, b'0xffffffffa02b9000\n')
f('sys/module/snd_mixer_oss/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd_mixer_oss/sections/__kcrctab', 0o664, b'0xffffffffa02bba70\n')
f('sys/module/snd_mixer_oss/sections/.data', 0o664, b'0xffffffffa02bcb20\n')
f('sys/module/snd_mixer_oss/sections/.symtab', 0o664, b'0xffffffffa02bba98\n')
f('sys/module/snd_mixer_oss/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02bce80\n')
f('sys/module/snd_mixer_oss/sections/.rodata.str1.1', 0o664, b'0xffffffffa02bb760\n')
d('sys/module/snd_mixer_oss/notes', 0o775)
f('sys/module/snd_mixer_oss/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xc3u\x0b\x0eh,\x90W\xba\xf3S\xa9\x16\\<\x97u\x0f.\xa6')
d('sys/module/pcmcia', 0o775)
f('sys/module/pcmcia/initstate', 0o664, b'live\n')
f('sys/module/pcmcia/srcversion', 0o664, b'6544CB07B8D7D79F19A4CD6\n')
f('sys/module/pcmcia/refcnt', 0o664, b'0\n')
d('sys/module/pcmcia/parameters', 0o775)
f('sys/module/pcmcia/parameters/io_speed', 0o664, b'0\n')
d('sys/module/pcmcia/sections', 0o775)
f('sys/module/pcmcia/sections/__ksymtab_strings', 0o664, b'0xffffffffa01f9890\n')
f('sys/module/pcmcia/sections/.exit.text', 0o664, b'0xffffffffa01f85c4\n')
f('sys/module/pcmcia/sections/.note.gnu.build-id', 0o664, b'0xffffffffa01f86c8\n')
f('sys/module/pcmcia/sections/.strtab', 0o664, b'0xffffffffa01fb138\n')
f('sys/module/pcmcia/sections/__ksymtab', 0o664, b'0xffffffffa01f9710\n')
f('sys/module/pcmcia/sections/.rodata', 0o664, b'0xffffffffa01f8700\n')
f('sys/module/pcmcia/sections/__param', 0o664, b'0xffffffffa01f9a00\n')
f('sys/module/pcmcia/sections/.bss', 0o664, b'0xffffffffa01fcb40\n')
f('sys/module/pcmcia/sections/.text', 0o664, b'0xffffffffa01f5000\n')
f('sys/module/pcmcia/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/pcmcia/sections/__kcrctab', 0o664, b'0xffffffffa01f9810\n')
f('sys/module/pcmcia/sections/.ref.data', 0o664, b'0xffffffffa01fc720\n')
f('sys/module/pcmcia/sections/.data', 0o664, b'0xffffffffa01fc3a0\n')
f('sys/module/pcmcia/sections/.devinit.text', 0o664, b'0xffffffffa01f85e2\n')
f('sys/module/pcmcia/sections/.symtab', 0o664, b'0xffffffffa01f9a28\n')
f('sys/module/pcmcia/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa01fc780\n')
f('sys/module/pcmcia/sections/.rodata.str1.1', 0o664, b'0xffffffffa01f913d\n')
d('sys/module/pcmcia/notes', 0o775)
f('sys/module/pcmcia/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00v\xd6BR5q\xd2\xd4\x05y\xe0\x84j\x17\xab\xb2\xe6o\xd43')
d('sys/module/nvram', 0o775)
f('sys/module/nvram/initstate', 0o664, b'live\n')
f('sys/module/nvram/srcversion', 0o664, b'ECD12B2949BC771BDCE8146\n')
f('sys/module/nvram/refcnt', 0o664, b'1\n')
d('sys/module/nvram/holders', 0o775)
l('sys/module/nvram/holders/thinkpad_acpi', '../../thinkpad_acpi')
d('sys/module/nvram/sections', 0o775)
f('sys/module/nvram/sections/__ksymtab_strings', 0o664, b'0xffffffffa00ad2c0\n')
f('sys/module/nvram/sections/.exit.text', 0o664, b'0xffffffffa00ace3c\n')
f('sys/module/nvram/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00ace5c\n')
f('sys/module/nvram/sections/.strtab', 0o664, b'0xffffffffa00adc38\n')
f('sys/module/nvram/sections/__ksymtab', 0o664, b'0xffffffffa00ad230\n')
f('sys/module/nvram/sections/.rodata', 0o664, b'0xffffffffa00ace80\n')
f('sys/module/nvram/sections/.bss', 0o664, b'0xffffffffa00ae640\n')
f('sys/module/nvram/sections/.text', 0o664, b'0xffffffffa00ac000\n')
f('sys/module/nvram/sections/.init.text', 0o664, b'0xffffffffa00b0000\n')
f('sys/module/nvram/sections/__kcrctab', 0o664, b'0xffffffffa00ad290\n')
f('sys/module/nvram/sections/.data', 0o664, b'0xffffffffa00ae200\n')
f('sys/module/nvram/sections/.symtab', 0o664, b'0xffffffffa00ad338\n')
f('sys/module/nvram/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00ae280\n')
f('sys/module/nvram/sections/.rodata.str1.1', 0o664, b'0xffffffffa00acfb8\n')
d('sys/module/nvram/notes', 0o775)
f('sys/module/nvram/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x00j\xf9\trf\x94M\xafD\\s\xced\xcc\x97c\xd5L\n')
d('sys/module/scsi_mod', 0o775)
d('sys/module/scsi_mod/parameters', 0o775)
f('sys/module/scsi_mod/parameters/scan', 0o664, b'sync\n')
f('sys/module/scsi_mod/parameters/default_dev_flags', 0o664, b'0\n')
f('sys/module/scsi_mod/parameters/scsi_logging_level', 0o664, b'0\n')
f('sys/module/scsi_mod/parameters/inq_timeout', 0o664, b'5\n')
f('sys/module/scsi_mod/parameters/max_luns', 0o664, b'512\n')
f('sys/module/scsi_mod/parameters/max_report_luns', 0o664, b'511\n')
d('sys/module/pcmcia_core', 0o775)
f('sys/module/pcmcia_core/initstate', 0o664, b'live\n')
f('sys/module/pcmcia_core/srcversion', 0o664, b'7C1220572DE8897E4D560EB\n')
f('sys/module/pcmcia_core/refcnt', 0o664, b'3\n')
d('sys/module/pcmcia_core/holders', 0o775)
l('sys/module/pcmcia_core/holders/yenta_socket', '../../yenta_socket')
l('sys/module/pcmcia_core/holders/rsrc_nonstatic', '../../rsrc_nonstatic')
l('sys/module/pcmcia_core/holders/pcmcia', '../../pcmcia')
d('sys/module/pcmcia_core/parameters', 0o775)
f('sys/module/pcmcia_core/parameters/setup_delay', 0o664, b'10\n')
f('sys/module/pcmcia_core/parameters/cis_speed', 0o664, b'300\n')
f('sys/module/pcmcia_core/parameters/unreset_delay', 0o664, b'10\n')
f('sys/module/pcmcia_core/parameters/shutdown_delay', 0o664, b'3\n')
f('sys/module/pcmcia_core/parameters/cis_width', 0o664, b'0\n')
f('sys/module/pcmcia_core/parameters/unreset_limit', 0o664, b'30\n')
f('sys/module/pcmcia_core/parameters/reset_time', 0o664, b'10\n')
f('sys/module/pcmcia_core/parameters/resume_delay', 0o664, b'20\n')
f('sys/module/pcmcia_core/parameters/unreset_check', 0o664, b'10\n')
f('sys/module/pcmcia_core/parameters/vcc_settle', 0o664, b'40\n')
d('sys/module/pcmcia_core/sections', 0o775)
f('sys/module/pcmcia_core/sections/__ksymtab_strings', 0o664, b'0xffffffffa0094310\n')
f('sys/module/pcmcia_core/sections/.exit.text', 0o664, b'0xffffffffa0093558\n')
f('sys/module/pcmcia_core/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0093668\n')
f('sys/module/pcmcia_core/sections/.strtab', 0o664, b'0xffffffffa0096898\n')
f('sys/module/pcmcia_core/sections/__ksymtab', 0o664, b'0xffffffffa0093e50\n')
f('sys/module/pcmcia_core/sections/.rodata', 0o664, b'0xffffffffa00936a0\n')
f('sys/module/pcmcia_core/sections/__param', 0o664, b'0xffffffffa0094180\n')
f('sys/module/pcmcia_core/sections/.bss', 0o664, b'0xffffffffa0098f40\n')
f('sys/module/pcmcia_core/sections/.text', 0o664, b'0xffffffffa008f000\n')
f('sys/module/pcmcia_core/sections/.init.text', 0o664, b'0xffffffffa009a000\n')
f('sys/module/pcmcia_core/sections/__kcrctab', 0o664, b'0xffffffffa0094070\n')
f('sys/module/pcmcia_core/sections/.data', 0o664, b'0xffffffffa0098700\n')
f('sys/module/pcmcia_core/sections/.symtab', 0o664, b'0xffffffffa00945d0\n')
f('sys/module/pcmcia_core/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0098b80\n')
f('sys/module/pcmcia_core/sections/.rodata.str1.1', 0o664, b'0xffffffffa0093a88\n')
f('sys/module/pcmcia_core/sections/.ref.text', 0o664, b'0xffffffffa0093580\n')
d('sys/module/pcmcia_core/notes', 0o775)
f('sys/module/pcmcia_core/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00M\xe5~\xb9\x10\x7f(\xb8\x18\xe2ma\xb4\xdc\x9a\x8er\xb2t\xbc')
d('sys/module/rtc_core', 0o775)
f('sys/module/rtc_core/initstate', 0o664, b'live\n')
f('sys/module/rtc_core/srcversion', 0o664, b'8684A14821938B7D8E50DB5\n')
f('sys/module/rtc_core/refcnt', 0o664, b'1\n')
d('sys/module/rtc_core/holders', 0o775)
l('sys/module/rtc_core/holders/rtc_cmos', '../../rtc_cmos')
d('sys/module/rtc_core/sections', 0o775)
f('sys/module/rtc_core/sections/__ksymtab_gpl', 0o664, b'0xffffffffa00ea380\n')
f('sys/module/rtc_core/sections/.smp_locks', 0o664, b'0xffffffffa00ea5b8\n')
f('sys/module/rtc_core/sections/__ksymtab_strings', 0o664, b'0xffffffffa00ea4d0\n')
f('sys/module/rtc_core/sections/.exit.text', 0o664, b'0xffffffffa00e9e60\n')
f('sys/module/rtc_core/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00e9ea4\n')
f('sys/module/rtc_core/sections/.strtab', 0o664, b'0xffffffffa00eb9c0\n')
f('sys/module/rtc_core/sections/.rodata', 0o664, b'0xffffffffa00e9ee0\n')
f('sys/module/rtc_core/sections/.bss', 0o664, b'0xffffffffa00ecd80\n')
f('sys/module/rtc_core/sections/.text', 0o664, b'0xffffffffa00e8000\n')
f('sys/module/rtc_core/sections/.init.text', 0o664, b'0xffffffffa007d000\n')
f('sys/module/rtc_core/sections/.data', 0o664, b'0xffffffffa00ec760\n')
f('sys/module/rtc_core/sections/.symtab', 0o664, b'0xffffffffa00ea5e0\n')
f('sys/module/rtc_core/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00ec9c0\n')
f('sys/module/rtc_core/sections/.rodata.str1.1', 0o664, b'0xffffffffa00ea098\n')
f('sys/module/rtc_core/sections/__kcrctab_gpl', 0o664, b'0xffffffffa00ea460\n')
d('sys/module/rtc_core/notes', 0o775)
f('sys/module/rtc_core/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xfbj\x90X\x98\xf7\x0e\x95\x19\xc6\xa1\xa1\xcd\xe0\x835$\x0e\x96\x17')
d('sys/module/led_class', 0o775)
f('sys/module/led_class/initstate', 0o664, b'live\n')
f('sys/module/led_class/srcversion', 0o664, b'B916A50F53ABBBB19DFF203\n')
f('sys/module/led_class/refcnt', 0o664, b'2\n')
d('sys/module/led_class/holders', 0o775)
l('sys/module/led_class/holders/iwl3945', '../../iwl3945')
l('sys/module/led_class/holders/thinkpad_acpi', '../../thinkpad_acpi')
d('sys/module/led_class/sections', 0o775)
f('sys/module/led_class/sections/__ksymtab_gpl', 0o664, b'0xffffffffa00f23c0\n')
f('sys/module/led_class/sections/__ksymtab_strings', 0o664, b'0xffffffffa00f2420\n')
f('sys/module/led_class/sections/.exit.text', 0o664, b'0xffffffffa00f2330\n')
f('sys/module/led_class/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00f2344\n')
f('sys/module/led_class/sections/.strtab', 0o664, b'0xffffffffa00f2bf8\n')
f('sys/module/led_class/sections/.bss', 0o664, b'0xffffffffa00f3500\n')
f('sys/module/led_class/sections/.text', 0o664, b'0xffffffffa00f2000\n')
f('sys/module/led_class/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/led_class/sections/.data', 0o664, b'0xffffffffa00f30c0\n')
f('sys/module/led_class/sections/.symtab', 0o664, b'0xffffffffa00f2478\n')
f('sys/module/led_class/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00f3140\n')
f('sys/module/led_class/sections/.rodata.str1.1', 0o664, b'0xffffffffa00f2368\n')
f('sys/module/led_class/sections/__kcrctab_gpl', 0o664, b'0xffffffffa00f2400\n')
d('sys/module/led_class/notes', 0o775)
f('sys/module/led_class/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00Iq;\xaf\xad\xf1\xe5\xcf8\xa5\x86u\xa9tMV\x02*\xfd\r')
d('sys/module/libata', 0o775)
d('sys/module/libata/parameters', 0o775)
f('sys/module/libata/parameters/acpi_gtf_filter', 0o664, b'7\n')
f('sys/module/libata/parameters/noacpi', 0o664, b'0\n')
f('sys/module/libata/parameters/atapi_passthru16', 0o664, b'1\n')
f('sys/module/libata/parameters/ata_probe_timeout', 0o664, b'0\n')
f('sys/module/libata/parameters/dma', 0o664, b'7\n')
f('sys/module/libata/parameters/atapi_dmadir', 0o664, b'0\n')
f('sys/module/libata/parameters/fua', 0o664, b'0\n')
f('sys/module/libata/parameters/ignore_hpa', 0o664, b'0\n')
f('sys/module/libata/parameters/atapi_enabled', 0o664, b'1\n')
f('sys/module/libata/parameters/allow_tpm', 0o664, b'0\n')
d('sys/module/thermal', 0o775)
f('sys/module/thermal/initstate', 0o664, b'live\n')
f('sys/module/thermal/srcversion', 0o664, b'1787CE9FEB053C917D031A9\n')
f('sys/module/thermal/refcnt', 0o664, b'0\n')
d('sys/module/thermal/parameters', 0o775)
f('sys/module/thermal/parameters/act', 0o664, b'0\n')
f('sys/module/thermal/parameters/tzp', 0o664, b'0\n')
f('sys/module/thermal/parameters/crt', 0o664, b'0\n')
f('sys/module/thermal/parameters/psv', 0o664, b'0\n')
d('sys/module/thermal/sections', 0o775)
f('sys/module/thermal/sections/.exit.text', 0o664, b'0xffffffffa00a1f14\n')
f('sys/module/thermal/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00a1f3c\n')
f('sys/module/thermal/sections/.strtab', 0o664, b'0xffffffffa00a3b28\n')
f('sys/module/thermal/sections/.rodata', 0o664, b'0xffffffffa00a1f60\n')
f('sys/module/thermal/sections/__param', 0o664, b'0xffffffffa00a29d0\n')
f('sys/module/thermal/sections/.bss', 0o664, b'0xffffffffa00a4bc0\n')
f('sys/module/thermal/sections/.text', 0o664, b'0xffffffffa00a0000\n')
f('sys/module/thermal/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/thermal/sections/.data', 0o664, b'0xffffffffa00a4640\n')
f('sys/module/thermal/sections/.init.data', 0o664, b'0xffffffffa0070090\n')
f('sys/module/thermal/sections/.symtab', 0o664, b'0xffffffffa00a2ac0\n')
f('sys/module/thermal/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00a4800\n')
f('sys/module/thermal/sections/.rodata.str1.1', 0o664, b'0xffffffffa00a242a\n')
d('sys/module/thermal/notes', 0o775)
f('sys/module/thermal/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x97\xa9\xec\xf2\x01\x86!\x94\x89\xa4n%q\t\xb7\xb4\x97a\x1a\x85')
d('sys/module/keyboard', 0o775)
d('sys/module/keyboard/parameters', 0o775)
f('sys/module/keyboard/parameters/brl_nbchords', 0o664, b'1\n')
f('sys/module/keyboard/parameters/brl_timeout', 0o664, b'300\n')
d('sys/module/ahci', 0o775)
d('sys/module/ahci/drivers', 0o775)
l('sys/module/ahci/drivers/pci:ahci', '../../../bus/pci/drivers/ahci')
d('sys/module/ahci/parameters', 0o775)
f('sys/module/ahci/parameters/marvell_enable', 0o664, b'1\n')
f('sys/module/ahci/parameters/ahci_em_messages', 0o664, b'1\n')
f('sys/module/ahci/parameters/skip_host_reset', 0o664, b'0\n')
d('sys/module/snd_pcm_oss', 0o775)
f('sys/module/snd_pcm_oss/initstate', 0o664, b'live\n')
f('sys/module/snd_pcm_oss/srcversion', 0o664, b'E3D05A52B04130012283029\n')
f('sys/module/snd_pcm_oss/refcnt', 0o664, b'0\n')
d('sys/module/snd_pcm_oss/parameters', 0o775)
f('sys/module/snd_pcm_oss/parameters/dsp_map', 0o664, b'0,0,0,0,0,0,0,0\n')
f('sys/module/snd_pcm_oss/parameters/nonblock_open', 0o664, b'Y\n')
f('sys/module/snd_pcm_oss/parameters/adsp_map', 0o664, b'1,1,1,1,1,1,1,1\n')
d('sys/module/snd_pcm_oss/sections', 0o775)
f('sys/module/snd_pcm_oss/sections/.exit.text', 0o664, b'0xffffffffa02c6e7c\n')
f('sys/module/snd_pcm_oss/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02c6e94\n')
f('sys/module/snd_pcm_oss/sections/.strtab', 0o664, b'0xffffffffa02c8818\n')
f('sys/module/snd_pcm_oss/sections/.rodata', 0o664, b'0xffffffffa02c6ec0\n')
f('sys/module/snd_pcm_oss/sections/__param', 0o664, b'0xffffffffa02c7378\n')
f('sys/module/snd_pcm_oss/sections/.bss', 0o664, b'0xffffffffa02c9b00\n')
f('sys/module/snd_pcm_oss/sections/.text', 0o664, b'0xffffffffa02bf000\n')
f('sys/module/snd_pcm_oss/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd_pcm_oss/sections/.data', 0o664, b'0xffffffffa02c96c0\n')
f('sys/module/snd_pcm_oss/sections/.symtab', 0o664, b'0xffffffffa02c73f0\n')
f('sys/module/snd_pcm_oss/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02c9740\n')
f('sys/module/snd_pcm_oss/sections/.rodata.str1.1', 0o664, b'0xffffffffa02c71a8\n')
d('sys/module/snd_pcm_oss/notes', 0o775)
f('sys/module/snd_pcm_oss/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xd7y|\xbf\xa5\x8aZm\x86\x85CF\x11s\xd3\xf5f\xbd7\xc6')
d('sys/module/ac', 0o775)
f('sys/module/ac/initstate', 0o664, b'live\n')
f('sys/module/ac/srcversion', 0o664, b'6E9CD20EE5BE8425A61B938\n')
f('sys/module/ac/refcnt', 0o664, b'0\n')
d('sys/module/ac/sections', 0o775)
f('sys/module/ac/sections/.exit.text', 0o664, b'0xffffffffa006d43c\n')
f('sys/module/ac/sections/.note.gnu.build-id', 0o664, b'0xffffffffa006d45c\n')
f('sys/module/ac/sections/.strtab', 0o664, b'0xffffffffa006dd28\n')
f('sys/module/ac/sections/.rodata', 0o664, b'0xffffffffa006d480\n')
f('sys/module/ac/sections/.bss', 0o664, b'0xffffffffa006e600\n')
f('sys/module/ac/sections/.text', 0o664, b'0xffffffffa006d000\n')
f('sys/module/ac/sections/.init.text', 0o664, b'0xffffffffa0070000\n')
f('sys/module/ac/sections/.data', 0o664, b'0xffffffffa006e0d0\n')
f('sys/module/ac/sections/.symtab', 0o664, b'0xffffffffa006d668\n')
f('sys/module/ac/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa006e240\n')
f('sys/module/ac/sections/.rodata.str1.1', 0o664, b'0xffffffffa006d5a0\n')
d('sys/module/ac/notes', 0o775)
f('sys/module/ac/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00^h\xf0\x19\xcc\x94"G\xf8\xbc\x1c\xf0\xa3\x8d?\xec\xa5\xc4\x95\x9f')
d('sys/module/dock', 0o775)
d('sys/module/dock/parameters', 0o775)
f('sys/module/dock/parameters/immediate_undock', 0o664, b'Y\n')
d('sys/module/pcie_aspm', 0o775)
d('sys/module/pcie_aspm/parameters', 0o775)
f('sys/module/pcie_aspm/parameters/policy', 0o664, b'[default] performance powersave \n')
d('sys/module/lockdep', 0o775)
d('sys/module/lockdep/parameters', 0o775)
f('sys/module/lockdep/parameters/lock_stat', 0o664, b'1\n')
d('sys/module/tcp_cubic', 0o775)
d('sys/module/tcp_cubic/parameters', 0o775)
f('sys/module/tcp_cubic/parameters/tcp_friendliness', 0o664, b'1\n')
f('sys/module/tcp_cubic/parameters/beta', 0o664, b'717\n')
f('sys/module/tcp_cubic/parameters/bic_scale', 0o664, b'41\n')
f('sys/module/tcp_cubic/parameters/fast_convergence', 0o664, b'1\n')
f('sys/module/tcp_cubic/parameters/initial_ssthresh', 0o664, b'0\n')
d('sys/module/ecb', 0o775)
f('sys/module/ecb/initstate', 0o664, b'live\n')
f('sys/module/ecb/srcversion', 0o664, b'5BBA8E5CDEA4EA8BA5ED31D\n')
f('sys/module/ecb/refcnt', 0o664, b'2\n')
d('sys/module/ecb/sections', 0o775)
f('sys/module/ecb/sections/.exit.text', 0o664, b'0xffffffffa0027300\n')
f('sys/module/ecb/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0027314\n')
f('sys/module/ecb/sections/.strtab', 0o664, b'0xffffffffa00277c0\n')
f('sys/module/ecb/sections/.bss', 0o664, b'0xffffffffa0027e80\n')
f('sys/module/ecb/sections/.text', 0o664, b'0xffffffffa0027000\n')
f('sys/module/ecb/sections/.init.text', 0o664, b'0xffffffffa006b000\n')
f('sys/module/ecb/sections/.data', 0o664, b'0xffffffffa0027a40\n')
f('sys/module/ecb/sections/.symtab', 0o664, b'0xffffffffa0027340\n')
f('sys/module/ecb/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0027ac0\n')
f('sys/module/ecb/sections/.rodata.str1.1', 0o664, b'0xffffffffa0027338\n')
d('sys/module/ecb/notes', 0o775)
f('sys/module/ecb/notes/.note.gnu.build-id', 0o664, b"\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xc20'\x18T\x8bW#\xf5\xe5\xf8JvI\xcb\xc4\x91\x95\nJ")
d('sys/module/snd_page_alloc', 0o775)
f('sys/module/snd_page_alloc/initstate', 0o664, b'live\n')
f('sys/module/snd_page_alloc/srcversion', 0o664, b'D9E59C27F28D5DA2B935016\n')
f('sys/module/snd_page_alloc/refcnt', 0o664, b'2\n')
d('sys/module/snd_page_alloc/holders', 0o775)
l('sys/module/snd_page_alloc/holders/snd_pcm', '../../snd_pcm')
l('sys/module/snd_page_alloc/holders/snd_hda_intel', '../../snd_hda_intel')
d('sys/module/snd_page_alloc/sections', 0o775)
f('sys/module/snd_page_alloc/sections/__ksymtab_strings', 0o664, b'0xffffffffa00350e8\n')
f('sys/module/snd_page_alloc/sections/.exit.text', 0o664, b'0xffffffffa0034cc0\n')
f('sys/module/snd_page_alloc/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0034cf8\n')
f('sys/module/snd_page_alloc/sections/.strtab', 0o664, b'0xffffffffa0035be8\n')
f('sys/module/snd_page_alloc/sections/__ksymtab', 0o664, b'0xffffffffa0035040\n')
f('sys/module/snd_page_alloc/sections/.rodata', 0o664, b'0xffffffffa0034d20\n')
f('sys/module/snd_page_alloc/sections/.bss', 0o664, b'0xffffffffa0036780\n')
f('sys/module/snd_page_alloc/sections/.text', 0o664, b'0xffffffffa0034000\n')
f('sys/module/snd_page_alloc/sections/.init.text', 0o664, b'0xffffffffa0038000\n')
f('sys/module/snd_page_alloc/sections/__kcrctab', 0o664, b'0xffffffffa00350b0\n')
f('sys/module/snd_page_alloc/sections/.data', 0o664, b'0xffffffffa0036300\n')
f('sys/module/snd_page_alloc/sections/.symtab', 0o664, b'0xffffffffa0035180\n')
f('sys/module/snd_page_alloc/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00363c0\n')
f('sys/module/snd_page_alloc/sections/.rodata.str1.1', 0o664, b'0xffffffffa0034fd5\n')
d('sys/module/snd_page_alloc/notes', 0o775)
f('sys/module/snd_page_alloc/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00s\xc1\x8a\x9a\xf7g\x83-x\xc0\xbe\x98\xb0[\x95"\xd7\x97Q=')
d('sys/module/loop', 0o775)
f('sys/module/loop/initstate', 0o664, b'live\n')
f('sys/module/loop/srcversion', 0o664, b'7595F5D46DFDB4A2D489DDD\n')
f('sys/module/loop/refcnt', 0o664, b'0\n')
d('sys/module/loop/sections', 0o775)
f('sys/module/loop/sections/__bug_table', 0o664, b'0xffffffffa023ca1f\n')
f('sys/module/loop/sections/__ksymtab_strings', 0o664, b'0xffffffffa023caf0\n')
f('sys/module/loop/sections/.exit.text', 0o664, b'0xffffffffa023c72c\n')
f('sys/module/loop/sections/.note.gnu.build-id', 0o664, b'0xffffffffa023c7e4\n')
f('sys/module/loop/sections/.strtab', 0o664, b'0xffffffffa023d960\n')
f('sys/module/loop/sections/__ksymtab', 0o664, b'0xffffffffa023ca70\n')
f('sys/module/loop/sections/.rodata', 0o664, b'0xffffffffa023c808\n')
f('sys/module/loop/sections/__param', 0o664, b'0xffffffffa023caa0\n')
f('sys/module/loop/sections/.bss', 0o664, b'0xffffffffa023e780\n')
f('sys/module/loop/sections/.text', 0o664, b'0xffffffffa023a000\n')
f('sys/module/loop/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/loop/sections/__kcrctab', 0o664, b'0xffffffffa023ca90\n')
f('sys/module/loop/sections/.data', 0o664, b'0xffffffffa023e180\n')
f('sys/module/loop/sections/.symtab', 0o664, b'0xffffffffa023cb20\n')
f('sys/module/loop/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa023e3c0\n')
f('sys/module/loop/sections/.rodata.str1.1', 0o664, b'0xffffffffa023c88a\n')
d('sys/module/loop/notes', 0o775)
f('sys/module/loop/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x007;\xf4\xa1\x0e\xe8\x13\xedX\xe6\xbc\xa4\xcd\xde_\x7fe\x84\x90\x85')
d('sys/module/aes_generic', 0o775)
f('sys/module/aes_generic/initstate', 0o664, b'live\n')
f('sys/module/aes_generic/srcversion', 0o664, b'CE7DEF557FD2F72DE36DE4A\n')
f('sys/module/aes_generic/refcnt', 0o664, b'1\n')
d('sys/module/aes_generic/holders', 0o775)
l('sys/module/aes_generic/holders/aes_x86_64', '../../aes_x86_64')
d('sys/module/aes_generic/sections', 0o775)
f('sys/module/aes_generic/sections/__ksymtab_gpl', 0o664, b'0xffffffffa02ccef0\n')
f('sys/module/aes_generic/sections/__ksymtab_strings', 0o664, b'0xffffffffa02ccf80\n')
f('sys/module/aes_generic/sections/.exit.text', 0o664, b'0xffffffffa02cceb8\n')
f('sys/module/aes_generic/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02ccecc\n')
f('sys/module/aes_generic/sections/.strtab', 0o664, b'0xffffffffa02cd678\n')
f('sys/module/aes_generic/sections/.bss', 0o664, b'0xffffffffa02cdf80\n')
f('sys/module/aes_generic/sections/.text', 0o664, b'0xffffffffa02cb000\n')
f('sys/module/aes_generic/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/aes_generic/sections/.data', 0o664, b'0xffffffffa02cda80\n')
f('sys/module/aes_generic/sections/.init.data', 0o664, b'0xffffffffa0065300\n')
f('sys/module/aes_generic/sections/.symtab', 0o664, b'0xffffffffa02ccfe8\n')
f('sys/module/aes_generic/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02cdbc0\n')
f('sys/module/aes_generic/sections/__kcrctab_gpl', 0o664, b'0xffffffffa02ccf50\n')
d('sys/module/aes_generic/notes', 0o775)
f('sys/module/aes_generic/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00V@\xa6&\x10\xf7X\xe0q\xbf\xb9G\xd9d\x004\x06h\xe6\x13')
d('sys/module/snd_seq_device', 0o775)
f('sys/module/snd_seq_device/initstate', 0o664, b'live\n')
f('sys/module/snd_seq_device/srcversion', 0o664, b'A166653189DAF15E3F2B588\n')
f('sys/module/snd_seq_device/refcnt', 0o664, b'1\n')
d('sys/module/snd_seq_device/holders', 0o775)
l('sys/module/snd_seq_device/holders/snd_seq', '../../snd_seq')
d('sys/module/snd_seq_device/sections', 0o775)
f('sys/module/snd_seq_device/sections/__ksymtab_strings', 0o664, b'0xffffffffa02a5d10\n')
f('sys/module/snd_seq_device/sections/.exit.text', 0o664, b'0xffffffffa02a5a5c\n')
f('sys/module/snd_seq_device/sections/.note.gnu.build-id', 0o664, b'0xffffffffa02a5a90\n')
f('sys/module/snd_seq_device/sections/.strtab', 0o664, b'0xffffffffa02a66e0\n')
f('sys/module/snd_seq_device/sections/__ksymtab', 0o664, b'0xffffffffa02a5c80\n')
f('sys/module/snd_seq_device/sections/.bss', 0o664, b'0xffffffffa02a7280\n')
f('sys/module/snd_seq_device/sections/.text', 0o664, b'0xffffffffa02a5000\n')
f('sys/module/snd_seq_device/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/snd_seq_device/sections/__kcrctab', 0o664, b'0xffffffffa02a5ce0\n')
f('sys/module/snd_seq_device/sections/.data', 0o664, b'0xffffffffa02a6de0\n')
f('sys/module/snd_seq_device/sections/.symtab', 0o664, b'0xffffffffa02a5db0\n')
f('sys/module/snd_seq_device/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa02a6ec0\n')
f('sys/module/snd_seq_device/sections/.rodata.str1.1', 0o664, b'0xffffffffa02a5bd0\n')
d('sys/module/snd_seq_device/notes', 0o775)
f('sys/module/snd_seq_device/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xb6B\x93\xc2\xcc\x83\xa8\xfc\xc2r\xf6\x8c\xcej\n\xc5\x87ys\r')
d('sys/module/8250', 0o775)
d('sys/module/8250/parameters', 0o775)
f('sys/module/8250/parameters/share_irqs', 0o664, b'0\n')
f('sys/module/8250/parameters/nr_uarts', 0o664, b'4\n')
d('sys/module/ipv6', 0o775)
f('sys/module/ipv6/initstate', 0o664, b'live\n')
f('sys/module/ipv6/srcversion', 0o664, b'184C844E7A78529619B4C9D\n')
f('sys/module/ipv6/refcnt', 0o664, b'10\n')
d('sys/module/ipv6/sections', 0o775)
f('sys/module/ipv6/sections/__ksymtab_gpl', 0o664, b'0xffffffffa028ac10\n')
f('sys/module/ipv6/sections/.smp_locks', 0o664, b'0xffffffffa028a240\n')
f('sys/module/ipv6/sections/__bug_table', 0o664, b'0xffffffffa0288edd\n')
f('sys/module/ipv6/sections/__ksymtab_strings', 0o664, b'0xffffffffa028b108\n')
f('sys/module/ipv6/sections/.exit.text', 0o664, b'0xffffffffa0285e40\n')
f('sys/module/ipv6/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0286024\n')
f('sys/module/ipv6/sections/.strtab', 0o664, b'0xffffffffa02937f0\n')
f('sys/module/ipv6/sections/__ksymtab', 0o664, b'0xffffffffa028ad90\n')
f('sys/module/ipv6/sections/.rodata', 0o664, b'0xffffffffa0286060\n')
f('sys/module/ipv6/sections/.data.read_mostly', 0o664, b'0xffffffffa029cf20\n')
f('sys/module/ipv6/sections/.bss', 0o664, b'0xffffffffa029dd80\n')
f('sys/module/ipv6/sections/.text', 0o664, b'0xffffffffa0258000\n')
f('sys/module/ipv6/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/ipv6/sections/__kcrctab', 0o664, b'0xffffffffa028afe0\n')
f('sys/module/ipv6/sections/.data', 0o664, b'0xffffffffa02998c0\n')
f('sys/module/ipv6/sections/.init.data', 0o664, b'0xffffffffa0065d80\n')
f('sys/module/ipv6/sections/.symtab', 0o664, b'0xffffffffa028b4c8\n')
f('sys/module/ipv6/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa029d9c0\n')
f('sys/module/ipv6/sections/.rodata.str1.1', 0o664, b'0xffffffffa02883c0\n')
f('sys/module/ipv6/sections/.ref.text', 0o664, b'0xffffffffa0285f40\n')
f('sys/module/ipv6/sections/__kcrctab_gpl', 0o664, b'0xffffffffa028ad10\n')
d('sys/module/ipv6/notes', 0o775)
f('sys/module/ipv6/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x90\x01\xde\xf3>\xfd_r\xbc\xbb\x16\xbd\xceIf\xc2vC\xbd\xc9')
d('sys/module/microcode', 0o775)
f('sys/module/microcode/initstate', 0o664, b'live\n')
f('sys/module/microcode/srcversion', 0o664, b'FC3DA13E858B639D12F2043\n')
f('sys/module/microcode/refcnt', 0o664, b'0\n')
d('sys/module/microcode/sections', 0o775)
f('sys/module/microcode/sections/__bug_table', 0o664, b'0xffffffffa02516f7\n')
f('sys/module/microcode/sections/.exit.text', 0o664, b'0xffffffffa0250ff6\n')
f('sys/module/microcode/sections/.note.gnu.build-id', 0o664, b'0xffffffffa0251040\n')
f('sys/module/microcode/sections/.strtab', 0o664, b'0xffffffffa02520b8\n')
f('sys/module/microcode/sections/.rodata', 0o664, b'0xffffffffa0251080\n')
f('sys/module/microcode/sections/.bss', 0o664, b'0xffffffffa0252c40\n')
f('sys/module/microcode/sections/.text', 0o664, b'0xffffffffa0250000\n')
f('sys/module/microcode/sections/.init.text', 0o664, b'0xffffffffa0065000\n')
f('sys/module/microcode/sections/.ref.data', 0o664, b'0xffffffffa0252850\n')
f('sys/module/microcode/sections/.data', 0o664, b'0xffffffffa0252600\n')
f('sys/module/microcode/sections/.symtab', 0o664, b'0xffffffffa0251740\n')
f('sys/module/microcode/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa0252880\n')
f('sys/module/microcode/sections/.rodata.str1.1', 0o664, b'0xffffffffa0251218\n')
f('sys/module/microcode/sections/.cpuinit.text', 0o664, b'0xffffffffa0250d90\n')
d('sys/module/microcode/notes', 0o775)
f('sys/module/microcode/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x81 p\x0b\x99+\x14\x05\x9c\xb2\xf6{\xacs\xa9\xf3<\xb5\x99\x1a')
d('sys/module/i8042', 0o775)
d('sys/module/i8042/parameters', 0o775)
f('sys/module/i8042/parameters/debug', 0o664, b'N\n')
f('sys/module/i8042/parameters/panicblink', 0o664, b'500\n')
d('sys/module/usbcore', 0o775)
f('sys/module/usbcore/initstate', 0o664, b'live\n')
f('sys/module/usbcore/srcversion', 0o664, b'4FF77EB8DAB25F4076D7357\n')
f('sys/module/usbcore/refcnt', 0o664, b'5\n')
d('sys/module/usbcore/holders', 0o775)
l('sys/module/usbcore/holders/uhci_hcd', '../../uhci_hcd')
l('sys/module/usbcore/holders/usb_storage', '../../usb_storage')
l('sys/module/usbcore/holders/cdc_acm', '../../cdc_acm')
l('sys/module/usbcore/holders/usbhid', '../../usbhid')
l('sys/module/usbcore/holders/ehci_hcd', '../../ehci_hcd')
d('sys/module/usbcore/drivers', 0o775)
l('sys/module/usbcore/drivers/usb:usbfs', '../../../bus/usb/drivers/usbfs')
l('sys/module/usbcore/drivers/usb:usb', '../../../bus/usb/drivers/usb')
l('sys/module/usbcore/drivers/usb:hub', '../../../bus/usb/drivers/hub')
d('sys/module/usbcore/parameters', 0o775)
f('sys/module/usbcore/parameters/use_both_schemes', 0o664, b'Y\n')
f('sys/module/usbcore/parameters/blinkenlights', 0o664, b'N\n')
f('sys/module/usbcore/parameters/usbfs_snoop', 0o664, b'N\n')
f('sys/module/usbcore/parameters/autosuspend', 0o664, b'2\n')
f('sys/module/usbcore/parameters/nousb', 0o664, b'N\n')
f('sys/module/usbcore/parameters/old_scheme_first', 0o664, b'N\n')
d('sys/module/usbcore/sections', 0o775)
f('sys/module/usbcore/sections/__ksymtab_gpl', 0o664, b'0xffffffffa004ea30\n')
f('sys/module/usbcore/sections/.smp_locks', 0o664, b'0xffffffffa004f930\n')
f('sys/module/usbcore/sections/__ex_table', 0o664, b'0xffffffffa004fa58\n')
f('sys/module/usbcore/sections/__bug_table', 0o664, b'0xffffffffa004fa40\n')
f('sys/module/usbcore/sections/__ksymtab_strings', 0o664, b'0xffffffffa004f300\n')
f('sys/module/usbcore/sections/.exit.text', 0o664, b'0xffffffffa004ba30\n')
f('sys/module/usbcore/sections/.note.gnu.build-id', 0o664, b'0xffffffffa004bb30\n')
f('sys/module/usbcore/sections/.strtab', 0o664, b'0xffffffffa0055cf8\n')
f('sys/module/usbcore/sections/.fixup', 0o664, b'0xffffffffa004ba87\n')
f('sys/module/usbcore/sections/.rodata', 0o664, b'0xffffffffa004bb60\n')
f('sys/module/usbcore/sections/__param', 0o664, b'0xffffffffa004f210\n')
f('sys/module/usbcore/sections/.bss', 0o664, b'0xffffffffa005cec0\n')
f('sys/module/usbcore/sections/.text', 0o664, b'0xffffffffa0038000\n')
f('sys/module/usbcore/sections/.init.text', 0o664, b'0xffffffffa0027000\n')
f('sys/module/usbcore/sections/.data', 0o664, b'0xffffffffa005af00\n')
f('sys/module/usbcore/sections/.symtab', 0o664, b'0xffffffffa004fb18\n')
f('sys/module/usbcore/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa005cb00\n')
f('sys/module/usbcore/sections/.rodata.str1.1', 0o664, b'0xffffffffa004c044\n')
f('sys/module/usbcore/sections/__kcrctab_gpl', 0o664, b'0xffffffffa004ef70\n')
d('sys/module/usbcore/notes', 0o775)
f('sys/module/usbcore/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00!\xbeP\xa2@m\xffg\xff\x1b\xa8\x8f\xd2\xb3\x11\x14[\xb0\x89\x1e')
d('sys/module/intel_agp', 0o775)
f('sys/module/intel_agp/initstate', 0o664, b'live\n')
f('sys/module/intel_agp/srcversion', 0o664, b'98AFACFFF6A1ADC3A6FAB67\n')
f('sys/module/intel_agp/refcnt', 0o664, b'0\n')
d('sys/module/intel_agp/drivers', 0o775)
l('sys/module/intel_agp/drivers/pci:agpgart-intel', '../../../bus/pci/drivers/agpgart-intel')
d('sys/module/intel_agp/sections', 0o775)
f('sys/module/intel_agp/sections/.smp_locks', 0o664, b'0xffffffffa008ae38\n')
f('sys/module/intel_agp/sections/.exit.text', 0o664, b'0xffffffffa00893ac\n')
f('sys/module/intel_agp/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00896b0\n')
f('sys/module/intel_agp/sections/.strtab', 0o664, b'0xffffffffa008c1b8\n')
f('sys/module/intel_agp/sections/.devexit.text', 0o664, b'0xffffffffa00893be\n')
f('sys/module/intel_agp/sections/.rodata', 0o664, b'0xffffffffa00896e0\n')
f('sys/module/intel_agp/sections/.bss', 0o664, b'0xffffffffa008da00\n')
f('sys/module/intel_agp/sections/.text', 0o664, b'0xffffffffa0086000\n')
f('sys/module/intel_agp/sections/.init.text', 0o664, b'0xffffffffa008f000\n')
f('sys/module/intel_agp/sections/.data', 0o664, b'0xffffffffa008cf60\n')
f('sys/module/intel_agp/sections/.devinit.text', 0o664, b'0xffffffffa0089400\n')
f('sys/module/intel_agp/sections/.symtab', 0o664, b'0xffffffffa008ae50\n')
f('sys/module/intel_agp/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa008d640\n')
f('sys/module/intel_agp/sections/.rodata.str1.1', 0o664, b'0xffffffffa008a9c0\n')
d('sys/module/intel_agp/notes', 0o775)
f('sys/module/intel_agp/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\xb0\xb5IU\x04F\xdb\x80\xdf\x9a8\xedq\x0b\xd4\xc7K\x851\x86')
d('sys/module/evdev', 0o775)
f('sys/module/evdev/initstate', 0o664, b'live\n')
f('sys/module/evdev/srcversion', 0o664, b'6F6934C1F95317926EA0520\n')
f('sys/module/evdev/refcnt', 0o664, b'5\n')
d('sys/module/evdev/sections', 0o775)
f('sys/module/evdev/sections/.exit.text', 0o664, b'0xffffffffa00b3724\n')
f('sys/module/evdev/sections/.note.gnu.build-id', 0o664, b'0xffffffffa00b3738\n')
f('sys/module/evdev/sections/.strtab', 0o664, b'0xffffffffa00b4610\n')
f('sys/module/evdev/sections/.rodata', 0o664, b'0xffffffffa00b3760\n')
f('sys/module/evdev/sections/.bss', 0o664, b'0xffffffffa00b50c0\n')
f('sys/module/evdev/sections/.text', 0o664, b'0xffffffffa00b2000\n')
f('sys/module/evdev/sections/.init.text', 0o664, b'0xffffffffa00b7000\n')
f('sys/module/evdev/sections/.data', 0o664, b'0xffffffffa00b4be0\n')
f('sys/module/evdev/sections/.symtab', 0o664, b'0xffffffffa00b3b60\n')
f('sys/module/evdev/sections/.gnu.linkonce.this_module', 0o664, b'0xffffffffa00b4d00\n')
f('sys/module/evdev/sections/.rodata.str1.1', 0o664, b'0xffffffffa00b3a38\n')
d('sys/module/evdev/notes', 0o775)
f('sys/module/evdev/notes/.note.gnu.build-id', 0o664, b'\x04\x00\x00\x00\x14\x00\x00\x00\x03\x00\x00\x00GNU\x00\x8d/u!\xb1\xf2\xe7\x10z\x8a\xf7\x0c\x88\x17\x1c4i\\2\xa8')
d('sys/module/atkbd', 0o775)
d('sys/module/atkbd/drivers', 0o775)
l('sys/module/atkbd/drivers/serio:atkbd', '../../../bus/serio/drivers/atkbd')
d('sys/class', 0o755)
d('sys/class/rfkill', 0o755)
l('sys/class/rfkill/rfkill0', '../../devices/platform/thinkpad_acpi/rfkill/rfkill0')
d('sys/class/rtc', 0o755)
l('sys/class/rtc/rtc0', '../../devices/pnp0/00:07/rtc/rtc0')
d('sys/class/usb_endpoint', 0o755)
l('sys/class/usb_endpoint/usbdev1.1_ep00', '../../devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00')
l('sys/class/usb_endpoint/usbdev4.3_ep83', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83')
l('sys/class/usb_endpoint/usbdev4.2_ep04', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04')
l('sys/class/usb_endpoint/usbdev1.1_ep81', '../../devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81')
l('sys/class/usb_endpoint/usbdev3.1_ep81', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81')
l('sys/class/usb_endpoint/usbdev4.1_ep00', '../../devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00')
l('sys/class/usb_endpoint/usbdev3.4_ep00', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00')
l('sys/class/usb_endpoint/usbdev4.2_ep00', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00')
l('sys/class/usb_endpoint/usbdev5.7_ep81', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81')
l('sys/class/usb_endpoint/usbdev5.9_ep81', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81')
l('sys/class/usb_endpoint/usbdev5.9_ep83', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83')
l('sys/class/usb_endpoint/usbdev5.9_ep84', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84')
l('sys/class/usb_endpoint/usbdev5.9_ep00', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00')
l('sys/class/usb_endpoint/usbdev4.2_ep03', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03')
l('sys/class/usb_endpoint/usbdev4.2_ep84', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84')
l('sys/class/usb_endpoint/usbdev3.1_ep00', '../../devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00')
l('sys/class/usb_endpoint/usbdev4.2_ep83', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83')
l('sys/class/usb_endpoint/usbdev4.2_ep02', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02')
l('sys/class/usb_endpoint/usbdev4.3_ep00', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00')
l('sys/class/usb_endpoint/usbdev5.7_ep00', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00')
l('sys/class/usb_endpoint/usbdev4.1_ep81', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81')
l('sys/class/usb_endpoint/usbdev5.9_ep02', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02')
l('sys/class/usb_endpoint/usbdev4.2_ep82', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82')
l('sys/class/usb_endpoint/usbdev2.1_ep81', '../../devices/pci0000:00/0000:00:1d.1/usb2/2-0:1.0/usb_endpoint/usbdev2.1_ep81')
l('sys/class/usb_endpoint/usbdev4.3_ep81', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81')
l('sys/class/usb_endpoint/usbdev2.1_ep00', '../../devices/pci0000:00/0000:00:1d.1/usb2/usb_endpoint/usbdev2.1_ep00')
l('sys/class/usb_endpoint/usbdev5.1_ep81', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81')
l('sys/class/usb_endpoint/usbdev4.2_ep81', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81')
l('sys/class/usb_endpoint/usbdev3.4_ep81', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81')
l('sys/class/usb_endpoint/usbdev4.3_ep02', '../../devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02')
l('sys/class/usb_endpoint/usbdev5.7_ep02', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02')
l('sys/class/usb_endpoint/usbdev5.1_ep00', '../../devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00')
l('sys/class/usb_endpoint/usbdev5.9_ep01', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01')
l('sys/class/usb_endpoint/usbdev5.9_ep82', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82')
d('sys/class/ieee80211', 0o755)
l('sys/class/ieee80211/phy0', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0')
d('sys/class/scsi_disk', 0o755)
l('sys/class/scsi_disk/7:0:0:0', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0')
l('sys/class/scsi_disk/0:0:0:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0')
d('sys/class/vc', 0o755)
l('sys/class/vc/vcsa3', '../../devices/virtual/vc/vcsa3')
l('sys/class/vc/vcsa6', '../../devices/virtual/vc/vcsa6')
l('sys/class/vc/vcsa2', '../../devices/virtual/vc/vcsa2')
l('sys/class/vc/vcsa', '../../devices/virtual/vc/vcsa')
l('sys/class/vc/vcs2', '../../devices/virtual/vc/vcs2')
l('sys/class/vc/vcs7', '../../devices/virtual/vc/vcs7')
l('sys/class/vc/vcsa5', '../../devices/virtual/vc/vcsa5')
l('sys/class/vc/vcsa10', '../../devices/virtual/vc/vcsa10')
l('sys/class/vc/vcs10', '../../devices/virtual/vc/vcs10')
l('sys/class/vc/vcsa1', '../../devices/virtual/vc/vcsa1')
l('sys/class/vc/vcsa4', '../../devices/virtual/vc/vcsa4')
l('sys/class/vc/vcs', '../../devices/virtual/vc/vcs')
l('sys/class/vc/vcs1', '../../devices/virtual/vc/vcs1')
l('sys/class/vc/vcs4', '../../devices/virtual/vc/vcs4')
l('sys/class/vc/vcsa7', '../../devices/virtual/vc/vcsa7')
l('sys/class/vc/vcs6', '../../devices/virtual/vc/vcs6')
l('sys/class/vc/vcs3', '../../devices/virtual/vc/vcs3')
l('sys/class/vc/vcs5', '../../devices/virtual/vc/vcs5')
d('sys/class/mem', 0o755)
l('sys/class/mem/zero', '../../devices/virtual/mem/zero')
l('sys/class/mem/kmsg', '../../devices/virtual/mem/kmsg')
l('sys/class/mem/mem', '../../devices/virtual/mem/mem')
l('sys/class/mem/urandom', '../../devices/virtual/mem/urandom')
l('sys/class/mem/port', '../../devices/virtual/mem/port')
l('sys/class/mem/null', '../../devices/virtual/mem/null')
l('sys/class/mem/kmem', '../../devices/virtual/mem/kmem')
l('sys/class/mem/full', '../../devices/virtual/mem/full')
l('sys/class/mem/random', '../../devices/virtual/mem/random')
d('sys/class/pci_bus', 0o755)
l('sys/class/pci_bus/0000:00', '../../devices/pci0000:00/pci_bus/0000:00')
l('sys/class/pci_bus/0000:01', '../../devices/pci0000:00/0000:00:01.0/pci_bus/0000:01')
l('sys/class/pci_bus/0000:03', '../../devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03')
l('sys/class/pci_bus/0000:0c', '../../devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c')
l('sys/class/pci_bus/0000:16', '../../devices/pci0000:00/0000:00:1e.0/0000:15:00.0/pci_bus/0000:16')
l('sys/class/pci_bus/0000:04', '../../devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04')
l('sys/class/pci_bus/0000:02', '../../devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02')
l('sys/class/pci_bus/0000:15', '../../devices/pci0000:00/0000:00:1e.0/pci_bus/0000:15')
d('sys/class/hwmon', 0o755)
l('sys/class/hwmon/hwmon0', '../../devices/platform/thinkpad_hwmon/hwmon/hwmon0')
d('sys/class/net', 0o755)
l('sys/class/net/wmaster0', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0')
l('sys/class/net/wlan0', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0')
l('sys/class/net/lo', '../../devices/virtual/net/lo')
l('sys/class/net/eth0', '../../devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0')
d('sys/class/dmi', 0o755)
l('sys/class/dmi/id', '../../devices/virtual/dmi/id')
d('sys/class/firmware', 0o755)
f('sys/class/firmware/timeout', 0o644, b'60\n')
d('sys/class/scsi_generic', 0o755)
l('sys/class/scsi_generic/sg0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0')
l('sys/class/scsi_generic/sg1', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1')
l('sys/class/scsi_generic/sg2', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2')
d('sys/class/bdi', 0o755)
l('sys/class/bdi/7:1', '../../devices/virtual/bdi/7:1')
l('sys/class/bdi/7:0', '../../devices/virtual/bdi/7:0')
l('sys/class/bdi/7:2', '../../devices/virtual/bdi/7:2')
l('sys/class/bdi/7:4', '../../devices/virtual/bdi/7:4')
l('sys/class/bdi/7:3', '../../devices/virtual/bdi/7:3')
l('sys/class/bdi/11:0', '../../devices/virtual/bdi/11:0')
l('sys/class/bdi/7:5', '../../devices/virtual/bdi/7:5')
l('sys/class/bdi/8:0', '../../devices/virtual/bdi/8:0')
l('sys/class/bdi/default', '../../devices/virtual/bdi/default')
l('sys/class/bdi/0:16', '../../devices/virtual/bdi/0:16')
l('sys/class/bdi/8:16', '../../devices/virtual/bdi/8:16')
l('sys/class/bdi/9:0', '../../devices/virtual/bdi/9:0')
l('sys/class/bdi/7:7', '../../devices/virtual/bdi/7:7')
l('sys/class/bdi/7:6', '../../devices/virtual/bdi/7:6')
d('sys/class/leds', 0o755)
l('sys/class/leds/iwl-phy0:TX', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX')
l('sys/class/leds/tpacpi::bay_active', '../../devices/platform/thinkpad_acpi/leds/tpacpi::bay_active')
l('sys/class/leds/tpacpi:orange:batt', '../../devices/platform/thinkpad_acpi/leds/tpacpi:orange:batt')
l('sys/class/leds/tpacpi::dock_active', '../../devices/platform/thinkpad_acpi/leds/tpacpi::dock_active')
l('sys/class/leds/tpacpi::unknown_led', '../../devices/platform/thinkpad_acpi/leds/tpacpi::unknown_led')
l('sys/class/leds/iwl-phy0:assoc', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc')
l('sys/class/leds/tpacpi:green:batt', '../../devices/platform/thinkpad_acpi/leds/tpacpi:green:batt')
l('sys/class/leds/iwl-phy0:radio', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio')
l('sys/class/leds/iwl-phy0:RX', '../../devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX')
l('sys/class/leds/tpacpi::dock_batt', '../../devices/platform/thinkpad_acpi/leds/tpacpi::dock_batt')
l('sys/class/leds/tpacpi::thinklight', '../../devices/platform/thinkpad_acpi/leds/tpacpi::thinklight')
l('sys/class/leds/tpacpi::power', '../../devices/platform/thinkpad_acpi/leds/tpacpi::power')
l('sys/class/leds/tpacpi::standby', '../../devices/platform/thinkpad_acpi/leds/tpacpi::standby')
d('sys/class/usb_host', 0o755)
l('sys/class/usb_host/usb_host1', '../../devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1')
l('sys/class/usb_host/usb_host2', '../../devices/pci0000:00/0000:00:1d.1/usb_host/usb_host2')
l('sys/class/usb_host/usb_host4', '../../devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4')
l('sys/class/usb_host/usb_host5', '../../devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5')
l('sys/class/usb_host/usb_host3', '../../devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3')
d('sys/class/pcmcia_socket', 0o755)
l('sys/class/pcmcia_socket/pcmcia_socket0', '../../devices/pci0000:00/0000:00:1e.0/0000:15:00.0/pcmcia_socket/pcmcia_socket0')
d('sys/class/sound', 0o755)
l('sys/class/sound/audio', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/audio')
l('sys/class/sound/dsp', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/dsp')
l('sys/class/sound/timer', '../../devices/virtual/sound/timer')
l('sys/class/sound/card0', '../../devices/pci0000:00/0000:00:1b.0/sound/card0')
l('sys/class/sound/seq', '../../devices/virtual/sound/seq')
l('sys/class/sound/mixer', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/mixer')
l('sys/class/sound/pcmC0D0p', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p')
l('sys/class/sound/controlC0', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0')
l('sys/class/sound/pcmC0D0c', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c')
l('sys/class/sound/adsp', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/adsp')
l('sys/class/sound/pcmC0D1p', '../../devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p')
d('sys/class/misc', 0o755)
l('sys/class/misc/uinput', '../../devices/virtual/misc/uinput')
l('sys/class/misc/fuse', '../../devices/virtual/misc/fuse')
l('sys/class/misc/hpet', '../../devices/virtual/misc/hpet')
l('sys/class/misc/cpu_dma_latency', '../../devices/virtual/misc/cpu_dma_latency')
l('sys/class/misc/mcelog', '../../devices/virtual/misc/mcelog')
l('sys/class/misc/device-mapper', '../../devices/virtual/misc/device-mapper')
l('sys/class/misc/psaux', '../../devices/virtual/misc/psaux')
l('sys/class/misc/nvram', '../../devices/virtual/misc/nvram')
l('sys/class/misc/network_latency', '../../devices/virtual/misc/network_latency')
l('sys/class/misc/network_throughput', '../../devices/virtual/misc/network_throughput')
l('sys/class/misc/microcode', '../../devices/virtual/misc/microcode')
d('sys/class/vtconsole', 0o755)
l('sys/class/vtconsole/vtcon1', '../../devices/virtual/vtconsole/vtcon1')
l('sys/class/vtconsole/vtcon0', '../../devices/virtual/vtconsole/vtcon0')
d('sys/class/bsg', 0o755)
l('sys/class/bsg/7:0:0:0', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0')
l('sys/class/bsg/0:0:0:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0')
l('sys/class/bsg/4:0:0:0', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0')
d('sys/class/input', 0o755)
l('sys/class/input/event0', '../../devices/platform/i8042/serio0/input/input0/event0')
l('sys/class/input/input0', '../../devices/platform/i8042/serio0/input/input0')
l('sys/class/input/event4', '../../devices/LNXSYSTM:00/device:00/PNP0C0D:00/input/input4/event4')
l('sys/class/input/mouse0', '../../devices/platform/i8042/serio1/input/input1/mouse0')
l('sys/class/input/mouse1', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1')
l('sys/class/input/event2', '../../devices/platform/pcspkr/input/input2/event2')
l('sys/class/input/input2', '../../devices/platform/pcspkr/input/input2')
l('sys/class/input/input5', '../../devices/LNXSYSTM:00/device:00/PNP0C0E:00/input/input5')
l('sys/class/input/input8', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8')
l('sys/class/input/input3', '../../devices/LNXSYSTM:00/LNXPWRBN:00/input/input3')
l('sys/class/input/input1', '../../devices/platform/i8042/serio1/input/input1')
l('sys/class/input/mice', '../../devices/virtual/input/mice')
l('sys/class/input/event6', '../../devices/virtual/input/input6/event6')
l('sys/class/input/input6', '../../devices/virtual/input/input6')
l('sys/class/input/event3', '../../devices/LNXSYSTM:00/LNXPWRBN:00/input/input3/event3')
l('sys/class/input/event1', '../../devices/platform/i8042/serio1/input/input1/event1')
l('sys/class/input/input4', '../../devices/LNXSYSTM:00/device:00/PNP0C0D:00/input/input4')
l('sys/class/input/event7', '../../devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7')
l('sys/class/input/event5', '../../devices/LNXSYSTM:00/device:00/PNP0C0E:00/input/input5/event5')
d('sys/class/thermal', 0o755)
l('sys/class/thermal/thermal_zone1', '../../devices/virtual/thermal/thermal_zone1')
l('sys/class/thermal/cooling_device1', '../../devices/virtual/thermal/cooling_device1')
l('sys/class/thermal/cooling_device0', '../../devices/virtual/thermal/cooling_device0')
l('sys/class/thermal/thermal_zone0', '../../devices/virtual/thermal/thermal_zone0')
d('sys/class/power_supply', 0o755)
l('sys/class/power_supply/AC', '../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/ACPI0003:00/power_supply/AC')
l('sys/class/power_supply/BAT0', '../../devices/LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C09:00/PNP0C0A:00/power_supply/BAT0')
d('sys/class/block', 0o755)
l('sys/class/block/loop1', '../../devices/virtual/block/loop1')
l('sys/class/block/loop7', '../../devices/virtual/block/loop7')
l('sys/class/block/loop0', '../../devices/virtual/block/loop0')
l('sys/class/block/sda10', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10')
l('sys/class/block/sr0', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0')
l('sys/class/block/loop5', '../../devices/virtual/block/loop5')
l('sys/class/block/loop3', '../../devices/virtual/block/loop3')
l('sys/class/block/sda9', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9')
l('sys/class/block/sda7', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7')
l('sys/class/block/loop6', '../../devices/virtual/block/loop6')
l('sys/class/block/sdb', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb')
l('sys/class/block/sda1', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1')
l('sys/class/block/loop2', '../../devices/virtual/block/loop2')
l('sys/class/block/sda8', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8')
l('sys/class/block/sda5', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5')
l('sys/class/block/loop4', '../../devices/virtual/block/loop4')
l('sys/class/block/sda6', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6')
l('sys/class/block/sda', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda')
l('sys/class/block/md0', '../../devices/virtual/block/md0')
l('sys/class/block/sdb1', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1')
d('sys/class/scsi_host', 0o755)
l('sys/class/scsi_host/host0', '../../devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0')
l('sys/class/scsi_host/host5', '../../devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5')
l('sys/class/scsi_host/host1', '../../devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1')
l('sys/class/scsi_host/host4', '../../devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4')
l('sys/class/scsi_host/host2', '../../devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2')
l('sys/class/scsi_host/host7', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7')
l('sys/class/scsi_host/host3', '../../devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3')
d('sys/class/graphics', 0o755)
l('sys/class/graphics/fbcon', '../../devices/virtual/graphics/fbcon')
l('sys/class/graphics/fb0', '../../devices/platform/vesafb.0/graphics/fb0')
d('sys/class/tty', 0o755)
l('sys/class/tty/tty51', '../../devices/virtual/tty/tty51')
l('sys/class/tty/tty24', '../../devices/virtual/tty/tty24')
l('sys/class/tty/tty61', '../../devices/virtual/tty/tty61')
l('sys/class/tty/ttyS1', '../../devices/platform/serial8250/tty/ttyS1')
l('sys/class/tty/tty40', '../../devices/virtual/tty/tty40')
l('sys/class/tty/tty60', '../../devices/virtual/tty/tty60')
l('sys/class/tty/tty15', '../../devices/virtual/tty/tty15')
l('sys/class/tty/tty27', '../../devices/virtual/tty/tty27')
l('sys/class/tty/tty7', '../../devices/virtual/tty/tty7')
l('sys/class/tty/tty43', '../../devices/virtual/tty/tty43')
l('sys/class/tty/tty23', '../../devices/virtual/tty/tty23')
l('sys/class/tty/tty14', '../../devices/virtual/tty/tty14')
l('sys/class/tty/tty56', '../../devices/virtual/tty/tty56')
l('sys/class/tty/tty3', '../../devices/virtual/tty/tty3')
l('sys/class/tty/ttyS3', '../../devices/platform/serial8250/tty/ttyS3')
l('sys/class/tty/tty49', '../../devices/virtual/tty/tty49')
l('sys/class/tty/tty47', '../../devices/virtual/tty/tty47')
l('sys/class/tty/tty57', '../../devices/virtual/tty/tty57')
l('sys/class/tty/tty30', '../../devices/virtual/tty/tty30')
l('sys/class/tty/tty45', '../../devices/virtual/tty/tty45')
l('sys/class/tty/tty22', '../../devices/virtual/tty/tty22')
l('sys/class/tty/tty55', '../../devices/virtual/tty/tty55')
l('sys/class/tty/tty11', '../../devices/virtual/tty/tty11')
l('sys/class/tty/tty53', '../../devices/virtual/tty/tty53')
l('sys/class/tty/tty12', '../../devices/virtual/tty/tty12')
l('sys/class/tty/tty28', '../../devices/virtual/tty/tty28')
l('sys/class/tty/tty58', '../../devices/virtual/tty/tty58')
l('sys/class/tty/tty32', '../../devices/virtual/tty/tty32')
l('sys/class/tty/tty44', '../../devices/virtual/tty/tty44')
l('sys/class/tty/tty6', '../../devices/virtual/tty/tty6')
l('sys/class/tty/tty9', '../../devices/virtual/tty/tty9')
l('sys/class/tty/tty26', '../../devices/virtual/tty/tty26')
l('sys/class/tty/tty10', '../../devices/virtual/tty/tty10')
l('sys/class/tty/tty39', '../../devices/virtual/tty/tty39')
l('sys/class/tty/tty0', '../../devices/virtual/tty/tty0')
l('sys/class/tty/tty36', '../../devices/virtual/tty/tty36')
l('sys/class/tty/tty34', '../../devices/virtual/tty/tty34')
l('sys/class/tty/tty16', '../../devices/virtual/tty/tty16')
l('sys/class/tty/tty25', '../../devices/virtual/tty/tty25')
l('sys/class/tty/tty13', '../../devices/virtual/tty/tty13')
l('sys/class/tty/tty52', '../../devices/virtual/tty/tty52')
l('sys/class/tty/ttyS0', '../../devices/platform/serial8250/tty/ttyS0')
l('sys/class/tty/tty31', '../../devices/virtual/tty/tty31')
l('sys/class/tty/tty35', '../../devices/virtual/tty/tty35')
l('sys/class/tty/tty54', '../../devices/virtual/tty/tty54')
l('sys/class/tty/tty63', '../../devices/virtual/tty/tty63')
l('sys/class/tty/tty1', '../../devices/virtual/tty/tty1')
l('sys/class/tty/tty33', '../../devices/virtual/tty/tty33')
l('sys/class/tty/tty18', '../../devices/virtual/tty/tty18')
l('sys/class/tty/ttyACM0', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0')
l('sys/class/tty/tty42', '../../devices/virtual/tty/tty42')
l('sys/class/tty/tty8', '../../devices/virtual/tty/tty8')
l('sys/class/tty/tty37', '../../devices/virtual/tty/tty37')
l('sys/class/tty/tty5', '../../devices/virtual/tty/tty5')
l('sys/class/tty/tty48', '../../devices/virtual/tty/tty48')
l('sys/class/tty/tty17', '../../devices/virtual/tty/tty17')
l('sys/class/tty/tty62', '../../devices/virtual/tty/tty62')
l('sys/class/tty/tty59', '../../devices/virtual/tty/tty59')
l('sys/class/tty/tty50', '../../devices/virtual/tty/tty50')
l('sys/class/tty/tty21', '../../devices/virtual/tty/tty21')
l('sys/class/tty/ttyS2', '../../devices/platform/serial8250/tty/ttyS2')
l('sys/class/tty/tty19', '../../devices/virtual/tty/tty19')
l('sys/class/tty/ptmx', '../../devices/virtual/tty/ptmx')
l('sys/class/tty/tty46', '../../devices/virtual/tty/tty46')
l('sys/class/tty/tty2', '../../devices/virtual/tty/tty2')
l('sys/class/tty/console', '../../devices/virtual/tty/console')
l('sys/class/tty/tty41', '../../devices/virtual/tty/tty41')
l('sys/class/tty/tty4', '../../devices/virtual/tty/tty4')
l('sys/class/tty/tty29', '../../devices/virtual/tty/tty29')
l('sys/class/tty/tty', '../../devices/virtual/tty/tty')
l('sys/class/tty/tty38', '../../devices/virtual/tty/tty38')
l('sys/class/tty/tty20', '../../devices/virtual/tty/tty20')
d('sys/class/scsi_device', 0o755)
l('sys/class/scsi_device/7:0:0:0', '../../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0')
l('sys/class/scsi_device/0:0:0:0', '../../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0')
l('sys/class/scsi_device/4:0:0:0', '../../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0')
d('sys/block', 0o755)
l('sys/block/loop1', '../devices/virtual/block/loop1')
l('sys/block/loop7', '../devices/virtual/block/loop7')
l('sys/block/loop0', '../devices/virtual/block/loop0')
l('sys/block/sr0', '../devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0')
l('sys/block/loop5', '../devices/virtual/block/loop5')
l('sys/block/loop3', '../devices/virtual/block/loop3')
l('sys/block/loop6', '../devices/virtual/block/loop6')
l('sys/block/sdb', '../devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb')
l('sys/block/loop2', '../devices/virtual/block/loop2')
l('sys/block/loop4', '../devices/virtual/block/loop4')
l('sys/block/sda', '../devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda')
l('sys/block/md0', '../devices/virtual/block/md0')
d('sys/devices', 0o755)
d('sys/devices/system', 0o755)
d('sys/devices/system/machinecheck', 0o755)
d('sys/devices/system/machinecheck/machinecheck1', 0o755)
f('sys/devices/system/machinecheck/machinecheck1/trigger', 0o644, b'\n')
f('sys/devices/system/machinecheck/machinecheck1/bank0ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck1/check_interval', 0o644, b'12c\n')
f('sys/devices/system/machinecheck/machinecheck1/tolerant', 0o644, b'1\n')
f('sys/devices/system/machinecheck/machinecheck1/bank1ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck1/bank5ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck1/bank3ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck1/bank2ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck1/bank4ctl', 0o644, b'ffffffffffffffff\n')
d('sys/devices/system/machinecheck/machinecheck0', 0o755)
f('sys/devices/system/machinecheck/machinecheck0/trigger', 0o644, b'\n')
f('sys/devices/system/machinecheck/machinecheck0/bank0ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck0/check_interval', 0o644, b'12c\n')
f('sys/devices/system/machinecheck/machinecheck0/tolerant', 0o644, b'1\n')
f('sys/devices/system/machinecheck/machinecheck0/bank1ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck0/bank5ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck0/bank3ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck0/bank2ctl', 0o644, b'ffffffffffffffff\n')
f('sys/devices/system/machinecheck/machinecheck0/bank4ctl', 0o644, b'ffffffffffffffff\n')
d('sys/devices/system/clocksource', 0o755)
d('sys/devices/system/clocksource/clocksource0', 0o755)
f('sys/devices/system/clocksource/clocksource0/available_clocksource', 0o644, b'hpet acpi_pm jiffies tsc \n')
f('sys/devices/system/clocksource/clocksource0/current_clocksource', 0o644, b'hpet\n')
d('sys/devices/system/cpu', 0o755)
f('sys/devices/system/cpu/possible', 0o644, b'0-1\n')
f('sys/devices/system/cpu/online', 0o644, b'0-1\n')
f('sys/devices/system/cpu/present', 0o644, b'0-1\n')
f('sys/devices/system/cpu/sched_mc_power_savings', 0o644, b'1\n')
d('sys/devices/system/cpu/cpu0', 0o755)
f('sys/devices/system/cpu/cpu0/crash_notes', 0o644, b'7fb20920\n')
d('sys/devices/system/cpu/cpu0/topology', 0o755)
f('sys/devices/system/cpu/cpu0/topology/thread_siblings', 0o644, b'01\n')
f('sys/devices/system/cpu/cpu0/topology/thread_siblings_list', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu0/topology/core_siblings_list', 0o644, b'0-1\n')
f('sys/devices/system/cpu/cpu0/topology/core_siblings', 0o644, b'03\n')
f('sys/devices/system/cpu/cpu0/topology/core_id', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu0/topology/physical_package_id', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu0/thermal_throttle', 0o755)
f('sys/devices/system/cpu/cpu0/thermal_throttle/count', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu0/cache', 0o755)
d('sys/devices/system/cpu/cpu0/cache/index0', 0o755)
f('sys/devices/system/cpu/cpu0/cache/index0/ways_of_associativity', 0o644, b'8\n')
f('sys/devices/system/cpu/cpu0/cache/index0/type', 0o644, b'Data\n')
f('sys/devices/system/cpu/cpu0/cache/index0/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu0/cache/index0/number_of_sets', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu0/cache/index0/size', 0o644, b'32K\n')
f('sys/devices/system/cpu/cpu0/cache/index0/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cache/index0/shared_cpu_map', 0o644, b'01\n')
f('sys/devices/system/cpu/cpu0/cache/index0/level', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cache/index0/shared_cpu_list', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu0/cache/index2', 0o755)
f('sys/devices/system/cpu/cpu0/cache/index2/ways_of_associativity', 0o644, b'16\n')
f('sys/devices/system/cpu/cpu0/cache/index2/type', 0o644, b'Unified\n')
f('sys/devices/system/cpu/cpu0/cache/index2/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu0/cache/index2/number_of_sets', 0o644, b'4096\n')
f('sys/devices/system/cpu/cpu0/cache/index2/size', 0o644, b'4096K\n')
f('sys/devices/system/cpu/cpu0/cache/index2/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cache/index2/shared_cpu_map', 0o644, b'03\n')
f('sys/devices/system/cpu/cpu0/cache/index2/level', 0o644, b'2\n')
f('sys/devices/system/cpu/cpu0/cache/index2/shared_cpu_list', 0o644, b'0-1\n')
d('sys/devices/system/cpu/cpu0/cache/index1', 0o755)
f('sys/devices/system/cpu/cpu0/cache/index1/ways_of_associativity', 0o644, b'8\n')
f('sys/devices/system/cpu/cpu0/cache/index1/type', 0o644, b'Instruction\n')
f('sys/devices/system/cpu/cpu0/cache/index1/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu0/cache/index1/number_of_sets', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu0/cache/index1/size', 0o644, b'32K\n')
f('sys/devices/system/cpu/cpu0/cache/index1/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cache/index1/shared_cpu_map', 0o644, b'01\n')
f('sys/devices/system/cpu/cpu0/cache/index1/level', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cache/index1/shared_cpu_list', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu0/cpuidle', 0o755)
d('sys/devices/system/cpu/cpu0/cpuidle/state2', 0o755)
f('sys/devices/system/cpu/cpu0/cpuidle/state2/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x10\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state2/power', 0o644, b'500\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state2/latency', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state2/usage', 0o644, b'689\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state2/name', 0o644, b'C2\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state2/time', 0o644, b'209561\n')
d('sys/devices/system/cpu/cpu0/cpuidle/state3', 0o755)
f('sys/devices/system/cpu/cpu0/cpuidle/state3/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x30\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state3/power', 0o644, b'100\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state3/latency', 0o644, b'57\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state3/usage', 0o644, b'11774\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state3/name', 0o644, b'C3\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state3/time', 0o644, b'116651688\n')
d('sys/devices/system/cpu/cpu0/cpuidle/state1', 0o755)
f('sys/devices/system/cpu/cpu0/cpuidle/state1/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x0\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state1/power', 0o644, b'1000\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state1/latency', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state1/usage', 0o644, b'2\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state1/name', 0o644, b'C1\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state1/time', 0o644, b'1044\n')
d('sys/devices/system/cpu/cpu0/cpuidle/state0', 0o755)
f('sys/devices/system/cpu/cpu0/cpuidle/state0/desc', 0o644, b'<null>\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state0/power', 0o644, b'4294967295\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state0/latency', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state0/usage', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state0/name', 0o644, b'<null>\n')
f('sys/devices/system/cpu/cpu0/cpuidle/state0/time', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu0/cpufreq', 0o755)
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq', 0o644, b'1000000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/cpuinfo_max_freq', 0o644, b'2333000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies', 0o644, b'2333000 2000000 1667000 1333000 1000000 \n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_governor', 0o644, b'ondemand\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_available_governors', 0o644, b'ondemand performance \n')
f('sys/devices/system/cpu/cpu0/cpufreq/cpuinfo_cur_freq', 0o644, b'1000000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/affected_cpus', 0o644, b'0 1\n')
f('sys/devices/system/cpu/cpu0/cpufreq/related_cpus', 0o644, b'0 1\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_setspeed', 0o644, b'<unsupported>\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_driver', 0o644, b'acpi-cpufreq\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_max_freq', 0o644, b'1000000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/scaling_min_freq', 0o644, b'1000000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/cpuinfo_min_freq', 0o644, b'1000000\n')
d('sys/devices/system/cpu/cpu0/cpufreq/stats', 0o755)
f('sys/devices/system/cpu/cpu0/cpufreq/stats/total_trans', 0o644, b'12627\n')
f('sys/devices/system/cpu/cpu0/cpufreq/stats/time_in_state', 0o644, b'''2333000 92206
2000000 442
1667000 1009
1333000 1713
1000000 740740
''')
d('sys/devices/system/cpu/cpu0/cpufreq/ondemand', 0o755)
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/powersave_bias', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/sampling_rate_max', 0o644, b'10000000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/sampling_rate', 0o644, b'20000\n')
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/up_threshold', 0o644, b'90\n')
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/ignore_nice_load', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu0/cpufreq/ondemand/sampling_rate_min', 0o644, b'10000\n')
d('sys/devices/system/cpu/cpu0/microcode', 0o755)
f('sys/devices/system/cpu/cpu0/microcode/version', 0o644, b'0xc7\n')
f('sys/devices/system/cpu/cpu0/microcode/processor_flags', 0o644, b'0x20\n')
d('sys/devices/system/cpu/cpuidle', 0o755)
f('sys/devices/system/cpu/cpuidle/current_driver', 0o644, b'acpi_idle\n')
f('sys/devices/system/cpu/cpuidle/current_governor_ro', 0o644, b'menu\n')
d('sys/devices/system/cpu/cpu1', 0o755)
l('sys/devices/system/cpu/cpu1/cpufreq', '../cpu0/cpufreq')
f('sys/devices/system/cpu/cpu1/crash_notes', 0o644, b'7fb20b68\n')
f('sys/devices/system/cpu/cpu1/online', 0o644, b'1\n')
d('sys/devices/system/cpu/cpu1/topology', 0o755)
f('sys/devices/system/cpu/cpu1/topology/thread_siblings', 0o644, b'02\n')
f('sys/devices/system/cpu/cpu1/topology/thread_siblings_list', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/topology/core_siblings_list', 0o644, b'0-1\n')
f('sys/devices/system/cpu/cpu1/topology/core_siblings', 0o644, b'03\n')
f('sys/devices/system/cpu/cpu1/topology/core_id', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/topology/physical_package_id', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu1/thermal_throttle', 0o755)
f('sys/devices/system/cpu/cpu1/thermal_throttle/count', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu1/cache', 0o755)
d('sys/devices/system/cpu/cpu1/cache/index0', 0o755)
f('sys/devices/system/cpu/cpu1/cache/index0/ways_of_associativity', 0o644, b'8\n')
f('sys/devices/system/cpu/cpu1/cache/index0/type', 0o644, b'Data\n')
f('sys/devices/system/cpu/cpu1/cache/index0/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu1/cache/index0/number_of_sets', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu1/cache/index0/size', 0o644, b'32K\n')
f('sys/devices/system/cpu/cpu1/cache/index0/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cache/index0/shared_cpu_map', 0o644, b'02\n')
f('sys/devices/system/cpu/cpu1/cache/index0/level', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cache/index0/shared_cpu_list', 0o644, b'1\n')
d('sys/devices/system/cpu/cpu1/cache/index2', 0o755)
f('sys/devices/system/cpu/cpu1/cache/index2/ways_of_associativity', 0o644, b'16\n')
f('sys/devices/system/cpu/cpu1/cache/index2/type', 0o644, b'Unified\n')
f('sys/devices/system/cpu/cpu1/cache/index2/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu1/cache/index2/number_of_sets', 0o644, b'4096\n')
f('sys/devices/system/cpu/cpu1/cache/index2/size', 0o644, b'4096K\n')
f('sys/devices/system/cpu/cpu1/cache/index2/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cache/index2/shared_cpu_map', 0o644, b'03\n')
f('sys/devices/system/cpu/cpu1/cache/index2/level', 0o644, b'2\n')
f('sys/devices/system/cpu/cpu1/cache/index2/shared_cpu_list', 0o644, b'0-1\n')
d('sys/devices/system/cpu/cpu1/cache/index1', 0o755)
f('sys/devices/system/cpu/cpu1/cache/index1/ways_of_associativity', 0o644, b'8\n')
f('sys/devices/system/cpu/cpu1/cache/index1/type', 0o644, b'Instruction\n')
f('sys/devices/system/cpu/cpu1/cache/index1/coherency_line_size', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu1/cache/index1/number_of_sets', 0o644, b'64\n')
f('sys/devices/system/cpu/cpu1/cache/index1/size', 0o644, b'32K\n')
f('sys/devices/system/cpu/cpu1/cache/index1/physical_line_partition', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cache/index1/shared_cpu_map', 0o644, b'02\n')
f('sys/devices/system/cpu/cpu1/cache/index1/level', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cache/index1/shared_cpu_list', 0o644, b'1\n')
d('sys/devices/system/cpu/cpu1/cpuidle', 0o755)
d('sys/devices/system/cpu/cpu1/cpuidle/state2', 0o755)
f('sys/devices/system/cpu/cpu1/cpuidle/state2/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x10\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state2/power', 0o644, b'500\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state2/latency', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state2/usage', 0o644, b'271\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state2/name', 0o644, b'C2\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state2/time', 0o644, b'47233\n')
d('sys/devices/system/cpu/cpu1/cpuidle/state3', 0o755)
f('sys/devices/system/cpu/cpu1/cpuidle/state3/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x30\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state3/power', 0o644, b'100\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state3/latency', 0o644, b'57\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state3/usage', 0o644, b'10601\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state3/name', 0o644, b'C3\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state3/time', 0o644, b'117038054\n')
d('sys/devices/system/cpu/cpu1/cpuidle/state1', 0o755)
f('sys/devices/system/cpu/cpu1/cpuidle/state1/desc', 0o644, b'ACPI FFH INTEL MWAIT 0x0\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state1/power', 0o644, b'1000\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state1/latency', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state1/usage', 0o644, b'1\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state1/name', 0o644, b'C1\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state1/time', 0o644, b'8\n')
d('sys/devices/system/cpu/cpu1/cpuidle/state0', 0o755)
f('sys/devices/system/cpu/cpu1/cpuidle/state0/desc', 0o644, b'<null>\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state0/power', 0o644, b'4294967295\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state0/latency', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state0/usage', 0o644, b'0\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state0/name', 0o644, b'<null>\n')
f('sys/devices/system/cpu/cpu1/cpuidle/state0/time', 0o644, b'0\n')
d('sys/devices/system/cpu/cpu1/microcode', 0o755)
f('sys/devices/system/cpu/cpu1/microcode/version', 0o644, b'0xc7\n')
f('sys/devices/system/cpu/cpu1/microcode/processor_flags', 0o644, b'0x20\n')
d('sys/devices/pnp0', 0o755)
f('sys/devices/pnp0/uevent', 0o644, b'')
d('sys/devices/pnp0/00:04', 0o755)
l('sys/devices/pnp0/00:04/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0200:00')
l('sys/devices/pnp0/00:04/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:04/options', 0o644, b'')
f('sys/devices/pnp0/00:04/id', 0o644, b'PNP0200\n')
f('sys/devices/pnp0/00:04/resources', 0o644, b'''state = active
io 0x0-0xf
io 0x80-0x8f
io 0xc0-0xdf
dma 4
''')
f('sys/devices/pnp0/00:04/uevent', 0o644, b'')
d('sys/devices/pnp0/00:04/power', 0o755)
f('sys/devices/pnp0/00:04/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:0a', 0o755)
l('sys/devices/pnp0/00:0a/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/ATM1200:00')
l('sys/devices/pnp0/00:0a/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:0a/options', 0o644, b'')
f('sys/devices/pnp0/00:0a/id', 0o644, b'''ATM1200
PNP0c31
''')
f('sys/devices/pnp0/00:0a/resources', 0o644, b'''state = active
mem 0xfed40000-0xfed40fff
''')
f('sys/devices/pnp0/00:0a/uevent', 0o644, b'')
d('sys/devices/pnp0/00:0a/power', 0o755)
f('sys/devices/pnp0/00:0a/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:03', 0o755)
l('sys/devices/pnp0/00:03/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0103:00')
l('sys/devices/pnp0/00:03/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:03/options', 0o644, b'')
f('sys/devices/pnp0/00:03/id', 0o644, b'PNP0103\n')
f('sys/devices/pnp0/00:03/resources', 0o644, b'''state = active
mem 0xfed00000-0xfed003ff
''')
f('sys/devices/pnp0/00:03/uevent', 0o644, b'')
d('sys/devices/pnp0/00:03/power', 0o755)
f('sys/devices/pnp0/00:03/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:02', 0o755)
l('sys/devices/pnp0/00:02/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C02:00')
l('sys/devices/pnp0/00:02/subsystem', '../../../bus/pnp')
l('sys/devices/pnp0/00:02/driver', '../../../bus/pnp/drivers/system')
f('sys/devices/pnp0/00:02/options', 0o644, b'')
f('sys/devices/pnp0/00:02/id', 0o644, b'PNP0c02\n')
f('sys/devices/pnp0/00:02/resources', 0o644, b'''state = active
io 0x10-0x1f
io 0x90-0x9f
io 0x24-0x25
io 0x28-0x29
io 0x2c-0x2d
io 0x30-0x31
io 0x34-0x35
io 0x38-0x39
io 0x3c-0x3d
io 0xa4-0xa5
io 0xa8-0xa9
io 0xac-0xad
io 0xb0-0xb5
io 0xb8-0xb9
io 0xbc-0xbd
io 0x50-0x53
io 0x72-0x77
io 0x164e-0x164f
io 0x2e-0x2f
io 0x1000-0x107f
io 0x1180-0x11bf
io 0x800-0x80f
io 0x15e0-0x15ef
io 0x1600-0x165f
mem 0xf0000000-0xf3ffffff
mem 0xfed1c000-0xfed1ffff
mem 0xfed14000-0xfed17fff
mem 0xfed18000-0xfed18fff
mem 0xfed19000-0xfed19fff
''')
f('sys/devices/pnp0/00:02/uevent', 0o644, b'DRIVER=system\n')
d('sys/devices/pnp0/00:02/power', 0o755)
f('sys/devices/pnp0/00:02/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:00', 0o755)
l('sys/devices/pnp0/00:00/firmware_node', '../../LNXSYSTM:00/device:00/PNP0C01:00')
l('sys/devices/pnp0/00:00/subsystem', '../../../bus/pnp')
l('sys/devices/pnp0/00:00/driver', '../../../bus/pnp/drivers/system')
f('sys/devices/pnp0/00:00/options', 0o644, b'')
f('sys/devices/pnp0/00:00/id', 0o644, b'PNP0c01\n')
f('sys/devices/pnp0/00:00/resources', 0o644, b'''state = active
mem 0x0-0x9ffff
mem 0xc0000-0xc3fff
mem 0xc4000-0xc7fff
mem 0xc8000-0xcbfff
mem 0xcc000-0xcffff
mem disabled
mem disabled
mem disabled
mem 0xdc000-0xdffff
mem 0xe0000-0xe3fff
mem 0xe4000-0xe7fff
mem 0xe8000-0xebfff
mem 0xec000-0xeffff
mem 0xf0000-0xfffff
mem 0x100000-0x7fffffff
mem 0xfec00000-0xfed3ffff
mem 0xfed41000-0xffffffff
''')
f('sys/devices/pnp0/00:00/uevent', 0o644, b'DRIVER=system\n')
d('sys/devices/pnp0/00:00/power', 0o755)
f('sys/devices/pnp0/00:00/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/power', 0o755)
f('sys/devices/pnp0/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:09', 0o755)
l('sys/devices/pnp0/00:09/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/IBM0057:00')
l('sys/devices/pnp0/00:09/subsystem', '../../../bus/pnp')
l('sys/devices/pnp0/00:09/driver', '../../../bus/pnp/drivers/i8042 aux')
f('sys/devices/pnp0/00:09/options', 0o644, b'')
f('sys/devices/pnp0/00:09/id', 0o644, b'''IBM0057
PNP0f13
''')
f('sys/devices/pnp0/00:09/resources', 0o644, b'''state = active
irq 12
''')
f('sys/devices/pnp0/00:09/uevent', 0o644, b'DRIVER=i8042 aux\n')
d('sys/devices/pnp0/00:09/power', 0o755)
f('sys/devices/pnp0/00:09/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:07', 0o755)
l('sys/devices/pnp0/00:07/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0B00:00')
l('sys/devices/pnp0/00:07/subsystem', '../../../bus/pnp')
l('sys/devices/pnp0/00:07/driver', '../../../bus/pnp/drivers/rtc_cmos')
f('sys/devices/pnp0/00:07/options', 0o644, b'')
f('sys/devices/pnp0/00:07/id', 0o644, b'PNP0b00\n')
f('sys/devices/pnp0/00:07/nvram', 0o644, b'\x00\x00\x00\x00\x00\x00\x03\x80\x02\x00\xfc\x01\x00\x03\x00\x9f\x16\x00\x00\x02\x00\x00\x00\x01\x05\x00\x00\x11\x11\x01\x00\x00\x02e\x00\xfc \x8c\xc1\x01\x00|\x00\x00\x00\x00\x00\x00\x00\x00\x08D\xb0\x11\x00c\x11T/\x06\xd00\x80\x00\x00\x00\x00\x00\xc0\x05R\xfc\x01\x1d\x00[\x0f\x02\x03\x00\x00\x04\x04\x00\xdf8\xd3\xa0\x13r\xf2\x06\x13\xf9\xa7\xf1\x0e\x00\xc5\xfe\x00\x00\xf1\xff\xff\xff\xff\x00\x00\x00IS\xb2\x00')
f('sys/devices/pnp0/00:07/resources', 0o644, b'''state = active
io 0x70-0x71
irq 8
''')
f('sys/devices/pnp0/00:07/uevent', 0o644, b'DRIVER=rtc_cmos\n')
d('sys/devices/pnp0/00:07/rtc', 0o755)
d('sys/devices/pnp0/00:07/rtc/rtc0', 0o755)
l('sys/devices/pnp0/00:07/rtc/rtc0/subsystem', '../../../../../class/rtc')
l('sys/devices/pnp0/00:07/rtc/rtc0/device', '../../../00:07')
f('sys/devices/pnp0/00:07/rtc/rtc0/date', 0o644, b'2008-09-29\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/dev', 0o644, b'253:0\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/max_user_freq', 0o644, b'64\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/wakealarm', 0o644, b'')
f('sys/devices/pnp0/00:07/rtc/rtc0/since_epoch', 0o644, b'1222655451\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/name', 0o644, b'rtc_cmos\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/time', 0o644, b'02:30:51\n')
f('sys/devices/pnp0/00:07/rtc/rtc0/uevent', 0o644, b'''MAJOR=253
MINOR=0
''')
d('sys/devices/pnp0/00:07/rtc/rtc0/power', 0o755)
f('sys/devices/pnp0/00:07/rtc/rtc0/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:07/power', 0o755)
f('sys/devices/pnp0/00:07/power/wakeup', 0o644, b'enabled\n')
d('sys/devices/pnp0/00:06', 0o755)
l('sys/devices/pnp0/00:06/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0C04:00')
l('sys/devices/pnp0/00:06/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:06/options', 0o644, b'')
f('sys/devices/pnp0/00:06/id', 0o644, b'PNP0c04\n')
f('sys/devices/pnp0/00:06/resources', 0o644, b'''state = active
io 0xf0-0xf0
irq 13
''')
f('sys/devices/pnp0/00:06/uevent', 0o644, b'')
d('sys/devices/pnp0/00:06/power', 0o755)
f('sys/devices/pnp0/00:06/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:08', 0o755)
l('sys/devices/pnp0/00:08/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0303:00')
l('sys/devices/pnp0/00:08/subsystem', '../../../bus/pnp')
l('sys/devices/pnp0/00:08/driver', '../../../bus/pnp/drivers/i8042 kbd')
f('sys/devices/pnp0/00:08/options', 0o644, b'')
f('sys/devices/pnp0/00:08/id', 0o644, b'PNP0303\n')
f('sys/devices/pnp0/00:08/resources', 0o644, b'''state = active
io 0x60-0x60
io 0x64-0x64
irq 1
''')
f('sys/devices/pnp0/00:08/uevent', 0o644, b'DRIVER=i8042 kbd\n')
d('sys/devices/pnp0/00:08/power', 0o755)
f('sys/devices/pnp0/00:08/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:05', 0o755)
l('sys/devices/pnp0/00:05/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01/PNP0800:00')
l('sys/devices/pnp0/00:05/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:05/options', 0o644, b'')
f('sys/devices/pnp0/00:05/id', 0o644, b'PNP0800\n')
f('sys/devices/pnp0/00:05/resources', 0o644, b'''state = active
io 0x61-0x61
''')
f('sys/devices/pnp0/00:05/uevent', 0o644, b'')
d('sys/devices/pnp0/00:05/power', 0o755)
f('sys/devices/pnp0/00:05/power/wakeup', 0o644, b'\n')
d('sys/devices/pnp0/00:01', 0o755)
l('sys/devices/pnp0/00:01/subsystem', '../../../bus/pnp')
f('sys/devices/pnp0/00:01/options', 0o644, b'')
f('sys/devices/pnp0/00:01/id', 0o644, b'''PNP0a08
PNP0a03
''')
f('sys/devices/pnp0/00:01/resources', 0o644, b'''state = active
io 0xcf8-0xcff
''')
f('sys/devices/pnp0/00:01/uevent', 0o644, b'')
d('sys/devices/pnp0/00:01/power', 0o755)
f('sys/devices/pnp0/00:01/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00', 0o755)
l('sys/devices/pci0000:00/firmware_node', '../LNXSYSTM:00/device:00/PNP0A08:00')
f('sys/devices/pci0000:00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.3', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:20')
l('sys/devices/pci0000:00/0000:00:1d.3/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1d.3/driver', '../../../bus/pci/drivers/uhci_hcd')
f('sys/devices/pci0000:00/0000:00:1d.3/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1d.3/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/modalias', 0o644, b'pci:v00008086d000027CBsv000017AAsd0000200Abc0Csc03i00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000001860 0x000000000000187f 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1d.3/config', 0o644, b"\x86\x80\xcb'\x05\x00\x80\x02\x02\x00\x03\x0c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00a\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\n \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1d.3/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.3/subsystem_device', 0o644, b'0x200a\n')
f('sys/devices/pci0000:00/0000:00:1d.3/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1d.3/irq', 0o644, b'19\n')
f('sys/devices/pci0000:00/0000:00:1d.3/pools', 0o644, b'''poolinfo - 0.1
uhci_qh            13   32  128  1
uhci_td             3   64   64  1
buffer-2048         0    0 2048  0
buffer-512          0    0  512  0
buffer-128          0    0  128  0
buffer-32           1  128   32  1
''')
f('sys/devices/pci0000:00/0000:00:1d.3/device', 0o644, b'0x27cb\n')
f('sys/devices/pci0000:00/0000:00:1d.3/class', 0o644, b'0x0c0300\n')
f('sys/devices/pci0000:00/0000:00:1d.3/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.3/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1d.3/uevent', 0o644, b'''DRIVER=uhci_hcd
PCI_CLASS=C0300
PCI_ID=8086:27CB
PCI_SUBSYS_ID=17AA:200A
PCI_SLOT_NAME=0000:00:1d.3
MODALIAS=pci:v00008086d000027CBsv000017AAsd0000200Abc0Csc03i00
''')
f('sys/devices/pci0000:00/0000:00:1d.3/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.3/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1d.3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4/subsystem', '../../../../../class/usb_host')
l('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4/device', '../../../0000:00:1d.3')
f('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb_host/usb_host4/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/subsystem', '../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/driver', '../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/ep_00', 'usb_endpoint/usbdev4.1_ep00')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/version', 0o644, b' 1.10\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/authorized_default', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/maxchild', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bDeviceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bcdDevice', 0o644, b'0206\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/devnum', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/idProduct', 0o644, b'0001\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/product', 0o644, b'UHCI Host Controller\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/dev', 0o644, b'189:384\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/urbnum', 0o644, b'62\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/descriptors', 0o644, b'\x12\x01\x10\x01\t\x00\x00@k\x1d\x01\x00\x06\x02\x03\x02\x01\x01\t\x02\x19\x00\x01\x01\x00\xe0\x00\t\x04\x00\x00\x01\t\x00\x00\x00\x07\x05\x81\x03\x02\x00\xff')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/serial', 0o644, b'0000:00:1d.3\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/speed', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bmAttributes', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/busnum', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/uevent', 0o644, b'''MAJOR=189
MINOR=384
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=1d6b/1/206
TYPE=9/0/0
BUSNUM=004
DEVNUM=001
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/manufacturer', 0o644, b'Linux 2.6.27-rc7-00106-g6ef190c-dirty uhci_hcd\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/idVendor', 0o644, b'1d6b\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/bMaxPower', 0o644, b'  0mA\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/subsystem', '../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/device', '../../../usb4')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/dev', 0o644, b'252:7\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=7
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/usb_endpoint/usbdev4.1_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/driver', '../../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/ep_00', 'usb_endpoint/usbdev4.3_ep00')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/version', 0o644, b' 1.00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/maxchild', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bDeviceClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bcdDevice', 0o644, b'0001\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/devnum', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/idProduct', 0o644, b'2016\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/product', 0o644, b'Biometric Coprocessor\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/dev', 0o644, b'189:386\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/urbnum', 0o644, b'9\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/descriptors', 0o644, b"\x12\x01\x00\x01\x00\x00\x00\x08\x83\x04\x16 \x01\x00\x01\x02\x00\x01\t\x02'\x00\x01\x01\x00\xa02\t\x04\x00\x00\x03\xff\x00\x00\x00\x07\x05\x81\x02@\x00\x00\x07\x05\x02\x02@\x00\x00\x07\x05\x83\x03\x04\x00\x14")
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/speed', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bMaxPacketSize0', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bmAttributes', 0o644, b'a0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/busnum', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/uevent', 0o644, b'''MAJOR=189
MINOR=386
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=483/2016/1
TYPE=0/0/0
BUSNUM=004
DEVNUM=003
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/manufacturer', 0o644, b'STMicroelectronics\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/idVendor', 0o644, b'0483\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/bMaxPower', 0o644, b'100mA\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/device', '../../../4-2')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/dev', 0o644, b'252:23\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/wMaxPacketSize', 0o644, b'0008\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=23
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/usb_endpoint/usbdev4.3_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/active_duration', 0o644, b'8370363\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/connected_duration', 0o644, b'8370362\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/persist', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/power/level', 0o644, b'on\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/ep_83', 'usb_endpoint/usbdev4.3_ep83')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/ep_81', 'usb_endpoint/usbdev4.3_ep81')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/ep_02', 'usb_endpoint/usbdev4.3_ep02')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/modalias', 0o644, b'usb:v0483p2016d0001dc00dsc00dp00icFFisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bNumEndpoints', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bInterfaceClass', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=483/2016/1
TYPE=0/0/0
INTERFACE=255/0/0
MODALIAS=usb:v0483p2016d0001dc00dsc00dp00icFFisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/device', '../../../4-2:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/interval', 0o644, b'20ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/bEndpointAddress', 0o644, b'83\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/dev', 0o644, b'252:22\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/wMaxPacketSize', 0o644, b'0004\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/uevent', 0o644, b'''MAJOR=252
MINOR=22
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/bInterval', 0o644, b'14\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep83/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/device', '../../../4-2:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/dev', 0o644, b'252:20\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=20
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/device', '../../../4-2:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/bEndpointAddress', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/dev', 0o644, b'252:21\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/uevent', 0o644, b'''MAJOR=252
MINOR=21
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/usb_endpoint/usbdev4.3_ep02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-2/4-2:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/ep_81', 'usb_endpoint/usbdev4.1_ep81')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/driver', '../../../../../bus/usb/drivers/hub')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/modalias', 0o644, b'usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bInterfaceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=hub
PRODUCT=1d6b/1/206
TYPE=9/0/0
INTERFACE=9/0/0
MODALIAS=usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/device', '../../../4-0:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/interval', 0o644, b'255ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/dev', 0o644, b'252:6\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/wMaxPacketSize', 0o644, b'0002\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=6
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/usb_endpoint/usbdev4.1_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-0:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/power/active_duration', 0o644, b'8372033\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/power/connected_duration', 0o644, b'8372033\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/power/level', 0o644, b'auto\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/driver', '../../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/ep_00', 'usb_endpoint/usbdev4.2_ep00')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/version', 0o644, b' 2.00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/maxchild', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bDeviceClass', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bcdDevice', 0o644, b'0100\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/devnum', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bDeviceProtocol', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/idProduct', 0o644, b'2110\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/product', 0o644, b'BCM2045B\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bDeviceSubClass', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/dev', 0o644, b'189:385\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/urbnum', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/descriptors', 0o644, b'\x12\x01\x00\x02\xe0\x01\x01@\\\n\x10!\x00\x01\x01\x02\x00\x01\t\x02\xd8\x00\x04\x01\x00\xe0\x00\t\x04\x00\x00\x03\xe0\x01\x01\x00\x07\x05\x81\x03\x10\x00\x01\x07\x05\x82\x02@\x00\x01\x07\x05\x02\x02@\x00\x01\t\x04\x01\x00\x02\xe0\x01\x01\x00\x07\x05\x83\x01\x00\x00\x01\x07\x05\x03\x01\x00\x00\x01\t\x04\x01\x01\x02\xe0\x01\x01\x00\x07\x05\x83\x01\t\x00\x01\x07\x05\x03\x01\t\x00\x01\t\x04\x01\x02\x02\xe0\x01\x01\x00\x07\x05\x83\x01\x11\x00\x01\x07\x05\x03\x01\x11\x00\x01\t\x04\x01\x03\x02\xe0\x01\x01\x00\x07\x05\x83\x01 \x00\x01\x07\x05\x03\x01 \x00\x01\t\x04\x01\x04\x02\xe0\x01\x01\x00\x07\x05\x83\x01@\x00\x01\x07\x05\x03\x01@\x00\x01\t\x04\x01\x05\x02\xe0\x01\x01\x00\x07\x05\x83\x01@\x00\x01\x07\x05\x03\x01@\x00\x01\t\x04\x02\x00\x02\xff\xff\xff\x00\x07\x05\x84\x02 \x00\x01\x07\x05\x04\x02 \x00\x01\t\x04\x03\x00\x00\xfe\x01\x00\x00\x07!\x07\x88\x13@\x00')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/speed', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bmAttributes', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/busnum', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/uevent', 0o644, b'''MAJOR=189
MINOR=385
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=a5c/2110/100
TYPE=224/1/1
BUSNUM=004
DEVNUM=002
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/manufacturer', 0o644, b'Broadcom Corp\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/idVendor', 0o644, b'0a5c\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bNumInterfaces', 0o644, b' 4\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/bMaxPower', 0o644, b'  0mA\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/modalias', 0o644, b'usb:v0A5Cp2110d0100dcE0dsc01dp01icFEisc01ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bInterfaceNumber', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bInterfaceSubClass', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bInterfaceClass', 0o644, b'fe\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=a5c/2110/100
TYPE=224/1/1
INTERFACE=254/1/0
MODALIAS=usb:v0A5Cp2110d0100dcE0dsc01dp01icFEisc01ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/device', '../../../4-1')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/dev', 0o644, b'252:19\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=19
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/usb_endpoint/usbdev4.2_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/ep_81', 'usb_endpoint/usbdev4.2_ep81')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/ep_02', 'usb_endpoint/usbdev4.2_ep02')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/ep_82', 'usb_endpoint/usbdev4.2_ep82')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/modalias', 0o644, b'usb:v0A5Cp2110d0100dcE0dsc01dp01icE0isc01ip01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bNumEndpoints', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bInterfaceSubClass', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bInterfaceClass', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=a5c/2110/100
TYPE=224/1/1
INTERFACE=224/1/1
MODALIAS=usb:v0A5Cp2110d0100dcE0dsc01dp01icE0isc01ip01
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/bInterfaceProtocol', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/device', '../../../4-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/bEndpointAddress', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/dev', 0o644, b'252:14\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/uevent', 0o644, b'''MAJOR=252
MINOR=14
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/device', '../../../4-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/bEndpointAddress', 0o644, b'82\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/dev', 0o644, b'252:13\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/uevent', 0o644, b'''MAJOR=252
MINOR=13
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep82/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/device', '../../../4-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/interval', 0o644, b'1ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/dev', 0o644, b'252:12\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/wMaxPacketSize', 0o644, b'0010\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=12
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/usb_endpoint/usbdev4.2_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/active_duration', 0o644, b'8370651\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/connected_duration', 0o644, b'8370650\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/persist', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/power/level', 0o644, b'on\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/ep_84', 'usb_endpoint/usbdev4.2_ep84')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/ep_04', 'usb_endpoint/usbdev4.2_ep04')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/modalias', 0o644, b'usb:v0A5Cp2110d0100dcE0dsc01dp01icFFiscFFipFF\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bInterfaceNumber', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bNumEndpoints', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bInterfaceSubClass', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bInterfaceClass', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=a5c/2110/100
TYPE=224/1/1
INTERFACE=255/255/255
MODALIAS=usb:v0A5Cp2110d0100dcE0dsc01dp01icFFiscFFipFF
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/bInterfaceProtocol', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/device', '../../../4-1:1.2')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/bEndpointAddress', 0o644, b'04\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/dev', 0o644, b'252:18\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/wMaxPacketSize', 0o644, b'0020\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/uevent', 0o644, b'''MAJOR=252
MINOR=18
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep04/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/device', '../../../4-1:1.2')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/bEndpointAddress', 0o644, b'84\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/dev', 0o644, b'252:17\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/wMaxPacketSize', 0o644, b'0020\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/uevent', 0o644, b'''MAJOR=252
MINOR=17
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/usb_endpoint/usbdev4.2_ep84/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.2/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/ep_83', 'usb_endpoint/usbdev4.2_ep83')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/ep_03', 'usb_endpoint/usbdev4.2_ep03')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/modalias', 0o644, b'usb:v0A5Cp2110d0100dcE0dsc01dp01icE0isc01ip01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bInterfaceNumber', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bNumEndpoints', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bInterfaceSubClass', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bInterfaceClass', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=a5c/2110/100
TYPE=224/1/1
INTERFACE=224/1/1
MODALIAS=usb:v0A5Cp2110d0100dcE0dsc01dp01icE0isc01ip01
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/bInterfaceProtocol', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/device', '../../../4-1:1.1')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/interval', 0o644, b'1ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/bEndpointAddress', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/type', 0o644, b'Isoc\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/dev', 0o644, b'252:16\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/wMaxPacketSize', 0o644, b'0000\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/bmAttributes', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/uevent', 0o644, b'''MAJOR=252
MINOR=16
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/device', '../../../4-1:1.1')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/interval', 0o644, b'1ms\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/bEndpointAddress', 0o644, b'83\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/type', 0o644, b'Isoc\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/dev', 0o644, b'252:15\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/wMaxPacketSize', 0o644, b'0000\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/bmAttributes', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/uevent', 0o644, b'''MAJOR=252
MINOR=15
''')
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/bInterval', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/usb_endpoint/usbdev4.2_ep83/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.3/usb4/4-1/4-1:1.1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.2', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.2/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:0d')
l('sys/devices/pci0000:00/0000:00:1c.2/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.2/driver', '../../../bus/pci/drivers/pcieport-driver')
f('sys/devices/pci0000:00/0000:00:1c.2/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.2/enable', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.2/modalias', 0o644, b'pci:v00008086d000027D4sv00000000sd00000000bc06sc04i00\n')
f('sys/devices/pci0000:00/0000:00:1c.2/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000006000 0x0000000000007fff 0x0000000000000100
0x00000000e8000000 0x00000000e9ffffff 0x0000000000000200
0x00000000e4100000 0x00000000e41fffff 0x0000000000001201
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.2/config', 0o644, b"\x86\x80\xd4'\x07\x05\x10\x00\x02\x00\x04\x06\x10\x00\x81\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04\x0b\x00`p\x00\x00\x00\xe8\xf0\xe9\x11\xe4\x11\xe4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x0b\x03\x04\x00\x10\x80A\x01\xc0\x0f\x00\x00\x00\x00\x10\x00\x11L\x11\x03\x00\x00\x01\x10\xe0\xa0 \x00(\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x90\x01\x00\x0c0\xe0\xfe\x89A\x00\x00\x00\x00\x00\x00\r\xa0\x00\x00\xaa\x17\x11 \x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x11\xc0\x00\x00\x00\x00\x00\x00\xc7\x00\x06\x07\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00\x02\x00\x01\x18\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xff\x00\x00\x80\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x11\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x02\x03\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x01\xc0\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07`\x00`\x00\x00\x00\x00`\x02\x00\x00\x0c\x00\x06\x00\x00\x00\x00\x00[`\xc9\xc0\x00p&u\x00\x10\x00\x00\xff\x0f\x00\x00\x16\x00\x00\x14\xb5\xbcJ\xbc\x0b*\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00 \x00\x10\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x02\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1c.2/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.2/subsystem_device', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.2/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.2/irq', 0o644, b'508\n')
f('sys/devices/pci0000:00/0000:00:1c.2/device', 0o644, b'0x27d4\n')
f('sys/devices/pci0000:00/0000:00:1c.2/class', 0o644, b'0x060400\n')
f('sys/devices/pci0000:00/0000:00:1c.2/msi_bus', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.2/subsystem_vendor', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.2/uevent', 0o644, b'''DRIVER=pcieport-driver
PCI_CLASS=60400
PCI_ID=8086:27D4
PCI_SUBSYS_ID=0000:0000
PCI_SLOT_NAME=0000:00:1c.2
MODALIAS=pci:v00008086d000027D4sv00000000sd00000000bc06sc04i00
''')
f('sys/devices/pci0000:00/0000:00:1c.2/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.2/pci_bus', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/subsystem', '../../../../../class/pci_bus')
l('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/device', '../../../0000:00:1c.2')
f('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.2/pci_bus/0000:04/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.2/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.2/0000:00:1c.2:pcie03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:25')
l('sys/devices/pci0000:00/0000:00:1b.0/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1b.0/driver', '../../../bus/pci/drivers/HDA Intel')
f('sys/devices/pci0000:00/0000:00:1b.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1b.0/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1b.0/modalias', 0o644, b'pci:v00008086d000027D8sv000017AAsd00002010bc04sc03i00\n')
f('sys/devices/pci0000:00/0000:00:1b.0/resource', 0o644, b'''0x00000000ee400000 0x00000000ee403fff 0x0000000000020204
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1b.0/config', 0o644, b"\x86\x80\xd8'\x06\x01\x10\x00\x02\x00\x03\x04\x10\x00\x00\x00\x04\x00@\xee\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x10 \x00\x00\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x0b\x02\x00\x00\x03\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x01`B\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05p\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x91\x00\x00\x00\x00\x00\x00\x08\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00\x02\x00\x01\x13\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x81\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x02\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x00\xc0\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1b.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1b.0/subsystem_device', 0o644, b'0x2010\n')
f('sys/devices/pci0000:00/0000:00:1b.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1b.0/irq', 0o644, b'17\n')
f('sys/devices/pci0000:00/0000:00:1b.0/device', 0o644, b'0x27d8\n')
f('sys/devices/pci0000:00/0000:00:1b.0/class', 0o644, b'0x040300\n')
f('sys/devices/pci0000:00/0000:00:1b.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1b.0/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1b.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1b.0/uevent', 0o644, b'''DRIVER=HDA Intel
PCI_CLASS=40300
PCI_ID=8086:27D8
PCI_SUBSYS_ID=17AA:2010
PCI_SLOT_NAME=0000:00:1b.0
MODALIAS=pci:v00008086d000027D8sv000017AAsd00002010bc04sc03i00
''')
f('sys/devices/pci0000:00/0000:00:1b.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1b.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound', 0o755)
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/subsystem', '../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/device', '../../../0000:00:1b.0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/dev', 0o644, b'14:4\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/uevent', 0o644, b'''MAJOR=14
MINOR=4
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/audio/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/dev', 0o644, b'14:3\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/uevent', 0o644, b'''MAJOR=14
MINOR=3
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/dsp/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/dev', 0o644, b'14:0\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/uevent', 0o644, b'''MAJOR=14
MINOR=0
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/mixer/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/pcm_class', 0o644, b'generic\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/dev', 0o644, b'116:16\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/uevent', 0o644, b'''MAJOR=116
MINOR=16
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0p/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/dev', 0o644, b'116:0\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/uevent', 0o644, b'''MAJOR=116
MINOR=0
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/controlC0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/pcm_class', 0o644, b'generic\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/dev', 0o644, b'116:24\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/uevent', 0o644, b'''MAJOR=116
MINOR=24
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D0c/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/dev', 0o644, b'14:12\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/uevent', 0o644, b'''MAJOR=14
MINOR=12
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/adsp/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p', 0o755)
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/subsystem', '../../../../../../class/sound')
l('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/device', '../../card0')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/pcm_class', 0o644, b'generic\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/dev', 0o644, b'116:17\n')
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/uevent', 0o644, b'''MAJOR=116
MINOR=17
''')
d('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1b.0/sound/card0/pcmC0D1p/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/pci_bus', 0o755)
d('sys/devices/pci0000:00/pci_bus/0000:00', 0o755)
l('sys/devices/pci0000:00/pci_bus/0000:00/subsystem', '../../../../class/pci_bus')
l('sys/devices/pci0000:00/pci_bus/0000:00/device', '../../../pci0000:00')
f('sys/devices/pci0000:00/pci_bus/0000:00/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/pci_bus/0000:00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/pci_bus/0000:00/power', 0o755)
f('sys/devices/pci0000:00/pci_bus/0000:00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/power', 0o755)
f('sys/devices/pci0000:00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:21')
l('sys/devices/pci0000:00/0000:00:1d.7/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1d.7/driver', '../../../bus/pci/drivers/ehci_hcd')
f('sys/devices/pci0000:00/0000:00:1d.7/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1d.7/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/modalias', 0o644, b'pci:v00008086d000027CCsv000017AAsd0000200Bbc0Csc03i20\n')
f('sys/devices/pci0000:00/0000:00:1d.7/resource', 0o644, b'''0x00000000ee404000 0x00000000ee4043ff 0x0000000000020200
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1d.7/config', 0o644, b"\x86\x80\xcc'\x06\x01\x90\x02\x02 \x03\x0c\x00\x00\x00\x00\x00@@\xee\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x0b \x00\x00\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x0b\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01X\xc2\xc9\x00\x80\x00\x00\n\x00\xa0 \x00\x00\x00\x00  \x9f\x01\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x08\x80\x00\x00\xd7?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xff\x00\xff\x00\xff\x00 \x00\x00\x88\x00\x00\x00\x00\xdb\xb6m\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\t\x88\x85@\x00\x86\x0f\x02\x00\x06\x17\x02 ")
f('sys/devices/pci0000:00/0000:00:1d.7/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.7/subsystem_device', 0o644, b'0x200b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1d.7/irq', 0o644, b'19\n')
f('sys/devices/pci0000:00/0000:00:1d.7/pools', 0o644, b'''poolinfo - 0.1
ehci_sitd           0    0   96  0
ehci_itd            0    0  192  0
ehci_qh             5   25  160  1
ehci_qtd            5   42   96  1
buffer-2048        16   32 2048 16
buffer-512          0    0  512  0
buffer-128          2   32  128  1
buffer-32           2  128   32  1
''')
f('sys/devices/pci0000:00/0000:00:1d.7/device', 0o644, b'0x27cc\n')
f('sys/devices/pci0000:00/0000:00:1d.7/class', 0o644, b'0x0c0320\n')
f('sys/devices/pci0000:00/0000:00:1d.7/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1d.7/uevent', 0o644, b'''DRIVER=ehci_hcd
PCI_CLASS=C0320
PCI_ID=8086:27CC
PCI_SUBSYS_ID=17AA:200B
PCI_SLOT_NAME=0000:00:1d.7
MODALIAS=pci:v00008086d000027CCsv000017AAsd0000200Bbc0Csc03i20
''')
f('sys/devices/pci0000:00/0000:00:1d.7/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/subsystem', '../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/driver', '../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/ep_00', 'usb_endpoint/usbdev5.1_ep00')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/version', 0o644, b' 2.00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/authorized_default', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/maxchild', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bDeviceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bcdDevice', 0o644, b'0206\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/devnum', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/idProduct', 0o644, b'0002\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/product', 0o644, b'EHCI Host Controller\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/dev', 0o644, b'189:512\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/urbnum', 0o644, b'151\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/descriptors', 0o644, b'\x12\x01\x00\x02\t\x00\x00@k\x1d\x02\x00\x06\x02\x03\x02\x01\x01\t\x02\x19\x00\x01\x01\x00\xe0\x00\t\x04\x00\x00\x01\t\x00\x00\x00\x07\x05\x81\x03\x04\x00\x0c')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/serial', 0o644, b'0000:00:1d.7\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/speed', 0o644, b'480\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bmAttributes', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/busnum', 0o644, b'5\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/uevent', 0o644, b'''MAJOR=189
MINOR=512
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=1d6b/2/206
TYPE=9/0/0
BUSNUM=005
DEVNUM=001
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/manufacturer', 0o644, b'Linux 2.6.27-rc7-00106-g6ef190c-dirty ehci_hcd\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/idVendor', 0o644, b'1d6b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/bMaxPower', 0o644, b'  0mA\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/subsystem', '../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/device', '../../../usb5')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/dev', 0o644, b'252:9\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=9
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/usb_endpoint/usbdev5.1_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/driver', '../../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/ep_00', 'usb_endpoint/usbdev5.7_ep00')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/version', 0o644, b' 2.00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/maxchild', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bDeviceClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bcdDevice', 0o644, b'0100\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/devnum', 0o644, b'7\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/idProduct', 0o644, b'8012\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/product', 0o644, b'Flash Disk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/dev', 0o644, b'189:518\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/urbnum', 0o644, b'743\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/descriptors', 0o644, b'\x12\x01\x00\x02\x00\x00\x00@C\x10\x12\x80\x00\x01\x01\x02\x00\x01\t\x02 \x00\x01\x01\x00\x802\t\x04\x00\x00\x02\x08\x06P\x00\x07\x05\x81\x02\x00\x02\x00\x07\x05\x02\x02\x00\x02\x00')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/speed', 0o644, b'480\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bmAttributes', 0o644, b'80\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/busnum', 0o644, b'5\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/uevent', 0o644, b'''MAJOR=189
MINOR=518
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=1043/8012/100
TYPE=0/0/0
BUSNUM=005
DEVNUM=007
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/manufacturer', 0o644, b'Generic\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/idVendor', 0o644, b'1043\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/bMaxPower', 0o644, b'100mA\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/device', '../../../5-1')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/dev', 0o644, b'252:24\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=24
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/usb_endpoint/usbdev5.7_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/active_duration', 0o644, b'65113\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/wakeup', 0o644, b'\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/connected_duration', 0o644, b'65113\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/persist', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/power/level', 0o644, b'on\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/ep_81', 'usb_endpoint/usbdev5.7_ep81')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/ep_02', 'usb_endpoint/usbdev5.7_ep02')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/driver', '../../../../../../bus/usb/drivers/usb-storage')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/modalias', 0o644, b'usb:v1043p8012d0100dc00dsc00dp00ic08isc06ip50\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bNumEndpoints', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bInterfaceSubClass', 0o644, b'06\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bInterfaceClass', 0o644, b'08\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=usb-storage
PRODUCT=1043/8012/100
TYPE=0/0/0
INTERFACE=8/6/80
MODALIAS=usb:v1043p8012d0100dc00dsc00dp00ic08isc06ip50
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/bInterfaceProtocol', 0o644, b'50\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/device', '../../../5-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/dev', 0o644, b'252:10\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=10
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/device', '../../../5-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/bEndpointAddress', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/dev', 0o644, b'252:11\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/uevent', 0o644, b'''MAJOR=252
MINOR=11
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/usb_endpoint/usbdev5.7_ep02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/subsystem', '../../../../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/subsystem', '../../../../../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/uevent', 0o644, b'DEVTYPE=scsi_target\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/generic', 'scsi_generic/sg2')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/subsystem', '../../../../../../../../../bus/scsi')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/driver', '../../../../../../../../../bus/scsi/drivers/sd')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/iodone_cnt', 0o644, b'0xc3\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/device_blocked', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/max_sectors', 0o644, b'240\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/modalias', 0o644, b'scsi:t-0x00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_level', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/queue_depth', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/rev', 0o644, b'1.00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/iocounterbits', 0o644, b'32\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/vendor', 0o644, b'Generic \n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/queue_type', 0o644, b'none\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/iorequest_cnt', 0o644, b'0xc3\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/evt_media_change', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/model', 0o644, b'USB Flash Drive \n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/ioerr_cnt', 0o644, b'0x2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/uevent', 0o644, b'''DEVTYPE=scsi_device
DRIVER=sd
MODALIAS=scsi:t-0x00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/timeout', 0o644, b'60\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/subsystem', '../../../../../../../../../../../class/scsi_disk')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/device', '../../../7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/app_tag_own', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/FUA', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/cache_type', 0o644, b'write through\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/protection_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/manage_start_stop', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/allow_restart', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_disk/7:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/subsystem', '../../../../../../../../../../../class/scsi_generic')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/device', '../../../7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/dev', 0o644, b'21:2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/uevent', 0o644, b'''MAJOR=21
MINOR=2
''')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_generic/sg2/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/subsystem', '../../../../../../../../../../../class/bsg')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/device', '../../../7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/dev', 0o644, b'254:2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/uevent', 0o644, b'''MAJOR=254
MINOR=2
''')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/bsg/7:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/subsystem', '../../../../../../../../../../../class/block')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/bdi', '../../../../../../../../../../virtual/bdi/8:16')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/device', '../../../7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/capability', 0o644, b'13\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/ro', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/size', 0o644, b'257024\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/dev', 0o644, b'8:16\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/range', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/removable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/stat', 0o644, b'     117      409     2103      272        0        0        0        0        0      194      272\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/uevent', 0o644, b'''MAJOR=8
MINOR=16
DEVTYPE=disk
''')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/bsg', '../../../bsg/7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/nr_requests', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/nomerges', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/scheduler', 0o644, b'noop anticipatory deadline [cfq] \n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/hw_sector_size', 0o644, b'512\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/max_hw_sectors_kb', 0o644, b'120\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/read_ahead_kb', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/max_sectors_kb', 0o644, b'120\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/slice_async_rq', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/back_seek_max', 0o644, b'16384\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/slice_sync', 0o644, b'100\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/slice_async', 0o644, b'40\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/fifo_expire_sync', 0o644, b'125\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/slice_idle', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/back_seek_penalty', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/fifo_expire_async', 0o644, b'250\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/queue/iosched/quantum', 0o644, b'4\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/subsystem', '../../../../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/start', 0o644, b'32\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/size', 0o644, b'256992\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/dev', 0o644, b'8:17\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/stat', 0o644, b'     109      392     1903      246        0        0        0        0        0      169      246\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/uevent', 0o644, b'''MAJOR=8
MINOR=17
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/block/sdb/sdb1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0/subsystem', '../../../../../../../../../../../class/scsi_device')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0/device', '../../../7:0:0:0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/target7:0:0/7:0:0:0/scsi_device/7:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/subsystem', '../../../../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/device', '../../../host7')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/unique_id', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/can_queue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/sg_tablesize', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/proc_name', 0o644, b'usb-storage\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-1/5-1:1.0/host7/scsi_host/host7/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/driver', '../../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/ep_00', 'usb_endpoint/usbdev5.9_ep00')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/version', 0o644, b' 2.00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/maxchild', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bDeviceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bcdDevice', 0o644, b'0100\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/devnum', 0o644, b'9\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/idProduct', 0o644, b'007b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/product', 0o644, b'N78\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/configuration', 0o644, b'Bulk transfer method configuration\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/dev', 0o644, b'189:520\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/urbnum', 0o644, b'20\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/descriptors', 0o644, b'''\x12\x01\x00\x02\x02\x00\x00@!\x04{\x00\x00\x01\x01\x02\x03\x01\t\x02\xab\x01\x0e\x01\x04\xc02\t\x04\x00\x00\x01\x02\x02\x01\x06\x05$\x00\x00\x01\x04$\x02\x0f\x05$\x06\x00\x01\x07\x05\x81\x03@\x00\x08\t\x04\x01\x00\x02
\x00\x00\x07\x07\x05\x82\x02\x00\x02\xff\x07\x05\x01\x02\x00\x02\xff\t\x04\x02\x00\x01\x02\x02\xff\x08\x05$\x00\x00\x01\x04$\x02\x0f\x05$\x06\x02\x03\x07\x05\x83\x03@\x00\x08\t\x04\x03\x00\x02
\x00\x00\t\x07\x05\x84\x02\x00\x02\xff\x07\x05\x02\x02\x00\x02\xff\t\x04\x04\x00\x00\x02\x08\x00\x00\x05$\x00\x10\x01\x05$\x11\x00\x01\r$\x06\x04\x05\x06\x07\x08\t
\x0b\x0c\r\t\x04\x05\x00\x00\x02\x0b\x00
\x05$\x00\x10\x01\x05$\x15\x00\x01\x05$\x06\x05\x06\t\x04\x06\x00\x00
\x00\x00\x00\t\x04\x06\x01\x02
\x00\x00\x00\x07\x05\x85\x02\x00\x02\xff\x07\x05\x03\x02\x00\x02\xff\t\x04\x07\x00\x00\x02\x0b\x00\x0b\x05$\x00\x10\x01\x05$\x15\x00\x01\x05$\x06\x07\x08\t\x04\x08\x00\x00
\x00\x00\x00\t\x04\x08\x01\x02
\x00\x00\x00\x07\x05\x86\x02\x00\x02\xff\x07\x05\x04\x02\x00\x02\xff\t\x04\t\x00\x00\x02\x08\x01\x00\x05$\x00\x10\x01\x05$\x08\x00\x01\x06$\x06\t
\x0b\t\x04
\x00\x00\x02\xfe\x00\x00\x05$\x00\x10\x01\x05$\xab\x05\x15\x05$\x06
\x0b\t\x04\x0b\x00\x00
\x00\x00\x00\t\x04\x0b\x01\x02
\x00\x00\x00\x04$\xfd\x01\x07\x05\x05\x02\x00\x02\xff\x07\x05\x87\x02\x00\x02\xff\t\x04\x0c\x00\x00\x02\xfd\x00\x0c\x05$\x00\x10\x01\x05$\xfc\x00\x01\x05$\x06\x0c\r\x04$\xfd\x00\t\x04\r\x00\x00
\x00\x00\x00\t\x04\r\x01\x02
\x00\x00\x00\x07\x05\x88\x03\x00\x02\x06\x07\x05\x06\x02\x00\x02\xff\t\x04\r\x02\x02
\x00\x00\x00\x07\x05\x88\x02\x00\x02\xff\x07\x05\x06\x02\x00\x02\xff''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/serial', 0o644, b'354172020305000\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/speed', 0o644, b'480\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bmAttributes', 0o644, b'c0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/busnum', 0o644, b'5\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/uevent', 0o644, b'''MAJOR=189
MINOR=520
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=421/7b/100
TYPE=2/0/0
BUSNUM=005
DEVNUM=009
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/manufacturer', 0o644, b'Nokia\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/idVendor', 0o644, b'0421\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bNumInterfaces', 0o644, b'14\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/bMaxPower', 0o644, b'100mA\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc08ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bInterfaceNumber', 0o644, b'04\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bInterfaceSubClass', 0o644, b'08\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/8/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc08ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.4/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/device', '../../../5-2')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/dev', 0o644, b'252:33\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=33
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/usb_endpoint/usbdev5.9_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/ep_83', 'usb_endpoint/usbdev5.9_ep83')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc02ipFF\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bInterfaceNumber', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/interface', 0o644, b'CDC Comms Interface\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bInterfaceSubClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/2/255
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc02ipFF
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/bInterfaceProtocol', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/device', '../../../5-2:1.2')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/interval', 0o644, b'16ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/bEndpointAddress', 0o644, b'83\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/dev', 0o644, b'252:30\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/uevent', 0o644, b'''MAJOR=252
MINOR=30
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/bInterval', 0o644, b'08\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/usb_endpoint/usbdev5.9_ep83/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.2/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/ep_01', 'usb_endpoint/usbdev5.9_ep01')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/ep_82', 'usb_endpoint/usbdev5.9_ep82')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/driver', '../../../../../../bus/usb/drivers/cdc_acm')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bInterfaceNumber', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bNumEndpoints', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/interface', 0o644, b'CDC Data Interface\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=cdc_acm
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/device', '../../../5-2:1.1')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/interval', 0o644, b'31875us\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/bEndpointAddress', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/dev', 0o644, b'252:29\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/uevent', 0o644, b'''MAJOR=252
MINOR=29
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep01/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/device', '../../../5-2:1.1')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/bEndpointAddress', 0o644, b'82\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/dev', 0o644, b'252:28\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/uevent', 0o644, b'''MAJOR=252
MINOR=28
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/usb_endpoint/usbdev5.9_ep82/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02iscFEip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bInterfaceNumber', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bInterfaceSubClass', 0o644, b'fe\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/254/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02iscFEip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.10/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc0Bip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bInterfaceNumber', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/interface', 0o644, b'PC Suite Services\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bInterfaceSubClass', 0o644, b'0b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/11/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc0Bip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.7/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/ep_81', 'usb_endpoint/usbdev5.9_ep81')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/driver', '../../../../../../bus/usb/drivers/cdc_acm')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc02ip01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bmCapabilities', 0o644, b'15')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/interface', 0o644, b'CDC Comms Interface\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bInterfaceSubClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=cdc_acm
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/2/1
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc02ip01
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/bInterfaceProtocol', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/device', '../../../5-2:1.0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/interval', 0o644, b'16ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/dev', 0o644, b'252:27\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=27
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/bInterval', 0o644, b'08\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/usb_endpoint/usbdev5.9_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/subsystem', '../../../../../../../../class/tty')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/device', '../../../5-2:1.0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/dev', 0o644, b'166:0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/uevent', 0o644, b'''MAJOR=166
MINOR=0
DEVNAME=ttyACM0
''')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.0/tty/ttyACM0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc08ip01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bInterfaceNumber', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bInterfaceSubClass', 0o644, b'08\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/8/1
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc08ip01
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/bInterfaceProtocol', 0o644, b'01\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.9/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bInterfaceNumber', 0o644, b'08\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.8/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/active_duration', 0o644, b'20902\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/wakeup', 0o644, b'\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/connected_duration', 0o644, b'20901\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/persist', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/power/level', 0o644, b'on\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bInterfaceNumber', 0o644, b'0d\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.13/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02isc0Bip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bInterfaceNumber', 0o644, b'05\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/interface', 0o644, b'SYNCML-SYNC\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bInterfaceSubClass', 0o644, b'0b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/11/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02isc0Bip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.5/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bInterfaceNumber', 0o644, b'0b\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.11/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic02iscFDip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bInterfaceNumber', 0o644, b'0c\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/interface', 0o644, b'LCIF_Alt0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bInterfaceSubClass', 0o644, b'fd\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bInterfaceClass', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=2/253/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic02iscFDip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.12/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/ep_02', 'usb_endpoint/usbdev5.9_ep02')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/ep_84', 'usb_endpoint/usbdev5.9_ep84')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bInterfaceNumber', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bNumEndpoints', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/interface', 0o644, b'CDC Data Interface\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/device', '../../../5-2:1.3')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/bEndpointAddress', 0o644, b'84\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/dev', 0o644, b'252:31\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/uevent', 0o644, b'''MAJOR=252
MINOR=31
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep84/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/device', '../../../5-2:1.3')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/interval', 0o644, b'31875us\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/bEndpointAddress', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/type', 0o644, b'Bulk\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/dev', 0o644, b'252:32\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/direction', 0o644, b'out\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/wMaxPacketSize', 0o644, b'0200\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/bmAttributes', 0o644, b'02\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/uevent', 0o644, b'''MAJOR=252
MINOR=32
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/usb_endpoint/usbdev5.9_ep02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/subsystem', '../../../../../../bus/usb')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/modalias', 0o644, b'usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bInterfaceNumber', 0o644, b'06\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bNumEndpoints', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bInterfaceClass', 0o644, b'0a\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/uevent', 0o644, b'''DEVTYPE=usb_interface
PRODUCT=421/7b/100
TYPE=2/0/0
INTERFACE=10/0/0
MODALIAS=usb:v0421p007Bd0100dc02dsc00dp00ic0Aisc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-2/5-2:1.6/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/power/active_duration', 0o644, b'547102\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/power/connected_duration', 0o644, b'8371932\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/power/level', 0o644, b'auto\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/ep_81', 'usb_endpoint/usbdev5.1_ep81')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/driver', '../../../../../bus/usb/drivers/hub')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/modalias', 0o644, b'usb:v1D6Bp0002d0206dc09dsc00dp00ic09isc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bInterfaceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=hub
PRODUCT=1d6b/2/206
TYPE=9/0/0
INTERFACE=9/0/0
MODALIAS=usb:v1D6Bp0002d0206dc09dsc00dp00ic09isc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/device', '../../../5-0:1.0')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/interval', 0o644, b'256ms\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/dev', 0o644, b'252:8\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/wMaxPacketSize', 0o644, b'0004\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=8
''')
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/bInterval', 0o644, b'0c\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/usb_endpoint/usbdev5.1_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb5/5-0:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1d.7/usb_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/subsystem', '../../../../../class/usb_host')
l('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/device', '../../../0000:00:1d.7')
f('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/companion', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.7/usb_host/usb_host5/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:0c')
l('sys/devices/pci0000:00/0000:00:1c.1/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.1/driver', '../../../bus/pci/drivers/pcieport-driver')
f('sys/devices/pci0000:00/0000:00:1c.1/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.1/enable', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.1/modalias', 0o644, b'pci:v00008086d000027D2sv00000000sd00000000bc06sc04i00\n')
f('sys/devices/pci0000:00/0000:00:1c.1/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000004000 0x0000000000005fff 0x0000000000000100
0x00000000ec000000 0x00000000edffffff 0x0000000000000200
0x00000000e4000000 0x00000000e40fffff 0x0000000000001201
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.1/config', 0o644, b"\x86\x80\xd2'\x07\x05\x10\x00\x02\x00\x04\x06\x10\x00\x81\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x03\x00@P\x00\x00\x00\xec\xf0\xed\x01\xe4\x01\xe4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x0b\x02\x04\x00\x10\x80A\x01\xc0\x0f\x00\x00\x00\x00\x11\x00\x11,\x11\x02B\x00\x110\xe0\xa0\x18\x00\x00\x00H\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x90\x01\x00\x0c0\xe0\xfe\x81A\x00\x00\x00\x00\x00\x00\r\xa0\x00\x00\xaa\x17\x11 \x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x11\x00\x00\x00\x00\x00\x00\x0f\xc7\x80\x06\x07\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00\x02\x00\x01\x18\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x80\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x00\x06\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x02\x02\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x01\xc0\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07`\x00`\x00\x00\x00\x00`\x02\x00\x00\x0c\x00\x06\x00\x00\x00\x001[`\xc9\xc0\x00p&u\xa2\x178\x00\xa1\x07G\x02\x16\x00\x00\x14\xb5\xbcJ\xbc\x0b*\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb4\x0c\xb4\x00\xc4\x0c\xb6\x00A\x08z\x00Y\x00\x1b\x00[\x00\x1d\x00\x01\x00\xba\x00\xe3\x0f\xd3\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x02\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1c.1/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.1/subsystem_device', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.1/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.1/irq', 0o644, b'509\n')
f('sys/devices/pci0000:00/0000:00:1c.1/device', 0o644, b'0x27d2\n')
f('sys/devices/pci0000:00/0000:00:1c.1/class', 0o644, b'0x060400\n')
f('sys/devices/pci0000:00/0000:00:1c.1/msi_bus', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/subsystem_vendor', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.1/uevent', 0o644, b'''DRIVER=pcieport-driver
PCI_CLASS=60400
PCI_ID=8086:27D2
PCI_SUBSYS_ID=0000:0000
PCI_SLOT_NAME=0000:00:1c.1
MODALIAS=pci:v00008086d000027D2sv00000000sd00000000bc06sc04i00
''')
f('sys/devices/pci0000:00/0000:00:1c.1/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.1/pci_bus', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/subsystem', '../../../../../class/pci_bus')
l('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/device', '../../../0000:00:1c.1')
f('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/pci_bus/0000:03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/subsystem', '../../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/driver', '../../../../bus/pci/drivers/iwl3945')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/rs_window', 0o644, b''' 54Mbs: 00000000000000000000000000000000000000000000000000000000000000
 48Mbs: 00000000000000000000000000000000000000000000000000000000000000
 36Mbs: 00000000000000000000000000000000000000000000000000000000000000
 24Mbs: 00000000000000000000000000000000000000000000000000000000000000
 18Mbs: 00000000000000000000000000000000000000000000000000000000000000
 12Mbs: 00000000000000000000000000000000000000000000000000000000000000
 11Mbs: 00000000000000000000000000000000000000000000000000000000000000
  9Mbs: 00000000000000000000000000000000000000000000000000000000000000
  6Mbs: 00000000000000000000000000000000000000000000000000000000000000
  5Mbs: 00000000000000000000000000000000000000000000000000000000000000
  2Mbs: 00000000000000000000000000000000000000000000000000000000000000
  1Mbs: 00000000000000000000000000000000000000000000000000000000000000

Average rate: 0Mbs
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/modalias', 0o644, b'pci:v00008086d00004227sv00008086sd00001011bc02sc80i00\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/resource', 0o644, b'''0x00000000edf00000 0x00000000edf00fff 0x0000000000020200
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/statistics', 0o644, b'''02 00 01 00 a4 c0 02 00 c6 27 01 00 80 16 01 00  .........\'......
e9 0c 00 00 00 00 00 00 00 00 00 00 69 05 00 00  ............i...
de 98 01 00 00 00 00 00 00 00 00 00 00 00 00 00  ................
00 00 00 00 00 00 00 00 bd 01 00 00 00 00 00 00  ................
fd 68 08 00 ae d3 06 00 78 65 04 00 b8 2b 00 00  .h......xe...+..
00 00 00 00 00 00 00 00 7e 42 02 00 4f 95 01 00  ........~B..O...
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................
00 00 00 00 da 01 00 00 00 00 00 00 00 00 00 00  ................
00 00 00 00 d8 dc 00 00 59 10 01 00 00 00 00 00  ........Y.......
df 0c 00 00 16 00 00 00 00 00 00 00 00 00 00 00  ................
88 00 00 00 00 00 00 00 04 00 00 00 00 00 00 00  ................
00 00 00 00 72 ff ff ff 00 00 00 00 00 00 00 00  ....r...........
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................
00 00 00 00 01 00 00 00 bb 45 98 35 1c 91 e7 f1  .........E.5....
4c 19 00 00 c2 0c 00 00 e7 ed 00 00 1c 4c 00 00  L............L..
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/config', 0o644, b'\x86\x80\'B\x06\x05\x10\x00\x02\x00\x80\x02\x10\x00\x00\x00\x00\x00\xf0\xed\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x80\x11\x10\x00\x00\x00\x00\xc8\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\xd0"H\x00\x00\x00\r\x05\xe0\x81\x00\x0c\x10\xe0\xfe\x00\x00\x00\x00\xd1A\x00\x00\x10\x00\x11\x00\xc0\x0e\x00\x00\x10\x08\n\x00\x11\x1c\x07\x00B\x01\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01\x14\x00\x00\x10\x00\x00\x00\x00\x00\x11 \x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14\x00\x00\x00\x01\x00\x00\x04\x0f\x1a\x00\x00\x00\x01\x01\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x01\x00\xd1\xaeu\xff\xff\xd2\x19\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/filter_flags', 0o644, b'0x0024\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/retry_rate', 0o644, b'1')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/channels', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/subsystem_device', 0o644, b'0x1011\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/tx_power', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/irq', 0o644, b'504\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/antenna', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/device', 0o644, b'0x4227\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/class', 0o644, b'0x028000\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/power_level', 0o644, b'6 (AC) OFF\n\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/flags', 0o644, b'0x8015\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/temperature', 0o644, b'-142\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/measurement', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/subsystem_vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/uevent', 0o644, b'''DRIVER=iwl3945
PCI_CLASS=28000
PCI_ID=8086:4227
PCI_SUBSYS_ID=8086:1011
PCI_SLOT_NAME=0000:03:00.0
MODALIAS=pci:v00008086d00004227sv00008086sd00001011bc02sc80i00
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/status', 0o644, b'0x000002e4\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/subsystem', '../../../../../../class/ieee80211')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/device', '../../../0000:03:00.0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/index', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/macaddress', 0o644, b'00:19:d2:75:ae:d1\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/ieee80211/phy0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/subsystem', '../../../../../../class/net')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/device', '../../../0000:03:00.0')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/phy80211', '../../ieee80211/phy0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/ifindex', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/features', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/iflink', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/dormant', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/dev_id', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/type', 0o644, b'801\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/operstate', 0o644, b'unknown\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/carrier', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/link_mode', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/tx_queue_len', 0o644, b'1000\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/flags', 0o644, b'0x1003\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/addr_len', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/address', 0o644, b'00:19:d2:75:ae:ff\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/uevent', 0o644, b'''INTERFACE=wmaster0
IFINDEX=3
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/mtu', 0o644, b'1500\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/broadcast', 0o644, b'ff:ff:ff:ff:ff:ff\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_aborted_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_crc_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_bytes', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_packets', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_carrier_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_length_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_missed_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_window_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/collisions', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_frame_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_packets', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/multicast', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/rx_over_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_bytes', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/statistics/tx_heartbeat_errors', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wmaster0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/subsystem', '../../../../../../class/net')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/device', '../../../0000:03:00.0')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/phy80211', '../../ieee80211/phy0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/ifindex', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/features', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/iflink', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/dormant', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/dev_id', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/type', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/operstate', 0o644, b'up\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/carrier', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/link_mode', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/tx_queue_len', 0o644, b'1000\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/flags', 0o644, b'0x1003\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/addr_len', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/address', 0o644, b'00:19:d2:75:ae:ff\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/uevent', 0o644, b'''INTERFACE=wlan0
IFINDEX=4
''')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/mtu', 0o644, b'1500\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/broadcast', 0o644, b'ff:ff:ff:ff:ff:ff\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_aborted_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_crc_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_bytes', 0o644, b'289996\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_packets', 0o644, b'1029\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_carrier_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_length_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_missed_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_window_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/collisions', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_frame_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_packets', 0o644, b'788\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/multicast', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/rx_over_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_bytes', 0o644, b'530013\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/statistics/tx_heartbeat_errors', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/beacon', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/link', 0o644, b'77\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/nwid', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/retries', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/fragment', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/misc', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/crypt', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/noise', 0o644, b'178\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/level', 0o644, b'199\n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/net/wlan0/wireless/status', 0o644, b'0x0\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/subsystem', '../../../../../../class/leds')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/device', '../../../0000:03:00.0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/trigger', 0o644, b'none AC-online BAT0-charging-or-full BAT0-charging BAT0-full rfkill0 phy0rx [phy0tx] phy0assoc phy0radio \n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/brightness', 0o644, b'0\n\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:TX/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/subsystem', '../../../../../../class/leds')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/device', '../../../0000:03:00.0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/trigger', 0o644, b'none AC-online BAT0-charging-or-full BAT0-charging BAT0-full rfkill0 phy0rx phy0tx [phy0assoc] phy0radio \n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/brightness', 0o644, b'255\n\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:assoc/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/subsystem', '../../../../../../class/leds')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/device', '../../../0000:03:00.0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/trigger', 0o644, b'none AC-online BAT0-charging-or-full BAT0-charging BAT0-full rfkill0 phy0rx phy0tx phy0assoc [phy0radio] \n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/brightness', 0o644, b'255\n\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:radio/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/subsystem', '../../../../../../class/leds')
l('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/device', '../../../0000:03:00.0')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/trigger', 0o644, b'none AC-online BAT0-charging-or-full BAT0-charging BAT0-full rfkill0 [phy0rx] phy0tx phy0assoc phy0radio \n')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/brightness', 0o644, b'0\n\x00')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:03:00.0/leds/iwl-phy0:RX/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.1/0000:00:1c.1:pcie00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:00.0', 0o755)
l('sys/devices/pci0000:00/0000:00:00.0/subsystem', '../../../bus/pci')
f('sys/devices/pci0000:00/0000:00:00.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:00.0/enable', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:00.0/modalias', 0o644, b'pci:v00008086d000027A0sv000017AAsd00002015bc06sc00i00\n')
f('sys/devices/pci0000:00/0000:00:00.0/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:00.0/config', 0o644, b"\x86\x80\xa0'\x06\x01\x90 \x03\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x15 \x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x90\xd1\xfe\x01@\xd1\xfe\x05\x00\x00\xf0\x01\x80\xd1\xfe\x00\x00\x02\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x11\x11\x000\x11\x11\x00\xff\x03\x00\x00\x80\x1a9\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04\x01\x00\x00\t\x00\tQJ\x00J\xb0\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x05\x00\x10\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:00.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:00.0/subsystem_device', 0o644, b'0x2015\n')
f('sys/devices/pci0000:00/0000:00:00.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:00.0/irq', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:00.0/device', 0o644, b'0x27a0\n')
f('sys/devices/pci0000:00/0000:00:00.0/class', 0o644, b'0x060000\n')
f('sys/devices/pci0000:00/0000:00:00.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:00.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:00.0/uevent', 0o644, b'''PCI_CLASS=60000
PCI_ID=8086:27A0
PCI_SUBSYS_ID=17AA:2015
PCI_SLOT_NAME=0000:00:00.0
MODALIAS=pci:v00008086d000027A0sv000017AAsd00002015bc06sc00i00
''')
f('sys/devices/pci0000:00/0000:00:00.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:00.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:00.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.3', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.3/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:0f')
l('sys/devices/pci0000:00/0000:00:1c.3/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.3/driver', '../../../bus/pci/drivers/pcieport-driver')
f('sys/devices/pci0000:00/0000:00:1c.3/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.3/enable', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.3/modalias', 0o644, b'pci:v00008086d000027D6sv00000000sd00000000bc06sc04i00\n')
f('sys/devices/pci0000:00/0000:00:1c.3/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000008000 0x0000000000009fff 0x0000000000000100
0x00000000ea000000 0x00000000ebffffff 0x0000000000000200
0x00000000e4200000 0x00000000e42fffff 0x0000000000001201
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.3/config', 0o644, b"\x86\x80\xd6'\x07\x05\x10\x00\x02\x00\x04\x06\x10\x00\x81\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\x13\x00\x80\x90\x00\x00\x00\xea\xf0\xeb!\xe4!\xe4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x0b\x04\x04\x00\x10\x80A\x01\xc0\x0f\x00\x00\x00\x00\x10\x00\x11L\x11\x04\x00\x00\x01\x10\xe0\xa0(\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x90\x01\x00\x0c0\xe0\xfe\x91A\x00\x00\x00\x00\x00\x00\r\xa0\x00\x00\xaa\x17\x11 \x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x11\x00\x00\x00\x00\x00\x00\x00\xc7\x00\x06\x07\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00\x02\x00\x01\x18\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xff\x00\x00\x80\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x02\x04\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x01\xc0\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07`\x00`\x00\x00\x00\x00`\x02\x00\x00\x0c\x00\x06\x00\x00\x00\x00\x00[`\xc9\xc0\x00p&u\x00\x10\x00\x00\xff\x0f\x00\x00\x16\x00\x00\x14\xb5\xbcJ\xbc\x0b*\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x02\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1c.3/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.3/subsystem_device', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.3/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.3/irq', 0o644, b'507\n')
f('sys/devices/pci0000:00/0000:00:1c.3/device', 0o644, b'0x27d6\n')
f('sys/devices/pci0000:00/0000:00:1c.3/class', 0o644, b'0x060400\n')
f('sys/devices/pci0000:00/0000:00:1c.3/msi_bus', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.3/subsystem_vendor', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.3/uevent', 0o644, b'''DRIVER=pcieport-driver
PCI_CLASS=60400
PCI_ID=8086:27D6
PCI_SUBSYS_ID=0000:0000
PCI_SLOT_NAME=0000:00:1c.3
MODALIAS=pci:v00008086d000027D6sv00000000sd00000000bc06sc04i00
''')
f('sys/devices/pci0000:00/0000:00:1c.3/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.3/pci_bus', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/subsystem', '../../../../../class/pci_bus')
l('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/device', '../../../0000:00:1c.3')
f('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.3/pci_bus/0000:0c/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.3/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.3/0000:00:1c.3:pcie00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.0/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:01')
l('sys/devices/pci0000:00/0000:00:1f.0/subsystem', '../../../bus/pci')
f('sys/devices/pci0000:00/0000:00:1f.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1f.0/enable', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.0/modalias', 0o644, b'pci:v00008086d000027B9sv000017AAsd00002009bc06sc01i00\n')
f('sys/devices/pci0000:00/0000:00:1f.0/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1f.0/config', 0o644, b'''\x86\x80\xb9\'\x07\x00\x10\x02\x02\x00\x01\x06\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\t \x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x10\x00\x00\x80\x00\x00\x00\x81\x11\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x8b\x8b\x8b\x8b\x92\x00\x00\x00\x8b\x8b\x8b\x8b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00
\x1f\x01\x16|\x00\xe1\x15\x0c\x00\x81\x16\x1c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00$\x06\x02\x00\x00\x00\x00\x00\x13\x1c
\x00\x00\x03\x00\x00\x00\x00\xf0\x00\x00\x00\x00\x00\x00\x00\x12
\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00D\x12\x11\x00gE\x00\x00\xc0\xff\x00\x00\x01\x00\x00\x00\t\x00\x0c\x10\xb4\x02$\x17\x00\x00\x00\x00\x00\x00\x00\x00\x01\xc0\xd1\xfe\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00''')
f('sys/devices/pci0000:00/0000:00:1f.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1f.0/subsystem_device', 0o644, b'0x2009\n')
f('sys/devices/pci0000:00/0000:00:1f.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1f.0/irq', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.0/device', 0o644, b'0x27b9\n')
f('sys/devices/pci0000:00/0000:00:1f.0/class', 0o644, b'0x060100\n')
f('sys/devices/pci0000:00/0000:00:1f.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1f.0/uevent', 0o644, b'''PCI_CLASS=60100
PCI_ID=8086:27B9
PCI_SUBSYS_ID=17AA:2009
PCI_SLOT_NAME=0000:00:1f.0
MODALIAS=pci:v00008086d000027B9sv000017AAsd00002009bc06sc01i00
''')
f('sys/devices/pci0000:00/0000:00:1f.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:0b')
l('sys/devices/pci0000:00/0000:00:1c.0/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.0/driver', '../../../bus/pci/drivers/pcieport-driver')
f('sys/devices/pci0000:00/0000:00:1c.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.0/enable', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.0/modalias', 0o644, b'pci:v00008086d000027D0sv00000000sd00000000bc06sc04i00\n')
f('sys/devices/pci0000:00/0000:00:1c.0/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000003000 0x0000000000003fff 0x0000000000000100
0x00000000ee000000 0x00000000ee0fffff 0x0000000000000200
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.0/config', 0o644, b"\x86\x80\xd0'\x07\x05\x10\x00\x02\x00\x04\x06\x10\x00\x81\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x02\x0000\x00\x00\x00\xee\x00\xee\xf1\xff\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x04\x00\x10\x80A\x01\xc0\x0f\x00\x00\x00\x00\x11\x00\x11,\x11\x01B\x00\x110\xe0\xa0\x10\x00(\x00@\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x90\x01\x00\x0c0\xe0\xfeyA\x00\x00\x00\x00\x00\x00\r\xa0\x00\x00\xaa\x17\x11 \x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x11\xc0\x00\x00\x00\x00\x00\x0f\xc7\x00\x06\x07\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00\x02\x00\x01\x18\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x80\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x11\x00\x06\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x02\x01\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x01\xc0\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07`\x00`\x00\x00\x00\x00`\x02\x00\x00\x0c\x00\x06\x00\x00\x00\x00\x08[`\xc9\xc0\x00p&u:\x188\x009\x08\x9e\x0e\x16\x00\x00\x14\xb5\xbcJ\xbc\x0b*\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xec\t\xec\x00\xfc\t\xee\x00c\x00W\x00i\x00J\x00k\x00L\x00\x01\x00\x05\x00\x10\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x02\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1c.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.0/subsystem_device', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.0/irq', 0o644, b'510\n')
f('sys/devices/pci0000:00/0000:00:1c.0/device', 0o644, b'0x27d0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/class', 0o644, b'0x060400\n')
f('sys/devices/pci0000:00/0000:00:1c.0/msi_bus', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.0/subsystem_vendor', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:1c.0/uevent', 0o644, b'''DRIVER=pcieport-driver
PCI_CLASS=60400
PCI_ID=8086:27D0
PCI_SUBSYS_ID=0000:0000
PCI_SLOT_NAME=0000:00:1c.0
MODALIAS=pci:v00008086d000027D0sv00000000sd00000000bc06sc04i00
''')
f('sys/devices/pci0000:00/0000:00:1c.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.0/pci_bus', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/subsystem', '../../../../../class/pci_bus')
l('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/device', '../../../0000:00:1c.0')
f('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/pci_bus/0000:02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/subsystem', '../../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/driver', '../../../../bus/pci/drivers/e1000e')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/modalias', 0o644, b'pci:v00008086d0000109Asv000017AAsd00002001bc02sc00i00\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/resource', 0o644, b'''0x00000000ee000000 0x00000000ee01ffff 0x0000000000020200
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000003000 0x000000000000301f 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/config', 0o644, b'\x86\x80\x9a\x10\x07\x05\x10\x00\x00\x00\x00\x02\x10\x00\x00\x00\x00\x00\x00\xee\x00\x00\x00\x00\x010\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x01 \x00\x00\x00\x00\xc8\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\xd0"\xc8\x00 \x00\x0f\x05\xe0\x81\x00\x0c\x10\xe0\xfe\x00\x00\x00\x00\xc9A\x00\x00\x10\x00\x01\x00\xc1\x0c\x00\x00\x10(\x1a\x00\x11\x1c\x07\x00@\x01\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01\x14\x00\x00\x10\x00\x00\x00\x00\x00\x11 \x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14\x00\x00\x00\x01\x00\x00\x04\x0f\x03\x00\x00\x00\x01\x01\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x01\x00\xc7\x8d\xe2\xff\xffA\x16\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/resource2', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/subsystem_device', 0o644, b'0x2001\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/irq', 0o644, b'505\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/device', 0o644, b'0x109a\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/class', 0o644, b'0x020000\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/uevent', 0o644, b'''DRIVER=e1000e
PCI_CLASS=20000
PCI_ID=8086:109A
PCI_SUBSYS_ID=17AA:2001
PCI_SLOT_NAME=0000:02:00.0
MODALIAS=pci:v00008086d0000109Asv000017AAsd00002001bc02sc00i00
''')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net', 0o755)
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/subsystem', '../../../../../../class/net')
l('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/device', '../../../0000:02:00.0')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/ifindex', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/features', 0o644, b'0x111ba9\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/iflink', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/dormant', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/dev_id', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/type', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/operstate', 0o644, b'down\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/carrier', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/link_mode', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/tx_queue_len', 0o644, b'1000\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/flags', 0o644, b'0x1003\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/addr_len', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/address', 0o644, b'00:16:41:e2:8d:ff\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/uevent', 0o644, b'''INTERFACE=eth0
IFINDEX=2
''')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/mtu', 0o644, b'1500\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/broadcast', 0o644, b'ff:ff:ff:ff:ff:ff\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_aborted_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_crc_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_bytes', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_packets', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_carrier_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_length_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_dropped', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_fifo_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_missed_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_window_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/collisions', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_frame_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_packets', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_compressed', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/multicast', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/rx_over_errors', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_bytes', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/statistics/tx_heartbeat_errors', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:02:00.0/net/eth0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02', 0o755)
l('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1c.0/0000:00:1c.0:pcie02/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.3', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.3/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:18')
l('sys/devices/pci0000:00/0000:00:1f.3/subsystem', '../../../bus/pci')
f('sys/devices/pci0000:00/0000:00:1f.3/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1f.3/enable', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.3/modalias', 0o644, b'pci:v00008086d000027DAsv000017AAsd0000200Fbc0Csc05i00\n')
f('sys/devices/pci0000:00/0000:00:1f.3/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x00000000000018e0 0x00000000000018ff 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1f.3/config', 0o644, b"\x86\x80\xda'\x01\x01\x80\x02\x02\x00\x05\x0c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe1\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x0f \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1f.3/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1f.3/subsystem_device', 0o644, b'0x200f\n')
f('sys/devices/pci0000:00/0000:00:1f.3/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1f.3/irq', 0o644, b'11\n')
f('sys/devices/pci0000:00/0000:00:1f.3/device', 0o644, b'0x27da\n')
f('sys/devices/pci0000:00/0000:00:1f.3/class', 0o644, b'0x0c0500\n')
f('sys/devices/pci0000:00/0000:00:1f.3/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.3/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1f.3/uevent', 0o644, b'''PCI_CLASS=C0500
PCI_ID=8086:27DA
PCI_SUBSYS_ID=17AA:200F
PCI_SLOT_NAME=0000:00:1f.3
MODALIAS=pci:v00008086d000027DAsv000017AAsd0000200Fbc0Csc05i00
''')
f('sys/devices/pci0000:00/0000:00:1f.3/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.3/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:19')
l('sys/devices/pci0000:00/0000:00:1d.0/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1d.0/driver', '../../../bus/pci/drivers/uhci_hcd')
f('sys/devices/pci0000:00/0000:00:1d.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1d.0/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/modalias', 0o644, b'pci:v00008086d000027C8sv000017AAsd0000200Abc0Csc03i00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000001800 0x000000000000181f 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1d.0/config', 0o644, b"\x86\x80\xc8'\x05\x00\x80\x02\x02\x00\x03\x0c\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\n \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1d.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.0/subsystem_device', 0o644, b'0x200a\n')
f('sys/devices/pci0000:00/0000:00:1d.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1d.0/irq', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1d.0/pools', 0o644, b'''poolinfo - 0.1
uhci_qh            11   32  128  1
uhci_td             1   64   64  1
buffer-2048         0    0 2048  0
buffer-512          0    0  512  0
buffer-128          0    0  128  0
buffer-32           1  128   32  1
''')
f('sys/devices/pci0000:00/0000:00:1d.0/device', 0o644, b'0x27c8\n')
f('sys/devices/pci0000:00/0000:00:1d.0/class', 0o644, b'0x0c0300\n')
f('sys/devices/pci0000:00/0000:00:1d.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1d.0/uevent', 0o644, b'''DRIVER=uhci_hcd
PCI_CLASS=C0300
PCI_ID=8086:27C8
PCI_SUBSYS_ID=17AA:200A
PCI_SLOT_NAME=0000:00:1d.0
MODALIAS=pci:v00008086d000027C8sv000017AAsd0000200Abc0Csc03i00
''')
f('sys/devices/pci0000:00/0000:00:1d.0/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/subsystem', '../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/driver', '../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/ep_00', 'usb_endpoint/usbdev1.1_ep00')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/version', 0o644, b' 1.10\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/authorized_default', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/maxchild', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bDeviceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bcdDevice', 0o644, b'0206\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/devnum', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/idProduct', 0o644, b'0001\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/product', 0o644, b'UHCI Host Controller\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/dev', 0o644, b'189:0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/urbnum', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/descriptors', 0o644, b'\x12\x01\x10\x01\t\x00\x00@k\x1d\x01\x00\x06\x02\x03\x02\x01\x01\t\x02\x19\x00\x01\x01\x00\xe0\x00\t\x04\x00\x00\x01\t\x00\x00\x00\x07\x05\x81\x03\x02\x00\xff')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/serial', 0o644, b'0000:00:1d.0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/speed', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bmAttributes', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/busnum', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/uevent', 0o644, b'''MAJOR=189
MINOR=0
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=1d6b/1/206
TYPE=9/0/0
BUSNUM=001
DEVNUM=001
''')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/manufacturer', 0o644, b'Linux 2.6.27-rc7-00106-g6ef190c-dirty uhci_hcd\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/idVendor', 0o644, b'1d6b\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/bMaxPower', 0o644, b'  0mA\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/subsystem', '../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/device', '../../../usb1')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/dev', 0o644, b'252:1\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=1
''')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/usb_endpoint/usbdev1.1_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/ep_81', 'usb_endpoint/usbdev1.1_ep81')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/driver', '../../../../../bus/usb/drivers/hub')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/modalias', 0o644, b'usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bInterfaceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=hub
PRODUCT=1d6b/1/206
TYPE=9/0/0
INTERFACE=9/0/0
MODALIAS=usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/device', '../../../1-0:1.0')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/interval', 0o644, b'255ms\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/dev', 0o644, b'252:0\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/wMaxPacketSize', 0o644, b'0002\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=0
''')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/usb_endpoint/usbdev1.1_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/1-0:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/power/active_duration', 0o644, b'2532\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/power/connected_duration', 0o644, b'8372376\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.0/usb1/power/level', 0o644, b'auto\n')
d('sys/devices/pci0000:00/0000:00:1d.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1d.0/usb_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1/subsystem', '../../../../../class/usb_host')
l('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1/device', '../../../0000:00:1d.0')
f('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.0/usb_host/usb_host1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:16')
l('sys/devices/pci0000:00/0000:00:1f.2/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1f.2/driver', '../../../bus/pci/drivers/ahci')
f('sys/devices/pci0000:00/0000:00:1f.2/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1f.2/resource3', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/modalias', 0o644, b'pci:v00008086d000027C5sv000017AAsd0000200Dbc01sc06i01\n')
f('sys/devices/pci0000:00/0000:00:1f.2/resource', 0o644, b'''0x00000000000018c8 0x00000000000018cf 0x0000000000020101
0x00000000000018ac 0x00000000000018af 0x0000000000020101
0x00000000000018c0 0x00000000000018c7 0x0000000000020101
0x00000000000018a8 0x00000000000018ab 0x0000000000020101
0x00000000000018b0 0x00000000000018bf 0x0000000000020101
0x00000000ee404400 0x00000000ee4047ff 0x0000000000020200
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1f.2/config', 0o644, b"\x86\x80\xc5'\x07\x04\xb0\x02\x02\x01\x06\x01\x00\x00\x00\x00\xc9\x18\x00\x00\xad\x18\x00\x00\xc1\x18\x00\x00\xa9\x18\x00\x00\xb1\x18\x00\x00\x00D@\xee\x00\x00\x00\x00\xaa\x17\r \x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x0b\x02\x00\x00\x07\xa3\x00\x80\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05p\x01\x00\x0c0\xe0\xfe\x99A\x00\x00\x00\x00\x00\x00@\x00\x11\x10\x80\x01\x00J\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1f.2/resource1', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/resource2', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1f.2/subsystem_device', 0o644, b'0x200d\n')
f('sys/devices/pci0000:00/0000:00:1f.2/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1f.2/irq', 0o644, b'506\n')
f('sys/devices/pci0000:00/0000:00:1f.2/device', 0o644, b'0x27c5\n')
f('sys/devices/pci0000:00/0000:00:1f.2/class', 0o644, b'0x010601\n')
f('sys/devices/pci0000:00/0000:00:1f.2/resource5', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1f.2/uevent', 0o644, b'''DRIVER=ahci
PCI_CLASS=10601
PCI_ID=8086:27C5
PCI_SUBSYS_ID=17AA:200D
PCI_SLOT_NAME=0000:00:1f.2
MODALIAS=pci:v00008086d000027C5sv000017AAsd0000200Dbc01sc06i01
''')
f('sys/devices/pci0000:00/0000:00:1f.2/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/subsystem', '../../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/uevent', 0o644, b'DEVTYPE=scsi_target\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/generic', 'scsi_generic/sg0')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/subsystem', '../../../../../../bus/scsi')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/driver', '../../../../../../bus/scsi/drivers/sd')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/iodone_cnt', 0o644, b'0x6877\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/device_blocked', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/modalias', 0o644, b'scsi:t-0x00\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_level', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/queue_depth', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/rev', 0o644, b'4.06\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/sw_activity', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/whitespace_test', 0o644, b'WHITE  SPACE   \n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/iocounterbits', 0o644, b'32\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/vendor', 0o644, b'ATA     \n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/queue_type', 0o644, b'none\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/iorequest_cnt', 0o644, b'0x6877\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/evt_media_change', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/model', 0o644, b'ST910021AS      \n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/ioerr_cnt', 0o644, b'0x6\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/uevent', 0o644, b'''DEVTYPE=scsi_device
DRIVER=sd
MODALIAS=scsi:t-0x00
''')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/timeout', 0o644, b'60\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/subsystem', '../../../../../../../../class/scsi_disk')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/device', '../../../0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/app_tag_own', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/FUA', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/cache_type', 0o644, b'write back\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/protection_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/manage_start_stop', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/allow_restart', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_disk/0:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/subsystem', '../../../../../../../../class/scsi_generic')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/device', '../../../0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/dev', 0o644, b'21:0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/uevent', 0o644, b'''MAJOR=21
MINOR=0
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_generic/sg0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/subsystem', '../../../../../../../../class/bsg')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/device', '../../../0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/dev', 0o644, b'254:0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/uevent', 0o644, b'''MAJOR=254
MINOR=0
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/bsg/0:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/subsystem', '../../../../../../../../class/block')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/bdi', '../../../../../../../virtual/bdi/8:0')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/device', '../../../0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/capability', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/ro', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/test:colon+plus', 0o644, b'colon\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/size', 0o644, b'195371568\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/dev', 0o644, b'8:0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/range', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/removable', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/stat', 0o644, b'   12939     5801   590192   194190    13755    63323   617056  2678026        0   114408  2872209\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/uevent', 0o644, b'''MAJOR=8
MINOR=0
DEVTYPE=disk
DEVNAME=sda
''')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/test_empty_file', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/bsg', '../../../bsg/0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/nr_requests', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/nomerges', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/scheduler', 0o644, b'noop anticipatory deadline [cfq] \n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/hw_sector_size', 0o644, b'512\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/max_hw_sectors_kb', 0o644, b'32767\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/read_ahead_kb', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/max_sectors_kb', 0o644, b'512\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/slice_async_rq', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/back_seek_max', 0o644, b'16384\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/slice_sync', 0o644, b'100\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/slice_async', 0o644, b'40\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/fifo_expire_sync', 0o644, b'125\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/slice_idle', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/back_seek_penalty', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/fifo_expire_async', 0o644, b'250\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/queue/iosched/quantum', 0o644, b'4\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/start', 0o644, b'190820133\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/size', 0o644, b'4546332\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/dev', 0o644, b'8:10\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/stat', 0o644, b'      42       43      384      417        0        0        0        0        0      286      417\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/uevent', 0o644, b'''MAJOR=8
MINOR=10
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda10/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/start', 0o644, b'143942463\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/size', 0o644, b'46877607\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/dev', 0o644, b'8:9\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/stat', 0o644, b'      70      322      896      800        3        0       40      240        0      945     1039\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/uevent', 0o644, b'''MAJOR=8
MINOR=9
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/holders', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/holders/md0', '../../../../../../../../../virtual/block/md0')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda9/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/start', 0o644, b'104872383\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/size', 0o644, b'19534977\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/dev', 0o644, b'8:7\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/stat', 0o644, b'      35      263      298      419        0        0        0        0        0      298      419\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/uevent', 0o644, b'''MAJOR=8
MINOR=7
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda7/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/start', 0o644, b'63\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/size', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/dev', 0o644, b'8:1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/stat', 0o644, b'       2        0        4       36        0        0        0        0        0       36       36\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/uevent', 0o644, b'''MAJOR=8
MINOR=1
DEVTYPE=partition
DEVNAME=sda1
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/start', 0o644, b'124407423\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/size', 0o644, b'19534977\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/dev', 0o644, b'8:8\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/stat', 0o644, b'      34      264      298      498        0        0        0        0        0      368      498\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/uevent', 0o644, b'''MAJOR=8
MINOR=8
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda8/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/start', 0o644, b'126\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/size', 0o644, b'62926479\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/dev', 0o644, b'8:5\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/stat', 0o644, b'   11087     4819   566626   169007     2951     4548    59992   129572        0    65249   298573\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/uevent', 0o644, b'''MAJOR=8
MINOR=5
DEVTYPE=partition
DEVNAME=sda5
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda5/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/subsystem', '../../../../../../../../../class/block')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/start', 0o644, b'62926668\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/size', 0o644, b'41945652\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/dev', 0o644, b'8:6\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/stat', 0o644, b'    1648       67    21334    22824    10801    58775   557024  2548214        0    51231  2571037\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/uevent', 0o644, b'''MAJOR=8
MINOR=6
DEVTYPE=partition
''')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/block/sda/sda6/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0/subsystem', '../../../../../../../../class/scsi_device')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0/device', '../../../0:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/target0:0:0/0:0:0:0/scsi_device/0:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/device', '../../../host0')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/unique_id', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/can_queue', 0o644, b'31\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/sg_tablesize', 0o644, b'168\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/link_power_management_policy', 0o644, b'medium_power\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/proc_name', 0o644, b'ahci\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/em_message', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/em_message_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host0/scsi_host/host0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host1', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host1/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/device', '../../../host1')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/unique_id', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/can_queue', 0o644, b'31\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/sg_tablesize', 0o644, b'168\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/link_power_management_policy', 0o644, b'max_performance\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/proc_name', 0o644, b'ahci\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/em_message', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/em_message_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host1/scsi_host/host1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host2', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host2/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host2/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/device', '../../../host2')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/unique_id', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/can_queue', 0o644, b'31\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/sg_tablesize', 0o644, b'168\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/link_power_management_policy', 0o644, b'max_performance\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/proc_name', 0o644, b'ahci\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/em_message', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/em_message_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host2/scsi_host/host2/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host3', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host3/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/device', '../../../host3')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/unique_id', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/can_queue', 0o644, b'31\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/sg_tablesize', 0o644, b'168\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/link_power_management_policy', 0o644, b'max_performance\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/proc_name', 0o644, b'ahci\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/em_message', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/em_message_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.2/host3/scsi_host/host3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:1d')
l('sys/devices/pci0000:00/0000:00:1d.2/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1d.2/driver', '../../../bus/pci/drivers/uhci_hcd')
f('sys/devices/pci0000:00/0000:00:1d.2/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1d.2/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/modalias', 0o644, b'pci:v00008086d000027CAsv000017AAsd0000200Abc0Csc03i00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000001840 0x000000000000185f 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1d.2/config', 0o644, b"\x86\x80\xca'\x05\x00\x80\x02\x02\x00\x03\x0c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00A\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\n \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1d.2/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1d.2/subsystem_device', 0o644, b'0x200a\n')
f('sys/devices/pci0000:00/0000:00:1d.2/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1d.2/irq', 0o644, b'18\n')
f('sys/devices/pci0000:00/0000:00:1d.2/pools', 0o644, b'''poolinfo - 0.1
uhci_qh            13   32  128  1
uhci_td             4   64   64  1
buffer-2048         0    0 2048  0
buffer-512          0    0  512  0
buffer-128          3   32  128  1
buffer-32           2  128   32  1
''')
f('sys/devices/pci0000:00/0000:00:1d.2/device', 0o644, b'0x27ca\n')
f('sys/devices/pci0000:00/0000:00:1d.2/class', 0o644, b'0x0c0300\n')
f('sys/devices/pci0000:00/0000:00:1d.2/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.2/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1d.2/uevent', 0o644, b'''DRIVER=uhci_hcd
PCI_CLASS=C0300
PCI_ID=8086:27CA
PCI_SUBSYS_ID=17AA:200A
PCI_SLOT_NAME=0000:00:1d.2
MODALIAS=pci:v00008086d000027CAsv000017AAsd0000200Abc0Csc03i00
''')
f('sys/devices/pci0000:00/0000:00:1d.2/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.2/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1d.2/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3/subsystem', '../../../../../class/usb_host')
l('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3/device', '../../../0000:00:1d.2')
f('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb_host/usb_host3/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/subsystem', '../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/driver', '../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/ep_00', 'usb_endpoint/usbdev3.1_ep00')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/version', 0o644, b' 1.10\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/authorized_default', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/maxchild', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bDeviceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bcdDevice', 0o644, b'0206\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/devnum', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/idProduct', 0o644, b'0001\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/product', 0o644, b'UHCI Host Controller\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/dev', 0o644, b'189:256\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/urbnum', 0o644, b'68\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/descriptors', 0o644, b'\x12\x01\x10\x01\t\x00\x00@k\x1d\x01\x00\x06\x02\x03\x02\x01\x01\t\x02\x19\x00\x01\x01\x00\xe0\x00\t\x04\x00\x00\x01\t\x00\x00\x00\x07\x05\x81\x03\x02\x00\xff')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/serial', 0o644, b'0000:00:1d.2\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/speed', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bMaxPacketSize0', 0o644, b'64\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bmAttributes', 0o644, b'e0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/busnum', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/uevent', 0o644, b'''MAJOR=189
MINOR=256
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=1d6b/1/206
TYPE=9/0/0
BUSNUM=003
DEVNUM=001
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/manufacturer', 0o644, b'Linux 2.6.27-rc7-00106-g6ef190c-dirty uhci_hcd\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/idVendor', 0o644, b'1d6b\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/bMaxPower', 0o644, b'  0mA\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/subsystem', '../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/device', '../../../usb3')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/dev', 0o644, b'252:5\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/wMaxPacketSize', 0o644, b'0040\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=5
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/usb_endpoint/usbdev3.1_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/ep_81', 'usb_endpoint/usbdev3.1_ep81')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/driver', '../../../../../bus/usb/drivers/hub')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/modalias', 0o644, b'usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bInterfaceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bInterfaceClass', 0o644, b'09\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=hub
PRODUCT=1d6b/1/206
TYPE=9/0/0
INTERFACE=9/0/0
MODALIAS=usb:v1D6Bp0001d0206dc09dsc00dp00ic09isc00ip00
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/bInterfaceProtocol', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/device', '../../../3-0:1.0')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/interval', 0o644, b'255ms\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/dev', 0o644, b'252:4\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/wMaxPacketSize', 0o644, b'0002\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=4
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/bInterval', 0o644, b'ff\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/usb_endpoint/usbdev3.1_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-0:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/power/active_duration', 0o644, b'603809\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/power/connected_duration', 0o644, b'8372166\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/power/level', 0o644, b'auto\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/subsystem', '../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/driver', '../../../../../bus/usb/drivers/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/ep_00', 'usb_endpoint/usbdev3.4_ep00')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/version', 0o644, b' 2.00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/maxchild', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bDeviceClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bcdDevice', 0o644, b'2000\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/devnum', 0o644, b'4\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bDeviceProtocol', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/idProduct', 0o644, b'c03e\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/product', 0o644, b'USB-PS/2 Optical Mouse\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bDeviceSubClass', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/configuration', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/dev', 0o644, b'189:259\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/urbnum', 0o644, b'12\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/authorized', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/descriptors', 0o644, b'\x12\x01\x00\x02\x00\x00\x00\x08m\x04>\xc0\x00 \x01\x02\x00\x01\t\x02"\x00\x01\x01\x00\xa01\t\x04\x00\x00\x01\x03\x01\x02\x00\t!\x10\x01\x00\x01"2\x00\x07\x05\x81\x03\x04\x00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bNumConfigurations', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/speed', 0o644, b'1.5\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bMaxPacketSize0', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bmAttributes', 0o644, b'a0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/busnum', 0o644, b'3\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/quirks', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bConfigurationValue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/uevent', 0o644, b'''MAJOR=189
MINOR=259
DEVTYPE=usb_device
DRIVER=usb
PRODUCT=46d/c03e/2000
TYPE=0/0/0
BUSNUM=003
DEVNUM=004
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/manufacturer', 0o644, b'Logitech\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/idVendor', 0o644, b'046d\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bNumInterfaces', 0o644, b' 1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/bMaxPower', 0o644, b' 98mA\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/subsystem', '../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/device', '../../../3-1')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/interval', 0o644, b'0ms\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/bEndpointAddress', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/type', 0o644, b'Control\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/dev', 0o644, b'252:26\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/direction', 0o644, b'both\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/wMaxPacketSize', 0o644, b'0008\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/bmAttributes', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/uevent', 0o644, b'''MAJOR=252
MINOR=26
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/bInterval', 0o644, b'00\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/usb_endpoint/usbdev3.4_ep00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/active_duration', 0o644, b'51384\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/wakeup', 0o644, b'enabled\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/connected_duration', 0o644, b'51383\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/autosuspend', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/persist', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/power/level', 0o644, b'on\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/subsystem', '../../../../../../bus/usb')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/ep_81', 'usb_endpoint/usbdev3.4_ep81')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/driver', '../../../../../../bus/usb/drivers/usbhid')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/modalias', 0o644, b'usb:v046DpC03Ed2000dc00dsc00dp00ic03isc01ip02\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bInterfaceNumber', 0o644, b'00\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bNumEndpoints', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bInterfaceSubClass', 0o644, b'01\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bAlternateSetting', 0o644, b' 0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bInterfaceClass', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/uevent', 0o644, b'''DEVTYPE=usb_interface
DRIVER=usbhid
PRODUCT=46d/c03e/2000
TYPE=0/0/0
INTERFACE=3/1/2
MODALIAS=usb:v046DpC03Ed2000dc00dsc00dp00ic03isc01ip02
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/bInterfaceProtocol', 0o644, b'02\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/subsystem', '../../../../../../../../class/usb_endpoint')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/device', '../../../3-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/interval', 0o644, b'10ms\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/bEndpointAddress', 0o644, b'81\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/type', 0o644, b'Interrupt\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/dev', 0o644, b'252:25\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/direction', 0o644, b'in\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/bLength', 0o644, b'07\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/wMaxPacketSize', 0o644, b'0004\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/bmAttributes', 0o644, b'03\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/uevent', 0o644, b'''MAJOR=252
MINOR=25
''')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/bInterval', 0o644, b'0a\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/usb_endpoint/usbdev3.4_ep81/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input', 0o755)
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/subsystem', '../../../../../../../../class/input')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/device', '../../../3-1:1.0')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/uniq', 0o644, b'\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/phys', 0o644, b'usb-0000:00:1d.2-1/input0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/modalias', 0o644, b'input:b0003v046DpC03Ee0110-e0,1,2,4,k110,111,112,r0,1,8,am4,lsfw\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/name', 0o644, b'Logitech USB-PS/2 Optical Mouse\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/uevent', 0o644, b'''PRODUCT=3/46d/c03e/110
NAME="Logitech USB-PS/2 Optical Mouse"
PHYS="usb-0000:00:1d.2-1/input0"
UNIQ=""
EV==17
KEY==70000 0 0 0 0
REL==103
MSC==10
MODALIAS=input:b0003v046DpC03Ee0110-e0,1,2,4,k110,111,112,r0,1,8,am4,lsfw
''')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/subsystem', '../../../../../../../../../class/input')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/device', '../../input8')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/dev', 0o644, b'13:33\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/uevent', 0o644, b'''MAJOR=13
MINOR=33
''')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/mouse1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/id', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/id/version', 0o644, b'0110\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/id/product', 0o644, b'c03e\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/id/vendor', 0o644, b'046d\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/id/bustype', 0o644, b'0003\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7', 0o755)
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/subsystem', '../../../../../../../../../class/input')
l('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/device', '../../input8')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/dev', 0o644, b'13:71\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/uevent', 0o644, b'''MAJOR=13
MINOR=71
''')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/event7/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities', 0o755)
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/msc', 0o644, b'10\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/abs', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/snd', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/ff', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/key', 0o644, b'70000 0 0 0 0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/rel', 0o644, b'103\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/sw', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/ev', 0o644, b'17\n')
f('sys/devices/pci0000:00/0000:00:1d.2/usb3/3-1/3-1:1.0/input/input8/capabilities/led', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:01.0', 0o755)
l('sys/devices/pci0000:00/0000:00:01.0/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:06')
l('sys/devices/pci0000:00/0000:00:01.0/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:01.0/driver', '../../../bus/pci/drivers/pcieport-driver')
f('sys/devices/pci0000:00/0000:00:01.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:01.0/enable', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:01.0/modalias', 0o644, b'pci:v00008086d000027A1sv00000000sd00000000bc06sc04i00\n')
f('sys/devices/pci0000:00/0000:00:01.0/resource', 0o644, b'''0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000002000 0x0000000000002fff 0x0000000000000100
0x00000000ee100000 0x00000000ee1fffff 0x0000000000000200
0x00000000d0000000 0x00000000dfffffff 0x0000000000001201
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:01.0/config', 0o644, b'\x86\x80\xa1\'\x07\x05\x10\x00\x03\x00\x04\x06\x10\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x01\x00  \x00\x00\x10\xee\x10\xee\x01\xd0\xf1\xdf\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x01\x90\x02\xc8\x00\x00\x00\x00\r\x80\x00\x00\xaa\x17\x14 \x05\xa0\x01\x00\x0c0\xe0\xfeqA\x00\x00\x00\x00\x00\x00\x10\x00A\x01\x00\x00\x00\x00\x00\x00\x01\x00\x01M\x01\x02\x03\x00\x01\x11\x80%\x08\x00\xc0\x01H\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x0f\x00\x00\x00\x00\x86\x0f\x05\x00\x00\x00\x00\x80\x02\x00\x01\x14\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x80\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x01\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x01\x00\x00\x01\x01\x02\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x90\xd1\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x06\x00\x01\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00n\x7f \x08\xf4\x0c\x00\x00\xb04 \x00\x00\x00\x00\x003\x0f\x00\x00\xff\xff\x03\x00\xd0\x0f\xf1\x05\xd1\x0f\xd1\x0f\x08\x00\x00\x00\x1f\x00\x00\x00\x00\x00\x00\x00\r/B\x00\x0c\x00\x00\x14\xb5\xbcJ\xbc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x003\x0b\xe0\x00Q\x0b\xfe\x00 \x00\x08\x00\x85\x03\xf1\x00\x87\x03\xf3\x00\x00\x00\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00 \x00\x04\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x14\x08\x88\x00\x00\x00\x00\x00\x00\x00\x00"wD\xb23\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\n\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\r\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x0c\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x0c\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x003\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x004\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x0e\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x002\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x004\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x002\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00;\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x003\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x009\x06`\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00:\x06b\x00 \x00\x00`\x02\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x007')
f('sys/devices/pci0000:00/0000:00:01.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:01.0/subsystem_device', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:01.0/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:01.0/irq', 0o644, b'511\n')
f('sys/devices/pci0000:00/0000:00:01.0/device', 0o644, b'0x27a1\n')
f('sys/devices/pci0000:00/0000:00:01.0/class', 0o644, b'0x060400\n')
f('sys/devices/pci0000:00/0000:00:01.0/msi_bus', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:01.0/subsystem_vendor', 0o644, b'0x0000\n')
f('sys/devices/pci0000:00/0000:00:01.0/uevent', 0o644, b'''DRIVER=pcieport-driver
PCI_CLASS=60400
PCI_ID=8086:27A1
PCI_SUBSYS_ID=0000:0000
PCI_SLOT_NAME=0000:00:01.0
MODALIAS=pci:v00008086d000027A1sv00000000sd00000000bc06sc04i00
''')
f('sys/devices/pci0000:00/0000:00:01.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:01.0/pci_bus', 0o755)
d('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01', 0o755)
l('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/subsystem', '../../../../../class/pci_bus')
l('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/device', '../../../0000:00:01.0')
f('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/cpuaffinity', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/power', 0o755)
f('sys/devices/pci0000:00/0000:00:01.0/pci_bus/0000:01/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:01.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:01.0/power/wakeup', 0o644, b'disabled\n')
d('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03', 0o755)
l('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03/power', 0o755)
f('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie03/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0', 0o755)
l('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/firmware_node', '../../../LNXSYSTM:00/device:00/PNP0A08:00/device:06/device:07')
l('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/subsystem', '../../../../bus/pci')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/enable', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/modalias', 0o644, b'pci:v00001002d000071D4sv000017AAsd000020A4bc03sc00i00\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/resource', 0o644, b'''0x00000000d0000000 0x00000000dfffffff 0x0000000000021208
0x0000000000002000 0x00000000000020ff 0x0000000000020101
0x00000000ee100000 0x00000000ee10ffff 0x0000000000020200
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x00000000ee120000 0x00000000ee13ffff 0x0000000000027202
''')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/config', 0o644, b'\x02\x10\xd4q\x07\x01\x10\x00\x00\x00\x00\x03\x10\x00\x00\x00\x08\x00\x00\xd0\x01 \x00\x00\x00\x00\x10\xee\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\xa4 \x00\x00\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\xa4 \x01X\x02\x06\x00\x00\x00\x00\x10\x80\x11\x00\xa0\x0f,\x01\x10\x08\x00\x00\x01\r\x00\x00C\x00\x01\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11 \x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00@\x03\x00\x00\x00$\x8f\x0b\x00\xff\xff\x07e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/resource1', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/resource2', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/subsystem_device', 0o644, b'0x20a4\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/vendor', 0o644, b'0x1002\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/irq', 0o644, b'11\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/device', 0o644, b'0x71d4\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/class', 0o644, b'0x030000\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/rom', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/uevent', 0o644, b'''PCI_CLASS=30000
PCI_ID=1002:71D4
PCI_SUBSYS_ID=17AA:20A4
PCI_SLOT_NAME=0000:01:00.0
MODALIAS=pci:v00001002d000071D4sv000017AAsd000020A4bc03sc00i00
''')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/resource0_wc', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:01.0/0000:01:00.0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00', 0o755)
l('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00/subsystem', '../../../../bus/pci_express')
f('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00/power', 0o755)
f('sys/devices/pci0000:00/0000:00:01.0/0000:00:01.0:pcie00/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/firmware_node', '../../LNXSYSTM:00/device:00/PNP0A08:00/device:13')
l('sys/devices/pci0000:00/0000:00:1f.1/subsystem', '../../../bus/pci')
l('sys/devices/pci0000:00/0000:00:1f.1/driver', '../../../bus/pci/drivers/ata_piix')
f('sys/devices/pci0000:00/0000:00:1f.1/local_cpulist', 0o644, b'0-7\n')
f('sys/devices/pci0000:00/0000:00:1f.1/resource3', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/enable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/modalias', 0o644, b'pci:v00008086d000027DFsv000017AAsd0000200Cbc01sc01i8a\n')
f('sys/devices/pci0000:00/0000:00:1f.1/resource', 0o644, b'''0x00000000000001f0 0x00000000000001f7 0x0000000000000110
0x00000000000003f6 0x00000000000003f6 0x0000000000000110
0x0000000000000170 0x0000000000000177 0x0000000000000110
0x0000000000000376 0x0000000000000376 0x0000000000000110
0x0000000000001880 0x000000000000188f 0x0000000000020101
0x0000000000000000 0x0000000000000000 0x0000000000000000
0x0000000000000000 0x0000000000000000 0x0000000000000000
''')
f('sys/devices/pci0000:00/0000:00:1f.1/config', 0o644, b"\x86\x80\xdf'\x05\x00\x80\x02\x02\x8a\x01\x01\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x81\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xaa\x17\x0c \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\x03\x00\x00\x03\xe3\x00@\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x0f\x02\x00\x00\x00\x00\x00")
f('sys/devices/pci0000:00/0000:00:1f.1/resource1', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/resource2', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/local_cpus', 0o644, b'ff\n')
f('sys/devices/pci0000:00/0000:00:1f.1/subsystem_device', 0o644, b'0x200c\n')
f('sys/devices/pci0000:00/0000:00:1f.1/vendor', 0o644, b'0x8086\n')
f('sys/devices/pci0000:00/0000:00:1f.1/irq', 0o644, b'16\n')
f('sys/devices/pci0000:00/0000:00:1f.1/device', 0o644, b'0x27df\n')
f('sys/devices/pci0000:00/0000:00:1f.1/class', 0o644, b'0x01018a\n')
f('sys/devices/pci0000:00/0000:00:1f.1/msi_bus', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/resource0', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/subsystem_vendor', 0o644, b'0x17aa\n')
f('sys/devices/pci0000:00/0000:00:1f.1/uevent', 0o644, b'''DRIVER=ata_piix
PCI_CLASS=1018A
PCI_ID=8086:27DF
PCI_SUBSYS_ID=17AA:200C
PCI_SLOT_NAME=0000:00:1f.1
MODALIAS=pci:v00008086d000027DFsv000017AAsd0000200Cbc01sc01i8a
''')
f('sys/devices/pci0000:00/0000:00:1f.1/resource4', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/broken_parity_status', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host5', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host5/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host5/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/device', '../../../host5')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/unique_id', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/can_queue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/sg_tablesize', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/proc_name', 0o644, b'ata_piix\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host5/scsi_host/host5/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/subsystem', '../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/uevent', 0o644, b'DEVTYPE=scsi_host\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/subsystem', '../../../../../bus/scsi')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/uevent', 0o644, b'DEVTYPE=scsi_target\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/generic', 'scsi_generic/sg1')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/subsystem', '../../../../../../bus/scsi')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/driver', '../../../../../../bus/scsi/drivers/sr')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/iodone_cnt', 0o644, b'0x20b3\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/device_blocked', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/modalias', 0o644, b'scsi:t-0x05\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_level', 0o644, b'6\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/queue_depth', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/rev', 0o644, b'RB01\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/type', 0o644, b'5\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/iocounterbits', 0o644, b'32\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/vendor', 0o644, b'MATSHITA\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/queue_type', 0o644, b'none\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/iorequest_cnt', 0o644, b'0x82a9\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/evt_media_change', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/model', 0o644, b'DVD-RAM UJ-842  \n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/ioerr_cnt', 0o644, b'0x0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/uevent', 0o644, b'''DEVTYPE=scsi_device
DRIVER=sr
MODALIAS=scsi:t-0x05
''')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/timeout', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/subsystem', '../../../../../../../../class/scsi_generic')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/device', '../../../4:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/dev', 0o644, b'21:1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/uevent', 0o644, b'''MAJOR=21
MINOR=1
''')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_generic/sg1/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/subsystem', '../../../../../../../../class/bsg')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/device', '../../../4:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/dev', 0o644, b'254:1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/uevent', 0o644, b'''MAJOR=254
MINOR=1
''')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/bsg/4:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/subsystem', '../../../../../../../../class/block')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/bdi', '../../../../../../../virtual/bdi/11:0')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/device', '../../../4:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/capability', 0o644, b'19\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/ro', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/make-it-fail', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/size', 0o644, b'2097151\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/dev', 0o644, b'11:0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/range', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/removable', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/stat', 0o644, b'       0        0        0        0        0        0        0        0        0        0        0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/uevent', 0o644, b'''MAJOR=11
MINOR=0
DEVTYPE=disk
''')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/nr_requests', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/nomerges', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/scheduler', 0o644, b'noop anticipatory deadline [cfq] \n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/hw_sector_size', 0o644, b'512\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/max_hw_sectors_kb', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/read_ahead_kb', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/max_sectors_kb', 0o644, b'128\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/slice_async_rq', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/back_seek_max', 0o644, b'16384\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/slice_sync', 0o644, b'100\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/slice_async', 0o644, b'40\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/fifo_expire_sync', 0o644, b'125\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/slice_idle', 0o644, b'8\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/back_seek_penalty', 0o644, b'2\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/fifo_expire_async', 0o644, b'250\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/queue/iosched/quantum', 0o644, b'4\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/block/sr0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0/subsystem', '../../../../../../../../class/scsi_device')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0/device', '../../../4:0:0:0')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0/uevent', 0o644, b'')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/target4:0:0/4:0:0:0/scsi_device/4:0:0:0/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/power', 0o755)
f('sys/devices/pci0000:00/0000:00:1f.1/host4/power/wakeup', 0o644, b'\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host', 0o755)
d('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4', 0o755)
l('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/subsystem', '../../../../../../class/scsi_host')
l('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/device', '../../../host4')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/unique_id', 0o644, b'5\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/can_queue', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/unchecked_isa_dma', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/active_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/sg_tablesize', 0o644, b'128\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/prot_guard_type', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/host_busy', 0o644, b'0\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/proc_name', 0o644, b'ata_piix\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/state', 0o644, b'running\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/cmd_per_lun', 0o644, b'1\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/supported_mode', 0o644, b'Initiator\n')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/uevent', 0o644, b'')
f('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/prot_capabilities', 0o644, b'0\n')
d('sys/devices/pci0000:00/0000:00:1f.1/host4/scsi_host/host4/power', 