# ui_cli.py - command line user interface
# Copyright (C) 2008  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import os
import logging
import sys
import textwrap

import cruftremover
_ = cruftremover.setup_gettext()


class UnknownCommand(cruftremover.Exception):

    def __init__(self, name):
        self._str = _("Unknown command: %s") % name


class UnknownCruft(cruftremover.Exception):

    def __init__(self, name):
        self._str = _("Unknown cruft: %s") % name


class MustBeRoot(cruftremover.Exception):

    def __init__(self):
        self._str = _("cruft-remover must be run as root, sorry.")


class CommandLineUserInterface(cruftremover.UserInterface):

    def run(self, options, args):
        if self.mustberoot and os.getuid() != 0:
            raise MustBeRoot()
    
        dict = {
            "find": self.show_cruft,
            "cleanup": self.cleanup,
            "ignore": self.ignore,
            "unignore": self.unignore,
            "help": self.help,
        }

        if args:
            cmd = args[0]
            args = args[1:]
        else:
            cmd = "help"
            args = []
        
        if cmd in dict:
            app = self.app
            app.state.load(options.state_file)
            try:
                dict[cmd](options, args)
            except Exception, e: # pragma: no cover
                logging.critical(unicode(e))
                sys.exit(1)
        else:
            raise UnknownCommand(cmd)
    
    def find_cruft(self):
        list = []
        for plugin in self.pm.get_plugins():
            for cruft in plugin.get_cruft():
                list.append(cruft)
        return list
    
    def show_one_cruft(self, name, desc, s, width): #pragma: no cover
        if width is None:
            max = len(name) + len(desc)
        else:
            max = width
            max -= 9 # state column
            max -= 2 # two spaces
            max -= 1 # avoid the last column, some terminals word wrap there

        print "%-9s  %.*s " % (s, max, name)
        if desc:
            for line in textwrap.wrap(desc, max):
                print "%9s  %s" % ("", line)
            print
    
    def show_cruft(self, options, args):
        list = []
        maxname = ""
        state = self.app.state
        for cruft in self.find_cruft():
            name = cruft.get_name()
            if options.verbose:
                desc = cruft.get_description() # pragma: no cover
            else:
                desc = None
            if state.is_enabled(name):
                s = _("removable")
            else:
                s = _("ignored")
            list.append((name, desc, s))
            if len(name) > len(maxname):
                maxname = name

        rows, cols = cruftremover.get_terminal_size()
        for name, desc, s in sorted(list):
            self.show_one_cruft(name, desc, s, cols)

    def cleanup(self, options, args):
        crufts = {}
        for cruft in self.find_cruft():
            crufts[cruft.get_name()] = cruft
        
        if args:
            for arg in args:
                if arg not in crufts:
                    raise UnknownCruft(arg)
        elif options.all:
            state = self.app.state
            args = []
            for name in crufts.keys():
                if state.is_enabled(name):
                    args.append(name)
                else:
                    logging.info(_("Ignored: %s") % name)
    
        for arg in args:
            if options.no_act:
                logging.info(_("Pretending to remove cruft: %s") % arg)
            else:
                logging.info(_("Removing cruft: %s") % arg)
                crufts[arg].cleanup()
        for plugin in self.pm.get_plugins():
            if options.no_act:
                logging.info(_("Pretending to post-cleanup: %s") % plugin)
            else:
                logging.info(_("Post-cleanup: %s") % plugin)
                plugin.post_cleanup()

    def ignore(self, options, cruft_names):
        state = self.app.state
        for cruft_name in cruft_names:
            state.disable(cruft_name)
        if not options.no_act:
            state.save(options.state_file)

    def unignore(self, options, cruft_names):
        state = self.app.state
        for cruft_name in cruft_names:
            state.enable(cruft_name)
        if not options.no_act:
            state.save(options.state_file)

    def help(self, options, args):
        self.app.parser.print_help()
