# plugin_tests.py - unit tests for plugin.py
# Copyright (C) 2008  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import os
import tempfile
import unittest

import cruftremover


class PluginTests(unittest.TestCase):

    def testGetCruftRaisesException(self):
        p = cruftremover.Plugin()
        self.assertRaises(cruftremover.UnimplementedMethod, p.get_cruft)

    def testPostCleanupReturnsNone(self):
        p = cruftremover.Plugin()
        self.assertEqual(p.post_cleanup(), None)

    def testDoesNotHaveAppAttributeByDefault(self):
        p = cruftremover.Plugin()
        self.assertFalse(hasattr(p, "app"))

    def testSetApplicationSetsApp(self):
        p = cruftremover.Plugin()
        p.set_application("foo")
        self.assertEqual(p.app, "foo")


class PluginManagerTests(unittest.TestCase):

    def testFindsNoPluginsInEmptyDirectory(self):
        tempdir = tempfile.mkdtemp()
        pm = cruftremover.PluginManager(None, [tempdir])
        plugins = pm.get_plugins()
        os.rmdir(tempdir)
        self.assertEqual(plugins, [])

    def testFindsOnePluginFileInTestPluginDirectory(self):
        pm = cruftremover.PluginManager(None, ["testplugins"])
        self.assertEqual(pm.get_plugin_files(), 
                         ["testplugins/hello_plugin.py"])

    def testFindsOnePluginInTestPluginDirectory(self):
        pm = cruftremover.PluginManager(None, ["testplugins"])
        self.assertEqual(len(pm.get_plugins()), 1)

    def testFindPluginsSetsApplicationInPluginsFound(self):
        pm = cruftremover.PluginManager("foo", ["testplugins"])
        self.assertEqual(pm.get_plugins()[0].app, "foo")

    def callback(self, filename, index, count):
        self.callback_called = True

    def testCallsCallbackWhenFindingPlugins(self):
        pm = cruftremover.PluginManager(None, ["testplugins"])
        self.callback_called = False
        pm.get_plugins(callback=self.callback)
        self.assert_(self.callback_called)
