/*  Screem:  screem-site-model.c,
 * 
 *  file tree model handling for a ScreemSite
 *
 *  Copyright (C) 2001, 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include <time.h>

#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <glade/glade.h>

#include <gtk/gtk.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <glib/gi18n.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "screem-site.h"
#include "screem-site-model.h"

#include "screem-application.h"

#include "fileops.h"

#include "support.h"

typedef struct {
	gchar *uri;
	const gchar *mime_type;
	gint number;
	gint total_size;
} Stats;

static void screem_site_model_row_deleted( GtkTreeModel *model, 
					  GtkTreePath *path,
					  GtkTreeIter *it, gpointer data );

static gboolean screem_site_model_find_iter( ScreemSiteViewNodeInfo *info,
					    const gchar *uri,
					    GtkTreeIter *parent,
					    GtkTreeIter *it );

static GList *screem_site_model_calc_stats( ScreemSite *site,
					GtkTreeModel *model, GtkTreeIter *it, 
					GList *types );
gint screem_site_model_compare_func( GtkTreeModel *model, 
				    GtkTreeIter *a, GtkTreeIter *b,
				    ScreemSite *site )
{
        ScreemSiteViewNodeInfo *ainfo;
        ScreemSiteViewNodeInfo *binfo;
	gchar *aname;
	gchar *bname;
	const gchar *spath;
	
	int ret = 0;

	spath = screem_site_get_pathname( site );

	ainfo = binfo = NULL;
	aname = bname = NULL;
	
	if( a ) {
		gtk_tree_model_get( model, a, 
				FILE_BROWSER_DATA_COL, &ainfo,
				FILE_BROWSER_NAME_COL, &aname,
				-1 );
	}
	if( b ) {
		gtk_tree_model_get( model, b, 
				FILE_BROWSER_DATA_COL, &binfo,
				FILE_BROWSER_NAME_COL, &bname,
				-1 );
	}

	/* fake site case */
	if( screem_site_get_fake_flag( site ) ) {
		if( aname && bname ) {
			ret = strcmp( aname, bname );
		} else {
			return aname - bname;
		}
	} else if( ainfo && ! strcmp( spath, ainfo->fullname ) ) {
		/* special root case */
		ret = -1;
	} else if( binfo && ! strcmp( spath, binfo->fullname ) ) {
		ret = 1;
	/* folders come before files */
	} else if( ainfo && binfo && 
		   ainfo->type == SCREEM_SITE_VIEW_FOLDER &&
		   binfo->type != SCREEM_SITE_VIEW_FOLDER ) {
		ret = -1;
	} else if( ainfo && binfo && 
		 binfo->type == SCREEM_SITE_VIEW_FOLDER &&
		 ainfo->type != SCREEM_SITE_VIEW_FOLDER ) {
		ret = 1;
	/* handle unset names */
	} else if( aname && ! bname ) {
		ret = 1;
	} else if( bname && ! aname ) {
		ret = -1;
	/* default name comparison */
	} else {
		ret = strcmp( aname, bname );
	}

	g_free( aname );
	g_free( bname );

	return ret;
}


ScreemSiteViewNodeInfo* 
screem_site_model_node_info_new( ScreemSiteViewNodeType type, ScreemSite *site,
				gpointer data, struct stat *s )
{
	ScreemSiteViewNodeInfo *info;

	info = g_new0( ScreemSiteViewNodeInfo, 1 );

	info->type = type;
	if( data ) {
		info->fullname = g_strdup( data );
	}
	info->site = site;

	return info;
}

void screem_site_model_node_info_free( ScreemSiteViewNodeInfo *info )
{
	g_return_if_fail( info != NULL );

	g_free( info->fullname );
	g_free( info );
}

void screem_site_model_set_val_at( GtkTreeStore *store, GtkTreeIter *iter,
				  ScreemSiteViewNodeInfo *info)
{
	ScreemSite *site;
	gchar flags[ 3 ] = { '.', '.', '\0' };
	site = info->site;

	if( screem_site_is_excluded( site, info->fullname ) ) {
		flags[ 0 ] = 'E';
	}
	if( screem_site_is_ignored( site, info->fullname ) ) {
		flags[ 1 ] = 'I';
	}

	/* name is what we want to display */
	gtk_tree_store_set( store, iter, 
			    FILE_BROWSER_USER_COL, flags, 
			    FILE_BROWSER_DATA_COL, info, 
			    -1 );
}

void screem_site_model_destroy( ScreemSite *site )
{
	ScreemSitePrivate *priv;
	ScreemSiteModel *model;

	priv = site->priv;
	model = priv->model;

	g_free( model->images );
	g_free( model->styles );
	g_free( model->scripts );
		
	g_object_unref( model->browser );

	g_free( model );
}

GtkTreeModel *screem_site_get_model( ScreemSite *site, gboolean base )
{
	ScreemSitePrivate *priv;
	ScreemSiteModel *smodel;
	GtkTreeModel *model;
	
	g_return_val_if_fail( SCREEM_IS_SITE( site ), NULL );

	priv = site->priv;
	smodel = priv->model;
	
	model = screem_file_browser_get_model( smodel->browser );
	if( base ) {
		model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );
		model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );
	}

	return model;
}

gboolean screem_site_model_create_dir( ScreemSite *site, 
		GtkWidget *widget, const gchar *dir )
{
	gboolean ret = FALSE;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter pit;
	gchar *tdir;
	const gchar *pathname;

	gchar *newdir = NULL;


	pathname = screem_site_get_pathname( site );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	tdir = NULL;
	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {

		gtk_tree_model_get( model, &it, 
				FILE_BROWSER_URI_COL, &tdir, -1 );
	}

	if( ! tdir ) {
		if( pathname ) {
			tdir = g_strdup( pathname );
		}
	} else if( ! screem_uri_is_dir( tdir ) ) {
		gtk_tree_model_iter_parent( model, &pit, &it );
		g_free( tdir );
		gtk_tree_model_get( model, &pit, 
				FILE_BROWSER_URI_COL, &tdir,
				-1 );
		if( ! tdir ) {
			return FALSE;
		}
	}
	ret = FALSE;
	if( tdir ) {
		newdir = g_strconcat( tdir, G_DIR_SEPARATOR_S,
				dir, NULL );

		ret = mkdir_recursive( newdir,
				GNOME_VFS_PERM_USER_ALL |
				GNOME_VFS_PERM_GROUP_ALL |
				GNOME_VFS_PERM_OTHER_READ |
				GNOME_VFS_PERM_OTHER_EXEC,
				screem_site_model_file_op, 
				site->priv->model );
		g_free( newdir );
		g_free( tdir );
	}
	
	return ret;
}

static void screem_site_model_row_deleted( GtkTreeModel *model, 
		GtkTreePath *path, GtkTreeIter *it, gpointer data )
{
	gchar *string;
	GtkTreeIter iter;
	ScreemSiteViewNodeInfo *info;
	ScreemSiteModel *smodel;
	
	string = gtk_tree_path_to_string( path );

	info = NULL;
	if( string && *string == '0' ) {
		gtk_tree_model_get( model, it,
				FILE_BROWSER_DATA_COL, &info, -1 );
		
	}
	if( info ) {
		/* before destroying we need to remove it from
		   the quick access nodes, Images, Scripts, Stylesheets etc */
		smodel = info->site->priv->model;
		if( smodel && smodel->images &&
		    screem_site_model_find_iter( info, NULL, 
			    smodel->images, &iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		if( smodel && smodel->styles &&
		    screem_site_model_find_iter( info, NULL, 
			    smodel->styles, &iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		if( smodel && smodel->scripts &&
		    screem_site_model_find_iter( info, NULL, 
			    smodel->scripts, &iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		
		screem_site_model_node_info_free( info );
	}
	g_free( string );
}

static gboolean screem_site_model_find_iter( ScreemSiteViewNodeInfo *info,
					    const gchar *uri,
					    GtkTreeIter *parent,
					    GtkTreeIter *it )
{
	GtkTreeModel *model;
	ScreemSiteViewNodeInfo *ninfo;
	gboolean got;
	gboolean found = FALSE;

	if( ! uri ) {
		uri = info->fullname;
	}

	model = screem_site_get_model( info->site, TRUE );
	
	if( ! parent ) {
		got = gtk_tree_model_get_iter_first( model, it );
	} else {
		got = gtk_tree_model_iter_children( model, it, parent );
	}

	while( got && ! found ) {
		GValue value = { 0 };

		gtk_tree_model_get_value( model, it, FILE_BROWSER_DATA_COL,
					  &value );
		ninfo = g_value_get_pointer( &value );
		g_value_unset( &value );

		if( ninfo && ! strcmp( uri, ninfo->fullname ) ) {
			/* found it */
			found = TRUE;
		} else {
			GtkTreeIter cit;
			if( gtk_tree_model_iter_has_child( model, it ) ) {
				if( screem_site_model_find_iter( info, uri,
								it, &cit ) ) {
					/* found it */
					*it = cit;
					found = TRUE;
				}
			}
		}
		/* if not found check siblings */
		if( ! found ) {
			got = gtk_tree_model_iter_next( model, it );
		}
	}

	return found;
}

static GList *screem_site_model_calc_stats( ScreemSite *site,
	GtkTreeModel *model, GtkTreeIter *it, GList *types )
{
	GnomeVFSFileInfo *info;
	ScreemSiteViewNodeInfo *ninfo;
	GtkTreeIter iter;
	GtkTreeIter tmp;
	const gchar *mime_type;

	if( ! it ) {
		/* get root */
		if( gtk_tree_model_get_iter_first( model, &iter ) &&
		    gtk_tree_model_iter_has_child( model, &iter ) ) {
			tmp = iter;
			it = &tmp;
			gtk_tree_model_iter_children( model, it, &iter );
		}
	}
	if( ! it ) {
		return types;
	}

	do {
		gtk_tree_model_get( model, it, 
				FILE_BROWSER_DATA_COL, &ninfo,
				FILE_BROWSER_MIME_COL, 
				(gchar**)&mime_type, -1 );
		if( ! ninfo || ! ninfo->fullname || 
		    screem_site_is_excluded( site, ninfo->fullname ) ) {
			continue;
		}
		if( ! mime_type ) {
			mime_type = "application/octet-stream";
		}
		info = gnome_vfs_file_info_new();

		gnome_vfs_get_file_info( ninfo->fullname, 
				 info, 0 );

		if( info->type == GNOME_VFS_FILE_TYPE_DIRECTORY ) {
			if( gtk_tree_model_iter_has_child( model, it ) ) {
				GtkTreeIter child;

				if( gtk_tree_model_iter_children( model, &child, it ) ) {
					types = screem_site_model_calc_stats( site, model, &child, types );
				}
			}
		} else {
			GList *list;

			for( list = types; list; list = list->next ) {
				Stats *stats;

				stats = (Stats*)list->data;
				if( ! strcmp( stats->mime_type,
							mime_type ) ) {
					stats->number ++;
					stats->total_size += info->size;
					break;
				}
			}
			if( ! list ) {
				Stats *stats;
			
				stats = g_new0( Stats, 1 );
				stats->uri = ninfo->fullname;
				stats->number = 1;
				stats->total_size = info->size;
				stats->mime_type = mime_type;
			
				types = g_list_prepend( types, stats );
			}
		}
		gnome_vfs_file_info_unref( info );
	} while( gtk_tree_model_iter_next( model, it ) );

	return types;
}

GtkTreeModel *screem_site_get_statistics( ScreemSite *site )
{
	GtkTreeModel *store;
	GtkTreeModel *model;
	GList *list;
	GList *tmp;
	gint total;
	gint i;
	ScreemApplication *app;
	ScreemIconCache *cache;
		
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	cache = screem_application_get_icon_cache( app );
	g_object_unref( app );

	store = screem_site_get_model( site, TRUE );
	
	/* FIXME: should do this in an idle handler, processing
	 * 1 file / dir at a time */
	
	list = screem_site_model_calc_stats( site, store, NULL, NULL );

	model = GTK_TREE_MODEL( gtk_list_store_new( 7, 
						    G_TYPE_STRING, 
						    GDK_TYPE_PIXBUF,
						    G_TYPE_INT,
						    G_TYPE_STRING,
						    G_TYPE_STRING,
						    G_TYPE_INT,
						    G_TYPE_INT,
						    NULL ) );
	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( model ),
			0, GTK_SORT_ASCENDING );

	for( i = 0, tmp = list, total = 0; tmp; tmp = tmp->next, ++ i ) {
		Stats *stats;
		GtkTreeIter it;
		GdkPixbuf *pixbuf;
		const gchar *desc;
		gchar *ttol;
		gchar *avg;
		
		stats = (Stats*)tmp->data;

		total += stats->total_size;

		desc = gnome_vfs_mime_get_description( stats->mime_type );
		
		ttol = gnome_vfs_format_file_size_for_display( stats->total_size );
		avg = gnome_vfs_format_file_size_for_display( stats->total_size / stats->number );
		
		gtk_list_store_insert_with_values( GTK_LIST_STORE( model ), 
				&it, i,
				0, desc,
				2, stats->number,
				3, ttol,
				4, avg,
				5, stats->total_size,
				6, stats->total_size / stats->number,
				-1 );
		g_free( ttol );
		g_free( avg );

		pixbuf = screem_icon_cache_get_pixbuf( cache,
				stats->uri, stats->mime_type,
				SCREEM_ICON_CACHE_DEFAULT_SIZE );
		
		if( pixbuf ) {
			gtk_list_store_set( GTK_LIST_STORE( model ), 
					&it, 1, pixbuf, -1 );
			g_object_unref( pixbuf );
		}

		g_free( stats );
	}
	g_object_set_data( G_OBJECT( model ), "total",
			   GINT_TO_POINTER( total ) );
	g_list_free( list );

	g_object_unref( cache );
	
	return model;
}

static gboolean screem_site_model_update_node( GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, ScreemSite *site )
{
	ScreemSiteViewNodeInfo *info;
	gchar *pflags;
	
	info = NULL;
	gtk_tree_model_get( model, iter, 
			FILE_BROWSER_DATA_COL, &info, 
			FILE_BROWSER_USER_COL, &pflags,
			-1 );
	
	if( info ) {
		gchar flags[ 3 ] = { '.', '.', '\0' };

		if( screem_site_is_excluded( site, info->fullname ) ) {
			flags[ 0 ] = 'E';
		}
		if( screem_site_is_ignored( site, info->fullname ) ) {
			flags[ 1 ] = 'I';
		}

		if( strcmp( pflags, flags ) ) {
			gtk_tree_store_set( GTK_TREE_STORE( model ),
					iter, FILE_BROWSER_USER_COL,
					flags, -1 );
		}
	}
	g_free( pflags );

	return FALSE;
}

void screem_site_model_update( ScreemSite *site )
{
	GtkTreeModel *store;

	store = screem_site_get_model( site, TRUE );
	
	gtk_tree_model_foreach( store, 
				(GtkTreeModelForeachFunc)screem_site_model_update_node,
				site );
}

void screem_site_model_removed( ScreemFileBrowser *browser, 
		const gchar *uri, GtkTreeIter *it, gpointer data )
{
	ScreemSite *site;
	GtkTreePath *path;
	GtkTreeModel *model;

	site = SCREEM_SITE( data );

	model = screem_site_get_model( site, TRUE );
	
	path = gtk_tree_model_get_path( model, it );
	if( path ) {	
		screem_site_model_row_deleted( model, path, it, NULL );
		gtk_tree_path_free( path );
	}

	/* FIXME: need to offer cvs delete */
}

static GtkTreeIter *screem_site_model_add_virtual_root( GtkTreeModel *model,
		const gchar *name, ScreemIconCache *cache )
{
	GtkTreeIter *ret;
	GdkPixbuf *pixbuf;
	
	ret = g_new0( GtkTreeIter, 1 );
	gtk_tree_store_append( GTK_TREE_STORE( model ), ret, NULL );
	
	/* pass root as we know its a directory,
	 * and we just want the directory icon */
	gtk_tree_store_set( GTK_TREE_STORE( model ), ret,
			FILE_BROWSER_NAME_COL, name,
			FILE_BROWSER_URI_COL, NULL,
			FILE_BROWSER_SHOW_COL, TRUE,
			-1 );
	
	pixbuf = screem_icon_cache_get_pixbuf( cache, 
			"/", "x-directory/normal", 
			SCREEM_ICON_CACHE_DEFAULT_SIZE);
	if( pixbuf ) {
		gtk_tree_store_set( GTK_TREE_STORE( model ), ret,
				FILE_BROWSER_ICON_COL, pixbuf,
				-1 );
		g_object_unref( pixbuf );
	}

	return ret;
}

static void screem_site_model_add_virtual( GtkTreeModel *model,
		GtkTreeIter *it, GtkTreeIter *parent,
		const gchar *uri,
		ScreemSiteViewNodeInfo *info )
{
	GtkTreeIter iiter;
	gchar *base;
	GdkPixbuf *pixbuf;
	gboolean visible;
	const gchar *mime_type;
		
	gtk_tree_store_append( GTK_TREE_STORE( model ), &iiter, 
			parent );
	base = g_path_get_basename( uri );
	gtk_tree_model_get( model, it,
			FILE_BROWSER_ICON_COL, &pixbuf, 
			FILE_BROWSER_SHOW_COL, &visible,
			FILE_BROWSER_MIME_COL, &mime_type,
			-1 );

	gtk_tree_store_set( GTK_TREE_STORE( model ), &iiter,
			FILE_BROWSER_NAME_COL, base,
			FILE_BROWSER_ICON_COL, pixbuf,
			FILE_BROWSER_URI_COL, uri,
			FILE_BROWSER_MIME_COL, mime_type,
			FILE_BROWSER_SHOW_COL, visible,
			-1 );
	g_free( base );
	screem_site_model_set_val_at( GTK_TREE_STORE( model),
			&iiter, info );

	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
}

void screem_site_model_added( ScreemFileBrowser *browser, 
		const gchar *uri, const gchar *mime_type,
		GtkTreeIter *it, gpointer data )
{
	ScreemSite *site;
	ScreemApplication *app;
	ScreemIconCache *cache;
	ScreemSiteModel *priv;
	ScreemSiteViewNodeInfo *info;
	GtkTreeModel *model;
	gboolean is_dir;

	site = SCREEM_SITE( data );

	if( screem_site_get_fake_flag( site ) || ! it ) {
		return;
	}

	g_assert( mime_type );
	
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	cache = screem_application_get_icon_cache( app );
	
	priv = site->priv->model;

	model = screem_site_get_model( site, TRUE );
	
	is_dir = ! strcmp( "x-directory/normal", mime_type );
	
	if( ! is_dir ) {
		/* if mime type is application/octet-stream
		 * do a slow mime type check, we don't want
		 * to miss any pages
		 */
		if( ! strcmp( "application/octet-stream", mime_type ) ){
			mime_type = screem_get_mime_type( uri, FALSE );
		}
		
		info = screem_site_model_node_info_new( SCREEM_SITE_VIEW_FILE,
						       site,
						       (gpointer)uri,
						       NULL );
		info->model = priv;

		if( screem_page_is_mime_type_page( mime_type ) &&
			! screem_site_locate_page( site, uri ) ) {
			ScreemPage *page;
	
			page = screem_page_new( G_OBJECT( app ) );
			screem_page_set_pathname( page, uri );
			if( ! screem_site_add_page( site, page ) ) {
				/* shouldn't actually happen */
				g_assert( FALSE );
			}
			g_object_unref( page );
		}
		if( g_str_has_prefix( mime_type, "image/" ) ) {
			/* add to images branch as well */
			if( ! priv->images ) {
				priv->images = screem_site_model_add_virtual_root( model, _( "Images" ), cache );
			}
			screem_site_model_add_virtual( model, it,
					priv->images, uri, info );
		} else if( ! strcmp( "text/css", mime_type ) ) {
			/* add to stylesheets branch as well */
			if( ! priv->styles ) {
				priv->styles = screem_site_model_add_virtual_root( model, _( "Stylesheets" ), cache );
			}
			screem_site_model_add_virtual( model, it,
					priv->styles, uri, info );
		} else if( ! strcmp( "text/javascript", mime_type ) ||
			   ! strcmp( "text/x-javascript", mime_type ) ) {
			/* add to scripts branch as well */
			if( ! priv->scripts ) {
				priv->scripts = screem_site_model_add_virtual_root( model, _( "Scripts" ), cache );
			}
			screem_site_model_add_virtual( model, it,
					priv->scripts, uri, info );
		}  
	} else {
		info = screem_site_model_node_info_new( SCREEM_SITE_VIEW_FOLDER,
						       site,
						       (gpointer)uri,
						       NULL );
		info->model = priv;
	}
	if( it ) {
		screem_site_model_set_val_at( GTK_TREE_STORE( model ),
					     it, info );
	}
	g_object_unref( cache );
	g_object_unref( app );
}

void screem_site_model_icon_change( ScreemFileBrowser *browser, const gchar *uri,
				     const gchar *mime_type,
				     GtkTreeIter *it,
				     gpointer data )
{
	ScreemSite *site;
	ScreemSiteModel *priv;
	GtkTreeModel *model;
	gchar *url;
	GdkPixbuf *pixbuf;
	ScreemApplication *app;
	ScreemIconCache *cache;
	
	site = SCREEM_SITE( data );
	priv = site->priv->model;
		
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	cache = screem_application_get_icon_cache( app );
	g_object_unref( app );

	model = screem_site_get_model( site, TRUE );
	
	gtk_tree_model_get( model, it,
			FILE_BROWSER_URI_COL, &url,
			-1 );
	
	if( ! url ) {
		pixbuf = screem_icon_cache_get_pixbuf( cache, "/", 
				"x-directory/normal", SCREEM_ICON_CACHE_DEFAULT_SIZE );
		gtk_tree_store_set( GTK_TREE_STORE(model),
				    it,
				    FILE_BROWSER_ICON_COL,
				    pixbuf,
				    -1 );
		if( pixbuf ) {
			g_object_unref( pixbuf );
		}
	}

	g_object_unref( cache );
	
	g_free( url );
}

void screem_site_model_file_op( GnomeVFSMonitorEventType type,
				const gchar *uri, gpointer data )
{
	ScreemSiteModel *model;
	ScreemSite *site;
	ScreemApplication *app;
	
	model = (ScreemSiteModel*)data;
	site = model->site;
	app = SCREEM_APPLICATION( site->priv->application );
	screem_application_file_op( type, uri, app );
}

