/*  Screem:  support.c,
 *  some useful functions
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <errno.h>
#include <fnmatch.h>
#include <unistd.h>

#include <libgnome/gnome-util.h>
#include <libgnome/gnome-exec.h>

#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gdk/gdkkeysyms.h>

#include <gtk/gtkliststore.h>

#include <gconf/gconf-client.h>

#include <stdio.h>
#include <string.h>

#include "support.h"
#include "readtags.h"

/**
 * change_state:
 *
 * toggles widget sensitivity
 *
 * return values: none
 */
void change_state( GtkWidget *widget, gpointer data )
{
        gtk_widget_set_sensitive( widget, !GTK_WIDGET_IS_SENSITIVE( widget ) );
}

void change_visible( GtkWidget *widget, gpointer data )
{
	if( GTK_WIDGET_VISIBLE( widget ) ) {
		gtk_widget_hide( widget );
	} else {
		gtk_widget_show( widget );
	}
}

gchar *convert_keysym_state_to_string( guint keysym, guint state )
{
        gchar *key;
	GString *str;

        if( keysym == GDK_VoidSymbol || keysym == GDK_BackSpace ||
	    keysym == GDK_Delete || keysym == 0 ) {
                return NULL;
	}

        gdk_error_trap_push();
        key = gdk_keyval_name( keysym );
        gdk_error_trap_pop();
        if( ! key )
		return NULL;

	str = g_string_new( NULL );

        if( state & GDK_CONTROL_MASK ) {
		g_string_append( str, "Control-" );
	}
        if( state & GDK_LOCK_MASK ) {
		g_string_append( str, "Lock-" );
        }
        if( state & GDK_SHIFT_MASK ) {
		g_string_append( str, "Shift-" );
        }
        if( state & GDK_MOD1_MASK ) {
		g_string_append( str, "Mod1-" );
        }
        if( state & GDK_MOD2_MASK ) {
		g_string_append( str, "Mod2-" );
        }
        if( state & GDK_MOD3_MASK ) {
		g_string_append( str, "Mod3-" );
        }
        if( state & GDK_MOD4_MASK ) {
		g_string_append( str, "Mod4-" );
        }
        if( state & GDK_MOD5_MASK ) {
		g_string_append( str, "Mod5-" );
        }

	if( str->len ) {
		g_string_append( str, key );
	}

	key = str->str;
	g_string_free( str, FALSE );

	return key;
}

gchar *screem_support_ctags( const gchar *dirname,
			     const gchar *word,
			     gchar **pattern )
{
	gchar *ret;
	tagFile *file;
	tagFileInfo *info;
	tagEntry *entry;

	gchar *tagfile;

	if( ! word ) {
		return NULL;
	}
	
	info = g_new0( tagFileInfo, 1 );
	entry = g_new0( tagEntry, 1 );

	ret = NULL;

	tagfile = g_strconcat( dirname, G_DIR_SEPARATOR_S, "tags", NULL );
	file = tagsOpen( tagfile, info );
	if( ! file ) {
		g_free( tagfile );
		tagfile = g_strconcat( dirname, G_DIR_SEPARATOR_S, "TAGS",
				       NULL );
		file = tagsOpen( tagfile, info );
	}
	g_free( tagfile );

	if( file ) {
		if(tagsFind(file, entry, word, 
			    TAG_OBSERVECASE | TAG_FULLMATCH) == TagSuccess) {
			ret = g_strconcat( dirname, G_DIR_SEPARATOR_S,
					   entry->file, NULL );
			if( pattern ) {
				if( entry->address.pattern ) {
					*pattern = screem_support_escape_regexp( entry->address.pattern );				
				} else {
					*pattern = NULL;
				}
			}
		}
	}
	
	g_free( entry );
	g_free( info );

	return ret;
}

/* process regexp, strip leading and trailing '/' characters,
   escape ( ) { } $ * + ? */
gchar *screem_support_escape_regexp( const gchar *pattern )
{
	gchar *ret;
	GString *temp;

	g_return_val_if_fail( pattern != NULL, NULL );
	g_return_val_if_fail( *pattern == '/', NULL );

	temp = g_string_new( NULL );
	
	pattern ++;
	while( *pattern != '\0' &&
	       ( *pattern != '/' || *( pattern - 1 ) == '\\' ) ) {
		gchar p;
		gchar c;

		p = *( pattern - 1 );
		c = *pattern;

		if( p != '\\' &&
		    ( c == '(' || c == ')' || c == '{' || c == '}' ||
		      c == '*' || c == '+' || c == '?' ) ) {
			g_string_append_c( temp, '\\' );
			g_string_append_c( temp, c );
		} else if( p != '\\' && c == '$' && *(pattern+1) != '/' ) {
			g_string_append_c( temp, '\\' );
			g_string_append_c( temp, c );
		} else {
			g_string_append_c( temp, c );
		}

		pattern ++;
	}

	if( temp->len ) {
		ret = g_strdup( temp->str );
	} else {
		ret = NULL;
	}
	g_string_free( temp, TRUE );
	

	return ret;
}


GtkTreeIter *screem_support_find_in_list( GtkListStore *store, guint column, const gchar *str )
{
	GtkTreeIter it;
	GtkTreeIter *ret;
	gchar *val;
	
	ret = NULL;

	g_assert( gtk_tree_model_get_column_type( GTK_TREE_MODEL( store ),
						  column ) == G_TYPE_STRING );
	
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( store ), &it ) ) {
		do {
			gtk_tree_model_get( GTK_TREE_MODEL( store ), &it, 
					    column, &val, -1 );

			if( val && ! strncmp( "regex:", val, strlen( "regex:" ) ) )  {
				/* regexp pattern */
				g_warning( "regex: pattern match not supported yet\n" );
			} else if( val && ! strncmp( "glob:", val, strlen( "glob:" ) ) ) {
				/* glob */
				if( fnmatch( val + strlen( "glob:" ), str, 0 ) == 0 ) {
					ret = gtk_tree_iter_copy( &it );
				}
			} else 	if( val && ! strcmp( val, str ) ) {
				/* string pattern */
				
				ret = gtk_tree_iter_copy( &it );
			}
			g_free( val );
		} while( ( ! ret ) && 
			 gtk_tree_model_iter_next( GTK_TREE_MODEL( store ),
				 		   &it ) );
	}
	
	return ret;
}

gchar *screem_support_charset_convert( const gchar *data )
{
	const gchar *end;
	gchar *ret;
	guint len;
	
	ret = NULL;
	if( data && ! g_utf8_validate( data, strlen( data ), &end ) ) {
		gchar *temp = NULL;
		const gchar *charset;
		gboolean utf8;
		const char *encodings_to_try[ 2 ] = { 0, 0 };

		utf8 = g_get_charset( &charset );

		if( ! utf8 ) {
			encodings_to_try[ 0 ] = charset;
		}
		if( g_ascii_strcasecmp( charset, "ISO-8859-1" ) ) {
			encodings_to_try[ 1 ] = "ISO-8859-1";
		}

		if( encodings_to_try[ 0 ] ) {
			temp = g_convert( data, strlen( data ), "UTF-8", 
					  encodings_to_try[ 0 ], 
					  NULL, &len, NULL );
		}
		if( ! temp && encodings_to_try[ 1 ] ) {
			temp = g_convert( data, strlen( data ), "UTF-8", 
					  encodings_to_try[ 1 ], 
					  NULL, &len, NULL );
		}

		ret = temp;
	} else if( data ) {
		ret = g_strdup( data );
	}

	return ret;
}

gchar *screem_support_charset_convert_to( const gchar *data, const gchar *charset )
{
	gchar *ret;

	ret = NULL;
	
	if( charset && g_strcasecmp( "UTF-8", charset ) ) {
		/* need to convert from UTF-8 to charset */
		gchar *temp;
		gint len;
		
		temp = g_convert( data, strlen( data ), charset, 
				  "UTF-8", 
				  NULL, &len, NULL );

		ret = temp;
	}

	return ret;
}

/* HACK due to the huge annoyance of gnome-vfs mime data
 * determining that anything starting <?xml  is text/xml
 * no matter what any *.mime files say about the extension */

gchar *screem_get_mime_type( const gchar *filename )
{
	gchar *type;
	gchar *ret;
	
	type = gnome_vfs_get_mime_type( filename );
	ret = screem_get_mime_type_override( filename, type );
	g_free( type );
	
	return ret;
}

gchar *screem_get_mime_type_override( const gchar *filename,
					const gchar *type ) 
{
	const gchar *ext;
	gchar *ret;
	
	ret = NULL;
	if( type ) {

		ext = g_extension_pointer( filename );
		
		if( ! strcmp( "screem", ext ) ) {
			ret = g_strdup("application/x-screem");
		} else if( ! strcmp( "tagtree", ext ) ) { 
			ret = g_strdup( "application/x-screem-tag-tree" );
		} else if( ! strcmp( "wml", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wml" );
		} else if( ! strcmp( "wmls", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlscript" );
		} else if( ! strcmp( "wmlb", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlb" );
		} else if( ! strcmp( "wmlc", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlc" );
		} else if( ! strcmp( "wmlsc", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlscriptc" );
		} else {
			ret = g_strdup( type );
		}
	} else if( type ) {
		ret = g_strdup( type );
	}

	return ret;
}

const gchar *g_utf8_skip_space( const gchar *txt )
{
	gunichar c;

	g_return_val_if_fail( txt != NULL, NULL );
	
	while( *txt ) {
		c = g_utf8_get_char( txt );
		if( ! g_unichar_isspace( c ) ) {
			break;
		}
		txt = g_utf8_next_char( txt );
	}

	return txt;
}

const gchar *screem_utf8_skip_to_space( const gchar *txt )
{
	gunichar c;

	g_return_val_if_fail( txt != NULL, NULL );
	
	while( *txt ) {
		c = g_utf8_get_char( txt );
		if( g_unichar_isspace( c ) ) {
			break;
		}
		txt = g_utf8_next_char( txt );
	}

	return txt;
}



/* ripped out of gedit2 */
gchar* screem_escape_underlines( const gchar* text )
{
	GString *str;
	gint length;
	const gchar *p;
 	const gchar *end;

  	g_return_val_if_fail( text != NULL, NULL );

    	length = strlen( text );

	str = g_string_new( NULL );

  	p = text;
  	end = text + length;

  	while( p != end ) {
      		const gchar *next;
      		next = g_utf8_next_char( p );

		switch( *p ) {
       			case '_':
          			g_string_append( str, "__" );
          			break;
        		default:
          			g_string_append_len( str, p, next - p );
          			break;
        	}
      		p = next;
    	}

	return g_string_free( str, FALSE );
}

gboolean screem_execute_default_app( const gchar *uri )
{
	const gchar *mime;
	GnomeVFSMimeApplication *app;
	gchar *url;
	gchar *exstr;
	gchar *temp;

	mime = gnome_vfs_get_mime_type( uri );
	app = gnome_vfs_mime_get_default_application( mime );
	if( app ) {
		url = NULL;
		if( ! app->expects_uris ) {
			url = gnome_vfs_get_local_path_from_uri( uri );
		} else {
			/* FIXME check app supports the method uri
			   is using */
		}
		
		if( ! url ) {
			url = g_strdup( uri );
		}
		uri = strstr( app->command, "%s" );
		if( ! uri ) {
			exstr = g_strconcat( app->command, " \"", url, "\"", NULL );
		} else {
			temp = g_strndup( app->command, uri - app->command );
			exstr = g_strconcat( temp, url, 
					uri + strlen( "%s" ), NULL );
			g_free( temp );
		}
		if( app->requires_terminal ) {
			gnome_execute_terminal_shell( NULL, exstr );
		} else {
			gnome_execute_shell( NULL, exstr );
		}
		g_free( exstr );
	}

	return ( app != NULL );
}

void g_string_append_utf8_len( GString *str, const gchar *utf8,
				guint len )
{
	gunichar c;
	
	while( len ) {
		c = g_utf8_get_char( utf8 );
		g_string_append_unichar( str, c );
		utf8 = g_utf8_next_char( utf8 );
		len --;
	}
}

GtkPositionType screem_get_tab_position( void )
{
	GConfClient *client;
	static GConfEnumStringPair positions[] = {
		{ GTK_POS_LEFT, "left" },
		{ GTK_POS_RIGHT, "right" },
		{ GTK_POS_TOP, "top" },
		{ GTK_POS_BOTTOM, "bottom" },
		{ GTK_POS_TOP, "" },
		{ GTK_POS_TOP, NULL }
	};
	gchar *pos;
	gint position;

	client = gconf_client_get_default();
		
	pos = gconf_client_get_string( client,
			"/apps/screem/ui/tab_position",
			NULL );
	if( ! pos ) {
		pos = g_strdup( "" );
	}
	gconf_string_to_enum( positions, pos, &position );
	g_free( pos );

	g_object_unref( client );

	return position;
}
