/*  Screem:  screem-window-menus.c
 *
 *  Handle menus/toolbars
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <pwd.h>
#include <string.h>
#include <sys/types.h>

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-util.h>

#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-stock-icons.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <bonobo/bonobo-file-selector-util.h>

#include <glade/glade.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkiconfactory.h>

#include <glib-object.h>
#include <glib/glist.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-window-private.h"

#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-site-druid.h"

#include "screem-editor.h"
#include "screem-tree-view.h"

#include "screem-site-view.h"    /* don't like including this */

#include "screem-preferences.h"

#include "screem-page.h"
#include "pageUI.h"

#include "screem-markup.h"

#include "screem-todo.h"

#include "screem-cvs.h"

#include "screem-search.h"

#include "support.h"

#include "fileops.h"

#include "libegg/dock/egg-dock.h"
#include "libegg/menu/egg-action-group.h"
#include "libegg/menu/egg-toggle-action.h"
#include "libegg/menu/egg-menu-merge.h"

#include "libegg/toolbar/eggtoolbar.h"

static void file_menu_save_document_as_callback( EggAction *action, 
					   	 gpointer user_data );


static void add_widget( EggMenuMerge *merge, GtkWidget *widget, ScreemWindow *window )
{
       	if( GTK_IS_MENU_SHELL( widget ) ) {
		gtk_box_pack_start( GTK_BOX( window->content_hbox ), widget,
				    FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ), widget,
					0 );
	} else if( EGG_IS_TOOLBAR( widget ) ) {
		guint toolbars;
		
		toolbars = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( merge ),
						"toolbars" ) );
		
		toolbars ++;
		
		egg_toolbar_set_show_arrow( EGG_TOOLBAR( widget ), TRUE );
		egg_toolbar_set_tooltips( EGG_TOOLBAR( widget ), TRUE );
		
		gtk_box_pack_start( GTK_BOX( window->content_hbox ), widget,
				    FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ), widget,
					toolbars );
		g_object_set_data( G_OBJECT( merge ), "toolbars",
					GUINT_TO_POINTER( toolbars ) );
	} 
}

static gboolean file_menu_new_window_after_cb( gpointer data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( data );
	app = SCREEM_APPLICATION( window->application );
	site = screem_application_get_default_site( app );

gdk_threads_enter();
	screem_window_set_current( window, site );
	gdk_threads_leave();
	
	return FALSE;
}

static void file_menu_new_window_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	app = SCREEM_APPLICATION( window->application );

	window = screem_application_create_window( app );

	g_idle_add( file_menu_new_window_after_cb, window );

	gtk_widget_show( GTK_WIDGET( window ) );
}

static void file_menu_new_site_callback( EggAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_site_druid_begin( application, window );
}

static void file_menu_new_document_callback( EggAction *action,
					     gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_page_druid_begin( application, window );
}

static void file_menu_new_blank_document_callback( EggAction *action, 
						   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	page = screem_page_new( G_OBJECT( app ) );
	screem_page_set_data( page, NULL );

	screem_site_add_page( site, page );
	screem_window_set_document( window, page );
	g_object_unref( app );
}

static void file_menu_new_site_template_callback( EggAction *action, 
						  gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );
	screem_site_create_template( window );
}

static void file_menu_open_document_callback( EggAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemSite *dsite;
	ScreemSite *site;
	gchar *spath;
	gchar **filenames;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );
	dsite = screem_application_get_default_site( application );

	site = screem_window_get_current( window );
	spath = screem_get_local_site_path( site );

	filenames = bonobo_file_selector_open_multi( GTK_WINDOW( window ),
						TRUE,
					      _( "Open document..." ),
					      NULL, spath );
	if( filenames ) {
		const gchar *filename;
		gint i;
		for( i = 0; filenames[ i ]; ++ i ) {
			filename = filenames[ i ];
			site = screem_window_get_current( window );
			if( ! screem_site_locate_page( site, filename ) ) {
				site = dsite;
			}
			screem_page_open_with_filename( site, 
						window, 
						filename );
		}
		g_strfreev( filenames );
	}
	g_free( spath );
	
}

static void file_menu_open_location_callback( EggAction *action,
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemWindowDetails *details;
	GladeXML *xml;
	GConfClient *client;
	gchar *glade_path;
	GtkWidget *widget;

	gchar *name;
	gboolean opage;
	gchar *txt;
	const gchar *prompt;

	const gchar *widg;
	
	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	if( ! strcmp( "Open Location", name ) ) {
		opage = TRUE;
		widg = "open_file_location";
	} else {
		opage = FALSE;
		widg = "open_site_location";
	}
	g_free( name );

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );
	details = window->details;
	
	client = gconf_client_get_default();
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, widg, NULL );
	g_free( glade_path );
	g_object_unref( client );

	widget = glade_xml_get_widget( xml, "prompt" );
	if( opage ) {
		prompt = _( "Enter the URL of the file to open" );
	} else {
		prompt = _( "Enter the URL of the site to open" );
	}
	txt = g_strconcat( "<b>", prompt, "</b>", NULL );
	gtk_label_set_markup( GTK_LABEL( widget ), txt );
	g_free( txt );

	widget = glade_xml_get_widget( xml, widg );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		const gchar *url;
		ScreemSite *site;
		
		widget = glade_xml_get_widget( xml, "url" );
		url = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		if( opage ) {
			site = screem_window_get_current( window );
			if( ! screem_site_locate_page( site, url ) ) {
				site = screem_application_get_default_site( application );
			}
			screem_page_open_with_filename( site, window,
							url );
		} else {
			screem_site_open_with_filename( window, 
						application, url );
		}
	}
	widget = glade_xml_get_widget( xml, widg );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}

static void file_menu_save_document_callback( EggAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemPage *page;
	const gchar *filepath;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	/* Check if the page has a file path. */
	filepath = screem_page_get_pathname(page);
	if( filepath && *filepath != '\0' ) {
		/* Filename is normal, try a save straight away */
		if( ! screem_page_save( page ) ) {
			/* failed to save! */
			GtkWidget *widget;
			gchar *temp;

			temp = gnome_vfs_unescape_string_for_display( filepath );			
			widget = gtk_message_dialog_new( GTK_WINDOW( window ),
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_OK,
							 _( "Failed to save %s" ),
							 temp );
			g_free( temp );
			gtk_window_set_transient_for( GTK_WINDOW( widget ),
						      GTK_WINDOW( window ) );
			gtk_dialog_run( GTK_DIALOG( widget ) );
			gtk_widget_destroy( widget );
		}
	} else {
		/* If there's no file name, request one. */
		file_menu_save_document_as_callback( action, user_data );
	}
}

static void file_menu_save_document_as_callback( EggAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	
	window = SCREEM_WINDOW( user_data );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	if( screem_page_save_as( page, window ) == SAVE_AS_FAILED ) {
		GtkWidget *widget;

		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _( "Failed to save page" ) );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		gtk_dialog_run( GTK_DIALOG( widget ) );
		gtk_widget_destroy( widget );
	} 
}

static void file_menu_save_template_as_callback( EggAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	const gchar *pathname;
	gchar *pdata;
	gchar *ndata;
	gint status;
	
	window = SCREEM_WINDOW( user_data );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	status = screem_page_save_as( page, window );
	if( status == SAVE_AS_FAILED ) {
		GtkWidget *widget;

		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _( "Failed to save template" ) );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		gtk_dialog_run( GTK_DIALOG( widget ) );
		gtk_widget_destroy( widget );
	} else if( status == SAVE_AS_OK ) {
		pathname = screem_page_get_pathname( page );
		pdata = screem_page_get_data( page );

		ndata = g_strconcat( "<!-- #BeginTemplate \"", 
				pathname, "\" -->\n",
				pdata, "\n<!-- #EndTemplate -->", NULL );
		g_free( pdata );
		screem_page_set_data( page, ndata );
		g_free( ndata );
		screem_page_save( page );
	}
}


static void file_menu_close_document_callback( EggAction *action, 
					       gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	ScreemSite *site;
	gboolean ret;
	
	window = SCREEM_WINDOW( user_data );
	
	site = screem_window_get_current( window );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );
	ret = screem_page_close_page( page, window, TRUE );
	if( ret ) {
		screem_site_save( site );

		if( screem_site_get_fake_flag( site ) ) {
			screem_site_remove_page( site, page );
			g_object_unref( page );
		}
	}
}

static void file_menu_close_all_callback( EggAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	ScreemSite *site;
	GList *tmp;
	GList *tmp2;
	
	window = SCREEM_WINDOW( user_data );

	site = screem_window_get_current( window );

	tmp = g_list_copy( screem_mdi_get_document_list( window->details->mdi ) );
	screem_mdi_remove_all( window->details->mdi );
	
	if( screem_site_get_fake_flag( site ) ) {
		for( tmp2 = tmp; tmp2; tmp2 = tmp2->next ) {
			page = SCREEM_PAGE( tmp2->data );
			screem_site_remove_page( site, page );
			g_object_unref( page );
		}
	}
	g_list_free( tmp );
}

static void file_menu_open_site_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	
	gchar *filename;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	filename = bonobo_file_selector_open( NULL, FALSE,
					 _( "Select Site..." ),
					 NULL, NULL );

	if( filename ) {
		screem_site_open_with_filename( window, app, filename );

		g_free( filename );
	}
	g_object_unref( app );
}

static void file_menu_save_all_callback( EggAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	g_list_foreach( (GList*)screem_site_get_pages( site ),
			(GFunc)screem_page_save, NULL );
}


static void file_menu_close_site_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemEditor *editor;
	ScreemSite *site;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	editor = SCREEM_EDITOR( window->details->editor );

	site = screem_window_get_current( window );

	if( screem_site_get_fake_flag( site ) ) {
		g_assert( FALSE );
	} else if( screem_site_save_confirmation( site ) ) {
		ScreemSite *next;
		screem_site_close_site( site );
	
		screem_application_close_site( application, site );
	
		next = screem_application_get_default_site( application );
		screem_window_set_current( window, next );

		g_object_unref( site );
	}  
}

static void file_menu_print_callback( EggAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_print( window, FALSE );
}

static void file_menu_print_preview_callback( EggAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_print( window, TRUE );
}

static void file_menu_page_setup_callback( EggAction *action,
					   gpointer user_data )
{
	screem_preferences_page_setup();
}

static void file_menu_offline_callback( EggAction *action, 
					gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_toggle_online_status( window, GINT_TO_POINTER( 1 ) );
}

static void file_menu_exit_application_callback( EggAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_application_close_all_windows( application );
}

static void edit_menu_undo_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_view_undo( SCREEM_VIEW( window->details->current ) );
}

static void edit_menu_redo_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_view_redo( SCREEM_VIEW( window->details->current ) );
}

static void edit_menu_cut_callback( EggAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_cut( editor );
	}
}

static void edit_menu_copy_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_copy( editor );
	}
}

static void edit_menu_paste_callback( EggAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_paste( editor );
	}
}

static void edit_menu_paste_enc_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_paste_encoded( editor );
	}
}

static void edit_menu_clear_callback( EggAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_clear_selection( editor );
	}
}

static void edit_menu_select_all_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_select_region( editor, 0, 0 );
	}
}

static void edit_menu_select_context_callback( EggAction *action, 
					       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_context( page, 
				pos, &start, &end, FALSE ) ) {
		screem_editor_select_region( editor, start, 
					end - start );	
	}
}

static void edit_menu_mark_editable_callback( EggAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	GladeXML *xml;
	GConfClient *client;
	gchar *glade_path;
	GtkWidget *widget;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	client = gconf_client_get_default();
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "mark_editable", NULL );
	g_free( glade_path );
	g_object_unref( client );
	
	widget = glade_xml_get_widget( xml, "mark_editable" );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		const gchar *name;
		gchar *start;
		
		widget = glade_xml_get_widget( xml, "region_name" );
		name = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		start = g_strconcat( "<!-- #BeginEditable \"", name,
					"\" -->\n", NULL );
		
		screem_editor_insert_markup( SCREEM_EDITOR( details->editor ),
						start, 
						"<!-- #EndEditable -->\n" );
		g_free( start );
	}
	widget = glade_xml_get_widget( xml, "mark_editable" );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}

static void edit_menu_auto_indent_callback( EggAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_auto_indent( editor, pos );
}

static void edit_menu_indent_callback( EggAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_indent( editor, pos );
}

static void edit_menu_unindent_callback( EggAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_unindent( editor, pos );
}

static void edit_menu_encode_entities_callback( EggAction *action,
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	screem_editor_encode_text( editor, FALSE );
}

static void edit_menu_url_encode_callback( EggAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	screem_editor_encode_text( editor, TRUE );
}

static void edit_menu_case_callback( EggAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;
	gchar *name;
	guint start;
	guint end;
	gchar *text;
	gchar *tmp;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	if( page ) {
		if( ! screem_editor_has_selection( editor, 
						&start, &end ) ) {
			start = 0;
			end = gtk_text_buffer_get_char_count( GTK_TEXT_BUFFER( page ) );
		}

		g_object_get( G_OBJECT( action ), "name", &name, NULL );
		text = screem_editor_get_text( editor, start,
				end - start );
		tmp = screem_markup_change_case( text,
				! strcmp( "TagUpper", name ) );
		screem_editor_delete_forward( editor, start,
				end - start );
		screem_editor_insert( editor, start, tmp );
		g_free( text );
		g_free( tmp );
		g_free( name );
	}
}

static void edit_menu_bookmarks_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	page = screem_window_get_document( window );

	screem_page_edit_bookmarks( window, editor, page );
}

static void edit_menu_search_callback( EggAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	if( ! strcmp( "Find", action->name ) ) {
		screem_search_show_dialog( window->search, FALSE );
	} else if( ! strcmp( "FindAgain", action->name ) ) {
		screem_search_set_from( window->search,
				SCREEM_SEARCH_CURRENT );
		screem_search_do_find( window->search );	
	} else {
		screem_search_show_dialog( window->search, TRUE );
	}
}

static void edit_menu_goto_line_callback( EggAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	GladeXML *xml;
	GConfClient *client;
	gchar *glade_path;
	GtkWidget *widget;
	ScreemPage *page;
	guint lines;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	page = screem_window_get_document( window );

	if( ! page ) {
		return;
	}
	
	client = gconf_client_get_default();
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "gotoline", NULL );
	g_free( glade_path );
	g_object_unref( client );

	widget = glade_xml_get_widget( xml, "line" );
	lines = gtk_text_buffer_get_line_count( GTK_TEXT_BUFFER( page ) );
	gtk_spin_button_set_range( GTK_SPIN_BUTTON( widget ),
				0.0, (gdouble)lines );
	
	widget = glade_xml_get_widget( xml, "gotoline" );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( xml, "line" );
		lines = (guint)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
	
		editor = SCREEM_EDITOR( window->details->editor );
		screem_editor_goto_line( editor, lines );
	}
	widget = glade_xml_get_widget( xml, "gotoline" );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}

static void view_menu_dock_show_callback( EggAction *action,
					  gpointer user_data )
{
	ScreemWindow *window;
	EggDockItem *item;
	gboolean show;	
		
	window = SCREEM_WINDOW( user_data );
	
	show = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );
	
	item = EGG_DOCK_ITEM( g_object_get_data( G_OBJECT( action ), "item" ) );
	
	/* check item here as it is NULL when we set the initial toggle state */
	if( item ) {
		if( show ) {
			egg_dock_item_show_item( item );
		} else {
			egg_dock_item_hide_item( item );
		}
	} else if( ! strcmp( "ViewMainToolbar", action->name ) ) {
		GtkWidget *widget;
	
		widget = egg_menu_merge_get_widget( EGG_MENU_MERGE( window->merge ),
							"/Main Toolbar" );	
		if( widget ) {
			if( show ) {
				gtk_widget_show( widget );
			} else { 
				gtk_widget_hide( widget );
			}
		}
		gconf_client_set_bool( window->application->client,
					"/apps/screem/ui/show_main_toolbar",
					show, NULL );
	} else if( ! strcmp( "ViewWizardToolbar", action->name ) ) {
		GtkWidget *widget;
	
		widget = egg_menu_merge_get_widget( EGG_MENU_MERGE( window->merge ),
							"/Wizards Toolbar" );	
		if( widget ) {
			if( show ) {
				gtk_widget_show( widget );
			} else { 
				gtk_widget_hide( widget );
			}
		}
		gconf_client_set_bool( window->application->client,
					"/apps/screem/ui/show_wizard_toolbar",
					show, NULL );
	}
}

static void view_menu_editor_callback( EggAction *action, 
					gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );

	screem_window_change_view( window, window->details->editor );
}

static void view_menu_preview_callback( EggAction *action, 
					gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );

	screem_window_change_view( window, window->details->preview );
}

static void view_menu_linkview_callback( EggAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );

	screem_window_change_view( window, window->details->link_view );
}

static void view_menu_treeview_callback( EggAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );

	screem_window_change_view( window, window->details->tree_view );
}

static void insert_doctype_sel_callback( GtkTreeSelection *sel,
					 GtkWidget *entry )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	gchar *public;
	
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				0, &public,
				-1 );
		
		gtk_entry_set_text( GTK_ENTRY( entry ), public );
		g_free( public );
	}
}

static void insert_doctype_activate_callback( GtkTreeView *view,
					GtkTreePath *path,
					GtkTreeViewColumn *col,
					GtkDialog *dialog )
{
	gtk_dialog_response( dialog, GTK_RESPONSE_OK );
}

static void insert_menu_doctype_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );
	ScreemApplication *application;
	ScreemDTDDB *db;
	GladeXML *xml;
	gchar *glade_path;
	gchar *defaultdtd;
	GtkWidget *widget;
	gint button;

	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkListStore *store;
	GtkTreeSelection *sel;
	GtkWidget *dialog;
	
	application = SCREEM_APPLICATION( window->application );

	glade_path = gconf_client_get_string( application->client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "insertdtd", NULL );
	dialog = glade_xml_get_widget( xml, "insertdtd" );
	g_free( glade_path );

	db = screem_application_get_dtd_db( application );
	store = screem_dtd_db_get_store( db );

	widget = glade_xml_get_widget( xml, "doctypes" );
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 0, NULL ); 
	gtk_tree_view_column_set_title( col, "PUBLIC" );

	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 1, NULL ); 
	gtk_tree_view_column_set_title( col, "SYSTEM" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), 
			GTK_TREE_MODEL( store ) );

	g_signal_connect( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK(insert_doctype_activate_callback),
			  dialog );
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	widget = glade_xml_get_widget( xml, "public" );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( insert_doctype_sel_callback ),
			  widget );
	
	defaultdtd = gconf_client_get_string(application->client,
					     "/apps/screem/editor/default_dtd",
					     NULL );
	gtk_entry_set_text( GTK_ENTRY( widget ), defaultdtd );
	g_free( defaultdtd );

	gtk_window_set_transient_for( GTK_WINDOW( dialog ),
				      GTK_WINDOW( window ) );
	button = gtk_dialog_run( GTK_DIALOG( dialog ) );

	if( button == GTK_RESPONSE_OK ) {
		ScreemEditor *editor;
		const gchar *txt;

		editor = SCREEM_EDITOR( window->details->editor );

		widget = glade_xml_get_widget( xml, "public" );
		txt = gtk_entry_get_text( GTK_ENTRY( widget ) );

		if( ! strncmp( "<!DOCTYPE", txt, strlen( "<!DOCTYPE" ) ) ) {
			/* full doctype tag was entered, insert */
			screem_editor_insert( editor, 0, txt );
		} else {
			/* build doctype tag */
			gchar *tag;
			gchar *systemid;
			gchar *root;
			ScreemDTD *dtd;
		
			dtd = screem_dtd_db_get_dtd( db, txt, NULL ); 
			root = screem_dtd_db_get_root( db, txt, NULL );
			if( ! root ) {
				root = g_strdup( "unknown" );
			}

			systemid = screem_dtd_db_get_system_id( db,
							     txt,
							     NULL );
		
			tag = g_strconcat( "<!DOCTYPE ", root,
					   " PUBLIC \"", txt, 
					   "\" \"", systemid,
					   "\">", NULL );
			screem_editor_insert( editor, 0, tag );
			g_free( root );
			g_free( systemid );
			g_free( tag );
		}
	}

	gtk_widget_destroy( dialog );
	g_object_unref( G_OBJECT( xml ) );
}

static void insert_menu_file_callback( EggAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	gchar *filename;
	gchar *spath;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;

	site = screem_window_get_current( window );
	spath = screem_get_local_site_path( site );

	filename = bonobo_file_selector_open( NULL, TRUE,
					      _( "Open document..." ),
					      NULL, spath );
	g_free( spath );

	if( filename && details->current == details->editor ) {
		screem_editor_insert_file( SCREEM_EDITOR( details->editor ),
					   filename );
	}

	g_free( filename );
}

static void site_menu_site_settings_callback( EggAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;

	window = SCREEM_WINDOW( user_data );

	site = screem_window_get_current( window );

	screem_site_settings_dialog( site, window );
}

static void edit_menu_toolbars_callback( EggAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	
	window = SCREEM_WINDOW( user_data );
	
}

static void edit_menu_preferences_callback( EggAction *action, 
					    gpointer user_data )
{
	ScreemApplication *app;

	g_object_get( G_OBJECT( user_data ), "app", &app, NULL );
	
	screem_preferences_edit( app );

	g_object_unref( app );
}

/* FIXME: put some place sensible, doesn't belong in ScreemCVS though */
static gboolean screem_cvs_dialog( ScreemWindow *window,
				   gchar **log,
				   gchar **root,
				   gchar **module,
				   gchar **base,
				   gchar **flags )
{
	ScreemApplication *application;
	GConfClient *client;
       	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *dialog;
	GtkWidget *widget;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	const gchar *value;
	gint response;

	application = SCREEM_APPLICATION( window->application );
	client = application->client;
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "cvs_dialog", NULL );
	g_free( glade_path );

	dialog = glade_xml_get_widget( xml, "cvs_dialog" );

	if( log ) {
		widget = glade_xml_get_widget( xml, "logframe" );
		gtk_widget_show( widget );
	}

	if( root || module || base ) {
		widget = glade_xml_get_widget( xml, "checkoutframe" );
		gtk_widget_show( widget );
	}

	widget = glade_xml_get_widget( xml, "root" );
	gtk_widget_set_sensitive( widget, root != NULL );
	if( root ) {
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *root );
	}
	widget = glade_xml_get_widget( xml, "module" );
	gtk_widget_set_sensitive( widget, module != NULL );
	if( module ) {
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *module );
	}
	widget = glade_xml_get_widget( xml, "checkoutto" );
	gtk_widget_set_sensitive( widget, base != NULL );
	if( base ) {
		widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY(widget));
		gtk_entry_set_text( GTK_ENTRY( widget ), *base );
	}
	
	if( flags ) {
		widget = glade_xml_get_widget( xml, "flagframe" );
		gtk_widget_show( widget );

		widget = glade_xml_get_widget( xml, "cvsflags" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *flags );
	}

	gtk_window_set_transient_for( GTK_WINDOW( dialog ),
				      GTK_WINDOW( window ) );
	response = gtk_dialog_run( GTK_DIALOG( dialog ) );

	if( response == GTK_RESPONSE_OK ) {
		if( log ) {
			widget = glade_xml_get_widget( xml, "cvslog" );
			buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
			gtk_text_buffer_get_start_iter( buffer, &it );
			gtk_text_buffer_get_end_iter( buffer, &eit );
			*log = gtk_text_buffer_get_text( buffer, &it, &eit,
							  TRUE );
		}

		if( root ) {
			widget = glade_xml_get_widget( xml, "root" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*root = g_strdup( value );
		}
		if( module ) {
			widget = glade_xml_get_widget( xml, "module" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*module = g_strdup( value );
		}
		if( base ) {
			widget = glade_xml_get_widget( xml, "checkoutto" );
			*base = gtk_editable_get_chars( GTK_EDITABLE(widget),
							0, -1 );
		}
		
		if( flags ) {
			widget = glade_xml_get_widget( xml, "cvsflags" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*flags = g_strdup( value );
		}
	}
	gtk_widget_destroy( dialog );
	g_object_unref( G_OBJECT( xml ) );

	return ( response == GTK_RESPONSE_OK );
}

static void cvs_menu_checkout_site_callback( EggAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	gchar *flags;
	gchar *module;
	gchar *root;
	gchar *base;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	flags = "";
	module = "";
	root = (gchar*)g_getenv( "CVSROOT" );
	if( ! root ) {
		root = "";
	}
	base = "";

	if( screem_cvs_dialog( window, NULL, &root, &module, &base, 
			       &flags ) ) {
		
		g_object_set( G_OBJECT( cvs ), 
			      "root", root,
			      "base", base, NULL );
		screem_cvs_checkout( cvs, flags, module );
		
		g_free( base );
		g_free( flags );
		g_free( module );
		g_free( root );
	}
}

static void cvs_menu_update_site_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *root;
	gchar *module;
	gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	root = (gchar*)screem_site_get_cvs_root( site );
	module = (gchar*)screem_site_get_name( site );
	base = (gchar*)screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, NULL, &root,
			       &module, &base, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );

		screem_cvs_update( cvs, NULL, flags );

		g_free( module );
		g_free( base );
		g_free( flags );
	}
}


static void cvs_menu_update_page_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *root;
	gchar *module;
	gchar *flags;
	gchar *base;
	
	gchar *temp;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	if( strcmp( "UpdateFile", action->name ) ) {
		page = screem_window_get_document( window );
		
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );
	if( ! screem_site_get_fake_flag( site ) ) {
		root = (gchar*)screem_site_get_cvs_root( site );
	       	module = (gchar*)screem_site_get_name( site );
		base = (gchar*)screem_site_get_pathname( site );
		temp = NULL;
	} else {
		root = (gchar*)g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		module = "";
		temp = base = g_dirname( pathname );
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, NULL, 
					   &root, &module, &base,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );

		screem_cvs_update( cvs, pathname, flags );

		g_free( root );
		g_free( module );
		g_free( base );
		g_free( flags );
	} 

	if( temp ) {
		g_free( temp );
	}
}

static void cvs_menu_import_site_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *log;
	gchar *root;
	gchar *module;
	gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	log = "";
	root = (gchar*)screem_site_get_cvs_root( site );
	if( ! root ) {
		root = (gchar*)g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
	}
	module = (gchar*)screem_site_get_name( site );
	base = (gchar*)screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, &log, 
			       &root, &module, &base, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_import( cvs, flags, log, module );

		g_free( log );
		g_free( root );
		g_free( module );
		g_free( base );
		g_free( flags );
	}
}

static void cvs_menu_commit_site_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	log = "";
	root = screem_site_get_cvs_root( site );
	if( ! root ) {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
	}
	base = screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, &log, 
			       NULL, NULL, NULL, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_commit( cvs, NULL, flags, log );

		g_free( log );
		g_free( flags );
	}
}

static void cvs_menu_commit_page_callback( EggAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *module;
	gchar *flags;

	gchar *temp;

	gchar **troot;
	gchar **tbase;
	gchar **tmodule;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	if( strcmp( "CommitFile", action->name ) ) {
		page = screem_window_get_document( window );
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );

	if( ! screem_site_get_fake_flag( site ) ) {
		log = "";
		root = screem_site_get_cvs_root( site );
		if( ! root ) {
			root = g_getenv( "CVSROOT" );
			if( ! root ) {
				root = "";
			}
		}
		base = screem_site_get_pathname( site );
		troot = NULL;
		tmodule = NULL;
		tbase = NULL;
		temp = NULL;
	} else {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		if( pathname ) {
			temp =  g_dirname( pathname );
		} else {
			temp = NULL;
		}
		base = (const gchar*)temp;

		module = "";
		tbase = (gchar**)&base;
		tmodule = &module;
		troot = (gchar**)&root;
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, &log, 
					   troot, tmodule, tbase,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_commit( cvs, pathname, flags, log );

		g_free( log );
		g_free( flags );

		if( troot ) {
			g_free( *troot );
		}
		if( tmodule ) {
			g_free( *tmodule );
		}
		if( tbase ) {
			g_free( *tbase );
		}
	} 

	if( temp ) {
		g_free( temp );
	}
}

static void cvs_menu_add_page_callback( EggAction *action, 
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *module;
	gchar *flags;

	gchar *temp;

	gchar **troot;
	gchar **tbase;
	gchar **tmodule;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	if( strcmp( "AddFile", action->name ) ) {
		page = screem_window_get_document( window );
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );

	if( ! screem_site_get_fake_flag( site ) ) {
		log = "";
		root = screem_site_get_cvs_root( site );
		if( ! root ) {
			root = g_getenv( "CVSROOT" );
			if( ! root ) {
				root = "";
			}
		}
		base = screem_site_get_pathname( site );
		troot = NULL;
		tmodule = NULL;
		tbase = NULL;
		temp = NULL;
	} else {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		if( pathname ) {
			temp =  g_dirname( pathname );
		} else {
			temp = NULL;
		}
		base = (const gchar*)temp;

		module = "";
		tbase = (gchar**)&base;
		tmodule = &module;
		troot = (gchar**)&root;
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, &log, 
					   troot, tmodule, tbase,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_add( cvs, pathname, flags, log );

		g_free( log );
		g_free( flags );

		if( troot ) {
			g_free( *troot );
		}
		if( tmodule ) {
			g_free( *tmodule );
		}
		if( tbase ) {
			g_free( *tbase );
		}
	} 

	if( temp ) {
		g_free( temp );
	}
}

static void cvs_menu_login_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;

	gchar *root;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	root = (gchar*)g_getenv( "CVSROOT" );
	if( ! root ) {
		root = "";
	}
	flags = "";

	if( screem_cvs_dialog( window, NULL, 
			       &root, NULL, NULL, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root, NULL );
	
		//	screem_cvs_import( cvs, flags, log, module );

		g_free( root );
		g_free( flags );
	}
}

static void tool_menu_fix_links_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	GConfClient *client;
	gchar *glade_path;
	GladeXML *xml;
	GtkWidget *widget;
	
	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );
	client = application->client;
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "fixlinks", NULL );
	g_free( glade_path );

	widget = glade_xml_get_widget( xml, "fixlinks" ); 
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		ScreemSite *site;
		GtkWidget *entry;
		const gchar *sitepath;
		gchar *src;
		gchar *dest;
		
		site = screem_window_get_current( window );
		g_assert( ! screem_site_get_fake_flag( site ) );
		sitepath = screem_site_get_pathname( site );
		
		entry = glade_xml_get_widget( xml, "original" );
		src = (gchar*)gtk_entry_get_text( GTK_ENTRY( entry ) );
		if( strncmp( sitepath, src, strlen( sitepath ) ) ) {
			src = g_strconcat( sitepath, src + 1, NULL );
		} else {
			src = g_strdup( src );
		}
		entry = glade_xml_get_widget( xml, "new" );
		dest = (gchar*)gtk_entry_get_text( GTK_ENTRY( entry ) );
		if( strncmp( sitepath, dest, strlen( sitepath ) ) ) {
			dest = g_strconcat( sitepath, dest + 1 , NULL );
		} else {
			dest = g_strdup( dest );
		}		
		screem_site_file_change( site, src, dest );
		g_free( src );
		g_free( dest );
	}
	gtk_widget_destroy( widget );
	g_object_unref( G_OBJECT( xml ) );
}

static void tool_menu_template_update_callback( EggAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemPage *page;
	gchar *tag;
	gchar *path;
	const gchar *pathname;
	ScreemPage *template;
	GSList *blocks;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	site = screem_window_get_current( window );
	page = screem_window_get_document( window );
	if( page ) {
		if( screem_page_is_template( page, &tag, &path, NULL ) ) {

			/* the page appears to use/be a template, check which */
			
			pathname = screem_page_get_pathname( page );
			g_free( tag );
			
			if( pathname && strcmp( pathname, path ) ) {
				
				/* it is using a template */
			
				template = screem_page_new( G_OBJECT( app ) );
				screem_page_set_pathname( template, path );

				/* we need the blocks from template, which
				 * may be different to those in the current page,
				 * which is why we didn't get them before */
			
				g_free( path );
				blocks = NULL;
				if( screem_page_is_template( template, 
								&tag, &path,
								&blocks ) ) {
					screem_markup_update_from_template( site, template, page, tag, blocks );
					g_slist_foreach( blocks, 
							(GFunc)g_free, NULL );
					g_slist_free( blocks );
				}
				g_object_unref( template );
			}
		}
	}
	g_object_unref( app );
}

static void edit_macros_callback( EggAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	screem_preferences_edit_macros( app );
	g_object_unref( app );
}

static void edit_helpers_callback( EggAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	screem_preferences_edit_helpers( app );
	g_object_unref( app );
}

static void doc_menu_previous_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	gtk_notebook_prev_page( GTK_NOTEBOOK( details->notebook ) );
}

static void doc_menu_next_callback( EggAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
		
	gtk_notebook_next_page( GTK_NOTEBOOK( details->notebook ) );
}


static void help_menu_help_callback( EggAction *action, 
				     gpointer user_data )
{
	GError *error;
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	error = NULL;

	gnome_help_display( "screem", NULL, &error );

	if (error) {
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new (GTK_WINDOW (window),
						 0,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Could not display help for Screem.\n"
						   "%s"),
						 error->message);
		
		g_signal_connect_swapped (dialog, "response",
					  G_CALLBACK (gtk_widget_destroy),
					  dialog);
		gtk_widget_show (dialog);
		
		g_error_free (error);
	}
}

static void help_menu_about_screem_callback( EggAction *action, 
					     gpointer user_data )
{
	static GtkWidget *about = NULL;

	const gchar *COPYRIGHT = "1999-2003 David A Knight";
	const gchar *AUTHORS[] = {
		"David A Knight - Main Author",
		"Lee Mallabone - Sitecopy Integration / General ",
		"Joe Orton - Sitecopy",
		"Sven Liessem - General",
		"Matt Colyer - General",
		NULL
	};
	const gchar *DESCRIPTION = _( "\
SCREEM (Site CReation and Editing EnvironMent)\n\
Screem provides an integrated development environment for creating and maintaining websites\n\
SCREEM is licensed under the GNU General Public License version 2 or above" );
        GdkPixbuf *logo = NULL;
	const gchar *DOCUMENTERS[] = {
		"Matt Colyer",
		"David A Knight",
		NULL
	};
	const gchar *TRANSLATERS = _( "Translater Credits" );

	if( ! about ) {
		logo = gdk_pixbuf_new_from_file( SPLASH_DIR"/splash.png", NULL );
	
		about = gnome_about_new( "Screem", VERSION,
					 COPYRIGHT,
					 DESCRIPTION,
					 AUTHORS,
					 DOCUMENTERS,
				 	TRANSLATERS,
				 	logo );
		g_signal_connect_swapped( G_OBJECT( about ), "delete_event",
					G_CALLBACK( g_nullify_pointer ),
					&about );
		g_signal_connect_swapped( G_OBJECT( about ), "destroy",
					G_CALLBACK( g_nullify_pointer ),
					&about );
	}
	gtk_window_set_transient_for( GTK_WINDOW( about ),
				      	GTK_WINDOW( user_data ) );
	gtk_widget_show( about );
	gdk_window_raise( GDK_WINDOW( about->window ) );
}

static void help_menu_tip_callback( EggAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	gtk_window_set_transient_for(GTK_WINDOW(application->hint),
				     GTK_WINDOW(window));
	
	gtk_widget_show( application->hint );
}

static void help_menu_debug_callback( EggAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

}


/* Site view menu: FIXME doesn't belong here */

static void create_dir_callback( const gchar *string, gpointer data )
{
	ScreemWindow *window;
	ScreemSite *site;
	GtkWidget *widget;

	window = SCREEM_WINDOW( data );
	site = screem_window_get_current( window );

	widget = GTK_WIDGET( g_object_get_data( G_OBJECT( window ),
						"siteview" ) );

	if( ! string ) {
                screem_window_show_message( window,
					    _( "No directory name specified." ) );
	} else if( ! screem_site_view_create_dir( site, widget, string ) ) {
                screem_window_show_message( window,
					    _( "Directory could not be created." ) );
        }
}

static void site_view_menu_new_dir_callback( EggAction *action, 
					     gpointer user_data )
{
	GtkWidget *widget;

	widget = gnome_request_dialog( FALSE, 
				       _("Enter the directory name to create:"),
				       "", 255, 
				       (GnomeStringCallback)create_dir_callback,
				       user_data, NULL );
	gtk_window_set_title( GTK_WINDOW( widget ), 
				"Create Directory: - Screem" );
}

static void site_view_menu_delete_file_callback( EggAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;
	GtkWidget *confirm;
	GtkWidget *label;
	GConfClient *client;
	gchar *glade_path;
	gchar *txt;
	gchar *base;
	gchar *temp;
	GladeXML *xml;
		
	window = SCREEM_WINDOW( user_data );

	site = screem_window_get_current( window );
	view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );
	
	gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	if( uri ) {		
		client = gconf_client_get_default();
		glade_path = gconf_client_get_string( client,
						      "/apps/screem/general/glade_path",
						      NULL );
		xml = glade_xml_new( glade_path, "delete_file", NULL );
		g_free( glade_path );
		g_object_unref( client );
	
		confirm = glade_xml_get_widget( xml, "delete_file" );
		label = glade_xml_get_widget( xml, "delete_label" );
		base = g_path_get_basename( uri );
		temp = gnome_vfs_unescape_string_for_display( base );
		txt = g_strdup_printf( _( "Are you sure you want to permanently delete \"%s\"?" ),
					temp );
		g_free( temp );
		g_free( base );
		gtk_label_set( GTK_LABEL( label ), txt );
		g_free( txt );
		
		gtk_window_set_wmclass( GTK_WINDOW( confirm ),
					"Screem",
					"delete_confirm_prompt" );
		gtk_window_set_transient_for( GTK_WINDOW( confirm ),
					      GTK_WINDOW( window ) );
		gtk_window_set_type_hint( GTK_WINDOW( confirm ), 
					  GDK_WINDOW_TYPE_HINT_DIALOG );
		if(gtk_dialog_run( GTK_DIALOG( confirm ) ) == GTK_RESPONSE_OK){
			delete_dir( uri );
		}
		gtk_widget_destroy( confirm );
		g_object_unref( G_OBJECT( xml ) );
		g_free( uri );
	}
}

static void site_view_menu_file_props_callback( EggAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	ScreemSite *site;
	GtkTreeModel *model;
	GtkWidget *view;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;

	window = SCREEM_WINDOW( user_data );
	details = window->details;

	site = screem_window_get_current( window );
	view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
						"siteview" ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	gtk_tree_selection_get_selected( selection, &model, &it );
	
	gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	
	if( uri ) {
		/* display info on info fullname */
		screem_site_view_show_properties( site, uri,
						  GTK_WINDOW( window ) );
		g_free( uri );
	}
}

void screem_window_recent_page_cb( GeditRecent *recent, const gchar *uri,
				   gpointer data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemSite *site;

	window = SCREEM_WINDOW( data );
	application = SCREEM_APPLICATION( window->application );
	site = screem_application_get_default_site( application );

	screem_page_open_with_filename( site, window, uri );
}

void screem_window_recent_site_cb( GeditRecent *recent, const gchar *uri,
				   gpointer data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	
	window = SCREEM_WINDOW( data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	screem_site_open_with_filename( window, app, uri );
	g_object_unref( app );
}

/* toolbar callback stuff */
typedef enum {
	LEFT, CENTRE, JUSTIFY, RIGHT
} Align;

static void
toolbar_task_list_callback( EggAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;
	ScreemTodo *todo;

	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	todo = SCREEM_TODO( screem_site_get_todo( site ) );

	screem_todo_view( todo );
}

static void screem_toolbar_spell_check( EggAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	page = screem_window_get_document( window );

	g_object_set_data( G_OBJECT( window->details->spell ), "page", page );
	screem_spell_check_interactive( window->details->spell );
}

static void screem_toolbar_external_browser_notify( GConfClient *client,
						    guint cnxn_id,
						    GConfEntry *entry,
						    gpointer data )
{
	ScreemApplication *application;
	ScreemWindow *window;
	GSList *list;

	gchar *browser;
	gchar *conf;
	gchar *tip;
	gchar *stock_id;
	gint i;
	gboolean set;
	
	EggAction *action;
	
	window = SCREEM_WINDOW( data );
	application = SCREEM_APPLICATION( window->application );

	list = window->details->browser_notify_handles;


	i = g_slist_index( list, GINT_TO_POINTER( cnxn_id ) );
	i /= 2;
	i ++;

	browser = g_strdup_printf( "Browser%i", i );

	conf = g_strdup_printf( "/apps/screem/Browser%i_path", i );
	tip = gconf_client_get_string( application->client, conf,
				       NULL );
	g_free( conf );

	set = TRUE;
	if( ! tip || *tip == '\0' ) {
		set = FALSE;
		tip = g_strdup_printf( "External Browser %i Not Set",
				       i );
	}
	
	stock_id = g_strconcat( "Screem_", browser, NULL );
	
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						browser );
	g_object_set( G_OBJECT( action ), 
			"tooltip", tip,
			"visible", set, 
			"stock_id", stock_id, NULL );
	
	g_free( tip );
	g_free( stock_id );
	g_free( browser );
}

void screem_toolbar_external_browser( EggAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemPage *page;
	const gchar *pagepath;

	gchar *conf;
	gchar *browserpath;
	gboolean use_url;
	gboolean dynamic;

	window = SCREEM_WINDOW( user_data );

	/* the verb holds the browser we want, eg Browser1, Browser2 etc */

	application = SCREEM_APPLICATION( window->application );

	/* get browser settings */
	conf = g_strconcat( "/apps/screem/", action->name, "_path", NULL );
	browserpath = gconf_client_get_string( application->client,
					       conf, NULL );
	g_free( conf );

	if( ! browserpath && ! strcmp( "Browser0", action->name ) ) {
		/* default browser, hasn't been overridden for
		   SCREEM, so use gnome default,
		   /desktop/gnome/applications/browser/exec */
		browserpath = gconf_client_get_string( application->client,
						       "/desktop/gnome/applications/browser/exec", NULL );
	}

	page = screem_window_get_document( window );
	dynamic = screem_page_is_dynamic( page );

	if( ! dynamic ) {
		conf = g_strconcat( "/apps/screem/", action->name, "_static", NULL );
	} else {
		conf = g_strconcat( "/apps/screem/", action->name, "_dynamic", NULL );
	}
	use_url = gconf_client_get_bool( application->client, conf, NULL );
	g_free( conf );

	if( ! browserpath ) {
		/* FIXME: hmm, no browserpath available, alert user */
	} else if( (screem_page_save_confirmation( page ) != 
		    GTK_RESPONSE_CANCEL) ) {
		pagepath = screem_page_get_pathname( page );

		if( use_url ) {
			ScreemSite *site;
			const gchar *sitepath;
			gchar *uri;
			
			site = screem_window_get_current( window );
			
			sitepath = screem_site_get_pathname( site );
			
			uri = g_strconcat( browserpath, " http://",
					   screem_site_get_http_url( site ),
					   pagepath + strlen( sitepath ), 
					   NULL );
			
			gnome_execute_shell( NULL, uri );
			g_free( uri );
		} else {
			gchar *file;
			gchar *rest;
			gchar *uri;
			gint pos;

			file = strstr( browserpath, "%f" );
			if( file ) {
				pos = file - browserpath;
				rest = file + 2;
				if( rest > ( browserpath + strlen( browserpath ) ) )
					rest = "";
				file = g_strndup( browserpath, pos );
			} else {
				/* no %f, assume path just goes on the end */
				file = g_strdup( browserpath );
				rest = "";
			}
			uri = g_strconcat( file, " ", pagepath, " ", rest,
					   NULL );
			g_free( file );

			gnome_execute_shell( NULL, uri );

			g_free( uri );
		}
	}
	if( browserpath )
		g_free( browserpath );
}

static void screem_toolbar_create_browsers( ScreemWindow *window )
{
	gint i;

	ScreemApplication *application;
	gchar *conf;
	const int NO_OF_CUSTOM_BROWSERS = 3;

	EggActionGroupEntry *entry;
	
	static EggActionGroupEntry browser_action_entries[] = {
		{ "Action Name", "Action Label",
	  	   GTK_STOCK_NEW, NULL, "tip",
	  	   G_CALLBACK( NULL ), NULL },
	};
	
	application = SCREEM_APPLICATION( window->application );

	for( i = 1; i <= NO_OF_CUSTOM_BROWSERS; ++ i ) {
		gchar *data;
		gchar *label;
		gchar *tip;
		gchar *browser;
		gchar *menu;
		gint handle;
		gboolean set;
		
		browser = g_strdup_printf( "Browser%i", i );
		menu = g_strdup_printf( "<Root><menu><submenu name=\"View\" verb=\"View\"><menuitem name=\"%s\" verb=\"%s\" /></submenu></menu><dockitem name=\"Main Toolbar\"><toolitem name=\"%s\" verb=\"%s\" /></dockitem></Root>",
					browser, browser, browser, browser );
		
		conf = g_strdup_printf( "/apps/screem/Browser%i_icon", i );
		data = gconf_client_get_string( application->client, conf,
						NULL );
		g_free( conf );
		if( data ) {
			/* change default stock icon */
			GdkPixbuf *pixbuf;
			gchar *id = g_strconcat( "Screem_", browser, NULL );
			GtkIconSet *set = gtk_icon_factory_lookup_default( id );
			GtkIconSource *src = gtk_icon_source_new();
				
			pixbuf = gdk_pixbuf_new_from_file( data, NULL );
			g_free( data );
			if( pixbuf ) {
				gtk_icon_source_set_pixbuf( src, pixbuf );
			
				gtk_icon_source_set_direction_wildcarded( src, TRUE );
				gtk_icon_source_set_state_wildcarded( src, TRUE );
				gtk_icon_source_set_size_wildcarded( src, TRUE );
				gtk_icon_source_set_size( src, GTK_ICON_SIZE_BUTTON );
				gtk_icon_set_add_source( set, src );
				
				g_object_unref( G_OBJECT( pixbuf ) );
				gtk_icon_source_free( src );
				data = id;
			} else {
				data = g_strconcat( "Screem_", browser, NULL );
				g_free( id );
			}
		} else {
			data = g_strconcat( "Screem_", browser, NULL );
		}

		conf = g_strdup_printf( "/apps/screem/Browser%i_path", i );
		tip = gconf_client_get_string( application->client, conf,
					       NULL );
		g_free( conf );
		set = TRUE;
		if( ! tip || *tip == '\0' ) {
			set = FALSE;
			g_free( tip );
			tip = g_strdup_printf( "External Browser %i Not Set",
					       i );
		}
		label = g_strdup_printf( "External Browser %i", i );

		entry = browser_action_entries;
		entry->name = browser;
		entry->label = label;
		entry->tooltip = tip;
		entry->stock_id = data;
		entry->callback = G_CALLBACK( screem_toolbar_external_browser );
		entry->user_data = window;
		egg_action_group_add_actions( EGG_ACTION_GROUP( window->action_group ),
						entry, 1 );
		g_free( label );
		g_free( tip );
		g_free( data );
		
		egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( window->merge ),
							menu, strlen( menu ), NULL );
		
		g_free( menu );
		
		if( ! set ) {
			EggAction *action;
			
			action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
								browser );
			g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		}
		
		g_free( browser );

		browser = g_strdup_printf( "/apps/screem/Browser%i_icon",
					   i );
		handle = gconf_client_notify_add( application->client, browser,
						  
						  screem_toolbar_external_browser_notify, 
						  window, NULL, NULL );
		window->details->browser_notify_handles =
			g_slist_append(window->details->browser_notify_handles,
				       GINT_TO_POINTER( handle ) );
		g_free( browser );

		browser = g_strdup_printf( "/apps/screem/Browser%i_path",
					   i );
		handle = gconf_client_notify_add( application->client, browser,
						  
						  screem_toolbar_external_browser_notify, 
						  window, NULL, NULL );
		window->details->browser_notify_handles =
			g_slist_append(window->details->browser_notify_handles,
				       GINT_TO_POINTER( handle ) );
		g_free( browser );

	}
}

void screem_toolbar_markup( ScreemEditor *editor, const gchar *tag )
{
	gchar *open;
	gchar *close;

	open = g_strconcat( "<", tag, ">", NULL );
	close = g_strconcat( "</", tag, ">", NULL );

	screem_editor_insert_markup( editor, open, close );

	g_free( close );
	g_free( open );
}

void screem_toolbar_align( ScreemEditor *editor, Align a )
{
	const gchar *open = NULL;
	const gchar *close = NULL;

	switch( a ) {
	case CENTRE:
		open = "<center>";
		close = "</center>";
		break;
	case RIGHT:
		open = "<div align=\"right\">";
		close = "</div>";
		break;
	case JUSTIFY:
		open = "<div align=\"justify\">";
		close = "</div>";
		break;
	case LEFT:
		open = "<div align=\"left\">";
		close = "</div>";
		break;
	}
	if( open && close )
		screem_editor_insert_markup( editor, open, close );
}


/* normal tag insertion handling */
static void quickbar_bold_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "b" );
}
static void quickbar_italic_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "i" );
}
static void quickbar_underline_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "u" );
}
static void quickbar_para_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "p" );
}
static void quickbar_bullet_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "ul" );
}
static void quickbar_numbered_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "ol" );
}

/* align handling */

static void quickbar_alignleft_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, LEFT );
}
static void quickbar_aligncentre_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, CENTRE );
}
static void quickbar_alignright_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, RIGHT );
}
static void quickbar_justify_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, JUSTIFY );
}


static void fontbar_font_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "font" );
}
static void fontbar_pre_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "pre" );
}
static void fontbar_subscript_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "sub" );
}

static void fontbar_superscript_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "sup" );
}



static void tablebar_table_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "table" );
}

static void tablebar_tablebody_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "tbody" );
}

static void tablebar_tr_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "tr" );
}

static void tablebar_th_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "th" );
}

static void tablebar_td_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "td" );
}

static void tablebar_caption_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "caption" );
}

static void tablebar_summary_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "summary" );
}


static void formbar_textarea_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "textarea" );	
}

static void formbar_input_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, "<input name=\"\">", NULL );
}

static void formbar_fileinput_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, "<input type=\"file\" name=\"\">",
					NULL );
}

static void formbar_button_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "button" );
}

static void formbar_checkbutton_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<input type=\"checkbox\" name=\"\">",
					NULL );
}

static void formbar_radiobutton_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<input type=\"radio\" name=\"\">", NULL );
}

static void formbar_optionmenu_callback( EggAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<select><option>",
					"</option></select>" );
}

static void insert_basic_html_callback( EggAction *action, 
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemDTDDB *db;
	ScreemEditor *editor;
	ScreemPage *page;
	const gchar *pathname;
	gchar *doctype;
	gchar *text;
	gboolean xhtml;
	const gchar *head;
	gint pos;
	const gchar *metas[] = {
		"Generator",
		"Author",
		"Content-Type",
		NULL
	};
	
	window = SCREEM_WINDOW( user_data );

	application = SCREEM_APPLICATION( window->application );
	db = screem_application_get_dtd_db( application );

	editor = SCREEM_EDITOR( window->details->editor );

	page = screem_window_get_document( window );
	pathname = screem_page_get_pathname( page );
	
	doctype = gconf_client_get_string( application->client,
				     "/apps/screem/editor/default_dtd",
					     NULL );
	xhtml = FALSE;
	if( doctype ) {
		xhtml = ( strstr( doctype, " XHTML " ) != NULL );
	}
	if( pathname && ! xhtml ) {
		xhtml = ! strcmp( ".xml", 
				g_extension_pointer( pathname ) );
		xhtml |= ! strcmp( ".xhtml",
				g_extension_pointer( pathname ) );
	}
	if( ! xhtml ) {
		text = screem_page_get_data( page );
		xhtml = ! strncmp( "<?xml", text, strlen( "<?xml" ) );
		g_free( text );
	}
	
	text = screem_markup_basic_html( db, doctype, FALSE, xhtml );
	g_free( doctype );

	screem_editor_insert( editor, -1, text );

	head = strstr( text, "</head>" );
	if( head ) {
		pos = ( head - text );
		screem_editor_set_pos( editor, pos );
		for( pos = 0; metas[ pos ]; ++ pos ) {
			action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), metas[ pos ] );
			egg_action_activate( action );
			screem_editor_insert( editor, -1, "\n" );
		}
	}

	/* auto indent the inserted text */
	screem_editor_select_region( editor, 0, 0 );
	screem_editor_auto_indent( editor, 0 );
	
	g_free( text );
}

static void meta_insert_callback( EggAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;
	ScreemDTD *dtd;
	gchar *id;
	const gchar *pathname;
	gchar *data;
	gchar *name;
	GString *markup;
	const gchar *pattern1 = "<meta name=\"%s\" content=\"%s\" ";
	const gchar *pattern2 = "<meta http-equiv=\"%s\" content=\"%s\" ";
	const gchar *pattern;
	const gchar *type;
	gchar *content;
	gboolean xhtml;
	gchar *charset;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	page = screem_window_get_document( window );
	pathname = screem_page_get_pathname( page );
	dtd = screem_page_get_dtd( page );

	if( ! page ) {
		return;
	}
	
	id = NULL;
	xhtml = FALSE;
	if( dtd ) {
		g_object_get( G_OBJECT( dtd ), "public", &id, NULL );
	}
	if( id ) {
		xhtml = ( strstr( id, " XHTML " ) != NULL );
		g_free( id );
	}
	if( pathname && ! xhtml ) {
		xhtml = ! strcmp( ".xml", 
				g_extension_pointer( pathname ) );
		xhtml |= ! strcmp( ".xhtml",
				g_extension_pointer( pathname ) );
	}
	if( ! xhtml ) {
		data = screem_page_get_data( page );
		xhtml = ! strncmp( "<?xml", data, strlen( "<?xml" ) );
		g_free( data );
	}

	g_object_get( G_OBJECT( action ), "name", &name, NULL );

	markup = g_string_new( NULL );

	pattern = pattern1;
	type = "";
	content = NULL;
	if( ! strcmp( "Generator", name ) ) {
		pattern = pattern1;
		type = "generator";
		content = g_strdup( PACKAGE_STRING );
	} else if( ! strcmp( "Author", name ) ) {
		pattern = pattern1;
		type = "author";
		content = g_strdup( g_get_real_name() );
	} else if( ! strcmp( "Content-Type", name ) ) {
		pattern = pattern2;
		type = "Content-Type";
		charset = screem_page_get_charset( page );
		content = g_strdup_printf( "%s;charset=%s",
				screem_page_get_mime_type( page ),
				charset );
		g_free( charset );
	} else if( ! strcmp( "MetaRefresh", name ) ) {
		pattern = pattern2;
		type = "Refresh";
		content = g_strdup( "10" );
	}

	if( ! content ) {
		content = g_strdup( "" );
	}
	g_string_append_printf( markup, pattern,
				type, content );

	if( xhtml ) {
		g_string_append( markup, "/>" );
	} else {
		g_string_append( markup, ">" );
	}

	screem_editor_insert_markup( editor, markup->str, NULL );

	g_string_free( markup, TRUE );

	g_free( content );
	g_free( name );
}

/* initialise stuff */
static EggActionGroupEntry screem_window_action_entries[] = {
	{ "File Menu", N_("_File"), NULL, NULL, NULL, NULL, NULL },
	{ "New Menu", N_("_New"), GTK_STOCK_NEW, NULL, NULL, NULL, NULL },
	{ "RecentPages", N_("Recent Pages"), NULL, NULL, NULL, NULL, NULL },
	{ "Edit", N_("_Edit"), NULL, NULL, NULL, NULL, NULL },
	{ "Editable Regions", N_("Editable Regions"), NULL, NULL, NULL, NULL, NULL },
	{ "Block Operations", N_("Block Operations"), NULL, NULL, NULL, NULL, NULL },
	{ "View", N_("_View"), NULL, NULL, NULL, NULL, NULL },
	{ "Search", N_("Se_arch"), NULL, NULL, NULL, NULL, NULL },
	{ "Insert", N_("_Insert"), NULL, NULL, NULL, NULL, NULL },
	{ "Wizards", N_("Wizards"), NULL, NULL, NULL, NULL, NULL },
	{ "HTML", N_("HTML"), NULL, NULL, NULL, NULL, NULL },
	{ "Formatting", N_("Formatting"), NULL, NULL, NULL, NULL, NULL },
	{ "Fonts", N_("Fonts"), NULL, NULL, NULL, NULL, NULL },
	{ "Tables", N_("Tables"), NULL, NULL, NULL, NULL, NULL },
	{ "Forms", N_("Forms"), NULL, NULL, NULL, NULL, NULL },
	{ "Site", N_("_Site"), NULL, NULL, NULL, NULL, NULL },
	{ "RecentSites", N_("Recent Sites"), NULL, NULL, NULL, NULL, NULL },
	{ "SwitchTo", N_("Switch to site"), NULL, NULL, NULL, NULL, NULL },
	{ "Tools", N_("_Tools"), NULL, NULL, NULL, NULL, NULL },
	{ "CVS", N_("CVS"), NULL, NULL, NULL, NULL, NULL },
	{ "HelpMenu", N_("_Help"), NULL, NULL, NULL, NULL, NULL },
	{ "Helpers", N_( "_Helpers" ), NULL, NULL, NULL, NULL, NULL },
	{ "Documents", N_("_Documents"), NULL, NULL, NULL, NULL, NULL },

	{ "FakeTop", (""), NULL, NULL, NULL, NULL, NULL },
	
	{ "New Window", N_("New _Window"),
	  GTK_STOCK_NEW, NULL, N_("Open a new window"),
	  G_CALLBACK( file_menu_new_window_callback ), NULL },

	{ "New Site", N_("New _Site..."),
	  GTK_STOCK_NEW, NULL, N_("Create a new site"),
	  G_CALLBACK( file_menu_new_site_callback ), NULL },

	{ "New Document", N_("New Document..."),
	  GTK_STOCK_NEW, "<Control>N", N_("Create a new document"),
	  G_CALLBACK( file_menu_new_document_callback ), NULL },

	{ "New Blank Document", N_("New Blank Document"),
	  GTK_STOCK_NEW, NULL, N_("Create a new empty document"),
	  G_CALLBACK( file_menu_new_blank_document_callback ), NULL },

	{ "New Site Template", N_("New Site Template..."),
	  GTK_STOCK_NEW, NULL, N_("Create a new site template"), 
	  G_CALLBACK( file_menu_new_site_template_callback ), NULL },

	{ "Open Site", N_("Open..."),
	  GTK_STOCK_OPEN, NULL, N_("Open a site"), 
	  G_CALLBACK( file_menu_open_site_callback ), NULL },

	{ "Open Site Location", N_("Open Location..."),
	  GTK_STOCK_OPEN, NULL, N_("Open a site from a specified location"), 
	  G_CALLBACK( file_menu_open_location_callback ), NULL },

	{ "Save All", N_("Save All"),
	  GTK_STOCK_SAVE, "<Control><Shift>L",
	  N_("Save All modified documents in the current site"),
	  G_CALLBACK( file_menu_save_all_callback ), NULL },

	{ "Close Site", N_("Close"),
	  GTK_STOCK_CLOSE, NULL, N_("Close a site"), 
	  G_CALLBACK( file_menu_close_site_callback ), NULL },

	{ "Open Document", N_("Open..."),
	  GTK_STOCK_OPEN, "<Control>O", N_("Open a document"), 
	  G_CALLBACK( file_menu_open_document_callback ), NULL },

	  { "Open Location", N_("Open Location..."),
	  GTK_STOCK_OPEN, "<Control>L", N_("Open a document from a specified location"), 
	  G_CALLBACK( file_menu_open_location_callback ), NULL },

	{ "Save Document", N_("Save"),
	  GTK_STOCK_SAVE, "<Control>S",  N_("Save a document"), 
	  G_CALLBACK( file_menu_save_document_callback ), NULL },

	{ "Save Document As", N_("Save As..."),
	  GTK_STOCK_SAVE_AS, "<Shift><Control>S", N_("Save document as"), 
	  G_CALLBACK( file_menu_save_document_as_callback ), NULL },

	  { "Save Template", N_("Save As Template..."),
	  GTK_STOCK_SAVE_AS, NULL, N_("Save document as a template"), 
	  G_CALLBACK( file_menu_save_template_as_callback ), NULL },

	{ "Close Document", N_("Close"),
	  GTK_STOCK_CLOSE, "<Control>W", N_("Close a document"),
	  G_CALLBACK( file_menu_close_document_callback ), NULL },

	{ "CloseAll", N_( "Close All" ),
	  GTK_STOCK_CLOSE, "<Control><Shift>W",
	  N_( "Close all open documents in the current site" ),
	  G_CALLBACK( file_menu_close_all_callback ), NULL },
	  
	{ "Print", N_("Print..."),
	  GTK_STOCK_PRINT, "<Control>P",  N_("Print the current view"),
	  G_CALLBACK( file_menu_print_callback ), NULL },

	{ "PrintPreview", N_("Print Preview"),
	  GTK_STOCK_PRINT_PREVIEW, "<Shift><Control>P",
	  N_("Preview Printing of the current view"), 
	  G_CALLBACK( file_menu_print_preview_callback ), NULL },

	{ "PageSetup", N_("Page Set_up"),
	  NULL, NULL,
	  N_("Setup page printing settings"), 
	  G_CALLBACK( file_menu_page_setup_callback ), NULL },

	{ "Offline", N_("Work Offline"),
	  "Screem_Offline", NULL,  N_("Prevent Screem from attempting to load remote files"),
	  G_CALLBACK( file_menu_offline_callback ), NULL },

	{ "Online", N_("Work Online"),
	  "Screem_Online", NULL, N_("Allow Screem to attempt to load remote files"), 
	  G_CALLBACK( file_menu_offline_callback ), NULL },

	{ "Exit", N_("_Quit"),
	  GTK_STOCK_QUIT, "<Control>Q",  N_("Quit Screem"), 
	  G_CALLBACK( file_menu_exit_application_callback ), NULL },

	{ "Undo", N_("Undo"),
	  GTK_STOCK_UNDO, "<Control>Z", N_("Undo"),
	  G_CALLBACK( edit_menu_undo_callback ), NULL },

	{ "Redo", N_("Redo"),
	  GTK_STOCK_REDO, "<Shift><Control>Z", N_("Redo"), 
	  G_CALLBACK( edit_menu_redo_callback ), NULL },

	{ "Cut", N_("Cut"),
	  GTK_STOCK_CUT, "<Control>X", N_("Cut current selection"), 
	  G_CALLBACK( edit_menu_cut_callback ), NULL },

	{ "Copy", N_("Copy"),
	  GTK_STOCK_COPY, "<Control>C", N_("Copy current selection"), 
	  G_CALLBACK( edit_menu_copy_callback ), NULL },

	{ "Paste", N_("Paste"),
	  GTK_STOCK_PASTE, "<Control>V", N_("Paste clipboard contents"),
	  G_CALLBACK( edit_menu_paste_callback ), NULL },

	{ "PasteEncoded", N_("Paste Encoded"),
	  GTK_STOCK_PASTE, NULL,
	  N_("Paste clipboard contents, encoding html entites where needed"),
	  G_CALLBACK( edit_menu_paste_enc_callback ), NULL },

	{ "Clear", N_("Delete"),
	  GTK_STOCK_DELETE, NULL,  N_("Erase the current selection"), 
	  G_CALLBACK( edit_menu_clear_callback ), NULL },

	{ "SelectAll", N_("Select All"),
	  NULL, "<Control>A",  N_("Select All"), 
	  G_CALLBACK( edit_menu_select_all_callback ), NULL },

	{ "SelectContext", N_("Select Context"),
	  NULL, "<Shift><Control>A",
	  N_("Select the element which the cursor is currently within"),
	  G_CALLBACK( edit_menu_select_context_callback ), NULL },

	{ "Find", N_("Find..."),
	  GTK_STOCK_FIND, "<Control>F", N_("Find matches in document(s)"),
	  G_CALLBACK( edit_menu_search_callback ), NULL },

	{ "FindAgain", N_("Find Again"),
	  GTK_STOCK_FIND, "<Control>G", N_("Find matches in document(s)"),
	 G_CALLBACK( edit_menu_search_callback ), NULL },
	  
	{ "Replace", N_("Replace..."),
	  GTK_STOCK_FIND_AND_REPLACE, "<Control>H",
	  N_("Find and Replace matches in document(s)"),
	  G_CALLBACK( edit_menu_search_callback ), NULL },

	{ "GotoLine", N_( "Goto _Line..." ),
	  GTK_STOCK_JUMP_TO, NULL,  N_( "Goto a specified line" ),
	  G_CALLBACK( edit_menu_goto_line_callback ), NULL },

	  
	{ "Bookmarks", N_("Bookmarks"),
	  "Screem_Bookmarks", "<Control>B", N_("Edit Bookmarks"), 
	  G_CALLBACK( edit_menu_bookmarks_callback ), NULL },

	{ "MarkEditable", N_("Create Editable Region..."),
	  NULL, NULL,
	  N_("Add a new editable region, or make a region out of the selected text"),
	  G_CALLBACK( edit_menu_mark_editable_callback ), NULL },

	{ "Auto Indent", N_( "Auto Indent" ),
	  NULL, NULL,
	  N_("Indent block according to the document tree"),
	  G_CALLBACK( edit_menu_auto_indent_callback ), NULL },
	{ "Indent", N_( "Indent" ),
	  NULL, "<Shift><Control>I",
	  N_("Indent block"),
	  G_CALLBACK( edit_menu_indent_callback ), NULL },
	{ "UnIndent", N_( "Unindent" ),
	  NULL, "<Shift><Control>U",
	  N_("Unindent block"),
	  G_CALLBACK( edit_menu_unindent_callback ), NULL },
	{ "Encode Entities", N_( "Encode Entities" ),
	  NULL, NULL,
	  N_("Encode necessary characters as entities"),
	  G_CALLBACK( edit_menu_encode_entities_callback ), NULL },
	{ "URL Encode", N_( "URL Encode" ),
	  NULL, NULL,
	  N_("Encode using URL encoding. eg. %20 for space"),
	  G_CALLBACK( edit_menu_url_encode_callback ), NULL },
	{ "TagLower", N_( "Tags to lowercase" ),
	  NULL, NULL,
	  N_("Change all tags to lowercase"),
	  G_CALLBACK( edit_menu_case_callback ), NULL },
	{ "TagUpper", N_( "Tags to uppercase" ),
	  NULL, NULL,
	  N_("Change all tags to uppercase"),
	  G_CALLBACK( edit_menu_case_callback ), NULL },
	  
	{ "BonoboCustomize", N_("Toolbars"),
	  NULL, NULL,
	  N_("Edit toolbar display and appearance"), 
	  G_CALLBACK( edit_menu_toolbars_callback ), NULL},

	{ "EditMacros", N_( "Macros" ),
	  NULL, NULL,
	  N_( "Edit Macros" ),
	  G_CALLBACK( edit_macros_callback ), NULL },
	  
	{ "Preferences", N_("_Preferences"),
	  GTK_STOCK_PREFERENCES, NULL,  N_("Edit Preferences for Screem"), 
	  G_CALLBACK( edit_menu_preferences_callback ), NULL },

	{ "DockItems", N_("Dock Items"),
	  NULL, NULL, NULL,
	  G_CALLBACK( NULL ), NULL },
	{ "ViewMessages", N_("Messages"),
	  NULL, "<Control>F1", N_("Show/Hide the Messages dockitem"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewErrors", N_("Errors"),
	  NULL, "<Control>F2", N_("Show/Hide the Messages dockitem"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewShortcut", N_("Shortcut bar"),
	  NULL, "<Control>F3", N_("Show/Hide the Shortcut bar"),
	  G_CALLBACK( view_menu_dock_show_callback), NULL, TOGGLE_ACTION },
	{ "ViewFiles", N_("Files"),
	  NULL, "<Control>F4", N_("Show/Hide the file browser"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewResources", N_("Resources"),
	  NULL, "<Control>F5", N_("Show/Hide the resources browser"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewTree", N_("Structure"),
	  NULL, "<Control>F6", N_("Show/Hide the document structure browser"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewAttributes", N_("Attributes"),
	  NULL, "<Control>F7", N_("Show/Hide the tag attribute editor"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewSymbols", N_("Symbols"),
	  NULL, "<Control>F8", N_("Show/Hide the ctags browser"),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	{ "ViewMainToolbar", N_( "Main Toolbar" ),
	  NULL, "<Shift><Control>T", N_("Show/Hide the Main Toolbar" ),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
  	{ "ViewWizardToolbar", N_( "Wizard Toolbar" ),
	  NULL, NULL, N_("Show/Hide the Wizards Toolbar" ),
	  G_CALLBACK( view_menu_dock_show_callback ), NULL, TOGGLE_ACTION },
	  
	{ "SwitchEditor", N_("_Editor"),
	  NULL, NULL, N_("Switch to the editor view"),
	  G_CALLBACK( view_menu_editor_callback ), NULL, RADIO_ACTION, NULL},

	{ "SwitchPreview", N_("_Preview"),
	  NULL, NULL, N_("Switch to the preview view"),
	  G_CALLBACK( view_menu_preview_callback ), NULL, RADIO_ACTION, "SwitchEditor" },

	{ "SwitchLinkView", N_("_Link View"),
	  "Screem_LinkView", NULL, N_("Switch to the link view"),
	  G_CALLBACK( view_menu_linkview_callback ), NULL, RADIO_ACTION, "SwitchEditor" },

	{ "SwitchTreeView", N_("_Tree View"),
	  NULL, NULL,  N_("Switch to the tree view"),
	  G_CALLBACK( view_menu_treeview_callback ), NULL, RADIO_ACTION, "SwitchEditor" },

	{ "Browser0", N_("External Preview"),
	  "Screem_Preview", NULL, N_("Preview in Default external browser"),
	  G_CALLBACK( screem_toolbar_external_browser ), NULL },

	{ "TaskList", N_("Site Tasks"),
	  "Screem_Todo", NULL,  N_("Tasks to be performed on the site"),
	  G_CALLBACK( toolbar_task_list_callback ), NULL},

	{ "SpellCheck", N_("Spell Check"),
	  GTK_STOCK_SPELL_CHECK, NULL, N_("Spell check the current document"), 
	  G_CALLBACK( screem_toolbar_spell_check ), NULL},

	{"Bold", N_("Bold"),
	 GTK_STOCK_BOLD, NULL,
	 N_("Bold Text"), G_CALLBACK( quickbar_bold_callback ), NULL},

	{"Italic", N_("Italic"),
	 GTK_STOCK_ITALIC, NULL,
	 N_("Italic Text"), G_CALLBACK( quickbar_italic_callback ), NULL},

	{"Underline", N_("Underline"),
	 GTK_STOCK_UNDERLINE, NULL,
	 N_("Underlined Text"), G_CALLBACK( quickbar_underline_callback ), NULL},

	{"Para", N_("Paragraph"),
	 "Screem_Paragraph", NULL,
	 N_("Paragraph"), G_CALLBACK( quickbar_para_callback ), NULL},

	{"Bullet", N_("Bullet List"),
	 GNOME_STOCK_TEXT_BULLETED_LIST, NULL,
	 N_("Bulleted List"), G_CALLBACK( quickbar_bullet_callback ), NULL},

	{"Numbered", N_("Numbered List"),
	 GNOME_STOCK_TEXT_NUMBERED_LIST, NULL,
	 N_("Numbered List"), G_CALLBACK( quickbar_numbered_callback ), NULL},

	{"AlignLeft", N_("Left Align"),
	 GTK_STOCK_JUSTIFY_LEFT, NULL,
	 N_("Align Text Left"), G_CALLBACK( quickbar_alignleft_callback ), NULL},

	{"AlignCentre", N_("Centre Align"),
	 GTK_STOCK_JUSTIFY_CENTER, NULL,
	 N_("Align Text Centre"), G_CALLBACK( quickbar_aligncentre_callback ), NULL},

	{"AlignRight", N_("Right Align"),
	 GTK_STOCK_JUSTIFY_RIGHT, NULL,
	 N_("Align Text Right"), G_CALLBACK( quickbar_alignright_callback ), NULL},

	{"Justify", N_("Justify"),
	 GTK_STOCK_JUSTIFY_FILL, NULL,
	 N_("Justify Text"), G_CALLBACK( quickbar_justify_callback ), NULL},

	{"Font", N_("Font"),
	 GTK_STOCK_SELECT_FONT, NULL,
	 N_("Font Attributes"), G_CALLBACK( fontbar_font_callback ), NULL},

	{"Pre", N_("Pre"),
	 "Screem_Pre", NULL,
	 N_("Preformatted Text"), G_CALLBACK( fontbar_pre_callback ), NULL},

	{"Subscript", N_("Subscript"),
	 "Screem_Sub", NULL,
	 N_("Subscripted Text"), G_CALLBACK( fontbar_subscript_callback ), NULL},

	{"Superscript", N_("Superscript"),
	 "Screem_Sup", NULL,
	 N_("Superscripted Text"), G_CALLBACK( fontbar_superscript_callback ), NULL},

	{"Table", N_("Table"),
	 "Screem_Table", NULL,
	 N_("Table"), G_CALLBACK( tablebar_table_callback ), NULL},

	{"TBody", N_("Table Body"),
	 "Screem_Table", NULL,
	 N_("Table Body"), G_CALLBACK( tablebar_tablebody_callback ), NULL},

	{"Tr", N_("Table Row"),
	 "Screem_Tr", NULL,
	 N_("Table Row"), G_CALLBACK( tablebar_tr_callback ), NULL},

	{"Th", N_("Table Heading"),
	 "Screem_Th", NULL,
	 N_("Table Heading"), G_CALLBACK( tablebar_th_callback ), NULL},

	{"Td", N_("Table Column"),
	 "Screem_Td", NULL,
	 N_("Table Column"), G_CALLBACK( tablebar_td_callback ), NULL},

	{"Caption", N_("Table Caption"),
	 "Screem_Caption", NULL,
	 N_("Table Caption"), G_CALLBACK( tablebar_caption_callback ), NULL},

	{"Summary", N_("Table Summary"),
	 "Screem_Caption", NULL,
	 N_("Table Summary"), G_CALLBACK( tablebar_summary_callback ), NULL},

	{"Textarea", N_("Textarea"),
	 "Screem_Text", NULL,
	 N_("Text Area"), G_CALLBACK( formbar_textarea_callback ), NULL},

	{"Input", N_("Input"),
	 "Screem_Entry", NULL,
	 N_("Text Input"), G_CALLBACK( formbar_input_callback ), NULL},

	{"Fileinput", N_("File Input"),
	 "Screem_FileEntry", NULL,
	 N_("File Input"), G_CALLBACK( formbar_fileinput_callback ), NULL},

	{"Button", N_("Button"),
	 "Screem_Button", NULL,
	 N_("Button"), G_CALLBACK( formbar_button_callback ), NULL},

	{"Checkbutton", N_("Checkbutton"),
	 "Screem_Checkbutton", NULL,
	 N_("Checkbutton"), G_CALLBACK( formbar_checkbutton_callback ), NULL},

	{"Radiobutton", N_("Radiobutton"),
	 "Screem_Radiobutton", NULL,
	 N_("Radiobutton"), G_CALLBACK( formbar_radiobutton_callback ), NULL},

	{"Optionmenu", N_("Optionmenu"),
	 "Screem_Optionmenu", NULL,
	 N_("Optionmenu"), G_CALLBACK( formbar_optionmenu_callback ), NULL},

	{ "Basic Structure", N_( "Basic Structure" ),
	  NULL, NULL,
	  N_( "Insert basic HTML page, based on the default doctype" ),
	  G_CALLBACK( insert_basic_html_callback ), NULL },
	 
	{ "Generator", N_( "Generator meta" ),
	  NULL, NULL,
	  N_( "Insert a generator meta tag" ),
	  G_CALLBACK( meta_insert_callback ), NULL },

	{ "Author", N_( "Author meta" ),
	  NULL, NULL,
	  N_( "Insert an author meta tag" ),
	  G_CALLBACK( meta_insert_callback ), NULL },

	{ "Content-Type", N_( "Content-Type meta" ),
	  NULL, NULL,
	  N_( "Insert a content type meta tag" ),
	  G_CALLBACK( meta_insert_callback ), NULL },
	
	{ "MetaRefresh", N_( "Refresh meta" ),
	  NULL, NULL,
	  N_( "Insert a refresh meta tag" ),
	  G_CALLBACK( meta_insert_callback ), NULL },
	  
	{"InsertDoctype", N_("_Doctype tag..."),
	 NULL, NULL,
	 N_("Insert a <!DOCTYPE>"), G_CALLBACK( insert_menu_doctype_callback ), NULL},

	{"InsertFile", N_("_From File..."),
	 NULL, NULL,
	 N_("Insert text from a file"), G_CALLBACK( insert_menu_file_callback ), NULL},

	{"CheckoutSite", N_("Checkout Site"),
	 "Screem_CVSCheckout", NULL,
	 N_("Checkout a Site from CVS"), G_CALLBACK( cvs_menu_checkout_site_callback ), NULL},

	{"UpdateSite", N_("Update Site"),
	 "Screem_CVSUpdate", NULL,
	 N_("Update a Site from CVS"), G_CALLBACK( cvs_menu_update_site_callback ), NULL},

	{"UpdatePage", N_("Update Page"),
	 "Screem_CVSUpdate", NULL,
	 N_("Update a Page from CVS"), G_CALLBACK( cvs_menu_update_page_callback ), NULL},

	{"ImportSite", N_("Import Site"),
	 NULL, NULL,
	 N_("Place a Site in CVS"), G_CALLBACK( cvs_menu_import_site_callback ), NULL},

	{"CommitSite", N_("Commit Site"),
	 NULL, NULL,
	 N_("Commit Site changes to CVS"), G_CALLBACK( cvs_menu_commit_site_callback ), NULL},

	{"CommitPage", N_("Commit Page"),
	 NULL, NULL,
	 N_("Commit Page changes to CVS"), G_CALLBACK( cvs_menu_commit_page_callback ), NULL},

	{"AddPage", N_("Add Page"),
	 "Screem_CVSAdd", NULL,
	 N_("Add Page to CVS"), G_CALLBACK( cvs_menu_add_page_callback ), NULL},

	{ "AddFile", N_("Add File"),
	  "Screem_CVSAdd", NULL, N_("Add File to CVS"),
	  G_CALLBACK( cvs_menu_add_page_callback ), NULL },

	{ "CommitFile", N_("Commit File"),
	  NULL, NULL, N_("Commit File to CVS"),
	  G_CALLBACK( cvs_menu_commit_page_callback ), NULL },

	{ "UpdateFile", N_("Update File"),
	  "Screem_CVSUpdate", NULL, N_("Update File from CVS"),
	  G_CALLBACK( cvs_menu_update_page_callback ), NULL },

	{ "CVSLogin", N_("Login"),
	  NULL, NULL, N_("Login to a CVS server"),
	  G_CALLBACK( cvs_menu_login_callback ), NULL },

	{ "FixLinks", N_( "Fix Links..." ),
	  NULL, NULL, 
	  N_( "Update all links to a source to point to a new location" ),
	  G_CALLBACK( tool_menu_fix_links_callback ), NULL },

	{ "TemplateUpdate", N_( "Update from Template" ),
	  NULL, NULL, 
	  N_( "Updates the page from the template it is using" ),
	  G_CALLBACK( tool_menu_template_update_callback ), NULL },

	{ "EditHelpers", N_( "Edit Helpers" ),
	  NULL, NULL,
	  N_( "Edit helpers" ),
	  G_CALLBACK( edit_helpers_callback ), NULL },

	{ "PreviousDocument", N_( "_Previous Document" ),
	  GTK_STOCK_GO_BACK, "<Control>Page_Up", 
	  N_( "Switch to the previous open document" ),
	  G_CALLBACK( doc_menu_previous_callback ), NULL },

	{ "NextDocument", N_( "_Next Document" ),
	  GTK_STOCK_GO_FORWARD, "<Control>Page_Down", 
	  N_( "Switch to the next open document" ),
	  G_CALLBACK( doc_menu_next_callback ), NULL },
	  
	{ "Help", N_("Contents"),
	  GTK_STOCK_HELP, "F1", N_("Help On this application"), 
	  G_CALLBACK( help_menu_help_callback ), NULL },

	{ "About Screem", N_("_About Screem"),
	  GNOME_STOCK_ABOUT, NULL,
	  N_("Display credits for the creators of Screem"), 
	  G_CALLBACK( help_menu_about_screem_callback ), NULL },

	{ "Tip Of The Day", N_("Tip Of The Day"),
	  GTK_STOCK_DIALOG_INFO, NULL,  N_("Show the Tip Of The Day"),
	  G_CALLBACK( help_menu_tip_callback ), NULL },

	{ "Debug", N_("Debug"),
	  GTK_STOCK_DIALOG_INFO, NULL,  N_("Debug"),
	  G_CALLBACK( help_menu_debug_callback ), NULL },

	{ "New Directory", N_("New Directory"),
	  GTK_STOCK_NEW, NULL, NULL, 
	  G_CALLBACK( site_view_menu_new_dir_callback ), NULL },

	{ "Delete File", N_("Delete"),
	  GTK_STOCK_DELETE, NULL, NULL, 
	  G_CALLBACK( site_view_menu_delete_file_callback ), NULL },

	{ "File Properties", N_("Properties"),
	  GTK_STOCK_PROPERTIES, NULL, NULL, 
	  G_CALLBACK( site_view_menu_file_props_callback ), NULL },

	{ "SiteSettings", N_("_Site Settings"),
	  GTK_STOCK_PROPERTIES, NULL, N_("Edit Settings for the current Site"), 
	  G_CALLBACK( site_menu_site_settings_callback ), NULL },

	{ "ExcludeFlag", N_("Exclude"),
	  NULL, NULL, NULL, 
	  G_CALLBACK( NULL ), NULL, TOGGLE_ACTION },

	{ "IgnoreFlag", N_("Ignore"),
	  NULL, NULL, NULL,
	  G_CALLBACK( NULL ), NULL, TOGGLE_ACTION },

	{ "ASCIIFlag", N_("ASCII"),
	  NULL, NULL, NULL,
	  G_CALLBACK( NULL ), NULL, TOGGLE_ACTION }
};
static 
guint screem_window_action_n_entries=G_N_ELEMENTS(screem_window_action_entries);


void screem_window_initialise_menus( ScreemWindow *window )
{
	gint i;
	EggMenuMerge *merge;
	EggAction *action;
	gint major;
	gint minor;
	gint patch;
	gboolean show;
	
	window->action_group = egg_action_group_new( "ScreemActions" );
	
	/* set user data field in action table */
	for( i = 0; i < screem_window_action_n_entries; ++ i ) {
		screem_window_action_entries[ i ].user_data = window;
	}

	egg_action_group_add_actions( EGG_ACTION_GROUP( window->action_group ),
					screem_window_action_entries,
					screem_window_action_n_entries );

	/* disable debug action for stable versions */
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "Debug" );
	show = TRUE;
	major = minor = patch = 0;
	if( sscanf( VERSION, "%i.%i.%i", &major, &minor, &patch ) > 2 ){
		show = ( minor % 2 );
	}
	g_object_set( G_OBJECT( action ),
			"visible", show, NULL );
	
	
	merge = egg_menu_merge_new ();
	egg_menu_merge_insert_action_group( merge, 
					    EGG_ACTION_GROUP( window->action_group ), 
					    0 );
	window->merge = G_OBJECT( merge );
	
	g_signal_connect( merge, "add_widget", G_CALLBACK( add_widget ), window );

  	egg_menu_merge_add_ui_from_file( merge, 
					 UIDATADIR"/screem-window-menu.xml", 
					 NULL );

	egg_menu_merge_add_ui_from_file( merge,
					 UIDATADIR"/screem-window-toolbar.xml",
					 NULL );
	
	egg_menu_merge_add_ui_from_file( merge,
					 UIDATADIR"/screem-window-popups.xml",
					 NULL );
		
	screem_toolbar_create_browsers( window );

	gtk_window_add_accel_group( GTK_WINDOW( window ), merge->accel_group );
}

